/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_channel_list.cpp
* @brief       Implementation for channel list related functionalities and maintainance of
*              information pertaining to all channels reported by SMS. 
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_common.h"
#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_channel_list.h"
#include "fc_sxm_tcl_advisories.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_tcl_states.h"
#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_category_list_if.h"
#include "fc_sxm_tcl_presets_if.h"
#include "fc_sxm_tcl_playback_if.h"
#include "fc_sxm_tcl_content_alert_proxy.h"
#include "fc_sxm_tcl_audio_utils.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_CHANNEL_LIST
#include "trcGenProj/Header/fc_sxm_tcl_channel_list.cpp.trc.h"
#endif

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

static tVoid cb_DecoderChannelAccessCallback(DECODER_OBJECT /* hDecoder */, CHANNEL_OBJECT hChannel, void * /* pvChannelAccessCallbackArg*/)
{
   ETG_TRACE_USR4(("cb_DecoderChannelAccessCallback()"));

   CHANNEL_EVENT_MASK tEventMask = (CHANNEL_OBJECT_EVENT_SERVICE_ID | CHANNEL_OBJECT_EVENT_CHANNEL_ID | CHANNEL_OBJECT_EVENT_ATTRIBUTES | 
                                    CHANNEL_OBJECT_EVENT_NAME | CHANNEL_OBJECT_EVENT_CATEGORY);

   fc_sxm_trMsgChannelEventCallback prMsgChannelEventUpdate = fc_sxm_trMsgChannelEventCallback(hChannel, tEventMask);
   fc_sxm_tclAudioApp::instance()->vPostMsgNew(prMsgChannelEventUpdate);
}

/*****************************************************************************
 * FUNCTION    : fc_sxm_tclChannelList                                       *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class constructor                                           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclChannelList::fc_sxm_tclChannelList()
    : _bChannelListReady(FALSE)
    , _bUpdateChannelList(FALSE)
    , _bChannelListModified(FALSE)
    , _bIsAudioServiceSubscribed(FALSE)
    , _bChannelArtUpdate(FALSE)
    , _bEnableMatureChannels(FALSE)
    , _u16TunedChannelChannelID(FC_SXM_AUDIO_DEFAULT_CHANNEL)
    , _u16TunedChannelServiceID(SERVICE_INVALID_ID)
    , _u16TunedChannelCategoryID(CATEGORY_INVALID_ID)
    , _poAudioApp(OSAL_NULL)

{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList constructor"));

   /* Mute audio */
   //fc_sxm_arl_tclISource::instance()->fc_sxm_SetMute(TRUE);
}

/*****************************************************************************
 * FUNCTION    : ~fc_sxm_tclChannelList                                      *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class destructor                                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclChannelList::~fc_sxm_tclChannelList()
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList destructor"));

    /* Set all the class member variables to default values */
    _u16RequestedUnavailableChannel   = CHANNEL_INVALID_ID;
    _bChannelListReady                = FALSE;
    _bUpdateChannelList               = FALSE;
    _bChannelListModified             = FALSE;
    _bIsAudioServiceSubscribed        = FALSE;
    _u16TunedChannelChannelID         = CHANNEL_INVALID_ID;
    _u16TunedChannelServiceID         = SERVICE_INVALID_ID;
    _u16TunedChannelCategoryID        = CATEGORY_INVALID_ID;
    _poAudioApp                       = OSAL_NULL;
}

/*****************************************************************************
 * FUNCTION    : vUpdateDecoderState                                         *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to start channel list creation and initiate          *
 *               channel updates from SMS once decoder is in ready state     *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hDecoderObject - Valid decoder object from SMS service      *
 * PARAMETER   : enDecoderState - Current decoder state                      *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vUpdateDecoderState(DECODER_OBJECT hDecoderObject,DECODER_STATE_ENUM enDecoderState)
{
    ETG_TRACE_USR1(("fc_sxm_tclChannelList::vUpdateDecoderState(%d)",
                   ETG_CENUM(DECODER_STATE_ENUM, enDecoderState)));

   /* Register for channel event notification for the first time */
   if((DECODER_STATE_READY == enDecoderState) && 
      (DECODER_INVALID_OBJECT != hDecoderObject)) {
      ETG_TRACE_USR4(("Registring for channel object notifications"));

      // Clear the SXM Initializing advisory
      fc_sxm_tclAdvisories::instance()->vClearSXMInitializingAdvisory();

      /* Set channel list availability flag to true */
      _bChannelListReady = TRUE;
      /* Display current tuned channel information */
      vUpdateCurrentTunedChannel();
      if(FC_SXM_AUDIO_RADIO_ID_CHANNEL == _u16TunedChannelChannelID) {
         _u16TunedChannelChannelID = FC_SXM_AUDIO_DEFAULT_CHANNEL;
         vRestorePreviewChannel();
      }else {
         vNotifyChannelInformationUpdate(e_ADVISORY_UPDATE);
      }

      /* Notify auto compare properties */
      vNotifyTunedChannelPreset(u16GetSIDOfTunedChannel(), FALSE);

      /* Channel list is created and the same is updated to HMI here */
      _bChannelListModified = TRUE;
      vNotifyChannelListUpdate();
      vSetChannelBrowseStatus();
      /* Mute audio */
      //fc_sxm_arl_tclISource::instance()->fc_sxm_SetMute(FALSE);
   } 
}

/* Clear class member variables */
tVoid fc_sxm_tclChannelList::vClearMemberData(tVoid)
{
   ETG_TRACE_USR1(("fc_sxm_tclChannelList::vClearMemberData()"));

   /* Set the status info and notify all clients */
   vTriggerChannelListStatus(FALSE);

   /* Mute audio */
   //fc_sxm_arl_tclISource::instance()->fc_sxm_SetMute(TRUE);

   /* Clear member variables */
   _u16RequestedUnavailableChannel   = CHANNEL_INVALID_ID;
   _bChannelListReady                = FALSE;
   _bUpdateChannelList               = FALSE;
   _bChannelListModified             = FALSE;
   _bIsAudioServiceSubscribed        = FALSE;
   _bChannelArtUpdate                = FALSE;
   _u16TunedChannelChannelID         = FC_SXM_AUDIO_DEFAULT_CHANNEL;
   _u16TunedChannelServiceID         = SERVICE_INVALID_ID;
   _u16TunedChannelCategoryID        = CATEGORY_INVALID_ID;
   _trCategoryMonitorConfig.vReset();

   /* Clear channel list */
   _mChannelList.clear();
   _trRapidChnBrowse.vReset();
}

/* Update current tuned channel info */
tVoid fc_sxm_tclChannelList::vUpdateCurrentTunedChannel(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::vUpdateCurrentTunedChannel()"));
   DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
   
   CHANNEL_ID u16ChannelId = DECODER.tCurrentChannelId(hDecoder);
   ETG_TRACE_USR2(("DECODER.tCurrentChannelId - %d", u16ChannelId));

   if(CHANNEL_INVALID_ID == u16ChannelId) {
      u16ChannelId  = DECODER.tLastTunedChannelId(hDecoder);
   }
   if((CHANNEL_INVALID_ID != u16ChannelId) && (!bIsDwellTimerRunning()))
   {
      _u16TunedChannelChannelID  = u16ChannelId;
      _u16TunedChannelServiceID  = DECODER.tCurrentServiceId(hDecoder);
      _u16TunedChannelCategoryID = DECODER.tCurrentCategoryId(hDecoder);
   }
}

/* Check if dwell timer is rumming */
tBool fc_sxm_tclChannelList::bIsDwellTimerRunning(tVoid)
{
   tBool bIsTimerOn = _mChannelSelectDwellTimer.bIsRunning();
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::bIsDwellTimerRunning TimerStatus = %u", bIsTimerOn));
   return bIsTimerOn;
}

/* Stop channel select dwell timer */
tVoid fc_sxm_tclChannelList::vStopDwellTimer(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::vStopDwellTimerRunning"));
   return (_mChannelSelectDwellTimer.vStop());
}

/*****************************************************************************
 * FUNCTION    : vUpdateChannelInfo                                          *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Update channel information in internal channel list         *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hChannelObject - Valid channel object                       *
 * PARAMETER   : enEventMask    - Channel update event data                  *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vUpdateChannelInfo(fc_sxm_trChannel const &rChannel, CHANNEL_EVENT_MASK enEventMask)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vUpdateChannelInfo() CHANNEL_ID = %u \t eventMask = 0x%04x",
            rChannel.u16ChannelId, enEventMask));
    /* Channel deleted */
    if(CHANNEL_OBJECT_EVENT_REMOVED & enEventMask) {
        vRemoveExistingChannel(rChannel);
    } else {
        /* Update other parameters from channel object */
        vModifyChannelParams(rChannel, enEventMask);
    }
    vUpdatePresetBands(rChannel, enEventMask);
}

/*****************************************************************************
 * FUNCTION    : vAddNewChannel                                              *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Add a new channel into internal channel list                *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hChannelObject - Valid channel object                       *
 * PARAMETER   : bGetAllInfo    - Flag to get all attributes or not          *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vAddNewChannel(fc_sxm_trChannel const& rChannel)
{
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::vAddNewChannel()"));


   if(rChannel.bIsValid()) {
       enChannelLockAction enAction = fc_sxm_tclAudioUtils::enGetChannelLockAction(
               rChannel.bIsLocked(), rChannel.bIsMature(), _bEnableMatureChannels);
       if (enChannelAction_Ignore == enAction)
       {
           _mChannelList.insert(pair<CHANNEL_ID, fc_sxm_trChannel>(rChannel.u16ChannelId, rChannel));
       }
       else if (enChannelAction_Lock == enAction)
       {
           fc_sxm_trChannel chnLock = rChannel;
           (tVoid) bLockChannel(chnLock);
           _mChannelList.insert(pair<CHANNEL_ID, fc_sxm_trChannel>(rChannel.u16ChannelId, chnLock));
       }
       else // if (enChannelAction_UnLock == enAction)
       {
           fc_sxm_trChannel chnUnlock = rChannel;
           (tVoid) bUnLockChannel(chnUnlock);
           _mChannelList.insert(pair<CHANNEL_ID, fc_sxm_trChannel>(rChannel.u16ChannelId, chnUnlock));
       }
       // Update channel list modified flag
       _bChannelListModified = TRUE;
       ETG_TRACE_USR2(("Channel added in list"));
   }
   
   if(!_bIsAudioServiceSubscribed) {
      vCheckAudioSubscriptionStatus();
   }

   vNotifyChannelListUpdate();
}

/*****************************************************************************
 * FUNCTION    : vRemoveExistingChannel                                      *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Remove an existing channel from internal channel list       *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hChannelObject - Valid channel object                       *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vRemoveExistingChannel(fc_sxm_trChannel const &rChannel)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vRemoveExistingChannel()"));

    /* Get service ID of the channel object */
    SERVICE_ID u16ServiceID = rChannel.u16ServiceId;
    if(SERVICE_INVALID_ID == u16ServiceID) {
        ETG_TRACE_ERR(("Failed to get service ID from channel object"));
    } else {
        ETG_TRACE_USR4(("Removed channel service ID - %d", u16ServiceID));

        CHANNEL_ID u16ChannelID = CHANNEL_INVALID_ID;
        /* Find for the service ID in the list */
        map<CHANNEL_ID, fc_sxm_trChannel>::iterator mIter;
        for(mIter = _mChannelList.begin();mIter != _mChannelList.end();++mIter) {
            /* Search for the channel ID in the channel list */
            if( u16ServiceID == mIter->second.u16ServiceId) {
                vNotifyCategoryChange(mIter->second, CHANNEL_OBJECT_EVENT_REMOVED, mIter->second.bIsSubscribed());
                /* Get channel id */
                 u16ChannelID = mIter->second.u16ChannelId;
                mIter->second.vChannelRemoved();
                break;
            }
        }

        /* Update current audio subscription status */
        if(_bIsAudioServiceSubscribed) {
           vCheckAudioSubscriptionStatus();
        }

       /* If channel removed is current tuned channel the display off air system message */
       if(_u16TunedChannelChannelID == u16ChannelID) {
           /* Set unauthorised channel as current channel */
           _u16RequestedUnavailableChannel = u16ChannelID;
           /* Set channel unavailable system message  */
           (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, TRUE);
           vNotifyChannelInformationUpdate(e_ADVISORY_UPDATE);
       }

       /* Update channel list modified flag */
       _bChannelListModified = TRUE;
       vNotifyChannelListUpdate();
    }
}

/*****************************************************************************
 * FUNCTION    : vModifyChannelParams                                        *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Modify channel parameters from channel list                 *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hChannelObject   - Valid channel object                     *
 * PARAMETER   : enChannelEvtMask - Channel update event data                *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vModifyChannelParams(fc_sxm_trChannel const& rChannel,CHANNEL_EVENT_MASK enChannelEvtMask)
{
    tBool bChannelInfoModified = FALSE;
    tBool bSubscriptionUpdate = FALSE;

    /* Get channel ID of the channel object */
    CHANNEL_ID u16ChannelID = rChannel.u16ChannelId;
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vModifyChannelParams(mask=0x%04x) u16ChannelID=%d", enChannelEvtMask, u16ChannelID));

    if(CHANNEL_INVALID_ID == u16ChannelID) {
        ETG_TRACE_ERR(("Failed to get channel ID from channel object"));
    } else {
        /* Check channel ID updates */
        if(CHANNEL_OBJECT_EVENT_CHANNEL_ID & enChannelEvtMask) {
            /* Update channel ID */
            vUpdateChannelID(rChannel);
            /* Update channel list modified flag */
            _bChannelListModified = TRUE;
            bChannelInfoModified = TRUE;
        }

        /* Find for the service ID in the list */
        map<CHANNEL_ID, fc_sxm_trChannel>::iterator mIter;
        mIter = _mChannelList.find(u16ChannelID);
        tBool bChannelTuned = (_u16TunedChannelChannelID == u16ChannelID);
        if (mIter == _mChannelList.end()) {
            ETG_TRACE_ERR(("Element not found in channel list : "
                           "Service ID - %d,  Channel ID - %d",
                           rChannel.u16ServiceId, 
                           u16ChannelID));
        } else {
            fc_sxm_trChannel &rListChannel = mIter->second;
            tBool bChnSubscriptionState  = rListChannel.bIsSubscribed();
            if(CHANNEL_OBJECT_EVENT_SERVICE_ID & enChannelEvtMask) {
                rListChannel.u16ServiceId = rChannel.u16ServiceId;  /* Update service ID */
            }   /* Check service ID updates */

            /* Update channel info modified flag */
            bChannelInfoModified = TRUE;

            if(CHANNEL_OBJECT_EVENT_ATTRIBUTES & enChannelEvtMask) {
                rListChannel.vCopyAttributes(rChannel); /* Update channel attribute changes */
                bSubscriptionUpdate = TRUE;

                /* If current tuned channel become unsubscribed then send system message */
                if( bChannelTuned && _bChannelListReady ) {
                    vUpdateTunedChannelAdvisory(u16ChannelID, rChannel);
                }
            }   /* Check for attribute updates */
            if(CHANNEL_OBJECT_EVENT_NAME & enChannelEvtMask) {
                rListChannel.oAudioChannelName = rChannel.oAudioChannelName; /* Update channel name info */
                _bChannelListModified = TRUE; /* Update channel list modified flag */
            }   /* Check for channel name updates */
            if(CHANNEL_OBJECT_EVENT_CATEGORY & enChannelEvtMask) {
                rListChannel.vCopyCategoryInfo(rChannel); /* Update category info */
                /* If it is current tuned channel then update its category info */
                if(bChannelTuned) {
                    _u16TunedChannelCategoryID = rListChannel.u16CategoryId;
                }
            }   /* Check for category updates */
            if(CHANNEL_OBJECT_EVENT_TITLE & enChannelEvtMask) {
                rListChannel.oChannelSongName=rChannel.oChannelSongName; /* Update title details */
            }   /* Check for title updates */
            if(CHANNEL_OBJECT_EVENT_ARTIST & enChannelEvtMask) {
                rListChannel.oChannelArtistName=rChannel.oChannelArtistName; /* Update artist details */
            }   /* Check for artist updates */
            if(CHANNEL_OBJECT_EVENT_ART & enChannelEvtMask) {
               rListChannel.bIsChannelArtAvailable=rChannel.bIsChannelArtAvailable; /* Update artist details */
               rListChannel.bIsAlbumArtAvailable = rChannel.bIsAlbumArtAvailable;
               ETG_TRACE_USR4(("fc_sxm_tclChannelList bIsChannelArtAvailable %d", rListChannel.bIsChannelArtAvailable));
               ETG_TRACE_USR4(("fc_sxm_tclChannelList bIsAlbumArtAvailable %d", rListChannel.bIsAlbumArtAvailable));
               ETG_TRACE_USR4(("fc_sxm_tclChannelList u16ChannelId %d", rChannel.u16ChannelId));               
               _bChannelArtUpdate = TRUE;
            }   /* Check for channel art updates */
            // Check if category change trigger has to be made
            vNotifyCategoryChange(rListChannel, enChannelEvtMask, bChnSubscriptionState);
        }

        /* Update current audio subscription status */
        if(bSubscriptionUpdate) {
           vCheckAudioSubscriptionStatus();
        }

         if(!bChannelTuned) {
             /* Do not notify HMI if the update is not for current tuned channel */
             bChannelInfoModified = FALSE;
         } else if(CHANNEL_OBJECT_EVENT_CHANNEL_ID & enChannelEvtMask) {
             if(_u16TunedChannelServiceID != rChannel.u16ServiceId) {
                bChannelInfoModified = FALSE;
             }
         }

         if( (bChannelInfoModified) && (!bIsDwellTimerRunning())) {
            if(_bChannelArtUpdate) {
                vNotifyChannelInformationUpdate(e_CHANNEL_ART_UPDATE);
            }
            vNotifyChannelInformationUpdate(e_ADVISORY_UPDATE);
            if(bSubscriptionUpdate) {
               vNotifyChannelInformationUpdate(e_ADVISORY_UPDATE);
            }
         }
    }
}

/*****************************************************************************
 * FUNCTION    : vUpdateChannelID                                               *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID from channel object                          *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hChannelObject - Valid channel object                       *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vUpdateChannelID(fc_sxm_trChannel const& rChannel)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vUpdateChannelID()"));

    SERVICE_ID u16ServiceID = rChannel.u16ServiceId;
    CHANNEL_ID u16ChannelID = rChannel.u16ChannelId;

    if((CHANNEL_INVALID_ID == u16ChannelID) || (SERVICE_INVALID_ID) == u16ServiceID) {
        ETG_TRACE_ERR(("Failed to get service ID and channel ID from the channel object"));
    } else {

        /* List iterator */
        map<CHANNEL_ID, fc_sxm_trChannel>::iterator mIter;

        /* Find if the entry already exists and remove old data */
        mIter = _mChannelList.find(u16ChannelID);
        if(mIter != _mChannelList.end()) {
            _mChannelList.erase(mIter);
        }

        /* Get service ID whose channel ID got modified */
        for(mIter = _mChannelList.begin(); mIter != _mChannelList.end(); ++mIter) {
            /* Fetch corresponding element service ID */
            if(u16ServiceID == ((*mIter).second).u16ServiceId) {
                _mChannelList.erase(mIter);
                break;
            }
        }

        /* Then add this as a new entry as key got
           modified in the list */
        vAddNewChannel(rChannel); 

        /* If the update is for current tuned service ID then update channel ID accordingly */
        if(_u16TunedChannelServiceID == u16ServiceID) {
            _u16TunedChannelChannelID = u16ChannelID;
        }
    }
}


/*****************************************************************************
 * FUNCTION    : u16GetCurrentTunedChannel                                   *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get current tuned channel ID                                *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : CHANNEL_ID - Current tuned channel ID                       *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclChannelList::u16GetCurrentTunedChannel(tVoid) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u16GetCurrentTunedChannel() _u16TunedChannelChannelID = %u", _u16TunedChannelChannelID));

    return _u16TunedChannelChannelID;
}

/*****************************************************************************
 * FUNCTION    : u16GetSIDOfTunedChannel                                     *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get SID of current tuned channel                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : SERVICE_ID - Service ID of current tuned channel            *
 *****************************************************************************/
SERVICE_ID fc_sxm_tclChannelList::u16GetSIDOfTunedChannel(tVoid) const
{
    SERVICE_ID u16ServiceID = SERVICE_INVALID_ID;

    /* Find for tuned channel from channel list */
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
    mIter = _mChannelList.find(_u16TunedChannelChannelID);

    if (mIter != _mChannelList.end())
    {
        u16ServiceID = mIter->second.u16ServiceId;
    }
    else
    {
        ETG_TRACE_ERR(("Failed to find tuned channel in channel list"));
    }
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u16GetSIDOfTunedChannel() servID = %u", u16ServiceID));

    return u16ServiceID;
}

SERVICE_ID fc_sxm_tclChannelList::u16GetSIDOfPreviewChannel(tVoid) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u16GetSIDOfPreviewChannel()"));

    SERVICE_ID u16ServiceID = SERVICE_INVALID_ID;

    /* Find for tuned channel from channel list */
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
    mIter = _mChannelList.find(FC_SXM_AUDIO_DEFAULT_CHANNEL);

    if (mIter != _mChannelList.end())
    {
        u16ServiceID = mIter->second.u16ServiceId;
    }
    else
    {
        ETG_TRACE_ERR(("Failed to find preview channel in channel list"));
    }

    return u16ServiceID;
}

/*****************************************************************************
 * FUNCTION    : enGetPresetChannelInfo                                      *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get details of preset channel and publish if tuned          *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : u16ServiceID  - Valid service ID for preset info            *
 * PARAMETER   : bChannelTuned - Flag to handle notifications                *
 * RETURNVALUE : midw_ext_fi_tcl_PresetListEntry - Preset info               *
 *****************************************************************************/
midw_ext_fi_tcl_PresetListEntry 
    fc_sxm_tclChannelList::enGetPresetChannelInfo(
        SERVICE_ID u16ServiceID, 
        tBool bChannelTuned, tTunable& tDisplayAdvisory)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::enGetPresetChannelInfo() ServiceID(%u) \t bChannelTuned(%u)",
            u16ServiceID, bChannelTuned));

    tDisplayAdvisory = e_UNAVAILABLE;

    /* Copy of preset list entry */
    midw_ext_fi_tcl_PresetListEntry lPresetEntry;

    /* Fill default data in preset list entry */
    lPresetEntry.ServiceID = SERVICE_INVALID_ID;
    lPresetEntry.ChannelID = CHANNEL_INVALID_ID;
    fc_sxm_vString2Fi("", lPresetEntry.ChannelName);
    lPresetEntry.ChannelActive = FALSE;
    DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();

    // Find for tuned channel from channel list
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter = _mChannelList.begin();
    for ( ; mIter != _mChannelList.end(); ++mIter) {
    if(u16ServiceID == ((*mIter).second).u16ServiceId) {
        const fc_sxm_trChannel& rChannel = mIter->second;
        // Fill channel info
        lPresetEntry.ServiceID = rChannel.u16ServiceId;
        lPresetEntry.ChannelID =rChannel.u16ChannelId;
        fc_sxm_vString2Fi(rChannel.oAudioChannelName.c_str(), lPresetEntry.ChannelName);

        // If the channel is already tuned, check only for the Tune status else check for channels Browse status
        tDisplayAdvisory = (_u16TunedChannelServiceID == u16ServiceID) ?
                rChannel.eIsTunable() : rChannel.enBrowseStatus();

        ETG_TRACE_USR4(("fc_sxm_tclChannelList::enGetPresetChannelInfo() ChannelID(%u) \t BrowseStatus(%u)",
                rChannel.u16ChannelId, ETG_CENUM(tTunable, tDisplayAdvisory)));
        if ((rChannel.bIsAvailable()) && (rChannel.bIsAuthorized()))
        {
            lPresetEntry.ChannelActive = TRUE;
        }
        if (rChannel.bIsChannelArtAvailable)
        {
            lPresetEntry.ChnGraphicAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH;
        }
        else
        {
            lPresetEntry.ChnGraphicAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
        }

        const CATEGORY_ID sfCatID (SMART_FAVORITES.tCategoryId(hDecoder));
        lPresetEntry.IsSmartFavorite = rChannel.bBelongsToCategory(sfCatID);

        lPresetEntry.IsLocked = rChannel.bIsLocked();
        lPresetEntry.IsAvailable = rChannel.bIsAvailable();
        lPresetEntry.IsMature = rChannel.bIsMature();
        lPresetEntry.IsSkipped = rChannel.bIsSkipped();
        // Update current tuned channel and publish channel info
        if(TRUE == bChannelTuned)
        {
            _u16TunedChannelChannelID  = rChannel.u16ChannelId;
            _u16TunedChannelServiceID = rChannel.u16ServiceId;
            _u16TunedChannelCategoryID = rChannel.u16CategoryId;
        }
        break;
        }
    }

    return lPresetEntry;
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartGetChnList)                 *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to get channel list                             *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Get channel list method start request               *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vProcess(
    fc_sxm_trMsgAudioMStartGetChnList const *prMsg)
{
    tU32 u32Index = (prMsg->oFiMsg).Index;
    tU32 u32Range = (prMsg->oFiMsg).Range;
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vProcess"
                        "(fc_sxm_trMsgAudioMStartGetChnList) u32Index = %u \t u32Range = %u", u32Index, u32Range));

    /* Method result data to return */
    midw_ext_sxm_audiofi_tclMsgGetChannelListMethodResult ofiTxObj;

    tBool bValidInputs = FALSE;

    if((0 == u32Index) && (0 == u32Range))
    {
        /* If index and range both are zero, then update the entire list */
        u32Range = (tU32)(_mChannelList.size());
        bValidInputs = TRUE;
    }
    else if( (_mChannelList.size() > u32Index) &&
             (_mChannelList.size() > (
                 u32Index + u32Range - 1)) )
    {
        /* If the index and range values are valid then 
           update the requested list elements */
        bValidInputs = TRUE;
    }
    else
    {
        /* If the index and range values are not valid then do not update */
    }

    /* If the request is valid then send the requested list elements */
    if(TRUE == bValidInputs)
    {
        ETG_TRACE_USR3(("Sending requested data success status"));

        ofiTxObj.Status.enType = 
            midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;

        if(_mChannelList.size() >= (u32Index + u32Range))
        {
            /* If further elements are available send the next list element */
            ofiTxObj.NextIndex = (tU16)(u32Index + u32Range);
        }
        else
        {
            /* If no more list elements are available then 
               send the last list element index */
            ofiTxObj.NextIndex = (tU16)((_mChannelList.size() - 1));
        }

        tU32 u32Count = 0;
        // Enquire & Collect the list of preset indices up front to improve run time performance
        map<SERVICE_ID, tU8> mPresetIndices;
        vGetPresetIndices(mPresetIndices);
        map<SERVICE_ID, tU8>::const_iterator mPresetIter;

        map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
        for(u32Count=0, mIter = _mChannelList.begin();
            mIter!=_mChannelList.end();
            u32Count++, ++mIter)
        {
            /* Iterate till the list index and get the data */
            if((u32Count >= u32Index) && (u32Count < u32Range)) 
            {
                const fc_sxm_trChannel& channel((*mIter).second);
                // GetChannelList would provide all channels irrespective of their skip status
               if(e_TUNABLE == channel.eIsTunable())
               {
                   /* Increment list index count */
                   midw_ext_fi_tcl_ChannelListEntry tChannelListEntry;

                   /* Copy service ID */
                   tChannelListEntry.ServiceID = 
                           channel.u16ServiceId;
                   /* Copy channel ID */
                   tChannelListEntry.ChannelID = 
                           channel.u16ChannelId;
                   /* Copy channel name availability */
                   tChannelListEntry.ChannelNameAvail = 
                           channel.bIsChannelNameAvailable();
                   /* Copy channel name */
                   fc_sxm_vString2Fi(channel.oAudioChannelName.c_str(),
                                     tChannelListEntry.ChannelName);

                   fc_sxm_vString2Fi(channel.oChannelArtistName.c_str(),
                                                        tChannelListEntry.ChannelArtistName);

                   fc_sxm_vString2Fi(channel.oChannelSongName.c_str(),
                                                        tChannelListEntry.ChannelSongName);

                   if (etg_bIsTraceActive(TR_CLASS_FC_SXM_CHANNEL_LIST, TR_LEVEL_USER_2)) {
                       ETG_TRACE_USR2(("ArtistName --------------- '%s'", (tChannelListEntry).ChannelArtistName.szValue));
                       ETG_TRACE_USR2(("SongName --------------- '%s'", (tChannelListEntry).ChannelSongName.szValue));
                   }

                   ETG_TRACE_USR2((
                       "Channel ID - %d, service ID - %d, channel name - %s", 
                       tChannelListEntry.ChannelID, 
                       tChannelListEntry.ServiceID, 
                       ((tChannelListEntry).ChannelName).szValue));

                   // Copy preset number 
                   mPresetIter = mPresetIndices.find(channel.u16ServiceId);
                   tChannelListEntry.PresetNumber = (mPresetIter != mPresetIndices.end()) ? mPresetIter->second : 0;
                   if (((*mIter).second).bIsChannelArtAvailable)
                   {
                       tChannelListEntry.ChannelGraphicAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH;
                   }
                   else
                   {
                       tChannelListEntry.ChannelGraphicAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
                   }

                   /* Copy Lock, Availability, Mature and Skipped attributes*/
                   tChannelListEntry.IsLocked = channel.bIsLocked();
                   tChannelListEntry.IsAvailable = channel.bIsAvailable();
                   tChannelListEntry.IsMature = channel.bIsMature();
                   tChannelListEntry.IsSkipped = channel.bIsSkipped();

                   ETG_TRACE_USR2((
                      "Locked - %u, Available - %u, Mature - %u, Skipped - %u",
                      tChannelListEntry.IsLocked,
                      tChannelListEntry.IsAvailable,
                      tChannelListEntry.IsMature,
                      tChannelListEntry.IsSkipped));

                   /* Push the data into method result list */
                   (ofiTxObj.ChannelList).push_back(tChannelListEntry);
               }
               else
               {
                  u32Count--;
               }
            }

            if(u32Count == (u32Range - 1))
                break;
        }
    }
    else
    {
        ETG_TRACE_USR3(("Sending requested data with rejected status"));
        ofiTxObj.Status.enType = 
            midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_REQUEST_REJECTED;
    }

    /* Post the Method Result */
    if(AIL_EN_N_NO_ERROR == 
          fc_sxm_tclAudioService::instance()->enSendFiMessage(
                                                  prMsg->rAdressing, 
                                                  ofiTxObj))
    {
        ETG_TRACE_USR3(("Channel list method result sent successfully"));
    }
    else
    {
        ETG_TRACE_ERR(("Failed to send method result for channel list"));
    }
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartGetChnInfo)             *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to get channel info                             *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Get channel information list method start request   *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vProcess(
        fc_sxm_trMsgAudioMStartGetChnInfo const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartGetChnInfo)"));
    /* Method result data to return */
    midw_ext_sxm_audiofi_tclMsgGetChannelInfoMethodResult ofiTxObj;
    midw_ext_fi_tcl_ChannelInformation lChannelInfo;
    std::vector<tU16, std::allocator<tU16> >::const_iterator cii;

    for(cii = (prMsg->oFiMsg).SIDList.begin(); cii != (prMsg->oFiMsg).SIDList.end(); ++cii)
    {
        ETG_TRACE_USR4(("SID - %d", *cii));
        /* Find the channel from channel list */
        map<CHANNEL_ID, fc_sxm_trChannel>::iterator mIter;
        for(mIter = _mChannelList.begin(); mIter != _mChannelList.end(); ++mIter) {
            if(*cii == ((*mIter).second).u16ServiceId) {
                lChannelInfo.SID = ((*mIter).second).u16ServiceId;
                lChannelInfo.ChannelNumber =
                    ((*mIter).second).u16ChannelId;
                fc_sxm_vString2Fi(((*mIter).second).oAudioChannelName.c_str(),
                                                    lChannelInfo.ChannelName);
                fc_sxm_vString2Fi(((*mIter).second).oChannelArtistName.c_str(),
                                                    lChannelInfo.ArtistName);
                fc_sxm_vString2Fi(((*mIter).second).oChannelSongName.c_str(),
                                                    lChannelInfo.SongName);
                if (etg_bIsTraceActive(TR_CLASS_FC_SXM_CHANNEL_LIST, TR_LEVEL_USER_2)) {
                   ETG_TRACE_USR2(("ArtistName --------------- '%s'", (lChannelInfo).ArtistName.szValue));
                   ETG_TRACE_USR2(("SongName --------------- '%s'", (lChannelInfo).SongName.szValue));
                }
                if (((*mIter).second).bIsChannelArtAvailable) {
                    lChannelInfo.ChnArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH;
                }
                else {
                    lChannelInfo.ChnArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
                }
                if (((*mIter).second).bIsAlbumArtAvailable) {
                    lChannelInfo.AlbumArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH;
                }
                else {
                    lChannelInfo.AlbumArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
                }
                lChannelInfo.IsLocked = ((*mIter).second).bIsLocked();
                lChannelInfo.IsAvailable = ((*mIter).second).bIsAvailable();
                lChannelInfo.IsMature = ((*mIter).second).bIsMature();
                lChannelInfo.IsSkipped = ((*mIter).second).bIsSkipped();

                /* Push the data into method result list */
                (ofiTxObj.ChannelInfoList).push_back(lChannelInfo);
            }
        }
    }
    /* Post the Method Result */
    if(AIL_EN_N_NO_ERROR ==
          fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing,ofiTxObj)) {
        ETG_TRACE_USR3(("Channel list info method result sent successfully"));
    }
    else {
        ETG_TRACE_ERR(("Failed to send method result for channel list info"));
    }
}

#if(0)
/*****************************************************************************/
SelectChannel
|
|-----> FI_EN_XMTUN_SELECT_SID
|  |--> CidOrSid     : Holds ServiceID value
|  |--> CategoryType : Ignore
|  |--> Steps        : Ignore
|  |--> DwellTime    : Holds delay in milliseconds
|
|-----> FI_EN_XMTUN_SELECT_CHANNEL
|  |
|  |-----> [On Channel List]
|  |  |--> CidOrSid     : Holds list index value
|  |  |--> CategoryType : 0
|  |  |--> Steps        : Ignore
|  |  |--> DwellTime    : Holds delay in milliseconds
|  |
|  |-----> [On Category List]
|  |  |--> CidOrSid     : Holds list index value
|  |  |--> CategoryType : 1 to 31
|  |  |--> Steps        : Ignore
|  |  |--> DwellTime    : Holds delay in milliseconds
|
|-----> FI_EN_XMTUN_SELECT_CHANNEL_UP
|  |
|  |-----> [On Channel List]
|  |  |--> CidOrSid     : Ignore
|  |  |--> CategoryType : 0
|  |  |--> Steps        : Step [+1]
|  |  |--> DwellTime    : Holds delay in milliseconds
|  |
|  |-----> [On Category List]
|  |  |--> CidOrSid     : Ignore
|  |  |--> CategoryType : 1 to 31
|  |  |--> Steps        : Step [+1]
|  |  |--> DwellTime    : Holds delay in milliseconds
|
|-----> FI_EN_XMTUN_SELECT_CHANNEL_DOWN
|  |
|  |-----> [On Channel List]
|  |  |--> CidOrSid     : Ignore
|  |  |--> CategoryType : 0
|  |  |--> Steps        : Step [+1]
|  |  |--> DwellTime    : Holds delay in milliseconds
|  |
|  |-----> [On Category List]
|  |  |--> CidOrSid     : Ignore
|  |  |--> CategoryType : 1 to 31
|  |  |--> Steps        : Step [+1]
|  |  |--> DwellTime    : Holds delay in milliseconds
|
|-----> FI_EN_XMTUN_SELECT_PROGRAM_UP
|  |--> CidOrSid     : Ignore
|  |--> CategoryType : Ignore
|  |--> Steps        : Ignore
|  |--> DwellTime    : Holds delay in milliseconds
|
|-----> FI_EN_XMTUN_SELECT_PROGRAM_DOWN
|  |--> CidOrSid     : Ignore
|  |--> CategoryType : Ignore
|  |--> Steps        : Ignore
|  |--> DwellTime    : Holds delay in milliseconds
/*****************************************************************************/
#endif

/*****************************************************************************
 * FUNCTION    : u16GetCidFromSidInChannelList                               *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID from service ID from channel list            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : u16ServiceID - Valid service ID to query                    *
 * RETURNVALUE : CHANNEL_ID - Valid queried channel ID                       *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclChannelList::u16GetCidFromSidInChannelList(SERVICE_ID u16ServiceID)
{
    CHANNEL_ID u16ChannelID = CHANNEL_INVALID_ID;

    /* Find channel id from service id */
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
    for(mIter = _mChannelList.begin(); mIter != _mChannelList.end(); ++mIter) {
        if(u16ServiceID ==((*mIter).second).u16ServiceId) {
            u16ChannelID = ((*mIter).second).u16ChannelId;
            break;
        }
    }
    /* Display channel unavailable message if channel not found */
    if(mIter == _mChannelList.end()) {
        /* Set channel unavailable system message  */
        (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, FALSE);
    }
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u16GetCidFromSidInChannelList() serviceID = %u \t channelID = %u", u16ServiceID, u16ChannelID));
    return u16ChannelID;
}

/*****************************************************************************
 * FUNCTION    : u16GetCidFromChannelListIndex                               *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID from channel index in channel list           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : u16ListIndex - Valid list index to query                    *
 * RETURNVALUE : CHANNEL_ID - Valid queried channel ID                       *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclChannelList::u16GetCidFromChannelListIndex(tU16 u16ListIndex)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u16GetCidFromChannelListIndex() u16ListIndex = %u", u16ListIndex));

    CHANNEL_ID u16ChannelID = CHANNEL_INVALID_ID;

    if(_mChannelList.size() > u16ListIndex) {
        /* Find channel id from service id */
        map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
        tU16 u16LoopCount = 0;
        for(mIter = _mChannelList.begin();
            mIter != _mChannelList.end(); u16LoopCount++, ++mIter) {
           if(mIter->second.bIsBrowseable()) {
               if(u16LoopCount == u16ListIndex) {
                   break;
               }
           } else {
              u16LoopCount--;
           }
        }

        if(mIter != _mChannelList.end()) {
            u16ChannelID = mIter->second.u16ChannelId;
        } else {
            ETG_TRACE_ERR(("Index not found"));
        }
    } else {
        ETG_TRACE_ERR(("Invalid list index received"));
        /* Set channel unavailable system message  */
        (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, FALSE);
    }

    return u16ChannelID;
}

/*****************************************************************************
 * FUNCTION    : u16GetCidFromChannelListRelativeIndex                       *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID from relative index to current tuned channel *
 *               in channel list                                             *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : s8ListRelativeIndex - Valid relative index to query         *
 * RETURNVALUE : CHANNEL_ID         - Valid queried channel ID               *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclChannelList::u16GetCidFromChannelListRelativeIndex(tS8 s8ListRelativeIndex)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u16GetCidFromChannelListRelativeIndex()"
            "Current channelID = %d, Relative Index = %d",
            _u16TunedChannelChannelID, s8ListRelativeIndex));

    CHANNEL_ID u16ChannelID = CHANNEL_INVALID_ID;

    /* Find for tuned channel from channel list */
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
    mIter = _mChannelList.find(_u16TunedChannelChannelID);
    if (mIter != _mChannelList.end()) {
        if(0 < s8ListRelativeIndex) {
            tS8 s8IndexCount = 0;
            for(s8IndexCount = 0; s8IndexCount < s8ListRelativeIndex;) {
                /* Wrap to first element if the iterator has reached last element in the list */
                if(mIter == _mChannelList.end()) {
                    mIter = _mChannelList.begin();
                } else {
                    /* Increment to next element */
                    mIter++;
                }

                /* Search next valid entry to tune */
                if(mIter != _mChannelList.end() && (FALSE != mIter->second.bIsBrowseable())) {
                    s8IndexCount++;
                }
            }
        }
        else if(0 > s8ListRelativeIndex) {
            tS8 s8IndexCount = 0;
            for(s8IndexCount = s8ListRelativeIndex; s8IndexCount < 0;) {
                /* Wrap to first element if the iterator has reached last element in the list */
                if(mIter == _mChannelList.begin()) {
                    mIter = _mChannelList.end();
                } else {
                    /* Increment to next element */
                    mIter--;
                }

                /* Search next valid entry to tune */
                if(mIter != _mChannelList.end() && (FALSE != mIter->second.bIsBrowseable())) {
                    s8IndexCount++;
                }
            }
        }

        /* Get the channel ID */
        if (mIter != _mChannelList.end()) {
            u16ChannelID = ((*mIter).second).u16ChannelId;
        } else {
            ETG_TRACE_ERR(("Index not found"));
        }

    } else {
        ETG_TRACE_ERR(("Failed to find tunned channel ID from channel list"));
    }

    return u16ChannelID;
}

/* Method Processes FI request to DisableMatureChannels*/
tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartDisableMatureChannels const *prMsg)
{
    ETG_TRACE_USR3(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartDisableMatureChannels) ENTERED \t status = %u", prMsg->oFiMsg.Status));
    vDisableAllMatureChannels(prMsg->oFiMsg.Status);
    midw_ext_sxm_audiofi_tclMsgDisableMatureChannelsMethodResult oMRes;
    oMRes.LockStatus = TRUE;
    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
    ETG_TRACE_USR3(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartDisableMatureChannels) EXIT"));
}

/* Method Processes FI request to SkipChannels*/
tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartSkipChannels const *prMsg)
{
    midw_ext_fi_tcl_e8_ChannelSkipType skipType = prMsg->oFiMsg.SkipChannelType;
    DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
    _u16TunedChannelChannelID = DECODER.tCurrentChannelId(hDecoder);
    ETG_TRACE_USR3(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartSkipChannels) skipType(%u) \t currentTunedChannel(%u)",
            skipType.enType, _u16TunedChannelChannelID));

    midw_ext_sxm_audiofi_tclMsgSkipChannelMethodResult oMRes;
    if ( midw_ext_fi_tcl_e8_ChannelSkipType::FI_EN_SKIP_ALL == skipType.enType)
    {
        oMRes.Status = bSkipAllChannels();
    }
    else if ( midw_ext_fi_tcl_e8_ChannelSkipType::FI_EN_UNSKIP_ALL == skipType.enType)
    {
        oMRes.Status = bUnSkipAllChannels();
    }
    else if ( midw_ext_fi_tcl_e8_ChannelSkipType::FI_EN_SKIP_SELECTION == skipType.enType)
    {
        oMRes.Status = bSkipSelectedChannel(prMsg->oFiMsg.ServiceID, prMsg->oFiMsg.ChannelID);
    }
    else if ( midw_ext_fi_tcl_e8_ChannelSkipType::FI_EN_UNSKIP_SELECTION == skipType.enType)
    {
        oMRes.Status = bUnSkipSelectedChannel(prMsg->oFiMsg.ServiceID, prMsg->oFiMsg.ChannelID);
    }

    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
    ETG_TRACE_USR3(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartSkipChannels) EXIT oMRes.Status = %u", oMRes.Status));
}

/*****************************************************************************
 * FUNCTION    : u16GetCidFromSelectionMode                                  *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID from selection mode based on category type   *
 *               either from channel list or category list                   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : enSelectionMode - Channel select selection mode             *
 * PARAMETER   : u16CidOrSid      - Service ID or index                      *
 * PARAMETER   : u16CategoryType  - Category type to query                    *
 * PARAMETER   : u8Steps         - Relative index                            *
 * RETURNVALUE : CHANNEL_ID    - Valid queried channel ID                    *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclChannelList::u16GetCidFromSelectionMode(
    midw_ext_fi_tcl_e8_SelectChannelMode enSelectionMode,
    tU16 u16CidOrSid,
    tU16 u16CategoryType,
    tU8 u8Steps)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u16GetCidFromSelectionMode() enSelectionMode(%u) u16CidOrSid(%u) u16CategoryType(%u) u8Steps(%u)",
            enSelectionMode.enType, u16CidOrSid, u16CategoryType, u8Steps));
    CHANNEL_ID u16ChannelID = CHANNEL_INVALID_ID;
    switch(enSelectionMode.enType)
    {
        /* CidOrSid contains service ID, ignore all other parameters */
        case midw_ext_fi_tcl_e8_SelectChannelMode::
             FI_EN_XMTUN_SELECT_SID:
        {
            u16ChannelID = u16GetCidFromSidInChannelList(u16CidOrSid);
        }
        break;
        /* CidOrSid contains channel ID, ignore all other parameters */
        case midw_ext_fi_tcl_e8_SelectChannelMode::
             FI_EN_XMTUN_SELECT_CID:
        {
            u16ChannelID = u16CidOrSid;
        }
        break;
        /* CidOrSid contains list index, category type defines category list 
           or channel list to access, ignore other parameters */
        case midw_ext_fi_tcl_e8_SelectChannelMode::
             FI_EN_XMTUN_SELECT_CHANNEL:
        {
            if(0 == u16CategoryType)
            {
                /* Refer channel list */
                u16ChannelID = u16GetCidFromChannelListIndex(u16CidOrSid);
            }
            else
            {
                /* Refer category list */
                CATEGORY_ID u16CategoryID = CATEGORY_INVALID_ID;

                if(((FC_SXM_AUDIO_CATEGORY_TYPE_MIN <= u16CategoryType) && (FC_SXM_AUDIO_CATEGORY_TYPE_MAX >= u16CategoryType))
                     || (( FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_START < u16CategoryType) && (FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_END > u16CategoryType) ))
                {
                    u16CategoryID = u16CategoryType;
                }
                else if((254 == u16CategoryType) || (255 == u16CategoryType))
                {
                    u16CategoryID = _u16TunedChannelCategoryID;
                }
                else
                {
                    ETG_TRACE_ERR(("Unknown category type received"));
                    
                    /* Set category unavailable system message  */
                    (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                        midw_ext_fi_tcl_e8_SystemMessage::
                            FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND, FALSE);
                }

                if(CATEGORY_INVALID_ID != u16CategoryID)
                {
                    u16ChannelID = getCidForIndex(u16CategoryID, u16CidOrSid);
                }
            }
        }
        break;
        /* category type defines category list or channel list to access
           Steps holds relative index to tune to next valid list element from 
           current tuned channel, for specific category type u16CidOrSid holds 
           channel index else tune from current tuned channel, 
           ignore all other parameters */
        case midw_ext_fi_tcl_e8_SelectChannelMode::
             FI_EN_XMTUN_SELECT_CHANNEL_UP:
        {
            tS8 s8RelativeIndex = ((+1) * (u8Steps));
            if(0 == u16CategoryType)
            {
                /* Refer channel list */
                u16ChannelID = 
                    u16GetCidFromChannelListRelativeIndex(s8RelativeIndex);
            }
            else
            {
                /* Refer category list */
                CATEGORY_ID u16CategoryID = CATEGORY_INVALID_ID;
                CHANNEL_ID u16TunedChannelID = CHANNEL_INVALID_ID;

                if(((FC_SXM_AUDIO_CATEGORY_TYPE_MIN <= u16CategoryType) && (FC_SXM_AUDIO_CATEGORY_TYPE_MAX >= u16CategoryType))
                    || (( FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_START < u16CategoryType) && (FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_END > u16CategoryType) ))
                {
                    u16CategoryID = u16CategoryType;
                    u16TunedChannelID = u16CidOrSid;
                }
                else if((254 == u16CategoryType) || (255 == u16CategoryType))
                {
                    u16CategoryID = _u16TunedChannelCategoryID;
                    u16TunedChannelID = _u16TunedChannelChannelID;
                }
                else
                {
                    ETG_TRACE_ERR(("Unknown category type received"));

                    /* Set category unavailable system message  */
                    (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                        midw_ext_fi_tcl_e8_SystemMessage::
                            FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND, FALSE);
                }

                if( (CATEGORY_INVALID_ID != u16CategoryID) && 
                    (CHANNEL_INVALID_ID != u16TunedChannelID) )
                {
                    u16ChannelID = getCidForRelativeIndex ( u16CategoryID, u16TunedChannelID, s8RelativeIndex);
                }
                else
                {
                    ETG_TRACE_ERR((
                        "Failed to get valid category ID and channel ID"));
                }
            }
        }
        break;
        /* Steps holds relative index to tune to previous valid list element 
           from current tuned channel in channel list, for specific category 
           type CidOrSid holds channel index else tune from current tuned 
           channel, ignore all other parameters */
        case midw_ext_fi_tcl_e8_SelectChannelMode::
             FI_EN_XMTUN_SELECT_CHANNEL_DOWN:
        {
            tS8 s8RelativeIndex = (tS8) ((-1) * (u8Steps));
            if(0 == u16CategoryType)
            {
                /* Refer channel list */
                u16ChannelID = 
                    u16GetCidFromChannelListRelativeIndex(s8RelativeIndex);
            }
            else
            {
                /* Refer category list */
                CATEGORY_ID u16CategoryID = CATEGORY_INVALID_ID;
                CHANNEL_ID u16TunedChannelID = CHANNEL_INVALID_ID;

                if(((FC_SXM_AUDIO_CATEGORY_TYPE_MIN <= u16CategoryType) && (FC_SXM_AUDIO_CATEGORY_TYPE_MAX >= u16CategoryType))
                     || (( FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_START < u16CategoryType) && (FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_END > u16CategoryType) ))
                {
                    u16CategoryID = u16CategoryType;
                    u16TunedChannelID = u16CidOrSid;
                }
                else if((254 == u16CategoryType) || (255 == u16CategoryType))
                {
                    u16CategoryID = _u16TunedChannelCategoryID;
                    u16TunedChannelID = _u16TunedChannelChannelID;
                }
                else
                {
                    ETG_TRACE_ERR(("Unknown category type received"));

                    /* Set category unavailable system message  */
                    (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                        midw_ext_fi_tcl_e8_SystemMessage::
                            FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND, FALSE);
                }

                if( (CATEGORY_INVALID_ID != u16CategoryID) && 
                    (CHANNEL_INVALID_ID != u16TunedChannelID) )
                {
                    u16ChannelID = getCidForRelativeIndex ( u16CategoryID, u16TunedChannelID, s8RelativeIndex);
                }
                else
                {
                    ETG_TRACE_ERR((
                        "Failed to get valid category ID and channel ID"));
                }
            }
        }
        break;
        /* Ignore all parameters and tune to next category first channel 
           from current tuned channel category type */
        case midw_ext_fi_tcl_e8_SelectChannelMode::
             FI_EN_XMTUN_SELECT_PROGRAM_UP:
        {
            u16ChannelID = getNextProgramCid ( _u16TunedChannelCategoryID, FC_SXM_AUDIO_TUNE_PROGRAM_UP);
        }
        break;
        /* Ignore all parameters and tune to previous category first channel 
           from current tuned channel category type */
        case midw_ext_fi_tcl_e8_SelectChannelMode::
             FI_EN_XMTUN_SELECT_PROGRAM_DOWN:
        {
            u16ChannelID = getNextProgramCid ( _u16TunedChannelCategoryID, FC_SXM_AUDIO_TUNE_PROGRAM_DOWN);
        }
        break;

        default:
            break;
    }
    return(u16ChannelID);
}

/* Check if channel info should be sent, even if identical to previous */
tBool fc_sxm_tclChannelList::bForceChannelInfoUpdate(tFCSXM_SelectMode enMode) const
{
   if (e_CHANNEL_ART_UPDATE == enMode ||
       e_RECALL_PRESET == enMode ||
       e_SELECT_CHANNEL == enMode ||
       e_CHANNEL_BROWSE == enMode ||
       e_RADIO_ID_CHANNEL == enMode ||
       e_ADVISORY_TIMEOUT == enMode)
   {
       return TRUE;
   }

   return FALSE;
}

/* Check for current audio service subscription status */
tVoid fc_sxm_tclChannelList::vCheckAudioSubscriptionStatus(tVoid)
{
   tBool bNewStatus = FALSE;
   map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
   for(mIter = _mChannelList.begin(); mIter != _mChannelList.end(); ++mIter) {
      // Get channel subscription status
      if (mIter->second.bIsSubscribedChannel()) {
          bNewStatus = TRUE;
          break;
      }
   }

   ETG_TRACE_USR4(("vCheckAudioSubscriptionStatus :Old Status = %u \t New Status = %u",
           _bIsAudioServiceSubscribed, bNewStatus));
   _bIsAudioServiceSubscribed = bNewStatus;

   // Notify audio subscription status to all registered clients
   fc_sxm_tcl_trAudioPropertyAudioSubStatus tAudioSubscriptionStatus;
   tAudioSubscriptionStatus.oFiRes.Data.SubStatus = _bIsAudioServiceSubscribed;
   // Update only if there is change in subscription status
   fc_sxm_tclAudioProperties::instance()->oAudioSubscriptionStatus.vSetAndNotify(tAudioSubscriptionStatus);
}

/*****************************************************************************
 * FUNCTION    : vRestorePreviewChannel                                      *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID of the last tuned channel from SMS and tune  *
 *               to that channel                                             *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vRestorePreviewChannel(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::vRestorePreviewChannel()"));

   /* Tune to last tuned channel */
   if(SMSAPI_RETURN_CODE_SUCCESS == DECODER.eTuneDirect(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),FC_SXM_AUDIO_DEFAULT_CHANNEL,TRUE)) {
      ETG_TRACE_USR2(("Tuned successfully to last tuned channel having channel ID - %d", FC_SXM_AUDIO_DEFAULT_CHANNEL));
      _u16TunedChannelChannelID = FC_SXM_AUDIO_DEFAULT_CHANNEL;
      /* Notify auto compare properties */
      /* Set IR Mode when SF channel is tuned */
      vNotifyTunedChannelPreset (u16GetSIDOfTunedChannel(), TRUE);
   } else {
      ETG_TRACE_ERR(("Failed to restore last tuned channel with Id - %d", FC_SXM_AUDIO_DEFAULT_CHANNEL));

      tTunable enDefaultChannelValidity = e_UNAVAILABLE;
      map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
      mIter = _mChannelList.find(FC_SXM_AUDIO_DEFAULT_CHANNEL);
      if(mIter != _mChannelList.end()) {
          enDefaultChannelValidity = enTuneStatus(mIter->second, TRUE);
      }

      switch(enDefaultChannelValidity)
      {
         case e_UNAVAILABLE: {
            _u16RequestedUnavailableChannel = FC_SXM_AUDIO_DEFAULT_CHANNEL;
            (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, TRUE);
            vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
         } break;
         case e_UNAUTHORIZED: {
            _u16RequestedUnavailableChannel = FC_SXM_AUDIO_DEFAULT_CHANNEL;
            (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, TRUE);
            vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
         } break;
         case e_TUNABLE:
         default:
            break;
      }
   }
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartSelChn)                     *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to select channel                               *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Select channel method start request                 *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vProcess(
    fc_sxm_trMsgAudioMStartSelChn const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vProcess"
                    "(fc_sxm_trMsgAudioMStartSelChn)"));

    /* Method result data */
    midw_ext_sxm_audiofi_tclMsgSelectChannelMethodResult ofiTxObj;

    /* Load values for method result */
    ofiTxObj.SelectionMode = (prMsg->oFiMsg).SelectionMode;
    ofiTxObj.Status        = FALSE;

    /* Based on selection mode, fetch channel ID */
    CHANNEL_ID u16RequestedChannel = u16GetCidFromSelectionMode(
         (prMsg->oFiMsg).SelectionMode              // Selection mode
        ,(prMsg->oFiMsg).CidOrSid                   // SID or Index
        ,(prMsg->oFiMsg).CategoryType               // Category type
        ,(prMsg->oFiMsg).Steps);                    // Steps

    tBool bIsDirectTune = static_cast <tBool> (((ofiTxObj.SelectionMode.enType == midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CID) ||
            (ofiTxObj.SelectionMode.enType == midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_SID)));

    ETG_TRACE_USR4(("u16RequestedChannel = %u \t bIsDirectTune = %u", u16RequestedChannel, bIsDirectTune));
    if (FALSE == bIsDirectTune)
    {
        vHandleSelectChannel(u16RequestedChannel, (prMsg->oFiMsg).Steps);
    }
    else
    {
        // In case if its a direct tune, clear any existing channel advisories and stop the channel advisory timer.
        fc_sxm_tclAdvisories::instance()->vStopAndClearChannelAdvisory();
    }

    /* Find channel ID to tune */
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
    mIter = _mChannelList.find(u16RequestedChannel);
    if (mIter != _mChannelList.end())
    {
        tTunable enTunableMode = enTuneStatus(mIter->second, bIsDirectTune);

        switch(enTunableMode)
       {
         case e_UNAVAILABLE: /* Check if channel available */
         {
            /* Store requested unavailable channel ID */
            _u16RequestedUnavailableChannel = u16RequestedChannel;

            /* If not subscribed then send system message */
            (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                midw_ext_fi_tcl_e8_SystemMessage::
                    FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, FALSE);
            vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
         }
         break;
         case e_UNAUTHORIZED:
         {
            /* Store requested unavailable channel ID */
            _u16RequestedUnavailableChannel = u16RequestedChannel;

            /* If not subscribed then send system message */
            (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                midw_ext_fi_tcl_e8_SystemMessage::
                    FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, FALSE);
            vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
          }
          break;
         case e_LOCKED:
         {
             _u16RequestedUnavailableChannel = u16RequestedChannel;
             // Channel is locked.... send a system message
             (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                midw_ext_fi_tcl_e8_SystemMessage::
                    FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE, FALSE);
             vNotifyChannelInformationUpdate(e_MATURE_LOCKED_CHANNEL);
         }
         break;
         case e_SKIPPED:
         {
             _u16RequestedUnavailableChannel = u16RequestedChannel;
             // Channel is a skipped channel.... send a system message
             (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                midw_ext_fi_tcl_e8_SystemMessage::
                    FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED, FALSE);
             vNotifyChannelInformationUpdate(e_SKIPPED_CHANNEL);
         }
         break;
          case e_TUNABLE:
          {
            Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                          _poAudioApp->poGetPresetsInstance(): NULL;
            SXM_ASSERT_RETURN(NULL != poPresets)
            poPresets->vSetTuneScanStatus(FALSE);
            /* Tune to selected channel */
            SMSAPI_RETURN_CODE_ENUM enSMSReturnCodeVal =
                DECODER.eTuneDirect(
                    fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),
                    mIter->second.u16ChannelId,
                    TRUE);
            if(SMSAPI_RETURN_CODE_SUCCESS == enSMSReturnCodeVal)
            {
                /* fill method result */
                ofiTxObj.Status = TRUE;
                _u16TunedChannelChannelID  =
                    mIter->second.u16ChannelId;
                _u16TunedChannelServiceID  =
                    mIter->second.u16ServiceId;
                _u16TunedChannelCategoryID =
                    mIter->second.u16CategoryId;

                ETG_TRACE_USR4(("Successfully tuned to channel %d",
                                _u16TunedChannelChannelID));
                fc_sxm_tclAdvisories::instance()->vStopAndClearChannelAdvisory();
                vNotifyChannelInformationUpdate(e_SELECT_CHANNEL);

                /* Notify auto compare properties */
                vNotifyTunedChannelPreset (u16GetSIDOfTunedChannel(), TRUE);
            }
            else
            {
                ETG_TRACE_ERR(("Failed to tune to specified channel, "
                               "unknown error, SMS_API_RET_CODE - %d, "
                               "ERROR_CODE - %d",
                ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enSMSReturnCodeVal),
                ETG_CENUM(DECODER_ERROR_CODE_ENUM,
                          DECODER.eErrorCode(
                          fc_sxm_tclSmsDecoder::instance()->
                              hGetSmsDecoder())) ));
            }
          }
          break;
          default:
            break;
        }
    }
    else
    {
        ETG_TRACE_ERR((
            "Failed to get requested channel ID from channel list"));

        /* Store requested unavailable channel ID */
        _u16RequestedUnavailableChannel = u16RequestedChannel;

        /* If not available then send system message */
        (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
            midw_ext_fi_tcl_e8_SystemMessage::
                FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, FALSE);
        vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
    }

    /* Post the Method Result */
    if(AIL_EN_N_NO_ERROR == 
           fc_sxm_tclAudioService::instance()->enSendFiMessage(
                                                   prMsg->rAdressing, 
                                                   ofiTxObj))
    {
        ETG_TRACE_USR3(("Channel select method result sent successfully"));
    }
    else
    {
        ETG_TRACE_ERR(("Failed to send method result for channel select"));
    }
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartSelChnWitDwellTime)         *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to select channel with delay                    *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Select channel with delay method start request      *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartSelChnWitDwellTime const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartSelChnWitDwellTime)"));
    if (FALSE == bAbortAdvancedAudioPlayback())
    {
        // Store request for further processing
        _oPendingChnSelWitDelayRequest = *prMsg;
        // Restart timer if it is already running, else start new timer
        _mChannelSelectDwellTimer.vStart((prMsg->oFiMsg).DwellTime);

        /* Also update channel info to requested clients once timer is *
         * started but do not tune to that channel until timer expires */

        // Based on selection mode, fetch channel ID
        CHANNEL_ID u16RequestedChannel = u16GetCidFromSelectionMode(
             (prMsg->oFiMsg).SelectionMode              // Selection mode
            ,(prMsg->oFiMsg).CidOrSid                   // SID or Index
            ,(prMsg->oFiMsg).CategoryType               // Category type
            ,(prMsg->oFiMsg).Steps);                    // Steps

        ETG_TRACE_USR4(("u16RequestedChannel = %d", u16RequestedChannel));

        vHandleSelectChannel(u16RequestedChannel, (prMsg->oFiMsg).Steps);

        //Find channel ID to tune
        map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
        mIter = _mChannelList.find(u16RequestedChannel);
        if (mIter != _mChannelList.end())
        {
            _u16TunedChannelChannelID  =
                mIter->second.u16ChannelId;
            _u16TunedChannelServiceID  =
                mIter->second.u16ServiceId;
            _u16TunedChannelCategoryID =
                mIter->second.u16CategoryId;
            vNotifyChannelInformationUpdate(e_CHANNEL_BROWSE);
            /* Notify auto compare properties */
            vNotifyTunedChannelPreset( _u16TunedChannelServiceID, FALSE);
        }
        else
        {
            ETG_TRACE_ERR(("Failed to get requested channel ID from channel list"));
        }
    }
}

/***********************************************************************************
 * FUNCTION    : vHandleSelectChannel(CHANNEL_ID &u16RequestedChannel, tU8 u8Steps)*
 * CLASS       : fc_sxm_tclChannelList                                             *
 * --------------------------------------------------------------------------------*
 * DESCRIPTION : Handle use case when tune scan or any FlashEvent playback is on   *
 *                and any channel is selected  *
 * ------------------------------------------------------------------------------- *
 * PARAMETER   : CHANNEL_ID &u16RequestedChannel, tU8 u8Steps                      *
 * RETURNVALUE : None                                                              *
 ***********************************************************************************/
tVoid fc_sxm_tclChannelList::vHandleSelectChannel(CHANNEL_ID &u16RequestedChannel, tU8 u8Steps)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vHandleSelectChannel u16RequestedChannel = %u, u8Steps = %u", u16RequestedChannel, u8Steps));

    Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                _poAudioApp->poGetPresetsInstance(): NULL;
    SXM_ASSERT_RETURN(NULL != poPresets)

    if (poPresets->bGetTuneScanStatus())
    {
        midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult ofiTxObj;
        poPresets->vTuneScanStop(ofiTxObj);

        vGetSelectedChanId(u16RequestedChannel, u8Steps, poPresets->u16GetSavedChannelId());

        fc_sxm_tcl_trAudioPropertyTuneScanContentStatus oProperty;
        oProperty.oFiMsg.TuneScanStatus.enType = midw_ext_fi_tcl_e8_TuneScanStatus::FI_EN_TUNE_SCAN_STOPPED;
        fc_sxm_tclAudioProperties::instance()->oTuneScanContentStatus.vSet(oProperty);
        fc_sxm_tclAudioProperties::instance()->oTuneScanContentStatus.vNotify();
    }
    else if ((_poAudioApp) && (_poAudioApp->bIsFlashEventPlayback()))
    {
        DECODER_OBJECT hDecoder = _poAudioApp->oGetDecoderObject();
        if(DECODER_INVALID_OBJECT != hDecoder)
        {
            CHANNEL_ID chnLastTuned = DECODER.tLastTunedChannelId(hDecoder);
            vGetSelectedChanId(u16RequestedChannel, u8Steps, chnLastTuned);
            ETG_TRACE_USR4(("FlashEvent Playback = TRUE  \t chnLastTuned = %u", chnLastTuned));
        }
    }
}

// Method aborts playback of TWNow, Sports Flash or Tune Scan whichever is in progress
// Return TRUE if TWNow, Sports Flash or Tune Scan was aborted
tBool fc_sxm_tclChannelList::bAbortAdvancedAudioPlayback()
{
    tBool bAbort = FALSE;
    if ((_poAudioApp) && (_poAudioApp->bIsFlashEventPlayback()))
    {
        _poAudioApp->vAbortFlashEventPlayback();
        bAbort = TRUE;
    }
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::bAbortAdvancedAudioPlayback bAbortStatus(%u)", bAbort));
    return bAbort;
}

/**********************************************************************************************************
 * FUNCTION    : vGetSelectedChanId(CHANNEL_ID &u16RequestedChannel, tU8 u8Steps, CHANNEL_ID savedChnId)*
 * CLASS       : fc_sxm_tclChannelList                                                                    *
 * -------------------------------------------------------------------------------------------------------*
 * DESCRIPTION : Method sets next channel id to be tuned                                                  *
 * ------------------------------------------------------------------------------------------------------ *
 * PARAMETER   : CHANNEL_ID &u16RequestedChannel, tU8 u8Steps , CHANNEL_ID savedChnId                     *
 * RETURNVALUE : None                                                                                     *
 **********************************************************************************************************/
tVoid fc_sxm_tclChannelList::vGetSelectedChanId(CHANNEL_ID &u16RequestedChannel, tU8 u8Steps, CHANNEL_ID savedChnId)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vGetSelectedChanId"));

    CHANNEL_ID u16CurrentChannel = u16GetCurrentTunedChannel();
    _u16TunedChannelChannelID = savedChnId;

    if(u16RequestedChannel > u16CurrentChannel)
    {
        tS8 s8RelativeIndex = ((+1) * (u8Steps));
        u16RequestedChannel = u16GetCidFromChannelListRelativeIndex(s8RelativeIndex);
    }
    else if(u16RequestedChannel < u16CurrentChannel)
    {
        tS8 s8RelativeIndex = (tS8)((-1) * (u8Steps));
        u16RequestedChannel = u16GetCidFromChannelListRelativeIndex(s8RelativeIndex);
    }
    else
    {
        u16RequestedChannel = _u16TunedChannelChannelID;
    }

    ETG_TRACE_USR4(("_u16TunedChannelChannelID = %d, Steps = %d, u16RequestedChannel = %d", _u16TunedChannelChannelID, u8Steps, u16RequestedChannel));
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgChannelListTimer)                      *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Process timeout of channel list update timer                *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Channel list timer struct                           *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vProcessTimer(
    fc_sxm_trMsgChannelListTimer const * /* prMsg */ )
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vProcess"
                    "(fc_sxm_trMsgChannelListTimer)"));

    /* Stop the timer and notify time out to channel list class */
    _mChannelListUpdateTimer.vStop();

    /* If there are any further updates in channel list, notify them */
    if(TRUE == _bUpdateChannelList)
    {
        vNotifyChannelListUpdate();
    }
}  

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgChannelSelectDwellTimer)               *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Process channel select timer time out                       *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Select channel with delay timer class               *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vProcessTimer(
    fc_sxm_trMsgChannelSelectDwellTimer const * /* prMsg */ )
{
    ETG_TRACE_USR1(("fc_sxm_tclChannelList::vProcess"
                    "(fc_sxm_trMsgChannelSelectDwellTimer)"));

    /* Stop the timer and notify time out to channel list class */
    vStopDwellTimer();

    /* Method result data */
    midw_ext_sxm_audiofi_tclMsgSelectChannelWithDwellTimeMethodResult ofiTxObj;
    ofiTxObj.Status = FALSE;

    /* Find channel ID to tune */
    map<CHANNEL_ID, fc_sxm_trChannel>::iterator mIter;
    mIter = _mChannelList.find(_u16TunedChannelChannelID);
    if (mIter != _mChannelList.end())
    {
        tTunable enTuneState = mIter->second.eIsTunable();
        ETG_TRACE_USR2(("_u16TunedChannelChannelID = (%u), Tune Status = (%u)",
                _u16TunedChannelChannelID, ETG_CENUM(tTunable, enTuneState)));
        switch(enTuneState)
        {
        case e_UNAVAILABLE:
        {
            _u16RequestedUnavailableChannel = _u16TunedChannelChannelID;
            /* If not subscribed then send system message */
            (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                midw_ext_fi_tcl_e8_SystemMessage::
                    FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, FALSE);

            vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
        }
        break;
        case e_UNAUTHORIZED:
        {
            _u16RequestedUnavailableChannel = _u16TunedChannelChannelID;
            /* If not subscribed then send system message */
            (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                midw_ext_fi_tcl_e8_SystemMessage::
                    FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, FALSE);
            vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
        }
        break;
        case e_TUNABLE:
        {
            Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                          _poAudioApp->poGetPresetsInstance(): NULL;
            SXM_ASSERT_RETURN(NULL != poPresets)
            poPresets->vSetTuneScanStatus(FALSE);
            /* Tune to selected channel */
            SMSAPI_RETURN_CODE_ENUM enSMSReturnCodeVal = 
                DECODER.eTuneDirect(
                    fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),
                    mIter->second.u16ChannelId,
                    TRUE);
            if(SMSAPI_RETURN_CODE_SUCCESS == enSMSReturnCodeVal)
            {
                /* fill method result */
                ofiTxObj.Status = TRUE;
                _u16TunedChannelChannelID  = 
                    mIter->second.u16ChannelId;
                _u16TunedChannelServiceID  =
                    mIter->second.u16ServiceId;
                _u16TunedChannelCategoryID = 
                    mIter->second.u16CategoryId;
                ETG_TRACE_USR4(("Successfully tuned to channel %d",
                                _u16TunedChannelChannelID));

                enAdvisoryMsgType eCheckAdvisory = enNO_ADVISORY;
                midw_ext_fi_tcl_e8_SystemMessage::tenType eCurrentSystemMessage  = (fc_sxm_tclAdvisories::instance())->eGetCurrentAdvisoryType(FALSE, eCheckAdvisory);
                eCurrentSystemMessage = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;
                if(enHIGH_PRIO_ADVISORY != eCheckAdvisory) {
                   vNotifyChannelInformationUpdate(e_TUNING_CHANNEL);
                }
                /* Notify auto compare properties */
                vNotifyTunedChannelPreset(u16GetSIDOfTunedChannel(), TRUE);
            }
            else
            {
                ETG_TRACE_ERR(("Failed to tune to specified channel, "
                               "unknown error, SMS_API_RET_CODE - %d, "
                               "ERROR_CODE - %d", 
                ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enSMSReturnCodeVal), 
                ETG_CENUM(DECODER_ERROR_CODE_ENUM, 
                          DECODER.eErrorCode(
                              fc_sxm_tclSmsDecoder::instance()->
                                  hGetSmsDecoder())) ));
            }
        }
        break;
        default:
           break;
        }
    }
    else
    {
        ETG_TRACE_ERR((
            "Failed to get requested channel ID from channel list"));
    }

    /* Post the Method Result */
    if(AIL_EN_N_NO_ERROR == 
          fc_sxm_tclAudioService::instance()->
              enSendFiMessage(
                  _oPendingChnSelWitDelayRequest.rAdressing, 
                  ofiTxObj))
    {
        ETG_TRACE_USR3((
            "Channel select with delay method result sent successfully"));
    }
    else
    {
        ETG_TRACE_ERR((
            "Failed to send method result for channel select with delay"));
    }
}

tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioTestToneEvent const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioTestToneEvent)"));
   switch(prMsg->u8EnableTone)
   {
      case 0:
      {
         /* Stop tone generation */
         if(SMSAPI_RETURN_CODE_SUCCESS == DECODER.eStopToneGeneration(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder())) {
            ETG_TRACE_USR4(("Tone generation stopped successfully"));
         } else {
            ETG_TRACE_ERR(("Failed to stop tone generation"));
         }
      }
      break;
      case 1:
      {
         UN32 un32ToneFreqHz = 100;
         N8 n8Volume = 5;
         if(SMSAPI_RETURN_CODE_SUCCESS == DECODER.eStartToneGeneration(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), 
                                                                       un32ToneFreqHz, n8Volume, DECODER_TONE_GENERATION_BALANCE_BOTH)) {
            ETG_TRACE_USR4(("Tone generation started successfully"));
         } else {
            ETG_TRACE_ERR(("Failed to start tone generation"));
         }
      }
      break;
      default:
      {
         ETG_TRACE_ERR(("Unknown value received"));
      }
      break;
   }
}

/*****************************************************************************
 * FUNCTION    : vNotifyChannelListUpdate                                    *
 * CLASS       : fc_sxm_tclChannelList                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Notify channel list changes to all clients                  *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclChannelList::vNotifyChannelListUpdate(tVoid)
{
    /* If there is any updates in channel list elements then only notify HMI */
    if((TRUE == _bChannelListModified) && (TRUE == _bChannelListReady))
    {
        ETG_TRACE_USR4(("fc_sxm_tclChannelList::vNotifyChannelListUpdate()"));

        /* If the timer is still running just update the flag else update HMI 
           and start timer */
        if(FALSE == 
           _mChannelListUpdateTimer.bIsRunning())
        {
            /* Reset channel list modified flag */
            _bChannelListModified = FALSE;

            /* Set channel list property to true */
            vTriggerChannelListStatus(TRUE);

            ETG_TRACE_USR3(("vNotifyChannelListUpdate - channel list status "
                            "changed notified"));

            /* Now start the timer */
            _mChannelListUpdateTimer.vStart(
                FC_SXM_AUDIO_DEFAULT_CHANNEL_LIST_UPDATE_PERIOD);

            /* Set update flag to false */
            _bUpdateChannelList = FALSE;
        }
        else
        {
            /* Update channel list update flag */
            _bUpdateChannelList = TRUE;
        }
    }
}

tVoid fc_sxm_tclChannelList::vNotifyChannelInformationUpdate(tFCSXM_SelectMode tSelectMode)
{
    if(TRUE == _bChannelListReady) {
      if ( e_SUBSCRIPTION_UPDATED == tSelectMode) {
          vRetuneCurrentChannel();
      }
      // Search for required channel ID
      CHANNEL_ID u16TuneChannelID = _u16TunedChannelChannelID;

      fc_sxm_tcl_trAudioPropertyChannelInformation tChannelInfoProp;
      midw_ext_fi_tcl_ChannelInformation &oFiChn=tChannelInfoProp.oFiRes.ChannelInformation;

      fc_sxm_tclAdvisories* poAdvisories = fc_sxm_tclAdvisories::instance();
      SXM_ASSERT_RETURN(NULL != poAdvisories)

      if((FC_SXM_AUDIO_RADIO_ID_CHANNEL == u16TuneChannelID) &&
              (tSelectMode != e_UNAVAILABLE_CHANNEL) &&
              (FALSE == poAdvisories->bIsAdvisoryTimerRunning()) &&
              (tSelectMode != e_MATURE_LOCKED_CHANNEL) &&
              (tSelectMode != e_SKIPPED_CHANNEL) &&
              (tSelectMode != e_SUBSCRIPTION_UPDATED)) {
         tSelectMode = e_RADIO_ID_CHANNEL;
      } /* Do not display advisory on RADIO ID channel */

      enAdvisoryMsgType enCheckAdvisory = enNO_ADVISORY;
      midw_ext_fi_tcl_e8_SystemMessage::tenType enCurrentAdvisory = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;
      // Get appropriate CheckAdvisory and CurrentAdvisory Messages
      vApplyAdvisory(tSelectMode, u16TuneChannelID, enCheckAdvisory, enCurrentAdvisory);
      tBool bMuteStatus = TRUE;
      tBool  bReplayModeAdvisory = (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_LOADING_XM == enCurrentAdvisory) ||
               (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SXM_INITIALIZING == enCurrentAdvisory);
      ETG_TRACE_USR2(("vNotifyChannelInformationUpdate (tSelectMode, enCheckAdvisory, bReplayModeAdvisory) = %d %d, %d", tSelectMode,
               enCheckAdvisory,
               bReplayModeAdvisory));
      if(enHIGH_PRIO_ADVISORY != enCheckAdvisory)
      {
          vFillLowPrioAdvisoryMsg(u16TuneChannelID, enCheckAdvisory, oFiChn, tSelectMode);
          bMuteStatus = FALSE;
      } else if((bIsReplayModeOn()) && (!bReplayModeAdvisory)) {
          vFillReplayModeElements(u16TuneChannelID, oFiChn);
          // The following condition is applicable whenever the ReplayModeis ON and a subscription updation happens
          if (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SUBSCRIPTION_UPDATED != enCurrentAdvisory) {
              enCurrentAdvisory = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;
          }
          bMuteStatus = FALSE;
      }
      else if((enHIGH_PRIO_ADVISORY == enCheckAdvisory) &&
              (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_ANTENNA_DISCONNECTED == enCurrentAdvisory)) {
          bMuteStatus = FALSE;
          ETG_TRACE_USR4(("vNotifyChannelListUpdate() : Audio unmuted for check antenna use case"));
      }
      vSetMuteState(bMuteStatus);

      tBool bSaveArtistStatus, bSaveSongStatus;
      vCheckSongInfoAvailibilityStatus(u16TuneChannelID, bSaveArtistStatus, bSaveSongStatus);

      oFiChn.ArtistAlertStatus = bSaveArtistStatus;
      oFiChn.SongAlertStatus = bSaveSongStatus;

      vNotifySystemMsg(enCurrentAdvisory);
      vChannelInformationUpdate(tChannelInfoProp, tSelectMode);
      vPrintChnInfoUpdate(oFiChn);

   }
}

/**
 * Check artist and song info availability status
 * @param hChannelID[in] valid channel id
 * @param bSaveArtistStatus[out] TRUE when request channel has Artist info available else FALSE
 * @param bSaveSongStatus[out] TRUE when request channel has Song info available else FALSE
 */
tVoid fc_sxm_tclChannelList::vCheckSongInfoAvailibilityStatus(CHANNEL_ID hChannelID, tBool& bSaveArtistStatus, tBool& bSaveSongStatus) const
{
   DECODER_OBJECT hDecoder = (_poAudioApp) ? _poAudioApp->oGetDecoderObject() : DECODER_INVALID_OBJECT;
   if(DECODER_INVALID_OBJECT != hDecoder)
   {
      SEEK_AVAILABILTY_ENUM enSongAvailablity = SEEK_AVAILIBILITY_UNKNOWN;
      SEEK_AVAILABILTY_ENUM enArtistAvailablity = SEEK_AVAILIBILITY_UNKNOWN;

      fc_sxm_tclContentAlertProxy::instance()->vIsAvailableToRegister(hDecoder, hChannelID, &enSongAvailablity, &enArtistAvailablity);

      bSaveSongStatus = bGetSaveSongInfoStatus(enSongAvailablity);
      bSaveArtistStatus = bGetSaveSongInfoStatus(enArtistAvailablity);

      ETG_TRACE_USR2(("vCheckSongInfoStatus, hChannelId=%d bSaveArtistStatus=%d bSaveSongStatus=%d", hChannelID,
               bSaveArtistStatus,
               bSaveSongStatus));
   }
}

/**
 * Get save song info status
 * @param enSeekAvailablity[in] contain the availability of artist or song info
 * @param TRUE when artist or song info available or already registered else FALSE
 */
tBool fc_sxm_tclChannelList::bGetSaveSongInfoStatus(SEEK_AVAILABILTY_ENUM enSeekAvailablity) const
{
   return ((SEEK_AVAILIBILITY_UNAVAILABLE == enSeekAvailablity) ||
           (SEEK_AVAILIBILITY_NOT_ALLOWED == enSeekAvailablity) ||
           (SEEK_AVAILIBILITY_UNKNOWN == enSeekAvailablity)) ?
               FALSE :
               TRUE;
}

// Calculate the advisory to be applied. Based on the current set Select Mode,
// 1) the AdvisoryMsgType and CheckAdvisory is calculated
// 2) Timers are started for indicating the duration for which the advisories needs to be displayed
tVoid fc_sxm_tclChannelList::vApplyAdvisory(tFCSXM_SelectMode tSelectMode,
        CHANNEL_ID& u16TuneChannelID,
        enAdvisoryMsgType& enCheckAdvisory,
        midw_ext_fi_tcl_e8_SystemMessage::tenType& enCurrentAdvisory) const
{
    fc_sxm_tclAdvisories* poAdvisories = fc_sxm_tclAdvisories::instance();
    SXM_ASSERT_RETURN(NULL != poAdvisories)
    // selectMode acts as the input based on which the advisory type is retrieved
    switch(tSelectMode) {
         case e_RADIO_ID_CHANNEL: {
            enCheckAdvisory = enRADIO_ID_DISPLAY;
         } break;
         case e_MATURE_LOCKED_CHANNEL:
         case e_SKIPPED_CHANNEL:
         case e_UNAVAILABLE_CHANNEL: {
            u16TuneChannelID = _u16RequestedUnavailableChannel;
            enCurrentAdvisory = poAdvisories->eGetCurrentAdvisoryType(TRUE, enCheckAdvisory);
            poAdvisories->vStartAdvisoryTimer(FC_SXM_AUDIO_DEFAULT_ADVISORY_UPDATE_PERIOD);
         } break;
         case e_MATURE_LOCKED_CHANNEL_STARTUP:
         case e_SKIPPED_CHANNEL_STARTUP:
         case e_UNAVAILABLE_CHANNEL_STARTUP: {
             enCurrentAdvisory = poAdvisories->eGetCurrentAdvisoryType(TRUE, enCheckAdvisory);
            poAdvisories->vStartAdvisoryTimer(FC_SXM_AUDIO_DEFAULT_ADVISORY_UPDATE_PERIOD);
         } break;
         case e_CHANNEL_BROWSE: {
            poAdvisories->vStopAndClearChannelAdvisory();
            enCurrentAdvisory = poAdvisories->eGetCurrentAdvisoryType(FALSE, enCheckAdvisory);
            if(enHIGH_PRIO_ADVISORY == enCheckAdvisory) {
               poAdvisories->vStartAdvisoryTimer(FC_SXM_AUDIO_HIGH_PRIO_ADVISORY_UPDATE_PERIOD);
                enCheckAdvisory = enCHANNEL_ADVISORY;
            } else {
                enCheckAdvisory = enNO_ADVISORY;
            }
            enCurrentAdvisory = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;
         } break;
         case e_ADVISORY_TIMEOUT:
         case e_CHANNEL_ART_UPDATE:
         case e_ADVISORY_UPDATE: {
            enCurrentAdvisory = poAdvisories->eGetCurrentAdvisoryType(FALSE, enCheckAdvisory);
         } break;
         case e_SUBSCRIPTION_UPDATED:
         case e_FACTORY_RESET: {
            enCurrentAdvisory = poAdvisories->eGetCurrentAdvisoryType(FALSE, enCheckAdvisory);
            enCheckAdvisory = enHIGH_PRIO_ADVISORY;
         } break;
         case e_SELECT_CHANNEL:
         /* fall throught */  /* TO REMOVE LINT WARNING DO NOT DELETE THIS COMMENT */
         case e_RECALL_PRESET:
         /* fall throught */  /* TO REMOVE LINT WARNING DO NOT DELETE THIS COMMENT */
         case e_TUNING_CHANNEL:
         /* fall throught */  /* TO REMOVE LINT WARNING DO NOT DELETE THIS COMMENT */
         case e_TUNED_CHANNEL: {
            enCurrentAdvisory = poAdvisories->eGetCurrentAdvisoryType(FALSE, enCheckAdvisory);
            if(enHIGH_PRIO_ADVISORY == enCheckAdvisory) {
               if(!(poAdvisories->bIsAdvisoryTimerRunning())) {
                  poAdvisories->vStartAdvisoryTimer(FC_SXM_AUDIO_HIGH_PRIO_ADVISORY_UPDATE_PERIOD);
               }
               enCheckAdvisory = enCHANNEL_ADVISORY;
            } else {
               enCheckAdvisory = enNO_ADVISORY;
            }
            enCurrentAdvisory = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;
         } break;
         default:
            break;
      }
    ETG_TRACE_USR3(("vApplyAdvisory tSelectMode=(%u) u16TuneChannelID=(%u) enCheckAdvisory=(%u) enCurrentAdvisory=(%u)",
            tSelectMode, u16TuneChannelID, enCheckAdvisory, enCurrentAdvisory));
}

// Tune to the given Channel ID
tVoid fc_sxm_tclChannelList::vUseChannel(CHANNEL_ID channelID) const
{
    // Get Decoder object from AudioApp
    DECODER_OBJECT hDecoder = (_poAudioApp) ?
            _poAudioApp->oGetDecoderObject() : DECODER_INVALID_OBJECT;
    // Use the given channelID
    SMSAPI_RETURN_CODE_ENUM enRetCode = DECODER.eUseChannel(hDecoder,
            channelID, cb_DecoderChannelAccessCallback, NULL);
    ETG_TRACE_USR1(("fc_sxm_tclChannelList::vUseChannel channelID(%u) DECODER.eUseChannel() returns (%u)",
            channelID, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode)));
    if( SMSAPI_RETURN_CODE_SUCCESS != enRetCode)
    {
        ETG_TRACE_ERR(("DECODER.eUseChannel() Failed  Error Code = %u",
                ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode)));
    }
}

// Fill FI structure when ReplayMode is ON and Current Advisory != midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_LOADING_XM
tVoid fc_sxm_tclChannelList::vFillReplayModeElements(CHANNEL_ID channelID, midw_ext_fi_tcl_ChannelInformation &oFiChn) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vFillReplayModeElements()"));
    // Find for tuned channel from channel list
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter = _mChannelList.find(channelID);
    if (mIter != _mChannelList.end()) {
       fc_sxm_trChannel const &rChn=(*mIter).second;
       fc_sxm_vString2Fi(rChn.oChannelSongName.c_str(), oFiChn.SongName);
       oFiChn.SongNameAvail = fc_sxm_pValidateString(oFiChn.SongName.szValue);
       fc_sxm_vString2Fi(rChn.oChannelArtistName.c_str(), oFiChn.ArtistName);
       oFiChn.ArtistNameAvail = fc_sxm_pValidateString(oFiChn.ArtistName.szValue);
       fc_sxm_vString2Fi(rChn.oChannelSongName.c_str(), oFiChn.ExtSongName);
       oFiChn.ExtSongNameAvail = rChn.bIsSongNameAvailable();
       fc_sxm_vString2Fi(rChn.oChannelArtistName.c_str(), oFiChn.ExtArtistName);
       oFiChn.ExtArtistNameAvail = rChn.bIsArtistNameAvailable();
       oFiChn.ChannelNumber = rChn.u16ChannelId;
       oFiChn.SID = rChn.u16ServiceId;
       oFiChn.CategoryType = rChn.u16CategoryId;
       oFiChn.PrimaryCategoryType = rChn.u16CategoryId;
       fc_sxm_vString2Fi(rChn.oAudioChannelName.c_str(), oFiChn.ChannelName);
       oFiChn.ChannelNameAvail = rChn.bIsChannelNameAvailable();
       fc_sxm_vString2Fi(rChn.oChannelCategoryName.c_str(), oFiChn.CategoryName);
       oFiChn.CategoryNameAvail = rChn.bIsCategoryNameAvailable();
       oFiChn.ChnGraphicAvail = rChn.bIsChannelArtAvailable;
       if (rChn.bIsChannelArtAvailable) {
           oFiChn.ChnArtAvail.enType = (_bChannelArtUpdate) ?
                   midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH :
                   midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_NOREFRESH;
           ETG_TRACE_USR2(("midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH "));
       }
       else {
           oFiChn.ChnArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
           ETG_TRACE_USR2(("midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT "));
       }
       if (rChn.bIsAlbumArtAvailable) {
           oFiChn.AlbumArtAvail.enType = (_bChannelArtUpdate) ?
                   midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH :
                   midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_NOREFRESH;
       }
       else {
           oFiChn.AlbumArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
       }
       oFiChn.IsAvailable = rChn.bIsAvailable();
       oFiChn.IsLocked = rChn.bIsLocked();
       oFiChn.IsMature = rChn.bIsMature();
       oFiChn.IsSkipped = rChn.bIsSkipped();
    }
}

// Method decides which Low Prio the advisory msg needs to be displayed and fills up the corresponding FI structure
tVoid fc_sxm_tclChannelList::vFillLowPrioAdvisoryMsg(CHANNEL_ID u16TuneChannelID, enAdvisoryMsgType enCheckAdvisory, midw_ext_fi_tcl_ChannelInformation &oFiChn, tFCSXM_SelectMode tSelectMode) const
{
    ETG_TRACE_USR3(("vFillLowPrioAdvisoryMsg u16TuneChannelID(%u) enCheckAdvisory(%u)", u16TuneChannelID, enCheckAdvisory));
    // Find for tuned channel from channel list
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
    mIter = _mChannelList.find(u16TuneChannelID);
    if (mIter != _mChannelList.end())
    {
        ETG_TRACE_USR3(("vFillLowPrioAdvisoryMsg Found CID"));
        fc_sxm_trChannel const &rChn = mIter->second;
        switch(enCheckAdvisory) {
          case enRADIO_ID_DISPLAY: {
              ETG_TRACE_USR3(("enRADIO_ID_DISPLAY"));
              vFillRadioIDAdvisoryMsg(rChn, oFiChn);
          } break;
          case enNO_ADVISORY: {
              ETG_TRACE_USR3(("enNO_ADVISORY"));
              vFillChannelAdvisoryMsg(rChn, oFiChn, tSelectMode);             
          } break;
          case enCHANNEL_ADVISORY: {
              ETG_TRACE_USR3(("enCHANNEL_ADVISORY"));
              vFillChannelAdvisoryMsg(rChn, oFiChn, tSelectMode);
          } break;
          default:
            break;
       }
    } else {
        ETG_TRACE_USR3(("USING CHANNEL"));
        vUseChannel(u16TuneChannelID);
    }
}

// Method Fills Details needed for RadioID Advisory Msg to FI Structure midw_ext_fi_tcl_ChannelInformation
tVoid fc_sxm_tclChannelList::vFillRadioIDAdvisoryMsg(fc_sxm_trChannel const &rChn, midw_ext_fi_tcl_ChannelInformation &oFiChn) const
{
    oFiChn.ChannelNumber = rChn.u16ChannelId;
    oFiChn.SID = rChn.u16ServiceId;
    oFiChn.ChnGraphicAvail = rChn.bIsChannelArtAvailable;
    if (rChn.bIsChannelArtAvailable) {
        oFiChn.ChnArtAvail.enType = (_bChannelArtUpdate) ?
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH :
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_NOREFRESH;
        ETG_TRACE_USR2(("vFillRadioIDAdvisoryMsg::FI_EN_GRAPHICS_NEEDS_REFRESH "));
    }
    else {
        oFiChn.ChnArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
        ETG_TRACE_USR2(("vFillRadioIDAdvisoryMsg::FI_EN_GRAPHICS_NEEDS_DEFAULT "));
    }

    if (rChn.bIsAlbumArtAvailable) {
        oFiChn.AlbumArtAvail.enType = (_bChannelArtUpdate) ?
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH :
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_NOREFRESH;
    }
    else {
        oFiChn.AlbumArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
    }

    oFiChn.ChannelNameAvail = TRUE;
    tChar c8RadioIDName[SXM_AUDIO_DISP_TEXT_LENGTH] = { '\0' };
    sprintf(c8RadioIDName, "RADIO ID: %s", rChn.oChannelArtistName.c_str());
    fc_sxm_vString2Fi(c8RadioIDName, oFiChn.ChannelName);
}

// Method Fills Details needed for Channel Advisory Msg to FI Structure midw_ext_fi_tcl_ChannelInformation
tVoid fc_sxm_tclChannelList::vFillChannelAdvisoryMsg(fc_sxm_trChannel const &rChn, midw_ext_fi_tcl_ChannelInformation &oFiChn, tFCSXM_SelectMode tSelectMode) const
{
    ETG_TRACE_USR1(("fc_sxm_tclChannelList::vFillChannelAdvisoryMsg"));
    oFiChn.ChannelNumber = rChn.u16ChannelId;
    ETG_TRACE_USR4(("fc_sxm_tclChannelList oFiChn.ChannelNumber %d", oFiChn.ChannelNumber));
    oFiChn.SID = rChn.u16ServiceId;
    oFiChn.CategoryType = rChn.u16CategoryId;
    oFiChn.PrimaryCategoryType = rChn.u16CategoryId;
    fc_sxm_vString2Fi(rChn.oAudioChannelName.c_str(), oFiChn.ChannelName);
    oFiChn.ChannelNameAvail = rChn.bIsChannelNameAvailable();
    fc_sxm_vString2Fi(rChn.oChannelCategoryName.c_str(), oFiChn.CategoryName);
    oFiChn.CategoryNameAvail = rChn.bIsCategoryNameAvailable();
    oFiChn.ChnGraphicAvail = rChn.bIsChannelArtAvailable;
    if (rChn.bIsChannelArtAvailable) {
        oFiChn.ChnArtAvail.enType = (_bChannelArtUpdate) ?
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH :
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_NOREFRESH;
        ETG_TRACE_USR2(("vFillChannelAdvisoryMsg::FI_EN_GRAPHICS_NEEDS_REFRESH '%d'", oFiChn.ChnArtAvail.enType));
    }
    else {
        oFiChn.ChnArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
        ETG_TRACE_USR2(("vFillChannelAdvisoryMsg::FI_EN_GRAPHICS_NEEDS_DEFAULT "));
    }
    if (rChn.bIsAlbumArtAvailable) {
        oFiChn.AlbumArtAvail.enType = (_bChannelArtUpdate) ?
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH :
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_NOREFRESH;
    }
    else {
        oFiChn.AlbumArtAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
    }
    oFiChn.IsAvailable = rChn.bIsAvailable();
    oFiChn.IsLocked = rChn.bIsLocked();
    oFiChn.IsMature = rChn.bIsMature();
    oFiChn.IsSkipped = rChn.bIsSkipped();

   /*In case of select channel using rotary encoder, don't update artist and song info*/
   if (!(tSelectMode == e_CHANNEL_BROWSE))
   {
      // If no advisory needs to be displayed, instead of adding In addition to filling up of
       fc_sxm_vString2Fi(rChn.oChannelSongName.c_str(), oFiChn.SongName);
       oFiChn.SongNameAvail = fc_sxm_pValidateString(oFiChn.SongName.szValue);
       fc_sxm_vString2Fi(rChn.oChannelArtistName.c_str(), oFiChn.ArtistName);
       oFiChn.ArtistNameAvail = fc_sxm_pValidateString(oFiChn.ArtistName.szValue);
       fc_sxm_vString2Fi(rChn.oChannelSongName.c_str(), oFiChn.ExtSongName);
       oFiChn.ExtSongNameAvail = rChn.bIsSongNameAvailable();
       fc_sxm_vString2Fi(rChn.oChannelArtistName.c_str(), oFiChn.ExtArtistName);
       oFiChn.ExtArtistNameAvail = rChn.bIsArtistNameAvailable();
   }
}

// Method sends channel information property update to registered clients
tVoid fc_sxm_tclChannelList::vChannelInformationUpdate(fc_sxm_tcl_trAudioPropertyChannelInformation const& tChannelInfoProp,
        tFCSXM_SelectMode tSelectMode) const
{
    if(bForceChannelInfoUpdate(tSelectMode))
    {
        ETG_TRACE_USR4(("fc_sxm_tclChannelList::vChannelInformationUpdate ---> bForceChannelInfoUpdate() returned TRUE"));
        // Set the status info and notify all clients
        fc_sxm_tclAudioProperties::instance()->oChannelInformation.vSet(tChannelInfoProp);
        fc_sxm_tclAudioProperties::instance()->oChannelInformation.vNotify();
    }
    else
    {
        ETG_TRACE_USR4(("fc_sxm_tclChannelList::vChannelInformationUpdate ---> bForceChannelInfoUpdate() returned FALSE"));
        // Set the status info and notify all clients only if there is any change from earlier update
        fc_sxm_tclAudioProperties::instance()->oChannelInformation.vSetAndNotify(tChannelInfoProp);
    }
}


// Method updates system messages to registered clients
tVoid fc_sxm_tclChannelList::vNotifySystemMsg(midw_ext_fi_tcl_e8_SystemMessage::tenType enCurrentAdvisory) const
{
    ETG_TRACE_USR1(("fc_sxm_tclChannelList::vNotifySystemMsg Advisory Notified to Registered Clients = %u", enCurrentAdvisory));
    // Update system message
    fc_sxm_tcl_trAudioPropertySystemMessage tSystemMessage;
    // Set no error as advisory
    tSystemMessage.oFiRes.SystemErrorType.enType = enCurrentAdvisory;
    // Notify System and ChannelInformation Update
    if (fc_sxm_tclAudioUtils::bIsChannelTuneError(enCurrentAdvisory))
    {
        tSystemMessage.oFiRes.CID = _u16RequestedUnavailableChannel;
    }
    else
    {
        tSystemMessage.oFiRes.CID = _u16TunedChannelChannelID;
    }
    ETG_TRACE_USR2(("tSystemMessage ChannelNumber -- '%u'", tSystemMessage.oFiRes.CID));
    // Notify all clients
    fc_sxm_tclAudioProperties::instance()->oSystemMessage.vSet(tSystemMessage);
    fc_sxm_tclAudioProperties::instance()->oSystemMessage.vNotify();
}

// Utility method to print FI data to be sent using class midw_ext_fi_tcl_ChannelInformation
tVoid fc_sxm_tclChannelList::vPrintChnInfoUpdate(const midw_ext_fi_tcl_ChannelInformation& oFiChn) const
{
    // Print all channel related info
    if (etg_bIsTraceActive(TR_CLASS_FC_SXM_CHANNEL_LIST, TR_LEVEL_USER_2))
    {
          ETG_TRACE_USR2(("ChannelNumber ------------ '%d'", oFiChn.ChannelNumber));
          ETG_TRACE_USR2(("SID ---------------------- '%d'", oFiChn.SID));
          ETG_TRACE_USR2(("ChannelNameAvail --------- '%d'", oFiChn.ChannelNameAvail));
          ETG_TRACE_USR2(("ChannelName -------------- '%s'", oFiChn.ChannelName.szValue));
          ETG_TRACE_USR2(("CategoryType ------------- '%d'", oFiChn.CategoryType));
          ETG_TRACE_USR2(("CategoryNameAvail -------- '%d'", oFiChn.CategoryNameAvail));
          ETG_TRACE_USR2(("CategoryName ------------- '%s'", oFiChn.CategoryName.szValue));
          ETG_TRACE_USR2(("ArtistNameAvail ---------- '%d'", oFiChn.ArtistNameAvail));
          ETG_TRACE_USR2(("ArtistName --------------- '%s'", oFiChn.ArtistName.szValue));
          ETG_TRACE_USR2(("SongNameAvail ------------ '%d'", oFiChn.SongNameAvail));
          ETG_TRACE_USR2(("SongName ----------------- '%s'", oFiChn.SongName.szValue));
          ETG_TRACE_USR2(("ChnGraphicAvail ---------- '%d'", oFiChn.ChnGraphicAvail));
          ETG_TRACE_USR2(("ChnArtAvail--------------- '%d'", oFiChn.ChnArtAvail.enType));
          ETG_TRACE_USR2(("AlbumArtAvail ------------ '%d'", oFiChn.AlbumArtAvail.enType));
          ETG_TRACE_USR2(("IsAvailable--------------- '%u'", oFiChn.IsAvailable));
          ETG_TRACE_USR2(("IsLocked  ---------------- '%u'", oFiChn.IsLocked));
          ETG_TRACE_USR2(("IsMature   --------------- '%u'", oFiChn.IsMature));
          ETG_TRACE_USR2(("IsSkipped ---------------- '%u'", oFiChn.IsSkipped));
          ETG_TRACE_USR2(("SaveArtistStatus --------- '%u'", oFiChn.ArtistAlertStatus));
          ETG_TRACE_USR2(("SaveSongStatus ----------- '%u'", oFiChn.SongAlertStatus));
    }
}

tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgChannelEventCallback const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgChannelEventCallback) ChannelID=(%u) Mask=(%u)",
           prMsg->rChannel.u16ChannelId, prMsg->tEventMask));
   if (fc_sxm_tclAudioProxy::instance()->bAreHandleChnCatCallbacksAllowed()) {
       vUpdateChannelInfo(prMsg->rChannel, prMsg->tEventMask);
   }
}

tVoid fc_sxm_tclChannelList::vUpdateAllChannels(midw_ext_fi_tcl_ChannelList &rChannelList)
{
   ETG_TRACE_USR4(("fc_sxm_tclChannelList::vUpdateAllChannels()"));

   map<SERVICE_ID, tU8> mPresetIndices;
   vGetPresetIndices(mPresetIndices);
   map<SERVICE_ID, tU8>::const_iterator mPresetIter;

   map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
   for(mIter=_mChannelList.begin(); mIter!=_mChannelList.end(); ++mIter)
   {
      const fc_sxm_trChannel& rChannel(mIter->second);
      if(rChannel.bIsBrowseable())
      {
          /* Increment list index count */
          midw_ext_fi_tcl_ChannelListEntry tChannelListEntry;

          /* Copy service ID */
          tChannelListEntry.ServiceID = rChannel.u16ServiceId;
          /* Copy channel ID */
          tChannelListEntry.ChannelID = rChannel.u16ChannelId;
          /* Copy channel name availability */
          tChannelListEntry.ChannelNameAvail = rChannel.bIsChannelNameAvailable();
          /* Copy channel name */
          fc_sxm_vString2Fi(rChannel.oAudioChannelName.c_str(), tChannelListEntry.ChannelName);
          // Copy Artist name
          fc_sxm_vString2Fi( rChannel.oChannelArtistName.c_str(), tChannelListEntry.ChannelArtistName);
          // Copy Song name
          fc_sxm_vString2Fi( rChannel.oChannelSongName.c_str(), tChannelListEntry.ChannelSongName);

          ETG_TRACE_USR2(("Channel ID - %d, service ID - %d, channel name - %s", 
              tChannelListEntry.ChannelID, 
              tChannelListEntry.ServiceID, 
              ((tChannelListEntry).ChannelName).szValue));

          // Copy preset number
          mPresetIter = mPresetIndices.find(rChannel.u16ServiceId);
          tChannelListEntry.PresetNumber = (mPresetIter != mPresetIndices.end()) ? mPresetIter->second : 0;


          // Copy Lock, Availability, Mature and Skipped attributes
          tChannelListEntry.IsLocked = rChannel.bIsLocked();
          tChannelListEntry.IsAvailable = rChannel.bIsAvailable();
          tChannelListEntry.IsMature = rChannel.bIsMature();
          tChannelListEntry.IsSkipped = rChannel.bIsSkipped();

          // Push the data into method resuly list
          ((rChannelList).ChannelList).push_back(tChannelListEntry);
      }
   }
}

CHANNEL_ID fc_sxm_tclChannelList::getCidForIndex(tU16 u16CategoryID, tU16 u16Id) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::getCidForIndex() categoryID = (%u) u16Id = (%u)", u16CategoryID, u16Id));
    Ifc_sxm_tclCategoryList* poCategoryList = (_poAudioApp) ?
                          _poAudioApp->poGetCategoryListInstance(): NULL;
    SXM_ASSERT_RETURN_VAL (NULL != poCategoryList, CHANNEL_INVALID_ID)

    return (poCategoryList->u16GetCidFromCategoryListIndex(u16CategoryID, u16Id));
}

CHANNEL_ID fc_sxm_tclChannelList::getCidForRelativeIndex(CATEGORY_ID categoryID, CHANNEL_ID tunedChannelId, tS16 s16RelativeIndex) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::getCidForRelativeIndex() categoryID = (%u) tunedChannelId = (%u) s16RelativeIndex = (%u)",
            categoryID, tunedChannelId, s16RelativeIndex));
    Ifc_sxm_tclCategoryList* poCategoryList = (_poAudioApp) ?
                              _poAudioApp->poGetCategoryListInstance(): NULL;
    SXM_ASSERT_RETURN_VAL (NULL != poCategoryList, CHANNEL_INVALID_ID)

    return (poCategoryList->u16GetCidFromCategoryListRelativeIndex(categoryID, tunedChannelId, s16RelativeIndex));
}

CHANNEL_ID fc_sxm_tclChannelList::getNextProgramCid(tU16 u16CategoryID, tS16 s16TunedChannelId) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::getNextProgramCid() u16CategoryID = %u \t s16TunedChannelId = %u", u16CategoryID, s16TunedChannelId));
    Ifc_sxm_tclCategoryList* poCategoryList = (_poAudioApp) ?
                              _poAudioApp->poGetCategoryListInstance(): NULL;
    SXM_ASSERT_RETURN_VAL (NULL != poCategoryList, CHANNEL_INVALID_ID)

    return (poCategoryList->u16GetCidOfNextProgramFromCategoryList(u16CategoryID, s16TunedChannelId));
}

tVoid fc_sxm_tclChannelList::vNotifyTunedChannelPreset(SERVICE_ID serviceID, tBool bNeedsUpdate) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vNotifyTunedChannelPreset() serviceID = %u \t bNeedsUpdate = %u", serviceID, bNeedsUpdate));
    Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                          _poAudioApp->poGetPresetsInstance(): NULL;
    SXM_ASSERT_RETURN(NULL != poPresets)
    poPresets->vNotifyTunedChannelPresetIndex(serviceID);
    if ( TRUE == bNeedsUpdate)
        poPresets->vUpdateIRMode();
}

tU8 fc_sxm_tclChannelList::u8GetPresetsIndex(SERVICE_ID serviceID) const
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::u8GetPresetsIndex()"));
    Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                          _poAudioApp->poGetPresetsInstance(): NULL;
    SXM_ASSERT_RETURN_VAL(NULL != poPresets, 0)
    return (poPresets->u8GetPresetIndex(serviceID));
}

tVoid fc_sxm_tclChannelList::vGetPresetIndices(map<SERVICE_ID, tU8>& mPresetIndices) const
{
    Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                              _poAudioApp->poGetPresetsInstance(): NULL;
    SXM_ASSERT_RETURN(NULL != poPresets)
    poPresets->vGetPresetIndices(mPresetIndices);
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vGetPresetIndices() mPresetIndices.size = %u", mPresetIndices.size()));
}

tBool fc_sxm_tclChannelList::bIsReplayModeOn() const
{
    Ifc_sxm_tclPlayback* poPlayback = (_poAudioApp) ?
                              _poAudioApp->poGetPlaybackInstance(): NULL;
    SXM_ASSERT_RETURN_VAL(NULL != poPlayback, FALSE)

    tBool bReplayOn = poPlayback->bIsReplayModeActive();
    ETG_TRACE_USR3(("fc_sxm_tclChannelList::bIsReplayModeOn() %u", bReplayOn));

    return bReplayOn;
}

/* Update Tuned channel's advisory */
tVoid fc_sxm_tclChannelList::vUpdateTunedChannelAdvisory(CHANNEL_ID channelID, const fc_sxm_trChannel& trChannel)
{
    midw_ext_fi_tcl_e8_SystemMessage::tenType enSysMsg;
    // Don't send system messages for locked and skipped status change for currently tuned channel
    if (!trChannel.bIsAvailable())
    {
        // Channel not available
        enSysMsg = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE;
    }
    else if(!trChannel.bIsAuthorized())
    {
        // Channel unauthorized
        enSysMsg = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED;
    }
    else
    {
        // Channel can continue to be tuned
        enSysMsg = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;
    }

    if (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR != enSysMsg)
    {
        ETG_TRACE_USR1(("fc_sxm_tclChannelList::vUpdateTunedChannelAdvisory(%u) Set tuned channel msg (%u) ", channelID, enSysMsg));
        // Set unauthorised channel as current channel
        _u16RequestedUnavailableChannel = channelID;
        // Set channel unavailable system message
        (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(enSysMsg, TRUE);
    }
    ETG_TRACE_USR3(("fc_sxm_tclChannelList::vUpdateTunedChannelAdvisory enSysMsg = %u", enSysMsg));
}

// Method checks if the channel's browse status and sets appropriate display advisory if its not browseable
tVoid fc_sxm_tclChannelList::vSetChannelBrowseStatus()
{
    // Get Decoder object from Audio App
    DECODER_OBJECT hDecoder = (_poAudioApp) ? _poAudioApp->oGetDecoderObject() :
            DECODER_INVALID_OBJECT;
    // Get Last Tuned channel id from Decoder
    CHANNEL_ID u16LastTunedChannel = DECODER.tLastTunedChannelId(hDecoder);
    ETG_TRACE_USR2(("Current tuned channel id ------ %u", _u16TunedChannelChannelID));
    ETG_TRACE_USR2(("Last tuned valid channel id --- %u", u16LastTunedChannel));

    tTunable enChannelBrowseStatus = e_UNAVAILABLE;
    if(_u16TunedChannelChannelID != u16LastTunedChannel) {
      _u16RequestedUnavailableChannel = _u16TunedChannelChannelID;
    } else {
      enChannelBrowseStatus = e_TUNABLE;
    }
    // Iterate thru the channel_list and find the Browse Status of Tuned channel
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator mIter;
    mIter = _mChannelList.find(u16LastTunedChannel);
    if(mIter != _mChannelList.end()) {
     enChannelBrowseStatus = mIter->second.enBrowseStatus();
    }

    // If the Browse status is other than "e_TUNABLE", set appropriate advisory to be displayed
    switch(enChannelBrowseStatus) {
     case e_UNAVAILABLE: {
        _u16RequestedUnavailableChannel = FC_SXM_AUDIO_DEFAULT_CHANNEL;
        (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, TRUE);
        vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL_STARTUP);
     } break;
     case e_UNAUTHORIZED: {
        _u16RequestedUnavailableChannel = FC_SXM_AUDIO_DEFAULT_CHANNEL;
        (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, TRUE);
        vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL_STARTUP);
     } break;
     case e_LOCKED: {
         (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE, TRUE);
         vNotifyChannelInformationUpdate(e_MATURE_LOCKED_CHANNEL_STARTUP);
     } break;
     case e_SKIPPED: {
         (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED, TRUE);
         vNotifyChannelInformationUpdate(e_SKIPPED_CHANNEL_STARTUP);
     } break;
     default:
       break;
    }
}

/* Set Audio App Pointer Reference*/
tVoid fc_sxm_tclChannelList::vSetApp(Ifc_sxm_tclAudioApp* audioApp)
{
    ETG_TRACE_USR4(("fc_sxm_tclChannelList::vSetApp()"));
    _poAudioApp = audioApp;
}
/* Change to one of the preset channels has occurred... Notify Presets about the change*/
tVoid fc_sxm_tclChannelList::vUpdatePresetBands(fc_sxm_trChannel const &rChannel, CHANNEL_EVENT_MASK enEventMask) const
{
    // There is a separate property update sent for changes to TunedChannel
    if ( (rChannel.bIsPreset())   &&
            (TRUE == fc_sxm_tclAudioUtils::bSendPresetBandNotifications(static_cast <tU32> (enEventMask))) )
    {
        Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                                      _poAudioApp->poGetPresetsInstance(): NULL;
        SXM_ASSERT_RETURN (NULL != poPresets)
        return (poPresets->vPresetBandUpdate(rChannel, enEventMask));
    }
}

/* Change to one of the category channels has occurred... Notify Category list about the change*/
tVoid fc_sxm_tclChannelList::vUpdateCategoryList() const
{
    Ifc_sxm_tclCategoryList* poCatList = (_poAudioApp) ?
                                  _poAudioApp->poGetCategoryListInstance(): NULL;
    SXM_ASSERT_RETURN (NULL != poCatList)
    return (poCatList->vCategoryListUpdate());
}

/* Lock or Unlock Mature Channels*/
tVoid fc_sxm_tclChannelList::vDisableAllMatureChannels(tBool bDisableAllMatureChn)
{
    DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
    _u16TunedChannelChannelID = DECODER.tCurrentChannelId(hDecoder);
    _bEnableMatureChannels = bDisableAllMatureChn;
    Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
                _poAudioApp->poGetPresetsInstance(): NULL;
    SXM_ASSERT_RETURN(NULL != poPresets)
    poPresets->vTuneScanConfigure();
    map<CHANNEL_ID, fc_sxm_trChannel>::iterator chnIt = _mChannelList.begin();
    for( ;chnIt != _mChannelList.end(); ++chnIt)
    {
        if (chnIt->second.bIsMature())
        {
            tBool bRequestStatus = FALSE;
            if (bDisableAllMatureChn)
            {
                bRequestStatus = DECODER.bLockChannel(hDecoder, chnIt->second.u16ChannelId);
                if (chnIt->second.u16ChannelId == _u16TunedChannelChannelID)
                {
                    vHandleTunedChannelLock();
                }
                ETG_TRACE_USR3(("LOCK A Mature channel with Channel ID: %u\t RequestType: LOCK\t bRequestStatus = %u",
                                    chnIt->second.u16ChannelId, bRequestStatus));
            }
            else
            {
                bRequestStatus = DECODER.bUnlockChannel(hDecoder, chnIt->second.u16ChannelId);
                ETG_TRACE_USR3(("UNLOCK A Mature channel with Channel ID: %u\t RequestType: UNLOCK\t bRequestStatus = %u",
                                    chnIt->second.u16ChannelId, bRequestStatus));
            }
            if (bRequestStatus)
            {
                chnIt->second.vLockChannel(bDisableAllMatureChn);
            }
        }
    } // End for loop
}
/* Method sets skip state to true for all channels*/
tBool fc_sxm_tclChannelList::bSkipAllChannels()
{
    DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
    const SERVICE_ID sidPreviewChn = u16GetSIDOfPreviewChannel();
    map<CHANNEL_ID, fc_sxm_trChannel>::iterator chnIt = _mChannelList.begin();
    for( ;chnIt != _mChannelList.end(); ++chnIt)
    {
        if ((!chnIt->second.bIsSkipped()) &&
                (sidPreviewChn != chnIt->second.u16ServiceId) &&
                (FC_SXM_AUDIO_RADIO_ID_CHANNEL != chnIt->second.u16ChannelId))
        {
            tBool bRequestStatus = DECODER.bSkipChannel(hDecoder, chnIt->second.u16ChannelId);
            if (bRequestStatus)
            {
                chnIt->second.vSkipChannel(TRUE);
                if (chnIt->second.u16ChannelId == _u16TunedChannelChannelID)
                {
                    vHandleTunedChannelSkip();
                }
            }
            ETG_TRACE_USR3(("One channel Skipped with Channel ID: %u \t bRequestStatus = %u \t chnIt->second.enSkip = %u",
                    (chnIt->second).u16ChannelId, bRequestStatus, chnIt->second.enSkip));
        }
    }
    return TRUE;
}

/* Method sets skip state to false for all channels*/
tBool fc_sxm_tclChannelList::bUnSkipAllChannels()
{
    DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
    // User must not set preview channel state to skip/unskip
    const SERVICE_ID sidPreviewChn = u16GetSIDOfPreviewChannel();
    map<CHANNEL_ID, fc_sxm_trChannel>::iterator chnIt = _mChannelList.begin();
    for( ;chnIt != _mChannelList.end(); ++chnIt)
    {
        if ((chnIt->second.bIsSkipped()) &&
                (sidPreviewChn != chnIt->second.u16ServiceId) &&
                (FC_SXM_AUDIO_RADIO_ID_CHANNEL != chnIt->second.u16ChannelId))
        {
            tBool bRequestStatus = DECODER.bUnskipChannel(hDecoder, chnIt->second.u16ChannelId);
            if (bRequestStatus)
                chnIt->second.vSkipChannel(FALSE);
            ETG_TRACE_USR3(("One channel Unskipped with Channel ID: %u \t bRequestStatus = %u \t chnIt->second.enSkip = %u",
                    (chnIt->second).u16ChannelId, bRequestStatus, chnIt->second.enSkip));
        }
    }
    return TRUE;
}

/* Method sets skip state to true for selected channel*/
tBool fc_sxm_tclChannelList::bSkipSelectedChannel (SERVICE_ID serviceID, CHANNEL_ID channelID)
{
    tBool bSkipStatus = FALSE;
    if ((serviceID != u16GetSIDOfPreviewChannel()) && (FC_SXM_AUDIO_RADIO_ID_CHANNEL != channelID))
    {
        DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
        map<CHANNEL_ID, fc_sxm_trChannel>::iterator chnIt = _mChannelList.find(channelID);
        if (chnIt != _mChannelList.end())
        {
            if (!chnIt->second.bIsSkipped())
            {
                bSkipStatus = DECODER.bSkipChannel(hDecoder, chnIt->second.u16ChannelId);
                if (bSkipStatus)
                {
                    chnIt->second.vSkipChannel(TRUE);
                    if (chnIt->second.u16ChannelId == _u16TunedChannelChannelID)
                    {
                        vHandleTunedChannelSkip();
                    }
                }
                ETG_TRACE_USR3(("One channel Skipped with Channel ID: %u \t bSkipStatus = %u \t chnIt->second.enSkip = %u",
                                    chnIt->second.u16ChannelId, bSkipStatus, chnIt->second.enSkip));
            }
        }
    }
    return bSkipStatus;
}

/* Method sets skip state to false for selected channel*/
tBool fc_sxm_tclChannelList::bUnSkipSelectedChannel(SERVICE_ID serviceID, CHANNEL_ID channelID)
{
    tBool bUnSkipStatus = FALSE;
    if ((serviceID != u16GetSIDOfPreviewChannel()) && (FC_SXM_AUDIO_RADIO_ID_CHANNEL != channelID))
    {
        DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
        map<CHANNEL_ID, fc_sxm_trChannel>::iterator chnIt = _mChannelList.find(channelID);
        if (chnIt != _mChannelList.end())
        {
            if (chnIt->second.bIsSkipped())
            {
                bUnSkipStatus = DECODER.bUnskipChannel(hDecoder, chnIt->second.u16ChannelId);
                if (bUnSkipStatus)
                    chnIt->second.vSkipChannel(FALSE);
                ETG_TRACE_USR3(("One channel Unskipped with Channel ID: %u \t bRequestStatus = %u \t chnIt->second.enSkip = %u",
                                    chnIt->second.u16ChannelId, bUnSkipStatus, chnIt->second.enSkip));
            }
        }
    }
    return bUnSkipStatus;
}

// Method returns channel details for the given channel ID
const fc_sxm_trChannel* fc_sxm_tclChannelList::trGetChannel(CHANNEL_ID channelID) const
{
    const fc_sxm_trChannel* poChannel = OSAL_NULL;
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator chnIt = _mChannelList.find(channelID);
    if (chnIt != _mChannelList.end())
    {
        poChannel = &(chnIt->second);
    }
    return poChannel;
}

// Get channel's browse status
tBool fc_sxm_tclChannelList::bGetChannelBrowseStatus(CHANNEL_ID channelID) const
{
    tBool bChnBrowse = TRUE;
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator chnIt = _mChannelList.find(channelID);
    if (chnIt != _mChannelList.end())
    {
        bChnBrowse = chnIt->second.bIsBrowseable();
    }
    return bChnBrowse;
}

// Method prints the contents stored in _trCategoryMonitorConfig variable
tVoid fc_sxm_trCategoryMonitorConfig::vPrint() const
{
    ETG_TRACE_USR4(("CategoryID = %u \t tEventMask = %u \t bReportBrowsableChannels = %u \t bEnableMonitoring = %u",
            categoryID, tEventMask, bReportBrowsableChannels, bEnableMonitoring));
}


// Method handles HMI request for Handling "SetCategoryMonitor"
tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartSetCategoryMonitor const *prMsg)
{
    if (prMsg)
    {
        // Store Parameters passed by HMI
        _trCategoryMonitorConfig.vReset(
                static_cast <CATEGORY_ID> (prMsg->oFiMsg.CategoryID)
                , fc_sxm_tclAudioUtils::u16ConvertChannelEventBitSet(prMsg->oFiMsg.ChannelEvents)
                , prMsg->oFiMsg.MonitorBrowsableChannels
                , prMsg->oFiMsg.EnableMonitoring);

        _trCategoryMonitorConfig.vPrint();
        // Post method result
        midw_ext_sxm_audiofi_tclMsgSetCategoryMonitorMethodResult oMRes;
        oMRes.Status = TRUE;
        fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
    }
}

// Method decides if the reported CHANNEL_EVENT needs to be propagated to HMI or not
tVoid fc_sxm_tclChannelList::vNotifyCategoryChange(fc_sxm_trChannel const &rChannel, CHANNEL_EVENT_MASK enEventMask,tBool bOldSubscriptionState) const
{
    /*
     * Conditions to be checked before sending property update
     * 1. Is monitoring channel enabled?
     * 2. a. Does client wants to monitor any all channels?
     * 2. b. Does client wants to monitor channels from specific category?
     * 3. Does the changed channel belong to either 2.a. or 2.b.?
     * 4. Is the channel Browesable?
     * 5. Does the reported channel event belongs to the set of channel change events that client is interested in?
     */
    // Condition 5
    tBool bCompareEvent = fc_sxm_tclAudioUtils::bCompareChannelEventMask(enEventMask, _trCategoryMonitorConfig.tEventMask);
    tBool bChnSubscriptionChange = static_cast <tBool> (bOldSubscriptionState != rChannel.bIsSubscribed());

    // Though checking for condition 5 could be postponed after checking condition 4, processing involved in checking for conditions
    // 2 and 3 is huge. Hence Condition 5 is validated before other validations

    // Condition 1
    if ((_trCategoryMonitorConfig.bEnableMonitoring) && ((bCompareEvent) || (bChnSubscriptionChange)))
    {
        ETG_TRACE_USR4(("fc_sxm_tclChannelList::vNotifyCategoryChange ChannelID = %u \t eventMask = %u \t bCompareEvent = %u \t bChnSubscriptionChange = %u",
                    rChannel.u16ChannelId, enEventMask, bCompareEvent, bChnSubscriptionChange));
        // Condition 2 a and 2 b ==> Condition 3
        if ((_trCategoryMonitorConfig.categoryID == 0) ||
                (rChannel.bBelongsToCategory(_trCategoryMonitorConfig.categoryID)))
        {
            if (_trCategoryMonitorConfig.bReportBrowsableChannels)
            {
                // As Client has requested only to report for Browseable CHannels, check channel browse status and then report
                if ((rChannel.bIsBrowseable()) || (bChnSubscriptionChange))
                {
                    ETG_TRACE_USR4(("Trigger CategoryList timer to start for bReportBrowsableChannels = TRUE"));
                    vUpdateCategoryList();
                }
            }
            else
            {
                ETG_TRACE_USR4(("Trigger CategoryList timer to start for bReportBrowsableChannels = FALSE"));
                // Report irrespective of whether the channel is locked(also skipped) or not
                vUpdateCategoryList();
            }
        }
    }
}

// Method triggers ChannelList property update
tVoid fc_sxm_tclChannelList::vTriggerChannelListStatus(tBool bStatusChanged) const
{
    // Notify clients about the changes to ChannelList
    fc_sxm_tcl_trAudioPropertyChannelListStatus tChannelListStatus;
    tChannelListStatus.oFiRes.Availability = bStatusChanged;
    fc_sxm_tclAudioProperties::instance()->oChannelListStatus.vSet(tChannelListStatus);
    fc_sxm_tclAudioProperties::instance()->oChannelListStatus.vNotify();
    ETG_TRACE_USR4(("vTriggerChannelListStatus - channel list status notified with value %u", bStatusChanged));
}

// Method handle methodstart requests  for "RapidChannelBrowseStart" from Clients
tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartRapidChannelBrowseStart const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartRapidChannelBrowseStart"));
    DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
    CHANNEL_ID prevTunedChnID  = DECODER.tLastTunedChannelId(hDecoder);
    CHANNEL_ID startChnID = _u16TunedChannelChannelID = DECODER.tCurrentChannelId(hDecoder);
    if (TRUE == bAbortAdvancedAudioPlayback())
    {
        startChnID  = prevTunedChnID;
    }
    else
    {
       vHandleSelectChannel(startChnID);
    }
    vector<CHANNEL_ID> vectChnBrowse;
    if (midw_ext_fi_tcl_e8_BrowseChannelType::FI_EN_BROWSETYPE_ALL_CHANNELS == prMsg->oFiMsg.Config.Type.enType )
    {
        vFetchAllChannels(vectChnBrowse);
    }
    else if (midw_ext_fi_tcl_e8_BrowseChannelType::FI_EN_BROWSETYPE_CATEGORY_CHANNELS == prMsg->oFiMsg.Config.Type.enType )
    {
        vFetchCategoryChannels(vectChnBrowse, prMsg->oFiMsg.CategoryID);
    }
    tBool bPrepare = static_cast <tBool> ( vectChnBrowse.size() > 1);
    // Post method result
    midw_ext_sxm_audiofi_tclMsgRapidChannelBrowseStartMethodResult oMRes;
    oMRes.Status = bPrepare;
    // Irrespective of the size of Channel ID Listm send FI message with Status bit set to 0 or 1
    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
    if (bPrepare)
    {
        vPrepareRapidChannelBrowse(startChnID, prMsg->oFiMsg.Direction, prMsg->oFiMsg.Config, vectChnBrowse);
    }
}

// Method prepares RapidChannelBrowse object for next task
tVoid fc_sxm_tclChannelList::vPrepareRapidChannelBrowse(CHANNEL_ID chnID, midw_ext_fi_tcl_e8_BrowseDirection enDirection,
        const midw_ext_fi_tcl_BrowseChannelConfig& configuration,
        const vector<CHANNEL_ID>& vectChannels)
{
    _trRapidChnBrowse.vReset();
    ETG_TRACE_USR4(("vPrepareRapidChannelBrowse enDirection = %u _u16TunedChannelChannelID = %u",
            enDirection.enType, chnID));
    _trRapidChnBrowse.vInit(enDirection, configuration, chnID, vectChannels);
    vStartRapidChannelBrowse();
}

// Start Using Functionalities offered to Rapid Channel Browse
tVoid fc_sxm_tclChannelList::vStartRapidChannelBrowse()
{
    CHANNEL_ID chnID = _trRapidChnBrowse.u16GetNextChannelID();
    ETG_TRACE_USR4((" vStartRapidChannelBrowse chnID = %u", chnID));
    if( CHANNEL_INVALID_ID != chnID )
    {
        vNotifyRapidChannelBrowseInfo(chnID);
        tU16 u16TimerExpiry = _trRapidChnBrowse.u16GetNextDuration();
        ETG_TRACE_USR4(("vStartRapidChannelBrowse NextTick = %u ", u16TimerExpiry));
        vStartRapidChannelBrowseTimer(u16TimerExpiry);
    }
}

// Notify Property Update fir RapidChannelBrowseInfo to Clients
tVoid fc_sxm_tclChannelList::vNotifyRapidChannelBrowseInfo(CHANNEL_ID chnID)
{
    _bChannelArtUpdate = TRUE;
    // Notify clients about the changes to ChannelList
    fc_sxm_tcl_trAudioPropertyRapidChannelBrowseInfo trNotify;
    vFillReplayModeElements(chnID, trNotify.oFiMsg.ChannelInfo);
    //tChannelListStatus.oFiRes.Availability = bStatusChanged;
    fc_sxm_tclAudioProperties::instance()->oRapidChannelBrowseInfo.vSet(trNotify);
    fc_sxm_tclAudioProperties::instance()->oRapidChannelBrowseInfo.vNotify();
    ETG_TRACE_USR2(("vTriggerChannelListStatus - vNotifyRapidChannelBrowseInfo for CHANNEL %u", chnID));
}

// Method collects all browseable channels irrespective of their category for Rapid Channel Browsing
tVoid fc_sxm_tclChannelList::vFetchAllChannels(vector<CHANNEL_ID>& vectChnBrowse) const
{
    vectChnBrowse.clear();
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator cIt = _mChannelList.begin();
    while (cIt != _mChannelList.end())
    {
        // Perform first level validation if the channel is browseable or not
        if (cIt->second.bIsBrowseable())
            vectChnBrowse.push_back(cIt->second.u16ChannelId);
        ++cIt;
    }
    ETG_TRACE_USR3(("vFetchAllChannels for RapidChnBrowse has size %u", vectChnBrowse.size()));
}

// Method collects browseable channels from a category for Rapid Channel Browsing
tVoid fc_sxm_tclChannelList::vFetchCategoryChannels(vector<CHANNEL_ID>& vectChnBrowse, CATEGORY_ID tCategoryID) const
{
    Ifc_sxm_tclCategoryList* poCategoryList = (_poAudioApp) ?
                                  _poAudioApp->poGetCategoryListInstance(): NULL;
    SXM_ASSERT_RETURN(NULL != poCategoryList)
    poCategoryList->vFetchCategoryChannels(vectChnBrowse, tCategoryID);
    ETG_TRACE_USR3(("vFetchCategoryChannels for RapidChnBrowse has size %u", vectChnBrowse.size()));
}

// Method handle methodstart requests for "RapidChannelBrowseStop" from Clients
tVoid fc_sxm_tclChannelList::vProcess(fc_sxm_trMsgAudioMStartRapidChannelBrowseStop const *prMsg)
{
    tBool bStatus = FALSE;
    _mRapidChannelBrowseTimer.vStop();
    _trRapidChnBrowse.vReset();

    CHANNEL_ID chnID = static_cast <CHANNEL_ID> (prMsg->oFiMsg.ChannelID);
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator cIt = _mChannelList.find(chnID);
    if ((cIt != _mChannelList.end()) && (cIt->second.bIsBrowseable()))
    {
        SMSAPI_RETURN_CODE_ENUM enRetCode = DECODER.eTuneDirect(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),
                            chnID,
                            TRUE);
        if (SMSAPI_RETURN_CODE_SUCCESS == enRetCode)
        {
            bStatus = TRUE;
        }
    }

    midw_ext_sxm_audiofi_tclMsgRapidChannelBrowseStopMethodResult oMRes;
    oMRes.Status = bStatus;
    ETG_TRACE_USR3(("Rapid Channel Browse Stop issued... Tune to channel %u --> Status(%u)",
            chnID, bStatus));
    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
}


// Method processes Timer expiry event for RapidChannelBrowse
tVoid fc_sxm_tclChannelList::vProcessTimer(fc_sxm_trMsgRapidChannelBrowseTimer const * /* prMsg */)
{
    vStopRapidChannelBrowseTimer();
    vStartRapidChannelBrowse();
}

// Stop Rapid Channel Browsing Timer
tVoid fc_sxm_tclChannelList::vStopRapidChannelBrowseTimer()
{
    _mRapidChannelBrowseTimer.vStop();
}

// Start Rapid Channel Browsing Timer with the given time duration
tVoid fc_sxm_tclChannelList::vStartRapidChannelBrowseTimer(tU16 u16Duration)
{
    if (!_mRapidChannelBrowseTimer.bIsRunning())
    {
        _mRapidChannelBrowseTimer.vStart(u16Duration);
    }
}

// Based on the project parameter, allow or stop tuning to Locked and Skipped Channel
tTunable fc_sxm_tclChannelList::enTuneStatus(const fc_sxm_trChannel& trChannel, tBool bIsDirectTune) const
{
#ifndef SXM_AUDIO_RNAIVI_PROJECT_SPECIFIC
    // Followed for G3G
    tTunable enTunable = (bIsDirectTune) ? trChannel.eIsTunable() :
            trChannel.enBrowseStatus();
#else
    // Followed for RN_AIVI .... Here whether the user has selected Direct Tune or not has no significance
    (tVoid) bIsDirectTune;
    tTunable enTunable = trChannel.enBrowseStatus();
#endif
    return enTunable;
}

// Method decides what action to be taken if current tuned channel is skipped.
// For G3G - Take no action and continue playing the current tuned channel ... Inline with UIRR
// For RN_AIVI - a) Stop playing current tuned channel; b) Fire a system error message; c) Tune to preview channel .... Violates many UIRR points
tVoid fc_sxm_tclChannelList::vHandleTunedChannelSkip()
{
#ifndef SXM_AUDIO_RNAIVI_PROJECT_SPECIFIC
    ETG_TRACE_USR3(("vHandleTunedChannelSkip -- Take no action"));
#else
    _u16RequestedUnavailableChannel = _u16TunedChannelChannelID;
    ETG_TRACE_USR3(("vHandleTunedChannelSkip -- _u16TunedChannelChannelID = %u", _u16TunedChannelChannelID));
    fc_sxm_tclAdvisories* poAdvisories = fc_sxm_tclAdvisories::instance();
    if ((poAdvisories) && (FALSE == poAdvisories->bIsAdvisoryTimerRunning()))
    {
        poAdvisories->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED, FALSE);
        vNotifyChannelInformationUpdate(e_SKIPPED_CHANNEL);
        vRestorePreviewChannel();
    }
#endif
}

// Method decides what action to be taken if current tuned channel which is identified to broadcast MATURE content is locked.
// For G3G - Take no action and continue playing the current tuned channel ... Inline with UIRR
// For RN_AIVI - a) Stop playing current tuned channel; b) Fire a system error message; c) Tune to preview channel .... Violates many UIRR points
tVoid fc_sxm_tclChannelList::vHandleTunedChannelLock()
{
#ifndef SXM_AUDIO_RNAIVI_PROJECT_SPECIFIC
    ETG_TRACE_USR3(("vHandleTunedChannelLock -- Take no action"));
#else
    _u16RequestedUnavailableChannel = _u16TunedChannelChannelID;
    ETG_TRACE_USR3(("vHandleTunedChannelLock -- _u16TunedChannelChannelID = %u", _u16TunedChannelChannelID));
    fc_sxm_tclAdvisories* poAdvisories = fc_sxm_tclAdvisories::instance();
    if ((poAdvisories) && (FALSE == poAdvisories->bIsAdvisoryTimerRunning()))
    {
        poAdvisories->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED, FALSE);
        vNotifyChannelInformationUpdate(e_MATURE_LOCKED_CHANNEL);
        vRestorePreviewChannel();
    }
#endif
}

// Lock a mature channel and return TRUE if lock operation is success
tBool fc_sxm_tclChannelList::bLockChannel(fc_sxm_trChannel& trChannel)
{
    tBool bRequestStatus = DECODER.bLockChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),
                    trChannel.u16ChannelId);
    ETG_TRACE_USR1(("Channel # %u is now broadcasting Mature Content... so Lock it as EnableMatureChannels(%u) \t Request(%u)",
                    trChannel.u16ChannelId, _bEnableMatureChannels, bRequestStatus));
    if (bRequestStatus)
    {
        trChannel.vLockChannel(TRUE);
    }
    return bRequestStatus;
}

// Unlock a mature channel and return TRUE if unlock operation is success
tBool fc_sxm_tclChannelList::bUnLockChannel(fc_sxm_trChannel& trChannel)
{
    tBool bRequestStatus = DECODER.bUnlockChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),
                    trChannel.u16ChannelId);
    ETG_TRACE_USR1(("Channel # %u is NOT broadcasting Mature Content... so UnLock it as EnableMatureChannels(%u) \t Request(%u)",
                    trChannel.u16ChannelId, _bEnableMatureChannels, bRequestStatus));
    if (bRequestStatus)
    {
        trChannel.vLockChannel(FALSE);
    }
    return bRequestStatus;
}

// Upon receiving "DECODER_OBJECT_EVENT_SUBSCRIPTION_UPDATED" event, retune to the current tuned channel
tVoid fc_sxm_tclChannelList::vRetuneCurrentChannel()
{
    DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
    tBool bIsChnSubscribed;
    SMSAPI_RETURN_CODE_ENUM enRetCode = DECODER.eIsChannelSubscribed ( hDecoder, _u16TunedChannelChannelID, &bIsChnSubscribed );
    if ((( SMSAPI_RETURN_CODE_SUCCESS == enRetCode) && (TRUE == bIsChnSubscribed)))
    {
        enRetCode = DECODER.eTuneDirect( hDecoder, _u16TunedChannelChannelID, FALSE);
    }
    ETG_TRACE_USR4(("vRetuneCurrentChannel _u16TunedChannelChannelID(%u) \t bIsChnSubscribed(%u) \t enRetCode(%u)",
            _u16TunedChannelChannelID, bIsChnSubscribed, enRetCode));
}

// Mute SXM Audio source
tVoid fc_sxm_tclChannelList::vSetMuteState (tBool bState)
{
    if ((_poAudioApp) && (bState != bGetMuteStatus()))
        _poAudioApp->vMuteSXMAudio(bState);
}

// Get the current mute status of SXM Audio
tBool fc_sxm_tclChannelList::bGetMuteStatus() const
{
    tBool bMuteState = FALSE;
    if (_poAudioApp)
        bMuteState = _poAudioApp->bGetSXMAudioMuteState();
    return bMuteState;
}

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
