/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_category_list.cpp
* @brief       Category List Handler implementation.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_common.h"
#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_category_list.h"
#include "fc_sxm_tcl_advisories.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_tcl_states.h"
#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_channel_list_if.h"
#include "fc_sxm_tcl_presets_if.h"
#include "fc_sxm_tcl_content_alert_proxy_if.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_CATEGORY_LIST
#include "trcGenProj/Header/fc_sxm_tcl_category_list.cpp.trc.h"
#endif

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*****************************************************************************
 * FUNCTION    : cb_s16CategoryIterateHandler                                *
 * CLASS       : None                                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Channel iterator event callback                             *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategory       - Valid category object                     *
 * PARAMETER   : s16CurrentIndex - Channel index in category iterated        *
 * PARAMETER   : hChannel        - Valid channel object                      *
 * PARAMETER   : pvIterateArg    - Data from calling function                *
 * RETURNVALUE : N16             - Steps to iterate further                  *
 *****************************************************************************/
static N16 cb_s16CategoryIterateHandler(
        CATEGORY_OBJECT hCategory,
        CATEGORY_CHANNEL_INDEX s16CurrentIndex,
        CHANNEL_OBJECT hChannel,
        tVoid* /* pvIterateArg */ )
{
    ETG_TRACE_USR4(("cb_s16CategoryIterateHandler(), CategoryId - %d", 
            CATEGORY.tGetCategoryId(hCategory)));
    // To handle further list elements
    // Process channel object
    N16 hReturnData = fc_sxm_tclCategoryList::instance()->s16updateChannelInfo(
            hCategory,
            s16CurrentIndex,
            hChannel);

    return (hReturnData);
}

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*****************************************************************************
 * FUNCTION    : fc_sxm_tclCategoryList                                      *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class constructor                                           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclCategoryList::fc_sxm_tclCategoryList()
: _bCategoryNotificationsReg(FALSE)
, _bUpdateCategoryList(FALSE)
, _bListUpdate(FALSE)
, _u16CurrentSelectedCategoryID(0)
, _u16MySeekCatId(0)
, _enCurrentDecoderState(DECODER_STATE_INVALID)
, _poAudioApp(OSAL_NULL)
,_u32FeaturedFavCount(0)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList constructor"));
    _mCategoryUserIdList.clear();
#ifdef FC_SXM_ENABLE_FEATURED_FAVORITE_SORTING
    _mFeaturedFavCategoryList.clear();
#endif
}

/*****************************************************************************
 * FUNCTION    : ~fc_sxm_tclCategoryList                                     *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class destructor                                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclCategoryList::~fc_sxm_tclCategoryList()
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList destructor"));

    /* Set all the class member variables to default values */
    _bCategoryNotificationsReg = FALSE;
    _bUpdateCategoryList = FALSE;
    _u16CurrentSelectedCategoryID = 0;
    _u16MySeekCatId = 0;
    _enCurrentDecoderState = DECODER_STATE_INVALID;
    _poAudioApp = OSAL_NULL;
    _u32FeaturedFavCount = 0;
    _mCategoryUserIdList.clear();
#ifdef FC_SXM_ENABLE_FEATURED_FAVORITE_SORTING
    _mFeaturedFavCategoryList.clear();
#endif
}

/*****************************************************************************
 * FUNCTION    : vUpdateDecoderState                                         *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to start category list creation and initiate         *
 *               category updates from SMS once decoder is in ready state    *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hDecoderObject - Valid decoder object from SMS service      *
 * PARAMETER   : enDecoderState - Current decoder state                      *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclCategoryList::vUpdateDecoderState(
        DECODER_OBJECT hDecoderObject,
        DECODER_STATE_ENUM enDecoderState)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vUpdateDecoderState()"));

    /* Store current decoder state */
    _enCurrentDecoderState = enDecoderState;

    /* Register for category event notification for the first time */
    if( (DECODER_STATE_READY == enDecoderState) &&
            (FALSE == _bCategoryNotificationsReg)  && (DECODER_INVALID_OBJECT != hDecoderObject))
    {
        ETG_TRACE_USR3(("Registring for category notifications"));

        _bCategoryNotificationsReg = TRUE;

        /* Update category list availability property update */
        vNotifyCategoryListUpdate();
    }
}

/* Clear class member variables */
tVoid fc_sxm_tclCategoryList::vClearMemberData(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vClearMemberData()"));

    /* Clear member variables */
    _bCategoryNotificationsReg = FALSE;
    _bUpdateCategoryList = FALSE;
    _u16CurrentSelectedCategoryID = 0;
    _enCurrentDecoderState = DECODER_STATE_INVALID;

    /* Clear category list */
    _mCategoryList.clear();

    _mCategoryUserIdList.clear();
    _u32FeaturedFavCount = 0;
#ifdef FC_SXM_ENABLE_FEATURED_FAVORITE_SORTING
    _mFeaturedFavCategoryList.clear();
#endif

    /* Update category list availability property update */
    vNotifyCategoryListUpdate(FALSE);
}

/*****************************************************************************
 * FUNCTION    : vUpdateCategoryInfo                                         *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method Update category information in internal list         *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategoryObject - Valid category object from SMS service    *
 * PARAMETER   : hEventMask      - Update info of category object            *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclCategoryList::vUpdateCategoryInfo(
        fc_sxm_trCategory const &rCategory,
        CATEGORY_EVENT_MASK hEventMask)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vUpdateCategoryInfo(catId=%u)", rCategory._u16CategoryId));

    /* Category removed */
    if(CATEGORY_OBJECT_EVENT_REMOVED & hEventMask)
    {
        ETG_TRACE_USR2(("Category removed"));
        vRemoveExistingCategory(rCategory);

        //Updating userid list also
        vRemoveExistingUserId(rCategory._u16CategoryId);

#ifdef FC_SXM_ENABLE_FEATURED_FAVORITE_SORTING
        vRemoveFeaturedFavorite(rCategory._u16CategoryId);
#endif
    } else {
        ETG_TRACE_USR2(("Category param modified"));
        vModifyCategoryParams(rCategory, hEventMask);
    }

    /* Update category list availability property update */
    vNotifyCategoryListUpdate();
}

/*****************************************************************************
 * FUNCTION    : vAddNewCategory                                             *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Add new category to internal category list                  *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategoryObject - Valid category object from SMS service    *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclCategoryList::vAddNewCategory(
        fc_sxm_trCategory const &rCategory)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vAddNewCategory()"));

    CATEGORY_ID nCatId = rCategory._u16CategoryId;
    if(CATEGORY_INVALID_ID == nCatId)
    {
        ETG_TRACE_ERR(("Failed to get category id from category object"));
    }
    else if(FALSE == bIsValidRangeCategoryId(nCatId))
    {
        ETG_TRACE_ERR(("Category not in range of broadcast and user/virtual category type"));
    }
    else 
    {
        /* Push the data to category list */
        _mCategoryList.insert(pair<CATEGORY_ID, fc_sxm_trCategory>(nCatId, rCategory));

        //Add userid into the list only if it is a new category
        USER_ID nUserId = CATEGORY.tGetUserId(( (_poAudioApp)? _poAudioApp->oGetDecoderObject() :
                DECODER_INVALID_OBJECT), nCatId);

        if(0 == _mCategoryUserIdList.count(nUserId))
        {
            nUserId = CATEGORY_INVALID_ID;
            if(rCategory._enCatType == CATEGORY_TYPE_BROADCAST)
            {
                nUserId = u16ConvertBroadcastCategoryIdtoUserId(nCatId);
                _mCategoryUserIdList.insert(pair<USER_ID, CATEGORY_ID>(nUserId, nCatId));
            }
            else if(rCategory._enCatType == CATEGORY_TYPE_USER)
            {
                nUserId = u16ConvertUserCategoryIdtoUserId(nCatId);
                _mCategoryUserIdList.insert(pair<USER_ID, CATEGORY_ID>(nUserId, nCatId));

#ifdef FC_SXM_ENABLE_FEATURED_FAVORITE_SORTING
                vAddFeaturedFavoriteToSort(rCategory._oCategoryName, rCategory._u16CategoryId);
#endif
            }
        }
    }
}

/*****************************************************************************
 * FUNCTION    : vRemoveExistingCategory                                     *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Remove existing category from internal category list        *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategoryObject - Valid category object from SMS service    *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclCategoryList::vRemoveExistingCategory(
        fc_sxm_trCategory const &rCategory)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vRemoveExistingCategory(catId=%u)", rCategory._u16CategoryId));

    /* Get category ID from category object */
    CATEGORY_ID nCatId = rCategory._u16CategoryId;
    if(CATEGORY_INVALID_ID == nCatId)
    {
        ETG_TRACE_ERR(("Failed to get category ID from category object"));
    }
    else
    {
        /* Find for the category ID in the list */
        map<CATEGORY_ID, fc_sxm_trCategory>::iterator mIter;
        mIter = _mCategoryList.find(nCatId);
        if (mIter != _mCategoryList.end())
        {
            /* If found delete the entry from the list */
            _mCategoryList.erase(mIter);
        }

    }
}

/*****************************************************************************
 * FUNCTION    : vRemoveExistingUserId                                       *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Remove existing userid from internal category list          *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : CATEGORY_ID - Valid category ID                             *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclCategoryList::vRemoveExistingUserId(CATEGORY_ID nCatId)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vRemoveExistingUserId(catId=%u)", nCatId));
    if(CATEGORY_INVALID_ID == nCatId)
    {
        ETG_TRACE_ERR(("Invalid Category ID"));
    }
    else
    {
        USER_ID nUserId = CATEGORY_INVALID_ID;

        /* Find for the category ID in the Category UserID list */
        map<USER_ID, CATEGORY_ID>::iterator categoryuseridIterator;
        for(categoryuseridIterator=_mCategoryUserIdList.begin();
                categoryuseridIterator!=_mCategoryUserIdList.end();
                ++categoryuseridIterator)
        {
            if(categoryuseridIterator->second == nCatId)
            {
                nUserId = categoryuseridIterator->first;
                break;
            }
        }
        DECODER_OBJECT hDecoder = ( (_poAudioApp)? _poAudioApp->oGetDecoderObject() :
                DECODER_INVALID_OBJECT);
        if(hDecoder != DECODER_INVALID_OBJECT) {

            for(tU16 u16Index=nUserId; u16Index<_u32FeaturedFavCount; u16Index++)
            {
                _mCategoryUserIdList[u16Index] = _mCategoryUserIdList[(tU16)(u16Index + 1)];
                CATEGORY.eSetUserId(hDecoder, _mCategoryUserIdList[(tU16)(u16Index + 1)], u16Index);
            }

            map<USER_ID, CATEGORY_ID>::iterator mIter;
            mIter = _mCategoryUserIdList.lower_bound((tU16)_u32FeaturedFavCount);
            if(mIter != _mCategoryUserIdList.end())
            {
                _mCategoryUserIdList.erase(mIter);
            }

            /* Updating the sequence for featured favorite after deletion */
            _u32FeaturedFavCount -= 1;
        }
        else {
            ETG_TRACE_ERR(("Invalid Decoder"));
        }
    }
}

/*****************************************************************************
 * FUNCTION    : vModifyCategoryParams                                       *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Modify category params from category list                   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategoryObject  - Valid category object from SMS service   *
 * PARAMETER   : hCategoryEvtMask - Category update event mask               *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclCategoryList::vModifyCategoryParams(
        fc_sxm_trCategory const &roCatData,
        CATEGORY_EVENT_MASK hCategoryEvtMask)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vModifyCategoryParams()"));

    /* Get category ID from category object */
    CATEGORY_ID nCatId = roCatData._u16CategoryId;
    if(CATEGORY_INVALID_ID == nCatId)
    {
        ETG_TRACE_ERR(("Failed to get category ID from category object"));
    }
    else
    {
        /* Find for the category ID in the list */
        map<CATEGORY_ID, fc_sxm_trCategory>::iterator mIter;
        mIter = _mCategoryList.find(nCatId);
        if( (mIter != _mCategoryList.end()) && (hCategoryEvtMask & CATEGORY_OBJECT_EVENT_NAME) ) {
            /* If found update the list element params */
            /* Update category name */
            ((*mIter).second)._oCategoryName = roCatData._oCategoryName;
            ((*mIter).second)._enCatType = roCatData._enCatType;
        } else {
            vAddNewCategory(roCatData);
        }
    }
}


/* Notify category list update */
tVoid fc_sxm_tclCategoryList::vNotifyCategoryListUpdate(tBool bIsDecoderStateRequired)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vNotifyCategoryListUpdate()"));

    if((DECODER_STATE_READY == _enCurrentDecoderState) || (FALSE == bIsDecoderStateRequired)) {
        tBool bListAvailable = FALSE;
        map<CATEGORY_ID, fc_sxm_trCategory>::iterator categoryIterator;
        DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
        for(categoryIterator = _mCategoryList.begin();
                categoryIterator != _mCategoryList.end(); ++categoryIterator) {
            if(0 != CATEGORY.tSize(hDecoder, (*categoryIterator).first)) {
                bListAvailable = TRUE;
                break;
            }
        }

        /* Set the status info and notify all clients */
        fc_sxm_tcl_trAudioPropertyCategoryListStatus tCategoryListStatus;
        tCategoryListStatus.oFiRes.Availability = bListAvailable;
        fc_sxm_tclAudioProperties::instance()->oCategoryListStatus.vSet(tCategoryListStatus);
        fc_sxm_tclAudioProperties::instance()->oCategoryListStatus.vNotify();

        if(bListAvailable) {
            ETG_TRACE_USR2(("vNotifyCategoryListUpdate - TRUE"));
        } else {
            ETG_TRACE_USR2(("vNotifyCategoryListUpdate - FALSE"));
        }
    }
}

/* Update channel list for each category */
tVoid fc_sxm_tclCategoryList::vUpdateCategoryChannels(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vUpdateCategoryChannels()"));

    CATEGORY_ID u16CategoryArray[USER_CATEGORY_ID_MAX] = { 0 };
    tU16 u16ListCount = (tU16)(_mCategoryList.size());
    if(0 != u16ListCount) {
        map<CATEGORY_ID, fc_sxm_trCategory>::iterator categoryIterator;
        tU16 u16Count = 0;
        for(categoryIterator = _mCategoryList.begin();
                categoryIterator != _mCategoryList.end(); ++categoryIterator, u16Count++) {
            u16CategoryArray[u16Count] = ((*categoryIterator).second)._u16CategoryId;
            ((*categoryIterator).second).mCategoryChannelList.clear();
        }
    }

    if(0 != u16ListCount) {
        tU16 u16Count = 0;
        DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
        for(u16Count=0; u16Count<u16ListCount; u16Count++) {
            SMSAPI_RETURN_CODE_ENUM hReturnCode =
                    CATEGORY.eIterate(hDecoder,
                            u16CategoryArray[u16Count], cb_s16CategoryIterateHandler, NULL);
            if(SMSAPI_RETURN_CODE_SUCCESS != hReturnCode) {
                ETG_TRACE_ERR(("Failed to register for channel callback for category %d, ERROR_CODE - %d",
                        u16CategoryArray[u16Count], ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, hReturnCode) ));
            }
        }
    }
}

/*****************************************************************************
 * FUNCTION    : s16updateChannelInfo                                        *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Update channel info in category list                        *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategory     - Valid category object                       *
 * PARAMETER   : tCurrentIndex - Channel index being iterated                *
 * PARAMETER   : hChannel      - Valid channel object                        *
 * RETURNVALUE : N16           - Steps to iterate further                    *
 *****************************************************************************/
N16 fc_sxm_tclCategoryList::s16updateChannelInfo(
        CATEGORY_OBJECT hCategory,
        CATEGORY_CHANNEL_INDEX tCurrentIndex,
        CHANNEL_OBJECT hChannel)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::s16updateChannelInfo()"));

    /* Channel count to fetch further list elements */
    size_t u32ChannelCount = 0;

    /* Get category ID from category object */
    CATEGORY_ID hCategoryID = CATEGORY.tGetCategoryId(hCategory);
    if(CATEGORY_INVALID_ID == hCategoryID)
    {
        ETG_TRACE_ERR(("Invalid category received, ID - %d", hCategoryID));
    }
    else
    {
        /* Category list iterator */
        map<CATEGORY_ID, fc_sxm_trCategory>::iterator categoryIterator;
        categoryIterator = _mCategoryList.find(hCategoryID);
        if(categoryIterator == _mCategoryList.end()) {
            ETG_TRACE_ERR(("Failed to find current category type in category list"));
        } else {
            /* Get Service ID and Channel ID from the channel object that are required to insert a new entry into channel list */
            SERVICE_ID hServiceID = CHANNEL.tServiceId(hChannel);
            CHANNEL_ID hChannelID = CHANNEL.tChannelId(hChannel);

            if( (SERVICE_INVALID_ID == hServiceID) ||
                    (CHANNEL_INVALID_ID == hChannelID) )   {
                ETG_TRACE_ERR(("Invalid ID's received, hServiceID - %d and hChannelID - %d", hServiceID, hChannelID));
            } else {
                // Check if the entry for the above service ID already exist
                // If data to above mentioned service ID doen not exist
                //   then insert channel details into channel list of current
                //   category type
                if(0 == categoryIterator->second.mCategoryChannelList.count(hChannelID))
                {
                    if(bChannelBrowseStatus(hChannelID)) {
                        fc_sxm_trChannel rChn(hChannel);
                        /* Push the data to channel list */
                        categoryIterator->second.mCategoryChannelList.insert(pair<CHANNEL_ID, fc_sxm_trChannel>(hChannelID, rChn));
                    }
                }
            }
        }

        /* Get total number of channels available in the category */
        u32ChannelCount = CATEGORY.tSize(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),hCategoryID);
    }

    /* If data recived is not the last element, then request for 
       next element data */
    N16 n16HandlerReturnValue = 0;
    if(u32ChannelCount != (tU32)(tCurrentIndex+1))
    {
        n16HandlerReturnValue = 1;
    }

    return n16HandlerReturnValue;
}

/*****************************************************************************
 * FUNCTION    : u16GetCidFromCategoryListIndex                              *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get category ID from category list index based on           *
 *               category type                                               *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategoryType - Category type to query                      *
 * PARAMETER   : u16ListIndex  - Index in category list                      *
 * RETURNVALUE : CHANNEL_ID    - Channel ID of list element queried          *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclCategoryList::u16GetCidFromCategoryListIndex(
        tU16 hCategoryType,
        tU16 u16ListIndex)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::u16GetCidFromCategoryListIndex()"));

    CHANNEL_ID hChannelID = CHANNEL_INVALID_ID;

    /* Update category channels */
    vUpdateCategoryChannels();

    /* For requested category type get channel list */
    map<CATEGORY_ID, fc_sxm_trCategory>::const_iterator categoryIterator;
    categoryIterator = _mCategoryList.find(hCategoryType);
    if(categoryIterator != _mCategoryList.end()) {
        /* Validate channel index */
        if (u16ListIndex <= categoryIterator->second.mCategoryChannelList.size()) {
            map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator channelIterator;
            tU16 loopCount = 0;

            /* Fetch channel ID for requested channel list index */
            for(channelIterator = categoryIterator->second.mCategoryChannelList.begin();
                    channelIterator != categoryIterator->second.mCategoryChannelList.end();
                    ++channelIterator, loopCount++) {
                if (channelIterator->second.bIsBrowseable()) {
                    /* Get channel ID*/
                    if(loopCount == u16ListIndex) {
                        hChannelID = ((*channelIterator).second).u16ChannelId;
                        break;
                    }
                } else {
                    loopCount--;
                }
            }
        } else {
            ETG_TRACE_ERR(("Invalid channel index received"));
        }
    } else {
        ETG_TRACE_ERR(("Failed to find specified category from category list"));

        /* Set channel unavailable system message  */
        (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(
                midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND, FALSE);
    }

    return (hChannelID);
}

/*****************************************************************************
 * FUNCTION    : u16GetCidFromCategoryListRelativeIndex                      *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID from category list index based on relative   *
 *               index. This should also wrap to next list element if the    *
 *               index received is the last element in the list              *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategoryType      - Category type to query                 *
 * PARAMETER   : hChannelID         - Channel ID to query                    *
 * PARAMETER   : lRelativeListIndex - Relative index in category list        *
 * RETURNVALUE : CHANNEL_ID         - Channel ID of list element queried     *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclCategoryList::u16GetCidFromCategoryListRelativeIndex(
        CATEGORY_ID hCategoryType,
        CHANNEL_ID hChannelID,
        tS16 lRelativeListIndex)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::u16GetCidFromCategoryListRelativeIndex()"));

    CHANNEL_ID lChannelID = CHANNEL_INVALID_ID;

    /* Update category channels */
    vUpdateCategoryChannels();

    /* For requested category type get channel list */
    map<CATEGORY_ID, fc_sxm_trCategory>::iterator categoryIterator = _mCategoryList.begin();
    for(;;)
    {
        categoryIterator++;
        if(categoryIterator == _mCategoryList.end()) {
            categoryIterator = _mCategoryList.begin();
        }
        if(FALSE == (((*categoryIterator).second).mCategoryChannelList).empty()) {
            break;
        }
    }
    map<CHANNEL_ID, fc_sxm_trChannel>::iterator channelIterator = (((*categoryIterator).second).mCategoryChannelList).begin();

    categoryIterator = _mCategoryList.find(hCategoryType); /* Find current category */
    if(categoryIterator != _mCategoryList.end()) {
        channelIterator = (((*categoryIterator).second).mCategoryChannelList).find(hChannelID);  /* Find current channel */
    }

    for(tS16 s16RelativeListIndex = lRelativeListIndex; lRelativeListIndex != 0; )
    {
        if(0 < lRelativeListIndex) {
            s16RelativeListIndex = FC_SXM_AUDIO_CATEGOEY_CHANNEL_UP;
        } else if (0 > lRelativeListIndex){
            s16RelativeListIndex = FC_SXM_AUDIO_CATEGOEY_CHANNEL_DOWN;
        }

        tBool bChannelFound = FALSE;
        for(;(categoryIterator != _mCategoryList.end()) && (bChannelFound != TRUE);)
        {  /* In current category search for next valid channel*/
            for(;(channelIterator != (((*categoryIterator).second).mCategoryChannelList).end()) &&
            (bChannelFound != TRUE) && (((*categoryIterator).second).bIsCategoryAvailable());)
            {  /* If next available channel is tunable and not the current channel then break out all all loops */
                if( (channelIterator != (((*categoryIterator).second).mCategoryChannelList).end()) && /* End has no data */
                        (((*channelIterator).second).bIsBrowseable()) && /* Tunable channel */
                        (((*channelIterator).second).u16ChannelId != hChannelID) ) /* Must not be current tuned channel */
                {  /* Set flag and get channel id */
                    bChannelFound = TRUE;
                    lChannelID = ((*channelIterator).second).u16ChannelId;
                } else if( /* If next available channel is tunable but the current channel and only one channel and one category is available then break out of all loops */
                        (channelIterator != (((*categoryIterator).second).mCategoryChannelList).end()) && /* End has no data */
                        (((*channelIterator).second).bIsBrowseable()) && /* Tunable channel */
                        (1 == (((*categoryIterator).second).mCategoryChannelList).size()) &&
                        (1 == _mCategoryList.size()) ) /* only one category and one channel is available in the list  */
                {  /* Set flag and get channel id */
                    bChannelFound = TRUE;
                    lChannelID = ((*channelIterator).second).u16ChannelId;
                } else {
                    switch(s16RelativeListIndex)
                    {  /* Based on browse direction move to next channel */
                    case FC_SXM_AUDIO_CATEGOEY_CHANNEL_UP: {
                        channelIterator++;
                    } break;
                    case FC_SXM_AUDIO_CATEGOEY_CHANNEL_DOWN: {
                        if(channelIterator != (((*categoryIterator).second).mCategoryChannelList).begin()) {
                            channelIterator--;
                        } else { /* If channel begin is reached then set pointer to end to break out of for loop */
                            channelIterator = (((*categoryIterator).second).mCategoryChannelList).end();
                        }
                    } break;
                    default:
                        break;
                    }/* Make sure the iterator points to valid memory */
                }
            }

            if((0 < lRelativeListIndex) && (!bChannelFound)) {
                for(;;)
                {
                    categoryIterator++;
                    if(categoryIterator == _mCategoryList.end()) {
                        categoryIterator = _mCategoryList.begin();
                    }
                    if(FALSE == (((*categoryIterator).second).mCategoryChannelList).empty()) {
                        break;
                    }
                }
                channelIterator = (((*categoryIterator).second).mCategoryChannelList).begin();
            } else if ((0 > lRelativeListIndex) && (!bChannelFound)) {
                for(;;)
                {
                    if(categoryIterator == _mCategoryList.begin()) {
                        categoryIterator = _mCategoryList.end();
                    }
                    categoryIterator--;
                    if(FALSE == (((*categoryIterator).second).mCategoryChannelList).empty()) {
                        break;
                    }
                }
                channelIterator = (((*categoryIterator).second).mCategoryChannelList).end();
                channelIterator--; /* Point to last channel in current category */
            }
        }

        if(0 < lRelativeListIndex) {
            lRelativeListIndex--;
        } else if (0 > lRelativeListIndex){
            lRelativeListIndex++;
        }
    } /* End of steps */

    return lChannelID;
}

/*****************************************************************************
 * FUNCTION    : u16GetCidOfNextProgramFromCategoryList                      *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get channel ID from category list based on current program  *
 *               type                                                        *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : u16CategoryType  - Category type to query                    *
 * PARAMETER   : s16RelativeIndex - Relative index in category list          *
 * RETURNVALUE : CHANNEL_ID     - Channel ID of list element queried         *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclCategoryList::u16GetCidOfNextProgramFromCategoryList(
        tU16 u16CategoryType,
        tS16 s16RelativeIndex)
{
    ETG_TRACE_USR4((
            "fc_sxm_tclCategoryList::u16GetCidOfNextProgramFromCategoryList()"));

    CHANNEL_ID hChannelID = CHANNEL_INVALID_ID;

    /* Update category channels */
    vUpdateCategoryChannels();

    /* For requested category type get channel list */
    map<CATEGORY_ID, fc_sxm_trCategory>::const_iterator categoryIterator = _mCategoryList.begin();
    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator channelIterator = ((*categoryIterator).second).mCategoryChannelList.begin();

    /* Look for required category */
    categoryIterator = _mCategoryList.find(u16CategoryType);

    /* In that find for requested channel type */
    if(categoryIterator == _mCategoryList.end())
    {
        /* This is required when user tunes to channel 0 and presses 
           program up or program down */
        ETG_TRACE_ERR(("Failed to find specified category from category "
                "list hence tuning to first channel of first "
                "category in category list"));

        if(0 != _mCategoryList.size())
        {
            /* Pointing to first category */
            categoryIterator = _mCategoryList.begin();
            if(0 != 
                    (((*categoryIterator).second).mCategoryChannelList).size())
            {
                /* Point to first channel in the list */
                channelIterator = 
                        (((*categoryIterator).second).
                                mCategoryChannelList).begin();
                hChannelID = 
                        ((*channelIterator).second).u16ChannelId;
            }
            else
            {
                ETG_TRACE_ERR((
                        "No valid channels available in category list"));
            }
        }
        else
        {
            ETG_TRACE_ERR(("Category list is empty"));
        }
    }
    else
    {
        for(;;) {
            tBool bElementFound = FALSE;
            switch(s16RelativeIndex) {
            case FC_SXM_AUDIO_TUNE_PROGRAM_UP:  {
                /* Increment category pointer to next category */
                categoryIterator++;
                /* If last category is reached then point to first category in the list */
                if(categoryIterator == _mCategoryList.end()) {
                    categoryIterator = _mCategoryList.begin();
                }
            } break;
            case FC_SXM_AUDIO_TUNE_PROGRAM_DOWN: {
                /* Check if category pointer is point to first category */
                if(categoryIterator == _mCategoryList.begin()) {
                    /* If yes then point to list end  */
                    categoryIterator = _mCategoryList.end();
                }
                /* Decrement category pointer */
                categoryIterator--;
            } break;
            default: {
                ETG_TRACE_ERR(("This part of code should never be reached"));
            } break;
            }

            for(channelIterator = (((*categoryIterator).second).mCategoryChannelList).begin();
                    channelIterator != (((*categoryIterator).second).mCategoryChannelList).end();
                    ++channelIterator) {
                if(((*channelIterator).second).bIsBrowseable()) {
                    bElementFound = TRUE;
                    break;
                }
            }

            if(bElementFound)
                break;
        }
        hChannelID = ((*channelIterator).second).u16ChannelId;
    }

    return (hChannelID);
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartGetCatList)                 *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to get category list                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Get category list method start request              *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclCategoryList::vProcess(
        fc_sxm_trMsgAudioMStartGetCatList const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vProcess"
            "(fc_sxm_trMsgAudioMStartGetCatList)"));

    midw_ext_sxm_audiofi_tclMsgGetCategoryListMethodResult ofiTxObj;

    /* Set default method result as rejected */
    ((ofiTxObj).Status).enType =
            midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_REQUEST_REJECTED;

    /* Set category type in method result */
    (ofiTxObj).CategoryType = (prMsg->oFiMsg).CategoryType;

    /* Update category channels */
    vUpdateCategoryChannels();

    if(  (FC_SXM_AUDIO_CATEGORY_TYPE_ALL == (prMsg->oFiMsg).CategoryType) ||
            (bIsValidRangeCategoryId((prMsg->oFiMsg).CategoryType)  ) ||
            (FC_SXM_AUDIO_CATEGORY_TYPE_CURRENT == (prMsg->oFiMsg).CategoryType)
    )
    {
        ETG_TRACE_USR4(("Category type received is valid type"));
        ETG_TRACE_USR4(("Category Id: %d", (prMsg->oFiMsg).CategoryType));

        if(  (  (FC_SXM_AUDIO_CATEGORY_TYPE_CURRENT == 
                (prMsg->oFiMsg).CategoryType) &&
                (FC_SXM_AUDIO_CATEGORY_TYPE_ALL == 
                        _u16CurrentSelectedCategoryID)
        ) ||
        (FC_SXM_AUDIO_CATEGORY_TYPE_ALL ==
                (prMsg->oFiMsg).CategoryType
        )
        )
        {
            /* Set method result status as success */
            ((ofiTxObj).Status).enType =
                    midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;

            if(TRUE == ((ofiTxObj).CategoryList).empty()) {
                Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                        _poAudioApp->poGetChannelListInstance(): NULL;
                SXM_ASSERT_RETURN(NULL != poChannelList)

                /* Create All Channels category here */
                /* Copy of category list entry */
                midw_ext_fi_tcl_CategoryListEntry lCategoryListEntry;
                /* Copy category ID */
                lCategoryListEntry.CategoryType = 0;
                /* Copy category name availability */
                lCategoryListEntry.CategoryNameAvail = TRUE;
                /* Copy category name */
                string rString = "All Channels";
                fc_sxm_vString2Fi(rString.c_str(), lCategoryListEntry.CategoryName);
                poChannelList->vUpdateAllChannels(lCategoryListEntry.ChannelList);
                /* Insert new category list element into method result */
                ((ofiTxObj).CategoryList).push_back(lCategoryListEntry);
            } 

            /* Get channel list for all category types */
            map<USER_ID, CATEGORY_ID>::iterator categoryuseridIterator;
            for(categoryuseridIterator=_mCategoryUserIdList.begin();
                    categoryuseridIterator!=_mCategoryUserIdList.end();
                    ++categoryuseridIterator)
            {
                map<CATEGORY_ID, fc_sxm_trCategory>::iterator categoryIterator
                = _mCategoryList.find((*categoryuseridIterator).second);
                if(categoryIterator != _mCategoryList.end())
                {
                    /* Get channel list for all category types */
                    if(FALSE == (((*categoryIterator).second).mCategoryChannelList).empty())
                    {
                        if(((*categoryIterator).second).bIsCategoryAvailable())
                        {
                            /* Copy of category list entry */
                            midw_ext_fi_tcl_CategoryListEntry lCategoryListEntry;

                            /* Copy category ID */
                            lCategoryListEntry.CategoryType =
                                    ((*categoryIterator).second)._u16CategoryId;

                            /* Copy category name availability */
                            lCategoryListEntry.CategoryNameAvail =
                                    ((*categoryIterator).second).
                                    bIsCategoryNameAvailable();
                            /* Copy category name */
                            if(((*categoryIterator).second)._enCatType == CATEGORY_TYPE_USER)
                            {
                                string rString = "Featured Favorites - " + ((*categoryIterator).second)._oCategoryName;
                                ETG_TRACE_USR4(("Category - %s",rString.c_str()));
                                fc_sxm_vString2Fi(rString.c_str(), lCategoryListEntry.CategoryName);
                            }
                            else
                            {
                                fc_sxm_vString2Fi(
                                        (((*categoryIterator).second)._oCategoryName).c_str(),
                                        lCategoryListEntry.CategoryName);
                            }
                            /* Copy channel list data */
                            map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator channelIterator;
                            for(channelIterator =
                                    (((*categoryIterator).second).
                                            mCategoryChannelList).begin();
                                    channelIterator !=
                                            (((*categoryIterator).second).
                                                    mCategoryChannelList).end();
                                    ++channelIterator)
                            {
                                const fc_sxm_trChannel& rChannel((*channelIterator).second);
                                // Add only those channels that are browseable
                                if( rChannel.bIsBrowseable() )
                                {
                                    midw_ext_fi_tcl_ChannelListEntry lChannelList;
                                    vCopyChannelEntry(lChannelList, rChannel);
                                    /* Insert new rChannel list element into channel list */
                                    lCategoryListEntry.ChannelList.ChannelList.push_back(lChannelList);
                                }
                            }
                            /* Insert new category list element into method result */
                            ((ofiTxObj).CategoryList).push_back(lCategoryListEntry);
                        }
                    }
                    else
                    {
                        ETG_TRACE_ERR(("Category %d has no channel elements, "
                                "hence not added in category list",
                                ((*categoryIterator).second).
                                _u16CategoryId));
                    }
                }
                else
                {
                    ETG_TRACE_ERR(("Failed to find category details for "
                            "specified category USER ID"));
                }
            }

            /* Save current category type */
            _u16CurrentSelectedCategoryID = FC_SXM_AUDIO_CATEGORY_TYPE_ALL;
        }
        else
        {
            /* Get requested category type */
            CATEGORY_ID lRequestedCategory = (FC_SXM_AUDIO_CATEGORY_TYPE_CURRENT == (prMsg->oFiMsg).CategoryType) ?
                    _u16CurrentSelectedCategoryID : (prMsg->oFiMsg).CategoryType;
            /* For requested category type get channel list */
            map<CATEGORY_ID, fc_sxm_trCategory>::iterator categoryIterator;
            categoryIterator = _mCategoryList.find(lRequestedCategory);
            if(categoryIterator != _mCategoryList.end())
            {
                if(((*categoryIterator).second).bIsCategoryAvailable())
                {
                    /* Set method result status as success */
                    ((ofiTxObj).Status).enType =
                            midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
                    /* Copy of category list entry */
                    midw_ext_fi_tcl_CategoryListEntry lCategoryListEntry;

                    /* Copy category ID */
                    lCategoryListEntry.CategoryType =
                            ((*categoryIterator).second)._u16CategoryId;
                    /* Copy category name availability */
                    lCategoryListEntry.CategoryNameAvail =
                            ((*categoryIterator).second).
                            bIsCategoryNameAvailable();
                    /* Copy category name */
                    fc_sxm_vString2Fi(
                            (((*categoryIterator).second)._oCategoryName).c_str(),
                            lCategoryListEntry.CategoryName);
                    /* Copy channel list data */
                    map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator channelIterator;
                    for(channelIterator =
                            (((*categoryIterator).second).
                                    mCategoryChannelList).begin();
                            channelIterator !=
                                    (((*categoryIterator).second).
                                            mCategoryChannelList).end();
                            ++channelIterator)
                    {
                        const fc_sxm_trChannel& rChannel((*channelIterator).second);
                        if ( rChannel.bIsBrowseable() )
                        {
                            midw_ext_fi_tcl_ChannelListEntry lChannelList;
                            vCopyChannelEntry(lChannelList, rChannel);
                            // Insert new channel list element into channel list
                            lCategoryListEntry.ChannelList.ChannelList.push_back(lChannelList);
                        }
                    }

                    /* Insert new category list element into method result */
                    ((ofiTxObj).CategoryList).push_back(lCategoryListEntry);

                    /* Save current category type */
                    _u16CurrentSelectedCategoryID = lRequestedCategory;
                }
            }
            else
            {
                ETG_TRACE_ERR(("Failed to find category details for "
                        "specified category ID"));
            }
        }
    }
    else
    {
        ETG_TRACE_ERR(("Recived invalid category type in method start"));
    }

    /* Post method result */
    if(AIL_EN_N_NO_ERROR == 
            fc_sxm_tclAudioService::instance()->enSendFiMessage(
                    prMsg->rAdressing,
                    ofiTxObj))
    {
        ETG_TRACE_USR3(("Category list method result sent successfully"));
    }
    else
    {
        ETG_TRACE_ERR(("Failed to send method result for category list"));
    }
}

tVoid fc_sxm_tclCategoryList::vProcess(fc_sxm_trMsgCategoryEventCallback const *prMsg)
{
    ETG_TRACE_USR4(("vProcess(fc_sxm_trMsgCategoryEventCallback): _u16CategoryId=%u mask=0x%08x", 
            prMsg->rCategory._u16CategoryId, prMsg->tEventMask));
    if (fc_sxm_tclAudioProxy::instance()->bAreHandleChnCatCallbacksAllowed()) {
        vUpdateCategoryInfo(prMsg->rCategory, prMsg->tEventMask);

        if(prMsg->rCategory._oCategoryName == "My Seeks") {
            _u16MySeekCatId = prMsg->rCategory._u16CategoryId;
        }

        if((prMsg->rCategory._u16CategoryId == _u16MySeekCatId) && (_poAudioApp))
        {
           fc_sxm_trMsgAudioSeekCategoryUpdate oSeekCategoryUpdate;
           _poAudioApp->vPostSeekCategoryUpdate(oSeekCategoryUpdate);
        }
    }
}

/* Method returns PresetsIndex for the given service ID*/
tU8 fc_sxm_tclCategoryList::u8GetPresetsIndex(SERVICE_ID serviceID) const
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::u8GetPresetsIndex()"));
    Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
            _poAudioApp->poGetPresetsInstance(): NULL;
    SXM_ASSERT_RETURN_VAL(NULL != poPresets, 0)
    return (poPresets->u8GetPresetIndex(serviceID));
}

/* Set Audio App Pointer Reference*/
tVoid fc_sxm_tclCategoryList::vSetApp(Ifc_sxm_tclAudioApp* audioApp)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vSetApp()"));
    _poAudioApp = audioApp;
}

/* Copy channel object contents into midw_ext_fi_tcl_ChannelListEntry FI structure*/
tVoid fc_sxm_tclCategoryList::vCopyChannelEntry (midw_ext_fi_tcl_ChannelListEntry& trChannelEntry,
        const fc_sxm_trChannel& rChannel) const
{
    // Copy service ID
    trChannelEntry.ServiceID = rChannel.u16ServiceId;
    // Copy channel ID
    trChannelEntry.ChannelID = rChannel.u16ChannelId;
    // Copy rChannel name availability
    trChannelEntry.ChannelNameAvail = rChannel.bIsChannelNameAvailable();
    // Copy rChannel name
    fc_sxm_vString2Fi( rChannel.oAudioChannelName.c_str(), trChannelEntry.ChannelName);
    // Copy Artist name
    fc_sxm_vString2Fi( rChannel.oChannelArtistName.c_str(), trChannelEntry.ChannelArtistName);
    // Copy Song name
    fc_sxm_vString2Fi( rChannel.oChannelSongName.c_str(), trChannelEntry.ChannelSongName);
    // Copy preset value
    trChannelEntry.PresetNumber = u8GetPresetsIndex(rChannel.u16ServiceId);
    // Copy Lock, Availability, Mature and Skipped attributes
    trChannelEntry.IsLocked = rChannel.bIsLocked();
    trChannelEntry.IsAvailable = rChannel.bIsAvailable();
    trChannelEntry.IsMature = rChannel.bIsMature();
    trChannelEntry.IsSkipped = rChannel.bIsSkipped();
}



/* Method sends property "CategoryListUpdate" to HMI */
tVoid fc_sxm_tclCategoryList::vNotifyCategoryListContentUpdate (tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vNotifyCategoryListUpdate()"));

    /*Stop the timer*/
    _mCategoryListUpdateTimer.vStop();

    if(_bListUpdate) {

        /* Set the status info and notify all clients */
        fc_sxm_tcl_trAudioPropertyCategoryListStatus tCategoryListStatus;
        tCategoryListStatus.oFiRes.CategoryListContentUpdate = TRUE;
        fc_sxm_tclAudioProperties::instance()->oCategoryListStatus.vSet(tCategoryListStatus);
        fc_sxm_tclAudioProperties::instance()->oCategoryListStatus.vNotify();

        _bListUpdate = FALSE;

        /*Start the timer again*/
        vStartCategoryListTimer();
    }
}

/* Process timeout of category list update timer */
tVoid fc_sxm_tclCategoryList::vProcessTimer (fc_sxm_trMsgCategoryListTimer const * /* prMsg */ )
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::fc_sxm_trMsgCategoryListTimer"));
    /*Notify the CategoryListUpdate*/
    vNotifyCategoryListContentUpdate();
}

/* Method to start category list timer */
tVoid fc_sxm_tclCategoryList::vStartCategoryListTimer (tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vStartCategoryListTimer()"));
    _mCategoryListUpdateTimer.vStart(FC_SXM_AUDIO_CATEGORY_LIST_UPDATE_PERIOD);
}

/* Method handles artist/song updates from channel list */
tVoid fc_sxm_tclCategoryList::vCategoryListUpdate (tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vCategoryListUpdate()"));
    _bListUpdate = TRUE;
    if(!(_mCategoryListUpdateTimer.bIsRunning())) {
        /*Notify the CategoryListUpdate*/
        vNotifyCategoryListContentUpdate();
    }
}

/* Method to check for valid category type range */
tBool fc_sxm_tclCategoryList::bIsValidRangeCategoryId(CATEGORY_ID nCatId) const
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vCheckCategoryIdValidRange()"));
    return (! (((FC_SXM_AUDIO_CATEGORY_TYPE_BROADCAST_START > nCatId) ||
            (FC_SXM_AUDIO_CATEGORY_TYPE_BROADCAST_END < nCatId)) &&
            ((FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_START > nCatId) ||
                    (FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_END < nCatId))));
}

/* Method to convert broadcast categoryId to UserId*/
USER_ID fc_sxm_tclCategoryList::u16ConvertBroadcastCategoryIdtoUserId(CATEGORY_ID rCatId) const
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vConvertBroadcastCategoryIdtoUserId()"));
    USER_ID nUserId = (tU16)(FC_SXM_AUDIO_FEATURED_FAV_NUM_BANDS + rCatId);
    SMSAPI_RETURN_CODE_ENUM eSetRetVal = CATEGORY.eSetUserId(( (_poAudioApp)?
            _poAudioApp->oGetDecoderObject() :
            DECODER_INVALID_OBJECT), rCatId, nUserId);
    if(eSetRetVal == SMSAPI_RETURN_CODE_SUCCESS)
    {
        return nUserId;
    }
    else
    {
        ETG_TRACE_ERR(("Failed to set UserId for the respective CategoryId"));
        return CATEGORY_INVALID_ID;
    }
}


/* Method to convert User categoryId to UserId*/
USER_ID fc_sxm_tclCategoryList::u16ConvertUserCategoryIdtoUserId(CATEGORY_ID rCatId)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vConvertUserCategoryIdtoUserId()"));

    /* incrementing featured favorite count */
    _u32FeaturedFavCount += 1;

    /* setting featured favorite count as userid for the new user category*/
    USER_ID nUserId = (USER_ID)_u32FeaturedFavCount;
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vConvertUserCategoryIdtoUserId()(nUserId: %u) (CatId: %u)", nUserId, rCatId));

    SMSAPI_RETURN_CODE_ENUM eSetRetVal = CATEGORY.eSetUserId(( (_poAudioApp)?
            _poAudioApp->oGetDecoderObject() :
            DECODER_INVALID_OBJECT), rCatId, nUserId);
    if(eSetRetVal == SMSAPI_RETURN_CODE_SUCCESS)
    {
        return nUserId;
    }
    else
    {
        ETG_TRACE_ERR(("Failed to set UserId for the respective CategoryId"));
        return CATEGORY_INVALID_ID;
    }
}

// Method returns the channel browse status stored by class channel_list for the given channelID
tBool fc_sxm_tclCategoryList::bChannelBrowseStatus(CHANNEL_ID tChannelID) const
{
    Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
            _poAudioApp->poGetChannelListInstance(): NULL;
    SXM_ASSERT_RETURN_VAL(NULL != poChannelList, 0)
    return (poChannelList->bGetChannelBrowseStatus(tChannelID));
}

// Method collects browseable channels from a category for Rapid Channel Browsing
tVoid fc_sxm_tclCategoryList::vFetchCategoryChannels(vector<CHANNEL_ID>& vectChnBrowse, CATEGORY_ID tCategoryID) const
{
    vectChnBrowse.clear();
    // Find requested category ID
    map<CATEGORY_ID, fc_sxm_trCategory>::const_iterator catIter = _mCategoryList.find(tCategoryID);
    if (catIter != _mCategoryList.end())
    {
        // Iterate thru channels belonging category "tCategoryID"
        map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator chnIter = catIter->second.mCategoryChannelList.begin();
        while (chnIter != catIter->second.mCategoryChannelList.end())
        {
            // Check Channel's Browse Status
            if (bChannelBrowseStatus(chnIter->first))
            {
                vectChnBrowse.push_back(chnIter->first);
            }
            ++chnIter;
        }
    }
    ETG_TRACE_USR3(("vFetchCategoryChannels Category(%u) has (%u) browseable channels", tCategoryID, vectChnBrowse.size()));
}

#ifdef FC_SXM_ENABLE_FEATURED_FAVORITE_SORTING

//Adding string and category id into Featured favorite list for Sorting
tVoid fc_sxm_tclCategoryList::vAddFeaturedFavoriteToSort(
        string sCategoryName, CATEGORY_ID u16CategoryId)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vAddFeaturedfavtoSort"));

    //Adding into featured favorite list, introduced for sorting
    _mFeaturedFavCategoryList.insert(pair<string, CATEGORY_ID>
    (sCategoryName, u16CategoryId));
    if(_mFeaturedFavCategoryList.size() > 1) {
        vSortFeaturedFavorite();
    }
}

//Removing from featured favorite list, introduced for sorting
tVoid fc_sxm_tclCategoryList::vRemoveFeaturedFavorite(CATEGORY_ID nCatId)
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vRemoveFeaturedFavorite"));
    if(CATEGORY_INVALID_ID == nCatId)
    {
        ETG_TRACE_ERR(("Invalid Category ID"));
    }
    else
    {
        map<string, CATEGORY_ID>::iterator mIterFeaturedFavCategoryList;
        for(mIterFeaturedFavCategoryList=_mFeaturedFavCategoryList.begin();
                mIterFeaturedFavCategoryList !=_mFeaturedFavCategoryList.end();
                ++mIterFeaturedFavCategoryList)
        {
            if(mIterFeaturedFavCategoryList->second == nCatId)
            {
                _mFeaturedFavCategoryList.erase(mIterFeaturedFavCategoryList);
                break;
            }
        }
    }
}

//Method to sort Featured Favorite Categories only
tVoid fc_sxm_tclCategoryList::vSortFeaturedFavorite()
{
    ETG_TRACE_USR4(("fc_sxm_tclCategoryList::vSortFeaturedFavorite()"));

    map<string, CATEGORY_ID>::const_iterator mIter;
    tU8 u8Index = 0;
    DECODER_OBJECT hDecoder = ( (_poAudioApp)? _poAudioApp->oGetDecoderObject() :
            DECODER_INVALID_OBJECT);
    if(hDecoder != DECODER_INVALID_OBJECT) {
        for(mIter = _mFeaturedFavCategoryList.begin();
                mIter != _mFeaturedFavCategoryList.end(); ++ mIter)
        {
            SMSAPI_RETURN_CODE_ENUM eSetRetVal = CATEGORY.eSetUserId(hDecoder, mIter->second, (tU16)(u8Index +1));
            if(eSetRetVal == SMSAPI_RETURN_CODE_SUCCESS) {
                _mCategoryUserIdList[(tU16)(u8Index +1)] = mIter->second;
                u8Index = (tU8)(u8Index + 1);
            }
            else {
                ETG_TRACE_ERR(("Failed to set UserId for the respective CategoryId"));
                break;
            }
        }
    }
    else {
        ETG_TRACE_ERR(("Invalid Decoder"));
    }
}
#endif

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
