/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_audio_utils.cpp
* @brief       Implementation for all utility methods used by fc_sxm audio service. 
*              File controls all advisory notifications being triggered from audio service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_audio_fi.h"
#include "fc_sxm_tcl_audio_utils.h"
#include "fc_sxm_generic_utils.h"

/*
 * Method returns the display advisory to be shown based on priority calculations
 */
enAdvisoryMsgType fc_sxm_tclAudioUtils::enGetAdvisoryType(const tU32 u32Value)
{
	enAdvisoryMsgType enAdvisory;
	if (TRUE == fc_sxm_tclAudioUtils::bIsHighPriorityAdvisory(u32Value))
		enAdvisory = enHIGH_PRIO_ADVISORY;
	else if (TRUE == fc_sxm_tclAudioUtils::bIsChannelAdvisory(u32Value))
		enAdvisory = enCHANNEL_ADVISORY;
	else
		enAdvisory = enNO_ADVISORY;
	return enAdvisory;
}

/*
 * Method returns TRUE if the given value is qualified for Showing channel specific advisory msg
 */
tBool fc_sxm_tclAudioUtils::bIsChannelAdvisory(const tU32 u32Value)
{
	// Defining priority order for Channel specific advisories
	midw_ext_fi_tcl_e8_SystemMessage::tenType priority[] = {
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND, //0
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_OFFAIR, //1
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, //2
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, //3
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE, //4
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED, //5
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_LOADING_XM //6
	};

	for (size_t u8Index = 0U; u8Index <= 6U; u8Index++)
	{
		tU32 u32BitPos = static_cast <tU32> (priority[u8Index]);
		// Check if status bit is set for channel attributes
		if (fc_sxm_tclGenericUtils::bCheckBit(u32Value, u32BitPos))
			return TRUE;
	}
	return FALSE;
}
/*
 * Method returns TRUE if the given value is qualified for Showing higher priority advisory msg
 */
tBool fc_sxm_tclAudioUtils::bIsHighPriorityAdvisory(const tU32 u32Value)
{
	// Defining priority order for HighPrio advisory types
	midw_ext_fi_tcl_e8_SystemMessage::tenType priority[] = {
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SXM_INITIALIZING, // 0
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_HARDWARE_ERROR, //1
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SUBSCRIPTION_UPDATED, //2
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_ANTENNA_DISCONNECTED, //3
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_SIGNAL //4
	};
	// Iterate thru the priority array
	for (size_t u8Index = 0U; u8Index <= 4U; u8Index++)
	{
		tU32 u32BitPos = static_cast <tU32> (priority[u8Index]);
		// Check if status bit is set for high priority attributes
		if (fc_sxm_tclGenericUtils::bCheckBit(u32Value, u32BitPos))
			return TRUE;
	}
	return FALSE;
}

/*
 * Method checks and sets the correct Advisory Msg Type based on the priority order of channel advisories
 */
tVoid fc_sxm_tclAudioUtils::vSetChannelAdvisoryMsgType(const tU32 u32Value,
			midw_ext_fi_tcl_e8_SystemMessage::tenType& enCurrentSystemMsg,
			enAdvisoryMsgType &enCheckAdvisory)
{
	// Set default values only for channel specific advisories
	enCheckAdvisory = enNO_ADVISORY;
	enCurrentSystemMsg = midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;

	// Defining priority order for Channel specific advisories
	midw_ext_fi_tcl_e8_SystemMessage::tenType priority[] = {
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND, //0
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_OFFAIR, //1
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, //2
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, //3
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE, //4
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED, //5
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_LOADING_XM //6
	};

	// Iterate thru the priority array
	for (size_t u8Index = 0U; u8Index <= 6U; u8Index++)
	{
		tU32 u32BitPos = static_cast <tU32> (priority[u8Index]);
		if (fc_sxm_tclGenericUtils::bCheckBit(u32Value, u32BitPos))
		{
			// Set the current System Msg and CheckAdvisory
			enCurrentSystemMsg = priority[u8Index];
			enCheckAdvisory = enCHANNEL_ADVISORY;
			break; // break out after deciding which advisory to be displayed
		}
	}
}
/*
 * Method checks and sets the correct Advisory Msg Type based on the priority order of HighPrio advisories
 */
tVoid fc_sxm_tclAudioUtils::vSetHighPrioAdvisoryMsgType(const tU32 u32Value,
			midw_ext_fi_tcl_e8_SystemMessage::tenType& enCurrentSystemMsg,
			enAdvisoryMsgType &enCheckAdvisory)
{
	// Defining priority order for HighPrio advisory types
	midw_ext_fi_tcl_e8_SystemMessage::tenType priority[] = {
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SXM_INITIALIZING, // 0
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_HARDWARE_ERROR, //1
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SUBSCRIPTION_UPDATED, //2
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_ANTENNA_DISCONNECTED, //3
			midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_SIGNAL //4
	};
	// Iterate thru the priority array
	for (size_t u8Index = 0U; u8Index <= 4U; u8Index++)
	{
		tU32 u32BitPos = static_cast <tU32> (priority[u8Index]);
		if (fc_sxm_tclGenericUtils::bCheckBit(u32Value, u32BitPos))
		{
			// Set the current System Msg and CheckAdvisory
			enCurrentSystemMsg = priority[u8Index];
			enCheckAdvisory = enHIGH_PRIO_ADVISORY;
			break; // break out after deciding which advisory to be displayed
		}
	}
}
/*
 * Method returns true if Channel Advisory bit is set and tuned channel must be restored to Preview channel
 */
tBool fc_sxm_tclAudioUtils::bRestoreToPreviewChannel(const tU32 u32Value)
{
	tBool bChnUnAuthourized = fc_sxm_tclGenericUtils::bCheckBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED));
	tBool bChnUnAvailable = fc_sxm_tclGenericUtils::bCheckBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE));
	tBool bChnMatureLocked = fc_sxm_tclGenericUtils::bCheckBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE));
	tBool bChnSkipped = fc_sxm_tclGenericUtils::bCheckBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED));

	tBool bShouldRestorePrvChn = static_cast <tBool> ((bChnUnAuthourized | bChnUnAvailable | bChnMatureLocked | bChnSkipped));
	return bShouldRestorePrvChn;
}

/*
 * Clear All Channel Advisory Bits
 */
tVoid fc_sxm_tclAudioUtils::vClearAllChannelAdvisories(tU32& u32Value)
{
	// Reset unauthorised system message
	fc_sxm_tclGenericUtils::vClearBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED));

	// Reset channel unavailable system message
	fc_sxm_tclGenericUtils::vClearBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE));

	// Reset category unavailable system message
	fc_sxm_tclGenericUtils::vClearBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND));

	// Reset category unavailable system message
	fc_sxm_tclGenericUtils::vClearBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE));

	// Reset category unavailable system message
	fc_sxm_tclGenericUtils::vClearBit(u32Value, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED));
}

/*
 * Method returns TRUE if an update for preset band channel change notification has to be triggered from MIDW
 */
tBool fc_sxm_tclAudioUtils::bSendPresetBandNotifications(const tU32 u32Mask)
{
	// Method decides if preset band notification has to be sent to HMI or not based on the channel event mask notified
	tU32 u32SuppressMask = (CHANNEL_OBJECT_EVENT_CATEGORY |
			CHANNEL_OBJECT_EVENT_TITLE |
			CHANNEL_OBJECT_EVENT_ARTIST |
			CHANNEL_OBJECT_EVENT_ALBUM);
	tU32 u32EventMask = (u32Mask) & (~u32SuppressMask);
	tBool bSendNotification = static_cast <tBool> (u32EventMask != 0);
	return bSendNotification;
}

/*
 * Method Decides the change that needs to be performed on Smart Favorites List for every change in preset channels
 * Arguments:
 * 1) u8Presets_RemoveCount -- The number(count) of entries of channel to be removed in presets list
 * 2) u8SF_AddCount -- The number(count) of entries of channel to be added in SF list
 * 3) u8SF_RemoveCount -- The number(count) of entries of channel to be removed from SF list
 */
enPresetChange_Action fc_sxm_tclAudioUtils::enGetPresetChangeAction(tU8 u8Presets_RemoveCount,
			tU8 u8SF_AddCount, tU8 u8SF_RemoveCount)
{
	enPresetChange_Action enAction = enAction_Invalid;
	// Make decision on how to handle store preset channel

	if (u8Presets_RemoveCount > 1)
	{
		// Condition 1: If SF channel that has to be removed is stored in more than one preset index
		enAction = fc_sxm_tclAudioUtils::enGetDuplicatePresetAction(u8SF_AddCount, u8SF_RemoveCount);
	}
	else if (u8Presets_RemoveCount == 1)
	{
		// Condition 2: If SF channel that has to be removed is stored in only one preset index
		enAction = fc_sxm_tclAudioUtils::enGetSingularPresetAction(u8SF_AddCount, u8SF_RemoveCount);
	}
	else if (u8Presets_RemoveCount == 0)
	{
		// Condition 3: If SF channel that has to be removed is not stored in any presets list
		enAction = fc_sxm_tclAudioUtils::enGetNonPresetAction(u8SF_AddCount, u8SF_RemoveCount);
	}
	return enAction;
}

/*
 * Method evaluates the action to be  taken on Smart Favorites List when the same preset channel
 *     is stored in more than one place
 * Arguments:
 * 1) u8SF_AddCount -- The number(count) of entries of channel to be added in SF list
 * 2) u8SF_RemoveCount -- The number(count) of entries of channel to be removed from SF list
 */
enPresetChange_Action fc_sxm_tclAudioUtils::enGetDuplicatePresetAction(tU8 u8SF_AddCount, tU8 u8SF_RemoveCount)
{
	enPresetChange_Action enAction = enAction_Invalid;
	// Condition 1: the channel to be added is not part of SF list
	if (u8SF_AddCount == 0)
	{
		if (u8SF_RemoveCount == 0)
		{
			enAction = enAction_AddChn;
		}
		else if (u8SF_RemoveCount == 1)
		{
			enAction = enAction_AddChn;
		}
	}
	// Condition 2: the channel to be added is already part of SF list
	else if (u8SF_AddCount == 1)
	{
		if (u8SF_RemoveCount == 0)
		{
			enAction = enAction_NoAction;
		}
		else if (u8SF_RemoveCount == 1)
		{
			enAction = enAction_NoAction;
		}
	}
	return enAction;
}

/*
 * Method evaluates the action to be  taken on Smart Favorites List when a preset channel which
 *    is present in only one preset channel is removed
 * Arguments:
 * 1) u8SF_AddCount -- The number(count) of entries of channel to be added in SF list
 * 2) u8SF_RemoveCount -- The number(count) of entries of channel to be removed from SF list
 */
enPresetChange_Action fc_sxm_tclAudioUtils::enGetSingularPresetAction(tU8 u8SF_AddCount, tU8 u8SF_RemoveCount)
{
	enPresetChange_Action enAction = enAction_Invalid;
	// Condition 1: the channel to be added is not part of SF list
	if (u8SF_AddCount == 0)
	{
		if (u8SF_RemoveCount == 0)
		{
			enAction = enAction_AddChn;
		}
		else if (u8SF_RemoveCount == 1)
		{
			enAction = enAction_ReplaceChn;
		}
	}
	// Condition 2: the channel to be added is already part of SF list
	else if (u8SF_AddCount == 1)
	{
		if (u8SF_RemoveCount == 0)
		{
			enAction = enAction_NoAction;
		}
		else if (u8SF_RemoveCount == 1)
		{
			enAction = enAction_DeleteChn;
		}
	}
	return enAction;
}

/*
 * Method evaluates the action to be  taken on Smart Favorites List when a Non-preset channel
 *    is added to presets
 * Arguments:
 * 1) u8SF_AddCount -- The number(count) of entries of channel to be added in SF list
 * 2) u8SF_RemoveCount -- The number(count) of entries of channel to be removed from SF list
 */
enPresetChange_Action fc_sxm_tclAudioUtils::enGetNonPresetAction(tU8 u8SF_AddCount, tU8 u8SF_RemoveCount)
{
	enPresetChange_Action enAction = enAction_Invalid;
	// Condition 1: the channel to be added is not part of SF list
	if (u8SF_AddCount == 0)
	{
		if (u8SF_RemoveCount == 0)
		{
			enAction = enAction_ReplaceChn;
		}
		else if (u8SF_RemoveCount == 1)
		{
			enAction = enAction_Invalid;
		}
	}
	// Condition 2: the channel to be added is already part of SF list
	else if (u8SF_AddCount == 1)
	{
		if (u8SF_RemoveCount == 0)
		{
			enAction = enAction_NoAction;
		}
		else if (u8SF_RemoveCount == 1)
		{
			enAction = enAction_Invalid;
		}
	}
	return enAction;
}

/*
 * Method does a one-one mapping between FI Channel event mask passed by HMI with Channel event mask
 * provided by SMS lib
 */
CHANNEL_EVENT_MASK fc_sxm_tclAudioUtils::u16ConvertChannelEventBitSet(const midw_ext_fi_tcl_b16_CHANNEL_EVENT& event)
{
	CHANNEL_EVENT_MASK tEventMask = CHANNEL_OBJECT_EVENT_NONE;
	if (event.bCHANNEL_EVENT_SERVICE_ID())
		tEventMask |= CHANNEL_OBJECT_EVENT_SERVICE_ID;
	if (event.bCHANNEL_EVENT_CHANNEL_ID())
		tEventMask |= CHANNEL_OBJECT_EVENT_CHANNEL_ID;
	if (event.bCHANNEL_EVENT_ATTRIBUTES())
		tEventMask |= CHANNEL_OBJECT_EVENT_ATTRIBUTES;
	if (event.bCHANNEL_EVENT_NAME())
		tEventMask |= CHANNEL_OBJECT_EVENT_NAME;
	if (event.bCHANNEL_EVENT_CATEGORY())
		tEventMask |= CHANNEL_OBJECT_EVENT_CATEGORY;
	if (event.bCHANNEL_EVENT_TITLE())
		tEventMask |= CHANNEL_OBJECT_EVENT_TITLE;
	if (event.bCHANNEL_EVENT_ARTIST())
		tEventMask |= CHANNEL_OBJECT_EVENT_ARTIST;
	if (event.bCHANNEL_EVENT_ALBUM())
		tEventMask |= CHANNEL_OBJECT_EVENT_ALBUM;
	if (event.bCHANNEL_EVENT_ART())
		tEventMask |= CHANNEL_OBJECT_EVENT_ART;
	if (event.bCHANNEL_EVENT_REMOVED())
		tEventMask |= CHANNEL_OBJECT_EVENT_REMOVED;
	return tEventMask;
}

/*
 * Method returns TRUE if the saved mask has atleast one bit set as the compare mask
 */
tBool fc_sxm_tclAudioUtils::bCompareChannelEventMask(const CHANNEL_EVENT_MASK tCompareMask, const CHANNEL_EVENT_MASK tSavedMask)
{
	return (static_cast <tBool> ((0 != (tCompareMask & tSavedMask))));
}

/*
 * Method returns TRUE if the system error message to be sent carries error specific for Channel Tune
 */
tBool fc_sxm_tclAudioUtils::bIsChannelTuneError(const midw_ext_fi_tcl_e8_SystemMessage::tenType& enErrorType)
{
	tBool bChannelTuneError = ((midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE == enErrorType) ||
			(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED == enErrorType) ||
			(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_OFFAIR == enErrorType) ||
			(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE == enErrorType) ||
			(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED == enErrorType) );
	return bChannelTuneError;
}

/*
 * Method implements the following MATURE CHANNEL logic
 * Is channel locked	Is channel mature	Lock All Bit (Yes/No)	Lock/Unlock/No action
 *      Yes	                  Yes	            Yes	                   No action
 *      Yes	                  Yes	            No	                   Unlock
 *      Yes	                   No	            Yes	                   Unlock
 *      Yes	                   No	            No	                   Unlock
 *      No	                  Yes	            Yes	                   Lock
 *      No	                  Yes	            No	                   No action
 *      No 	                   No	            Yes	                   No action
 *      No	                   No	            No	                   No action
 *
 *  Method returns the values stated in the last column
 */

enChannelLockAction fc_sxm_tclAudioUtils::enGetChannelLockAction (tBool bIsChnLocked, tBool bIsChnMature, tBool bMatureLockOn)
{
	enChannelLockAction enAction = enChannelAction_Ignore;
	if (bIsChnLocked)
	{
		if (bIsChnMature)
		{
			if (!bMatureLockOn)
			{
				enAction = enChannelAction_UnLock;
			}
		}
		else // bIsChnMature is false
		{
			enAction = enChannelAction_UnLock;
		}
	}
	else if ((bIsChnMature) && (bMatureLockOn)) // bIsChnLocked is false
	{
		enAction = enChannelAction_Lock;
	}
	return enAction;
}

