/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_properties.h
* @brief       Declaration of messages used for property notifications within SXM Audio.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef SXM_AUDIO_PROPERTIES_H
#define SXM_AUDIO_PROPERTIES_H

#include "fc_sxm_audio_fi.h"

/* Channel list status info */
struct fc_sxm_tcl_trAudioPropertyChannelListStatus 
{
    /* Constructor */
    fc_sxm_tcl_trAudioPropertyChannelListStatus(tBool bChanged = FALSE) 
    {
    	(tVoid)bChanged;
        oFiRes.Availability = FALSE;
    }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trAudioPropertyChannelListStatus const& b) const
    {
        SXM_RET_1_IF_NEQ(oFiRes.Availability);
        return FALSE;
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgChannelListStatusStatus() const
    {
        return oFiRes;
    }
    
    /* Data containing channel list status info */
    midw_ext_sxm_audiofi_tclMsgChannelListStatusStatus oFiRes; 
};

/* Channel information */
struct fc_sxm_tcl_trAudioPropertyChannelInformation 
{
    /* Constructor */
    fc_sxm_tcl_trAudioPropertyChannelInformation() 
    {
        (oFiRes.ChannelInformation).ChannelNameAvail    = FALSE;    /* Set channel name available to false */
        (oFiRes.ChannelInformation).CategoryNameAvail   = FALSE;    /* Set category name available to false */
        (oFiRes.ChannelInformation).ArtistNameAvail     = FALSE;    /* Set artist name available to false */
        (oFiRes.ChannelInformation).SongNameAvail       = FALSE;    /* Set song name available to false */
        (oFiRes.ChannelInformation).ExtSongNameAvail    = FALSE;    /* Set extended song name available to false */
        (oFiRes.ChannelInformation).ExtArtistNameAvail  = FALSE;    /* Set extended artist name available to false */
        (oFiRes.ChannelInformation).ChnGraphicAvail     = FALSE;    /* Set channel graphics available to false */
        (oFiRes.ChannelInformation).ChannelNumber       = 0xFFFF;   /* Set default channel number */
        (oFiRes.ChannelInformation).SID                 = 0xFFFF;   /* Set default service id */
        (oFiRes.ChannelInformation).CategoryType        = 0xFF;     /* Set default category id */
        (oFiRes.ChannelInformation).PrimaryCategoryType = 0xFF;     /* Set default primary category */
        (oFiRes.ChannelInformation).ChannelName         = "";       /* Clear channel name */
        (oFiRes.ChannelInformation).CategoryName        = "";       /* Clear category name */
        (oFiRes.ChannelInformation).SongName            = "";       /* Set song name */
        (oFiRes.ChannelInformation).ArtistName          = "";       /* Set artist name */
        (oFiRes.ChannelInformation).ExtSongName         = "";       /* Set extended song name */
        (oFiRes.ChannelInformation).ExtArtistName       = "";       /* Set extended artist name */
    }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trAudioPropertyChannelInformation const& b) const
    {
        return !(oFiRes == b.oFiRes);
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgChannelInformationStatus() const
    {
        return oFiRes;
    }
    
    /* Data containing channel information */
    midw_ext_sxm_audiofi_tclMsgChannelInformationStatus oFiRes; 
};

/* Auto compare */
struct fc_sxm_tcl_trAudioPropertyAutoCompare 
{
    /* Constructor */
    fc_sxm_tcl_trAudioPropertyAutoCompare() 
    {
        oFiRes.TunerElemNr = 0;
    }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trAudioPropertyAutoCompare const& b) const
    {
        return !(oFiRes == b.oFiRes);
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgAutoCompareStatus() const
    {
        return oFiRes;
    }
    
    /* Data containing channel information */
    midw_ext_sxm_audiofi_tclMsgAutoCompareStatus oFiRes; 
};

/* System Message */
struct fc_sxm_tcl_trAudioPropertySystemMessage 
{
    /* Constructor */
    fc_sxm_tcl_trAudioPropertySystemMessage() 
    {
        ((oFiRes).SystemErrorType).enType = 
            midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_ERROR;
        oFiRes.CID = CHANNEL_INVALID_ID;
    }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trAudioPropertySystemMessage const& b) const
    {
        return !( (oFiRes.SystemErrorType == b.oFiRes.SystemErrorType) &&
        		(oFiRes.CID == b.oFiRes.CID));
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgSystemMessageStatus() const
    {
        return oFiRes;
    }
    
    /* Data containing channel information */
    midw_ext_sxm_audiofi_tclMsgSystemMessageStatus oFiRes; 
};

/* Category list status info */
struct fc_sxm_tcl_trAudioPropertyCategoryListStatus 
{
    /* Constructor */
    fc_sxm_tcl_trAudioPropertyCategoryListStatus(tBool bChanged = FALSE) 
    {
    	(tVoid)bChanged;
        oFiRes.Availability = FALSE;
    }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trAudioPropertyCategoryListStatus const& b) const
    {
        SXM_RET_1_IF_NEQ(oFiRes.Availability);
        return FALSE;
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgCategoryListStatusStatus() const
    {
        return oFiRes;
    }
    
    /* Data containing channel list status info */
    midw_ext_sxm_audiofi_tclMsgCategoryListStatusStatus oFiRes; 
};

/* Audio service subscription status info */
struct fc_sxm_tcl_trAudioPropertyAudioSubStatus 
{
    /* Constructor */
    fc_sxm_tcl_trAudioPropertyAudioSubStatus() 
    { }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trAudioPropertyAudioSubStatus const& b) const
    {
        return !(oFiRes == b.oFiRes);
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgSxmAudioSubInfoStatus() const
    {
        return oFiRes;
    }

    /* Data containing channel list status info */
    midw_ext_sxm_audiofi_tclMsgSxmAudioSubInfoStatus oFiRes; 
};

struct fc_sxm_tcl_trAudioPropertyInstantReplayStatus
{
   /* Constructor */
   fc_sxm_tcl_trAudioPropertyInstantReplayStatus()
   {
      ((oFiRes).InstantReplayStatus).TimeToLive = 0;     /* Time to LIVE for IR controls */
      ((oFiRes).InstantReplayStatus).PlaybackStatus.enType = 
         midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_INVALID_STATE; /* Current play state for IR controls */
   }

   /* Overloading comparision operator */
   bool operator!=(fc_sxm_tcl_trAudioPropertyInstantReplayStatus const& b) const
   {
       return !(oFiRes == b.oFiRes);
   }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgInstantReplayStatusStatus() const
    {
        return oFiRes;
    }

   /* Data containing instant replay status info */
   midw_ext_sxm_audiofi_tclMsgInstantReplayStatusStatus oFiRes;
};

/* IR Mode */
struct fc_sxm_tcl_trAudioPropertyIRMode 
{
    /* Constructor */
    fc_sxm_tcl_trAudioPropertyIRMode() 
    {
        oFiRes.IsIRModeActive = FALSE;
    }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trAudioPropertyIRMode const& b) const
    {
        return !(oFiRes == b.oFiRes);
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgInstantReplayModeStatus() const
    {
        return oFiRes;
    }
    
    /* Data containing channel information */
    midw_ext_sxm_audiofi_tclMsgInstantReplayModeStatus oFiRes; 
};

/* Data services antenna status */
struct fc_sxm_tcl_trDSAntennaStatus 
{
    /* Constructor */
    fc_sxm_tcl_trDSAntennaStatus() 
    {
        oFiRes.IsAntennaConnected = FALSE;
    }

    /* Overloading comparision operator */
    bool operator!=(fc_sxm_tcl_trDSAntennaStatus const& b) const
    {
        return !(oFiRes == b.oFiRes);
    }

    /* Overloading conversion operator */
    operator midw_ext_sxm_audiofi_tclMsgSxmDataServiceAntennaStatusStatus() const
    {
        return oFiRes;
    }
    
    /* Data containing channel information */
    midw_ext_sxm_audiofi_tclMsgSxmDataServiceAntennaStatusStatus oFiRes; 
};

/* Tune Scan Content status update*/
struct fc_sxm_tcl_trAudioPropertyTuneScanContentStatus
{
	fc_sxm_tcl_trAudioPropertyTuneScanContentStatus()
    {  /* Deafult CTOR */   }

    /* Overloading comparision operator */
    bool operator != (fc_sxm_tcl_trAudioPropertyTuneScanContentStatus const& b) const
    {
        return !(oFiMsg == b.oFiMsg);
    }
    /* Data containing Tune Scan Content information */
    midw_ext_sxm_audiofi_tclMsgTuneScanContentStatusStatus oFiMsg;
};


/* Preset channel update */
struct fc_sxm_tcl_trAudioPropertyPresetChannelUpdate
{
	fc_sxm_tcl_trAudioPropertyPresetChannelUpdate()
    {  /* Deafult CTOR */   }

    /* Overloading comparision operator */
    bool operator != (fc_sxm_tcl_trAudioPropertyPresetChannelUpdate const& b) const
    {
        return !(oFiMsg == b.oFiMsg);
    }
    /* Data containing preset channel information */
    midw_ext_sxm_audiofi_tclMsgPresetChannelUpdateStatus oFiMsg;
};

/* Preset channel update */
struct fc_sxm_tcl_trAudioPropertyPresetListUpdate
{
	fc_sxm_tcl_trAudioPropertyPresetListUpdate()
    {  /* Deafult CTOR */   }

    /* Overloading comparison operator */
    bool operator != (fc_sxm_tcl_trAudioPropertyPresetListUpdate const& b) const
    {
        return !(oFiMsg == b.oFiMsg);
    }
    /* Data containing preset list update information */
    midw_ext_sxm_audiofi_tclMsgPresetListUpdateStatus oFiMsg;
};

/* Clock parameters update */
struct fc_sxm_tcl_trAudioPropertySetClockParams
{
	fc_sxm_tcl_trAudioPropertySetClockParams()
    {  /* Deafult CTOR */   }

    /* Overloading comparison operator */
    bool operator != (fc_sxm_tcl_trAudioPropertySetClockParams const& b) const
    {
        return !(oFiMsg == b.oFiMsg);
    }
    /* Data containing preset list update information */
    midw_ext_sxm_audiofi_tclMsgSetClockParamsStatus oFiMsg;
};


struct fc_sxm_tcl_trFavoriteContentAlertInfo
{
   /* Constructor */
   fc_sxm_tcl_trFavoriteContentAlertInfo()
   {
      // Assigned to Invalid Maximum value to avoid the alert pop up when it send by default Get to HMI
      oFiMsg.AlertInfo.ServiceID = SERVICE_INVALID_ID;
      oFiMsg.AlertInfo.ChannelID = CHANNEL_INVALID_ID;
      oFiMsg.AlertInfo.IsExpired = FALSE;
   }

   /* Overloading comparison operator */
   bool operator!=(fc_sxm_tcl_trFavoriteContentAlertInfo const& b) const
   {
       return !(oFiMsg == b.oFiMsg);
   }

   /* Overloading conversion operator */
   operator midw_ext_sxm_audiofi_tclMsgFavoriteContentAlertInfoStatus() const
   {
       return oFiMsg;
   }

   /* Data containing alert notifications for registered favorites */
   midw_ext_sxm_audiofi_tclMsgFavoriteContentAlertInfoStatus oFiMsg;
};

struct fc_sxm_tcl_trAlertNoticeListUpdate
{
   /* Constructor */
   fc_sxm_tcl_trAlertNoticeListUpdate()
   {
   }

   /* Overloading comparison operator */
   bool operator!=(fc_sxm_tcl_trAlertNoticeListUpdate const& b) const
   {
       return !(oFiMsg == b.oFiMsg);
   }

   /* Overloading conversion operator */
   operator midw_ext_sxm_audiofi_tclMsgAlertNoticeListUpdateStatus() const
   {
       return oFiMsg;
   }

   /* Data containing alert notifications for registered favorites */
   midw_ext_sxm_audiofi_tclMsgAlertNoticeListUpdateStatus oFiMsg;
};

/* TWNowServiceStatus update */
struct fc_sxm_tcl_trAudioPropertyTWNowServiceStatus
{
	fc_sxm_tcl_trAudioPropertyTWNowServiceStatus()
    {
		oFiMsg.ServiceStatus.enType = midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_STOPPED;
    }

    /* Overloading comparision operator */
    bool operator != (fc_sxm_tcl_trAudioPropertyTWNowServiceStatus const& b) const
    {
        return !(oFiMsg.ServiceStatus.enType == b.oFiMsg.ServiceStatus.enType);
    }

    bool operator == (fc_sxm_tcl_trAudioPropertyTWNowServiceStatus const& b) const
	{
		return (oFiMsg.ServiceStatus.enType == b.oFiMsg.ServiceStatus.enType);
	}
    /* Data containing TWNowServiceStatus */
    midw_ext_sxm_audiofi_tclMsgTWNowServiceStatusStatus oFiMsg;
};

/* TWNowBulletinStatus update */
struct fc_sxm_tcl_trAudioPropertyTWNowBulletinStatus
{
	fc_sxm_tcl_trAudioPropertyTWNowBulletinStatus()
    {
		oFiMsg.BulletinID = TW_NOW_INVALID_BULLETIN_ID;
		oFiMsg.BulletinStatus.enType = midw_ext_fi_tcl_e8_TWNow_Bulletin_Status::FI_EN_TWNOW_NO_BULLETIN_AVAILABLE;
    }

    /* Overloading comparision operator */
    bool operator != (fc_sxm_tcl_trAudioPropertyTWNowBulletinStatus const& b) const
    {
        return !( (oFiMsg.BulletinID == b.oFiMsg.BulletinID) &&
        		(oFiMsg.BulletinStatus.enType == b.oFiMsg.BulletinStatus.enType) );
    }

    bool operator == (fc_sxm_tcl_trAudioPropertyTWNowBulletinStatus const& b) const
	{
		return ( (oFiMsg.BulletinID == b.oFiMsg.BulletinID) &&
				(oFiMsg.BulletinStatus.enType == b.oFiMsg.BulletinStatus.enType) );
	}
    /* Data containing TWNowBulletinStatus */
    midw_ext_sxm_audiofi_tclMsgTWNowBulletinStatusStatus oFiMsg;
};


/* SportsFlashServiceStatus update */
struct fc_sxm_tcl_trAudioPropertySportsFlash_ServiceStatus
{
	fc_sxm_tcl_trAudioPropertySportsFlash_ServiceStatus()
    {
		oFiMsg.ServiceStatus.enType = midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_STOPPED;
    }

    /* Overloading comparision operator */
    bool operator != (fc_sxm_tcl_trAudioPropertySportsFlash_ServiceStatus const& b) const
    {
        return !(oFiMsg.ServiceStatus.enType == b.oFiMsg.ServiceStatus.enType);
    }

    bool operator == (fc_sxm_tcl_trAudioPropertySportsFlash_ServiceStatus const& b) const
	{
		return (oFiMsg.ServiceStatus.enType == b.oFiMsg.ServiceStatus.enType);
	}
    /* Data containing TWNowServiceStatus */
    midw_ext_sxm_audiofi_tclMsgSportsFlash_ServiceStatusStatus oFiMsg;
};

/* SportsFlash_GameEventNotification*/
struct fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification
{
	fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification()
	{
		oFiMsg.ProgramStatus.enType = midw_ext_fi_tcl_e8_SportsFlash_ProgramStatus::FI_EN_PROGRAM_ENDED;
		oFiMsg.GameInfo.ChannelID = CHANNEL_INVALID_ID;
		oFiMsg.GameInfo.ServiceID = SERVICE_INVALID_ID;
	}

	/* Overloading comparision operator */
	bool operator != (fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification const& b) const
	{
		return !((oFiMsg.ProgramStatus.enType == b.oFiMsg.ProgramStatus.enType) &&
				(oFiMsg.GameInfo.ChannelID == b.oFiMsg.GameInfo.ChannelID) &&
				(oFiMsg.GameInfo.HomeTeam.TeamID == b.oFiMsg.GameInfo.HomeTeam.TeamID));
	}

	bool operator == (fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification const& b) const
	{
		return ((oFiMsg.ProgramStatus.enType == b.oFiMsg.ProgramStatus.enType) &&
				(oFiMsg.GameInfo.ChannelID == b.oFiMsg.GameInfo.ChannelID) &&
				(oFiMsg.GameInfo.HomeTeam.TeamID == b.oFiMsg.GameInfo.HomeTeam.TeamID));
	}
	midw_ext_sxm_audiofi_tclMsgSportsFlash_GameEventNotificationStatus oFiMsg;
};

/* SportsFlash_FlashEventNotification*/
struct fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification
{
	fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification()
	{
		oFiMsg.FlashEventID = static_cast<tU32>(SPORTS_FLASH_INVALID_EVENT_ID);
		oFiMsg.FlashEventStatus.enType = midw_ext_fi_tcl_e8_SportsFlash_FlashEventStatus::FI_EN_SPORTS_FLASH_EVENT_EXPIRED;
		oFiMsg.GameInfo.ChannelID = CHANNEL_INVALID_ID;
		oFiMsg.GameInfo.ServiceID = SERVICE_INVALID_ID;
	}

	/* Overloading comparision operator */
	bool operator != (fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification const& b) const
	{
		return !((oFiMsg.FlashEventStatus.enType == b.oFiMsg.FlashEventStatus.enType) &&
				(oFiMsg.GameInfo.ChannelID == b.oFiMsg.GameInfo.ChannelID) &&
				(oFiMsg.FlashEventID == b.oFiMsg.FlashEventID));
	}

	bool operator == (fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification const& b) const
	{
		return ((oFiMsg.FlashEventStatus.enType == b.oFiMsg.FlashEventStatus.enType) &&
				(oFiMsg.GameInfo.ChannelID == b.oFiMsg.GameInfo.ChannelID) &&
				(oFiMsg.FlashEventID == b.oFiMsg.FlashEventID));
	}
	bool bIsSavedEvent(fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification const& b) const
	{
		return ((oFiMsg.FlashEventID == b.oFiMsg.FlashEventID) &&
				(oFiMsg.GameInfo.ChannelID == b.oFiMsg.GameInfo.ChannelID));
	}
	midw_ext_sxm_audiofi_tclMsgSportsFlash_FlashEventNotificationStatus oFiMsg;
};

/* TWNowPlaybackStatus notification*/
struct fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus
{
	fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus()
	{
		oFiMsg.FlashEventPlayState.enType = midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_UNKNOWN;
	}
	/* Overloading comparison operators */
	bool operator != (fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus const& b) const
	{
		return !(oFiMsg.FlashEventPlayState.enType == b.oFiMsg.FlashEventPlayState.enType);
	}

	bool operator == (fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus const& b) const
	{
		return (oFiMsg.FlashEventPlayState.enType == b.oFiMsg.FlashEventPlayState.enType);
	}
	midw_ext_sxm_audiofi_tclMsgTWNowPlaybackStatusStatus oFiMsg;
};

/* SportsFlash_PlaybackStatus notification*/
struct fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus
{
	fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus()
	{
		oFiMsg.FlashEventPlayState.enType = midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_UNKNOWN;
	}
	/* Overloading comparison operators */
	bool operator != (fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus const& b) const
	{
		return !(oFiMsg.FlashEventPlayState.enType == b.oFiMsg.FlashEventPlayState.enType);
	}

	bool operator == (fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus const& b) const
	{
		return (oFiMsg.FlashEventPlayState.enType == b.oFiMsg.FlashEventPlayState.enType);
	}
	midw_ext_sxm_audiofi_tclMsgSportsFlash_PlaybackStatusStatus oFiMsg;
};

/* Rapid Channel Browse Info*/
struct fc_sxm_tcl_trAudioPropertyRapidChannelBrowseInfo
{
	fc_sxm_tcl_trAudioPropertyRapidChannelBrowseInfo()
	{
		oFiMsg.ChannelInfo.ChannelNumber = CHANNEL_INVALID_ID;
		oFiMsg.ChannelInfo.SID = SERVICE_INVALID_ID;
		oFiMsg.ChannelInfo.IsAvailable = FALSE;
	}
	// Overloading comparison operators
	bool operator != (fc_sxm_tcl_trAudioPropertyRapidChannelBrowseInfo const& b) const
	{
		return !((oFiMsg.ChannelInfo.ChannelNumber == b.oFiMsg.ChannelInfo.ChannelNumber) &&
				(oFiMsg.ChannelInfo.SID == b.oFiMsg.ChannelInfo.SID));
	}

	// Overloading Assignment operators
	bool operator == (fc_sxm_tcl_trAudioPropertyRapidChannelBrowseInfo const& b) const
	{
		return ((oFiMsg.ChannelInfo.ChannelNumber == b.oFiMsg.ChannelInfo.ChannelNumber) &&
				(oFiMsg.ChannelInfo.SID == b.oFiMsg.ChannelInfo.SID));
	}
	midw_ext_sxm_audiofi_tclMsgRapidChannelBrowseInfoStatus oFiMsg;
};

/*
 * Message sent from SMS context containing tune update info
 */
struct fc_sxm_trMsgAudioSmsEvtTuneState
    : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio,
                    FC_SXM_AUDIO_ACTION_BASE_ADVISORY + 1);
    TUNE_STATE_ENUM eTuneState;
    CHANNEL_ID eTunedChannelID;
    CATEGORY_ID eTunedCategoryID;
};

/*
 * Audio property class that holds property update data
 */
class fc_sxm_tclAudioProperties:
    public fc_sxm_tclSingleton<fc_sxm_tclAudioProperties>
{
    /*=========================================================================
    =======                     private section                         =======
    =========================================================================*/
    private:

        /* Defining fc_sxm_tclSingleton as friend class */
        friend class fc_sxm_tclSingleton<fc_sxm_tclAudioProperties>;

    /*=========================================================================
    =======                     public section                          =======
    =========================================================================*/
    public:

        /* Channel list status property */
        fc_sxm_tclAutoProperty< 
            fc_sxm_tcl_trAudioPropertyChannelListStatus, 
            CCA_C_U16_SRV_SXM_AUDIO, 
            MIDW_EXT_SXM_AUDIOFI_C_U16_CHANNELLISTSTATUS> oChannelListStatus;

        /* Channel information property */
        fc_sxm_tclAutoProperty< 
            fc_sxm_tcl_trAudioPropertyChannelInformation, 
            CCA_C_U16_SRV_SXM_AUDIO, 
            MIDW_EXT_SXM_AUDIOFI_C_U16_CHANNELINFORMATION> oChannelInformation;

        /* Auto compare property */
        fc_sxm_tclAutoProperty< 
            fc_sxm_tcl_trAudioPropertyAutoCompare, 
            CCA_C_U16_SRV_SXM_AUDIO, 
            MIDW_EXT_SXM_AUDIOFI_C_U16_AUTOCOMPARE> oAutoCompare;

        /* Auto compare property */
        fc_sxm_tclAutoProperty< 
            fc_sxm_tcl_trAudioPropertySystemMessage, 
            CCA_C_U16_SRV_SXM_AUDIO, 
            MIDW_EXT_SXM_AUDIOFI_C_U16_SYSTEMMESSAGE> oSystemMessage;

        /* Category list status property */
        fc_sxm_tclAutoProperty< 
            fc_sxm_tcl_trAudioPropertyCategoryListStatus, 
            CCA_C_U16_SRV_SXM_AUDIO, 
            MIDW_EXT_SXM_AUDIOFI_C_U16_CATEGORYLISTSTATUS> oCategoryListStatus;

        /* Audio service subscription status property */
        fc_sxm_tclAutoProperty< 
            fc_sxm_tcl_trAudioPropertyAudioSubStatus, 
            CCA_C_U16_SRV_SXM_AUDIO, 
            MIDW_EXT_SXM_AUDIOFI_C_U16_SXMAUDIOSUBINFO> oAudioSubscriptionStatus;

        /* Instant Replay status property */
        fc_sxm_tclAutoProperty<
            fc_sxm_tcl_trAudioPropertyInstantReplayStatus,
            CCA_C_U16_SRV_SXM_AUDIO,
            MIDW_EXT_SXM_AUDIOFI_C_U16_INSTANTREPLAYSTATUS> oInstantReplayStatus;

        /* Instant Replay mode property */
        fc_sxm_tclAutoProperty<
            fc_sxm_tcl_trAudioPropertyIRMode,
            CCA_C_U16_SRV_SXM_AUDIO,
            MIDW_EXT_SXM_AUDIOFI_C_U16_INSTANTREPLAYMODE> oInstantReplayMode;

        /* Data services antenna status property */
        fc_sxm_tclAutoProperty<
            fc_sxm_tcl_trDSAntennaStatus,
            CCA_C_U16_SRV_SXM_AUDIO,
            MIDW_EXT_SXM_AUDIOFI_C_U16_SXMDATASERVICEANTENNASTATUS> oDSAntennaStatus;

        /* PresetChannelUpdate status property */
        fc_sxm_tclAutoProperty<
            fc_sxm_tcl_trAudioPropertyPresetChannelUpdate,
            CCA_C_U16_SRV_SXM_AUDIO,
            MIDW_EXT_SXM_AUDIOFI_C_U16_PRESETCHANNELUPDATE> oPresetInfo;
			
		  /* TuneScanContent status property */
		  fc_sxm_tclAutoProperty<
			fc_sxm_tcl_trAudioPropertyTuneScanContentStatus,
			CCA_C_U16_SRV_SXM_AUDIO,
			MIDW_EXT_SXM_AUDIOFI_C_U16_TUNESCANCONTENTSTATUS> oTuneScanContentStatus;

         /* Favorite alert info property */
        fc_sxm_tclAutoProperty<
         fc_sxm_tcl_trFavoriteContentAlertInfo,
         CCA_C_U16_SRV_SXM_AUDIO,
         MIDW_EXT_SXM_AUDIOFI_C_U16_FAVORITECONTENTALERTINFO> oAlertInfo;

        /* Favorite alert notification list update property */
        fc_sxm_tclAutoProperty<
           fc_sxm_tcl_trAlertNoticeListUpdate,
           CCA_C_U16_SRV_SXM_AUDIO,
           MIDW_EXT_SXM_AUDIOFI_C_U16_ALERTNOTICELISTUPDATE> oAlertNoticListUpdate;

        /* TWNowService status property */
        fc_sxm_tclAutoProperty<
        	fc_sxm_tcl_trAudioPropertyTWNowServiceStatus,
            CCA_C_U16_SRV_SXM_AUDIO,
            MIDW_EXT_SXM_AUDIOFI_C_U16_TWNOWSERVICESTATUS> oTWNowServiceStatus;

        /* TWNowBulletin status property */
        fc_sxm_tclAutoProperty<
        	fc_sxm_tcl_trAudioPropertyTWNowBulletinStatus,
            CCA_C_U16_SRV_SXM_AUDIO,
            MIDW_EXT_SXM_AUDIOFI_C_U16_TWNOWBULLETINSTATUS> oTWNowBulletinStatus;

        /* Sports Flash Service status property */
		fc_sxm_tclAutoProperty<
			fc_sxm_tcl_trAudioPropertySportsFlash_ServiceStatus,
			CCA_C_U16_SRV_SXM_AUDIO,
			MIDW_EXT_SXM_AUDIOFI_C_U16_SPORTSFLASH_SERVICESTATUS> oSportsFlashServiceStatus;

		/* Sports Flash GameEvent Notification */
		fc_sxm_tclAutoProperty<
			fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification,
			CCA_C_U16_SRV_SXM_AUDIO,
			MIDW_EXT_SXM_AUDIOFI_C_U16_SPORTSFLASH_GAMEEVENTNOTIFICATION> oSportsFlashGameEvent;

		/* Sports Flash FlashEvent Notification */
		fc_sxm_tclAutoProperty<
			fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification,
			CCA_C_U16_SRV_SXM_AUDIO,
			MIDW_EXT_SXM_AUDIOFI_C_U16_SPORTSFLASH_FLASHEVENTNOTIFICATION> oSportsFlashFlashEvent;

		/* TWNow PlaybackStatus Notification */
		fc_sxm_tclAutoProperty<
			fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus,
			CCA_C_U16_SRV_SXM_AUDIO,
			MIDW_EXT_SXM_AUDIOFI_C_U16_TWNOWPLAYBACKSTATUS> oTWNowPlaybackStatus;

		/* Sports Flash PlaybackStatus Notification */
		fc_sxm_tclAutoProperty<
			fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus,
			CCA_C_U16_SRV_SXM_AUDIO,
			MIDW_EXT_SXM_AUDIOFI_C_U16_SPORTSFLASH_PLAYBACKSTATUS> oSportsFlashPlaybackStatus;

		/* Rapid Channel Browse Info Notification*/
		fc_sxm_tclAutoProperty<
				fc_sxm_tcl_trAudioPropertyRapidChannelBrowseInfo,
				CCA_C_U16_SRV_SXM_AUDIO,
				MIDW_EXT_SXM_AUDIOFI_C_U16_RAPIDCHANNELBROWSEINFO> oRapidChannelBrowseInfo;

		/* PresetChannelUpdate status property */
		fc_sxm_tclAutoProperty<
			fc_sxm_tcl_trAudioPropertyPresetListUpdate,
			CCA_C_U16_SRV_SXM_AUDIO,
			MIDW_EXT_SXM_AUDIOFI_C_U16_PRESETLISTUPDATE> oPresetListUpdate;

		/* SetClockParams status property */
		fc_sxm_tclAutoProperty<
		fc_sxm_tcl_trAudioPropertySetClockParams,
		CCA_C_U16_SRV_SXM_AUDIO,
		MIDW_EXT_SXM_AUDIOFI_C_U16_SETCLOCKPARAMS> oClockParmsUpdate;

};

#endif
