/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_audio_app.cpp
* @brief       Application handler implementation for Audio service. 
*              Maintains ownership of all helper classes that implements different 
*                   functionalities of SXM Audio.
*              File redirects all messages posted from Audio Service to appropriate helper classes.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_sms_util.h"

#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_audiorouting_lib.h"

#include "fc_sxm_tcl_diaghandler.h"
#include "fc_sxm_tcl_diagtm_handler.h"
#include "fc_sxm_tcl_diag_download.h"

#include "fc_sxm_tcl_states.h"
#include "fc_sxm_tcl_channel_list.h"
#include "fc_sxm_tcl_category_list.h"
#include "fc_sxm_tcl_presets.h"
#include "fc_sxm_tcl_advisories.h"
#include "fc_sxm_tcl_playback.h"
#include "fc_sxm_tcl_content_alerts.h"
#include "fc_sxm_tcl_content_alert_proxy.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_tcl_twnow.h"
#include "fc_sxm_tcl_twnow_proxy.h"
#include "fc_sxm_tcl_sports_flash.h"
#include "fc_sxm_tcl_sports_flash_proxy.h"
#include "fc_sxm_tcl_audio_app.h"
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
#include "fc_sxm_tcl_profilemanager.h"
#endif

typedef enum 
{
    /* Channel list method */
    fc_sxm_enTtfisCmdsAudio_GET_CHANNEL_LIST                 =0x01,
    /* Channel list property */
    fc_sxm_enTtfisCmdsAudio_GET_CHANNEL_LIST_STATUS          =0x02,
    /* Select channel method */
    fc_sxm_enTtfisCmdsAudio_SELECT_CHANNEL                   =0x03,
    /* Select channel with dwell time method */
    fc_sxm_enTtfisCmdsAudio_SELECT_CHANNEL_WITH_DWELL_TIME   =0x04,
    /* Channel information property */
    fc_sxm_enTtfisCmdsAudio_GET_CHANNEL_INFORMATION_STATUS   =0x05,
    /* Category list method */
    fc_sxm_enTtfisCmdsAudio_GET_CATEGORY_LIST                =0x06,
    /* Recall preset method */
    fc_sxm_enTtfisCmdsAudio_RECALL_PRESET                    =0x07,
    /* Store preset method */
    fc_sxm_enTtfisCmdsAudio_STORE_PRESET                     =0x08,
    /* Set auto compare method */
    fc_sxm_enTtfisCmdsAudio_SET_AUTO_COMPARE                 =0x09,
    /* Auto compare property */
    fc_sxm_enTtfisCmdsAudio_AUTO_COMPARE_STATUS              =0x0a,
    /* System message property */
    fc_sxm_enTtfisCmdsAudio_SYSTEM_MESSAGE                   =0x0b,
    /* Set destination position method */
    fc_sxm_enTtfisCmdsAudio_SET_DESTINATION_POSITION         =0x0c,
    /* Set cvm-event */
    fc_sxm_enTtfisCmdsAudio_CVM_EVENT                        =0x0d,
    /* Playback method to play sxm audio */
    fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_PLAYBACK               =0x0e,
    /* Tune start method */
    fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TUNE_START             =0x0f,
    /* Smart favorite list method */
    fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_LIST_CONTROL        =0x10,
    /* Set current volume */
    fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_CURRENT_VOLUME         =0x11,
    /* Set sxm module to factory default settings */
    fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_MODULE_RESET           =0x12,
    /* Test tone */
    fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TEST_TONE              =0x13,
    /* Tune Scan */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TUNE_SCAN              =0x14,
	/* GetPresetList */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_GET_PRESET_LIST        =0x15,
	/* Disable Mature Channels */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_DISABLE_MATURE         =0x16,
	/* Skip/Unskip Channel */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SKIP_CHANNEL           =0x17,
    /* Save favorite */
    fc_sxm_enTtfisCmdsAudio_SAVE_FAVORITE                    =0x18,
    /* Get favorite list */
    fc_sxm_enTtfisCmdsAudio_GET_FAVORITE_LIST                =0x19,
    /* Modify favorite to enable/disable, delete selected favorite,
     * block/unblock selected favorite type,
     * enable/disable, delete all favorite of selected type */
    fc_sxm_enTtfisCmdsAudio_MODIFY_SELECTED_FAVORITE         =0x1a,
    fc_sxm_enTtfisCmdsAudio_MODIFY_ALL_FAVORITE              =0x1b,
	/* TWNow SetMarket */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_SETMARKET         =0x1c,
	/* TWNow GetActiveMarket */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_GETACTIVEMARKET   =0x1d,
	/* TWNow BulletinPlayback */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_BULLETINPLAYBACK  =0x1e,
	/* TWNow GetBulletinID */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_GETBULLETINID     =0x1f,
	/* TWNow GetAvailable Markets */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_GETAVLMARKETS     =0x20,
	/* TWNow Enable */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_ENABLE            =0x21,
	/* SportsFlash Enable */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_ENABLE               =0x22,
	/* SportsFlash Add Favorite Team */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_ADDFAVORITE          =0x23,
	/* SportsFlash Remove Favorite  */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_REMOVEFAVORITE       =0x24,
	/* SportsFlash Remove All Favorites */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_REMOVEALLFAVORITES   =0x25,
	/* SportsFlash Get All Favorite Teams */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETALLFAVORITES      =0x26,
	/* SportsFlash Get Current Games involving favorite teams*/
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETCURRENTGAMES      =0x27,
	/* SportsFlash Set games Monitor */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_SETGAMESMONITOR      =0x28,
	/* SportsFlash Get current flash events */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETFLASHEVENTS       =0x29,
	/* SportsFlash PLAY FLASH EVENT */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_PLAYFLASHEVENT       =0x2a,
	/* SportsFlash Abort Flash Event */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_ABORTFLASHEVENT      =0x2b,
	/* SportsFlash Get Current playing event's flash ID */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_CURRENTFLASHEVENT    =0x2c,
	/* SportsFlash Get Flash Enabled Leagues */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETLEAGUES           =0x2d,
	/* SportsFlash Get Teams from a League */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETTEAMS             =0x2e,
	/* SportsFlash Get Teams from a League */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_MONITORTEAM          =0x2f,
	/* for Clock parameters */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_CLOCK_PARAMS            =0x30,
	/* Set Category Monitor */
	fc_sxm_enTtfisCmdsAudio_SET_CATEGORY_MONITOR              =0x31,
	/* RapidChannelBrowse Start */
	fc_sxm_enTtfisCmdsAudio_RAPIDCHANNELBROWSE_START          =0x32,
	/* RapidChannelBrowse Stop */
	fc_sxm_enTtfisCmdsAudio_RAPIDCHANNELBROWSE_STOP           =0x33,
	/* UserProfile */
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_USER_PROFILE            =0x34,
	/*Get preset configuration*/
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_GET_PRESETS_CONFIG	  =0x35,
	/*Set preset list*/
	fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SET_PRESET_LIST		  =0x36,
    /* TTFis audio max value */
    fc_sxm_enTtfisCmdsAudio_MAX
} fc_sxm_tenTtfisCmdsAudio;
// note: commands for diag start from 0xF0

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AUDIO_APP
#include "trcGenProj/Header/fc_sxm_tcl_audio_app.cpp.trc.h"
#endif

/* 
constructor: give it a name and provice the used sxmServiceId (not the ccaServiceId!!) 
*/
fc_sxm_tclAudioApp::fc_sxm_tclAudioApp()
    : fc_sxm_tclBaseApp("sxmAudioApp", _enStaticServiceId, 0, TR_CLASS_FC_SXM_AUDIO_APP)
, _hDecoder(DECODER_INVALID_OBJECT)
, _bIsSMSCfgError(FALSE)
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp constructor"));
}

/*
Destructor
*/
fc_sxm_tclAudioApp::~fc_sxm_tclAudioApp()
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp destructor"));

    _hDecoder = DECODER_INVALID_OBJECT;
    _bIsSMSCfgError = FALSE;
}

/* Attention: called in cca-context by sxmapp-manager */
tVoid fc_sxm_tclAudioApp::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vInitialize"));

    /* call vInitialize of base-class to start the worker thread */
    fc_sxm_tclBaseApp::vInitialize();


    /*Diagnosis runs on Audio thread.Hence Responsibility
    to create and start it is given to Audio app */
    /* Initialize all the helper class for sxm audio here */
    _oMemberList.poAdd(fc_sxm_tclSystemStates::instance());

    _oMemberList.poAdd(fc_sxm_tclDiagHandler::instance());
    _oMemberList.poAdd(fc_sxm_tclChannelList::instance());
    _oMemberList.poAdd(fc_sxm_tclCategoryList::instance());
    _oMemberList.poAdd(fc_sxm_tclPresets::instance());
    _oMemberList.poAdd(fc_sxm_tclAdvisories::instance());
    _oMemberList.poAdd(fc_sxm_tclPlayback::instance());
    _oMemberList.poAdd(fc_sxm_tclContentAlert::instance());
    _oMemberList.poAdd(fc_sxm_tclTWNow::instance());
    _oMemberList.poAdd(fc_sxm_tclSportsFlash::instance());

    fc_sxm_tclSystemStates::instance()->vInit();

    fc_sxm_tclChannelList::instance()->vSetApp(this);
    fc_sxm_tclCategoryList::instance()->vSetApp(this);
    fc_sxm_tclPresets::instance()->vSetApp(this);
    fc_sxm_tclAdvisories::instance()->vSetApp(this);
    fc_sxm_tclPlayback::instance()->vSetApp(this);
    fc_sxm_tclContentAlert::instance()->vSetApp(this, fc_sxm_tclContentAlertProxy::instance());
    fc_sxm_tclTWNow::instance()->vSetApp(this, fc_sxm_tclTWNowProxy::instance());
    fc_sxm_tclSportsFlash::instance()->vSetApp(this, fc_sxm_tclSportsFlashProxy::instance());

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
//Register for Personalization events
    fc_sxm_tcl_profilemanager::instance()->vRegister(this);
#endif
}

/* Attention: called in cca-context by sxmapp-manager */
tVoid fc_sxm_tclAudioApp::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vDeInitialize"));
    /* call vDeInitialize() of base-class to stop the worker thread */
    fc_sxm_tclBaseApp::vDeInitialize();
   
    /* DeInitialize all the helper classes */
    _oMemberList.vClear();

    fc_sxm_tclContentAlertProxy* poContentAlertProxy = fc_sxm_tclContentAlertProxy::instance();
    FC_SXM_MEMORY_CLEAR(poContentAlertProxy)

    fc_sxm_tclTWNowProxy* poTWNowProxy = fc_sxm_tclTWNowProxy::instance();
    FC_SXM_MEMORY_CLEAR(poTWNowProxy)

    fc_sxm_tclSportsFlashProxy* poSportsFlashProxy = fc_sxm_tclSportsFlashProxy::instance();
    FC_SXM_MEMORY_CLEAR(poSportsFlashProxy)

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    //Deregister for Personalization events
        fc_sxm_tcl_profilemanager::instance()->vDeRegister(this);
#endif
}

// Get the decoder object
DECODER_OBJECT fc_sxm_tclAudioApp::oGetDecoderObject(tVoid) const
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::oGetDecoderObject()"));
    //return(_hDecoder);
    return(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder());
}

/* 
initialization of audio in sms is now handled in worker-thread-context 
*/
tVoid fc_sxm_tclAudioApp::vStop() const {
    /* Do neccessary action before stopping decoder
       when everything is done, send ack.
       this can also by postponed to a later point in time.
    */
    fc_sxm_tclChannelList::instance()->vClearMemberData();
    fc_sxm_tclCategoryList::instance()->vClearMemberData();
    fc_sxm_tclAdvisories::instance()->vClearMemberData();
    fc_sxm_tclPresets::instance()->vClearMemberData();
    fc_sxm_tclPlayback::instance()->vClearMemberData();
    fc_sxm_tclContentAlert::instance()->vClearMemberData();
    fc_sxm_tclTWNow::instance()->vClearMemberData();
    fc_sxm_tclSportsFlash::instance()->vClearMemberData();

    //Moved to audio proxy
    //vAllowService(FALSE, CCA_C_U16_SRV_SXM_AUDIO);
    fc_sxm_tclAudioProxy::instance()->vSetAudioStopped();
}
tVoid fc_sxm_tclAudioApp::vSetDecoder(DECODER_OBJECT hDecoder) {
    _hDecoder=hDecoder;
    tBool bDecoderReady=(_hDecoder != DECODER_INVALID_OBJECT);
    //Moved to audio proxy
    //vAllowService(bDecoderReady, CCA_C_U16_SRV_SXM_AUDIO);

    DECODER_STATE_ENUM hDecoderState = DECODER_STATE_INVALID;
    if(bDecoderReady) {
       hDecoderState = DECODER_STATE_READY;
    }

    /* Update the decoder ready state to audio helper classes */
    _oMemberList.vUpdateDecoderState(_hDecoder, hDecoderState);
}
/*
  Handle the ttfis commands for audio
  */
tVoid fc_sxm_tclAudioApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
{
    /* get msg code */
    fc_sxm_tenTtfisCmdsAudio enMsgCode=(fc_sxm_tenTtfisCmdsAudio)prMsg->u8MsgCode;
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgCmdAppTtfisCmd) code=%d",
        ETG_CENUM(fc_sxm_tenTtfisCmdsAudio, enMsgCode)));

    tU8 u8Param1 = prMsg->au8Data[0];
    tU8 u8Param2 = prMsg->au8Data[1];
    tU8 u8Param3 = prMsg->au8Data[2];
    tU8 u8Param4 = prMsg->au8Data[3];
    tU8 u8Param5 = prMsg->au8Data[4];

    switch (enMsgCode) 
    {
        /* Channel list method */
    case fc_sxm_enTtfisCmdsAudio_GET_CHANNEL_LIST:
        {
            midw_ext_sxm_audiofi_tclMsgGetChannelListMethodStart oMStart;
            oMStart.Index  = u8Param1;
            oMStart.Range  = u8Param2;
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

        /* Channel list property */
    case fc_sxm_enTtfisCmdsAudio_GET_CHANNEL_LIST_STATUS:
        {
            midw_ext_sxm_audiofi_tclMsgChannelListStatusGet oPGet;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        /* Select channel method */
    case fc_sxm_enTtfisCmdsAudio_SELECT_CHANNEL:
        {
            midw_ext_fi_tcl_e8_SelectChannelMode ChannelSelectMode;
            /* Select channel with dwell time method */
            switch(u8Param1)
            {
            case 1:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_SID           ; break;
            case 2:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CID           ; break;
            case 3:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CHANNEL       ; break;
            case 4:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CHANNEL_UP    ; break;
            case 5:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CHANNEL_DOWN  ; break;
            case 6:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_PROGRAM_UP    ; break;
            case 7:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_PROGRAM_DOWN  ; break;
            default: ChannelSelectMode.enType = (midw_ext_fi_tcl_e8_SelectChannelMode::tenType)u8Param1                ; break;
            }

            midw_ext_sxm_audiofi_tclMsgSelectChannelMethodStart oMStart;
            oMStart.SelectionMode = ChannelSelectMode;
            oMStart.CidOrSid      = u8Param2;
            oMStart.CategoryType  = u8Param3;
            oMStart.Steps         = u8Param4;
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

        /* Select channel with dwell time method */
    case fc_sxm_enTtfisCmdsAudio_SELECT_CHANNEL_WITH_DWELL_TIME:
        {
            midw_ext_fi_tcl_e8_SelectChannelMode ChannelSelectMode;
            /* based on select channel mode */
            switch(u8Param1)
            {
            case 1:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_SID           ; break;
            case 2:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CID           ; break;
            case 3:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CHANNEL       ; break;
            case 4:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CHANNEL_UP    ; break;
            case 5:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_CHANNEL_DOWN  ; break;
            case 6:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_PROGRAM_UP    ; break;
            case 7:  ChannelSelectMode.enType = midw_ext_fi_tcl_e8_SelectChannelMode::FI_EN_XMTUN_SELECT_PROGRAM_DOWN  ; break;
            default: ChannelSelectMode.enType = (midw_ext_fi_tcl_e8_SelectChannelMode::tenType)u8Param1                ; break;
            }

            midw_ext_sxm_audiofi_tclMsgSelectChannelWithDwellTimeMethodStart oMStart;
            oMStart.SelectionMode = ChannelSelectMode;
            oMStart.CidOrSid      = u8Param2;
            oMStart.CategoryType  = u8Param3;
            oMStart.Steps         = u8Param4;
            oMStart.DwellTime     = (tU16)(u8Param5 * 1000);
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

        /* Channel information property */
    case fc_sxm_enTtfisCmdsAudio_GET_CHANNEL_INFORMATION_STATUS:
        {
            midw_ext_sxm_audiofi_tclMsgChannelInformationGet oPGet;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        /* Category list method */
    case fc_sxm_enTtfisCmdsAudio_GET_CATEGORY_LIST:
        {
            midw_ext_sxm_audiofi_tclMsgGetCategoryListMethodStart oPGet;
            oPGet.CategoryType = u8Param1;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        /* Recall preset method */
    case fc_sxm_enTtfisCmdsAudio_RECALL_PRESET:
        {
            midw_ext_sxm_audiofi_tclMsgRecallPresetMethodStart oPGet;
            oPGet.PresetElement = u8Param1;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        /* Store preset method */
    case fc_sxm_enTtfisCmdsAudio_STORE_PRESET:
        {
            midw_ext_sxm_audiofi_tclMsgStorePresetMethodStart oPGet;
            oPGet.PresetElement = u8Param1;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        /* Set auto compare method */
    case fc_sxm_enTtfisCmdsAudio_SET_AUTO_COMPARE:
        {
            midw_ext_sxm_audiofi_tclMsgSetAutoCompareParametersMethodStart oPGet;
            oPGet.u8FirstElement = u8Param1;
            oPGet.u8LastElement  = u8Param2;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        /* Auto compare property */
    case fc_sxm_enTtfisCmdsAudio_AUTO_COMPARE_STATUS:
        {
            midw_ext_sxm_audiofi_tclMsgAutoCompareGet oPGet;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        /* System message property */
    case fc_sxm_enTtfisCmdsAudio_SYSTEM_MESSAGE:
        {
            midw_ext_sxm_audiofi_tclMsgSystemMessageGet oPGet;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;
            /* set destination */
    case fc_sxm_enTtfisCmdsAudio_SET_DESTINATION_POSITION:
        {
            midw_ext_sxm_audiofi_tclMsgSetDestinationPositionMethodStart oPGet;
            oPGet.s32Latitude  = u8Param1;
            oPGet.s32Longitude = u8Param2;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_CVM_EVENT:
        {
            fc_sxm_trMsgAudioCvmEvent oMsg;
#ifdef VARIANT_S_FTR_ENABLE_SPM_CORE_FI
			oMsg.oFiMsg.CvmEvent.enType=(spm_fi_tcl_SPM_e32_CVM_EVENT::tenType)u8Param1;
#else
            oMsg.oFiMsg.CvmEvent.enType=(cfc_fi_tcl_SPM_e32_CVM_EVENT::tenType)u8Param1;
#endif
            fc_sxm_tclAppManager::instance()->vPostMsgNew(oMsg);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_PLAYBACK:
        {
           midw_ext_sxm_audiofi_tclMsgInstantReplayControlMethodStart oPGet;
           ((oPGet).u8IRPlaybackMode).enType = (midw_ext_fi_tcl_e8_IRPlaybackMode::tenType)u8Param1;
           fc_sxm_vSendSelfMessage(oPGet);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TUNE_START:
        {
           midw_ext_sxm_audiofi_tclMsgEnableTuneStartMethodStart oPGet;
           oPGet.bEnableTuneStart = (tBool)u8Param1;
           fc_sxm_vSendSelfMessage(oPGet);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_LIST_CONTROL:
        {
           midw_ext_sxm_audiofi_tclMsgSmartFavoriteListControlMethodStart oPGet;
           ((oPGet).Operation).enType = (midw_ext_fi_tcl_e8_SFOperation::tenType) u8Param1;
           oPGet.ListIndex = u8Param2;
           fc_sxm_vSendSelfMessage(oPGet);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_CURRENT_VOLUME:
        {
           midw_ext_sxm_audiofi_tclMsgSetCurrentVolumeMethodStart oPGet;
           (oPGet).IsMuteActive = u8Param1;
           fc_sxm_vSendSelfMessage(oPGet);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_MODULE_RESET:
        {
        	midw_ext_sxm_audiofi_tclMsgResetSXMModuleMethodStart oMStart;
        	fc_sxm_vSendSelfMessage (oMStart);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TEST_TONE:
        {
           fc_sxm_trMsgAudioTestToneEvent oMsg;
           oMsg.u8EnableTone = u8Param1;
           vPostMsgNew(oMsg);
        }
        break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TUNE_SCAN:
        {
			midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodStart oPGet;
			(oPGet.TuneScanMode).enType = (midw_ext_fi_tcl_e8_TuneScanMode::tenType) u8Param1;
			fc_sxm_vSendSelfMessage(oPGet);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_GET_PRESET_LIST:
        {
        	// GetPresetList
        	midw_ext_sxm_audiofi_tclMsgGetPresetListMethodStart oMStart;
        	oMStart.PresetBand = u8Param1;
            fc_sxm_vSendSelfMessage (oMStart);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_DISABLE_MATURE:
        {
        	// Disable Mature Channels
        	midw_ext_sxm_audiofi_tclMsgDisableMatureChannelsMethodStart oMStart;
        	oMStart.Status = u8Param1;
            fc_sxm_vSendSelfMessage (oMStart);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SKIP_CHANNEL:
        {
        	// Skip/Unskip Channels
        	midw_ext_sxm_audiofi_tclMsgSkipChannelMethodStart oMStart;
        	oMStart.SkipChannelType.enType = static_cast <midw_ext_fi_tcl_e8_ChannelSkipType::tenType> (u8Param1);
        	oMStart.ServiceID = u8Param2;
        	oMStart.ChannelID = u8Param3;
        	fc_sxm_vSendSelfMessage (oMStart);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SAVE_FAVORITE:
       {
          midw_ext_sxm_audiofi_tclMsgSaveFavoriteMethodStart prSaveFavoriteMessage;
          prSaveFavoriteMessage.ServiceID = (tU16)SXM_GET_U16(&prMsg->au8Data[0]);
          prSaveFavoriteMessage.ChannelID = (tU16)SXM_GET_U16(&prMsg->au8Data[2]);
          prSaveFavoriteMessage.FavoriteType.enType = midw_ext_fi_tcl_e8_FavoriteType::tenType(prMsg->au8Data[4]);

          prSaveFavoriteMessage.MaxFavoriteCount = prMsg->au8Data[5];
          fc_sxm_vSendSelfMessage(prSaveFavoriteMessage);
       }
       break;
    case fc_sxm_enTtfisCmdsAudio_GET_FAVORITE_LIST:
       {
          midw_ext_sxm_audiofi_tclMsgGetFavoriteListMethodStart prGetFavoriteListMessage;
          prGetFavoriteListMessage.FavoriteType.enType = midw_ext_fi_tcl_e8_FavoriteType::tenType(prMsg->au8Data[0]);
          fc_sxm_vSendSelfMessage(prGetFavoriteListMessage);
       }
       break;
    case fc_sxm_enTtfisCmdsAudio_MODIFY_SELECTED_FAVORITE:
       {
          midw_ext_sxm_audiofi_tclMsgModifyFavoriteMethodStart prModifyFavoriteMessage;
          prModifyFavoriteMessage.FavoriteType.enType = midw_ext_fi_tcl_e8_FavoriteType::tenType(prMsg->au8Data[0]);
          prModifyFavoriteMessage.Action.enType  = midw_ext_fi_tcl_e8_Action::tenType(prMsg->au8Data[1]);
          vConvertTTFisString2FIString((tCString)&prMsg->au8Data[2] , prModifyFavoriteMessage.ContentName);
          fc_sxm_vSendSelfMessage(prModifyFavoriteMessage);
       }
       break;
    case fc_sxm_enTtfisCmdsAudio_MODIFY_ALL_FAVORITE:
       {
          midw_ext_sxm_audiofi_tclMsgModifyFavoriteMethodStart prModifyFavoriteMessage;
          prModifyFavoriteMessage.FavoriteType.enType = midw_ext_fi_tcl_e8_FavoriteType::tenType(prMsg->au8Data[0]);
          prModifyFavoriteMessage.Action.enType = midw_ext_fi_tcl_e8_Action::tenType(prMsg->au8Data[1]);

          string sContentName = "";
          fc_sxm_vString2Fi(sContentName.c_str(), prModifyFavoriteMessage.ContentName);
          fc_sxm_vSendSelfMessage(prModifyFavoriteMessage);
       }
       break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_SETMARKET:
		{
			// TWNow SetMarket
			midw_ext_sxm_audiofi_tclMsgTWNowSetMarketMethodStart oMStart;
			oMStart.Status = u8Param1;
			vConvertTTFisString2FIString((tCString)&prMsg->au8Data[1] , oMStart.Market);
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_GETACTIVEMARKET:
		{
			// TWNow GetActiveMarket
			midw_ext_sxm_audiofi_tclMsgTWNowGetActiveMarketMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_BULLETINPLAYBACK:
		{
			// TWNow BulletinPlayback
			midw_ext_sxm_audiofi_tclMsgTWNowBulletinPlaybackMethodStart oMStart;
			oMStart.BulletinID = u8Param1;
			oMStart.Action.enType =
					static_cast <midw_ext_fi_tcl_e8_TWNow_Bulletin_PlayBackAction::tenType>
			         (u8Param2);
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_GETBULLETINID:
		{
			// TWNow GetBulletinID
			midw_ext_sxm_audiofi_tclMsgTWNowGetBulletinIDMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_GETAVLMARKETS:
		{
			// TWNow GetAvailable Markets
			midw_ext_sxm_audiofi_tclMsgTWNowGetAvailableMarketsMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_TWNOW_ENABLE:
		{
			// TWNow Enable
			midw_ext_sxm_audiofi_tclMsgTWNowEnableMethodStart oMStart;
			oMStart.Status = u8Param1;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;	   
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_ENABLE:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashEnableNotificationMethodStart oMStart;
			oMStart.Enable = u8Param1;;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_ADDFAVORITE:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashAddFavoriteTeamMethodStart oMStart;
			oMStart.SportType.enType =
					static_cast <midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType> (u8Param1);
			oMStart.LeagueID = u8Param2;
			oMStart.TeamID = u8Param3;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_REMOVEFAVORITE:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashRemoveFavoriteTeamMethodStart oMStart;
			oMStart.SportType.enType =
					static_cast <midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType> (u8Param1);
			oMStart.LeagueID = u8Param2;
			oMStart.TeamID = u8Param3;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_REMOVEALLFAVORITES:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashRemoveAllFavoriteTeamsMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETALLFAVORITES:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashGetAllFavoritesMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETCURRENTGAMES:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentGamesMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_SETGAMESMONITOR:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashSetGamesMonitorMethodStart oMStart;
			oMStart.ChannelIDs.push_back(u8Param1);
			oMStart.ChannelIDs.push_back(u8Param2);
			oMStart.ChannelIDs.push_back(u8Param3);
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETFLASHEVENTS:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentFlashEventsMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_PLAYFLASHEVENT:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashPlayFlashEventMethodStart oMStart;
			oMStart.FlashEventID = u8Param1;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_ABORTFLASHEVENT:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashAbortFlashEventMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_CURRENTFLASHEVENT:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentFlashEventIDMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETLEAGUES:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashEnabledLeaguesMethodStart oMStart;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_GETTEAMS:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashGetTeamsMethodStart oMStart;
			oMStart.SportType.enType =
					static_cast <midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType> (u8Param1);
			oMStart.LeagueID = u8Param2;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
	case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SF_MONITORTEAM:
		{
			midw_ext_sxm_audiofi_tclMsgSportsFlashMonitorTeamsMethodStart oMStart;
			midw_ext_fi_tcl_SportsFlash_MonitorTeam fiTeam;
			fiTeam.LeagueID = u8Param1;
			fiTeam.TeamID = u8Param2;
			fiTeam.Status = u8Param3;
			oMStart.FavoriteTeams.push_back(fiTeam);
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_CLOCK_PARAMS:
        {
            midw_ext_sxm_audiofi_tclMsgSetClockParamsSet oMes;
            oMes.DST = static_cast<tBool>(u8Param1);
            oMes.LocalTimeDate.u8Weekday = u8Param2;
            oMes.LocalTimeDate.u8Hours = u8Param3;
            oMes.LocalTimeDate.u8Minutes = u8Param4;
            vConvertTTFisString2FIString((tCString)&u8Param5 , oMes.TimeZone);
            fc_sxm_vSendSelfMessage (oMes);
        }
        break;
    case fc_sxm_enTtfisCmdsAudio_SET_CATEGORY_MONITOR:
		{
			midw_ext_sxm_audiofi_tclMsgSetCategoryMonitorMethodStart oMStart;
			oMStart.CategoryID = u8Param1;
			oMStart.ChannelEvents.u16Value = static_cast <tU16> (u8Param2);
			oMStart.MonitorBrowsableChannels = u8Param3;
			oMStart.EnableMonitoring = u8Param4;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
    case fc_sxm_enTtfisCmdsAudio_RAPIDCHANNELBROWSE_START:
		{
			midw_ext_sxm_audiofi_tclMsgRapidChannelBrowseStartMethodStart oMStart;
			oMStart.Direction.enType = static_cast<midw_ext_fi_tcl_e8_BrowseDirection::tenType> (u8Param1);
			vAssignBrowseRateConfig(oMStart.Config.Configuration, u8Param2);
			oMStart.Config.Loop.enType = static_cast<midw_ext_fi_tcl_e8_BrowseChannelLoop::tenType> (u8Param3);
			oMStart.Config.Type.enType = static_cast<midw_ext_fi_tcl_e8_BrowseChannelType::tenType> (u8Param4);
			oMStart.CategoryID = u8Param5;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
    case fc_sxm_enTtfisCmdsAudio_RAPIDCHANNELBROWSE_STOP:
		{
			midw_ext_sxm_audiofi_tclMsgRapidChannelBrowseStopMethodStart oMStart;
			oMStart.ChannelID = u8Param1;
			fc_sxm_vSendSelfMessage (oMStart);
		}
		break;
    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_USER_PROFILE:
        {
        	// User profile
        	midw_ext_sxm_audiofi_tclMsgUserProfileMethodStart oMStart;
        	oMStart.ProfileAction.enType = static_cast<midw_ext_fi_tcl_ProfileAction::tenType> (u8Param1);
        	oMStart.ProfileStatus.enType = static_cast<midw_ext_fi_tcl_ProfileStatus::tenType> (u8Param2);
        	oMStart.SourceID = u8Param3;
        	oMStart.DestinationID = u8Param4;
            fc_sxm_vSendSelfMessage (oMStart);
        }
        break;

    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_GET_PRESETS_CONFIG:
	{
    	//Get preset configuration
    	midw_ext_sxm_audiofi_tclMsgGetPresetsConfigMethodStart oMStart;
    	fc_sxm_vSendSelfMessage (oMStart);
	}
    break;

    case fc_sxm_enTtfisCmdsAudio_SXM_AUDIO_SET_PRESET_LIST:
    {
		//Set preset list
		midw_ext_sxm_audiofi_tclMsgSetPresetListMethodStart oMStart;
		tU8 u8Valid = u8Param1;

		midw_ext_fi_tcl_PresetEntry PresetList;

		for (tU8 u8LoopCount = 1; u8LoopCount <= 4; u8LoopCount++) {
			PresetList.Index = u8LoopCount;		//Presets for index 1,2,3,4
			PresetList.SID = u8LoopCount;		//SID 1,2,3,4 will be filled in index 1,2,3,4 respectively
			oMStart.PresetEntry.push_back(PresetList);
		}

		if (!u8Valid) {
			//for invalid preset list push index 0
			PresetList.Index = 0;
			PresetList.SID = 0;
			oMStart.PresetEntry.push_back(PresetList);
		}

		fc_sxm_vSendSelfMessage(oMStart);
	}
    break;

    case fc_sxm_enTtfisCmdsAudio_MAX:
        default:
        {
            tBool bDispatched=fc_sxm_tclDiagHandler::instance()->bDispatchMsg(prMsg);
            if (!bDispatched) {
                ETG_TRACE_ERR(("fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) unknown code=%d",
                               ETG_CENUM(fc_sxm_tenTtfisCmdsAudio, enMsgCode)));
            }
        }
        break;
    }
}

/* 
this method is called by the thread-function.
For each message that shall be handled put and entry here.
*/
tVoid fc_sxm_tclAudioApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg) {
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vDispatchMsgFromQ: SID=%x action=%d",
        ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()), 
        poThreadMsg->u16GetActionOnly()));
    tU32 u32Action=poThreadMsg->u32GetAction();
    switch (u32Action) {
        // use macro SXM_MSGQ_DISPATCH  for each message specific for audio
        /* next line to call vProcess(fc_sxm_trMsgAudioSmsEvtSrmState *) */
        SXM_MSGQ_DISPATCH_TO_2OBJ(fc_sxm_trMsgAudioSmsEvtAntennaQuality, 
                                  fc_sxm_tclAdvisories::instance(),
                                  fc_sxm_tclDiagHandler::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioDiagSetSXMAntennaDTC, fc_sxm_tclDiagHandler::instance());

        SXM_MSGQ_DISPATCH_TO_4OBJ(fc_sxm_trMsgAudioSmsEvtTuneState,
        		fc_sxm_tclAdvisories::instance(),
        		fc_sxm_tclTWNow::instance(),
        		fc_sxm_tclSportsFlash::instance(),
        		fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_3OBJ(fc_sxm_trMsgAudioSmsEvtChannelEvt, fc_sxm_tclAdvisories::instance(),
        		fc_sxm_tclSportsFlash::instance(),
        		fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_2OBJ(fc_sxm_trMsgAudioSmsEvtSignalState,
                                  fc_sxm_tclAdvisories::instance(),
                                  fc_sxm_tclDiagHandler::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMuteState, fc_sxm_tclPlayback::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgChannelEventCallback, fc_sxm_tclChannelList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgPresetBandUpdateCallback, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCategoryEventCallback, fc_sxm_tclCategoryList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSmsEvtPlaybackEvt, fc_sxm_tclPlayback::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSmsEvtTuneScanEvt, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioATAlerts, fc_sxm_tclContentAlert::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSeekCategoryUpdate, fc_sxm_tclContentAlert::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSeekAlertEvent, fc_sxm_tclContentAlert::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTWNowEventAlert, fc_sxm_tclTWNow::instance());

        /* method-start messages from cca-service */
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartGetChnList, fc_sxm_tclChannelList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSelChn, fc_sxm_tclChannelList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSelChnWitDwellTime, fc_sxm_tclChannelList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartGetCatList, fc_sxm_tclCategoryList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartRecallPreset, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartStorePreset, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSetAutoCompare, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartIRPlaybackControl, fc_sxm_tclPlayback::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTuneStart, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSFListControl, fc_sxm_tclPresets::instance());        
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartCurrentVolume, fc_sxm_tclPlayback::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartResetSxmModule, fc_sxm_tclSystemStates::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioTestToneEvent, fc_sxm_tclChannelList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartGetPresetList, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartDisableMatureChannels, fc_sxm_tclChannelList::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSkipChannels, fc_sxm_tclChannelList::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTuneScanControl, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSaveFavorite, fc_sxm_tclContentAlert::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartGetFavoriteList, fc_sxm_tclContentAlert::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartModifyFavorite, fc_sxm_tclContentAlert::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartGetAlertNoticeList, fc_sxm_tclContentAlert::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTWNowSetMarket, fc_sxm_tclTWNow::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTWNowGetActiveMarket, fc_sxm_tclTWNow::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTWNowBulletinPlayback, fc_sxm_tclTWNow::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTWNowGetBulletinID, fc_sxm_tclTWNow::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTWNowGetAvailableMarkets, fc_sxm_tclTWNow::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartTWNowEnable, fc_sxm_tclTWNow::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashEnableNotification, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashAddFavoriteTeam, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashRemoveFavoriteTeam, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashRemoveAllFavoriteTeams, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashGetAllFavorites, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashGetCurrentGames, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashSetGamesMonitor, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashGetCurrentFlashEvents, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashPlayFlashEvent, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashAbortFlashEvent, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashGetCurrentFlashEventID, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashEnabledLeagues, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashGetTeams, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSportsFlashMonitorTeam, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgSFGameAlert, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgSFFlashEventAlert, fc_sxm_tclSportsFlash::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartGetChnInfo, fc_sxm_tclChannelList::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSetCategoryMonitor, fc_sxm_tclChannelList::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartRapidChannelBrowseStart, fc_sxm_tclChannelList::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartRapidChannelBrowseStop, fc_sxm_tclChannelList::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartGetPresetsConfig, fc_sxm_tclPresets::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSetPresetList, fc_sxm_tclPresets::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartUserProfile, fc_sxm_tclPresets::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSMSCfgError, fc_sxm_tclSystemStates::instance());

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioProfileChange, this);
#endif

#if 0
        // if another member wants this message:
        SXM_MSGQ_DISPATCH_TO_2OBJ(fc_sxm_trMsgEvtAudioSmsModuleSubStatusChanged, 
                                  fc_sxm_tclDiagHandler::instance(), 
                                  fc_sxm_tclAdvisories::instance());
#endif
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgEvtAudioSmsModuleSubStatusChanged, 
                                  fc_sxm_tclDiagTmHandler::instance());

        // system states
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioCmdNewAppState, fc_sxm_tclSystemStates::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSmEvent, fc_sxm_tclSystemStates::instance()); 
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSmsEvtSrmState, fc_sxm_tclSmsSrm::instance()); 
        SXM_MSGQ_DISPATCH_TO_4OBJ(fc_sxm_trMsgAudioSmsEvtModuleState, 
                                  fc_sxm_tclSmsModule::instance(),
                                  fc_sxm_tclDiagTmHandler::instance(),
                                  fc_sxm_tclDiagHandler::instance(),
                                  fc_sxm_tclDiagDownload::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSmsEvtDecoderState, fc_sxm_tclSmsDecoder::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioDataStopped, fc_sxm_tclDataProxy::instance());
       
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSetDestinationPostion,fc_sxm_tclAudioApp::instance());
        
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioDiagAudioTestStart,fc_sxm_tclAudioApp::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioPropSetClockParams,fc_sxm_tclAudioApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioSmsSubscriptionUpdatedEvt, fc_sxm_tclAdvisories::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppSportsMetaDataUpdate, this);

        default:
            /* forward to diag */
            tBool bDispatched=fc_sxm_tclDiagHandler::instance()->bDispatchMsg(poThreadMsg);
            if (!bDispatched) {
                ETG_TRACE_ERR(("fc_sxm_tclAudioApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                               poThreadMsg->u32GetAction()));
            }
            break;         
    }
}





/* Process set destination position method start msg from client */
tVoid fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioMStartSetDestinationPostion const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioMStartSetDestinationPostion)"));

    //Send a message to the main application
    fc_sxm_trMsgAppMgrDestination rMsgSetDestinationPosition;
    rMsgSetDestinationPosition.rLocation.s32Latitude=fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(prMsg->oFiMsg.s32Latitude);
    rMsgSetDestinationPosition.rLocation.s32Longitude=fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(prMsg->oFiMsg.s32Longitude);

    fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsgSetDestinationPosition);

    midw_ext_sxm_audiofi_tclMsgSetDestinationPositionMethodResult ofiTxObj;
    ofiTxObj.Status = TRUE;

    /* Post the Method Result */
    if(AIL_EN_N_NO_ERROR == 
          fc_sxm_tclAudioService::instance()->enSendFiMessage(
                                                  prMsg->rAdressing, 
                                                  ofiTxObj))
    {
        ETG_TRACE_USR4((
            "Set destination method result sent successfully"));
    }
    else
    {
        ETG_TRACE_ERR((
            "Failed to send method result for set destination"));
    }
}


//Process to start the audio test trigerred from FC_Diagnosis
tVoid fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioDiagAudioTestStart const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioDiagAudioTestStart) Channel Id = %d",prMsg->u16ChannelId)); 

    //Do the handling here.
    /*implement the functionality here .Get the channel id from the strucutre fc_sxm_trMsgAudioDiagAudioTestStart
      Tune to the channel.Just print the result.I will take care of posting it to the diaglib */

    if(SMSAPI_RETURN_CODE_SUCCESS == DECODER.eTuneDirect(_hDecoder, prMsg->u16ChannelId, TRUE))
    {
        ETG_TRACE_USR4(("Channel tuned successfully"));
    }
    else
    {
        ETG_TRACE_ERR(("Failed to tune to specified channel"));
    }
    
}
/* Process Clock Params from client */
tVoid fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioPropSetClockParams const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioPropSetClockParams weekDay=%d",prMsg->oFiMsg.LocalTimeDate.u8Weekday));
    ETG_TRACE_USR4(("timezone=%s",prMsg->oFiMsg.TimeZone.szValue));
    ETG_TRACE_USR4(("dst=%d",prMsg->oFiMsg.DST));

    //Send a message to the main application
    fc_sxm_trMsgAppMgrClockParams rMsgClockParams;
    rMsgClockParams.strTimeZone = prMsg->oFiMsg.TimeZone;
    rMsgClockParams.bDSTStatus = prMsg->oFiMsg.DST;
    rMsgClockParams.oLocalTime = prMsg->oFiMsg.LocalTimeDate;

    fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsgClockParams);
    
}


//Here we print the report for fc_sxm status
tVoid fc_sxm_tclAudioApp::vPrintReport() {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "GENERAL:"));

    /* todo: store data for which we have notifications in protected data when we receive them.
       here only printing should be done 
       (to avoid duplicated code)
       add reason-text and phone-number for subscription
    */
    // SRM
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\tSRM:"));
    SRM_OBJECT hSRM=fc_sxm_tclSmsSrm::instance()->hGetSmsSrm();
    if (hSRM != SRM_INVALID_OBJECT)
    {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tState: %u",
                             ETG_CENUM(SRM_STATE_ENUM, SRM.eState(hSRM))));
    }
    else
    {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tState: SRM INVALID:"));
    }

    // Module
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\tMODULE:"));
    MODULE_OBJECT hModule=fc_sxm_tclSmsModule::instance()->hGetSmsModule();
    if (hModule != MODULE_INVALID_OBJECT)
    {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tState: %u",
                             ETG_CENUM(MODULE_STATE_ENUM, MODULE.eState(hModule))));
        MODULE_SUBSTATUS_ENUM eStatus;
        MODULE_SUBSTATUS_REASON_CODE u8ReasonCode;
        UN32 u32UTCSuspendDate;
        BOOLEAN bIsAudioSubscribed = FALSE;

        char szReasonText=0;
        char szPhoneNumber=0;
        STRING_OBJECT hReasonText=STRING.hCreate(&szReasonText, 100);
        STRING_OBJECT hPhoneNumber=STRING.hCreate(&szPhoneNumber, 100);


        MODULE.eSubscription(hModule,&eStatus,
                             &u8ReasonCode,&u32UTCSuspendDate,hReasonText,hPhoneNumber,&bIsAudioSubscribed);
		ETG_TRACE_USR4(("Subscription bIsAudioSubscribed: %d",bIsAudioSubscribed));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tSubscription:"));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tStatus: %d",
                        ETG_CENUM(MODULE_SUBSTATUS_ENUM, eStatus)));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tReason: 0x%02x", u8ReasonCode));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tUTCSuspendDate: 0x%08x", u32UTCSuspendDate));

        //cleanup
        STRING.vDestroy(hReasonText);
        STRING.vDestroy(hPhoneNumber);


    }
    else
    {
        //ETG_TRACE_ERR(("\tMODULE invalid"));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tState: MODULE INVALID:"));
        return;
    }

    // Decoder
    if (_hDecoder != DECODER_INVALID_OBJECT)
    {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\tDECODER:"));
        SIGNAL_QUALITY_STRUCT rSignalQuality;
        DECODER.eSignalQuality(_hDecoder,&rSignalQuality);
        
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tState: %d", 
                             ETG_CENUM(DECODER_STATE_ENUM, DECODER.eState(_hDecoder))
                             ));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tSignal Quality:"));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\teState: %d",
                             ETG_CENUM(SIGNAL_STATE_ENUM, rSignalQuality.eState)
                             ));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\teSatellite: %d", 
                             rSignalQuality.eSatellite
                             ));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tAntenna:"));

        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tState: %d", 
                             ETG_CENUM(ANTENNA_STATE_ENUM, DECODER.eAntennaState(_hDecoder,1))
                             ));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tNumber of Antennas: %d", 
                             DECODER.n8NumAntennas(_hDecoder)
                             ));

    }
    else
    {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t 0tDECODER invalid"));
        return;
    }

    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, ""));


    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "%s:", szGetName()));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\tChannelId: %d", fc_sxm_tclChannelList::instance()->u16GetCurrentTunedChannel()));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\tSID: 0x%x", fc_sxm_tclChannelList::instance()->u16GetSIDOfTunedChannel()));
    
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\ttodo: add data for audio (e.g. content of fc_sxm_trChannel for current channel)"));
    
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, ""));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "Data Service States (fi)"));
    std::vector<midw_ext_fi_tcl_SxmListElemDataSrvState> rList=fc_sxm_tclAppManager::instance()->oSxmDataServiceStatusFi.rGet().oFiMsg.List;
    SXM_FOREACH(vector<midw_ext_fi_tcl_SxmListElemDataSrvState>, iter, rList) {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t%x: %20s(%d)", 
                             ETG_CENUM(fc_sxm_tenServiceID, ((tU16)iter->Type.enType<<8) | 0x01),
                             fc_sxm_szSmsDataServiceStateFiToString(iter->Status.Status),
                             (tU16)iter->Status.Status.enType));
    }

    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, ""));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "Data-Services Detailed:"));

}

// Send Method Result for classes that have delegated responsibility to AudioApp
tVoid fc_sxm_tclAudioApp::vSendAudioFiMsg(fc_sxm_trAdressing const& rAdressing, const midw_ext_sxm_audiofi_tclMsgBaseMessage& oMRes)
{
	fc_sxm_tclAudioService::instance()->enSendFiMessage(rAdressing, oMRes);
}

// Send Method Errors for classes that have delegated responsibility to AudioApp
tVoid fc_sxm_tclAudioApp::vSendAudioErrorMsg(fc_sxm_trAdressing const  &rAdressing, tInt s32ErrorMsg)
{
	fc_sxm_tclAudioService::vSendError(rAdressing, (tU16)s32ErrorMsg);
}

// Notify Alert info
tVoid fc_sxm_tclAudioApp::vFavoriteContentAlertInfo(const fc_sxm_tcl_trFavoriteContentAlertInfo& oProperty)
{
   fc_sxm_tclAudioProperties::instance()->oAlertInfo.vSet(oProperty);
   fc_sxm_tclAudioProperties::instance()->oAlertInfo.vNotify();
}

// Notify Alert List Update status
tVoid fc_sxm_tclAudioApp::vAlertListUpdateStatus(const fc_sxm_tcl_trAlertNoticeListUpdate& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oAlertNoticListUpdate.vSet(oProperty);
   fc_sxm_tclAudioProperties::instance()->oAlertNoticListUpdate.vNotify();
}

// Notify TWNow Service's updated status
tVoid fc_sxm_tclAudioApp::vTWNowServiceStatus(const fc_sxm_tcl_trAudioPropertyTWNowServiceStatus& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oTWNowServiceStatus.vSetAndNotify(oProperty);
}

// Notify TWNow Bulletin Status
tVoid fc_sxm_tclAudioApp::vTWNowBulletinStatus(const fc_sxm_tcl_trAudioPropertyTWNowBulletinStatus& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oTWNowBulletinStatus.vSetAndNotify(oProperty);
}

// Notify Sports Flash service status
tVoid fc_sxm_tclAudioApp::vSFlashServiceStatus(const fc_sxm_tcl_trAudioPropertySportsFlash_ServiceStatus& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oSportsFlashServiceStatus.vSetAndNotify(oProperty);
}
// Notify Game Event Update
tVoid fc_sxm_tclAudioApp::vSFlashGameEventNotify(const fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oSportsFlashGameEvent.vSetAndNotify(oProperty);
}

// Notify Flash Event Update
tVoid fc_sxm_tclAudioApp::vSFlashFlashEventNotify(const fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oSportsFlashFlashEvent.vSetAndNotify(oProperty);
}

// Method returns ChannelList instance
Ifc_sxm_tclChannelList* fc_sxm_tclAudioApp::poGetChannelListInstance() const
{
	return fc_sxm_tclChannelList::instance();
}

// Method returns CategoryList instance
Ifc_sxm_tclCategoryList* fc_sxm_tclAudioApp::poGetCategoryListInstance() const
{
	return fc_sxm_tclCategoryList::instance();
}

// Method returns Presets instance
Ifc_sxm_tclPresets* fc_sxm_tclAudioApp::poGetPresetsInstance() const
{
	return fc_sxm_tclPresets::instance();
}

// Method returns Playback instance
Ifc_sxm_tclPlayback* fc_sxm_tclAudioApp::poGetPlaybackInstance() const
{
	return fc_sxm_tclPlayback::instance();
}
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
Ifc_sxm_tclProfileManager* fc_sxm_tclAudioApp::poGetProfileManagerInstance() const
{
	return fc_sxm_tcl_profilemanager::instance();
}
#endif

// Method requests Audio routing library to apply mute status "bMuteAudio"
tVoid fc_sxm_tclAudioApp::vMuteSXMAudio(tBool bMuteAudio) const
{
	fc_sxm_arl_tclISource::instance()->vSetMute(bMuteAudio);
}

// Return the current Mute state of SXM Audio
tBool fc_sxm_tclAudioApp::bGetSXMAudioMuteState() const
{
	return fc_sxm_arl_tclISource::instance()->bGetMuteState();
}

tVoid fc_sxm_tclAudioApp::vPostATAlertMsg(const fc_sxm_trMsgAudioATAlerts &oATAlerts)
{
   vPostMsgNew(oATAlerts);
}

tVoid fc_sxm_tclAudioApp::vPostSeekCategoryUpdate(const fc_sxm_trMsgAudioSeekCategoryUpdate &oSeekCategoryUpdate)
{
   vPostMsgNew(oSeekCategoryUpdate);
}

// Post Sports Flash GameAlert Message
tVoid fc_sxm_tclAudioApp::vSFGameAlertMsg (fc_sxm_trMsgSFGameAlert const &oGameAlert )
{
	vPostMsgNew(oGameAlert);
}

// Post Sports Flash FlashEventAlert Message
tVoid fc_sxm_tclAudioApp::vSFFlashEventAlertMsg (fc_sxm_trMsgSFFlashEventAlert const &oFlashEventAlert )
{
	vPostMsgNew(oFlashEventAlert);
}

// Post TWNow Event Alert Message
tVoid fc_sxm_tclAudioApp::vTWNowEventAlertMsg(fc_sxm_trMsgTWNowEventAlert const &oTWNowEvent)
{
	vPostMsgNew(oTWNowEvent);
}

// Utility method which would fetch channel Information for the given channel ID
const fc_sxm_trChannel* fc_sxm_tclAudioApp::trGetChannel(CHANNEL_ID channelID) const
{
	return fc_sxm_tclChannelList::instance()->trGetChannel(channelID);
}

// Notify TWNow Playback Status
tVoid fc_sxm_tclAudioApp::vTWNowPlaybackNotify(const fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oTWNowPlaybackStatus.vSetAndNotify(oProperty);
}

// Notify SportsFlash Playback Status
tVoid fc_sxm_tclAudioApp::vSFlashPlaybackNotify(const fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus& oProperty)
{
	fc_sxm_tclAudioProperties::instance()->oSportsFlashPlaybackStatus.vSetAndNotify(oProperty);
}

// Handle Broadscope changes for SportsFlash
tVoid fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAppSportsMetaDataUpdate const *prMsg)
{
	if ((prMsg) && (TRUE == prMsg->bMetaDataUpdate))
	{
		// Instruct Sports Flash to prepare for MetaDataUpdate
		fc_sxm_tclSportsFlash::instance()->vMetaDataUpdateStart();
	}
	else
	{
		// Instruct Sports Flash that MetaData Updation is complete
		fc_sxm_tclSportsFlash::instance()->vMetaDataUpdateComplete();
	}
}

// Abort TWNow and SportsFlashEventsPlayback
tVoid fc_sxm_tclAudioApp::vAbortFlashEventPlayback()
{
	if (fc_sxm_tclTWNow::instance()->bIsTWNowPlaybackInProgress())
	{
		(tVoid) fc_sxm_tclTWNow::instance()->enAbortTWNowBulletin(TW_NOW_INVALID_BULLETIN_ID);
	}
	else if (fc_sxm_tclSportsFlash::instance()->bIsSFPlaybackInProgress())
	{
		(tVoid) fc_sxm_tclSportsFlash::instance()->enAbortFlashEvent();
	}
}

// Return true if either Sports Flash playback or TWNow playback is in progress
bool fc_sxm_tclAudioApp::bIsFlashEventPlayback() const
{
	return ((fc_sxm_tclTWNow::instance()->bIsTWNowPlaybackInProgress()) ||
			(fc_sxm_tclSportsFlash::instance()->bIsSFPlaybackInProgress()));
}

// Return Current Tuned Channel's ID
CHANNEL_ID fc_sxm_tclAudioApp::u16GetTunedChnID() const
{
	return (fc_sxm_tclChannelList::instance()->u16GetCurrentTunedChannel());
}

// Method assigns BrowseRateConfigruation values for every entry made in TTFIs for Rapid Channel Browse
tVoid fc_sxm_tclAudioApp::vAssignBrowseRateConfig(vector<midw_ext_fi_tcl_BrowseRateConfiguration,
		allocator<midw_ext_fi_tcl_BrowseRateConfiguration> >& vectConfig,
		const tU8 u8Param) const
{
	midw_ext_fi_tcl_BrowseRateConfiguration config;
	vectConfig.clear();
	// This method never gets executed in run time and can be invoked only by providing TTFIs command for RapidChannelBrowseStart
	switch(u8Param)
	{
	case 0u:
	{
		// +ve case....
		// Use case assumption for the first 10 seconds, channel change rate = 0.5 seconds.
		// For the next 8 seconds, channel change rate = 0.4 seconds.
		// For the next 5 seconds, channel change rate = 0.25 seconds.
		// After that, channel change rate = 0.1 seconds forever.
		config.duration = 10u;
		config.rate = 500u;
		vectConfig.push_back(config);

		config.duration = 8u;
		config.rate = 400u;
		vectConfig.push_back(config);

		config.duration = 5u;
		config.rate = 250u;
		vectConfig.push_back(config);

		config.duration = 0u;
		config.rate = 100u;
		vectConfig.push_back(config);
	}
	break;
	case 1u:
	{
		// +ve case....
		// Use case assumption for the first 5 seconds, channel change rate = 1 second
		// For the next 3 seconds, channel change rate = 0.5 seconds
		// For the next 2 seconds, channel change rate = 0.25 seconds
		// After that, channel change rate = 0.1 seconds forever.
		config.duration = 5u;
		config.rate = 1000u;
		vectConfig.push_back(config);

		config.duration = 3u;
		config.rate = 500u;
		vectConfig.push_back(config);

		config.duration = 2u;
		config.rate = 250u;
		vectConfig.push_back(config);

		config.duration = 0u;
		config.rate = 100u;
		vectConfig.push_back(config);
	}
	break;
	case 2u:
	{
		// +ve case...
		// Use case assumption Constant channel change rate for 0.5 seconds forever
		config.duration = 0u;
		config.rate = 500u;
		vectConfig.push_back(config);
	}
	break;
	case 3u:
	{
		// +ve case ... 2 step change and change change rate deccelerate
		// Use case assumption. For the first 5 seconds, channel change rate = 0.1 seconds
		// After that, channel change rate = 0.5 seconds forever
		config.duration = 5u;
		config.rate = 100u;
		vectConfig.push_back(config);

		config.duration = 0u;
		config.rate = 500u;
		vectConfig.push_back(config);
	}
	break;
	case 4u:
	{
		// +ve case .. 5 step change with channel change rate first accelerates, then deccelerates and then again accelerates
		// Use case assumption for the first 5 seconds, channel change rate = 1 second
		// For the next 3 seconds, channel change rate = 0.5 seconds
		// For the next 2 seconds, channel change rate = 0.25 seconds
		// For the next 3 seconds, channel change rate = 0.5 seconds
		// After that, channel change rate = 1 second forever.
		config.duration = 5u;
		config.rate = 1000u;
		vectConfig.push_back(config);

		config.duration = 3u;
		config.rate = 500u;
		vectConfig.push_back(config);

		config.duration = 2u;
		config.rate = 250u;
		vectConfig.push_back(config);

		config.duration = 3u;
		config.rate = 500u;
		vectConfig.push_back(config);

		config.duration = 0u;
		config.rate = 1000u;
		vectConfig.push_back(config);
	}
	break;
	case 5u:
	{
		// -ve case ... Error condition
		// 0.1 seconds is minimum value designated for configuration rate. Any value lesser than that, channels are changed at 0.5 seconds forever
		config.duration = 10u;
		config.rate = 50u; // Must have been at least 100u
		vectConfig.push_back(config);
	}
	break;
	case 6u:
	{
		// -ve case ... Error condition .. Values for first step is proper but the next step is not
		// Use case assumption for the first 5 seconds, channel change rate = 1 second
		// After that, channel change rate = 0.05 seconds forever which is an error
		config.duration = 5u;
		config.rate = 1000u;
		vectConfig.push_back(config);

		config.duration = 0u;
		config.rate = 25u; // Must have been at least 100u
		vectConfig.push_back(config);
	}
	break;
	default:
	{
		(tVoid) config; // Avoids prio 3 lint warning
	}
		break;
	}
	ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vAssignBrowseRateConfig u8Param = %u \t vectConfig.size() = %u", u8Param, vectConfig.size()));
}

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION

//Method to handle profile change event for all audio services
tVoid fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioProfileChange const * prMsg )
{
	ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioProfileChange) prMsg->bIsGuestUser: %d",prMsg->bIsGuestUser));
	tBool bReturn = FALSE;

	Ifc_sxm_tclPresets* poPresets = poGetPresetsInstance();
    if(poPresets != NULL) {
    	bReturn = poPresets->bHandleChangeProfilePresetList();
    	ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vProcess(fc_sxm_trMsgAudioProfileChange) bReturn: %d",bReturn));
    }

    //Based on return value acknowledge PM about profile change
    Ifc_sxm_tclProfileManager* poProfileMgr = poGetProfileManagerInstance();
    if(TRUE != prMsg->bIsGuestUser) {

		#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    	((poProfileMgr) && (bReturn)) ?

    			poProfileMgr->vAckChangeProfileStatus(FC_SXM_EN_PROFILE_CHANGED) :
    			poProfileMgr->vAckChangeProfileStatus(FC_SXM_EN_FAILED);
		#endif

    } else {
    	poProfileMgr->vSetProfileChange(TRUE);
    }
}

#endif

//Callback method about profile change
tVoid fc_sxm_tclAudioApp::vChangeProfileUpdate(tBool bIsGuestUser)
{
	ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vChangeProfileUpdate()"));
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
	fc_sxm_trMsgAudioProfileChange oAudioProfileChangeEvent;
	oAudioProfileChangeEvent.bIsGuestUser = bIsGuestUser;

	//Post profile change event message to presets
	vPostMsgNew(oAudioProfileChangeEvent);
#endif
}

//Callback method about copy profile
tVoid fc_sxm_tclAudioApp::vCopyProfileUpdate()
{
	ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vCopyProfileUpdate()"));
}

//Callback method about delete profile
tVoid fc_sxm_tclAudioApp::vDeleteProfileUpdate()
{
	ETG_TRACE_USR4(("fc_sxm_tclAudioApp::vDeleteProfileUpdate()"));
}


//Method to check about profile change
tBool fc_sxm_tclAudioApp::bIsProfileChange() const
{
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
	Ifc_sxm_tclProfileManager* poProfileMgr = poGetProfileManagerInstance();
	tBool bReturn FALSE;
	if(poProfileMgr) {
		bReturn = poProfileMgr->bIsProfileChange();
		ETG_TRACE_USR4(("fc_sxm_tclAudioApp::bIsProfileChange() bReturn: %d",bReturn));
	}
	return bReturn;
#endif
	return true;
}

//Method to get active profileID
tBool fc_sxm_tclAudioApp::u8GetActiveProfileId() const
{
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
	Ifc_sxm_tclProfileManager* poProfileMgr = poGetProfileManagerInstance();
	tU8 u8ActiveProfileID = 0;
	if(poProfileMgr) {
		u8ActiveProfileID = poProfileMgr->u8GetActiveProfileId();
		ETG_TRACE_USR4(("fc_sxm_tclAudioApp::u8GetActiveProfileId() u8ActiveProfileID: %d",u8ActiveProfileID));
	}
	return u8ActiveProfileID;
#endif
	return true;
}

