/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diagtm_handler.cpp
* @brief       SXM Diagnosis Testmode handler. Responsible for creation ,Mediator for other
*              Diagnosis classes and also destruction.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"

#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_sms_util.h"

#include "fc_sxm_tcl_diag_handler_test.h"
#include "fc_sxm_tcl_diag_properties.h"
#include "fc_sxm_tcl_diaghandler.h"
#include "fc_sxm_tcl_diagtm_handler.h"
#include "fc_sxm_tcl_sports_app.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diagtm_handler.cpp.trc.h"
#endif


/*******************************************************************************
*
* FUNCTION: fc_sxm_tclDiagTmHandler::
*             fc_sxm_tclDiagTmHandler()
*
* DESCRIPTION: Constructor .
*              
*
* PARAMETER: None
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_sxm_tclDiagTmHandler::fc_sxm_tclDiagTmHandler():
m_bActive(FALSE)
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler constructor"));

    vPopulateDbErrorString();
}

/*******************************************************************************
*
* FUNCTION: fc_sxm_tclDiagTmHandler::
*             ~fc_sxm_tclDiagTmHandler()
*
* DESCRIPTION: Destructor .
*              
*
* PARAMETER: None
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_sxm_tclDiagTmHandler::~fc_sxm_tclDiagTmHandler()
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler destructor"));
    m_bActive = FALSE;

    vRemoveDbErrorString();
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vProcess
*(
*  fc_sxm_trMsgDiagMStartTestModeEnable const *prMsg
*)
*
* DESCRIPTION: This method is called on dispatch msg for enabling test mode .
*
* PARAMETER: const *prMsg : 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vProcess
(
 fc_sxm_trMsgDiagMStartTestModeEnable const *prMsg
 )
{
    tBool bEnable = prMsg->oFiMsg.bEnable;
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vProcess(fc_sxm_trMsgDiagMStartTestModeEnable)bEnable=%d",bEnable));

    /*Update data base version on Module state*/
    vUpdateDataBaseVersion();

    vActivate(bEnable);

    /* send the answer to the client using adressing-infomation: */
    midw_ext_sxm_audiofi_tclMsgDiagTestModeEnableMethodResult oFiTxObj;
    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oFiTxObj);
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vActivate
*(
*  tBool bEnable
*)
*
* DESCRIPTION: This method is called on dispatch msg for enabling test mode .
*
* PARAMETER: const *prMsg : 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vActivate
(
 tBool bEnable
 )
{
   if (m_bActive != bEnable) 
   {
      DECODER_OBJECT hDecoderObj = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();

      m_bActive=bEnable;

      if (TRUE == m_bActive) 
      {
         /*Enable the Engineering data in the Decoder call back.Updating the event mask 
         in Decoder*/
         if(
            SMSAPI_RETURN_CODE_ERROR != DECODER.eModifyRegisteredEventMask(hDecoderObj,                                                
            DECODER_OBJECT_EVENT_ENGINEERING_DATA ,
            SMSAPI_MODIFY_EVENT_MASK_ENABLE                                                 
            )
            )
         {               
            ETG_TRACE_USR2(("fc_sxm_tclDiagTmHandler::vActivate - Engineering data enabled "));
            m_oTestModeTimer.vStart(DIAG_TESTMODEDATA_TIMER_VAL);

         } //if(SMSAPI_RETURN_CODE_ERROR..

         /*Update data base version on Module state*/
         vUpdateDataBaseVersion();
      } //if (m_bActive) 
      else 
      {
         /*Disable the Engineering data in the Decoder call back*/
         if(SMSAPI_RETURN_CODE_ERROR != DECODER.eModifyRegisteredEventMask(hDecoderObj,                                                
            DECODER_OBJECT_EVENT_ENGINEERING_DATA ,
            SMSAPI_MODIFY_EVENT_MASK_DISABLE                                                 
            ))
         {
            m_oTestModeTimer.vStop();
         }
      }
   }
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vProcess
*(
*  fc_sxm_trMsgDiagMFetchUpcValue const *prMsg
*) const
*
* DESCRIPTION: This method is called on dispatch msg for enabling test mode .
*
* PARAMETER: const *prMsg : 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vProcess
(
 fc_sxm_trMsgDiagMFetchUpcValue const *prMsg
 ) const
{
    tU8 u8Source = prMsg->oFiMsg.GetVal;

    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vProcess(fc_sxm_trMsgDiagMFetchUpcValue)u8Source=%d",u8Source));  

    tU32 u32Iterator = 0;
    size_t u32Respsize = 0;
    size_t pu32Bytesrcvd = 0 ;
    tU8 u8Buffer[FC_SXM_DIAG_UPC_WORD] = {0};

    midw_ext_sxm_audiofi_tclMsgDiagUpcValuesMethodResult oFiTxObj;

    if (FC_SXM_DIAG_STM_MENU == u8Source

        ||

        FC_SXM_DIAG_SETTING_MENU == u8Source) 
    {
        /*SMSAPI_RETURN_CODE_ENUM eReturnCode;
        Call to query the memory size required for Package Info*/
        if(
            (SMSAPI_RETURN_CODE_SUCCESS == MODULE.ePackage(fc_sxm_tclSmsModule::instance()->hGetSmsModule(),MODULE_PACKGE_CMD_REPORT,0,OSAL_NULL,0,&pu32Bytesrcvd))

            &&

            (FC_SXM_DIAG_UPC_PACKAGE_ID_LIMIT <= pu32Bytesrcvd) //atleast 12 bytes should be available
            )
        {
            ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::fc_sxm_trMsgAudioSmsEvtModuleState package size = %d,",pu32Bytesrcvd));

            u32Respsize = pu32Bytesrcvd;

            UN8 *pu8Resp = OSAL_NEW tU8[pu32Bytesrcvd];

            if(OSAL_NULL != pu8Resp)
            {
                //Get the Package info
                if(SMSAPI_RETURN_CODE_SUCCESS == MODULE.ePackage(fc_sxm_tclSmsModule::instance()->hGetSmsModule(),MODULE_PACKGE_CMD_REPORT,0,pu8Resp,u32Respsize,&pu32Bytesrcvd))
                {
                    //get the last 12 bytes
                    u32Iterator = (tU32)(pu32Bytesrcvd - FC_SXM_DIAG_UPC_PACKAGE_ID_LIMIT);
                    //similar to sprintf() but additionally takes care of null termination also
                    /*avoiding lint error 564.cannot use ++*/
                    OSALUTIL_s32SaveNPrintFormat((tString)u8Buffer,FC_SXM_DIAG_UPC_WORD,"%02x%02x%02x%02x%02x%02x",\
                       pu8Resp[u32Iterator],pu8Resp[u32Iterator+1],\
                       pu8Resp[u32Iterator+2],pu8Resp[u32Iterator+3],\
                       pu8Resp[u32Iterator+4],pu8Resp[u32Iterator+5] );
                    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::ModuleState package ULID0 = %s",u8Buffer));
                    fc_sxm_vString2Fi((tCString)u8Buffer,oFiTxObj.UpcLid0);

                    OSAL_pvMemorySet(u8Buffer,0,FC_SXM_DIAG_UPC_WORD);
                    //similar to sprintf() but additionally takes care of null termination also
                    OSALUTIL_s32SaveNPrintFormat((tString)u8Buffer,FC_SXM_DIAG_UPC_WORD,"%02x%02x%02x%02x%02x%02x",\
                       pu8Resp[u32Iterator+6],pu8Resp[u32Iterator+7],\
                       pu8Resp[u32Iterator+8],pu8Resp[u32Iterator+9],\
                       pu8Resp[u32Iterator+10],pu8Resp[u32Iterator+11] );
                    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::ModuleState package ULID1 = %s",u8Buffer));
                    fc_sxm_vString2Fi((tCString)u8Buffer,oFiTxObj.UpcLid1);
                }
                else
                {
                    ETG_TRACE_USR4(("DiagTmHandler::vProcess(fc_sxm_trMsgDiagMFetchUpcValue)Package info SMSAPI_RETURN=Fail "));
                }

                //Delete the allocated memory
                OSAL_DELETE[] pu8Resp;
                pu8Resp = OSAL_NULL;

            }
            else
            {
                ETG_TRACE_USR4(("DiagTmHandler::vProcess(fc_sxm_trMsgDiagMFetchUpcValue)pu8Resp is null"));
            }  
        }
        else
        {
            ETG_TRACE_USR4(("DiagTmHandler::vProcess(fc_sxm_trMsgDiagMFetchUpcValue)mem alloc SMSAPI_RETURN=Fail bytes = %d",pu32Bytesrcvd));
        }

    }  
    /* send the answer to the client using adressing-infomation: */
    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oFiTxObj);
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vProcess
*(
*  fc_sxm_trMsgCmdEmitTestModeData const *prMsg
*)
*
* DESCRIPTION: This method is called on dispatch msg for enabling test mode .
*
* PARAMETER: const *prMsg : 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vProcessTimer
(
 fc_sxm_trMsgCmdEmitTestModeData const *prMsg
 )
{
    (tVoid)prMsg;
    ETG_TRACE_USR1(("fc_sxm_tclDiagTmHandler::vProcess(fc_sxm_trMsgCmdEmitTestModeData)timer"));

    if (TRUE == m_bActive)
    {

        ETG_TRACE_USR1(("fc_sxm_tclDiagTmHandler::vProcess(fc_sxm_trMsgCmdEmitTestModeData) m_bActive  is active"));

        //Update the test mode data with pending SXM service status and SXM Version info
        vUpdateSxmVerServiceStatus();

        fc_sxm_tclDiagProperties::instance()->oTestModeData.vNotify();       

        m_oTestModeTimer.vStart(DIAG_TESTMODEDATA_TIMER_VAL);
    }
    else
    {
        ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vProcess(fc_sxm_trMsgCmdEmitTestModeData)else - m_bActive=%d",m_bActive));
    }
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vUpdateSxmVerServiceStatus(tVoid)
*
*
* DESCRIPTION: This method is called to update SXM version info and service status 
*              before sending the test mode data to client.
*
* PARAMETER: NONE 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vUpdateSxmVerServiceStatus()
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagTmHandler::vUpdateSxmVerServiceStatus"));

    /*Get the handle to test mode property.Update the property with data for SXM version and service status .
    The property is already updated with Data for Detailed signal quality and overlay signal quality*/
    midw_ext_sxm_audiofi_tclMsgDiagTestModeDataStatus &rFiData =fc_sxm_tclDiagProperties::instance()->oTestModeData.rAccess().oFiMsg;
    midw_ext_fi_tcl_SxmVersionInfo &SxmVersionInfo = rFiData.Data.SxmVersionInfo;

    /*Update the SXM version info*/
    fc_sxm_trDiagSxmVersionInfo &rfSxmVerInfo = m_DiagVersionInfo.rAccess();

    fc_sxm_vString2Fi(rfSxmVerInfo.oModuleId.c_str(),SxmVersionInfo.ModuleId);
    fc_sxm_vString2Fi(rfSxmVerInfo.oHdwRev.c_str(),SxmVersionInfo.HwRev);
    fc_sxm_vString2Fi(rfSxmVerInfo.oSwRev.c_str(),SxmVersionInfo.SwRev);
    fc_sxm_vString2Fi(rfSxmVerInfo.oSXIRev.c_str(),SxmVersionInfo.SxiRev);
    fc_sxm_vString2Fi(rfSxmVerInfo.oBBRev.c_str(),SxmVersionInfo.BbRev);
    fc_sxm_vString2Fi(rfSxmVerInfo.oHDecRev.c_str(),SxmVersionInfo.HDecRev);
    fc_sxm_vString2Fi(rfSxmVerInfo.oRFRev.c_str(),SxmVersionInfo.RFRev);
    fc_sxm_vString2Fi(rfSxmVerInfo.oCapability.c_str(),SxmVersionInfo.Capability);
	fc_sxm_vString2Fi(rfSxmVerInfo.oSPLRev.c_str(),SxmVersionInfo.SPLRev);

    /*Release the semaphore for version info*/
    m_DiagVersionInfo.vRelease();

    /*Update the Subscription status info*/
    midw_ext_fi_tcl_SxmDataSubStatus &SxmServiceStatus = rFiData.Data.SxmSubscriptionState;

    fc_sxm_trDiagSxmSubStatus &rfSxmServiceSt = m_DiagSubStatus.rAccess();

    SxmServiceStatus.SubStatus =  rfSxmServiceSt.u8SubsStat;
    SxmServiceStatus.ReasonCode = rfSxmServiceSt.u8ReasonCode;
    SxmServiceStatus.SuspendedDate = rfSxmServiceSt.u32SusPendDate;

    /*Release the semaphore for subscription status*/
    m_DiagSubStatus.vRelease();

    fc_sxm_tclDiagProperties::instance()->oTestModeData.vRelease();

}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vHandleEngineeringDataCallBack
*          (DECODER_OBJECT& hDecoder)
*
* DESCRIPTION: This function is used to update the diagnosis related info got from
*              the decoder call back.must be called in context of sms-decoder-callback
*
* PARAMETER: [IN] hDecoder - Reference to Decoder object
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vHandleEngineeringDataCallBack
(
 DECODER_OBJECT hDecoder
 ) const
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler vHandleEngineeringDataCallBack"));

    /*Get the handle to the signal quality object*/
    DETAILED_SIGNAL_QUALITY_OBJECT oSignalQuality = ENGINEERING_DATA.hSignalQualityDetailed(hDecoder);
    DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT oOverlaySigQw = ENGINEERING_DATA.hOverlaySignalQualityDetailed(hDecoder);

    /*get the Test mode data status Fi object*/
    midw_ext_sxm_audiofi_tclMsgDiagTestModeDataStatus &rFiData =fc_sxm_tclDiagProperties::instance()->oTestModeData.rAccess().oFiMsg;
    midw_ext_fi_tcl_SxmDetSignalQuality &rDetSigQual = rFiData.Data.DetSignalQuality;

    /*Get the signal strength value-Enum values matching with the value from SMS*/
    STRING_OBJECT oStrSigStrngth = DETAILED_SIGNAL_QUALITY.hSigStrength(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oStrSigStrngth,rDetSigQual.SigQuality);
    //rDetSigQual.SigQuality = (tU8)fc_sxm_vConvertSmsString2Int_base16(oStrSigStrngth);

    /*Get the tuner status-Hexadecimal value from SMS*/
    STRING_OBJECT oTunerStatus = DETAILED_SIGNAL_QUALITY.hTunerStatus(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oTunerStatus,rDetSigQual.TunerStatus);
    //rDetSigQual.TunerStatus = (tU8)fc_sxm_vConvertSmsString2Int_base16(oTunerStatus);

    STRING_OBJECT oEnsaLock = DETAILED_SIGNAL_QUALITY.hENSALockStatus(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oEnsaLock,rDetSigQual.ENSALockStatus);
    //rDetSigQual.ENSALockStatus =(tU8)fc_sxm_vConvertSmsString2Int_base16(oEnsaLock);

    STRING_OBJECT oEnsbLock = DETAILED_SIGNAL_QUALITY.hENSBLockStatus(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oEnsbLock,rDetSigQual.ENSBLockStatus);
    //rDetSigQual.ENSBLockStatus = (tU8)fc_sxm_vConvertSmsString2Int_base16(oEnsbLock);

    /*Get the tuner Ber S1-Float values from SMS*/
    STRING_OBJECT oBERS1 = DETAILED_SIGNAL_QUALITY.hBERS1(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oBERS1,rDetSigQual.BerS1);
    //rDetSigQual.BerS1 = fc_sxm_vConvertSmsString2Double(oBERS1);

    STRING_OBJECT oBERS2 = DETAILED_SIGNAL_QUALITY.hBERS2(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oBERS2,rDetSigQual.BerS2);
    //rDetSigQual.BerS2 = fc_sxm_vConvertSmsString2Double(oBERS2);

    STRING_OBJECT oBERT = DETAILED_SIGNAL_QUALITY.hBERT(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oBERT,rDetSigQual.BerT);
    //rDetSigQual.BerT = fc_sxm_vConvertSmsString2Double(oBERT);

    STRING_OBJECT oCNS1A = DETAILED_SIGNAL_QUALITY.hCNS1A(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oCNS1A,rDetSigQual.CtoNS1A);
    //rDetSigQual.CtoNS1A = fc_sxm_vConvertSmsString2Double(oCNS1A);

    STRING_OBJECT oCNS1B = DETAILED_SIGNAL_QUALITY.hCNS1B(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oCNS1B,rDetSigQual.CtoNS1B);
    //rDetSigQual.CtoNS1B = fc_sxm_vConvertSmsString2Double(oCNS1B);

    STRING_OBJECT oCNS2A = DETAILED_SIGNAL_QUALITY.hCNS2A(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oCNS2A,rDetSigQual.CtoNS2A);
    //rDetSigQual.CtoNS2A = fc_sxm_vConvertSmsString2Double(oCNS2A);

    STRING_OBJECT oCNS2B = DETAILED_SIGNAL_QUALITY.hCNS2B(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oCNS2B,rDetSigQual.CtoNS2B);
    //rDetSigQual.CtoNS2B = fc_sxm_vConvertSmsString2Double(oCNS2B);

    STRING_OBJECT oRSErrsWord = DETAILED_SIGNAL_QUALITY.hRSErrsWord(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oRSErrsWord,rDetSigQual.RSErrsWords);
    //rDetSigQual.RSErrsWords = fc_sxm_vConvertSmsString2Double(oRSErrsWord);

    STRING_OBJECT oRSErrsSatSymb = DETAILED_SIGNAL_QUALITY.hRSErrsSatSymb(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oRSErrsSatSymb,rDetSigQual.RSErrSatSymb);
    //rDetSigQual.RSErrSatSymb = fc_sxm_vConvertSmsString2Double(oRSErrsSatSymb);

    STRING_OBJECT oRSErrsTerrSymb = DETAILED_SIGNAL_QUALITY.hRSErrsTerrSymb(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oRSErrsTerrSymb,rDetSigQual.RSErrTerrSymb);
    //rDetSigQual.RSErrTerrSymb = fc_sxm_vConvertSmsString2Double(oRSErrsTerrSymb);

    STRING_OBJECT oTunerCarrierFreqOffset = DETAILED_SIGNAL_QUALITY.hTunerCarrierFreqOffset(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oTunerCarrierFreqOffset,rDetSigQual.TunerCarrFreqOfset);
    //rDetSigQual.TunerCarrFreqOfset = fc_sxm_vConvertSmsString2int(oTunerCarrierFreqOffset);

    STRING_OBJECT oRSSI = DETAILED_SIGNAL_QUALITY.hRSSI(oSignalQuality);
    fc_sxm_vCopySmsString2Fi(oRSSI,rDetSigQual.RSSI);
    //rDetSigQual.RSSI = fc_sxm_vConvertSmsString2int(oRSSI);

    /*Getting the overlay signal properties*/
    midw_ext_fi_tcl_SxmDetOvrlSignalQuality &rDetOvrSigQual = rFiData.Data.DetOvrlSignalQuality;

    STRING_OBJECT oReceiverState = DETAILED_OVERLAY_SIGNAL_QUALITY.hReceiverState(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oReceiverState,rDetOvrSigQual.ReceiverState);
    //rDetOvrSigQual.ReceiverState = (tU8)fc_sxm_vConvertSmsString2Int_base16(oReceiverState);

    STRING_OBJECT oOBERS1A = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERS1A(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERS1A,rDetOvrSigQual.OBerS1A);
    //rDetOvrSigQual.OBerS1A = fc_sxm_vConvertSmsString2Double(oOBERS1A);

    STRING_OBJECT oOBERS1B = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERS1B(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERS1B,rDetOvrSigQual.OBerS1B);
    //rDetOvrSigQual.OBerS1B = fc_sxm_vConvertSmsString2Double(oOBERS1B);

    STRING_OBJECT oOBERS2A = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERS2A(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERS2A,rDetOvrSigQual.OBerS2A);
    //rDetOvrSigQual.OBerS2A = fc_sxm_vConvertSmsString2Double(oOBERS2A);

    STRING_OBJECT oOBERS2B = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERS2B(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERS2B,rDetOvrSigQual.OBerS2B);
    //rDetOvrSigQual.OBerS2B = fc_sxm_vConvertSmsString2Double(oOBERS2B);

    STRING_OBJECT oOBERTerrA = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTerrA(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTerrA,rDetOvrSigQual.OBerTA);
    //rDetOvrSigQual.OBerTA = fc_sxm_vConvertSmsString2Double(oOBERTerrA);

    STRING_OBJECT oOBERTerrB = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTerrB(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTerrB,rDetOvrSigQual.OBerTB);
    //rDetOvrSigQual.OBerTB = fc_sxm_vConvertSmsString2Double(oOBERTerrB);

    STRING_OBJECT oOBERTurboWord0a = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTurboWordErrRate0A(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTurboWord0a,rDetOvrSigQual.OBERTrbWrdErRt0A);
    //rDetOvrSigQual.OBERTrbWrdErRt0A = fc_sxm_vConvertSmsString2Double(oOBERTurboWord0a);

    STRING_OBJECT oOBERTurboWord0b = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTurboWordErrRate0B(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTurboWord0b,rDetOvrSigQual.OBERTrbWrdErRt0B);
    //rDetOvrSigQual.OBERTrbWrdErRt0B = fc_sxm_vConvertSmsString2Double(oOBERTurboWord0b);

    STRING_OBJECT oOBERTurboWord1a = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTurboWordErrRate1A(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTurboWord1a,rDetOvrSigQual.OBERTrbWrdErRt1A);

    STRING_OBJECT oOBERTurboWord1b = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTurboWordErrRate1B(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTurboWord1b,rDetOvrSigQual.OBERTrbWrdErRt1B);

    STRING_OBJECT oOBERTurboWord2a = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTurboWordErrRate2A(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTurboWord2a,rDetOvrSigQual.OBERTrbWrdErRt2A);

    STRING_OBJECT oOBERTurboWord2b = DETAILED_OVERLAY_SIGNAL_QUALITY.hOBERTurboWordErrRate2B(oOverlaySigQw);
    fc_sxm_vCopySmsString2Fi(oOBERTurboWord2b,rDetOvrSigQual.OBERTrbWrdErRt2B);


    fc_sxm_tclDiagProperties::instance()->oTestModeData.vRelease();

}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vProcess
*(
* fc_sxm_trMsgEvtAudioSmsModuleSubStatusChanged const *prMsg
* )
*
* DESCRIPTION: This is method called on Module state change to inform subscription status
*
* PARAMETER: fc_sxm_trMsgEvtAudioSmsModuleSubStatusChanged const *prMsg : 
*            subscription status change
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vProcess
(
 fc_sxm_trMsgEvtAudioSmsModuleSubStatusChanged const *prMsg
 )
{
    (tVoid)prMsg;
    BOOLEAN bIsAudioSubscribed = FALSE;
    MODULE_OBJECT hModule = fc_sxm_tclSmsModule::instance()->hGetSmsModule();
    if (!hModule) {
        return;
    }

    MODULE_SUBSTATUS_ENUM peStatus = MODULE_SUBSTATUS_INVALID;
    MODULE_SUBSTATUS_REASON_CODE ptReasonCode = 0;
    UN32 pun32UTCSuspendDate = 0;

    char ReasonText[100] = "\0";
    char PhoneNumber[100] = "\0";
    STRING_OBJECT hReasonText = STRING.hCreate(ReasonText, 100);
    STRING_OBJECT hPhoneNumber = STRING.hCreate(PhoneNumber, 100);

    MODULE.eSubscription(hModule,&peStatus,
        &ptReasonCode,
        &pun32UTCSuspendDate,
        hReasonText,
        hPhoneNumber,&bIsAudioSubscribed);

    STRING.tCopyToCStr(hReasonText,ReasonText,100);
    STRING.tCopyToCStr(hPhoneNumber,PhoneNumber,100);

    ETG_TRACE_USR4(("Subscription peStatus: %d",peStatus));
    ETG_TRACE_USR4(("Subscription ptReasonCode: %d",ptReasonCode));
    ETG_TRACE_USR4(("Subscription pun32UTCSuspendDate: %d",pun32UTCSuspendDate));
    ETG_TRACE_USR4(("Subscription hReasonText: %s",hReasonText));
    ETG_TRACE_USR4(("Subscription hPhoneNumber: %s",hPhoneNumber));
	ETG_TRACE_USR4(("Subscription bIsAudioSubscribed: %d",bIsAudioSubscribed));

    /*Updating the Diag handler with available data*/
    /*get the Test mode data status Fi object*/
    fc_sxm_trDiagSxmSubStatus &rfSubStatus = m_DiagSubStatus.rAccess();

    rfSubStatus.u8SubsStat = peStatus;    
    rfSubStatus.u8ReasonCode = ptReasonCode;
    /*pun32UTCSuspendDate is a UTC time.This can be converted to a date by using the 
    Ansi time function gmtime() which returns a structure(struct tm):
    Currently not done from Diagnosis.Expecting PDAL to convert and format according to Modal 
    requirements .Passing the UTC time -pai2hi*/
    rfSubStatus.u32SusPendDate = pun32UTCSuspendDate;    

    m_DiagSubStatus.vRelease();

    /*Update the property-Updating module subcription status - Start*/
    midw_ext_sxm_audiofi_tclMsgSxmModuleSubInfoStatus &rModuleFiData =fc_sxm_tclDiagProperties::instance()->oSxmModuleSubInfo.rAccess().oFiMsg;
    midw_ext_fi_tcl_SxmDataSubStatus &rModuleSubStatus = rModuleFiData.Data;
    rModuleSubStatus.SubStatus = peStatus;
    rModuleSubStatus.ReasonCode = ptReasonCode;
    rModuleSubStatus.SuspendedDate = pun32UTCSuspendDate;
    fc_sxm_tclDiagProperties::instance()->oSxmModuleSubInfo.vRelease();

    fc_sxm_tclDiagProperties::instance()->oSxmModuleSubInfo.vNotify();
    /*Module subscription status  - End*/

    // cleanup
    STRING.vDestroy(hReasonText);
    STRING.vDestroy(hPhoneNumber);

}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vProcess
*           (
*            fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg
*            )
*
* DESCRIPTION: This function is used to update the SXM version related info got from
*              the module call back
*
* PARAMETER: tBool bSuccess, : Result of user data clear
*            fc_sxm_trAdressing const &_rAdressing :information about the sender
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vProcess
(
 fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg
 )
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::fc_sxm_trMsgAudioSmsEvtModuleState"));
    if (prMsg->eState != MODULE_STATE_READY) {
        return;
    }

    /*Update module version on Module state*/
    vUpdateModuleVersion();
    
    /*Update module version on Module state*/
    vUpdateRadioId();  
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vUpdateModuleVersion()
*
* DESCRIPTION: This function is used to update the SXM version related info got from
*              the module call back
*
* PARAMETER:  MODULE_OBJECT &hModule : Handle to the module
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vUpdateModuleVersion()
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateModuleVersion"));

    MODULE_OBJECT hModule = fc_sxm_tclSmsModule::instance()->hGetSmsModule();
    if (!hModule) {
        return;
    }
   /*get the Test mode data status Fi object.Writing to a storage rather than writing to 
      Fi directly as this value might be required by other internal modules as well- Start*/
    fc_sxm_trDiagSxmVersionInfo &rfSxmVersion = m_DiagVersionInfo.rAccess();

    STRING_OBJECT oModuleId = MODULE_VERSION.hTypeID(hModule);
    fc_sxm_vCopySmsString2Stl(oModuleId,rfSxmVersion.oModuleId);

    STRING_OBJECT oHwRev = MODULE_VERSION.hHardwareRevision(hModule);
    fc_sxm_vCopySmsString2Stl(oHwRev,rfSxmVersion.oHdwRev);

    STRING_OBJECT oSwRev = MODULE_VERSION.hSoftwareRevision(hModule);
    fc_sxm_vCopySmsString2Stl(oSwRev,rfSxmVersion.oSwRev);

    STRING_OBJECT oSXIRev = MODULE_VERSION.hProtocolRevision(hModule);
    fc_sxm_vCopySmsString2Stl(oSXIRev,rfSxmVersion.oSXIRev);

    STRING_OBJECT oBbRev = MODULE_VERSION.hBaseBandRevision(hModule);
    fc_sxm_vCopySmsString2Stl(oBbRev,rfSxmVersion.oBBRev);

    STRING_OBJECT oHDecRev = MODULE_VERSION.hDecoderRevision(hModule);
    fc_sxm_vCopySmsString2Stl(oHDecRev,rfSxmVersion.oHDecRev);

    STRING_OBJECT oRFRev = MODULE_VERSION.hRFRevision(hModule);
    fc_sxm_vCopySmsString2Stl(oRFRev,rfSxmVersion.oRFRev);

    STRING_OBJECT oCapability = MODULE_VERSION.hCapability(hModule);
    fc_sxm_vCopySmsString2Stl(oCapability,rfSxmVersion.oCapability);

    STRING_OBJECT oSPLRev = MODULE_VERSION.hSPLRevision(hModule);
    fc_sxm_vCopySmsString2Stl(oSPLRev,rfSxmVersion.oSPLRev);

    m_DiagVersionInfo.vRelease();

    /*Module version end*/   
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vUpdateRadioId()
*
* DESCRIPTION: This function is used to update SXM Radio ID to the clients
*
* PARAMETER: MODULE_OBJECT &hModule : Handle to the module
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vUpdateRadioId() const
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagTmHandler::vUpdateRadioId"));
   
    MODULE_OBJECT hModule = fc_sxm_tclSmsModule::instance()->hGetSmsModule();
    if (!hModule) {
        return;
    }
    /*Get Data from SMS */
    STRING_OBJECT oRadioId = MODULE.hESN(hModule);

    /*Update the Fi*/
    midw_ext_sxm_audiofi_tclMsgSxmRadioIdStatus &rFiData =fc_sxm_tclDiagProperties::instance()->oSxmRadioId.rAccess().oFiMsg;
    fc_sxm_vCopySmsString2Fi(oRadioId,rFiData.RadioId);
    fc_sxm_tclDiagProperties::instance()->oSxmRadioId.vRelease();

    fc_sxm_tclDiagProperties::instance()->oSxmRadioId.vNotify();
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion
*           (
*           tVoid
*            )
*
* DESCRIPTION: This function is used to update the Data base version related info 
*
* PARAMETER: 
*            
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion()
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion ")); 

    DATASERVICE_ERROR_CODE_ENUM eReasonFail;
    DATASERVICE_REF_DATA_VER
                tCurVersion = DATASERVICE_REF_DATA_INVALID_VER,
                tNextVersion = DATASERVICE_REF_DATA_INVALID_VER;

    UN8 u8MajorVersion= 0;
    UN8 u8MinorVersion= 0;
    UN8 u8subVersion= 0;
    tU8 u8SmsVersion[FC_SXM_DIAG_VERSION_MAX]= {0};

    midw_ext_sxm_audiofi_tclMsgSxmSmsDbVersionsStatus &oFiData = fc_sxm_tclDiagProperties::instance() -> oSxmDbVersions.rAccess().oFiMsg;
#if 0
    /*------------------Channel art database----------------*/
    eReasonFail = CHANNEL_ART.eGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion channel art reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.CgsDb);
#endif

    /*-----------------------Fuel---------------------------*/

    eReasonFail = FUEL.eGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion Fuel reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.FuelDb);

    /*-----------------------canadian Fuel---------------------------*/

    eReasonFail = FUEL.eCanadianGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion Canadian_Fuel reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.FuelCanadaDB);

    /*-----------------------Sports---------------------------*/

    eReasonFail = (sxm_sports_get_version(&tCurVersion) == SXM_E_OK) ?
        		DATASERVICE_ERROR_CODE_NONE:
        		DATASERVICE_ERROR_CODE_GENERAL;

    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion sports reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.SportsDB);

    /*-----------------------Parking---------------------------*/

    eReasonFail = (sxm_parking_get_version(&tCurVersion) == SXM_E_OK) ?
    		DATASERVICE_ERROR_CODE_NONE:
    		DATASERVICE_ERROR_CODE_GENERAL;

    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion parking reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.ParkingDB);

    /*-----------------------Movie----------------------------*/

    eReasonFail = MOVIES.eGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion Movie reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.MovieDb);

    /*-----------------------Tabular Weather----------------------------*/

    eReasonFail = WEATHER.eGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion Weather reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.TabWeatherDb);
#if 0
    /*-----------------------Stocks----------------------------*/

    eReasonFail = STOCK_TICKER.eGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion Stocks reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.StockDb);
#endif
    /*-----------------------WS Alerts----------------------------*/

    eReasonFail = WS_ALERTS.eGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion WS Alerts reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.WsAlertsDb);
    /*-----------------------Safe view----------------------------*/

    eReasonFail = SAFETY_CAMERAS.eGetReferenceDataVersion(OSAL_NULL,&tCurVersion,&tNextVersion);
    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vUpdateDataBaseVersion Safe view reason=%d",eReasonFail));
    vCheckNWriteDbVersion(tCurVersion,eReasonFail,oFiData.Data.SafeViewdb);

    /*-----------------------SMS Version----------------------------*/

    tCString sStrSmsVersion = SMS.pacGetVersion(&u8MajorVersion,&u8MinorVersion,&u8subVersion);
     OSALUTIL_s32SaveNPrintFormat((tString)u8SmsVersion,FC_SXM_DIAG_VERSION_MAX,"v%d.%d.%d",\
                       u8MajorVersion,\
                       u8MinorVersion,\
                       u8subVersion);
    fc_sxm_vString2Fi((tCString)u8SmsVersion,oFiData.Data.SmsVersion);
     if(OSAL_NULL != sStrSmsVersion)
    {
        ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::pacGetVersion full=%s",sStrSmsVersion));
    }
    else
    {
        ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::pacGetVersion sStrSmsVersion is null"));
    }
    /*-----------------------SMS Version----------------------------*/

    fc_sxm_tclDiagProperties::instance() -> oSxmDbVersions.vRelease();

    fc_sxm_tclDiagProperties::instance()->oSxmDbVersions.vNotify();
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vPopulateDbErrorString()
*
* DESCRIPTION: Method to populate DB error string on object construction
*
* PARAMETER: NONE
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vPopulateDbErrorString()
{
    m_vecDbErrorStrings.push_back(string("NoError"));
    m_vecDbErrorStrings.push_back(string("ErGen"));
    m_vecDbErrorStrings.push_back(string("DevFail"));
    m_vecDbErrorStrings.push_back(string("InvalDev"));
    m_vecDbErrorStrings.push_back(string("NoDb"));
    m_vecDbErrorStrings.push_back(string("DbVerMis"));
    m_vecDbErrorStrings.push_back(string("NoDbAcs"));
    m_vecDbErrorStrings.push_back(string("StopFail"));
    m_vecDbErrorStrings.push_back(string("DbCorupt"));
    m_vecDbErrorStrings.push_back(string("DbRdOnly"));
    m_vecDbErrorStrings.push_back(string("DbOld"));
    m_vecDbErrorStrings.push_back(string("ErUnKnwn"));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vRemoveDbErrorString()
*
* DESCRIPTION: Method to remove DB error string on object destruction
*
* PARAMETER: NONE
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vRemoveDbErrorString()
{
    m_vecDbErrorStrings.clear();
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagTmHandler::vCheckNWriteDbVersion()
*
* DESCRIPTION: Method to check the status of DbVersion read and update the 
*              HMI with DB version or system error
*
* PARAMETER: DATASERVICE_REF_DATA_VER u32Version : Database version
*            DATASERVICE_ERROR_CODE_ENUM eStatus :Database read status
*            midw_ext_fi_tclString& rffiObj : the Fi string
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagTmHandler::vCheckNWriteDbVersion
(
 DATASERVICE_REF_DATA_VER u32Version,
 DATASERVICE_ERROR_CODE_ENUM eStatus,
 midw_ext_fi_tclString& rffiObj
 )
{   
    if(DATASERVICE_ERROR_CODE_NONE == eStatus)
    {
        fc_sxm_vCopyInt2FiString(u32Version,rffiObj,FC_SXM_DB_VERSIONS_MAX_LEN);
    }
    else
    {
        tCString sStrIn = ((m_vecDbErrorStrings[eStatus]).c_str());
        fc_sxm_vString2Fi(sStrIn,rffiObj);
    }
}


