/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diag_download.cpp
* @brief       SMS Firmware download functionality provider.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"
#include "fc_sxm_tcl_states.h"
//#include "fc_sxm_sms_util.h"
#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_tcl_diag_properties.h"
#include "fc_sxm_diaglibhandler.h"
#include "fc_sxm_tcl_diaghandler.h"
#include "fc_sxm_tcl_diag_download.h"
#include "fc_sxm_client_download.h"
#include "fc_sxm_client_download_fi.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diag_download.cpp.trc.h"
#endif
 
/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownload::vPreRun()
 *
 * DESCRIPTION: This method is called before the actual run to give some time for the
 *              SMS to shut down.
 *
 * PARAMETER: NONE
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vPreRun() {
    SXM_ASSERT_RETURN(_enState==fc_sxm_enDiagDownloadState_Pending);
    vSetState(fc_sxm_enDiagDownloadState_PreRun);
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownload::vRun()
 *
 * DESCRIPTION: This method starts the firware download activity
 *
 * PARAMETER: NONE
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vRun() {
    SXM_ASSERT_RETURN(_enState==fc_sxm_enDiagDownloadState_PreRun);
    string oCurSwVersion=fc_sxm_tclDiagHandler::instance()->rGetModuleVersInfo().oSwRev;
    ETG_TRACE_USR1(("fc_sxm_tclDiagDownloadActivity::vRun START, Ver:%20s->%20s bForced=%d oFileName=%s",
                    oCurSwVersion.c_str(), 
                    _rRequestMsg.oSwVersion.c_str(), 
                    _rRequestMsg.bForced, 
                    _rRequestMsg.oFileName.c_str()));

    if (_rRequestMsg.oSwVersion==oCurSwVersion && !_rRequestMsg.bForced) {
        // version is already OK
        ETG_TRACE_USR1(("fc_sxm_tclDiagDownloadActivity::vRun: version unchanged"));
        // terminate activity, correct system-state should be entered
        vSendProgress(TRUE,100);
        vSetState(fc_sxm_enDiagDownloadState_Idle, fc_sxm_enDiagDownloadResult_Ok);
        return;
    }
    // request progress-messages from SMS-module
    fc_sxm_tclSmsModule::instance()->vChangeMask(fc_sxm_enMaskMode_Add, MODULE_OBJECT_EVENT_UPDATE_PROGRESS);

    SMSAPI_RETURN_CODE_ENUM enSmsRes = 
        MODULE.eFirmwareUpdate (fc_sxm_tclSmsModule::instance()->hGetSmsModule(),
                                _rRequestMsg.oFileName.c_str());
    
    if(enSmsRes==SMSAPI_RETURN_CODE_SUCCESS)
    {
        vSetState(fc_sxm_enDiagDownloadState_Running);
    }
    else
    {
        ETG_TRACE_USR1(("fc_sxm_tclDiagDownloadActivity::vRun: could not start update in sms (%d)",
                        ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enSmsRes)));

        if(enSmsRes==SMSAPI_RETURN_CODE_MODULE_FWUPDATE_FILE_VERSION_INVALID)
        {
        	vSetState(fc_sxm_enDiagDownloadState_Idle,fc_sxm_enDiagDownloadResult_InvalidVersion);
        }
        else
        {
        	vSetState(fc_sxm_enDiagDownloadState_Idle, fc_sxm_enDiagDownloadResult_Failed);
        }

        fc_sxm_tclSmsModule::instance()->vChangeMask(fc_sxm_enMaskMode_Remove, MODULE_OBJECT_EVENT_UPDATE_PROGRESS);

        return;
    }

}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagLog::vSendMethodStartSaveTestResult
 *(
 *  fc_sxm_trMsgDiagModuleUpdateEvent const *prMsg
 *)
 *
 * DESCRIPTION: This method is called whenever the SMS gives the update for firmware
 *              download progress
 *
 * PARAMETER: fc_sxm_trMsgDiagModuleUpdateEvent const *prMsg : event handle
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vProcess
(
		fc_sxm_trMsgDiagModuleUpdateEvent const *prMsg
)
{
    
    (tVoid)prMsg; //not used resolving prio 2 lint
    tU8 u8UpdateProgress=0;
    SMSAPI_RETURN_CODE_ENUM enSmsRes = 
        MODULE.eUpdateProgress (fc_sxm_tclSmsModule::instance()->hGetSmsModule(),
                                &u8UpdateProgress);
    if (fc_sxm_enDiagDownloadState_Running != _enState) {
        ETG_TRACE_ERR(("fc_sxm_tclDiagDownload::vProcess(fc_sxm_trMsgDiagModuleUpdateEvent): not running (%d)",
                       ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState)));
        vSendResult(fc_sxm_enDiagDownloadResult_Abort);
        return;
    }

    if (SMSAPI_RETURN_CODE_SUCCESS != enSmsRes) {
        ETG_TRACE_ERR(("fc_sxm_tclDiagDownload:: MODULE.eUpdateProgress returned: %d",
                       ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enSmsRes)));
        return;
    }
    if (u8UpdateProgress > 100) {
        ETG_TRACE_ERR(("fc_sxm_tclDiagDownload:: invalid (>100) u8UpdateProgress=%d",
                       u8UpdateProgress));
        return;
    }
    vSendProgress(TRUE,u8UpdateProgress);
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownload::vSetState
 * (
 *		fc_sxm_tenDiagDownloadState enState,
 *		fc_sxm_tenDiagDownloadResult enResult
 *)
 *
 *
 * DESCRIPTION: This method is used to set the state and result for download
 *
 * PARAMETER: fc_sxm_tenDiagDownloadState enState :The current download state
 *		      fc_sxm_tenDiagDownloadResult enResult :The current result
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vSetState
(
		fc_sxm_tenDiagDownloadState enState,
		fc_sxm_tenDiagDownloadResult enResult
)
{
    if (_enState != enState) {
        ETG_TRACE_USR1(("fc_sxm_tclDiagDownload::vSetState: %d->%d",
                        ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState),
                        ETG_CENUM(fc_sxm_tenDiagDownloadState, enState)));
        _enState = enState;
    }
    switch (_enState) {
        case fc_sxm_enDiagDownloadState_Idle: 
        {
            _oSupervisionTimer.vStop();
            _rRequestMsg=fc_sxm_trMsgDiagDownload();
            if (_oDownloadActivity.enGetState()!=fc_sxm_enActivityState_Idle) {
                _oDownloadActivity.vOnDone();
            }
            vSendResult(enResult);
        }
        break;


        case fc_sxm_enDiagDownloadState_Pending:
            _oSupervisionTimer.vStart(FC_SXM_DIAG_DL_MAX_WAIT_START_MS);
            _oDownloadActivity.bRequest();
            break;

        case fc_sxm_enDiagDownloadState_PreRun:
            _oSupervisionTimer.vStart(FC_SXM_DIAG_DL_WAIT_PRE_RUN_MS);
            _oDownloadActivity.bRequest();
            break;


        case fc_sxm_enDiagDownloadState_Running:
            _oSupervisionTimer.vStart(FC_SXM_DIAG_DL_MAX_WAIT_MS);
            break;

        case fc_sxm_enDiagDownloadState_WaitRestart:
            _oDownloadActivity.vOnDone();
            _oSupervisionTimer.vStart(FC_SXM_DIAG_DL_MAX_WAIT_RESTART_MS);
            break;

        default:
            ETG_TRACE_ERR(("fc_sxm_tclDiagDownload::vSetState: invalid %d->%d",
                           ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState),
                           ETG_CENUM(fc_sxm_tenDiagDownloadState, enState)))
                vSetState(fc_sxm_enDiagDownloadState_Idle);
            break;
    }
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagLog::vSendMethodStartSaveTestResult
 *(
 *  fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg
 *)
 *
 * DESCRIPTION: This method is called whenever the SMS module status changes
 *
 * PARAMETER: fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg : event handle
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vProcess
(
		fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg
)
{
    MODULE_STATE_ENUM enModuleState = prMsg->eState;
    ETG_TRACE_USR1(("fc_sxm_tclDiagDownload::vProcess(fc_sxm_trMsgAudioSmsEvtModuleState) moduleState=%d _enState=%d",
                    ETG_CENUM(MODULE_STATE_ENUM, enModuleState),
                    ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState)));
    switch (_enState) 
    {
        case fc_sxm_enDiagDownloadState_Idle:
            break;

        case fc_sxm_enDiagDownloadState_Running:
        {
            if (MODULE_STATE_INITIAL == enModuleState) {
                /* module has restarted after download has finished
                   now lets wait until module is ready again to check if it
                   has the correct versio now
                */
                vSetState(fc_sxm_enDiagDownloadState_WaitRestart);
            } else if (MODULE_STATE_UPDATING != enModuleState) {
                /* even in error-conditions fc_sxm_tclSystemStates should bring MODULE
                   back to MODULE_STATE_INITIAL,
                   so no error-handling her.
                */
                ETG_TRACE_ERR(("fc_sxm_tclDiagDownload::vProcess(fc_sxm_trMsgAudioSmsEvtModuleState) unexpected moduleState(%d) in state (%d)",
                               ETG_CENUM(MODULE_STATE_ENUM, enModuleState),
                               ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState)));
            }
        }
        break;
        case fc_sxm_enDiagDownloadState_WaitRestart:
        {
            if (MODULE_STATE_READY == enModuleState) {
                
                string oCurSwVersion=fc_sxm_tclDiagHandler::instance()->rGetModuleVersInfo().oSwRev;
                if (_rRequestMsg.oSwVersion==oCurSwVersion) {
                    ETG_TRACE_USR1(("fc_sxm_tclDiagDownload: new version %s installed",
                                    oCurSwVersion.c_str()));
                    vSetState(fc_sxm_enDiagDownloadState_Idle, fc_sxm_enDiagDownloadResult_Ok);
                } else {
                    ETG_TRACE_USR1(("fc_sxm_tclDiagDownload: new version not installed requested:%20s current:%s",
                                    _rRequestMsg.oSwVersion.c_str(),
                                    oCurSwVersion.c_str()));
                    vSetState(fc_sxm_enDiagDownloadState_Idle, fc_sxm_enDiagDownloadResult_Failed);
                }
            }
        }
        break;
        default:
            break;
    }
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownload::vSendProgress
 * (
 *    tBool bActive,
 *		tU8 u8Progress
 * ) const
 *
 * DESCRIPTION: This method is used to send firmware update progress to FC_Diagnosis
 *
 * PARAMETER: tU8 u8Progress : Progress in %
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vSendProgress
(
      tBool bActive,
		tU8 u8Progress
) const
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagDownload:: u8Progress=%d", u8Progress));

    midw_ext_sxm_audiofi_tclMsgFirmwareDownloadProgressStatus &rFiDnlProgressData = fc_sxm_tclDiagProperties::instance()->oSxmDownloadProgress.rAccess().oFiMsg;
    rFiDnlProgressData.bActive = bActive;
    rFiDnlProgressData.Progress = u8Progress;
    fc_sxm_tclDiagProperties::instance()->oSxmDownloadProgress.vRelease();
    fc_sxm_tclDiagProperties::instance()->oSxmDownloadProgress.vNotify();

}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownload::vSendResult
 * (
 *		fc_sxm_tenDiagDownloadResult enResult
 * ) const
 *
 * DESCRIPTION: This method is used to send firmware update result to FC_Diagnosis
 *
 * PARAMETER: enResult : method result
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vSendResult
(
		fc_sxm_tenDiagDownloadResult enResult
) const
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagDownload::vSendResult(%d)",
                    ETG_CENUM(fc_sxm_tenDiagDownloadResult, enResult)));
    midw_ext_sxm_audiofi_tclMsgFirmwareDownloadMethodResult oFiTxObj;
    oFiTxObj.enResult.enType = static_cast<midw_ext_fi_tcl_e8_SxmFirmwareDownloadResult::tenType>(enResult);
    /* send the answer to the client using adressing-infomation: */
    fc_sxm_tclAudioService::instance()->enSendFiMessage(m_rAdressing, oFiTxObj);

    vSendProgress(FALSE,0);
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownloadActivity::vRun()
 *
 * DESCRIPTION: This actual run function of avtivity class
 *
 * PARAMETER: NONE
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownloadActivity::vRun()
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagDownloadActivity::vRun START"));
    fc_sxm_tclDiagDownload::instance()->vPreRun();
    ETG_TRACE_USR1(("fc_sxm_tclDiagDownloadActivity::vRun END"));
}


/*******************************************************************************
 *
 * FUNCTION: fc_sxm_tclDiagDownload::fc_sxm_tclDiagDownload()
 *
 * DESCRIPTION: Construcutor
 *
 * PARAMETER: NONE
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
fc_sxm_tclDiagDownload::fc_sxm_tclDiagDownload():
    _enState(fc_sxm_enDiagDownloadState_Idle) 
{}



/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownload::vProcess
 * (
 *		fc_sxm_trMsgDiagDownloadMStart const *prMsg
 * ) const
 *
 * DESCRIPTION: This method is used to start download after receiving trigger from FC_Diagnosis
 *
 * PARAMETER: fc_sxm_trMsgDiagDownloadMStart const *prMsg:Contains the file path and bforced flag
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vProcess
(
		fc_sxm_trMsgDiagDownloadMStart const *prMsg
)
{
	ETG_TRACE_USR1(("fc_sxm_tclDiagDownload::fc_sxm_trMsgDiagDownloadMStart received"));

   /*Send progress as we accepted method start*/
   vSendProgress(TRUE,0);

	if (fc_sxm_enDiagDownloadState_Idle != _enState)
	{
		ETG_TRACE_ERR(("fc_sxm_trMsgDiagDownload; BUSY! (%d)",
				ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState)));
		vSendResult(fc_sxm_enDiagDownloadResult_Busy);
		return;
	}

	m_rAdressing = prMsg->rAdressing;

	tString sFileName = prMsg->oFiMsg.File.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
	tString sSWVersion = prMsg->oFiMsg.Version.szGet(midw_ext_fi_tclString::FI_EN_UTF8);

	string oFileName(sFileName);
	string oSwVersion(sSWVersion);
	tBool bForced = prMsg->oFiMsg.bForced;

	ETG_TRACE_ERR(("fc_sxm_trMsgDiagDownload:  (state=%d) bForced=%d oSwVersion=%30s oFileName=%s",
	                   ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState),
	                   bForced, oSwVersion.c_str(), oFileName.c_str()));

	_rRequestMsg = fc_sxm_trMsgDiagDownload(oFileName,oSwVersion,bForced);
	OSAL_DELETE [] sFileName;
	OSAL_DELETE [] sSWVersion;

	vSetState(fc_sxm_enDiagDownloadState_Pending);
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDownload::vSendResult
 * (
 *		fc_sxm_trMsgDiagDlSvTimer const *prMsg
 * ) const
 *
 * DESCRIPTION: Supervision-timeout. (always error-condition)
 *
 * PARAMETER: fc_sxm_trMsgDiagDlSvTimer const *prMsg:Timer event handle
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDownload::vProcessTimer
(
		fc_sxm_trMsgDiagDlSvTimer const *prMsg
)
{
    (tVoid)prMsg;
    ETG_TRACE_ERR(("fc_sxm_tclDiagDownload::fc_sxm_trMsgDiagDlSvTimer state:%d",
                   ETG_CENUM(fc_sxm_tenDiagDownloadState, _enState)));
    switch(_enState) {
        case fc_sxm_enDiagDownloadState_Idle:  
            break;
        case fc_sxm_enDiagDownloadState_PreRun:
            vRun();
            break;
        case fc_sxm_enDiagDownloadState_Pending:
        case fc_sxm_enDiagDownloadState_Running:
        case fc_sxm_enDiagDownloadState_WaitRestart:
            vSetState(fc_sxm_enDiagDownloadState_Idle, fc_sxm_enDiagDownloadResult_Failed);
            break;
        default:
            break;
    }
}


