/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diag_defset.cpp
* @brief       SXM Diagnosis Default settings handler.
*              Responsible for Handling the Default settings request from FC_Diagnosis and HMI.
*              Default settings include Restoring the Databases to the factory default and 
*              removing all user data.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <errno.h>

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"

#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_tcl_pers_data.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_tcl_diag_usr_data_clear.h"
#include "fc_sxm_tcl_diagtm_handler.h"
#include "fc_sxm_tcl_diag_defset.h"
#include "fc_sxm_diaglibhandler.h"
#include "fc_sxm_tcl_config.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diag_defset.cpp.trc.h"
#endif

#define MAX_ARGS 6
#define BUFF_SIZE 256

tVoid fc_sxm_tclDiagRestoreSmsDbActivity::vRun() {
    ETG_TRACE_USR1(("fc_sxm_tclDiagRestoreSmsDbActivity::vRun START"));
    fc_sxm_tclPersData::instance()->vSetPersistentDbError(TRUE);
    /*Restore the DB*/
    fc_sxm_tclSmsDirRestorer::instance()->bRestore();

    /*Set the persistent data to default*/
    fc_sxm_tclPersData::instance()->vSetDefaultSettings();
    ETG_TRACE_USR1(("fc_sxm_tclDiagRestoreSmsDbActivity::call vOnDone()"));
    /*The activity done*/
    vOnDone();

    /*Send the result to HMI*/
    fc_sxm_tclDiagDefset::instance()->vHandleDbRestoreResult();

    ETG_TRACE_USR1(("fc_sxm_tclDiagRestoreSmsDbActivity::vRun END"));
}


tVoid fc_sxm_tclDiagDefsetFactoryActivity::vRun() {
    ETG_TRACE_USR1(("fc_sxm_tclDiagDefsetFactoryActivity::vRun START"));
    /*Restore the DB*/
    fc_sxm_tclSmsDirRestorer::instance()->bRestore();
    /*Set the persistent data to default*/
    fc_sxm_tclPersData::instance()->vSetDefaultSettings();
    /* don't restart sms, this would change the data again */
    /*Send the result to FC_Diagnosis*/
    fc_sxm_tclDiagDefset::instance()->vHandleFactoryDefSet();
   

    ETG_TRACE_USR1(("fc_sxm_tclDiagDefsetFactoryActivity::vRun END"));
}

tVoid fc_sxm_tclDiagDefsetUserActivity::vRun() {
    ETG_TRACE_USR1(("fc_sxm_tclDiagDefsetUserActivity::vRun START"));

    /*Restore sms.cfg*/
    fc_sxm_tclSmsDirRestorer::instance()->bRestoreSmsCfg(TRUE);

    /*Set the persistent data to default*/
    fc_sxm_tclPersData::instance()->vSetDefaultSettings();

    /*The activity done*/
    vOnDone();
    /*Send the result to FC_Diagnosis*/
    fc_sxm_tclDiagDefset::instance()->vHandleResult(fc_sxm_enDiagResult_Ok);
    ETG_TRACE_USR1(("fc_sxm_tclDiagDefsetUserActivity::vRun END"));
}

fc_sxm_tclDiagDefset::fc_sxm_tclDiagDefset():
    _enDefsetType(fc_sxm_enDiagDefsetType_Invalid),m_bInformHMI(FALSE),
    _enSmsCfgState(enSmsCfgState_OK),m_enDefsetState(fc_sxm_en_Defset_Idle)
{
    fc_sxm_tclSmsDirRestorer::instance()->vStartupCheck();

    if (fc_sxm_tclPersData::instance()->bGetPersistentDbError()) {
        // start timer. if everything is still ok after 10 min, we delete the error-flag
        _oDbErrorDetectTimer.vStart(600 * 1000);
    }

}

fc_sxm_tclDiagDefset::~fc_sxm_tclDiagDefset() {
    /*Delete the instance*/
    OSAL_DELETE fc_sxm_tclSmsDirRestorer::instance();
}

tVoid fc_sxm_tclDiagDefset::vHandleResult(fc_sxm_tenDiagResult enResult) {
    ETG_TRACE_USR1(("fc_sxm_tclDiagDefset::vOnDefsetResult: _enDefsetType=%d enResult=%d",
                    ETG_CENUM(fc_sxm_tenDiagDefsetType, _enDefsetType), 
                    ETG_CENUM(fc_sxm_tenDiagResult, enResult)));
    /*Send the result to FC_Diagnosis if defset type is not valid*/
    if (_enDefsetType!=fc_sxm_enDiagDefsetType_Invalid) {
        // todo: notify diag-lib, use: fc_sxm_vPostMsgToCcaObj()
        fc_sxm_trMsgDiagDefSetResult rDMsg(enResult);
        /*Post the message directly to CCA thread*/
        fc_sxm_vPostMsgToCcaObj(fc_sxm_diagLibHandler::instance(), rDMsg);

        _enDefsetType=fc_sxm_enDiagDefsetType_Invalid;
    }

}

// we get notified, that the data-base of a data-service is missing or corrupted
tVoid fc_sxm_tclDiagDefset::vProcess(fc_sxm_trMsgDiagSmsDbError const *prMsg) {
    fc_sxm_tenServiceID enServiceId=prMsg->enServiceId;
    tBool bAppError=fc_sxm_tclPersData::instance()->bGetPersistentDbError(enServiceId);
    tBool bGlobalError=fc_sxm_tclPersData::instance()->bGetPersistentDbError();
    ETG_TRACE_USR1(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagSmsDbError: App=%d bAppError=%d bGlobalError=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, enServiceId), bAppError, bGlobalError));
    

    if (bAppError) {
        // Data-service had already reported an db-error, so try to reset all
#ifdef FC_SXM_ENABLE_AUTO_RESTORE_SMS_HB
        if (!bGlobalError) {
            tBool bAccepted=_oRestoreSmsDbActivity.bRequest();
            if (bAccepted) {
                fc_sxm_tclPersData::instance()->vSetPersistentDbError(TRUE); 
            } else {
                ETG_TRACE_ERR(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagSmsDbError: Could not start Activity"));
            }
        }
#endif
    } else {
#ifdef FC_SXM_ENABLE_AUTO_RESTORE_SMS_HB_SERVICE
        fc_sxm_tclPersData::instance()->vSetPersistentDbError(TRUE, prMsg->enServiceId);
        // let data-service try to restore its db
        fc_sxm_trMsgAppMgrRestoreSmsDb rMsg(prMsg->enServiceId);
        fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsg);
#endif
    }

}

/*
  Supervision-timeout.
  After restoring the complete sms-sb, the timer is started.
  When no new db-error occured until timeout we assume that the databases are ok again and we
  remove the persitent error-flag.
*/
tVoid fc_sxm_tclDiagDefset::vProcessTimer(fc_sxm_trMsgDiagDbErrorDetectTimer const *prMsg) const {
    (tVoid)prMsg;
    /* no db-error has been detected until now, so db seems to be ok again.
       this handling prevents us from restoring the db on every startup if we cannot repair it
    */
    fc_sxm_tclPersData::instance()->vSetPersistentDbError(FALSE);
}

tVoid fc_sxm_tclDiagDefset::vProcess(fc_sxm_trMsgDiagDefset const *prMsg) {
   ETG_TRACE_USR1(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagDefset type=%d",
      ETG_CENUM(fc_sxm_tenDiagDefsetType, prMsg->enType)));
   if ((_enDefsetType != fc_sxm_enDiagDefsetType_Invalid)
      &&
      (prMsg->enState == m_enDefsetState)//If Idle 
      ) {
      ETG_TRACE_USR1(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagDefset busy: type=%d",
         ETG_CENUM(fc_sxm_tenDiagDefsetType, _enDefsetType)));
      // todo: send error-message to diaglib
      return;
   }

   _enSmsCfgState=enSmsCfgState_OK;
   // store type first, result might come synchronously
   _enDefsetType=prMsg->enType;
   tBool bSuccess = FALSE;
   switch (_enDefsetType)
   {    
   case fc_sxm_enDiagDefsetType_Factory:      
      vHandleFactoryDefSet( prMsg->enState);
      bSuccess = TRUE; 
      break;
   case fc_sxm_enDiagDefsetType_User:
      if (_oDefsetUserActivity.bRequest()) {
         bSuccess = TRUE;
      } else {
         ETG_TRACE_ERR(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagDefset(User): bRequest failed"));
      }
      break;
   case fc_sxm_enDiagDefsetType_UserOld:
      if (fc_sxm_tclDiagUsrDataClear::instance()->bStart()) {
         bSuccess = TRUE;
      } else {
         ETG_TRACE_ERR(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagDefset(User): bRequest failed"));
      }
      break;
   default:
      ETG_TRACE_ERR(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagDefset(%d): invalid defset type",
         _enDefsetType));
      break;        
   }

   if (!bSuccess) {
      vHandleResult(fc_sxm_enDiagResult_Failed);
   }
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagDefset::vHandleFactoryDefSet
*(
*  fc_sxm_tenDefsetState enDefsetState
*)
*
* DESCRIPTION: This method is used to determine the factory defset result to be
* send to Diagnosis.Actual defset to be performed when we get the Pre defset trigger 
* by routine control mechanism and send the result when we get the actual system set trigger
*
* PARAMETER: fc_sxm_tenDefsetState enDefsetState 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagDefset::vHandleFactoryDefSet
(
 fc_sxm_tenDefsetState enDefsetState
 )
{

   ETG_TRACE_USR4(("fc_sxm_tclDiagDefset::vHandleFactoryDefSet received defset state = %d",
      ETG_CENUM(fc_sxm_tenDefsetState, enDefsetState)));  

   tBool bSuccess = FALSE;

   switch(m_enDefsetState)
   {
   case fc_sxm_en_Defset_Idle:
      {
         ETG_TRACE_USR2(("fc_sxm_tclDiagDefset::vHandleFactoryDefSet[Actual State]  Idle"));
         m_enDefsetState = enDefsetState;

         if (TRUE != (_oDefsetFactoryActivity.bRequest()))
         {
            ETG_TRACE_ERR(("fc_sxm_tclDiagDefset::vHandleFactoryDefSet bRequest failed"));
         }
      }
      break;
   case fc_sxm_en_Defset_Pending:
      {
         ETG_TRACE_USR2(("fc_sxm_tclDiagDefset::vHandleFactoryDefSet[Actual State] Pending"));         
         bSuccess = TRUE;
         m_enDefsetState = fc_sxm_en_Defset_Done;
      }
      break;
   case fc_sxm_en_PreDefset_Active:
      {
         ETG_TRACE_USR4(("fc_sxm_tclDiagDefset::vHandleFactoryDefSet[Actual State]Active"));
         m_enDefsetState=fc_sxm_en_Defset_Done;
      }
      break;
   case fc_sxm_en_Defset_Done:
      {
         ETG_TRACE_USR4(("fc_sxm_tclDiagDefset::vHandleFactoryDefSet[Actual State] done"));
         //Send OK
         bSuccess = TRUE; 
         m_enDefsetState = fc_sxm_en_Defset_Idle;

      }
      break;
   default :
      ETG_TRACE_ERR(("fc_sxm_tclDiagDefset::vHandleFactoryDefSet type not required to handle"));  
      break;
   }

   if(TRUE == bSuccess)
   {
      vHandleResult(fc_sxm_enDiagResult_Ok);
   }

}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDefset::vProcess
 *(
 *  fc_sxm_trMsgDiagRestoreSmsDb const *prMsg
 *)
 *
 * DESCRIPTION: This method is called on dispatch of message for resetting SXM data bases .
 *
 * PARAMETER: const *prMsg : fc_sxm_trMsgDiagRestoreSmsDb 
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDefset::vProcess
(
 fc_sxm_trMsgDiagRestoreSmsDb const *prMsg
 )
{
    tU8 u8Service = prMsg->oFiMsg.tU8Request;

    ETG_TRACE_USR4(("fc_sxm_tclDiagTmHandler::vProcess(fc_sxm_trMsgDiagRestoreSmsDb)u8Service=%d",u8Service));  

    vStartDbReset(u8Service);

    /* send the answer to the client using adressing-infomation: */
    /*midw_ext_sxm_audiofi_tclMsgDiagSxmResetMethodResult oFiTxObj;
    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oFiTxObj);*/
    m_rAdressing = prMsg->rAdressing;
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDefset::vStartDbReset
 *(
 *  tU8 u8Service
 *)
 *
 * DESCRIPTION: This method is used to perform of the data base reset .
 *
 * PARAMETER: tU8 u8Servic : Identifier to find out the request from HMI
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagDefset::vStartDbReset
(
 tU8 u8Service
 )
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagDefset::vStartDbReset start")); 

    //Operation to convert the service info got from HMI to the internal Service ID Format
    tU16 u16ServiceIdInternal = (tU16)((((u8Service > 0)) ? (((u8Service << 8)) | 0x1 ):u8Service));

    if (u16ServiceIdInternal == fc_sxm_enServiceID_All) {
        // reset complete sms_hb
        tBool bAccepted=_oRestoreSmsDbActivity.bRequest();
        if (!bAccepted) {
            ETG_TRACE_ERR(("fc_sxm_tclDiagDefset::vProcess(fc_sxm_trMsgDiagRestoreSmsDb): Could not start Activity"));
        }
    } else {
        fc_sxm_trMsgAppMgrRestoreSmsDb rMsg((fc_sxm_tenServiceID)u16ServiceIdInternal);
        ETG_TRACE_USR1(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagRestoreSmsDb App=%d",
                        ETG_CENUM(fc_sxm_tenServiceID, (fc_sxm_tenServiceID)u16ServiceIdInternal)));
        /*Only in this case HMI response is required*/
        m_bInformHMI = TRUE;
        fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsg);
    }

}

//extern int SMSLIB_bSmsCfgError;//manoj

tVoid fc_sxm_tclDiagDefset::vHandleSmsInitError(SMSAPI_RETURN_CODE_ENUM eReturnCode) {
    /*ETG_TRACE_ERR(("vHandleSmsInitError(%d) _enSmsCfgState=%d SMSLIB_bSmsCfgError=%d",
                   ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode),
                   ETG_CENUM(fc_sxm_tclDiagDefset::tenSmsCfgState, _enSmsCfgState),*/
 
 ETG_TRACE_USR1(("fc_sxm_tclDiagDefset::vHandleSmsInitError returncode(%d)",
                   ETG_CENUM(SMSAPI_RETURN_CODE_ENUM,eReturnCode)));
#if 0
    switch (eReturnCode) {
        case SMSAPI_RETURN_CODE_CFG_VALUE_TOO_LONG:
        case SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET:
        case SMSAPI_RETURN_CODE_CFG_TAG_REMOVED:
        case SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE:
        case SMSAPI_RETURN_CODE_CFG_FILE_BAD_CHECKSUM:
        case SMSAPI_RETURN_CODE_CFG_BASE64_ERROR:
        case SMSAPI_RETURN_CODE_CFG_NO_PARENT:
        {
            break;
        }
        default:
            return;
    }
#endif
    /*if (!SMSLIB_bSmsCfgError) {
        return;
    }*/
    switch (_enSmsCfgState) {
        case enSmsCfgState_OK:
        {
            // restore sms.cfg
            _enSmsCfgState=enSmsCfgState_Replaced;
            fc_sxm_tclSmsDirRestorer::instance()->bRestoreSmsCfg(TRUE);
            break;
        }
        case enSmsCfgState_Replaced:
        {
            _enSmsCfgState=enSmsCfgState_Deleted;
            fc_sxm_tclSmsDirRestorer::instance()->vDeleteSmsCfg();
            break;
        }
        case enSmsCfgState_Deleted:
        {
            _enSmsCfgState=enSmsCfgState_FAILED;
            break;
        }
        default:
            break;
    }
}

tVoid fc_sxm_tclSmsDirRestorer::vStartupCheck() {
    if (bMarkerExists()) {
        // everything seems to be ok, check sms.cfg in case sms has lost it
        fc_sxm_tclSmsDirRestorer::instance()->bRestoreSmsCfg(FALSE);
    } else {
        ETG_TRACE_ERR(("fc_sxm_tclSmsDirRestorer::vStartupCheck(): Marker Missing, restoring SMS"));
        bRestore();
    }
}

tVoid fc_sxm_tclSmsDirRestorer::vDoRestoreSpecificDb(string const &oSubDir,
        string &oSrcDir, string &oDstDir) const
{
	ETG_TRACE_USR1(("vDoRestoreSpecificDb Start"));
	string cmd;

	oDstDir += "/";
	oDstDir += oSubDir;
	oDstDir += "/";
	oDstDir += oSubDir;

	if(oSubDir == "stock_ticker")
	{
		oDstDir += "_ref.db";
	}

	else
	{
		oDstDir += ".db";
	}

	cmd += oDstDir;
	ETG_TRACE_USR1(("remove service specific db=%s", cmd.c_str()));
	vRemove(cmd);

	//for channelart clear all the files
	if(oSubDir == "channelart")
	{
		vClearChannelArt();
	}

	//for all the services data will be copied from static to dynamic folder
	cmd = "rsync -r ";
	oSrcDir += "/";
	oSrcDir += oSubDir;
	cmd += oSrcDir;
	cmd += " ";
	cmd += FC_SXM_DEFAULT_SMS_CFG_PATH;
	cmd += "/";
	ETG_TRACE_USR1(("Differential Copy from Static2Dynamic=%s", cmd.c_str()));
	vExecvpCmd(cmd);
	ETG_TRACE_USR1(("vDoRestoreSpecificDb End"));
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagDefset::vParseCmd
 *(
 *  char *argPath,char **argv
 *)
 *
 * DESCRIPTION: This method takes an input line and returns a zero-terminated
 * array of char pointers,each of which points to a zero-terminated character string.
 * Once vParseCmd() finds a non-white space,the address of that location is saved to
 * the current position of argv and the index is advanced.Then, parse() skips all
 * non-whtitespace characters. This process repeats until the end of string line is
 * reached and at that moment argv is terminated with a zero.
 *
 * PARAMETER: char *argPath,char **argv
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclSmsDirRestorer::vParseCmd(char *argPath,char **argv) const
{
    while (*argPath != '\0') {       /* if not the end of argPath ....... */
         while (*argPath == ' ' || *argPath == '\t' || *argPath == '\n')
              *argPath++ = '\0';     /* replace white spaces with 0    */
         *argv++ = argPath;          /* save the argument position     */

         while (((*argPath) != '\0') && ((*argPath) != ' ') &&
        		 (*argPath) != '\t' && (*argPath) != '\n')
        	 	 argPath++;             /* skip the argument until ...    */
    }
    *argv = '\0';                 /* mark the end of argument list  */
}


/*It takes array argv[], treats it as a command line arguments with the program name in argv[0],
forks a child process, and executes the indicated program in that child process. While the child
process is executing the command,the parent executes a wait(), waiting for the completion of the child.
In this special case, the parent knows the child's process ID and therefore is able to wait a specific
child to complete.*/
tVoid fc_sxm_tclSmsDirRestorer::vExecuteCmd(char **argv) const
{
	ETG_TRACE_USR1(("vExecvpCmd:: START"));
	pid_t pid;
	tS32 status;

	if ((pid = fork()) < 0) {     /* fork a child process  */
          ETG_TRACE_ERR(("vExecuteCmd:: ERROR forking child process failed\n"));
          exit(1);
     }
     else if (pid == 0) 
	 {   /* for the child process:         */
          if (execvp(*argv, argv) < 0) 
		  {     /* execute the command  */
               ETG_TRACE_ERR(("*** ERROR: exec failed\n"));
               exit(1);
          }
     }
     else 
	 {   /* for the parent:  */
          while (wait(&status) != pid)  /* wait for completion of the child process  */
               ;
     }
	ETG_TRACE_USR1(("vExecvpCmd:: END"));
}

tVoid fc_sxm_tclSmsDirRestorer::vExecvpCmd(string &cmd) const
{
	char *argv[MAX_ARGS];
	tChar arCmd[BUFF_SIZE];

	size_t u32Length = cmd.copy(arCmd,cmd.size(),0);
	arCmd[u32Length] = '\0';

	ETG_TRACE_USR1(("vExecvpCmd Length=%d",u32Length));
	vParseCmd(arCmd,argv);
	vExecuteCmd(argv);
}


tVoid fc_sxm_tclSmsDirRestorer::vDoRestoreFactory(string const &oSubDir,string &oSrcDir, string &oDstDir) const
{
	ETG_TRACE_USR1(("vDoRestoreFactory Start"));
	string oBaseDir = FC_SXM_SMS_FFS_BASE_PATH;
	string cmd;

	oSrcDir += "/";
	oSrcDir += oSubDir;
	oDstDir += "/";
	oDstDir += oSubDir;
	vRemove(oDstDir);

	cmd = "rsync -r ";
	cmd += oSrcDir;
	cmd += " ";
	cmd += oBaseDir+"/";
	ETG_TRACE_USR1(("Differential Copy from Static2Dynamic =%s", cmd.c_str()));
	vExecvpCmd(cmd);

	fc_sxm_tclConfig::instance()->vCheckKDSPresetType();

	ETG_TRACE_USR1(("vDoRestoreFactory End"));
}

tVoid fc_sxm_tclSmsDirRestorer::vDoRestoreFactoryMixed(string const &oSubDir,string &oSrcDir, string &oDstDir) const
{
    ETG_TRACE_USR1(("vDoRestoreFactoryMixed Start"));
    string oBaseDir = FC_SXM_SMS_FFS_BASE_PATH;
    string cmd;

    oSrcDir += "/";
    oSrcDir += oSubDir;
    oDstDir += "/";
    oDstDir += "sms.cfg";
    vRemove(oDstDir);

    cmd = "rsync -r ";
    cmd += oSrcDir;
    cmd += " ";
    cmd += oDstDir;
    cmd += "/";
    cmd +=  "sms.cfg";
    ETG_TRACE_USR1(("Differential Copy from Static2Dynamic =%s", cmd.c_str()));
    vExecvpCmd(cmd);
    ETG_TRACE_USR1(("vDoRestoreFactoryMixed End"));
}


tVoid fc_sxm_tclSmsDirRestorer::vDoRestoreSdkService(string const &oSubDir) const
{
	ETG_TRACE_USR1(("vDoRestoreSdkService Start"));
	string oSrcDir = FC_SXM_STATIC_SDK_W_PATH;
	string oDstDir = FC_SXM_DEFAULT_SDK_W_PATH;
    string cmd;

    oDstDir += "/";
    oDstDir += oSubDir;
    oDstDir += "/";
    oDstDir += "parkingdb";

	ETG_TRACE_USR1(("Parking db path =%s",oDstDir.c_str()));
	vRemove(oDstDir);
	cmd.clear();

	cmd = "rsync -r ";
	cmd += FC_SXM_STATIC_SDK_W_PATH;
	cmd += "/";
	cmd += oSubDir;
	cmd += " ";
	cmd += FC_SXM_DEFAULT_SDK_W_PATH;
	cmd += "/";
	ETG_TRACE_USR1(("Differential Copy from Static2Dynamic=%s", cmd.c_str()));
	vExecvpCmd(cmd);

	ETG_TRACE_USR1(("vDoRestoreSdkService End"));
}

tVoid fc_sxm_tclSmsDirRestorer::vDoRestoreStatesFile(string const &oSubDir,string &oSrcDir, string &oDstDir) const
{
	ETG_TRACE_USR1(("vDoRestoreStatesFile Start"));
	string cmd;

	oSrcDir += "/";
	oSrcDir += oSubDir;
	oDstDir += "/";
	oDstDir += oSubDir;
	vRemove(oDstDir);

	cmd = "rsync -r ";
	cmd += oSrcDir;
	cmd += " ";
	cmd += oDstDir+"/";
	ETG_TRACE_USR1(("Differential Copy from Static2Dynamic =%s", cmd.c_str()));
	vExecvpCmd(cmd);
	ETG_TRACE_USR1(("vDoRestoreStatesFile End"));

}

tVoid fc_sxm_tclSmsDirRestorer::vRestoreDefConfig(string const &oSubDir,string &oSrcDir, string &oDstDir) const
{
	ETG_TRACE_USR1(("vRestoreDefConfig  Start"));
    string cmd;
	// Factory restore
	if (oSubDir == "sms.cfg") {
		vDoRestoreFactory(oSubDir,oSrcDir,oDstDir);
	}
	//ClearAll databases execpt sdk service
	if (oSubDir == "") {

		//clear channelart folder contents
		vClearChannelArt();

		//copy data from static folder to dynamic folder
		cmd = "rsync -r ";
		cmd += oSrcDir;
		cmd += " ";
		cmd += FC_SXM_SMS_FFS_BASE_PATH;
		cmd += "/";
		ETG_TRACE_USR1(("Differential Copy from Static2Dynamic =%s", cmd.c_str()));
		vExecvpCmd(cmd);

	}
	// Added for Sports - Do this operation for Restore Factory Settings clear All and Clear Sports alone
	if ((oSubDir == "sms.cfg") || (oSubDir == "") || (oSubDir == "sports")) {
		cmd = FC_SXM_DEFAULT_T_SPORTS_CYCLE_PATH;
		vRemove(cmd);
	}
	if ((oSubDir == "sms.cfg") || (oSubDir == "") || (oSubDir == "parking")) {
		cmd = FC_SXM_DEFAULT_T_PARKING_CYCLE_PATH;
		vRemove(cmd);
	}

	if(oSubDir == "parking")
	{
		vDoRestoreSdkService(oSubDir);
	}

	if((oSubDir == "") || (oSubDir =="sms.cfg"))
	{
    	//Do nothing
	}
	else if(oSubDir !="parking")
	{
		vDoRestoreSpecificDb(oSubDir, oSrcDir, oDstDir);
	}

	//Set the path for states file
	if (oSubDir == "states")
	{
		oDstDir = FC_SXM_DEFAULT_SDK_R_PATH;
		oSrcDir = FC_SXM_STATIC_SDK_R_PATH;
		vDoRestoreStatesFile(oSubDir,oSrcDir, oDstDir);
	}
	vSyncFs();
	ETG_TRACE_USR1(("vRestoreDefConfig  End"));
}

//To clear all the files inside a Directory
//input : Directory path
//output : returns TRUE if function call is success else FALSE

tBool fc_sxm_tclSmsDirRestorer::bRemoveDirFiles(string const &oDir) const {

	ETG_TRACE_USR1(("bRemoveDirFiles Start"));
	tBool bRes = false;
	DIR *dir = OSAL_NULL;
	struct dirent *dirent = OSAL_NULL;

	//check directory is empty
	if ((dir = opendir(oDir.c_str())) != OSAL_NULL) {
		string FilePath = oDir;
		FilePath += "/";
		//initially set bRes to true
		bRes = true;

		//condition :: directory should not be empty and
		//remove file/DIR should be success (if remove(file/DIR) fails/returns non zero value, come out of the loop and close DIR)
		while (((dirent = readdir(dir)) != OSAL_NULL) && (bRes)) {
			if (strcmp(dirent->d_name, ".") && strcmp(dirent->d_name, "..")) {

				string path = FilePath + dirent->d_name;

				ETG_TRACE_USR1(("RemoveDirFiles :: Directory/File %s to be removed ", path.c_str()));
				//check for a Sub Directory
				if(dirent->d_type == DT_DIR)
				{
					//Recursively call bRemoveDirFiles to remove files inside sub directory
					(void) bRemoveDirFiles(path.c_str());

					//remove Sub directory
					bRes = OSAL.bFileSystemRemoveDir(path.c_str());
				}
				else
				{
					if(remove(path.c_str()) == -1)
					{
						//come out of while loop
						ETG_TRACE_ERR(("RemoveDirFiles ::Failed to remove file"));
						bRes = false;
					}
					else{
						bRes = true;
					}
				}
				ETG_TRACE_USR1(("RemoveDirFiles :: Directory/File removed, Result = %d", bRes));
			}
		}

		//close Directory
		if (dir != OSAL_NULL) {

			//if close DIR fails return false else return bRes value updated for remove file/DIR osal calls
			if (closedir(dir) == -1) {
				ETG_TRACE_ERR(("RemoveDirFiles :: Close Directory failed"));
				bRes = false;
			}
		}

	} else {
		//folder is already empty so bRes set as true
		bRes = true;
		ETG_TRACE_USR1(("bRemoveDirFiles :: Directory is empty"));
	}
	ETG_TRACE_USR1(("bRemoveDirFiles END bRes = %d", bRes));

	return bRes;
}

//Function call to clear channel art folder contents
tVoid fc_sxm_tclSmsDirRestorer::vClearChannelArt() const
{
	string oDir = FC_SXM_DEFAULT_SMS_CFG_PATH;
	oDir += "/channelart";
	(void)bRemoveDirFiles(oDir.c_str());
}


/*
 * Restore/clear the database for sms services
 * @brief
 * 1. Clear specific service db other than parking
 *    from "/var/opt/bosch/dynamic/ffs/siriusxm/sms_hb" & replace with "/var/opt/bosch/static/dynamic_init/ffs/siriusxm/sms_hb"
 * 2. Clear parking service db
 *    from "/var/opt/bosch/dynamic/ffs/siriusxm/W" & replace with "/var/opt/bosch/static/dynamic_init/ffs/siriusxm/W"
 * 3. Clear All db
 *    from "/var/opt/bosch/dynamic/ffs/siriusxm/sms_hb" & replace with "/var/opt/bosch/static/dynamic_init/ffs/siriusxm/sms_hb"
 *    And also from "/var/opt/bosch/dynamic/ffs/siriusxm/W" & replace with "/var/opt/bosch/static/dynamic_init/ffs/siriusxm/W"
 * @param  oSubDir service name
 * @param  bSync   always TURE
 * @return TRUE
 */
tBool fc_sxm_tclSmsDirRestorer::bRestore(string const &oSubDir, tBool bSync)
{
	fc_sxm_tclScopedLock oLock(_oSem);
	_bSync = bSync;
	string cmd;
	string oDstDir = FC_SXM_DEFAULT_SMS_CFG_PATH;
	string oSrcDir = FC_SXM_STATIC_SMS_CFG_PATH;

	ETG_TRACE_USR1(("fc_sxm_tclSmsDirRestorer::bRestore() bSync=%d dir=%s START", bSync, oSubDir.c_str()));

	vRestoreDefConfig(oSubDir,oSrcDir,oDstDir);
	vSetMarker();

	ETG_TRACE_USR1(("fc_sxm_bRestoreSmsDir(%s) END", oSubDir.c_str()));

	return TRUE;
}

string fc_sxm_tclSmsDirRestorer::oGetMarkerFileName() {
    string oRes=FC_SXM_DEFAULT_SMS_CFG_PATH;
    oRes+="/";
    oRes+="SmsDirRestoreMarker.txt";
    return oRes;
}

string fc_sxm_tclSmsDirRestorer::oGetMarkerFilePath() {
    string oRes=FC_SXM_DEFAULT_SMS_CFG_PATH;
    return oRes;
}



tVoid fc_sxm_tclSmsDirRestorer::vSyncFs() const {
    if (_bSync) {
        sync();
    }
}

tVoid fc_sxm_tclSmsDirRestorer::vSyncFileAndDir(string const &oFileNameAndPath, 
                                                string const &oFilePath) const {
    if (!_bSync) {
        return;
    }
    int fd;
    DIR *dirp;


    if (oFileNameAndPath.length()) {
        fd =open(oFileNameAndPath.c_str(),  O_RDWR);
        if (fd==-1) {
            ETG_TRACE_ERR(("vSyncFile: open failed on %s", oFileNameAndPath.c_str()));
            return;
        }
        if (fsync(fd) == -1) {
            ETG_TRACE_ERR(("vSyncFile: fsync failed on %s", oFileNameAndPath.c_str()));
        }
        if (close(fd) == -1) {
            ETG_TRACE_ERR(("vSyncFile: close failed on %s", oFileNameAndPath.c_str()));
            return;
        }
    }

    if ((dirp = opendir(oFilePath.c_str())) == 0) {
        ETG_TRACE_ERR(("vSyncFile: Failed to open dir: %s", oFilePath.c_str()));
        return;
    }
    fd = dirfd(dirp);
    if (fd == -1) {
        ETG_TRACE_ERR(("vSyncFile: Failed get fp: %s", oFilePath.c_str()));
        return;
    }
    if (fsync(fd) == -1) {
        ETG_TRACE_ERR(("vSyncFile: fsync failed for dir: %s", oFilePath.c_str()));
    }
    if ((closedir(dirp)) == -1) {
    	ETG_TRACE_ERR(("vSyncFile: closedir failed for dir: %s", oFilePath.c_str()));
    }
}

tVoid fc_sxm_tclSmsDirRestorer::vSetMarker() const {
    if (_bSync) {
        string cmd;
        cmd = "rsync -r ";
        cmd += FC_SXM_STATIC_SMS_CFG_PATH;
        cmd += "SmsDirRestoreMarker.txt";
        cmd += "";
        cmd += oGetMarkerFileName();
        vExecvpCmd(cmd);
        vSyncFileAndDir(oGetMarkerFileName(), oGetMarkerFilePath());
    }
}

tVoid fc_sxm_tclSmsDirRestorer::vRemove(string &cmd) const
{
	tChar arCmd[BUFF_SIZE];
	tS32 retVal;

	size_t u32Length = cmd.copy(arCmd,cmd.size(),0);
	arCmd[u32Length] = '\0';

	retVal = remove(arCmd);
	if (retVal == 0)
	{
		ETG_TRACE_USR1(("file removed successfully"));
	}
	else
	{
		ETG_TRACE_ERR(("Error: unable to remove the file"));
	}
	cmd.clear();
}


tVoid fc_sxm_tclSmsDirRestorer::vRemoveMarker() const {
    if (_bSync && bMarkerExists()) {
    	string cmd;
        cmd+=oGetMarkerFileName();
        vRemove(cmd);
        vSyncFileAndDir("", oGetMarkerFilePath());
    }
}

tVoid fc_sxm_tclSmsDirRestorer::vRestoreSmsHbOnNextRestart() {
    _bSync=TRUE;
    vRemoveMarker();
    _bSync=FALSE;
}

tBool fc_sxm_tclSmsDirRestorer::bSmsCfgFileExists() {
    string oRes=FC_SXM_DEFAULT_SMS_CFG_PATH;
    oRes+="/";
    oRes+="/sms.cfg";
    return fc_sxm_bFileExists(oRes);
}



tBool fc_sxm_tclSmsDirRestorer::bRestoreSmsCfg(tBool bForced) {
    tBool bFileExists=bSmsCfgFileExists();
    ETG_TRACE_USR4(("bRestoreSmsCfg: bForced=%d bFileExists=%d", bForced, bFileExists));
    if (bForced || !bFileExists) {
        return bRestore("sms.cfg", TRUE);
    }
    return TRUE;
}

tVoid fc_sxm_tclSmsDirRestorer::vDeleteSmsCfg() {
    ETG_TRACE_USR3(("vDeleteSmsCfg"));
    string cmd;
    cmd = FC_SXM_DEFAULT_SMS_CFG_PATH;
    cmd += "/sms.cfg";
    vRemove(cmd);
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagDefset::vProcess
*           (
*            fc_sxm_trMsgDiagSxmDbReplaced const *prMsg
*            )
*
* DESCRIPTION: This function is called whenever the Data base is replaced
*
* PARAMETER: fc_sxm_trMsgDiagSxmDbReplaced : message containing DB replacement
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagDefset::vProcess
(
 fc_sxm_trMsgDiagSxmDbReplaced const *prMsg
 )
{
   ETG_TRACE_USR4(("fc_sxm_tclDiagDefset::fc_sxm_trMsgDiagSxmDbReplaced"));
   (tVoid)prMsg;//not used resolving prio 2 lint

   /*SMS DB error also initiates a database restore of individual services
   the method result to HMI is only required if the data base restore is initiated from HMI
   Otherwise information to fc_sxm diagnosis is enough*/
   if(TRUE == m_bInformHMI)
   {
      vHandleDbRestoreResult();
      m_bInformHMI = FALSE;
   }    
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagDefset::vHandleDbRestoreResult
*
*
* DESCRIPTION: This function is used to send the method result for database restore
*
* PARAMETER: NONE
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagDefset::vHandleDbRestoreResult()
{
	ETG_TRACE_USR4(("fc_sxm_tclDiagDefset::vHandleDbRestoreResult"));
	midw_ext_sxm_audiofi_tclMsgDiagSxmResetMethodResult oFiTxObj;
   //Send the method result to HMI
	fc_sxm_tclAudioService::instance()->enSendFiMessage(m_rAdressing, oFiTxObj);
}

#if 0
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h> 
int main() {     
    int fd;
    DIR *dirp;
    int error;
    char path[] = "test_dir";
    if ((dirp = opendir(path)) == 0) {
        printf("Failed to open dir: %d\n", errno);
        return -1;
    }     
    fd = dirfd(dirp);
    if (fd == -1) {
        printf("Failed to open: %d\n", errno);
        return -1;     }       
    if (fsync(fd) == -1) {
        printf("fsync failed: %d\n", errno);
        return -1;     
    }     
return 0; 
}
#endif 
