/******************************************************************************
 *
 * FILE: tclDSMApp.cpp
 * PROJECT: AaRS
 * SW-COMPONENT: Data Service Manager
 *
 * DESCRIPTION: This file contains the DSM class interface specification
 *
 * AUTHOR: CM-AI/ECB2-Scholz
 *
 * COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
 * HISTORY:
 * Date | Rev. | Author | Modification
 * ----------------------------------------------------------------------------
 * 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
 *
 *****************************************************************************/
#include <stdexcept>
#include <stdarg.h>  // for va_start, etc
#include <memory>    // for std::unique_ptr
#include <algorithm> // for find

#include "tclDSM.h"
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_DSM_DAB_WIZARD
#include "trcGenProj/Header/tclDSMDABWizard.cpp.trc.h"
#endif

using namespace AaRSDABLib;
using namespace AaRSDABLib::API;
using namespace dsm;

#define WAITING_FOR_APP_SELECTION_TIMEOUT (1*60000)
#define WAITING_FOR_APP_ACTIVATION_TIMEOUT (10*60000)

#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABWizard::tclDSMDABWizard(tclDSM* dsm )
{
    if( dsm == NULL )
    {
        throw std::invalid_argument( "tclDSMDABWizard constructer argument must not NULL" );
    }

    _dsm                       = dsm;
    _u8NumTuner                = 0;
    _tDataServiceList          = NULL;
    _u8WizardDBQueryTag        = 0xff;
    _u64DataServiceListTimeout = 0;

    ETG_TRACE_USR1(("%p.tclDSMDABWizard(%p)",(void*)this,dsm));

    _wizard_job_list.clear();
    _blacklist_apps.clear();

}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABWizard::tclDSMDABWizardJob::tclDSMDABWizardJob(tU8 u8TunerIdx, std::list<tU16>  u16UserAppType_list, tU32 u32Strategy, tU8 u8DBQueryTag,tclDSMDABWizard* parent)
    : _parent(parent)
{
    tU16 u16UserAppType;

    u8TunerIdx_md            = u8TunerIdx;
    u16UserAppType_list_md   = u16UserAppType_list;
    u32Strategy_md           = u32Strategy;
    u8WizardJobDBQueryTag_md = u8DBQueryTag;
    u64StateTimeout_md       = 0;
    tCurrentState            = DAB_WIZARD_STATE__INIT;
    r_uri                    = "";
    u16user_app_type_current = 0;
    u8JobIdx_md              = 0;
    w_cmd                    = AaRSDABLib::API::DDM::tR_SELECT_URI::w_cmd__STOP;
    w_response               = AaRSDABLib::API::DDM::tR_SELECT_URI::w_response__OTHER_ERROR;

    for(std::list<tU16>::iterator u16UserAppType_list_iter = u16UserAppType_list.begin();
                u16UserAppType_list_iter != u16UserAppType_list.end();
                u16UserAppType_list_iter++)
    {
        u16UserAppType = *u16UserAppType_list_iter;
        ETG_TRACE_USR1(("%p.tclDSMDABWizardJob(%d,0x%16x,0x%08x,%d,%p)",(void*)this,u8TunerIdx,u16UserAppType,u32Strategy,u8DBQueryTag,parent));
        (void)u16UserAppType;
    }

    _parent->_dsm->vAddDSMEventListener(this);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABWizard::tclDSMDABWizardJob::~tclDSMDABWizardJob()
{
    ETG_TRACE_USR1(("~tclDSMDABWizardJob()"));

    _parent->_dsm->vRemoveDSMEventListener(this);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::tclDSMDABWizardJob::vDSMEvent( tDSMEvent ev, std::string uri )
{
    tCurrentStateType tNextState = tCurrentState;
    if( (ev==DSM_EV_APPLICATION_COMPLETE)||(ev==DSM_EV_APPLICATION_ALLREADY_COMPLETE) )
    {
        if( uri == r_uri )
        {
            if( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE )
            {
                tNextState = DAB_WIZARD_STATE__DONE;
            }
        }
    }
    if( ev == DSM_EV_APPLICATION_ERROR )
    {
        if( uri == r_uri )
        {
            if( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE )
            {
                _parent->vAddToBlacklist(r_uri);
                tNextState = DAB_WIZARD_STATE__DONE;
                _parent->_dsm->log("tclDSMDABWizard.vDSMEvent","%p Error %s -> Blacklist",this,r_uri.c_str());
            }
        }
    }
    if( tCurrentState != tNextState )
    {
        _parent->_dsm->log("tclDSMDABWizard.vDSMEvent","%p %d State (%d=>%d)",this,ev,tCurrentState,tNextState);
    }

    tCurrentState = tNextState;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABWizard::~tclDSMDABWizard()
{
    /* normaly, the application should call SwitchOff.
     * If not, this will free all job instances
     */
    while( 1 )
    {
        std::list<tclDSMDABWizardJob*>::iterator wizard_job_list_iter = _wizard_job_list.begin();
        if( wizard_job_list_iter == _wizard_job_list.end() )
        {
            break;
        }
        else
        {
            tclDSMDABWizardJob* wizard_job_inst = *wizard_job_list_iter;

            _wizard_job_list.remove(wizard_job_inst);
            delete wizard_job_inst;
        }
    }

    if( _tDataServiceList != NULL )
    {
        delete _tDataServiceList;
        _tDataServiceList = NULL;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vDSMEvent( tDSMEvent ev, std::string uri )
{
    for(std::list<tclDSMDABWizardJob*>::iterator list_iter = _wizard_job_list.begin();
        list_iter != _wizard_job_list.end();
        list_iter++)
    {
        tclDSMDABWizardJob* inst = *list_iter;
        inst->vDSMEvent(ev,uri);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vInitializeTables(tclDSM* dsm)
{
    (void)dsm;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABWizard::bIsBlackListApp( std::string uri )
{
    for(std::list<tclBlackListApp>::iterator list_iter = _blacklist_apps.begin();
        list_iter != _blacklist_apps.end();
        list_iter++)
    {
        if( ((tclBlackListApp)*list_iter).uri == uri )
        {
            return TRUE;
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vAddToBlacklist( std::string uri )
{
    _dsm->log("tclDSMDABWizard.vAddToBlacklist","%p %s",this,uri.c_str());
    if( bIsBlackListApp(uri) == FALSE )
    {
        tclBlackListApp app;
        app.uri = uri;
        app.host_timestamp = _dsm->u64ElapsedTimeMS()/1000;
        _blacklist_apps.insert(_blacklist_apps.end(),app);
    }
    else
    {
        for(std::list<tclBlackListApp>::iterator list_iter = _blacklist_apps.begin();
            list_iter != _blacklist_apps.end();
            list_iter++)
        {
            tclBlackListApp app = *list_iter;
            if( app.uri == uri )
            {
                app.host_timestamp = _dsm->u64ElapsedTimeMS()/1000;
                *list_iter = app;
            }
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vDeleteOlderAppsFromBlackList( tU64 host_timestamp )
{
    tBool bRun = TRUE;
    while(bRun == TRUE)
    {
        bRun = FALSE;
        for(std::list<tclBlackListApp>::iterator list_iter = _blacklist_apps.begin();
            list_iter != _blacklist_apps.end();
            list_iter++)
        {
            tclBlackListApp app = *list_iter;
            if( app.host_timestamp < host_timestamp )
            {
                _dsm->log("tclDSMDABWizard.vDeleteOlderAppsFromBlackList","%p %s",this,app.uri.c_str());
                _blacklist_apps.remove(app);
                bRun = TRUE;
                break;
            }
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vSwitchOff( )
{
    while( 1 )
    {
        std::list<tclDSMDABWizardJob*>::iterator wizard_job_list_iter = _wizard_job_list.begin();
        if( wizard_job_list_iter == _wizard_job_list.end() )
        {
            break;
        }
        else
        {
            tclDSMDABWizardJob* wizard_job_inst = *wizard_job_list_iter;

            wizard_job_inst->vSwitchOff();
            _wizard_job_list.remove(wizard_job_inst);
            delete wizard_job_inst;
        }
    }

    /*just for fear*/
    _dsm->oDAB()->vSendDSMMeCaCommand(
                 0,
                 DDM::tC_SELECT_URI(
                     DDM::tC_SELECT_URI::w_cmd__STOP_ALL,
                     (DDM::tC_SELECT_URI::t__b_on)0,
                     (DDM::tC_SELECT_URI::t__b_pause)0,
                     (DDM::tC_SELECT_URI::t__b_priority)0,
                     DDM::tC_SELECT_URI::b_decode_level_auto,
                     (DDM::tC_SELECT_URI::t__w_ua_type)0,
                     (DDM::tC_SELECT_URI::t__w_scidi)0,
                     (DDM::tC_SELECT_URI::t__w_uri_len)0,
                    (DDM::tC_SELECT_URI::t__r_uri)"",
                     (DDM::tC_SELECT_URI::t__w_rfu_len)0,
                     (DDM::tC_SELECT_URI::t__r_rfu)NULL));

     _dsm->log("tclDSMDABWizard.vSwitchOff","STOP_ALL");
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vSetNumTuner( tU8 u8NumTuner )
{
    if (u8NumTuner<2)
    {
        _u8NumTuner = u8NumTuner;
    }
    else
    {
        _u8NumTuner = 0;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU8 tclDSMDABWizard::u8GetNumTuner()
{
    return _u8NumTuner;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vSetDecodingCapability( tU8 u8TunerIdx, tU32 u32NumCurrentENSJobs, tU32 u32NumOtherENSJobs )
{
    (void)u8TunerIdx;
    (void)u32NumCurrentENSJobs;
    (void)u32NumOtherENSJobs;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vSetDecodingStrategy( tU8 u8TunerIdx, std::list<tU16>  u16UserAppType_list, tU32 u32Strategy )
{
    _wizard_job_list.insert(_wizard_job_list.end(),new tclDSMDABWizardJob(u8TunerIdx,
                                                                          u16UserAppType_list,
                                                                          u32Strategy,
                                                                          0xf0,
                                                                          this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABWizard::bNotSelected(std::string uri)
{
    /*check for the uri in the wizard job list*/
    for(std::list<tclDSMDABWizardJob*>::iterator list_iter = _wizard_job_list.begin();
        list_iter != _wizard_job_list.end();
        list_iter++)
    {
        tclDSMDABWizardJob* wizard_job_inst = *list_iter;

        if (uri==wizard_job_inst->sGetURI())
        {
            /*if found return FALSE*/
            return FALSE;
        }
    }
    /*if uri not found return TRUE*/
    return TRUE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::tclDSMDABWizardJob::vHandle()
{
    tU64 u64CurrentTime = _parent->_dsm->u64ElapsedTimeMS();
    tCurrentStateType tNextState = tCurrentState;
    tBool bStateTimeoutEvent = FALSE;

    if( u64StateTimeout_md == 0 )
    {
        bStateTimeoutEvent = TRUE;
        u64StateTimeout_md = u64CurrentTime;
    }
    else if( u64StateTimeout_md < u64CurrentTime )
    {
        bStateTimeoutEvent = TRUE;
        u64StateTimeout_md = u64CurrentTime;
    }

    if( tCurrentState == DAB_WIZARD_STATE__INIT )
    {
        u64StateTimeout_md = u64CurrentTime + 4000;
        u16user_app_type_current = 0;

        if (r_uri.empty() != TRUE)
        {
            /*delete the old uri from the members data*/
            r_uri.clear();
        }
        tNextState = DAB_WIZARD_STATE__START;
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__START )&&
            ( bStateTimeoutEvent == TRUE ))
    {
        if (0==_parent->u8GetCurrentReceiveableDataServiceList(false))
        {
            /*request list query send to ADR, wait for the response*/
            u64StateTimeout_md = u64CurrentTime + 5000;
            tNextState = DAB_WIZARD_STATE__WAITING_FOR_SERVICE_LIST;
        }
        else
        {
            /*request list query not send to ADR, use current list*/
            if( bSelectNextApp(u8TunerIdx_md) == TRUE )
            {
                u64StateTimeout_md = u64CurrentTime + WAITING_FOR_APP_SELECTION_TIMEOUT;
                tNextState = DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION;
            }
            else
            {
                u64StateTimeout_md = u64CurrentTime + 1000;
                tNextState = DAB_WIZARD_STATE__INIT;
            }
        }
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_SERVICE_LIST )&&
            ( bStateTimeoutEvent == FALSE ))
    {
        /*  ... waiting ... */
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_SERVICE_LIST )&&
            ( bStateTimeoutEvent == TRUE ))
    {
        /* timeout, restart ...
         */
        _parent->_dsm->log("tclDSMDABWizard.vHandle","wating for service list timeout, abort");
        u64StateTimeout_md = u64CurrentTime + 1000;
        tNextState = DAB_WIZARD_STATE__INIT;
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION )&&
            ( bStateTimeoutEvent == FALSE ))
    {
        /*  ... waiting ... */
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION )&&
            ( bStateTimeoutEvent == TRUE ))
    {
        /* timeout, switch off ...
         */
        _parent->_dsm->log("tclDSMDABWizard.vHandle","wating for app selection timeout, abort");
        _parent->vAddToBlacklist(r_uri);
        tNextState = DAB_WIZARD_STATE__DONE;
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_ACTIVATION )&&
            ( bStateTimeoutEvent == FALSE ))
    {
        /*  ... waiting ... */
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_ACTIVATION )&&
            ( bStateTimeoutEvent == TRUE ))
    {
        /* timeout, switch off ...
         */
        _parent->_dsm->log("tclDSMDABWizard.vHandle","wating for app activation timeout, abort");
        _parent->vAddToBlacklist(r_uri);
        tNextState = DAB_WIZARD_STATE__DONE;
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE )&&
            ( bStateTimeoutEvent == FALSE ))
    {
        /*  ... waiting ... */
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE )&&
            ( bStateTimeoutEvent == TRUE ))
    {
        /* done, stop app ...tNextState
         */
        _parent->_dsm->log("tclDSMDABWizard.vHandle","wating for app complete timeout, abort");

        if (((u32Strategy_md == DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE)&&
             (*u16UserAppType_list_md.begin()==2))
             ||
            (u32Strategy_md == DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE_COMPONENT))
        {
            /*do not stop the apps from current comp and the SLS from current service*/
            u64StateTimeout_md = u64CurrentTime + 120000;
        }
        else
        {
            tNextState = DAB_WIZARD_STATE__DONE;
        }

    }
    else if( tCurrentState == DAB_WIZARD_STATE__DONE )
    {
        _parent->_dsm->oDAB()->vSendDSMMeCaCommand(
                    0,
                    DDM::tC_SELECT_URI(
                        DDM::tC_SELECT_URI::w_cmd__STOP,
                        (DDM::tC_SELECT_URI::t__b_on)0,
                        (DDM::tC_SELECT_URI::t__b_pause)0,
                        (DDM::tC_SELECT_URI::t__b_priority)0,
                        DDM::tC_SELECT_URI::b_decode_level_auto,
                        (DDM::tC_SELECT_URI::t__w_ua_type)0,
                        (DDM::tC_SELECT_URI::t__w_scidi)0,
                        (DDM::tC_SELECT_URI::t__w_uri_len)r_uri.length(),
                        (DDM::tC_SELECT_URI::t__r_uri)r_uri.data(),
                        (DDM::tC_SELECT_URI::t__w_rfu_len)0,
                        (DDM::tC_SELECT_URI::t__r_rfu)NULL));

        _parent->_dsm->log("tclDSMDABWizard.vHandle","%p STOP %s selected)",this,r_uri.data());

        u64StateTimeout_md = u64CurrentTime + 3000;
        tNextState = DAB_WIZARD_STATE__WAITING_FOR_APP_DESELECTION;
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_DESELECTION )&&
            ( bStateTimeoutEvent == FALSE ))
    {
        /*  ... waiting ... */
    }
    else if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_DESELECTION )&&
            ( bStateTimeoutEvent == TRUE ))
    {
        /* timeout, switch off ...
         */
        _parent->_dsm->log("tclDSMDABWizard.vHandle","wating for app deselection timeout, go to init state");
        u64StateTimeout_md = u64CurrentTime + 1000;
        tNextState = DAB_WIZARD_STATE__INIT;
    }

    if( tCurrentState != tNextState )
    {
        _parent->_dsm->log("tclDSMDABWizard.vHandle","%p State (%d=>%d)",this,tCurrentState,tNextState);
    }

    /* make current next state to current state
     */
    tCurrentState = tNextState;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vHandle()
{
    for(std::list<tclDSMDABWizardJob*>::iterator list_iter = _wizard_job_list.begin();
        list_iter != _wizard_job_list.end();
        list_iter++)
    {
        tclDSMDABWizardJob* inst = *list_iter;
        inst->vHandle();
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU8 tclDSMDABWizard::u8GetCurrentReceiveableDataServiceList(tBool force)
{
    tU64 u64CurrentTime = _dsm->u64ElapsedTimeMS();

    /*request the list only every 10 sec*/
    if((_u64DataServiceListTimeout < u64CurrentTime) || (force==TRUE))
    {
        //char sql[1024] = "";
        _u64DataServiceListTimeout=u64CurrentTime+10000;

        /* store the current receiveable data service list for later use*/
        if( _tDataServiceList != NULL )
        {
            delete _tDataServiceList;
            _tDataServiceList = NULL;
        }

        _dsm->oDAB()->vSendDSMMeCaCommand(
                    0,
                    DB::tC_QUERY(
                        DB::tC_QUERY::b_cmd__EVALUATE,
                        _u8WizardDBQueryTag,
                        0,
                        0,
                        0,
                        "SELECT "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__DAB_URI     0
                        "dab_uri(C.o,U.user_app_x_pad_data), "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__E_LABEL     1
                        "E.label,"
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__S_LABEL     2
                        "S.label,"
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__C_LABEL     3
                        "C.label, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__U_LABEL     4
                        "U.label, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__USER_APP_TYPE 5
                        "U.user_app_type, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__CURRENT_COMP 6
                        "rdm_current(C.o), "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__CURRENT_SERV 7
                        "rdm_current(S.o), "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__CURRENT_ENS 8
                        "rdm_current(E.o), "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__USER_APP_DATA 9
                        "U.user_app_data, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__EID 10
                        "E.eid, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__TMID 11
                        "C.tmid, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__SUBCHID 12
                        "C.subchid, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__PADDR 13
                        "C.paddr, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__SCIDS 14
                        "C.scids, "
                #define DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__SID 15
                        "S.sid "
                        "FROM "
                        "v1f0_dab_ensemble AS E,"
                        "dab_service AS S,"
                        "dab_service_component AS C, "
                        "dab_user_app_information AS U "
                        "WHERE "
                        "("
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
                        " (U.user_app_type = 7) OR "
#endif
#ifdef CONFIG_DSM__ENABLE_DAB_BWS
                        " (U.user_app_type = 3) OR "
#endif
#ifdef CONFIG_DSM__ENABLE_DAB_SLS
                        " (U.user_app_type = 2 AND "
                        "  rdm_current_s() = S.o AND "
                        "  rdm_current_e() = E.o) OR "
#endif
                        " (1==2)) AND "
                        "E.o=S.e AND "
                        "S.o=C.s AND "
                        "E.o=U.e AND "
                        "C.o=U.c AND "
                        "(E.quality_word&0xC0)=0 ",
                        DB::tC_QUERY::b_result_protocol__SDXF_TABLE,
                        0,
                        DB::tC_QUERY::b_parameter_protocol__UNDEF,
                        0,
                        0,AARSLIB_NULL));
        return 0;
    }
    else
    {
        return 1;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABWizard::tclStoredAppList tclDSMDABWizard::getStoredApps()
{
    std::list<tclStoredApp> stored_apps;

    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;
    rc = _dsm->oDB()->dbPrepare(
                "SELECT uri,age_received,host_timestamp FROM dab_data_application WHERE host_timestamp IS NOT NULL ORDER BY host_timestamp ASC",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        /* copy dsmdb dab_data_application table into _stored_apps list
         */
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                tclStoredApp app;
                app.uri = std::string((const char*)tclDB::dbColumnText( pStmt, 0 ));
                app.age_received = tclDB::dbColumnInt( pStmt, 1 );
                app.host_timestamp = tclDB::dbColumnInt64( pStmt, 2 );
                stored_apps.insert(stored_apps.end(),app);
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                /* no new App selected
                 */
                return stored_apps;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }

    return stored_apps;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABWizard::tclReceiveableAppList tclDSMDABWizard::getReceivebleApps()
{
    return (_receiveable_apps);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::vHandleMeCaResponse( tU8 u8TunerIdx, AaRSLib::MSG::tResponseMessage message)
{
    /*handle the query response here*/
    if( message.opcode() == DB::tR_QUERY::OP_CODE )
    {
        DB::tR_QUERY m(message);
        if( (m.b_tag() == _u8WizardDBQueryTag)&&
            (m.b_response() == DB::tR_QUERY::b_response__PARA_OK)&&
            (m.b_result_protocol() == DB::tR_QUERY::b_result_protocol__SDXF_TABLE))
        {
            _dsm->log("tclDSMDABWizard.vHandleMeCaResponse","%p tR_QUERY)",this);

            /* store the current receiveable data service list for later use*/
            if( _tDataServiceList != NULL )
            {
                delete _tDataServiceList;
                _tDataServiceList = NULL;
            }

            _tDataServiceList = new DB::tR_QUERY(message);

            /*move the data service lists to the instance of the wizard*/

            if( _tDataServiceList != NULL )
            {
                tPU8 sdxf_chunk = _tDataServiceList->r_result();
                tU32 sdxf_chunklen = _tDataServiceList->l_result_length();
                int sdxf_rowcnt = AaRSDABLib::SDXFHelper::
                        table_get_num_rows(sdxf_chunk,sdxf_chunklen);
                int r=0;

                /*delete old list*/
                if (_receiveable_apps.size()!=0)
                {
                    _receiveable_apps.clear();
                }

                /* copy SDXF table into _receiveable_apps list ...
                 */

#define STORE_APP    0
//#define REPLACE_APP  1 this is realized by removing the old app from the list and store the new to the list
#define IGNORE_APP   2

                /*for debugging show the _receiveble_apps list*/
                ETG_TRACE_USR1(("Start building of _receiveble_apps list:"));

                while( sdxf_rowcnt > r )
                {
                    tclReceiveableApp app;
                    tU8 store_replace_ignore_app = STORE_APP;

                    app.label_bytes = 0;
                    app.e_label_bytes = 0;
                    app.e_label[0]    = 0;
                    app.s_label_bytes = 0;
                    app.s_label[0]    = 0;
                    app.c_label_bytes = 0;
                    app.c_label[0]    = 0;
                    app.u_label_bytes = 0;
                    app.u_label[0]    = 0;

                    tU8* row_chunk = AaRSDABLib::SDXFHelper::table_get_row(sdxf_chunk,sdxf_chunklen,r++);
                    AaRSDABLib::SDXFHelper::SDXFBlob uri_blob = AaRSDABLib::SDXFHelper::chunk_get_blob(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__DAB_URI));
                    app.uri = std::string((const char*)uri_blob.content,uri_blob.length);
                    AaRSDABLib::SDXFHelper::SDXFBlob e_label_blob = AaRSDABLib::SDXFHelper::chunk_get_blob(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__E_LABEL));
                    if( e_label_blob.length == 20 )
                    {
                        memcpy(app.e_label,e_label_blob.content,20);
                        app.e_label_bytes = (tU8)e_label_blob.length;
                        memcpy(app.label,e_label_blob.content,20);
                        app.label_bytes = (tU8)e_label_blob.length;
                    }
                    AaRSDABLib::SDXFHelper::SDXFBlob s_label_blob = AaRSDABLib::SDXFHelper::chunk_get_blob(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__S_LABEL));
                    if( s_label_blob.length == 20 )
                    {
                        memcpy(app.s_label,s_label_blob.content,20);
                        app.s_label_bytes = (tU8)s_label_blob.length;
                        memcpy(app.label,s_label_blob.content,20);
                        app.label_bytes = (tU8)s_label_blob.length;
                    }
                    AaRSDABLib::SDXFHelper::SDXFBlob c_label_blob = AaRSDABLib::SDXFHelper::chunk_get_blob(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__C_LABEL));
                    if( c_label_blob.length == 20 )
                    {
                        memcpy(app.c_label,c_label_blob.content,20);
                        app.c_label_bytes = (tU8)c_label_blob.length;
                        memcpy(app.label,c_label_blob.content,20);
                        app.label_bytes = (tU8)c_label_blob.length;
                    }
                    AaRSDABLib::SDXFHelper::SDXFBlob u_label_blob = AaRSDABLib::SDXFHelper::chunk_get_blob(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__U_LABEL));
                    if( u_label_blob.length == 20 )
                    {
                        memcpy(app.u_label,u_label_blob.content,20);
                        app.u_label_bytes = (tU8)u_label_blob.length;
                        memcpy(app.label,u_label_blob.content,20);
                        app.label_bytes = (tU8)u_label_blob.length;
                    }

                    app.u16user_app_type = (tU8)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__USER_APP_TYPE));
                    app.u8current_comp   = (tU8)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__CURRENT_COMP));
                    app.u8current_serv   = (tU8)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__CURRENT_SERV));
                    app.u8current_ens    = (tU8)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__CURRENT_ENS));

                    AaRSDABLib::SDXFHelper::SDXFBlob u_app_data_blob = AaRSDABLib::SDXFHelper::chunk_get_blob(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__USER_APP_DATA));
                    if( u_app_data_blob.length <= 20 )
                    {
                        memcpy(app.u_app_data,u_app_data_blob.content,u_app_data_blob.length);
                        app.u_app_data_bytes = (tU8)u_app_data_blob.length;
                    }

                    app.u16eid      = (tU16)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__EID));
                    app.u16tmid     = (tU16)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__TMID));
                    app.u16subchid  = (tU16)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__SUBCHID));
                    app.u16paddr    = (tU16)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__PADDR));
                    app.u8scids     = (tU8)AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__SCIDS));
                    app.u32sid      = AaRSDABLib::SDXFHelper::chunk_get_numeric_tU32(AaRSDABLib::SDXFHelper::row_get_chunk(sdxf_chunk,sdxf_chunklen,row_chunk,1+DB_QUERY_STMT_ID__RECEIVEABLE_TPEG_SERVICE_LIST__SID));

                    /*check if the component is already in the list*/
                    for(std::list<tclReceiveableApp>::iterator receiveable_apps_iter = _receiveable_apps.begin();
                        receiveable_apps_iter != _receiveable_apps.end();
                        receiveable_apps_iter++)
                    {
                        tclReceiveableApp list_item_app = *receiveable_apps_iter;
                        tBool leave_loop = FALSE;

                        if (app.u16tmid == 0)
                        {
                            /* MSC - Stream mode - audio the app is PAD !!!*/
                            /* always store PAD Apps, just leave the loop*/
                            break;
                        }

                        if (list_item_app.u16tmid == app.u16tmid)
                        {
                            switch (app.u16tmid)
                            {
                              case 0:
                                /*MSC - Stream mode - audio the app is PAD !!!*/
                                /*this statement should be unreacheble*/
                                break;

                              case 1: /*MSC - Stream mode - data*/
                                if((list_item_app.u16eid     == app.u16eid)&&
                                   (list_item_app.u16subchid == app.u16subchid))
                                {
                                    /* ignore this app because its component is already in the list*/
                                    /* so far no apps are defined for stream mode data, maybe later*/
                                    store_replace_ignore_app = IGNORE_APP;
                                    leave_loop = TRUE;
                                }
                                break;

                              case 2: /*Reserved*/
                                /*in former times FIDC, but not used any longer*/
                                break;

                              case 3: /*MSC - Packet mode - data*/
                                if((list_item_app.u16eid     == app.u16eid)&&
                                   (list_item_app.u16subchid == app.u16subchid)&&
                                   (list_item_app.u16paddr   == app.u16paddr))
                                {
                                    if (list_item_app.u32sid == app.u32sid)
                                    {
                                        /* we have the same packet mode component several times in the same service*/
                                        /* isn't that stupid? however ... ->*/
                                        if (list_item_app.u8scids > app.u8scids)
                                        {
                                            ETG_TRACE_USR1(("ERASE sid-EQ scids lower!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",list_item_app.u16eid,list_item_app.u32sid,list_item_app.u8scids,list_item_app.u16subchid,list_item_app.u16paddr));
                                            /* if the new app is primary it will have SCIDS = 0*/
                                            /* else lower SCIDS will have priority*/
                                            /* remove old item from the app list*/
                                            /* store_replace_ignore_app == STORE_APP will finish the work*/
                                            _receiveable_apps.erase(receiveable_apps_iter);
                                            leave_loop = TRUE;
                                        }
                                        if (list_item_app.u8scids <= app.u8scids)
                                        {
                                            ETG_TRACE_USR1(("IGNORE sid-EQ scids higher!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",app.u16eid,app.u32sid,app.u8scids,app.u16subchid,app.u16paddr));
                                            /* ignore this app because its component is already in the list*/
                                            store_replace_ignore_app = IGNORE_APP;
                                            leave_loop = TRUE;
                                        }
                                    }
                                    else
                                    {
                                        /* we have another service (e.g. service id)*/
                                        /* and the same packet component in it*/
                                        /* this is complex: */
                                        /* prefer the primary comps with SCIDS == 0*/
                                        /* and as secondary level the lower SID*/
                                        /* and as third level the lower SCIDS*/

                                        if(((list_item_app.u8scids == 0) &&
                                            (app.u8scids == 0))||
                                           ((list_item_app.u8scids != 0) &&
                                            (app.u8scids != 0)))
                                        {
                                            /*first check for data service id*/
                                            if ((app.u32sid>0xffff)&&
                                                (list_item_app.u32sid<0xffff))
                                            {
                                                /*stored item is audio service and new is data service*/
                                                ETG_TRACE_USR1(("ERASE both scidss == 0 or !=0 sid DATA!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",list_item_app.u16eid,list_item_app.u32sid,list_item_app.u8scids,list_item_app.u16subchid,list_item_app.u16paddr));
                                                /* remove old item from the app list*/
                                                /* store_replace_ignore_app == STORE_APP will finish the work*/
                                                _receiveable_apps.erase(receiveable_apps_iter);
                                                leave_loop = TRUE;
                                            }
                                            else if ((list_item_app.u32sid < app.u32sid)||
                                                     ((app.u32sid<0xffff)&&
                                                      (list_item_app.u32sid>0xffff)))
                                            {
                                                /*prefer lower sid or*/
                                                /*stored item is data service and new is audio service*/
                                                ETG_TRACE_USR1(("IGNORE both scidss == 0 or !=0 sid higher or sid AUDIO!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",app.u16eid,app.u32sid,app.u8scids,app.u16subchid,app.u16paddr));
                                                /* ignore this app because its component is already in the list*/
                                                store_replace_ignore_app = IGNORE_APP;
                                                leave_loop = TRUE;
                                            }
                                            else /* list_item_app.u32sid > app.u32sid*/
                                            {
                                                ETG_TRACE_USR1(("ERASE both scidss == 0 or !=0 sid lower!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",list_item_app.u16eid,list_item_app.u32sid,list_item_app.u8scids,list_item_app.u16subchid,list_item_app.u16paddr));
                                                /* if the new app is in SID which is lower*/
                                                /* remove old item from the app list*/
                                                /* store_replace_ignore_app == STORE_APP will finish the work*/
                                                _receiveable_apps.erase(receiveable_apps_iter);
                                                leave_loop = TRUE;
                                            }
                                        }
                                        else if ((list_item_app.u8scids == 0) &&
                                                 (app.u8scids != 0))
                                        {
                                            ETG_TRACE_USR1(("IGNORE new scids!=0!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",app.u16eid,app.u32sid,app.u8scids,app.u16subchid,app.u16paddr));
                                            /* ignore this app because its component is already in the list*/
                                            store_replace_ignore_app = IGNORE_APP;
                                            leave_loop = TRUE;
                                        }
                                        else if ((list_item_app.u8scids != 0) &&
                                                 (app.u8scids == 0))
                                        {
                                            ETG_TRACE_USR1(("ERASE new scids==0!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",list_item_app.u16eid,list_item_app.u32sid,list_item_app.u8scids,list_item_app.u16subchid,list_item_app.u16paddr));
                                            /* if the new app is primary it will have SCIDS = 0*/
                                            /* remove old item from the app list*/
                                            /* store_replace_ignore_app == STORE_APP will finish the work*/
                                            _receiveable_apps.erase(receiveable_apps_iter);
                                            leave_loop = TRUE;
                                        }
                                    }
                                }
                                break;
                              default:
                                break;
                            }
                            if(leave_loop == TRUE)
                            {
                                /* this is true if old entry was erased and new shall be stored or
                                   we decided to ignore the new entry because the already stored one fits*/
                                break;
                            }
                        }
                    }

                    if (store_replace_ignore_app == STORE_APP)
                    {   /*check if the profile id is supported by this implementation*/
                        if (tclDSMApp::u32AppProfileSupported(app.u16user_app_type,app.u_app_data,app.u_app_data_bytes))
                        {
                            ETG_TRACE_USR1(("INSERT!!!: eid=0x%x sid=0x%x scids=%d subch=%d packaddr=%d",app.u16eid, app.u32sid, app.u8scids, app.u16subchid, app.u16paddr));

                            _receiveable_apps.insert(_receiveable_apps.end(),app);
                        }
                    }
                }

                /*for debugging show the _receiveble_apps list*/
                ETG_TRACE_USR1(("List of _receiveble_apps:"));

                for(std::list<tclReceiveableApp>::iterator receiveable_apps_iter = _receiveable_apps.begin();
                    receiveable_apps_iter != _receiveable_apps.end();
                    receiveable_apps_iter++)
                {
                    tclReceiveableApp list_item_app = *receiveable_apps_iter;
                    ETG_TRACE_USR1(("eid=0x%x tmid=%d uatype=%d subch=%d packaddr=%d %s ",list_item_app.u16eid, list_item_app.u16tmid,list_item_app.u16user_app_type,list_item_app.u16subchid,list_item_app.u16paddr,list_item_app.uri.c_str()));
                }

            }
            /*notify to the job subclass*/
            for(std::list<tclDSMDABWizardJob*>::iterator wizard_job_list_iter = _wizard_job_list.begin();
                wizard_job_list_iter != _wizard_job_list.end();
                wizard_job_list_iter++)
            {
                tclDSMDABWizardJob* wizard_job_inst = *wizard_job_list_iter;
                wizard_job_inst->vHandleNewDataServiceList(u8TunerIdx);
            }
        }
    }
    else
    {
        /*pass other responses to the job subclass*/
        for(std::list<tclDSMDABWizardJob*>::iterator wizard_job_list_iter = _wizard_job_list.begin();
            wizard_job_list_iter != _wizard_job_list.end();
            wizard_job_list_iter++)
        {
            tclDSMDABWizardJob* wizard_job_inst = *wizard_job_list_iter;
            wizard_job_inst->vHandleMeCaResponse(u8TunerIdx,message);
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::tclDSMDABWizardJob::vHandleNewDataServiceList(tU8 u8TunerIdx)
{
    tU64          u64CurrentTime = _parent->_dsm->u64ElapsedTimeMS();
    tCurrentStateType tNextState = tCurrentState;

    if( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_SERVICE_LIST )
    {
        if( bSelectNextApp(u8TunerIdx) == TRUE )
        {
            u64StateTimeout_md = u64CurrentTime + WAITING_FOR_APP_SELECTION_TIMEOUT;
            tNextState = DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION;
        }
        else
        {
            u64StateTimeout_md = u64CurrentTime + 1000;
            tNextState = DAB_WIZARD_STATE__INIT;
        }
    }

    if( tCurrentState != tNextState )
    {
        _parent->_dsm->log("tclDSMDABWizard.vHandleNewDataServiceList","%p State (%d=>%d)",this,tCurrentState,tNextState);
    }

    tCurrentState = tNextState;
}


/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::tclDSMDABWizardJob::vHandleMeCaResponse( tU8 u8TunerIdx, AaRSLib::MSG::tResponseMessage message)
{
    (void)u8TunerIdx;

    tU64 u64CurrentTime = _parent->_dsm->u64ElapsedTimeMS();
    tCurrentStateType tNextState = tCurrentState;

    if( message.opcode() == DDM::tR_SELECT_URI::OP_CODE )
    {
        DDM::tR_SELECT_URI m(message);
        if( std::string((const char*)m.r_uri(),m.w_uri_len()) == r_uri )
        {
            w_cmd = m.w_cmd();
            w_response = m.w_response();

            _parent->_dsm->log("tclDSMDABWizard.vHandleMeCaResponse","%p tR_SELECT_URI w_cmd=0x%04x r_response=0x%04x)",this,w_cmd,w_response);

            if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION )||
               ( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_ACTIVATION ))
            {
                if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                    ( w_response == DDM::tR_SELECT_URI::w_response__PARA_OK ))
                {
                    u64StateTimeout_md = u64CurrentTime + WAITING_FOR_APP_ACTIVATION_TIMEOUT;
                    tNextState = DAB_WIZARD_STATE__WAITING_FOR_APP_ACTIVATION;
                }
                else if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                    ( w_response == DDM::tR_SELECT_URI::w_response__URI_ACTIVATED ))
                {
                    u64StateTimeout_md = u64CurrentTime + 120000;
                    tNextState = DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE;
                }
                else if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                          (( w_response == DDM::tR_SELECT_URI::w_response__NO_RESOURCES )||
                          ( w_response == DDM::tR_SELECT_URI::w_response__DATA_ALREADY_SELECTED )||
                          ( w_response == DDM::tR_SELECT_URI::w_response__OTHER_ERROR )))
                {
                    u64StateTimeout_md = u64CurrentTime + 1000;
                    tNextState = DAB_WIZARD_STATE__INIT;
                }
                else if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                          (( w_response == DDM::tR_SELECT_URI::w_response__URI_ACTIVATE_FAIL_NO_DB_INFO )||
                          ( w_response == DDM::tR_SELECT_URI::w_response__URI_ACTIVATE_FAIL_NO_SCIDI )||
                          ( w_response == DDM::tR_SELECT_URI::w_response__URI_ACTIVATE_FAIL_RECEPT )||
                          ( w_response == DDM::tR_SELECT_URI::w_response__URI_ACTIVATE_FAIL_SCHED )||
                          ( w_response == DDM::tR_SELECT_URI::w_response__URI_ACTIVATE_FAIL_TDC_ENABLE)))
                {
                    u64StateTimeout_md = u64CurrentTime + 1000;
                    tNextState = DAB_WIZARD_STATE__DONE;
                }
                else if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                         (( w_response == DDM::tR_SELECT_URI::w_response__COMMAND_NOT_VALID )||
                          ( w_response == DDM::tR_SELECT_URI::w_response__URI_NOT_VALID )))
                {
                    u64StateTimeout_md = u64CurrentTime + 1000;
                    _parent->vAddToBlacklist(r_uri);
                    tNextState = DAB_WIZARD_STATE__INIT;
                }
                else if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                         ( w_response == DDM::tR_SELECT_URI::w_response__URI_DEACTIVATED_BY_SCHED ))
                {
                    /* NOP */
                }
                else if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                    ( w_response == DDM::tR_SELECT_URI::w_response__URI_PROCESSING_IN_PROGRESS ))
                {
                    /* NOP */
                }
                else
                {
                    /* NOP */
                }
            }
            else if( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_DESELECTION)
            {
                if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__STOP)&&
                    ( w_response == DDM::tR_SELECT_URI::w_response__PARA_OK ))
                {
                    u64StateTimeout_md = u64CurrentTime + 1000;
                    tNextState = DAB_WIZARD_STATE__INIT;

                    /*delete the old uri from the members data*/
                    r_uri.clear();
                }
                else
                {
                    /* NOP */
                }
            }
            else if( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE )
            {
                if( (w_cmd == DDM::tR_SELECT_URI::w_cmd__START)&&
                    (( w_response == DDM::tR_SELECT_URI::w_response__URI_DEACTIVATED_BY_SCHED )||
                     ( w_response == DDM::tR_SELECT_URI::w_response__URI_DEACTIVATED_BY_SYNC_LOSS)))
                {                    
                    if ((u32Strategy_md == DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE)||
                        (u32Strategy_md == DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE_COMPONENT))
                    {
                       /*for PAD assume a service change and try again immediately (0.5sec)*/
                       u64StateTimeout_md = u64CurrentTime + 500;
                    }
                    else
                    {
                       /*try again in 2 sec*/
                       u64StateTimeout_md = u64CurrentTime + 2000;
                    }
                    tNextState = DAB_WIZARD_STATE__DONE;
                }
            }
        }
    } else if ( message.opcode() == RDM::tR_GET_RDM_STATUS::OP_CODE )
    {
        RDM::tR_GET_RDM_STATUS m(message);
		
        if (RDM::tR_GET_RDM_STATUS::b_rdm_events_AUDIO_SERVICE_CHANGED&m.b_rdm_events_dab())
        {
            tU64 u64CurrentTime = _parent->_dsm->u64ElapsedTimeMS();

            if (_parent->_u64DataServiceListTimeout < (u64CurrentTime+3000))
            {
                /*set the list timeout to prevent rapid updade*/
                _parent->_u64DataServiceListTimeout = u64CurrentTime+3000;
            }
        }
	
        if (RDM::tR_GET_RDM_STATUS::b_rdm_events_AUDIO_SERVICE_COMPONENT__CHANGED&m.b_rdm_events_dab())
        {            
	        /*in case of SLS stop activity*/
            if (u16user_app_type_current == DAB_SERVICE_MANAGER_UAPPTYPE_SLS)
            {
                /*if(( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION )||
                   ( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_ACTIVATION )||
                   ( tCurrentState == DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE))*/
                {
                    tNextState = DAB_WIZARD_STATE__DONE;
                }
                _parent->_u64DataServiceListTimeout = u64CurrentTime+3000;
            }
            else
            {
                /*set the list timeout to prevent rapid updade*/
                if (_parent->_u64DataServiceListTimeout < (u64CurrentTime+3000))
                {
                    /*set the list timeout to prevent rapid updade*/
                    _parent->_u64DataServiceListTimeout = u64CurrentTime+3000;
                }
            }
        }
    }

    if( tCurrentState != tNextState )
    {
        _parent->_dsm->log("tclDSMDABWizard.vHandleMeCaResponse","%p State (%d=>%d)",this,tCurrentState,tNextState);
    }

    tCurrentState = tNextState;
}


/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
std::string tclDSMDABWizard::tclDSMDABWizardJob::sGetURI()
{
    return r_uri;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABWizard::tclDSMDABWizardJob::vSwitchOff()
{
    tU64 u64CurrentTime = _parent->_dsm->u64ElapsedTimeMS();

    /*deselect URI if selected*/
    if((tCurrentState==DAB_WIZARD_STATE__WAITING_FOR_APP_ACTIVATION)||
       (tCurrentState==DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION)||
       (tCurrentState==DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE))
    {
        _parent->_dsm->oDAB()->vSendDSMMeCaCommand(
                    0,
                    DDM::tC_SELECT_URI(
                        DDM::tC_SELECT_URI::w_cmd__STOP,
                        (DDM::tC_SELECT_URI::t__b_on)0,
                        (DDM::tC_SELECT_URI::t__b_pause)0,
                        (DDM::tC_SELECT_URI::t__b_priority)0,
                        DDM::tC_SELECT_URI::b_decode_level_auto,
                        (DDM::tC_SELECT_URI::t__w_ua_type)0,
                        (DDM::tC_SELECT_URI::t__w_scidi)0,
                        (DDM::tC_SELECT_URI::t__w_uri_len)r_uri.length(),
                        (DDM::tC_SELECT_URI::t__r_uri)r_uri.data(),
                        (DDM::tC_SELECT_URI::t__w_rfu_len)0,
                        (DDM::tC_SELECT_URI::t__r_rfu)NULL));

        _parent->_dsm->log("tclDSMDABWizard.tclDSMDABWizardJob.vSwitchOff","%p STOP %s selected)",this,r_uri.data());
        _parent->_dsm->log("tclDSMDABWizard.tclDSMDABWizardJob.vSwitchOff","%p STOP state:%d->%d)",this,tCurrentState,DAB_WIZARD_STATE__WAITING_FOR_APP_DESELECTION);

        u64StateTimeout_md = u64CurrentTime + 2000;
        tCurrentState = DAB_WIZARD_STATE__WAITING_FOR_APP_DESELECTION;
        return;
    }
    else
    {
        _parent->_dsm->log("tclDSMDABWizard.tclDSMDABWizardJob.vSwitchOff","%p STOP state:%d->%d)",this,tCurrentState,DAB_WIZARD_STATE__INIT);
    }

    u64StateTimeout_md = u64CurrentTime + 1000;
    tCurrentState = DAB_WIZARD_STATE__INIT;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABWizard::tclDSMDABWizardJob::bFilterStrategy(tclReceiveableApp& app)
{
    if((find(u16UserAppType_list_md.begin(), u16UserAppType_list_md.end(), app.u16user_app_type) != u16UserAppType_list_md.end())&&
         (((app.u8current_comp == 1)&&
         ((u32Strategy_md&DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE_COMPONENT)||
          (u32Strategy_md&DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE)))||

        ((app.u8current_serv == 1)&&
         (u32Strategy_md&DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE)&&
         (NULL==strstr(app.uri.c_str(),"xpadapptype")))||

        ((app.u8current_ens == 1)&&
         (u32Strategy_md&DAB_SERVICE_MANAGER_STRATEGY__CURRENT_ENSEMBLE)&&
         (NULL==strstr(app.uri.c_str(),"xpadapptype")))||

        ((u32Strategy_md&DAB_SERVICE_MANAGER_STRATEGY__OTHER_ENSEMBLE)&&
         (NULL==strstr(app.uri.c_str(),"xpadapptype")))))
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABWizard::tclDSMDABWizardJob::bSelectNextApp( tU8 u8TunerIdx )
{
    (void)u8TunerIdx;

    tclStoredAppList       _stored_apps;      /* DSM DB list */
    tclReceiveableAppList  rec_apps; /* Wizard list from ADR*/

    rec_apps = _parent->getReceivebleApps();

    if(rec_apps.size() != 0 )
    {
        _stored_apps = _parent->getStoredApps();

        /* Check Blacklist and remove ...
         */
        if( 1 )
        {
            tU64 youngest_decoded_host_timestamp = 0;
            tU64 oldest_decoded_host_timestamp = 0xffffffffffffffff;
            /*
             * calculate the oldest and newest storeed and receiveable host-timestamp
             */
            for(tclStoredAppList::iterator list_iterB = _stored_apps.begin();
                list_iterB != _stored_apps.end();
                list_iterB++)
            {
                tclDSMDABWizard::tclStoredApp stored_app = *list_iterB;

                if( stored_app.host_timestamp > youngest_decoded_host_timestamp )
                {
                    youngest_decoded_host_timestamp = stored_app.host_timestamp;
                }

                for(std::list<tclReceiveableApp>::iterator list_iterA = rec_apps.begin();
                    list_iterA != rec_apps.end();
                    list_iterA++)
                {
                    tclReceiveableApp receiveable_app = *list_iterA;                    

                    /*filter the decoding strategie*/
                    if(bFilterStrategy(receiveable_app)==TRUE)
                    {
                        if((receiveable_app.uri == stored_app.uri)&&(stored_app.host_timestamp!=0))
                        {

                            if( stored_app.host_timestamp < oldest_decoded_host_timestamp )
                            {
                                oldest_decoded_host_timestamp = stored_app.host_timestamp;
                            }

                        }
                    }
                }
            }

            /* delete newer app/uri blacklist
             */
            if((youngest_decoded_host_timestamp != 0 )&&(oldest_decoded_host_timestamp != 0xffffffffffffffff)&&(oldest_decoded_host_timestamp > 30))
            {
                /* delete apps from blacklist,
                 */
                _parent->vDeleteOlderAppsFromBlackList(oldest_decoded_host_timestamp - 30 );
            }
        }

        /* A.) find new APPs
         */
        if( 1 )
        {
            /* find next app ...
             */

            /* search first "NEW" APP in rec_apps which is not in _stored_apps
             */
            tclReceiveableApp tNewApp;
            tBool bFoundNewApp = FALSE;
            tclReceiveableApp receiveable_app;
            for(std::list<tclReceiveableApp>::iterator list_iterA = rec_apps.begin();
                (list_iterA != rec_apps.end()) && (bFoundNewApp == FALSE);
                list_iterA++)
            {
                receiveable_app = *list_iterA;

                /*check if the app meets the decoding strategy*/
                if (bFilterStrategy(receiveable_app)==TRUE)
                {
                    /*check if another wizard job already does the decoding*/
                    if (_parent->bNotSelected(receiveable_app.uri)==TRUE)
                    {
                        if( _stored_apps.size() == 0 )
                        {
                            bFoundNewApp             = TRUE;
                            tNewApp                  = receiveable_app;
                            r_uri                    = receiveable_app.uri;
                            u16user_app_type_current = receiveable_app.u16user_app_type;
                        }
                        else
                        {
                            tclStoredApp stored_app;
                            tBool bFlag = FALSE;
                            for(std::list<tclStoredApp>::iterator list_iterB = _stored_apps.begin();
                                (list_iterB != _stored_apps.end()) && (bFoundNewApp == FALSE);
                                list_iterB++)
                            {
                                stored_app = *list_iterB;
                                if( receiveable_app.uri == stored_app.uri )
                                {
                                    bFlag = TRUE;
                                }
                            }
                            if((bFlag == FALSE)&&(_parent->bIsBlackListApp(receiveable_app.uri)==FALSE))
                            {
                                bFoundNewApp             = TRUE;
                                tNewApp                  = receiveable_app;
                                r_uri                    = receiveable_app.uri;
                                u16user_app_type_current = receiveable_app.u16user_app_type;
                            }
                        }
                    }
                }
            }

            if( bFoundNewApp == TRUE )
            {
                /* insert some meta information (e.g. labels) into the database
                 */
                SQLITE_API int rc;
                sqlite3_stmt* pStmt = NULL;
                rc = _parent->_dsm->oDB()->dbPrepare(
                            "REPLACE INTO dab_data_application (uri,e_label,s_label,c_label,u_label,label,w_user_app_type) VALUES(?,?,?,?,?,?,?)",
                            -1,
                            &pStmt,
                            NULL);
                if( pStmt != NULL )
                {
                    tclDB::dbBindText(pStmt,
                                      1,
                                      tNewApp.uri.data(),
                                      (int)tNewApp.uri.length(),
                                      NULL);

                    if( tNewApp.e_label_bytes == 0 )
                    {
                        tclDB::dbBindNull(pStmt,
                                          2);
                    }
                    else
                    {
                        tclDB::dbBindBlob(pStmt,
                                          2,
                                          &tNewApp.e_label,
                                          tNewApp.e_label_bytes,
                                          NULL);
                    }

                    if( tNewApp.s_label_bytes == 0 )
                    {
                        tclDB::dbBindNull(pStmt,
                                          3);
                    }
                    else
                    {
                        tclDB::dbBindBlob(pStmt,
                                          3,
                                          &tNewApp.s_label,
                                          tNewApp.s_label_bytes,
                                          NULL);
                    }

                    if( tNewApp.c_label_bytes == 0 )
                    {
                        tclDB::dbBindNull(pStmt,
                                          4);
                    }
                    else
                    {
                        tclDB::dbBindBlob(pStmt,
                                          4,
                                          &tNewApp.c_label,
                                          tNewApp.c_label_bytes,
                                          NULL);
                    }

                    if( tNewApp.u_label_bytes == 0 )
                    {
                        tclDB::dbBindNull(pStmt,
                                          5);
                    }
                    else
                    {
                        tclDB::dbBindBlob(pStmt,
                                          5,
                                          &tNewApp.u_label,
                                          tNewApp.u_label_bytes,
                                          NULL);
                    }

                    if( tNewApp.label_bytes == 0 )
                    {
                        tclDB::dbBindNull(pStmt,
                                          6);
                    }
                    else
                    {
                        tclDB::dbBindBlob(pStmt,
                                          6,
                                          &tNewApp.label,
                                          tNewApp.label_bytes,
                                          NULL);
                    }

                    tclDB::dbBindInt(pStmt,
                                      7,
                                      tNewApp.u16user_app_type);


                    while(1)
                    {
                        rc = _parent->_dsm->oDB()->dbStep(pStmt);
                        if( rc == SQLITE_DONE )
                        {
                            break;
                        }
                        else if( tclDB::dbOnStepError(rc) )
                        {
                            /* no new App selected
                             */
                            break;
                        }
                    }
                    _parent->_dsm->oDB()->dbReset(pStmt);
                }
            }

            if( bFoundNewApp == TRUE )
            {
                /* enable URI ...
                 */
                tU8 t_pause = 0;
                tU8 t_on = 5;
                tU8 prio = 0;
                _parent->_dsm->oDAB()->vSendDSMMeCaCommand(
                            0,
                            DDM::tC_SELECT_URI(
                                DDM::tC_SELECT_URI::w_cmd__START,
                                (DDM::tC_SELECT_URI::t__b_on)t_on,
                                (DDM::tC_SELECT_URI::t__b_pause)t_pause,
                                (DDM::tC_SELECT_URI::t__b_priority)prio,
                                DDM::tC_SELECT_URI::b_decode_level_auto,
                                (DDM::tC_SELECT_URI::t__w_ua_type)0,
                                (DDM::tC_SELECT_URI::t__w_scidi)0,
                                (DDM::tC_SELECT_URI::t__w_uri_len)r_uri.length(),
                                (DDM::tC_SELECT_URI::t__r_uri)r_uri.data(),
                                (DDM::tC_SELECT_URI::t__w_rfu_len)0,
                                (DDM::tC_SELECT_URI::t__r_rfu)NULL));

                _parent->_dsm->log("tclDSMDABWizard.bSelectNextApp","%p NEW %s selected)",this,r_uri.data());

                return TRUE;
            }

            /* B.) update existing APPs
             */
            if( 1 )
            {
                /* find next app ...
                 */

                /* search the oldest receiveable APP
                 */
                tBool bFoundUpdateApp = FALSE;
                for(std::list<tclStoredApp>::iterator list_iterB = _stored_apps.begin();
                    (list_iterB != _stored_apps.end()) && (bFoundUpdateApp == FALSE);
                    list_iterB++)
                {
                    tclStoredApp stored_app = *list_iterB;
                    for(std::list<tclReceiveableApp>::iterator list_iterA = rec_apps.begin();
                        (list_iterA != rec_apps.end()) && (bFoundUpdateApp == FALSE);
                        list_iterA++)
                    {
                        tclReceiveableApp receiveable_app = *list_iterA;

                        if(bFilterStrategy(receiveable_app)==TRUE)
                        {
                            /*check if another wizard job already does the decoding*/
                            if (_parent->bNotSelected(receiveable_app.uri)==TRUE)
                            {
                                if(( receiveable_app.uri == stored_app.uri )&&(_parent->bIsBlackListApp(receiveable_app.uri)==FALSE))
                                {
                                    bFoundUpdateApp          = TRUE;
                                    r_uri                    = receiveable_app.uri;
                                    u16user_app_type_current = receiveable_app.u16user_app_type;
                                }
                            }
                        }
                    }
                }
                if( bFoundUpdateApp == TRUE )
                {
                    /* enable URI ...
                     */
                    tU8 t_pause = 0;
                    tU8 t_on = 5;
                    tU8 prio = 0;
                    _parent->_dsm->oDAB()->vSendDSMMeCaCommand(
                                0,
                                DDM::tC_SELECT_URI(
                                    DDM::tC_SELECT_URI::w_cmd__START,
                                    (DDM::tC_SELECT_URI::t__b_on)t_on,
                                    (DDM::tC_SELECT_URI::t__b_pause)t_pause,
                                    (DDM::tC_SELECT_URI::t__b_priority)prio,
                                    DDM::tC_SELECT_URI::b_decode_level_auto,
                                    (DDM::tC_SELECT_URI::t__w_ua_type)0,
                                    (DDM::tC_SELECT_URI::t__w_scidi)0,
                                    (DDM::tC_SELECT_URI::t__w_uri_len)r_uri.length(),
                                    (DDM::tC_SELECT_URI::t__r_uri)r_uri.data(),
                                    (DDM::tC_SELECT_URI::t__w_rfu_len)0,
                                    (DDM::tC_SELECT_URI::t__r_rfu)NULL));

                    _parent->_dsm->log("tclDSMDABWizard.bSelectNextApp","%p UPDATE %s selected)",this,r_uri.data());

                    return TRUE;
                }

                _parent->_dsm->log("tclDSMDABWizard.bSelectNextApp","blacklist cleared",this);
                _parent->_blacklist_apps.clear();
            }
        }
    }
    /* no new App selected
     */
    return FALSE;
}

#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */

/******************************************************************************
 * EOF
 *****************************************************************************/
 
