/******************************************************************************
 *
 * FILE: tclDSMApp.cpp
 * PROJECT: AaRS
 * SW-COMPONENT: Data Service Manager
 *
 * DESCRIPTION: This file contains the DSM class interface specification
 *
 * AUTHOR: CM-AI/ECB2-Scholz
 *
 * COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
 * HISTORY:
 * Date | Rev. | Author | Modification
 * ----------------------------------------------------------------------------
 * 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
 *
 *****************************************************************************/
#include <stdarg.h>  // for va_start, etc
#include <memory>    // for std::unique_ptr
#include "tclDSM.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_DSM_DAB_APP
#include "trcGenProj/Header/tclDSMDABApp.cpp.trc.h"
#endif

using namespace dsm;

#ifdef CONFIG_DSM__ENABLE_DAB
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABApp::tclDSMDABApp( tclDSM* dsm, tU8 u8TunerIndex, std::string uri )
    : tclDSMApp(dsm)
{
    _a = DSM_INVALID_APPID;
    _u8TunerIndex = u8TunerIndex;
    _uri = uri;
    _u16SCIDI = 0xffff;
    _u16XPADAppType = 0;
    _ctime = 0;
    _ctime_utc = 0;
    _ctime_60s = 0;
	_ctime_old_host_time = 0;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABApp::~tclDSMDABApp()
{
    if( s64GetA() != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "UPDATE dab_data_application SET w_scidi=NULL,w_xpad_app_type=NULL WHERE o=?",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64GetA());
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
            pStmt = NULL;
        }
    }

    if( bGetErrorFlag() == TRUE )
    {
        if( s64GetA() != 0 )
        {
            if( tGetURI().length() != 0 )
            {
                _dsm->vBroadCastEvent( DSM_EV_APPLICATION_ERROR,
                                       string_sprintf("%s",tGetURI().data()) );
            }
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABApp::vInitializeTables(tclDSM* dsm)
{
    SQLITE_API int rc;
    /*
   * create table dab_data_service
   */
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_data_application ("
                /***/ "o INTEGER PRIMARY KEY,"
                /***/ "min_received,"  /* time in minutes this data application was received. */
                /***/ "age_received,"  /* time in minutes since this data application was received. */
                /***/ "age_used,"      /* time in minutes since this data application was used. */
                /***/ "dab_timestamp DATETIME,"
                /***/ "dab_utc_timestamp DATETIME,"
                /***/ "host_timestamp DATETIME,"
                /***/ "refresh_timestamp DATETIME," /* if not NULL, refresh/redecode application timestamp */
                /***/ "label,"
                /***/ "uri NOT NULL,"
                /***/ "w_user_app_type NOT NULL,"
                /***/ "r_user_app_data,"
                /***/ "w_scidi,"
                /***/ "w_xpad_app_type,"
                /***/ "e_label,"
                /***/ "s_label,"
                /***/ "c_label,"
                /***/ "u_label,"
                /***/ "complete,"  /* complete flag */
                /***/ "nxtmotdir," /* next still unprocessed MOT directory transpid */
                /***/ "nxtmotdirobjectsleft," /* number of missing objects for next mot directory */
#ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
                /***/ "size,"
#endif /* CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN */
                /***/ "UNIQUE(uri))",
                NULL,NULL,NULL);
    (void)rc;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tS64 tclDSMDABApp::s64GetA()
{
    if( _a == DSM_INVALID_APPID )
    {
        if( _u16SCIDI != 0xffff )
        {
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
                                     "SELECT o FROM dab_data_application WHERE w_scidi=? AND (w_xpad_app_type=? OR w_xpad_app_type IS NULL) LIMIT 1",
                                     -1,
                                     &pStmt,
                                     NULL);
            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                 1,
                                 _u16SCIDI);
                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                 1,
                                 _u16XPADAppType);
                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( rc == SQLITE_ROW )
                    {
                        if( tclDB::dbColumnCount(pStmt) == 1 )
                        {
                            _a = tclDB::dbColumnInt64( pStmt, 0 );
                        }
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        _dsm->oDB()->dbReset(pStmt);
                        return DSM_INVALID_APPID;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
            }
        }
    }
    if( _a == DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                                 "SELECT o FROM dab_data_application WHERE uri=? LIMIT 1",
                                 -1,
                                 &pStmt,
                                 NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindText(pStmt,
                              1,
                              _uri.c_str(),
                              (int)_uri.length(),
                              NULL);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        _a = tclDB::dbColumnInt64( pStmt, 0 );
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return DSM_INVALID_APPID;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return _a;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tS64 tclDSMDABApp::s64GetApplicationTime()
{
    return (_ctime + ((_dsm->u64ElapsedTimeMS()/1000) - _ctime_old_host_time));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABApp::vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg)
{
    (void)msg;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABApp::vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg)
{
    tS64 host_timestamp = _dsm->u64ElapsedTimeMS()/1000;
    tBool bCallGC = FALSE;
    tBool b60sTick = FALSE;
    tBool bDABTimeChanged = FALSE;
    tU64 unixepoch_time_local = 0;
    tU64 unixepoch_time_utc = 0; /* UTC */
    tU32 utc = msg.d_utc_comp_time();
    tU32 mjd = msg.d_mjd_comp_time();
    tU32 utc_comp_time_uncorrected = 0;
    tU32 mjd_comp_time_uncorrected = 0;
    if( msg.size() >= 22 )
    {
        utc_comp_time_uncorrected = msg.d_utc_comp_time_uncorrected();
        mjd_comp_time_uncorrected = msg.d_mjd_comp_time_uncorrected();
    }

    if( mjd >= MJD_1_1_1970 )
    {
        unixepoch_time_local = (tU64)(mjd - MJD_1_1_1970) * 24 * 60 * 60;
        if( utc != 0 )
        {
            unixepoch_time_local += ((0x1f&(utc >> (6+6+10))) * (60*60));	/* add hours */
            unixepoch_time_local += ((0x3f&(utc >> (6+10))) * (60));  	/* add minutes */
            //unixepoch_time_local += ((0x3f&(utc >> (10))) * (1));  		/* add seconds */
            // we shall not use seconds comming from the DAB system, because this results into negative timestamp deltas
        }
    }

    if( mjd_comp_time_uncorrected >= MJD_1_1_1970 )
    {
        unixepoch_time_utc = (tU64)(mjd_comp_time_uncorrected - MJD_1_1_1970) * 24 * 60 * 60;
        if( utc_comp_time_uncorrected != 0 )
        {
            unixepoch_time_utc += ((0x1f&(utc_comp_time_uncorrected >> (6+6+10))) * (60*60));	/* add hours */
            unixepoch_time_utc += ((0x3f&(utc_comp_time_uncorrected >> (6+10))) * (60));  	/* add minutes */
            //unixepoch_time_utc += ((0x3f&(utc_comp_time_uncorrected >> (10))) * (1));  		/* add seconds */
            // we shall not use seconds comming from the DAB system, because this results into negative timestamp deltas
        }
    }

    if( unixepoch_time_utc == 0 )
    {
        unixepoch_time_utc = unixepoch_time_local;
    }

    if( _ctime == 0 )
    {
        /* negative time skip happen, e.g. from ETI file rewind
         */
        bCallGC = TRUE;
    }
    
    if( _ctime != unixepoch_time_local )
    {
    _ctime = unixepoch_time_local;
    bDABTimeChanged = TRUE;
    }
    if( _ctime_utc != unixepoch_time_utc )
    {
    _ctime_utc = unixepoch_time_utc;
    bDABTimeChanged = TRUE;
    }

    _ctime_old_host_time = _dsm->u64ElapsedTimeMS()/1000;

    ETG_TRACE_USR1(("%p.vHandleMsgR_TIMESTAMP()  w_scidi=0x%04x  mjd=0x%08x/0x%08x utc=0x%08x/0x%08x/(%d,%d,%d) ",
                    (void*)this,
                    msg.w_scidi(),
                    msg.d_mjd_comp_time(),
                    msg.d_mjd_comp_time_uncorrected(),
                    msg.d_utc_comp_time(),
                    msg.d_utc_comp_time_uncorrected(),
                    (0x1f&(utc >> (6+6+10))),
                    (0x3f&(utc >> (6+10))),
                    (0x3f&(utc >> (10)))));

		ETG_TRACE_USR1((" a=%d  t=%d _ctime=%d _ctime_utc=%d _ctime_old_host_time=%d bCallGC=%d",
			(int)s64GetA(),
			msg.d_internal_timestamp(), (int)_ctime, (int)_ctime_utc, (int)_ctime_old_host_time, bCallGC));

    if( host_timestamp != 0 )
    {
        if(_ctime_60s == 0 )
        {
            b60sTick = TRUE;
            bCallGC = TRUE;
            _ctime_60s = host_timestamp + 60;
        }
        else if(_ctime_60s < (tU64)host_timestamp )
        {
            b60sTick = TRUE;
            bCallGC = TRUE;
            _ctime_60s = host_timestamp + 60;
        }
    }

    if((s64GetA() != DSM_INVALID_APPID)&&(bDABTimeChanged == TRUE))
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                                 "UPDATE dab_data_application SET min_received=(CASE WHEN min_received IS NULL THEN 1 ELSE min_received+1 END),age_received=0,dab_timestamp=?,dab_utc_timestamp=?,host_timestamp=? WHERE o=?",
                                 -1,
                                 &pStmt,
                                 NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                               1,
                               _ctime);

            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                               2,
                               _ctime_utc);

            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                               3,
                               host_timestamp);

            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                               4,
                               s64GetA());
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }

    if( bCallGC == TRUE )
    {
        ETG_TRACE_USR1(("%p.vHandleMsgR_TIMESTAMP() a=%d _ctime=%d => vHandleGarbageCollection",(void*)this,(int)s64GetA(),(int)_ctime));
        vHandleGarbageCollection();
    }

    if( b60sTick == TRUE )
    {
        ETG_TRACE_USR1(("%p.vHandleMsgR_TIMESTAMP() a=%d _ctime=%d => vHandle60sCTimeTick",(void*)this,(int)s64GetA(),(int)_ctime));
        vHandle60sCTimeTick();
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABApp::vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg)
{
    if( ( msg.w_response() == AaRSDABLib::API::DDM::tR_SELECT_URI::w_response__URI_DEACTIVATED_BY_SYNC_LOSS )||
            ( msg.w_response() == AaRSDABLib::API::DDM::tR_SELECT_URI::w_response__URI_DEACTIVATED_BY_SCHED ) )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;

        ETG_TRACE_USR1(("%p.vHandleMsgR_SELECT_URI(w_response__URI_DEACTIVATED_BY_SYNC_LOSS|w_response__URI_DEACTIVATED_BY_SCHED) a=%d",
                        (void*)this,(int)s64GetA()));

        rc = _dsm->oDB()->dbPrepare(
                                 "UPDATE dab_data_application SET w_scidi=NULL,w_xpad_app_type=NULL,age_received=0 WHERE uri=?",
                                 -1,
                                 &pStmt,
                                 NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindText(pStmt,
                              1,
                              msg.r_uri(),
                              msg.w_uri_len(),
                              NULL);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
            pStmt = NULL;
        }

        _a = DSM_INVALID_APPID;
        _u16SCIDI = 0xffff;
        _u16XPADAppType = 0;

        _dsm->log("tclDABServiceManager","%s deactivated",msg.r_uri());
    }
    else if( ( msg.w_response() == AaRSDABLib::API::DDM::tR_SELECT_URI::w_response__URI_ACTIVATED ) )
    {
        char* xpad_app_type_str = NULL;
        int   xpad_app_type     = 0;

        ETG_TRACE_USR1(("%p.vHandleMsgR_SELECT_URI(w_response__URI_ACTIVATED) a=%d",
                        (void*)this,(int)s64GetA()));

        if( s64GetA() != DSM_INVALID_APPID )
        {
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;

            /* if the  application allready exists in the table e.g uri present UPDATE will work
           * if update fails we create a new line with REPLACE INTO*/

            rc = _dsm->oDB()->dbPrepare(
                                     "UPDATE dab_data_application SET w_scidi=?,w_user_app_type=?,r_user_app_data=?,w_xpad_app_type=?,age_received=0,host_timestamp=? WHERE o=?",
                                     -1,
                                     &pStmt,
                                     NULL);
            if( pStmt != NULL )
            {
                //int   sqlite_changes    = 0;
                tU16  w_scidi = msg.w_scidi();
                tU16  w_user_app_type = msg.w_ua_type();

                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                 1,
                                 w_scidi);

                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                 2,
                                 (w_user_app_type&0x0fff));

                _dsm->oDB()->tclDB::dbBindBlob(pStmt,
                                 3,
                                 msg.r_rfu(),
                                 msg.w_rfu_len(),
                                 NULL);

                xpad_app_type_str   = strstr(msg.r_uri(),"xpadapptype=");

                if (xpad_app_type_str!=NULL)
                {
                    sscanf(xpad_app_type_str+12,"%x8",&xpad_app_type);

                    _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                     4,
                                     xpad_app_type);
                }
                else
                {
                    _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                      4);
                }

                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                  5,
                                  _dsm->u64ElapsedTimeMS()/1000);

                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                  6,
                                  s64GetA());

                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        //sqlite_changes=_dsm->oDB()->dbChanges();
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        break;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
                pStmt = NULL;
            }
        }
        else
        {
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
#ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
                                    "REPLACE INTO dab_data_application (uri,w_scidi,w_user_app_type,w_xpad_app_type,min_received,age_received,age_used,size,complete) VALUES(?,?,?,?,0,0,99,0,0)",
#else
                                    "REPLACE INTO dab_data_application (uri,w_scidi,w_user_app_type,w_xpad_app_type,min_received,age_received,age_used,complete) VALUES(?,?,?,?,0,0,99,0)",
#endif
                                     -1,
                                     &pStmt,
                                     NULL);
            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindText(pStmt,
                                  1,
                                  msg.r_uri(),
                                  msg.w_uri_len(),
                                  NULL);

                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                 2,
                                 msg.w_scidi());

                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                 3,
                                 (msg.w_ua_type()&0x0fff));

                xpad_app_type_str   = strstr(msg.r_uri(),"xpadapptype=");

                if (xpad_app_type_str!=NULL)
                {
                    sscanf(xpad_app_type_str+12,"%x8",&xpad_app_type);

                    _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                     4,
                                     xpad_app_type);
                }
                else
                {
                    _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                      4);
                }

                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        _a = sqlite3_last_insert_rowid(_dsm->oDB()->pHandle());
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        break;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
                pStmt = NULL;
            }
        }

        if( s64GetA() != DSM_INVALID_APPID )
        {
            _u16SCIDI       = msg.w_scidi();
            _u16XPADAppType = (tU16)xpad_app_type;
            _dsm->log("tclDABServiceManager","%s activated",msg.r_uri());
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABApp::bIsAppComplete( tS64 s64AppId )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                                 "SELECT complete FROM v_dab_mot_application WHERE o=? LIMIT 1",
                                 -1,
                                 &pStmt,
                                 NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        tS32 complete = tclDB::dbColumnInt( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        return (complete==0)?FALSE:TRUE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABApp::vHandle60sCTimeTick()
{

}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABApp::vDeleteAppData( tU64 a )
{
    DSM_PERF_ENTRY();
    vDeleteAppData(_dsm,a);
    DSM_PERF_EXIT("tclDSMDABApp::vDeleteAppData");
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABApp::vDeleteAppData( tclDSM* dsm, tU64 a )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    ETG_TRACE_USR1(("vDeleteAppData(a=%d)",
                    (int)a));

    rc = dsm->oDB()->dbPrepare(
                             "DELETE FROM dab_data_application WHERE o=?",
                             -1,
                             &pStmt,
                             NULL);
    if( pStmt != NULL )
    {
        dsm->oDB()->tclDB::dbBindInt64(pStmt, 1, a);
        while(1)
        {
            rc = dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        dsm->oDB()->dbReset(pStmt);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABApp::vHandleGarbageCollection()
{
    DSM_PERF_ENTRY();

    int max_num_loops = 10;
    while(max_num_loops--)
    {
        tBool bFileError = (_dsm->oDB()->u32FileErrorCounter()>0)?TRUE:FALSE;
        tBool bIsFullError = FALSE;
        _dsm->oDB()->vWALCheckPoint();

        if( bFileError == FALSE )
        {
            bIsFullError = _dsm->oDB()->bIsFull();
        }

        if( bFileError || bIsFullError )
        {
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;

            rc = _dsm->oDB()->dbPrepare(
                        "SELECT o,uri FROM dab_data_application WHERE o!=? ORDER BY age_used DESC,age_received DESC,host_timestamp ASC LIMIT 1",
                        -1,
                        &pStmt,
                        NULL);
            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt, 1, s64GetA());
                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        _dsm->oDB()->dbReset(pStmt);
                        DSM_PERF_EXIT("tclDSMDABApp::vHandleGarbageCollection");
                        return FALSE;
                    }
                    else if( rc == SQLITE_ROW )
                    {
                        if( tclDB::dbColumnCount(pStmt) == 2 )
                        {
                            tS64 a = tclDB::dbColumnInt64( pStmt, 0 );
                            if( a == s64GetA() )
                            {
                                _dsm->oDB()->dbReset(pStmt);
                                DSM_PERF_EXIT("tclDSMDABApp::vHandleGarbageCollection");
                                return FALSE;
                            }
                            else
                            {
                                const char* uri = (const char*)tclDB::dbColumnText( pStmt, 1 );
                                _dsm->log("vHandleGarbageCollection","App bFileError=%d bIsFullError=%d a=%lld uri=%s deleted",bFileError,bIsFullError,a,uri);

                                ETG_TRACE_USR1(("%p.vHandleGarbageCollection() App bFileError=%d bIsFullError=%d a=%d uri=%s deleted",
                                                (void*)this,
                                               bFileError,bIsFullError,(int)a,uri));

                                _dsm->oDAB()->vDestructApp(a);
                                vDeleteAppData(a);
                            }
                            break;
                        }
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        _dsm->oDB()->dbReset(pStmt);
                        DSM_PERF_EXIT("tclDSMDABApp::vHandleGarbageCollection");
                        return FALSE;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
            }
        }
        else
        {
            DSM_PERF_EXIT("tclDSMDABApp::vHandleGarbageCollection");
            return TRUE;
        }
    }
    DSM_PERF_EXIT("tclDSMDABApp::vHandleGarbageCollection");
    return FALSE;
}

#endif /* CONFIG_DSM__ENABLE_DAB */

/******************************************************************************
 * EOF
 *****************************************************************************/

