/******************************************************************************
 *
 * FILE: tclDSMApp.cpp
 * PROJECT: AaRS
 * SW-COMPONENT: Data Service Manager
 *
 * DESCRIPTION: This file contains the DSM class interface specification
 *
 * AUTHOR: CM-AI/ECB2-Scholz
 *
 * COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
 * HISTORY:
 * Date | Rev. | Author | Modification
 * ----------------------------------------------------------------------------
 * 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
 *
 *****************************************************************************/
#include <stdarg.h>  // for va_start, etc
#include <memory>    // for std::unique_ptr
#include <vector>
#include "tclDSM.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_DSM_DAB_EPG
#include "trcGenProj/Header/tclDSMAppDABEPG.cpp.trc.h"
#endif

#if !defined(DSM_PERF_ENTRY) || !defined(DSM_PERF_EXIT)
/* create dummy PERF macros, when not defined
 */
#define DSM_PERF_ENTRY()
#define DSM_PERF_EXIT(_name_)
#endif

#undef ENABLE_PRINT_EPG
using namespace dsm;

#ifdef CONFIG_DSM__EPG_OBJECT_SIZE_LIMIT
tU32 tclDSMDABEPGApp::_u32_object_size_limit = CONFIG_DSM__EPG_OBJECT_SIZE_LIMIT;
#endif
#ifdef CONFIG_DSM__EPG_APP_SIZE_LIMIT
tU32 tclDSMDABEPGApp::_u32_app_size_limit = CONFIG_DSM__EPG_APP_SIZE_LIMIT;
#endif

#ifdef CONFIG_DSM__ENABLE_DAB_EPG
#define tclDSMDABEPGApp_Tag_UNDEF_0x00                   0x00
#define tclDSMDABEPGApp_Tag_epg_0x02                     0x02
#define tclDSMDABEPGApp_Tag_serviceInformation_0x03      0x03
#define tclDSMDABEPGApp_Tag_tokenTableElement_0x04       0x04
#define tclDSMDABEPGApp_Tag_defaultcontentIDElement_0x05 0x05   /* not used in latest EPG spec */
#define tclDSMDABEPGApp_Tag_languageElement_0x06         0x06
#define tclDSMDABEPGApp_Tag_shortName_0x10               0x10
#define tclDSMDABEPGApp_Tag_mediumName_0x11              0x11
#define tclDSMDABEPGApp_Tag_longName_0x12                0x12
#define tclDSMDABEPGApp_Tag_mediaDescription_0x13        0x13
#define tclDSMDABEPGApp_Tag_genre_0x14                   0x14
//#define tclDSMDABEPGApp_Tag_CA_0x15                      0x15
#define tclDSMDABEPGApp_Tag_keywords_0x16                0x16
#define tclDSMDABEPGApp_Tag_memberOf_0x17                0x17
#define tclDSMDABEPGApp_Tag_link_0x18                    0x18
#define tclDSMDABEPGApp_Tag_location_0x19                0x19
#define tclDSMDABEPGApp_Tag_shortDescription_0x1A        0x1A
#define tclDSMDABEPGApp_Tag_longDescription_0x1B         0x1B
#define tclDSMDABEPGApp_Tag_programme_0x1C               0x1C
#define tclDSMDABEPGApp_Tag_programmeGroups_0x20         0x20
#define tclDSMDABEPGApp_Tag_schedule_0x21                0x21
#define tclDSMDABEPGApp_Tag_alternateSource_0x22         0x22
#define tclDSMDABEPGApp_Tag_programmeGroup_0x23          0x23
#define tclDSMDABEPGApp_Tag_scope_0x24                   0x24
#define tclDSMDABEPGApp_Tag_serviceScope_0x25            0x25
#define tclDSMDABEPGApp_Tag_ensemble_0x26                0x26
//#define tclDSMDABEPGApp_Tag_frequency_0x27               0x27
#define tclDSMDABEPGApp_Tag_service_0x28                 0x28
#define tclDSMDABEPGApp_Tag_serviceID_0x29               0x29
//#define tclDSMDABEPGApp_Tag_epg_Language_0x2A            0x2A
#define tclDSMDABEPGApp_Tag_multimedia_0x2B              0x2B
#define tclDSMDABEPGApp_Tag_time_0x2C                    0x2C
#define tclDSMDABEPGApp_Tag_bearer_0x2D                  0x2D
#define tclDSMDABEPGApp_Tag_programmeEvent_0x2E          0x2E
#define tclDSMDABEPGApp_Tag_relativeTime_0x2F            0x2F
#define tclDSMDABEPGApp_Tag_simulcast_0x30               0x30
#define tclDSMDABEPGApp_Tag_ATTR01                       0x01
#define tclDSMDABEPGApp_Tag_ATTR80                       0x80
#define tclDSMDABEPGApp_Tag_ATTR81                       0x81
#define tclDSMDABEPGApp_Tag_ATTR82                       0x82
#define tclDSMDABEPGApp_Tag_ATTR83                       0x83
#define tclDSMDABEPGApp_Tag_ATTR84                       0x84
#define tclDSMDABEPGApp_Tag_ATTR85                       0x85

#define tclDSMDABEPGApp_16Bit_Tag(_parenttag_,_tag_)                ((((_parenttag_)&0xff)<<8) | ((_tag_)&0xff))
#define tclDSMDABEPGApp_Attribute_programmeGroups_creationTime      tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_programmeGroups_0x20,0x81)
#define tclDSMDABEPGApp_Attribute_schedule_creationTime             tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_schedule_0x21,0x81)
#define tclDSMDABEPGApp_Attribute_scope_startTime                   tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_scope_0x24,0x80)
#define tclDSMDABEPGApp_Attribute_scope_stopTime                    tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_scope_0x24,0x81)
#define tclDSMDABEPGApp_Attribute_serviceInformation_creationTime   tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_serviceInformation_0x03,0x81)
#define tclDSMDABEPGApp_Attribute_time_time                         tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_time_0x2C,0x80)
#define tclDSMDABEPGApp_Attribute_time_duration                     tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_time_0x2C,0x81)
#define tclDSMDABEPGApp_Attribute_time_actualTime                   tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_time_0x2C,0x82)
#define tclDSMDABEPGApp_Attribute_link_expiryTime                   tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_link_0x18,0x84)

#define tclDSMDABEPGApp_Attribute_multimedia_mimeValue              tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_multimedia_0x2B,0x80)
#define tclDSMDABEPGApp_Attribute_multimedia_url                    tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_multimedia_0x2B,0x82)

typedef enum {
    tclDSMDABEPGApp_TAGType_Integer,
    tclDSMDABEPGApp_TAGType_timePointType,
    tclDSMDABEPGApp_TAGType_URL,
    tclDSMDABEPGApp_TAGType_MIMEtype,
    tclDSMDABEPGApp_TAGType_CRID,
    tclDSMDABEPGApp_TAGType_ShortCRID,
    tclDSMDABEPGApp_TAGType_UNDEF
} tclDSMDABEPGApp_TAGType;

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABEPGApp::tclDSMDABEPGApp(tclDSM* dsm, tU8 u8TunerIndex,AaRSDABLib::API::DDM::tR_SELECT_URI msg)
    : tclDSMDABMOTApp(dsm,u8TunerIndex,msg.r_uri())
{
    ETG_TRACE_USR1(("%p.tclDSMDABEPGApp(%p,%d,%s)",(void*)this,dsm,u8TunerIndex,msg.r_uri()));
    vInitMOTDirectory();
    tclDSMDABMOTApp::vHandleMsgR_SELECT_URI(msg);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABEPGApp::~tclDSMDABEPGApp()
{
    ETG_TRACE_USR1(("%p.~tclDSMDABEPGApp()",this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
static void tclDSMDABEPGApp_timePoint2UnixTime(
        sqlite3_context *context,
        int argc,
        sqlite3_value **argv
        )
{
    tS64 unixepoch_time = 0;
    if( argc == 1 )
    {
        if( tclDB::dbValueType(argv[0]) == SQLITE_BLOB )
        {
            int timePointbytes = tclDB::dbValueBytes(argv[0]);
            if( timePointbytes >= 4 )
            {
                tU8* timePoint = (tU8*)tclDB::dbValueBlob(argv[0]);
                if( timePoint != NULL )
                {
#define MJD_1_1_1970 40587
                    tU32 mjd = tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1,17);
                    tU32 utc = 0;
                    if( mjd >= MJD_1_1_1970 )
                    {
                        unixepoch_time = (tU64)(mjd - MJD_1_1_1970) * 24 * 60 * 60;
                        if((tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1+1,1) == 1 )&&( timePointbytes >= 6 ))
                        {
                            /* long form
                             */
                            utc = tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1+1+1,27);
                            unixepoch_time += ((0x1f&(utc >> (6+6+10))) * (60*60));     /* add hours */
                            unixepoch_time += ((0x3f&(utc >> (6+10))) * (60));  		/* add minutes */
                            unixepoch_time += ((0x3f&(utc >> (10))) * (1));  			/* add seconds */
                            if((timePointbytes >= 7 )&&( tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1,1) == 1 ))
                            {
#if 0
                                /* with LTO
                                 */
                                tU8 lto = timePoint[6]&0x1f;
                                if ( tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1+1+1+27+2,1) == 1 )
                                {
                                    unixepoch_time += lto*30*60;
                                }
                                else
                                {
                                    unixepoch_time -= lto*30*60;
                                }
#endif
                            }
                        }
                        else if ((tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1+1,1) == 0 )&&( timePointbytes >= 4 ))
                        {
                            /* short form
                             */
                            utc = tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1+1+1,11);
                            utc = (utc<<(6+10));
                            unixepoch_time += ((0x1f&(utc >> (6+6+10))) * (60*60));     /* add hours */
                            unixepoch_time += ((0x3f&(utc >> (6+10))) * (60));  		/* add minutes */
                            unixepoch_time += ((0x3f&(utc >> (10))) * (1));  			/* add seconds */
                            if((timePointbytes >= 5 )&&( tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1,1) == 1 ))
                            {
#if 0
                                /* with LTO
                                 */
                                tU8 lto = timePoint[4]&0x1f;
                                if ( tclDSMApp::u32BitStreamValue(timePoint,timePointbytes,1+17+1+1+1+11+2,1) == 1 )
                                {
                                    unixepoch_time += lto*30*60;
                                }
                                else
                                {
                                    unixepoch_time -= lto*30*60;
                                }
#endif
                            }
                        }
                    }
                }
            }
        }
    }
    sqlite3_result_int64(context,unixepoch_time);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
static void tclDSMDABEPGApp_dsmCheckContentId(
        sqlite3_context *context,
        int argc,
        sqlite3_value **argv
        )
{
    int result = 0;
    if( argc == 6 )
    {
        if( tclDB::dbValueType(argv[0]) == SQLITE_BLOB )
        {
            int contentIdbytes = tclDB::dbValueBytes(argv[0]);
            if( contentIdbytes > 1 )
            {
                tU8* contentId = (tU8*)tclDB::dbValueBlob(argv[0]);
                tS64 r_ensemble = tclDB::dbValueInt64(argv[1]);
                int b_service_type = tclDB::dbValueInt(argv[2]);
                tS64 r_service = tclDB::dbValueInt64(argv[3]);
                int b_scids = tclDB::dbValueInt(argv[4]);
                int b_xpad_app_type = tclDB::dbValueInt(argv[5]);

                if( b_xpad_app_type != 0 )
                {
                    /* not implemented ...
                     */
                }
                else if(( b_scids != 0xff )&&( b_scids != 0x0 ))
                {
                    /* not implemented ...
                     */
                }
                else if(( b_service_type != 0 ) &&( r_service != 0 ))
                {
                    int n = 1;
                    if( (contentId[0]&0x40) != 0 )
                    {
                        n += 3; /* ecc + 2 byte eid */
                    }
                    if( (contentId[0]&0x20) != 0 )
                    {
                        n += 1; /* 1 byte xpad */
                    }
                    if( (contentId[0]&0x10) != 0 )
                    {
                        n += 4; /* 32 bit sid */
                    }
                    else
                    {
                        n += 2; /* 16 bit sid */
                    }
                    if( contentIdbytes == n )
                    {
                        /* check SID first
                         */
                        if ( (contentId[0]&0x40) != 0 )
                        {
                            /* with ECC & EID ...
                             */
                            if( (contentId[0]&0x10) != 0 )
                            {
                                /* 32 bit sid */
                                if( (b_service_type == 2)&&
                                        (((r_service>>0)&0xffffffff) == (((contentId[4]&0xff)<<24)|((contentId[5]&0xff)<<16)|((contentId[6]&0xff)<<8)|((contentId[7]&0xff)<<0))) )
                                {
                                    result++;
                                }
                            }
                            else
                            {
                                /* 16 bit sid */
                                if( (b_service_type == 1)&&
                                        (((r_service>>24)&0xff) == 0x01)&&
                                        (((r_service>>0)&0xffff) == (((contentId[4]&0xff)<<8)|((contentId[5]&0xff)<<0))) )
                                {
                                    result++;
                                }
                            }
                        }
                        else
                        {
                            /* without ECC & EID ...
                             */
                            if( (contentId[0]&0x10) != 0 )
                            {
                                /* 32 bit sid */
                                if( (b_service_type == 2)&&
                                        (((r_service>>0)&0xffffffff) == (((contentId[1]&0xff)<<24)|((contentId[2]&0xff)<<16)|((contentId[3]&0xff)<<8)|((contentId[4]&0xff)<<0))) )
                                {
                                    result++;
                                }
                            }
                            else
                            {
                                /* 16 bit sid */
                                if( (b_service_type == 1)&&
                                        (((r_service>>24)&0xff) == 0x01)&&
                                        (((r_service>>0)&0xffff) == (((contentId[1]&0xff)<<8)|((contentId[2]&0xff)<<0))) )
                                {
                                    result++;
                                }
                            }
                        }

                        if((result != 0 )&&(b_service_type == 1))
                        {
                            /* when SID is maching ...
                             */
                            if ( (contentId[0]&0x40) != 0 )
                            {
                                /* check service ECC
                                 */
                                if( ((r_service>>16)&0xff) == contentId[1] )
                                {
                                    result++;
                                }
                                else if( ((r_service>>16)&0xff) == 0 )
                                {
                                    result++;
                                }
                                else if( 0 == contentId[1] )
                                {
                                    result++;
                                }
                                else
                                {
                                    result = 0;
                                }
                            }
                        }

                        if( r_ensemble != 0 )
                        {
                            if( result != 0 )
                            {
                                if ( (contentId[0]&0x40) != 0 )
                                {
                                    /* check enable ECC
                                     */
                                    if( ((r_ensemble>>16)&0xff) == contentId[1] )
                                    {
                                        result++;
                                    }
                                    if( ((r_ensemble>>16)&0xff) == 0 )
                                    {
                                        result++;
                                    }
                                    else if( 0 == contentId[1] )
                                    {
                                        result++;
                                    }
                                }
                            }

                            if( result != 0 )
                            {
                                if(( (contentId[0]&0x40) != 0 )&&( r_ensemble != 0 )&&(((r_ensemble>>24)&0xff)==0x01))
                                {
                                    if( result != 0 )
                                    {
                                        /* check EID
                                         */
                                        if( ((r_ensemble>>0)&0xffff) == (((contentId[3]&0xff)<<8)|((contentId[4]&0xff)<<0)) )
                                        {
                                            result++;
                                        }
                                        /* check ECC
                                         */
                                        if( ((r_ensemble>>16)&0xff) == contentId[1] )
                                        {
                                            result++;
                                        }
                                        if( ((r_ensemble>>16)&0xff) == 0 )
                                        {
                                            result++;
                                        }
                                        else if( 0 == contentId[1] )
                                        {
                                            result++;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                else if( r_ensemble != 0 )
                {
                    /* not implemented ...
                     */
                }
            }
        }
    }
    sqlite3_result_int(context,result);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
static void tclDSMDABEPGApp_dsmContentNameEQUrl(
        sqlite3_context *context,
        int argc,
        sqlite3_value **argv
        )
{
    if( argc == 2 )
    {
        tU8* contentName = (tU8*)tclDB::dbValueBlob(argv[0]);
        tU32 contentNameBytes = tclDB::dbValueBytes(argv[0]);
        char* url = (char*)tclDB::dbValueText(argv[1]);
        tU32 urlBytes = tclDB::dbValueBytes(argv[1]);
        if( urlBytes == (contentNameBytes-1) )
        {
            if( strncmp((char*)(&contentName[1]),url,urlBytes) == 0 )
            {
                sqlite3_result_int(context,1);
                return;
            }
        }
    }
    sqlite3_result_int(context,0);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABEPGApp::vInitializeTables(tclDSM* dsm)
{
    SQLITE_API int rc;

    ETG_TRACE_USR1(("vInitializeTables(%p) ...",dsm));

    rc = dsm->oDB()->dbCreateFunction(
                "timePoint2UnixTime",
                1,
                SQLITE_DETERMINISTIC,
                NULL,
                tclDSMDABEPGApp_timePoint2UnixTime,
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbCreateFunction(
                "dsmCheckContentId",
                6,
                SQLITE_DETERMINISTIC,
                NULL,
                tclDSMDABEPGApp_dsmCheckContentId,
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbCreateFunction(
                "dsmContentNameEQUrl",
                1,
                SQLITE_DETERMINISTIC,
                NULL,
                tclDSMDABEPGApp_dsmContentNameEQUrl,
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbCreateFunction(
                "dsmContentNameEQUrl",
                2,
                SQLITE_DETERMINISTIC,
                NULL,
                tclDSMDABEPGApp_dsmContentNameEQUrl,
                NULL,NULL,NULL);

    /*
     * create tables dab_epg*
     */
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_epg_root ("
                /***/ "a INTEGER NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "transpid INTEGER NOT NULL,"
                /***/ "dab_epg_o,"
                /***/ "root_tag INTEGER)",
                NULL,NULL,NULL);

    /*
     * create tables dab_epg*
     */
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_epg ("
                /***/ "o INTEGER PRIMARY KEY,"
                /***/ "p_o INTEGER," /* parent o */
            #ifdef CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN
                /***/ "addr INTEGER,"
                /***/ "p_addr INTEGER," /* parent addr */
            #endif
                /***/ "tag INTEGER,"
                /***/ "value)",
                NULL,NULL,NULL);

#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_epg_schedule_scope ("
                /***/ "a INTEGER NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "transpid INTEGER NOT NULL,"
                /***/ "dab_epg_schedule_o INTEGER,"
                /***/ "contentID BLOB)",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */

    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_epg_root_01 "
                /***/ "BEFORE DELETE ON dab_epg_root "
                /***/ "BEGIN "
                /****/ "DELETE FROM dab_epg WHERE o IN ("
                /****/ "WITH RECURSIVE "
                /****/ "dab_epg_nodes(o) AS ( "
                /****/ "VALUES(old.dab_epg_o) "
                /****/ "UNION "
                /****/ "SELECT dab_epg.o FROM dab_epg, dab_epg_nodes "
                /****/ "WHERE dab_epg.p_o=dab_epg_nodes.o "
                /****/ ") "
                /****/ "SELECT o FROM dab_epg_nodes);"
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                /****/ "DELETE FROM dab_epg_schedule_scope WHERE a=old.a AND transpid=old.transpid;"
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                /***/ "END",
                NULL,NULL,NULL);

#ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_epg_01 "
                /***/ "AFTER INSERT ON dab_epg "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size+20 WHERE o=new.a; "
                /***/ "END",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_epg_02 "
                /***/ "BEFORE DELETE ON dab_epg "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size-20 WHERE o=old.a; "
                /***/ "END",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN */

#ifndef CONFIG_DSM_AVOID_ANY_INDEX
#if 0
    /* this index makes no sense, because it is not very selective and
     * creates many flat updates while DG reception
     */
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_epg_1 ON dab_epg (tag)",
                NULL,NULL,NULL);
#endif
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_epg_2 ON dab_epg (p_o)",
                NULL,NULL,NULL);
    /* index for contentIDs */
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_epg_4 ON dab_epg (tag) WHERE tag=0x2980",
                NULL,NULL,NULL);
#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
#else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
    /* index for defaultContentIDs */
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_epg_5 ON dab_epg (tag) WHERE tag=0x05",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_epg_root_1 ON dab_epg_root (dab_epg_o)",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM_AVOID_ANY_INDEX */

    /* e.g.:
0003/serviceInformation
0003.0026/ensemble
0003.0026.0080/ATTR [blob]=0xE010BC
0003.0026.0010/shortName
0003.0026.0010.0001/ [text]=DR
0003.0026.0011/mediumName
0003.0026.0011.0001/ [text]=DR-Deutschland
0003.0026.0028/service
0003.0026.0028.0029/serviceID
0003.0026.0028.0029.0080/ATTR [blob]=0x40E010BCD210
0003.0026.0028.0010/shortName
0003.0026.0028.0010.0001/ [text]=DLF
0003.0026.0028.0011/mediumName
0003.0026.0028.0011.0001/ [text]=Deutschlandfunk
0003.0026.0028.0013/mediaDescription
0003.0026.0028.0013.002B/multimedia
0003.0026.0028.0013.002B.0080/ATTR [text]=image/png
0003.0026.0028.0013.002B.0082/ATTR [text]=radio_dradio-deutschlandfunk_128x128v.png
0003.0026.0028.0013.002B.0083/ATTR [blob]=0x02
0003.0026.0028.0013.002B.0084/ATTR [blob]=0x0080
0003.0026.0028.0013.002B.0085/ATTR [blob]=0x0080
     */
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                "v_dab_epg_service_multimedia "
                "AS "
                "SELECT "
                "R.a,"
                "R.transpid,"
                "N_serviceID_contentId.value AS contentId,"
                "("
                "SELECT X_multimedia_value.value FROM "
                "dab_epg AS X_multimedia_value "
                "WHERE "
                "N_multimedia.o=X_multimedia_value.p_o AND "
                "X_multimedia_value.tag=0x2B80 "
                ") AS mimeValue, "
                "N_multimedia_value.value AS url,"
                "("
                "SELECT blob2Int(X_multimedia_value.value) FROM "
                "dab_epg AS X_multimedia_value "
                "WHERE "
                "N_multimedia.o=X_multimedia_value.p_o AND "
                "X_multimedia_value.tag=0x2B83 "
                ") AS type, "
                "("
                "SELECT blob2Int(X_multimedia_value.value) FROM "
                "dab_epg AS X_multimedia_value "
                "WHERE "
                "N_multimedia.o=X_multimedia_value.p_o AND "
                "X_multimedia_value.tag=0x2B84 "
                ") AS width, "
                "("
                "SELECT blob2Int(X_multimedia_value.value) FROM "
                "dab_epg AS X_multimedia_value "
                "WHERE "
                "N_multimedia.o=X_multimedia_value.p_o AND "
                "X_multimedia_value.tag=0x2B85 "
                ") AS height "
                "FROM "
                "dab_epg_root AS R,"
                "dab_epg AS N_serviceInformation,"
                "dab_epg AS N_ensemble,"
                "dab_epg AS N_service,"
                "dab_epg AS N_mediaDescription,"
                "dab_epg AS N_multimedia,"
                "dab_epg AS N_multimedia_value,"
                "dab_epg AS N_serviceID,"
                "dab_epg AS N_serviceID_contentId "
                "WHERE "
                "R.dab_epg_o=N_serviceInformation.o AND "
                "N_serviceInformation.tag=0x03 AND "
                "N_serviceInformation.o=N_ensemble.p_o AND "
                "N_ensemble.o=N_service.p_o AND "
                "N_service.o=N_mediaDescription.p_o AND "
                "N_mediaDescription.o=N_multimedia.p_o AND "
                "N_multimedia.o=N_multimedia_value.p_o AND "
                "N_multimedia_value.tag=0x2B82 AND "
                "N_service.o=N_serviceID.p_o AND "
                "N_service.tag=0x2628 AND "
                "N_serviceID.o=N_serviceID_contentId.p_o AND "
                "N_serviceID_contentId.tag=0x2980",
                NULL,NULL,NULL);

    /* e.g.:
0002.0021.001C/programme
0002.0021.001C.0081/ATTR [integer]=631601
0002.0021.001C.0011/mediumName
0002.0021.001C.0011.0001/ [text]=Corso - Kultur n
0002.0021.001C.0012/longName
0002.0021.001C.0012.0001/ [text]=Corso - Kultur nach 3
0002.0021.001C.0013/mediaDescription
0002.0021.001C.0013.001A/shortDescription
0002.0021.001C.0013.001A.0001/ [text]=15:30 - 15:35 Nachrichten
0002.0021.001C.0019/location
0002.0021.001C.0019.002C/time
0002.0021.001C.0019.002C.0080/ATTR [integer]=1416837900/2014-11-24 14:05:00
0002.0021.001C.0019.002C.0081/ATTR [integer]=3300
     */

    /*
0002.0021.001C/programme
0002.0021.001C.0081/ATTR [integer]=1138510
0002.0021.001C.0011/mediumName
0002.0021.001C.0011.0001/ [text]=Nachrichten, Wet
0002.0021.001C.0012/longName
0002.0021.001C.0012.0001/ [text]=Nachrichten, Wetter
0002.0021.001C.0014/genre
0002.0021.001C.0014.0080/ATTR [blob]=0x030101
0002.0021.001C.0014/genre
0002.0021.001C.0014.0080/ATTR [blob]=0x0301010D
0002.0021.001C.0014/genre
0002.0021.001C.0014.0080/ATTR [blob]=0x0301010C
0002.0021.001C.0019/location
0002.0021.001C.0019.002C/time
0002.0021.001C.0019.002C.0080/ATTR [integer]=1385161202/2013-11-22 23:00:02
0002.0021.001C.0019.002C.0081/ATTR [integer]=178
     */

    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_epg_programme "
                /**/ "AS "
                     "SELECT "
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                     "T_epg_scope.a AS a,"
                     "T_epg_scope.contentID AS contentId,"
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "R.a AS a,"
                     "E_epg_defaultcontentid.value AS contentId,"
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "E_epg_schedule_programme_location_time_time.value AS startTime,"
                     "E_epg_schedule_programme_location_time_duration.value AS duration,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1101) AS mediumName,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1201) AS longName"
            #if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
                     ",(SELECT X3.value FROM dab_epg AS X1,dab_epg AS X2,dab_epg AS X3 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X3.p_o=X2.o AND X2.tag=0x131A AND X3.tag=0x1A01) AS shortDescription"
            #endif
            #ifdef CONFIG_DSM__ENABLE_DAB_EPG_ALL
                     ",(SELECT X3.value FROM dab_epg AS X1,dab_epg AS X2,dab_epg AS X3 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X3.p_o=X2.o AND X2.tag=0x131B AND X3.tag=0x1B01) AS longDescription,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1481) AS genre_type,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1480) AS genre_href,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1880) AS link_url,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1881) AS link_mimeValue,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1882) AS link_lang,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1883) AS link_description,"
                     "(SELECT X2.value FROM dab_epg AS X1,dab_epg AS X2 WHERE X1.p_o=E_epg_schedule_programme.o AND X2.p_o=X1.o AND X2.tag=0x1884) AS link_expirytime"
            #endif
                     " "
                     "FROM "
            #ifndef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                     "dab_epg_root AS R,"
            #endif
                     "dab_epg AS E_epg_schedule_programme,"
                     "dab_epg AS E_epg_schedule_programme_location,"
                     "dab_epg AS E_epg_schedule_programme_location_time,"
                     "dab_epg AS E_epg_schedule_programme_location_time_time,"
                     "dab_epg AS E_epg_schedule_programme_location_time_duration,"
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                     "dab_epg_schedule_scope AS T_epg_scope "
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "dab_epg AS E_epg_defaultcontentid,"
                     "dab_epg AS E_epg_schedule "
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "WHERE "
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                     "E_epg_schedule_programme.p_o=T_epg_scope.dab_epg_schedule_o AND "
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "R.dab_epg_o=E_epg_schedule.p_o AND "
                     "R.dab_epg_o=E_epg_defaultcontentid.p_o AND "
                     "R.root_tag=0x02 AND "
                     "E_epg_schedule.o=E_epg_schedule_programme.p_o AND "
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "E_epg_schedule_programme.o=E_epg_schedule_programme_location.p_o AND "
                     "E_epg_schedule_programme_location.o=E_epg_schedule_programme_location_time.p_o AND "
                     "E_epg_schedule_programme_location_time.o=E_epg_schedule_programme_location_time_time.p_o AND "
                     "E_epg_schedule_programme_location_time.o=E_epg_schedule_programme_location_time_duration.p_o AND "
                     "E_epg_schedule_programme_location_time_time.tag=0x2C80 AND "
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "E_epg_defaultcontentid.tag=0x05 AND "
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                     "E_epg_schedule_programme_location_time_duration.tag=0x2C81 "
                     "",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_epg "
                /**/ "AS "
                /**/ "WITH RECURSIVE "
            #ifdef CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN
                /**/ "epg_node(o,level,roottag,path,a,transpid,addr,tag,value) AS ( "
                /**/ "SELECT o,0,tag,printf('%04p',tag&0xff) AS path,dab_epg_root.a,dab_epg_root.transpid,addr,tag,value FROM dab_epg,dab_epg_root WHERE dab_epg.o=dab_epg_root.dab_epg_o AND p_o IS NULL AND (tag=0x02 OR tag=0x03) "
                /**/ "UNION ALL "
                /**/ "SELECT dab_epg.o,epg_node.level+1,epg_node.roottag,printf('%s.%04p',epg_node.path,dab_epg.tag&0xff),a,transpid,dab_epg.addr,dab_epg.tag,dab_epg.value "
            #else
                /**/ "epg_node(o,level,roottag,path,a,transpid,tag,value) AS ( "
                /**/ "SELECT o,0,tag,printf('%04p',tag&0xff) AS path,a,transpid,tag,value FROM dab_epg,dab_epg_root WHERE dab_epg.o=dab_epg_root.dab_epg_o AND p_o IS NULL AND (tag=0x02 OR tag=0x03) "
                /**/ "UNION ALL "
                /**/ "SELECT dab_epg.o,epg_node.level+1,epg_node.roottag,printf('%s.%04p',epg_node.path,dab_epg.tag&0xff),a,transpid,dab_epg.tag,dab_epg.value "
            #endif
                /**/ "FROM "
                /**/ "epg_node,"
                /**/ "dab_epg "
                /**/ "WHERE "
                /**/ "epg_node.o=dab_epg.p_o AND "
                /**/ "dab_epg.p_o NOT NULL "
                /**/ ") "
                /**/ "SELECT "
                /**/ "a, "
                /**/ "(SELECT label FROM dab_data_application WHERE o=epg_node.a) AS label, "
                /**/ "transpid, "
            #ifdef CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN
                /**/ "printf('0x%08p',addr) AS addr, "
            #else
            #endif
                /**/ "printf('0x%04p',roottag) AS roottag, "
                /**/ "level, "
                /**/ "printf('0x%04p',tag) AS tag, "
                /**/ "path || '/' || "
                /**/ "CASE WHEN tag&0xff=0x02 THEN 'epg' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x03 THEN 'serviceInformation' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x04 THEN 'tokenTableElement' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x05 THEN 'defaultcontentIDElement' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x10 THEN 'shortName' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x11 THEN 'mediumName' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x12 THEN 'longName' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x13 THEN 'mediaDescription' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x14 THEN 'genre' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x15 THEN 'CA' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x16 THEN 'keywords' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x17 THEN 'memberOf' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x18 THEN 'link' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x19 THEN 'location' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x1A THEN 'shortDescription' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x1B THEN 'longDescription' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x1C THEN 'programme' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x20 THEN 'programmeGroups' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x21 THEN 'schedule' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x22 THEN 'alternateSource' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x23 THEN 'programmeGroup' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x24 THEN 'scope' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x25 THEN 'serviceScope' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x26 THEN 'ensemble' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x27 THEN 'frequency' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x28 THEN 'service' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x29 THEN 'serviceID' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x2A THEN 'epgLanguage' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x2B THEN 'multimedia' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x2C THEN 'time' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x2D THEN 'bearer' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x2E THEN 'programmeEvent' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x2F THEN 'relativeTime' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff=0x30 THEN 'simulcast' ELSE '' END || "
                /**/ "CASE WHEN tag&0xff>=0x80 THEN 'ATTR' ELSE '' END || "
                /**/ "CASE WHEN value NOT NULL AND typeof(value)='blob' THEN ' [' || typeof(value) || ']=0x' || hex(value)  ELSE '' END || "
                /**/ "CASE WHEN value NOT NULL AND typeof(value)='text' THEN ' [' || typeof(value) || ']=""' || replace(replace(value,char(10),'\\r'),char(13),'\\n') || '""'  ELSE '' END || "
                /**/ "CASE WHEN value NOT NULL AND typeof(value)='integer' THEN ' [' || typeof(value) || ']=' || value || ''  ELSE '' END || "
                /**/ "CASE WHEN tag=0x2081 THEN '/' || datetime(value, 'unixepoch') ELSE '' END || "
                /**/ "CASE WHEN tag=0x2181 THEN '/' || datetime(value, 'unixepoch') ELSE '' END || "
                /**/ "CASE WHEN tag=0x2480 THEN '/' || datetime(value, 'unixepoch') ELSE '' END || "
                /**/ "CASE WHEN tag=0x2481 THEN '/' || datetime(value, 'unixepoch') ELSE '' END || "
                /**/ "CASE WHEN tag=0x2C80 THEN '/' || datetime(value, 'unixepoch') ELSE '' END || "
                /**/ "CASE WHEN tag=0x2C82 THEN '/' || datetime(value, 'unixepoch') ELSE '' END || "
                /**/ "'' "
                /**/ "AS value "
            #ifdef CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN
                /**/ "FROM epg_node ORDER BY a,transpid,addr; "
            #else
                /**/ "FROM epg_node ORDER BY a,transpid; "
            #endif
                     "",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS v_dab_epg_0x0002 AS SELECT * FROM v_dab_epg WHERE roottag='0x0002'",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS v_dab_epg_0x0003 AS SELECT * FROM v_dab_epg WHERE roottag='0x0003'",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_epg_file "
                /**/ "AS "
                /**/ "SELECT E.a,E.label,E.transpid,"
                /**/ "(SELECT F.name FROM v_dab_mot_file AS F WHERE F.a=E.a AND F.transpid=E.transpid) AS name,"
                /**/ "(SELECT datetime(P.datafield, 'unixepoch') FROM dab_mot_directory_param AS P WHERE P.a=E.a AND P.transpid=E.transpid AND paramid=13) AS uniquebodyversion,"
                /**/ "E.roottag,group_concat(E.value,char(10)) AS content FROM v_dab_epg AS E GROUP BY E.a,E.transpid"
                     "",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS v_dab_epg_file_0x0002 AS SELECT * FROM v_dab_epg_file WHERE roottag='0x0002'",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS v_dab_epg_file_0x0003 AS SELECT * FROM v_dab_epg_file WHERE roottag='0x0003'",
                NULL,NULL,NULL);

    (void)rc;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABEPGApp::vHandle60sCTimeTick()
{
    ETG_TRACE_USR1(("%p.vHandle60sCTimeTick() ...",(void*)this));
    tclDSMDABMOTApp::vHandle60sCTimeTick();
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::vHandleGarbageCollection()
{
    tBool ret;
    ETG_TRACE_USR1(("%p.vHandleGarbageCollection() ...",(void*)this));
    ret = tclDSMDABMOTApp::vHandleGarbageCollection();
    ETG_TRACE_USR1(("%p.vHandleGarbageCollection() ... %d done",
                    (void*)this,(int)ret));
    return ret;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABEPGApp::vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg)
{
    tclDSMDABMOTApp::vHandleMsgR_DATACHANNEL_FRAME(msg);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABEPGApp::vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg)
{
    tclDSMDABMOTApp::vHandleMsgR_TIMESTAMP(msg);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABEPGApp::vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg)
{
    tclDSMDABMOTApp::vHandleMsgR_SELECT_URI(msg);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
std::string tclDSMDABEPGApp::pInsertCDATAToken( tDecodeEPGObjectStatus* status, const char* cdata, tU32 cdata_bytes, tU8 ptag )
{
    std::string cdata_str = "";
    /*
     * All CDATA and other strings shall use ISO/IEC 10646 [4] with UTF-8 encoding.
     * The ISO/IEC 10646 [4] characters
     * 0xE000 to 0xF8FF shall not be included within any binary encoded strings.
     *
     * UTF-8 Encoding:
     * 0xxxxxxx:                                1Byte
     * 110xxxxx.10xxxxxx:                       2Byte
     * 1110xxxx.10xxxxxx.10xxxxxx:              3Byte
     * 11110xxx.10xxxxxx.10xxxxxx.10xxxxxx:     4Byte
     */
    tU32 maxstrlen;
    tU32 maxbytelen;
    switch(ptag)
    {
    case tclDSMDABEPGApp_Tag_shortName_0x10:
        maxstrlen = 8;
        break;
    case tclDSMDABEPGApp_Tag_mediumName_0x11:
        maxstrlen = 16;
        break;
    case tclDSMDABEPGApp_Tag_longName_0x12:
        maxstrlen = 128;
        break;
    case tclDSMDABEPGApp_Tag_shortDescription_0x1A:
        maxstrlen = 180;
        break;
    case tclDSMDABEPGApp_Tag_longDescription_0x1B:
        maxstrlen = 1200;
        break;
    default:
        maxstrlen = 256; /* by default, limit to 256 character */
        break;
    }
    maxbytelen = 4*maxstrlen+1; /* limit size in bytes */

    tU32 i;
    tU32 b_cnt=0; /* byte counter */
    tU32 c_cnt=0; /* UTF-8 character counter */
    for(i=0;i<cdata_bytes;i++)
    {
        char c = cdata[i];

        if( (c==0x01)||(c==0x02)||(c==0x03)||(c==0x04)||(c==0x05)||(c==0x06)||(c==0x07)||(c==0x08)||(c==0x0B)||(c==0x0C)||(c==0x0E)||(c==0x0F)||(c==0x10)||(c==0x11)||(c==0x12)||(c==0x13) )
        {
            if(( status->token_ptr[c&0x1f] != 0 )&&( status->token_length[c&0x1f] != 0 ))
            {
                /*
                 * check the token table ...
                 */
                tU32 k = 0;
                while (k<status->token_length[c&0x1f])
                {
                    /* process each token character ...
                     */
                    char tc = status->token_ptr[c&0x1f][k];
                    if( (((tU8)tc)&0xC0) != 0x80 )
                    {
                        c_cnt++;
                    }
                    if(maxstrlen!=0)
                    {
                        if((c_cnt>maxstrlen)||(b_cnt>maxbytelen))
                        {
#ifdef ENABLE_PRINT_EPG
                            printf(" (pInsertCDATAToken: warning: string (tag=0x%02x) limited to %d character)\n",ptag,(int)maxstrlen);
#endif
                            break;
                        }
                    }

                    cdata_str.append(1,tc);
                    b_cnt++;
                    k++;
                }
            }
            else
            {
#ifdef ENABLE_PRINT_EPG
                printf(" (pInsertCDATAToken: error undefined token %d) ",(int)c);
#endif
            }
        }
        else
        {
            if( (((tU8)c)&0xC0) != 0x80 )
            {
                c_cnt++;
            }
            if(maxstrlen!=0)
            {
                if((c_cnt>maxstrlen)||(b_cnt>maxbytelen))
                {
#ifdef ENABLE_PRINT_EPG
                    printf(" (pInsertCDATAToken: warning: string (tag=0x%02x) limited to %d character)\n",ptag,(int)maxstrlen);
#endif
                    break;
                }
            }

            if( c != 0 )
            {
                cdata_str.append(1,c);
                b_cnt++;
            }
        }
    }

    return cdata_str;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bUTF8Validate( const std::string& string )
{
    int l = (int)string.length();
    for (int i=0; i < l; i++)
    {
        int n = 0;
        int c = (unsigned char) string[i];
        // 00000000 -- 0000007F: 	0/0xxxxxxx
        if ((0x00 <= c) && (c <= 0x7f))
        {
            n=0; // 0bbbbbbb
        }
        /* 00000080 -- 000007FF: 	C/110xxxxx 10xxxxxx
         */
        else if ((c & 0xE0) == 0xC0)
        {
            n=1; // 110bbbbb
        }
        /* 00000800 -- 0000FFFF: 	E/1110xxxx 10xxxxxx 10xxxxxx
         */
        else if ((c & 0xF0) == 0xE0)
        {
            n=2; // 1110bbbb
        }
        /* 00010000 -- 001FFFFF: 	F/11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
         */
        else if ((c & 0xF8) == 0xF0)
        {
            n=3; // 11110bbb
        }
        else
        {
            return false;
        }
        /* check the subsequent bytes when available ...
         */
        for (int j=0; (j<n) && (i<l); j++)
        {
            /* n bytes matching
             * 10bbbbbb follow ?
             */
            if ( (++i == l) ||
                 (( (unsigned char)string[i] & 0xC0) != 0x80) )
            {
                return false;
            }
        }
    }
    return true;
}

static tclDSMDABEPGApp_TAGType getTagType(tU16 tag)
{
    switch(tag)
    {
    case tclDSMDABEPGApp_Attribute_programmeGroups_creationTime:
    case tclDSMDABEPGApp_Attribute_schedule_creationTime:
    case tclDSMDABEPGApp_Attribute_scope_startTime:
    case tclDSMDABEPGApp_Attribute_scope_stopTime:
    case tclDSMDABEPGApp_Attribute_serviceInformation_creationTime:
    case tclDSMDABEPGApp_Attribute_time_time:
    case tclDSMDABEPGApp_Attribute_time_actualTime:
    case tclDSMDABEPGApp_Attribute_link_expiryTime:
        return tclDSMDABEPGApp_TAGType_timePointType;

    case tclDSMDABEPGApp_Attribute_time_duration:
        return tclDSMDABEPGApp_TAGType_Integer;

    case tclDSMDABEPGApp_Attribute_multimedia_url:
        return tclDSMDABEPGApp_TAGType_URL;

    case tclDSMDABEPGApp_Attribute_multimedia_mimeValue:
        return tclDSMDABEPGApp_TAGType_MIMEtype;

    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_programmeGroup_0x23,0x80):
    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_memberOf_0x17,0x80):
    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_programme_0x1C,0x80):
    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_programmeEvent_0x2E,0x80):
        return tclDSMDABEPGApp_TAGType_CRID;

    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_programmeGroup_0x23,0x81):
    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_memberOf_0x17,0x81):
    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_programme_0x1C,0x81):
    case tclDSMDABEPGApp_16Bit_Tag(tclDSMDABEPGApp_Tag_programmeEvent_0x2E,0x81):
        return tclDSMDABEPGApp_TAGType_ShortCRID;
    default:
        break;
    }
    return tclDSMDABEPGApp_TAGType_UNDEF;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
static tBool bEPGTagSupported( tU32 level, tU8 ptag, tU8 tag )
{
    if( level > CONFIG_DSM__EPG_PARSER_DEPTH )
    {
        return FALSE;
    }

#ifdef CONFIG_DSM__ENABLE_DAB_EPG_ALL
    return TRUE;
#else /* CONFIG_DSM__ENABLE_DAB_EPG_ALL */
#ifdef CONFIG_DSM__ENABLE_DAB_EPG_BASIC_STATIONLOGOS
    if(( ptag == tclDSMDABEPGApp_Tag_UNDEF_0x00 )&&( tag == tclDSMDABEPGApp_Tag_serviceInformation_0x03 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_serviceInformation_0x03 )&&( tag == tclDSMDABEPGApp_Tag_ensemble_0x26 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_serviceInformation_0x03 )&&( tag == tclDSMDABEPGApp_Tag_tokenTableElement_0x04 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_ensemble_0x26 )&&( tag == tclDSMDABEPGApp_Tag_service_0x28 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_service_0x28 )&&( tag == tclDSMDABEPGApp_Tag_serviceID_0x29 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_serviceID_0x29 )&&( tag == tclDSMDABEPGApp_Tag_ATTR80 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_service_0x28 )&&( tag == tclDSMDABEPGApp_Tag_mediaDescription_0x13 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_mediaDescription_0x13 )&&( tag == tclDSMDABEPGApp_Tag_multimedia_0x2B )) return TRUE;
    /*if(( ptag == tclDSMDABEPGApp_Tag_multimedia_0x2B )&&( tag == tclDSMDABEPGApp_Tag_ATTR80 )) return TRUE;*/
    if(( ptag == tclDSMDABEPGApp_Tag_multimedia_0x2B )&&( tag == tclDSMDABEPGApp_Tag_ATTR82 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_multimedia_0x2B )&&( tag == tclDSMDABEPGApp_Tag_ATTR83 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_multimedia_0x2B )&&( tag == tclDSMDABEPGApp_Tag_ATTR84 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_multimedia_0x2B )&&( tag == tclDSMDABEPGApp_Tag_ATTR85 )) return TRUE;
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_BASIC_STATIONLOGOS */
#ifdef CONFIG_DSM__ENABLE_DAB_EPG_BASIC_SCHEDULE
    if(( ptag == tclDSMDABEPGApp_Tag_UNDEF_0x00 )&&( tag == tclDSMDABEPGApp_Tag_epg_0x02 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_epg_0x02 )&&( tag == tclDSMDABEPGApp_Tag_schedule_0x21 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_epg_0x02 )&&( tag == tclDSMDABEPGApp_Tag_tokenTableElement_0x04 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_schedule_0x21 )&&( tag == tclDSMDABEPGApp_Tag_programme_0x1C )) return TRUE;

    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_location_0x19 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_location_0x19 )&&( tag == tclDSMDABEPGApp_Tag_time_0x2C )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_time_0x2C )&&( tag == tclDSMDABEPGApp_Tag_ATTR80 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_time_0x2C )&&( tag == tclDSMDABEPGApp_Tag_ATTR81 )) return TRUE;

#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
    if(( ptag == tclDSMDABEPGApp_Tag_schedule_0x21 )&&( tag == tclDSMDABEPGApp_Tag_scope_0x24 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_scope_0x24 )&&( tag == tclDSMDABEPGApp_Tag_serviceScope_0x25 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_serviceScope_0x25 )&&( tag == tclDSMDABEPGApp_Tag_ATTR80 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_epg_0x02 )&&( tag == tclDSMDABEPGApp_Tag_defaultcontentIDElement_0x05 )) return TRUE;
#else /**/
    if(( ptag == tclDSMDABEPGApp_Tag_epg_0x02 )&&( tag == tclDSMDABEPGApp_Tag_defaultcontentIDElement_0x05 )) return TRUE;
#endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */

    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_mediumName_0x11 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_mediumName_0x11 )&&( tag == tclDSMDABEPGApp_Tag_ATTR01 )) return TRUE;

    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_longName_0x12 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_longName_0x12 )&&( tag == tclDSMDABEPGApp_Tag_ATTR01 )) return TRUE;

    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_mediumName_0x11 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_mediumName_0x11 )&&( tag == tclDSMDABEPGApp_Tag_ATTR01 )) return TRUE;

#if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_mediaDescription_0x13 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_mediaDescription_0x13 )&&( tag == tclDSMDABEPGApp_Tag_shortDescription_0x1A )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_shortDescription_0x1A )&&( tag == tclDSMDABEPGApp_Tag_ATTR01 )) return TRUE;
#endif

#if 0
    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_shortName_0x10 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_shortName_0x10 )&&( tag == tclDSMDABEPGApp_Tag_ATTR01 )) return TRUE;

    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_mediaDescription_0x13 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_mediaDescription_0x13 )&&( tag == tclDSMDABEPGApp_Tag_ATTR01 )) return TRUE;

    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_genre_0x14 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_genre_0x14 )&&( tag == tclDSMDABEPGApp_Tag_ATTR80 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_genre_0x14 )&&( tag == tclDSMDABEPGApp_Tag_ATTR81 )) return TRUE;

    if(( ptag == tclDSMDABEPGApp_Tag_programme_0x1C )&&( tag == tclDSMDABEPGApp_Tag_link_0x18 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_link_0x18 )&&( tag == tclDSMDABEPGApp_Tag_ATTR80 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_link_0x18 )&&( tag == tclDSMDABEPGApp_Tag_ATTR81 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_link_0x18 )&&( tag == tclDSMDABEPGApp_Tag_ATTR82 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_link_0x18 )&&( tag == tclDSMDABEPGApp_Tag_ATTR83 )) return TRUE;
    if(( ptag == tclDSMDABEPGApp_Tag_link_0x18 )&&( tag == tclDSMDABEPGApp_Tag_ATTR84 )) return TRUE;
#endif
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_BASIC_SCHEDULE */
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_ALL */

    (void)ptag;
    (void)tag;
    (void)level;
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bTryDecodeEPGObjectInsertRoot( tDecodeEPGObjectStatus* status, tS64 dab_epg_o, tU8 tag )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    rc = _dsm->oDB()->dbPrepare(
                "REPLACE INTO dab_epg_root (a,transpid,dab_epg_o,root_tag) VALUES(?,?,?,?)",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        status->a);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      2,
                                      status->u16_transport_id);

        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        3,
                                        dab_epg_o);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      4,
                                      tag);


        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                _dsm->oDB()->dbReset(pStmt);
                return TRUE;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    return FALSE;
}

#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bTryDecodeEPGObjectInsertScope( tDecodeEPGObjectStatus* status, tS64 schedule_o, unsigned char* contentID, tU32 contentIDBytes )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    rc = _dsm->oDB()->dbPrepare(
                "REPLACE INTO dab_epg_schedule_scope (a,transpid,dab_epg_schedule_o,contentID) VALUES(?,?,?,?)",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        status->a);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      2,
                                      status->u16_transport_id);

        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        3,
                                        schedule_o);

        _dsm->oDB()->tclDB::dbBindBlob(pStmt,
                                       4,
                                       contentID,
                                       contentIDBytes,
                                       0);


        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                _dsm->oDB()->dbReset(pStmt);
                return TRUE;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    return FALSE;
}
#endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bTryDecodeEPGObject( tDecodeEPGObjectStatus* status, tU32 level, tU8 ptag, tS64 p_o, tU32 p_addr, tU32 addr, tU32 taglen )
{
    SQLITE_API int rc = SQLITE_OK;

    if( status->pStmt == NULL )
    {
        _dsm->oDB()->dbPrepare(
            #ifdef CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN
                    "INSERT OR IGNORE INTO dab_epg (p_o,p_addr,addr,tag,value) VALUES(?,?,?,?,?)",
            #else /* CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN */
                    "INSERT OR IGNORE INTO dab_epg (p_o,tag,value) VALUES(?,?,?)",
            #endif /* CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN */
                    -1,
                    &status->pStmt,
                    NULL);
    }


    if(_dsm->oDB()->dbGetAutoCommit() != 0)
    {
        DSM_EPG_FATAL_DECODING_ERROR();
        return FALSE;
    }

    if( status->pStmt != NULL )
    {
        tU8* p = status->object_body;
        tU32 j = 0;
        while( ( (taglen) >= (addr+j+2) )&&( (addr+j) < status->object_body_bytes )&&( (addr+j) < taglen ) )
        {
            int paramidx = 1;
            tS64 o = 0;
            volatile tU8* pFrame = &p[addr+j];
            tU8 element_tag;
            tU32 element_length;
            tU32 element_addr = addr+j;

#ifdef ENABLE_PRINT_EPG
            printf("0x%08x: ",addr+j);
#endif

            if( p_o == 0 )
            {
                _dsm->oDB()->tclDB::dbBindNull(status->pStmt,
                                               paramidx++);
            }
            else
            {
                _dsm->oDB()->tclDB::dbBindInt64(status->pStmt,
                                                paramidx++,
                                                p_o);
            }

#ifdef CONFIG_DSM_ENABLE_DAB_EPG_ADDR_COLUMN
            /* p_addr
             */
            _dsm->oDB()->tclDB::dbBindInt(status->pStmt,
                                          paramidx++,
                                          p_addr);

            /* addr
             */
            _dsm->oDB()->tclDB::dbBindInt(status->pStmt,
                                          paramidx++,
                                          addr+j);
#else
            (void)p_addr;
#endif

            status->u32NumDecodedElements++;

            element_tag = p[addr+j++];
            element_length = p[addr+j++];

            if( ptag == 0x04 )
            {
                /* TOKEN element
 * This element is not defined in the XML specification. Frequently recurring strings in the EPG character data ("tokens")
 * can be encoded using a token table. A maximum of 16 tokens are allowed per table. This table defines tags (bytes that
 * can be identified in the character data stream) and their equivalent strings. Whenever a decoder finds a token tag in a
 * character stream it shall replace the tag with its equivalent string. This element can only occur within the two top-level
 * elements (epg and serviceInformation) and, if present, it shall occur before any other elements. This element applies to
 * all character data within the parent top-level element (i.e. epg or serviceInformation) and all children of the parent
 * element. This element shall be encoded as defined in clause 4.3, with the following provisos:
 * element_tag: This shall always be 0x04.
 * element_data_byte: These bytes contain a sequence of one or more tokens (see below).
 *
 * Entries in the token table are encoded as a unique tag and its associated string.
 * Token strings shall never include references to other tokens.
 *
 * token_ tag: This byte identifies the token. There are 16 possible tag values (these are all non-printing characters):
 * 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x0B, 0x0C, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13
 * NOTE: This excludes the values 0x00 (null), 0x09 (tab), 0x0A (linefeed) and 0x0D (carriage return).
 * Each tag shall occur at most once within the token table.
 *
 * token_length: This field indicates the number of data bytes in the token string. The range of this is 0x00 to 0xFF
 * (i.e. 0 to 255).
 *
 * token_data_byte: The token string.
 */
#ifdef ENABLE_PRINT_EPG
                for( tU32 i=0;i<level;i++ )
                {
                    printf(" ");
                }
                printf("TOKEN=0x%02x[%d] ",element_tag,element_length);
#endif

                if( element_length > (status->object_body_bytes-addr-j) )
                {
                    /* invalid length information
                     */
                    _dsm->oDB()->dbReset(status->pStmt);
                    DSM_EPG_FATAL_DECODING_ERROR();
                    return FALSE;
                }

                if( element_tag <= 0xf )
                {
                    status->token_ptr[element_tag&0x1f] = (char*)&p[addr+j];
                    status->token_length[element_tag&0x1f] = element_length;
                }

#ifdef ENABLE_PRINT_EPG
                {
                    for(tU32 i=0;i<element_length;i++)
                    {
                        if(( p[addr+j+i] < 32 )||( p[addr+j+i] > 127 ))
                        {
                            printf(".");
                        }
                        else
                        {
                            printf("%c",p[addr+j+i]);
                        }
                    }
                }
                printf("\n");
#endif
            }
            else
            {
                if( element_length == 0xFE )
                {
                    /* 16 bit length
                 */
                    if( (taglen-addr) < (j+2) )
                    {
                        _dsm->oDB()->dbReset(status->pStmt);
                        DSM_EPG_FATAL_DECODING_ERROR();
                        return FALSE;
                    }
                    element_length = 0;
                    element_length |= (p[addr+j++]<<8);
                    element_length |= (p[addr+j++]<<0);

                }
                else if( element_length == 0xFF )
                {
                    if( (taglen-addr) < (j+3) )
                    {
                        _dsm->oDB()->dbReset(status->pStmt);
                        DSM_EPG_FATAL_DECODING_ERROR();
                        return FALSE;
                    }
                    element_length = 0;
                    element_length |= (p[addr+j++]<<16);
                    element_length |= (p[addr+j++]<<8);
                    element_length |= (p[addr+j++]<<0);
                }

                if( element_length > (status->object_body_bytes-addr-j) )
                {
                    /* invalid length information
                     */
                    _dsm->oDB()->dbReset(status->pStmt);
                    DSM_EPG_FATAL_DECODING_ERROR();
                    return FALSE;
                }

#ifdef ENABLE_PRINT_EPG
                for( tU32 i=0;i<level;i++ )
                {
                    printf(" ");
                }
                printf("TAG=0x%02x[%d] ",element_tag,element_length);
#endif

                /* check possible top level tags
                 */
                if((level==0)&&((element_tag!=0x02)&&(element_tag!=0x03)))
                {
#ifdef ENABLE_PRINT_EPG
                    printf("invalid top level tag, abort");
#endif
                    _dsm->oDB()->dbReset(status->pStmt);
                    DSM_EPG_FATAL_DECODING_ERROR();
                    return FALSE;
                }

                if( element_tag == 0x7f )
                {
#ifdef ENABLE_PRINT_EPG
                    printf("INVALID\n");
#endif
                    _dsm->oDB()->dbReset(status->pStmt);
                    DSM_EPG_FATAL_DECODING_ERROR();
                    return FALSE;
                }
                else if( element_tag == 0x01 )
                {
                    int tag = (((ptag&0xff)<<8)|(element_tag&0xff));
                    /* CDATA element
                     */
                    /* tag
                     */
                    _dsm->oDB()->tclDB::dbBindInt(status->pStmt,
                                                  paramidx++,
                                                  tag);

                    std::string cdata_str = "";

                    /* tag
                     */
                    if( element_length == 0 )
                    {
                        _dsm->oDB()->tclDB::dbBindNull(status->pStmt,
                                                       paramidx++);
                    }
                    else
                    {
                        char* t = (char*)&p[addr+j];
                        if( (taglen-addr) < (j+element_length) )
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            DSM_EPG_FATAL_DECODING_ERROR();
                            return FALSE;
                        }

                        cdata_str = pInsertCDATAToken(status,(const char*)t,element_length,ptag);
                        if( bUTF8Validate(cdata_str) == false )
                        {
                            /* don't store invalid UTF strings
                             */
#ifdef ENABLE_PRINT_EPG
                            printf(" (UTF8 Error:%s) ",cdata_str.c_str());
#endif
                            cdata_str = "";
                        }

                        _dsm->oDB()->tclDB::dbBindText(status->pStmt,
                                                       paramidx++,
                                                       (const char*)cdata_str.c_str(),
                                                       (int)cdata_str.length(),
                                                       0);
                    }
#ifdef ENABLE_PRINT_EPG
                    printf("CDATA ");
                    {
                        for(tU32 i=0;i<element_length;i++)
                        {
                            if(( p[addr+j+i] < 32 )||( p[addr+j+i] > 127 ))
                            {
                                printf(".");
                            }
                            else
                            {
                                printf("%c",p[addr+j+i]);
                            }
                        }
                    }
                    printf("\n");
#endif

                    if( bEPGTagSupported(level,ptag,element_tag) == TRUE )
                    {
                        while(1)
                        {
                            rc = _dsm->oDB()->dbStep(status->pStmt);
                            if( rc == SQLITE_DONE )
                            {
                                break;
                            }
                            else if( tclDB::dbOnStepError(rc) )
                            {
                                _dsm->oDB()->dbReset(status->pStmt);
                                _doRollbackTransaction = TRUE;
                                return FALSE;
                            }
                        }
                    }
                }
#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                else if(((ptag == tclDSMDABEPGApp_Tag_serviceScope_0x25)&&( element_tag == tclDSMDABEPGApp_Tag_ATTR80 ))&&
                        ( bEPGTagSupported(level,ptag,element_tag) == TRUE ))
                {
                    status->pu8ContentID = (unsigned char*)&p[addr+j];
                    status->u32ContentIDBytes = element_length;
                    if( (taglen-addr) < (j+element_length) )
                    {
                        _dsm->oDB()->dbReset(status->pStmt);
                        DSM_EPG_FATAL_DECODING_ERROR();
                        return FALSE;
                    }
                }
#endif
                else if(element_tag >= 0x80 )
                {
                    int tag = (((ptag&0xff)<<8)|(element_tag&0xff));
                    /* attribute tag ...
                     */

                    _dsm->oDB()->tclDB::dbBindInt(status->pStmt,
                                                  paramidx++,
                                                  tag);

                    tclDSMDABEPGApp_TAGType tagtype = getTagType((tU16)tag);

                    if( ( tagtype == tclDSMDABEPGApp_TAGType_URL )||
                            ( tagtype == tclDSMDABEPGApp_TAGType_CRID)||
                            ( tagtype == tclDSMDABEPGApp_TAGType_MIMEtype ) )
                    {
                        unsigned char* t = (unsigned char*)&p[addr+j];
                        if( (taglen-addr) < (j+element_length) )
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            DSM_EPG_FATAL_DECODING_ERROR();
                            return FALSE;
                        }
                        _dsm->oDB()->tclDB::dbBindText(status->pStmt,
                                                       paramidx++,
                                                       (const char*)t,
                                                       element_length,
                                                       0);
                    }
                    else if( tagtype == tclDSMDABEPGApp_TAGType_ShortCRID )
                    {
                        unsigned char* v = (unsigned char*)&p[addr+j];

                        if( (taglen-addr) < (j+element_length) )
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            DSM_EPG_FATAL_DECODING_ERROR();
                            return FALSE;
                        }

                        _dsm->oDB()->tclDB::dbBindInt64(status->pStmt,
                                                        paramidx++,
                                                        tclDSMApp::u32BitStreamValue(v,element_length,0,8*element_length));
                    }
                    else if( tagtype == tclDSMDABEPGApp_TAGType_Integer )
                    {
                        unsigned char* v = (unsigned char*)&p[addr+j];

                        if( (taglen-addr) < (j+element_length) )
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            DSM_EPG_FATAL_DECODING_ERROR();
                            return FALSE;
                        }

                        _dsm->oDB()->tclDB::dbBindInt64(status->pStmt,
                                                        paramidx++,
                                                        tclDSMApp::u32BitStreamValue(v,element_length,0,8*element_length));
                    }
                    else if( tagtype == tclDSMDABEPGApp_TAGType_timePointType )
                    {
                        unsigned char* v = (unsigned char*)&p[addr+j];

                        if( (taglen-addr) < (j+element_length) )
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            DSM_EPG_FATAL_DECODING_ERROR();
                            return FALSE;
                        }

                        tU64 unixepoch_time = 0;
                        tU32 mjd = tclDSMApp::u32BitStreamValue(v,element_length,1,17);
                        tU32 utc = 0;
                        if( mjd >= MJD_1_1_1970 )
                        {
                            unixepoch_time = (tU64)(mjd - 40587) * 24 * 60 * 60;
                            if((tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1,1) == 1 )&&( element_length >= 6 ))
                            {
                                /* long form
                             */
                                utc = tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1,27);
                                unixepoch_time += ((0x1f&(utc >> (6+6+10))) * (60*60));     /* add hours */
                                unixepoch_time += ((0x3f&(utc >> (6+10))) * (60));  		/* add minutes */
                                unixepoch_time += ((0x3f&(utc >> (10))) * (1));  			/* add seconds */
                                if((element_length >= 7 )&&( tclDSMApp::u32BitStreamValue(v,element_length,1+17+1,1) == 1 ))
                                {
#if 0
                                    tU8 lto = tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1+27+2+1,5);
                                    //printf("tclDSMDABEPGApp_TAGType_timePointType A %ld %d lto=%d\n",(long)unixepoch_time,tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1+27+2,1),lto);
                                    if ( tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1+27+2,1) == 1 )
                                    {
                                        unixepoch_time += lto*30*60;
                                    }
                                    else
                                    {
                                        unixepoch_time -= lto*30*60;
                                    }
#endif
                                }
                            }
                            else if ((tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1,1) == 0 )&&( element_length >= 4 ))
                            {
                                /* short form
                             */
                                utc = tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1,11);
                                utc = (utc<<(6+10));
                                unixepoch_time += ((0x1f&(utc >> (6+6+10))) * (60*60));     /* add hours */
                                unixepoch_time += ((0x3f&(utc >> (6+10))) * (60));  		/* add minutes */
                                unixepoch_time += ((0x3f&(utc >> (10))) * (1));  			/* add seconds */
                                if((element_length >= 5 )&&( tclDSMApp::u32BitStreamValue(v,element_length,1+17+1,1) == 1 ))
                                {
#if 0
                                    tU8 lto = tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1+11+2+1,5);
                                    //printf("tclDSMDABEPGApp_TAGType_timePointType B %ld %d lto=%d\n",(long)unixepoch_time,tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1+11+2,1),lto);
                                    if ( tclDSMApp::u32BitStreamValue(v,element_length,1+17+1+1+1+11+2,1) == 1 )
                                    {
                                        unixepoch_time += lto*30*60;
                                    }
                                    else
                                    {
                                        unixepoch_time -= lto*30*60;
                                    }
#endif
                                }
                            }
                        }

                        _dsm->oDB()->tclDB::dbBindInt64(status->pStmt,
                                                        paramidx++,
                                                        unixepoch_time);
                    }
                    else
                    {
                        unsigned char* t = (unsigned char*)&p[addr+j];
                        if( (taglen-addr) < (j+element_length) )
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            DSM_EPG_FATAL_DECODING_ERROR();
                            return FALSE;
                        }
                        _dsm->oDB()->tclDB::dbBindBlob(status->pStmt,
                                                       paramidx++,
                                                       t,
                                                       element_length,
                                                       0);
                    }
#ifdef ENABLE_PRINT_EPG
                    printf("Attribute ");
                    {
                        for(tU32 i=0;i<element_length;i++)
                        {
                            printf("%02x",p[addr+j+i]);
                        }
                        printf(" ");
                        for(tU32 i=0;i<element_length;i++)
                        {
                            if( p[addr+j+i] < 32 )
                            {
                                printf(".");
                            }
                            else
                            {
                                printf("%c",p[addr+j+i]);
                            }
                        }
                    }
                    printf("\n");
#endif
                    if( bEPGTagSupported(level,ptag,element_tag) == TRUE )
                    {
                        while(1)
                        {
                            rc = _dsm->oDB()->dbStep(status->pStmt);
                            if( rc == SQLITE_DONE )
                            {
                                break;
                            }
                            else if( tclDB::dbOnStepError(rc) )
                            {
                                _dsm->oDB()->dbReset(status->pStmt);
                                _doRollbackTransaction = TRUE;
                                return FALSE;
                            }
                        }
                    }
                }
                else if(( element_tag == 4 )||
                        ( element_tag == 2 )||
                        ( element_tag == 3 )||
                        (( element_tag >= 0x10 )&&
                         ( element_tag <= 0x7E )))
                {
                    int tag = (((ptag&0xff)<<8)|(element_tag&0xff));
                    /* element tag ...
                     */
                    _dsm->oDB()->tclDB::dbBindInt(status->pStmt,
                                                  paramidx++,
                                                  tag);

                    _dsm->oDB()->tclDB::dbBindNull(status->pStmt,
                                                   paramidx++);
#ifdef ENABLE_PRINT_EPG
                    printf("Element ");
                    printf("\n");
#endif
                    if( bEPGTagSupported(level,ptag,element_tag) == TRUE )
                    {
                        while(1)
                        {
                            rc = _dsm->oDB()->dbStep(status->pStmt);
                            if( rc == SQLITE_DONE )
                            {
                                o = _dsm->oDB()->tclDB::dbLastInsertRowId();
                                if( level == 0 )
                                {
                                    if( bTryDecodeEPGObjectInsertRoot(status,o,element_tag) != TRUE )
                                    {
                                        _dsm->oDB()->dbReset(status->pStmt);
                                        _doRollbackTransaction = TRUE;
                                        return FALSE;
                                    }
                                }
                                break;
                            }
                            else if( tclDB::dbOnStepError(rc) )
                            {
                                _dsm->oDB()->dbReset(status->pStmt);
                                _doRollbackTransaction = TRUE;
                                return FALSE;
                            }
                        }
                        if( level > CONFIG_DSM__EPG_PARSER_DEPTH )
                        {
                            DSM_EPG_FATAL_DECODING_ERROR();
                        }
                        else
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            if( o != 0 )
                            {
                                if( bTryDecodeEPGObject(status,
                                                        level+1,
                                                        element_tag,
                                                        o,
                                                        element_addr,
                                                        addr+j,
                                                        addr+j+element_length) == FALSE )
                                {
                                    _dsm->oDB()->dbReset(status->pStmt);
                                    return FALSE;
                                }
                            }
                            else
                            {
                                _dsm->oDB()->dbReset(status->pStmt);
                                return FALSE;
                            }
                        }
                    }
                }
#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                else if ( element_tag == tclDSMDABEPGApp_Tag_defaultcontentIDElement_0x05 )
                {
                    status->pu8DefaultContentID = (unsigned char*)&p[addr+j];
                    status->u32DefaultContentIDBytes = element_length;
                    if( (taglen-addr) < (j+element_length) )
                    {
                        _dsm->oDB()->dbReset(status->pStmt);
                        DSM_EPG_FATAL_DECODING_ERROR();
                        return FALSE;
                    }
                }
#endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                else if((
            #ifndef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                            ( element_tag == tclDSMDABEPGApp_Tag_defaultcontentIDElement_0x05 )||
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                            ( element_tag == tclDSMDABEPGApp_Tag_languageElement_0x06 ))&&
                        ( bEPGTagSupported(level,ptag,element_tag) == TRUE ))
                {
                    /* delault contentId or Language Element ...
                     */
                    _dsm->oDB()->tclDB::dbBindInt(status->pStmt,
                                                  paramidx++,
                                                  element_tag);

                    unsigned char* t = (unsigned char*)&p[addr+j];
                    if( (taglen-addr) < (j+element_length) )
                    {
                        _dsm->oDB()->dbReset(status->pStmt);
                        DSM_EPG_FATAL_DECODING_ERROR();
                        return FALSE;
                    }
                    _dsm->oDB()->tclDB::dbBindBlob(status->pStmt,
                                                   paramidx++,
                                                   t,
                                                   element_length,
                                                   0);
#ifdef ENABLE_PRINT_EPG
                    printf("defaultcontentIdElement ");
                    {
                        for(tU32 i=0;i<element_length;i++)
                        {
                            printf("%02x",p[addr+j+i]);
                        }
                    }
                    printf("\n");
#endif
                    while(1)
                    {
                        rc = _dsm->oDB()->dbStep(status->pStmt);
                        if( rc == SQLITE_DONE )
                        {
                            break;
                        }
                        else if( tclDB::dbOnStepError(rc) )
                        {
                            _dsm->oDB()->dbReset(status->pStmt);
                            _doRollbackTransaction = TRUE;
                            return FALSE;
                        }
                    }
                }
                else
                {
#ifdef ENABLE_PRINT_EPG
                    printf("INVALID ptag=0x%02x tag=0x%02x\n",ptag,element_tag);
#endif
                }
            }

            _dsm->oDB()->dbReset(status->pStmt);

            j+=element_length;

            if( (j+addr) > status->object_body_bytes )
            {
                _dsm->oDB()->dbReset(status->pStmt);
                DSM_EPG_FATAL_DECODING_ERROR();
                return FALSE;
            }

            (void)pFrame;
        }
        _dsm->oDB()->dbReset(status->pStmt);

#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
        if( ptag == tclDSMDABEPGApp_Tag_schedule_0x21 )
        {
            if( (status->pu8ContentID != NULL) && (status->u32ContentIDBytes != 0) )
            {
                if( bTryDecodeEPGObjectInsertScope(status,p_o,status->pu8ContentID,status->u32ContentIDBytes) != TRUE )
                {
                    _dsm->oDB()->dbReset(status->pStmt);
                    _doRollbackTransaction = TRUE;
                    return FALSE;
                }
            }
            else if( (status->pu8DefaultContentID != NULL) && (status->u32DefaultContentIDBytes != 0) )
            {
                if( bTryDecodeEPGObjectInsertScope(status,p_o,status->pu8DefaultContentID,status->u32DefaultContentIDBytes) != TRUE )
                {
                    _dsm->oDB()->dbReset(status->pStmt);
                    _doRollbackTransaction = TRUE;
                    return FALSE;
                }
            }
        }
#endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */

        addr += j;
        if( addr != taglen )
        {
            DSM_EPG_FATAL_DECODING_ERROR();
            return FALSE;
        }
    }

    if( tclDB::dbOnStepError(rc) )
    {
        DSM_EPG_FATAL_DECODING_ERROR();
        return FALSE;
    }

    return TRUE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bTryDecodeEPGObject( tDecodeEPGObjectStatus* status )
{
#ifdef ENABLE_PRINT_EPG
    printf("\n");
#endif
    if( status->object_body_bytes < 2 )
    {
        return FALSE;
    }
    if( (status->object_body[0]!=0x02)&&(status->object_body[0]!=0x03) )
    {
        return FALSE;
    }
#ifdef ENABLE_PRINT_EPG
    {
        printf("a=%ld transport_id=%04x: \n",(long)status->a,status->u16_transport_id);
        printf("bTryDecodeEPGObject() %d 0x",status->object_body_bytes);
        for(int i=0;i<status->object_body_bytes;i++)
        {
            printf("%02x",status->object_body[i]);
        }
        printf("\n");
    }
#endif
    ETG_TRACE_USR1(("%p.bTryDecodeEPGObject(%lld,0x%04x,%d) ...",
                    (void*)this,
                    status->a,
                    status->u16_transport_id,
                    status->object_body_bytes));

    if( bTryDecodeEPGObject(status,0,0,0,0,0,status->object_body_bytes) == TRUE )
    {
        ETG_TRACE_USR1(("%p.bTryDecodeEPGObject(%lld,%d/0x%04x,%d) OK u32NumDecodedElements=%d ... done",
                        (void*)this,
                        status->a,
                        status->u16_transport_id,
                        status->u16_transport_id,
                        status->object_body_bytes,
                        status->u32NumDecodedElements));
        _dsm->vBroadCastEvent( DSM_EV_APPLICATION_EPG_OBJECT_DECODED,
                               string_sprintf("/dsm/epg?a=%d?transpid=0x%04x",
                                              (int)status->a,
                                              status->u16_transport_id) );
        return TRUE;
    }
    else
    {
        ETG_TRACE_USR1(("%p.bTryDecodeEPGObject(%lld,%d/0x%04x,%d) FAIL u32NumDecodedElements=%d ... done",
                        (void*)this,
                        status->a,
                        status->u16_transport_id,
                        status->u16_transport_id,
                        status->object_body_bytes,
                        status->u32NumDecodedElements));
        //DSM_EPG_FATAL_DECODING_ERROR();
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
#ifdef CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER
#define MOTDIR_PARAMID_1            1
#define MOTDIR_PARAMID_TRIGGERTIME  5
#define MOTDIR_PARAMID_6            6
#define MOTDIR_PARAMID_9            9
#define MOTDIR_PARAMID_CNAME        12
#define MOTDIR_PARAMID_VERSION      13
#define MOTDIR_PARAMID_16           16
#define MOTDIR_PARAMID_17           17
#define MOTDIR_PARAMID_32           32
#define MOTDIR_PARAMID_34           34
#define MOTDIR_PARAMID_11           11
#define MOTDIR_PARAMID_33           33 /* 0x21 ProfileSubset */
#define MOTDIR_PARAMID_35           35 /* 0x23 CAInfo */

#define MOT_DIRECTORY_PARAMID__BODYSIZE         0x40
#define MOT_DIRECTORY_PARAMID__HEADERSIZE       0x41
#define MOT_DIRECTORY_PARAMID__CONTENTTYPE      0x42
#define MOT_DIRECTORY_PARAMID__CONTENTSUBTYPE   0x43

tBool tclDSMDABEPGApp::bDirectoryParamFilter( tU8 u8ContentType, tU16 u16ContentSubType, tU8 u8ParamId)
{
#if 0
    return TRUE;
#else
    switch( u8ContentType )
    {
    case 0: /* General Data */
        return FALSE;
    case 2: /* Application */
        switch( u16ContentSubType )
        {

        case 0: /* gif */
        case 1: /* jfif */
        case 2: /* bmp */
        case 3: /* png */
            switch( u8ParamId )
            {
            case MOTDIR_PARAMID_CNAME: return TRUE;
            case MOT_DIRECTORY_PARAMID__BODYSIZE: return TRUE;
            default:
                break;
            }
            break;
        default: break;
        }
        return FALSE;
    case 7: /* Application */
        switch( u16ContentSubType )
        {
        case 0: /* 7/0 Object contains Service Information */
        case 1: /* 7/1 Object contains Programme Information */
        case 2: /* 7/2 Object contains Group Information */
            switch( u8ParamId )
            {
            case MOTDIR_PARAMID_VERSION: return TRUE;
            case MOTDIR_PARAMID_17: return TRUE;

            case MOTDIR_PARAMID_16: return TRUE;
            case MOTDIR_PARAMID_9: return TRUE;
            case MOTDIR_PARAMID_33: return TRUE; /* ProfileSubset */

            case MOT_DIRECTORY_PARAMID__BODYSIZE: return TRUE;
            case MOT_DIRECTORY_PARAMID__CONTENTTYPE: return TRUE;
            case MOT_DIRECTORY_PARAMID__CONTENTSUBTYPE: return TRUE;

            default:
                break;
            }
            break;
        default: break;
        }
        return FALSE;
    default: break;
    }

    printf("unhandled u8ContentType=%02x u16ContentSubType=%02x u8ParamId=%d\n",u8ContentType,u16ContentSubType,u8ParamId);
    return FALSE;
#endif
}
#endif /* CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER */

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bTryDecodeMOTDirectoryAppSpecificPreHandling( tS64 a, tU16 u16_transport_id, tU8 u8DGType )
{
    (void)u16_transport_id;
    (void)u8DGType;

    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;
    /* Delete advance profile objects first
     * While object complete processing,
     * the segements from out of size objects (e.g. EPG advance profile objects) might be
     * cleared.
     * Now, any other wrong profile objects will we revoved
     * (e.g. advance profile objects within basic profile size limit).
     *
     */
    rc = _dsm->oDB()->dbPrepare(
                "SELECT P.transpid FROM dab_mot_directory_param AS P WHERE P.a=? AND P.paramid=33 AND P.datafield!=1",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        a);
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    tU16 u16WrongProfileSubsetTranspId = (tU16)tclDB::dbColumnInt( pStmt, 0 );
                    vClearDGSegemt(a,4,u16WrongProfileSubsetTranspId);
                    vForceSetObjectComplete( a, 4, u16WrongProfileSubsetTranspId );
                    //printf("delete %d\n",u16WrongProfileSubsetTranspId);
                }
            }
            else if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }

    return true;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
int tclDSMDABEPGApp::u32EPGObjectCompressionType( tS64 a, tU16 u16_transport_id )
{
    int compressiontype = -1;

    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    rc = _dsm->oDB()->dbPrepare(
                "SELECT datafield FROM dab_mot_directory_param WHERE a=?1 AND transpid=?2 AND paramid=17",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        a);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      2,
                                      u16_transport_id);

        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                compressiontype = 0;
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    compressiontype = tclDB::dbColumnInt( pStmt, 0 );
                }
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    return compressiontype;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU32 tclDSMDABEPGApp::u32ObjectSizeLimit(tU8 u8DGType,tU16 u16TranspID)
{
    (void)u8DGType;
    (void)u16TranspID;
    return _u32_object_size_limit;
}

#ifdef CONFIG_DSM_ENABLE_bTryDecodeEPGObjectIfNessesary_OPTIMIZIATION
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary( tU32* p32EPGByteLimit, tS64 a, tU16 u16_transport_id, const void* object_body, tU32 object_body_bytes )
{
    //SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    DSM_PERF_ENTRY();

#ifdef CONFIG_DSM__EPG_ENABLE_BURST_BYTE_LIMIT
    if( p32EPGByteLimit != NULL )
    {
        if( *p32EPGByteLimit < object_body_bytes )
        {
            DSM_PERF_EXIT("tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary");
            return FALSE;
        }
        *p32EPGByteLimit -= object_body_bytes;
    }
#else
    (void)p32EPGByteLimit;
#endif

    int compressiontype = u32EPGObjectCompressionType(a,u16_transport_id);

    if( compressiontype != -1 )
    {
        tU32 n = object_body_bytes;
        if( n != 0 )
        {
#ifndef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS
            tU8* p_unzip = NULL;
#endif
            tU8* p = (tU8*)object_body;

            if((compressiontype==1) ||
                    (( n > 0 )&&(p[0]!=tclDSMDABEPGApp_Tag_epg_0x02)&&(p[0]!=tclDSMDABEPGApp_Tag_serviceInformation_0x03)))
            {
                /* decompress ...
                 */
#ifndef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS
                int ret;

                tU32 p_unzip_size = 1024;
                p_unzip = (tU8*)malloc(p_unzip_size);
                tU32 p_unzip_used = 0;

                // inflate
                // zlib struct
                z_stream infstream;
                infstream.zalloc = Z_NULL;
                infstream.zfree = Z_NULL;
                infstream.opaque = Z_NULL;
                infstream.avail_in = (uInt)n; // size of input
                infstream.next_in = (Bytef *)p; // input char array

                inflateInit2(&infstream,16+MAX_WBITS);
                do
                {
                    infstream.next_out = (Bytef *)&p_unzip[p_unzip_used]; // output char array
                    infstream.avail_out = (uInt)1024; // size of output
                    ret = inflate(&infstream, Z_NO_FLUSH);

                    p_unzip_used += 1024-infstream.avail_out;
                    p_unzip_size += 1024;
                    p_unzip = (tU8*)realloc(p_unzip,p_unzip_size);
                } while( (ret != Z_STREAM_END) && (ret > 0) );
                inflateEnd(&infstream);

                if( ret >= 0 )
                {
                    compressiontype = 0;

                    p = p_unzip;
                    n = p_unzip_used;
                }
                else
                {
                    compressiontype = -1;
                }
#else /* CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS */
                compressiontype = -1;
#ifdef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE_AND_CLEAR_SEGM__COMPRESSEDOBJECTS
                vClearDGSegemt(a,4,u16_transport_id);
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_IGNORE_AND_CLEAR_SEGM__COMPRESSEDOBJECTS */
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS */
            }

            if((compressiontype==0)&&
                    ( n > 0 )&&
                    ((p[0]==tclDSMDABEPGApp_Tag_epg_0x02)||(p[0]==tclDSMDABEPGApp_Tag_serviceInformation_0x03)))
            {
                tclDSMDABEPGApp::tDecodeEPGObjectStatus status;
                memset(&status,0,sizeof(status));
                status.a = a;
                status.u16_transport_id = u16_transport_id;
                status.object_body_bytes = n;
                status.object_body = (tU8*)p;

                if( bTryDecodeEPGObject(&status) == TRUE )
                {
#ifdef ENABLE_CLEAR_EPG_DG_SEGM_WHEN_DECODED
                    vClearDGSegemt(a,4,u16_transport_id);
#endif /* ENABLE_CLEAR_EPG_DG_SEGM_WHEN_DECODED */
                }
            }
#ifndef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS
            if( p_unzip != NULL )
            {
                free(p_unzip);
            }
#endif
        }
        _dsm->oDB()->dbReset(pStmt);
        DSM_PERF_EXIT("tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary");
        return TRUE;
    }
    DSM_PERF_EXIT("tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary");
    return TRUE;
}
#endif /* CONFIG_DSM_ENABLE_bTryDecodeEPGObjectIfNessesary_OPTIMIZIATION */

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary( tU32* p32EPGByteLimit, tS64 a, tU16 u16_transport_id )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    DSM_PERF_ENTRY();

    int compressiontype = u32EPGObjectCompressionType(a,u16_transport_id);

#ifdef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS
    if(compressiontype==1)
    {
        /*
         * this file is compressed, ignore it
         */
#ifdef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE_AND_CLEAR_SEGM__COMPRESSEDOBJECTS
        vClearDGSegemt(a,4,u16_transport_id);
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_IGNORE_AND_CLEAR_SEGM__COMPRESSEDOBJECTS */
        DSM_PERF_EXIT("tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary");
        return TRUE;
    }
#endif

    rc = _dsm->oDB()->dbPrepare(
                "SELECT "
                "O.object "
                "FROM v_dab_mot_object AS O WHERE O.a=?1 AND O.transpid=?2 AND O.dgtype=4 AND O.transpid NOT IN (SELECT DISTINCT transpid FROM dab_epg_root WHERE a=O.a AND transpid=O.transpid) LIMIT 1",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        a);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      2,
                                      u16_transport_id);

        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    tU32 n = tclDB::dbColumnBytes( pStmt, 0 );

#ifdef CONFIG_DSM__EPG_ENABLE_BURST_BYTE_LIMIT
                    if( n != 0 )
                    {
                        if( p32EPGByteLimit != NULL )
                        {
                            if( *p32EPGByteLimit < n )
                            {
                                DSM_PERF_EXIT("tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary");
                                _dsm->oDB()->dbReset(pStmt);
                                return FALSE;
                            }
                            *p32EPGByteLimit -= n;
                        }
                    }
#else
                    (void)p32EPGByteLimit;
#endif

                    if( n != 0 )
                    {
#ifndef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS
                        tU8* p_unzip = NULL;
#endif
                        tU8* p = (tU8*)tclDB::dbColumnBlob( pStmt, 0 );

                        if((compressiontype==1) ||
                                (( n > 0 )&&(p[0]!=tclDSMDABEPGApp_Tag_epg_0x02)&&(p[0]!=tclDSMDABEPGApp_Tag_serviceInformation_0x03)))
                        {
                            /* decompress ...
                             */
#ifndef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS
                            int ret;

                            tU32 p_unzip_size = 1024;
                            p_unzip = (tU8*)malloc(p_unzip_size);
                            tU32 p_unzip_used = 0;

                            // inflate
                            // zlib struct
                            z_stream infstream;
                            infstream.zalloc = Z_NULL;
                            infstream.zfree = Z_NULL;
                            infstream.opaque = Z_NULL;
                            infstream.avail_in = (uInt)n; // size of input
                            infstream.next_in = (Bytef *)p; // input char array

                            inflateInit2(&infstream,16+MAX_WBITS);
                            do
                            {
                                infstream.next_out = (Bytef *)&p_unzip[p_unzip_used]; // output char array
                                infstream.avail_out = (uInt)1024; // size of output
                                ret = inflate(&infstream, Z_NO_FLUSH);

                                p_unzip_used += 1024-infstream.avail_out;
                                p_unzip_size += 1024;
                                p_unzip = (tU8*)realloc(p_unzip,p_unzip_size);
                            } while( (ret != Z_STREAM_END) && (ret > 0) );
                            inflateEnd(&infstream);

                            if( ret >= 0 )
                            {
                                compressiontype = 0;

                                p = p_unzip;
                                n = p_unzip_used;
                            }
                            else
                            {
                                compressiontype = -1;
                            }
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS */
                        }

                        if((compressiontype==0)&&
                                ( n > 0 )&&
                                ((p[0]==tclDSMDABEPGApp_Tag_epg_0x02)||(p[0]==tclDSMDABEPGApp_Tag_serviceInformation_0x03)))
                        {
                            tclDSMDABEPGApp::tDecodeEPGObjectStatus status;
                            memset(&status,0,sizeof(status));
                            status.a = a;
                            status.u16_transport_id = u16_transport_id;
                            status.object_body_bytes = n;
                            status.object_body = (tU8*)p;

                            if( bTryDecodeEPGObject(&status) == TRUE )
                            {
#ifdef ENABLE_CLEAR_EPG_DG_SEGM_WHEN_DECODED
                                vClearDGSegemt(a,4,u16_transport_id);
#endif /* ENABLE_CLEAR_EPG_DG_SEGM_WHEN_DECODED */
                            }
                        }
#ifndef CONFIG_DSM__ENABLE_DAB_EPG_IGNORE__COMPRESSEDOBJECTS
                        if( p_unzip != NULL )
                        {
                            free(p_unzip);
                        }
#endif
                    }
                    _dsm->oDB()->dbReset(pStmt);
                    DSM_PERF_EXIT("tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary");
                    return TRUE;
                }
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    DSM_PERF_EXIT("tclDSMDABEPGApp::bTryDecodeEPGObjectIfNessesary");
    return TRUE;
}
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */

/******************************************************************************
 * EOF
 *****************************************************************************/

