/******************************************************************************
 *
 * FILE: tclDSM.h
 * PROJECT: AaRS
 * SW-COMPONENT: Data Service Manager
 *
 * DESCRIPTION: This file contains the DSM class interface specification
 *
 * AUTHOR: CM-AI/ECB2-Scholz
 *
 * COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
 * HISTORY:
 * Date | Rev. | Author | Modification
 * ----------------------------------------------------------------------------
 * 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
 *
 *****************************************************************************/
#include <stdarg.h>  // for va_start, etc
#include <memory>    // for std::unique_ptr
#include <stdio.h>
#include <stdexcept>

#include "tclDSM.h"

#include "zlib.h"
#include "sqlite3.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_DSM
#include "trcGenProj/Header/tclDSM.cpp.trc.h"
#endif
using namespace dsm;

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSM::tclDSM(tclDSMTimeProvider* dsmTimeProvider,tclDSMEventListener* tclEventListener)
{
    _dsmeventlistener.clear();
    _event_queue.clear();

    _dsmtimeprovider = dsmTimeProvider;
    vAddDSMEventListener(tclEventListener);

    if( (sizeof(tU32) != 4)||(sizeof(tS32) != 4) )
    {
        throw std::length_error( "sizeof(tU32/tS32) must be 4 bytes, check type definition" );
    }

    if( (sizeof(tU64) != 8)||(sizeof(tS64) != 8) )
    {
        throw std::length_error( "sizeof(tU64/tS64) must be 8 bytes, check type definition" );
    }

    ETG_TRACE_USR1(("%p.tclDSM(%p,%p)",(void*)this,dsmTimeProvider,tclEventListener));
    _db = new tclDB(this);
#ifdef CONFIG_DSM__ENABLE_DAB
    _dab = new tclDABServiceManager(this);
#endif /* CONFIG_DSM__ENABLE_DAB */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSM::~tclDSM()
{
#ifdef CONFIG_DSM__ENABLE_DAB
    if( _dab != NULL )
    {
        delete _dab;
        _dab = NULL;
    }
#endif /* CONFIG_DSM__ENABLE_DAB */
    if(_db != NULL)
    {
        _db->vClose();
        delete _db;
        _db = NULL;
    }
}

#ifdef CONFIG_DSM__ENABLE_DAB
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDABServiceManager* tclDSM::oDAB()
{
    return _dab;
}
#endif /* CONFIG_DSM__ENABLE_DAB */

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSM::vAddDSMEventListener(tclDSMEventListener* tclEventListener)
{
    if( tclEventListener != NULL )
    {
        _dsmeventlistener.insert(_dsmeventlistener.end(),tclEventListener);

    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSM::vRemoveDSMEventListener(tclDSMEventListener* tclEventListener)
{
    if( tclEventListener != NULL )
    {
        _dsmeventlistener.remove(tclEventListener);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSM::vHandle()
{
    ETG_TRACE_USR1(("%p.vHandle() ...",(void*)this));
#ifdef CONFIG_DSM__ENABLE_DAB
    oDAB()->vHandle();
#endif /* CONFIG_DSM__ENABLE_DAB */

    ETG_TRACE_USR1(("%p.vHandle() ... vSendBroadCastEvents ...",(void*)this));

    vSendBroadCastEvents();

    ETG_TRACE_USR1(("%p.vHandle() ... done",(void*)this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSM::log(const char* src_str, const char* fmt_str, ...)
{    
#ifdef ENABLE_DSM_LOG_TABLE
    int size = 256;
    std::string str;
    va_list ap;
    while (1) {
        str.resize(size);
        va_start(ap, fmt_str);
        int n = vsnprintf((char *)str.c_str(), size, fmt_str, ap);
        va_end(ap);

        ETG_TRACE_USR1(("%p.tclDSM:log(%s)",(void*)this,str.c_str()));

        if (n > -1 && n < size) {
            str.resize(n);
            {
                sqlite3_stmt* pStmt = NULL;
                /* handling dsm_log GC is done by trigger
                 */
                oDB()->dbPrepare( "INSERT INTO dsm_log (timestamp,source,msg) VALUES(datetime('now'),?,?) ",
                                       -1,
                                       &pStmt,
                                       NULL);
                if( pStmt != NULL )
                {
                    oDB()->tclDB::dbBindText(pStmt,
                                             1,
                                             src_str,-1,0);
                    oDB()->tclDB::dbBindText(pStmt,
                                             2,
                                             str.c_str(),-1,0);
                    oDB()->dbStep(pStmt);
                    oDB()->dbReset(pStmt);
                }
            }
            return;
        }
        if (n > -1)
            size = n + 1;
        else
            size *= 2;
    }
    return;
#else /* ENABLE_DSM_LOG_TABLE */
    (void)src_str;
    (void)fmt_str;
#endif /* ENABLE_DSM_LOG_TABLE */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU64 tclDSM::u64ElapsedTimeMS()
{
    tU64 t = 0;
    if( _dsmtimeprovider != NULL )
    {
        t = _dsmtimeprovider->u64DSMGetUTCmS();
    }
    return t;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSM::vDeleteAppsWithFutureHostTimeStamp()
{
#ifdef CONFIG_DSM__ENABLE_DAB
    oDAB()->vDeleteAppsWithFutureHostTimeStamp();
#endif /* CONFIG_DSM__ENABLE_DAB */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU64 tclDSM::u64GetMaxAppHostTimestamp()
{
    tU64 u64_t = 0;
    tU64 u64_t_max = 0;
#ifdef CONFIG_DSM__ENABLE_DAB
    u64_t = oDAB()->u64GetMaxAppHostTimestamp();
    if( u64_t > u64_t_max ) u64_t_max = u64_t;
#endif /* CONFIG_DSM__ENABLE_DAB */
    return u64_t_max;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSM::vBroadCastEvent( tDSMEvent ev, std::string uri )
{
    tDSMEventObj e;
    e.ev = ev;
    e.uri = uri;

    ETG_TRACE_USR1(("%p.vBroadCastEvent(ev=%d,uri=%s)",(void*)this,ev,uri.c_str()));

    try {
        _event_queue.insert(_event_queue.end(),e);
    } catch( std::bad_alloc ex ) {
        (void)ex;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSM::vSendBroadCastEvents( )
{
    while( _event_queue.size() > 0 )
    {
        tDSMEventObj e = *_event_queue.begin();
        _event_queue.erase(_event_queue.begin());

        for (std::list<tclDSMEventListener*>::iterator it = _dsmeventlistener.begin(); it!=_dsmeventlistener.end(); ++it)
        {
            tclDSMEventListener* l = *it;
            tDSMEvent ev = e.ev;
            std::string uri = e.uri;
            l->vDSMEvent(ev,uri);
        }
    }
}



/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDB* tclDSM::oDB()
{
    return _db;
}

/******************************************************************************
 * EOF
 *****************************************************************************/

