/******************************************************************************
 *
 * FILE: tclDSMApp.cpp
 * PROJECT: AaRS
 * SW-COMPONENT: Data Service Manager
 *
 * DESCRIPTION: This file contains the DSM class interface specification
 *
 * AUTHOR: CM-AI/ECB2-Scholz
 *
 * COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
 * HISTORY:
 * Date | Rev. | Author | Modification
 * ----------------------------------------------------------------------------
 * 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
 *
 *****************************************************************************/
#include <stdarg.h>  // for va_start, etc
#include <memory>    // for std::unique_ptr
#include <math.h>
#include "tclDSM.h"

using namespace dsm;
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_DSM_DAB_SERVICE_MANAGER
#include "trcGenProj/Header/tclDABServiceManager.cpp.trc.h"
#endif

#define tclDABServiceManager_MIN(_a_,_b_) ((_a_)<(_b_)?(_a_):(_b_))

#if !defined(DSM_PERF_ENTRY) || !defined(DSM_PERF_EXIT)
/* create dummy PERF macros, when not defined
 */
#define DSM_PERF_ENTRY()
#define DSM_PERF_EXIT(_name_)
#endif

using namespace dsm;

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclServiceManager::tclServiceManager(class tclDSM* dsm)
{
    ETG_TRACE_USR1(("%p.tclServiceManager(dsm=%p)",(void*)this,dsm));
    _dsm = dsm;
}

/******************************************************************************
 ******************************************************************************
 * class tclDABServiceManager ...
 ******************************************************************************
 *****************************************************************************/

#ifdef CONFIG_DSM__ENABLE_DAB
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDABServiceManager::tclDABServiceManager(class tclDSM* dsm) :
    tclServiceManager(dsm),
    _sls_handler(this)
{
    ETG_TRACE_USR1(("%p.tclServiceManager(dsm=%p)",(void*)this,dsm));
    _dsm = dsm;
    _number_tuner = 1;
    _u64Timeout1000mS = 0;
    _u64Timeout60000mS = 0;

    _u64TimeStamp = 0;
    //_u64TimeOffset = 0;

    _active_apps.clear();
    _cmd_receiver = NULL;
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    _wizard = new tclDSMDABWizard(dsm);
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */

#ifdef CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT
    _u64HandleDatachannelFramePrevTimestamp = _dsm->u64ElapsedTimeMS();
    _u32HandleDatachannelSkiped = 0;
    _dHandleDatachannelPeriod = 0;
    _dHandleDatachannelProcessingTime = 0;
#endif
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDABServiceManager::~tclDABServiceManager()
{
    ETG_TRACE_USR1(("%p.~tclServiceManager()",(void*)this));
    vCloseDB();
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    delete _wizard;
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU64 tclDABServiceManager::u64GetMaxAppHostTimestamp()
{
    tU64 u64_t = 0;
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return 0;
    }

    rc = _dsm->oDB()->dbPrepare(
                "SELECT max(host_timestamp) FROM dab_data_application",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    u64_t = tclDB::dbColumnInt64(pStmt,0);
                }
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    return u64_t;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vSendDSMMeCaCommand( tU8 u8TunerIdx, AaRSLib::MSG::tCommandMessage msg)
{
    if( _cmd_receiver != NULL )
    {
        ETG_TRACE_USR1(("%p.vSendDSMMeCaCommand(%d,%s) ...",(void*)this,
                        u8TunerIdx,msg.toString().c_str()));
        _cmd_receiver->vSendDSMMeCaCommand(u8TunerIdx,msg);
        ETG_TRACE_USR1(("%p.vSendDSMMeCaCommand() ... done",(void*)this));
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vCloseDB()
{
    ETG_TRACE_USR1(("%p.vCloseDB() ...",(void*)this));

#ifdef CONFIG_DSM__KEEP_ONLY_SLS_FROM_CURRENT_SERVICE

    /* remove all SLS application if needed*/
    sqlite3_stmt* pStmt = NULL;
    SQLITE_API int rc;

    rc = _dsm->oDB()->dbPrepare(
                "DELETE FROM dab_data_application WHERE w_user_app_type=2",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
#endif

    while( 1 )
    {
        std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
        if( list_iter == _active_apps.end() )
        {
            break;
        }
        else
        {
            tclDSMDABApp* app = *list_iter;
            if( app != NULL )
            {
                _active_apps.remove(app);
                delete app;
            }
        }
    }
    ETG_TRACE_USR1(("%p.vCloseDB() ... done",(void*)this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vDestructApp( tS64 a )
{
    ETG_TRACE_USR1(("%p.vDestructApp(%d) ...",(void*)this,a));
    for(std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
        list_iter != _active_apps.end(); list_iter++)
    {
        tclDSMDABApp* app = *list_iter;
        if( app != NULL )
        {
            if( app->s64GetA() == a )
            {
                _active_apps.remove(app);
                delete app;
                return;
            }
        }
    }
    ETG_TRACE_USR1(("%p.vDestructApp(%d) ... done",(void*)this,a));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vDestructAllApps( )
{
    ETG_TRACE_USR1(("%p.vDestructAllApps() ...",(void*)this));
    for(std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
        list_iter != _active_apps.end(); list_iter++)
    {
        tclDSMDABApp* app = *list_iter;
        if( app != NULL )
        {
            _active_apps.remove(app);
            delete app;
        }
    }
    ETG_TRACE_USR1(("%p.vDestructAllApps() ... done",(void*)this));
}


/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vDeleteAll()
{
    SQLITE_API int rc;

    ETG_TRACE_USR1(("%p.vDeleteAll() ...",(void*)this));

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    rc = _dsm->oDB()->dbExec(
                "DELETE FROM dab_data_application",
                NULL,NULL,NULL);

    while( 1 )
    {
        std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
        if( list_iter == _active_apps.end() )
        {
            break;
        }
        else
        {
            tclDSMDABApp* app = *list_iter;
            if( app != NULL )
            {
                _active_apps.remove(app);
                delete app;
            }
        }
    }
    ETG_TRACE_USR1(("%p.vDeleteAll() ... done",(void*)this));
    (void)rc;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vDeleteAppsWithFutureHostTimeStamp()
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    rc = _dsm->oDB()->dbPrepare(
                "DELETE FROM dab_data_application WHERE host_timestamp>?",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        tS64 t_now = _dsm->u64ElapsedTimeMS()/1000;

        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        t_now);
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vHandleMsgR_DATACHANNEL_FRAME(tU8 u8TunerIdx,AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg)
{
    if (msg.l_body_size() > 0)
    {
        /*so this must be a mot DG*/
        if( ( msg.b_api_source() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_api_source__SOURCE_API_DDM )&&
                ( msg.b_data_source() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_source__SOURCE_TYPE_SERV_COMP)&&
                ((( msg.b_data_type() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_type__DATA_TYPE_DATA_GROUPS )&&
                  ( msg.l_header_size() == 2 ))||
                 (( msg.b_data_type() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_type__DATA_TYPE_PAD_DATA_GROUPS )&&
                  ( msg.l_header_size() == 4 )))&&
                ( msg.l_body_size() >= 007 )) /*check the segment flag and user access flag presence*/
        {
            tU16 u16_scidi = (tU16)((msg.r_header()[0]<<8)|(msg.r_header()[1]));
            tU16 u16_xpad_app_type = 0;

            if ( msg.b_data_type() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_type__DATA_TYPE_PAD_DATA_GROUPS )
            {
                u16_xpad_app_type = (tU16)((msg.r_header()[2]<<8)|(msg.r_header()[3]));
            }

            for(std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
                list_iter != _active_apps.end(); list_iter++)
            {
                tclDSMDABApp* app = *list_iter;
                if( app != NULL )
                {
                    if( msg.b_data_type() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_type__DATA_TYPE_DATA_GROUPS )
                    {
                        if((app->u8GetTunerIdx() == u8TunerIdx)&&
                                (app->u16tGetSCIDI() == u16_scidi))
                        {
                            app->vHandleMsgR_DATACHANNEL_FRAME(msg);
                        }
                    }
                    else
                    {
                        if((app->u8GetTunerIdx() == u8TunerIdx)&&
                                (app->u16tGetSCIDI() == u16_scidi)&&
                                (app->u16tGetXPADAppType() == u16_xpad_app_type))
                        {
                            app->vHandleMsgR_DATACHANNEL_FRAME(msg);
                        }
                    }
                }
            }
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vHandleMsgR_SELECT_URI(tU8 u8TunerIdx,AaRSDABLib::API::DDM::tR_SELECT_URI msg)
{
    for(std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
        list_iter != _active_apps.end(); list_iter++)
    {
        tclDSMDABApp* app = *list_iter;
        if( app != NULL )
        {
            if((app->u8GetTunerIdx() == u8TunerIdx)&&
                    (app->tGetURI() == msg.r_uri()))
            {
                if( ( msg.w_response() == AaRSDABLib::API::DDM::tR_SELECT_URI::w_response__URI_DEACTIVATED_BY_SYNC_LOSS )||
                        ( msg.w_response() == AaRSDABLib::API::DDM::tR_SELECT_URI::w_response__URI_DEACTIVATED_BY_SCHED ) )
                {
                    app->vHandleMsgR_SELECT_URI(msg);
                    _active_apps.remove(app);
                    delete app;
                    return;
                }
                else
                {
                    app->vHandleMsgR_SELECT_URI(msg);
                }
                return;
            }
        }
    }
    if( ( msg.w_response() == AaRSDABLib::API::DDM::tR_SELECT_URI::w_response__URI_ACTIVATED ) )
    {
        tclDSMDABApp* app = NULL;
        tU16  w_user_app_type = msg.w_ua_type();
        switch( w_user_app_type )
        {
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
        case DAB_SERVICE_MANAGER_UAPPTYPE_EPG:
            app = new tclDSMDABEPGApp(_dsm,u8TunerIdx,msg);
            break;
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
#ifdef CONFIG_DSM__ENABLE_DAB_SLS
        case DAB_SERVICE_MANAGER_UAPPTYPE_SLS:
            app = new tclDSMDABSLSApp(_dsm,u8TunerIdx,msg);
            break;
#endif
#ifdef CONFIG_DSM__ENABLE_DAB_BWS
        case DAB_SERVICE_MANAGER_UAPPTYPE_BWS:
            app = new tclDSMDABBWSApp(_dsm,u8TunerIdx,msg);
            break;
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
#ifdef CONFIG_DSM__ENABLE_DAB_JOURNALINE
        case DAB_SERVICE_MANAGER_UAPPTYPE_JRNL:
            app = new tclDSMDABJRLNApp(_dsm,u8TunerIdx,msg);
            break;
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
        }
        if( app != NULL )
        {
            _active_apps.insert(_active_apps.begin(),app);
        }
    }

}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vHandleMsgR_TIMESTAMP(tU8 u8TunerIdx,AaRSDABLib::API::DDM::tR_TIMESTAMP msg)
{
    for(std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
        list_iter != _active_apps.end(); list_iter++)
    {
        tclDSMDABApp* app = *list_iter;
        if( app != NULL )
        {
            if((app->u8GetTunerIdx() == u8TunerIdx)&&
                    (app->u16tGetSCIDI() == msg.w_scidi()))
            {
                app->vHandleMsgR_TIMESTAMP(msg);
            }
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vHandleMeCaResponse( tU8 u8TunerIdx, AaRSLib::MSG::tResponseMessage msg)
{
    /*
   * check if tuner index is in range
   */
    if ( u8TunerIdx >= _number_tuner )
    {
        return;
    }

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    DSM_PERF_ENTRY();

    ETG_TRACE_USR1(("%p.vHandleMeCaResponse(%d,%s) ... ",
                    (void*)this,
                    u8TunerIdx,
                    msg.toString().c_str()));

    /*
   * dispatch response ...
   */
    switch( msg.opcode() )
    {
    case AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::OP_CODE:
    {
#ifdef CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT
        tU64 u64NowA = _dsm->u64ElapsedTimeMS();
        tBool bSkipHandleMsgR_DATACHANNEL_FRAME = FALSE;
        tDouble dLoad = 0.0;
        if( _u64HandleDatachannelFramePrevTimestamp != 0 )
        {
            if( _dHandleDatachannelProcessingTime != 0 )
            {
                dLoad = (100.0 * _dHandleDatachannelProcessingTime / _dHandleDatachannelPeriod);
                //printf("%f %f %f\n",dLoad,_dHandleDatachannelProcessingTime,_dHandleDatachannelPeriod);
                if( dLoad > CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT )
                {
                    bSkipHandleMsgR_DATACHANNEL_FRAME = TRUE;
                }
            }
        }
        if( bSkipHandleMsgR_DATACHANNEL_FRAME == FALSE )
        {
            vHandleMsgR_DATACHANNEL_FRAME(u8TunerIdx,msg);
        }
        else
        {
            _u32HandleDatachannelSkiped++;
#if 0
            printf("vHandleMsgR_DATACHANNEL_FRAME skipped because high processing load %2.1f %2.1f %2.1f %d\n",
                   dLoad,
                   _dHandleDatachannelProcessingTime,
                   _dHandleDatachannelPeriod,
                   _u32HandleDatachannelSkiped);
#endif
            _dsm->log("tclDABServiceManager",
                      "vHandleMsgR_DATACHANNEL_FRAME skipped because high processing load %2.1f %2.1f %2.1f %d",
                      dLoad,
                      _dHandleDatachannelProcessingTime,
                      _dHandleDatachannelPeriod,
                      _u32HandleDatachannelSkiped);

        }
        if( _u64HandleDatachannelFramePrevTimestamp != 0 )
        {
            if( (u64NowA - CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT_MAX_DC_PERIOD) > _u64HandleDatachannelFramePrevTimestamp )
            {
                /* When the previous message was decoded before more then 3 seconds,
               * then assume that it was handled 3 seconds ago.
               */
                _u64HandleDatachannelFramePrevTimestamp = u64NowA - CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT_MAX_DC_PERIOD;
            }

            if( u64NowA > _u64HandleDatachannelFramePrevTimestamp )
            {
                double x = CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT_LPF_CONSTANT;
                tU64 u64NowB = _dsm->u64ElapsedTimeMS();
                tU64 u64DeltaA = (u64NowA - _u64HandleDatachannelFramePrevTimestamp);
                tU64 u64DeltaB = (u64NowB - u64NowA);
                _dHandleDatachannelPeriod = ( x*u64DeltaA + ( (1.0-x) * _dHandleDatachannelPeriod ));
                _dHandleDatachannelProcessingTime = ( x*u64DeltaB + ( (1.0-x) * _dHandleDatachannelProcessingTime ));
                //printf("%f %f\n",_dHandleDatachannelProcessingTime,_dHandleDatachannelPeriod);
            }
        }
        _u64HandleDatachannelFramePrevTimestamp = u64NowA;
#else /* CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT */
        vHandleMsgR_DATACHANNEL_FRAME(u8TunerIdx,msg);
#endif /* CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT */
    }
        break;
    case AaRSDABLib::API::DDM::tR_TIMESTAMP::OP_CODE:
        vHandleMsgR_TIMESTAMP(u8TunerIdx,msg);
        break;
    case AaRSDABLib::API::DDM::tR_SELECT_URI::OP_CODE:
        vHandleMsgR_SELECT_URI(u8TunerIdx,msg);
        break;
    }

    _dsm->vSendBroadCastEvents();

#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    if( _wizard != NULL )
    {
        _wizard->vHandleMeCaResponse(u8TunerIdx,msg);
    }
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */

    /* call vHandle from SLS manager
   */
    _sls_handler.vHandleMeCaResponse(u8TunerIdx,msg);

    _dsm->vSendBroadCastEvents();

    DSM_PERF_EXIT("tclDABServiceManager::vHandleMeCaResponse");
    ETG_TRACE_USR1(("%p.vHandleMeCaResponse() ... done",
                    (void*)this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vInitializeTables()
{
    SQLITE_API int rc;

    ETG_TRACE_USR1(("%p.vInitializeTables() ...",(void*)this));

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    /* create table for global DSM parameters (e.g. current global timestamp)
   */
    rc = _dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dsm_param ("
                /***/ "key INTEGER PRIMARY KEY,"
                /***/ "value"
                /**/ ")",
                NULL,NULL,NULL);

#ifdef ENABLE_DSM_LOG_TABLE
    rc = _dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dsm_log ("
                /***/ "timestamp,"
                /***/ "source,"
                /***/ "msg"
                /**/ ")",
                NULL,NULL,NULL);
    rc = _dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "dsm_log_tr BEFORE INSERT ON dsm_log "
                /**/ "BEGIN "
                /**/ "DELETE FROM dsm_log WHERE rowid NOT IN (SELECT rowid FROM dsm_log ORDER BY timestamp DESC LIMIT 99); "
                /**/ "END",
                NULL,NULL,NULL);
#endif /* ENABLE_DSM_LOG_TABLE */

    tclDSMDABApp::vInitializeTables(_dsm);
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    tclDSMDABWizard::vInitializeTables(_dsm);
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */
#if defined(CONFIG_DSM__ENABLE_DAB_EPG)||defined(CONFIG_DSM__ENABLE_DAB_BWS)||defined(CONFIG_DSM__ENABLE_DAB_SLS)
    tclDSMDABMOTApp::vInitializeTables(_dsm);
#endif /* defined(CONFIG_DSM__ENABLE_DAB_EPG)||defined(CONFIG_DSM__ENABLE_DAB_BWS)||defined(CONFIG_DSM__ENABLE_DAB_SLS) */
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
    tclDSMDABEPGApp::vInitializeTables(_dsm);
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
#ifdef CONFIG_DSM__ENABLE_DAB_BWS
    tclDSMDABBWSApp::vInitializeTables(_dsm);
#endif /* CONFIG_DSM__ENABLE_DAB_BWS */
#ifdef CONFIG_DSM__ENABLE_DAB_SLS
    tclDSMDABSLSApp::vInitializeTables(_dsm);
#endif /* CONFIG_DSM__ENABLE_DAB_SLS */
#ifdef CONFIG_DSM__ENABLE_DAB_JOURNALINE
    tclDSMDABJRLNApp::vInitializeTables(_dsm);
#endif /* CONFIG_DSM__ENABLE_DAB_JOURNALINE */

    rc = _dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS tr_dab_mot_delete "
            #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                "AFTER DELETE ON dab_mot_object BEGIN "
            #else
                "AFTER DELETE ON dab_datagroup_mot BEGIN "
            #endif
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                /**/ "DELETE FROM dab_epg_schedule_scope WHERE a=old.a AND transpid=old.transpid; "
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                /**/ "DELETE FROM dab_epg_root WHERE a=old.a AND transpid=old.transpid; "
                /**/ "DELETE FROM dab_mot_directory WHERE a=old.a AND transpid=old.transpid; "
                /**/ "DELETE FROM dab_mot_directory_param WHERE a=old.a AND transpid=old.transpid; "
                /**/ "DELETE FROM dab_mot_directory_cname WHERE a=old.a AND transpid=old.transpid; "
                     "END",
                NULL,NULL,NULL);
    (void)rc;

    ETG_TRACE_USR1(("%p.vInitializeTables() ... done",(void*)this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vTriggerAppCompleteEvent( )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    ETG_TRACE_USR1(("%p.vTriggerAppCompleteEvent() ...",(void*)this));

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    /* MOT Directory Compressed or MOT Directory Uncompressed */
    rc = _dsm->oDB()->dbPrepare(
                "SELECT uri FROM v_dab_mot_application AS A WHERE A.complete=1",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    const char* uri = (const char*)tclDB::dbColumnText(pStmt,0);
                    if( uri != NULL )
                    {
                        _dsm->vBroadCastEvent( DSM_EV_APPLICATION_COMPLETE, uri );
                        _dsm->log("bTryDecodeMOTDirectory","DSM_EV_APPLICATION_COMPLETE %s",uri);
                    }
                }
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    ETG_TRACE_USR1(("%p.vTriggerAppCompleteEvent() ... done",(void*)this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bAppComplete( const char* uri )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( uri != NULL )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        /* MOT Directory Compressed or MOT Directory Uncompressed */
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT 1 FROM v_dab_mot_application AS A WHERE uri=? AND A.complete=1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindText(pStmt,
                                           1,
                                           uri,
                                           -1,
                                           NULL);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        if( tclDB::dbColumnInt(pStmt,0) == 1 )
                        {
                            _dsm->oDB()->dbReset(pStmt);
                            return TRUE;
                        }
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU16 tclDABServiceManager::u16GetUserAppType( const char* uri )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return DAB_SERVICE_MANAGER_UAPPTYPE_UNDEF;
    }

    if( uri != NULL )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT w_user_app_type FROM dab_data_application WHERE uri=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindText(pStmt,
                                           1,
                                           uri,
                                           (int)strlen(uri),
                                           NULL);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        tU16 w_user_app_type = (tU16)tclDB::dbColumnInt( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        return w_user_app_type;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return DAB_SERVICE_MANAGER_UAPPTYPE_UNDEF;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return DAB_SERVICE_MANAGER_UAPPTYPE_UNDEF;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU16 tclDABServiceManager::u16GetUserAppType( tS64 s64AppId )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return DAB_SERVICE_MANAGER_UAPPTYPE_UNDEF;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT w_user_app_type FROM dab_data_application WHERE o=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        tU16 w_user_app_type = (tU16)tclDB::dbColumnInt( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        return w_user_app_type;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return DAB_SERVICE_MANAGER_UAPPTYPE_UNDEF;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return DAB_SERVICE_MANAGER_UAPPTYPE_UNDEF;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tS64 tclDABServiceManager::s64GetAppId( const char* uri )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return DSM_INVALID_APPID;
    }

    if( uri != NULL )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;

        rc = _dsm->oDB()->dbPrepare(
                    "SELECT o FROM dab_data_application WHERE uri=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindText(pStmt,
                                           1,
                                           uri,
                                           (int)strlen(uri),
                                           NULL);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        tS64 a = tclDB::dbColumnInt64( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        return a;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return DSM_INVALID_APPID;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return DSM_INVALID_APPID;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU32 tclDABServiceManager::u32getNumApps()
{
    tU32 u32NumApps = 0;
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return 0;
    }

    rc = _dsm->oDB()->dbPrepare(
                "SELECT count() FROM dab_data_application",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    u32NumApps = (tU32)tclDB::dbColumnInt( pStmt, 0 );
                }
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                _dsm->oDB()->dbReset(pStmt);
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    return u32NumApps;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU32  tclDABServiceManager::u32GetNumMOTObjects( tS64 s64AppId )
{
    tU32 u32NumMOTObjects = 0;
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return 0;
    }

    rc = _dsm->oDB()->dbPrepare(
                "SELECT count() FROM v_dab_mot_object WHERE a=?",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        s64AppId);

        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    u32NumMOTObjects = (tU32)tclDB::dbColumnInt( pStmt, 0 );
                }
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                _dsm->oDB()->dbReset(pStmt);
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    return u32NumMOTObjects;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bIsAppComplete( tS64 s64AppId )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT complete FROM v_dab_mot_application WHERE o=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        tS32 complete = tclDB::dbColumnInt( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        return (complete==0)?FALSE:TRUE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetRecentTranspId( tS64 s64AppId, tU16& u16TransportId, tS64 s64AppTimeOffset )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }
    if(s64AppId != DSM_INVALID_APPID)
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        tS64 s64CurrentTime = (_dsm->u64ElapsedTimeMS()/1000);
        tS64 s64CurrentAppTime = s64CurrentTime-s64AppTimeOffset;

        rc = _dsm->oDB()->dbPrepare(
            #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                    "SELECT MOC.transpid FROM dab_mot_directory_param AS P, dab_mot_object AS MOC "
                    "WHERE P.a=?1 AND MOC.a=?1 AND P.transpid=MOC.transpid AND MOC.dgtype=4 AND MOC.complete=1 AND "
                    "((P.paramid=5 AND P.datafield=0) OR "
                    "(P.paramid=5 AND P.datafield!=0 AND ((?2-P.datafield)>0))) ORDER BY (?2-P.datafield) ASC LIMIT 1",
            #else
                    "SELECT MOC.transpid FROM dab_mot_directory_param AS P, dab_mot_object_complete AS MOC "
                    "WHERE P.a=?1 AND MOC.a=?1 AND P.transpid=MOC.transpid AND MOC.dgtype=4 AND "
                    "((P.paramid=5 AND P.datafield=0) OR "
                    "(P.paramid=5 AND P.datafield!=0 AND ((?2-P.datafield)>0))) ORDER BY (?2-P.datafield) ASC LIMIT 1",
            #endif
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            2,
                                            s64CurrentAppTime);


            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        u16TransportId = (tU16)tclDB::dbColumnInt( pStmt, 0 );

                        ETG_TRACE_USR1(("%p.bGetRecentTranspId() u16TransportId=0x%08x",(void*)this,u16TransportId));

                        _dsm->oDB()->dbReset(pStmt);
                        return TRUE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFileTransPortId( tS64 s64AppId, const char* pFileName, tU16& u16TransportId )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if((s64AppId != DSM_INVALID_APPID )&&(pFileName != NULL))
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT transpid FROM v_dab_mot_file WHERE a=? AND name=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            _dsm->oDB()->tclDB::dbBindText(pStmt,
                                           2,
                                           pFileName,
                                           (int)strlen(pFileName),
                                           NULL);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        u16TransportId = (tU16)tclDB::dbColumnInt( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        return TRUE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFileParam( tS64 s64AppId, tU16 u16TransportId, tS32 u32ParamId, tU32& rValue )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT datafield FROM dab_mot_directory_param WHERE a=? AND transpid=? AND paramid=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          2,
                                          u16TransportId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          3,
                                          u32ParamId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        rValue = (tU32)tclDB::dbColumnInt64( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        return TRUE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFileParam( tS64 s64AppId, tU16 u16TransportId, tS32 u32ParamId, std::string& rValue )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT datafield FROM dab_mot_directory_param WHERE a=? AND transpid=? AND paramid=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          2,
                                          u16TransportId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          3,
                                          u32ParamId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        const char* pValue = (const char*)tclDB::dbColumnText( pStmt, 0 );
                        if( pValue != NULL )
                        {
                            rValue = std::string(pValue);
                            _dsm->oDB()->dbReset(pStmt);
                            return TRUE;
                        }
                        _dsm->oDB()->dbReset(pStmt);
                        return FALSE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFileParam( tS64 s64AppId, tU16 u16TransportId, tS32 u32ParamId, std::vector<tU8>& rValue )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT datafield FROM dab_mot_directory_param WHERE a=? AND transpid=? AND paramid=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          2,
                                          u16TransportId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          3,
                                          u32ParamId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        const unsigned char* pBody = (const unsigned char*)tclDB::dbColumnBlob( pStmt, 0 );
                        if( pBody != NULL )
                        {
                            tU32 n = tclDB::dbColumnBytes(pStmt,0);
                            rValue.resize(n);
                            if( n != 0 )
                            {
                                memcpy( (void*)&rValue[0],
                                        (void*)tclDB::dbColumnBlob(pStmt,0),
                                        n);
                            }
                        }
                        _dsm->oDB()->dbReset(pStmt);
                        return TRUE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFileName( tS64 s64AppId, tU16 u16TransportId, std::string& rName )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT name FROM v_dab_mot_file WHERE a=? AND transpid=? LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          2,
                                          u16TransportId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        const char* pName = (const char*)tclDB::dbColumnText( pStmt, 0 );
                        _dsm->oDB()->dbReset(pStmt);
                        if( pName != NULL )
                        {
                            rName = std::string(pName);
                            return TRUE;
                        }
                        return FALSE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFileBody( tS64 s64AppId, tU16 u16TransportId, std::vector<tU8>& resFileBody )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    resFileBody.clear();
    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT body FROM v_dab_mot_file WHERE a=? AND transpid=? AND complete=1 LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          2,
                                          u16TransportId);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        const unsigned char* pBody = (const unsigned char*)tclDB::dbColumnBlob( pStmt, 0 );
                        if( pBody != NULL )
                        {
                            tU32 n = tclDB::dbColumnBytes(pStmt,0);
                            resFileBody.resize(n);
                            if( n != 0 )
                            {
                                memcpy( (void*)&resFileBody[0],
                                        (void*)tclDB::dbColumnBlob(pStmt,0),
                                        n);
                            }
                        }
                        _dsm->oDB()->dbReset(pStmt);
                        return TRUE;
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetBWSRootPageName( const char* uri, std::string& rName, tU8 u8BWSProfileId )
{
    tS64 s64AppId = _dsm->oDAB()->s64GetAppId(uri);
    if( s64AppId != DSM_INVALID_APPID )
    {
        return bGetBWSRootPageName(s64AppId,rName,u8BWSProfileId);
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetBWSRootPageName( tS64 s64AppId, std::string& rName, tU8 u8BWSProfileId )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
                    "SELECT P.datafield FROM dab_mot_directory_param AS P,dab_mot_directory AS D WHERE D.a=P.a AND P.a=? AND P.transpid=D.transpid AND paramid=34",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64AppId);

            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        const unsigned char* pBody = (const unsigned char*)tclDB::dbColumnBlob( pStmt, 0 );
                        if( pBody != NULL )
                        {
                            tU32 n = tclDB::dbColumnBytes(pStmt,0);
                            if( n > 1 )
                            {
                                if(( u8BWSProfileId == 0 )||( pBody[0] == u8BWSProfileId ))
                                {
                                    rName.resize(n-1);
                                    memcpy( (void*)&rName[0],
                                        (void*)&pBody[1],
                                        n-1);
                                    _dsm->oDB()->dbReset(pStmt);

                                    _dsm->oDAB()->vInformUsed(s64AppId);
                                    return TRUE;
                                }
                            }
                        }
                    }
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFile( const char* uri, const char* filename, std::vector<tU8>& resFileBody )
{
    std::string resMimeType = "";
    return bGetFile(uri,filename,resMimeType,resFileBody);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFile( const char* uri, const char* filename, std::string& resMimeType, std::vector<tU8>& resFileBody )
{
    tS64 s64AppId = _dsm->oDAB()->s64GetAppId(uri);
    if( s64AppId != DSM_INVALID_APPID )
    {
        tU16 u16TransportId = 0;
        if(  _dsm->oDAB()->bGetFileTransPortId( s64AppId, filename, u16TransportId ) )
        {
            std::string sMimeType = "";
            std::vector<tU8> tFileBody;
            if( _dsm->oDAB()->bGetFileParam(s64AppId,u16TransportId,16,sMimeType) )
            {
                resMimeType = sMimeType;
            }
            if( _dsm->oDAB()->bGetFileBody(s64AppId,u16TransportId,tFileBody) )
            {
                resFileBody = tFileBody;

                _dsm->oDAB()->vInformUsed(uri);
                return TRUE;
            }
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetFile( tS64 s64AppId, const char* filename, std::string& resMimeType, std::vector<tU8>& resFileBody )
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    if( s64AppId != DSM_INVALID_APPID )
    {
        tU16 u16TransportId = 0;
        if(  _dsm->oDAB()->bGetFileTransPortId( s64AppId, filename, u16TransportId ) )
        {
            std::string sMimeType = "";
            std::vector<tU8> tFileBody;
            if( _dsm->oDAB()->bGetFileParam(s64AppId,u16TransportId,16,sMimeType) )
            {
                resMimeType = sMimeType;
            }
            if( _dsm->oDAB()->bGetFileBody(s64AppId,u16TransportId,tFileBody) )
            {
                resFileBody = tFileBody;

                _dsm->oDAB()->vInformUsed(s64AppId);

                return TRUE;
            }
        }
    }
    return FALSE;
}

#if defined(CONFIG_DSM__ENABLE_DAB_EPG) && (defined(CONFIG_DSM__ENABLE_DAB_EPG_BASIC_SCHEDULE) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL))
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetPrevEPGScheduleInfos( sqlite3_stmt** ppStmt, tEPGContentIdDataType& ContentIdData, tEPGScheduleInfoType& Info, tU32 u32SpanSeconds )
{
    SQLITE_API int rc;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    DSM_PERF_ENTRY();

    if(ppStmt!=NULL)
    {
        if((*ppStmt)==NULL)
        {
            rc = _dsm->oDB()->dbPrepare(
                        "SELECT "
                        "A.dab_utc_timestamp+G.value-A.host_timestamp, "
                        "P1.startTime,"
                        "P1.duration,"
                        "P1.mediumName,"
                        "P1.longName,"
            #if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
                        "P1.shortDescription,"
            #endif
                        "A.o AS a "
                        "FROM "
                        "v_dab_epg_programme AS P1,"
                        "dsm_param AS G,"
                        "dab_data_application AS A "
                        "WHERE "
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                        "A.o=(SELECT R.a FROM dab_data_application AS A,dab_epg_schedule_scope AS R WHERE A.o=R.a AND dsmCheckContentId(R.contentID,?1,?2,?3,?4,?5)!=0 ORDER BY A.host_timestamp DESC LIMIT 1) AND "
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                        "A.o=(SELECT A.o FROM dab_data_application AS A,dab_epg_root AS R,dab_epg AS P1,dab_epg AS P2 WHERE A.o=R.a AND R.dab_epg_o=P1.o AND P1.o=P2.p_o AND R.root_tag=2 AND P2.tag=5 AND dsmCheckContentId(P2.value,?1,?2,?3,?4,?5)!=0 ORDER BY A.host_timestamp DESC LIMIT 1) AND "
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                        "G.key=1 AND A.o=P1.a AND "
                        "dsmCheckContentId(P1.contentId,?1,?2,?3,?4,?5)!=0 AND "
                        "P1.startTime>(A.dab_utc_timestamp+G.value-A.host_timestamp-?6) AND "
                        "P1.startTime<=(A.dab_utc_timestamp+G.value-A.host_timestamp) "
                        "ORDER BY P1.startTime DESC",
                        -1, ppStmt, NULL);
			(void)rc;

            sqlite3_stmt* pStmt = (*ppStmt);
            if( pStmt != NULL )
            {
                /* call dbReset for avoiding issue with busy reused statements */

                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                1,
                                                ContentIdData.r_ensemble);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                2,
                                                ContentIdData.b_service_type);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                3,
                                                ContentIdData.r_service);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                4,
                                                ContentIdData.b_scids);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                5,
                                                ContentIdData.b_xpad_app_type);
                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                              6,
                                              u32SpanSeconds);
            }
        }
        if((*ppStmt)!=NULL)
        {
            sqlite3_stmt* pStmt = (*ppStmt);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_ROW )
                {
#if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
                    if( tclDB::dbColumnCount(pStmt) == 7 )
#else
                    if( tclDB::dbColumnCount(pStmt) == 6 )
#endif
                    {
                        Info.u64CurrTime = tclDB::dbColumnInt64(pStmt,0);
                        Info.u64StartTime = tclDB::dbColumnInt64(pStmt,1);
                        Info.u16Duration = tclDB::dbColumnInt(pStmt,2);
                        if( tclDB::dbColumnBytes(pStmt,3) == 0 )
                        {
                            Info.sMediumName = "";
                        }
                        else
                        {
                            Info.sMediumName = (const char*)tclDB::dbColumnText(pStmt,3);
                        }
                        if( tclDB::dbColumnBytes(pStmt,4) == 0 )
                        {
                            Info.sLongName = "";
                        }
                        else
                        {
                            Info.sLongName = (const char*)tclDB::dbColumnText(pStmt,4);
                        }
#if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
                        if( tclDB::dbColumnBytes(pStmt,5) == 0 )
                        {
                            Info.sShortDescription = "";
                        }
                        else
                        {
                            Info.sShortDescription = (const char*)tclDB::dbColumnText(pStmt,5);
                        }
#endif
                        _dsm->oDAB()->vInformUsed(tclDB::dbColumnInt64(pStmt,6));

                        DSM_PERF_EXIT("tclDABServiceManager::bGetPrevEPGScheduleInfos");
                        return TRUE;
                    }
                    DSM_PERF_EXIT("tclDABServiceManager::bGetPrevEPGScheduleInfos");
                    return FALSE;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    DSM_PERF_EXIT("tclDABServiceManager::bGetPrevEPGScheduleInfos");
                    return FALSE;
                }
                else
                {
                    DSM_PERF_EXIT("tclDABServiceManager::bGetPrevEPGScheduleInfos");
                    return FALSE;
                }
            }
        }
    }
    DSM_PERF_EXIT("tclDABServiceManager::bGetPrevEPGScheduleInfos");
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetNextEPGScheduleInfos( sqlite3_stmt** ppStmt, tEPGContentIdDataType& ContentIdData, tEPGScheduleInfoType& Info, tU32 u32SpanSeconds )
{
    SQLITE_API int rc;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    DSM_PERF_ENTRY();

    if(ppStmt!=NULL)
    {
        if((*ppStmt)==NULL)
        {
            rc = _dsm->oDB()->dbPrepare(
                        "SELECT "
                        "A.dab_utc_timestamp+G.value-A.host_timestamp, "
                        "P1.startTime,"
                        "P1.duration,"
                        "P1.mediumName,"
                        "P1.longName"
            #if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
                        ",P1.shortDescription"
            #endif
                        " "
                        "FROM "
                        "v_dab_epg_programme AS P1,"
                        "dsm_param AS G,"
                        "dab_data_application AS A "
                        "WHERE "
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                        "A.o=(SELECT R.a FROM dab_data_application AS A,dab_epg_schedule_scope AS R WHERE A.o=R.a AND dsmCheckContentId(R.contentID,?1,?2,?3,?4,?5)!=0 ORDER BY A.host_timestamp DESC LIMIT 1) AND "
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                        "A.o=(SELECT A.o FROM dab_data_application AS A,dab_epg_root AS R,dab_epg AS P1,dab_epg AS P2 WHERE A.o=R.a AND R.dab_epg_o=P1.o AND P1.o=P2.p_o AND R.root_tag=2 AND P2.tag=5 AND dsmCheckContentId(P2.value,?1,?2,?3,?4,?5)!=0 ORDER BY A.host_timestamp DESC LIMIT 1) AND "
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                        "G.key=1 AND A.o=P1.a AND "
                        "dsmCheckContentId(P1.contentId,?1,?2,?3,?4,?5)!=0 AND "
                        "P1.startTime<(A.dab_utc_timestamp+G.value-A.host_timestamp+?6) AND "
                        "P1.startTime>(A.dab_utc_timestamp+G.value-A.host_timestamp) "
                        "ORDER BY P1.startTime ASC",
                        -1, ppStmt, NULL);
			(void)rc;

            sqlite3_stmt* pStmt = (*ppStmt);

            if( pStmt != NULL )
            {
                /* call dbReset for avoiding issue with busy reused statements */

                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                1,
                                                ContentIdData.r_ensemble);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                2,
                                                ContentIdData.b_service_type);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                3,
                                                ContentIdData.r_service);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                4,
                                                ContentIdData.b_scids);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                5,
                                                ContentIdData.b_xpad_app_type);
                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                6,
                                                u32SpanSeconds);
            }
        }
        if((*ppStmt)!=NULL)
        {
            sqlite3_stmt* pStmt = (*ppStmt);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_ROW )
                {
#if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
                    if( tclDB::dbColumnCount(pStmt) == 6 )
#else
                    if( tclDB::dbColumnCount(pStmt) == 5 )
#endif
                    {
                        Info.u64CurrTime = tclDB::dbColumnInt64(pStmt,0);
                        Info.u64StartTime = tclDB::dbColumnInt64(pStmt,1);
                        Info.u16Duration = tclDB::dbColumnInt(pStmt,2);
                        if( tclDB::dbColumnBytes(pStmt,3) == 0 )
                        {
                            Info.sMediumName = "";
                        }
                        else
                        {
                            Info.sMediumName = (const char*)tclDB::dbColumnText(pStmt,3);
                        }
                        if( tclDB::dbColumnBytes(pStmt,4) == 0 )
                        {
                            Info.sLongName = "";
                        }
                        else
                        {
                            Info.sLongName = (const char*)tclDB::dbColumnText(pStmt,4);
                        }
#if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
                        if( tclDB::dbColumnBytes(pStmt,5) == 0 )
                        {
                            Info.sShortDescription = "";
                        }
                        else
                        {
                            Info.sShortDescription = (const char*)tclDB::dbColumnText(pStmt,5);
                        }
#endif
                        DSM_PERF_EXIT("tclDABServiceManager::bGetNextEPGScheduleInfos");
                        return TRUE;
                    }
                    DSM_PERF_EXIT("tclDABServiceManager::bGetNextEPGScheduleInfos");
                    return FALSE;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    DSM_PERF_EXIT("tclDABServiceManager::bGetNextEPGScheduleInfos");
                    return FALSE;
                }
                else
                {
                    DSM_PERF_EXIT("tclDABServiceManager::bGetNextEPGScheduleInfos");
                    return FALSE;
                }
            }
        }
    }
    DSM_PERF_EXIT("tclDABServiceManager::bGetNextEPGScheduleInfos");
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bHasEPGScheduleInfo( tEPGContentIdDataType& ContentIdData )
{
    SQLITE_API int rc;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    sqlite3_stmt* pStmt = NULL;
    rc = _dsm->oDB()->dbPrepare(
                "SELECT "
                "1 "
                "FROM "
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                "dab_epg_schedule_scope AS S "
                "WHERE "
                "dsmCheckContentId(S.contentID,?1,?2,?3,?4,?5)!=0",
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                "dab_epg_root AS R,"
                "dab_epg AS P2 "
                "WHERE "
                "R.dab_epg_o=P2.p_o AND "
                "R.root_tag=2 AND "
                "P2.tag=5 AND "
                "dsmCheckContentId(P2.value,?1,?2,?3,?4,?5)!=0",
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                -1, &pStmt, NULL);

    if( pStmt != NULL )
    {
        /* call dbReset for avoiding issue with busy reused statements */

        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        ContentIdData.r_ensemble);
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        2,
                                        ContentIdData.b_service_type);
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        3,
                                        ContentIdData.r_service);
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        4,
                                        ContentIdData.b_scids);
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        5,
                                        ContentIdData.b_xpad_app_type);
    }

    rc = _dsm->oDB()->dbStep(pStmt);
    if( rc == SQLITE_ROW )
    {
        if( tclDB::dbColumnCount(pStmt) == 1 )
        {
            if( tclDB::dbColumnInt( pStmt, 0 ) == 1 )
            {
                _dsm->oDB()->dbReset(pStmt);
                return TRUE;
            }
        }
    }
    else if( tclDB::dbOnStepError(rc) )
    {
        _dsm->oDB()->dbReset(pStmt);
        return FALSE;
    }
    _dsm->oDB()->dbReset(pStmt);
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU32 tclDABServiceManager::u32GetEPGScheduleInfoContentIds( std::vector<tEPGContentIdDataType>& result )
{
    int result_cnt = 0;
    SQLITE_API int rc;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return 0;
    }

    sqlite3_stmt* pStmt = NULL;
    rc = _dsm->oDB()->dbPrepare(
                "SELECT DISTINCT "
            #ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
                "contentID "
                "FROM "
                "dab_epg_schedule_scope",
            #else /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                "P2.value "
                "FROM "
                "dab_epg_root AS R,"
                "dab_epg AS P1,"
                "dab_epg AS P2 "
                "WHERE "
                "R.dab_epg_o=P1.o AND "
                "P1.o=P2.p_o AND "
                "R.root_tag=2 AND "
                "P2.tag=5",
            #endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
                -1, &pStmt, NULL);

    while(1)
    {
        rc = _dsm->oDB()->dbStep(pStmt);
        if( tclDB::dbOnStepError(rc) )
        {
            _dsm->oDB()->dbReset(pStmt);
            return 0;
        }
        else if( rc == SQLITE_DONE )
        {
            break;
        }
        else if( rc == SQLITE_ROW )
        {
            if( tclDB::dbColumnCount(pStmt) == 1 )
            {
                tclDABServiceManager::tEPGContentIdDataType id;
                tU32 n = tclDB::dbColumnBytes(pStmt,0);
                tU8* b = (tU8*)tclDB::dbColumnBlob(pStmt,0);

                id.r_ensemble = 0;
                id.b_service_type = 0;
                id.r_service = 0;
                id.b_scids = 0;
                id.b_xpad_app_type = 0;

                if( (n>0) && (b!=NULL) )
                {
                    if( tclDSMApp::u32BitStreamValue(b,n,0,1) == 0 )
                    {
                        int j = 1;
                        unsigned int ens_flag = tclDSMApp::u32BitStreamValue(b,n,j,1);
                        j+=1;
                        unsigned int xpad_flag = tclDSMApp::u32BitStreamValue(b,n,j,1);
                        j+=1;
                        unsigned int sid_flag = tclDSMApp::u32BitStreamValue(b,n,j,1);
                        j+=1;
                        id.b_scids = (tU8)tclDSMApp::u32BitStreamValue(b,n,j,4);
                        j+=4;
                        unsigned int ecc = 0;
                        if( ens_flag == 1 )
                        {
                            ecc = tclDSMApp::u32BitStreamValue(b,n,j,8);
                            j+=8;
                            id.r_ensemble = 0x01000000;
                            id.r_ensemble |= (ecc<<16);
                            id.r_ensemble |= tclDSMApp::u32BitStreamValue(b,n,j,16);
                            j+=16;
                        }
                        if( sid_flag == 0 )
                        {
                            id.b_service_type = 1;
                            id.r_service = 0x01000000;
                            id.r_service |= (ecc<<16);
                            id.r_service |= tclDSMApp::u32BitStreamValue(b,n,j,16);
                            j+=16;
                        }
                        else
                        {
                            id.b_service_type = 2;
                            id.r_service = tclDSMApp::u32BitStreamValue(b,n,j,32);
                            j+=32;
                        }
                        if( xpad_flag == 1 )
                        {
                            j+=3;
                            id.b_xpad_app_type = (tU8)tclDSMApp::u32BitStreamValue(b,n,j,5);
                            j+=5;
                        }

                        result.push_back(id);
                        result_cnt++;
                    }
                }
            }
        }
    }
    _dsm->oDB()->dbReset(pStmt);
    return result_cnt;
}
#endif

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDABServiceManager::bGetMultimediaObjects( std::vector<tU8>& resContentId, std::string& resMimeType, std::vector<tU8>& resFileBody, const char* uri, sqlite3_stmt** pContentIdStmt, tU8* multimedia_type , tU32* multimedia_w , tU32* multimedia_h )
{
    SQLITE_API int rc;

    if( uri == NULL )
    {
        return FALSE;
    }

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    DSM_PERF_ENTRY();

    if(pContentIdStmt!=NULL)
    {
        if((*pContentIdStmt)==NULL)
        {
            rc = _dsm->oDB()->dbPrepare(
                        "SELECT DISTINCT M.contentId,A.o FROM dab_data_application AS A,v_dab_epg_service_multimedia AS M WHERE A.o=M.a AND A.uri=?",
                        -1, pContentIdStmt, NULL);
			(void)rc;

            _dsm->oDB()->tclDB::dbBindText((*pContentIdStmt), 1, uri, -1, NULL);
        }
        if((*pContentIdStmt)!=NULL)
        {
            sqlite3_stmt* pStmt = (*pContentIdStmt);
            if( pStmt != NULL )
            {
                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( rc == SQLITE_ROW )
                    {
                        if( tclDB::dbColumnCount(pStmt) == 2 )
                        {
                            tU32 n = tclDB::dbColumnBytes(pStmt,0);
                            resContentId.resize(n);
                            if( n != 0 )
                            {
                                tBool r;
                                memcpy( (void*)&resContentId[0],
                                        (void*)tclDB::dbColumnBlob(pStmt,0),
                                        n);

                                r = bGetMultimediaObject(resMimeType,resFileBody,
                                                         uri,resContentId,
                                                         multimedia_type, multimedia_w, multimedia_h);
                                if( r == TRUE )
                                {
                                    DSM_PERF_EXIT("tclDABServiceManager::bGetMultimediaObjects");
                                    return TRUE;
                                }
                            }
                        }
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        break;
                    }
                }
            }

        }
    }
    DSM_PERF_EXIT("tclDABServiceManager::bGetMultimediaObjects");
    return FALSE;
}

/******************************************************************************
 * FUNCTION:        tclDABServiceManager_bGetMultimediaObject_MatchBestImage
 * DESCRIPTION:     match image requirements to available information
 *****************************************************************************/

/*
Image Parameter:
   Image Type (type): (optional)
    logo_unrestricted       = 0x02
    logo_mono_square        = 0x03
    logo_colour_square      = 0x04
    logo_mono_rectangle     = 0x05
    logo_colour_rectangle   = 0x06

   Image Width (w): (optional)
    widht in pixel

   Image Height (h): (optional)
    height in pixel

   Aspect Radio:
    * Asumed ratio for square types 32:32
    * Asumed ratio for rectangle types 112:32
    * Aspect ratio limit max(w/h,h/w) error is 20% (configurable)

   Image Size Limit:
    * Image Size limit (image_w < 2*wanted_w && image_h < 2*wanted_h) (configurable)

A.) Preprocessing on received attributes
    * When type is not defined (0 or 1) and w<256 and h<256 and w=h or aspect ratio is inside limit for square, then type is set to logo_colour_square.
    * When type is not defined (0 or 1) and w<256 and h<256 and w!=h or aspect ratio is inside limit for rectangle, then type is set to logo_colour_rectangle.

    * When type is logo_unrestricted and w<256 and h<256 and w=h or aspect ratio is inside limit for square, then type is set to logo_colour_square.
    * When type is logo_unrestricted and w<256 and h<256 and w!=h or aspect ratio is inside limit for rectangle, then type is set to logo_colour_rectangle.

    * When type is logo_colour_square or logo_mono_square and w is not defined, then set w to 32 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)
    * When type is logo_colour_square or logo_mono_square and h is not defined, then set h to 32 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)

    * When type is logo_colour_rectangle or logo_mono_rectangle and h is not defined, then set w to 32 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)
    * When type is logo_colour_rectangle or logo_mono_rectangle and h is not defined, then set h to 112 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)

B.) Best Image selection (not implemented yet)
    Search available Station Images for the given contentId and aspect ratio class (square/rectangle).
    The image size shall not exceed the size limits (image_w < 2*wanted_w && image_h < 2*wanted_w).
    Each available image will be qualified by a quality error.

    if(wanted_image_type is logo_square or logo_rectangle)
        aspect_ratio_error = 100 * deviation(image_aspect_ratio / wanted_aspect_ratio)

    image_size_error = 100 * deviation((image_w+image_h) / (wanted_w+wanted_h))

    if(color_requested && color_image)
        image_collor_error = 0
    else if (color_requested && monochrome_image)
        image_collor_error = 20

    quality_error = aspect_ratio_error + image_size_error + image_collor_error

    The image with the smallest (valid) quality error will be selected.
 */

#ifndef CONFIG_DSM__EPG_SQUARE_LOGO_MAX_W
#define CONFIG_DSM__EPG_SQUARE_LOGO_MAX_W (2*128)
#endif
#ifndef CONFIG_DSM__EPG_SQUARE_LOGO_MAX_H
#define CONFIG_DSM__EPG_SQUARE_LOGO_MAX_H (2*128)
#endif

#ifndef CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_W
#define CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_W (2*112)
#endif
#ifndef CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_H
#define CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_H (2*32)
#endif

int tclDABServiceManager::bGetMultimediaObject_MatchBestImage( tS64 a,
                                                               tGetMultimediaObjectFoundMultimediaObjectType* found_mobj,
                                                               tDABEPGLogoImageType image_t, /* image type */
                                                               int image_w,  /* image width */
                                                               int image_h, /* image height */
                                                               tDABEPGLogoImageType wanted_t, /* wanted type */
                                                               int wanted_w, /* wanted width */
                                                               int wanted_h, /* wanted height */
                                                               char* pUrl, int nUrl, /* filename */
                                                               char* pUri, int nUri  /* uri */
                                                               )
{    
    if( (image_w<0) || (image_h<0) || (wanted_w<0) || wanted_h<0 )
    {
        /*
         * reject invalid image parameters
         */
        return 0;
    }

    if(( image_t != logo_undef_0 )&&
        #if (CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_IMAGE_TYPE_1 == 1)
            ( image_t != logo_undef_1 )&&
        #endif
            ( image_t != logo_unrestricted )&&
            ( image_t != logo_mono_square )&&
            ( image_t != logo_colour_square )&&
            ( image_t != logo_mono_rectangle )&&
            ( image_t != logo_colour_rectangle ))
    {
        /*
         * ignore wrong image type
         */
        return 0;
    }

    if(( wanted_t != logo_undef_0 )&&
        #if (CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_IMAGE_TYPE_1 == 1)
            ( wanted_t != logo_undef_1 )&&
        #endif
            ( wanted_t != logo_unrestricted )&&
            ( wanted_t != logo_mono_square )&&
            ( wanted_t != logo_colour_square )&&
            ( wanted_t != logo_mono_rectangle )&&
            ( wanted_t != logo_colour_rectangle ))
    {
        /*
         * ignore wrong wanted type
         */
        return 0;
    }

    if((( image_w == 0 )||( image_h == 0 ))&&((image_t==logo_colour_square)||(image_t==logo_mono_square)))
    {
        /*
         * When type is logo_colour_square or logo_mono_square and w is not defined, then set w to 32 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)
         * When type is logo_colour_square or logo_mono_square and h is not defined, then set h to 32 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)
         */
        image_w = 32; /* (ETSI TS 102 818 V3.1.1 (2015-01), 5.8) */
        image_h = 32;
    }
    else if((( image_w == 0 )||( image_h == 0 ))&&((image_t==logo_colour_rectangle)||(image_t==logo_mono_rectangle)))
    {
        /*
         * When type is logo_colour_rectangle or logo_mono_rectangle and h is not defined, then set w to 32 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)
         * When type is logo_colour_rectangle or logo_mono_rectangle and h is not defined, then set h to 112 (ETSI TS 102 818 V3.1.1 (2015-01), 5.8)
         */
        image_w = 112; /* (ETSI TS 102 818 V3.1.1 (2015-01), 5.8) */
        image_h = 32;
    }

#if (CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_HR_IMAGE_UNRESTRICTED_HEIGHT_BUT_NO_WIDTH == 1)
    /*
     * speacial bugfix for DSMDecodeSingleDataServices.EPGTest_2014_08_27__15_10_45__21000_SSI32Log
     * HR images with type=2, width=undefined and height=128
     *
     */
    if(( image_w == 0 )&&( image_h != 0 )&&(image_h <= CONFIG_DSM__EPG_SQUARE_LOGO_MAX_H )&&( image_t==logo_unrestricted ))
    {
        image_w = image_h;
        image_t = logo_colour_square;
    }
#endif

    if(( image_w == 0 )||( image_h == 0 ))
    {
        /*
         * return because image size is not clear
         */
        return 0;
    }

    if(((wanted_t==logo_colour_square)||(wanted_t==logo_mono_square))&&((wanted_w == 0)||(wanted_h == 0)))
    {
        /*
         * by default, return the 32x32 sqare format
         */
        wanted_w = 32;
        wanted_h = 32;
    }

    if(((wanted_t==logo_colour_rectangle)||(wanted_t==logo_mono_rectangle))&&((wanted_w == 0)||(wanted_h == 0)))
    {
        /*
         * by default, return the 112x32 rectangle format
         */
        wanted_w = 112;
        wanted_h = 32;
    }

    if(( wanted_w == 0 )||( wanted_h == 0 ))
    {
        /*
         * return because wanted size is not clear
         */
        return 0;
    }

    /*
     * calculate wanted aspect ratio
     */
    double wanted_aspect_ratio = 1.0;
    if( (wanted_w != 0) && (wanted_h != 0) )
    {
        if((wanted_t==logo_colour_square)||(wanted_t==logo_mono_square))
        {
            wanted_aspect_ratio = 32.0/32.0;
        }
        else if((wanted_t==logo_colour_rectangle)||(wanted_t==logo_mono_rectangle))
        {
            wanted_aspect_ratio = 112.0/32.0;
        }
        else
        {
            wanted_aspect_ratio = (double)wanted_w/(double)wanted_h;
        }
    }

    /*
     * calculate image aspect ratio
     */
    double image_aspect_ratio = 1.0;
    if( (image_w != 0) && (image_h != 0) )
    {
        image_aspect_ratio = (double)image_w/(double)image_h;
    }

    /*
     * if(wanted_image_type is logo_square or logo_rectangle)
     *   aspect_ratio_error = 100 * deviation(image_aspect_ratio / wanted_aspect_ratio)
     */
    double aspect_ratio_error = 0;
    if( image_aspect_ratio > wanted_aspect_ratio )
    {
        aspect_ratio_error = image_aspect_ratio/wanted_aspect_ratio;
        aspect_ratio_error -= 1;
    }
    else
    {
        aspect_ratio_error = wanted_aspect_ratio/image_aspect_ratio;
        aspect_ratio_error -= 1;
    }

    if( (100*aspect_ratio_error) > CONFIG_DSM__GET_MULTIMEDIA_OBJECT_ASCPECT_RATIO_ERROR_LIMIT )
    {
        /*
         * abort, when aspect ratio does not fit at all
         */
        return 0;
    }

    if( ((image_t==logo_undef_0) ||
     #if (CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_IMAGE_TYPE_1 == 1)
         (image_t==logo_undef_1) ||
     #endif
         (image_t==logo_unrestricted)) &&
            (image_w != 0) && (image_h != 0) &&
            (image_aspect_ratio != 0))
    {
        /*
         * calucalte the image type if not defined
         */
        double square_acpect_ratio_error = 0;
        if(image_aspect_ratio > (32.0/32.0))
        {
            square_acpect_ratio_error = image_aspect_ratio / (32.0/32.0);
            square_acpect_ratio_error -= 1;
        }
        else
        {
            square_acpect_ratio_error = (32.0/32.0) / image_aspect_ratio;
            square_acpect_ratio_error -= 1;
        }
        if( (( image_w <= wanted_w ) && ( image_h <= wanted_h ) ) || ((image_w <= CONFIG_DSM__EPG_SQUARE_LOGO_MAX_W) && (image_h <= CONFIG_DSM__EPG_SQUARE_LOGO_MAX_H)) )
        {
            if( (100*square_acpect_ratio_error) <= CONFIG_DSM__GET_MULTIMEDIA_OBJECT_ASCPECT_RATIO_ERROR_LIMIT )
            {
                /*
                * When type is logo_unrestricted and w<256 and h<256 and w=h or aspect ratio is inside limit for square, then type is set to logo_colour_square.
                 */
                image_t = logo_colour_square;
            }
        }
    }

    if( ((image_t==logo_undef_0) ||
     #if (CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_IMAGE_TYPE_1 == 1)
         (image_t==logo_undef_1) ||
     #endif
         (image_t==logo_unrestricted)) &&
            (image_w != 0) && (image_h != 0) &&
            (image_aspect_ratio != 0) )
    {
        /*
         * calucalte the image type if not defined
         */
        double rectangle_acpect_ratio_error = 0;
        if(image_aspect_ratio > (112.0/32.0))
        {
            rectangle_acpect_ratio_error = image_aspect_ratio / (112.0/32.0);
            rectangle_acpect_ratio_error -= 1;
        }
        else
        {
            rectangle_acpect_ratio_error = (112.0/32.0) / image_aspect_ratio;
            rectangle_acpect_ratio_error -= 1;
        }
        if( (( image_w <= wanted_w ) && ( image_h <= wanted_h ) ) || ((image_w <= CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_W) && (image_h <= CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_H)) )
        {
            if( (100*rectangle_acpect_ratio_error) <= CONFIG_DSM__GET_MULTIMEDIA_OBJECT_ASCPECT_RATIO_ERROR_LIMIT )
            {
                /*
                * When type is logo_unrestricted and w<256 and h<256 and w!=h or aspect ratio is inside limit for rectangle, then type is set to logo_colour_rectangle.
                 */
                image_t = logo_colour_rectangle;
            }
        }
    }

    if((wanted_t==logo_colour_square)||(wanted_t==logo_mono_square))
    {
        if( !(( image_w <= wanted_w ) || ( image_h <= wanted_h ) ) && !((image_w <= CONFIG_DSM__EPG_SQUARE_LOGO_MAX_W) || (image_h <= CONFIG_DSM__EPG_SQUARE_LOGO_MAX_H)) )
        {
            /* logo image is to big and larger as requested
             */
            return 0;
        }
    }

    if((wanted_t==logo_colour_rectangle)||(wanted_t==logo_mono_rectangle))
    {
        if( !(( image_w <= wanted_w ) || ( image_h <= wanted_h ) ) && !((image_w <= CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_W) || (image_h <= CONFIG_DSM__EPG_RECTANGLE_LOGO_MAX_H)) )
        {
            /* logo image is to big and larger as requested
             */
            return 0;
        }
    }

    /*
     * calculate image size error
     */
    double image_size_error = 0;
    if( ((image_w+image_h)!=0) && (wanted_w+wanted_h) )
    {
        if( (image_w+image_h) > (wanted_w+wanted_h) )
        {
            image_size_error = (double)(image_w+image_h) / (double)(wanted_w+wanted_h);
            image_size_error -= 1;
        }
        else
        {
            image_size_error = (double)(wanted_w+wanted_h) / (double)(image_w+image_h);
            image_size_error -= 1;
        }
    }

    /*
     * calculate image color error
     */
    int image_color_error = 0;
    if((wanted_t==logo_colour_square)||(wanted_t==logo_colour_rectangle))
    {
        if((image_t==logo_colour_square)||(image_t==logo_colour_rectangle))
        {
            image_color_error = 0;
        }
        else if((image_t==logo_mono_square)||(image_t==logo_mono_rectangle))
        {
            image_color_error = 20;
        }
    }
    else if((wanted_t==logo_mono_square)||(wanted_t==logo_mono_rectangle))
    {
        if((image_t==logo_mono_square)||(image_t==logo_mono_rectangle))
        {
            image_color_error = 0;
        }
        else
        {
            /*
             * color image while monochrome requested, ignore it
             */
            return 0;
        }
    }

    /*
     * caculate image quality error value
     */
    int image_quality_error = 1 + image_color_error + (int)(100*image_size_error) + (int)(100*aspect_ratio_error);

    if(( image_quality_error < found_mobj->quality_error )||( found_mobj->quality_error == 0 ))
    {
        /*
         * take current image
         */
        found_mobj->quality_error = image_quality_error;
        found_mobj->a = a;
        memset(found_mobj->Url,0,sizeof(found_mobj->Url));
        strncpy(found_mobj->Url,
                pUrl,
                tclDABServiceManager_MIN((int)sizeof(found_mobj->Url)-1,nUrl));
        memset(found_mobj->Uri,0,sizeof(found_mobj->Uri));
        strncpy(found_mobj->Uri,
                pUri,
                tclDABServiceManager_MIN((int)sizeof(found_mobj->Uri)-1,nUri));
        found_mobj->w = image_w;
        found_mobj->h = image_h;
        found_mobj->t = image_t;

        return 1;
    }

    /* continue search
     */
    return 0;
}

/******************************************************************************
 * FUNCTION:        bGetMultimediaObject
 * DESCRIPTION:     get multimedia object by given ContentID
 * PARAMETER:
 * RETURNVALUE:
 *****************************************************************************/
tBool tclDABServiceManager::bGetMultimediaObject( std::string& resMimeType, std::vector<tU8>& resFileBody, const char* uri, std::vector<tU8> ContentId, tU8* pMultimediaType, tU32* pW, tU32* pH, tU64* pResA, std::string* pResURL )
{
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
    tBool bResult = FALSE;
    (void)resMimeType;

    if( uri == NULL )
    {
        return FALSE;
    }

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return FALSE;
    }

    /* request object for service
     */
    SQLITE_API int rc;
    tGetMultimediaObjectFoundMultimediaObjectType found_mobj;
    sqlite3_stmt* pStmt01 = NULL;
    memset(&found_mobj,0,sizeof(found_mobj));

    rc = _dsm->oDB()->dbPrepare(
                "SELECT M.a,M.width,M.height,M.type,M.url,A.uri FROM dab_data_application AS A,v_dab_epg_service_multimedia AS M ON A.o=M.a,dab_mot_directory_cname AS F ON M.a=F.a AND M.url=F.contentname WHERE A.uri=?1 AND M.contentId=?2 ORDER BY A.age_received ASC",
                -1,
                &pStmt01,
                NULL);

    if( pStmt01 != NULL )
    {
        _dsm->oDB()->tclDB::dbBindText(pStmt01, 1, uri, -1, NULL);
        _dsm->oDB()->tclDB::dbBindBlob(pStmt01, 2, &ContentId[0], (int)ContentId.size(), NULL);

        tS64 _a = DSM_INVALID_APPID;

        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt01);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt01) == 6 )
                {
                    tS64 a = tclDB::dbColumnInt64( pStmt01,0 );
                    int w = tclDB::dbColumnInt( pStmt01,1 );
                    int h = tclDB::dbColumnInt( pStmt01,2 );
                    tDABEPGLogoImageType t = (tDABEPGLogoImageType)tclDB::dbColumnInt( pStmt01,3 );

                    if(( a != _a)&&( _a != DSM_INVALID_APPID))
                    {
                        /*
                         * check images from the most recent application only
                         */
                        break;
                    }
                    _a = a;

                    bGetMultimediaObject_MatchBestImage(a,
                                                        &found_mobj,
                                                        (tDABEPGLogoImageType)t,
                                                        w,h,
                                                        (pMultimediaType==NULL)?logo_undef_0:(tDABEPGLogoImageType)(*pMultimediaType),
                                                        (pW==NULL)?0:(*pW),
                                                        (pH==NULL)?0:(*pH),
                                                        (char*)tclDB::dbColumnText( pStmt01,4 ), tclDB::dbColumnBytes( pStmt01,4 ),
                                                        (char*)tclDB::dbColumnText( pStmt01,5 ), tclDB::dbColumnBytes( pStmt01,5 ));
                }
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt01);

        if( found_mobj.quality_error != 0 )
        {
            /* read image body ...
             */
            if( pW != 0 )
            {
                *pW = found_mobj.w;
            }
            if( pH != 0 )
            {
                *pH = found_mobj.h;
            }
            if( pMultimediaType != 0 )
            {
                *pMultimediaType = found_mobj.t;
            }

            /* update the "used" timestamp
             */
            vInformUsed( found_mobj.Uri );

            bResult = bGetFile( found_mobj.a, found_mobj.Url, resMimeType, resFileBody );

            if( pResA != NULL )
            {
                *pResA =  found_mobj.a;
            }
            if( pResURL != NULL )
            {
                *pResURL = found_mobj.Url;
            }
        }
    }
    return bResult;
#else /* CONFIG_DSM__ENABLE_DAB_EPG */
    (void)pMultimediaType;
    (void)pW;
    (void)pH;
    (void)mimeValue;
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
    return FALSE;
}

/******************************************************************************
 * FUNCTION:        bGetMultimediaObject
 * DESCRIPTION:     get multimedia object by given ID
 * PARAMETER:
 * RETURNVALUE:
 *****************************************************************************/
tPVoid tclDABServiceManager::bGetMultimediaObject( tU32* u32ResultBytes, tU32 r_ensemble, tU8 b_service_type, tU32 r_service, tU8 b_scids, tU8 b_xpad_app_type, tU8* pMultimediaType, tU32* pW, tU32* pH, tPChar* mimeValue, tU64* pResA, std::string* pResURL )
{
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
    tPVoid pResult = NULL;
    (void)mimeValue;

    DSM_PERF_ENTRY();

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        DSM_PERF_EXIT("tclDABServiceManager::bGetMultimediaObject");
        return NULL;
    }

    if( b_xpad_app_type != 0 )
    {
        /* request object for XPad application
         */
        DSM_PERF_EXIT("tclDABServiceManager::bGetMultimediaObject");
        return NULL; /* not implemented */
    }
    else if( b_scids != 0xff )
    {
        /* request object for service component
         */
        DSM_PERF_EXIT("tclDABServiceManager::bGetMultimediaObject");
        return NULL; /* not implemented */
    }
    else if( r_service != 0xff )
    {
        /* request object for service
         */
        SQLITE_API int rc;
        tGetMultimediaObjectFoundMultimediaObjectType found_mobj;
        sqlite3_stmt* pStmt01 = NULL;
        memset(&found_mobj,0,sizeof(found_mobj));

        rc = _dsm->oDB()->dbPrepare(
                    "SELECT "
                    "M.a,M.width,M.height,M.type,M.url,A.uri "
                    "FROM dab_data_application AS A,"
                    "v_dab_epg_service_multimedia AS M,"
                    "dab_mot_directory_cname AS F "
                    "WHERE "
                    "A.o=M.a AND "
                    "M.a=F.a AND "
                    "M.url=F.contentname AND "
                    "dsmCheckContentId(M.contentId,?1,?2,?3,?4,?5)!=0 "
                    "ORDER BY A.age_received ASC",
                    -1,
                    &pStmt01,
                    NULL);

        if( pStmt01 != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt01,
                                            1,
                                            r_ensemble);
            _dsm->oDB()->tclDB::dbBindInt64(pStmt01,
                                            2,
                                            b_service_type);
            _dsm->oDB()->tclDB::dbBindInt64(pStmt01,
                                            3,
                                            r_service);
            _dsm->oDB()->tclDB::dbBindInt64(pStmt01,
                                            4,
                                            b_scids);
            _dsm->oDB()->tclDB::dbBindInt64(pStmt01,
                                            5,
                                            b_xpad_app_type);

            tS64 _a = DSM_INVALID_APPID;

            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt01);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt01) == 6 )
                    {
                        tS64 a = tclDB::dbColumnInt64( pStmt01,0 );
                        int w = tclDB::dbColumnInt( pStmt01,1 );
                        int h = tclDB::dbColumnInt( pStmt01,2 );
                        tDABEPGLogoImageType t = (tDABEPGLogoImageType)tclDB::dbColumnInt( pStmt01,3 );

                        if(( a != _a)&&( _a != DSM_INVALID_APPID))
                        {
                            /*
                             * check images from the most recent application only
                             */
                            break;
                        }
                        _a = a;

                        bGetMultimediaObject_MatchBestImage(a,
                                                            &found_mobj,
                                                            (tDABEPGLogoImageType)t,
                                                            w,h,
                                                            (pMultimediaType==NULL)?logo_undef_0:(tDABEPGLogoImageType)(*pMultimediaType),
                                                            (pW==NULL)?0:(*pW),
                                                            (pH==NULL)?0:(*pH),
                                                            (char*)tclDB::dbColumnText( pStmt01,4 ), tclDB::dbColumnBytes( pStmt01,4 ),
                                                            (char*)tclDB::dbColumnText( pStmt01,5 ), tclDB::dbColumnBytes( pStmt01,5 ));
                    }
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt01);

            if( found_mobj.quality_error != 0 )
            {
                /* read image body ...
                 */
                if( pW != 0 )
                {
                    *pW = found_mobj.w;
                }
                if( pH != 0 )
                {
                    *pH = found_mobj.h;
                }
                if( pMultimediaType != 0 )
                {
                    *pMultimediaType = found_mobj.t;
                }

                sqlite3_stmt* pStmt02 = NULL;
                rc = _dsm->oDB()->dbPrepare(
                            "SELECT F.body FROM v_dab_mot_file AS F WHERE F.a=?1 AND F.name=?2 LIMIT 1",
                            -1,
                            &pStmt02,
                            NULL);

                if( pStmt02 != NULL )
                {
                    _dsm->oDB()->tclDB::dbBindInt64(pStmt02,
                                                    1,
                                                    found_mobj.a);

                    _dsm->oDB()->tclDB::dbBindText(pStmt02,
                                                   2,
                                                   found_mobj.Url,
                                                   -1,
                                                   NULL);

                    while(1)
                    {
                        rc = _dsm->oDB()->dbStep(pStmt02);
                        if( rc == SQLITE_DONE )
                        {
                            break;
                        }
                        else if( rc == SQLITE_ROW )
                        {
                            if( tclDB::dbColumnCount(pStmt02) == 1 )
                            {
                                int n = tclDB::dbColumnBytes( pStmt02,0 );
                                if( n > 0 )
                                {
                                    pResult = sqlite3_malloc( n );
                                    if( pResult != NULL )
                                    {
                                        tPVoid p = (tPVoid)tclDB::dbColumnBlob( pStmt02,0 );
                                        if( u32ResultBytes != NULL )
                                        {
                                            *u32ResultBytes = n;
                                        }
                                        memcpy( pResult, p, n);
                                    }
                                }

                                /* update the "used" timestamp
                                 */
                                vInformUsed( found_mobj.Uri );
                            }
                            break;
                        }
                        else if( tclDB::dbOnStepError(rc) )
                        {
                            break;
                        }
                    }
                    _dsm->oDB()->dbReset(pStmt02);
                }

                if( pResA != NULL )
                {
                    *pResA =  found_mobj.a;
                }
                if( pResURL != NULL )
                {
                    *pResURL = found_mobj.Url;
                }

                DSM_PERF_EXIT("tclDABServiceManager::bGetMultimediaObject");
                return pResult;
            }
        }
    }
#else /* CONFIG_DSM__ENABLE_DAB_EPG */
    (void)u32ResultBytes;
    (void)r_ensemble;
    (void)b_service_type;
    (void)r_service;
    (void)b_scids;
    (void)b_xpad_app_type;
    (void)pMultimediaType;
    (void)pW;
    (void)pH;
    (void)mimeValue;
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
    DSM_PERF_EXIT("tclDABServiceManager::bGetMultimediaObject");
    return NULL;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vSetCommandReceiver( tclDABServiceManagerCommandReceiver* cmdrx )
{
    _cmd_receiver = cmdrx;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vSwitchOff( )
{
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    if( _wizard != NULL )
    {
        _wizard->vSwitchOff();
    }
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vSetNumTuner( tU8 u32NumTuner )
{
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    if( _wizard != NULL )
    {
        _wizard->vSetNumTuner(u32NumTuner);
    }
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vSetDecodingCapability( tU8 u8TunerIdx, tU32 u32NumCurrentENSJobs, tU32 u32NumOtherENSJobs )
{
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    if( _wizard != NULL )
    {
        _wizard->vSetDecodingCapability(u8TunerIdx,u32NumCurrentENSJobs,u32NumOtherENSJobs);
    }
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vSetDecodingStrategy( tU8 u8TunerIdx, std::list<tU16> u16UserAppType_list, tU32 u32Strategy )
{
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    if( _wizard != NULL )
    {
        _wizard->vSetDecodingStrategy(u8TunerIdx,u16UserAppType_list,u32Strategy);
    }
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vInformUsed( std::string uri )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    rc = _dsm->oDB()->dbPrepare(
                             "UPDATE dab_data_application SET age_used=0 WHERE age_used!=0 AND uri=?",
                             -1,
                             &pStmt,
                             NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindText(pStmt,
                          1,
                          uri.data(),
                          (int)uri.length(),
                          NULL);
        rc = _dsm->oDB()->dbStep(pStmt);
        _dsm->oDB()->dbReset(pStmt);
    }
    (void)rc;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vInformUsed( tS64 s64AppId )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    rc = _dsm->oDB()->dbPrepare(
                             "UPDATE dab_data_application SET age_used=0 WHERE age_used!=0 AND o=?",
                             -1,
                             &pStmt,
                             NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                          1,
                          s64AppId);
        rc = _dsm->oDB()->dbStep(pStmt);
        _dsm->oDB()->dbReset(pStmt);
    }
    (void)rc;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vIncreateAgeCounter()
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    rc = _dsm->oDB()->dbPrepare(
                             "UPDATE dab_data_application SET age_used=(CASE WHEN age_used IS NULL THEN 1 ELSE age_used+1 END),age_received=(CASE WHEN age_received IS NULL THEN 1 ELSE age_received+1 END)",
                             -1,
                             &pStmt,
                             NULL);
    if( pStmt != NULL )
    {
        rc = _dsm->oDB()->dbStep(pStmt);
        _dsm->oDB()->dbReset(pStmt);
    }
    (void)rc;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDABServiceManager::vHandle()
{
    if( _dsm->oDB()->bIsOpen() == FALSE )
    {
        return;
    }

    DSM_PERF_ENTRY();

    ETG_TRACE_USR1(("%p.vHandle() ... ",
                    (void*)this));

    tU64 t = _dsm->u64ElapsedTimeMS();
    if(( t - _u64Timeout60000mS ) >= 60000 )
    {
        _u64Timeout60000mS = t;
        /*
         * executed every 60000 mS
         */
        vIncreateAgeCounter();

#ifdef CONFIG_DSM_ENABLE_CYCLIC_WAL_CHECKPOINT_60s
        if( _dsm->oDB()->pHandle() != NULL )
        {
            _dsm->oDB()->vWALCheckPoint();
        }
#endif /* CONFIG_DSM_ENABLE_CYCLIC_WAL_CHECKPOINT_60s */
    }
    if(( t - _u64Timeout1000mS ) >= 1000 )
    {
        _u64Timeout1000mS = t;
        /*
         * executed every 1000 mS
         */

        _u64TimeStamp = t/1000;
        //_u64TimeOffset = 2*60*60;

        {
            /* update current timestamp in param table
             */
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
                                     "REPLACE INTO dsm_param (key,value) VALUES(?,?)",
                                     -1,
                                     &pStmt,
                                     NULL);
            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                1,
                                                DSM_PARAM_KEY__CURRENT_UTC_TIMESTAMP);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                2,
                                                t/1000);
                rc = _dsm->oDB()->dbStep(pStmt);
                _dsm->oDB()->dbReset(pStmt);
            }
            (void)rc;
        }

        {
            /* update current timeoffset in param table
             */
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
                                     "REPLACE INTO dsm_param (key,value) VALUES(?,?)",
                                     -1,
                                     &pStmt,
                                     NULL);
            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                1,
                                                DSM_PARAM_KEY__CURRENT_UTC_TIMEOFFSET);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                2,
                                                2*60*60);
                rc = _dsm->oDB()->dbStep(pStmt);
                _dsm->oDB()->dbReset(pStmt);
            }
            (void)rc;
        }

#ifdef CONFIG_DSM_ENABLE_CYCLIC_WAL_CHECKPOINT_1s
        if( _dsm->oDB()->pHandle() != NULL )
        {
            _dsm->oDB()->vWALCheckPoint();
        }
#endif /* CONFIG_DSM_ENABLE_CYCLIC_WAL_CHECKPOINT_1s */
    }

#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    if( _wizard != NULL )
    {
        _wizard->vHandle();
    }
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */

    /* call vHandle for each active app ...
     */
    for(std::list<tclDSMDABApp*>::iterator list_iter = _active_apps.begin();
        list_iter != _active_apps.end(); list_iter++)
    {
        tclDSMDABApp* app = *list_iter;
        if( app != NULL )
        {
            app->vHandle();
        }
    }

    /* call vHandle from SLS manager
     */
    _sls_handler.vHandle();

    DSM_PERF_EXIT("tclDABServiceManager::vHandle");
    ETG_TRACE_USR1(("%p.vHandle() ... done",
                    (void*)this));
}

#endif /* CONFIG_DSM__ENABLE_DAB */

/******************************************************************************
 * EOF
 *****************************************************************************/

