/************************************************************************
 * FILE:        fc_dabtuner_timer.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of fc_dabtuner_timer
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#ifndef FC_DABTUNER_TIMER_HPP
#define FC_DABTUNER_TIMER_HPP

//#include "fc_dabtuner_dispatcher.h"
#include "fc_dabtuner_messageHandler.h"
#include "fc_dabtuner_trace.h"
#include "fc_dabtuner_profiler.h"

//#include <vector>
#include <map>


//#include <map>

//using namespace std;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_DISP 
#include "trcGenProj/Header/fc_dabtuner_timer.hpp.trc.h"
#endif

namespace DAB {
#define DAB_TMR_MAGIC 0x11335577
#define DAB_TMR_BAD_MAGIC 0x22446688
typedef enum {
    DAB_eTimerInitial,
    DAB_eTimerRunning,
    DAB_eTimerQueued,
    DAB_eTimerCanceled,
    DAB_eTimerInvalid
} DAB_tenTimerState;

class DAB_tclTimer;
class DAB_tclTimerMgr;

typedef tU32 DAB_tTimerHandle;
    tVoid DAB_vTmrMgrCallbackFn(tVoid *pvArg);

//lint -esym(1712, DAB_tclTimerMgr) prio3 prio3: default constructor not defined
class DAB_tclTimerMgr {
	friend class DAB_tclTimer;
 public:
	typedef std::map<DAB_tTimerHandle, DAB_tclTimer*> tTimerMap;

	DAB_tclTimerMgr(OSAL_tEventHandle hEvent, tU32 u32EventMask){
		_hLastHandle=0;
        _hEvent=hEvent;
        _u32EventMask=u32EventMask;
	};


//lint -esym(1762,DAB::DAB_tclTimerMgr::vHandleTimerEvent)  // Member function '...' could be made const
	tVoid vHandleTimerEvent(){
		ETG_TRACE_USR4(("DAB_tclTimerMgr::vHandleTimerEvent START"));
        DAB_tTimerHandle hHandle=OSAL_C_INVALID_HANDLE;
        static DAB_Profiler rProfiler("vHandleTimerEvent");
        while (_oSemFreeQ.bPop(&hHandle)) {
            ETG_TRACE_USR4(("DAB_tclTimerMgr::TimerDebug:execute _hTmr=0x%08x",
                            hHandle));
            rProfiler.vStart( hHandle);
            vExecuteTimer(hHandle);
            rProfiler.u32GetMs();
        }        
		ETG_TRACE_USR4(("DAB_tclTimerMgr::vHandleTimerEvent DONE"));
	};


 private:
	DAB_tTimerHandle hRegisterTimer(DAB_tclTimer * poTmr){
		_hLastHandle++;
        //lint -save -esym(1702, operator!=) both member and non-member ops
		while(_timerHandleMap.find(_hLastHandle) != _timerHandleMap.end()) {
        //lint -restore
            _hLastHandle++;
		}
        _timerHandleMap.insert(std::pair<DAB_tTimerHandle, DAB_tclTimer*>(_hLastHandle, poTmr));
		return _hLastHandle;
	};
	tBool bDeregisterTimer(DAB_tTimerHandle *hHandle) {
		tBool bRes=FALSE;
        tTimerMap::iterator iter = _timerHandleMap.find(*hHandle);
        //lint -save -esym(1702, operator!=) both member and non-member ops
        if (iter != _timerHandleMap.end()) {
        //lint -restore
            _timerHandleMap.erase(iter);
			bRes=TRUE;
        }
        *hHandle=(tU32)OSAL_C_INVALID_HANDLE;
		return bRes;
    };

	tBool bIsRegistered(DAB_tTimerHandle hHandle) {
		tBool bRes=FALSE;
        tTimerMap::iterator iter = _timerHandleMap.find(hHandle);
        //lint -save -esym(1702, operator!=) both member and non-member ops
        if (iter != _timerHandleMap.end()) {
        //lint -restore
            bRes=TRUE;
		}
		return bRes;
	};

    tVoid vExecuteTimer(DAB_tTimerHandle hHandle);
	DAB_tTimerHandle _hLastHandle;
    OSAL_tEventHandle _hEvent;
    tU32 _u32EventMask;
	DAB_tclTimer *poGetTimer(DAB_tTimerHandle hHandle) {
        //lint -save -esym(1702, operator!=) both member and non-member ops
        tTimerMap::iterator iter = _timerHandleMap.find(hHandle);
        //lint -restore
        
        if ((iter) != _timerHandleMap.end()) { 
            return iter->second;
        }
        return OSAL_NULL;
    };

	tTimerMap _timerHandleMap;
    DAB_SemFreeQ<DAB_tTimerHandle, 100> _oSemFreeQ;
};




class DAB_tclTimer {
	friend class DAB_tclTimerMgr;

 public: 
	template<class D, class M>
    static DAB_tclTimer *poCreate(D* pMh, M const *pUsrMsg) {
        (tVoid) pMh;
        ETG_TRACE_USR4(("tclTimer::poCreate"));
        DAB_tclTimer *pTmr= OSAL_NEW DAB_tclTimer(DAB_poGetMsgCaller<D,M>(), pUsrMsg);
        return pTmr;
    }

    tVoid vDeInit() {
        if (!bValid()) {
            return;
        }
        if (_poTmrMgr->bDeregisterTimer(&_hTmr)) {
            tS32 s32OsalRet;
            s32OsalRet = OSAL_s32TimerDelete(_hOsalTmrHandle);
            DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);
        }
        if (_pUsrMsg) {
            delete _pUsrMsg;
            _pUsrMsg=OSAL_NULL;
        }
        _hOsalTmrHandle=(tU32)OSAL_C_INVALID_HANDLE;
        _hTmr = (tU32)OSAL_C_INVALID_HANDLE;
        _eTimerState = DAB_eTimerInvalid;
        _poCaller= OSAL_NULL;

    }

    tBool bValid() const {
        if (_u32Magic!=DAB_TMR_MAGIC) {
            return FALSE;
        }
        return _eTimerState != DAB_eTimerInvalid;
    }

	template<class D, class M>
    tVoid vInit(D* pMh, M const *pUsrMsg) {
        (tVoid) pMh;
        DAB_Message *pUsrMsg_=OSAL_NEW M(*pUsrMsg);

        ETG_TRACE_USR4(("tclTimer::vInit(D,M)"));
        vInit(DAB_poGetMsgCaller<D,M>(), pUsrMsg_);
    }

	template<class D, class M>
    tVoid vInit(D* pMh, M const &rUsrMsg) {
        (tVoid) pMh;
        DAB_Message *pUsrMsg_=OSAL_NEW M(rUsrMsg);

        //        ETG_TRACE_USR4(("tclTimer::vInit(D,M)"));
        vInit(DAB_poGetMsgCaller<D,M>(), pUsrMsg_);
    }


    DAB_tclTimer():
        _u32Magic(DAB_TMR_MAGIC),
        _pUsrMsg(OSAL_NULL),
        _poCaller(OSAL_NULL),
        _eTimerState(DAB_eTimerInvalid),
        _u32DurationMs(0),
        _bCyclic(FALSE),
        _hOsalTmrHandle(OSAL_C_INVALID_HANDLE),
        _hTmr(OSAL_C_INVALID_HANDLE),
        _poTmrMgr(OSAL_NULL)
    {};

    //lint -esym(1551, DAB_tclTimer::~DAB_tclTimer) prio3: Function may throw exception in destructor
    ~DAB_tclTimer() {
        ETG_TRACE_USR4(("tclTimer::Destructor"));
        vDeInit();
        _u32Magic=DAB_TMR_BAD_MAGIC;
        _poCaller=OSAL_NULL;
        _pUsrMsg=OSAL_NULL;
        _poTmrMgr=OSAL_NULL;
    };

    tVoid vStart(tU32 u32DurationMs, tBool bCyclic=FALSE) {
        //        ETG_TRACE_USR4(("tclTimer::vStart :u32DurationMs=%u bCyclic=%u", u32DurationMs, bCyclic));
        if (!bValid()) {
            ETG_TRACE_COMP(("tclTimer::vStart: Timer Invalid"));
            return;
        }
        DAB_ASSERT_RETURN(OSAL_NULL!=_poTmrMgr);
        DAB_ASSERT_RETURN(_poTmrMgr->bIsRegistered(_hTmr));
        _u32DurationMs=u32DurationMs;
        _bCyclic=bCyclic;
        if (u32DurationMs) {
            _eTimerState=DAB_eTimerRunning;
        }
        else {
            _eTimerState=DAB_eTimerCanceled;
        }
		tS32 s32OsalRet;
        if (OSAL_NULL != _poCaller) {
            ETG_TRACE_USR2(("tclTimer::vStart :u32DurationMs=%u bCyclic=%u _hTmr=0x%08x receiver=%s", 
                            u32DurationMs, 
                            bCyclic,
                            _hTmr,
                            _poCaller->poGetTarget()->pcGetName()));
        }

#if 0
        if (OSAL_NULL != _pUsrMsg) {
            _pUsrMsg->vTrace();
        }
#endif

        s32OsalRet = OSAL_s32TimerSetTime(_hOsalTmrHandle, _u32DurationMs, bCyclic?_u32DurationMs:0);
        DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);

    };

    tVoid vStop() {
        if (!bValid()) {
            ETG_TRACE_USR1(("tclTimer::vStop: timer invalid"));
            return;
        }
        if (OSAL_NULL != _poCaller) {
            ETG_TRACE_USR2(("tclTimer::vStop _hTmr=0x%08x receiver=%s", 
                            _hTmr, _poCaller->poGetTarget()->pcGetName()));
        }
        _eTimerState=DAB_eTimerCanceled;
        DAB_ASSERT_RETURN(OSAL_NULL != _poTmrMgr);
        DAB_ASSERT_RETURN(_poTmrMgr->bIsRegistered(_hTmr));
        tS32 s32OsalRet;
        s32OsalRet = OSAL_s32TimerSetTime(_hOsalTmrHandle, 0, 0);
        DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);           
    };

    tBool bIsRunning() const {
#ifdef DAB_ENABLE_TIMER_DEBUG
        ETG_TRACE_USR4(("tclTimer::bIsRunning _hTmr=0x%08x:%d", 
                        _hTmr, _eTimerState==DAB_eTimerRunning));
#endif

        return _eTimerState==DAB_eTimerRunning;
    }

    tVoid vPush2Q() {
#ifdef DAB_ENABLE_TIMER_DEBUG
        ETG_TRACE_USR4(("DAB_tclTimerMgr::TimerDebug:push _hTmr=0x%08x start",
                        _hTmr));
#endif        
        DAB_ASSERT_RETURN(OSAL_NULL != _poTmrMgr);
        _poTmrMgr->_oSemFreeQ.vPush(_hTmr);
#ifdef DAB_ENABLE_TIMER_DEBUG
        ETG_TRACE_USR4(("DAB_tclTimerMgr::TimerDebug:push _hTmr=0x%08x done",
                        _hTmr));
#endif
        tS32 s32OsalRet;
        s32OsalRet = OSAL_s32EventPost(_poTmrMgr->_hEvent, 
                          _poTmrMgr->_u32EventMask,
                          OSAL_EN_EVENTMASK_OR);
        DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);

    }

	template<class M>
    DAB_tclTimer(DAB_MsgCallerBase *poCaller, M const *pUsrMsg) {
        DAB_Message *pUsrMsg_=OSAL_NEW M(*pUsrMsg);
        vInit(poCaller, pUsrMsg_);
    }



 private:
    tVoid poSetTmrMgr();
    //lint -esym(1704, DAB_tclTimer::DAB_tclTimer) prio3 constructor has private access specification
    //lint -esym(1712, DAB_tclTimer::DAB_tclTimer) prio3 prio3: default constructor not defined
    DAB_tclTimer(DAB_tclTimer const&);             // copy constructor is private
    DAB_tclTimer& operator=(DAB_tclTimer const&);  // assignment operator is private

    tVoid vInit(DAB_MsgCallerBase *poCaller, DAB_Message *pUsrMsg) {
        ETG_TRACE_USR4(("tclTimer::vInit"));
        
        _poCaller=poCaller;
        poSetTmrMgr();
        _hTmr = _poTmrMgr->hRegisterTimer(this);
        tS32 s32OsalRet;
        s32OsalRet = (tS32)OSAL_s32TimerCreate(DAB_vTmrMgrCallbackFn, (tPVoid)this,&_hOsalTmrHandle);
        DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);
        _eTimerState=DAB_eTimerInitial;
        _u32DurationMs=0;
        _bCyclic=FALSE;
        _pUsrMsg=pUsrMsg;
	}


    tVoid vExecute() {
        if (!bValid()) {
            ETG_TRACE_COMP(("tclTimer::vExecute: timer invalid"));
            return;
        }
        if (_eTimerState == DAB_eTimerRunning) {
            if (!_bCyclic) {
                _eTimerState=DAB_eTimerCanceled;
            }
            if (OSAL_NULL != _poCaller) {
                _poCaller->vCall(_pUsrMsg);
            }
        } else {
            ETG_TRACE_USR2(("tclTimer::vExecute: timer not running"));

        }
    };

    tU32 _u32Magic;
    DAB_Message * _pUsrMsg;
    DAB_MsgCallerBase *_poCaller;
    DAB_tenTimerState _eTimerState;
    tU32 _u32DurationMs;
    tBool _bCyclic;
    OSAL_tTimerHandle _hOsalTmrHandle;
    DAB_tTimerHandle _hTmr;
    DAB_tclTimerMgr *_poTmrMgr;


};




}
#endif
