/************************************************************************
 * FILE:        fc_dabtuner_stringUtil.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of fc_dabtuner_stringUtil
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#ifndef DAB_STRING_UTIL_HPP
#define DAB_STRING_UTIL_HPP

#define MIDW_COMMON_S_IMPORT_INTERFACE_UTF8_SORT
#include "midw_common_if.h"

#include "fc_dabtuner_util.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL
#include "trcGenProj/Header/fc_dabtuner_stringUtil.hpp.trc.h"
#endif

namespace DAB {

#define DAB_MAX_UTF8_CHAR_LEN 3
#define TUN_INVALID_UTF8_CHAR          0xffff
#define TUN_INVALID_UNICODE_CHARACTER  0xffff
#define TUN_CHAR_SPACE                 0x20
#define TUN_UNICODE_CHAR_SPACE         0x0020

    typedef enum {
        enDab_CharSetEBU = 0,
        enDab_CharSetUTF8 = 1,
        enDab_CharSetASCII = 2,
        enDab_CharSetUCS = 3
    } tenDab_CharSet;
    
    /** Maping of HMI List ID with ADR List Id */
    typedef std::map<tU8, tU16> mapFMRdsEBU2Unicode;
    static mapFMRdsEBU2Unicode m_mapFMRdsEBU2Unicode;


    class DAB_stringUtil 
    {
    private:
    	
    	DAB_stringUtil();
        static tU16 u16ConvertFromRdsEBU2Unicode( tU8 u8RdsChar );
        static tU32 u32ConvertFromUnicode2UTF8( tU16 u16UnicodeChar );
        static tU32 u32ConvertFromRdsEBU2UTF8( tU8 u8RdsChar );
		//Trims left and right of the string and returns.
		static tU32  u32GetTrimmedString(tU32 u32LeadingSpaceCount, tU32 u32TrailingSpaceCount, const tChar *acInText, tU32 u32MaxLen,tU8 *pu8OutString);
    public:
        static tU32 u32ConvertMecaStringToUtf8( tenDab_CharSet enCharSet, const tChar *acInText,  tU8 *pu8OutString, tU32 u32MaxLen=0);
        static tU32 u32ConvertUnicodeStringToUtf8( const tChar *acInText, tU32 acInTextLen,  tU8 *pu8OutString, tU32 u32MaxLen=0);
        static tU32 u32ConvertEBUStringToUtf8( const tChar *acInText, tU32 u32TextLen,  tU8 *pu8OutString, tU32 u32MaxLen);
        static tU8 getUt8CharLengthinBytes(char c);
        static tVoid vConvertRDSPSNametoUtf8( const char *aPSName, char* s8abuffer);
        static tU32 u32ConvertFromFMRdsEBU2UTF8( tU8 u8RdsChar );
        static tU16 u16ConvertFromFMRdsEBU2Unicode( tU8 u8RdsChar );
    };
    
    template<int MAXLEN> 
    class DAB_tclUtf8String {     
    public:   
        DAB_tclUtf8String() :
            _oBuf(),
            _oStrImpl(_oBuf._au8Data,0,FALSE)
        {};
            
        tVoid vTrace() const {
            ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL,
                                "DAB_tclUtf8String vTrace: MAXLEN=%d this=0x%08x string=%s",
                                MAXLEN,
                                this,
                                _oBuf._au8Data));

        };

        DAB_tclUtf8String( const DAB_tclUtf8String& r):
            _oBuf(r._oBuf),
            _oStrImpl(_oBuf._au8Data) {};

        DAB_tclUtf8String(tenDab_CharSet enCharSet, tChar const *pu8String ) : 
            _oBuf(enCharSet, pu8String),
            _oStrImpl(_oBuf._au8Data)
        {};




        /********************************************************************
         *FUNCTION:     constructor of DAB_tclUtf8String   
         *DESCRIPTION:  destroys the DAB_tclUtf8String object   
         *PARAMETER:    none   
         *RETURNVALUE:  none
         *******************************************************************/
        virtual ~DAB_tclUtf8String(){};

        virtual DAB_tclUtf8String<MAXLEN> &operator=  ( const DAB_tclUtf8String<MAXLEN>& r) {
            if(this == &r) {
                return *this;
            }
            _oBuf=r._oBuf;
            _oStrImpl.vAssign(tclUtf8StringBase(_oBuf._au8Data));

            return *this;
        };
		
		virtual DAB_tclUtf8String<MAXLEN> &operator=  ( DAB_tclUtf8String<MAXLEN> &&r ) {
            if(this == &r) {
                return *this;
            }
            _oBuf=r._oBuf;
            _oStrImpl.vAssign(tclUtf8StringBase(_oBuf._au8Data));
			r._oBuf._u32Len=0;
			memset(r._oBuf._au8Data, '\0', MAXLEN+1);
			r._oStrImpl.vAssign(tclUtf8StringBase(r._oBuf._au8Data));
            return *this;
        };


        /********************************************************************
         *FUNCTION:     compare operator   
         *DESCRIPTION:  compare an object with another   
         *PARAMETER:    the other object   
         *RETURNVALUE:  TRUE if the left one is lower, FLASE if the right one is lower
         *******************************************************************/
        bool operator<( const DAB_tclUtf8String& r) const {
            return _oStrImpl < r._oStrImpl;
        }
        /********************************************************************
         *FUNCTION:     compare operator   
         *DESCRIPTION:  compare an object with another   
         *PARAMETER:    the other object   
         *RETURNVALUE:  TRUE 
         *******************************************************************/
        virtual tBool operator!= ( const DAB_tclUtf8String& r)  const {return !(_oBuf==r._oBuf);};

        /********************************************************************
         *FUNCTION:     compare operator   
         *DESCRIPTION:  compare an object with another   
         *PARAMETER:    the other object   
         *RETURNVALUE:  FALSE 
         *******************************************************************/
        virtual tBool operator== ( const DAB_tclUtf8String& r) const {return _oBuf==r._oBuf;};

        /********************************************************************
         *FUNCTION:     pcu8GetString  
         *DESCRIPTION:  give read access pointer to data
         *PARAMETER:    none  
         *RETURNVALUE:  pointer to the data 
         *******************************************************************/
        virtual tChar const *pcGetCString() const {   return pcu8GetString();};
        tChar const *pcu8GetString() const {   return (tChar const*)_oBuf._au8Data; };

    private:
        template<int MAXBUFLEN> 
        class tclBuf {
        public:            
            tclBuf() {
				for(tU16 u16Index=0;u16Index<MAXBUFLEN+1;u16Index++){
					_au8Data[u16Index]=0;
				}
				_u32Len =0;
            }
            tclBuf(tenDab_CharSet enCharSet, tChar const *pu8String) {
                _u32Len=DAB_stringUtil::u32ConvertMecaStringToUtf8(enCharSet, pu8String, _au8Data, MAXBUFLEN);
            }
            tclBuf(tclBuf const &roOther):
                _u32Len(roOther._u32Len)
            {
                memcpy(_au8Data, roOther._au8Data, MAXBUFLEN+1);
            }
                
            bool operator==(tclBuf const &r) const {
                if (_u32Len != r._u32Len) {
                    return FALSE;
                }
                return OSAL_s32MemoryCompare(_au8Data, r._au8Data, _u32Len) ? FALSE : TRUE;
            }
            tclBuf &operator=(tclBuf const &r) {
                if(this == &r) {
                    return *this;
                }
                _u32Len = r._u32Len;
                memcpy(_au8Data, r._au8Data, MAXBUFLEN+1);
                return *this;
            }
			
			tclBuf &operator=(tclBuf &&r) {
                if(this == &r) {
                    return *this;
                }
                _u32Len = r._u32Len;
                memcpy(_au8Data, r._au8Data, MAXBUFLEN+1);
				r._u32Len = 0;
				memset(r._au8Data, '\0', MAXBUFLEN+1);
                return *this;
            }


            tU8 _au8Data[MAXBUFLEN+1];
            tU32 _u32Len;
        };
        /********************************************************************
         *FUNCTION:     compare operator   
         *DESCRIPTION:  compare an object with another   
         *PARAMETER:    the other object   
         *RETURNVALUE:  TRUE if the right one is lower, FLASE if the left one is lower
         *******************************************************************/
        virtual tBool operator>  ( const DAB_tclUtf8String& r)  const {return !(*this<r);};



                
    protected:
        tclBuf<MAXLEN> _oBuf;
        tclUtf8StringBase _oStrImpl;
    };









} // namespace DAB

#endif 
