/************************************************************************
 * FILE:        fc_dabtuner_prioQueue.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of fc_dabtuner_prioQueue
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#ifndef FC_DABTUNER_PRIOQUEUE_HPP
#define FC_DABTUNER_PRIOQUEUE_HPP

#include <list>


#include "fc_dabtuner_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL
#include "trcGenProj/Header/fc_dabtuner_prioQueue.hpp.trc.h"
#endif

typedef enum {
    DAB_eMsgPrioHighest,
    DAB_eMsgPrioNormal,
    DAB_eMsgPrioLowest,
    DAB_eMsgNumPrios
} DAB_tenMsgPrio;

template <class M>
class DAB_tclPrioQueue {
    typedef std::list<M> DAB_tMsgQueue;

 public:

    DAB_tclPrioQueue():
        _enMaxPrio(DAB_eMsgNumPrios),
        _bProtected(FALSE)
    {};

    ~DAB_tclPrioQueue() {
        clear();
    };

    tVoid push(M const &oMsg, DAB_tenMsgPrio ePrio=DAB_eMsgPrioNormal) {
        vLock();

        if (((tU32)ePrio >=  (tU32)DAB_eMsgNumPrios)) {
            ETG_TRACE_USR4(("DAB_tclPrioQueue::push: invalid prio %d", ePrio));
            ePrio = DAB_eMsgNumPrios;
        }
        _aMsgQueues[ePrio].push_back(oMsg);
        if (ePrio<_enMaxPrio) {
            _enMaxPrio=ePrio;
        }
        ETG_TRACE_USR4(("DAB_tclPrioQueue::push: new Size=%d",
                        size(FALSE)));
        vUnLock();
    };


    tVoid pop_front() {
		vLock();
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)DAB_eMsgNumPrios;u32ListNr++) {
            DAB_tMsgQueue *pQ=&_aMsgQueues[u32ListNr];
            if(!pQ->empty()) {
                pQ->pop_front();
                break;
            } else {
                _enMaxPrio=(DAB_tenMsgPrio)(u32ListNr+1);
            }
        }
        ETG_TRACE_USR4(("DAB_tclPrioQueue::pop_front: new Size=%d",
                        size(FALSE)));
		vUnLock();


    }

    M const &front() {
        static M oDummyRes;
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)DAB_eMsgNumPrios;u32ListNr++) {
            DAB_tMsgQueue *pQ=&_aMsgQueues[u32ListNr];
            if(!pQ->empty()) {
                return pQ->front();
            }
        }
        return oDummyRes;
        
    }
    tBool bPop(M *pMsg) {
        tBool bRes=FALSE;
        if (OSAL_NULL==pMsg) {
            ETG_TRACE_USR4(("DAB_tclPrioQueue::pop: NULL_POINTER"));
            return FALSE;
        }
        vLock();
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)DAB_eMsgNumPrios;u32ListNr++) {
            DAB_tMsgQueue *pQ=&_aMsgQueues[u32ListNr];
            if(!pQ->empty()) {
                *pMsg=pQ->front();
                pQ->pop_front();
                bRes=TRUE;
                break;
            } else {
                _enMaxPrio=(DAB_tenMsgPrio)(u32ListNr+1);
            }
        }
        ETG_TRACE_USR4(("DAB_tclPrioQueue::pop: new Size=%d",
                        size(FALSE)));
        vUnLock();
        return bRes;
    };

    tVoid clear() {
        vLock();
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)DAB_eMsgNumPrios;u32ListNr++) {
            _aMsgQueues[u32ListNr].clear();
        }
        _enMaxPrio=DAB_eMsgNumPrios;
        vUnLock();
    };

    tU32 size(tBool bLock=TRUE) const {
        if (bLock) {
            vLock();
        }
        tU32 u32Size=0;
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)DAB_eMsgNumPrios;u32ListNr++) {
           u32Size+= (tU32)(_aMsgQueues[u32ListNr].size());
        }
        if (bLock) {
            vUnLock();
        }
        return u32Size;
    };

    bool empty() {
        return size()==0;
    };

    tVoid vSetProtection(tChar const *pcSemName=OSAL_NULL) {
        if (_bProtected) {
            _oSem.vClose();
            _bProtected=FALSE;
        }
        if (OSAL_NULL!=pcSemName) {
            _oSem.vOpen(pcSemName);
            _bProtected=TRUE;
        }
    }
    tVoid vLock() const {
        if (_bProtected) {
            (const_cast<dab_sem *>(&_oSem))->vGet();
        }
    }
    tVoid vUnLock() const {
        if (_bProtected) {
            (const_cast<dab_sem *>(&_oSem))->vPost();
        }
    }
private:
    DAB_tMsgQueue _aMsgQueues[DAB_eMsgNumPrios+1];
    DAB_tenMsgPrio _enMaxPrio; // contains prio of non-empty queue with highest priority
    dab_sem _oSem;
    tBool _bProtected;

};

#endif
