/************************************************************************
* FILE:         DAB_ServiceBase.cpp
* PROJECT:      g3g
* SW-COMPONENT: FC_DABTUNER
*----------------------------------------------------------------------
*
* DESCRIPTION: Definition of generic handlers to send method result,
 *         service handlers, update clients, send status message to clients.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      	| Author                      | Modification
* 06.11.2009    | Ruchi Mishra(RBEI/ECG3)      |  Original version
*
*************************************************************************/


#include "fc_dabtuner_util.h"
#include "fc_dabtuner_mainProxy.h"

#include "fc_dabtuner_serviceBase.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_SRV_BASE
#include "trcGenProj/Header/fc_dabtuner_serviceBase.cpp.trc.h"
#endif

/*
Required:
1.) Dispatch on fid+opCode
2.) automatic handling of upreg(->get)+relupreg
3.) Broadcast(fid) to registered clients

*/
using namespace DAB;



trServerConfig::trServerConfig(tU16 u16ServiceId, 
                               tU16 u16AppId, 
                               char const *pcName,
                               tU16 u16MajorVersion, 
                               tU16 u16MinorVersion,
                               tU16 u16PatchVersion):
    _u16ServiceId(u16ServiceId),
    _u16AppId(u16AppId),
    _pcName(pcName),
    _u16MajorVersion(u16MajorVersion),
    _u16MinorVersion(u16MinorVersion),
    _u16PatchVersion(u16PatchVersion),
    _bAvail(FALSE){};

/*---------------------------------------------------------------------------*/
/*     FUNCTION:    DAB_ServiceBase
*/
/*!    @brief      constructor
*                  
*     @Param       poMain, u16ServiceId
*     @return      None
*    Ver History:
*    Initial version:
*    27.10.2009                                      Ruchi Mishra(RBEI/ECG3)
*/
/*---------------------------------------------------------------------------*/
DAB_ServiceBase::DAB_ServiceBase() {

};
/*---------------------------------------------------------------------------*/
/*     FUNCTION:    DAB_ServiceBase
*/
/*!    @brief      destructor
*                  
*     @Param       poMain, u16ServiceId
*     @return      None
*    Ver History:
*    Initial version:
*    27.10.2009                                      Ruchi Mishra(RBEI/ECG3)
*/
/*---------------------------------------------------------------------------*/
DAB_ServiceBase::~DAB_ServiceBase() {
}

tVoid DAB_ServiceBase::vInit() {
    trServerConfig &roConf=roGetConfig();
    roConf._oNotSem.vOpen(roConf._pcName);
}

tVoid DAB_ServiceBase::vDeInitServiceBase() {
    trServerConfig &roConf=roGetConfig();

   roConf._u16ServiceId = 0; 
   roConf._bAvail=FALSE;
   for (map<tU16, trSrvFidData>::iterator fidIter=roConf._mapFidData.begin();
        //lint -save -esym(1702, operator!=) both member and non-member ops
        fidIter!=roConf._mapFidData.end();
        //lint -restore
        ++fidIter) {
       (*fidIter).second._lClients.clear();
   }
   roConf._mapFidData.clear();
   roConf._oNotSem.vClose();
}

tBool DAB_ServiceBase::bGetServiceVersion (tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion)
{
    trServerConfig &roConf=roGetConfig();

   rfu16MajorVersion = roConf._u16MajorVersion;
   rfu16MinorVersion = roConf._u16MinorVersion;
   rfu16PatchVersion = roConf._u16PatchVersion;

   ETG_TRACE_USR4(("DAB_ServiceBase::bGetServiceVersion: %d.%d.%d App-Id=%d Service-Id=%d", rfu16MajorVersion, rfu16MinorVersion, rfu16PatchVersion, roConf._u16AppId, roConf._u16ServiceId )); 

   return TRUE;
}

tU16 DAB_ServiceBase::u16GetServiceId() {
    return roGetConfig()._u16ServiceId;
}





tVoid DAB_ServiceBase::vSetAvailability(tBool bAvail) {
    trServerConfig &roConf=roGetConfig();

    roConf._bAvail=bAvail;

    ETG_TRACE_USR4(("DAB_ServiceBase::vSetAvailability: bAvail: %d ServiceId= %x", bAvail, roConf._u16ServiceId )); 

    DAB_mainProxy::vServiceAvailabilityChanged(roConf._u16ServiceId, 
                                               bAvail ? AMT_C_U8_SVCSTATE_AVAILABLE : AMT_C_U8_SVCSTATE_NOT_AVAILABLE);
}


tVoid DAB_ServiceBase::vOnUnknownMessage(amt_tclBaseMessage* poMessage) {
	amt_tclServiceData oServiceData(poMessage);
      
    ETG_TRACE_COMP(("%10s::vOnUnknownMessage:fid=0x%x opCode=%d MsgType=0x%x",
                    roGetConfig()._pcName,
                    oServiceData.u16GetFunctionID(),
                    oServiceData.u8GetOpCode(),
                    poMessage->u8GetType())); 

}
/* todo provide implementation of some helpers as in serviceHandler of fc,
No separate class is needed, but you may add them to a separated cpp-file(DAB_ServiceHandler.cpp)
*/




tVoid DAB_ServiceBase::vSendError(trAdressing const *prAdressing, tU8 u8ErrorCode) {
    trServerConfig &roConf=roGetConfig();

    if (roConf._u16AppId == prAdressing->_u16AppId ){
        return;
    }

    gm_tclU8Message oErrorMsg( 
                              roConf._u16AppId,    // Source app-ID: DAB_APP_ID       
                              prAdressing->_u16AppId,    // Dest.  app-ID: 
                              prAdressing->_u16RegId, 
                              prAdressing->_u16CmdCount, 
                              roConf._u16ServiceId, 
                              prAdressing->_u16Fid, 
                              AMT_C_U8_CCAMSG_OPCODE_ERROR );

  oErrorMsg.vSetByte( u8ErrorCode );
  DAB_mainProxy::enPostMsg(&oErrorMsg);
}


tVoid DAB_ServiceBase::vNotify(fi_tclMessageBase const & roSendFiObj) {
    trServerConfig &roConf=roGetConfig();
    ail_tenCommunicationError enComError;
    tU16 u16InvalidRegId=0;
    tBool bFoundInvalidRegId=FALSE;
    map<tU16, trSrvFidData>::iterator fidIter =  roConf._mapFidData.find(roSendFiObj.u16GetFunctionID());
    //lint -save -esym(1702, operator!=) both member and non-member ops
    if (fidIter != roConf._mapFidData.end()) {
        //lint -restore
        trSrvFidData &roFidData = (*fidIter).second;
        roConf._oNotSem.vGet();
        for (list<trAdressing>::iterator clientIter=roFidData._lClients.begin();
             //lint -save -esym(1702, operator!=) both member and non-member ops
             clientIter!=roFidData._lClients.end();
             //lint -restore
             ++clientIter) {
            if ((*clientIter)._u16AppId!= roConf._u16AppId) {
                enComError =enSendFiMessage(&(*clientIter), roSendFiObj);
                if (AIL_EN_N_REGISTRATION_DENIED==enComError){
                    bFoundInvalidRegId = TRUE;
                    u16InvalidRegId=(*clientIter)._u16RegId;  
                }
            }
            else {
                ETG_TRACE_USR1(("vNotify: No Notification of self fid=0x%x opcode=0x%x",
                                roSendFiObj.u16GetFunctionID(), roSendFiObj.u8GetOpCode())); 
            }
        }
        roConf._oNotSem.vPost();
    }
    if (bFoundInvalidRegId) {
        vUnregisterRegId(u16InvalidRegId);
    }
}
tVoid DAB_ServiceBase::vNotifyOnce(fi_tclMessageBase const & roSendFiObj) {
    vNotify(roSendFiObj);
    vReleaseNotification(DAB_APPID_ALL, roSendFiObj.u16GetFunctionID());
}


// craete generic fns to send fi-objects according to ServiceHandler of fc, use fn-template if neccessary
ail_tenCommunicationError DAB_ServiceBase::enSendFiMessage(trAdressing const *prAdressing, fi_tclMessageBase const & roSendFiObj) {
    DAB_ASSERT_RETURN_VAL(OSAL_NULL != prAdressing, AIL_EN_N_INVALID_PARAMETER);
    trServerConfig &roConf=roGetConfig();

    if (!roConf._bAvail) {
        return AIL_EN_N_NO_ERROR;
    }
   fi_tclVisitorMessage oOutVisitorMsg( roSendFiObj.corfoGetTypeBase() );

   /* Set the CCA message information */
   oOutVisitorMsg.vInitServiceData
        (
          roConf._u16AppId,                  /* Source app-ID   */
          prAdressing->_u16AppId,     /* Dest. app-ID    */
          AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,        /* stream type     */ 
          0,                    /* stream counter  */
          prAdressing->_u16RegId,                  /* Register ID     */
          prAdressing->_u16CmdCount,                  /* Command counter */
          roConf._u16ServiceId,                  /* Service-ID      */
          roSendFiObj.u16GetFunctionID(),                  /* Function-ID     */
          roSendFiObj.u8GetOpCode() ,             /* Opcode          */
          0,                    /* ACT             */
          0,                    /* Source sub-ID   */
          0                     /* Dest. sub-ID    */
        );

   return DAB_mainProxy::enPostMsg(&oOutVisitorMsg);
   
}


tVoid DAB_ServiceBase::vStoreNotification(trAdressing const *prAdressing) {
    trServerConfig &roConf=roGetConfig();
   ETG_TRACE_COMP(("DAB_ServiceBase::vStoreNotification: (%10s)opCode=%d u16Fid=%d u16AppId=%d",
                   roConf._pcName,
                   prAdressing->_u8OpCode,
                   prAdressing->_u16Fid,
                   prAdressing->_u16AppId)); 
   roConf._oNotSem.vGet();
   trSrvFidData &rFidData =roConf._mapFidData[prAdressing->_u16Fid];
   list<trAdressing> &lClients = rFidData._lClients;
   list<trAdressing>::iterator clientIter=find_if(lClients.begin(),
                                                  lClients.end(),
                                                  trAppIdFinder(prAdressing->_u16AppId));
    //lint -save -esym(1702, operator!=) both member and non-member ops
   if (clientIter==lClients.end()) {
       lClients.push_back(*prAdressing);
   }
   //lint -restore
   
   roConf._oNotSem.vPost();
}




tVoid DAB_ServiceBase::vReleaseNotification(tU16 u16AppId, tU16 u16Fid) {
    trServerConfig &roConf=roGetConfig();

    roConf._oNotSem.vGet();
    map<tU16, trSrvFidData>::iterator fidIter=roConf._mapFidData.find(u16Fid);
    //lint -save -esym(1702, operator!=) both member and non-member ops

    if (fidIter!=roConf._mapFidData.end()) {
        //lint -restore

        list<trAdressing> &lClients = (*fidIter).second._lClients;
        if (u16AppId == DAB_APPID_ALL) {
            lClients.clear();
        }
        list<trAdressing>::iterator clientIter=find_if(lClients.begin(),
                                                       lClients.end(),
                                                       trAppIdFinder(u16AppId));
        //lint -save -esym(1702, operator!=) both member and non-member ops
        if (clientIter!=lClients.end()) {
            //lint -restore
            lClients.erase(clientIter);
        }
    }
    roConf._oNotSem.vPost();
}

tVoid DAB_ServiceBase::vUnregisterRegId(tU16 u16RegId) {
    trServerConfig &roConf=roGetConfig();
        

    //lint -save -esym(1702, operator!=) both member and non-member ops
    for (map<tU16, trSrvFidData>::iterator fidIter=roConf._mapFidData.begin();
         //lint -restore
         fidIter!=roConf._mapFidData.end();
         ++fidIter) {
        roConf._oNotSem.vGet();
        
        list<trAdressing> &lClients = (*fidIter).second._lClients;
        list<trAdressing>::iterator clientIter=find_if(lClients.begin(),
                                                       lClients.end(),
                                                       trRegIdFinder(u16RegId));
        //lint -save -esym(1702, operator!=) both member and non-member ops
        if (clientIter!=lClients.end()) {
        //lint -restore
            lClients.erase(clientIter);
        }

        roConf._oNotSem.vPost();
        
    }
}


tVoid  DAB_ServiceBase::vTraceRequest(tPCChar name, trAdressing const *prAdressing)
{
   if (OSAL_NULL != prAdressing)
   {
      ETG_TRACE_USR3(("%10s::%10s: OpCode: %d, Fid: %d, Src: %d",
                      roGetConfig()._pcName,
                      name,
                      prAdressing->_u8OpCode,
                      prAdressing->_u16Fid,
                      prAdressing->_u16AppId)); 
   }
}

/*tVoid xxx_vTracefnId(tU16 fnId) {
  ETG_TRACE_FATAL(("xxx_vTracefnId=0x%04x", fnId));
}*/

