/************************************************************************

 * FILE:         fc_dabtuner_main.cpp
 * PROJECT:      g3g 
 * SW-COMPONENT: FC_DABTUNER
 *----------------------------------------------------------------------
 *
 * DESCRIPTION: DABTuner main application class
 *              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:      
 * Date      | Author                      | Modification
 * 14.06.07  | CM-DI/PJ-VW36-Liebanas      | initial version
 *
 *************************************************************************/

    
//-----------------------------------------------------------------------------
// file define
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------


#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#include "fc_dabtuner_util.h"
#include "dabdrv_main.hpp"
#include "fc_dabtuner_main.h"
#include "fc_dabtuner_config.hpp"

#include "fc_dabtuner_service.h"

#include "fc_dabtuner_DiagLibHandler.h"
#include "fc_dabtuner_ttfisCmd.h"
#include "dabdrv_main.hpp"

#include "fc_dabtuner_clientSpm.h"
#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK
#include "fc_dabtuner_clientVdClock.h"
#endif
#include "fc_dabtuner_clientTuner.h"
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_INF4CV
#include "fc_dabtuner_clientDiagLog.h"
#endif
#ifdef KDS_FOR_G3G_VARIANT
#include "fc_dabtuner_clientAudio.h"
#endif
//KDS
#define SYSTEM_S_IMPORT_INTERFACE_KDS_DEF
#include "system_pif.h"

#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
#include "AarsDBusProxy/tuner_gio_dbus_handler.h"
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_MAIN 
#include "trcGenProj/Header/fc_dabtuner_main.cpp.trc.h"
#endif

using namespace DAB;


tBool           fc_dabtuner_tclApp::m_bAppClosed     = FALSE;

/*************************************************************************
 *
 * FUNCTION: fc_dabtuner_tclApp::fc_dabtuner_tclApp()
 * 
 * DESCRIPTION: 
 *
 *  Constructor: - Initialize AMT, create Sem, create notification table 
 * 
 *
 * PARAMETER: void
 *
 * RETURNVALUE: none
 *
 *************************************************************************/
fc_dabtuner_tclApp::fc_dabtuner_tclApp()
{
	/* --- initialize the AT/CCA framework for this component --- */

	/**Vnd4kor: to sole prio 2 lint warning*/
	_bDabEnable = FALSE;
    DAB_u32CurTraceCheck++;
	ETG_TRACE_USR4(("--------- fc_dabtuner_tclApp::fc_dabtuner_tclApp()"));


	amt_bInit();
    _u32OldAppState = AMT_C_U32_STATE_UNINITALIZED;
    _u32AppState = AMT_C_U32_STATE_UNINITALIZED;
#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
    m_pGioDbusHandler = nullptr;
#endif
    if (!DAB_GET_CONFIG().bDabModulePresent) {
        ETG_TRACE_USR1(("fc_dabtuner_tclApp::fc_dabtuner_tclApp(): No DAB-Module"));
        return;
    }
    _bDabEnable=DAB_GET_CONFIG().bDabEnable;
	fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
#ifdef FEATURE_PSA_RCC
	if(_bDabEnable)
	{		
		if(enConfigCountryZone_USA != (tenConfigCountryZone)poConfig->u8GetCountryZone())
		{
			if(enConfigBandActivation_DeactiveAll == (tenConfigBandActivation)poConfig->u8GetActiveBand())
				_bDabEnable = FALSE;
		}
		else 
			_bDabEnable = FALSE;
	}
#endif

	// Create semaphore to avoid parallel processing
    _oProcessingSem.vOpen(DABTUNER_PROCESSMSG_SEM_NAME);

    DAB_Profiler rProfiler("fc_dabtuner_tclApp");
    // start dispatcher-thread an dispatcher-functionality
    DAB_tclDispatcher::vSetDefaultDispathcher(OSAL_NEW(DAB_tclDispatcher));
    DAB_tclDispatcher::poGetDefaultDispatcher()->vInit(DAB_GET_CONFIG().u32DispatcherPrio, &_oProcessingSem);

    if (_bDabEnable) {
        _oMhQ.poAdd(dabdrv_main::instance());
		#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
	if(poConfig != nullptr) {
		if(poConfig->bIsSBRVariant() ){
			m_pGioDbusHandler = tuner_gio_dbus_handler::instance();
		}
	}
#endif
        /* create the cca service handler. 
           It analyses the received cca messages and calls the corresponding handler function --- */
        _oMhQ.poAdd(fc_dabtuner_tclCCAService::instance());
        _qServices.push_back(fc_dabtuner_tclCCAService::instance());



        _oMhQ.poAdd(fc_dabtuner_DiagLibHandler::instance());
        _qServices.push_back(fc_dabtuner_DiagLibHandler::instance());  
        /* create the cca message handlers. 
           It analyses the received cca messages and calls the corresponding handler function --- ~~~ */
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_INF4CV
        _oMhQ.poAdd(fc_dabtuner_clientDiagLog::instance());
        _qClients.push_back(fc_dabtuner_clientDiagLog::instance());
#endif

        _oMhQ.poAdd(fc_dabtuner_clientSpm::instance());
        _qClients.push_back(fc_dabtuner_clientSpm::instance());

        _oMhQ.poAdd(fc_dabtuner_clientTuner::instance());
        _qClients.push_back(fc_dabtuner_clientTuner::instance());
		
		#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK
		_oMhQ.poAdd(fc_dabtuner_clientVdClock::instance());
        _qClients.push_back(fc_dabtuner_clientVdClock::instance());
		#endif
        
        // disable FC_AUDIO interface
#ifdef DAB_OLD_AUDIO_HANDLING
        _oMhQ.poAdd(fc_dabtuner_clientAudio::instance());
        _qClients.push_back(fc_dabtuner_clientAudio::instance());
#endif //#if DAB_OLD_AUDIO_HANDLING
        
        
        // Ttfis Cmd Handler
        _oMhQ.poAdd(fc_dabtuner_tclTrace::instance());
    } else {
        _oMhQ.poAdd(fc_dabtuner_DiagLibHandler::instance());
        _qServices.push_back(fc_dabtuner_DiagLibHandler::instance());

    }

    rProfiler.u32GetMs();


	//Board ID to see whether it is a PLUS variant is checked by the RadioLib, I just receive the response to
	// ComponentStatus to know if there is a DAB module or not.

}

/*************************************************************************
 *
 * FUNCTION: fc_dabtuner_tclApp::~fc_dabtuner_tclApp()
 * 
 * DESCRIPTION: destructor, frees all ressources 
 *             ( if not done by vOnApplicationClose before )
 *
 * PARAMETER: void
 * 
 * RETURNVALUE: none
 *
 *************************************************************************/
//lint -esym(1551, fc_dabtuner_tclApp::~fc_dabtuner_tclApp) prio3 Function may throw exception '...' in destructor
fc_dabtuner_tclApp::~fc_dabtuner_tclApp()
{
    DAB_u32CurTraceCheck++;
    static DAB_Profiler rProfiler("~fc_dabtuner_tclApp");

	/* ---
       Normally the SPM calls vOnApplicationClose before this application - object is
       destroyed. Here we check if the ressources are already freed by vOnApplicationClose else 
       we free the ressources
       --- */ 
	if(!m_bAppClosed) 
	{
		vOnApplicationClose(); //lint !e1551 !e1506: Function may throw exception; Call to virtual function 'fc_dabtuner_tclApp::vOnApplicationClose(void)' within a constructor or destructor
	}
    rProfiler.u32GetMs();

}

tVoid fc_dabtuner_tclApp::vFreeMem() {

    _oMhQ.vClear();

    OSAL_DELETE DAB_tclDispatcher::poGetDefaultDispatcher();
    OSAL_DELETE fc_dabtuner_config::instance();


}
/*************************************************************************
 *
 * FUNCTION: tVoid fc_dabtuner_tclApp::vOnApplicationClose()
 * 
 * DESCRIPTION: releases all resources and sends close - message
 *
 * PARAMETER: void
 *
 * RETURNVALUE: void
 *
 *************************************************************************/
tVoid fc_dabtuner_tclApp::vOnApplicationClose()
{  
    DAB_u32CurTraceCheck++;

#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
		   if(m_pGioDbusHandler!= NULL)
		   {
			   m_pGioDbusHandler->vDeInit();
			   delete m_pGioDbusHandler;
			   m_pGioDbusHandler = NULL;
		   }
#else

    if (m_bAppClosed) {
        return;
    }
	m_bAppClosed = TRUE;
    ETG_TRACE_USR1(("--------- fc_dabtuner_tclApp::vOnApplicationClose()"));
    if (DAB_GET_CONFIG().bDabModulePresent) {
        // kill dispatcher-thread
        DAB_tclDispatcher::poGetDefaultDispatcher()->vClose();
        _oProcessingSem.vGet();
        DAB_Profiler rProfiler("vOnApplicationClose");

        _oMhQ.vDeInit();
        
        
        _oProcessingSem.vClose();
        
        _qServices.clear();
        _qClients.clear();
        vFreeMem();
        rProfiler.u32GetMs();
    } else {
        ETG_TRACE_USR1(("fc_dabtuner_tclApp::vOnApplicationClose(): No DAB-Module"));
    }
#endif
	/* --- inform SPM about successful shut down --- */
	vApplicationCloseAcknowledge(AIL_C_U8_APP_END_SUCCESSFUL);
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     fc_dabtuner_tclApp::theServer
 *
 *DESCRIPTION:  create new server object
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  fc_dabtuner_tclApp* _theServer
 *
 *HISTORY:
 *27.01.07 kol2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
fc_dabtuner_tclApp* fc_dabtuner_tclApp::theServer()
{
    return instance();
}

/*************************************************************************
 *
 * FUNCTION: tBool fc_dabtuner_tclApp::bOnInit ()
 * 
 * DESCRIPTION: is called after the initialisation of the framework is finished
 *
 * PARAMETER: void
 *
 * RETURNVALUE: TRUE
 *
 *************************************************************************/
tBool fc_dabtuner_tclApp::bOnInit()
{
    DAB_u32CurTraceCheck++;
    static DAB_Profiler rProfiler("bOnInit");
    ETG_TRACE_USR1(("fc_dabtuner_tclApp::bOnInit()"));
    if (!DAB_GET_CONFIG().bDabModulePresent) {
        ETG_TRACE_USR1(("fc_dabtuner_tclApp::bOnInit(): No DAB-Module"));
        return TRUE;
    }
#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
    fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
	if(poConfig == nullptr)
	{
		return true;
	}
	if(poConfig->bIsSBRVariant())
	{
		tBool bSuccess = false;
		if(m_pGioDbusHandler!= NULL)
		{
		   ETG_TRACE_USR3(("fc_dabtuner_tclApp::bOnInit() Initializing gio handler."));
		   bSuccess = m_pGioDbusHandler->bInit("DAB_SBR_COM");
		   //dabdrv_adrIf::instance()->vDeInit();
		}
		if(!bSuccess)
		{
			ETG_TRACE_USR4(("fc_dabtuner_tclApp::bOnInit() Initializing gio handler failed"));
		}
	}
	else
	{
		ETG_TRACE_USR4(("fc_dabtuner_tclApp::bOnInit() Not SBR variant"));
	}
#endif
    ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "bOnInit: calling semGet"));
    _oProcessingSem.vGet();
    ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "bOnInit: semGet OK"));
    rProfiler.vStart();
    /* +++
       HERE YOU CAN INITIALIZE YOUR APPLICATION 
       - start your threads
       - register your call backs
       - create anything you need
       +++ */
    _oMhQ.vInit();

    ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "bOnInit: semGet semPost"));
    rProfiler.u32GetMs();

    _oProcessingSem.vPost();
	return  TRUE;
}

/*************************************************************************
 *
 * FUNCTION: tVoid fc_dabtuner_tclApp::vOnServiceState
 * 
 * DESCRIPTION: the framework uses this method to inform the appliction about 
 *              a service state change of another server
 *              this can only happen if the application has a client part and 
 *              is registered with this server
 *
 * PARAMETER: 
 *
 * RETURNVALUE: void
 *
 *************************************************************************/
tVoid fc_dabtuner_tclApp::vOnServiceState(tU16 u16ServiceId,tU16 u16ServerId,
                                          tU16 u16RegisterId,tU8  u8ServiceState,tU16 u16SubId)
{
    DAB_u32CurTraceCheck++;
    static DAB_Profiler rProfiler("vOnServiceState");

    if (DAB_GET_CONFIG().bDabModulePresent) {

        ETG_TRACE_USR4(("fc_dabtuner_tclApp::vOnServiceState():  Service Id:0x%x, New Service State:%u",
                        ETG_ENUM(SERVICE_ID, u16ServiceId),
                        ETG_ENUM(SERVICE_STATE, u8ServiceState)));
        /* --- check which service has changed --- */
        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnServiceState: calling semGet"));
        _oProcessingSem.vGet();
        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnServiceState:semGet OK"));
        rProfiler.vStart();
        for (deque<DAB::DAB_ClientBase *>::iterator iter=_qClients.begin();
             //lint -save -esym(1702, operator!=) both member and non-member ops
             iter!=_qClients.end();
             //lint -restore
             ++iter) {
            if ((*iter)->u16GetServiceId()==u16ServiceId) {
                (*iter)->vOnServiceState(
                                         u16ServiceId
                                         , u16ServerId
                                         , u16RegisterId 
                                         , u8ServiceState
                                         , u16SubId 
                                         );
            }
        }

        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnServiceState: semGet semPost"));
        rProfiler.u32GetMs();

        _oProcessingSem.vPost();

    }
    else {
        ETG_TRACE_USR1(("fc_dabtuner_tclApp::vOnServiceState(): No DAB-Module"));
    }

    /* --- call the corresponding method of the super class to give AIL the chance to do the  rest --- */
    ail_tclAppInterface::vOnServiceState(u16ServiceId,u16ServerId,u16RegisterId,u8ServiceState,u16SubId);
}

/*************************************************************************
 *
 * FUNCTION: tVoid fc_dabtuner_tclApp::vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState)
 * 
 * DESCRIPTION: * is call from SPM to inform this application about new power states
 *              * changes the state of the application
 *              * sends ServiceAvailability state to all notified clients
 *              * changes state of downloadservice(-object)  
 *
 * PARAMETER: tU32 u32OldAppState (actual state), 
 *            tU32 u32AppState (state to change to)
 *
 * RETURNVALUE: void
 *
 *************************************************************************/
/* +++ you have to implement the behavior concerning power states here +++ */
tVoid fc_dabtuner_tclApp::vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState)
{
    DAB_u32CurTraceCheck++;
    //static DAB_Profiler rProfiler("vOnNewAppState");

    ETG_TRACE_USR4(("fc_dabtuner_tclApp::vOnNewAppState(): Should switch to Applikation state: %u(APP_STATE,8)",
                    ETG_ENUM(APP_STATE, u32AppState)));
    _u32AppState=u32AppState;
    if (DAB_GET_CONFIG().bDabModulePresent) {

        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnNewAppState: calling semGet"));
        _oProcessingSem.vGet();
        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnNewAppState: semGet OK"));
		// Fix for NCG3D-97241 
       // rProfiler.vStart(u32AppState);


        /* --- analyse the state we have to go to  --- */
        if(u32AppState == u32OldAppState)
        {
            //switching to same state
            /* ---  the server has already entered this state, so it has nothing to do --- */
        }
        else
        {
            _u32OldAppState=u32OldAppState;

            trMsgSrvCmdNewAppState oNewAppState(u32OldAppState, u32AppState);
            DAB_vCallMsg(&oNewAppState);

            for (deque<DAB::DAB_ServiceBase *>::iterator iter1=_qServices.begin();
                 //lint -save -esym(1702, operator!=) both member and non-member ops
                 iter1!=_qServices.end();
                 //lint -restore
                 ++iter1) {
                (*iter1)->vOnNewAppState(u32AppState);
            }
            for (deque<DAB::DAB_ClientBase *>::iterator iter2=_qClients.begin();
                 //lint -save -esym(1702, operator!=) both member and non-member ops
                 iter2!=_qClients.end();
                 //lint -restore
                 ++iter2) {
			    ETG_TRACE_USR4(("fc_dabtuner_tclApp::vOnNewAppState()Client Request"));
                (*iter2)->vOnNewAppState(u32AppState);
            }



        }//else (u32AppState != u32OldAppState)

        //rProfiler.u32GetMs();
        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnNewAppState: calling semGet semPost"));
        _oProcessingSem.vPost();
        //if (u32AppState != AMT_C_U32_STATE_OFF) {
            /* --- create and send an acknowledge message so the SPM knows you have reached the new state --- */
        ETG_TRACE_USR4(("fc_dabtuner_tclApp::vAppStateChanged(): Applikation state: %u(APP_STATE,8)",
                        ETG_ENUM(APP_STATE, u32AppState)));
        if (u32AppState == AMT_C_U32_STATE_OFF) {
            vAppStateChanged(u32AppState,10000);
        }
        else {
            vAppStateChanged(u32AppState,0);
        //} else {
        //    vAppStateChanged(u32AppState,40000);
        }
    } else {
        ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_SYNC, "fc_dabtuner_tclApp::vOnNewAppState(): No DAB-Module"));
        /* --- create and send an acknowledge message so the SPM knows you have reached the new state --- */
        ETG_TRACE_USR4(("fc_dabtuner_tclApp::vAppStateChanged(): Applikation state: %u(APP_STATE,8)",
                            ETG_ENUM(APP_STATE, u32AppState)));
        vAppStateChanged(u32AppState,0);
    }


} 


// called when a client has unregistered from a service; clear 
// notification table entries then.
tVoid fc_dabtuner_tclApp::vOnUnregister (tU16 u16ServiceId, tU16 u16RegisterId) {
    DAB_u32CurTraceCheck++;
    if (DAB_GET_CONFIG().bDabModulePresent) {
        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnUnregister: calling semGet"));
        _oProcessingSem.vGet();
        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnUnregister: semGet OK"));

        for (deque<DAB::DAB_ServiceBase *>::iterator iter=_qServices.begin();
             //lint -save -esym(1702, operator!=) both member and non-member ops
             iter!=_qServices.end();
             //lint -restore
             ++iter) {
            if ((*iter)->u16GetServiceId()==u16ServiceId) {
                (*iter)->vUnregisterRegId(u16RegisterId);
            }
        }
        ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnUnregister: calling semGet semPost"));
        _oProcessingSem.vPost();
    } else {
        ETG_TRACE_USR1(("fc_dabtuner_tclApp::vOnUnregister(): No DAB-Module"));
    }
    // call to base class for completness
    ail_tclAppInterface::vOnUnregister (u16ServiceId, u16RegisterId);
}

ail_tenCommunicationError fc_dabtuner_tclApp::enPostMsg(amt_tclServiceData* poMessage) const {
    ail_tenCommunicationError enCommError = AIL_EN_N_NO_ERROR;
    enCommError=fc_dabtuner_tclApp::instance()->enPostMessage(poMessage);
    if (AIL_EN_N_NO_ERROR != enCommError) {
        ETG_TRACE_USR4(("Result PostMessage: %d",
                        enCommError)); 
        poMessage->bDelete();
    }
    return enCommError;
}

/*************************************************************************
 *
 * FUNCTION: tVoid fc_dabtuner_tclApp::vOnNewMessage (amt_tclBaseMessage* poMessage)
 * 
 * DESCRIPTION: * casts Message to amt_tclServiceData (if legal)
 *              * gives amt_tclServiceData - Message to service for analyzing
 *              * ignores all unexpected messsages
 *
 * PARAMETER: amt_tclBaseMessage* poMessage ( Message to analyse)
 *
 * RETURNVALUE: void
 *
 *************************************************************************/
tVoid fc_dabtuner_tclApp::vOnNewMessage(amt_tclBaseMessage* poMessage) 
{
    DAB_u32CurTraceCheck++;
    static DAB_Profiler rProfiler("vOnNewMessage");

    ETG_TRACE_USR4(("fc_dabtuner_tclApp::vOnNewMessage(): Entry function"));
    if (!DAB_GET_CONFIG().bDabModulePresent) {
        ETG_TRACE_USR1(("fc_dabtuner_tclApp::vOnNewMessage(): No DAB-Module"));
        poMessage->bDelete();
        return;
    }

	/* ---  check if the received message is a AMT Service data message --- */
    ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnNewMessage: calling semGet"));
    _oProcessingSem.vGet();
    ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnNewMessage: semGet OK"));
    rProfiler.vStart((tU32)((uintptr_t)poMessage));

    ETG_TRACE_USR4(("fc_dabtuner_tclApp::vOnNewMessage(): type=%d", poMessage->u8GetType()));

	if(poMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA)
	{

        /* --- cast the received message to a AMT Service data message so we can access the standard members --- */
        amt_tclServiceData oServiceData(poMessage);
      
        tU16 u16ServiceId = oServiceData.u16GetServiceID();
        ETG_TRACE_USR4(("fc_dabtuner_tclApp::vOnNewMessage(): u16ServiceId=0x%x fid=0x%x opCode=0x%x", 
                        u16ServiceId, 
                        oServiceData.u16GetFunctionID(), 
                        oServiceData.u8GetOpCode()));

        /* --- one server can offer more than one service and it can act as a client. So analyse the ServiceID
           to know which service handler we have to call. --- */
        tBool bDone=FALSE;
        /* +++ enter your Service ID here +++ */
        for (deque<DAB::DAB_ServiceBase *>::iterator iterSrv=_qServices.begin();
             //lint -save -esym(1702, operator!=) both member and non-member ops
             iterSrv!=_qServices.end();
             //lint -restore
             ++iterSrv) {
            if ((*iterSrv)->u16GetServiceId()==u16ServiceId) {
                (*iterSrv)->vDispatchMessage_(&oServiceData);
                bDone=TRUE;
                break;
            }
        }
      
        if (!bDone) 
        {
            //lint -save -esym(1702, operator!=) both member and non-member ops
            for (deque<DAB::DAB_ClientBase *>::iterator iterClient=_qClients.begin();
                 iterClient!=_qClients.end();
                 ++iterClient)
                //lint -restore
            {
                if ((*iterClient)->u16GetServiceId()==u16ServiceId) {
                    ETG_TRACE_USR4(("fc_dabtuner_tclApp::vOnNewMessage(): u16ServiceId=0x%x",u16ServiceId)); 
                    (*iterClient)->vDispatchMessage_(&oServiceData);
                    break;
                }
            }
        }


	}
	else
	{
		/* +++ we have received a message with an unknown format. 
           this is an error so make a trace or assert here +++ */
        ETG_TRACE_ERR(("fc_dabtuner_tclApp::vOnNewMessage(): Unknown Message Type (0x%x)",
                       poMessage->u8GetType()));


		/* --- we can't do anything with this message so free the ressources --- */
		
	}

	poMessage->bDelete();
    rProfiler.u32GetMs();

    ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "vOnNewMessage: calling semGet semPost"));
    _oProcessingSem.vPost();

}

/*************************************************************************
 *
 * FUNCTION: tBool fc_dabtuner_tclApp::bGetServiceVersion (tU16 u16ServiceID, tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion)
 * 
 * DESCRIPTION: stores data about the version of this service (magics) 
 *
 * PARAMETER: * u16ServiceID: the service we should describe
 *            * rfu16MajorVersion, rfu16MinorVersion, rfu16PatchVersion: targets for storing the data
 *
 * RETURNVALUE: success: true / false
 *
 *************************************************************************/
tBool fc_dabtuner_tclApp::bGetServiceVersion (tU16 u16ServiceID, 
											  tU16& rfu16MajorVersion, 
											  tU16& rfu16MinorVersion, 
											  tU16& rfu16PatchVersion)
{
    for (deque<DAB::DAB_ServiceBase *>::iterator iter1=_qServices.begin();
        //lint -save -esym(1702, operator!=) both member and non-member ops
        iter1!=_qServices.end();
        //lint -restore
        ++iter1) {

            tU16 _u16ServiceID = (*iter1)->u16GetServiceId();
             
            ETG_TRACE_USR4(("fc_dabtuner_tclApp::bGetServiceVersion(): for u16ServiceId=0x%x in list _u16ServiceId=0x%x",u16ServiceID, _u16ServiceID));

            if (u16ServiceID == _u16ServiceID) {
                ETG_TRACE_USR4(("fc_dabtuner_tclApp::bGetServiceVersion(): return _u16ServiceId=0x%x",_u16ServiceID));
        return (*iter1)->bGetServiceVersion(rfu16MajorVersion, rfu16MinorVersion, rfu16PatchVersion);
        }
    }

    /* +++ we have received a request with an unkown service ID.
       this is an error so make a trace or assert here +++ */
    return FALSE;
}


/************************************************************************
 *
 * FUNCTION: tBool fc_dabtuner_tclApp::bOnWatchdog()
 * 
 * DESCRIPTION: ????
 *
 * PARAMETER: void
 *
 * RETURNVALUE: true
 *
 *************************************************************************/
tBool fc_dabtuner_tclApp::bOnWatchdog()
{
	return TRUE;
}




tVoid fc_dabtuner_tclApp::vOnAppStateReached(tU32 u32AppState) {
    ETG_TRACE_USR4(("vOnAppStateReached: %d (OFF=%d)",
                        u32AppState, AMT_C_U32_STATE_OFF));
    if (u32AppState == AMT_C_U32_STATE_OFF) {
        ETG_TRACE_USR4(("fc_dabtuner_tclApp::vAppStateChanged(): Applikation state: %u(APP_STATE,8)",
                            ETG_ENUM(APP_STATE, u32AppState)));
        vAppStateChanged(u32AppState,0);
    }
}
