/************************************************************************
* FILE:         fc_dabtuner_config.cpp
* PROJECT:      g3g
* SW-COMPONENT: DAB Tuner
*----------------------------------------------------------------------
*
* DESCRIPTION:  Implementation of fc_dabtuner_config
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author             | Modification

*
*************************************************************************/


#include "fc_dabtuner_util.h"
#include "fc_dabtuner_config.hpp"

// linking registry access functions
#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define DIAGNOSTICS_KDS_ENTRY_IMPORT_INTERFACE
#include "fc_diagnosis_if.h"


#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#ifndef DABTUNER_UTEST
	#define DP_S_IMPORT_INTERFACE_FI
	#include "dp_fc_dabtuner_if.h"
	#include "dp_generic_if.h"
#endif
#include "fc_dabtuner_util.h"
#include "dabdrv_types.hpp"
#include "dabdrv_presets.hpp"
#include "dabmeca_rdm.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_MAIN
#include "trcGenProj/Header/fc_dabtuner_config.cpp.trc.h"
#endif

namespace DAB {

/*****************************************************************************
 * Constructor
 ****************************************************************************/
trKdsDabTunerParameter::trKdsDabTunerParameter( tVoid)
{
  ETG_TRACE_USR1(( " trKdsDabTunerParameter::trKdsDabTunerParameter() -> Constructor" ));
  trKdsDabTunerParameter::vSetToDefaultValues();
}

/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid trKdsDabTunerParameter::vSetToDefaultValues( tVoid)
{
  tUInt unIdx;
  ETG_TRACE_USR1(( " trKdsDabTunerParameter::vSetToDefaultValues() -> "));

  bKdsDataRead                    = FALSE;
  sTag[ 0]     = 'd';
  sTag[ 1]     = 'a';
  sTag[ 2]     = 'b';
  sTag[ 3]     = 't';
  sTag[ 4]     = 'u';
  sTag[ 5]     = 'n';
  sTag[ 6]     = 'e';
  sTag[ 7]     = 'r';
  sVersion[ 0] = '2';
  sVersion[ 1] = '0';
  sVersion[ 2] = '1';
  u8FeatureControl1             = DAB_CONFIG_DEFAULT_FEATURE_CONTROL1;
  u8FeatureControl2             = DAB_CONFIG_DEFAULT_FEATURE_CONTROL2;
  u8ServiceFollowingConfig      = DAB_CONFIG_SERVICEFOLLOWING_CONFIG;
  u8ConcealmentLevel            = DAB_CONFIG_DEFAULT_CONCEALMENT_LEVEL;

  for( unIdx = 0; unIdx < enKdsDabTunerParameter_len_TsuMatrixDabFm; unIdx++) {
      u8TsuMatrixDabFm[ unIdx]    = DAB_CONFIG_DEFAULT_TSU_MATRIX_DAB_FM[unIdx];
  }
  for( unIdx = 0; unIdx < enKdsDabTunerParameter_len_TsuMatrixDabFm; unIdx++) {
      u8TsuMatrixFmDab[ unIdx]    = DAB_CONFIG_DEFAULT_TSU_MATRIX_FM_DAB[unIdx];
  }



 u8DabFmSwitchBackDelay			= DAB_FM_SWITCHBACKDELAY;		
 u8DabFmSmoothingT_Up			= DAB_FM_SMOOTHINGT_UP;		
 u8DabFmSmoothingT_Down			= DAB_FM_SMOOTHINGT_DOWN;	
 u8DabFmComparisionDelay		= DAB_FM_COMPARISIONDELAY;
 u8DabFmSmoothingT_Long			= DAB_FM_SMOOTHINGT_LONG;
 u8DabFmQualityThreshold		= FM_QUALITYTHRESHOLD;

 u8TsuTimeConstantsDabFm[0]		= u8DabFmSwitchBackDelay;
 u8TsuTimeConstantsDabFm[1]		= u8DabFmSmoothingT_Up;
 u8TsuTimeConstantsDabFm[2]		= u8DabFmSmoothingT_Down;
 u8TsuTimeConstantsDabFm[3]		= u8DabFmComparisionDelay;
 u8TsuTimeConstantsDabFm[4]		= u8DabFmSmoothingT_Long;
 u8TsuTimeConstantsDabFm[5]		= u8DabFmQualityThreshold;

 u8FmDABSwitchBackDelay			= FM_DAB_SWITCHBACKDELAY;	
 u8FmDABSmoothingT_Up		    = FM_DAB_SMOOTHINGT_UP;	
 u8FmDABSmoothingT_Down			= FM_DAB_SMOOTHINGT_DOWN;	
 u8FmDABPSIDChangeDelay			= FM_DAB_PSID_CHANGEDELAY;	
 u8FmDABSmoothingT_Long			= FM_DAB_SMOOTHINGT_LONG;	
 u8FmDABQualityThreshold		= DAB_QUALITYTHRESHOLD;

 u8TsuTimeConstantsFmDab[0]		= u8FmDABSwitchBackDelay;
 u8TsuTimeConstantsFmDab[1]		= u8FmDABSmoothingT_Up;
 u8TsuTimeConstantsFmDab[2]		= u8FmDABSmoothingT_Down;
 u8TsuTimeConstantsFmDab[3]		= u8FmDABPSIDChangeDelay;
 u8TsuTimeConstantsFmDab[4]		= u8FmDABSmoothingT_Long;
 u8TsuTimeConstantsFmDab[5]		= u8FmDABQualityThreshold;

 u8RdmQualityConfigDecreaseSpeedFactorCount = DAB_CONFIG_DEFAULT_QUALITY_DECREASE_SPEED_FACTOR_COUNT; 

  for( unIdx = 0; unIdx < u8RdmQualityConfigDecreaseSpeedFactorCount; unIdx++) {
      u8RdmQualityConfigDecreaseSpeedFactor[ unIdx]    = DAB_CONFIG_DEFAULT_QUALITY_DECREASE_SPEED_FACTOR[unIdx];
  }

#ifdef VARIANT_S_FTR_ENABLE_FEATURE_SUZUKI
  u8RdmQualityConfigDecreaseSpeedFactor[ 1] = DAB_CONFIG_DEFAULT_CONCEALMENT_LEVEL_SUZUKI;
#endif
 u8RdmQualityConfigMode			= DAB_CONFIG_DEFAULT_RDM_QUALITY_CONFIG_MODE;
 u8RdmQualityConfigCount		= DAB_CONFIG_DEFAULT_RDM_QUALITY_CONFIG_COUNT;

 

 u8Ensemble_RecIndicator			= DAB_CONFIG_ENSEMBLE_RECINDICATOR;		
 u8Ensemble_RecIndicatorSyncDelay	= DAB_CONFIG_ENSEMBLE_RECINDICATORSYNCDELAY;	 
 u8EnsembleVerAlg_TimeTickDivider	= DAB_CONFIG_ENSEMBLEVERALG_TIMETICKDIVIDER;
 u8EnsembleVerAlg_TimePeriod		= DAB_CONFIG_ENSEMBLEVERALG_TIMEPERIOD;
 u8EnsembleVerAlg_MinFICBERTime		= DAB_CONFIG_ENSEMBLEVERALG_MINFICBERTIME;
 u8EnsembleVerAlg_MinFICBERLevel	= DAB_CONFIG_ENSEMBLEVERALG_MINFICBERLEVEL;
 u8EnsembleVerAlg_TargetCntrValue   = DAB_CONFIG_ENSEMBLEVERALG_TARGETCNTRVALUE;
 u8EnsembleQualAlg_MinFICBERTim		= DAB_CONFIG_ENSEMBLEQUALALG_MINFICBERTIM;
 u8EnsembleQualAlg_MinFICBERLevel	= DAB_CONFIG_ENSEMBLEQUALALG_MINFICBERLEVEL;
 u8BGSnooping1						= DAB_CONFIG_BGSNOOPING1;	                 
 u8BGSnooping_LearnFieldstrength	= DAB_CONFIG_BGSNOOPING_LEARNFIELDSTRENGTH;
 u8BGSnooping_UnLearnFieldstrength	= DAB_CONFIG_BGSNOOPING_UNLEARNFIELDSTRENGTH;
 u8BGSnooping_LearnCntrMaxValue		= DAB_CONFIG_BGSNOOPING_LEARNCNTRMAXVALUE;
 u8BGSnooping_LearnCntrValue		= DAB_CONFIG_BGSNOOPING_LEARNCNTRVALUE;
 u8BGSnooping_UnLearnCntrValue		= DAB_CONFIG_BGSNOOPING_UNLEARNCNTRVALUE;
 u8BGSnooping_UnLearnCntrMinValue	= DAB_CONFIG_BGSNOOPING_UNLEARNCNTRMINVALUE;


 u8RdmQualityConfig[0] = u8Ensemble_RecIndicator;
 u8RdmQualityConfig[1] = u8Ensemble_RecIndicatorSyncDelay;
 u8RdmQualityConfig[2] = u8EnsembleVerAlg_TimeTickDivider;
 u8RdmQualityConfig[3] = u8EnsembleVerAlg_TimePeriod;
 u8RdmQualityConfig[4] = u8EnsembleVerAlg_MinFICBERTime;
 u8RdmQualityConfig[5] = u8EnsembleVerAlg_MinFICBERLevel;
 u8RdmQualityConfig[6] = u8EnsembleVerAlg_TargetCntrValue;
 u8RdmQualityConfig[7] = u8EnsembleQualAlg_MinFICBERTim;
 u8RdmQualityConfig[8] = u8EnsembleQualAlg_MinFICBERLevel;
 u8RdmQualityConfig[9] = u8BGSnooping1;
 u8RdmQualityConfig[10] = u8BGSnooping_LearnFieldstrength;
 u8RdmQualityConfig[11] = u8BGSnooping_UnLearnFieldstrength;
 u8RdmQualityConfig[12] = u8BGSnooping_LearnCntrMaxValue;
 u8RdmQualityConfig[13] = u8BGSnooping_LearnCntrValue;
 u8RdmQualityConfig[14] = u8BGSnooping_UnLearnCntrValue;
 u8RdmQualityConfig[15] = u8BGSnooping_UnLearnCntrMinValue;


  u8AnnouncementConfig              = DAB_CONFIG_DEFAULT_ANNOUNCEMENT_CONFIG;
  u8DabScanConfig					= DAB_CONFIG_DEFAULT_SCAN_CONFIG;
  u8ScanMinSignalQualType			= DAB_CONFIG_DEFAULT_SCAN_MIN_SIGNAL_QUALITY;
  u8ScanAudioThreshold				= DAB_CONFIG_DEFAULT_SCAN_MIN_AUDIO_THRESHOLD;
  u8ScanPlayDuration				= DAB_CONFIG_DEFAULT_SCAN_PLAY_DURATION;

  u8EnsembleList_SortingType		= DAB_CONFIG_DEFAULT_ENSEMBLE_LIST_SORTING;
  u8ServiceList_SortingType			= DAB_CONFIG_DEFAULT_SERVICE_LIST_SORTING;
  u8ComponentList_SortingType		= DAB_CONFIG_DEFAULT_COMPONENT_LIST_SORTING;
  u16MaxStations					= DAB_CONFIG_DEFAULT_MAX_STATIONS;
  u8StationListConfig				= DAB_CONFIG_DEFAULT_STATIONLIST_CONFIG;

  u8T_FMDABSwitchMuteDab			= DAB_CONFIG_DEFAULT_T_FMDABSWITCHMUTEDAB;
  u8T_FMDABSwitchMuteDabPlus		= DAB_CONFIG_DEFAULT_T_FMDABSWITCHMUTEDABPLUS;
  u8T_FMDABSwitchMuteDmba			= DAB_CONFIG_DEFAULT_T_FMDABSWITCHMUTEDMBA;
  u8T_BG_LearnStart					= DAB_CONFIG_DEFAULT_T_BG_LEARNSTART;
  u8T_Learn_Start_SF_Off			= DAB_CONFIG_DEFAULT_T_LEARN_START_SF_OFF;
  u8T_Man_Mode_SF_Off				= DAB_CONFIG_DEFAULT_T_MAN_MODE_SF_OFF;
  u8T_Station_Select_SF_Off			= DAB_CONFIG_DEFAULT_T_STATION_SELECT_SF_OFF;
  u8T_SLS_Signal_Lost				= DAB_CONFIG_DEFAULT_T_SLS_SIGNAL_LOST;
  u8T_Clear_RT_Delay				= DAB_CONFIG_DEFAULT_T_CLEAR_RT_DELAY;
  u8T_Clear_RT_After_Sync_Lost		= DAB_CONFIG_DEFAULT_T_CLEAR_RT_AFTER_SYNC_LOST;

  u16LSMConfig_EID 					= DAB_CONFIG_DEFAULT_ENSEMBLE_ID;
  for( unIdx = 0; unIdx < enKdsDabTunerParameter_len_LSMConfig_EnsembleLabel; unIdx++) {
	  u16LSMConfig_EnsembleLabel[ unIdx]    = DAB_CONFIG_DEFAULT_ENSEMBLE_LABEL[unIdx];
  }
  u16LSMConfig_EnsembleShortLabelMask = DAB_CONFIG_DEFAULT_ENSEMBLE_LABEL_INFO;
  u16LSMConfig_SID = DAB_CONFIG_DEFAULT_LSM_ASID;	
  for( unIdx = 0; unIdx < enKdsDabTunerParameter_len_LSMConfig_ServiceLabel; unIdx++) {
	  u16LSMConfig_ServiceLabel[ unIdx]    = DAB_CONFIG_DEFAULT_SERVICE_LABEL[unIdx];
  }
  u16LSMConfig_ServiceShortLabelMask = DAB_CONFIG_DEFAULT_SERVICE_LABEL_INFO;
  u16LSMConfig_CompID = DAB_CONFIG_DEFAULT_LSM_ASID; 
  for( unIdx = 0; unIdx < enKdsDabTunerParameter_len_LSMConfig_ServiceLabel; unIdx++) {
	  u16LSMConfig_ComponentLabel[ unIdx]    = DAB_CONFIG_DEFAULT_SERVICE_LABEL[unIdx];
  }
  u16LSMConfig_ComponentShortLabelMask = DAB_CONFIG_DEFAULT_SERVICE_LABEL_INFO; 
  for( unIdx = 0; unIdx < enKdsDabTunerParameter_len_LSMConfig_Channel; unIdx++) {
	  u16LSMConfig_Channel[ unIdx]    =(tU8)(DAB_CONFIG_DEFAULT_CHANNEL[unIdx]);  }


  u8DabBandConfig                   = DAB_CONFIG_DEFAULT_DAB_BAND; 
  u16CRC = 0x00;	//Coverity 17668
}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid trKdsDabTunerParameter::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
		(tVoid)pu8KdsBuffer;
	(tVoid)u32Length;

  //tUInt unIdx;
  //ETG_TRACE_USR1(( " trKdsDabTunerParameter::vParse() ->\n"
  //                 " u32Length                   : %d bytes\n"
  //                 " KDS config                  : %d bytes\n"
  //                 " expected                    : %d bytes\n"
  //                 ,u32Length
  //                 ,DP_U8_KDSLEN_DABPARAMETERSET_COMPLETE
  //                 ,DAB_CONFIG_KDS_SET_DAB_TUNER_LEN
  //              ));

  if( u32Length < DAB_CONFIG_KDS_SET_DAB_TUNER_LEN)
  {
    /*ETG_TRACE_USR1(( " tun_trKdsDabTunerParameter::vParse() -> Error: wrong length of KDS entry, default values are used now!\n"
                     " u32Length                   : %d bytes\n"
                     " KDS config                  : %d bytes\n"
                     " expected                    : %d bytes\n"
                     ,u32Length
                     ,DP_U8_KDSLEN_DABPARAMETERSET_COMPLETE
                     ,DAB_CONFIG_KDS_SET_DAB_TUNER_LEN
                  ));*/
  }
  else
  {
	  OSAL_pvMemoryCopy( sTag, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_Tag], enKdsDabTunerParameter_len_Tag);
	  OSAL_pvMemoryCopy( sVersion, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_MajorVersion], enKdsDabTunerParameter_len_MajorVersion + enKdsDabTunerParameter_len_MinorVersion);
	
	  u8FeatureControl1             = pu8KdsBuffer[ enKdsDabTunerParameter_idx_FeatureControl1];
	  u8FeatureControl2             = pu8KdsBuffer[ enKdsDabTunerParameter_idx_FeatureControl2];
	  u8ServiceFollowingConfig      = pu8KdsBuffer[ enKdsDabTunerParameter_idx_ServiceFollowConfig];
	  u8ConcealmentLevel            = pu8KdsBuffer[ enKdsDabTunerParameter_idx_ConcealmentLevel];

	  OSAL_pvMemoryCopy( u8TsuMatrixDabFm, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_TsuMatrixDabFm], enKdsDabTunerParameter_len_TsuMatrixDabFm);
      OSAL_pvMemoryCopy( u8TsuMatrixFmDab, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_TsuMatrixFmDab], enKdsDabTunerParameter_len_TsuMatrixFmDab);

	 u8DabFmSwitchBackDelay			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabFmSwitchBackDelay];		
	 u8DabFmSmoothingT_Up			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabFmSmoothingT_Up];		
	 u8DabFmSmoothingT_Down			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabFmSmoothingT_Down];	
	 u8DabFmComparisionDelay		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabFmComparisionDelay];
	 u8DabFmSmoothingT_Long			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabFmSmoothingT_Long];
	 u8DabFmQualityThreshold		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabFmQualityThreshold];

	 u8TsuTimeConstantsDabFm[0]		= u8DabFmSwitchBackDelay;
	 u8TsuTimeConstantsDabFm[1]		= u8DabFmSmoothingT_Up;
	 u8TsuTimeConstantsDabFm[2]		= u8DabFmSmoothingT_Down;
	 u8TsuTimeConstantsDabFm[3]		= u8DabFmComparisionDelay;
	 u8TsuTimeConstantsDabFm[4]		= u8DabFmSmoothingT_Long;
	 u8TsuTimeConstantsDabFm[5]		= u8DabFmQualityThreshold;

	 u8FmDABSwitchBackDelay			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_FmDABSwitchBackDelay];	
	 u8FmDABSmoothingT_Up		    = pu8KdsBuffer[ enKdsDabTunerParameter_idx_FmDABSmoothingT_Up];	
	 u8FmDABSmoothingT_Down			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_FmDABSmoothingT_Down];	
	 u8FmDABPSIDChangeDelay			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_FmDABPSIDChangeDelay];	
	 u8FmDABSmoothingT_Long			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_FmDABSmoothingT_Long];	
	 u8FmDABQualityThreshold		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_FmDABQualityThreshold];

	 u8TsuTimeConstantsFmDab[0]		= u8FmDABSwitchBackDelay;
	 u8TsuTimeConstantsFmDab[1]		= u8FmDABSmoothingT_Up;
	 u8TsuTimeConstantsFmDab[2]		= u8FmDABSmoothingT_Down;
	 u8TsuTimeConstantsFmDab[3]		= u8FmDABPSIDChangeDelay;
	 u8TsuTimeConstantsFmDab[4]		= u8FmDABSmoothingT_Long;
	 u8TsuTimeConstantsFmDab[5]		= u8FmDABQualityThreshold;

	tU8 u8RdmQualityConfigDecreaseSpeedFactorValue	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_RDMQConfig_QDecreaseFactor];
	u8RdmQualityConfigDecreaseSpeedFactor[ 1]    = u8RdmQualityConfigDecreaseSpeedFactorValue;

	/*Work around for PSA as this value is still reserved in PSA KDS scripts*/
	if(u8RdmQualityConfigDecreaseSpeedFactor[ 1] ==0){
		u8RdmQualityConfigDecreaseSpeedFactor[ 1]    = DAB_CONFIG_DEFAULT_QUALITY_DECREASE_SPEED_FACTOR[1];
	}

	u8RdmQualityConfigMode			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_RDMQConfig_Mode];
	u8RdmQualityConfigCount			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_RDMQConfig_Count];
	 

	 u8Ensemble_RecIndicator			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_Ensemble_RecIndicator];		
	 u8Ensemble_RecIndicatorSyncDelay	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_Ensemble_RecIndicatorSyncDelay];	 
	 u8EnsembleVerAlg_TimeTickDivider	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleVerAlg_TimeTickDivider];
	 u8EnsembleVerAlg_TimePeriod		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleVerAlg_TimePeriod];
	 u8EnsembleVerAlg_MinFICBERTime		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleVerAlg_MinFICBERTime];
	 u8EnsembleVerAlg_MinFICBERLevel	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleVerAlg_MinFICBERLevel];
	 u8EnsembleVerAlg_TargetCntrValue   = pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleVerAlg_TargetCntrValue];
	 u8EnsembleQualAlg_MinFICBERTim		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleQualAlg_MinFICBERTime];
	 u8EnsembleQualAlg_MinFICBERLevel	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleQualAlg_MinFICBERLevel];
	 u8BGSnooping1						= pu8KdsBuffer[ enKdsDabTunerParameter_idx_BGSnooping1];	                 
	 u8BGSnooping_LearnFieldstrength	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_BGSnooping_LearnFieldstrength];
	 u8BGSnooping_UnLearnFieldstrength	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_BGSnooping_UnLearnFieldstrength];
	 u8BGSnooping_LearnCntrMaxValue		= pu8KdsBuffer[enKdsDabTunerParameter_idx_BGSnooping_LearnCntrMaxValue ];
	 u8BGSnooping_LearnCntrValue		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_BGSnooping_LearnCntrValue];
	 u8BGSnooping_UnLearnCntrValue		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_BGSnooping_UnLearnCntrValue];
	 u8BGSnooping_UnLearnCntrMinValue	= pu8KdsBuffer[ enKdsDabTunerParameter_idx_BGSnooping_UnLearnCntrMinValue];

	  u8RdmQualityConfig[0] = u8Ensemble_RecIndicator;
	 u8RdmQualityConfig[1] = u8Ensemble_RecIndicatorSyncDelay;
	 u8RdmQualityConfig[2] = u8EnsembleVerAlg_TimeTickDivider;
	 u8RdmQualityConfig[3] = u8EnsembleVerAlg_TimePeriod;
	 u8RdmQualityConfig[4] = u8EnsembleVerAlg_MinFICBERTime;
	 u8RdmQualityConfig[5] = u8EnsembleVerAlg_MinFICBERLevel;
	 u8RdmQualityConfig[6] = u8EnsembleVerAlg_TargetCntrValue;
	 u8RdmQualityConfig[7] = u8EnsembleQualAlg_MinFICBERTim;
	 u8RdmQualityConfig[8] = u8EnsembleQualAlg_MinFICBERLevel;
	 u8RdmQualityConfig[9] = u8BGSnooping1;
	 u8RdmQualityConfig[10] = u8BGSnooping_LearnFieldstrength;
	 u8RdmQualityConfig[11] = u8BGSnooping_UnLearnFieldstrength;
	 u8RdmQualityConfig[12] = u8BGSnooping_LearnCntrMaxValue;
	 u8RdmQualityConfig[13] = u8BGSnooping_LearnCntrValue;
	 u8RdmQualityConfig[14] = u8BGSnooping_UnLearnCntrValue;
	 u8RdmQualityConfig[15] = u8BGSnooping_UnLearnCntrMinValue;

		
	

	  u8AnnouncementConfig              = pu8KdsBuffer[ enKdsDabTunerParameter_idx_AnnouncementConfig];
	  u8DabScanConfig					= pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabScanConfig];
	  u8ScanMinSignalQualType			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_ScanMinSignalQuality];
	  u8ScanAudioThreshold				= pu8KdsBuffer[ enKdsDabTunerParameter_idx_ScanAudioThreshold];
	  u8ScanPlayDuration				= pu8KdsBuffer[ enKdsDabTunerParameter_idx_ScanPlayDuration];

	  u8EnsembleList_SortingType		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_EnsembleListSorting];
	  u8ServiceList_SortingType			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_ServiceListSorting];
	  u8ComponentList_SortingType		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_ComponentListSorting];
	  u16MaxStations					= DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_MaxStations]);
	  u8StationListConfig				= pu8KdsBuffer[ enKdsDabTunerParameter_idx_StationListConfig];


	  u8T_FMDABSwitchMuteDab			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_FmDab_SwitchMute_Dab];
	  u8T_FMDABSwitchMuteDabPlus		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_FmDab_SwitchMute_DabPlus];
	  u8T_FMDABSwitchMuteDmba			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_FmDab_SwitchMute_Dmba];
	  u8T_BG_LearnStart					= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_BGLearnStart];
	  u8T_Learn_Start_SF_Off			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_LearnStart_SF_Off];
	  u8T_Man_Mode_SF_Off				= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_ManMode_SF_Off];
	  u8T_Station_Select_SF_Off			= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_Station_Select_SF_Off];
	  u8T_SLS_Signal_Lost				= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_SLS_Signal_Lost];
	  u8T_Clear_RT_Delay				= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_Clear_RT_Delay];
	  u8T_Clear_RT_After_Sync_Lost		= pu8KdsBuffer[ enKdsDabTunerParameter_idx_t_Clear_RT_AfterSyncLost];

	  OSAL_pvMemoryCopy( u16LSMConfig_EnsembleLabel, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_EnsembleLabel], enKdsDabTunerParameter_len_LSMConfig_EnsembleLabel);
	  OSAL_pvMemoryCopy( u16LSMConfig_EnsembleLabel, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_ServiceLabel], enKdsDabTunerParameter_len_LSMConfig_ServiceLabel);
	  OSAL_pvMemoryCopy( u16LSMConfig_EnsembleLabel, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_CompLabel], enKdsDabTunerParameter_len_LSMConfig_CompLabel);
	   OSAL_pvMemoryCopy( u16LSMConfig_Channel, &pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_Channel], enKdsDabTunerParameter_len_LSMConfig_Channel);

	  u16LSMConfig_EnsembleShortLabelMask      = DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_EnsShortLabelMask]);
	  u16LSMConfig_ServiceShortLabelMask      = DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_SrvcShortLabelMask]);
	  u16LSMConfig_ComponentShortLabelMask      = DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_CompShortLabelMask]);
	  u16LSMConfig_EID =  DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_EID]);
	  u16LSMConfig_SID =  DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_SID]);
	  u16LSMConfig_CompID = DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_LSMConfig_CompID]);	 
	   
      u8DabBandConfig                = pu8KdsBuffer[ enKdsDabTunerParameter_idx_DabBandConfig];
	  u16CRC                = DABDRV_GET_U16 (&pu8KdsBuffer[ enKdsDabTunerParameter_idx_CRC]);
	

   
    bKdsDataRead = TRUE;
  }

}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid trKdsDabTunerParameter::vTrace( tVoid)
{
      
    ETG_TRACE_USR2(( " BEGIN ------------------------------------------------------------------------------------"            ));
    ETG_TRACE_USR2(( " bKdsDataRead                 : 0x%02X  (0x00: Default values, 0x01: date from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " sTag                         : %8s"    ,sTag                  ));
    ETG_TRACE_USR2(( " sVersion                     : %3s"    ,sVersion              ));
    ETG_TRACE_USR2(( " u8FeatureControl1            : 0x%02X" ,u8FeatureControl1    ));
    ETG_TRACE_USR2(( " u8FeatureControl2            : 0x%02X" ,u8FeatureControl2        ));
    ETG_TRACE_USR2(( " u8ServiceFollowingConfig     : 0x%02X" ,u8ServiceFollowingConfig             ));
    ETG_TRACE_USR2(( " u8ConcealmentLevel           : 0x%02X" ,u8ConcealmentLevel                 ));
    ETG_TRACE_USR2(( " u8TsuMatrixDabFm             : %*p" ,ETG_LIST_LEN(enKdsDabTunerParameter_len_TsuMatrixDabFm),ETG_LIST_PTR_T8(u8TsuMatrixDabFm)));
    ETG_TRACE_USR2(( " u8TsuMatrixFmDab             : %*p" ,ETG_LIST_LEN(enKdsDabTunerParameter_len_TsuMatrixFmDab),ETG_LIST_PTR_T8(u8TsuMatrixFmDab)));
	ETG_TRACE_USR2(( " u8DabFmSwitchBackDelay           : 0x%02X" ,u8DabFmSwitchBackDelay                 ));
	ETG_TRACE_USR2(( " u8DabFmSmoothingT_Up           : 0x%02X" ,u8DabFmSmoothingT_Up                 ));
	ETG_TRACE_USR2(( " u8DabFmSmoothingT_Down           : 0x%02X" ,u8DabFmSmoothingT_Down                 ));
	ETG_TRACE_USR2(( " u8DabFmComparisionDelay           : 0x%02X" ,u8DabFmComparisionDelay                 ));
	ETG_TRACE_USR2(( " u8DabFmSmoothingT_Long           : 0x%02X" ,u8DabFmSmoothingT_Long                 ));
	ETG_TRACE_USR2(( " u8DabFmQualityThreshold           : 0x%02X" ,u8DabFmQualityThreshold                 ));
	ETG_TRACE_USR2(( " u8FmDABSwitchBackDelay           : 0x%02X" ,u8FmDABSwitchBackDelay                 ));
	ETG_TRACE_USR2(( " u8FmDABSmoothingT_Up           : 0x%02X" ,u8FmDABSmoothingT_Up                 ));
	ETG_TRACE_USR2(( " u8FmDABSmoothingT_Down           : 0x%02X" ,u8FmDABSmoothingT_Down                 ));
	ETG_TRACE_USR2(( " u8FmDABPSIDChangeDelay           : 0x%02X" ,u8FmDABPSIDChangeDelay                 ));
	ETG_TRACE_USR2(( " u8FmDABSmoothingT_Long           : 0x%02X" ,u8FmDABSmoothingT_Long                 ));
	ETG_TRACE_USR2(( " u8FmDABQualityThreshold           : 0x%02X" ,u8FmDABQualityThreshold                 ));

	ETG_TRACE_USR2(( " u8RdmQualityConfigDecreaseSpeedFactor at 0: 0x%02X" ,u8RdmQualityConfigDecreaseSpeedFactor[0]));
	ETG_TRACE_USR2(( " u8RdmQualityConfigDecreaseSpeedFactor at 1: 0x%02X" ,u8RdmQualityConfigDecreaseSpeedFactor[1]));
	ETG_TRACE_USR2(( " u8RdmQualityConfigMode       : 0x%02X" ,u8RdmQualityConfigMode));
    ETG_TRACE_USR2(( " u8RdmQualityConfigCount      : 0x%02X" ,u8RdmQualityConfigCount));
    ETG_TRACE_USR2(( " u8AnnouncementConfig         : 0x%02X" ,u8AnnouncementConfig             ));
	 ETG_TRACE_USR2(( " u8Ensemble_RecIndicator           : 0x%02X" ,u8Ensemble_RecIndicator                 ));
	 ETG_TRACE_USR2(( " u8Ensemble_RecIndicatorSyncDelay           : 0x%02X" ,u8Ensemble_RecIndicatorSyncDelay                 ));
	 ETG_TRACE_USR2(( " u8EnsembleVerAlg_TimeTickDivider           : 0x%02X" ,u8EnsembleVerAlg_TimeTickDivider                 ));
	 ETG_TRACE_USR2(( " u8EnsembleVerAlg_TimePeriod           : 0x%02X" ,u8EnsembleVerAlg_TimePeriod                 ));
	 ETG_TRACE_USR2(( " u8EnsembleVerAlg_MinFICBERTime           : 0x%02X" ,u8EnsembleVerAlg_MinFICBERTime                 ));
	 ETG_TRACE_USR2(( " u8EnsembleVerAlg_MinFICBERLevel           : 0x%02X" ,u8EnsembleVerAlg_MinFICBERLevel                 ));
	 ETG_TRACE_USR2(( " u8EnsembleVerAlg_TargetCntrValue           : 0x%02X" ,u8EnsembleVerAlg_TargetCntrValue                 ));
	 ETG_TRACE_USR2(( " u8EnsembleQualAlg_MinFICBERTim           : 0x%02X" ,u8EnsembleQualAlg_MinFICBERTim                 ));
	 ETG_TRACE_USR2(( " u8EnsembleQualAlg_MinFICBERLevel           : 0x%02X" ,u8EnsembleQualAlg_MinFICBERLevel                 ));
	 ETG_TRACE_USR2(( " u8BGSnooping1           : 0x%02X" ,u8BGSnooping1                 ));
	 ETG_TRACE_USR2(( " u8BGSnooping_LearnFieldstrength           : 0x%02X" ,u8BGSnooping_LearnFieldstrength                 ));
	 ETG_TRACE_USR2(( " u8BGSnooping_UnLearnFieldstrength           : 0x%02X" ,u8BGSnooping_UnLearnFieldstrength                 ));
	 ETG_TRACE_USR2(( " u8BGSnooping_LearnCntrMaxValue           : 0x%02X" ,u8BGSnooping_LearnCntrMaxValue                 ));
	 ETG_TRACE_USR2(( " u8BGSnooping_LearnCntrValue           : 0x%02X" ,u8BGSnooping_LearnCntrValue                 ));
	 ETG_TRACE_USR2(( " u8BGSnooping_UnLearnCntrValue           : 0x%02X" ,u8BGSnooping_UnLearnCntrValue                 ));
	 ETG_TRACE_USR2(( " u8BGSnooping_UnLearnCntrMinValue           : 0x%02X" ,u8BGSnooping_UnLearnCntrMinValue                 ));
  
    ETG_TRACE_USR2(( " u16LSMConfig_EID                   : 0x%08X" ,u16LSMConfig_EID            ));
	ETG_TRACE_USR2(( " u16LSMConfig_SID                   : 0x%08X" ,u16LSMConfig_SID            ));
	ETG_TRACE_USR2(( " u16LSMConfig_CompID                : 0x%08X" ,u16LSMConfig_CompID            ));
   

   
    ETG_TRACE_USR2(( " END   ------------------------------------------------------------------------------------"            ));
}



/*****************************************************************************
 * Constructor
 ****************************************************************************/
trKdsHWConfigurationString::trKdsHWConfigurationString( tVoid)
{
  ETG_TRACE_USR1((" trKdsHWConfigurationString::tun_trKdsHWConfigurationString() -> Constructor"));
  trKdsHWConfigurationString::vSetToDefaultValues();
}


/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid trKdsHWConfigurationString::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1(( " trKdsHWConfigurationString::vSetToDefaultValues() -> "));

  bKdsDataRead = FALSE;
  u8OEMType = (tU8)enKdsOEMType_SKODA;
  u8TunerMode = 0x20;
  //enFmDabHdTunerMode = enFmDabHdTunerMode_reserved_0x00; //Need to be retained when KDS reading for PSA is in place
  enFmDabHdTunerMode = enFmDabHdTunerMode_TunVar20_3FE_DDA_DDS_DAB;  //Currently since we are not reading the KDS value for PSA, make this the defualt.
																	 //This needs to be removed later.
}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid trKdsHWConfigurationString::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
	#ifdef KDS_FOR_G3G_VARIANT
  ETG_TRACE_USR1(( " trKdsHWConfigurationString::vParse() ->\n"
                   " u32Length             : %d bytes\n"
                   " KDS config            : %d bytes\n"
                   ,u32Length
                   ,DIA_C_U16_KDS_CM_CONFIG_STRING_LEN
                ));
#endif

  /**Vnd4kor : To solve prio 2 lint warning*/
    ETG_TRACE_USR1(( " trKdsHWConfigurationString::vParse() ->\n"
                   " u32Length             : %d bytes\n"
                    ,u32Length
                   ));
  u8OEMType = pu8KdsBuffer[ enKdsHWConfigurationString_idx_OEMType];
  u8TunerMode = pu8KdsBuffer[ enKdsHWConfigurationString_idx_TunerMode];
  enFmDabHdTunerMode = (tenFmDabHdTunerMode)pu8KdsBuffer[ enKdsHWConfigurationString_idx_FmDabHdTunerMode];
  bKdsDataRead = TRUE;
}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid trKdsHWConfigurationString::vTrace( tVoid)
{
    ETG_TRACE_USR2(( " bKdsDataRead         : 0x%02X  (0x00: Default values, 0x01: date from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " u8TunerMode          : 0x%02X"       ,u8TunerMode         ));
    ETG_TRACE_USR2(( " enFmDabHdTunerMode   : %02x"         ,ETG_CENUM( tenFmDabHdTunerMode,    enFmDabHdTunerMode)  ));
}

trKdsCodingData::trKdsCodingData( tVoid){
	  ETG_TRACE_USR1((" trKdsCodingData::trKdsCodingData() -> Constructor"));
     trKdsCodingData::vSetToDefaultValues();
}
tVoid trKdsCodingData::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1(( " trKdsCodingData::vSetToDefaultValues() -> "));
  bKdsDataRead = FALSE;
  u8TunerBand1 = (tU8)enCodingTunerBand1_EU_BAND3_WO_NEW;
  u8TunerBand2 =(tU8)enCodingTunerBand2_L_BAND;
  bAlarmEnabled = TRUE; 
}
tVoid trKdsCodingData::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " trKdsCodingData::vParse() ->\n"
                   " u32Length             : %d bytes\n"                  
                   ,u32Length                  
               ));
  u8TunerBand1  = pu8KdsBuffer[ enKdsCodingValue_TunerBand ]&(MASK_LOWER_NIBBLE);
  u8TunerBand2  = (pu8KdsBuffer[ enKdsCodingValue_TunerBand ]&(MASK_UPPER_NIBBLE))>>NUMBER_FOUR;
  bAlarmEnabled = (tBool)(pu8KdsBuffer[ enKdsCodingValue_Alarm ]>>NUMBER_SIX);
  bKdsDataRead  = TRUE;
}
tVoid trKdsCodingData::vTrace( tVoid)
{
    ETG_TRACE_USR2(( " bKdsDataRead         : 0x%02X  (0x00: Default values, 0x01: data from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " u8TunerBand1         : 0x%02X"       ,u8TunerBand1         ));
    ETG_TRACE_USR2(( " u8TunerBand2			: 0x%02X"         ,u8TunerBand2		  ));
	ETG_TRACE_USR2(( " bAlarmEnabled	    : 0x%02X"         ,bAlarmEnabled		  ));
}
//trKdsCodingVariant::trKdsCodingVariant( tVoid){
//	  ETG_TRACE_USR1((" trKdsCodingData::trKdsCodingData() -> Constructor"));
//	  trKdsCodingVariant::vSetToDefaultValues();
//}
//tVoid trKdsCodingVariant::vSetToDefaultValues( tVoid)
//{
//  ETG_TRACE_USR1(( " trKdsCodingVariant::vSetToDefaultValues() -> "));
//  bKdsDataRead = FALSE;
//  u8OEMType = (tU8)enKdsOEMType_SKODA;
//}
//tVoid trKdsCodingVariant::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
//{
//  ETG_TRACE_USR1(( " trKdsCodingData::vParse() ->\n"
//                   " u32Length             : %d bytes\n"
//                   ,u32Length
//               ));
//  u8OEMType  = pu8KdsBuffer[ enKdsOEMType ];
//  bKdsDataRead  = TRUE;
//}
//tVoid trKdsCodingVariant::vTrace( tVoid)
//{
//    ETG_TRACE_USR2(( " bKdsDataRead         : 0x%02X  (0x00: Default values, 0x01: data from KDS read)"  ,bKdsDataRead ));
//    ETG_TRACE_USR2(( "   u8OEMType          : 0x%02X"       ,  u8OEMType         ));
//}

#ifdef FEATURE_PSA_RCC
trKdsPSAEOLParamRadio::trKdsPSAEOLParamRadio( tVoid){
	  ETG_TRACE_USR1((" trKdsPSAEOLParamRadio::trKdsCodingData() -> Constructor"));
	  trKdsPSAEOLParamRadio::vSetToDefaultValues();
}
tVoid trKdsPSAEOLParamRadio::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1(( " trKdsPSAEOLParamRadio::vSetToDefaultValues() -> "));
  u8Radio_CountryZone = 0x00;
  bRadio_RdsFollowUp=0;
}
tVoid trKdsPSAEOLParamRadio::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " trKdsPSAEOLParamRadio::vParse() ->\n"
                   " u32Length             : %d bytes\n"
                   ,u32Length
               ));
  u8Radio_CountryZone  = pu8KdsBuffer[ enKdsPSAEOLRadio_Country ] >>NUMBER_SIX;  
  bKdsDataRead  = TRUE;
  bRadio_RdsFollowUp = (pu8KdsBuffer[ enKdsPSAEOLRadio_Cluster]>>NUMBER_FIVE)& 0x01;
}
tVoid trKdsPSAEOLParamRadio::vTrace( tVoid)
{
    ETG_TRACE_USR2(( " bKdsDataRead         : 0x%02X  (0x00: Default values, 0x01: data from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( "   u8Radio_CountryZone          : 0x%02X"       ,  u8Radio_CountryZone         ));
    ETG_TRACE_USR2(( "   bRadio_RdsFollowUp          : 0x%02X"       ,  bRadio_RdsFollowUp         ));
}

trKdsPSAEOLParamRadioNum::trKdsPSAEOLParamRadioNum( tVoid){
	  ETG_TRACE_USR1((" trKdsPSAEOLParamRadioNum::trKdsCodingData() -> Constructor"));
	  trKdsPSAEOLParamRadioNum::vSetToDefaultValues();
}
tVoid trKdsPSAEOLParamRadioNum::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1(( " trKdsPSAEOLParamRadioNum::vSetToDefaultValues() -> "));
  u8RadioNum_BandActivation = 0x00; //Disable DAB by default.
  u8RadioNum_BandActivityThreshold = 0x00;
}
tVoid trKdsPSAEOLParamRadioNum::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " trKdsPSAEOLParamRadioNum::vParse() ->\n"
                   " u32Length             : %d bytes\n"
                   ,u32Length
               ));
  u8RadioNum_BandActivation  = pu8KdsBuffer[ enKdsPSAEOL_BandActivation ] >>NUMBER_SIX;
  u8RadioNum_BandActivityThreshold = pu8KdsBuffer[ enKdsPSAEOL_ActivityThreshold ];
  bKdsDataRead  = TRUE;
}
tVoid trKdsPSAEOLParamRadioNum::vTrace( tVoid)
{
    ETG_TRACE_USR2(( " bKdsDataRead         : 0x%02X  (0x00: Default values, 0x01: data from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( "   u8RadioNum_BandActivation          : 0x%02X"       ,  u8RadioNum_BandActivation         ));
	ETG_TRACE_USR2(( "   u8RadioNum_BandActivityThreshold   : 0x%02X"       ,  u8RadioNum_BandActivityThreshold         ));
}
#endif

trDabConfig::trDabConfig():
        u32CcaPrio(DAB_DEFAULT_CCA_PRIO),
        u32DispatcherPrio(DAB_DEFAULT_DISPATCHER_PRIO),
        u32DispatcherStackSize(DAB_DEFAULT_DISPATCHER_STACK_SIZE),
        u32IpnPrio(DAB_IPN_PRIO),
        u32IpnStackSize(DAB_IPN_STACK_SIZE),
        bDabModulePresent(FALSE),
        bDabEnable(FALSE)
    {}

fc_dabtuner_config::fc_dabtuner_config(){
	//Default values to solve linit prio2 warnings.
    u8NumberofBanks = NUMBER_TWO;
    u8NumOfPresetsPerBank = NUMBER_SIX;
    bMixedPresetList = 0;
    m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE;
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
        bClearPreset = FALSE;
#endif


        vReloadConfigration();
}

tVoid fc_dabtuner_config::vReloadConfigration()
{       vGetRegistryValue(&rDabConfig.u32CcaPrio, "APP_THREAD", "AB_PRIO");
        vGetRegistryValue(&rDabConfig.u32DispatcherPrio, "APP_THREAD", "DAB_DISP_PRIO");
        vGetRegistryValue(&rDabConfig.u32DispatcherStackSize, "APP_THREAD", "DAB_DISP_STCK");
        vGetRegistryValue(&rDabConfig.u32IpnPrio, "APP_THREAD", "DAB_IPN_PRIO");
        vGetRegistryValue(&rDabConfig.u32IpnStackSize, "APP_THREAD", "DAB_IPN_STCK");


        // read KDS configuration
        vReadKDSConfig(DAB_KDS_CM_CONFIG_STRING);
        vReadKDSConfig(DAB_KDS_PARAMETER_SET);		
		vReadKDSConfig(DAB_KDS_CODING_VARIANT);
        vReadKDSConfig(DAB_KDS_GENERIC_PARAM_SET);
#ifdef FEATURE_PSA_RCC
		vReadKDSConfig(DAB_KDS_EOL_PARAM_RADIO);    
		vReadKDSConfig(DAB_KDS_EOL_PARAM_RADIONUM); 		
#endif

        tU8 u8TunerMode = m_rKdsHWConfigurationString.u8TunerMode;
        tBool bDabEnable = (u8TunerMode & 0x20)?TRUE:FALSE;

		vSetDabFrequencyTable_Part1(m_rKdsDabTunerParameter.u8DabBandConfig);
		vSetFrequencyTableBasedOnVariantSelection();


#ifdef FEATURE_PSA_RCC			
		tU8 u8ActiveBand  = m_rKdsPSAEOLParamRadioNum.u8RadioNum_BandActivation;
		tU8 u8CountryZone = m_rKdsPSAEOLParamRadio.u8Radio_CountryZone;
		vSetEOLBasedFrequencyTable(u8ActiveBand,u8CountryZone);
#endif
        rDabConfig.bDabModulePresent = bDabEnable;
#   ifdef LSIM
        rDabConfig.bDabModulePresent = TRUE;
#   endif // LSIM
        
        ETG_TRACE_USR4(("fc_dabtuner_config:KDS: bDabModulePresent=%u",
                        ETG_ENUM(BOOL, rDabConfig.bDabModulePresent)));

        if (rDabConfig.bDabModulePresent) {
            // this may change for later projects that want to disable the dab-module
            rDabConfig.bDabEnable = TRUE;
            //vReadKdsEolSysConfiguration1();
#ifdef FEATURE_PSA_RCC
		if(enConfigBandActivation_DeactiveAll == (tenConfigBandActivation)m_rKdsPSAEOLParamRadioNum.u8RadioNum_BandActivation)
		{
			 rDabConfig.bDabEnable = FALSE;
			 rDabConfig.bDabModulePresent = rDabConfig.bDabEnable;
		}
		ETG_TRACE_USR4(("fc_dabtuner_config:KDS: bDabEnable=%u",
                        ETG_ENUM(BOOL, rDabConfig.bDabEnable)));
#endif
        }
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
        tBool bDABAvailability = 0;
        dp_tclDABTunerDPDABAvailability oDABAvailability;
        tU32 status = oDABAvailability.s32GetData(bDABAvailability);
        if(!status)
        {
            if(!bDABAvailability && rDabConfig.bDabModulePresent)
            {
                bClearPreset = TRUE;
            }
        }
                
        status = oDABAvailability.s32SetData(rDabConfig.bDabModulePresent);
        ETG_TRACE_USR1(("Status : %d",status));
#endif

    }


    tVoid fc_dabtuner_config::vGetRegistryValue (tU32 *pu32KeyData, char const *cszSection, char const *cszKey) const
    {
        tBool bSuccess;
        tU32 u32KeyData;

        bSuccess = scd_bGetAppConfigurationValue
            (
             CCA_C_U16_APP_FC_DABTUNER, // CCA App ID
             cszSection,                // section name
             cszKey,                    // key name
             &u32KeyData                // location for Value
             );
        if ( bSuccess ) {
            *pu32KeyData = u32KeyData;
            ETG_TRACE_USR4(("fc_dabtuner_config::vGetRegistryValue: gotKey(%10s/%10s):0x%x",
                            cszSection, cszKey, u32KeyData));

        } else {
            ETG_TRACE_COMP(("fc_dabtuner_config::vGetRegistryValue FAILED: gotKey(%10s/%10s): default=0x%x",
                            cszSection, cszKey, *pu32KeyData));
        }
    };

    tVoid fc_dabtuner_config::vSetSfDabDabEnable(tBool bSet) { 
        DAB_SET_BIT_U8(bSet,m_rKdsDabTunerParameter.u8FeatureControl1,(tU8)0);
        ETG_TRACE_USR4(("fc_dabtuner_config::vSetSfDabDabEnable m_rKdsDabTunerParameter.u8FeatureSelection 0x%x bSet(%d)",m_rKdsDabTunerParameter.u8FeatureControl1, bSet));
    }
    
    tVoid fc_dabtuner_config::vSetSfDabFmEnable(tBool bSet)  { 
        DAB_SET_BIT_U8(bSet,m_rKdsDabTunerParameter.u8FeatureControl1,(tU8)1);
        ETG_TRACE_USR4(("fc_dabtuner_config::vSetSfDabFmEnable m_rKdsDabTunerParameter.u8FeatureSelection 0x%x bSet(%d)",m_rKdsDabTunerParameter.u8FeatureControl1, bSet));
    }

    tVoid fc_dabtuner_config::vSetSfFmDabEnable(tBool bSet){
    	 DAB_SET_BIT_U8(bSet,m_rKdsDabTunerParameter.u8FeatureControl1,(tU8)2);
    }

    tenConfigServiceLinking fc_dabtuner_config::enReadSfMode()
    {   
        tBool bSfDab = bGetSfDabDabEnable();
        tBool bSfFm  = bGetSfDabFmEnable();
        ETG_TRACE_USR4(("fc_dabtuner_config::enReadSfMode bSfDab %d  bSfFm %d",bSfDab, bSfFm));

        if(bSfDab) { 
            return bSfFm ? enConfigServiceLinking_DAB_FM : enConfigServiceLinking_DAB; }
        else { 
            return bSfFm ? enConfigServiceLinking_FM : enConfigServiceLinking_OFF; 
        }
    }

    DAB_tenMtcMode fc_dabtuner_config::enReadMtcMode()
    {
    	tBool bSfDabFm = bGetSfDabFmEnable();
        tBool bSfFmDab = bGetSfFmDabEnable();

        ETG_TRACE_USR4(("fc_dabtuner_config::enReadMtcMode bSfFmDab %d  bSfDabFm %d",bSfFmDab, bSfDabFm));

            return bSfDabFm ? (bSfFmDab ? DAB_enMtcMode_BOTH_On : DAB_enMtcMode_DAB_FM_On)
            		: (bSfFmDab ? DAB_enMtcMode_FM_DAB_On : DAB_enMtcMode_Off);

    }


    /*************************************************************************
    * FUNCTION:    vReadKDSConfig()
    * DESCRIPTION: Read various KDS configuration parameters 
    *************************************************************************/
    tVoid fc_dabtuner_config::vReadKDSConfig( tenKDSParameter enKDSParameter)
    {
        ETG_TRACE_USR1(( " fc_dabtuner_config::vReadKDSConfig() ->"));

        OSAL_tIODescriptor IOKdsHandle;
        IOKdsHandle    = OSAL_ERROR;

        //Open the KDS device before reading it
        IOKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS,OSAL_EN_READWRITE);

        if( OSAL_ERROR == IOKdsHandle )
        {
            ETG_TRACE_ERR(( " fc_dabtuner_config::vReadKDSConfig() -> KDS open operation failed, OSAL_u32ErrorCode=0x%08X"
                            ,OSAL_u32ErrorCode()));
            return;
        }
        ETG_TRACE_USR1(( " fc_dabtuner_config::vReadKDSConfig() -> Read KDS parameter enKDSParameter=%x"
                            ,ETG_CENUM( tenKDSParameter, enKDSParameter)));
        
        switch(enKDSParameter)
        {
            /*----------------------------------------------------------------------*/
            case DAB_KDS_PARAMETER_SET: {

#ifndef DABTUNER_UTEST
#ifdef DP_U16_KDSADR_DABPARAMETERSET

				dp_tclKdsDABParameterSet oDab;			

				
				tU8 au8Buf[DP_U8_KDSLEN_DABPARAMETERSET_COMPLETE]={0};
				tU8 u8MajorVersion =0;
				oDab.u8GetMajorVersion(u8MajorVersion);
				if(u8MajorVersion!=0){
					oDab.u8GetDABParameterSet(au8Buf, DP_U8_KDSLEN_DABPARAMETERSET_COMPLETE);
					m_rKdsDabTunerParameter.vParse( au8Buf, DP_U8_KDSLEN_DABPARAMETERSET_COMPLETE);
				}
				/*Worek around of setting concealment level to 1 for PSARCCB-10045 */
					#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
					m_rKdsDabTunerParameter.u8ConcealmentLevel = 1;
					#endif
				m_rKdsDabTunerParameter.vTrace();
				  ETG_TRACE_USR1(( " fc_dabtuner_config::dp_tclKdsDABParameterSet Value=%x"
                            ,au8Buf[0]));

#endif
#endif
                break;
            }

            /*----------------------------------------------------------------------*/

            case DAB_KDS_CM_CONFIG_STRING: {

                tsKDSEntry sKDSEntryData;

                sKDSEntryData.u16Entry       = DIA_C_U16_KDS_CM_CONFIG_STRING;
                sKDSEntryData.u16EntryLength = DIA_C_U16_KDS_CM_CONFIG_STRING_LEN;

                sKDSEntryData.u16EntryFlags  = M_KDS_ENTRY_FLAG_NONE;
                sKDSEntryData.au8EntryData[0]= 0;

                tS32 s32RetVal = OSAL_s32IORead(IOKdsHandle,(tPS8)&sKDSEntryData, sizeof(sKDSEntryData));

                if( OSAL_ERROR != s32RetVal ) {
                    m_rKdsHWConfigurationString.vParse( sKDSEntryData.au8EntryData, sKDSEntryData.u16EntryLength);
                    m_rKdsHWConfigurationString.vTrace();
                }
                else {
                    ETG_TRACE_ERR(("vReadKDSConfig(): Read KDS error %u", OSAL_u32ErrorCode()));
                }
                break;
            }
			case DAB_KDS_CODING_BAND:{
				tsKDSEntry sKDSEntryData;
				sKDSEntryData.u16Entry       = DIA_CODING_VALUE;
				sKDSEntryData.u16EntryLength = DIA_CODING_VALUE_LEN;
				sKDSEntryData.u16EntryFlags  = M_KDS_ENTRY_FLAG_NONE;
				sKDSEntryData.au8EntryData[0]= 0;
				tS32 s32RetVal = OSAL_s32IORead(IOKdsHandle,(tPS8)&sKDSEntryData, sizeof(sKDSEntryData));
				if( OSAL_ERROR != s32RetVal ) {
					m_rKdsDabCodingData.vParse(sKDSEntryData.au8EntryData, sKDSEntryData.u16EntryLength);
					m_rKdsDabCodingData.vTrace();
                }
                else {
                    ETG_TRACE_ERR(("vReadKDSConfig(): Read KDS error %u", OSAL_u32ErrorCode()));
                }
                break;
            }
#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
			case DAB_KDS_CODING_VARIANT:
			{
				dp_tclKdsCMVariantCoding oDab;
				tU8 u8KDSVal = 0xff;
				if(DP_U8_ELEM_STATUS_VALID !=oDab.u8GetADR3VariantInfo(u8KDSVal))
				{
					ETG_TRACE_ERR(("vReadKDSConfig(): DAB_KDS_CODING_VARIANT Read KDS error"));
				}
				else if(u8KDSVal == 0xff)
				{
					m_bIsSBRVariant = true;
				}
				break;
			}
#endif
#ifdef FEATURE_PSA_RCC
									 /** Reading End Of Line Parameters */
			case DAB_KDS_EOL_PARAM_RADIO:{

#ifdef DP_U16_KDSADR_TEL_FCT_RADIO 

				dp_tclKdsTEL_Fct_RADIO oDab;
				tU8 u8RadioValue= 0;
				oDab.u8GetRADIOCountryZone(u8RadioValue);
				m_rKdsPSAEOLParamRadio.u8Radio_CountryZone = u8RadioValue;
				m_rKdsPSAEOLParamRadio.bKdsDataRead = TRUE;
				oDab.u8GetRADIORDSFollowUp(u8RadioValue);
				m_rKdsPSAEOLParamRadio.bRadio_RdsFollowUp = u8RadioValue;
				m_rKdsPSAEOLParamRadio.vTrace();

#endif

				break;
										 }
			case DAB_KDS_EOL_PARAM_RADIONUM:{

#ifdef DP_U16_KDSADR_TEL_FCT_RADIONUM 

				dp_tclKdsTEL_Fct_RADIONUM oDab;				
				tU8   u8RadioNumBandActivation = 0x04;
				tU8   u8RadioNumBandActivityThreshold = 0;
				oDab.u8GetRADIONUMBandActivation(u8RadioNumBandActivation);
				oDab.u8GetRADIONUMRadioActivityThreshold(u8RadioNumBandActivityThreshold);
				m_rKdsPSAEOLParamRadioNum.u8RadioNum_BandActivation = u8RadioNumBandActivation;
				m_rKdsPSAEOLParamRadioNum.u8RadioNum_BandActivityThreshold = u8RadioNumBandActivityThreshold;
				m_rKdsPSAEOLParamRadioNum.bKdsDataRead = TRUE;
				m_rKdsPSAEOLParamRadioNum.vTrace();

#else
				tU8   u8RadioNumBandActivation = 0x03;
				tU8   u8RadioNumBandActivityThreshold = 0;
				m_rKdsPSAEOLParamRadioNum.u8RadioNum_BandActivation = u8RadioNumBandActivation;
				m_rKdsPSAEOLParamRadioNum.u8RadioNum_BandActivityThreshold = u8RadioNumBandActivityThreshold;
				m_rKdsPSAEOLParamRadioNum.bKdsDataRead = TRUE;
				m_rKdsPSAEOLParamRadioNum.vTrace();
#endif

				break;

			}
#endif

            /*----------------------------------------------------------------------*/
            case DAB_KDS_GENERIC_PARAM_SET:{
#ifndef DABTUNER_UTEST
                dp_tclKdsGenericTunerParameter oDab;			

                oDab.u8GetPresetListType(bMixedPresetList);
                if((tenPresetListType)bMixedPresetList == enPrestListType_Mixed_AllBands)
                {
                    oDab.u8GetNumberOfBanksHmiForMixedAllBands(u8NumberofBanks);
                    oDab.u8GetNumberOfPresetsPerBankForMixedAllBands(u8NumOfPresetsPerBank);
                }
                else
                {
				    oDab.u8GetNumberOfPresetsPerBankDAB(u8NumOfPresetsPerBank);
                    oDab.u8GetNumberOfBanksHmiDAB(u8NumberofBanks);	
                }
                ETG_TRACE_USR4(("vReadKDSConfig(): DAB_KDS_GENERIC_PARAM_SET: u8NumberofBanks= %d, u8NumOfPresetsPerBank = %d", u8NumberofBanks, u8NumOfPresetsPerBank));
#endif
                  break;      
               }


            default: {
                break;
        }

        /*----------------------------------------------------------------------*/
      }

      tS32 s32ReturnValue = OSAL_s32IOClose(IOKdsHandle);

       // Close KDS
       if (OSAL_OK  == s32ReturnValue) {
           ETG_TRACE_USR1(( " fc_dabtuner_config::vReadKDSConfig() -> KDS close operation succeeded."));
       }
       else {
           ETG_TRACE_USR1(( " fc_dabtuner_config::vReadKDSConfig() -> KDS close operation failed."));
       }
    }

	tVoid fc_dabtuner_config::vReadKdsEolSysConfiguration1(){
	#ifdef DP_U16_KDSADR_SYSTEMCONFIGURATION1
		tU8  u8DABStatus=0;		
		tS32 s32ReturnCode=DP_s32GetConfigItem("SystemConfiguration1","DAB",&u8DABStatus,1,FALSE); // 1 byte
		ETG_TRACE_COMP(("fc_dabtuner_config:u8DABStatus %d",u8DABStatus));
		if(s32ReturnCode != DP_S32_NO_ERR)
		{
			ETG_TRACE_COMP(("fc_dabtuner_config::DAB Enable from EOL CD Error %d",s32ReturnCode));
		}
		rDabConfig.bDabModulePresent = u8DABStatus?TRUE:FALSE;	
	#endif
	}
	
    tBool fc_dabtuner_config::bGetKdsBit(tU8 u8KdsByte, tU8 u8BitPos) const {
        if (u8BitPos>7) {
            return FALSE;
        }
        return (tBool)((u8KdsByte >> u8BitPos) & 0x01);
    }
    
    tU8 fc_dabtuner_config::u8GetKdsBits(tU8 u8KdsByte, tU8 u8BitPos, tU8 u8NumBits) const {
        if (u8BitPos>7) {
            return 0;
        }
        if (u8NumBits > (u8BitPos+7)) {
            return 0;
        }
        u8KdsByte = (tU8)(u8KdsByte>>(u8BitPos +(tU8)1 -u8NumBits));

        tU8 u8Mask= (tU8)(0xFF >> (8-u8NumBits));
        return u8Mask & u8KdsByte;
    }

    tVoid fc_dabtuner_config::vSetKdsBit(tU8 u8KdsByte, tU8 u8BitPos) const {
        u8KdsByte |= (tU8)(1L << u8BitPos);
    }

#if 0
    tVoid fc_dabtuner_config::vTraceAll() {
        ETG_TRACE_USR4(("fc_dabtuner_config:vTraceAll START"));

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("fc_dabtuner_config:REGISTRY: u32CcaPrio=%u u32DispatcherPrio=%u u32DispatcherStackSize=%u "
                        "u32IpnPrio=%u u32IpnStackSize=%u",
                        rDabConfig.u32CcaPrio, 
                        rDabConfig.u32DispatcherPrio, 
                        rDabConfig.u32DispatcherStackSize,
                        rDabConfig.u32IpnPrio,
                        rDabConfig.u32IpnStackSize));

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("fc_dabtuner_config:KDS: bDabModulePresent=%u",
                        ETG_ENUM(BOOL, rDabConfig.bDabModulePresent)));

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("  bGetSfDabDabEnable=%u bGetSfDabFmEnable=%u",
                        bGetSfDabDabEnable(),
                        bGetSfDabFmEnable()));
        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("u32GetLastFreq=%d (0x%08x) u32GetLastAsid=0x%08x pcu8GetLastSrvLabel=%16s (%02x)",
                        u32GetLastFreq(),
                        u32GetLastFreq(),
                        u32GetLastAsid(),
                        (tCString)pcu8GetLastSrvLabel(),
                        ETG_LIST_LEN(16), ETG_LIST_PTR_T8(pcu8GetLastSrvLabel())));

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("pcu8GetLastEnsLabel=%16s (%02x) u32GetLastEid=0x%04x",
                        (tCString)pcu8GetLastEnsLabel(),
                        ETG_LIST_LEN(16), ETG_LIST_PTR_T8(pcu8GetLastEnsLabel()),
                        u32GetLastEid()));

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("pcu8GetTsuMatrixDabFm=%02x",
                        ETG_LIST_LEN(32), ETG_LIST_PTR_T8(pcu8GetTsuMatrixDabFm())));                    

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("pcu8GetTsuMatrixFmDab=%02x",
                        ETG_LIST_LEN(32), ETG_LIST_PTR_T8(pcu8GetTsuMatrixFmDab())));                    

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("pcu8GetTsuConstantsDabFm=%02x",
                        ETG_LIST_LEN(6), ETG_LIST_PTR_T8(pcu8GetTsuConstantsDabFm())));                    

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("pcu8GetTsuConstantsFmDab=%02x",
                        ETG_LIST_LEN(6), ETG_LIST_PTR_T8(pcu8GetTsuConstantsFmDab())));  

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("pcu8GetRdmQualityConfig=%02x",
                        ETG_LIST_LEN(20), ETG_LIST_PTR_T8(pcu8GetRdmQualityConfig())));  

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("u8GetKdsBits(37, 7, 4)=%02x",
                        u8GetKdsBits(37, 7, 4))); 

        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("u8GetKdsVal(37)=%02x",
                        u8GetKdsVal(37))); 
        OSAL_s32ThreadWait(200);
        ETG_TRACE_USR4(("u16GetKdsVal(37)=%04x",
                        u16GetKdsVal(37))); 
                  

        for (tU32 i =0; i<sizeof(DAB_au8KdsDefaults);i++) {
            OSAL_s32ThreadWait(200);
            ETG_TRACE_USR4(("KDS[%u]=0x%02x",
                            i, DAB_au8KdsDefaults[i]));
            
        }

        ETG_TRACE_USR4(("fc_dabtuner_config:vTraceAll END"));

    }
#endif

#ifdef FEATURE_PSA_RCC
  tVoid fc_dabtuner_config::vSetEOLBasedFrequencyTable(tU8 u8ActiveBand, tU8 u8CountryZone){
		ETG_TRACE_USR4(( " fc_dabtuner_config::vSetEOLBasedFrequencyTable() u8ActiveBand %d  u8CountryZone %d",
								u8ActiveBand,u8CountryZone ));
		if(enConfigCountryZone_EU == (tenConfigCountryZone)u8CountryZone){
			switch((tenConfigBandActivation)u8ActiveBand){
				case enConfigBandActivation_LBandActive:
					m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_LBAND;
					break;
				case enConfigBandActivation_Band3Active:
				default:
//					m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_BAND3_NEW;
					break;
			}
		}
	}
#endif	

	tVoid fc_dabtuner_config::vSetDabFrequencyTable_Part1(tU8 u8BandConfig){

		if(u8BandConfig == 0){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE;
			return;
						
		}

		//kus9kor: Change in reading the upper and lower nibble
		tU8 u8Band2  = (u8BandConfig &(MASK_UPPER_NIBBLE))>>NUMBER_FOUR;
		tU8 u8Band1  = (u8BandConfig &(MASK_LOWER_NIBBLE));

		ETG_TRACE_USR4(( " fc_dabtuner_config::vSetDabFrequencyTable_Part1() tunerband1 %d  tunerBand2 %d",
								u8Band1,u8Band2 ));
		if((enCodingTunerBand1_EU_BAND3_WO_NEW == u8Band1)&&(enCodingTunerBand2_L_BAND == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE;
		}
		else if((enCodingTunerBand1_EU_BAND3 == u8Band1)&&(enCodingTunerBand2_L_BAND == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_NEW;
		}
		else if ((enCodingTunerBand1_EU_BAND3_WO_NEW == u8Band1)&&(enCodingTunerBand2_OFF == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_BAND3;
		}
		else if ((enCodingTunerBand1_EU_BAND3 == u8Band1)&&(enCodingTunerBand2_OFF == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_BAND3_NEW;
		}
		else if ((enCodingTunerBand1_OFF == u8Band1)&&(enCodingTunerBand2_L_BAND == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_LBAND;
		}
		else
		{
			vSetDabFrequencyTable_Part2(u8Band1,u8Band2);
		}
		ETG_TRACE_USR4(( " fc_dabtuner_config::vSetDabFrequencyTable_Part1() SetFrequencyTable %x",
								ETG_CENUM(tenMeca_RdmFrequencyTable,m_u8FreqeuncyTable)));
	
	}

    tVoid fc_dabtuner_config::vSetFrequencyTableBasedOnVariantSelection()
    {
        tU8 u8VariantType = 0;
        tS32 s32Res = DP_s32GetConfigItem("AIVIVariantCoding", "VariantSelection", &u8VariantType, sizeof(tU8));
        if (DP_S32_NO_ERR == s32Res)
        {
            ETG_TRACE_USR4((" AIVIVariantCoding_VariantSelection : 0x0%d", u8VariantType));
            if (u8VariantType == enKdsAIVIVariantType_DA)
                m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQ_LIST_EUROPE_BAND3_5A12D_NEW;
        }
        else
        {
            ETG_TRACE_USR4((" AIVIVariantCoding_VariantSelection error"));
        }
    }

    tVoid fc_dabtuner_config::vSetDabFrequencyTable_Part2(tU8 u8Band1, tU8 u8Band2){

		 if ((enCodingTunerBand1_CANADA_L_BAND == u8Band1)&&(enCodingTunerBand2_OFF == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_CANADA;
		}
		else if ((enCodingTunerBand1_KOREA_BAND3 == u8Band1)&&(enCodingTunerBand2_OFF == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_KOREA;
		}
		else if ((enCodingTunerBand1_CHINA_BAND3 == u8Band1)&&(enCodingTunerBand2_L_BAND == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_CHINA;
		}
		else if ((enCodingTunerBand1_CHINA_BAND3 == u8Band1)&&(enCodingTunerBand2_OFF == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_CHINA_BAND3;
		}
		else if ((enCodingTunerBand1_NEWZEALAND_BAND3 == u8Band1)&&(enCodingTunerBand2_OFF == u8Band2)){
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_NEWZEALAND;
		}		
		else{
			m_u8FreqeuncyTable = enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE;
		}
		ETG_TRACE_USR4(( " fc_dabtuner_config::vSetDabFrequencyTable_Part2() SetFrequencyTable %x",
								ETG_CENUM(tenMeca_RdmFrequencyTable,m_u8FreqeuncyTable)));

	}

 
}


