/************************************************************************
* FILE:         DAB_clientBase.cpp
* PROJECT:      g3g
* SW-COMPONENT: FC_DABTUNER
*----------------------------------------------------------------------
*
* DESCRIPTION: Definition of generic handlers to send method result,
 *         service handlers, update clients, send status message to clients.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                      | Modification
* 06.11.2009  Ruchi Mishra(RBEI/ECG3) 	   Original version
*************************************************************************/

#include "fc_dabtuner_util.h"
#include "fc_dabtuner_mainProxy.h"

#include "fc_dabtuner_clientBase.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_CLIENT_BASE
#include "trcGenProj/Header/fc_dabtuner_clientBase.cpp.trc.h"
#endif

/*
Required:
1.) Dispatch on fid+opCode
2.) automatic handling of upreg(->get)+relupreg
3.) Broadcast(fid) to registered clients

*/
using namespace DAB;


/*---------------------------------------------------------------------------*/
/*     FUNCTION:    DAB_ClientBase
*/
/*!    @brief      constructor
*                  
*     @Param       poMain, u16ServiceId
*     @return      None
*    Ver History:
*    Initial version:
*    27.10.2009                                      Ruchi Mishra(RBEI/ECG3)
*/
/*---------------------------------------------------------------------------*/
DAB_ClientBase::DAB_ClientBase(){};
/*---------------------------------------------------------------------------*/
/*     FUNCTION:    DAB_ClientBase
*/
/*!    @brief      destructor
*                  
*     @Param       poMain, u16ServiceId
*     @return      None
*    Ver History:
*    Initial version:
*    27.10.2009                                      Ruchi Mishra(RBEI/ECG3)
*/
/*---------------------------------------------------------------------------*/
DAB_ClientBase::~DAB_ClientBase() {};


tU16 DAB_ClientBase::u16RegisterService(tVoid) {
    trClientConfig &roConf=roGetConfig();
  /* --- 
   registration to be done is proofed in base class
   */
   tU16 u16RegID = AMT_C_U16_REGID_INVALID;
   
   /* ---
   try to register
   */
   /* +++ register to the keyboard service */
   if((roConf._u16ServiceId == CCA_C_U16_SRV_DIAGLOG)||
		(roConf._u16ServiceId == CCA_C_U16_SRV_CLOCK))
   {
	   u16RegID = DAB_mainProxy::bRegisterAsync( roConf._u16ServiceId,
	                                                    roConf._u16MajorVersion,
	                                                    roConf._u16MinorVersion,
	                                                    roConf._u16SrvAppId);
   }
   else
   {
	   u16RegID = DAB_mainProxy::u16RegisterService( roConf._u16ServiceId,
                                                 roConf._u16MajorVersion,
                                                 roConf._u16MinorVersion,
                                                 roConf._u16SrvAppId);
   }

   ETG_TRACE_USR4(("DAB_ClientBase::u16RegisterService: _u16ServiceId: 0x%x Ver %d.%d AppId= 0x%x  u16RegId= %d", 
       roConf._u16ServiceId, roConf._u16MajorVersion, roConf._u16MinorVersion, roConf._u16SrvAppId, u16RegID )); 

   return u16RegID; // will be stored in _u16RegID from base class.

}

tVoid DAB_ClientBase::vUnregisterService(tVoid) {
    trClientConfig &roConf=roGetConfig();

    DAB_mainProxy::vUnregisterService(roConf._u16ServiceId, roConf._u16SrvAppId );
}

tU16 DAB_ClientBase::u16GetServiceId() {
    return roGetConfig()._u16ServiceId;
}

tU16 DAB_ClientBase::u16GetRegId() {
    return roGetConfig()._u16RegID;
}


tVoid DAB_ClientBase::vOnNewAppState(tU32 u32AppState)
{
    trClientConfig &roConf=roGetConfig();

   switch(u32AppState)
   {
   case AMT_C_U32_STATE_NORMAL:
   case AMT_C_U32_STATE_DIAGNOSIS:
   case AMT_C_U32_STATE_PAUSE:
      {    
         if(roConf._u16RegID == AMT_C_U16_REGID_INVALID){
            roConf._u16RegID = u16RegisterService();         
         }
         break;
      }
   case AMT_C_U32_STATE_RECEIVE_READY:
      // receive ready is not in use yet but basically behaves same as pause
   case AMT_C_U32_STATE_OFF:
      {
         vUnregisterService();
         roConf._u16RegID = AMT_C_U16_REGID_INVALID;       
         break;
      }
   default:
      {
         break;
      }
   }
}


tVoid DAB_ClientBase::vRegisterFid(tU16 u16Fid) {
    trClientConfig &roConf=roGetConfig();
   if ( roConf._u16RegID == AMT_C_U16_REGID_INVALID )
   {
      return;
   }
   ETG_TRACE_USR3(("%10s: vRegisterFid: %d",
                   roConf._pcName,
                   u16Fid)); 
   gm_tclEmptyMessage oUpRegMessage
       (          
        roConf._u16MyAppId,     /* --- AppID of this application */
        roConf._u16SrvAppId,         /* --- AppID of the other Server */
        roConf._u16RegID,                    /* --- RegId for the service (see bRegisterForService ) */
        0,                            /* --- always 0 */
        roConf._u16ServiceId,            /* --- the SID of the service    */
        u16Fid,       /* --- the FID to register for   */
        AMT_C_U8_CCAMSG_OPCODE_UPREG  /* --- this is an Upreg-message! ---*/
        );
   vPostMsg(&oUpRegMessage);

}

tVoid DAB_ClientBase::vRegisterAllFIDs(tVoid)
{
    trClientConfig &roConf=roGetConfig();

   for (int s32index=0; s32index<roConf._u8NumUpRegs;++s32index){
       vRegisterFid(roConf._pu16UpregFids[s32index]);
   }
}


tVoid DAB_ClientBase::vUnregisterFid(tU16 u16Fid) {
    trClientConfig &roConf=roGetConfig();
    if ( roConf._u16RegID == AMT_C_U16_REGID_INVALID )
    {
        return;
    }
   ETG_TRACE_USR3(("%10s: vUnregisterFid: %d",
                   roConf._pcName,
                   u16Fid)); 

    gm_tclEmptyMessage oRelupRegMessage
        (          
         roConf._u16MyAppId,     /* --- AppID of this application */
         roConf._u16SrvAppId,         /* --- AppID of the other Server */
         roConf._u16RegID,                    /* --- RegId for the service (see bRegisterForService ) */
         0,                            /* --- always 0 */
         roConf._u16ServiceId,            /* --- the SID of the service    */
         u16Fid,       /* --- the FID to register for   */
         AMT_C_U8_CCAMSG_OPCODE_RELUPREG  /* --- this is an Upreg-message! ---*/
         );
    vPostMsg(&oRelupRegMessage);
    
}

tVoid DAB_ClientBase::vUnregisterAllFIDs(tVoid)
{
    trClientConfig &roConf=roGetConfig();
   for (int s32index=0; s32index<roConf._u8NumUpRegs;++s32index){
       vUnregisterFid(roConf._pu16UpregFids[s32index]);
   }
}

/*************************************************************************
*
* FUNCTION: tVoid vOnServiceState
* 
* DESCRIPTION: handle a state change of the service we use
*
* PARAMETER: 
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid DAB_ClientBase::vOnServiceState(tU16 u16ServiceId,
                                      tU16 /*u16ServerId*/,
                                      tU16 u16RegisterId,
                                      tU8  u8ServiceState,
                                      tU16 /*u16SubId*/)
{
    trClientConfig &roConf=roGetConfig();
    (tVoid)u16ServiceId;
   switch(u8ServiceState)
   {
   case AMT_C_U8_SVCSTATE_AVAILABLE:
      {
         /* ---
         Let's start with communication to all functions we want from the service.
         */
    	  if((roConf._u16ServiceId == CCA_C_U16_SRV_DIAGLOG)||
			(roConf._u16ServiceId == CCA_C_U16_SRV_CLOCK))
    	  {
    		  ETG_TRACE_USR3(("CCA_C_U16_SRV_DIAGLOG: %d u16RegisterId= %d",roConf._u16RegID,u16RegisterId));
    		  roConf.vSetRegID(u16RegisterId);
    	  }
         if( roConf._u16RegID != AMT_C_U16_REGID_INVALID)
         {
             vRegisterAllFIDs();
             vOnSrvUp();
         }
         break;
      }
   case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
      {
         roConf._u16FIDUpregsOpen=0;
         /* ---
         make sure notification tables are cleaned. 
         Unregister works also if service not avail.
         */
         if( roConf._u16RegID != AMT_C_U16_REGID_INVALID)
         {
            vUnregisterAllFIDs();
            vOnSrvDown(); 
         }
         break;
      }
   case AMT_C_U8_SVCSTATE_REG_INVALID:
      {
         roConf._u16FIDUpregsOpen=0;
         /* ---
         register again.
         */
         roConf._u16RegID = u16RegisterService();
         break;
      }
   default:
      {
          ETG_TRACE_ERR(("vOnServiceState: unknown SVCSTATE=0x%x",
                         u8ServiceState));
      }
   }//switch
}









tVoid DAB_ClientBase::vOnUnknownMessage(amt_tclBaseMessage* poMessage) {
	amt_tclServiceData oServiceData(poMessage);

    ETG_TRACE_COMP(("%10s::vOnUnknownMessage:fid=%d opCode=%d MsgType=%x",
                    roGetConfig()._pcName,
                    oServiceData.u16GetFunctionID(),
                    oServiceData.u8GetOpCode(),
                    poMessage->u8GetType())); 

}
/* todo provide implementation of some helpers as in serviceHandler of fc,
No separate class is needed, but you may add them to a separated cpp-file(DAB_ServiceHandler.cpp)
*/



tVoid DAB_ClientBase::vPostMsg(amt_tclServiceData* poMessage) const{
    ail_tenCommunicationError enCommError = AIL_EN_N_NO_ERROR;
    enCommError=DAB_mainProxy::enPostMsg(poMessage);
    if (AIL_EN_N_NO_ERROR != enCommError) {
        ETG_TRACE_USR4(("Result PostMessage: %d",
                        enCommError)); 
    }
}




// craete generic fns to send fi-objects according to ServiceHandler of fc, use fn-template if neccessary
tVoid DAB_ClientBase::vSendFiMessage(fi_tclMessageBase const & roSendFiObj, tU16 u16CmdCount) {
    trClientConfig &roConf=roGetConfig();

    if (roConf._u16RegID == AMT_C_U16_REGID_INVALID) {
        return;
    }
   fi_tclVisitorMessage oOutVisitorMsg( roSendFiObj.corfoGetTypeBase() );

   /* Set the CCA message information */
   oOutVisitorMsg.vInitServiceData
        (
          roConf._u16MyAppId,                  /* Source app-ID   */
          roConf._u16SrvAppId,     /* Dest. app-ID    */
          AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,        /* stream type     */ 
          0,                    /* stream counter  */
          roConf._u16RegID,                  /* Register ID     */
          u16CmdCount,                  /* Command counter */
          roConf._u16ServiceId,                  /* Service-ID      */
          roSendFiObj.u16GetFunctionID(),                  /* Function-ID     */
          roSendFiObj.u8GetOpCode(),             /* Opcode          */
          0,                    /* ACT             */
          0,                    /* Source sub-ID   */
          0                     /* Dest. sub-ID    */
        );

   vPostMsg(&oOutVisitorMsg);
   
}




tVoid  DAB_ClientBase::vTraceRequest(tPCChar name, trAdressing const *prAdressing)
{
   if (OSAL_NULL != prAdressing)
   {
      ETG_TRACE_USR3(("%10s::%10s: OpCode: %d, Fid: %d, Src: %d",
                      roGetConfig()._pcName,
                      name,
                      prAdressing->_u8OpCode,
                      prAdressing->_u16Fid,
                      prAdressing->_u16AppId)); 
   }
}

tVoid DAB_ClientBase::vTraceError(trAdressing const *prAdressing, tenClientErrorCode enClientErrorCode) {

    ETG_TRACE_COMP(("%20s::vTraceError:fid=%d opCode=%d errorCode=%d",
                    roGetConfig()._pcName,
                    prAdressing->_u16Fid,
                    prAdressing->_u8OpCode,
                    ETG_CENUM(tenClientErrorCode, enClientErrorCode))); 

}
