/************************************************************************
 * FILE:        rfc3072_helper.c
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  header contents of rfc3072_helper
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#ifndef __RFC3072_HELPER_H__
#define __RFC3072_HELPER_H__
/*******************************************************************************
 * Modulename:         rfc3072_helper
 * Module function:    helper Functions for RFC3072
 *
 *
 * Project:            AaRS
 *
 * Copyright (C) Robert Bosch GmbH, 2011
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is forbidden.
 *******************************************************************************
 */
#ifdef CONFIG_COMPILE_DABLIB
#include "dablibtypes.h"
#endif

#include "DBAPITypes.h"

/*
 * !!! disable the module test for production code !!!
 */
#undef ENABLE_RFC3072_MODULE_TEST

/*
 * 0x21 = Structured Chunk
 * 0x
 */
#define RFC3072_FLAGS_ARRAY                     (1<<6)
#define RFC3072_FLAGS_SHORT                     (1<<5)
#define RFC3072_FLAGS_DATATYPE_MASK             (7)
#define RFC3072_FLAGS_DATATYPE_STRUCTURE        (1)
#define RFC3072_FLAGS_DATATYPE_BITSTRING        (2)
#define RFC3072_FLAGS_DATATYPE_NUMERIC          (3)
#define RFC3072_FLAGS_DATATYPE_CHARACTER        (4)

#define RFC3072_DATA_ID_HEADLINE                (0x0fff)
#define RFC3072_DATA_ID(_id_)                   ((_id_)&(0xfff))
#define RFC3072_TABLE_LEVEL(_id_)               (((_id_)>>12)&(0xf))


#define RFC3072_Structure_Chunk(_Name_,_ID_,_Size_)                     \
uint32 tmp_ ## _Name_ ## _1 = (_ID_) ;      /* make lint happy */       \
uint32 tmp_ ## _Name_ ## _2 = (_Size_) ;    /* make lint happy */       \
uint8 (_Name_)[2+1+3] = {                                               \
    (((tmp_ ## _Name_ ## _1)>>8)&0xff),                                 \
    (((tmp_ ## _Name_ ## _1)>>0)&0xff),                                 \
    RFC3072_FLAGS_DATATYPE_STRUCTURE ,                                  \
    (((tmp_ ## _Name_ ## _2)>>16)&0xff),                                \
    (((tmp_ ## _Name_ ## _2)>>8)&0xff),                                 \
    (((tmp_ ## _Name_ ## _2)>>0)&0xff)  }

#define RFC3072_Structure_Chunk_SetSize(_PtrName_,_Offset_,_Size_)      \
    (_PtrName_)[(_Offset_)+3] = (((_Size_)>>16)&0xff);                  \
    (_PtrName_)[(_Offset_)+4] = (((_Size_)>>8)&0xff);                   \
    (_PtrName_)[(_Offset_)+5] = (((_Size_)>>0)&0xff);


#define RFC3072_NUMERIC_Short_Chunk(_Name_,_ID_,_Value_)                \
uint8 (_Name_)[2+1+3] = {                                               \
    (((_ID_)>>8)&0xff),                                                 \
    (((_ID_)>>0)&0xff),                                                 \
    RFC3072_FLAGS_SHORT | RFC3072_FLAGS_DATATYPE_NUMERIC ,              \
    (((_Value_)>>16)&0xff),                                             \
    (((_Value_)>>8)&0xff),                                              \
    (((_Value_)>>0)&0xff)  }

#define RFC3072_NUMERIC_UINT32_Chunk(_Name_,_ID_,_Value_)               \
uint8 (_Name_)[2+1+3+4] = {                                             \
    (((_ID_)>>8)&0xff),                                                 \
    (((_ID_)>>0)&0xff),                                                 \
    RFC3072_FLAGS_DATATYPE_NUMERIC ,                                    \
    0x00,                                                               \
    0x00,                                                               \
    0x04,                                                               \
    (((_Value_)>>24)&0xff),                                             \
    (((_Value_)>>16)&0xff),                                             \
    (((_Value_)>>8)&0xff),                                              \
    (((_Value_)>>0)&0xff) }

#define RFC3072_ARRAY_ChunkHeader(_Name_,_ID_,_Size_)                   \
uint8 (_Name_)[2+1+3] = {                                               \
    (((_ID_)>>8)&0xff),                                                 \
    (((_ID_)>>0)&0xff),                                                 \
    RFC3072_FLAGS_DATATYPE_BITSTRING ,                                  \
    (((_Size_)>>16)&0xff),                                              \
    (((_Size_)>>8)&0xff),                                               \
    (((_Size_)>>0)&0xff)  }

#define RFC3072_STRING_ChunkHeader(_Name_,_ID_,_Size_)                  \
uint8 (_Name_)[2+1+3] = {                                               \
    (((_ID_)>>8)&0xff),                                                 \
    (((_ID_)>>0)&0xff),                                                 \
    RFC3072_FLAGS_DATATYPE_CHARACTER ,                                  \
    (((_Size_)>>16)&0xff),                                              \
    (((_Size_)>>8)&0xff),                                               \
    (((_Size_)>>0)&0xff)  }

/*some types*/
typedef struct
{
  uint16 chunkID;
  uint8 flags;
  uint32 length;
  uint8* content;
} SDXFChunkInfo;

typedef struct
{
  uint32 length;
  uint8* content;
} SDXFBlob;

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_id</b>

 Returns chunk ID

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk identification
 *******************************************************************************
 */
uint16
sdxf_chunk_get_id(const uint8* sdxf_data);

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_flags</b>

 Returns chunk flags

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk flags
 *******************************************************************************
 */
uint8
sdxf_chunk_get_flags(const uint8* sdxf_data);

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_chunk_body_length</b>

 Returns the length of chunk data

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk body length
 *******************************************************************************
 */
uint32
sdxf_chunk_get_chunk_body_length(uint8* sdxf_data);

/*!
 *******************************************************************************
 @brief        <b>sdxf_get_chunk_body_pointer</b>

 Returns a pointer to a chunk content

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk body pointer
 *******************************************************************************
 */
uint8*
sdxf_get_chunk_body_pointer(uint8* sdxf_data);

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_info</b>

 Returns a chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk info structure
 *******************************************************************************
 */
SDXFChunkInfo
sdxf_chunk_get_info(uint8* sdxf_data);

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_datatype</b>

 Returns a chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       data type of chunk
 *******************************************************************************
 */
int
sdxf_chunk_get_datatype(uint8* sdxf_data);

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_blob</b>

 Returns the blob (content) of the chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk blob pointer
 *******************************************************************************
 */
SDXFBlob
sdxf_chunk_get_blob(uint8 *chunk);

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_numeric_sint32</b>

 Returns the numeric value of the chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       int32 numeric value
 *******************************************************************************
 */
sint32
sdxf_chunk_get_numeric_sint32(uint8 *chunk);

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_numeric_uint32</b>

 Returns the numeric value of the chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       uint32 numeric value
 *******************************************************************************
 */
uint32
sdxf_chunk_get_numeric_uint32(uint8 *chunk);

/*!
 *******************************************************************************
 @brief        <b>sdxf_row_get_chunk</b>

 Return the pointer to the chunk of the sdxf row pointed
 by row identified by chunk_idx

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @param[in]    row, pointer to row chunk
 @param[in]    column_idx, row number (0,1,2,3,4,...)
 @return       pointer to chunk
 *******************************************************************************
 */
uint8*
sdxf_row_get_chunk(const uint8* sdxf_data, uint32 sdxf_length, uint8* row,
    uint32 column_idx);

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_chunk</b>

 Returns the ptr to chunk identified by row_idx and column_idx

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @param[in]    row_idx, column number (0,1,2,3,4,...)
 @param[in]    column_idx, row number (0,1,2,3,4,...)
 @return       pointer to chunk
 *******************************************************************************
 */
uint8*
sdxf_table_get_chunk(uint8* sdxf_data, uint32 sdxf_length, uint32 row_idx,
    uint32 column_idx);

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_num_columns</b>

 Returns the number of comlumns

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @return       number of columns in table
 *******************************************************************************
 */
uint32
sdxf_table_get_num_columns(uint8* sdxf_data, uint32 sdxf_length);

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_num_rows</b>

 Returns the number of rows

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @return       number of rows in table
 *******************************************************************************
 */
uint32
sdxf_table_get_num_rows(uint8* sdxf_data, uint32 sdxf_length);

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_row</b>

 Return the pointer to the row of the sdxf table pointed
 by sdxf data

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @param[in]    row_idx, row index
 @return       pointer to row chunk
 *******************************************************************************
 */
uint8*
sdxf_table_get_row(uint8* sdxf_data, uint32 sdxf_length, uint32 row_idx);

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_headline_row</b>

 Return the pointer to the headline row of the sdxf table pointed
 by sdxf data

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @return       pointer to headline row chunk
 *******************************************************************************
 */
uint8*
sdxf_table_get_headline_row(uint8* sdxf_data, uint32 sdxf_length);

/*!
 *******************************************************************************
 @brief        <b>sdxf_labelblob_2_apilabel</b>

 Converts the label blob into the label structure

 @param[in]    label, pointer to APILabel
 @param[in]    label_blob, pointer to SDXF label blob
 @return       label valid or not
 *******************************************************************************
 */
uint32
sdxf_labelblob_2_apilabel(APILabel *label, SDXFBlob label_blob);

/*!
 *******************************************************************************
 @brief        <b>sdxf_apiserviceblob_2_apiservice</b>

 Converts the api_service blob into the api_service structure

 @param[in]    serv, pointer to APIService
 @param[in]    api_service_blob, pointer to SDXF service blob
 @return       success
 *******************************************************************************
 */
uint32
sdxf_apiserviceblob_2_apiservice(APIService *serv, SDXFBlob api_service_blob);

/*!
 *******************************************************************************
 @brief        <b>sdxf_apiserviceblob_2_apiservicetype</b>

 Converts the api_service blob into the api_service_type

 @param[in]    api_serv_type, api service type
 @param[in]    api_service_blob, pointer to SDXF service blob
 @return       success
 *******************************************************************************
 */
uint32
sdxf_apiserviceblob_2_apiservicetype(uint8 *api_serv_type,
    SDXFBlob api_service_blob);

/*!
 *******************************************************************************
 @brief        <b>sdxf_check_structure</b>

 Check the chunk structure

 @param[in]    sdxf_data        pointer to SDXF structure
 @param[in]    sdxf_length      length of SDXF structure
 @return       -1 => error
 *******************************************************************************
 */
int
sdxf_check_structure(uint8* sdxf_data, uint32 sdxf_length);

/*!
 *******************************************************************************
 @brief        <b>sdxf_print_chunk</b>

 Print a SDXF chunk

 @return       pointer to first char in line buffer
 *******************************************************************************
 */
#define SDXF_CONFIG_MAX_NUM_COLUMNS 256
#define SDXF_CONFIG_MIN_BUFFER_SIZE 32

typedef enum {
    COLUMN_FORMAT_MODE_SIGNED_DEZIMAL=1,
    COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL,
    COLUMN_FORMAT_MODE_UINT_HEX,
    COLUMN_FORMAT_MODE_UINT8_HEX,
    COLUMN_FORMAT_MODE_UINT16_HEX,
    COLUMN_FORMAT_MODE_UINT32_HEX,
    COLUMN_FORMAT_MODE_AUTO_HEX,
    COLUMN_FORMAT_MODE_DABLABEL,
    COLUMN_FORMAT_MODE_UNDEF=0
} tSDXFPrintColumnFormatMode;

typedef struct
{
	const char* name;
	tSDXFPrintColumnFormatMode 	mode;
}tSDXFPrintColumnFormatConfig;

#define SDXF_WELL_KNOWN_COLUMN_FORMAT_HINTS					                    \
                        { "o", COLUMN_FORMAT_MODE_AUTO_HEX },   		                    \
                        { "e", COLUMN_FORMAT_MODE_AUTO_HEX },		                            \
                        { "s", COLUMN_FORMAT_MODE_AUTO_HEX },		                            \
                        { "c", COLUMN_FORMAT_MODE_AUTO_HEX },		                            \
                        { "u", COLUMN_FORMAT_MODE_AUTO_HEX },		                            \
		{ "t", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },				\
                        { "f", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },				    \
                        { "f0", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		                    \
                        { "f1", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		                    \
                        { "f2", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		                    \
                        { "f3", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		                    \
                        { "subchid", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		            \
                        { "frequency", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		            \
                        { "freq", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		                    \
                        { "importance", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		            \
                        { "imp", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		                    \
                        { "user_app_type", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		            \
                        { "ua", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },	                            \
                        { "uat", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		                    \
                        { "uad", COLUMN_FORMAT_MODE_AUTO_HEX },		                            \
                        { "user_app_x_pad_data", COLUMN_FORMAT_MODE_AUTO_HEX },	                    \
                        { "uaxpadd", COLUMN_FORMAT_MODE_AUTO_HEX },		                    \
                        { "data", COLUMN_FORMAT_MODE_AUTO_HEX },				    \
                        { "bytes", COLUMN_FORMAT_MODE_AUTO_HEX },				    \
                        { "scids", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },			            \
                        { "id", COLUMN_FORMAT_MODE_UINT16_HEX },				    \
                        { "pi", COLUMN_FORMAT_MODE_UINT16_HEX },				    \
                        { "scidi", COLUMN_FORMAT_MODE_UINT16_HEX },				    \
                        { "w_scidi", COLUMN_FORMAT_MODE_UINT16_HEX },			            \
                        { "rid", COLUMN_FORMAT_MODE_UINT16_HEX },				    \
                        { "eid", COLUMN_FORMAT_MODE_UINT16_HEX },				    \
                        { "sid", COLUMN_FORMAT_MODE_AUTO_HEX },					    \
                        { "ecc", COLUMN_FORMAT_MODE_UINT8_HEX },				    \
                        { "ensecc", COLUMN_FORMAT_MODE_UINT8_HEX },				    \
                        { "servecc", COLUMN_FORMAT_MODE_UINT8_HEX },			            \
		{ "lsn", COLUMN_FORMAT_MODE_AUTO_HEX },                 \
		{ "dbkey", COLUMN_FORMAT_MODE_AUTO_HEX },               \
                        { "asuflags", COLUMN_FORMAT_MODE_AUTO_HEX },			            \
                        { "asuf", COLUMN_FORMAT_MODE_AUTO_HEX },		                    \
                        { "asu", COLUMN_FORMAT_MODE_AUTO_HEX },					    \
                        { "utc", COLUMN_FORMAT_MODE_AUTO_HEX },					    \
                        { "mjd", COLUMN_FORMAT_MODE_AUTO_HEX },					    \
                        { "pty", COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL },			            \
                        { "s_intpty", COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL },	                    \
                        { "s_comppty", COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL },	                    \
                        { "d_intpty", COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL },	                    \
                        { "d_comppty", COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL },	                    \
                        { "language", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		            \
                        { "lng", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },			            \
                        { "v", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },				    \
                        { "i", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },				    \
		{ "fst", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },           \
		{ "sig", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },           \
		{ "fav", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },           \
                        { "age", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },			            \
                        { "version", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		            \
                        { "int_table_id", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },	                    \
                        { "itid", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },			            \
                        { "scty", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },			            \
                        { "r_ensemble", COLUMN_FORMAT_MODE_AUTO_HEX },			            \
                        { "r_ens", COLUMN_FORMAT_MODE_AUTO_HEX },				    \
                        { "r_service", COLUMN_FORMAT_MODE_AUTO_HEX },			            \
                        { "r_serv", COLUMN_FORMAT_MODE_AUTO_HEX },				    \
		{ "quality", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },		\
		{ "nPS", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },           \
		{ "nDS", COLUMN_FORMAT_MODE_SIGNED_DEZIMAL },           \
                        { "q", COLUMN_FORMAT_MODE_UINT16_HEX },					    \
                        { "quality_word", COLUMN_FORMAT_MODE_UINT16_HEX },		            \
                        { "qw", COLUMN_FORMAT_MODE_UINT16_HEX },				    \
                        { "label", COLUMN_FORMAT_MODE_DABLABEL },				    \
                        { "reglabel", COLUMN_FORMAT_MODE_DABLABEL },			            \
                        { "enslabel", COLUMN_FORMAT_MODE_DABLABEL },			            \
                        { "servlabel", COLUMN_FORMAT_MODE_DABLABEL },			            \
                        { "complabel", COLUMN_FORMAT_MODE_DABLABEL },			            \
                        { "ualabel", COLUMN_FORMAT_MODE_DABLABEL },				    \
                        { "sihash", COLUMN_FORMAT_MODE_AUTO_HEX },				    \


typedef struct
{
	char*   tmp_buffer;
	uint32  tmp_buffer_size;

	char* line_buffer;
	uint32  line_buffer_size;
	uint32  line_buffer_used;
	uint32  line_buffer_is_full;                         

	struct
	{
		tSDXFPrintColumnFormatConfig	format_hint;
		uint32      					width;
	}column[SDXF_CONFIG_MAX_NUM_COLUMNS];

	uint32 							state;

	uint32 							max_num_columns;
	uint32							table_num_columns;

	tSDXFPrintColumnFormatConfig* 	column_format_hint;

	boolean							print_header;

	char 							headline_splitter;
	char 							column_splitter;
} tSDXFPrintChunkConfig;

char* sdxf_print_chunk( tSDXFPrintChunkConfig* config, uint32 sdxf_length, uint8* sdxf_data );

#if 0
static void
my_meca_dbapi_query_callback( ObjectRef oref,
							  uint8  b_response,
							  uint8  b_tag,
							  uint8  b_stmt_id,
							  uint8  b_rfu1,
							  uint16 w_num_result_tables,
							  uint16 w_num_result_rows,
							  uint16 w_num_result_columns,
							  uint8  b_result_protocol,
							  uint8  b_result_option,
							  uint32 l_result_length,
							  uint8* r_result )
 {
	switch (b_result_protocol)
	{
		case DBAPI_R_QUERY__B_RESULT_PROTOCOL__PARA_SDXF_TABLE:
		case DBAPI_R_QUERY__B_RESULT_PROTOCOL__PARA_SDXF_ROW:
		case DBAPI_R_QUERY__B_RESULT_PROTOCOL__PARA_SDXF_ROW_DONE:
			  {
					tSDXFPrintColumnFormatConfig column_format_hint[] = {
							SDXF_WELL_KNOWN_COLUMN_FORMAT_HINTS
							{ NULL, COLUMN_FORMAT_MODE_UNDEF }
					};

					char* line = NULL;
					char tmp_buffer[128];	/* temporary cell buffer */
					char line_buffer[1024]; /* line buffer */
					tSDXFPrintChunkConfig print_config;
					memset(&print_config,0,sizeof(tSDXFPrintChunkConfig));
					print_config.state = 0;
					print_config.print_header = TRUE;
					print_config.tmp_buffer = tmp_buffer;
					print_config.tmp_buffer_size = sizeof(tmp_buffer);
					print_config.line_buffer = line_buffer;
					print_config.line_buffer_size = sizeof(line_buffer);
					print_config.column_format_hint = column_format_hint;					
					print_config.headline_splitter = '=';
					print_config.column_splitter = '|';
					print_config.max_num_columns = SDXF_CONFIG_MAX_NUM_COLUMNS;
					while( (line=sdxf_print_chunk(&print_config,l_result_length,r_result)) != NULL )
					{
						printf("LINE= %s\n",line);
					}
			  }
}
#endif

/*!
 *******************************************************************************
 @brief        <b>sdxf_module_test</b>

 Test this functionality

 @return       1 if no error found
 *******************************************************************************
 */
#ifdef ENABLE_RFC3072_MODULE_TEST
int
sdxf_module_test();
#endif /* ENABLE_RFC3072_MODULE_TEST */

#endif /* __RFC3072_HELPER_H__ */
/*******************************************************************************
 * EOF
 *******************************************************************************
 */

