/************************************************************************
 * FILE:        dabmeca_tsu.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabmeca_tsu
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      	| Author                       | Modification
				  
				
 *************************************************************************/

#ifndef DABMECA_TSU_HPP
#define DABMECA_TSU_HPP

#include "fc_dabtuner_util.h"
#include "dabdrv_meca.h"
#include "dabmeca_db.hpp"
#include "dabdrv_typeEnums.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_MSG
#include "trcGenProj/Header/dabmeca_tsu.hpp.trc.h"
#endif

namespace DAB {

    typedef enum {
        enMeca_TsuConfigFlag_ENABLE_SERVICE_LINKING = 0x01,
        enMeca_TsuConfigFlag_IGNORE_DAB_DAB_SF = 0x02,
        enMeca_TsuConfigFlag_DISABLE_PSID_CNG_CNT_RESET = 0x04,
        enMeca_TsuConfigFlag_USE_5_8_FOR_DAB_QUALITY_THRESHOLDS = 0x08,
        enMeca_TsuConfigFlag_STALL_LONG_TIME_QUALITY_DOWN = 0x10,
        enMeca_TsuConfigFlag_USE_TAU_LONG_DOWN_FACTOR_V_1 = 0x20,
        enMeca_TsuConfigFlag_DISABLE_FM_LINKING_BUG_FIX = 0x40,
        enMeca_TsuConfigFlag_DISABLE_DRN_WORKAROUND = 0x80
    } tenMeca_TsuConfigFlag;

    // _DAB_C_TSU_SET_MODE
    struct trMeca_CTsuSetMode:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CTsuSetMode():
            enTsuMode(DAB_enMtcMode_Off),
            enTsuConfigFlags((tenMeca_TsuConfigFlag)0)
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_TSU_SET_MODE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 2;
            pRawOutput->au8Data[0]= (tU8)enTsuMode;
            pRawOutput->au8Data[1]= (tU8)enTsuConfigFlags;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CTsuSetMode:enTsuMode=%x enTsuConfigFlags=%x",
                            ETG_CENUM(DAB_tenMtcMode, enTsuMode),
                            ETG_CENUM(tenMeca_TsuConfigFlag, enTsuConfigFlags)));
        }
    
        DAB_tenMtcMode enTsuMode;
        tenMeca_TsuConfigFlag enTsuConfigFlags;

    };


    // DAB_*_R_TSU_SET_MODE
    struct trMeca_RTsuSetMode:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTsuMode=(DAB_tenMtcMode)pu8Data[0];
            enTsuConfigFlags=(tenMeca_TsuConfigFlag)pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RTsuSetMode:enTsuMode=%x enTsuConfigFlags=%x",
                            ETG_CENUM(DAB_tenMtcMode, enTsuMode),
                            ETG_CENUM(tenMeca_TsuConfigFlag, enTsuConfigFlags)));
        };
    
        DAB_tenMtcMode enTsuMode;
        tenMeca_TsuConfigFlag enTsuConfigFlags;
    };

    typedef enum {
        enMeca_TsuTunerId_INVALID = 0x00, // not defined in meca
        enMeca_TsuTunerId_MAIN_DAB_TUNER = 0x10,
        enMeca_TsuTunerId_MAIN_FM_TUNER  = 0x20
    } tenMeca_TsuTunerId;


    // _DAB_C_TSU_SWITCH_TUNER
    struct trMeca_CTsuSwitchTuner:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CTsuSwitchTuner():
            enTunerId(enMeca_TsuTunerId_MAIN_DAB_TUNER),
            rProgrammeService()
        {}

        trMeca_CTsuSwitchTuner(tenMeca_TsuTunerId enTunerId_,  trMecaProgrammeService const &rProgrammeService_):
            enTunerId(enTunerId_),
            rProgrammeService(rProgrammeService_)
        {}


        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_TSU_SWITCH_TUNER;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 6;
            pRawOutput->au8Data[0]= (tU8)enTunerId;
            pRawOutput->au8Data[1]= 0; // rfu
            rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CTsuSwitchTuner:enTunerId=%x Service=0x%08x",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId),
                            rProgrammeService.u32GetSID()));
        };
    
        tenMeca_TsuTunerId enTunerId;
        trMecaProgrammeService rProgrammeService;

    };


    // DAB_*_R_TSU_SWITCH_TUNER
    struct trMeca_RTsuSwitchTuner:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 6) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTunerId=(tenMeca_TsuTunerId)pu8Data[0];
            bSwitchDone = (pu8Data[1] == 1) ? TRUE:FALSE;
            rProgrammeService.vParse(&pu8Data[2]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RTsuSwitchTuner:enTunerId=%x bSwitchDone=%u",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId),
                            bSwitchDone));
            rProgrammeService.vTrace();

        };
    
        tenMeca_TsuTunerId enTunerId;
        tBool bSwitchDone;
        trMecaProgrammeService rProgrammeService;
    };



    // DAB_*_R_TSU_SWITCH_RECOMMENDATION
    struct trMeca_RTsuSwitchRecommendation:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 6) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTunerId=(tenMeca_TsuTunerId)pu8Data[0];
            rProgrammeService.vParse(&pu8Data[2]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RTsuSwitchRecommendation:enTunerId=%x",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId)));
            rProgrammeService.vTrace();

        };
    
        tenMeca_TsuTunerId enTunerId;
        trMecaProgrammeService rProgrammeService;
    };



    // _DAB_C_TSU_SET_CONFIG
    struct trMeca_CTsuSetConfig:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CTsuSetConfig():
            enTunerIdA(enMeca_TsuTunerId_INVALID),
            enTunerIdB(enMeca_TsuTunerId_INVALID)
        {
            OSAL_pvMemorySet(au8Matrix, 0, sizeof(au8Matrix));
            OSAL_pvMemorySet(au8Constants, 0, sizeof(au8Constants));
        }

        trMeca_CTsuSetConfig(tenMeca_TsuTunerId enTunerIdA_,
                             tenMeca_TsuTunerId enTunerIdB_,
                             tU8 const *au8Matrix_,
                             tU8 const *au8Constants_):
            enTunerIdA(enTunerIdA_),
            enTunerIdB(enTunerIdB_)
        {
            if (au8Matrix_!=OSAL_NULL && au8Constants_ != OSAL_NULL) {
                OSAL_pvMemoryCopy(au8Matrix, au8Matrix_, sizeof(au8Matrix));
                OSAL_pvMemoryCopy(au8Constants, au8Constants_, sizeof(au8Constants));
            } 
            else {
                OSAL_pvMemorySet(au8Matrix, 0, sizeof(au8Matrix));
                OSAL_pvMemorySet(au8Constants, 0, sizeof(au8Constants));
            }
        }

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_TSU_SET_CONFIG;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 40;
            pRawOutput->au8Data[0]= (tU8)enTunerIdA;
            pRawOutput->au8Data[1]= (tU8)enTunerIdB;
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[2],au8Matrix, 32);
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[34],au8Constants, 6);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CTsuSetConfig:enTunerIdA=%x enTunerIdB=%x Matrix[]=%*p Constants=%*p",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerIdA),
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerIdB),
                            ETG_LIST_LEN(32),
                            ETG_LIST_PTR_T8(au8Matrix),
                            ETG_LIST_LEN(6),
                            ETG_LIST_PTR_T8(au8Constants)));
        };
    
        tenMeca_TsuTunerId enTunerIdA;
        tenMeca_TsuTunerId enTunerIdB;
        tU8 au8Matrix[32];
        tU8 au8Constants[6];

    };


    // DAB_*_R_TSU_SET_CONFIG
    struct trMeca_RTsuSetConfig:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTunerIdA=(tenMeca_TsuTunerId)pu8Data[0];
            enTunerIdB=(tenMeca_TsuTunerId)pu8Data[1];
            bConfigOk = (pu8Data[2] == 1) ? TRUE:FALSE;

            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CTsuSetConfig:enTunerIdA=%x enTunerIdB=%x bConfigOk=%u",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerIdA),
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerIdB),
                            bConfigOk));
        };
        tenMeca_TsuTunerId enTunerIdA;
        tenMeca_TsuTunerId enTunerIdB;
        tBool bConfigOk;
    };



    // DAB_*_R_TSU_SELECT_PSID (command)
    struct trMeca_RTsuSelectPsid:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 6) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTunerId=(tenMeca_TsuTunerId)pu8Data[0];
            rProgrammeService.vParse(&pu8Data[2]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RTsuSelectPsid:enTunerId=%x",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId)));
            rProgrammeService.vTrace();

        };
    
        tenMeca_TsuTunerId enTunerId;
        trMecaProgrammeService rProgrammeService;
    };


    typedef enum {
        enMeca_TsuSelectState_SELECTION_REJECTED = 0,
        enMeca_TsuSelectState_SELECTION_DONE = 1,
        enMeca_TsuSelectState_WAITING_FOR_SELECTION = 4,
        enMeca_TsuSelectState_PSID_NOT_AVAILABLE = 6
    } tenMeca_TsuSelectState;

    // _DAB_C_TSU_SET_SELECT_PSID
    struct trMeca_CTsuSelectPsid:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL


        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_TSU_SELECT_PSID;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 6;
            pRawOutput->au8Data[0]= (tU8)enTunerId;
            pRawOutput->au8Data[1]= (tU8)enSelectState;
            rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CTsuSelectPsid:enTunerId=%x enSelectState=%d rProgrammeService=0x%08x",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId),
                            ETG_CENUM(tenMeca_TsuSelectState, enSelectState),
                            rProgrammeService.u32GetSID()));
        };
    
        tenMeca_TsuTunerId enTunerId;
        tenMeca_TsuSelectState enSelectState;
        trMecaProgrammeService rProgrammeService;


    };

    // _DAB_C_TSU_GET_PSID
    struct trMeca_CTsuGetPsid:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL


        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_TSU_GET_PSID;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 6;
            pRawOutput->au8Data[0]= (tU8)enTunerId;
            pRawOutput->au8Data[1]= 0;
            rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CTsuGetPsid:enTunerId=%x rProgrammeService=0x%08x",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId),
                            rProgrammeService.u32GetSID()));
        };
    
        tenMeca_TsuTunerId enTunerId;
        trMecaProgrammeService rProgrammeService;


    };


    // DAB_*_R_TSU_GET_QUALITY (command)
    struct trMeca_RTsuGetQuality:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    



        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTunerId=(tenMeca_TsuTunerId)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RTsuGetQuality:enTunerId=%x",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId)));

        };
    
        tenMeca_TsuTunerId enTunerId;
    };



    // _DAB_C_TSU_GET_QUALITY
    struct trMeca_CTsuGetQuality:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CTsuGetQuality() {
            enTunerId= enMeca_TsuTunerId_MAIN_DAB_TUNER; 
            u8QualityIndicator=0;
        }

        trMeca_CTsuGetQuality(tenMeca_TsuTunerId enTunerId_, tU8 u8QualityIndicator_) {
            enTunerId= enTunerId_;
            u8QualityIndicator = u8QualityIndicator_;
        }

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_TSU_GET_QUALITY;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 4;
            pRawOutput->au8Data[0]= (tU8)enTunerId;
            pRawOutput->au8Data[1]= 0;
            pRawOutput->au8Data[2]= 0;
            pRawOutput->au8Data[3]= (tU8)u8QualityIndicator;

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CTsuGetQuality:enTunerId=%x u8QualityIndicator=%d",
                            ETG_CENUM(tenMeca_TsuTunerId, enTunerId),
                            u8QualityIndicator));
        };
    
        tenMeca_TsuTunerId enTunerId;
        // quality-indicator should be sent by fm-tuner according to dab-module, no conversion here
        // definition is inconsistent with rdm, check with maik scholz
        tU8 u8QualityIndicator;
    };

/**********************************************************************************************/

     typedef enum {
        enMeca_MtcNotify_SET_ALL                      = 0x00,
        enMeca_MtcNotify_SET_FUNCTION                 = 0x01,
        enMeca_MtcNotify_CLEAR_ALL                    = 0x02,
        enMeca_MtcNotify_CLEAR_FUNCTION               = 0x03
    } tenMeca_MtcNotifyCtrl;

    // _DAB_C_MTC_NOTIFICATION
    struct trMeca_CMtcNotification:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcNotification():
                enMtcNotifyCtrl(enMeca_MtcNotify_SET_FUNCTION),
                u16MtcClientId(0x01)
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_NOTIFICATION;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 3;
            pRawOutput->au8Data[0]= (tU8)enMtcNotifyCtrl;
            DABDRV_SET_U16(&pRawOutput->au8Data[1],u16MtcClientId);
          /*  for (tU16 i=0,j=0;i< lFktIds.size();i++) {				
				DABDRV_SET_U16(&pRawOutput->au8Data[3+j],(tU16)lFktIds[i]);
				j=j+2;
            }
			for(tU16 i=0;i< pRawOutput->u16DataLen;i++)
			{
			ETG_TRACE_USR1(("  trMeca_CMtcNotification:Data=%x",
                            pRawOutput->au8Data[i]));
			}*/
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcNotification:enMtcNotifyCtrl=%x",
                            ETG_CENUM(tenMeca_MtcNotifyCtrl, enMtcNotifyCtrl)));
        }
    
        tenMeca_MtcNotifyCtrl enMtcNotifyCtrl;
        tU16 u16MtcClientId;
        vector<DAB_tenMecaTsuMsgId>lFktIds;
    };

    // _R_MTC_NOTIFICATION
    struct trMeca_RMtcNotification:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * /*pu8Data*/) {
            _rMostHdr=rMostHdr;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RMtcNotification"));
        };
    };

    // _DAB_C_MTC_SET_MODE
    struct trMeca_CMtcSetMode:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcSetMode():
            enTsuMode(DAB_enMtcMode_Off)
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_SET_MODE;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= (tU8)enTsuMode;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcSetMode:enTsuMode=%x",
                            ETG_CENUM(DAB_tenMtcMode, enTsuMode)));
        }
    
        DAB_tenMtcMode enTsuMode;
    };


   
    // DAB_*_R_MTC_SET_MODE
    struct trMeca_RMtcSetMode:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTsuMode=(DAB_tenMtcMode)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RTsuSetMode:enMtcMode=%x ",
                            ETG_CENUM(DAB_tenMtcMode, enTsuMode)));
        };
    
        DAB_tenMtcMode enTsuMode;
    };


	//MTC_REQUEST_SWITCH_STATUS
	/*struct trMeca_CMtcRequestSwitch:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcRequestSwitch():
            enSwitchMode(DAB_enMtcRequestSwitch_DAB)
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_REQUEST_SWITCH;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= (tU8)enSwitchMode;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcRequestSwitch:enSwitchMode=%x",
                            ETG_CENUM(DAB_tenMtcRequestSwitch, enSwitchMode)));
        }
    
        DAB_tenMtcRequestSwitch enSwitchMode;
    };*/

    /*struct trMeca_RMtcRequestSwitch:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enSwitchMode=(DAB_tenMtcRequestSwitch)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RMtcRequestSwitch:enSwitchMode=%x ",
                            ETG_CENUM(DAB_tenMtcRequestSwitch, enSwitchMode)));
        };
    
        DAB_tenMtcRequestSwitch enSwitchMode;
    };*/



	//MTC_ANNOUNCEMENT
	/*struct trMeca_CMtcAnnouncementState:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcAnnouncementState():
            enAnno_State(DAB_enMtcActivityState_Stop)
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_ANNOUNCEMENT;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= (tU8)enAnno_State;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcAnnouncementState:enSwitchMode=%x",
                            ETG_CENUM(DAB_tenMtcActivityState, enAnno_State)));
        };
    
        DAB_tenMtcActivityState enAnno_State;
    };*/

    /*struct trMeca_RMtcAnnouncementState:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enAnno_State=(DAB_tenMtcActivityState)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RMtcAnnouncementState:enSwitchMode=%x ",
                            ETG_CENUM(DAB_tenMtcActivityState, enAnno_State)));
        };
    
        DAB_tenMtcActivityState enAnno_State;
    };*/

	

	//MTC_SURVEILLANCE_MONITOR
    struct trMeca_RMtcSurveillanceMonitor:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen > 64) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enMtcStationType=(DAB_tenMtcStationType)pu8Data[0];
			u32MtcStationFreq = DABDRV_GET_U32(&pu8Data[1]);
			u32MtcStationId = DABDRV_GET_U32(&pu8Data[5]);
			u32MtcEnsID = DABDRV_GET_U32(&pu8Data[9]);
			if(enMtcStationType == DAB_enMtcStationType_DAB)
				OSAL_pvMemoryCopy(u8StationNameDAB, &pu8Data[13], 16);
			else
				OSAL_pvMemoryCopy(u8MtcStationNameFM, &pu8Data[21], 8);


			u8MtcStationQuality = pu8Data[30];
			enMtcTPState=(DAB_tenMtcTPState)pu8Data[31];
			
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_RMtcSurveillanceMonitor: e8MtcMtcStationType =%d "
			"  e8MtcStationFreq=%d  e8MtcStationId=%d  e8MtcEnsID=%d"
            "  e8MtcTPState=%d  u8MtcStationNameFM=%s",
             ETG_CENUM(DAB_tenMtcStationType, enMtcStationType), u32MtcStationFreq,
			 u32MtcStationId, u32MtcEnsID, 
			 ETG_CENUM(DAB_tenMtcTPState, enMtcTPState), u8MtcStationNameFM));
        };
    
		DAB_tenMtcStationType enMtcStationType;
		tU32 u32MtcStationFreq;
		tU32 u32MtcStationId;
		tU32 u32MtcEnsID;
		DAB_tenMtcTPState enMtcTPState;
		tU8 u8MtcStationNameFM[8];
		tU8 u8MtcStationQuality;
		tU8 u8StationNameDAB[16];
    };




    // _DAB_C_MTC_TUNER_STATUS
    struct trMeca_CMtcTunerStatus:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcTunerStatus()
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_TUNER_STATUS;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcTunerStatus"));
        }
    };
   
    typedef enum {
        enMeca_MtcTuner_MAIN_DAB_TUNER = 0x00,
        enMeca_MtcTuner_MAIN_FM_TUNER  = 0x01,
		enMeca_MtcTuner_MAIN_TUNER_UNKNOWN  = 0x02
    } tenMeca_MtcTuner;

    struct trMeca_RFMSurveillanceInfos:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    trMeca_RFMSurveillanceInfos():enMtcAudibleModulation(enMeca_MtcTuner_MAIN_DAB_TUNER),u32MtcFreq(0),u16MtcPI(0)
    { }
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen > 7) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        enMtcAudibleModulation = (tenMeca_MtcTuner)pu8Data[0];
        u32MtcFreq = DABDRV_GET_U32(&pu8Data[1]);
        u16MtcPI =   DABDRV_GET_U16(&pu8Data[5]);

        return TRUE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  trMeca_RFMSurveillanceInfos: e8MtcAudibleModulation =%d e8MtcFreq=%d  e8MtcPI=%d " ,
        ETG_CENUM(tenMeca_MtcTuner, enMtcAudibleModulation), u32MtcFreq,u16MtcPI));

    };

    tenMeca_MtcTuner enMtcAudibleModulation;
    tU32 u32MtcFreq;
    tU16 u16MtcPI;
};

    // DAB_*_R_MTC_TUNER_STATUS
    struct trMeca_RMtcTunerStatus:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enMtcTuner=(tenMeca_MtcTuner)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RMtcTunerStatus:enMtcTuner=%x ",
                            ETG_CENUM(tenMeca_MtcTuner, enMtcTuner)));
        };
    
        tenMeca_MtcTuner enMtcTuner;
    };


    // _C_MTC_SET_CONFIG
    struct trMeca_CMtcSetConfig:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcSetConfig()
        {
            OSAL_pvMemorySet(au8DabFmMatrix, 0, sizeof(au8DabFmMatrix));
            OSAL_pvMemorySet(au8FmDabMatrix, 0, sizeof(au8FmDabMatrix));
            OSAL_pvMemorySet(au8DabFmConstants, 0, sizeof(au8DabFmConstants));
            OSAL_pvMemorySet(au8FmDabConstants, 0, sizeof(au8FmDabConstants));
        }

        trMeca_CMtcSetConfig(tU8 const *au8DabFmMatrix_,
                             tU8 const *au8FmDabMatrix_,
                             tU8 const *au8DabFmConstants_,
                             tU8 const *au8FmDabConstants_)
        {
            if (au8DabFmMatrix_!=OSAL_NULL && au8DabFmConstants_ != OSAL_NULL) {
                OSAL_pvMemoryCopy(au8DabFmMatrix, au8DabFmMatrix_, sizeof(au8DabFmMatrix));
                OSAL_pvMemoryCopy(au8FmDabMatrix, au8FmDabMatrix_, sizeof(au8FmDabMatrix));
                OSAL_pvMemoryCopy(au8DabFmConstants, au8DabFmConstants_, sizeof(au8DabFmConstants));
                OSAL_pvMemoryCopy(au8FmDabConstants, au8FmDabConstants_, sizeof(au8FmDabConstants));
            } 
            else {
                OSAL_pvMemorySet(au8DabFmMatrix, 0, sizeof(au8DabFmMatrix));
                OSAL_pvMemorySet(au8FmDabMatrix, 0, sizeof(au8FmDabMatrix));
                OSAL_pvMemorySet(au8DabFmConstants,0, sizeof(au8DabFmConstants));
                OSAL_pvMemorySet(au8FmDabConstants, 0, sizeof(au8FmDabConstants));
            }
        }

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_SET_CONFIG;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = 76;
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[0],au8DabFmMatrix, 32);
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[32],au8FmDabMatrix, 32);
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[64],au8DabFmConstants, 6);
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[70],au8FmDabConstants, 6);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcSetConfig: Dab Matrix[]=%*p Constants=%*p",
                            ETG_LIST_LEN(32),
                            ETG_LIST_PTR_T8(au8DabFmMatrix),
                            ETG_LIST_LEN(6),
                            ETG_LIST_PTR_T8(au8DabFmConstants)));
            ETG_TRACE_USR1(("  trMeca_CMtcSetConfig: Fm  Matrix[]=%*p Constants=%*p",
                            ETG_LIST_LEN(32),
                            ETG_LIST_PTR_T8(au8FmDabMatrix),
                            ETG_LIST_LEN(6),
                            ETG_LIST_PTR_T8(au8FmDabConstants)));
        };
    
        tU8 au8DabFmMatrix[32];
        tU8 au8FmDabMatrix[32];
        tU8 au8DabFmConstants[6];
        tU8 au8FmDabConstants[6];

    };


    // DAB_*_R_TSU_SET_CONFIG
    struct trMeca_RMtcSetConfig:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            ETG_TRACE_USR1(("  trMeca_RMtcSetConfig: LNGE  %d data[0]= %d",
                            rMostHdr._u16MecaLen, pu8Data[0]));

            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            bConfigOk = (pu8Data[0] == 1) ? TRUE:FALSE;

            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RTsuSetConfig: bConfigOk=%u",
                            bConfigOk));
        };
        tBool bConfigOk;
    };
    struct trMeca_CMtcSfRestart:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcSfRestart()
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_SF_RESTART;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcSfRestart"));
        }
    };
    struct trMeca_RMtcSfRestart:
		public trMsgMecaInput {
		DAB_DISPATCH_IMPL

        trMeca_RMtcSfRestart()
            {}

        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * /*pu8Data*/) {
            _rMostHdr=rMostHdr;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RMtcSfRestart"));
        };
	};

    struct trMeca_RMtcTestmode:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
        trMeca_RMtcTestmode():
            u32FmFrequency(0),
            u16Pi(0),
            u8FmQuality(0)
        {}
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 7) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            u32FmFrequency=DABDRV_GET_U32(&pu8Data[0]);
            u16Pi = DABDRV_GET_U16(&pu8Data[4]);
            u8FmQuality = pu8Data[6];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RMtcTestmode:u32FmFrequency=%x u16Pi=%x u8FmQuality=%d",
                            u32FmFrequency,
                            u16Pi,
                            u8FmQuality));
        };
        tU32 u32FmFrequency;
        tU16 u16Pi;
        tU8  u8FmQuality;
    };
	 // MECA_DISPATCHER_C_MTC_SET_AUDIO
    struct trMeca_CMtcSetAudio:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CMtcSetAudio():
            u8AudioSrc1(0),
            u8AudioSrc2(0)
        {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_MTC_FBLOCK_ID;
            pRawOutput->enMecaTsuMsgId = MECA_DISPATCHER_C_MTC_SET_AUDIO;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = 2;
            pRawOutput->au8Data[0]= u8AudioSrc1;
            pRawOutput->au8Data[1]= u8AudioSrc2;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CMtcSetAudio:u8AudioSrc1=%d u8AudioSrc2=%d",
                            u8AudioSrc1,
                            u8AudioSrc2));
        }
    
        tU8 u8AudioSrc1;
        tU8 u8AudioSrc2;

    };
	
	  struct trMeca_RMtcSetAudio:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
        trMeca_RMtcSetAudio():            
            u8AudioSrc1(0),
            u8AudioSrc2(0)
        {}
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            u8AudioSrc1=pu8Data[0];
            u8AudioSrc2 = pu8Data[1];
           
            return TRUE;
        }
       virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RMtcSetAudio:u8AudioSrc1=%d u8AudioSrc2=%d",
                            u8AudioSrc1,
                            u8AudioSrc2));
        }     
       tU8 u8AudioSrc1;
        tU8 u8AudioSrc2;	
    };
} // namespace DAB

#endif
