/************************************************************************
 * FILE:        dabmeca_sys.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabmeca_sys
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      	| Author                       | Modification
				  
				
 *************************************************************************/

#ifndef DABMECA_SYS_HPP
#define DABMECA_SYS_HPP

#include "fc_dabtuner_util.h"
#include "dabdrv_meca.h"
#include "dabmeca_db.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_MSG
#include "trcGenProj/Header/dabmeca_sys.hpp.trc.h"
#endif

namespace DAB {
 //lint -esym(793,*identifiers in one block*)
    typedef enum {
        enMeca_SysPowerMode_ON                       = 0x01,
        enMeca_SysPowerMode_OFF                      = 0x02,
        enMeca_SysPowerMode_POWER_OFF_RESTRICTED     = 0x03, // only as result
        enMeca_SysPowerMode_ENABLE_POWER_SHTDN_INPUT = 0x04
    } tenMeca_SysPowerMode;



    // _DAB_C_SYS_POWER
    struct trMeca_CSysPower:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_SYS_POWER;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= (tU8)enPowerMode;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CSysPower: enPowerMode=%x",
                            ETG_CENUM(tenMeca_SysPowerMode, enPowerMode)));
        };
    
        tenMeca_SysPowerMode enPowerMode;

    };


    // DAB_*_R__SYS_POWER
    struct trMeca_RSysPower:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enPowerMode=(tenMeca_SysPowerMode)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RSysPower: enPowerMode=%x",
                            ETG_CENUM(tenMeca_SysPowerMode, enPowerMode)));
        };
    
        tenMeca_SysPowerMode enPowerMode;

    };



    // _DAB_C_SYS_GET_VERSION_INFO
    struct trMeca_CSysVersionInfo:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CSysVersionInfo(){};

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_SYS_GET_VERSION_INFO;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("CSysVersionInfo"));
        };
    

    };

#define DAB_MECA_SYS_VER_MAX_STR_LEN 32
    // DAB_*_R_SYS_GET_VERSION_INFO
    struct trMeca_RSysVersionInfo:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 128) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            OSALUTIL_szSaveStringNCopy(sProductName,(tCString)&pu8Data[0], DAB_MECA_SYS_VER_MAX_STR_LEN);
            OSALUTIL_szSaveStringNCopy(sVersionTag,(tCString)&pu8Data[32], DAB_MECA_SYS_VER_MAX_STR_LEN);
            OSALUTIL_szSaveStringNCopy(sVersionDate,(tCString)&pu8Data[64], DAB_MECA_SYS_VER_MAX_STR_LEN);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RSysVersionInfo: sProductName=%32s sVersionTag=%32s sVersionDate=%32s",
                            sProductName,
                            sVersionTag,
                            sVersionDate));
        };

    
        tChar sProductName[DAB_MECA_SYS_VER_MAX_STR_LEN + 1];
        tChar sVersionTag[DAB_MECA_SYS_VER_MAX_STR_LEN + 1];
        tChar sVersionDate[DAB_MECA_SYS_VER_MAX_STR_LEN + 1];
        

    };


    // _DAB_C_SYS_STORE
    struct trMeca_CSysStore:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CSysStore(tU8 u8UserId_=0, tU8 u8Tag_=0):
            u8UserId(u8UserId_),
            u8Tag(u8Tag_)
        {}

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_SYS_STORE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 2;
            pRawOutput->au8Data[0]= u8UserId;
            pRawOutput->au8Data[1]= u8Tag;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CSysStore u8UserId=%d u8Tag=%d",
                            u8UserId, u8Tag));
        };
    
        tU8 u8UserId;
        tU8 u8Tag;
    };

    typedef enum {
        enMeca_SysStoreStatus_STORE_ERROR       = 0,
        enMeca_SysStoreStatus_STORE_DONE        = 1,
        enMeca_SysStoreStatus_STORE_IN_PROGRESS = 2
    } tenMeca_SysStoreStatus;

    // DAB_*_R_SYS_STORE
    struct trMeca_RSysStore:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enStatus=(tenMeca_SysStoreStatus)pu8Data[0];
            u8UserId = pu8Data[1];
            u8Tag = pu8Data[2];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RSysStore enStatus=%x u8UserId=%d u8Tag=%d",
                            ETG_CENUM(tenMeca_SysStoreStatus, enStatus),
                            u8UserId,
                            u8Tag));
        };
    
        tenMeca_SysStoreStatus enStatus;
        tU8 u8UserId;
        tU8 u8Tag;

    };

    // _DAB_C_SYS_ReSTORE
    struct trMeca_CSysReStore:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_SYS_RESTORE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 2;
            pRawOutput->au8Data[0]= u8UserId;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CSysReStore u8UserId=%d",
                            u8UserId));
        };
    
        tU8 u8UserId;

    };

    typedef enum {
        enMeca_SysStoreStatus_RESTORE_ERROR       = 0,
        enMeca_SysStoreStatus_RESTORE_DONE        = 1
    } tenMeca_SysReStoreStatus;

    // DAB_*_R_SYS_STORE
    struct trMeca_RSysReStore:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enStatus=(tenMeca_SysReStoreStatus)pu8Data[0];
            u8UserId = pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RSysReStore enStatus=%x u8UserId=0x%x",
                            ETG_CENUM(tenMeca_SysStoreStatus, enStatus),
                            u8UserId));
        };
    
        tenMeca_SysReStoreStatus enStatus;
        tU8 u8UserId;

    };

    // C_SYS_STORAGE_GET_FRAME omitted
    // R_SYS_STORAGE_GET_FRAME omitted


    // _DAB_C_SYS_STORAGE_PUT_FRAME
#define DAB_MECA_SYS_STORAGE_PUT_FRAME_MAX_DATA_LEN 100
    typedef enum {
        enMeca_SysStorageDeviceId_PARAMETER_MEMORY      = 0x01,
        enMeca_SysStorageDeviceId_WRITEONCE_MEMORY      = 0x02,
        enMeca_SysStorageDeviceId_ERRORLOG_MEMORY      = 0x03
    } tenMeca_SysStorageDeviceId;
    struct trMeca_CSysStoragePutFrame:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_SYS_STORAGE_PUT_FRAME;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = (tU16)(6+u16DataLen);
            pRawOutput->au8Data[0]= u8UserId;
            pRawOutput->au8Data[1]= (tU8)enDeviceId;
            DABDRV_SET_U16(&pRawOutput->au8Data[2],u16DataType);
            DABDRV_SET_U16(&pRawOutput->au8Data[4],u16DataLen);
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[6],au8Data, DAB_MECA_SYS_STORAGE_PUT_FRAME_MAX_DATA_LEN);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CSysStoragePutFrame u8UserId=%d u8DeviceId=%d u16DataType=0x%04x u16DataLen=%d au8Data[]=%*p",
                            u8UserId,
                            ETG_CENUM(tenMeca_SysStorageDeviceId, enDeviceId),
                            u16DataType,
                            u16DataLen,
                            ETG_LIST_LEN(u16DataLen),
                            ETG_LIST_PTR_T8(au8Data)));
        };
    
        tU8 u8UserId;
        tenMeca_SysStorageDeviceId enDeviceId;
        tU16 u16DataType;
        tU16 u16DataLen;
        tU8 au8Data[DAB_MECA_SYS_STORAGE_PUT_FRAME_MAX_DATA_LEN];

    };



    // _DAB_R_SYS_STORAGE_PUT_FRAME
    struct trMeca_RSysStoragePutFrame:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 5) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            u8UserId = pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RSysStoragePutFrame u8UserId=%x enDeviceId=0x%x u16DataType=0x%04x bStatus(bOk)=%d",
                            u8UserId,
                            ETG_CENUM(tenMeca_SysStorageDeviceId, enDeviceId),
                            u16DataType,
                            bStatus));
        };

        tU8 u8UserId;
        tenMeca_SysStorageDeviceId enDeviceId;
        tU16 u16DataType;  
        tBool bStatus;

    };

    // C_SYS_STORAGE_PUT_FRAME omitted
    // R_SYS_STORAGE_PUT_FRAME omitted

    /*
start_gtkDAB_MOSTFB.cmd: GTK-GUI:
To enable messages via uart:
in GTK-GUI:
Meca->Sys->STORAGE-PUT-FRAME:
b-user-id=0
b-device-id=1
w-data-type=0x0001
r-data=
0x38050002010000000438050002020000000444000080005000
    */

    typedef enum {
        enMeca_SysStorageMemory_ERROR = 0x00, // not meca, default value for error in response
        enMeca_SysStorageMemory_PARAMETER = 0x01,
        enMeca_SysStorageMemory_WRITEONCE = 0x02,
        enMeca_SysStorageMemory_ERRORLOG  = 0x03
    } tenMeca_SysStorageMemory;

    // _DAB_C_SYS_STORAGE_FORMAT
    struct trMeca_CSysStorageFormat:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_SYS_STORAGE_FORMAT;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 2;
            pRawOutput->au8Data[0]= (tU8)enDeviceId;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CSysStorageFormat enDeviceId=%d",
                            ETG_CENUM(tenMeca_SysStorageMemory, enDeviceId)));
        };
    
        tenMeca_SysStorageMemory enDeviceId;

    };



    // DAB_*_R_SYS_STORAGE_FORMAT
    struct trMeca_RSysStorageFormat:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enDeviceId=(tenMeca_SysStorageMemory)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RSysStorageFormat enDeviceId=%d",
                            ETG_CENUM(tenMeca_SysStorageMemory, enDeviceId)));
        };
    
        tenMeca_SysStorageMemory enDeviceId;


    };

    // C_SYS_STORAGE_PROTECT omitted
    // R_SYS_STORAGE_PROTECT omitted



    typedef enum {
        enMeca_SysDiagFeature_DIAG_ENABLE_TEST_TONE    = 0,
        enMeca_SysDiagFeature_DIAG_TEST_ANTENNA_SUPPLY = 1,
        enMeca_SysDiagFeature_DIAG_TEST_PARAMETER_MEM  = 2,
        enMeca_SysDiagFeature_DIAG_TEST_FRONTEND       = 3
    } tenMeca_SysDiagFeature;

    typedef enum {
        enMeca_SysDiagCommand_START       = 1,
        enMeca_SysDiagCommand_STOP        = 2,
        enMeca_SysDiagCommand_TEST        = 3,
        enMeca_SysDiagCommand_START_LEFT  = 4,
        enMeca_SysDiagCommand_START_RIGHT = 5
    } tenMeca_SysDiagCommand;

    // _DAB_C_SYS_DIAG
    struct trMeca_CSysDiag:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_SYS_DIAG;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 3;
            pRawOutput->au8Data[0]= (tU8)enFeature;
            pRawOutput->au8Data[1]= (tU8)enCommand;
            pRawOutput->au8Data[2]=0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CSysDiag: enFeature=%x enCommand=%x",
                            ETG_CENUM(tenMeca_SysDiagFeature, enFeature),
                            ETG_CENUM(tenMeca_SysDiagCommand, enCommand)));
        };
    
        tenMeca_SysDiagFeature enFeature;
        tenMeca_SysDiagCommand enCommand;

    };

    typedef enum {
        enMeca_SysDiagResult_UNDEF = 0,
        enMeca_SysDiagResult_OK    = 1,
        enMeca_SysDiagResult_PASS  = 2,
        enMeca_SysDiagResult_FAIL  = 3
    } tenMeca_SysDiagResult;


    // DAB_*_R_SYS_DIAG
    struct trMeca_RSysDiag:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enFeature=(tenMeca_SysDiagFeature)pu8Data[0];
            enCommand=(tenMeca_SysDiagCommand)pu8Data[1];
            enResult=(tenMeca_SysDiagResult)pu8Data[2];
            OSAL_pvMemoryCopy(u8ResultData, &pu8Data[4], 32);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RSysDiag: enFeature=%x enCommand=%x enResult=%x u8ResultData=%*p",
                            ETG_CENUM(tenMeca_SysDiagFeature, enFeature),
                            ETG_CENUM(tenMeca_SysDiagCommand, enCommand),
                            ETG_CENUM(tenMeca_SysDiagResult, enResult),
                            ETG_LIST_LEN(32), ETG_LIST_PTR_T8(u8ResultData)));
        };
    
        tenMeca_SysDiagFeature enFeature;
        tenMeca_SysDiagCommand enCommand;
        tenMeca_SysDiagResult enResult;
        tU8 u8ResultData[32];

    };

} // namespace DAB



#endif
