/************************************************************************
 * FILE:        dabmeca_db.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabmeca_db
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      	| Author                       | Modification
				  
				
 *************************************************************************/

#ifndef DABMECA_DB_HPP
#define DABMECA_DB_HPP


#include "fc_dabtuner_util.h"
#include "dabdrv_meca.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_MSG
#include "trcGenProj/Header/dabmeca_db.hpp.trc.h"
#endif


#ifndef __RFC3072_HELPER_H__
#include "rfc3072_helper.h"
#endif



#define COMP_SERV_LIST 1


#define DAB_GET_ID(a) ((a) & 0xFF00FFFF)
#define DAB_ID_EQ(a,b) (DAB_GET_ID(a) == DAB_GET_ID(b))
#define DAB_ID_NEQ(a,b) (DAB_GET_ID(a) != DAB_GET_ID(b))
#define DAB_ID_LESS(a,b) (DAB_GET_ID(a) < DAB_GET_ID(b))
#define DAB_ID_GREATER(a,b) (DAB_GET_ID(a) > DAB_GET_ID(b))

//CRQ-431566
#define DAB_LABEL_MAX_NUM_CHAR 50
#define DAB_SHORT_LABEL_MAX_CHAR 16
#define DAB_LABEL_SHORT_MAX_NUM_CHAR 8
#define SLABEL_MAX_SIZE 151
#define DAB_EXT_LABEL_MIX_NUM_CHAR 16
#define DAB_LABEL_INFO_LENGTH 6

#define DAB_b0 0x01
#define DAB_b1 0x02
#define DAB_b2 0x04
#define DAB_b3 0x08
#define DAB_b4 0x10
#define DAB_b5 0x20
#define DAB_b6 0x40
#define DAB_b7 0x80

#define DAB_b8  0x0100
#define DAB_b9  0x0200
#define DAB_b10 0x0300
#define DAB_b11 0x0800
#define DAB_b12 0x1000
#define DAB_b13 0x2000
#define DAB_b14 0x4000
#define DAB_b15 0x8000

#define PTY_COARSE_PTY 0x03
#define PTY_COUNT 32

#define BASIC_DAB_SHORT_LABEL_MASK 0xFF00

namespace DAB {

    typedef enum {
        DAB_FALSE = 0,
        DAB_TRUE  = 1
    } DAB_tenBool;
    
    typedef enum {
        DAB_OFF = 0,
        DAB_ON  = 1
    } DAB_tenOnOff;

    typedef enum {
        enMeca_CharSet_EBU_LATIN    = 0x00,
        enMeca_CharSet_EBU_LATIN_CYR_GREEK    = 0x01,
        enMeca_CharSet_EBU_LATIN_ARAB_HEBREW_CYR_GREEK     = 0x02,
        enMeca_CharSet_ISO_LATIN_2 = 0x03,
        enMeca_CharSet_ISO_UCS_2 = 0x06,
        enMeca_CharSet_ISO_UTF_8 = 0x0F
    } tenMeca_CharSet;

    inline  tenDab_CharSet DAB_enGetDabCharSet(tenMeca_CharSet enMecaCharSet) {
        // todo: check mapping
        switch (enMecaCharSet){
            case enMeca_CharSet_EBU_LATIN:
            case enMeca_CharSet_EBU_LATIN_CYR_GREEK:
            case enMeca_CharSet_EBU_LATIN_ARAB_HEBREW_CYR_GREEK:
            case enMeca_CharSet_ISO_LATIN_2:
                return enDab_CharSetEBU;
                
            case enMeca_CharSet_ISO_UTF_8:
                return enDab_CharSetUTF8;
            case enMeca_CharSet_ISO_UCS_2:
                return enDab_CharSetUCS;
            default:
                return enDab_CharSetASCII;

        }
        
    }
	

     typedef enum {
        DAB_tenDbRecQuality_Hide=0,
        DAB_tenDbRecQuality_Show=1,
        DAB_tenDbRecQuality_Play=2
    } DAB_tenDbRecQuality;

    enum {
        DAB_enDbQualityReceivable=0x02,
        DAB_enDbQualityEnsembleReceivableBit=0x40,
        DAB_enDbQualityHideEnsembleBit=0x80,
        DAB_enDbQualityNotReceivableAndHide=0xC0,
        DAB_enDbQualityRelevantBits=0xF0
    };
    
    
    

    inline tBool bShowEnsemble(tU8 u8MecaReceptionQuality) {

        tBool bRes =  (u8MecaReceptionQuality & (tU8)DAB_enDbQualityHideEnsembleBit) ? FALSE : TRUE;

        ETG_TRACE_USR4(("  bShowEnsemble(0x%02x)=%d",
                        u8MecaReceptionQuality, bRes));
        return bRes;
    }

    inline tBool bIsReceptionOk(tU8 u8MecaReceptionQuality) {
        tBool bRes = TRUE;
        
        if(u8MecaReceptionQuality & (tU8)DAB_enDbQualityEnsembleReceivableBit) {
            bRes= FALSE;
        } 
        ETG_TRACE_USR4(("  bIsReceptionOk(0x%02x)=%d", u8MecaReceptionQuality, bRes));

        return bRes;
    }

    inline DAB_tenDbRecQuality enGetDbRecQuality(tU8 u8MecaReceptionQuality) {
        DAB_tenDbRecQuality enRes=DAB_tenDbRecQuality_Hide;
        if (bIsReceptionOk(u8MecaReceptionQuality)) {
            enRes= DAB_tenDbRecQuality_Play;
        }
        else if (bShowEnsemble(u8MecaReceptionQuality)) {
            enRes= DAB_tenDbRecQuality_Show;
        }
        ETG_TRACE_USR4(("  enGetDbRecQuality(%x)=%d",
                        u8MecaReceptionQuality, 
                        ETG_CENUM(DAB_tenDbRecQuality, enRes)));
        
        return enRes;
    }


    struct trMecaLabel {

        trMecaLabel():
            bLabelValid(FALSE),
			_u16LabelMask(0),
			oUtf8Label(),
			oUtf8LabelShort(),
			u8TextControlField(0),
			_enDabOrigCharSet(enDab_CharSetEBU)

        {
        	memset(_dabOrigString,0,DAB_LABEL_MAX_NUM_CHAR+1);
        };													  
        
        trMecaLabel(tU32 u32Id):
            bLabelValid(TRUE),
			_u16LabelMask(0xFF00),
			u8TextControlField(0),
			_enDabOrigCharSet(enDab_CharSetEBU)
        {// Handling UTF-8 characters. PSARCCB30-1911
            tChar sLabel[(DAB_LABEL_MAX_NUM_CHAR*3)+1];
            tChar sLabelShort[(DAB_LABEL_SHORT_MAX_NUM_CHAR*3)+1];
            OSAL_s32PrintFormat(sLabel, "0x%08x", u32Id);
            oUtf8Label=DAB_tclUtf8String<DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDab_CharSetASCII, sLabel);
            oUtf8LabelShort=DAB_tclUtf8String<DAB_LABEL_SHORT_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDab_CharSetASCII, pcGetLabelShort(sLabel, sLabelShort, 0xFF00));
            memset(_dabOrigString,0,DAB_LABEL_MAX_NUM_CHAR+1);
        };
        

    // persistant data serialization
    tVoid vSerialize(DAB_Serializer &rSerializer) const {
        rSerializer.vSerialize(bLabelValid);
        rSerializer.vSerializeCString((tChar const *)oUtf8Label.pcu8GetString());
        rSerializer.vSerializeCString((tChar const *)oUtf8LabelShort.pcu8GetString());
    }

	tVoid vParse(APILabel &rAPILabel, tBool _bLabelEqualsId = FALSE) {
	   tChar sLabel[(DAB_LABEL_MAX_NUM_CHAR*3)+1];
       memset(sLabel, 0, SLABEL_MAX_SIZE);
       tChar sLabelShort[(DAB_LABEL_SHORT_MAX_NUM_CHAR*3)+1];
       bLabelValid = rAPILabel.b_label_status;
	   u8TextControlField = rAPILabel.b_text_control_field;

	   tenDab_CharSet enDabCharSet =  DAB_enGetDabCharSet((tenMeca_CharSet)rAPILabel.b_charset);
	   _u16LabelMask = rAPILabel.w_8_char_abbreviation;

	   memcpy(_dabOrigString,rAPILabel.b_label,DAB_LABEL_MAX_NUM_CHAR);
	   
	   if ( enDabCharSet == enDab_CharSetUCS ){
			DAB_stringUtil::u32ConvertUnicodeStringToUtf8((const tChar*)&rAPILabel.b_label, DAB_LABEL_MAX_NUM_CHAR,(tU8*)sLabel, (DAB_LABEL_MAX_NUM_CHAR*3)+1);
			enDabCharSet = enDab_CharSetUTF8;
	   }else if( enDabCharSet == enDab_CharSetEBU ){
		   DAB_stringUtil::u32ConvertEBUStringToUtf8((const tChar*)&rAPILabel.b_label, DAB_LABEL_MAX_NUM_CHAR,(tU8*)sLabel, (DAB_LABEL_MAX_NUM_CHAR*3)+1);
                    enDabCharSet = enDab_CharSetUTF8;
	   }
	   else {
			OSALUTIL_szSaveStringNCopy(sLabel,(tCString)&rAPILabel.b_label, DAB_LABEL_MAX_NUM_CHAR+1);
	   }
	   ETG_TRACE_USR1(("enDabCharSet %d sLabel %s ",enDabCharSet,sLabel));
	   oUtf8Label = DAB_tclUtf8String<DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDabCharSet, sLabel);

	   if (_bLabelEqualsId) {
		   oUtf8LabelShort = DAB_tclUtf8String<DAB_LABEL_SHORT_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDabCharSet, pcGetCString());
	   }
	   else {
		   oUtf8LabelShort = DAB_tclUtf8String<DAB_LABEL_SHORT_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDabCharSet, pcGetLabelShort(sLabel, sLabelShort, _u16LabelMask));
	   }
	   _enDabOrigCharSet = (tenDab_CharSet)rAPILabel.b_charset;
	   ETG_TRACE_USR1(("Orig Meca charset %d _dabOrigString %s",_enDabOrigCharSet,_dabOrigString));
    }

    tVoid vParse(DAB_Parser &rParser) {
        rParser.vParse(bLabelValid);
        tChar sLabel[DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN + 1];
        u8TextControlField = 0;
        rParser.vParseCString(sLabel, DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN);
        oUtf8Label=DAB_tclUtf8String<DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDab_CharSetASCII, sLabel);
        rParser.vParseCString(sLabel, DAB_LABEL_SHORT_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN);
        oUtf8LabelShort=DAB_tclUtf8String<DAB_LABEL_SHORT_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDab_CharSetASCII, sLabel);

    }

        tVoid vParse(tU8 const *pu8Data, tBool bLong=TRUE) { // to be done for  u8TextControlField
            bLabelValid = (pu8Data[0] == 0x01) ? TRUE:FALSE;
            tenDab_CharSet enDabCharSet =  DAB_enGetDabCharSet((tenMeca_CharSet)pu8Data[1]);
            // temp-buffers for the meca-input-strings
            tChar sLabel[(DAB_LABEL_MAX_NUM_CHAR*3)+1];
            tChar sLabelShort[(DAB_LABEL_SHORT_MAX_NUM_CHAR*3)+1];
            sLabel[0]=0;
            sLabelShort[0]=0;
            tU16 u16Mask=0;
            if (!bLabelValid) {
                ETG_TRACE_ERR(("  trMecaLabel: Invalid label")); 
            }
            else if (!bLong) {
                if ( enDabCharSet == enDab_CharSetUCS ){
                    DAB_stringUtil::u32ConvertUnicodeStringToUtf8((const tChar*)&pu8Data[2], DAB_LABEL_MAX_NUM_CHAR,(tU8*)sLabel, (DAB_LABEL_MAX_NUM_CHAR*3)+1);
                    enDabCharSet = enDab_CharSetUTF8;
                }else if( enDabCharSet == enDab_CharSetEBU ){
				   DAB_stringUtil::u32ConvertEBUStringToUtf8((const tChar*)&pu8Data[2], DAB_LABEL_MAX_NUM_CHAR,(tU8*)sLabel, (DAB_LABEL_MAX_NUM_CHAR*3)+1);
                    enDabCharSet = enDab_CharSetUTF8;
                }
                else {
                OSALUTIL_szSaveStringNCopy(sLabel,(tCString)&pu8Data[2], DAB_LABEL_SHORT_MAX_NUM_CHAR+1);
            }
                memcpy(_dabOrigString,&pu8Data[2],DAB_LABEL_MAX_NUM_CHAR);
                u16Mask=0xFF00;
            }
            else {
                if ( enDabCharSet == enDab_CharSetUCS ){
                    DAB_stringUtil::u32ConvertUnicodeStringToUtf8((const tChar*)&pu8Data[4], DAB_LABEL_MAX_NUM_CHAR,(tU8*)sLabel, DAB_LABEL_MAX_NUM_CHAR+1);
                    enDabCharSet = enDab_CharSetUTF8;
            }
             else if( enDabCharSet == enDab_CharSetEBU ){
				   DAB_stringUtil::u32ConvertEBUStringToUtf8((const tChar*)&pu8Data[4], DAB_LABEL_MAX_NUM_CHAR,(tU8*)sLabel, (DAB_LABEL_MAX_NUM_CHAR*3)+1);
                    enDabCharSet = enDab_CharSetUTF8;
            }
            else {
                OSALUTIL_szSaveStringNCopy(sLabel,(tCString)&pu8Data[4], DAB_LABEL_MAX_NUM_CHAR+1);
                }
                u16Mask = DABDRV_GET_U16(&pu8Data[2]);
                memcpy(_dabOrigString,&pu8Data[4],DAB_LABEL_MAX_NUM_CHAR);
            }
			_u16LabelMask=u16Mask;
            oUtf8Label=DAB_tclUtf8String<DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDabCharSet, sLabel);
            oUtf8LabelShort=DAB_tclUtf8String<DAB_LABEL_SHORT_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDabCharSet, pcGetLabelShort(sLabel, sLabelShort, u16Mask));

           	_enDabOrigCharSet = (tenDab_CharSet)pu8Data[1];
           	ETG_TRACE_USR1(("_dabOrigCharset %d _dabOrigString %s",_enDabOrigCharSet,_dabOrigString));
            
        }
    private:
        char *pcGetLabelShort(char *pSrc, char *pDest, tU16 u16Mask) const {
            if (OSAL_NULL == pDest) {
                return OSAL_NULL;
            }
            tU8 numChars = 0;
            tU8 srcCharNum=0;
            tU8 destCharNum=0;
            tU8 MaskBit=0;
            ETG_TRACE_USR1(("short label src %s",pSrc));
			ETG_TRACE_USR1(("u16Mask %d", u16Mask));

			if (u16Mask==0)
				u16Mask = BASIC_DAB_SHORT_LABEL_MASK;

            while(pSrc[srcCharNum] != (char)NULL){
            	tU8 utf8charLenght = DAB_stringUtil::getUt8CharLengthinBytes(pSrc[srcCharNum]);
                if ((u16Mask >> ((DAB_SHORT_LABEL_MAX_CHAR-1) - MaskBit)) & 0x01) {
                	memcpy((pDest+destCharNum),(pSrc+srcCharNum),utf8charLenght);
                	destCharNum = (tU8)(destCharNum + utf8charLenght);
                    numChars++;
                	if(numChars == DAB_LABEL_SHORT_MAX_NUM_CHAR)
                        break;
                    }
                srcCharNum = (tU8)(srcCharNum + utf8charLenght);
                MaskBit++;
                if(srcCharNum>=DAB_LABEL_MAX_NUM_CHAR*3)
                	break;

            }
            pDest[destCharNum]=0;
            ETG_TRACE_USR1(("numChars %d pDest %s",numChars,pDest));
            return pDest;
        }

    public:

        char const *pcGetCString() const {
            return oUtf8Label.pcGetCString();
        }
        
        char const *pcGetCStringShort() const {
            return oUtf8LabelShort.pcGetCString();
        }

        tVoid vSerialize(tU8 *pu8Data, bool extendLabelMsg = false) const
        {
            tU16 index = 0;
            pu8Data[index++] = bLabelValid ? 0x01 : 0x00;
            pu8Data[index++] = (tU8)_enDabOrigCharSet;

            DABDRV_SET_U16(&pu8Data[index], _u16LabelMask);
            index += 2;

            if (isExtendedLabel() || extendLabelMsg)
            {
                pu8Data[index++] = (tU8)u8TextControlField;
                pu8Data[index++] = (tU8)(isExtendedLabel() ? getOrigLabelLength() : DAB_EXT_LABEL_MIX_NUM_CHAR);
            }
            OSAL_pvMemoryCopy(&pu8Data[index], _dabOrigString, DAB_LABEL_MAX_NUM_CHAR);
            ETG_TRACE_USR1(("vSerialize::_enDabOrigCharSet %d_dabOrigString %s", _enDabOrigCharSet, _dabOrigString));
        }
        tVoid vAddEnsembleLabel(trMecaLabel rEnsembleLabel) {
            tChar sBuildLabel[DAB_LABEL_MAX_NUM_CHAR+2+DAB_LABEL_SHORT_MAX_NUM_CHAR+1];
            OSAL_pvMemoryCopy(sBuildLabel,oUtf8Label.pcu8GetString(), DAB_LABEL_MAX_NUM_CHAR);
            OSALUTIL_szSaveStringNConcat(sBuildLabel," (",2);
            OSALUTIL_szSaveStringNConcat(sBuildLabel,(tCString)rEnsembleLabel.pcGetCStringShort(),DAB_LABEL_SHORT_MAX_NUM_CHAR);
            OSALUTIL_szSaveStringNConcat(sBuildLabel,")",1);
            oUtf8Label=DAB_tclUtf8String<DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN>(enDab_CharSetASCII, sBuildLabel);
        }
        tVoid vTrace() const {
            
            ETG_TRACE_USR1(("  trMecaLabel: bLabelValid=%u labelMask=%d au8DataShort=%12s au8Data=%s",
                            bLabelValid,
							_u16LabelMask,
                            oUtf8LabelShort.pcGetCString(),
                            oUtf8Label.pcGetCString())); 
#if 0
            oUtf8Label.vTrace();       
            oUtf8LabelShort.vTrace();       
#endif
        }
        tBool bIsLabelOk() const {
            return bLabelValid && oUtf8Label.pcGetCString()[0]!=0 && 0!=OSAL_s32StringNCompare(pcGetCString(), "ZZZZC ", 6);
        }
        
        //lint -esym(1739, DAB::trMecaLabel::operator==) Binary operator "" should be non-member function
        tBool operator==(const trMecaLabel &b) const{
            DAB_RET_0_IF_NEQ(bLabelValid);
            DAB_RET_0_IF_NEQ(oUtf8Label);
            DAB_RET_0_IF_NEQ(oUtf8LabelShort);
            DAB_RET_0_IF_NEQ(u8TextControlField);
            DAB_RET_0_IF_NEQ(_u16LabelMask);
            return TRUE;
        }
        //lint -esym(1739, DAB::trMecaLabel::operator!=) Binary operator "" should be non-member function
        tBool operator!=(const trMecaLabel &b) const{
            return !(*this==b);
        }
        //lint -esym(1739, DAB::trMecaLabel::operator<) Binary operator "" should be non-member function
        bool operator<(const trMecaLabel &b) const {
#if 0
            bool bRes=oUtf8Label<b.oUtf8Label ? true : false;
            ETG_TRACE_USR1(("  trMecaLabel: bResult=%d  operator< Label=%10s b.Label=%10s",
                            bRes,
                            oUtf8Label.pcGetCString(),
                            b.oUtf8Label.pcGetCString())); 
#endif
            return oUtf8Label<b.oUtf8Label ? true : false;
        }


        trMecaLabel & operator=(const trMecaLabel &b) {
            if (this == &b)  {
                return *this; 
            }
            bLabelValid = b.bLabelValid;
            oUtf8Label=b.oUtf8Label;
            oUtf8LabelShort=b.oUtf8LabelShort;
			_u16LabelMask=b._u16LabelMask;
			u8TextControlField = b.u8TextControlField;
			memcpy(_dabOrigString,b._dabOrigString,DAB_LABEL_MAX_NUM_CHAR);
			_enDabOrigCharSet=b._enDabOrigCharSet;
            return *this;
            
        }

        tSize getExtLabelLength() const
        {  
            //Updated size with 6 (size of b_label_status, b_charset, w_8_char_abbreviation, b_text_control_field, b_label_bytes)
            if(isExtendedLabel())
            {
                return (getOrigLabelLength() + DAB_LABEL_INFO_LENGTH);
            }
            else
            {
                return (DAB_EXT_LABEL_MIX_NUM_CHAR + DAB_LABEL_INFO_LENGTH);
            }
        }

        tSize getOrigLabelLength() const
        {
            tU8 orgLabel[DAB_LABEL_MAX_NUM_CHAR + 1];
            memcpy(orgLabel, _dabOrigString, DAB_LABEL_MAX_NUM_CHAR);
            return strlen(reinterpret_cast<char *>(orgLabel));
        }

        tBool isExtendedLabel() const
        {
            if (getOrigLabelLength() > DAB_EXT_LABEL_MIX_NUM_CHAR )
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        tBool bLabelValid;
		tU16 _u16LabelMask;
        DAB_tclUtf8String<DAB_LABEL_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN> oUtf8Label;
        DAB_tclUtf8String<DAB_LABEL_SHORT_MAX_NUM_CHAR*DAB_MAX_UTF8_CHAR_LEN> oUtf8LabelShort;
        tenDab_CharSet _enDabOrigCharSet;
        tU8 u8TextControlField;

        //+1 is added for EOL character, It will be useful while printing the string
        //After reading the data into _dabOrigString buffer it may not have EOL character.
        //if try to print it may lead to crash untill it finds another EOL in the memory.
        tU8 _dabOrigString[DAB_LABEL_MAX_NUM_CHAR+1];

    };


struct trMecaEnsemble {

    trMecaEnsemble():_u32Id(0) {}
    explicit trMecaEnsemble(tU32 u32Id):_u32Id(u32Id) {}

    tVoid vParse(tU8 const *pu8Data) {
        //        u8Tag= pu8Data[0];
        _u32Id = DABDRV_GET_U32(&pu8Data[0]);
    };

    // persistant data serialization
    tVoid vSerialize(DAB_Serializer &rSerializer) const {
        rSerializer.vSerialize(_u32Id);
    }

    tVoid vParse(DAB_Parser &rParser) {
        rParser.vParse(_u32Id);
    }

    // meca-serialization
    tVoid vSerialize(tU8 *pu8Data) const {
        DABDRV_SET_U32(&pu8Data[0],_u32Id);
    }

    tVoid vTrace() const {
        ETG_TRACE_USR1(("    trMecaEnsemble:u8ECC=0x%x, u16EID=0x%x id=0x%08x", 
                        u8GetECC(), u16GetEID(), _u32Id));
    };

    tU8 u8GetECC() const {
        return (tU8)((_u32Id & 0x00FF0000)>>16);
    }
    tVoid vSetECC(tU8 u8ECC) {
        _u32Id = (_u32Id & 0xFF00FFFF) | ((tU32)u8ECC << 16);
    }
    tU16 u16GetEID() const {
        return (tU16)(_u32Id & 0x0000FFFF);
    }
    tVoid vSetEID(tU16 u16EID) {
        _u32Id = ((_u32Id & 0xFFFF0000)| u16EID);
    }

    tU32 u32GetID() const {
        return _u32Id;
    }


    tBool bIsValid() const {
        return (_u32Id & 0xFFFF)!=0;
    }
    tVoid vInvalidate() {
        _u32Id=0;
    }

    tBool operator==(trMecaEnsemble const& b) const
    {
        return _u32Id == b._u32Id;
    }

    tBool operator!=(trMecaEnsemble const& b) const
    {
        return _u32Id != b._u32Id;
    }

    tBool operator<(trMecaEnsemble const& b) const
    {
        return _u32Id < b._u32Id;
    }

	trMecaEnsemble& operator=(trMecaEnsemble const& b) 
	{
		if(this == &b) {
			return *this;
		}
		_u32Id = b._u32Id;
		return *this; 
	}
    tU32 _u32Id;
};
    
struct trMecaEnsembleListElement {
    trMecaEnsemble rEnsemble;
    tU8 u8Quality;
    tU32 u32Freq;
    DAB::trMecaLabel rLabel;
    tU8 u8NumberOfAudioServices;
    tU8 u8NumberOfDataServices;	
	tBool bReception;
	string sSihash;
	string sFreqLabel;

    trMecaEnsembleListElement(trMecaEnsemble const &rEnsemble_=trMecaEnsemble(), tU8 u8Quality_=0, tU32 u32Freq_=0, tU8 u8NumberOfAudioServices_=0, tU8 u8NumberOfDataServices_=0,tBool bReception_=FALSE):
        rEnsemble(rEnsemble_),
        u8Quality(u8Quality_),
        u32Freq(u32Freq_),
        rLabel(),
        u8NumberOfAudioServices(u8NumberOfAudioServices_),
        u8NumberOfDataServices(u8NumberOfDataServices_),
		bReception(bReception_),
		sSihash(""),
		sFreqLabel("")
    {}

    tVoid vParse(tU8 const *pu8Data) {

        u8Quality= (pu8Data[1]);
        rEnsemble.vParse(&pu8Data[2]);
    };
    tVoid vTrace() const {
        ETG_TRACE_USR1(("    trMecaEnsembleListElement:ensId=%08x u8Quality=%x", 
                        rEnsemble.u32GetID(),
                        u8Quality)); 
    }

    tBool operator<(trMecaEnsembleListElement const& b) const
    {
        return rEnsemble < b.rEnsemble;
    }

    tBool operator==(trMecaEnsembleListElement const& b) const
    {
        return (rEnsemble == b.rEnsemble) && (u8Quality==b.u8Quality);
    }
    tBool operator!=(trMecaEnsembleListElement const& b) const
    {
        return !(*this==b);
    }
    
};

typedef enum {
    enMeca_ProgrammeServiceType_UNDEF = 0x00,
    enMeca_ProgrammeServiceType_DAB_PROGRAMME_SERVICE = 0x01,
    enMeca_ProgrammeServiceType_FM_PROGRAMME_SERVICE = 0x02,
    enMeca_ProgrammeServiceType_AMFM_PROGRAMME_SERVICE = 0x03
} tenMeca_ProgrammeServiceType;

typedef enum {
    enMeca_ServiceType_INVALID          = 0x00,
    enMeca_ServiceType_AUDIO_SERVICE    = 0x01,
    enMeca_ServiceType_DAB_DATA_SERVICE = 0x02
} tenMeca_ServiceType; // todo: rename to tenMeca_ServiceType

struct trMecaProgrammeService {

#ifdef COMP_SERV_LIST
	trMecaProgrammeService(tU32 u32Id=0, tenMeca_ServiceType enServiceType=enMeca_ServiceType_AUDIO_SERVICE, tU16 u16Scids=0):
        _enServiceType(enServiceType),
        _u32Id(u32Id),
		_u16Scids(u16Scids)
    {}

	trMecaProgrammeService(tenMeca_ProgrammeServiceType enProgrammeServiceType, tU8 u8ECC, tU16 u16SID, tU16 u16Scids=0):
        _enServiceType(enMeca_ServiceType_AUDIO_SERVICE),
		_u16Scids(u16Scids)
    {
        _u32Id=(tU8)((tU32)enProgrammeServiceType<<24) | ((tU16)u8ECC<<16) | ((tU32)u16SID);
    }
#else
    trMecaProgrammeService(tU32 u32Id=0, tenMeca_ServiceType enServiceType=enMeca_ServiceType_AUDIO_SERVICE):
        _enServiceType(enServiceType),
        _u32Id(u32Id)
    {}

    trMecaProgrammeService(tenMeca_ProgrammeServiceType enProgrammeServiceType, tU8 u8ECC, tU16 u16SID):
        _enServiceType(enMeca_ServiceType_AUDIO_SERVICE)
    {
        _u32Id=(tU8)((tU32)enProgrammeServiceType<<24) | ((tU16)u8ECC<<16) | ((tU32)u16SID);
    }
#endif


    tBool bIsDabAudioService() const {
        return enMeca_ServiceType_AUDIO_SERVICE==_enServiceType;
    }

#if 0
    tU8 u8GetScids() const {
        return (tU8)((_u32Id>>16) & 0xFF);
    }

    tU8 u8GetSubChannel() const {
        return (tU8)(u16GetScidi() & 0x3F);
    }
#endif
    

    // persistant data serialization
    tVoid vSerialize(DAB_Serializer &rSerializer) const {
        rSerializer.vSerialize(_u32Id);
        rSerializer.vSerialize(_u16Scids);
    }

    tVoid vParse(DAB_Parser &rParser) {
    	_enServiceType=enMeca_ServiceType_AUDIO_SERVICE;
        rParser.vParse(_u32Id);
        rParser.vParse(_u16Scids);
    }

    // meca-serialization
    tVoid vSerialize(tU8 *pu8Data) const {
        DABDRV_SET_U32(&pu8Data[0],_u32Id);
    }

    tVoid vParse(tU8 const *pu8Data,tenMeca_ServiceType enServiceType=enMeca_ServiceType_AUDIO_SERVICE) {
        _enServiceType=enServiceType;
        _u32Id = DABDRV_GET_U32(&pu8Data[0]);
    };


    tBool bIsValid() const {
        return (_u32Id & 0xFFFF) != 0;
    }

    tVoid vInvalidate() {
        _u32Id=0;
    }

    tVoid vTrace() const {
            ETG_TRACE_USR1(("    trMecaProgrammeService: u32Id=0x%08x srvType=%d scids %d",
                            _u32Id,
                            ETG_CENUM(tenMeca_ServiceType, _enServiceType),_u16Scids));
    };
    

    tU8 u8GetECC() const {
        return (tU8)((_u32Id >> 16) & 0xFF);
    }

    tU16 u16GetSID() const {
        return (tU16)(_u32Id & 0xFFFF);
    }

    tU32 u32GetSID() const {
        return _u32Id;
    }



    tenMeca_ServiceType _enServiceType;
    tU32 _u32Id;
#ifdef COMP_SERV_LIST
	tU16 _u16Scids;
    tU16 u16GetScids() const {
        return _u16Scids;
    }
#endif


};

inline tBool operator==(const trMecaProgrammeService& left, const trMecaProgrammeService& right) {
#ifdef COMP_SERV_LIST
	return (left._u32Id==right._u32Id) && (left._u16Scids==right._u16Scids);
#else
	return (left._u32Id==right._u32Id);
#endif
}

inline tBool operator!=(const trMecaProgrammeService& left, const trMecaProgrammeService& right) {
#ifdef COMP_SERV_LIST
    return (left._u32Id!=right._u32Id) && (left._u16Scids!=right._u16Scids);
#else
	return (left._u32Id!=right._u32Id);
#endif
}

inline tBool operator<(const trMecaProgrammeService& left, const trMecaProgrammeService& right) {
#ifdef COMP_SERV_LIST
//    return (left._u32Id<right._u32Id) && (left._u16Scids<right._u16Scids);
	/*tBool opRes;
	if(left._u32Id == right._u32Id){
		opRes = (left._u16Scids!=right._u16Scids);
	}
	else{
		opRes = (left._u32Id<right._u32Id);
	}

    return opRes;*/

	return (left._u32Id<right._u32Id || (left._u32Id==right._u32Id && left._u16Scids<right._u16Scids));
#else
	 return (left._u32Id<right._u32Id);
#endif
}

struct trMeca_ProgrammeServiceListElement {
    tU8 u8QualityOfService;
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rProgrammeService;
    trMeca_ProgrammeServiceListElement():u8QualityOfService(0),enServiceType(enMeca_ServiceType_AUDIO_SERVICE)
	{
	}
	//trMeca_ProgrammeServiceListElement ( tU8 u8QualityOfService =0,tenMeca_ServiceType enServiceType=enMeca_ServiceType_AUDIO_SERVICE){
		
	//}

    tVoid vTrace() const {
            ETG_TRACE_USR1(("    trMeca_ProgrammeServiceListElement: srvId=%08x u8QualityOfService=0x%x", 
                            rProgrammeService.u32GetSID(),
                            u8QualityOfService));
    }
    tVoid vParse(tU8 const *pu8Data) {
        u8QualityOfService = (pu8Data[1]);
		enServiceType=(tenMeca_ServiceType)pu8Data[0];//Coverity
        rProgrammeService.vParse(&pu8Data[2], enServiceType);        
    };
};




typedef enum {
    enMeca_DbRejectClass_DATABASE=0,
    enMeca_DbRejectClass_ENSEMBLE=1,
    enMeca_DbRejectClass_PROGRAMME_SERVICE=2,
    enMeca_DbRejectClass_DAB_DATA_SERVICE=3,
    enMeca_DbRejectClass_PROGRAMME_SERVICE_COMPONENT=4,
    enMeca_DbRejectClass_DAB_DATA_SERVICE_COMPONENT=5,
    enMeca_DbRejectClass_COMPONENT=6,
    enMeca_DbRejectClass_SUBCHANNEL=7,
    enMeca_DbRejectClass_REGION=8
} tenMeca_DbRejectClass;

typedef enum {
    enMeca_DbRejectCode_PARA_INVALID=0,
    enMeca_DbRejectCode_ENSID_INVALID=1,
    enMeca_DbRejectCode_PSID_INVALID=2,
    enMeca_DbRejectCode_DSID_INVALID=3,
    enMeca_DbRejectCode_SCIDI_INVALID=4,
    enMeca_DbRejectCode_SCIDS_INVALID=5,
    enMeca_DbRejectCode_SUBCH_INVALID=6,
    enMeca_DbRejectCode_REGID_INVALID=7,
    enMeca_DbRejectCode_FILTERID_INVALID=8
    
} tenMeca_DbRejectCode;

// DAB_*_R_DB_REJECT
struct trMeca_RDbReject:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 3) {
            return FALSE;
        }
        if (rMostHdr._u16MecaLen > 9) {
            return FALSE;
        }        

        _rMostHdr=rMostHdr;
        enRejectClass = (tenMeca_DbRejectClass)pu8Data[0];
        enRejectCode = (tenMeca_DbRejectCode)pu8Data[rMostHdr._u16MecaLen - 1];
        u8DataLen = (tU8)(rMostHdr._u16MecaLen - 3);
        OSAL_pvMemoryCopy(u8Data,&pu8Data[2], u8DataLen);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbReject: enRejectClass=%d enRejectCode=%d data=%*p",
                        ETG_CENUM(tenMeca_DbRejectClass, enRejectClass),
                        ETG_CENUM(tenMeca_DbRejectCode, enRejectCode),
                        ETG_LIST_LEN(u8DataLen), ETG_LIST_PTR_T8(u8Data)));
    };

    tenMeca_DbRejectClass enRejectClass;
    tenMeca_DbRejectCode enRejectCode;
    tU8 u8DataLen;
    tU8 u8Data[7];
};


// _DAB_C_DB_GET_DB_ENSEMBLE_LIST
struct trMeca_CDbGetDbEnsembleList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    trMeca_CDbGetDbEnsembleList():
        rRefEnsemble(),
        u8FilterId(0),
        u8NumEnsBefore(),
        u8NumEnsBehind(0xFF)
    {}

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_GET_DB_ENSEMBLE_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 7;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
        pRawOutput->au8Data[4] = u8FilterId;
        pRawOutput->au8Data[5] = u8NumEnsBefore;
        pRawOutput->au8Data[6] = u8NumEnsBehind;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbGetDbEnsembleList:u8FilterId=%u u8NumEnsBefore=%u u8NumEnsBehind=%x rRefEnsemble=0x%08x", 
                        u8FilterId,
                        u8NumEnsBefore,
                        u8NumEnsBehind,
                        rRefEnsemble.u32GetID()));
    };
    
    trMecaEnsemble rRefEnsemble;
    tU8 u8FilterId;
    tU8 u8NumEnsBefore;
    tU8 u8NumEnsBehind;
};


// DAB_*_R_DB_GET_DB_ENSEMBLE_LIST
struct trMeca_RDbGetDbEnsembleList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 6) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rRefEnsemble.vParse(&pu8Data[0]);
        u8FilterId = pu8Data[4];
        u8NumEnsBefore = pu8Data[5];
        u8NumEnsBehind = pu8Data[6];
        u8NumTotalBefore = pu8Data[7];
        u8NumTotalBehind = pu8Data[8];
        u8NumTotalEnsemble = (tU8)(u8NumEnsBefore + u8NumEnsBehind + (rRefEnsemble.bIsValid()?1:0));
        if ((rMostHdr._u16MecaLen -10) != u8NumTotalEnsemble*6) {
            return FALSE;
        }
        for (int i=0;i<u8NumTotalEnsemble;i++) {
            trMecaEnsembleListElement rEnsElem;
            rEnsElem.vParse(&pu8Data[10]+i*6);
            lEnsembleList.push_back(rEnsElem);
            ETG_TRACE_USR1(("  trMeca_RDbGetDbEnsembleList: EId=%x Ens-Quality=0x%x ",
                                            rEnsElem.rEnsemble._u32Id, 
                                            rEnsElem.u8Quality));

        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbGetDbEnsembleList: u8FilterId=%x u8NumEnsBefore=%u u8NumEnsBehind=%u "
                        "u8NumTotalBefore=%u u8NumTotalBehind=%u RefEns=0x%08x EnsList follows:", 
                        u8FilterId,
                        u8NumEnsBefore,
                        u8NumEnsBehind,
                        u8NumTotalBefore,
                        u8NumTotalBehind,
                        rRefEnsemble.u32GetID()));
        for (vector<trMecaEnsembleListElement>::const_iterator iter =lEnsembleList.begin();iter!=lEnsembleList.end();iter++) {
            (*iter).vTrace();
        }
    };

    trMecaEnsemble rRefEnsemble;
    tU8 u8FilterId;
    tU8 u8NumEnsBefore;
    tU8 u8NumEnsBehind;
    tU8 u8NumTotalBefore;
    tU8 u8NumTotalBehind;
    tU8 u8NumTotalEnsemble;
    vector<trMecaEnsembleListElement>lEnsembleList;
};


// _DAB_C_DB_ENSEMBLE_GET_LABEL
struct trMeca_CDbEnsembleGetLabel:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_LABEL;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 4;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetLabel:RefEns=0x%08x",
                        rRefEnsemble.u32GetID()));
    };
    
    trMecaEnsemble rRefEnsemble;

};


// DAB_*_R_DB_ENSEMBLE_GET_LABEL
struct trMeca_RDbEnsembleGetLabel:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 24) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rEnsemble.vParse(&pu8Data[0]);
        rLabel.vParse(&pu8Data[4]);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetLabel: rEnsemble=0x%08x label follows",
                        rEnsemble.u32GetID()));
        rLabel.vTrace();
    };

    trMecaEnsemble rEnsemble;
    trMecaLabel rLabel;
};

#if 0
// _DAB_C_DB_ENSEMBLE_GET_SHORT_LABEL
struct trMeca_CDbEnsembleGetShortLabel:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_SHORT_LABEL;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 4;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetShortLabel:rRefEnsemble=0x%08x",
                        rRefEnsemble.u32GetID()));
    };
    
    trMecaEnsemble rRefEnsemble;

};


// DAB_*_R_DB_ENSEMBLE_GET_SHORT_LABEL
struct trMeca_RDbEnsembleGetShortLabel:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 24) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rEnsemble.vParse(&pu8Data[0]);
        rLabel.vParse(&pu8Data[4], FALSE);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetShortLabel: rEnsemble=0x%08x label follows",
                        rEnsemble.u32GetID()));
        rLabel.vTrace();
    };

    trMecaEnsemble rEnsemble;
    trMecaLabel rLabel;
};

    // DAB_*_C_DB_ENSEMBLE_GET_TIME: not used
#endif

// _DAB_C_DB_ENSEMBLE_GET_INT_TABLE_ID
struct trMeca_CDbEnsembleGetIntTableId:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_INT_TABLE_ID;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 4;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetIntTableId:rRefEnsemble=0x%08x",
                        rRefEnsemble.u32GetID()));
    };
    
    trMecaEnsemble rRefEnsemble;

};


// DAB_*_R_DB_ENSEMBLE_GET_INT_TABLE_ID
struct trMeca_RDbEnsembleGetIntTableId:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 5) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rEnsemble.vParse(&pu8Data[0]);
        u8IntTableId= pu8Data[4];
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetIntTableId: u8IntTableId=0x%x rEnsemble=0x%08x",
                        u8IntTableId, rEnsemble.u32GetID()));
    };

    trMecaEnsemble rEnsemble;
    tU8 u8IntTableId;
};


// _DAB_C_DB_ENSEMBLE_GET_INFO
struct trMeca_CDbEnsembleGetInfo:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_INFO;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 4;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetInfo:rRefEnsemble=0x%08x",
                        rRefEnsemble.u32GetID()));
    };
    
    trMecaEnsemble rRefEnsemble;

};


// DAB_*_R_DB_ENSEMBLE_GET_INFO
struct trMeca_RDbEnsembleGetInfo:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {

        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetInfo:bParse() rMostHdr._u16MecaLen=%d  u8ReceptionQuality=0x%02x ",
                                        rMostHdr._u16MecaLen,
                                        (pu8Data[38])));

        if (rMostHdr._u16MecaLen != 39) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rEnsemble.vParse(&pu8Data[0]);
        u32UtcTime = DABDRV_GET_U32(&pu8Data[4]);
        u32MjdDate = DABDRV_GET_U32(&pu8Data[8]);
        u8Lto=pu8Data[12];
        u8TimeFlags=pu8Data[13];
        u8AlarmFlag=pu8Data[14];
        u8IntTableId=pu8Data[15];
        rLabel.vParse(&pu8Data[16]);
        u8NumProgServices=pu8Data[36];
        u8NumDataServices=pu8Data[37];
        u8ReceptionQuality=(pu8Data[38]);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetInfo: rEnsemble=0x%08x u32UtcTime=0x%x u32MjdDate=0x%x u8Lto=%u "
                        "u8TimeFlags=0x%x u8AlarmFlag=0x%x u8IntTableId=0x%x u8NumProgServices=%u "
                        "u8NumDataServices=%u u8ReceptionQuality=0x%02x EnsLabel follows",
                        rEnsemble.u32GetID(),
                        u32UtcTime,
                        u32MjdDate,
                        u8Lto,
                        u8TimeFlags,
                        u8AlarmFlag,
                        u8IntTableId,
                        u8NumProgServices,
                        u8NumDataServices,
                        u8ReceptionQuality));
        rLabel.vTrace();
    };

    trMecaEnsemble rEnsemble;
    tU32 u32UtcTime;
    tU32 u32MjdDate;
    tU8 u8Lto;
    tU8 u8TimeFlags;
    tU8 u8AlarmFlag;
    tU8 u8IntTableId;
    trMecaLabel rLabel;
    tU8 u8NumProgServices;
    tU8 u8NumDataServices;
    tU8 u8ReceptionQuality;
    
};


// _DAB_C_DB_ENSEMBLE_GET_SERVICE_LIST
struct trMeca_CDbEnsembleGetServiceList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

		trMeca_CDbEnsembleGetServiceList():
	rRefEnsemble(),
	u8FilterId(0),
	enServiceType(enMeca_ServiceType_AUDIO_SERVICE)
	{
	}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_SERVICE_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
        pRawOutput->au8Data[4] = u8FilterId;
        pRawOutput->au8Data[5] = (tU8)enServiceType;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetServiceList:rRefEnsemble=0x%08x u8FilterId=%u  enServiceType=%x", 
                        rRefEnsemble.u32GetID(),
                        u8FilterId,
                        ETG_CENUM(tenMeca_ServiceType, enServiceType)));
    };
    
    trMecaEnsemble rRefEnsemble;
    tU8 u8FilterId;
    tenMeca_ServiceType enServiceType;
};


// DAB_*_R_DB_ENSEMBLE_GET_SERVICE_LIST
struct trMeca_RDbEnsembleGetServiceList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 8) {
            return FALSE;
        }
        lServiceList.clear();
        _rMostHdr=rMostHdr;
        rRefEnsemble.vParse(&pu8Data[0]);
        u8FilterId = pu8Data[4];
        enServiceType = (tenMeca_ServiceType)pu8Data[5];
        u8NumServices = pu8Data[6];
        if ((rMostHdr._u16MecaLen -8) != u8NumServices*4) {
            return FALSE;
        }
        for (int i=0;i<u8NumServices;i++) {
            trMecaProgrammeService rMecaProgrammeService;
            rMecaProgrammeService.vParse(&pu8Data[8]+i*4, enServiceType);
            lServiceList.insert(rMecaProgrammeService);
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetServiceList: rRefEnsemble=0x%08x u8FilterId=%x enServiceType=%x u8NumServices=%d ServiceList follows:", 
                        rRefEnsemble.u32GetID(),
                        u8FilterId,
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u8NumServices));
        for (set<trMecaProgrammeService>::const_iterator iter =lServiceList.begin();iter!=lServiceList.end();++iter) {
            (*iter).vTrace();
        }
    };

    trMecaEnsemble rRefEnsemble;
    tU8 u8FilterId;
    tenMeca_ServiceType enServiceType;
    tU8 u8NumServices;

    set<trMecaProgrammeService>lServiceList;
};





// _DAB_C_DB_ENSEMBLE_GET_REGION_LIST
struct trMeca_CDbEnsembleGetRegionList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_REGION_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 7;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
        DABDRV_SET_U16(&pRawOutput->au8Data[4],u16RefExtdRegionId);
        pRawOutput->au8Data[6] = u8FilterId;
        pRawOutput->au8Data[7] = u8NumRegionsBefore;
        pRawOutput->au8Data[8] = u8NumRegionsBehind;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetRegionList:rRefEnsemble=0x%08x u16RefExtdRegionId=0x%x u8FilterId=%u u8NumRegionsBefore=%u u8NumRegionsBehind=%x", 
                        rRefEnsemble.u32GetID(),
                        u16RefExtdRegionId,
                        u8FilterId,
                        u8NumRegionsBefore,
                        u8NumRegionsBehind));
    };
    
    trMecaEnsemble rRefEnsemble;
    tU16 u16RefExtdRegionId;
    tU8 u8FilterId;
    tU8 u8NumRegionsBefore;
    tU8 u8NumRegionsBehind;
};


// DAB_*_R_DB_ENSEMBLE_GET_REGION_LIST
struct trMeca_RDbEnsembleGetRegionList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 6) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rRefEnsemble.vParse(&pu8Data[0]);
        u8FilterId = pu8Data[4];
        u8NumRegionsBefore = pu8Data[5];
        u8NumRegionsBehind = pu8Data[6];
        u8NumTotalBefore = pu8Data[7];
        u8NumTotalBehind = pu8Data[8];
        tU8 u8NumRegions = (tU8)(u8NumRegionsBefore + u8NumRegionsBehind + (rRefEnsemble.bIsValid()?1:0));
        if ((rMostHdr._u16MecaLen -12) != u8NumRegions*2) {
            return FALSE;
        }
        for (int i=0;i<u8NumRegions;i++) {
            tU16 u16Id = DABDRV_GET_U16(&pu8Data[12]+i*2);
            lExtdRegionIdList.push_back(u16Id);
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetRegionList: rRefEnsemble=0x%08x u8FilterId=%u u8NumRegionsBefore=%u u8NumRegionsBehind=%u "
                        "u8NumTotalBefore=%u u8NumTotalBehind=%u RegIds=%*p", 
                        rRefEnsemble.u32GetID(),
                        u8FilterId,
                        u8NumRegionsBefore,
                        u8NumRegionsBehind,
                        u8NumTotalBefore,
                        u8NumTotalBehind,
                        ETG_LIST_LEN((tU32)lExtdRegionIdList.size()),
                        ETG_LIST_PTR_T16(&lExtdRegionIdList[0])));
    };

    trMecaEnsemble rRefEnsemble;
    tU8 u8FilterId;
    tU8 u8NumRegionsBefore;
    tU8 u8NumRegionsBehind;
    tU8 u8NumTotalBefore;
    tU8 u8NumTotalBehind;
    vector<tU16>lExtdRegionIdList;
};


// _DAB_C_DB_ENSEMBLE_GET_REGION_LABEL
struct trMeca_CDbEnsembleGetRegionLabel:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_REGION_LABEL;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
        DABDRV_SET_U16(&pRawOutput->au8Data[4],u16ExtdRegionId);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetRegionLabel:rRefEnsemble=0x%08x",
                        rRefEnsemble.u32GetID()));
    };
    
    trMecaEnsemble rRefEnsemble;
    tU16 u16ExtdRegionId;


};


// DAB_*_R_DB_ENSEMBLE_GET_REGION_LABEL
struct trMeca_RDbEnsembleGetRegionLabel:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 26) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rEnsemble.vParse(&pu8Data[0]);
        u16ExtdRegionId = DABDRV_GET_U16(&pu8Data[4]);
        rLabel.vParse(&pu8Data[6]);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetRegionLabel: rEnsemble=0x%08x u16ExtdRegionId=0x%x Label follows",
                        rEnsemble.u32GetID(),
                        u16ExtdRegionId));
        rLabel.vTrace();

    };

    trMecaEnsemble rEnsemble;
    tU16 u16ExtdRegionId;
    trMecaLabel rLabel;

};




// _DAB_C_DB_ENSEMBLE_GET_FREQUENCY_LIST
struct trMeca_CDbEnsembleGetFrequencyList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_ENSEMBLE_GET_FREQUENCY_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
        pRawOutput->au8Data[4]= 0; // rfu
        pRawOutput->au8Data[5]= u8NumFrequencies;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbEnsembleGetFrequencyList:rRefEnsemble=0x%08x u8NumFrequencies=%u", 
                        rRefEnsemble.u32GetID(),
                        u8NumFrequencies));
    };
    
    trMecaEnsemble rRefEnsemble;
    tU8 u8NumFrequencies;
};


// DAB_*_R_DB_ENSEMBLE_GET_FREQUENCY_LIST
struct trMeca_RDbEnsembleGetFrequencyList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 6) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rRefEnsemble.vParse(&pu8Data[0]);
        u8NumFrequencies = pu8Data[5];

        if ((rMostHdr._u16MecaLen -6) != u8NumFrequencies*5) {
            return FALSE;
        }
        tU8 const *pu8QualStart = &pu8Data[6]+u8NumFrequencies*4;
        for (int i=0;i<u8NumFrequencies;i++) {
            tU32 u32Freq = DABDRV_GET_U32(&pu8Data[6]+i*4);
            tU8 u8Qual = (pu8QualStart[i]);
            lFrequencyList.push_back(u32Freq);
            lReceptionQualityList.push_back(u8Qual);
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        tU32 u32NumElem=(tU32)lFrequencyList.size(); 
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetFrequencyList: rRefEnsemble=0x%08x u8NumFrequencies=%u lFrequencyList=%*p ...", 
                        rRefEnsemble.u32GetID(),
                        u32NumElem,
                        ETG_LIST_LEN(u32NumElem),
                        ETG_LIST_PTR_T32(&lFrequencyList[0])));
        ETG_TRACE_USR1(("  trMeca_RDbEnsembleGetFrequencyList(cont):lReceptionQualityList=%*p",
                        ETG_LIST_LEN(u32NumElem),
                        ETG_LIST_PTR_T8(&lReceptionQualityList[0])));
    };

    trMecaEnsemble rRefEnsemble;
    tU8 u8NumFrequencies;
    vector<tU32>lFrequencyList;
    vector<tU8>lReceptionQualityList;
};




// _DAB_C_DB_GET_DB_SERVICE_LIST
struct trMeca_CDbGetServiceList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_GET_DB_SERVICE_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 8;
        pRawOutput->au8Data[0]= (tU8)enServiceType;
        pRawOutput->au8Data[1]= u8FilterId;
        pRawOutput->au8Data[2]= u8NumServBefore;
        pRawOutput->au8Data[3]= u8NumServBehind;
        rRefProgrammeService.vSerialize(&pRawOutput->au8Data[4]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbGetServiceList:enServiceType=%x u8FilterId=0x%x " 
                        "u8NumServBefore=%u u8NumServBehind=%u refService=0x%08x", 
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u8FilterId,
                        u8NumServBefore,
                        u8NumServBehind,
                        rRefProgrammeService.u32GetSID()));
    };
    
    tenMeca_ServiceType enServiceType;
    tU8 u8FilterId;
    tU8 u8NumServBefore;
    tU8 u8NumServBehind;
    trMecaProgrammeService rRefProgrammeService;

};


// DAB_*_R_DB_GET_DB_SERVICE_LIST
struct trMeca_RDbGetServiceList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if ((rMostHdr._u16MecaLen) < 10) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        enServiceType = (tenMeca_ServiceType)pu8Data[0];
        u8FilterId = pu8Data[1];
        u8NumServBefore = pu8Data[2];
        u8NumServBehind = pu8Data[3];
        u8NumTotalBefore = pu8Data[4];
        u8NumTotalBehind = pu8Data[5];
        rRefProgrammeService.vParse(&pu8Data[6], enServiceType);

        tU8 u8NumServices = (tU8)(u8NumServBefore + u8NumServBehind + (rRefProgrammeService.u32GetSID()?1:0));
        if ((rMostHdr._u16MecaLen -10) != u8NumServices*6) {
            return FALSE;
        }
        for (int i=0;i<u8NumServices;i++) {
            trMeca_ProgrammeServiceListElement rElem;
            rElem.vParse((&pu8Data[10])+i*6);
            lServiceElemList.push_back(rElem);
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbGetServiceList: enServiceType=%x u8FilterId=0x%x u8NumServBefore=%u "
                        "u8NumServBehind=%u u8NumTotalBefore=%u u8NumTotalBehind=%u RefService=0x%08x List follows:", 
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u8FilterId,
                        u8NumServBefore,
                        u8NumServBehind,
                        u8NumTotalBefore,
                        u8NumTotalBehind,
                        rRefProgrammeService.u32GetSID()));
        for (vector<trMeca_ProgrammeServiceListElement>::const_iterator iter =lServiceElemList.begin();iter!=lServiceElemList.end();iter++) {
            (*iter).vTrace();
        }
    };

    tenMeca_ServiceType enServiceType;
    tU8 u8FilterId;
    tU8 u8NumServBefore;
    tU8 u8NumServBehind;
    tU8 u8NumTotalBefore;
    tU8 u8NumTotalBehind;
    trMecaProgrammeService rRefProgrammeService;
    vector<trMeca_ProgrammeServiceListElement>lServiceElemList;
};



// _DAB_C_DB_SERVICE_GET_LABEL
struct trMeca_CDbServiceGetLabel:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    trMeca_CDbServiceGetLabel():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE) 
    {}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_GET_LABEL;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        pRawOutput->au8Data[0]=(tU8)enServiceType;
        rRefService.vSerialize(&pRawOutput->au8Data[2]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceGetLabel:rRefService=0x%08x",
                        rRefService.u32GetSID()));
    };
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rRefService;


};


// DAB_*_R_DB_SERVICE_GET_LABEL
struct trMeca_RDbServiceGetLabel:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 26) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        enServiceType=(tenMeca_ServiceType)pu8Data[0];
        rRefService.vParse(&pu8Data[22],enServiceType); 
        rLabel.vParse(&pu8Data[2]);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceGetLabel: sid=0x%08x ServiceInfo follows",
                        rRefService.u32GetSID()));
        rLabel.vTrace();

    };
    tenMeca_ServiceType enServiceType;
    trMecaLabel rLabel;
    trMecaProgrammeService rRefService;

};




// _DAB_C_DB_SERVICE_GET_COMPONENT_LIST
struct trMeca_CDbServiceGetComponentList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    trMeca_CDbServiceGetComponentList():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE),
        u8FilterId(0)
    { }
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_GET_COMPONENT_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        pRawOutput->au8Data[0]= u8FilterId;
        pRawOutput->au8Data[1]= (tU8)enServiceType;
        rRefProgrammeService.vSerialize(&pRawOutput->au8Data[2]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceGetComponentList:u8FilterId=0x%x sid=0x%08x enServiceType=%x" 
                        "refService follows", 
                        u8FilterId,
                        rRefProgrammeService.u32GetSID(),
                        ETG_CENUM(tenMeca_ServiceType, enServiceType)));
    };
    
    tenMeca_ServiceType enServiceType;
    tU8 u8FilterId;
    trMecaProgrammeService rRefProgrammeService;

};


    //static tU8 u8GetSubChannelFromScidi(tU16 u16Scidi) {
    //    return u16Scidi & 0x3F;
    //}
// DAB_*_R_SERVICE_GET_COMPONENT_LIST
struct trMeca_RDbServiceGetComponentList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 8) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8FilterId = pu8Data[0];
        enServiceType= (tenMeca_ServiceType)pu8Data[2];
        rRefProgrammeService.vParse(&pu8Data[4],enServiceType);

        u8NumComponents = pu8Data[3];
        if ((rMostHdr._u16MecaLen -8) != u8NumComponents*3) {
            return FALSE;
        }
        tU8 const *pu8SCIDS= &pu8Data[8+2*u8NumComponents];
        for (int i=0;i<u8NumComponents;i++) {
            tU16 u16SCIDI= DABDRV_GET_U16(&pu8Data[8]+i*2);
            tU8 u8SCIDS= pu8SCIDS[i];
            lu16SCIDI.push_back(u16SCIDI);
            lu8SCIDS.push_back(u8SCIDS);
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        tU32 u32NumElem=(tU32)lu16SCIDI.size(); 
        ETG_TRACE_USR1(("  trMeca_RDbServiceGetComponentList:ServiceType=%u u8FilterId=%u refService=0x%08x " 
                        "SCIDI[]=%*p Ref Service follows:", 
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u8FilterId,
                        rRefProgrammeService.u32GetSID(),
                        ETG_LIST_LEN(u32NumElem),
                        ETG_LIST_PTR_T16(&lu16SCIDI[0])));


        ETG_TRACE_USR1(("  trMeca_RDbServiceGetComponentList(cont) u8SCIDS[]=%*p",
                        ETG_LIST_LEN(u32NumElem),
                        ETG_LIST_PTR_T8(&lu8SCIDS[0])));
    };

    tU8 u8NumComponents;
    tU8 u8FilterId;
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rRefProgrammeService;
    vector<tU16>lu16SCIDI;
    vector<tU8>lu8SCIDS;
};





// _DAB_C_DB_SERVICE_GET_ANNOUNCEMENT_SUPPORT
struct trMeca_CDbServiceGetAnnouncementSupport:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_GET_ANNOUNCEMENT_SUPPORT;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 2;
        pRawOutput->au8Data[0]=(tU8)enMeca_ServiceType_AUDIO_SERVICE;
        pRawOutput->au8Data[1]=0; // rfu
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceGetAnnouncementSupport:"));
    };

};


// DAB_*_R_DB_SERVICE_GET_ANNOUNCEMENT_SUPPORT
struct trMeca_RDbServiceGetAnnouncementSupport:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 5) {
            return FALSE;
        }
        if (pu8Data[0]!=(tU8)enMeca_ServiceType_AUDIO_SERVICE) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8IntTableId= pu8Data[1];
        u16AnnoMask = DABDRV_GET_U16(&pu8Data[2]);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceGetAnnouncementSupport: u8IntTableId=0x%x u16AnnoMask=0x%x",
                        u8IntTableId,
                        u16AnnoMask));
    };

    tU8 u8IntTableId;
    tU16 u16AnnoMask;
};


// _DAB_C_DB_SERVICE_GET_PTY
struct trMeca_CDbServiceGetPty:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    
    trMeca_CDbServiceGetPty():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE)
    {}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_GET_PTY;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        pRawOutput->au8Data[0]=(tU8)enServiceType;
        pRawOutput->au8Data[1]=0; // rfu
        rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);

        
    }
    
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceGetPty:enServiceType=%x Service=0x%08x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rProgrammeService.u32GetSID()));
    };
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rProgrammeService;

};


typedef enum {
    enMeca_DbPtyMask_INTERNATIONAL_CODE         = DAB_b0,
    enMeca_DbPtyMask_INTERNATIONAL_CODE_DYNAMIC = DAB_b1,
    enMeca_DbPtyMask_SECOND_COARSE_CODE         = DAB_b2,
    enMeca_DbPtyMask_SECOND_COARSE_CODE_DYNAMIC = DAB_b3,
    enMeca_DbPtyMask_FINE_CODE                  = DAB_b4,
    enMeca_DbPtyMask_FINE_CODE_DYNAMIC          = DAB_b5,
    enMeca_DbPtyMask_SECOND_FINE_CODE           = DAB_b6,
    enMeca_DbPtyMask_SECOND_FINE_CODE_DYNAMIC   = DAB_b7
} tenMeca_DbPtyMask;

typedef enum {
    enMeca_DbPtyCodeIndex_INTERNATIONAL_CODE         = 0x00,
    enMeca_DbPtyCodeIndex_INTERNATIONAL_CODE_DYNAMIC = 0x01,
    enMeca_DbPtyCodeIndex_SECOND_COARSE_CODE         = 0x02,
    enMeca_DbPtyCodeIndex_SECOND_COARSE_CODE_DYNAMIC = 0x03,
    enMeca_DbPtyCodeIndex_FINE_CODE                  = 0x04,
    enMeca_DbPtyCodeIndex_FINE_CODE_DYNAMIC          = 0x05,
    enMeca_DbPtyCodeIndex_SECOND_FINE_CODE           = 0x06,
    enMeca_DbPtyCodeIndex_SECOND_FINE_CODE_DYNAMIC   = 0x07
} tenMeca_DbPtyCodeIndex;

// DAB_*_R_DB_SERVICE_GET_PTY
struct trMeca_RDbServiceGetPty:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 16) {
            return FALSE;
        }
        if (pu8Data[0]!=(tU8)enMeca_ServiceType_AUDIO_SERVICE) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rProgrammeService.vParse(&pu8Data[2]);
        u8IntTableId= pu8Data[6];
        u8PtyMask= pu8Data[7];
        OSAL_pvMemoryCopy(u8PtyCodes,&pu8Data[8], 8);

        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceGetPty: u8IntTableId=0x%x u8PtyMask=0x%x u8PtyCodes[]=%*p",
                        u8IntTableId,
                        u8PtyMask,
                        ETG_LIST_LEN(8),
                        ETG_LIST_PTR_T8(u8PtyCodes)));
    };

    trMecaProgrammeService rProgrammeService;
    tU8 u8IntTableId;
    tU8 u8PtyMask;
    tU8 u8PtyCodes[8];
};


typedef enum {
    enMeca_DbRangeModId_DAB_ENSEMBLE     = 0x00,
    enMeca_DbRangeModId_FMRDS            = 0x08,
    enMeca_DbRangeModId_FM_NO_RDS        = 0x09,
    enMeca_DbRangeModId_AM_MW9KHZ_AND_LW = 0x0a,
    enMeca_DbRangeModId_AM_MW5KHZ_AND_SW = 0x0c
} tenMeca_DbRangeModId;

// _DAB_C_DB_SERVICE_GET_AF
struct trMeca_CDbServiceGetAF:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    
    trMeca_CDbServiceGetAF():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE),
        u8FilterId(0),
        u16ExtdRegionId(0),
        enRangeModId(enMeca_DbRangeModId_DAB_ENSEMBLE)
    {}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_GET_AF;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 14;
        pRawOutput->au8Data[0]=(tU8)enServiceType;
        pRawOutput->au8Data[1]=u8FilterId;
        rEnsemble.vSerialize(&pRawOutput->au8Data[2]);
        DABDRV_SET_U16(&pRawOutput->au8Data[6], u16ExtdRegionId);
        pRawOutput->au8Data[8]=(tU8)enRangeModId;
        pRawOutput->au8Data[9]=0; // rfu
        rProgrammeService.vSerialize(&pRawOutput->au8Data[10]);

        
    }
    
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceGetAF:enServiceType=%x u8FilterId=0x%x u16ExtdRegionId=0x%x "
                        "enRangeModId=%x sid=0x%08x Ensemble follows",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u8FilterId,
                        u16ExtdRegionId,
                        ETG_CENUM(tenMeca_DbRangeModId, enRangeModId),
                        rProgrammeService.u32GetSID()));
    };

    tenMeca_ServiceType enServiceType;
    tU8 u8FilterId;
    trMecaEnsemble rEnsemble;
    tU16 u16ExtdRegionId;
    tenMeca_DbRangeModId enRangeModId;
    trMecaProgrammeService rProgrammeService;


};



// DAB_*_R_DB_SERVICE_GET_AF
struct trMeca_RDbServiceGetAF:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 14) {
            return FALSE;
        }
        if (pu8Data[0]!=(tU8)enMeca_ServiceType_AUDIO_SERVICE) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8FilterId = pu8Data[1];
        rEnsemble.vParse(&pu8Data[2]);
        u16ExtdRegionId=DABDRV_GET_U16(&pu8Data[6]);
        enRangeModId =(tenMeca_DbRangeModId)pu8Data[8];
        u8NumFrequencies=pu8Data[9];
        rProgrammeService.vParse(&pu8Data[10]);
        if (rMostHdr._u16MecaLen -14 != u8NumFrequencies*9) {
            return FALSE;
        }
        for (tU8 i=0;i<u8NumFrequencies;i++) {
            tU8 const *pu8EnsOrSrv=&pu8Data[14+i*9];
            if (enRangeModId == enMeca_DbRangeModId_DAB_ENSEMBLE) {
                trMecaEnsemble rEns;
                rEns.vParse(pu8EnsOrSrv);
                lEnsembleList[i]=rEns;
                //lEnsembleList.push_back(rEns);
            }
            else {
                trMecaProgrammeService rSrv;
                rSrv.vParse(pu8EnsOrSrv);
                lServiceList[i]=rSrv;
                //lServiceList.push_back(rSrv);
            }
            lu32FrequencyList[i]=DABDRV_GET_U32(pu8EnsOrSrv+4);
            //lu32FrequencyList.push_back(DABDRV_GET_U32(pu8EnsOrSrv+4));
            lu8FlagsList[i]=pu8EnsOrSrv[8];
            //lu8FlagsList.push_back(pu8EnsOrSrv[8]);
        }

        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceGetAF: u8FilterId=0x%x u16ExtdRegionId=0x%x "
                        "enRangeModId=%x u8NumFrequencies=%u",
                        u8FilterId,
                        u16ExtdRegionId,
                        enRangeModId,
                        u8NumFrequencies));
        for (tU8 i=0;i<u8NumFrequencies;i++) {
            if (enRangeModId == enMeca_DbRangeModId_DAB_ENSEMBLE) {
                lEnsembleList[i].vTrace();
            }
            else {
                lServiceList[i].vTrace();
            }

            ETG_TRACE_USR1(("    Frequency=%u Flag=0x%x",
                            lu32FrequencyList[i],
                            lu8FlagsList[i]));
        }
    };

    tU8 u8FilterId;
    trMecaEnsemble rEnsemble;
    tU16 u16ExtdRegionId;
    tenMeca_DbRangeModId enRangeModId;
    trMecaProgrammeService rProgrammeService;
    tU8 u8NumFrequencies;
    vector<trMecaEnsemble>lEnsembleList;
    vector<trMecaProgrammeService>lServiceList;
    vector<tU32>lu32FrequencyList;
    vector<tU8>lu8FlagsList;
};


// _DAB_C_DB_SERVICE_GET_INFO
struct trMeca_CDbServiceGetInfo:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    trMeca_CDbServiceGetInfo():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE)
    {}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_GET_INFO;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        pRawOutput->au8Data[0]=(tU8)enServiceType;
        pRawOutput->au8Data[1]=0; // rfu
        rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceGetInfo:enServiceType0%x Service=0x%08x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rProgrammeService.u32GetSID()));
    };
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rProgrammeService;



};


// DAB_*_R_DB_SERVICE_GET_INFO
struct trMeca_RDbServiceGetInfo:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        // default:Audio;
        tU8 const *pu8NumComp=&pu8Data[4];
        tU8 const *pu8TimeFlags=&pu8Data[5];
        tU8 const *pu8UtcTime=&pu8Data[6];
        tU8 const *pu8MjdData=&pu8Data[10];
        tU8 const *pu8Lto=&pu8Data[13];
        tU8 const *pu8Service=&pu8Data[14];
        tU8 const *pu8Label=&pu8Data[18];
        tU8 const *pu8ReceptionQual=&pu8Data[38];


        if (rMostHdr._u16MecaLen < 1) {
            return FALSE;
        }
        enServiceType= (tenMeca_ServiceType)pu8Data[0];
        if (enServiceType == enMeca_ServiceType_AUDIO_SERVICE) {
            if (rMostHdr._u16MecaLen != 48) {
                return FALSE;
            }
            
            u16AnnoMask = DABDRV_GET_U16(&pu8Data[2]); 
            u8IntTableId = pu8Data[1];
            u8PtyMask = pu8Data[39];
            OSAL_pvMemoryCopy(u8PtyCodes,&pu8Data[40], 8);


        }
        else if (enServiceType==enMeca_ServiceType_DAB_DATA_SERVICE) {
            if (rMostHdr._u16MecaLen != 36) {
                return FALSE;
            }
            pu8TimeFlags=&pu8Data[1];
            pu8UtcTime=&pu8Data[2];
            pu8MjdData=&pu8Data[6];
            pu8Lto=&pu8Data[9];
            pu8Service=&pu8Data[10];
            pu8Label=&pu8Data[14];
            
            pu8NumComp = &pu8Data[34];
            pu8ReceptionQual = &pu8Data[35];
        }
        else {
            return FALSE;
        }

        _rMostHdr=rMostHdr;
        u8TimeFlags=*pu8TimeFlags;
        u32UtcTime = DABDRV_GET_U32(pu8UtcTime);
        u32MjdDate = DABDRV_GET_U24(pu8MjdData); // todo: check
        u8Lto = *pu8Lto;
        rProgrammeService.vParse(pu8Service, enServiceType);
        rLabel.vParse(pu8Label);
        if (FALSE == rLabel.bLabelValid) {
            return FALSE;
        }
        u8NumberOfComponents = *pu8NumComp;
        u8ReceptionQuality = (*pu8ReceptionQual);

        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceGetInfo: serviceType=0x%x sid=0x%08x u8TimeFlags=0x%x u32UtcTime=0x%x "
                        "u32MjdDate=0x%x u8Lto=0x%x u8NumberOfComponents=%u u8ReceptionQuality=0x%x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rProgrammeService.u32GetSID(),
                        u8TimeFlags,
                        u32UtcTime,
                        u32MjdDate,
                        u8Lto,
                        u8NumberOfComponents,
                        u8ReceptionQuality));
        if (enServiceType == enMeca_ServiceType_AUDIO_SERVICE) {
            ETG_TRACE_USR1(("  trMeca_RDbServiceGetInfo:(AUDIO):u8IntTableId=0x%x u16AnnoMask=0x%x " 
                            "u8PtyMask=0x%x u8PtyCodes[]=%*p label follows",
                            u8IntTableId,
                            u16AnnoMask,
                            u8PtyMask,
                            ETG_LIST_LEN(8),
                            ETG_LIST_PTR_T8(u8PtyCodes)));
        }
        rLabel.vTrace();

    };
    // common:
    tenMeca_ServiceType enServiceType;
    tU8 u8TimeFlags;
    tU32 u32UtcTime;
    tU32 u32MjdDate;
    tU8 u8Lto;
    trMecaProgrammeService rProgrammeService;
    trMecaLabel rLabel;

    tU8 u8NumberOfComponents;
    tU8 u8ReceptionQuality;
    
    // audio only
    tU8 u8IntTableId;
    tU16 u16AnnoMask;
    tU8 u8PtyMask;
    tU8 u8PtyCodes[8];
};










// _DAB_C_DB_SERVICE_GET_LINK_INFO
struct trMeca_CDbServiceGetLinkInfo:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    
    trMeca_CDbServiceGetLinkInfo():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE)
    { }
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_GET_LINK_INFO;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        pRawOutput->au8Data[0]=(tU8)enServiceType;
        pRawOutput->au8Data[1]=0; // rfu
        rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);
    }
    
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceGetLinkInfo:enServiceType=%x Service=0x%08x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rProgrammeService.u32GetSID()));
    };
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rProgrammeService;

};


typedef enum {
    enMeca_DbLinkResponseStatus_ONEANDONLY_LINK   = 0x00,
    enMeca_DbLinkResponseStatus_FIRST_LINK        = 0x01,
    enMeca_DbLinkResponseStatus_INTERMEDIATE_LINK = 0x02,
    enMeca_DbLinkResponseStatus_LAST_LINK         = 0x03
} tenMeca_DbLinkResponseStatus;

typedef enum {
    enMeca_DbLinkFlags_LINK_ACTIVE         = DAB_b0,
    enMeca_DbLinkFlags_OTHER_ENSEMBLE         = DAB_b3,
    enMeca_DbLinkFlags_CHANGE_FLAG         = DAB_b5,
    enMeca_DbLinkFlags_DELINK_FLAG         = DAB_b6
} tenMeca_DbLinkFlags;

// DAB_*_R_DB_SERVICE_GET_LINK_INFO
struct trMeca_RDbServiceGetLinkInfo:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 12) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        enResponseStatus = (tenMeca_DbLinkResponseStatus)pu8Data[1];
        u16Lsn = DABDRV_GET_U16(&pu8Data[2]);
        u8LinkFlags = pu8Data[4];
        u16ShortHandFlags = DABDRV_GET_U16(&pu8Data[5]);
        u8NumberLinkedServices = pu8Data[7];
        rRefProgrammeService.vParse(&pu8Data[8]);
        enServiceType = (tenMeca_ServiceType)pu8Data[0];

        if (rMostHdr._u16MecaLen - 12 != u8NumberLinkedServices*4) {
            return FALSE;
        }
        for (tU8 i=0;i<u8NumberLinkedServices;i++) {
            trMecaProgrammeService rLinkedService;
            rLinkedService.vParse(&pu8Data[12+4*i], enServiceType);
            lLinkedServices.push_back(rLinkedService);
        }

        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceGetLinkInfo: enServiceType=%d enResponseStatus=%x u16Lsn=0x%x u8LinkFlags=0x%x "
                        "u16ShortHandFlags=0x%x u8NumberLinkedServices=%u refService=0x%08x List follows",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        ETG_CENUM(tenMeca_DbLinkResponseStatus, enResponseStatus),
                        u16Lsn,
                        u8LinkFlags,
                        u16ShortHandFlags,
                        u8NumberLinkedServices,
                        rRefProgrammeService.u32GetSID()));
        for (tU8 i=0;i<u8NumberLinkedServices;i++) {
            lLinkedServices[i].vTrace();
        }
    };
    tenMeca_ServiceType enServiceType;
    tenMeca_DbLinkResponseStatus enResponseStatus;
    tU16 u16Lsn;
    tU8 u8LinkFlags;
    tU16 u16ShortHandFlags;
    tU8 u8NumberLinkedServices;
    trMecaProgrammeService rRefProgrammeService;
    vector<trMecaProgrammeService> lLinkedServices;
    
};



#if 0
// _DAB_C_DB_PROGRAMME_SERVICE_GET_SHORT_LABEL
struct trMeca_CDbProgrammeServiceGetShortLabel:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_PROGRAMME_SERVICE_GET_SHORT_LABEL;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 4;
        rService.vSerialize(&pRawOutput->au8Data[0]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbProgrammeServiceGetShortLabel:Service=0x%08x",
                        rService.u32GetSID()));
    };
    
    trMecaProgrammeService rService;

};


// DAB_*_R_DB_PROGRAMME_SERVICE_GET_SHORT_LABEL
struct trMeca_RDbProgrammeServiceGetShortLabel:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 24) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rService.vParse(&pu8Data[0]);
        rLabel.vParse(&pu8Data[4], FALSE);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbProgrammeServiceGetShortLabel: Service=0x%08x and EnsInfo follows",
                        rService.u32GetSID()));
        rLabel.vTrace();

    };

    trMecaProgrammeService rService;
    trMecaLabel rLabel;

};

#endif

// _DAB_C_DB_SERVICE_COMPONENT_GET_SCIDI
struct trMeca_CDbServiceComponentGetSCIDI:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    trMeca_CDbServiceComponentGetSCIDI():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE),
        u8SCIDS(0)
    { }
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_COMPONENT_GET_SCIDI;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 6;
        pRawOutput->au8Data[0] = (tU8)enServiceType;
        pRawOutput->au8Data[1] = u8SCIDS;
        rService.vSerialize(&pRawOutput->au8Data[2]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceComponentGetSCIDI:ServiceType=%x u8SCIDS=0x%x Service=0x%08x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u8SCIDS,
                        rService.u32GetSID()));
    }
    tenMeca_ServiceType enServiceType;
    tU8 u8SCIDS;
    trMecaProgrammeService rService;

};


// DAB_*_R_DB_SERVICE_COMPONENT_GET_SCIDI
struct trMeca_RDbServiceComponentGetSCIDI:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 24) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8SCIDS = pu8Data[1];
        u16SCIDI = DABDRV_GET_U16(&pu8Data[2]);
        enServiceType = (tenMeca_ServiceType)pu8Data[0];
        rService.vParse(&pu8Data[4], enServiceType);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceComponentGetSCIDI:ServiceType=%x SID=0x%08x u8SCIDS=0x%x u16SCIDI=0x%04x Service follows",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rService.u32GetSID(),
                        u8SCIDS,
                        u16SCIDI));
    }

    tenMeca_ServiceType enServiceType;
    tU8 u8SCIDS;
    trMecaProgrammeService rService;
    tU16 u16SCIDI;
};






// _DAB_C_DB_SERVICE_COMPONENT_GET_LABEL
struct trMeca_CDbServiceComponentGetLabel:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    trMeca_CDbServiceComponentGetLabel():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE ),
        u16SCIDI(0)
    {}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_COMPONENT_GET_LABEL;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 8;
        pRawOutput->au8Data[0] = (tU8)enServiceType;
        pRawOutput->au8Data[1]= 0; // rfu
        DABDRV_SET_U16(&pRawOutput->au8Data[2], u16SCIDI);
        rRefService.vSerialize(&pRawOutput->au8Data[4]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceComponentGetLabel:enServiceType=%d rRefService=0x%08x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rRefService.u32GetSID()));
    };
    
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rRefService;
    tU16 u16SCIDI;



};


// DAB_*_R_DB_SERVICE_COMPONENT_GET_LABEL
struct trMeca_RDbServiceComponentGetLabel:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 28) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u16SCIDI = DABDRV_GET_U16(&pu8Data[22]);
        rLabel.vParse(&pu8Data[2]);
        enServiceType=(tenMeca_ServiceType)pu8Data[0];
        rRefService.vParse(&pu8Data[24], enServiceType);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceComponentGetLabel: serviceType=%x SID=0x%08x u16SCIDI=0x%04x ServiceLabel and ServiceInfo follows",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rRefService.u32GetSID(),
                        u16SCIDI));
        rLabel.vTrace();

    };
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rRefService;
    tU16 u16SCIDI;
    trMecaLabel rLabel;

};






// _DAB_C_DB_SERVICE_COMPONENT_GET_INFO
struct trMeca_CDbServiceComponentGetInfo:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    trMeca_CDbServiceComponentGetInfo():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE),
        u16SCIDI(0)
    { }
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_COMPONENT_GET_INFO;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 8;
        pRawOutput->au8Data[0] = (tU8)enServiceType;
        pRawOutput->au8Data[1] = 0; // rfu
        DABDRV_SET_U16(&pRawOutput->au8Data[2], u16SCIDI);
        rService.vSerialize(&pRawOutput->au8Data[4]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceComponentGetInfo:ServiceType=%x u16SCIDI=0x%04x SID=0x%08x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u16SCIDI,
                        rService.u32GetSID()));
    };
    tenMeca_ServiceType enServiceType;
    tU16 u16SCIDI;
    trMecaProgrammeService rService;

};


// DAB_*_R_DB_SERVICE_COMPONENT_GET_INFO
struct trMeca_RDbServiceComponentGetInfo:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 36) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8SCIDS = pu8Data[1];
        u16SCIDI = DABDRV_GET_U16(&pu8Data[4]);
        u8Language = pu8Data[6];
        u8Flags = pu8Data[7];
        u8Id = pu8Data[8];
        u8Scty = pu8Data[9];
        u16PacketAddress = DABDRV_GET_U16(&pu8Data[10]);
        rLabel.vParse(&pu8Data[12]);

        enServiceType=(tenMeca_ServiceType)pu8Data[0];
        rService.vParse(&pu8Data[32], enServiceType);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceComponentGetInfo:ServiceType=%x u8SCIDS=0x%02x u16SCIDI=0x%04x "
                        "u8Language=0x%x u8Flags=0x%x u8Id=0x%x u8Scty=0x%x u16PacketAddress=0x%x "
                        "SID=0x%08x label follows",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        u8SCIDS,
                        u16SCIDI,
                        u8Language,
                        u8Flags,
                        u8Id,
                        u8Scty,
                        u16PacketAddress,
                        rService.u32GetSID()));
        rLabel.vTrace();

    };

    tenMeca_ServiceType enServiceType;
    tU8 u8SCIDS;
    trMecaProgrammeService rService;
    tU16 u16SCIDI;
    tU8 u8Language;
    tU8 u8Flags;
    tU8 u8Id;
    tU8 u8Scty;
    tU16 u16PacketAddress;
    trMecaLabel rLabel;
};



// _DAB_C_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_LIST
struct trMeca_CDbServiceComponentGetUserApplicationList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    trMeca_CDbServiceComponentGetUserApplicationList():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE),
        u16SCIDI(0)
    { }
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 8;
        pRawOutput->au8Data[0] = (tU8)enServiceType;
        pRawOutput->au8Data[1] = 0; // rfu
        DABDRV_SET_U16(&pRawOutput->au8Data[2], u16SCIDI);
        rService.vSerialize(&pRawOutput->au8Data[4]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceComponentGetUserApplicationList:ServiceType=%x SID=0x%08x u16SCIDI=0x%04x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rService.u32GetSID(),
                        u16SCIDI));
    };
    tenMeca_ServiceType enServiceType;
    tU16 u16SCIDI;
    trMecaProgrammeService rService;

};


// DAB_*_R_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_LIST
struct trMeca_RDbServiceComponentGetUserApplicationList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 10) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8SCIDS = pu8Data[1];
        u16SCIDI = DABDRV_GET_U16(&pu8Data[2]);
        u8NumUaids = pu8Data[5];

        enServiceType = (tenMeca_ServiceType)pu8Data[0];
        rService.vParse(&pu8Data[6], enServiceType);
        if (rMostHdr._u16MecaLen - 10 !=  u8NumUaids*2) {
            return FALSE;
        }
        for (tU8 i=0;i<u8NumUaids;i++) {
            lu16Uaids.push_back(DABDRV_GET_U16(&pu8Data[10+i*2]));
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceComponentGetUserApplicationList:ServiceType=%x SID=0x%08x u8SCIDS=0x%02x u16SCIDI=0x%04x "
                        "u8NumUaids=%u "
                        "u16Uaids[]=%*p",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rService.u32GetSID(),
                        u8SCIDS,
                        u16SCIDI,
                        u8NumUaids,
                        ETG_LIST_LEN(u8NumUaids),
                        ETG_LIST_PTR_T16(&lu16Uaids[0])));
    };

    tenMeca_ServiceType enServiceType;
    tU8 u8SCIDS;
    trMecaProgrammeService rService;
    tU16 u16SCIDI;
    tU8 u8NumUaids;
    vector<tU16> lu16Uaids;
};






// _DAB_C_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_DATA
struct trMeca_CDbServiceComponentGetUserApplicationData:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    trMeca_CDbServiceComponentGetUserApplicationData():
        enServiceType(enMeca_ServiceType_AUDIO_SERVICE),
        u16SCIDI(0) 
    {}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_DATA;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 8;
        pRawOutput->au8Data[0] = (tU8)enServiceType;
        pRawOutput->au8Data[1] = 0; // rfu
        DABDRV_SET_U16(&pRawOutput->au8Data[2], u16SCIDI);
        rService.vSerialize(&pRawOutput->au8Data[4]);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbServiceComponentGetUserApplicationData:ServiceType=%x SID=0x%08x u16SCIDI=0x%04x",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rService.u32GetSID(),
                        u16SCIDI));
    };
    tenMeca_ServiceType enServiceType;
    tU16 u16SCIDI;
    trMecaProgrammeService rService;

};


// DAB_*_R_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_DATA
struct trMeca_RDbServiceComponentGetUserApplicationData:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 12) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8SCIDS = pu8Data[1];
        u16SCIDI = DABDRV_GET_U16(&pu8Data[2]);
        u16UserAppId = DABDRV_GET_U16(&pu8Data[4]);
        u8UserDataLenght = pu8Data[6];

        enServiceType= (tenMeca_ServiceType)pu8Data[0];
        rService.vParse(&pu8Data[8], enServiceType);
        if (rMostHdr._u16MecaLen - 12 !=  u8UserDataLenght) {
            return FALSE;
        }
        for (tU8 i=0;i<u8UserDataLenght;i++) {
            lu8UserData.push_back(pu8Data[12+i]);
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbServiceComponentGetUserApplicationData:ServiceType=%x SID=0x%08x u8SCIDS=0x%02x u16SCIDI=0x%04x "
                        "u16UserAppId=0x%x  u8UserDataLenght=%u"
                        "u8UserData[]=%*p",
                        ETG_CENUM(tenMeca_ServiceType, enServiceType),
                        rService.u32GetSID(),
                        u8SCIDS,
                        u16SCIDI,
                        u16UserAppId,
                        u8UserDataLenght,
                        ETG_LIST_LEN(u8UserDataLenght),
                        ETG_LIST_PTR_T8(&lu8UserData[0])));
    };

    tenMeca_ServiceType enServiceType;
    tU8 u8SCIDS;
    trMecaProgrammeService rService;
    tU16 u16SCIDI;
    tU16 u16UserAppId;
    tU8 u8UserDataLenght;
    vector<tU8> lu8UserData;
};


// _DAB_C_DB_COMPONENT_GET_INFO
struct trMeca_CDbComponentGetInfo:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_COMPONENT_GET_INFO;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 2;
        DABDRV_SET_U16(&pRawOutput->au8Data[0], u16SCIDI);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbComponentGetInfo:u16SCIDI=0x%04x",
                        u16SCIDI));
    };
    tU16 u16SCIDI;

};


// DAB_*_R_DB_COMPONENT_GET_INFO
struct trMeca_RDbComponentGetInfo:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 8) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u16SCIDI = DABDRV_GET_U16(&pu8Data[0]);
        u8Language = pu8Data[2];
        u8Flags = pu8Data[3];
        u8Id = pu8Data[4];
        u8Scty = pu8Data[5];
        u16PacketAddress = DABDRV_GET_U16(&pu8Data[6]);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbComponentGetInfo:u16SCIDI=0x%04x "
                        "u8Language=0x%x u8Flags=0x%x u8Id=0x%x u8Scty=0x%x u16PacketAddress=0x%x",
                        u16SCIDI,
                        u8Language,
                        u8Flags,
                        u8Id,
                        u8Scty,
                        u16PacketAddress));
    };

    tU16 u16SCIDI;
    tU8 u8Language;
    tU8 u8Flags;
    tU8 u8Id;
    tU8 u8Scty;
    tU16 u16PacketAddress;
};


// _DAB_C_DB_SUBCHANNEL_GET_INFO
struct trMeca_CDbSubchannelGetInfo:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SUBCHANNEL_GET_INFO;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 1;
        pRawOutput->au8Data[0] = u8SubchId;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbSubchannelGetInfo:u8SubchId=0x%02x",
                        u8SubchId));
    };
    tU8 u8SubchId;

};


// DAB_*_R_DB_SUBCHANNEL_GET_INFO
struct trMeca_RDbSubchannelGetInfo:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen != 8) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        u8SubchId = pu8Data[0];
        u8ProtectionLevel = pu8Data[1];
        u16BitRate = DABDRV_GET_U16(&pu8Data[2]);
        u16NumberCU = DABDRV_GET_U16(&pu8Data[4]);
        u16StartCU = DABDRV_GET_U16(&pu8Data[6]);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbSubchannelGetInfo:u8SubchId=0x%02x "
                        "u8ProtectionLevel=0x%x u16BitRate=0x%x u16NumberCU=0x%x u16StartCU=0x%x",
                        u8SubchId,
                        u8ProtectionLevel,
                        u16BitRate,
                        u16NumberCU,
                        u16StartCU));
    };

    tU8 u8SubchId;
    tU8 u8ProtectionLevel;
    tU16 u16BitRate;
    tU16 u16NumberCU;
    tU16 u16StartCU;
    
};





#if 0
// _DAB_C_DB_REGION_GET_LIST
struct trMeca_CDbRegionGetList:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_REGION_GET_LIST;
        pRawOutput->enOpType = enOpType_GET;
        pRawOutput->u16DataLen = 10;
        rRefEnsemble.vSerialize(&pRawOutput->au8Data[0]);
        DABDRV_SET_U16(&pRawOutput->au8Data[4], u16ExtdRegionId);
        pRawOutput->au8Data[6] = u8FilterId;
        pRawOutput->au8Data[7] = u8NumRegionsBefore;
        pRawOutput->au8Data[8] = u8NumRegionsBehind;
        pRawOutput->au8Data[8] = 0; // rfu
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbRegionGetList:rRefEnsemble=0x%08x u16ExtdRegionId=%x u8FilterId=0x%x u8NumRegionsBefore=%u "
                        "u8NumRegionsBehind=%u",
                        rRefEnsemble.u32GetID(),
                        u16ExtdRegionId,
                        u8FilterId,
                        u8NumRegionsBefore,
                        u8NumRegionsBehind));
    };
    trMecaEnsemble rRefEnsemble;
    tU16 u16ExtdRegionId;
    tU8 u8FilterId;
    tU8 u8NumRegionsBefore;
    tU8 u8NumRegionsBehind;

};


// DAB_*_R_DB_REGION_GET_LIST
struct trMeca_RDbRegionGetList:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 12) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        rRefEnsemble.vParse(&pu8Data[0]);
        u16ExtdRegionId=DABDRV_GET_U16(&pu8Data[4]);
        u8FilterId = pu8Data[6];
        u8NumRegionsBefore = pu8Data[7];
        u8NumRegionsBehind = pu8Data[8];
        u8NumTotalBefore = pu8Data[9];
        u8NumTotalBehind = pu8Data[10];
        u8NumRegions = pu8Data[11];
        if (rMostHdr._u16MecaLen -12 != u8NumRegions*6) {
            return FALSE;
        }
        tU8 const *pu8EnsStart =&pu8Data[12];
        tU8 const *pu8IdsStart =pu8EnsStart + u8NumRegions*4;
        for (tU8 i=0; i<u8NumRegions;i++) {
            trMecaEnsemble rEns;
            rEns.vParse(pu8EnsStart+i*4);
            lEnsembles.push_back(rEns);
            lu6ExtRegionIds.push_back(pu8IdsStart[i]);
            
        }
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbRegionGetList:rRefEnsemble=0x%08x u16ExtdRegionId=%x u8FilterId=0x%x u8NumRegionsBefore=%u "
                        "u8NumRegionsBehind=%u u8NumTotalBefore=%u u8NumTotalBehind=%u u8NumRegions=%u "
                        "Lists follows",
                        rRefEnsemble.u32GetID(),
                        u16ExtdRegionId,
                        u8FilterId,
                        u8NumRegionsBefore,
                        u8NumRegionsBehind,
                        u8NumTotalBefore,
                        u8NumTotalBehind,
                        u8NumRegions));
        for (tU8 i=0;i<u8NumRegions;i++) {
            lEnsembles[i].vTrace();
            ETG_TRACE_USR1(("   lu6ExtRegionId=0x%x",
                            lu6ExtRegionIds[i]));
        }
    };
    trMecaEnsemble rRefEnsemble;
    tU16 u16ExtdRegionId;
    tU8 u8FilterId;
    tU8 u8NumRegionsBefore;
    tU8 u8NumRegionsBehind;
    tU8 u8NumTotalBefore;
    tU8 u8NumTotalBehind;
    tU8 u8NumRegions;

    vector<trMecaEnsemble> lEnsembles;
    vector<tU16> lu6ExtRegionIds;
};
#endif

typedef enum {
    enMeca_DbNotifyCommand_INVALID = 0x00,
    enMeca_DbNotifyCommand_SET = 0x01,
    enMeca_DbNotifyCommand_DELETE = 0x02,
    enMeca_DbNotifyCommand_DELETE_ALL = 0x03,
    enMeca_DbNotifyCommand_SET_ALL = 0x04,
    enMeca_DbNotifyCommand_GET_SELECTED = 0x05,
    enMeca_DbNotifyCommand_SET_AUTO_RDMAPI_CURR = 0x06,
    enMeca_DbNotifyCommand_DEL_AUTO_RDMAPI_CURR = 0x07
} tenMeca_DbNotifyCommand;


typedef enum {
    enMeca_DbNotifyClass_INVALID = 0x00,
    enMeca_DbNotifyClass_DATABASE_GET_DB_ENSEMBLE_LIST = 0x02,
    enMeca_DbNotifyClass_DATABASE_GET_DB_SERVICE_LIST = 0x03,
    enMeca_DbNotifyClass_ENSEMBLE = 0x11,
    enMeca_DbNotifyClass_PROGRAMME_SERVICE = 0x21,
    enMeca_DbNotifyClass_DAB_DATA_SERVICE = 0x22,
    enMeca_DbNotifyClass_PROGRAMME_SERVICE_COMPONENT = 0x31,
    enMeca_DbNotifyClass_DATA_SERVICE_COMPONENT = 0x32,
    enMeca_DbNotifyClass_COMPONENT = 0x33,
    enMeca_DbNotifyClass_SUBCHANNEL = 0x41
} tenMeca_DbNotifyClass;


typedef enum {
    enMeca_DbNotifyMessageEns_INFO = 0x01,
    enMeca_DbNotifyMessageEns_LABEL = 0x02,
    enMeca_DbNotifyMessageEns_SHORT_LABEL = 0x04,
    enMeca_DbNotifyMessageEns_TIME = 0x08,
    enMeca_DbNotifyMessageEns_SRV_LIST_PROG = 0x10,
    enMeca_DbNotifyMessageEns_SRV_LIST_DATA = 0x20
} tenMeca_DbNotifyMessageEns;

typedef enum {
    enMeca_DbNotifyMessageSrv_LABEL = 0x0001,
    enMeca_DbNotifyMessageSrv_SHORT_LABEL = 0x0002,
    enMeca_DbNotifyMessageSrv_TIME = 0x0004,
    enMeca_DbNotifyMessageSrv_ANNO_SUPPORT = 0x0008,
    enMeca_DbNotifyMessageSrv_COMP_LIST = 0x0010,
    enMeca_DbNotifyMessageSrv_PTY = 0x0020,
    enMeca_DbNotifyMessageSrv_AF = 0x0040,
    enMeca_DbNotifyMessageSrv_INFO = 0x0080,
    enMeca_DbNotifyMessageSrv_LINK_INFO = 0x0100
} tenMeca_DbNotifyMessageSrv;

typedef enum {
    enMeca_DbNotifyMessageSrvComp_LABEL = 0x0001,
    enMeca_DbNotifyMessageSrvComp_USER_APP_LIST = 0x0002,
    //    enMeca_DbNotifyMessageSrvComp_INFO = 0x0004
    enMeca_DbNotifyMessageSrvComp_INFO = 0x0008
} tenMeca_DbNotifyMessageSrvComp;

typedef enum {
    enMeca_DbNotifyMessageComponent_INFO = 0x0001
} tenMeca_DbNotifyMessageComponent;

typedef enum {
    enMeca_DbNotifyMessageSubChannel_INFO = 0x0001
} tenMeca_DbNotifyMessageSubChannel;

//DAB_*_C_DB_SET_AUTO_NOTIFICATION
struct trMeca_CDbSetAutoNotification:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL
    
    trMeca_CDbSetAutoNotification() {
        u8FilterId = 0;
        enNotifyCommand = enMeca_DbNotifyCommand_INVALID;
        enNotifyClass = enMeca_DbNotifyClass_INVALID;
        u16NotifyMessage = 0;
        u8NumBefore = 0;
        u8NumBehind = 0;
        u16SCIDI = 0xFFFF;
        u8SubchId = 0;
        enServiceType=enMeca_ServiceType_AUDIO_SERVICE;
    }
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_SET_AUTO_NOTIFICATION;
        pRawOutput->enOpType = enOpType_SET;
        pRawOutput->u16DataLen = 10;
        pRawOutput->au8Data[0]= u8FilterId;
        pRawOutput->au8Data[1]= 0; // rfu
        pRawOutput->au8Data[2]=(tU8)enNotifyCommand;
        pRawOutput->au8Data[3]=(tU8)enNotifyClass;

        switch (enNotifyClass) {
            case enMeca_DbNotifyClass_DATABASE_GET_DB_ENSEMBLE_LIST:
            {
                rEnsemble.vSerialize(&pRawOutput->au8Data[4]);
                pRawOutput->au8Data[8]=u8NumBefore;
                pRawOutput->au8Data[9]=u8NumBehind;
                pRawOutput->u16DataLen = 10;
                break;
            }
            case enMeca_DbNotifyClass_DATABASE_GET_DB_SERVICE_LIST:
            {
                pRawOutput->au8Data[4] = (tU8)enServiceType;
                pRawOutput->au8Data[5]=0; // rfu
                pRawOutput->au8Data[6]=u8NumBefore;
                pRawOutput->au8Data[7]=u8NumBehind;
                rService.vSerialize(&pRawOutput->au8Data[8]);
                pRawOutput->u16DataLen = 12;                
                break;
            }
            case enMeca_DbNotifyClass_ENSEMBLE:
            {
                DABDRV_SET_U16(&pRawOutput->au8Data[4], u16NotifyMessage);
                rEnsemble.vSerialize(&pRawOutput->au8Data[6]);
                pRawOutput->u16DataLen = 10;
                break;
            }
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE:
            case enMeca_DbNotifyClass_DAB_DATA_SERVICE:
            {
                DABDRV_SET_U16(&pRawOutput->au8Data[4], u16NotifyMessage);
                rService.vSerialize(&pRawOutput->au8Data[6]);
                pRawOutput->u16DataLen = 10;
                break;
            }
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE_COMPONENT:
            case enMeca_DbNotifyClass_DATA_SERVICE_COMPONENT:
            {
                DABDRV_SET_U16(&pRawOutput->au8Data[4], u16NotifyMessage);
                rService.vSerialize(&pRawOutput->au8Data[6]);
                DABDRV_SET_U16(&pRawOutput->au8Data[10], u16SCIDI);
                pRawOutput->u16DataLen = 12;
                break;
            }
            case enMeca_DbNotifyClass_COMPONENT:
            {
                DABDRV_SET_U16(&pRawOutput->au8Data[4], u16NotifyMessage);
                DABDRV_SET_U16(&pRawOutput->au8Data[6], u16SCIDI);
                pRawOutput->u16DataLen = 8;
                break;
            }
            case enMeca_DbNotifyClass_SUBCHANNEL:
            {
                DABDRV_SET_U16(&pRawOutput->au8Data[4], u16NotifyMessage);
                pRawOutput->au8Data[6]=u8SubchId;
                pRawOutput->u16DataLen = 7;
                break;
            }
            case enMeca_DbNotifyClass_INVALID:
            default:
                break;
        }
    }
    
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbSetAutoNotification: u8FilterId=0x%x enNotifyCommand=%x enNotifyClass=%x ...",
                        u8FilterId,
                        ETG_CENUM(tenMeca_DbNotifyCommand, enNotifyCommand),
                        ETG_CENUM(tenMeca_DbNotifyClass, enNotifyClass)));
        switch (enNotifyClass) {
            case enMeca_DbNotifyClass_DATABASE_GET_DB_ENSEMBLE_LIST:
                ETG_TRACE_USR1(("    u8NumBefore=%u u8NumBehind=%u rEnsemble=0x%08x", 
                                u8NumBefore, u8NumBehind, rEnsemble.u32GetID()));
             break;
            case enMeca_DbNotifyClass_DATABASE_GET_DB_SERVICE_LIST:
                ETG_TRACE_USR1(("    u8NumBefore=%u u8NumBehind=%u enServiceType=%d refService=0x%08x", 
                                u8NumBefore, u8NumBehind,
                                ETG_CENUM(tenMeca_ServiceType, enServiceType),
                                rService.u32GetSID()));
                break;
            case enMeca_DbNotifyClass_ENSEMBLE:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x rEnsemble=0x%08x", 
                                ETG_CENUM(tenMeca_DbNotifyMessageEns,u16NotifyMessage),
                                rEnsemble.u32GetID()));
                break;
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE:
            case enMeca_DbNotifyClass_DAB_DATA_SERVICE:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x service=0x%08x", 
                                ETG_CENUM(tenMeca_DbNotifyMessageSrv,u16NotifyMessage),
                                rService.u32GetSID()));
                break;
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE_COMPONENT:
            case enMeca_DbNotifyClass_DATA_SERVICE_COMPONENT:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x u16SCIDI=0x%04x SID=0x%08x", 
                                ETG_CENUM(tenMeca_DbNotifyMessageSrvComp,u16NotifyMessage),
                                u16SCIDI,
                                rService.u32GetSID()));
                break;
            case enMeca_DbNotifyClass_COMPONENT:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x u16SCIDI=0x%04x", 
                                ETG_CENUM(tenMeca_DbNotifyMessageComponent,u16NotifyMessage),
                                u16SCIDI));    
                break;
            case enMeca_DbNotifyClass_SUBCHANNEL:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x u8SubchId=0x%02x", 
                                ETG_CENUM(tenMeca_DbNotifyMessageSubChannel,u16NotifyMessage),
                                u8SubchId));
                break;
            case enMeca_DbNotifyClass_INVALID:
            default:
                break;
        }
    };
    // generic
    tU8 u8FilterId;
    tenMeca_DbNotifyCommand enNotifyCommand;
    tenMeca_DbNotifyClass enNotifyClass;
    tU16 u16NotifyMessage;

    tU8 u8NumBefore;
    tU8 u8NumBehind;
    
    // only DATABASE_GET_DB_ENSEMBLE_LIST
    trMecaEnsemble rEnsemble;

    // only DATABASE_GET_DB_SERVICE_LIST
    trMecaProgrammeService rService;
    tenMeca_ServiceType enServiceType;
    // only PROGRAMME_SERVICE_COMPONENT and DATA_SERVICE_COMPONENT and COMPONENT
    tU16 u16SCIDI;

    // only SUBCHANNEL
    tU8 u8SubchId;
};


// DAB_*_R_DB_SET_AUTO_NOTIFICATION
struct trMeca_RDbSetAutoNotification:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        tU8 u8UsedLen=0;
        _rMostHdr=rMostHdr;
        /* buffer for content of variable size, to avoid range-check on every byte
           Data for the maximum size are allocated.
           So the range-check can be done as last step.
         */
        tU8 au8Data[10];
        if (rMostHdr._u16MecaLen < 2) {
            return FALSE;
        }
        if (rMostHdr._u16MecaLen > 12) {
            return FALSE;
        }
        OSAL_pvMemoryCopy(au8Data,&pu8Data[2], rMostHdr._u16MecaLen-2);
        u8FilterId=pu8Data[0];
        enNotifyClass = (tenMeca_DbNotifyClass)pu8Data[1];
        
        switch (enNotifyClass) { 
            case enMeca_DbNotifyClass_DATABASE_GET_DB_ENSEMBLE_LIST:
                rEnsemble.vParse(&au8Data[0]);
                u8NumBefore = au8Data[4];
                u8NumBehind = au8Data[5];
                u8UsedLen = 8;
                break;
            case enMeca_DbNotifyClass_DATABASE_GET_DB_SERVICE_LIST:
                enServiceType =(tenMeca_ServiceType)au8Data[0];
                u8NumBefore = au8Data[2];
                u8NumBehind = au8Data[3];
                rService.vParse(&au8Data[4], (tenMeca_ServiceType)au8Data[0]);
                u8UsedLen = 10;
                break;
            case enMeca_DbNotifyClass_ENSEMBLE:
                u16NotifyMessage = DABDRV_GET_U16(&au8Data[0]);
                rEnsemble.vParse(&au8Data[2]);
                u8UsedLen = 8;
                break;
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE:
            case enMeca_DbNotifyClass_DAB_DATA_SERVICE:
                u16NotifyMessage = DABDRV_GET_U16(&au8Data[0]);
                rService.vParse(&au8Data[2], 
                                enNotifyClass==enMeca_DbNotifyClass_PROGRAMME_SERVICE ?
                                enMeca_ServiceType_AUDIO_SERVICE : enMeca_ServiceType_DAB_DATA_SERVICE);
                u8UsedLen = 8;
                break;
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE_COMPONENT:
            case enMeca_DbNotifyClass_DATA_SERVICE_COMPONENT:
                u16NotifyMessage = DABDRV_GET_U16(&au8Data[0]);
                rService.vParse(&au8Data[2], 
                                enNotifyClass==enMeca_DbNotifyClass_PROGRAMME_SERVICE_COMPONENT ?
                                enMeca_ServiceType_AUDIO_SERVICE : enMeca_ServiceType_DAB_DATA_SERVICE);
                u16SCIDI=DABDRV_GET_U16(&au8Data[6]);
                u8UsedLen = 10;
                break;
            case enMeca_DbNotifyClass_COMPONENT:
                u16NotifyMessage = DABDRV_GET_U16(&au8Data[0]);
                u16SCIDI=DABDRV_GET_U16(&au8Data[2]);
                u8UsedLen = 6;
                break;
            case enMeca_DbNotifyClass_SUBCHANNEL:
                u16NotifyMessage = DABDRV_GET_U16(&au8Data[0]);
                u8SubchId=au8Data[2];
                u8UsedLen = 5;
                break;
            case enMeca_DbNotifyClass_INVALID:
            default:
                break;
        }
        return u8UsedLen==rMostHdr._u16MecaLen;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbSetAutoNotification: u8FilterId=0x%x  enNotifyClass=%x ...",
                        u8FilterId,
                        ETG_CENUM(tenMeca_DbNotifyClass, enNotifyClass)));
        switch (enNotifyClass) {
            case enMeca_DbNotifyClass_DATABASE_GET_DB_ENSEMBLE_LIST:
                ETG_TRACE_USR1(("    u8NumBefore=%u u8NumBehind=%u rEnsemble=0x%08x", 
                                u8NumBefore, u8NumBehind, rEnsemble.u32GetID()));
             break;
            case enMeca_DbNotifyClass_DATABASE_GET_DB_SERVICE_LIST:
                ETG_TRACE_USR1(("    srvType=%u u8NumBefore=%u u8NumBehind=%u SID=0x%08x", 
                                ETG_CENUM(tenMeca_ServiceType, enServiceType),
                                u8NumBefore, u8NumBehind, rService.u32GetSID()));
                break;
            case enMeca_DbNotifyClass_ENSEMBLE:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x rEnsemble=0x%08x", 
                                u16NotifyMessage, rEnsemble.u32GetID()));
                break;
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE:
            case enMeca_DbNotifyClass_DAB_DATA_SERVICE:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x SID=0x%08x", 
                                u16NotifyMessage,
                                rService.u32GetSID()));
                break;
            case enMeca_DbNotifyClass_PROGRAMME_SERVICE_COMPONENT:
            case enMeca_DbNotifyClass_DATA_SERVICE_COMPONENT:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x u16SCIDI=0x%04x SID=0x%08x", 
                                u16NotifyMessage,
                                u16SCIDI,
                                rService.u32GetSID()));
                break;
            case enMeca_DbNotifyClass_COMPONENT:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x u16SCIDI=0x%04x", 
                                u16NotifyMessage,
                                u16SCIDI));    
                break;
            case enMeca_DbNotifyClass_SUBCHANNEL:
                ETG_TRACE_USR1(("    u16NotifyMessage=0x%x u8SubchId=0x%02x", 
                                u16NotifyMessage,
                                u8SubchId));
                break;
            case enMeca_DbNotifyClass_INVALID:
            default:
                break;
        }
    };
    // generic
    tU8 u8FilterId;
    tenMeca_DbNotifyClass enNotifyClass;
    tU16 u16NotifyMessage;

    tU8 u8NumBefore;
    tU8 u8NumBehind;
    
    // only DATABASE_GET_DB_ENSEMBLE_LIST
    trMecaEnsemble rEnsemble;

    // only DATABASE_GET_DB_SERVICE_LIST
    tenMeca_ServiceType enServiceType;
    trMecaProgrammeService rService;

    // only PROGRAMME_SERVICE_COMPONENT and DATA_SERVICE_COMPONENT and COMPONENT
    tU16 u16SCIDI;

    // only SUBCHANNEL
    tU8 u8SubchId;

};


typedef enum {
    enMeca_DbFilterElemType_NOP                         = 0x0000,
    enMeca_DbFilterElemType_PTY_FILTER                  = 0x1000, /*not implemented*/
    enMeca_DbFilterElemType_ANNOUNCEMENT_FILTER         = 0x1001, //not implemented
    enMeca_DbFilterElemType_AUDIO_SERVICE_FILTER        = 0x1002, //not implemented
    enMeca_DbFilterElemType_ENSEMBLE_HAS_SERVICE_FILTER = 0x1003, //not implemented
    enMeca_DbFilterElemType_COMPONENT_LANGUAGE_SORTER   = 0x1004,
    enMeca_DbFilterElemType_SHORTLABEL_SORTER           = 0x1005,
    enMeca_DbFilterElemType_ID_SORTER                   = 0x1006, //not implemented
    enMeca_DbFilterElemType_COMPONENT_LANGUAGE_FILTER   = 0x1007,
    enMeca_DbFilterElemType_COMPONENT_TMC_FILTER        = 0x1008,
    enMeca_DbFilterElemType_ANNOCLUSTER_FILTER          = 0x1009,
    enMeca_DbFilterElemType_OEMSID_SORTER               = 0x100a,
    enMeca_DbFilterElemType_LABELVALID_FILTER           = 0x100b,
    enMeca_DbFilterElemType_ENSFREQUENCY_SORTER         = 0x100d,
    enMeca_DbFilterElemType_AUDIO_SERVICE_FILTER_NODMB  = 0x100e
} tenMeca_DbFilterElemType;

typedef enum {
    enMeca_DbFilterClassMask_ENSEMBLE  = DAB_b0,
    enMeca_DbFilterClassMask_SERVICE   = DAB_b1,
    enMeca_DbFilterClassMask_COMPONENT = DAB_b2
} tenMeca_DbFilterClassMask;

typedef enum {
    enMeca_DbFilterCommand_NEW                    = 0x10,
    enMeca_DbFilterCommand_NEW_WITH_AUTO_DESTRUCT = 0x11,
    enMeca_DbFilterCommand_CLEAR_CHAIN            = 0x20,
    enMeca_DbFilterCommand_GET_CHAIN              = 0x30,
    enMeca_DbFilterCommand_ADD_ELEMENT            = 0x40,
    enMeca_DbFilterCommand_REMOVE_ELEMENT         = 0x50
} tenMeca_DbFilterCommand;

// _DAB_C_DB_FILTERID
struct trMeca_CDbFilterId:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_FILTERID;
        pRawOutput->enOpType = enOpType_SETGET;
        pRawOutput->u16DataLen = (tU16)(8+u8DataFieldLen);
        pRawOutput->au8Data[0] = (tU8)enFilterCommand;
        pRawOutput->au8Data[1]= u8FilterId;
        DABDRV_SET_U16(&pRawOutput->au8Data[2], (tU16)enFilterElemType);
        DABDRV_SET_U16(&pRawOutput->au8Data[4], u16ClassMask);

        pRawOutput->au8Data[6]= 0; // rfu
        pRawOutput->au8Data[7]= u8DataFieldLen;
        OSAL_pvMemoryCopy(&pRawOutput->au8Data[8],au8DataField, u8DataFieldLen<64 ? u8DataFieldLen:64);
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbFilterId:enFilterCommand=%x u8FilterId=0x%x enFilterElemType=%x "
                        "u16ClassMask=0x%x u8DataFieldLen=%u au8DataField[]=%*p",
                        ETG_CENUM(tenMeca_DbFilterCommand, enFilterCommand),
                        u8FilterId,
                        ETG_CENUM(tenMeca_DbFilterElemType, enFilterElemType),
                        u16ClassMask,
                        u8DataFieldLen,
                        ETG_LIST_LEN(u8DataFieldLen>64?64:u8DataFieldLen),
                        ETG_LIST_PTR_T8(au8DataField)));
    };
    
    tenMeca_DbFilterCommand enFilterCommand;
    tU8 u8FilterId;
    tenMeca_DbFilterElemType enFilterElemType;
    tU16 u16ClassMask;
    tU8 u8DataFieldLen;
    tU8 au8DataField[64];
};


typedef enum {
    enMeca_DbFilterResponse_NEW_ERROR              = 0x10,
    enMeca_DbFilterResponse_NEW_OK                 = 0x11,
    enMeca_DbFilterResponse_NEW_ALLREADY_EXIST     = 0x12,
    enMeca_DbFilterResponse_CLEAR_CHAIN_ERROR      = 0x20,
    enMeca_DbFilterResponse_CLEAR_CHAIN_OK         = 0x21,
    enMeca_DbFilterResponse_GET_CHAIN_ERROR        = 0x30,
    enMeca_DbFilterResponse_GET_CHAIN_FIRST        = 0x31,
    enMeca_DbFilterResponse_GET_CHAIN_INTERMEDIATE = 0x32,
    enMeca_DbFilterResponse_GET_CHAIN_LAST         = 0x33,
    enMeca_DbFilterResponse_ADD_ELEMENT_ERROR      = 0x40,
    enMeca_DbFilterResponse_ADD_ELEMENT_OK         = 0x41
} tenMeca_DbFilterResponse;
// DAB_*_R_DB_FILTERID

#define DAB_MECA_DB_MAX_FILTER_DATA_FIELD_LEN 64
struct trMeca_RDbFilterId:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
        if (rMostHdr._u16MecaLen < 8) {
            return FALSE;
        }
        _rMostHdr=rMostHdr;
        enFilterResponse = (tenMeca_DbFilterResponse)pu8Data[0];
        u8FilterId = pu8Data[1];
        enFilterElemType = (tenMeca_DbFilterElemType)DABDRV_GET_U16(&pu8Data[2]);
        u16ClassMask = DABDRV_GET_U16(&pu8Data[4]);
        u8DataFieldLen = pu8Data[7];
        if (u8DataFieldLen > DAB_MECA_DB_MAX_FILTER_DATA_FIELD_LEN) {
            return FALSE;
        }
        if (rMostHdr._u16MecaLen != 8+u8DataFieldLen) {
            return FALSE;
        }
        OSAL_pvMemoryCopy(au8DataField,&pu8Data[8],u8DataFieldLen);
        return TRUE;
    }
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbFilterId:enFilterResponse=%x u8FilterId=0x%x enFilterElemType=%x "
                        "u16ClassMask=0x%x u8DataFieldLen=%u au8DataField[]=%*p",
                        ETG_CENUM(tenMeca_DbFilterResponse, enFilterResponse),
                        u8FilterId,
                        ETG_CENUM(tenMeca_DbFilterElemType, enFilterElemType),
                        u16ClassMask,
                        u8DataFieldLen,
                        ETG_LIST_LEN(u8DataFieldLen>DAB_MECA_DB_MAX_FILTER_DATA_FIELD_LEN?DAB_MECA_DB_MAX_FILTER_DATA_FIELD_LEN:u8DataFieldLen),
                        ETG_LIST_PTR_T8(au8DataField)));
    };
    tenMeca_DbFilterResponse enFilterResponse;
    tU8 u8FilterId;
    tenMeca_DbFilterElemType enFilterElemType;
    tU16 u16ClassMask;
    tU8 u8DataFieldLen;
    tU8 au8DataField[DAB_MECA_DB_MAX_FILTER_DATA_FIELD_LEN];
    trMeca_RDbFilterId()
    {
    	enFilterResponse = enMeca_DbFilterResponse_NEW_ERROR;
    	u8FilterId = 0;
    	enFilterElemType = enMeca_DbFilterElemType_NOP;
    	u16ClassMask = 0;
    	u8DataFieldLen = 0;
    	for (tU8 u8Index = 0; u8Index<DAB_MECA_DB_MAX_FILTER_DATA_FIELD_LEN ;u8Index++){
    	au8DataField[u8Index] = {0};}
    }
};

typedef enum {
    enMeca_DbQueryCmd_FREE              = 0x00,
    enMeca_DbQueryCmd_EVALUATE          = 0x01,
    enMeca_DbQueryCmd_PREPARE           = 0x02,
    enMeca_DbQueryCmd_REOPENE_READWRITE = 0x03, //not implemented
    enMeca_DbQueryCmd_REOPEN_READONLY   = 0x04  //not implemented
} tenMeca_DbQueryCmd;

typedef enum {
    enMeca_DbQueryAction_NOP                            = 0x00,
    enMeca_DbQueryAction_SEND_DB_QUERY_ONLY             = 0x01,
    enMeca_DbQueryAction_EXEC_DB_QUERY_AND_RES          = 0x02,
    enMeca_DbQueryAction_EXEC_DB_QUERY_AND_RES_CHANGED  = 0x03
} tenMeca_DbQueryAction;

typedef enum {
    enMeca_DbQueryTriggerEvent_ADDED                    = 0x0001,
    enMeca_DbQueryTriggerEvent_DATA_COMPLETE            = 0x1001,
    enMeca_DbQueryTriggerEvent_AUD_SVC_CHANGED          = 0x1002,
    enMeca_DbQueryTriggerEvent_AUD_COMP_SVC_CHANGED     = 0x1003,
    enMeca_DbQueryTriggerEvent_ENS_QUALITY_WORD         = 0x2104
} tenMeca_DbQueryTriggerEvent;


typedef enum {
    enMeca_DbQueryTriggerCmdRsp_NOP             	= 0x00,
    enMeca_DbQueryTriggerCmdRsp_ADD          		= 0x01,
    enMeca_DbQueryTriggerCmdRsp_DELETE           	= 0x02,
    enMeca_DbQueryTriggerCmdRsp_DELETE_ALL 			= 0x03,
    enMeca_DbQueryTriggerCmdRsp_UPDATE  			= 0x04,
    enMeca_DbQueryTriggerCmdRsp_TRIGGER  			= 0x05,
	enMeca_DbQueryTriggerCmdRsp_ONEVENT  			= 0x10
} tenMeca_DbQueryTriggerCmdRsp;

typedef enum {
    enMeca_DbQueryTriggerAction_NOP             								= 0x00,
    enMeca_DbQueryTriggerAction_SEND_R_DB_QUERY_TRIGGER_ONLY            		= 0x01,
    enMeca_DbQueryTriggerAction_EXEC_DB_QUERY_AND_SEND_RESULT             		= 0x02,
    enMeca_DbQueryTriggerAction_EXEC_DB_QUERY_AND_SEND_RESULT_WHEN_CHANGED   	= 0x03
} tenMeca_DbQueryTriggerAction;

typedef enum {
    enMeca_DbQueryTriggerList_TR__ADDED             			     		= 0x0001,
    enMeca_DbQueryTriggerList_TR__SERVICE_DATA_COMPLETE            		    = 0x0110,
    enMeca_DbQueryTriggerList_TR__CURR_RDM_ENS__CHANGED            		    = 0x0210,
    enMeca_DbQueryTriggerList_TR__CURR_RDM_AUDIO_SERV__CHANGED   			= 0x0310,
    enMeca_DbQueryTriggerList_TR__CURR_RDM_AUDIO_COMPONENT__CHANGED 		= 0x0410,
    enMeca_DbQueryTriggerList_TR__DAB_ENSEMBLE__QUALITY_WORD  				= 0x0421
} tenMeca_DbQueryTriggerList;

typedef enum {
    enMeca_DbQueryResultProtocol_CSV_TABLE     = 0x10, //not implemented
    enMeca_DbQueryResultProtocol_CSV_ROW       = 0x20, //not implemented
    enMeca_DbQueryResultProtocol_CSV_ROW_DONE  = 0x21, //not implemented
    enMeca_DbQueryResultProtocol_SDXF_TABLE    = 0x30,
    enMeca_DbQueryResultProtocol_SDXF_ROW      = 0x40, //not implemented
    enMeca_DbQueryResultProtocol_SDXF_ROW_DONE = 0x41, //not implemented
    enMeca_DbQueryResultProtocol_XML_TABLE     = 0x50, //not implemented
    enMeca_DbQueryResultProtocol_XML_ROW       = 0x60, //not implemented
    enMeca_DbQueryResultProtocol_XML_ROW_DONE  = 0x61  //not implemented
} tenMeca_DbQueryResultProtocol;

typedef enum {
    enMeca_DbQueryParameterProtocol_INT32_ARRAY  = 0x10,
    enMeca_DbQueryParameterProtocol_SDXF         = 0x20 //not implemented
} tenMeca_DbQueryParameterProtocol;

// _DAB_C_DB_QUERY
struct trMeca_CDbQuery:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

	trMeca_CDbQuery():
	u8QueryTag(0),
		u8QueryStateId(0),
		enDbQueryCmd(enMeca_DbQueryCmd_FREE),
		u8ResultProto(0),
		u8ResultOption(0),
		u8ParamProto(0)
	{
	}
    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        tU16 u16SqlStatementLenght = (tU16)lu8SqlStatement.size();
        tU8  u8SqlParameterLenght  = (tU8)lu8SqlParameter.size();

        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_QUERY;
        pRawOutput->enOpType = enOpType_SETGET;
        pRawOutput->u16DataLen = (tU16)(12+u16SqlStatementLenght+u8SqlParameterLenght);
        pRawOutput->au8Data[0] = (tU8)enDbQueryCmd;
        pRawOutput->au8Data[1]= u8QueryTag;
        pRawOutput->au8Data[2]= u8QueryStateId;
        pRawOutput->au8Data[3]= 0; // rfu
        DABDRV_SET_U16(&pRawOutput->au8Data[4], u16SqlStatementLenght);
        for (tU16 i=0;i<u16SqlStatementLenght;i++) {
            pRawOutput->au8Data[6+i] = lu8SqlStatement[i];
        }
       pRawOutput->au8Data[6+u16SqlStatementLenght]= (u8ResultProto?u8ResultProto:(tU8)enMeca_DbQueryResultProtocol_SDXF_TABLE);
		pRawOutput->au8Data[7+u16SqlStatementLenght]=(u8ResultOption?u8ResultOption: 0x03);
		pRawOutput->au8Data[8+u16SqlStatementLenght]= (u8ParamProto?u8ParamProto:(tU8)enMeca_DbQueryParameterProtocol_INT32_ARRAY);
        if ( u8SqlParameterLenght == 0 ) {
            pRawOutput->au8Data[9+u16SqlStatementLenght]= 0;//u8SqlParameterLenght;
            pRawOutput->au8Data[10+u16SqlStatementLenght]= 0;
            pRawOutput->au8Data[11+u16SqlStatementLenght]= 0;
        }
        else {
            pRawOutput->au8Data[9+u16SqlStatementLenght]= u8SqlParameterLenght;
            for (tU8 j=0;j<u8SqlParameterLenght;j++) {
                pRawOutput->au8Data[10+u16SqlStatementLenght+j] = lu8SqlParameter[j];
            }
        }
    }
	tVoid vUpdate(tU8* pu8Data){	
		enDbQueryCmd =(tenMeca_DbQueryCmd)pu8Data[0];
		u8QueryTag=pu8Data[1];
		u8QueryStateId=pu8Data[2];
		tU16 u16SqlStatementLength;
		u16SqlStatementLength=DABDRV_GET_U16(&pu8Data[4]);
	    for (tU16 i=0;i<u16SqlStatementLength;i++) {
            lu8SqlStatement.push_back(pu8Data[6+i]);
        }
		u8ResultProto=pu8Data[6+u16SqlStatementLength];
		u8ResultOption=pu8Data[7+u16SqlStatementLength];
		u8ParamProto=pu8Data[8+u16SqlStatementLength];	  
	}

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbQuery u8QueryTag=%d u8QueryStateId=%d u16SqlStatementLenght=%d",
                            u8QueryTag,
                            u8QueryStateId,
							(tU16)(lu8SqlStatement.size())));	

        ETG_TRACE_USR1(("  trMeca_CDbQuery enDbQueryCmd=%d",
        		ETG_CENUM(tenMeca_DbQueryCmd,enDbQueryCmd)));
    };
    tU8 u8QueryTag;
    tU8 u8QueryStateId;
    tenMeca_DbQueryCmd enDbQueryCmd;
    vector<tU8>lu8SqlStatement;
    vector<tU8>lu8SqlParameter;
	tU8 u8ResultProto;
	tU8 u8ResultOption;
	tU8 u8ParamProto;
};

// _DAB_C_DB_QUERY_TRIGGER
struct trMeca_CDbQueryTrigger:
	public  trMsgMecaOut {
	DAB_DISPATCH_IMPL

	virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {

		pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
		pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_QUERY_TRIGGER;
		pRawOutput->enOpType = enOpType_SETGET;
		pRawOutput->u16DataLen = 16;
		pRawOutput->au8Data[0] = (tU8)enMeca_DbQueryTriggerCmdRsp_ADD;
		pRawOutput->au8Data[1]= u8QueryTag;
		pRawOutput->au8Data[2]= u8Id;
		pRawOutput->au8Data[3]= u8QueryStateId;
		pRawOutput->au8Data[4]= 0; //b_condition_stmt_id - rfu
		pRawOutput->au8Data[5]= (tU8)enMeca_DbQueryAction;
		pRawOutput->au8Data[6]= 0; // rfu
		pRawOutput->au8Data[7]= 0; // rfu
		DABDRV_SET_U16(&pRawOutput->au8Data[8], u16TriggerListSize); //Trigger List size
		OSAL_pvMemoryCopy(&pRawOutput->au8Data[10],au8TriggerList,6);
	}

	virtual tVoid vTrace() const {
		ETG_TRACE_USR1(("  trMeca_CDbQueryTrigger u8QueryTag=%d u8QueryStateId=%d",
							u8QueryTag,
							u8QueryStateId));
	};
	tU8 u8QueryTag;
	tU8 u8Id;
	tU8 u8QueryStateId;
	tU8 au8TriggerList[20];
	tU16 u16TriggerListSize;
	tenMeca_DbQueryTriggerAction enMeca_DbQueryAction;
};

// _DAB_R_DB_QUERY_TRIGGER
struct trMeca_RDbQueryTrigger:
	public trMsgMecaInput {
	DAB_DISPATCH_IMPL

	virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
	   /* if (rMostHdr._u16MecaLen < 8) {
			return FALSE;
		}*/
		_rMostHdr=rMostHdr;

		enQueryTriggerResponse=(tenMeca_DbQueryTriggerCmdRsp)pu8Data[0];
		u8QueryTag = pu8Data[1];
		u8Id = pu8Data[2];
		u8QueryStateId = pu8Data[3];
		//rfu = pu8Data[4];
		enTriggerAction = (tenMeca_DbQueryTriggerAction)pu8Data[5];
		//rfu = pu8Data[6]; rfu = pu8Data[7];
		u16TriggerListSize = DABDRV_GET_U16(&pu8Data[8]);

		for (tU16 i=0;i<u16TriggerListSize;i++) {
			tU16 u16Data = DABDRV_GET_U16(&pu8Data[10+i]);
			lu16TriggerList.push_back(u16Data);

			ETG_TRACE_USR1(("  trMeca_RDbQueryTrigger:vTrace() lu16TriggerList[%d]=%d",i,
							ETG_CENUM(tenMeca_DbQueryTriggerList,lu16TriggerList[i])));
		}
		return TRUE;
	};
	virtual tVoid vTrace() const {
		ETG_TRACE_USR1(("  trMeca_RDbQueryTrigger:vTrace() enQueryTriggerResponse=%d u8QueryTag=%x u8Id=%d u8QueryStateId=%d enTriggerAction=%d u16TriggerListSize=%d",
				ETG_CENUM(tenMeca_DbQueryTriggerCmdRsp,enQueryTriggerResponse),
				u8QueryTag,
				u8Id,
				u8QueryStateId,
				ETG_CENUM(tenMeca_DbQueryTriggerAction,enTriggerAction),
				u16TriggerListSize));
	};

	tenMeca_DbQueryTriggerCmdRsp enQueryTriggerResponse;
	tU8 u8QueryTag;
	tU8 u8Id;
	tU8 u8QueryStateId;
	tenMeca_DbQueryTriggerAction enTriggerAction;
	tU16 u16TriggerListSize;
	vector<tU16>lu16TriggerList;
};

typedef enum {
    enMeca_DbQueryResponse_ERROR              = 0x00,
    enMeca_DbQueryResponse_OK                 = 0x01
} tenMeca_DbQueryResponse;

typedef enum {
	enMeca_TagID_LinkingInfo						=0x01,
	enMeca_TagID_Initial_Autostore					=0x02,
	enMeca_TagID_Global_List						=0x04,
	enMeca_TagID_Ensemble_List						=0x05,
	enMeca_TagID_Ensemble_Prog_Serv_Components_List =0x06,
	enMeca_TagID_Alt_Freq_Info						=0x07,
	enMeca_QUERY_TAG_TESTMODE_DB_REQUEST			=0x0a,
	enMeca_QUERY_TAG_COUNT_OF_TPEG_SERVICES			=0x0b,
	enMeca_QUERY_TAG_GET_TPEG_CHN_LIST				=0x0c,
	enMeca_QUERY_TAG_AVAIL_LIST 					=0x0d,
	enMeca_QUERY_TAG_COUNT_OF_TMC_SERVICES			=0x0e,
	enMeca_TagID_All_Preset_Info					=0x10,
	enMeca_TagID_Preset_Info_Start					=0x11,
	//enMeca_TagID_Preset_Info_END = (enMeca_TagID_Preset_Info_Start+DAB_NUM_PRESETS), 

	/* 0x70 - (0x70+0x7f) reserved for FC_DAB service list update
	*/
	enMeca_TagID_Global_List_START					=0x70,
	enMeca_TagID_Global_List_END					=(0x70+0x7F)

	/* 0xF0 - 0xFF reserved for FC_DATASERVICE DSM library
	*/
} tenMeca_DbQueryTagID;
typedef enum {
	enMeca_StateID_Ensemble_List						=0x01,
	enMeca_StateID_Global_List							=0x02,
	enMeca_DB_QUERY_STMT_ID_RECEIVEABLE_TPEG_CHN_LIST	=0x08
} tenMeca_DbQueryStateID;

// DAB_*_R_DB_QUERY
struct trMeca_RDbQuery:
    public trMsgMecaInput {
    DAB_DISPATCH_IMPL
trMeca_RDbQuery():
	enQueryResponse(enMeca_DbQueryResponse_OK),
    u8QueryTag(0),
    u8QueryStateId(0),
    u16QueryNumTables(0),
    u16QueryNumRows(0),
    u16QueryNumColumns(0),
    enQueryResultProtocol(enMeca_DbQueryResultProtocol_SDXF_TABLE),
    b8QueryResultOption(0),
    u32UserDataLenght(0),
    lu8UserData()
	{
	};
    
    virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
       /* if (rMostHdr._u16MecaLen < 8) {
            return FALSE;
        }*/
        _rMostHdr=rMostHdr;
     /*   if ( enMeca_DbQueryResponse_OK != (tenMeca_DbQueryResponse)pu8Data[0]) {
            return FALSE;
        }*/
	    enQueryResponse=(tenMeca_DbQueryResponse)pu8Data[0];
        u8QueryTag = pu8Data[1];
        u8QueryStateId = pu8Data[2];
        //rfu = pu8Data[3];
        u16QueryNumTables = DABDRV_GET_U16(&pu8Data[4]);
        u16QueryNumRows = DABDRV_GET_U16(&pu8Data[6]);
        u16QueryNumColumns = DABDRV_GET_U16(&pu8Data[8]);
        enQueryResultProtocol = (tenMeca_DbQueryResultProtocol)pu8Data[10];
        b8QueryResultOption = pu8Data[11];
        u32UserDataLenght = DABDRV_GET_U32(&pu8Data[12]);
        for (tU32 i=0;i<u32UserDataLenght;i++) {
            tU8 u8Data = pu8Data[16+i];
            lu8UserData.push_back(u8Data);
        }
        return TRUE;
    };
    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_RDbQuery:vTrace()QueryTag=%d enQueryResponse=%x NumRows=%d NumColumns=%d size=%d",
			u8QueryTag,
			ETG_CENUM(tenMeca_DbQueryResponse,enQueryResponse),
			u16QueryNumRows,
			u16QueryNumColumns,
			(tU16)(lu8UserData.size())));
    };

    tenMeca_DbQueryResponse enQueryResponse;
    tU8 u8QueryTag;
    tU8 u8QueryStateId;
    tU16 u16QueryNumTables;
    tU16 u16QueryNumRows;
    tU16 u16QueryNumColumns;
    tenMeca_DbQueryResultProtocol enQueryResultProtocol;
    tU8 b8QueryResultOption;
    tU32 u32UserDataLenght;
    vector<tU8>lu8UserData;
};

typedef enum {
	enMeca_PtyCommand_set_filter=0x01,
	enMeca_PtyCommand_add_filter=0x02,
	enMeca_PtyCommand_delete_filter=0x03,
	enMeca_PtyCommand_delete_all_filter=0x04
} tenMeca_PtyCommand;


 // _DAB_C_PTYM_FILTER
    struct trMeca_CPTYMSetFilter:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_PTYM_FILTER;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 5;
            pRawOutput->au8Data[0]=(tU8)enPTYCommand;
            pRawOutput->au8Data[1]=u8IntTableId;
            pRawOutput->au8Data[2]=u8PtyType;
			pRawOutput->au8Data[3]=u8PtyLanguage;
			pRawOutput->au8Data[4]=u8PtyCode;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CPTYMSetFilter:enPTYCommand=0x%x u8PtyType=%d u8PtyCode=%d", 
                            ETG_CENUM(tenMeca_PtyCommand, enPTYCommand),
                            u8PtyType,
							u8PtyCode));
        };
        tenMeca_PtyCommand enPTYCommand;
        tU8 u8IntTableId;
		tU8 u8PtyType;
		tU8 u8PtyLanguage;
		tU8 u8PtyCode;
    };

	struct trMeca_RPTYMSetFilter:
		public trMsgMecaInput {
			DAB_DISPATCH_IMPL

				virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
					if (rMostHdr._u16MecaLen != 5) {
						return FALSE;
					}
					_rMostHdr=rMostHdr;
				
					enPTYCommand=(tenMeca_PtyCommand)pu8Data[0];
					u8IntTableId=pu8Data[1];
					u8PtyType=pu8Data[2];
					u8PtyLanguage=pu8Data[3];
					u8PtyCode=pu8Data[4];
					return TRUE;
			};
			virtual tVoid vTrace() const {
				ETG_TRACE_USR1(("  trMeca_RPTYMSetFilter:vTrace()"));
			};
	
			tenMeca_PtyCommand enPTYCommand;
			tU8 u8IntTableId;
			tU8 u8PtyType;
			tU8 u8PtyLanguage;
			tU8 u8PtyCode;
	};

typedef enum {
	enMeca_DbConfig_ValueId_delete_All=0x07,	
	enMeca_DbConfig_ValueId_delete_presets=0x08
} tenMeca_DbConfig_ValueId;
struct trMeca_CDbConfig:
    public  trMsgMecaOut {
    DAB_DISPATCH_IMPL

    virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
        pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
        pRawOutput->u16MsgType = MECA_DISPATCHER_C_DB_CONFIG;
        pRawOutput->enOpType = enOpType_SETGET;
        pRawOutput->u16DataLen = 4;
        pRawOutput->au8Data[0] = (tU8)u8Cmd;
        pRawOutput->au8Data[1]= (tU8)enValueId;
		pRawOutput->au8Data[2]= 0;//rfu
		pRawOutput->au8Data[3]= 0;//no params
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1(("  trMeca_CDbConfig:u8Cmd=%d u8ValueId=0x%x ",
                        u8Cmd,
						ETG_CENUM(tenMeca_RdmScanCommand, enValueId)));
    };
	tU8 u8Cmd;
	tenMeca_DbConfig_ValueId enValueId;
};
struct trMeca_RDbConfig:
	public trMsgMecaInput {
		DAB_DISPATCH_IMPL
			virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
				_rMostHdr=rMostHdr;
				if (rMostHdr._u16MecaLen != 4) {
					return FALSE;
				}
				u8Cmd = pu8Data[0];            
				enValueId = (tenMeca_DbConfig_ValueId)pu8Data[1]; 
				return TRUE;
		}
		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMeca_RDbConfig: enCmd=%x enValueId=%x",
				u8Cmd,
				ETG_CENUM(tenMeca_DbConfig_ValueId, enValueId)));
		};
		tU8 u8Cmd;
	tenMeca_DbConfig_ValueId enValueId;
};
} // namespace DAB

#endif
