/************************************************************************
 * FILE:     dabdrv_mute.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  header of dabdrv_mute
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/
 
#include "dabdrv_mute.hpp"
#include "dabdrv_main.hpp"
#include "dabdrv_anno.hpp"
#include "dabdrv_chnInfo.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_DRV_MUTE 
#include "trcGenProj/Header/dabdrv_mute.cpp.trc.h"
#endif

namespace DAB {

    struct trMsgBlockAnnoMuteTimeOut:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgBlockAnnoMuteTimeOut"));
        };
    };
}

using namespace DAB;
dabdrv_mute::dabdrv_mute() {
    _bChnStable = FALSE;
    _bMuteRequested = FALSE;
    _bDabModuleMuted = TRUE;

    _bTaSameSubChannel=FALSE;
    _enAnnoSelectState=enAnnoSelectState_IDLE;
    _enSourceState = DAB_enSourceState_BG;
    _enMuteStateDab = enMuteState_muted;
    _enMuteStateDabTa = enMuteState_muted;
    _rRdmSid.vInvalidate();
    _rChnInfoSid.vInvalidate();
    _bRemoteCtrlActive=FALSE;
    _bDabAudioMuted = FALSE;
    _bDabTaMuted = FALSE;

    // commands from dabdrv_main
    vSubscribe<trMsgDrvStopComponent>();

    vSubscribe<trMsgDrvCmdChnSelectState>();
    vSubscribe<trMsgDrvIndActivityChangesChannel>();
    vSubscribe<trMsgDrvIndRemoteCtrl>();
    vSubscribe<trMsgDrvCmdRequestMute>();
    vSubscribe<trMsgDrvCmdUpdateMute>();
    vSubscribe<DAB_trAnnoStatusProperty>();
    vSubscribe<trMsgDrvCmdSourceState>();
    vSubscribe<trMsgDrvCmdSourceActivity>();
    vSubscribe<trMsgDrvCmdSetChnInfoPsid>();
    vSubscribe<trMsgDrvSetMuteStatus>();
    vSubscribe<trMsgDrvCmdMuteRequest>();
#ifdef MUTE_ACTIVE
    vSubscribe<trMsgDrvCmdTunerOpMuteReq>();
#endif
    vSubscribe<trMeca_RRdmGetRdmInfo>();
}

tVoid dabdrv_mute::vInit() {
    ETG_TRACE_USR1(("dabdrv_mute::vInit"));
    _oBlockAnnoMuteTimer.vInit(instance(),trMsgBlockAnnoMuteTimeOut());    
}

tVoid dabdrv_mute::vDeInit() {
    ETG_TRACE_USR1(("dabdrv_mute::vDeInit"));
    _oBlockAnnoMuteTimer.vDeInit();    
}

tVoid dabdrv_mute::vTraceState() const {
    ETG_TRACE_USR1(("  dabdrv_mute STATE: _enMuteStateDab=%d _enMuteStateDabTa=%d _bChnStable=%d AnnoTimerRunning=%d "
                    "_rChnInfoSid=0x%08x _rRdmSid=0x%08x _bMuteRequested=%d _bDabModuleMuted=%d _bRemoteCtrlActive=%d",
                    ETG_CENUM(tenMuteState, _enMuteStateDab),
                    ETG_CENUM(tenMuteState, _enMuteStateDabTa),
                    _bChnStable,
                    _oBlockAnnoMuteTimer.bIsRunning(),
                    _rChnInfoSid.u32GetSID(),
                    _rRdmSid.u32GetSID(),
                    _bMuteRequested,
                    _bDabModuleMuted,
                    _bRemoteCtrlActive));
}

// we get informed that dab-module is down
tVoid dabdrv_mute::vProcess(trMsgDrvStopComponent* StopComponent) {
    (tVoid)StopComponent;
    if (_bChnStable==TRUE ||
        _bDabModuleMuted == FALSE ||
        _oBlockAnnoMuteTimer.bIsRunning()) {
        _bTaSameSubChannel=FALSE;
        _oBlockAnnoMuteTimer.vStop();
        _bChnStable = FALSE;
        _bDabModuleMuted = TRUE;
        _rRdmSid.vInvalidate();
        vUpdate();
    }
}


// we get informed if a stable channel is selected by dab-module
tVoid dabdrv_mute::vProcess(trMsgDrvCmdChnSelectState* poChnSelectState) {
    if (_bChnStable!=poChnSelectState->bStable()) {
        _bChnStable=poChnSelectState->bStable();
        vUpdate();
    }
}

// an activity changed the channel, invalidate radio-text
tVoid dabdrv_mute::vProcess(trMsgDrvIndActivityChangesChannel* poActivityChangesChannel ) {
    (tVoid)poActivityChangesChannel;

    if (_bChnStable) {
        _bChnStable=FALSE;
        vUpdate();            
    }
}


tVoid dabdrv_mute::vProcess(trMsgDrvIndRemoteCtrl* poIndRemoteCtrl) {
    _bRemoteCtrlActive=poIndRemoteCtrl->bOn;
    vUpdate();
}

// someone wants us to be muted
tVoid dabdrv_mute::vProcess(trMsgDrvCmdRequestMute* poRequestMute) {
    (tVoid)poRequestMute;

    // inform e.g. channelSelect that we are muted now.
    trMsgDrvRspMuteDone oMsgMuteDone;
    DAB_vCallMsg(oMsgMuteDone);
}

tVoid dabdrv_mute::vProcess(trMsgDrvCmdUpdateMute* poUpdateMute) {
    (tVoid)poUpdateMute;
    vUpdate();
}

tVoid dabdrv_mute::vProcess(DAB_trAnnoStatusProperty* poAnnoStatus) {
    _enAnnoSelectState=poAnnoStatus->enAnnoSelectState;
    if (_enAnnoSelectState != enAnnoSelectState_PENDING && _enSourceState==DAB_enSourceState_FG) {
        _bTaSameSubChannel=(poAnnoStatus->enActiveAnnoSource == enAnnoSource_CURRENT_SUBCHANNEL);
    }
    vUpdate();
}


tVoid dabdrv_mute::vProcess(trMsgBlockAnnoMuteTimeOut* poBlockAnnoMuteTimeOut) {
    (tVoid)poBlockAnnoMuteTimeOut;
    vUpdate();
}


tVoid dabdrv_mute::vProcess(trMsgDrvCmdSourceState* poSourceState) {
    _enSourceState=poSourceState->enSourceState;
    if (_enSourceState != DAB_enSourceState_FG && _enSourceState != DAB_enSourceState_FG_TA) {
        _oBlockAnnoMuteTimer.vStop();
        _bTaSameSubChannel= FALSE;
    }
    else if (_enSourceState == DAB_enSourceState_FG && _bTaSameSubChannel) {
        _oBlockAnnoMuteTimer.vStart(DAB_MUTE_BLOCK_ANNO_MUTE_TIMER_MS);
        _bTaSameSubChannel= FALSE;
    }
    vUpdate();
}

tVoid dabdrv_mute::vProcess(trMsgDrvCmdSourceActivity* /*poSourceActivity*/) {
    //_rSourceActivity= *poSourceActivity;
    vUpdate();
}

tVoid dabdrv_mute::vProcess(trMsgDrvCmdSetChnInfoPsid* prChnInfoPsid) {
    _rChnInfoSid=prChnInfoPsid->rProgService;
    vUpdate();
}

tVoid dabdrv_mute::vProcess(trMeca_RRdmGetRdmInfo* poRdmInfo) {
    trMeca_RdmStatus const &rRdmStatus=poRdmInfo->rRdmStatus;
    tBool bDabModuleMuted = (rRdmStatus.u8Status & (tU8)enRdmStatusBits_MUTE) ? TRUE:FALSE;
    if (bDabModuleMuted != _bDabModuleMuted || _rRdmSid!=poRdmInfo->rProgrammeService) {
        _rRdmSid=poRdmInfo->rProgrammeService;
        _bDabModuleMuted=bDabModuleMuted;
        vUpdate();
    }
}


tVoid dabdrv_mute::vProcess(trMsgDrvSetMuteStatus* prSetMuteStatus) {

    ETG_TRACE_USR1(("  dabdrv_mute::vProcess(trMsgDrvSetMuteStatus)"));
    
    tBool bIsMuted = (prSetMuteStatus->enMuteState == (tenARLMuteState)enMuteState_muted) ? TRUE:FALSE;

    if ( prSetMuteStatus->enDabSource == enDabSource_Dab) {
        _bDabAudioMuted = bIsMuted;
        if(bIsMuted){
            if (_bMuteRequested) {
                // inform e.g. channelSelect that we are muted now.
                _bMuteRequested = FALSE;
                trMsgDrvRspMuteDone oMsgMuteDone;
                DAB_vCallMsg(oMsgMuteDone);
            }
        }
    } else {
        _bDabTaMuted = bIsMuted;
    }
}

// check if fc_audio may demute the dab-source
tBool dabdrv_mute::bIsAudioDemuteOK() const{
    dabdrv_main* poDrvMain= dabdrv_main::instance();
    ETG_TRACE_USR1(("  dabdrv_mute bIsAudioDemuteOK: bIsAdrUp=%d bActivityChangedChannel=%d _bChnStable=%d "
                    "_bMuteRequested=%d _bDabModuleMuted=%d",
                    poDrvMain->bIsAdrUp(),
                    dabdrv_main::instance()->bActivityChangedChannel(),
                    _bChnStable,
                    _bMuteRequested,
                    _bDabModuleMuted));
    if (!poDrvMain->bIsAdrUp()) return FALSE;
    if (_bRemoteCtrlActive) return TRUE;
    if (dabdrv_main::instance()->bActivityChangedChannel()) return FALSE;
    if (DAB_enSourceState_FG != _enSourceState &&
        DAB_enSourceState_FG_TA != _enSourceState) return FALSE;
    //if (!_bChnStable) return FALSE;
    if (_rRdmSid != _rChnInfoSid) return FALSE;
    if (!_rChnInfoSid.bIsValid()) return FALSE;
    if (_bMuteRequested) return FALSE;
    //if (_bDabModuleMuted) return FALSE;
    return TRUE;
}

tBool dabdrv_mute::bIsAudioDemuteDabOK() const{
    if (_bRemoteCtrlActive) {
        return TRUE;
    }
    // dab has to be muted if a TA is running on a different sub-channel
    if (!_bTaSameSubChannel && (_enAnnoSelectState==enAnnoSelectState_SELECTING || 
                                        _enAnnoSelectState==enAnnoSelectState_SELECTED)) return FALSE;
    return TRUE;
};

tBool dabdrv_mute::bIsAudioDemuteDabTaOK() const{
    // dabTA will be demuted instantly when a dabTA is available on the same subchannel
    if (_bTaSameSubChannel && (_enAnnoSelectState==enAnnoSelectState_PENDING ||
                               _enAnnoSelectState==enAnnoSelectState_SELECTING || 
                               _enAnnoSelectState==enAnnoSelectState_SELECTED)) return TRUE;
    
    // dabTA is played
    if (DAB_enSourceState_FG_TA == _enSourceState &&
        _enAnnoSelectState==enAnnoSelectState_SELECTED)return TRUE;
    
    // dabTA on same subchannel has ended but we let it continue some time
    if (_oBlockAnnoMuteTimer.bIsRunning()) return TRUE;

    return FALSE;
};


// check if the dab-source is muted in fc_audio
tBool dabdrv_mute::bIsDabAudioMuted() const{
    //return _rSourceActivity.bTA || _rSourceActivity.enSourceActivity == enSourceActivity_Inactive;
    return _bDabAudioMuted;
}

tBool dabdrv_mute::bIsDabTaMuted() const{
    //return !_rSourceActivity.bTA || _rSourceActivity.enSourceActivity == enSourceActivity_Inactive;
    return _bDabTaMuted;
}


/* vUpdate(): check current values of:
   - chn-stable: if !chn-stable: allways mute
   - sourceState(dabdrv_main): demute only in FG, else alway mut
   - sourceActivity(dabdrv_main): before chnSelection sourceActivity must be inactive
   - LearnMode(dabdrv_main): if learn=on: always mute
   
   Output:
   - Mute Flag in fc_dabtuner_service
   - trigger pending channel-selection
   
   states:
   - muted
   - demuting
   - demuted
   - muting
*/
tVoid dabdrv_mute::vUpdate(tBool bDemuteOk, tBool bAudioMuted, tenMuteState *penMuteStatus) const{
    tenMuteState enNewMuteState=*penMuteStatus;
    if (DAB_enSourceState_FG != _enSourceState && DAB_enSourceState_FG_TA != _enSourceState) {
        // if we are not in FG we are always muted
        *penMuteStatus=enMuteState_muted;
    }
    else {
        if (bDemuteOk) {
            if (bAudioMuted) {
                enNewMuteState=enMuteState_demuting;
            }
            else {
                enNewMuteState=enMuteState_demuted;
            }
        } 
        else {
            if (bAudioMuted) {
                enNewMuteState=enMuteState_muted;
            }
            else {
                enNewMuteState=enMuteState_muting;
            }

        }
    }
    ETG_TRACE_USR4(("dabdrv_mute vUpdate:enMuteState: %d-->%d",
                    ETG_CENUM(tenMuteState, *penMuteStatus),
                    ETG_CENUM(tenMuteState, enNewMuteState)));
    *penMuteStatus=enNewMuteState;
        

}

void dabdrv_mute::vUpdate() {
    ETG_TRACE_USR1(("  dabdrv_mute vUpdate: START _enMuteStateDab=%d _enMuteStateDabTa=%d _bMuteRequested=%d "
                    "_bTaSameSubChannel=%d _enAnnoSelectState=%d",
                    ETG_CENUM(tenMuteState, _enMuteStateDab),
                    ETG_CENUM(tenMuteState, _enMuteStateDabTa),
                    _bMuteRequested,
                    _bTaSameSubChannel,
                    ETG_CENUM(tenAnnoSelectState, _enAnnoSelectState)));

/*
    tBool bDemuteOk=bIsAudioDemuteOK();
    tBool bDemuteDabOk=bDemuteOk &&  bIsAudioDemuteDabOK();
    tBool bAudioMutedDab=bIsDabAudioMuted();
    ETG_TRACE_USR4(("  dabdrv_mute vUpdate: bDemuteDabOk=%d bAudioMutedDab=%d bDemuteOk=%d ",
                    bAudioMutedDab,
                    bDemuteDabOk,
                    bDemuteOk));
    vUpdate(bDemuteDabOk, bAudioMutedDab, &_enMuteStateDab);

    // indicate our DAB AUDIO mute-whish to tunermaster
    DAB_trMuteRequestProperty oPropVal=dabdrv_properties::instance()->oMuteRequestProperty.oGet();
    oPropVal.bMuteRequest=(_enMuteStateDab==enMuteState_muting || _enMuteStateDab==enMuteState_muted);
    oPropVal.bTA = FALSE;
    dabdrv_properties::instance()->oMuteRequestProperty.vSet(oPropVal);

    tBool bDemuteDabTaOk=bDemuteOk &&  bIsAudioDemuteDabTaOK();
    tBool bAudioMutedDabTa=bIsDabTaMuted();
    ETG_TRACE_USR4(("  dabdrv_mute vUpdate: bDemuteDabTaOk=%d bAudioMutedDabTa=%d",
                    bDemuteDabTaOk,
                    bAudioMutedDabTa));
    vUpdate(bDemuteDabTaOk, bAudioMutedDabTa, &_enMuteStateDabTa);

    // indicate our TA mute-whish to tunermaster
    DAB_trMuteRequestProperty oPropValTa=dabdrv_properties::instance()->oMuteRequestProperty.oGet();
    oPropValTa.bMuteRequest=(_enMuteStateDabTa==enMuteState_muting || _enMuteStateDabTa==enMuteState_muted);
    oPropValTa.bTA = TRUE;
    dabdrv_properties::instance()->oMuteRequestProperty.vSet(oPropValTa);

    // check if we have to answer a mute-request
    if (_bMuteRequested
#ifndef DAB_ENABLE_DEMUTE_ALLWAYS
        && ((_enSourceState==DAB_enSourceState_FG_TA && _enMuteStateDabTa==enMuteState_muted) || (_enMuteStateDab==enMuteState_muted) || (!dabdrv_main::instance()->bIsAppStateNormal()))
#endif
) {
        // inform e.g. channelSelect that we are muted now.
        _bMuteRequested = FALSE;
        trMsgDrvRspMuteDone oMsgMuteDone;
        DAB_vCallMsg(oMsgMuteDone);
    }
    ETG_TRACE_USR1(("  dabdrv_mute vUpdate: END _enMuteStateDab=%d _enMuteStateDabTa=%d bDemuteOk=%d bDemuteDabOk=%d bDemuteDabTaOk=%d _bMuteRequested=%d",
                    ETG_CENUM(tenMuteState, _enMuteStateDab),
                    ETG_CENUM(tenMuteState, _enMuteStateDabTa),
                    bDemuteOk,
                    bDemuteDabOk,
                    bDemuteDabTaOk,
                    _bMuteRequested));
    */

}

// someone wants us to be muted
tVoid dabdrv_mute::vProcess(trMsgDrvCmdMuteRequest* poMuteRequest) {
    ETG_TRACE_USR1(("  dabdrv_mute::vProcess(trMsgDrvCmdMuteRequest)"));

    DAB_trMuteRequestProperty oPropVal=dabdrv_properties::instance()->oMuteRequestProperty.oGet();
    DAB_trMuteRequestProperty oPropValTa=dabdrv_properties::instance()->oMuteRequestProperty.oGet();

    switch(poMuteRequest->enMuteRequest) {
        case enMuteRequest_ANNO_START:
            // indicate our DAB AUDIO mute-whish to tunermaster
            oPropVal.bMuteRequest=TRUE;
            oPropVal.bTA = FALSE;
            // indicate our TA mute-whish to tunermaster
            oPropValTa.bMuteRequest=FALSE;
            oPropValTa.bTA = TRUE;
            break;
        case enMuteRequest_ANNO_END:
            // indicate our DAB AUDIO mute-whish to tunermaster
            oPropVal.bMuteRequest=FALSE;
            oPropVal.bTA = FALSE;
            // indicate our DAB TA mute-whish to tunermaster
            oPropValTa.bMuteRequest=TRUE;
            oPropValTa.bTA = TRUE;
            break;
        case enMuteRequest_DAB_AUDIO:
            // indicate our DAB AUDIO mute-whish to tunermaster
            oPropVal.bMuteRequest=TRUE;
            oPropVal.bTA = FALSE;
            break;
        case enMuteRequest_NONE:
        default:
            break;
    }

    dabdrv_properties::instance()->oMuteRequestProperty.vSet(oPropVal);
    dabdrv_properties::instance()->oMuteRequestProperty.vSet(oPropValTa);

}

#ifdef MUTE_ACTIVE
tVoid dabdrv_mute::vProcess(trMsgDrvCmdTunerOpMuteReq* poMuteRequest) {

	DAB_trMuteRequestProperty oPropVal=dabdrv_properties::instance()->oMuteRequestProperty.oGet();

	if(poMuteRequest->enRequiredMuteState==enRequiredMuteState_Mute){
		oPropVal.bMuteRequest=TRUE;
	}
	else{
		oPropVal.bMuteRequest=FALSE;
	}

	oPropVal.u8Attenuation=0;

	switch(poMuteRequest->enMuteOperation)
	{
		case enMuteOperation_PresetLoad:
		case enMuteOperation_LoadFromList:
			oPropVal.bMute_Interrupt = TRUE;
			break;

		case enMuteOperation_EnsembleSeek:
		case enMuteOperation_ServiceSeek:
		case enMuteOperation_ManualTune:
		default:
			oPropVal.bMute_Interrupt = FALSE;
			break;
	}

	ETG_TRACE_USR4(("dabdrv_mute trMsgDrvCmdTunerOpMuteReq: bMuteRequest %d u8Attenuation %d bMute_Interrupt %d",
						 oPropVal.bMuteRequest,
						 oPropVal.u8Attenuation,
						 oPropVal.bMute_Interrupt));

	 dabdrv_properties::instance()->oMuteRequestProperty.vSet(oPropVal);
}
#endif
