/************************************************************************
 * FILE:       dabdrv_compList.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabdrv_compList
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#include "dabdrv_main.hpp"
#include "dabdrv_mecaIf.h"
#include "dabdrv_presets.hpp"
#include "dabdrv_compList.hpp"
#include "dabdrv_compInfo.hpp"
#include "dabdrv_chnInfo.hpp"
#include "dabdrv_presets.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_DRV_COMPLST 
#include "trcGenProj/Header/dabdrv_compList.cpp.trc.h"
#endif

// this class maintains a channellist filled with data from meca_db

/* interface for channel-selection:
   1.) get channel-information by intId, PSID, label
   2.) get newChannel(dir, steps, 
   
   todo: check for uncheched map-dereferencing that might create invalid entries, used DAB_IF_FIND_(MAP)
*/

using namespace DAB;

namespace DAB {

    // Timer-Event when collection of information for compList has to be finished
    struct trMsgCompListUpdateTimer:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgCompListUpdateTimer"));
        };
    };

    struct trMsgCompList1sTick:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgCompList1sTick"));
        };
    };
}

dabdrv_compList::dabdrv_compList():
    _mapCompInfo(),
    _mapDynSrvByLabel(),
    _mapFrozenIndexBySrv(),
    _vectorFrozenCompList(),
    _bCompListValid(FALSE),
    _u32UpdateId(0),
    _enListCheckState(enListCheckState_Done),
    _enFrozenCompListState(enFrozenCompListState_Closed),
    _u16LastUpdatedScidi(0),
    _bWaitCompList(FALSE),
    _u8NumComponents(0)
{
    vClearCompList();
    vClearFrozenCompList();
}


tVoid dabdrv_compList::vTraceCompInfo(trChnListElem const &rCompInfo) const{
    ETG_TRACE_USR4(("  %04x %17s ", 
                    rCompInfo.rMecaId.u16GetScidi(), 
                    rCompInfo.rLabel.pcGetCString()));
}

tVoid dabdrv_compList::vTraceCompInfo(trCompListServiceInfo const &rCompInfo) const{

    ETG_TRACE_USR4(("  %04x  %d %d %d %d %d %17s ", 
                    rCompInfo.u16Scidi, 
                    rCompInfo.bServiceInfoPresent, 
                    rCompInfo.bFoundInSrvList,
                    rCompInfo.bLabelPresent,
                    rCompInfo.bPrim,
                    (tU8)rCompInfo.enListElemUpdateState,
                    rCompInfo.bLabelPresent ? rCompInfo.rLabel.pcGetCString() : ""));
}

tVoid dabdrv_compList::vTraceAll() const {
    ETG_TRACE_USR4(("dabdrv_compList::vTraceAll:START"));

    ETG_TRACE_USR4(("dabdrv_compList::vTraceAll:_mapCompInfo"));
    ETG_TRACE_USR4(("  Srv  InfoPresent FoundInList labelPresent bPrm updState label"));
    DAB_FOREACH_MAP_CONST(trMecaId, trCompListServiceInfo,iterSrvMap, _mapCompInfo) {
        vTraceCompInfo((*iterSrvMap).second);
    }

    ETG_TRACE_USR4(("dabdrv_compList::vTraceAll:_mapDynSrvByLabel"));
    ETG_TRACE_USR4(("  Label                PSID"));

    DAB_FOREACH_MMAP_CONST(trMecaLabel, trMecaId, iterDynSrv, _mapDynSrvByLabel) {
        ETG_TRACE_USR4(("  %17s %04x ", 
                        iterDynSrv->first.pcGetCString(),
                        iterDynSrv->second.u16GetScidi()));
        
    }

    ETG_TRACE_USR4(("dabdrv_compList::vTraceAll:_vectorFrozenCompList"));
    ETG_TRACE_USR4(("  Label                PSID"));
    ETG_TRACE_USR4(("  Srv  InfoPresent FoundInList Quality labelPresent label EnsList"));
    for (tU32 i=0;i<_vectorFrozenCompList.size();++i) {
        vTraceCompInfo(_vectorFrozenCompList[i]);

    }

    ETG_TRACE_USR4(("dabdrv_compList::vTraceAll:_mapFrozenIndexBySrv (Srv : index)"));

    DAB_FOREACH_MAP_CONST(trMecaId, tU16, iterIndexBySrv, _mapFrozenIndexBySrv) {
        ETG_TRACE_USR4(("0x%x : %d",
                        iterIndexBySrv->first.u16GetScidi(),
                        iterIndexBySrv->second));
    }

    ETG_TRACE_USR4(("dabdrv_compList::vTraceAll:END"));
}

tVoid dabdrv_compList::vInit() {
    // commands from dabdrv_main
    vSubscribe<trMsgDrvStartComponent>();
    vSubscribe<trMsgDrvStopComponent>();

    vSubscribe<trMsgDrvCmdCompListScidiMonitor>();
    vSubscribe<trMsgDrvCmdSourceState>();
    vSubscribe<trMsgDrvCmdSetChnInfoPsid>();
    vSubscribe<trMsgDrvCmdChnSelect>();
    vSubscribe<trMsgDrvIndPreChnSelect>();

    // meca-responses
    vSubscribe<trMeca_RDbServiceGetComponentList>();
    vSubscribe<trMeca_RDbServiceComponentGetInfo>();

    // commands from server
    vSubscribe<trMsgSrvCmdSetCompList>();
    // timer can not be created in constructor
    vClearCompList();
    vClearFrozenCompList();
    vUpdateNumComponents();

    _oCompListUpdateTimer.vInit(instance(),trMsgCompListUpdateTimer());
}

tVoid dabdrv_compList::vDeInit() {
    // timer can not be created in constructor
    _oCompListUpdateTimer.vDeInit();
}

tVoid dabdrv_compList::vTraceState() const {
    ETG_TRACE_USR1(("  dabdrv_compList STATE: _bCompListValid=%d _enListCheckState=%d _enFrozenCompListState=%d _u32UpdateId=%d _u16LastUpdatedScidi=0x%04x _rCurSid=0x%08x _bWaitCompList=%d",
                    _bCompListValid,
                    ETG_CENUM(tenListCheckState, _enListCheckState),
                    ETG_CENUM(tenFrozenCompListState, _enFrozenCompListState),
                    _u32UpdateId,
                    _u16LastUpdatedScidi,
                    _rCurSid.u32GetSID(),
                    _bWaitCompList));
}

tVoid dabdrv_compList::vProcess(trMsgDrvStartComponent*) {
    _oCompListUpdateTimer.vStart(DAB_COMP_LIST_UPDATE_TIMER_MS);
    _rCompMonitor.vInvalidate();
}

tVoid dabdrv_compList::vProcess(trMsgDrvStopComponent*) {
    vClearCompList();
    vClearFrozenCompList();
    vUpdateNumComponents();
    vSendFrozenCompList();
}

tVoid dabdrv_compList::vProcess(trMsgCompList1sTick* ) {
    vRequestNextCompInfo();
}

tVoid dabdrv_compList::vProcess(trMsgDrvCmdCompListScidiMonitor *poSetScidiMonitor) {
    _rCompMonitor.rScidi = poSetScidiMonitor->rMecaId;
    _rCompMonitor.bPresent =rGetCompInfo(_rCompMonitor.rScidi).bValid;
    ETG_TRACE_USR4(("dabdrv_compList::trMsgDrvCmdChnListPsidMonitor:0x%08x bPresent=%d",
                    poSetScidiMonitor->rMecaId.u16GetScidi(), _rCompMonitor.bPresent));
}

tVoid dabdrv_compList::vProcess(trMsgDrvCmdSourceState* /*poSourceState*/) {
    DAB_trChnInfoProperty rChnInfoProperty = dabdrv_properties::instance()->oChnInfoProperty.oGet();
    if (rChnInfoProperty.bFmServiceFollowActive) {
        // close frozen comp-list
        vClearFrozenCompList();
        vSendFrozenCompList();        
    }
}

tVoid dabdrv_compList::vProcess(trMsgDrvCmdSetChnInfoPsid *poSidPsid) {
    (tVoid)poSidPsid;
    ETG_TRACE_COMP(("trMsgDrvCmdSetChnInfoPsid START"));
    vSetPsid(poSidPsid->rProgService);
    ETG_TRACE_COMP(("trMsgDrvCmdSetChnInfoPsid END"));

}

tVoid dabdrv_compList::vSetPsid(trMecaProgrammeService const &rProgSrv) {
    // the sid has changed, we have to invalidate the comp-list and collect the new data
    // todo: check if sid has really changed
    // we can not query the comp-data because the sid is not already tuned
    ETG_TRACE_COMP((" dabdrv_compList::vSetPsid START (0x%08x)",
                    rProgSrv.u32GetSID()));
    
    if (_rCurSid != rProgSrv) {
        _rCurSid = rProgSrv;
        ETG_TRACE_COMP((" dabdrv_compList::vSetPsid: sid changed, clearCompList"));
        vClearCompList(/* enListCheckState_Virgin */);
        if (_rCurSid.bIsValid()) {
            _bWaitCompList=TRUE;
        }
    }
    vClearFrozenCompList();
    vUpdateNumComponents();
	_enFrozenCompListState=enFrozenCompListState_Open;

    if (_rCurSid.bIsValid()) {
        _oCompListUpdateTimer.vStart(DAB_COMP_LIST_UPDATE_TIMER_MS);
    }
    vSendFrozenCompList();
    // just wait for the the componen-list
    ETG_TRACE_COMP((" dabdrv_compList::vSetPsid END"));

} 

tVoid dabdrv_compList::vProcess(trMsgDrvCmdChnSelect *poDrvCmdChnSelect) {
    (tVoid)poDrvCmdChnSelect;
    _rCompMonitor.vInvalidate();
}

tVoid dabdrv_compList::vProcess(trMsgDrvIndPreChnSelect *poIndPreChnSelect) {
    if (!poIndPreChnSelect->bCompMode && poIndPreChnSelect->bIdChanged) {
        _rCompMonitor.vInvalidate();
        vSetPsid();
    }
}



// todo: alway keep current channel
tVoid dabdrv_compList::vPrepareDynCompList() {
    ETG_TRACE_USR4(("dabdrv_compList::vPrepareDynCompList:START"));
    // clear all markers for changes
    _mapDynSrvByLabel.clear();
    DAB_FOREACH_MAP(trMecaId, trCompListServiceInfo,iterSrvMap, _mapCompInfo) {
        trCompListServiceInfo &rCompInfo = (*iterSrvMap).second;
        if (rCompInfo.rLabel.bLabelValid && !rCompInfo.bPrim) {
            ETG_TRACE_USR4(("dabdrv_compList::vPrepareDynCompList:ADDING:u16Scidi=%x label=%17s",
                            rCompInfo.u16Scidi,
                            rCompInfo.rLabel.pcGetCString()));
            _mapDynSrvByLabel.insert(pair<trMecaLabel, trMecaId>(rCompInfo.rLabel,trMecaId(rCompInfo.u16Scidi)));
        }
        else {
            ETG_TRACE_USR4(("dabdrv_compList::vPrepareDynCompList:IGNORE:srvId=%x label=%17s (infPres=%d, labelValid=%d bPrim=%d)",
                            rCompInfo.u16Scidi,
                            rCompInfo.rLabel.pcGetCString(),
                            rCompInfo.bServiceInfoPresent, 
                            rCompInfo.rLabel.bLabelValid,
                            rCompInfo.bPrim));
        }
    }
    ETG_TRACE_USR4(("dabdrv_compList::vPrepareDynCompList:END(updated)"));
}

tVoid dabdrv_compList::vOpenFrozenCompList() {
    ETG_TRACE_USR4(("dabdrv_compList::vOpenFrozenCompList:START"));

    vPrepareDynCompList();
    tU16 u16Index=0;
    _mapFrozenIndexBySrv.clear();
    _vectorFrozenCompList.clear();
    _vectorFrozenCompList.resize(_mapDynSrvByLabel.size());

    DAB_FOREACH_MMAP(trMecaLabel, trMecaId, iterDynSrvMap, _mapDynSrvByLabel) {
        trCompListServiceInfo &rServiceInfo=_mapCompInfo[(*iterDynSrvMap).second];
        trChnListElem oChnListElem;
        oChnListElem = rServiceInfo;
        oChnListElem.u16Id=(tU16)(u16Index+1);
        _vectorFrozenCompList[u16Index]= oChnListElem;
        ETG_TRACE_USR4(("dabdrv_compList::adding[%d]: id=%d u16Scidi=%x label=%s",
                        u16Index,
                        oChnListElem.u16Id,
                        rServiceInfo.u16Scidi,
                        rServiceInfo.rLabel.pcGetCString()));
        _mapFrozenIndexBySrv[trMecaId(rServiceInfo.u16Scidi)]=u16Index;
        u16Index++;
    }

    ETG_TRACE_USR4(("dabdrv_compList::vOpenFrozenCompList:END"));
}

tVoid dabdrv_compList::vClearFrozenCompList() {
    ETG_TRACE_COMP(("vClearFrozenCompList START"));

    _vectorFrozenCompList.clear();
    _mapFrozenIndexBySrv.clear();
    _enFrozenCompListState = enFrozenCompListState_Closed;
    _oCompListCollectTimer.vStop();
    ETG_TRACE_COMP(("vClearFrozenCompList END"));
}

tVoid dabdrv_compList::vClearCompList(tenListCheckState enListCheckState) {
    ETG_TRACE_COMP(("vClearCompList START"));

    _enListCheckState=enListCheckState;
    _u16LastUpdatedScidi=0;
    _bCompListValid=FALSE;
    _mapCompInfo.clear();
    _iterUpdate=_mapCompInfo.end();
    _bWaitCompList=FALSE;
    _u8NumComponents=0;
    ETG_TRACE_COMP(("vClearCompList END"));
}

tVoid dabdrv_compList::vProcess(trMsgSrvCmdSetCompList *poCmdOpenList) {
    ETG_TRACE_COMP(("trMsgSrvCmdSetCompList START"));

    DAB_tenResult enRes = DAB_enResult_OK;
    if (_enFrozenCompListState==enFrozenCompListState_Pending && 
        poCmdOpenList->enOperation != tenListOperation_CLOSE) {
        // we are busy
        enRes=DAB_enResult_FAILED;
    }

    switch (poCmdOpenList->enOperation) {
        case tenListOperation_OPEN:
            if (bIsFrozenCompListOpen()) {
                vClearFrozenCompList();
            }
            _enFrozenCompListState=enFrozenCompListState_Pending;
            vSendFrozenCompList();
            break;
        case tenListOperation_CLOSE:
            vClearFrozenCompList();
            vSendFrozenCompList();
            break;
        case tenListOperation_GET:
            vSendFrozenCompList(TRUE);
            break;
        case tenListOperation_INVALID:
        default:
            enRes=DAB_enResult_FAILED;
            break;
    }
    poCmdOpenList->enRes=enRes;
    vTraceAll();
    ETG_TRACE_COMP(("trMsgSrvCmdSetCompList END"));
}

tVoid dabdrv_compList::vStartCompInfoCollect() {
    ETG_TRACE_COMP(("vStartCompInfoCollect START"));
    poGet1sTickTimer()->vStart();
    _u32UpdateId++;
    _oCompListCollectTimer.vStop();
    _iterUpdate=_mapCompInfo.upper_bound(trMecaId(_u16LastUpdatedScidi));
    if (_iterUpdate==_mapCompInfo.end()) {
        _iterUpdate=_mapCompInfo.begin();
    }

    _enListCheckState=enListCheckState_Virgin;
    vRequestNextCompInfo();
    ETG_TRACE_COMP(("vStartCompInfoCollect END"));
}

tVoid dabdrv_compList::vRequestNextCompInfo() {
    tBool bSent=FALSE;
    ETG_TRACE_COMP(("vRequestNextCompInfo START"));

    tBool bDone=(_enListCheckState==enListCheckState_Done);
    while(!bDone) {
        if (_iterUpdate!=_mapCompInfo.end()) {
            tenListElemUpdateState enListElemUpdateState=_iterUpdate->second.enListElemUpdateState;
            ETG_TRACE_USR1(("vRequestNextSrvInfo:checking sid=0x%04x checkState=%d elemState=%d",
                            _iterUpdate->first.u16GetScidi(),
                            ETG_CENUM(tenListCheckState, _enListCheckState),
                            ETG_CENUM(tenListElemUpdateState, enListElemUpdateState)));            
            if ((tU32)_enListCheckState==(tU32)enListElemUpdateState && _iterUpdate->second.u32UpdateId !=_u32UpdateId) {
                // mark that chn has been updated
                _iterUpdate->second.u32UpdateId =_u32UpdateId;
                // request chnInfo
                _u16LastUpdatedScidi=_iterUpdate->first.u16GetScidi();
                trMeca_CDbServiceComponentGetInfo rMecaCComponentGetInfo;
                rMecaCComponentGetInfo.rService=rGetCurrentSid().rGetProgService();
                rMecaCComponentGetInfo.u16SCIDI=_u16LastUpdatedScidi;
                dabdrv_mecaIf::instance()->vSendMecaCommand(rMecaCComponentGetInfo);
                bSent=TRUE;
                bDone=TRUE;
            }
            if (++_iterUpdate == _mapCompInfo.end()) {
                ETG_TRACE_USR1(("vRequestNextSrvInfo:wrap MAP"));                        
                _iterUpdate=_mapCompInfo.begin();
            }
        }
        if (_iterUpdate==_mapCompInfo.begin()) {
            _enListCheckState = (tenListCheckState)((tU32)_enListCheckState+1);
            ETG_TRACE_USR1(("vRequestNextCompInfo:cycle done, new checkState=%d",
                            ETG_CENUM(tenListCheckState, _enListCheckState))); 
            if (_enListCheckState==enListCheckState_Done) {
                _u16LastUpdatedScidi=0;
                bDone=TRUE;
                break;
            } 
            
        }
    }

    if (!bSent && _enListCheckState==enListCheckState_Done) {
        ETG_TRACE_COMP(("vRequestNextCompInfo DONE"));
        //vTraceAll();
        _bCompListValid=TRUE;
        // update finished
        poGet1sTickTimer()->vStop();
        _oCompListUpdateTimer.vStart(DAB_COMP_LIST_UPDATE_TIMER_MS);
        vUpdateNumComponents();
        vSendFrozenCompList();
    }
    ETG_TRACE_COMP(("vRequestNextCompInfo END"));

}

tVoid dabdrv_compList::vProcess(trMsgCompListUpdateTimer *) {
    // todo: send method result "component-list"
    vStartCompInfoCollect();
}


tVoid dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList* poCompList) {
    ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList START"));

	if (poCompList->u8FilterId != (tU8)enDabDbFilterId_Default) {
        ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList Wrong Filter END"));
	    return;
	}

    if (poCompList->rRefProgrammeService!= rGetCurrentSid()) {
        ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList Wrong SID END"));
        return;
    }
    if (poCompList->enServiceType != enMeca_ServiceType_AUDIO_SERVICE) {
        ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList Wrong SrvType END"));
        return;
    }

    if (!poCompList->lu16SCIDI.size()) {
        ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList Empty END"));
        // ensemble is not tuned
        return;
    } else if (_bWaitCompList) {
        tU8 u8NewNumComp=(tU8)(poCompList->lu16SCIDI.size()-1);
        if (_u8NumComponents!=u8NewNumComp) {
            _u8NumComponents=u8NewNumComp;
            DAB_vCallMsgCtor(trMsgDrvIndNumComponentsOfSrv(_rCurSid, _u8NumComponents));
        }
        _bWaitCompList=FALSE;
    }
    
    ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList START"));
    tBool bFoundMonitor=FALSE;
    tBool bChanged=FALSE;
    tBool bStable=TRUE;
    // unset bFoundInSrvList for all services in map
    DAB_FOREACH_MAP(trMecaId, trCompListServiceInfo, iterSrvMap, _mapCompInfo) {
        (*iterSrvMap).second.bFoundInSrvList=FALSE;
    }
    // set bFoundInSrvList and store reception-quality map for all services received in list
    for(tU8 u8Index=0; u8Index<poCompList->lu16SCIDI.size(); u8Index++) {
        if (poCompList->lu8SCIDS[u8Index]==0xFF) {
            //the scidi will change later ...
            bStable=FALSE;
            continue;
        }
        trMecaId rMecaId(poCompList->lu16SCIDI[u8Index]);
        ETG_TRACE_USR4(("dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList): check monitor valid=%d monitoredId=0x%04x cur=0x%04x",
                        _rCompMonitor.bIsValid(),
                        _rCompMonitor.rScidi.u16GetScidi(),
                        rMecaId.u16GetScidi()));
        if (_rCompMonitor.bIsValid() && _rCompMonitor.rScidi==rMecaId) {
            ETG_TRACE_USR4(("dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList): found monitor"));
            bFoundMonitor=TRUE;
        }
        trCompListServiceInfo &rCompInfo = _mapCompInfo[rMecaId];
        if (rCompInfo.bNew) {
            bChanged=TRUE;
        }
        rCompInfo.u16Scidi = rMecaId.u16GetScidi();
        rCompInfo.bFoundInSrvList=TRUE;
    }

    // search and delete all services from map that were not contained in received list. (bFoundInSrvList==FALSE)
    // request label for all services where we have no label received yet
    tBool bDelete=FALSE;
    map<trMecaId, trCompListServiceInfo>::iterator iterDelete;
    DAB_FOREACH_MAP(trMecaId, trCompListServiceInfo, iterSrvMap2, _mapCompInfo) {
        if (bDelete) {
            bDelete=FALSE;
            ETG_TRACE_USR4(("dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList): deleting 0x%04x",
                            iterDelete->first.u16GetScidi()));
            _mapCompInfo.erase(iterDelete);
        }
        trCompListServiceInfo &rCompInfo=(*iterSrvMap2).second;
        // 1.) remove no longer valid services
        if (rCompInfo.bFoundInSrvList==FALSE) {
            bChanged=TRUE;
            iterDelete=iterSrvMap2;
            bDelete=TRUE;
        } 
        // 2.) request missing labels
        else if (!rCompInfo.bLabelPresent) {
            if (rCompInfo.bNew) {
                // set default-value
                rCompInfo.bNew=FALSE;
                //rCompInfo.rLabel = trMecaLabel(rCompInfo.u16Scidi);
                bChanged=TRUE;
            }
        }
    }

    if (bDelete) {
        ETG_TRACE_USR4(("dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList): deleting 0x%08x",
                        iterDelete->first.u16GetScidi()));
        _mapCompInfo.erase(iterDelete);
    }
    if (_rCompMonitor.bIsValid()) {
        tBool bSend=FALSE;
        ETG_TRACE_USR4(("dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList): monitor active bPresent=%d->%d bStable=%d",
                        _rCompMonitor.bPresent,
                        bFoundMonitor,
                        bStable));
        if (!_rCompMonitor.bPresent && bFoundMonitor) {
            // component we are waiting for arived
            bSend=TRUE;
            ETG_TRACE_USR4(("dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList): component arrived"));
        }
        else if (_rCompMonitor.bPresent && !bFoundMonitor && bStable) {
            // we have the stable compoent-list, and the monitored compoent is not included
            _rCompMonitor.bPresent=FALSE;
            bSend=TRUE;
            ETG_TRACE_USR4(("dabdrv_compList::vProcess(trMeca_RDbServiceGetComponentList): component lost"));
        }
        if (bSend) {
            trMsgDrvRspCompListScidiMonitor rMonitorRsp(_rCompMonitor.rScidi);
            if (_rCompMonitor.bPresent) {
                rMonitorRsp.bNew=TRUE;
            }
            else {
                rMonitorRsp.bLost=TRUE;
            }
            DAB_vCallMsg(rMonitorRsp);
        }
    }

    if (bChanged) {
        ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList CHANGED"));
        vStartCompInfoCollect();
        //vTraceAll();
    }
    ETG_TRACE_COMP(("trMeca_RDbServiceGetComponentList END"));

}

tVoid dabdrv_compList::vSendFrozenCompList(tBool bForced) {
    tBool bSend=FALSE;
    ETG_TRACE_COMP(("vSendFrozenCompList START bForced=%d", bForced));

    if (_enFrozenCompListState==enFrozenCompListState_Pending &&
        _bCompListValid) {
        ETG_TRACE_COMP(("vSendFrozenCompList PENDING"));
        _enFrozenCompListState=enFrozenCompListState_Open;
        vOpenFrozenCompList();
        bSend=TRUE;
    } else if (_enFrozenCompListState==enFrozenCompListState_Closed) {
        ETG_TRACE_COMP(("vSendFrozenCompList CLOSE"));

        _enFrozenCompListState=enFrozenCompListState_ClosedSent;
        bSend=TRUE;
    }
    if (bSend || bForced) {
        ETG_TRACE_COMP(("vSendFrozenCompList SEND"));
        // inform chn-info about status of frozen list first to let it fetch the frozenListId in case 
        trMsgDrvCmdFrozenCompListStatus rFrozenCompListStatus(bIsFrozenCompListOpen());
        DAB_vCallMsg(rFrozenCompListStatus);
        // inform clients of frozen comp-list
        DAB_trCompListProperty oProperty;
        oProperty.bActivatedElemId =u16GetActivatedElemId();
        oProperty.bOpen=bIsFrozenCompListOpen();
        oProperty.vectorCompEntries=_vectorFrozenCompList;
        dabdrv_properties::instance()->oCompListProperty.vSet(oProperty);
    }
    ETG_TRACE_COMP(("vSendFrozenCompList END"));

}

tU8 dabdrv_compList::u8GetNumComponents(trMecaId const &rMecaId) const {
    if (_rCurSid != rMecaId) {
        ETG_TRACE_USR1(("dabdrv_compList::u8GetNumComponents: 0 (_rCurSid=0x%08x !=  rMecaId=0x%08x)",
                        _rCurSid.u32GetSID(), rMecaId.u32GetSID()));
        return 0;
    }
        ETG_TRACE_USR4(("dabdrv_compList::u8GetNumComponents(rMecaId=0x%08x)=%d",
                        _rCurSid.u32GetSID(), _u8NumComponents));
    return _u8NumComponents;
}


tVoid dabdrv_compList::vUpdateNumComponents() {
    // todo: check if trMeca_RDbServiceGetComponentList-1 shall be used instead.
    // then we have number immediatelly, but it might be wrong, especially if component without label are not accepted
    tU8 u8NewNumComp=0;
    tBool bSend=TRUE;
    ETG_TRACE_COMP(("vUpdateNumComponents START"));

    if (!dabdrv_main::instance()->bIsCompMode()) {
        if (!_bCompListValid) {
            // let it unchanged
            bSend=FALSE;
        }
        else {
            tBool bPrimFound=FALSE;
            DAB_FOREACH_MAP(trMecaId, trCompListServiceInfo,iterSrvMap, _mapCompInfo) {
                trCompListServiceInfo &rCompInfo = (*iterSrvMap).second;
                if (rCompInfo.bPrim) {
                    bPrimFound=TRUE;
                }
                if (rCompInfo.rLabel.bLabelValid && !rCompInfo.bPrim) {
                    ETG_TRACE_USR4(("dabdrv_compList:: vUpdateNumComponents: count scidi=0x%04x",rCompInfo.u16Scidi));
                    u8NewNumComp++;
                }
            }
            if (!bPrimFound && u8NewNumComp) {
                u8NewNumComp--;
            }
        }
    }
    ETG_TRACE_USR4(("dabdrv_compList:: vUpdateNumComponents: num:%d-->%d bSend=%d",
                    _u8NumComponents, u8NewNumComp, bSend));

    if (bSend && _u8NumComponents!=u8NewNumComp) {
        _u8NumComponents=u8NewNumComp;
        DAB_vCallMsgCtor(trMsgDrvIndNumComponentsOfSrv(_rCurSid, _u8NumComponents));
    }
    ETG_TRACE_COMP(("vUpdateNumComponents END"));

}

tVoid dabdrv_compList::vProcess(trMeca_RDbServiceComponentGetInfo* poCompInfo) {

    ETG_TRACE_COMP(("RDbServiceComponentGetInfo START"));
    // if we received a valid label and the received label differs from the existing label, store it.
    if (poCompInfo->rService != rGetCurrentSid()) {        
        ETG_TRACE_COMP(("RDbServiceComponentGetInfo DISCARD"));
		return;
    }
    trMecaId rCompId(poCompInfo->u16SCIDI);
    trCompListServiceInfo &rCompInfo = _mapCompInfo[(trMecaId)poCompInfo->u16SCIDI];
    rCompInfo.bServiceInfoPresent=TRUE;
    rCompInfo.bPrim=DAB_BOOL_FROM_BIT(poCompInfo->u8Flags, DAB_b0); 
    rCompInfo.enListElemUpdateState=enListElemUpdateState_Ok;
    tBool bLabelChanged=FALSE;
    if (poCompInfo->rLabel.bLabelValid) {
        if (!rCompInfo.bLabelPresent || (rCompInfo.rLabel !=poCompInfo->rLabel )) {
            rCompInfo.rLabel=poCompInfo->rLabel;
            rCompInfo.bLabelPresent = TRUE;
            bLabelChanged=TRUE;
        }
    } else {
#ifdef DAB_REPLACE_INVALID_LABEL_BY_SID
        trMecaLabel rNewLabel(poCompInfo->u16SCIDI);
        if (!rCompInfo.bLabelPresent || rCompInfo.rLabel!=rNewLabel) {
            rCompInfo.bLabelPresent = TRUE;
            rCompInfo.rLabel = rNewLabel;
            bLabelChanged=TRUE;
        }
#endif
    }

    if (bLabelChanged && poCompInfo->u16SCIDI==_rCompMonitor.rScidi.u16GetScidi()) {
        DAB_vCallMsgCtor(trMsgDrvRspCompListScidiMonitor(_rCompMonitor.rScidi));
    }
    if (_u16LastUpdatedScidi && poCompInfo->u16SCIDI==_u16LastUpdatedScidi && _enListCheckState==enListCheckState_Virgin) {
        ETG_TRACE_COMP(("RDbServiceComponentGetInfo REQUEST"));
        poGet1sTickTimer()->vStart();
        vRequestNextCompInfo();
    }

    //vTraceAll();
    ETG_TRACE_COMP(("RDbServiceComponentGetInfo END"));
}

tU16 dabdrv_compList::u16GetFrozenListId(trMecaId const &rMecaId) {
    tU16 u16ResId=0;
    if (bIsFrozenCompListOpen()) {
        DAB_IF_FIND_MAP(trMecaId, tU16, iter, _mapFrozenIndexBySrv, rMecaId) {
            u16ResId = (tU16)(iter->second + 1);
        }
    }
	else{
		DAB_FOREACH_MAP_CONST(trMecaId, trCompListServiceInfo,iterSrvMap, _mapCompInfo) {
			u16ResId++;
			if(iterSrvMap->first==rMecaId){
				break;
			}			
		}
	}
    return u16ResId;
}

trChnListChnInfo dabdrv_compList::rGetCompInfo(trMecaId const &rMecaId) {
    trChnListChnInfo rResInfo;
    rResInfo.bValid=FALSE;
	//vTraceAll();
    DAB_IF_FIND_MAP(trMecaId, trCompListServiceInfo, iter, _mapCompInfo, rMecaId) {
        rResInfo.bValid=TRUE;
        trCompListServiceInfo &rDbSrvInfo =iter->second;
        rResInfo.rMecaId=rMecaId;
        rResInfo.bLabelPresent=rDbSrvInfo.bLabelPresent;
        rResInfo.rLabel=rDbSrvInfo.rLabel;
        rResInfo.u16FrozenListId=u16GetFrozenListId(rMecaId);
        // todo: check if auto-compare belongs to primary service
        rResInfo.u8PresetNumber=dabdrv_presets::instance()->u8GetPresetNumber(rGetCurrentSid(),(tU8)(dabdrv_chnInfo::instance()->rGetChnInfo().u16Scids));
    }
    return rResInfo;
}

trMecaLabel dabdrv_compList::rGetLabel(trMecaId const &rMecaId) {
    DAB_IF_FIND_MAP(trMecaId, trCompListServiceInfo, iter, _mapCompInfo, rMecaId) {
        return iter->second.rLabel;
    }
    trMecaLabel rLabel;
    return rLabel;
}

tU16 dabdrv_compList::u16GetActivatedElemId() const {
    return dabdrv_compInfo::instance()->rGetCompInfo().u16FrozenListId;
}

trChnListElem dabdrv_compList::rGetElemFromFrozenList(tU16 u16ElemId) {
    static trChnListElem rChnListElem;
    if (!bIsFrozenCompListOpen() || u16ElemId < 1) {
        return rChnListElem;
    }
    tU16 u16ElemIndex = (tU16)(u16ElemId-1);

    if (u16ElemIndex<(_vectorFrozenCompList.size())) {
        return _vectorFrozenCompList[u16ElemIndex];
    }

    return rChnListElem;
}

trChnListElem dabdrv_compList::rGetElemRelative(tS16 s16Steps)
{
    trChnListElem rChnListElem;
    
    if (bIsFrozenCompListOpen()) {
        tU16 u16FrozenListSize= (tU16)_vectorFrozenCompList.size();
        if (!u16FrozenListSize) {
            return rChnListElem;            
        }
        tU16 u16ActiveElemId= u16GetActivatedElemId();
        if (!u16ActiveElemId) {
            return (s16Steps>=0) ?  *_vectorFrozenCompList.begin() : *_vectorFrozenCompList.rbegin();
        }
        tS16 s16Index= (tS16)(u16ActiveElemId-1);
        s16Index=(tS16)(s16Index + s16Steps);
        while (s16Index <0) {
            s16Index=(tS16)(s16Index + u16FrozenListSize);
        }
        tU16 u16Index=(tU16)s16Index;
        u16Index %= u16FrozenListSize;
        return _vectorFrozenCompList[u16Index];

    }
    else {
        trMecaId rMecaId;
        vPrepareDynCompList();
        trChnListChnInfo rChnInfo = dabdrv_compInfo::instance()->rGetCompInfo();
        if (_mapDynSrvByLabel.empty()) {
            return rChnListElem;
        }
        tBool bOk=FALSE;
        DAB_IF_FIND_MAP(trMecaLabel, trMecaId, iterMapDynSrvLabel, _mapDynSrvByLabel, rChnInfo.rLabel) {
            bOk=TRUE;
            for (;;) {
                if (iterMapDynSrvLabel==_mapDynSrvByLabel.end()) {
                    iterMapDynSrvLabel=_mapDynSrvByLabel.begin();
                    bOk=FALSE;
                }
                if (iterMapDynSrvLabel->first!=rChnInfo.rLabel) {
                    // wrong label, use iter anyway
                    break; 
                }
                else if (iterMapDynSrvLabel->second==rChnInfo.rMecaId) {
                    // found
                    break;
                }
                else {
                    // try next, maybe it has the same label (multimap)
                    ++iterMapDynSrvLabel;
                }
            }

            if (bOk) {
                while (s16Steps>0) {
                    ++iterMapDynSrvLabel;
                    if (iterMapDynSrvLabel==_mapDynSrvByLabel.end()) {
                        iterMapDynSrvLabel=_mapDynSrvByLabel.begin();
                    }
                    s16Steps--;
                }
                while (s16Steps<0) {
                    if (iterMapDynSrvLabel==_mapDynSrvByLabel.begin()) {
                        iterMapDynSrvLabel=_mapDynSrvByLabel.end();
                    }
                    --iterMapDynSrvLabel;
                    s16Steps++;
                    
                }
                rMecaId = iterMapDynSrvLabel->second;
            }
        } 
        if (!bOk) {
            rMecaId= (s16Steps>=0)? _mapDynSrvByLabel.begin()->second : _mapDynSrvByLabel.rbegin()->second;
        }
        if (rMecaId.bIsValid()) {
            DAB_IF_FIND_MAP(trMecaId, trCompListServiceInfo, iterMapCompInfo, _mapCompInfo, rMecaId) {
                return iterMapCompInfo->second;
            } else {
                return _mapCompInfo.begin()->second;
            }
        }
        
    }
    return rChnListElem;
}


trMecaId dabdrv_compList::rGetCurrentSid() const {
    return dabdrv_chnInfo::instance()->rGetCurrentSid();
}


