/************************************************************************
 * FILE:       dabdrv_chnSelect.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabdrv_chnSelect
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#include "dabdrv_main.hpp"
#include "dabdrv_mecaIf.h"
#include "dabdrv_chnList.hpp"
#include "dabdrv_chnInfo.hpp"
#include "dabdrv_chnSelect.hpp"
#include "dabdrv_mute.hpp"
#include "dabdrv_presets.hpp"
#include "dabdrv_rdm.hpp"
#include "dabdrv_compSelect.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_DRV_SRV 
#include "trcGenProj/Header/dabdrv_chnSelect.cpp.trc.h"
#endif

using namespace DAB;

namespace DAB {
    // Timer used for supervision of chnSelect
    struct trMsgChnSelectSupervision:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgChnSelectSupervision"));
        };
    };
#ifdef MUTE_ACTIVE
	struct trMsgMuteSupervision:
			public DAB_Message
		{
			DAB_DISPATCH_IMPL
			virtual tVoid vTrace() const {
				ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG,
									"trMsgMuteSupervision"));
			};
		};
#endif

	// Timer used for supervision of Play_Asid response
    struct trMsgRspPlayAsidSupervisionTimer:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgRspPlayAsidSupervisionTimer"));
        };
    };
}


dabdrv_chnSelect::dabdrv_chnSelect(tVoid):
    _enChnSelectState(enChnSelectState_Idle),
    _enChnSelectStateOrig(enChnSelectState_Idle),
    _u16NumSkips(0),
    _bRetry(FALSE),
    _enChnState(enChnState_Invalid),
    _enMaxSrvFollowMode(enMeca_RdmServFollowMode_None),
    _bHardLinkingActiv(FALSE),
    _bForcedActivityTerminate(FALSE),
    _enTuneType(DAB_enTuneType_INVALID),
    _bDabDabSrvFollowActive(FALSE),
    _enSourceState(DAB_enSourceState_FG),
    bChnSelectFlag(TRUE)
 {
     vSubscribe<trMsgDrvStopComponent>();
    vSubscribe<trMeca_RRdmAudioPlayAsid>();
    vSubscribe<trMeca_RRdmGetRdmInfo>();
//    vSubscribe<trMeca_RRdmServFollowActive>();   
    vSubscribe<trMsgDrvCmdChnSelect>();
	vSubscribe<trMsgDrvCmdEnsSelect>();
    vSubscribe<trMsgDrvRspMuteDone>();
    vSubscribe<trMsgDrvRspSetDabSrvFollowClass>();
    vSubscribe<trMsgDrvRspChnListPsidMonitor>();
    vSubscribe<trMsgDrvFmServiceFollow>();
    vSubscribe<trMeca_RRdmServFollowActive>();
    vSubscribe<trMsgDrvCmdSourceState>();
	 vSubscribe<trMeca_RRdmTuneEid>();
	 vSubscribe<trMsgDrvCmdRdmTune>();
	 vSubscribe<trMeca_RRdmTune>();
#ifdef MUTE_ACTIVE
	 vSubscribe<trMsgDrvCmdTuneFreqlabel>();
#endif
	 vSubscribe<trMeca_RRdmTuneFrequencyLabel>();
}

tVoid dabdrv_chnSelect::vInit() {
    _oChnSelectSupervisionTimer.vInit(instance(), trMsgChnSelectSupervision());
	_otrMsgRspPlayAsidSupervisionTimer.vInit(instance(),trMsgRspPlayAsidSupervisionTimer());
#ifdef MUTE_ACTIVE
    _oMuteSupervisionTimer.vInit(instance(), trMsgMuteSupervision());
#endif
    
#if 0
    tS16 s16Ar[] = {1, -20, 50, 1234, 12345, 6, 7, 8, 9, 10};

      ETG_TRACE_USR1(("ETG This trace (list fixed S16) prints %d, list=%d, added %d",
                    10, ETG_LIST_LEN(10), ETG_LIST_PTR_T16(s16Ar), 1 ));


#endif

	_oChnInfoUpdateSuperVisionTimer.vInit(instance(), trMsgChnInfoUpdateTimer());

}

tVoid dabdrv_chnSelect::vProcess(trMsgDrvStopComponent *poStopComponent) {
    (tVoid)poStopComponent;
    _enMaxSrvFollowMode=enMeca_RdmServFollowMode_None;
    _bHardLinkingActiv = FALSE;
    _rLastSentPlayAsid=trMecaId();
    vUpdate(enChnSelectState_Failed);
}


tVoid dabdrv_chnSelect::vDeInit() {
    _oChnSelectSupervisionTimer.vDeInit();
	_otrMsgRspPlayAsidSupervisionTimer.vDeInit();
#ifdef MUTE_ACTIVE
    _oMuteSupervisionTimer.vDeInit();
#endif
	_oChnInfoUpdateSuperVisionTimer.vDeInit();
}

dabdrv_chnSelect::~dabdrv_chnSelect(tVoid) {}

tVoid dabdrv_chnSelect::vTraceState() const {
    ETG_TRACE_USR1(("  dabdrv_chnSelect STATE: _enChnSelectState=%d _u16NumSkips=%d _rUsrReqProgSrv=0x%08x _activeChnSelect:source=%d mode=%d value=0x%08x bNegative=%d "
                    "_rPendingChnListElem: u16Id=%d rMecaId=0x%08x u16Scidi=0x%04x",
                    ETG_CENUM(tenChnSelectState, _enChnSelectState),
                    _u16NumSkips,
                    _rUsrReqProgSrv.u32GetSID(),
                    ETG_CENUM(tenChnSelectSource, _activeChnSelect.enChnSelectSource),
                    ETG_CENUM(tenChnSelectMode, _activeChnSelect.enChnSelectMode),
                    _activeChnSelect.u32Value,
                    _activeChnSelect.bNegative,
                    _rPendingChnListElem.u16Id,
                    _rPendingChnListElem.rMecaId.u32GetSID(),
                    _rPendingChnListElem.u16Scidi));

    ETG_TRACE_USR1(("  dabdrv_chnSelect STATE (cont): svTimer running=%d _bRetry=%d _enChnState=%d",
                    _oChnSelectSupervisionTimer.bIsRunning(),
                    _bRetry,
                    ETG_CENUM(tenChnState, _enChnState)));
    
}

trEnsembleListElem dabdrv_chnSelect::rValidateRequest(trMsgDrvCmdEnsSelect const *poDrvCmdEnsSelect)const
{
		ETG_TRACE_USR4(("dabdrv_chnSelect::rValidateRequest: EnsSelect Mode =%d freq=0x%08x value=%d",
			poDrvCmdEnsSelect->enEnsSelectMode,
			poDrvCmdEnsSelect->u32Frequency,
			poDrvCmdEnsSelect->u32Value));

	 trEnsembleListElem rChnListEnsElem;
        switch (poDrvCmdEnsSelect->enEnsSelectMode) {
            case enEnsSelectMode_Eid:
            {
				ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enEnsSelectMode_Eid "));

                trMecaEnsemble rEnsemble(poDrvCmdEnsSelect->u32Value);
                // check if we have the channel in our channel-list
                rChnListEnsElem= dabdrv_chnList::instance()->rGetEnsembleInfo(rEnsemble);
                break;
            }
            case enEnsSelectMode_ListId:
				ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enEnsSelectMode_ListId "));
                // select from frozen list by ID
                rChnListEnsElem= dabdrv_chnList::instance()->rGetEnsembleElemFromFrozenList((tU16)poDrvCmdEnsSelect->u32Value);
                break;
            case enEnsSelectMode_Relative:
            {
                /* Select from list relativ to current position in list.
                   chnList will take care if the frozen list or the dynamic list has to be used.
                */
				 ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enEnsSelectMode_Relative steps=%d dir=%d ", poDrvCmdEnsSelect->u32Value, poDrvCmdEnsSelect->bNegative));
                tS32 s32Steps=(tU16)poDrvCmdEnsSelect->u32Value * (tS8)(poDrvCmdEnsSelect->bNegative ? -1 : 1);
                rChnListEnsElem = dabdrv_chnList::instance()->rGetEnsElemRelative((tS16)s32Steps);
				ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enEnsSelectMode_Relative SID=%x", rChnListEnsElem.rEnsemble._u32Id));
                break;
            }
            case enEnsSelectMode_Invalid:
            default:
                break;

        }

        return rChnListEnsElem;
}

/*
  check the parameters of the request an return appropriate channel to be selected.
  in case of error the sid 0 will be returned.

*/
trChnListElem dabdrv_chnSelect::rValidateRequest(trMsgDrvCmdChnSelect const *poDrvCmdChnSelect) const {
    trChnListElem rChnListElem;
        
    ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest: EnsSelect Mode =%d value=%d scids=%d",
          poDrvCmdChnSelect->enChnSelectMode,
          poDrvCmdChnSelect->u32Value,
          poDrvCmdChnSelect->u16scids));

	
	switch (poDrvCmdChnSelect->enChnSelectMode) {
            case enChnSelectMode_Psid:
            {
                 ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enChnSelectMode "));
				//trMecaProgrammeService rProgService(poDrvCmdChnSelect->u32Value,poDrvCmdChnSelect->u16scids);
                // check if we have the channel in our channel-list
                rChnListElem = dabdrv_chnList::instance()->rGetChnInfo(trMecaId(poDrvCmdChnSelect->u32Value),poDrvCmdChnSelect->u16scids);
                break;
            }
            case enChnSelectMode_ListId:
				ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enChnSelectMode_ListId "));
                // select from frozen list by ID
                rChnListElem= dabdrv_chnList::instance()->rGetElemFromFrozenList((tU16)poDrvCmdChnSelect->u32Value);
                break;
            case enChnSelectMode_Relative:
            {
                /* Select from list relativ to current position in list.
                   chnList will take care if the frozen list or the dynamic list has to be used.
                */
				ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enEnsSelectMode_Relative steps=%d dir=%d ", poDrvCmdChnSelect->u32Value, poDrvCmdChnSelect->bNegative));
                tS32 s32Steps=(tU16)poDrvCmdChnSelect->u32Value * (tS8)(poDrvCmdChnSelect->bNegative ? -1 : 1);
                rChnListElem = dabdrv_chnList::instance()->rGetElemRelative((tS16)s32Steps);
				ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest:enEnsSelectMode_Relative SID=%x", rChnListElem.rMecaId._u32Id));
                break;
            }
            case enChnSelectMode_Invalid:
            default:
                break;

        }

	   ETG_TRACE_USR1(("dabdrv_chnSelect::rValidateRequest: rChnListElem.bPSFlag =%d rChnListElem.u16Scids=%d",
		   rChnListElem.bPSFlag,rChnListElem.u16Scids));

		if(!rChnListElem.bPSFlag)
		{
			trMsgDrvCmdCompSelectFromSrvList oCompSelectFromSrvList;
			oCompSelectFromSrvList.ochnListElem = rChnListElem;
			DAB_vCallMsg(oCompSelectFromSrvList);
			
		}
		
		return rChnListElem;

}



tVoid dabdrv_chnSelect::vPropagateNewSid() {

    tBool bCurCompMode=dabdrv_main::instance()->bIsCompMode();
    tBool bNewCompMode=dabdrv_main::instance()->bIsCompMode() && _activeChnSelect.enChnSelectSource== enChnSelectSource_Lsm;
	trMecaProgrammeService rNewSrv(_rPendingChnListElem.rMecaId._u32Id, enMeca_ServiceType_AUDIO_SERVICE, _rPendingChnListElem.u16Scids);

    ETG_TRACE_USR1(("  dabdrv_chnSelect::vPropagateNewSid() _rLastSentPlayAsid=%x rNewSid=%x bCurCompMode=%d, bNewCompMode=%d _bRetry=%d",
                    _rLastSentPlayAsid._u32Id,
                    rNewSrv._u32Id,
                    bCurCompMode,
                    bNewCompMode,
                    _bRetry));


    if (!_bRetry && !bNewCompMode) {
        // inform caller that a chnSelectReq is now sent and the chnInfo will be updated: 
		ETG_TRACE_USR1(("  dabdrv_chnSelect::vPropagateNewSid() rChnIndPreChnSelect 1 "));
        trMsgDrvIndPreChnSelect rChnIndPreChnSelect(FALSE, _rLastSentPlayAsid!=rNewSrv, _activeChnSelect.enChnSelectSource);
        DAB_vCallMsg(rChnIndPreChnSelect);
    }

    if (_rLastSentPlayAsid!=rNewSrv || (bCurCompMode && !bNewCompMode)) {

		if(dabdrv_properties::instance()->oChnStateProperty.oGet().enChnState == enChnState_Unavail)
		{
			DAB_trChnStateProperty rChnStateProperty;
			rChnStateProperty.enChnState=enChnState_Selecting;
			dabdrv_properties::instance()->oChnStateProperty.vSet(rChnStateProperty);
		}
        // sid has changed or we left comp-mode
        _rLastSentPlayAsid=rNewSrv;
        // set new channel-info
        trMsgDrvCmdSetChnInfoPsid rDrvCmdSetChnInfoPsid(rNewSrv);
        DAB_vCallMsg(rDrvCmdSetChnInfoPsid);
        trMeca_CRdmGetRdmInfo rRequestRdmInfo;
        dabdrv_mecaIf::instance()->vSendMecaCommand(rRequestRdmInfo);
        // inform tsu about current dab-service
        trMsgDrvCmdTsuSetCurrSrv rTsuSetCurrentSrv;
        rTsuSetCurrentSrv.enTunerId=enMeca_TsuTunerId_MAIN_DAB_TUNER;
        rTsuSetCurrentSrv.rProgrammeService=rNewSrv;
        DAB_vCallMsg(rTsuSetCurrentSrv);

		ETG_TRACE_USR1(("  dabdrv_chnSelect::vPropagateNewSid() rChnIndPreChnSelect PlayAsid "));
		
		//Work-around by Surekha
  //      trMeca_CRdmAudioPlayAsid rRdmAudioPlayAsid;
  //      rRdmAudioPlayAsid.rMecaProgrammeService=rNewSrv;
  //      dabdrv_mecaIf::instance()->vSendMecaCommand(rRdmAudioPlayAsid);
 


    }

    if (!_bRetry && !bNewCompMode) {
        // inform caller that a chnSelectReq is now sent and the chnInfo will be updated: 
        trMsgDrvIndPostChnSelect rChnIndPostChnSelect(FALSE);
        DAB_vCallMsg(rChnIndPostChnSelect);
    }


}
tVoid dabdrv_chnSelect::vRequestServiceMute() {

    if (_enChnSelectState==enChnSelectState_WaitGo) {
        vPropagateNewSid();
        vUpdate(enChnSelectState_WaitMute);
        trMsgDrvCmdRequestMute rRequestMute;
        DAB_vCallMsg(rRequestMute);
    }
}
tVoid dabdrv_chnSelect::vSendSelectPsid() { //removed const for //CRQ 431566

    ETG_TRACE_USR1(("  dabdrv_chnSelect:vSendSelectPsid _bDabDabSrvFollowActive(%d)",
                    _bDabDabSrvFollowActive));
    trMeca_CRdmAudioPlayAsid rCRdmAudioPlayAsid;
    rCRdmAudioPlayAsid.rMecaProgrammeService=_rPendingChnListElem.rMecaId.rGetProgService();

	//if 2 second timer running, fast seek happens, timer needs to stop
	if (_oChnInfoUpdateSuperVisionTimer.bIsRunning()) {
		_oChnInfoUpdateSuperVisionTimer.vStop();
	}
    dabdrv_mecaIf::instance()->vSendMecaCommand(rCRdmAudioPlayAsid);

    //_oChnSelectSupervisionTimer.vStart(DAB_CHN_SELECT_WAIT_TIMER_MS);
}


/* when we receive the new service-follow-mode, the chnSelect can be sent to DAB-module
   we don't check for the correct answer */
tVoid dabdrv_chnSelect::vProcess(trMsgDrvRspSetDabSrvFollowClass *) {
    if (enChnSelectState_WaitSrvFollow==_enChnSelectState) {
        // only in this state we wait for the new service follow mode
        vSendSelectPsid();
        vUpdate(enChnSelectState_Sent);   
    }
}


tVoid dabdrv_chnSelect::vProcess(trMsgDrvRspChnListPsidMonitor *) {
    tU32 u32Sid=dabdrv_chnInfo::instance()->rGetCurrentSid().u32GetSID();
    ETG_TRACE_USR1(("dabdrv_chnSelect::trMsgDrvRspChnListPsidMonitor SID=0x%08x",
                    u32Sid));

     trMsgDrvCmdChnSelect oChnSelect;
    oChnSelect.enChnSelectSource=enChnSelectSource_Lsm;
    oChnSelect.enChnSelectMode= enChnSelectMode_Psid;
    oChnSelect.u32Value=u32Sid;
    DAB_vCallMsg(oChnSelect);
}

tVoid dabdrv_chnSelect::vProcess(trMeca_RRdmGetRdmInfo *poRdmInfo) {
    ETG_TRACE_USR1(("dabdrv_chnSelect::trMeca_RRdmGetRdmInfo  SID=0x%08x _enChnSelectState=%d",
                    poRdmInfo->rProgrammeService._u32Id,_enChnSelectState));
// On factory reset if the first freq 5A has some recieveable ensemble then it should play the 1st service of the ensemble.
	if((dabdrv_chnInfo::instance()->bGetFactoryResetStatus()) && (!poRdmInfo->rRdmStatus.bIsTuning()) && (poRdmInfo->rRdmStatus.bIsEnsValid()))
	{
		dabdrv_chnInfo::instance()->vSetFactoryResetStatus(FALSE);
		trMeca_CRdmAudioPlayService rRdmAudioPlayService;
		rRdmAudioPlayService.enServiceCommand = enMeca_RdmServiceCommand_PLAY_FIRST;
		dabdrv_mecaIf::instance()->vSendMecaCommand(rRdmAudioPlayService);
	}

    if(_enChnSelectState == enChnSelectState_TuningFreqLabel){

    	 ETG_TRACE_USR1(("dabdrv_chnSelect::trMeca_RRdmGetRdmInfo Freqlabel"));
    	if(poRdmInfo->rRdmStatus.bIsEnsValid() && !poRdmInfo->rRdmStatus.bIsTuning()){

    	    trMeca_CRdmAudioPlayService rPlayService;
    	    dabdrv_mecaIf::instance()->vSendMecaCommandCtor(rPlayService);
    	    _enChnSelectState = enChnSelectState_Idle;
    	}
    	if(!poRdmInfo->rRdmStatus.bIsTuning()){
    		vSetMute();
    	}
    }
    else if ( enChnSelectState_Hardlinking==_enChnSelectState ) {
  
        // chn shouldnt be changed, activity can be terminated
        trMsgDrvCmdSetChnInfoPsid rDrvCmdSetChnInfoPsid(poRdmInfo->rProgrammeService._u32Id);
        DAB_vCallMsg(rDrvCmdSetChnInfoPsid);
        /*trMsgDrvCmdChnSelect rChnSelectCmd(_activeChnSelect);
        rChnSelectCmd.enChnSelectSource=enChnSelectSource_Lsm;
        rChnSelectCmd.enChnSelectMode=enChnSelectMode_Psid;
        rChnSelectCmd.u32Value= poRdmInfo->rProgrammeService._u32Id;
        trChnListElem rChnListElem= rValidateRequest(&rChnSelectCmd);
        if (rChnListElem.bIsValid()) {
            vProcess(&rChnSelectCmd);
        }*/
    }
    else if (enChnSelectState_WaitRdmInfo==_enChnSelectState) {
        vPropagateNewSid();
        if (poRdmInfo->rProgrammeService == _rPendingChnListElem.rMecaId && !dabdrv_main::instance()->bIsCompMode()) {
            // chn is already selected, activity can be terminated
            vUpdate(enChnSelectState_Idle);
        }
        else {
            vUpdate(enChnSelectState_WaitMute);
            trMsgDrvCmdRequestMute rRequestMute;
            DAB_vCallMsg(rRequestMute);
        }
    }
    if ((enChnSelectState_Idle==_enChnSelectState || enChnSelectState_Failed==_enChnSelectState) && !dabdrv_main::instance()->bActivityChangedChannel()) {
        if ((poRdmInfo->rProgrammeService == _rPendingChnListElem.rMecaId)) {
            ETG_TRACE_USR1(("  dabdrv_chnSelect:trMeca_trMeca_RRdmGetRdmInfo svTimer STOP"));
            if (enChnSelectState_Failed==_enChnSelectState) {
                // rdm has now the correct SID, so everything is OK
                vUpdate(enChnSelectState_Idle);
            }

        }
    }
}

// centry update method
tVoid dabdrv_chnSelect::vUpdate(tenChnSelectState enChnSelectState, tBool bDirectRetry) {
    ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                        "dabdrv_chnSelect::vUpdate: enChnSelectState: %d --> %d",
                        ETG_CENUM(tenChnSelectState, _enChnSelectState),
                        ETG_CENUM(tenChnSelectState, enChnSelectState)));
    if (_enChnSelectState ==enChnSelectState) {
        // no change, do nothing
        return;
    }
    // set new chnSelectState
    _enChnSelectState=enChnSelectState;
    // inform system about new activity
    tBool bActivityTerminated=FALSE;
    if (_enChnSelectState==enChnSelectState_WaitGo) {
        vRequestServiceMute();
    } else if (_enChnSelectState==enChnSelectState_Idle) {
        bActivityTerminated=TRUE;
        // start timer in case that rdmInfo is not ok
        trMeca_RRdmGetRdmInfo rLastRdmInfo =dabdrv_rdm::instance()->rGetLastRdmInfo();
        DAB_vCallMsg(this, &rLastRdmInfo);
      
    }



    if (_enChnSelectState==enChnSelectState_Idle) {
        // chn-selection is successfully finished, delete retry-flag in case it is set
        _bRetry=FALSE;
    }
    tenChnState enChnState=enChnState_Invalid;
    if (_bRetry) {
        // dab-module could not select the channel, we try again but for the time beeing it is regarded as unavail
        enChnState=enChnState_Unavail;
    }
    else if (_u16NumSkips) {
        // we are searching, no update on chnState on each tried channel
        enChnState=enChnState_SelectingSearch;
    }
    else {
        // in all other cases chnState is derived from chnSelect state
        //lint -e{788} enum constant '...' not used within defaulted switch
        switch(enChnSelectState) {
            case enChnSelectState_Idle:
                enChnState= enChnState_Stable;
                break;
            case enChnSelectState_Sent:
            case enChnSelectState_WaitRdmInfo:
            case enChnSelectState_WaitSrvFollow:
                enChnState= enChnState_Selecting;
                break;
            case enChnSelectState_SentWait:
                enChnState=enChnState_SelectingWait;
                break;
            case enChnSelectState_Failed:
                enChnState=enChnState_Unavail;
                break;
            case enChnSelectState_WaitMute:
            case enChnSelectState_Hardlinking:
            default:
                break;
        }
    }
    ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                        "dabdrv_chnSelect::vUpdate?: enChnState: %d --> %d",
                        ETG_CENUM(tenChnState, _enChnState),
                        ETG_CENUM(tenChnState, enChnState)));
    if (enChnState!=enChnState_Invalid) {
        // inform about new chnState
        ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                            "dabdrv_chnSelect::vUpdate: enChnState: %d --> %d",
                            ETG_CENUM(tenChnState, _enChnState),
                            ETG_CENUM(tenChnState, enChnState)));
        _enChnState = enChnState;
        trMsgDrvCmdChnSelectState rCmdChnSelectState(enChnState);
        DAB_vCallMsg(rCmdChnSelectState);        
    }


    if (bActivityTerminated==TRUE) {
        // set service-following back to normal mode
        trMsgDrvCmdSetDabSrvFollowClass rSetSrvFollowClass(enSrvFollowClass_Normal);
        DAB_vCallMsg(rSetSrvFollowClass);
#ifdef DAB_CHN_SELECT_POST_POLL_RDM_INFO
        // sometimes we don't get the new rdm-info by auto-notification, so we poll it here
        dabdrv_mecaIf::instance()->vSendMecaCommandCtor(trMeca_CRdmGetRdmInfo());
#endif

    }
    else if (bDirectRetry) {
        /* try channel selection again without waiting for timeout */
        _bRetry=TRUE;
        _enChnSelectState=enChnSelectState_WaitGo;
        vRequestServiceMute();
    }


}


tVoid dabdrv_chnSelect::vProcess(trMsgDrvRspMuteDone *poDrvRspMuteDone) {
    (tVoid)poDrvRspMuteDone;
    if (_enChnSelectState == enChnSelectState_WaitMute) {
        vUpdate(enChnSelectState_WaitSrvFollow);
        tenSrvFollowClass enSrvFollowClass=enSrvFollowClass_ChnSelect;
        if (_u16NumSkips) {
            enSrvFollowClass=enSrvFollowClass_Skip;

        } else if (_bRetry) {
            enSrvFollowClass=enSrvFollowClass_Normal;
        }
        trMsgDrvCmdSetDabSrvFollowClass rSetSrvFollowClass(enSrvFollowClass);
        DAB_vCallMsg(rSetSrvFollowClass);
        ETG_TRACE_USR1(("  dabdrv_chnSelect:trMsgDrvRspMuteDone"));
    }
}
#ifdef MUTE_ACTIVE
tVoid dabdrv_chnSelect::vProcess(trMsgMuteSupervision* poMuteSupervision) {
(tVoid)poMuteSupervision;
	vSetMute();
	// trMsgDrvCmdTunerOpMuteReq rMuteReq;
	// rMuteReq.enRequiredMuteState=enRequiredMuteState_Demute;
    // DAB_vCallMsg(rMuteReq);

    // if(dabdrv_main::instance()->enGetMuteOperation() == enMuteOperation_EnsembleSeek)
    // {
        // DAB_trTunerStatusProperty oTunerStatus = dabdrv_properties::instance()->oTunerStatusProperty.oGet();
    	// oTunerStatus.enTunerActivityType = DAB_enTunerActivityType_NORMAL;
    	// dabdrv_main::instance()->enSetMuteOperation(enMuteOperation_None);
    	// dabdrv_properties::instance()->oTunerStatusProperty.vSet(oTunerStatus);
    // }
}
#endif
tVoid dabdrv_chnSelect::vProcess(trMsgRspPlayAsidSupervisionTimer* /*poRspPlayAsidSupervisionTimer*/) {

	ETG_TRACE_USR1(("trMsgRspPlayAsidSupervisionTimer Expired"));
	trMsgDrvDbQuery rDbQuery(Query_SelfLearnedLinkInfo);
	DAB_vCallMsg(rDbQuery);
	trMsgDrvDbQuery rDbQueryAltFreq(Query_AltFreqInformation);
	DAB_vCallMsg(rDbQueryAltFreq);

}
tVoid dabdrv_chnSelect::vProcess(trMsgChnSelectSupervision* poChnSelectSupervision) {
    (tVoid)poChnSelectSupervision;
    ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                        "vProcess::trMsgChnSelectSupervision _enChnSelectState=%d _bRetry=%d",
                        ETG_CENUM(tenChnSelectState, _enChnSelectState),
                        _bRetry));

    switch (_enChnSelectState) {

        case enChnSelectState_Hardlinking: 
        case enChnSelectState_WaitRdmInfo: 
        {
            // fake meca-rsp
            trMeca_RRdmGetRdmInfo rRdmInfo;
            DAB_vCallMsg(this, &rRdmInfo);
        }
        break;
        case enChnSelectState_Sent:
        {
            ETG_TRACE_USR1(("  dabdrv_chnSelect:trMsgChnSelectSupervision svTimer START(%d) (cause:failed)",
                            DAB_CHN_SELECT_FAILED_TIMER_MS));
            
            //_oChnSelectSupervisionTimer.vStart(DAB_CHN_SELECT_FAILED_TIMER_MS);
            vUpdate(enChnSelectState_SentWait);
            trMsgDrvCmdSetDabSrvFollowClass rSetSrvFollowClass(enSrvFollowClass_Normal);
            DAB_vCallMsg(rSetSrvFollowClass);
        }
        break;
        case enChnSelectState_SentWait:
        {
            // fake failed selection;
            // todo: different handling for preset
            
            trMeca_RRdmAudioPlayAsid rPlayAsid;
            rPlayAsid.rMecaProgrammeService=_rPendingChnListElem.rMecaId.rGetProgService();
            rPlayAsid.enServiceState=enMeca_RdmServiceState_PSID_NOT_AVAILABLE;
            DAB_vCallMsg(this, &rPlayAsid);
        }
        break;
        case enChnSelectState_Failed:
        case enChnSelectState_Idle:
        {
            if (dabdrv_main::instance()->bActivityChangedChannel()) {
                //_oChnSelectSupervisionTimer.vStart(DAB_CHN_SELECT_WAIT_TIMER_MS);
            } else {
                _bRetry=TRUE;
                vUpdate(enChnSelectState_WaitGo);        
            }
        }
        break;

        case enChnSelectState_WaitGo:
        case enChnSelectState_WaitMute:
        case enChnSelectState_WaitSrvFollow: 
        default:
            break;
    }
}


tVoid dabdrv_chnSelect::vProcess(trMsgDrvCmdChnSelect *poDrvCmdChnSelect) {
    if (_enChnSelectState == enChnSelectState_Hardlinking) {
        _enChnSelectState = _enChnSelectStateOrig;
    }

    ETG_TRACE_USR1(("  dabdrv_chnSelect:trMsgDrvCmdChnSelect _enChnSelectState=%d",
                        ETG_CENUM(tenChnSelectState, _enChnSelectState)));
    if (poDrvCmdChnSelect->enChnSelectSource==enChnSelectSource_RecallPreset) {
        ETG_TRACE_USR1(("dabdrv_chnSelect(trMsgDrvCmdChnSelect) SID=%x",
                            poDrvCmdChnSelect->u32Value));
        _rPendingChnListElem.rMecaId=trMecaId(poDrvCmdChnSelect->u32Value);
    }
    else if (poDrvCmdChnSelect->enChnSelectSource==enChnSelectSource_Lsm) {
        //_rPendingChnListElem=dabdrv_chnInfo::instance()->rGetChnInfo();
        _rPendingChnListElem= rValidateRequest(poDrvCmdChnSelect);
    }
    else {
        _rPendingChnListElem= rValidateRequest(poDrvCmdChnSelect);

//#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
		/*If DAB is in BG, store Station requested to Preset 0*/
		if(dabdrv_main::instance()->enGetSourceState()== DAB_enSourceState_BG)
		{
			trMeca_CRdmStationSetPresetStation rCRdmSetPresetStation;
			rCRdmSetPresetStation.enServiceType=enMeca_ServiceType_AUDIO_SERVICE;
			trMecaProgrammeService rProgSrv=_rPendingChnListElem.rMecaId.rGetProgService();
			rCRdmSetPresetStation.u8StationNumber=0;
			rCRdmSetPresetStation.rProgrammeService=rProgSrv;
			rCRdmSetPresetStation.u32Frequency=dabdrv_chnList::instance()->u32GetFrequencyOfService(rProgSrv);
			rCRdmSetPresetStation.rLabel = _rPendingChnListElem.rLabel;
			dabdrv_mecaIf::instance()->vSendMecaCommand(rCRdmSetPresetStation);

			trMsgSrvRspChnSelect rSrvRsp;
            rSrvRsp.enRes = DAB_enResult_OK;
            DAB_vCallMsg(rSrvRsp);
			
			return;

		}
//#endif
			

		//if(!_rPendingChnListElem.bPSFlag) 
		//	return;
		ETG_TRACE_USR1(("  dabdrv_chnSelect:trMsgDrvCmdChnSelect SID=%x _rPendingChnListElem.u16Scids=%d",
			_rPendingChnListElem.rMecaId._u32Id, _rPendingChnListElem.u16Scids));
        /* store the selected progService:
           in case the selection does not work and we are not in list-mode,
           the next service is automatically selected until success or we have reached 
           _rUsrReqProgSrv again.
        */
        _rUsrReqProgSrv =  _rPendingChnListElem.rMecaId;
        _u16NumSkips = 0;
        if (!_rPendingChnListElem.bIsValid()) {
            // add negative "return-code"
            bChnSelectFlag = FALSE;
            vSetMute();
            //poDrvCmdChnSelect->enRes=DAB_enResult_NOT_AVAILABLE;
            return;
        }
    }
    if(!_rPendingChnListElem.bPSFlag){
    	//incase of secondary service
		//If the state is left out incase of tune frequency label it may play the first service
    	_enChnSelectState = enChnSelectState_Idle;
			return;
    }
    _activeChnSelect = *poDrvCmdChnSelect;

    trMsgDrvCmdChnListPsidMonitor rMonitor; // default: disable
    if (poDrvCmdChnSelect->enChnSelectSource==enChnSelectSource_Lsm && poDrvCmdChnSelect->enRes==DAB_enResult_NOT_AVAILABLE) {
        // enable monitor
        rMonitor.rMecaId=trMecaId(poDrvCmdChnSelect->u32Value);
    }
    DAB_vCallMsg(rMonitor);

	ETG_TRACE_USR1(("  dabdrv_chnSelect:trMsgDrvCmdChnSelect _enChnSelectState=%d",_enChnSelectState));
    _bRetry=FALSE;
    ETG_TRACE_USR1(("  dabdrv_chnSelect:trMsgDrvCmdChnSelect svTimer STOP"));
    _oChnSelectSupervisionTimer.vStop();
    // todo: same short-cuts for comp-select
    switch (_enChnSelectState) {
        case enChnSelectState_Idle:
        case enChnSelectState_TuningFreqLabel:
            vUpdate(enChnSelectState_WaitGo);
            break;
        case enChnSelectState_WaitGo:            
        case enChnSelectState_WaitRdmInfo:
        case enChnSelectState_WaitMute:
        case enChnSelectState_WaitSrvFollow:
            //vPropagateNewSid();
            break;
        case enChnSelectState_Sent:
        case enChnSelectState_SentWait:
            vPropagateNewSid();
            vSendSelectPsid();
            vUpdate(enChnSelectState_Sent);
            break;
        case enChnSelectState_Failed:
            vUpdate(enChnSelectState_WaitGo);
            break;
        default:
            break;
    }
}

tVoid dabdrv_chnSelect::vProcess(trMsgDrvCmdEnsSelect* poDrvCmdEnsSelect)
{
	ETG_TRACE_USR4(("dabdrv_chnSelect::vProcess(trMsgDrvCmdEnsSelect):"));
         _rPendingChnListEnsembleElem= rValidateRequest(poDrvCmdEnsSelect);		
         vSendSelectEnsemble();
}

tVoid dabdrv_chnSelect::vSendSelectEnsemble() const {
	ETG_TRACE_USR4(("dabdrv_chnSelect::vSendSelectEnsemble:"));
	trMeca_CRdmTuneEid rCRdmTuneEid;
	rCRdmTuneEid.rEnsemble =_rPendingChnListEnsembleElem.rEnsemble;
    dabdrv_mecaIf::instance()->vSendMecaCommand(rCRdmTuneEid);
    //Coverity 16363
	//DAB_vCallMsgCtor(trMsgDrvIndRadioText(trRadioText()));
	trRadioText rRadioText;
    trMsgDrvIndRadioText oMsgDrvIndRadioText;
    oMsgDrvIndRadioText.rRadioText = rRadioText;
    DAB_vCallMsg(oMsgDrvIndRadioText);
}




// todo: timer-supervision if we don't get the correct psid or the final result enMeca_RdmServiceState_SERVICE_SELECTION_DONE
// todo: timer-supervision for advisorys like "searching"
tVoid dabdrv_chnSelect::vProcess(trMeca_RRdmAudioPlayAsid* poMecaRRdmPlayPsid) {

    ETG_TRACE_USR1(("  dabdrv_chnSelect:vProcess(trMeca_RRdmAudioPlayAsid) PSID=%08x  enServiceState=%d",
                        poMecaRRdmPlayPsid->rMecaProgrammeService._u32Id,
                        ETG_CENUM(tenMeca_RdmServiceState,poMecaRRdmPlayPsid->enServiceState)));
    

	if(ChannelRecall == dabdrv_main::instance()->enGetPresetOrChannelRecallRequested()){
		if (poMecaRRdmPlayPsid->rMecaProgrammeService != _rPendingChnListElem.rMecaId) {
			return;
		}
	}

    switch (poMecaRRdmPlayPsid->enServiceState) {
    	case enMeca_RdmServiceState_WAITING_FOR_SERVICE:
			if(_otrMsgRspPlayAsidSupervisionTimer.bIsRunning()){
				_otrMsgRspPlayAsidSupervisionTimer.vStop();
			}
    		_enChnSelectStateOrig = enChnSelectState_WaitGo;
        break;
        case enMeca_RdmServiceState_SERVICE_SELECTION_DONE:
		{
			ETG_TRACE_USR1(("  dabdrv_chnSelect:trMeca_RRdmAudioPlayAsid SELECTION_DONE svTimer STOP"));
			//trMeca_CDbServiceGetInfo rMecaCServiceGetInfo;
			//rMecaCServiceGetInfo.rProgrammeService = poMecaRRdmPlayPsid->rMecaProgrammeService;
			//dabdrv_mecaIf::instance()->vSendMecaCommand(rMecaCServiceGetInfo);

			char const *label = dabdrv_chnList::instance()->rGetChnLabel(poMecaRRdmPlayPsid->rMecaProgrammeService).pcGetCString();
			rPgmService = poMecaRRdmPlayPsid->rMecaProgrammeService;
			if (((DAB_CHNINFO_UPDATE_SUPERVISION_TIMER_MS)>0) && (dabdrv_chnInfo::instance()->bCompareIdLabel(poMecaRRdmPlayPsid->rMecaProgrammeService._u32Id, label))) {
					ETG_TRACE_USR4(("  dabdrv_chnSelect:trMeca_RRdmAudioPlayAsid SELECTION_DONE ChnInfoUpdateSuperVisionTimer STARTS"));
					_oChnInfoUpdateSuperVisionTimer.vStart(DAB_CHNINFO_UPDATE_SUPERVISION_TIMER_MS);
				}
				else{
					DAB_trChnInfoProperty rChnInfoProperty=dabdrv_properties::instance()->oChnInfoProperty.oGet();
					if(rChnInfoProperty.rMecaId == poMecaRRdmPlayPsid->rMecaProgrammeService._u32Id){
						if(!rChnInfoProperty.bComponentMode){
							rChnInfoProperty.rLabel = dabdrv_chnList::instance()->rGetChnLabel(rChnInfoProperty.rMecaId);
						}
						dabdrv_properties::instance()->oChnInfoProperty.vSet(rChnInfoProperty);
				
					}
					vUpdate(enChnSelectState_Idle);
					_enChnSelectStateOrig = enChnSelectState_Idle;
					_otrMsgRspPlayAsidSupervisionTimer.vStart(DAB_PLAY_ASID_SUPERVISION_TIMER_MS);
					//vSetMute();
			
					#ifdef MUTE_ACTIVE       
					/*if timer is running for an older service selection and if a new request comes,stop timer and start delay again*/
					if(_oMuteSupervisionTimer.bIsRunning()){
						_oMuteSupervisionTimer.vStop();
					}
						_oMuteSupervisionTimer.vStart(DAB_MUTE_SUPERVISION_TIMER_MS);
					#endif
				}
            break;
        }
        case enMeca_RdmServiceState_SERVICE_INVALID:
        case enMeca_RdmServiceState_PSID_NOT_AVAILABLE:
        {
        	_enChnSelectStateOrig = enChnSelectState_Idle;
        	vSetMute();
#ifdef MUTE_ACTIVE
            break;
#else
            vSetAutoSkip();
            break;
#endif
        }

        default:
        {
        	vSetAutoSkip();
           break;
        }
    }

}

tVoid dabdrv_chnSelect::vProcess(trMsgChnInfoUpdateTimer *poMsgChnInfoUpdateTimerSupervisionTimer) {
	(tVoid)poMsgChnInfoUpdateTimerSupervisionTimer;
	ETG_TRACE_USR4(("dabdrv_chnSelect::inside trMsgChnInfoUpdateTimerSupervisionTimer"));
	dabdrv_chnInfo::instance()->bUpdateProperty(rPgmService, TRUE);
	vUpdate(enChnSelectState_Idle);
	_enChnSelectStateOrig = enChnSelectState_Idle;
	_otrMsgRspPlayAsidSupervisionTimer.vStart(DAB_PLAY_ASID_SUPERVISION_TIMER_MS);
	//vSetMute();

#ifdef MUTE_ACTIVE       
	/*if timer is running for an older service selection and if a new request comes,stop timer and start delay again*/
	if (_oMuteSupervisionTimer.bIsRunning()) {
		_oMuteSupervisionTimer.vStop();
	}
	_oMuteSupervisionTimer.vStart(DAB_MUTE_SUPERVISION_TIMER_MS);
#endif
}

tVoid dabdrv_chnSelect::vProcess(trMeca_RRdmTuneEid* poMecaRRdmTuneEid) {

	if (poMecaRRdmTuneEid->rEnsemble != _rPendingChnListEnsembleElem.rEnsemble)
	{
		return;
	}

	switch (poMecaRRdmTuneEid->enEnsembleState) {
		case enMeca_RdmEnsembleState_TUNING_DONE:
			{
				ETG_TRACE_USR1(("  dabdrv_chnSelect:trMeca_RRdmTuneEid SELECTION_DONE ")); 
				trMsgDrvCmdSetEnsembleId rDrvCmdSetEnsembleId(poMecaRRdmTuneEid->rEnsemble);
				DAB_vCallMsg(rDrvCmdSetEnsembleId);

				break;
			}
		case enMeca_RdmEnsembleState_TUNING_ENSEMBLE: 
			{
				ETG_TRACE_USR1(("  dabdrv_chnSelect:trMeca_RRdmTuneEid SEARCH FOR ENSEMBLE RUNNING "));        

				break;
			}            
		case enMeca_RdmEnsembleState_ENSEMBLE_NOT_FOUND:
			{
				ETG_TRACE_USR1(("  dabdrv_chnSelect:trMeca_RRdmTuneEid ENSEMBLE NOT FOUND "));        

				break;
			}           
		case enMeca_RdmEnsembleState_ENSEMBLE_INVALID:
			{
				ETG_TRACE_USR1(("  dabdrv_chnSelect:trMeca_RRdmTuneEid ENSEMBLE INVALID "));        

				break;
			}           
		default:      

			break;
	}
}
tVoid dabdrv_chnSelect::vProcess(trMeca_RRdmTuneFrequencyLabel* poRRdmTuneFreqLabel) {
    ETG_TRACE_USR4(("  dabdrv_chnSelect::vProcess(trMeca_RRdmTuneFrequencyLabel) b_frequency_state=%d",
    		poRRdmTuneFreqLabel->enFrequencyState));


    vResetChnInfoProperty();
	trMsgSrvCmdRspTuneFreqlabel rFreqLabelRsp;
    if(poRRdmTuneFreqLabel->enFrequencyState != enMeca_RdmFrequencyState_FREQUENCY_OK){

    	rFreqLabelRsp.enRes = DAB_enResult_FAILED;
    	_enChnSelectState = enChnSelectState_Idle;
    }
    DAB_vCallMsg(rFreqLabelRsp);
}

tVoid dabdrv_chnSelect::vResetChnInfoProperty()
{
    //reset the channel info property value PSARCC30-1573
    DAB_trChnInfoProperty rProperty=dabdrv_properties::instance()->oChnInfoProperty.oGet();
    rProperty.rMecaId._u32Id=0;
    rProperty.u8Scids=0;
    rProperty.enChnState=enChnState_Unavail;
    rProperty.rRadioText=trRadioText();
    dabdrv_properties::instance()->oChnInfoProperty.vSet(rProperty);
}

tVoid dabdrv_chnSelect::vProcess(trMsgDrvFmServiceFollow* poFmServiceFollow) {
    ETG_TRACE_USR4(("  dabdrv_chnSelect::vProcess(trMsgDrvFmServiceFollow) bFmSrvFollowActive=%d",
                        poFmServiceFollow->bFmSrvFollowActive));
    if (poFmServiceFollow->bFmSrvFollowActive) {
        vUpdate(enChnSelectState_Idle);
    }
}
tVoid dabdrv_chnSelect::vProcess(trMeca_RRdmServFollowActive* poServFollowActive) {
    ETG_TRACE_USR4(("  dabdrv_chnSelect::vProcess(trMeca_RRdmServFollowActive) enServFollowState=%d u8ServFollowMode=%x",
                        ETG_CENUM(tenMeca_RdmServFollowState, poServFollowActive->enServFollowState),
                        poServFollowActive->u8ServFollowMode));
    if( poServFollowActive->enServFollowState == enMeca_RdmServFollowState_STATE_END ) {
        _bDabDabSrvFollowActive = FALSE;
    } else {
        _bDabDabSrvFollowActive = TRUE;
		// PSARCC30-2588
        //When Service following happens with different SID
        //then channel name is not changing, as the update rdminfo is blcoked
        // incase select source is preset_recall or chnl_Select. hence reset the selected source for hard linking.
        if(poServFollowActive->u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_HARD_LINKING){
        	dabdrv_presets::instance()->vResetSelectSource();
        	dabdrv_main::instance()->vSetPresetOrChannelRecallRequested(PresetOrChannelRecallInvalid);
        }
    }
}
tVoid dabdrv_chnSelect::vProcess(trMsgDrvCmdSourceState *poSourceState) {
    if (_enSourceState == poSourceState->enSourceState){
        return;
    }
    _enSourceState = poSourceState->enSourceState;
    vUpdate(enChnSelectState_Idle);
    return;
}

tVoid dabdrv_chnSelect::vProcess(trMsgDrvCmdRdmTune* poDrvCmdRdmTune) {


	ETG_TRACE_USR1((" dabdrv_chnSelect::trMsgDrvCmdRdmTune "));

	_enTuneType = poDrvCmdRdmTune->enTuneType;

	trMeca_CRdmTune oMsgMecaTune;

	oMsgMecaTune.enTuneCmd = enMeca_RdmTuneCmd_ENS;
	oMsgMecaTune.u32StartFrequency = 0;
	oMsgMecaTune.enDir = poDrvCmdRdmTune->enDirection;

	dabdrv_mecaIf::instance()->vSendMecaCommand(oMsgMecaTune);
	//Coverity 16363
	//DAB_vCallMsgCtor(trMsgDrvIndRadioText(trRadioText()));
	trRadioText rRadioText;
	trMsgDrvIndRadioText oMsgDrvIndRadioText;
	oMsgDrvIndRadioText.rRadioText = rRadioText;
	DAB_vCallMsg(oMsgDrvIndRadioText);

	poDrvCmdRdmTune->enRes = DAB_enResult_OK;
}


void dabdrv_chnSelect::vSetMute()
{
#ifdef MUTE_ACTIVE
	DAB_trTunerStatusProperty oTunerStatus = dabdrv_properties::instance()->oTunerStatusProperty.oGet();
	tenMuteOperation enMuteOp=dabdrv_main::instance()->enGetMuteOperation();
	ETG_TRACE_USR1(("  dabdrv_chnSelect:vSetMute enMuteOp(%d)",enMuteOp));

	if((enMuteOp==enMuteOperation_ServiceSeek) ||
			(enMuteOp==enMuteOperation_LoadFromList) ||
			(enMuteOp==enMuteOperation_PresetLoad) ||
			(enMuteOp==enMuteOperation_EnsembleSeek) ||
			enMuteOp==enMuteOperation_TuneFreqLabel){
		ETG_TRACE_USR1(("  dabdrv_chnSelect:vSetMute enMuteOp(%d)",enMuteOp));
	}
	else{
			return;
	}

	tenMuteOperation enNextMuteOp=dabdrv_main::instance()->enGetNextMuteOperation();

	ETG_TRACE_USR1(("  dabdrv_chnSelect:vSetMute enNextMuteOp(%d)",enNextMuteOp));

	if(enNextMuteOp != enMuteOperation_None){
		/*As mute actions are no more sequential, next mute operation would already be handled if it comes */
		dabdrv_main::instance()->enSetNextMuteOperation(enMuteOperation_None);
		
	}
	
		trMsgDrvCmdTunerOpMuteReq rMuteReq;
		rMuteReq.enRequiredMuteState=enRequiredMuteState_Demute;
		//rMuteReq.enMuteOperation=enMuteOp;
		DAB_vCallMsg(rMuteReq);

        if (enMuteOp == enMuteOperation_LoadFromList || enMuteOp == enMuteOperation_ServiceSeek)
        {
            trMsgSrvRspChnSelect rSrvRsp;
            rSrvRsp.enRes = bChnSelectFlag ? DAB_enResult_OK : DAB_enResult_NOT_AVAILABLE;
            DAB_vCallMsg(rSrvRsp);
            bChnSelectFlag = TRUE;            
        }

		if (enMuteOp != enMuteOperation_PresetLoad) // Activity type should not be changed to normal for preset recall
		{			
			oTunerStatus.enTunerActivityType = DAB_enTunerActivityType_NORMAL;
			dabdrv_properties::instance()->oTunerStatusProperty.vSet(oTunerStatus);
			ETG_TRACE_USR1(("dabdrv_chnSelect:vSetMute Sent Demute"));
		}
		dabdrv_main::instance()->enSetMuteOperation(enMuteOperation_None);
#endif
}

tVoid dabdrv_chnSelect::vCheckForMute(tBool bStatus){
	ETG_TRACE_USR1(("dabdrv_chnSelect::vCheckForMute bStatus = %d _enChnSelectStateOrig %d",bStatus,_enChnSelectStateOrig));
	if(!bStatus)
		return;
	//PSARCC21-3068 when selected preset is not available and the same is available in FM
	//Then DAB-FM linking happens but audio is muted due delayed response from stationPlay.

	 //No need to check for Back ground as if it is in back ground it will never come to wait go state
	if(_enChnSelectStateOrig == enChnSelectState_WaitGo){
		vSetMute();
	}
}

void dabdrv_chnSelect::vProcess(trMeca_RRdmTune* poRRdmTune) {
    ETG_TRACE_USR1(("dabdrv_chnSelect::trMeca_RRdmTune"));

    if(_enTuneType!=DAB_enTuneType_TUNE_ENSEMBLE){
        return;
    }
	
	DAB_trTunerStatusProperty oTunerStatus = dabdrv_properties::instance()->oTunerStatusProperty.oGet();
    
    if (poRRdmTune->enTuneState== enMeca_RdmTuneState_SEARCHING_DONE ||
        poRRdmTune->enTuneState== enMeca_RdmTuneState_TUNING_DONE) {

    	ETG_TRACE_USR1(("  dabdrv_chnSelect:trMeca_RRdmTune TUNING DONE "));
#ifdef DAB_ENABLE_MUTE_HANDLING
    	if(dabdrv_main::instance()->enGetMuteOperation() != enMuteOperation_EnsembleSeek){
#endif
			/* tune to first service of this ensemble*/			
			trMsgSrvCmdTestOperation rMsg;
			rMsg.enTestOperation=DAB_enTestOperation_TUNE_SERVICE;
			rMsg.u32Value1= (tU32)enMeca_RdmServiceCommand_PLAY_FIRST;			
			DAB_vCallMsg(rMsg);
    		oTunerStatus.enTunerActivityType = DAB_enTunerActivityType_NORMAL;
			dabdrv_properties::instance()->oTunerStatusProperty.vSet(oTunerStatus);
			return;
#ifdef DAB_ENABLE_MUTE_HANDLING
    	}
    	vSetMute();
#endif
    }
    else if (poRRdmTune->enTuneState== enMeca_RdmTuneState_TUNING ||
        poRRdmTune->enTuneState== enMeca_RdmTuneState_SEARCHING ||
        poRRdmTune->enTuneState== enMeca_RdmTuneState_SEARCHING_ENS) {

    	ETG_TRACE_USR4(("  dabdrv_chnSelect:trMeca_RRdmTune TUNING going on "));
		oTunerStatus.enTunerActivityType = DAB_enTunerActivityType_SEEK;

		dabdrv_properties::instance()->oTunerStatusProperty.vSet(oTunerStatus);
    }
    else if(poRRdmTune->enTuneState== enMeca_RdmTuneState_STOP)
    {
    	vSetMute();
    }
}
#ifdef MUTE_ACTIVE
void dabdrv_chnSelect::vProcess(trMsgDrvCmdTuneFreqlabel* poTuneFreqLabel) {
    ETG_TRACE_USR1(("dabdrv_chnSelect::trMsgSrvCmdTuneFreqlabel"));
    trMeca_CRdmTuneFrequencyLabel oMsgMecaLabel;
	ETG_TRACE_USR4(("  trMsgSrvCmdTuneFreqlabel:sFrequencyLabel=%s",poTuneFreqLabel->sfreqlabel.c_str()));
    oMsgMecaLabel.sfreqlabel = poTuneFreqLabel->sfreqlabel;
    dabdrv_mecaIf::instance()->vSendMecaCommand(oMsgMecaLabel);

    _enChnSelectState = enChnSelectState_TuningFreqLabel;

    //Before this operation if preset is recalled the state is not changed, hence reset the state.
    dabdrv_presets::instance()->vResetSelectSource();
}
#endif
void dabdrv_chnSelect::vSetAutoSkip(){
#ifdef DAB_CHN_SELCET_ENABLE_AUTO_SKIP
            if (!dabdrv_chnList::instance()->bIsFrozenListOpen() && _activeChnSelect.enChnSelectMode==enChnSelectMode_Relative) {
                /*
                  automatically select next station,
                  give up, if we have reached initial service again
                  remove channel-name from channel-info
                  A complete scan has already been conducted because in the first try the service follow
                  mode search had been enabled. For all further elements in the list the search will be disabled.

                */
                trMsgDrvCmdChnSelect rNewChnSelect(_activeChnSelect);
                rNewChnSelect.enChnSelectSource=enChnSelectSource_ChnSelect;
                rNewChnSelect.enChnSelectMode=enChnSelectMode_Relative;
                _u16NumSkips++;
                rNewChnSelect.u32Value+=_u16NumSkips;

                trChnListElem rNewChn =rValidateRequest(&rNewChnSelect);
                if (rNewChn.bIsValid() && _rUsrReqProgSrv != _rPendingChnListElem.rMecaId && _u16NumSkips<=0xFF) {
                    _rPendingChnListElem=rNewChn;
                    vSendSelectPsid();
                    vUpdate(enChnSelectState_Sent);
                    break;
                }
            }
#endif
}
