/************************************************************************
 * FILE:        dabdrv_activity.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabdrv_activity
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/


#ifndef DABDRV_ACTIVITY_HPP
#define DABDRV_ACTIVITY_HPP

#include "dabdrv_main.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_DRV_MAIN 
#include "trcGenProj/Header/dabdrv_activity.hpp.trc.h"
#endif
/* each message-handler can have one activity.
   An activity can be requested by the message-handler-
   If no activity with higher or equal priority is running, the requested activity will be started.
   If an activity with lower priority is running it will be stopped if it is interruptible.

   dabdrv_main is strongly linked to dabdrv_main. 
   dabdrv_main will be informed about requests to start an activity and indication that an activity is done:
   vOnActivityRequest() and vOnActivityDone()

*/
namespace DAB {

    // the priority
    typedef enum {
        enActivityPrio_Invalid,
        enActivityPrio_Lowest,
        enActivityPrio_Low,
        enActivityPrio_Normal,
		/**vnd4kor: supressing prio2 lint warning*/
		//lint -esym(793,*identifiers in one block*)
        enActivityPrio_High,
        enActivityPrio_Higher,
        enActivityPrio_Highest
    } tenActivityPrio;

    class dabdrv_activityBase {
        // Base-class of all activities
    public:
        dabdrv_activityBase():
            _enActivityState(DAB_enDrvActivityState_Idle),
            _enPrio(enActivityPrio_Normal),
            _bChangesChannel(TRUE),
            _bSetsChannel(FALSE)
        {}

        virtual DAB_tenDrvActivity enGetActivity()=0;
        DAB_tenDrvActivityState enGetState() const {
            return _enActivityState;
        }

        tenActivityPrio enGetPrio() const {return _enPrio;};
        tVoid vSetPrio(tenActivityPrio enPrio)  {_enPrio=enPrio;};

        tBool bGetChangesChannel() const {return _bChangesChannel;};
        tVoid vSetChangesChannel(tBool bChangesChannel)  {_bChangesChannel=bChangesChannel;};

        tBool bGetSetsChannel() const {return _bSetsChannel;};
        tVoid vSetSetsChannel(tBool bSetsChannel)  {_bSetsChannel=bSetsChannel;};

        tVoid vRequest() {
            if (bSetState(DAB_enDrvActivityState_Pending)) {
                ETG_TRACE_USR4(("dabdrv_activityBase::vRequest():%d",
                                ETG_CENUM(DAB_tenDrvActivity, enGetActivity())));
                dabdrv_main::instance()->vOnActivityRequest(enGetActivity());
            } 
        }
        tVoid vTerminated(tBool bSuccess=TRUE) {
            if (bSetState(DAB_enDrvActivityState_Idle)) {
                ETG_TRACE_USR4(("dabdrv_activityBase::vTerminated():%d",
                                ETG_CENUM(DAB_tenDrvActivity, enGetActivity())));
                dabdrv_main::instance()->vOnActivityDone(enGetActivity(), bSuccess);
            }
        }

        virtual tBool bRun()=0;
        virtual tBool bTerminate() {return FALSE;};
        virtual tBool bIsInterruptible() {return FALSE;};

    protected:
        tBool bSetState(DAB_tenDrvActivityState enActivityState) {
            if (_enActivityState != enActivityState) {
                ETG_TRACE_USR4(("dabdrv_activityBase::bSetState(%d):%d -> %d",
                                ETG_CENUM(DAB_tenDrvActivity, enGetActivity()),
                                ETG_CENUM(DAB_tenDrvActivityState, _enActivityState),
                                ETG_CENUM(DAB_tenDrvActivityState, enActivityState)));
                _enActivityState = enActivityState;
                return TRUE;
            }
            return FALSE;
        }
        DAB_tenDrvActivityState _enActivityState;
        tenActivityPrio _enPrio;
        tBool _bChangesChannel; // TRUE if the audible channel has to be restored when activity is done
        tBool _bSetsChannel;   // TRUE if activity sets a new channel. (restores)

    };
        
    /* template for a non-interruptible activity.
       Even if a activity with higher priority is requested, the running activity can 
       not be termitated */
    template<class D, int ACTIVITY>
    class dabdrv_activity:
        public dabdrv_activityBase{
        
    public:


        virtual DAB_tenDrvActivity enGetActivity() {
            return (DAB_tenDrvActivity)ACTIVITY;
        }


        virtual tBool bRun() {
            if (bSetState(DAB_enDrvActivityState_Running)) {
                ETG_TRACE_USR4(("dabdrv_activityBase::bRun():%d",
                                ETG_CENUM(DAB_tenDrvActivity, enGetActivity())));
                return D::instance()->bRunActivity();
            }
            // already running:
            return TRUE;
        }



    };



    /* template for interruptible activity.
       If a activity with higher priority is requested, the running activity will 
       be requested to terminate via  bTerminate*/
    template<class D, int ACTIVITY>
    class dabdrv_activityIr: public dabdrv_activity<D, ACTIVITY> {
    public:
        
        virtual tBool bTerminate() {
            if  (this->bSetState(DAB_enDrvActivityState_Terminating)) {
                ETG_TRACE_USR4(("dabdrv_activityIr::bTerminate():%d",
                                ETG_CENUM(DAB_tenDrvActivity, this->enGetActivity())));
                return D::instance()->bTerminateActivity();
            }
            // already terminating
            return TRUE;
        }
        virtual tBool bIsInterruptible() {
            return TRUE;
        }
        


    };

}



#endif
