/**
 * @copyright (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 * @addtogroup NavMiddleware
 * @{
 */

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_WEATHERINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_WEATHERINFOS_H_

#include <list>
#include <string>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{

enum WeatherReceptionState
{
   WEATHER_RECEPTION_STATE__UNKNOWN = 0,
   WEATHER_RECEPTION_STATE__OK,
   WEATHER_RECEPTION_STATE__INACTIVE,
   WEATHER_RECEPTION_STATE__NO_INTERNET_CONNECTION,
   WEATHER_RECEPTION_STATE__AUTHENTICATION_INVALID,
   WEATHER_RECEPTION_STATE__CONNECTION_LOST,
   WEATHER_RECEPTION_STATE__SUBSCRIPTION_EXPIRED,
   WEATHER_RECEPTION_STATE__NOT_PROVIDED
};

inline ::std::string toString(WeatherReceptionState weatherReceptionState)
{
   switch (weatherReceptionState)
   {
   case WEATHER_RECEPTION_STATE__UNKNOWN:
      return "WEATHER_RECEPTION_STATE__UNKNOWN";
   case WEATHER_RECEPTION_STATE__OK:
      return "WEATHER_RECEPTION_STATE__OK";
   case WEATHER_RECEPTION_STATE__INACTIVE:
      return "WEATHER_RECEPTION_STATE__INACTIVE";
   case WEATHER_RECEPTION_STATE__NO_INTERNET_CONNECTION:
      return "WEATHER_RECEPTION_STATE__NO_INTERNET_CONNECTION";
   case WEATHER_RECEPTION_STATE__AUTHENTICATION_INVALID:
      return "WEATHER_RECEPTION_STATE__AUTHENTICATION_INVALID";
   case WEATHER_RECEPTION_STATE__CONNECTION_LOST:
      return "WEATHER_RECEPTION_STATE__CONNECTION_LOST";
   case WEATHER_RECEPTION_STATE__SUBSCRIPTION_EXPIRED:
      return "WEATHER_RECEPTION_STATE__SUBSCRIPTION_EXPIRED";
   case WEATHER_RECEPTION_STATE__NOT_PROVIDED:
      return "WEATHER_RECEPTION_STATE__NOT_PROVIDED";
   default:
      return "<unknown>";
   }
}

class WeatherConfigurationInformation
{
public:
   WeatherConfigurationInformation()
      : m_timeInterval(0),
        m_numberOfHours(0)
   {}

   ~WeatherConfigurationInformation() {}

   void setTimeInterval(unsigned int timeInterval)
   {
      m_timeInterval = timeInterval;
   }

   unsigned int getTimeInterval() const
   {
      return m_timeInterval;
   }

   void setNumberOfHours(unsigned int numberOfHours)
   {
      m_numberOfHours = numberOfHours;
   }

   unsigned int getNumberOfHours() const
   {
      return m_numberOfHours;
   }

   const ValidValue<GeoCoordinateDegree>& getCoordinate() const
   {
      return m_geoCoordinate;
   }

   void setCoordinate(const ValidValue<GeoCoordinateDegree>& geoCoordinate)
   {
      m_geoCoordinate = geoCoordinate;
   }

   const ValidValue< unsigned int>& getDestinationIndex() const
   {
      return m_destinationIndex;
   }

   void setDestinationIndex( const ValidValue< unsigned int>& destinationIndex)
   {
      m_destinationIndex = destinationIndex;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "\tm_timeInterval: " << m_timeInterval << ::std::endl
             << "\tm_numberOfHours: "<< m_numberOfHours << ::std::endl
             << "\tm_destinationIndex: "<< m_destinationIndex.toString() << ::std::endl;
      if(m_geoCoordinate.isValid())
      {
         stream << m_geoCoordinate.getValue().getLatitude() << ::std::endl
                << m_geoCoordinate.getValue().getLongitude() << ::std::endl;
      }

      return stream.str();
   }

private:
   /**
    * if requested ReportType is TIMED then client has to provide the detail for how many next hours report is required
    * and TimeInterval between each report.Example(if client requires report for the next 12h and with the time interval
    *  of 2h, so in the response total of 6 reports will be provided.)
    */
   unsigned int m_timeInterval;
   unsigned int m_numberOfHours;
   /**
    * For the location type Arbitrary, coordinate is mandatory
    */
   ValidValue<GeoCoordinateDegree> m_geoCoordinate;

   /**
    * If the weather data is required for intermediate destinations, destination index is mandatory.
    * If the destination index is not provided, LocationType_Destination will be considered as final destination.
    */
   ValidValue< unsigned int> m_destinationIndex;
};

class ReportConfiguration
{
public:
   ReportConfiguration()
      :m_weatherReportType(WEATHER_REPORT_TYPE__OVERVIEW),
       m_locationType(LOCATION_TYPE__CURRENT_VEHICLE_POSITION)
   {}

   ~ReportConfiguration() {}

   void setWeatherReportType(WeatherReportType weatherReportType)
   {
      m_weatherReportType = weatherReportType;
   }

   WeatherReportType getWeatherReportType() const
   {
      return m_weatherReportType;
   }

   void setLocationType(LocationType locationType)
   {
      m_locationType = locationType;
   }

   LocationType getLocationType() const
   {
      return m_locationType;
   }

   void setWeatherConfigurationInformation(const WeatherConfigurationInformation& weatherConfigurationInformation)
   {
      m_weatherConfigurationInformation = weatherConfigurationInformation;
   }

   const WeatherConfigurationInformation& getWeatherConfigurationInformation() const
   {
      return m_weatherConfigurationInformation;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "\tm_weatherReportType: " << ::navmiddleware::toString(m_weatherReportType) << ::std::endl
             << "\tm_locationType: "<< ::navmiddleware::toString(m_locationType) << ::std::endl
             << "\tm_weatherConfigurationInformation: " << m_weatherConfigurationInformation.toString() << ::std::endl;

      return stream.str();
   }

private:
   WeatherReportType m_weatherReportType;
   LocationType m_locationType;
   WeatherConfigurationInformation m_weatherConfigurationInformation;
};

class WeatherDataInfos
{
public:
   WeatherDataInfos()
      : m_requestId(0) { }

   ~WeatherDataInfos() { }

   void setRequestId(RequestId requestId)
   {
      m_requestId = requestId;
   }

   RequestId getRequestId() const
   {
      return m_requestId;
   }

   void setWeatherDataInfoList(const ::std::vector<WeatherDataInfo>& weatherDataInfoList)
   {
      m_weatherDataInfoList = weatherDataInfoList;
   }

   const ::std::vector<WeatherDataInfo>& getWeatherDataInfoList() const
   {
      return m_weatherDataInfoList;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "WeatherDataInfo payload:" << ::std::endl
             << "\tm_requestId: " << m_requestId << ::std::endl;

      for (size_t i = 0; i < m_weatherDataInfoList.size(); i++)
      {
         stream << "\t-> " << m_weatherDataInfoList[i].toString();
      }

      return stream.str();
   }

   bool operator==(const WeatherDataInfos& rhs) const
   {
      return (m_requestId == rhs.m_requestId) &&
             (m_weatherDataInfoList == rhs.m_weatherDataInfoList);
   }

   bool operator!=(const WeatherDataInfos& rhs)const
   {
      return !(*this == rhs);
   }

private:
   RequestId m_requestId;
   ::std::vector<WeatherDataInfo> m_weatherDataInfoList;
};

}
#endif // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_WEATHERINFOS_H_

