/**
 * @copyright (c) 2015-2020 Robert Bosch Car Multimedia GmbH
 * @addtogroup NavMiddleware
 * @{
 */

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_TRAFFICMESSAGEINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_TRAFFICMESSAGEINFOS_H_

#include <string>
#include <sstream>
#include <vector>
#include <list>
#include <utility>
#include <map>
#include "InfoTypes.h"
#include "TrafficSettings.h"

namespace navmiddleware
{

enum TrafficMessageListType
{
   TRAFFICMESSAGELIST_TYPE__ALL_TM = 0,
   TRAFFICMESSAGELIST_TYPE__ALONG_CURRENT_ROUTE
};

inline ::std::string toString(TrafficMessageListType listType)
{
   switch (listType)
   {
   case TRAFFICMESSAGELIST_TYPE__ALL_TM:
      return "TRAFFICMESSAGELIST_TYPE__ALL_TM";
   case TRAFFICMESSAGELIST_TYPE__ALONG_CURRENT_ROUTE:
      return "TRAFFICMESSAGELIST_TYPE__ALONG_CURRENT_ROUTE";
   default:
      return "<unknown>";
   }
}

enum TrafficMessageTextType
{
   TRAFFICMESSAGE_TEXT_TYPE__INVALID = 0,
   TRAFFICMESSAGE_TEXT_TYPE__DETAILED,
   TRAFFICMESSAGE_TEXT_TYPE__LIST,
   TRAFFICMESSAGE_TEXT_TYPE__DETAILED_WITH_HEADING,
   TRAFFICMESSAGE_TEXT_TYPE__DETAILED_WITH_SHORTENED_EVENT_NAME
};

inline ::std::string toString(TrafficMessageTextType textType)
{
   switch (textType)
   {
   case TRAFFICMESSAGE_TEXT_TYPE__INVALID:
      return "TRAFFICMESSAGE_TEXT_TYPE__INVALID";
   case TRAFFICMESSAGE_TEXT_TYPE__DETAILED:
      return "TRAFFICMESSAGE_TEXT_TYPE__DETAILED";
   case TRAFFICMESSAGE_TEXT_TYPE__LIST:
      return "TRAFFICMESSAGE_TEXT_TYPE__LIST";
   case TRAFFICMESSAGE_TEXT_TYPE__DETAILED_WITH_HEADING:
      return "TRAFFICMESSAGE_TEXT_TYPE__DETAILED_WITH_HEADING";
   case TRAFFICMESSAGE_TEXT_TYPE__DETAILED_WITH_SHORTENED_EVENT_NAME:
      return "TRAFFICMESSAGE_TEXT_TYPE__DETAILED_WITH_SHORTENED_EVENT_NAME";
   default:
      return "<unknown>";
   }
}

enum TrafficMessageUserRule
{
   TRAFFICMESSAGEUSERRULE__NONE = 0,
   TRAFFICMESSAGEUSERRULE__BLOCK,
   TRAFFICMESSAGEUSERRULE__IGNORE
};

inline ::std::string toString(TrafficMessageUserRule userRule)
{
   switch (userRule)
   {
   case TRAFFICMESSAGEUSERRULE__NONE:
      return "TRAFFICMESSAGEUSERRULE__NONE";
   case TRAFFICMESSAGEUSERRULE__BLOCK:
      return "TRAFFICMESSAGEUSERRULE__BLOCK";
   case TRAFFICMESSAGEUSERRULE__IGNORE:
      return "TRAFFICMESSAGEUSERRULE__IGNORE";
   default:
      return "<unknown>";
   }
}

enum TrafficReceptionState
{
   TRAFFICRECEPTIONSTATE__UNKNOWN = 0,
   TRAFFICRECEPTIONSTATE__OK,                         // TmcTuner and ConnectTraffic
   TRAFFICRECEPTIONSTATE__TRIP_REPLAY_FAILED,
   TRAFFICRECEPTIONSTATE__NOT_SUBSCRIBED,             // TmcTuner
   TRAFFICRECEPTIONSTATE__NOT_AVAILABLE,              // TmcTuner
   TRAFFICRECEPTIONSTATE__NO_ANTENNA,                 // TmcTuner
   TRAFFICRECEPTIONSTATE__NO_SIGNAL,                  // TmcTuner
   TRAFFICRECEPTIONSTATE__TEMPORARILY_NOT_AVAILABLE,  // TmcTuner
   TRAFFICRECEPTIONSTATE__INACTIVE,                   // TmcTuner and ConnectTraffic
   TRAFFICRECEPTIONSTATE__NO_INTERNET_CONNECTION,     // ConnectTraffic
   TRAFFICRECEPTIONSTATE__AUTHENTICATION_INVALID,     // ConnectTraffic
   TRAFFICRECEPTIONSTATE__CONNECTION_LOST,            // ConnectTraffic
   TRAFFICRECEPTIONSTATE__SUBSCRIPTION_EXPIRED,       // ConnectTraffic - based on Config Item for Traffic Activation
   TRAFFICRECEPTIONSTATE__NOT_PROVIDED                // Traffic Source Not Provided
};

inline ::std::string toString(TrafficReceptionState receptionState)
{
   switch (receptionState)
   {
   case TRAFFICRECEPTIONSTATE__UNKNOWN:
      return "TRAFFICRECEPTIONSTATE__UNKNOWN";
   case TRAFFICRECEPTIONSTATE__OK:
      return "TRAFFICRECEPTIONSTATE__OK";
   case TRAFFICRECEPTIONSTATE__TRIP_REPLAY_FAILED:
      return "TRAFFICRECEPTIONSTATE__TRIP_REPLAY_FAILED";
   case TRAFFICRECEPTIONSTATE__NOT_SUBSCRIBED:
      return "TRAFFICRECEPTIONSTATE__NOT_SUBSCRIBED";
   case TRAFFICRECEPTIONSTATE__NOT_AVAILABLE:
      return "TRAFFICRECEPTIONSTATE__NOT_AVAILABLE";
   case TRAFFICRECEPTIONSTATE__NO_ANTENNA:
      return "TRAFFICRECEPTIONSTATE__NO_ANTENNA";
   case TRAFFICRECEPTIONSTATE__NO_SIGNAL:
      return "TRAFFICRECEPTIONSTATE__NO_SIGNAL";
   case TRAFFICRECEPTIONSTATE__TEMPORARILY_NOT_AVAILABLE:
      return "TRAFFICRECEPTIONSTATE__TEMPORARILY_NOT_AVAILABLE";
   case TRAFFICRECEPTIONSTATE__INACTIVE:
      return "TRAFFICRECEPTIONSTATE__INACTIVE";
   case TRAFFICRECEPTIONSTATE__NO_INTERNET_CONNECTION:
      return "TRAFFICRECEPTIONSTATE__NO_INTERNET_CONNECTION";
   case TRAFFICRECEPTIONSTATE__AUTHENTICATION_INVALID:
      return "TRAFFICRECEPTIONSTATE__AUTHENTICATION_INVALID";
   case TRAFFICRECEPTIONSTATE__CONNECTION_LOST:
      return "TRAFFICRECEPTIONSTATE__CONNECTION_LOST";
   case TRAFFICRECEPTIONSTATE__SUBSCRIPTION_EXPIRED:
      return "TRAFFICRECEPTIONSTATE__SUBSCRIPTION_EXPIRED";
   case TRAFFICRECEPTIONSTATE__NOT_PROVIDED:
      return "TRAFFICRECEPTIONSTATE__NOT_PROVIDED";
   default:
      return "<unknown>";
   }
}

enum TrafficMessageSource
{
   TRAFFIC_MESSAGE_SOURCE__UNKOWN = 0,
   TRAFFIC_MESSAGE_SOURCE__CONNECT_TRAFFIC,
   TRAFFIC_MESSAGE_SOURCE__TMC_TUNER,
   TRAFFIC_MESSAGE_SOURCE__TRAFFIC_TRIP_REPLAY
};

inline ::std::string toString(TrafficMessageSource trafficMessageSource)
{
   switch(trafficMessageSource)
   {
   case TRAFFIC_MESSAGE_SOURCE__UNKOWN:
      return "TRAFFIC_MESSAGE_SOURCE__UNKOWN";
   case TRAFFIC_MESSAGE_SOURCE__CONNECT_TRAFFIC:
      return "TRAFFIC_MESSAGE_SOURCE__CONNECT_TRAFFIC";
   case TRAFFIC_MESSAGE_SOURCE__TMC_TUNER:
      return "TRAFFIC_MESSAGE_SOURCE__TMC_TUNER";
   case TRAFFIC_MESSAGE_SOURCE__TRAFFIC_TRIP_REPLAY:
      return "TRAFFIC_MESSAGE_SOURCE__TRAFFIC_TRIP_REPLAY";
   default:
      return "<unknown>";
   }
}

enum TrafficMessageDetourResult
{
   TRAFFIC_MESSAGE_DETOUR_RESULT__SUCCESS = 0,
   TRAFFIC_MESSAGE_DETOUR_RESULT__FAILED
};

inline ::std::string toString(TrafficMessageDetourResult trafficMessageDetourResult)
{
   switch(trafficMessageDetourResult)
   {
   case TRAFFIC_MESSAGE_DETOUR_RESULT__SUCCESS:
      return "TRAFFIC_MESSAGE_DETOUR_RESULT__SUCCESS";
   case TRAFFIC_MESSAGE_DETOUR_RESULT__FAILED:
      return "TRAFFIC_MESSAGE_DETOUR_RESULT__FAILED";
   default:
      return "<unknown>";
   }
}

class TrafficMessageInfoBase
{
public:
   /*
    * below typedefs are @DEPRECATED. Will be removed once all clients are adapted to new Image class.
    */
   typedef ::std::pair<const unsigned char*, unsigned int> T_IconDataPair;
   typedef ::std::vector< T_IconDataPair > T_TrafficIconList;

   TrafficMessageInfoBase()
      : m_id(0)
      , m_index(0)
      , m_distanceToTmInMeters(0)
      , m_userRule(TRAFFICMESSAGEUSERRULE__NONE)
      , m_tmLengthInMeters(0)
      , m_isAlongRoute(false)
      , m_isUserDefinedTrafficMessage(false)
   {}

   TrafficMessageInfoBase(unsigned int id, unsigned int index, unsigned int distanceToTrafficMessage, const ::std::string& distance,
                          const ValidValue<DirectionDescription>& direction, TrafficMessageUserRule userRule, unsigned int tmLength, bool isAlongRoute,
                          const ValidValue<unsigned int>& trafficTimeDelayInMinutes, const ValidValue<unsigned int>& trafficTimeDelayInSeconds,
                          const ::std::vector< ::std::string>& textSections,
                          bool isUserDefinedTrafficMessage)
      : m_id(id)
      , m_index(index)
      , m_distanceToTmInMeters(distanceToTrafficMessage)
      , m_distance(distance)
      , m_direction(direction)
      , m_userRule(userRule)
      , m_tmLengthInMeters(tmLength)
      , m_isAlongRoute(isAlongRoute)
      , m_trafficTimeDelayInMinutes(trafficTimeDelayInMinutes)
      , m_trafficTimeDelayInSeconds(trafficTimeDelayInSeconds)
      , m_textSections(textSections)
      , m_isUserDefinedTrafficMessage(isUserDefinedTrafficMessage)
   {}

   virtual ~TrafficMessageInfoBase() {}

   virtual bool operator==(const TrafficMessageInfoBase& rhs) const
   {
      return m_id == rhs.m_id &&
             m_index == rhs.m_index &&
             m_distanceToTmInMeters == rhs.m_distanceToTmInMeters &&
             m_distance == rhs.m_distance &&
             m_direction == rhs.m_direction &&
             m_userRule == rhs.m_userRule &&
             m_tmLengthInMeters == rhs.m_tmLengthInMeters &&
             m_isAlongRoute == rhs.m_isAlongRoute &&
             m_flowColor == rhs.m_flowColor &&
             m_trafficTimeDelayInMinutes == rhs.m_trafficTimeDelayInMinutes &&
             m_trafficTimeDelayInSeconds == rhs.m_trafficTimeDelayInSeconds &&
             m_trafficIconData == rhs.m_trafficIconData &&
             m_textSections == rhs.m_textSections &&
             m_isUserDefinedTrafficMessage == rhs.m_isUserDefinedTrafficMessage;
   }

   virtual bool operator!=(const TrafficMessageInfoBase& rhs) const
   {
      return !(*this == rhs);
   }

   virtual void clear()
   {
      m_id = 0;
      m_index = 0;
      m_distanceToTmInMeters = 0;
      m_distance.clear();
      m_direction.invalidate();
      m_userRule = TRAFFICMESSAGEUSERRULE__NONE;
      m_tmLengthInMeters = 0;
      m_isAlongRoute = false;
      m_flowColor.invalidate();
      /*
       * m_imageBlobs is @DEPRECATED. Will be removed once all clients are adapted to new Image class.
       */
      m_imageBlobs.clear();
      m_trafficIconData = Image();
      m_trafficTimeDelayInMinutes.invalidate();
      m_trafficTimeDelayInSeconds.invalidate();
      m_textSections.clear();
      m_isUserDefinedTrafficMessage = false;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "\tm_id: " << m_id << ::std::endl
             << "\tm_index: " << m_index << ::std::endl
             << "\tm_distance: " << m_distance << ::std::endl
             << "\tm_directionDescription: " << m_direction.toString() << ::std::endl
             << "\tm_userRule: " << navmiddleware::toString(m_userRule) << ::std::endl
             << "\tm_tmLength: " << m_tmLengthInMeters << ::std::endl
             << "\tm_isAlongRoute: " << m_isAlongRoute << ::std::endl
             << "\tm_trafficIconData: "<< m_trafficIconData.toString() << ::std::endl
             << "\tm_trafficTimeDelayInMinutes: " << m_trafficTimeDelayInMinutes.toString() << ::std::endl
             << "\tm_trafficTimeDelayInSeconds: " << m_trafficTimeDelayInSeconds.toString() << ::std::endl
             << "\tm_isUserDefinedTrafficMessage:  " << m_isUserDefinedTrafficMessage << ::std::endl;

      if(m_flowColor.isValid())
      {
         stream << "\tm_flowColor: " << m_flowColor.getValue().toString();
      }

      /*
       * m_imageBlobs is @DEPRECATED. Will be removed once all clients are adapted to new Image class.
       */
      for(unsigned int index = 0; index < m_imageBlobs.size(); ++index)
      {
         stream << "\tIcon Size -->" << m_imageBlobs[index].second << ::std::endl;
      }

      for (unsigned int index = 0; index < m_textSections.size(); ++index)
      {
         stream << "\t-> " << m_textSections[index] << ::std::endl;
      }

      return stream.str();
   };

   /*
    * addTrafficIcon is @DEPRECATED. Will be removed once all clients are adapted to new Image class.
    */
   void addTrafficIcon(const unsigned char* imageBlob, unsigned int imageBlobLength)
   {
      m_imageBlobs.push_back(::std::make_pair(imageBlob, imageBlobLength));
   }

   void addTrafficIconData(const Image& trafficIconData)
   {
      m_trafficIconData = trafficIconData;
   }

   unsigned int m_id;
   unsigned int m_index;
   /*
    * Note: This value is used to plot traffic messages on Traffic information bar.
    */
   unsigned int m_distanceToTmInMeters;
   ::std::string m_distance;
   ValidValue<DirectionDescription> m_direction;
   TrafficMessageUserRule m_userRule;
   /*
    * This member contains the traffic message length in meters.
    * Note: Length is used to calculate the pixel count for Traffic information bar in Renault
    */
   unsigned int                       m_tmLengthInMeters;
   bool                               m_isAlongRoute;
   /*
    * Traffic incidents will not have a flow value, Hence this is made as ValidValue.
    */
   ValidValue<navmiddleware::Color>   m_flowColor;
   /*
    * m_imageBlobs is @DEPRECATED. Will be removed once all clients are adapted to new Image class.
    */
   T_TrafficIconList                  m_imageBlobs;
   Image                              m_trafficIconData;
   /*
    * @DEPRECATED This value indicates the delay caused by TrafficMessage in minutes.
    */
   ValidValue<unsigned int>           m_trafficTimeDelayInMinutes;

   /*
    * This value indicates the delay caused by TrafficMessage in seconds.
    */
   ValidValue<unsigned int>           m_trafficTimeDelayInSeconds;
   ::std::vector< ::std::string>      m_textSections;
   bool                               m_isUserDefinedTrafficMessage;
};


class TrafficMessageListInfo: public TrafficMessageInfoBase
{
public:
   TrafficMessageListInfo()
   {}

   TrafficMessageListInfo(unsigned int id, unsigned int index, const ::std::string& text, unsigned int distanceToTrafficMessage, const ::std::string& distance,
                          const ValidValue<DirectionDescription>& direction, TrafficMessageUserRule userRule, unsigned int tmLength, bool isAlongRoute,
                          const ValidValue<unsigned int>& trafficTimeDelayInMinutes, const ValidValue<unsigned int>& trafficTimeDelayInSeconds,
                          const ::std::vector< ::std::string>& textSections,
                          bool isUserDefinedTrafficMessage)
      : TrafficMessageInfoBase(id, index, distanceToTrafficMessage, distance, direction,
                               userRule, tmLength, isAlongRoute, trafficTimeDelayInMinutes,
                               trafficTimeDelayInSeconds,
                               textSections, isUserDefinedTrafficMessage)
      , m_text(text) /** @DEPRECATED.*/
   {}

   bool operator==(const TrafficMessageListInfo& rhs) const
   {
      return TrafficMessageInfoBase::operator==(rhs) &&
             m_text == rhs.m_text; /** @DEPRECATED.*/
   }

   bool operator!=(const TrafficMessageListInfo& rhs) const
   {
      return !(*this == rhs);
   }

   // needed to be able to sort TrafficMessageListInfos::m_trafficMessageListInfos by traffic message index
   bool operator<(const TrafficMessageListInfo& rhs) const
   {
      return m_index < rhs.m_index;
   }

   void clear()
   {
      TrafficMessageInfoBase::clear();
      /** @DEPRECATED.*/
      m_text.clear();
   }

   ::std::string toString() const
   {
      ::std::string str("TrafficMessageListInfo payload:\n");
      str.append(TrafficMessageInfoBase::toString());
      /** @DEPRECATED.*/
      str.append("\tm_text: " + m_text + "\n");

      return str;
   }
   /** @DEPRECATED. Base class m_textSections should be used.*/
   ::std::string     m_text;
};


class TrafficMessageDetailsInfo: public TrafficMessageInfoBase
{
public:
   TrafficMessageDetailsInfo()
   {}

   TrafficMessageDetailsInfo(unsigned int id, unsigned int index, const ::std::vector< ::std::string>& text, unsigned int distanceToTrafficMessage, const ::std::string& distance,
                             const ValidValue<DirectionDescription>& directionDescription, TrafficMessageUserRule userRule, unsigned int tmLength, bool isAlongRoute,
                             const ValidValue<unsigned int>& trafficTimeDelayInMinutes, const ValidValue<unsigned int>& trafficTimeDelayInSeconds,
                             bool isUserDefinedTrafficMessage,
                             const GeoCoordinateDegree& coordinate)   //<-- INF4CV
      : TrafficMessageInfoBase(id, index, distanceToTrafficMessage, distance,
                               directionDescription, userRule, tmLength, isAlongRoute,
                               trafficTimeDelayInMinutes, trafficTimeDelayInSeconds,
                               text, isUserDefinedTrafficMessage)
      , m_coordinate(coordinate)   //<-- INF4CV
   {}

   bool operator==(const TrafficMessageDetailsInfo& rhs) const
   {
      return (TrafficMessageInfoBase::operator==(rhs)
              && (m_coordinate == rhs.m_coordinate));   //<-- INF4CV
   }

   bool operator!=(const TrafficMessageDetailsInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void clear()
   {
      TrafficMessageInfoBase::clear();
      m_coordinate = GeoCoordinateDegree();   //<-- INF4CV
   }

   ::std::string toString() const
   {
      ::std::string str("TrafficMessageDetailsInfo payload:\n");
      str.append(TrafficMessageInfoBase::toString());
      //<-- INF4CV
      str.append(", Coordinate detail = ");
      str.append(m_coordinate.toString());
      //->
      return str;
   }

   //<-- INF4CV
   const GeoCoordinateDegree& getCoordinate() const
   {
      return m_coordinate;
   }

   void setCoordinate(const GeoCoordinateDegree& coordinate)
   {
      m_coordinate = coordinate;
   }
   //->

   private:

   //<-- INF4CV
   /* Traffic message coordinate in latitude : longitude.
    * Depending on the traffic message type the coordinate refers to e.g. the cause location of the traffic message.
    */
   GeoCoordinateDegree m_coordinate;
   //->
};

class TrafficMessageListInfos
{
public:

   TrafficMessageListInfos()
      : m_countOfAllTrafficMessages(0)
      , m_visualRangeStartIndex(0)
   {}

   ~TrafficMessageListInfos() {}

   const ::std::vector<TrafficMessageListInfo>& getTrafficMessageListInfos() const
   {
      return m_trafficMessageInfos;
   }
   ::std::vector<TrafficMessageListInfo>& getTrafficMessageListInfos()
   {
      return m_trafficMessageInfos;
   }

   unsigned int getCountOfAllTrafficMessages() const
   {
      return m_countOfAllTrafficMessages;
   }

   void setCountOfAllTrafficMessages(unsigned int countOfAllTrafficMessages)
   {
      m_countOfAllTrafficMessages = countOfAllTrafficMessages;
   }

   unsigned int getVisualRangeStartIndex() const
   {
      return m_visualRangeStartIndex;
   }

   void setVisualRangeStartIndex(unsigned int visualRangeStartIndex)
   {
      m_visualRangeStartIndex = visualRangeStartIndex;
   }

   const ValidValue<unsigned int>& getFocusedElementIndex() const
   {
      return m_focusedElementIndex;
   }

   void setFocusedElementIndex(const ValidValue<unsigned int>& focusedElementIndex)
   {
      m_focusedElementIndex = focusedElementIndex;
   }

   bool operator==(const TrafficMessageListInfos& rhs) const
   {
      return m_countOfAllTrafficMessages == rhs.m_countOfAllTrafficMessages &&
             m_visualRangeStartIndex == rhs.m_visualRangeStartIndex &&
             m_focusedElementIndex == rhs.m_focusedElementIndex &&
             m_trafficMessageInfos == rhs.m_trafficMessageInfos;
   }

   bool operator!=(const TrafficMessageListInfos& rhs) const
   {
      return !(*this == rhs);
   }

   void clear()
   {
      m_countOfAllTrafficMessages = 0;
      m_visualRangeStartIndex = 0;
      m_focusedElementIndex.invalidate();
      m_trafficMessageInfos.clear();
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "TrafficMessageListInfos payload:" << ::std::endl
             << "\tm_countOfAllTrafficMessages: " << m_countOfAllTrafficMessages << ::std::endl
             << "\tm_visualRangeStartIndex: " << m_visualRangeStartIndex << ::std::endl
             << "\tm_focusedElementIndex: " << m_focusedElementIndex.toString() << ::std::endl
             << "\tm_trafficMessageInfos: " << ::std::endl;

      for (unsigned int i = 0; i < m_trafficMessageInfos.size(); i++)
      {
         stream << "\n\t-> " << m_trafficMessageInfos[i].toString();
      }

      return stream.str();
   }

private:
   unsigned int m_countOfAllTrafficMessages;
   unsigned int m_visualRangeStartIndex;
   ValidValue<unsigned int> m_focusedElementIndex;
   ::std::vector<TrafficMessageListInfo> m_trafficMessageInfos;
};

class TrafficServiceReceptionStatusInfo
{
public:
   TrafficServiceReceptionStatusInfo()
      : m_trafficMessageSource(TRAFFIC_MESSAGE_SOURCE__UNKOWN)
      , m_trafficServiceReceptionStatus(false)
      , m_trafficReceptionState(TRAFFICRECEPTIONSTATE__UNKNOWN)
      , m_trafficReceptionTime()
   {}
   ~TrafficServiceReceptionStatusInfo() {}

   ::std::string getTrafficServiceName() const
   {
      return m_serviceName;
   }

   void setTrafficServiceName(const ::std::string& trafficServiceName)
   {
      m_serviceName = trafficServiceName;
   }

   void setTrafficMessageSource(TrafficMessageSource trafficMessageSource)
   {
      m_trafficMessageSource = trafficMessageSource;
   }

   TrafficMessageSource getTrafficMessageSource() const
   {
      return m_trafficMessageSource;
   }

   bool getTrafficServiceReceptionStatus() const
   {
      return m_trafficServiceReceptionStatus;
   }

   void setTrafficServiceReceptionStatus(bool trafficServiceRecStatus)
   {
      m_trafficServiceReceptionStatus = trafficServiceRecStatus;
   }

   TrafficReceptionState getTrafficReceptionState() const
   {
      return m_trafficReceptionState;
   }

   void setTrafficReceptionState(const TrafficReceptionState& trafficReceptionState)
   {
      m_trafficReceptionState = trafficReceptionState;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "TrafficServiceReceptionStatusInfo payload:" << ::std::endl
             << "\tm_serviceName: " << m_serviceName << ::std::endl
             << "\tm_trafficMessageSource: "<< ::navmiddleware::toString(m_trafficMessageSource) << ::std::endl
             << "\tm_trafficServiceReceptionStatus: " << m_trafficServiceReceptionStatus << ::std::endl
             << "\tm_trafficReceptionState: "<< navmiddleware::toString(m_trafficReceptionState) <<::std::endl
             << "\tm trafficReceptionTime: " << m_trafficReceptionTime.toString() << ::std::endl;
      return stream.str();
   }
   void setTrafficReceptionTime(const ::navmiddleware::ValidValue< ::std::string>& trafficReceptionTime)
   {
      m_trafficReceptionTime = trafficReceptionTime;
   }

   const ::navmiddleware::ValidValue< ::std::string>& getTrafficReceptionTime()
   {
      return m_trafficReceptionTime;
   }
private:
   ::std::string m_serviceName;
   TrafficMessageSource m_trafficMessageSource;
   bool m_trafficServiceReceptionStatus;
   TrafficReceptionState m_trafficReceptionState;
   ::navmiddleware::ValidValue< ::std::string> m_trafficReceptionTime;
};

class TrafficIconSelectionListInfo
{
public:
   typedef ::std::pair< ::std::string, Image> T_EventCategoryTextIconPair;
   typedef ::std::map< settings::TrafficIconSelectionCriteria, T_EventCategoryTextIconPair> T_TrafficIconSelectionCriteriaDetailsMap;

   TrafficIconSelectionListInfo()
   {}
   ~TrafficIconSelectionListInfo() {}

   const T_TrafficIconSelectionCriteriaDetailsMap& getTrafficIconSelectionItemList() const
   {
      return m_trafficIconSelectionItemList;
   }

   T_TrafficIconSelectionCriteriaDetailsMap& getTrafficIconSelectionItemListMutable()
   {
      return m_trafficIconSelectionItemList;
   }

   void setTrafficIconSelectionItemList(const T_TrafficIconSelectionCriteriaDetailsMap& trafficIconSelectionItemList)
   {
      m_trafficIconSelectionItemList = trafficIconSelectionItemList;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;

      stream << "\tTrafficIconSelectionListInfo payload:\n"<< ::std::endl
             << "\t m_trafficIconSelectionItemList: " << ::std::endl;

      for (T_TrafficIconSelectionCriteriaDetailsMap::const_iterator it = m_trafficIconSelectionItemList.begin();
            it != m_trafficIconSelectionItemList.end(); ++it)
      {
         stream << "\t Filter Criteria : "<< navmiddleware::settings::toString(it->first) <<::std::endl
                << "\t Filter Criteria Text-> " << it->second.first << ::std::endl
                << "\t Image Size-> " << it->second.second.getBlobSize() << ::std::endl;
      }
      return stream.str();
   }
private:
   T_TrafficIconSelectionCriteriaDetailsMap m_trafficIconSelectionItemList;
};

}
#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_TRAFFICMESSAGEINFOS_H_
