/**
* @copyright (c) 2017-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_SDSLOCATIONINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_SDSLOCATIONINFOS_H_

#include <string>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{

/**
 * Holds all data required to fill a SDS address input.
 */

enum SDSAddressElementType
{
   SDSADDRESSELEMENTTYPE_COUNTRY = 4278190080u,
   SDSADDRESSELEMENTTYPE_PLACE = 4278190081u,
   SDSADDRESSELEMENTTYPE_ROAD = 4278190082u,
   SDSADDRESSELEMENTTYPE_CROSSROAD = 4278190083u,
   SDSADDRESSELEMENTTYPE_HOUSENUMBER = 4278190084u,
   SDSADDRESSELEMENTTYPE_POSTALDISRICT = 4278190085u,
   SDSADDRESSELEMENTTYPE_CITYDISTRICT = 4278190086u,
   SDSADDRESSELEMENTTYPE_PLACEREFINEMENT = 4278190087u,
   SDSADDRESSELEMENTTYPE_ROADREFINEMENT = 4278190088u,
   SDSADDRESSELEMENTTYPE_STATE = 4278190089u,
   SDSADDRESSELEMENTTYPE_HOUSENUMBER_REFINEMENT = 4278190090u
};

inline ::std::string toString(SDSAddressElementType sdsAddressElementType)
{
   switch (sdsAddressElementType)
   {
   case SDSADDRESSELEMENTTYPE_COUNTRY:
      return "SDSADDRESSELEMENTTYPE_COUNTRY";
   case SDSADDRESSELEMENTTYPE_PLACE:
      return "SDSADDRESSELEMENTTYPE_PLACE";
   case SDSADDRESSELEMENTTYPE_ROAD:
      return "SDSADDRESSELEMENTTYPE_ROAD";
   case SDSADDRESSELEMENTTYPE_CROSSROAD:
      return "SDSADDRESSELEMENTTYPE_CROSSROAD";
   case SDSADDRESSELEMENTTYPE_HOUSENUMBER:
      return "SDSADDRESSELEMENTTYPE_HOUSENUMBER";
   case SDSADDRESSELEMENTTYPE_POSTALDISRICT:
      return "SDSADDRESSELEMENTTYPE_POSTALDISRICT";
   case SDSADDRESSELEMENTTYPE_CITYDISTRICT:
      return "SDSADDRESSELEMENTTYPE_CITYDISTRICT";
   case SDSADDRESSELEMENTTYPE_PLACEREFINEMENT:
      return "SDSADDRESSELEMENTTYPE_PLACEREFINEMENT";
   case SDSADDRESSELEMENTTYPE_ROADREFINEMENT:
      return "SDSADDRESSELEMENTTYPE_ROADREFINEMENT";
   case SDSADDRESSELEMENTTYPE_STATE:
      return "SDSADDRESSELEMENTTYPE_STATE";
   default:
      ::std::stringstream stream;
      stream << "SDSADDRESSELEMENTTYPE_<" << static_cast<unsigned int>(sdsAddressElementType) << ">";
      return stream.str();
   }
}


/** this class can transport additional flags per element of the address */
class SDSElementOptions
{
public:
   SDSElementOptions():
      m_isAmbiguous(false)
   {}

   bool isAmbiguous() const
   {
      return m_isAmbiguous;
   }

   void setIsAmbiguous(bool value)
   {
      m_isAmbiguous = value;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SDSElementOptions payload" << ::std::endl
             << "   Is SDS Element Options ambiguous = "<< m_isAmbiguous << ::std::endl;
      return stream.str();
   }

private:
   bool m_isAmbiguous;
};

/**  this class includes addtional flags for the complete address */
class SDSAddressOptions
{
public:

   SDSAddressOptions():
      m_isNavigable(false),
      m_isAmbiguous(false),
      m_hasRemovedInvalidEntry(false),
      m_areMoreEntriesPossible(false),
      m_isHousenumberAvailable(false),
      m_isHousenumberValid(false)
   {}

   bool isNavigable() const
   {
      return m_isNavigable;
   }

   void setIsNavigable(bool value)
   {
      m_isNavigable = value;
   }

   bool isAmbiguous() const
   {
      return m_isAmbiguous;
   }

   void setIsAmbiguous(bool value)
   {
      m_isAmbiguous = value;
   }

   bool hasRemovedInvalidEntry() const
   {
      return m_hasRemovedInvalidEntry;
   }

   void setHasRemovedInvalidEntry(bool value)
   {
      m_hasRemovedInvalidEntry = value;
   }

   bool areMoreEntriesPossible() const
   {
      return m_areMoreEntriesPossible;
   }

   void setAreMoreEntriesPossible(bool value)
   {
      m_areMoreEntriesPossible = value;
   }

   bool isHousenumberAvailable() const
   {
      return m_isHousenumberAvailable;
   }

   void setIsHousenumberAvailable(bool value)
   {
      m_isHousenumberAvailable = value;
   }

   bool isHousenumberValid() const
   {
      return m_isHousenumberValid;
   }

   void setIsHousenumberValid(bool value)
   {
      m_isHousenumberValid = value;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SDSAddressOptions payload:" << ::std::endl
             << "   Is navigation possible to this element = "<< m_isNavigable << ", "
             << "Is the element ambiguous = "<< m_isAmbiguous << ", "
             << "Is invalid entry removed = "<< m_hasRemovedInvalidEntry << ::std::endl
             << "   Are more entries possible = "<< m_areMoreEntriesPossible << ", "
             << "Is house number available = "<< m_isHousenumberAvailable << ", "
             << "Is house number valid = "<< m_isHousenumberValid << ::std::endl;
      return stream.str();
   }

private:
   bool m_isNavigable;
   bool m_isAmbiguous;
   bool m_hasRemovedInvalidEntry;
   bool m_areMoreEntriesPossible;
   bool m_isHousenumberAvailable;
   bool m_isHousenumberValid;
};

/** class of an element of the address entries with type and options */
class SDSAddressElement
{
public:

   SDSAddressElement():
      m_addressElementType(),
      m_data(),
      m_elementOptions()
   {}

   const SDSAddressElementType& getAddressElementType() const

   {
      return m_addressElementType;
   }

   void setAddressElementType(const SDSAddressElementType& addressElementType)
   {
      m_addressElementType = addressElementType;
   }

   const ::std::string& getData() const
   {
      return m_data;
   }

   void setData(const ::std::string& newData)
   {
      m_data = newData ;
   }

   const SDSElementOptions& getAddressElementOptions() const
   {
      return m_elementOptions;
   }

   void  setAddressElementOptions(const SDSElementOptions& newOpts)
   {
      m_elementOptions = newOpts;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SDSAddressElement payload:" << ::std::endl
             << "   Address Element type = "<< m_addressElementType << ", "
             << "Address data = \""<< m_data << "\"" << ::std::endl
             << "   Address element options = "<< m_elementOptions.toString() << ::std::endl;
      return stream.str();
   }

private:
   SDSAddressElementType m_addressElementType;
   ::std::string m_data;
   SDSElementOptions m_elementOptions;
};

/** class of an element of the address entries with type and options */
class SDSAddress
{
public:

   SDSAddress():
      m_addressElements()
   {}

   const ::std::vector<SDSAddressElement>& getAddressElements() const
   {
      return m_addressElements;
   }

   ::std::vector<SDSAddressElement>& getAddressElementsMutable()
   {
      return m_addressElements;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SDSAddress payload:" << ::std::endl;
      for(unsigned int addressElementIndex = 0; addressElementIndex < getAddressElements().size(); addressElementIndex++)
      {
         stream << "   AddressElement[" << addressElementIndex << "]:" << m_addressElements[addressElementIndex].toString();
      }
      return stream.str();
   }

private:
   ::std::vector<SDSAddressElement> m_addressElements;
};

/** class for an address with options and possible extension types */
class SDSAddressWithOptions
{
public:

   SDSAddressWithOptions():
      m_address(),
      m_addressOptions(),
      m_addressElementTypes()
   {}

   const SDSAddress& getAddress() const
   {
      return m_address;
   }

   SDSAddress& getAddressMutable()
   {
      return m_address;
   }


   const SDSAddressOptions& getAddressOptions() const
   {
      return m_addressOptions;
   }

   void setAddressOptions(const SDSAddressOptions& options)
   {
      m_addressOptions = options;
   }

   const ::std::vector<SDSAddressElementType>& getAddressElementTypes() const
   {
      return m_addressElementTypes;
   }

   ::std::vector<SDSAddressElementType>& getAddressElementTypesMutable()
   {
      return m_addressElementTypes;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SDSAddressWithOptions payload:" << ::std::endl
             << "   Address information = " << getAddress().toString() << ::std::endl
             << "   SDSAddressWithOptions Info = " << getAddressOptions().toString() << ::std::endl;
      for(unsigned int addressElementIndex = 0; addressElementIndex < getAddressElementTypes().size(); addressElementIndex++)
      {
         stream << "   ElementType[" << addressElementIndex <<"] = " << m_addressElementTypes[addressElementIndex] << ::std::endl;
      }
      return stream.str();
   }

private:
   SDSAddress m_address;
   SDSAddressOptions m_addressOptions;
   ::std::vector<SDSAddressElementType> m_addressElementTypes;
};


class RefinementWithIdInfos
{
public:

   RefinementWithIdInfos() {}

   class RefinementWithId
   {
   public:
      RefinementWithId():
         m_id(0)
      {
      }

      const ::std::string& getData() const
      {
         return m_data;
      }

      void setData(const ::std::string& newData)
      {
         m_data = newData ;
      }
      unsigned int  getId() const
      {
         return m_id;
      }

      void setId(unsigned int id)
      {
         m_id = id ;
      }

      const ::std::string& getAirDistance() const
      {
         return m_airDistance;
      }

      void setAirDistance(const ::std::string& airDistance)
      {
         m_airDistance = airDistance;
      }

      const ValidValue<DirectionDescription>& getDirection() const
      {
         return m_direction;
      }

      void setDirection(const ValidValue<DirectionDescription>& directionDescription)
      {
         m_direction = directionDescription;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "RefinementWithId payload:" << ::std::endl
                << "   Refinement ID = " << m_id << ::std::endl
                << "   Refinement data = \"" << m_data << "\"" << ::std::endl
                << "   Air Distance = \"" << m_airDistance << "\"" << ::std::endl
                << "   Direction = " << m_direction.toString() << ::std::endl;
         return stream.str();
      }

   public:
      ::std::string m_data;
      unsigned int m_id;
      ::std::string m_airDistance;
      ValidValue<DirectionDescription> m_direction;
   };

   const ::std::vector<RefinementWithId>& getElements() const
   {
      return m_listElements;
   }

   ::std::vector<RefinementWithId>& getElementsMutable()
   {
      return m_listElements;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "RefinementWithIdInfos payload:" << ::std::endl;
      for(unsigned int refinementListIndex = 0; refinementListIndex < getElements().size(); refinementListIndex++)
      {
         stream << "[" << refinementListIndex << "] - " << m_listElements[refinementListIndex].toString();
      }
      return stream.str();
   }

private:

   ::std::vector <RefinementWithId> m_listElements;
};

class SdsHouseNumberPatternsInfos
{
public:
   SdsHouseNumberPatternsInfos():
      m_minimum()
      ,m_maximum()
      ,m_patterns()
   {
   }

   const ::std::string& getMaximum() const
   {
      return m_maximum;
   }

   void setMaximum(const ::std::string& maximum)
   {
      m_maximum = maximum ;
   }

   const ::std::string& getMinimum() const
   {
      return m_minimum;
   }

   void setMinimum(const ::std::string& minimum)
   {
      m_minimum = minimum ;
   }

   const ::std::vector< ::std::string>& getPatterns() const
   {
      return m_patterns;
   }

   ::std::vector< ::std::string>& getPatternsMutable()
   {
      return m_patterns;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsHouseNumberPattern payload:" << ::std::endl
             << "   Minimum = " << m_minimum << ::std::endl
             << "   Maximum = " << m_maximum << ::std::endl;
      for (unsigned int i = 0; i < m_patterns.size(); i++)
      {
         stream << "   Pattern[" << i << "]: \"" << m_patterns[i].c_str() << "\"" << ::std::endl;
      }
      return stream.str();
   }

   void clear()
   {
      m_minimum.clear();
      m_maximum.clear();
      m_patterns.clear();
   }

private:
   ::std::string m_minimum;
   ::std::string m_maximum;
   ::std::vector < ::std::string> m_patterns;
};

enum NdsFeatureType
{
   NDS_FEATURE_TYPE__NAMED_OBJECT,
   NDS_FEATURE_TYPE__POI_NAME,
   NDS_FEATURE_TYPE__POI_CATEGORY,
   NDS_FEATURE_TYPE__POI_ATTRIBUTE
};

enum SdsPhoneticTranscriptionFormat
{
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__TA_IPA,
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__STARREC_SAMPA,
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__X_SAMPA,
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__LH_PLUS,
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__NT_SAMPA,
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__SVOX_PA,
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__VFA,
   SDS_PHONETIC_TRANSCRIPTION_FORMAT__I_FLYTEC
};

enum SdsPreRecordedVoiceFormat
{
   SDS_PRE_RECORDED_VOICE_FORMAT__ADPCM,
   SDS_PRE_RECORDED_VOICE_FORMAT__MP3
};

enum SdsBuildingBlockType
{
   SDS_BUILDING_BLOCK_TYPE__SPEECH,
   SDS_BUILDING_BLOCK_TYPE__EXTENSION
};

class SdsLanguageVersionInfo
{
public:
   SdsLanguageVersionInfo()
      : m_languageCode(0)
   {
   }

   void setLanguageCode(uint16_t languageCode)
   {
      m_languageCode = languageCode;
   }

   uint16_t getLanguageCode() const
   {
      return m_languageCode;
   }

   void setSpecificationVersion(const ::std::string& specificationVersion)
   {
      m_specificationVersion = specificationVersion;
   }

   const ::std::string& getSpecificationVersion() const
   {
      return m_specificationVersion;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsLanguageVersionInfo payload:" << ::std::endl
             << "   LanguageCode = " << m_languageCode << ::std::endl
             << "   LanguageSpecificationVersion = " << m_specificationVersion << ::std::endl;
      return stream.str();
   }

private:
   uint16_t       m_languageCode;
   ::std::string  m_specificationVersion;
};

class SdsPhoneticTraMetaDataInfo
{
public:
   SdsPhoneticTraMetaDataInfo()
      : m_sdsPhoneticTranscriptionFormat(SDS_PHONETIC_TRANSCRIPTION_FORMAT__TA_IPA)
   {
   }

   void setSdsPhoneticTranscriptionFormat(SdsPhoneticTranscriptionFormat sdsPhoneticTranscriptionFormat)
   {
      m_sdsPhoneticTranscriptionFormat = sdsPhoneticTranscriptionFormat;
   }

   SdsPhoneticTranscriptionFormat getSdsPhoneticTranscriptionFormat() const
   {
      return m_sdsPhoneticTranscriptionFormat;
   }

   void setSdsLanguageVersionInfos(const ::std::vector<SdsLanguageVersionInfo>& sdsLanguageVersionInfos)
   {
      m_sdsLanguageVersionInfos = sdsLanguageVersionInfos;
   }

   const ::std::vector<SdsLanguageVersionInfo>& getSdsLanguageVersionInfos() const
   {
      return m_sdsLanguageVersionInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsPhoneticTraMetaDataInfo payload:" << ::std::endl
             << "   SdsPhoneticTranscriptionFormat = " << m_sdsPhoneticTranscriptionFormat << ::std::endl;
      for(unsigned int languageVersionIndex = 0;
            languageVersionIndex < m_sdsLanguageVersionInfos.size(); ++languageVersionIndex)
      {
         stream << "[" << languageVersionIndex << "] - " <<
                m_sdsLanguageVersionInfos[languageVersionIndex].toString();
      }
      return stream.str();
   }

private:
   SdsPhoneticTranscriptionFormat         m_sdsPhoneticTranscriptionFormat;
   ::std::vector<SdsLanguageVersionInfo>  m_sdsLanguageVersionInfos;

};

class SdsPreRecordedVMetaDataInfo
{
public:
   SdsPreRecordedVMetaDataInfo()
      : m_sdsPreRecordedVoiceFormat(SDS_PRE_RECORDED_VOICE_FORMAT__ADPCM)
   {
   }

   void setSdsPreRecordedVoiceFormat(SdsPreRecordedVoiceFormat sdsPreRecordedVoiceFormat)
   {
      m_sdsPreRecordedVoiceFormat = sdsPreRecordedVoiceFormat;
   }

   SdsPreRecordedVoiceFormat getSdsPreRecordedVoiceFormat() const
   {
      return m_sdsPreRecordedVoiceFormat;
   }

   void setSdsLanguageVersionInfos(const ::std::vector<SdsLanguageVersionInfo>& sdsLanguageVersionInfos)
   {
      m_sdsLanguageVersionInfos = sdsLanguageVersionInfos;
   }

   const ::std::vector<SdsLanguageVersionInfo>& getSdsLanguageVersionInfos() const
   {
      return m_sdsLanguageVersionInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsPreRecordedVMetaDataInfo payload:" << ::std::endl
             << "   SdsPreRecordedVoiceFormat = " << m_sdsPreRecordedVoiceFormat << ::std::endl;
      for(unsigned int languageVersionIndex = 0;
            languageVersionIndex < m_sdsLanguageVersionInfos.size(); ++languageVersionIndex)
      {
         stream << "[" << languageVersionIndex << "] - " <<
                m_sdsLanguageVersionInfos[languageVersionIndex].toString();
      }
      return stream.str();
   }

private:
   SdsPreRecordedVoiceFormat              m_sdsPreRecordedVoiceFormat;
   ::std::vector<SdsLanguageVersionInfo>  m_sdsLanguageVersionInfos;
};

class SdsSpeechMetaDataInfo
{
public:
   SdsSpeechMetaDataInfo()
      : m_hasPhonetic(false)
      , m_hasPreRecorded(false)
   {
   }

   void setHasPhonetic(bool hasPhonetic)
   {
      m_hasPhonetic = hasPhonetic;
   }

   bool hasPhonetic() const
   {
      return m_hasPhonetic;
   }

   void setHasPreRecorded(bool hasPreRecorded)
   {
      m_hasPreRecorded = hasPreRecorded;
   }

   bool hasPreRecorded() const
   {
      return m_hasPreRecorded;
   }

   void setSdsPhoneticTraMetaDataInfo(const SdsPhoneticTraMetaDataInfo& sdsPhoneticTraMetaDataInfo)
   {
      m_sdsPhoneticTraMetaDataInfo = sdsPhoneticTraMetaDataInfo;
   }

   const SdsPhoneticTraMetaDataInfo& getSdsPhoneticTraMetaDataInfo() const
   {
      return m_sdsPhoneticTraMetaDataInfo;
   }

   void setSdsPreRecordedVMetaDataInfo(const SdsPreRecordedVMetaDataInfo& sdsPreRecordedVMetaDataInfo)
   {
      m_sdsPreRecordedVMetaDataInfo = sdsPreRecordedVMetaDataInfo;
   }

   const SdsPreRecordedVMetaDataInfo& getSdsPreRecordedVMetaDataInfo() const
   {
      return m_sdsPreRecordedVMetaDataInfo;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsSpeechMetaDataInfo payload:" << ::std::endl
             << "   hasPhonetic = " << m_hasPhonetic << ::std::endl
             << "   hasPreRecorded = " << m_hasPreRecorded << ::std::endl
             << "   SdsPhoneticTraMetaDataInfo = " <<
             m_sdsPhoneticTraMetaDataInfo.toString() << ::std::endl
             << "   SdsPreRecordedVMetaDataInfo = " <<
             m_sdsPreRecordedVMetaDataInfo.toString() << ::std::endl;

      return stream.str();
   }

private:
   bool                          m_hasPhonetic;
   bool                          m_hasPreRecorded;
   SdsPhoneticTraMetaDataInfo    m_sdsPhoneticTraMetaDataInfo;
   SdsPreRecordedVMetaDataInfo   m_sdsPreRecordedVMetaDataInfo;
};

class SdsBuildingBlockInfo
{
public:
   SdsBuildingBlockInfo()
      : m_buildingBlockId(0)
      , m_buildingBlockType(SDS_BUILDING_BLOCK_TYPE__SPEECH)
      , m_encryptionId(0)
      , m_typeNdsDbSupplierId(0)
      , m_typeExtId(0)
   {
   }

   void setBuildingBlockId(uint16_t buildingBlockId)
   {
      m_buildingBlockId = buildingBlockId;
   }
   uint16_t getBuildingBlockId() const
   {
      return m_buildingBlockId;
   }

   void setBuildingBlockType(SdsBuildingBlockType buildingBlockType)
   {
      m_buildingBlockType = buildingBlockType;
   }

   SdsBuildingBlockType getSdsBuildingBlockType() const
   {
      return m_buildingBlockType;
   }

   void setURI(const ::std::string& uri)
   {
      m_uri = uri;
   }

   const ::std::string& getURI() const
   {
      return m_uri;
   }

   void setEncryptionId(uint32_t encryptionId)
   {
      m_encryptionId = encryptionId;
   }

   uint32_t getEncryptionId() const
   {
      return m_encryptionId;
   }

   void setSdsSpeechMetaDataInfo(const SdsSpeechMetaDataInfo& sdsSpeechMetaDataInfo)
   {
      m_sdsSpeechMetaDataInfo = sdsSpeechMetaDataInfo;
   }

   const SdsSpeechMetaDataInfo& getSdsSpeechMetaDataInfo() const
   {
      return m_sdsSpeechMetaDataInfo;
   }

   void setTypeNdsDbSupplierID(uint8_t typeNdsDbSupplierId)
   {
      m_typeNdsDbSupplierId = typeNdsDbSupplierId;
   }

   uint8_t getTypeNdsDbSupplierID() const
   {
      return m_typeNdsDbSupplierId;
   }

   void setTypeExtId(uint32_t typeExtId)
   {
      m_typeExtId = typeExtId;
   }

   uint32_t getTypeExtId() const
   {
      return m_typeExtId;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsBuildingBlockInfo payload:" << ::std::endl
             << "   BuildingBlockId = " << m_buildingBlockId << ::std::endl
             << "   BuildingBlockType = " << m_buildingBlockType << ::std::endl
             << "   URI = " << m_uri << ::std::endl
             << "   EncryptionId = " << m_encryptionId << ::std::endl
             << "   SpeechMetaDataInfo = " <<
             m_sdsSpeechMetaDataInfo.toString() << ::std::endl
             << "   NdsSupplierId = " << static_cast<unsigned int>(m_typeNdsDbSupplierId) << ::std::endl
             << "   ExtId = " << m_typeExtId << ::std::endl;

      return stream.str();
   }


private:
   uint16_t                m_buildingBlockId;
   SdsBuildingBlockType    m_buildingBlockType;
   ::std::string           m_uri;
   uint32_t                m_encryptionId;
   SdsSpeechMetaDataInfo   m_sdsSpeechMetaDataInfo;
   uint8_t                 m_typeNdsDbSupplierId;
   uint32_t                m_typeExtId;
};

class SdsRegionMetaDataInfo
{
public:
   SdsRegionMetaDataInfo()
      : m_updateRegionId(0)
   {
   }

   void setUpdateRegionId(uint16_t updateRegionId)
   {
      m_updateRegionId = updateRegionId;
   }

   uint16_t getUpdateRegionId() const
   {
      return m_updateRegionId;
   }

   void setIsoCountryCode(const ::std::string& isoCountryCode)
   {
      m_isoCountryCode = isoCountryCode;
   }

   const ::std::string& getIsoCountryCode() const
   {
      return m_isoCountryCode;
   }

   void setIsoSubCountryCode(const ::std::string& isoSubCountryCode)
   {
      m_isoSubCountryCode = isoSubCountryCode;
   }

   const ::std::string& getIsoSubCountryCode() const
   {
      return m_isoSubCountryCode;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsRegionMetaDataInfo payload:" << ::std::endl
             << "   UpdateRegionId = " << m_updateRegionId << ::std::endl
             << "   isoCountryCode = " << m_isoCountryCode << ::std::endl
             << "   isoSubCountryCode = " << m_isoSubCountryCode << ::std::endl;

      return stream.str();
   }

private:
   uint16_t          m_updateRegionId;
   ::std::string     m_isoCountryCode;
   ::std::string     m_isoSubCountryCode;
};

class SdsUpdateRegionInfo
{
public:
   SdsUpdateRegionInfo()
      : m_updateRegionId(0)
   {
   }

   void setUpdateRegionId(uint16_t updateRegionId)
   {
      m_updateRegionId = updateRegionId;
   }

   uint16_t getUpdateRegionId() const
   {
      return m_updateRegionId;
   }

   void setSdsBuildingBlockInfos(const ::std::vector<SdsBuildingBlockInfo>& sdsBuildingBlockInfos)
   {
      m_sdsBuildingBlockInfos = sdsBuildingBlockInfos;
   }

   const ::std::vector<SdsBuildingBlockInfo>& getSdsBuildingBlockInfos() const
   {
      return m_sdsBuildingBlockInfos;
   }

   void setSdsRegionMetaDataInfos(const ::std::vector<SdsRegionMetaDataInfo>& sdsRegionMetaDataInfos)
   {
      m_sdsRegionMetaDataInfos = sdsRegionMetaDataInfos;
   }

   const ::std::vector<SdsRegionMetaDataInfo>& getSdsRegionMetaDataInfos() const
   {
      return m_sdsRegionMetaDataInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsUpdateRegionInfo payload:" << ::std::endl
             << "   UpdateRegionId = " << m_updateRegionId << ::std::endl;
      for(unsigned int buildingBlockInfoIndex = 0;
            buildingBlockInfoIndex < m_sdsBuildingBlockInfos.size(); ++buildingBlockInfoIndex)
      {
         stream << "[" << buildingBlockInfoIndex << "] - " <<
                m_sdsBuildingBlockInfos[buildingBlockInfoIndex].toString();
      }
      for(unsigned int regionMetaDataInfoIndex = 0;
            regionMetaDataInfoIndex < m_sdsRegionMetaDataInfos.size(); ++regionMetaDataInfoIndex)
      {
         stream << "[" << regionMetaDataInfoIndex << "] - " <<
                m_sdsRegionMetaDataInfos[regionMetaDataInfoIndex].toString();
      }

      return stream.str();
   }

private:
   uint16_t                               m_updateRegionId;
   ::std::vector<SdsBuildingBlockInfo>    m_sdsBuildingBlockInfos;
   ::std::vector<SdsRegionMetaDataInfo>   m_sdsRegionMetaDataInfos;
};

class SdsLanguageInfo
{
public:
   SdsLanguageInfo()
      : m_languageCode(0)
      , m_isTransliterationOf(0)
      , m_isDiacriticTransliterationOf(0)
   {
   }

   void setLanguageCode(uint16_t languageCode)
   {
      m_languageCode = languageCode;
   }

   uint16_t getLanguageCode() const
   {
      return m_languageCode;
   }

   void setIsoCountryCode(const ::std::string& isoCountryCode)
   {
      m_isoCountryCode = isoCountryCode;
   }

   const ::std::string& getIsoCountryCode() const
   {
      return m_isoCountryCode;
   }

   void setIsoLanguageCode(const ::std::string& isoLanguageCode)
   {
      m_isoLanguageCode = isoLanguageCode;
   }

   const ::std::string& getIsoLanguageCode() const
   {
      return m_isoLanguageCode;
   }

   void setIsoScriptCode(const ::std::string& isoScriptCode)
   {
      m_isoScriptCode = isoScriptCode;
   }

   const ::std::string& getIsoScriptCode() const
   {
      return m_isoScriptCode;
   }

   void setLanguageEntryInfo(const LanguageEntryInfo& languageEntryInfo)
   {
      m_languageEntryInfo = languageEntryInfo;
   }

   const LanguageEntryInfo& getLanguageEntryInfo() const
   {
      return m_languageEntryInfo;
   }

   void setIsTransliterationOf(uint16_t isTransliterationOf)
   {
      m_isTransliterationOf = isTransliterationOf;
   }

   uint16_t getIsTransliterationOf() const
   {
      return m_isTransliterationOf;
   }

   void setIsDiacriticTransliterationOf(uint16_t isDiacriticTransliterationOf)
   {
      m_isDiacriticTransliterationOf = isDiacriticTransliterationOf;
   }

   uint16_t getIsDiacriticTransliterationOf() const
   {
      return m_isDiacriticTransliterationOf;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsLanguageInfo payload:" << ::std::endl
             << "   LanguageCode = " << m_languageCode << ::std::endl
             << "   isoCountryCode = " << m_isoCountryCode << ::std::endl
             << "   isoLanguageCode = " << m_isoLanguageCode << ::std::endl
             << "   isoScriptCode = " << m_isoScriptCode << ::std::endl
             << "   LanguageEntryInfo = " << m_languageEntryInfo.toString() << ::std::endl
             << "   isTransliterationOf = " << m_isTransliterationOf << ::std::endl
             << "   isDiacriticTransliterationOf = " << m_isDiacriticTransliterationOf << ::std::endl;

      return stream.str();
   }

private:
   uint16_t          m_languageCode;
   ::std::string     m_isoCountryCode;
   ::std::string     m_isoLanguageCode;
   ::std::string     m_isoScriptCode;
   LanguageEntryInfo m_languageEntryInfo;
   uint16_t          m_isTransliterationOf;
   uint16_t          m_isDiacriticTransliterationOf;
};

class SdsProductInfo
{
public:
   SdsProductInfo()
      : m_supplierId(0)
      , m_productId(0)
   {
   }

   void setSupplierId(uint8_t supplierId)
   {
      m_supplierId = supplierId;
   }

   uint8_t getSupplierId() const
   {
      return m_supplierId;
   }

   void setProductId(uint32_t productId)
   {
      m_productId = productId;
   }

   uint32_t getProductId() const
   {
      return m_productId;
   }

   void setSdsUpdateRegionInfos(const ::std::vector<SdsUpdateRegionInfo>& sdsUpdateRegionInfos)
   {
      m_sdsUpdateRegionInfos = sdsUpdateRegionInfos;
   }

   const ::std::vector<SdsUpdateRegionInfo>& getSdsUpdateRegionInfos() const
   {
      return m_sdsUpdateRegionInfos;
   }

   void setSdsLanguageInfos(const ::std::vector<SdsLanguageInfo>& sdsLanguageInfos)
   {
      m_sdsLanguageInfos = sdsLanguageInfos;
   }

   const ::std::vector<SdsLanguageInfo>& getSdsLanguageInfos() const
   {
      return m_sdsLanguageInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsProductInfo payload:" << ::std::endl
             << "   SupplierId = " << static_cast<unsigned int>(m_supplierId) << ::std::endl
             << "   ProductId = " << m_productId << ::std::endl;
      for(unsigned int sdsUpdateRegionInfoIndex = 0;
            sdsUpdateRegionInfoIndex < m_sdsUpdateRegionInfos.size(); ++sdsUpdateRegionInfoIndex)
      {
         stream << "[" << sdsUpdateRegionInfoIndex << "] - " <<
                m_sdsUpdateRegionInfos[sdsUpdateRegionInfoIndex].toString();
      }
      for(unsigned int sdsLanguageInfoIndex = 0;
            sdsLanguageInfoIndex < m_sdsLanguageInfos.size(); ++sdsLanguageInfoIndex)
      {
         stream << "[" << sdsLanguageInfoIndex << "] - " <<
                m_sdsLanguageInfos[sdsLanguageInfoIndex].toString();
      }

      return stream.str();
   }

private:
   uint8_t                             m_supplierId;
   uint32_t                            m_productId;
   ::std::vector<SdsUpdateRegionInfo>  m_sdsUpdateRegionInfos;
   ::std::vector<SdsLanguageInfo>      m_sdsLanguageInfos;
};

class UpdateRegionSpecifierRelationInfo
{
public:
   UpdateRegionSpecifierRelationInfo()
      : m_updateRegionSpecifier(0)
      , m_supplierId(0)
      , m_productId(0)
      , m_updateRegionId(0)
   {
   }

   void setUpdateRegionSpecifier(uint16_t updateRegionSpecifier)
   {
      m_updateRegionSpecifier = updateRegionSpecifier;
   }

   uint16_t getUpdateRegionSpecifier() const
   {
      return m_updateRegionSpecifier;
   }

   void setSupplierId(uint8_t supplierId)
   {
      m_supplierId = supplierId;
   }

   uint8_t getSupplierId() const
   {
      return m_supplierId;
   }

   void setProductId(uint32_t productId)
   {
      m_productId = productId;
   }

   uint32_t getProductId() const
   {
      return m_productId;
   }

   void setUpdateRegionId(uint16_t updateRegionId)
   {
      m_updateRegionId = updateRegionId;
   }

   uint16_t getUpdateRegionId() const
   {
      return m_updateRegionId;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "UpdateRegionSpecifierRelationInfo payload:" << ::std::endl
             << "   updateRegionSpecifier = " << m_updateRegionSpecifier << ::std::endl
             << "   SupplierId = " << static_cast<unsigned int>(m_supplierId) << ::std::endl
             << "   ProductId = " << m_productId << ::std::endl
             << "   UpdateRegionId = " << m_updateRegionId << ::std::endl;

      return stream.str();
   }

private:
   uint16_t m_updateRegionSpecifier;
   uint8_t  m_supplierId;
   uint32_t m_productId;
   uint16_t m_updateRegionId;
};

class SdsGlobalDataInfo
{
public:

   SdsGlobalDataInfo()
      : m_datasetId(0)
   {
   }

   void setDeviceName(const ::std::string& deviceName)
   {
      m_deviceName = deviceName;
   }

   const ::std::string& getDeviceName() const
   {
      return m_deviceName;
   }

   void setDatasetId(uint32_t datasetId)
   {
      m_datasetId = datasetId;
   }

   uint32_t getDatasetId() const
   {
      return m_datasetId;
   }

   void setPersistentDatasetId(const ::std::string& persistentDatasetId)
   {
      m_persistentDatasetId = persistentDatasetId;
   }

   const ::std::string& getPersistentDatasetId() const
   {
      return m_persistentDatasetId;
   }

   void setSdsProductInfos(const ::std::vector<SdsProductInfo>& sdsProductInfos)
   {
      m_sdsProductInfos = sdsProductInfos;
   }

   const ::std::vector<SdsProductInfo>& getSdsProductInfos() const
   {
      return m_sdsProductInfos;
   }

   void setUpdateRegionSpecifierRelationInfos(
      const ::std::vector<UpdateRegionSpecifierRelationInfo>& updateRegionSpecifierRelationInfos)
   {
      m_updateRegionSpecifierRelationInfos = updateRegionSpecifierRelationInfos;
   }

   const ::std::vector<UpdateRegionSpecifierRelationInfo>& getUpdateRegionSpecifierRelationInfo() const
   {
      return m_updateRegionSpecifierRelationInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsGlobalDataInfo payload:" << ::std::endl
             << "   DeviceName = " << m_deviceName << ::std::endl
             << "   DatasetId = " << m_datasetId << ::std::endl
             << "   PersistentDatasetId = " << m_persistentDatasetId << ::std::endl;
      for(unsigned int sdsProductInfoIndex = 0;
            sdsProductInfoIndex < m_sdsProductInfos.size(); ++sdsProductInfoIndex)
      {
         stream << "[" << sdsProductInfoIndex << "] - " <<
                m_sdsProductInfos[sdsProductInfoIndex].toString();
      }
      for(unsigned int updateRegionSpecifierRelationInfoIndex = 0;
            updateRegionSpecifierRelationInfoIndex < m_updateRegionSpecifierRelationInfos.size();
            ++updateRegionSpecifierRelationInfoIndex)
      {
         stream << "[" << updateRegionSpecifierRelationInfoIndex << "] - " <<
                m_updateRegionSpecifierRelationInfos[updateRegionSpecifierRelationInfoIndex].toString();
      }

      return stream.str();
   }


private:
   ::std::string                                      m_deviceName;
   uint32_t                                           m_datasetId;
   ::std::string                                      m_persistentDatasetId;
   ::std::vector<SdsProductInfo>                      m_sdsProductInfos;
   ::std::vector<UpdateRegionSpecifierRelationInfo>   m_updateRegionSpecifierRelationInfos;
};

class SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo
{
public:
   SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo()
      : m_poiRegionId(0)
      , m_globalXrefId(0)
   {
   }

   void setSdsGlobalNamedObjectIdInfo(const SdsGlobalNamedObjectIdInfo& sdsGlobalNamedObjectIdInfo)
   {
      m_sdsGlobalNamedObjectIdInfo = sdsGlobalNamedObjectIdInfo;
   }

   const SdsGlobalNamedObjectIdInfo& getSdsGlobalNamedObjectIdInfo() const
   {
      return m_sdsGlobalNamedObjectIdInfo;
   }

   void setPoiRegionId(uint64_t poiRegionId)
   {
      m_poiRegionId = poiRegionId;
   }

   uint64_t getPoiRegionId() const
   {
      return m_poiRegionId;
   }

   void setGlobalXrefId(uint32_t globalXrefId)
   {
      m_globalXrefId = globalXrefId;
   }

   uint32_t getGlobalXrefId() const
   {
      return m_globalXrefId;
   }

   bool operator==(const SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo& other) const
   {
      return (this->m_sdsGlobalNamedObjectIdInfo == other.m_sdsGlobalNamedObjectIdInfo)
             && (this->m_poiRegionId == other.m_poiRegionId)
             && (this->m_globalXrefId == other.m_globalXrefId);
   }
   bool operator!=(const SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo& other) const
   {
      return !(*this == other);
   }

   SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo& operator=(const SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo& other)
   {
      if (&other != this)
      {
         m_sdsGlobalNamedObjectIdInfo = other.m_sdsGlobalNamedObjectIdInfo;
         m_poiRegionId = other.m_poiRegionId;
         m_globalXrefId = other.m_globalXrefId;
      }
      return *this;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo payload:" << ::std::endl
             << "   SdsGlobalNamedObjectIdInfo = " << m_sdsGlobalNamedObjectIdInfo.toString() << ::std::endl
             << "   PoiRegionId = " << m_poiRegionId << ::std::endl
             << "   GlobalXrefId = " << m_globalXrefId << ::std::endl;
      return stream.str();
   }

private:
   SdsGlobalNamedObjectIdInfo m_sdsGlobalNamedObjectIdInfo;
   uint64_t                   m_poiRegionId;
   uint32_t                   m_globalXrefId;
};

class SdsCountryInfo
{
public:
   SdsCountryInfo()
   {
   }

   void setSdsCountryCode(const ::std::string& sdsCountryCode)
   {
      m_sdsCountryCode = sdsCountryCode;
   }

   const ::std::string& getSdsCountryCode() const
   {
      return m_sdsCountryCode;
   }

   void setSdsGlobalNamedObjectIdWithPoiAndXrefIdInfo(
      const SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo sdsGlobalNamedObjectIdWithPoiAndXrefIdInfo)
   {
      m_sdsGlobalNamedObjectIdWithPoiAndXrefIdInfo = sdsGlobalNamedObjectIdWithPoiAndXrefIdInfo;
   }

   const SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo& getSdsGlobalNamedObjectIdWithPoiAndXrefIdInfo() const
   {
      return m_sdsGlobalNamedObjectIdWithPoiAndXrefIdInfo;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsCountryInfos payload:" << ::std::endl
             << "   sdsCountryCode = " << m_sdsCountryCode << ::std::endl
             << "   dsGlobalNamedObjectIdWithPoiAndXrefIdInfo = " << m_sdsGlobalNamedObjectIdWithPoiAndXrefIdInfo.toString() << ::std::endl;
      return stream.str();
   }

private:
   ::std::string                              m_sdsCountryCode; //ISO-3166-1 alpha 3 Code
   SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo m_sdsGlobalNamedObjectIdWithPoiAndXrefIdInfo;
};

class SdsSubCountryInfos
{
public:
   SdsSubCountryInfos()
   {
   }

   void setSdsCountryInfo(const SdsGlobalNamedObjectIdInfo& sdsCountryInfo)
   {
      m_sdsCountryInfo = sdsCountryInfo;
   }

   const SdsGlobalNamedObjectIdInfo& getSdsCountryInfo() const
   {
      return m_sdsCountryInfo;
   }

   void setSdsSubCountryInfos(const ::std::vector<SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo>& sdsSubCountryInfos)
   {
      m_sdsSubCountryInfos = sdsSubCountryInfos;
   }

   const ::std::vector<SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo>& getSdsSubCountryInfos() const
   {
      return m_sdsSubCountryInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsSubCountryInfos payload:" << ::std::endl
             << "   SdsGlobalNamedObjectIdInfo = " << m_sdsCountryInfo.toString() << ::std::endl;
      for(unsigned int sdsSubCountryInfoIndex = 0;
            sdsSubCountryInfoIndex < m_sdsSubCountryInfos.size();
            ++sdsSubCountryInfoIndex)
      {
         stream << "[" << sdsSubCountryInfoIndex << "] - " <<
                m_sdsSubCountryInfos[sdsSubCountryInfoIndex].toString();
      }

      return stream.str();
   }

private:
   SdsGlobalNamedObjectIdInfo                m_sdsCountryInfo;
   ::std::vector<SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo> m_sdsSubCountryInfos;
};

class SdsCountryAndSubCountryInfos
{
public:

   SdsCountryAndSubCountryInfos()
   {
   }

   void setSdsCountryInfos(const ::std::vector<SdsCountryInfo>& sdsCountryInfos)
   {
      m_SdsCountryInfos = sdsCountryInfos;
   }

   const ::std::vector<SdsCountryInfo>& getSdsCountryInfos() const
   {
      return m_SdsCountryInfos;
   }

   void setSdsSubCountryInfos(const ::std::vector<SdsSubCountryInfos>& sdsSubCountryInfos)
   {
      m_sdsSubCountryInfos = sdsSubCountryInfos;
   }

   const ::std::vector<SdsSubCountryInfos>& getSdsSubCountryInfos() const
   {
      return m_sdsSubCountryInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsCountryAndSubCountryInfos payload:" << ::std::endl;
      for(unsigned int sdsCountryInfoIndex = 0;
            sdsCountryInfoIndex < m_SdsCountryInfos.size();
            ++sdsCountryInfoIndex)
      {
         stream << "[" << sdsCountryInfoIndex << "] - " <<
                m_SdsCountryInfos[sdsCountryInfoIndex].toString();
      }
      for(unsigned int sdsSubCountryInfoIndex = 0;
            sdsSubCountryInfoIndex < m_sdsSubCountryInfos.size();
            ++sdsSubCountryInfoIndex)
      {
         stream << "[" << sdsSubCountryInfoIndex << "] - " <<
                m_sdsSubCountryInfos[sdsSubCountryInfoIndex].toString();
      }

      return stream.str();
   }

private:
   ::std::vector<SdsCountryInfo>      m_SdsCountryInfos;
   ::std::vector<SdsSubCountryInfos>   m_sdsSubCountryInfos;
};

class SdsNdsIdStringInfo
{
public:
   SdsNdsIdStringInfo()
   {
   }

   void setSdsGlobalNamedObjectIdInfo(const SdsGlobalNamedObjectIdInfo& sdsGlobalNamedObjectIdInfo)
   {
      m_sdsGlobalNamedObjectIdInfo = sdsGlobalNamedObjectIdInfo;
   }

   const SdsGlobalNamedObjectIdInfo& getSdsGlobalNamedObjectIdInfo() const
   {
      return m_sdsGlobalNamedObjectIdInfo;
   }

   void setLanguageEntryInfo(const LanguageEntryInfo& languageEntryInfo)
   {
      m_languageEntryInfo = languageEntryInfo;
   }

   const LanguageEntryInfo& getLanguageEntryInfo() const
   {
      return m_languageEntryInfo;
   }

   void setSDSAddressElementType(const ValidValue<SDSAddressElementType>& sdsAddressElementType)
   {
      m_sdsAddressElementType = sdsAddressElementType;
   }

   const ValidValue<SDSAddressElementType>& getSDSAdressElementType() const
   {
      return m_sdsAddressElementType;
   }

   void setSdsNamedStrings(const ::std::vector< ::std::string >& sdsNamedStrings)
   {
      m_sdsNamedStrings = sdsNamedStrings;
   }

   const ::std::vector< ::std::string >& getSdsNamedString() const
   {
      return m_sdsNamedStrings;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsNdsIdStringInfo payload:" << ::std::endl
             << "SdsGlobalNamedObjectIdInfo :" <<
             m_sdsGlobalNamedObjectIdInfo.toString() << ::std::endl;
      for(unsigned int sdsNamedStringIndex = 0;
            sdsNamedStringIndex < m_sdsNamedStrings.size();
            ++sdsNamedStringIndex)
      {
         stream << "[" << sdsNamedStringIndex << "] - " <<
                m_sdsNamedStrings[sdsNamedStringIndex] << ::std::endl;
      }

      return stream.str();
   }

private:
   SdsGlobalNamedObjectIdInfo        m_sdsGlobalNamedObjectIdInfo;
   LanguageEntryInfo                 m_languageEntryInfo;
   ValidValue<SDSAddressElementType> m_sdsAddressElementType;
   ::std::vector< ::std::string >    m_sdsNamedStrings;
};

class SdsNdsIdStringInfos
{
public:
   SdsNdsIdStringInfos() : m_requestId(0)
   {
   }

   void setRequestId(RequestId requestId)
   {
      m_requestId = requestId;
   }

   RequestId getRequestId() const
   {
      return m_requestId;
   }

   void setSdsNdsIdStringInfos(const ::std::vector<SdsNdsIdStringInfo>& sdsNdsIdStringInfos)
   {
      m_sdsNdsIdStringInfos = sdsNdsIdStringInfos;
   }

   const ::std::vector<SdsNdsIdStringInfo>& getSdsNdsIdStringInfos() const
   {
      return m_sdsNdsIdStringInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsNdsIdStringInfos payload:" << ::std::endl
             << "Request Id :" << m_requestId << ::std::endl;
      for(unsigned int sdsNdsIdStringInfoIndex = 0;
            sdsNdsIdStringInfoIndex < m_sdsNdsIdStringInfos.size();
            ++sdsNdsIdStringInfoIndex)
      {
         stream << "[" << sdsNdsIdStringInfoIndex << "] - " <<
                m_sdsNdsIdStringInfos[sdsNdsIdStringInfoIndex].toString().c_str() << ::std::endl;
      }

      return stream.str();
   }

private:
   RequestId                           m_requestId;
   ::std::vector<SdsNdsIdStringInfo>   m_sdsNdsIdStringInfos;
};

class SdsCityNamedObjectHasStreet
{
public:
   SdsCityNamedObjectHasStreet()
      : m_hasStreets(false)
   {
   }

   void setSdsGlobalNamedObjectIdInfo(const SdsGlobalNamedObjectIdInfo& sdsGlobalNamedObjectInfo)
   {
      m_sdsGlobalNamedObjectInfo = sdsGlobalNamedObjectInfo;
   }

   const SdsGlobalNamedObjectIdInfo& getSdsGlobalNamedObjectIdInfo() const
   {
      return m_sdsGlobalNamedObjectInfo;
   }

   void setHasStreets(bool hasStreets)
   {
      m_hasStreets = hasStreets;
   }

   bool getHasStreets() const
   {
      return m_hasStreets;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsCityNamedObjectHasStreet payload:" << ::std::endl
             << "SdsGlobalNamedObjectIdInfo :" <<
             m_sdsGlobalNamedObjectInfo.toString() << ::std::endl
             << "hasStreets :" << m_hasStreets << ::std::endl;

      return stream.str();
   }

private:
   SdsGlobalNamedObjectIdInfo m_sdsGlobalNamedObjectInfo;
   bool                       m_hasStreets;
};

} // namespace navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_SDSLOCATIONINFOS_H_
