/**
* @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_ROUTESETTINGS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_ROUTESETTINGS_H_

/*(TODO):InfoTypes is included here to fetch the definition of DistanceUnitSystem.
 * The following include has to be removed from here once the SettingTypes.h is available which should then hold
 * DistanceUnitSystem in it.
 */
#include <string>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{
namespace settings
{

enum RouteCriterion
{
   ROUTE_CRITERION_ECONOMIC=0,
   ROUTE_CRITERION_FAST,
   ROUTE_CRITERION_SHORT,
   ROUTE_CRITERION_MINIMIZE_FREEWAY //<-- JP
};

class RouteCriteriaSettings
{
public:
   RouteCriteriaSettings()
      : m_routeCriterion(ROUTE_CRITERION_FAST)
   {
   }

   RouteCriterion getRouteCriterion() const
   {
      return m_routeCriterion;
   }

   void setRouteCriterion(RouteCriterion routeCriterion)
   {
      m_routeCriterion = routeCriterion;
   }

   bool operator == (const RouteCriteriaSettings& rhs) const
   {
      return (m_routeCriterion == rhs.m_routeCriterion);
   }

   bool operator != (const RouteCriteriaSettings& rhs) const
   {
      return !(*this == rhs);
   }

private:
   RouteCriterion m_routeCriterion;
};

enum UseMotorwaysValue
{
   USE_MOTORWAYS_ON=0,
   USE_MOTORWAYS_MINIMIZE,
   USE_MOTORWAYS_OFF
};

enum UseSubscriptionRoadsValue
{
   USE_SUBSCRIPTION_ROADS_ON=0,
   USE_SUBSCRIPTION_ROADS_OFF
};

enum UseTimeRestrictedRoadsValue
{
   USE_TIME_RESTRICTED_ROADS_ALWAYS=0,
   USE_TIME_RESTRICTED_ROADS_WHEN_OPEN,
   USE_TIME_RESTRICTED_ROADS_NEVER
};

enum UseSeasonalRestrictedRoadsValue
{
   USE_SEASONAL_RESTRICTED_ROADS_ALWAYS=0,
   USE_SEASONAL_RESTRICTED_ROADS_WHEN_OPEN,
   USE_SEASONAL_RESTRICTED_ROADS_NEVER
};

enum UseTollRoadsValue
{
   USE_TOLL_ROADS_VALUE__ALWAYS=0,
   USE_TOLL_ROADS_VALUE__ASK,
   USE_TOLL_ROADS_VALUE__NEVER
};

enum UseFerriesValue
{
   USE_FERRIES_ON=0,
   USE_FERRIES_PREFER,
   USE_FERRIES_OFF
};

enum UseUnpavedRoadsValue
{
   USE_UNPAVED_ROADS_ON=0,
   USE_UNPAVED_ROADS_MINIMIZE,
   USE_UNPAVED_ROADS_OFF
};

enum UseIPDValue
{
   USE_IPD_ON=0,
   USE_IPD_OFF
};

enum MoreRoutesTypeValue
{
   MORE_ROUTES_TYPE_MULTIPLE=0,
   MORE_ROUTES_TYPE_ALTERNATIVE
};

enum UseRealTimeTrafficInfoValue
{
   USE_REALTIME_TRAFFIC_AUTOMATIC= 0,
   USE_REALTIME_TRAFFIC_PROMPT,
   USE_REALTIME_TRAFFIC_OFF
};

enum UseBorderCrossingsValue
{
   USE_BORDER_CROSSING_ON=0,
   USE_BORDER_CROSSING_OFF
};

enum UseAvoidAreaValue
{
   USE_AVOID_AREA_ON=0,
   USE_AVOID_AREA_OFF
};

enum UseCarShuttleTrainValue
{
   USE_CAR_SHUTTLE_TRAIN_ON=0,
   USE_CAR_SHUTTLE_TRAIN_OFF
};

enum UseLearnedRoutesValue
{
   USE_LEARNED_ROUTES_ON=0,
   USE_LEARNED_ROUTES_OFF
};

enum UseFreewaysValue
{
   USE_FREEWAYS_ON=0,
   USE_FREEWAYS_MINIMIZE,
   USE_FREEWAYS_OFF
};

enum UseHOVLanesValue
{
   USE_HOV_LANES_ON=0,
   USE_HOV_LANES_OFF
};

enum UseRouteAlternativeValue
{
   USE_ROUTE_ALTERNATIVE_ON=0,
   USE_ROUTE_ALTERNATIVE_OFF
};

//<-- JP
enum UseAutoRerouteValue
{
   USE_AUTO_REROUTE_ON = 0,
   USE_AUTO_REROUTE_OFF
};

enum UseSmartIC
{
   USE_SMART_IC_ON = 0,
   USE_SMART_IC_OFF
};

enum UseAlongside
{
   USE_ALONGSIDE_ON = 0,
   USE_ALONGSIDE_OFF
};

//<-- INF4CV
enum UseTunnelsValue
{
   USE_TUNNELS_ON = 0,
   USE_TUNNELS_MINIMIZE,
   USE_TUNNELS_OFF
};

enum UseUTurns
{
   USE_UTURNS_ON = 0,
   USE_UTURNS_OFF
};

enum UseLowClassifiedRoads
{
   USE_LOW_CLASSIFIED_ROADS_ON = 0,
   USE_LOW_CLASSIFIED_ROADS_OFF
};
//->

enum VehicleTypeValue
{
   USE_VEHICLETYPE_PASSENGER_CAR=0,
   USE_VEHICLETYPE_TRUCK,
   USE_VEHICLETYPE_BUS,
   USE_VEHICLETYPE_RESIDENT,
   USE_VEHICLETYPE_HEAVYDUTY,
   USE_VEHICLETYPE_PUBLIC_BUS,
   USE_VEHICLETYPE_PRIVATE_BUS,
   USE_VEHICLETYPE_EMERGENCY,
   USE_VEHICLETYPE_TAXI,
   USE_VEHICLETYPE_MILITARY,
   USE_VEHICLETYPE_MOTORCYCLE,
   USE_VEHICLETYPE_MOPED,
   USE_VEHICLETYPE_BIKE,
   USE_VEHICLETYPE_FARM,
   USE_VEHICLETYPE_LIGHT_TRUCK
};

/* csb1kor: Mapping from these three enums to the uint8 value(T_u8_Percent) of navi_fi is done based on the description mentioned in navi_fi
 * (TODO)To be validated */
enum RoofLoadValue
{
   USE_NO_ROOFLOAD=0,
   USE_ROOFLOAD_SMALL=33,
   USE_ROOFLOAD_MID=66,
   USE_ROOFLOAD_LARGE=100
};

/* csb1kor: Mapping from these three enums to the uint8 value(T_u8_Percent) of navi_fi is done based on the description mentioned in navi_fi
 * (TODO)To be validated */
enum TrailerValue
{
   USE_NO_TRAILER=0,
   USE_TRAILER_SMALL=33,
   USE_TRAILER_MID=66,
   USE_TRAILER_LARGE=100
};

enum CargoType
{
   USE_CARGOTYPE_HAS_WATER_POLLUTING=0,
   USE_CARGOTYPE_HAS_EXPLOSIVE,
   USE_CARGOTYPE_HAS_OTHER_DANGEROUS_GOODS,
   USE_CARGOTYPE_HAS_NO_LOAD,
   USE_CARGOTYPE_HAS_SPECIAL_LOAD,
   USE_CARGOTYPE_HAS_GASSES,
   USE_CARGOTYPE_HAS_FLAMMABLE_LIQUIDS,
   USE_CARGOTYPE_HAS_FLAMMABLE_SOLIDS,
   USE_CARGOTYPE_HAS_OXIDIZERS_AND_ORGANIC_PEROXIDES,
   USE_CARGOTYPE_HAS_TOXIC_AND_INFECTIOUS_SUBSTANCES,
   USE_CARGOTYPE_HAS_RADIOACTIVE_SUBSTANCES,
   USE_CARGOTYPE_HAS_CORROSIVE_SUBSTANCES
};

enum UseEndToEndNavigationValue
{
   USE_END_TO_END_NAVIGATION_ON=0,
   USE_END_TO_END_NAVIGATION_OFF
};

enum UseAutomaticGuidanceProposalValue
{
   USE_AUTOMATIC_GUIDANCE_PROPOSAL__ON = 0,
   USE_AUTOMATIC_GUIDANCE_PROPOSAL__OFF
};

enum UseChargingTimeEPois
{
   USE_CHARGING_TIME_EPOIS__ON = 0,
   USE_CHARGING_TIME_EPOIS__OFF
};

enum UseChargingStationAssistantValue
{
   USE_CHARGING_STATION_ASSISTANT__AUTO = 0,
   USE_CHARGING_STATION_ASSISTANT__OFF
};

enum UseLearningNavigationValue
{
   USE_LEARNING_NAVIGATION__ON = 0,
   USE_LEARNING_NAVIGATION__OFF
};

enum EstimatedTime
{
   ESTIMATED_TIME__CURRENT_DESTINATION = 0,
   ESTIMATED_TIME__FINAL_DESTINATION
};

enum DestinationTimeDisplay
{
   DESTINATION_TIME_DISPLAY__TIME_OF_ARRIVAL = 0,
   DESTINATION_TIME_DISPLAY__TRAVEL_TIME
};

inline ::std::string toString(UseMotorwaysValue motorwaysValue)
{
   switch (motorwaysValue)
   {
   case USE_MOTORWAYS_ON:
      return "USE_MOTORWAYS_ON";
   case USE_MOTORWAYS_MINIMIZE:
      return "USE_MOTORWAYS_MINIMIZE";
   case USE_MOTORWAYS_OFF:
      return "USE_MOTORWAYS_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_MOTORWAYS_<" << static_cast<unsigned int>(motorwaysValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseSubscriptionRoadsValue subscriptionRoadsValue)
{
   switch (subscriptionRoadsValue)
   {
   case USE_SUBSCRIPTION_ROADS_ON:
      return "USE_SUBSCRIPTION_ROADS_ON";
   case USE_SUBSCRIPTION_ROADS_OFF:
      return "USE_SUBSCRIPTION_ROADS_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_SUBSCRIPTION_ROADS_<" << static_cast<unsigned int>(subscriptionRoadsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseTimeRestrictedRoadsValue timeRestrictedRoadsValue)
{
   switch (timeRestrictedRoadsValue)
   {
   case USE_TIME_RESTRICTED_ROADS_ALWAYS:
      return "USE_TIME_RESTRICTED_ROADS_ALWAYS";
   case USE_TIME_RESTRICTED_ROADS_WHEN_OPEN:
      return "USE_TIME_RESTRICTED_ROADS_WHEN_OPEN";
   case USE_TIME_RESTRICTED_ROADS_NEVER:
      return "USE_TIME_RESTRICTED_ROADS_NEVER";
   default:
      ::std::stringstream stream;
      stream << "USE_TIME_RESTRICTED_ROADS_<" << static_cast<unsigned int>(timeRestrictedRoadsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseSeasonalRestrictedRoadsValue seasonalRestrictedRoadsValue)
{
   switch (seasonalRestrictedRoadsValue)
   {
   case USE_SEASONAL_RESTRICTED_ROADS_ALWAYS:
      return "USE_SEASONAL_RESTRICTED_ROADS_ALWAYS";
   case USE_SEASONAL_RESTRICTED_ROADS_WHEN_OPEN:
      return "USE_SEASONAL_RESTRICTED_ROADS_WHEN_OPEN";
   case USE_SEASONAL_RESTRICTED_ROADS_NEVER:
      return "USE_SEASONAL_RESTRICTED_ROADS_NEVER";
   default:
      ::std::stringstream stream;
      stream << "USE_SEASONAL_RESTRICTED_ROADS_<" << static_cast<unsigned int>(seasonalRestrictedRoadsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseTollRoadsValue tollRoadsValue)
{
   switch (tollRoadsValue)
   {
   case USE_TOLL_ROADS_VALUE__ALWAYS:
      return "USE_TOLL_ROADS_VALUE__ALWAYS";
   case USE_TOLL_ROADS_VALUE__ASK:
      return "USE_TOLL_ROADS_VALUE__ASK";
   case USE_TOLL_ROADS_VALUE__NEVER:
      return "USE_TOLL_ROADS_VALUE__NEVER";
   default:
      ::std::stringstream stream;
      stream << "USE_TOLL_ROADS_VALUE<" << static_cast<unsigned int>(tollRoadsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseFerriesValue ferriesValue)
{
   switch (ferriesValue)
   {
   case USE_FERRIES_ON:
      return "USE_FERRIES_ON";
   case USE_FERRIES_PREFER:
      return "USE_FERRIES_PREFER";
   case USE_FERRIES_OFF:
      return "USE_FERRIES_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_FERRIES_<" << static_cast<unsigned int>(ferriesValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseUnpavedRoadsValue unpavedRoadsValue)
{
   switch (unpavedRoadsValue)
   {
   case USE_UNPAVED_ROADS_ON:
      return "USE_UNPAVED_ROADS_ON";
   case USE_UNPAVED_ROADS_MINIMIZE:
      return "USE_UNPAVED_ROADS_MINIMIZE";
   case USE_UNPAVED_ROADS_OFF:
      return "USE_UNPAVED_ROADS_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_UNPAVED_ROADS_<" << static_cast<unsigned int>(unpavedRoadsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseIPDValue IPDValue)
{
   switch (IPDValue)
   {
   case USE_IPD_ON:
      return "USE_IPD_ON";
   case USE_IPD_OFF:
      return "USE_IPD_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_IPD_<" << static_cast<unsigned int>(IPDValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseRealTimeTrafficInfoValue realTimeTrafficInfoValue)
{
   switch (realTimeTrafficInfoValue)
   {
   case USE_REALTIME_TRAFFIC_AUTOMATIC:
      return "USE_REALTIME_TRAFFIC_AUTOMATIC";
   case USE_REALTIME_TRAFFIC_PROMPT:
      return "USE_REALTIME_TRAFFIC_PROMPT";
   case USE_REALTIME_TRAFFIC_OFF:
      return "USE_REALTIME_TRAFFIC_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_REALTIME_TRAFFIC_<" << static_cast<unsigned int>(realTimeTrafficInfoValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseBorderCrossingsValue borderCrossingsValue)
{
   switch (borderCrossingsValue)
   {
   case USE_BORDER_CROSSING_ON:
      return "USE_BORDER_CROSSING_ON";
   case USE_BORDER_CROSSING_OFF:
      return "USE_BORDER_CROSSING_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_BORDER_CROSSING_<" << static_cast<unsigned int>(borderCrossingsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseAvoidAreaValue avoidAreaValue)
{
   switch (avoidAreaValue)
   {
   case USE_AVOID_AREA_ON:
      return "USE_AVOID_AREA_ON";
   case USE_AVOID_AREA_OFF:
      return "USE_AVOID_AREA_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_AVOID_AREA_<" << static_cast<unsigned int>(avoidAreaValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseCarShuttleTrainValue carShuttleTrainValue)
{
   switch (carShuttleTrainValue)
   {
   case USE_CAR_SHUTTLE_TRAIN_ON:
      return "USE_CAR_SHUTTLE_TRAIN_ON";
   case USE_CAR_SHUTTLE_TRAIN_OFF:
      return "USE_CAR_SHUTTLE_TRAIN_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_CAR_SHUTTLE_TRAIN_<" << static_cast<unsigned int>(carShuttleTrainValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseLearnedRoutesValue learnedRoutesValue)
{
   switch (learnedRoutesValue)
   {
   case USE_LEARNED_ROUTES_ON:
      return "USE_LEARNED_ROUTES_ON";
   case USE_LEARNED_ROUTES_OFF:
      return "USE_LEARNED_ROUTES_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_LEARNED_ROUTES_<" << static_cast<unsigned int>(learnedRoutesValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseLearningNavigationValue learningNavigation)
{
   switch (learningNavigation)
   {
   case USE_LEARNING_NAVIGATION__ON:
      return "USE_LEARNING_NAVIGATION__ON";
   case USE_LEARNING_NAVIGATION__OFF:
      return "USE_LEARNING_NAVIGATION__OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_LEARNING_NAVIGATION__<" << static_cast<unsigned int>(learningNavigation) << ">";
      return stream.str();
   }
}

inline ::std::string toString(MoreRoutesTypeValue carShuttleTrainValue)
{
   switch (carShuttleTrainValue)
   {
   case MORE_ROUTES_TYPE_MULTIPLE:
      return "MORE_ROUTES_TYPE_MULTIPLE";
   case MORE_ROUTES_TYPE_ALTERNATIVE:
      return "MORE_ROUTES_TYPE_ALTERNATIVE";
   default:
      ::std::stringstream stream;
      stream << "MORE_ROUTES_TYPE_<" << static_cast<unsigned int>(carShuttleTrainValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseFreewaysValue freewaysValue)
{
   switch (freewaysValue)
   {
   case USE_FREEWAYS_ON:
      return "USE_FREEWAYS_ON";
   case USE_FREEWAYS_MINIMIZE:
      return "USE_FREEWAYS_MINIMIZE";
   case USE_FREEWAYS_OFF:
      return "USE_FREEWAYS_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_FREEWAYS_<" << static_cast<unsigned int>(freewaysValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseHOVLanesValue HOVLanesValue)
{
   switch (HOVLanesValue)
   {
   case USE_HOV_LANES_ON:
      return "USE_HOV_LANES_ON";
   case USE_HOV_LANES_OFF:
      return "USE_HOV_LANES_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_HOV_LANES_<" << static_cast<unsigned int>(HOVLanesValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseRouteAlternativeValue routeAlternativeValue)
{
   switch (routeAlternativeValue)
   {
   case USE_ROUTE_ALTERNATIVE_ON:
      return "USE_ROUTE_ALTERNATIVE_ON";
   case USE_ROUTE_ALTERNATIVE_OFF:
      return "USE_ROUTE_ALTERNATIVE_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_ROUTE_ALTERNATIVE_<" << static_cast<unsigned int>(routeAlternativeValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseEndToEndNavigationValue endToEndNavigationValue)
{
   switch (endToEndNavigationValue)
   {
   case USE_END_TO_END_NAVIGATION_ON:
      return "USE_END_TO_END_NAVIGATION_ON";
   case USE_END_TO_END_NAVIGATION_OFF:
      return "USE_END_TO_END_NAVIGATION_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_END_TO_END_NAVIGATION_<" << static_cast<unsigned int>(endToEndNavigationValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseAutomaticGuidanceProposalValue automaticGuidanceProposalValue)
{
   switch (automaticGuidanceProposalValue)
   {
   case USE_AUTOMATIC_GUIDANCE_PROPOSAL__ON:
      return "USE_AUTOMATIC_GUIDANCE_PROPOSAL__ON";
   case USE_AUTOMATIC_GUIDANCE_PROPOSAL__OFF:
      return "USE_AUTOMATIC_GUIDANCE_PROPOSAL__OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_AUTOMATIC_GUIDANCE_PROPOSAL__<" << static_cast<unsigned int>(automaticGuidanceProposalValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseChargingTimeEPois chargingTimeEPois)
{
   switch (chargingTimeEPois)
   {
   case USE_CHARGING_TIME_EPOIS__ON:
      return "USE_CHARGING_TIME_EPOIS__ON";
   case USE_CHARGING_TIME_EPOIS__OFF:
      return "USE_CHARGING_TIME_EPOIS__OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_CHARGING_TIME_EPOIS__<" << static_cast<unsigned int>(chargingTimeEPois) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseAutoRerouteValue autoRerouteValue)
{
   switch (autoRerouteValue)
   {
   case USE_AUTO_REROUTE_ON:
      return "USE_AUTO_REROUTE_ON";
   case USE_AUTO_REROUTE_OFF:
      return "USE_AUTO_REROUTE_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_AUTO_REROUTE_<" << static_cast<unsigned int>(autoRerouteValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseSmartIC smartIC)
{
   switch (smartIC)
   {
   case USE_SMART_IC_ON:
      return "USE_SMART_IC_ON";
   case USE_SMART_IC_OFF:
      return "USE_SMART_IC_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_SMART_IC_<" << static_cast<unsigned int>(smartIC) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseAlongside alongside)
{
   switch (alongside)
   {
   case USE_ALONGSIDE_ON:
      return "USE_ALONGSIDE_ON";
   case USE_ALONGSIDE_OFF:
      return "USE_ALONGSIDE_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_ALONGSIDE_<" << static_cast<unsigned int>(alongside) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseChargingStationAssistantValue useChargingStationAssistantValue)
{
   switch (useChargingStationAssistantValue)
   {
   case USE_CHARGING_STATION_ASSISTANT__AUTO:
      return "USE_CHARGING_STATION_ASSISTANT__AUTO";
   case USE_CHARGING_STATION_ASSISTANT__OFF:
      return "USE_CHARGING_STATION_ASSISTANT__OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_CHARGING_STATION_ASSISTANT__<" << static_cast<unsigned int>(useChargingStationAssistantValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(EstimatedTime estimatedTime)
{
   switch (estimatedTime)
   {
   case ESTIMATED_TIME__CURRENT_DESTINATION:
      return "ESTIMATED_TIME__CURRENT_DESTINATION";
   case ESTIMATED_TIME__FINAL_DESTINATION:
      return "ESTIMATED_TIME__FINAL_DESTINATION";
   default:
      ::std::stringstream stream;
      stream << "ESTIMATED_TIME__<" << static_cast<unsigned int>(estimatedTime) << ">";
      return stream.str();
   }
}

inline ::std::string toString(DestinationTimeDisplay destinationTimeDisplay)
{
   switch (destinationTimeDisplay)
   {
   case DESTINATION_TIME_DISPLAY__TIME_OF_ARRIVAL:
      return "DESTINATION_TIME_DISPLAY__TIME_OF_ARRIVAL";
   case DESTINATION_TIME_DISPLAY__TRAVEL_TIME:
      return "DESTINATION_TIME_DISPLAY__TRAVEL_TIME";
   default:
      ::std::stringstream stream;
      stream << "DESTINATION_TIME_DISPLAY__<" << static_cast<unsigned int>(destinationTimeDisplay) << ">";
      return stream.str();
   }
}

//<-- INF4CV
inline ::std::string toString(UseTunnelsValue tunnelsValue)
{
   switch (tunnelsValue)
   {
   case USE_TUNNELS_ON:
      return "USE_TUNNELS_ON";
   case USE_TUNNELS_MINIMIZE:
      return "USE_TUNNELS_MINIMIZE";
   case USE_TUNNELS_OFF:
      return "USE_TUNNELS_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_TUNNELS_<" << static_cast<unsigned int>(tunnelsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseUTurns uTurnsValue)
{
   switch (uTurnsValue)
   {
   case USE_UTURNS_ON:
      return "USE_UTURNS_ON";
   case USE_UTURNS_OFF:
      return "USE_UTURNS_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_UTURNS_<" << static_cast<unsigned int>(uTurnsValue) << ">";
      return stream.str();
   }
}

inline ::std::string toString(UseLowClassifiedRoads lowClassifiedRoadsValue)
{
   switch (lowClassifiedRoadsValue)
   {
   case USE_LOW_CLASSIFIED_ROADS_ON:
      return "USE_LOW_CLASSIFIED_ROADS_ON";
   case USE_LOW_CLASSIFIED_ROADS_OFF:
      return "USE_LOW_CLASSIFIED_ROADS_OFF";
   default:
      ::std::stringstream stream;
      stream << "USE_LOW_CLASSIFIED_ROADS_<" << static_cast<unsigned int>(lowClassifiedRoadsValue) << ">";
      return stream.str();
   }
}
//->

class VehicleProfile
{
public:
   VehicleProfile()
      :  m_optionVehicleType(USE_VEHICLETYPE_PASSENGER_CAR)
      ,  m_unitSystemForVehicleProfile(DISTANCEUNITSYSTEM_METRIC)
      ,  m_maxSpeed(250)
      //(TODO)csb1kor:Default values for these parameters are not known at the moment
      ,  m_heightOfVehicle(0)
      ,  m_widthOfVehicle(0)
      ,  m_lengthOfVehicle(0)
      ,  m_maxWeightOfVehicle(0)
      ,  m_maxLoadPerAxleOfVehicle(0)
      ,  m_numberOfAxles(0)
      ,  m_ecoVehicleType(0)
      ,  m_ecoEngineType(0)
      ,  m_optionRoofLoad(USE_NO_ROOFLOAD)
      ,  m_optionTrailer(USE_NO_TRAILER)
      ,  m_vehicleProfileIconId(0)
      ,  m_useTruckRestrictions(false)
      ,  m_isProfileDefinedByUser(false)
   {
   }

   VehicleTypeValue getVehicleType() const
   {
      return m_optionVehicleType;
   }

   void setVehicleType(VehicleTypeValue optionVehicleType)
   {
      m_optionVehicleType = optionVehicleType;
   }

   DistanceUnitSystem getUnitSystem() const
   {
      return m_unitSystemForVehicleProfile;
   }

   void setUnitSystem(DistanceUnitSystem unitSystem)
   {
      m_unitSystemForVehicleProfile = unitSystem;
   }

   RoofLoadValue getRoofLoad() const
   {
      return m_optionRoofLoad;
   }

   void setRoofLoad(RoofLoadValue optionRoofLoad)
   {
      m_optionRoofLoad = optionRoofLoad;
   }

   TrailerValue getTrailer() const
   {
      return m_optionTrailer;
   }

   void setTrailer(TrailerValue optionTrailer)
   {
      m_optionTrailer = optionTrailer;
   }

   const ::std::vector<CargoType>& getCargoTypes() const
   {
      return m_cargoTypes;
   }

   void setCargoTypes(::std::vector<CargoType> cargoTypes)
   {
      m_cargoTypes = cargoTypes;
   }

   unsigned int getMaxSpeed() const
   {
      return m_maxSpeed;
   }

   void setMaxSpeed(unsigned int maxSpeed)
   {
      m_maxSpeed = maxSpeed;
   }

   unsigned int getHeightOfVehicle() const
   {
      return m_heightOfVehicle;
   }

   void setHeightOfVehicle(unsigned int height)
   {
      m_heightOfVehicle = height;
   }

   unsigned int getWidthOfVehicle() const
   {
      return m_widthOfVehicle;
   }

   void setWidthOfVehicle(unsigned int width)
   {
      m_widthOfVehicle = width;
   }

   unsigned int getLengthOfVehicle() const
   {
      return m_lengthOfVehicle;
   }

   void setLengthOfVehicle(unsigned int length)
   {
      m_lengthOfVehicle = length;
   }

   unsigned int getMaxWeightOfVehicle() const
   {
      return m_maxWeightOfVehicle;
   }

   void setMaxWeightOfVehicle(unsigned int weight)
   {
      m_maxWeightOfVehicle = weight;
   }

   unsigned int getMaxLoadPerAxleOfVehicle() const
   {
      return m_maxLoadPerAxleOfVehicle;
   }

   void setMaxLoadPerAxleOfVehicle(unsigned int weight)
   {
      m_maxLoadPerAxleOfVehicle = weight;
   }

   unsigned int getNumberOfAxles() const
   {
      return m_numberOfAxles;
   }

   void setNumberOfAxles(unsigned int numberOfAxles)
   {
      m_numberOfAxles = numberOfAxles;
   }

   unsigned int getEcoVehicleType() const
   {
      return m_ecoVehicleType;
   }

   void setEcoVehicleType(unsigned int vehicleType)
   {
      m_ecoVehicleType = vehicleType;
   }

   unsigned int getEcoEngineType() const
   {
      return m_ecoEngineType;
   }

   void setEcoEngineType(unsigned int engineType)
   {
      m_ecoEngineType = engineType;
   }

   const ::std::string& getVehicleProfileName() const
   {
      return m_vehicleProfileName;
   }

   void setVehicleProfileName(const ::std::string& vehicleProfileName)
   {
      m_vehicleProfileName = vehicleProfileName;
   }

   uint64_t  getVehicleProfileIconId() const
   {
      return m_vehicleProfileIconId;
   }

   void setVehicleProfileIconId(uint64_t vehicleProfileIconId)
   {
      m_vehicleProfileIconId = vehicleProfileIconId;
   }

   bool getUseTruckRestrictions() const
   {
      return m_useTruckRestrictions;
   }

   void setUseTruckRestrictions(bool useTruckRestrictions)
   {
      m_useTruckRestrictions = useTruckRestrictions;
   }

   bool getIsProfileDefinedByUser() const
   {
      return m_isProfileDefinedByUser;
   }

   void setIsProfileDefinedByUser(bool isProfileDefinedByUser)
   {
      m_isProfileDefinedByUser = isProfileDefinedByUser;
   }

   bool operator == (const VehicleProfile& rhs) const
   {
      return (m_optionVehicleType == rhs.m_optionVehicleType) &&
             (m_unitSystemForVehicleProfile == rhs.m_unitSystemForVehicleProfile) &&
             (m_optionRoofLoad == rhs.m_optionRoofLoad) &&
             (m_optionTrailer == rhs.m_optionTrailer) &&
             (m_cargoTypes == rhs.m_cargoTypes) &&
             (m_maxSpeed == rhs.m_maxSpeed) &&
             (m_heightOfVehicle == rhs.m_heightOfVehicle) &&
             (m_widthOfVehicle == rhs.m_widthOfVehicle) &&
             (m_lengthOfVehicle == rhs.m_lengthOfVehicle) &&
             (m_maxWeightOfVehicle == rhs.m_maxWeightOfVehicle) &&
             (m_maxLoadPerAxleOfVehicle == rhs.m_maxLoadPerAxleOfVehicle) &&
             (m_numberOfAxles == rhs.m_numberOfAxles) &&
             (m_ecoVehicleType == rhs.m_ecoVehicleType) &&
             (m_ecoEngineType == rhs.m_ecoEngineType) &&
             (m_vehicleProfileName == rhs.m_vehicleProfileName) &&
             (m_vehicleProfileIconId == rhs.m_vehicleProfileIconId) &&
             (m_useTruckRestrictions == rhs.m_useTruckRestrictions) &&
             (m_isProfileDefinedByUser == rhs.m_isProfileDefinedByUser);
   }

   bool operator != (const VehicleProfile& rhs) const
   {
      return !(*this == rhs);
   }

private:
   VehicleTypeValue m_optionVehicleType;
   DistanceUnitSystem m_unitSystemForVehicleProfile;
   unsigned int m_maxSpeed;
   /**
    Client to set dimension in following units to avoid loss of precision while saving/restoring profile from db.
    Metric : millimeter
    Imperial: inch
    */
   unsigned int m_heightOfVehicle;
   /**
    Client to set dimension in following units to avoid loss of precision while saving/restoring profile from db.
    Metric : millimeter
    Imperial: inch
    */
   unsigned int m_widthOfVehicle;
   /**
    Client to set dimension in following units to avoid loss of precision while saving/restoring profile from db.
    Metric : millimeter
    Imperial: inch
    */
   unsigned int m_lengthOfVehicle;
   /**
    Client to set weight in following units to avoid loss of precision while saving/restoring profile from db.
    Metric : kilogram
    Imperial: pound
    */
   unsigned int m_maxWeightOfVehicle;
   /**
    Client to set weight in following units to avoid loss of precision while saving/restoring profile from db.
    Metric : kilogram
    Imperial: pound
    */
   unsigned int m_maxLoadPerAxleOfVehicle;
   unsigned int m_numberOfAxles;
   unsigned int m_ecoVehicleType;
   unsigned int m_ecoEngineType;
   RoofLoadValue m_optionRoofLoad;
   TrailerValue m_optionTrailer;
   ::std::vector<CargoType> m_cargoTypes;
   ::std::string m_vehicleProfileName;
   uint64_t m_vehicleProfileIconId;
   bool m_useTruckRestrictions;
   /**
    Client has to set following variable to true if the profile is defined by user.
    By default it is false.
    */
   bool m_isProfileDefinedByUser;
};

enum ActiveVehicleProfileType
{
   USE_VEHICLETYPE_USER_1 = 0,
   USE_VEHICLETYPE_USER_2 = 1,
   USE_VEHICLETYPE_USER_3 = 2,
   USE_VEHICLETYPE_DEFAULT = 3
};

class VehicleProfileSettings
{
public:
   VehicleProfileSettings()
   /**(TODO)csb1kor: Ways to configure the Number of LCV Profiles to be supported is not clear yet */
      :  m_configNumOfVehicleProfilesSupported(NUMBER_OF_USER_VEHICLE_PROFILES_SUPPORTED)
      ,  m_activeVehicleProfileType(USE_VEHICLETYPE_DEFAULT)
   {
   }

   const VehicleProfile& getCurrentVehicleProfile() const
   {
      if(USE_VEHICLETYPE_DEFAULT == m_activeVehicleProfileType)
      {
         return m_defaultVehicleProfile;
      }
      else
      {
         return m_vehicleProfilesList[m_activeVehicleProfileType];
      }
   }

   void setCurrentVehicleProfile(const VehicleProfile& currentVehicleProfile)
   {
      if(USE_VEHICLETYPE_DEFAULT != m_activeVehicleProfileType)
      {
         m_vehicleProfilesList[m_activeVehicleProfileType] = currentVehicleProfile;
      }
   }

   void setDefaultVehicleProfile(const VehicleProfile& vehicleProfile)
   {
      m_defaultVehicleProfile = vehicleProfile;
   }

   const VehicleProfile& getDefaultVehicleProfile() const
   {
      return m_defaultVehicleProfile;
   }

   const VehicleProfile& getUserVehicleProfile(unsigned int index) const
   {
      if(index >= m_configNumOfVehicleProfilesSupported)
      {
         index = 0;
      }
      return m_vehicleProfilesList[index];
   }

   VehicleProfile& getUserVehicleProfile(unsigned int index)
   {
      if(index >= m_configNumOfVehicleProfilesSupported)
      {
         index = 0;
      }
      return m_vehicleProfilesList[index];
   }

   void setUserVehicleProfile(unsigned int index,const VehicleProfile& vehicleProfile)
   {
      if(index < m_configNumOfVehicleProfilesSupported)
      {
         m_vehicleProfilesList[index] = vehicleProfile;
      }
   }

   ActiveVehicleProfileType getCurrentVehicleProfileType() const
   {
      return m_activeVehicleProfileType;
   }

   void setCurrentVehicleProfileType(ActiveVehicleProfileType activeVehicleProfileType)
   {
      m_activeVehicleProfileType = activeVehicleProfileType;
   }

   VehicleProfileSettings& operator = (const VehicleProfileSettings& rhs)
   {
      if (&rhs != this)
      {
         for(size_t vehicleProfileIdx = 0; vehicleProfileIdx < (sizeof(m_vehicleProfilesList)/sizeof(VehicleProfile));
               vehicleProfileIdx++)
         {
            m_vehicleProfilesList[vehicleProfileIdx] = rhs.m_vehicleProfilesList[vehicleProfileIdx];
         }

         m_defaultVehicleProfile = rhs.m_defaultVehicleProfile;
         m_configNumOfVehicleProfilesSupported = rhs.m_configNumOfVehicleProfilesSupported;
         m_activeVehicleProfileType = rhs.m_activeVehicleProfileType;
      }

      return *this;
   }

   bool operator == (const VehicleProfileSettings& rhs) const
   {
      bool areVehicleProfilesListEqual = true;

      for(size_t vehicleProfileIdx = 0; vehicleProfileIdx < (sizeof(m_vehicleProfilesList)/sizeof(VehicleProfile));
            vehicleProfileIdx++)
      {
         if( m_vehicleProfilesList[vehicleProfileIdx] != rhs.m_vehicleProfilesList[vehicleProfileIdx] )
         {
            areVehicleProfilesListEqual = false;
            break;
         }
      }

      return ((m_defaultVehicleProfile == rhs.m_defaultVehicleProfile) &&
              areVehicleProfilesListEqual &&
              (m_configNumOfVehicleProfilesSupported == rhs.m_configNumOfVehicleProfilesSupported) &&
              (m_activeVehicleProfileType == rhs.m_activeVehicleProfileType));
   }

   bool operator != (const VehicleProfileSettings& rhs) const
   {
      return !(*this == rhs);
   }

   static const unsigned int NUMBER_OF_USER_VEHICLE_PROFILES_SUPPORTED = 3;
private:
   VehicleProfile m_defaultVehicleProfile;
   VehicleProfile m_vehicleProfilesList[NUMBER_OF_USER_VEHICLE_PROFILES_SUPPORTED];
   unsigned int m_configNumOfVehicleProfilesSupported;
   ActiveVehicleProfileType m_activeVehicleProfileType;
};

class RouteOptionsSettings
{
public:
   RouteOptionsSettings()
      :  m_optionMotorways(USE_MOTORWAYS_ON)
      ,  m_optionSubscriptionRoads(USE_SUBSCRIPTION_ROADS_ON)
      ,  m_optionTimeRestrictedRoads(USE_TIME_RESTRICTED_ROADS_WHEN_OPEN)
      ,  m_optionSeasonalRestrictedRoads(USE_SEASONAL_RESTRICTED_ROADS_WHEN_OPEN)
      ,  m_optionTollRoads(USE_TOLL_ROADS_VALUE__ALWAYS)
      ,  m_optionFerries(USE_FERRIES_ON)
      ,  m_optionUnpavedRoads(USE_UNPAVED_ROADS_ON)
      ,  m_optionIPD(USE_IPD_ON)
      ,  m_optionRealTimeTraffic(USE_REALTIME_TRAFFIC_PROMPT)
      ,  m_optionBorderCrossings(USE_BORDER_CROSSING_ON)
      ,  m_optionAvoidArea(USE_AVOID_AREA_OFF)
      ,  m_optionCarShuttleTrain(USE_CAR_SHUTTLE_TRAIN_ON)
      ,  m_numPassengersHOVLanes(1)
      ,  m_optionLearnedRoutes(USE_LEARNED_ROUTES_OFF)
      ,  m_optionLearningNavigation(USE_LEARNING_NAVIGATION__OFF)
      ,  m_moreRoutesType(MORE_ROUTES_TYPE_MULTIPLE)
      ,  m_optionFreeways(USE_FREEWAYS_ON)
      ,  m_isHOVLaneGuidanceActive(USE_HOV_LANES_OFF)
      ,  m_optionUseRouteAlternative(USE_ROUTE_ALTERNATIVE_OFF)
      ,  m_optionUseEndToEndNavigation(USE_END_TO_END_NAVIGATION_ON)
      ,  m_optionUseAutomaticGuidanceProposal(USE_AUTOMATIC_GUIDANCE_PROPOSAL__ON)
      ,  m_optionChargingTimeEPois(USE_CHARGING_TIME_EPOIS__OFF)
      ,  m_optionUseChargingStationAssistant(USE_CHARGING_STATION_ASSISTANT__OFF)
      ,  m_estimatedTime(ESTIMATED_TIME__CURRENT_DESTINATION)
      ,  m_destinationTimeDisplay(DESTINATION_TIME_DISPLAY__TIME_OF_ARRIVAL)
      ,  m_optionAutoReroute(USE_AUTO_REROUTE_ON)
      ,  m_optionSmartIC(USE_SMART_IC_ON)
      ,  m_optionAlongside(USE_ALONGSIDE_ON)
      ,  m_optionClosedForWinter(USE_SEASONAL_RESTRICTED_ROADS_WHEN_OPEN)
      //<--INF4CV
      ,  m_optionTunnels(USE_TUNNELS_ON)
      ,  m_optionUTurns(USE_UTURNS_OFF)
      ,  m_optionLowClassifiedRoads(USE_LOW_CLASSIFIED_ROADS_ON)
      //->
   {
   }

   UseMotorwaysValue getUseMotorways() const
   {
      return m_optionMotorways;
   }

   void setUseMotorways(UseMotorwaysValue optionMotorways)
   {
      m_optionMotorways = optionMotorways;
   }

   UseSubscriptionRoadsValue getUseSubscriptionRoads() const
   {
      return m_optionSubscriptionRoads;
   }

   void setUseSubscriptionRoads(UseSubscriptionRoadsValue optionSubscriptionRoads)
   {
      m_optionSubscriptionRoads = optionSubscriptionRoads;
   }

   UseTimeRestrictedRoadsValue getUseTimeRestrictedRoads() const
   {
      return m_optionTimeRestrictedRoads;
   }

   void setUseTimeRestrictedRoads(UseTimeRestrictedRoadsValue optionTimeRestrictedRoads)
   {
      m_optionTimeRestrictedRoads = optionTimeRestrictedRoads;
   }

   UseSeasonalRestrictedRoadsValue getUseSeasonalRestrictedRoads() const
   {
      return m_optionSeasonalRestrictedRoads;
   }

   void setUseSeasonalRestrictedRoads(UseSeasonalRestrictedRoadsValue optionSeasonalRestrictedRoads)
   {
      m_optionSeasonalRestrictedRoads = optionSeasonalRestrictedRoads;
   }

   UseTollRoadsValue getUseTollRoads() const
   {
      return m_optionTollRoads;
   }

   void setUseTollRoads(UseTollRoadsValue optionTollRoads)
   {
      m_optionTollRoads = optionTollRoads;
   }

   UseFerriesValue getUseFerries() const
   {
      return m_optionFerries;
   }

   void setUseFerries(UseFerriesValue optionFerries)
   {
      m_optionFerries = optionFerries;
   }

   UseUnpavedRoadsValue getUseUnpavedRoads() const
   {
      return m_optionUnpavedRoads;
   }

   void setUseUnpavedRoads(UseUnpavedRoadsValue optionUnpavedRoads)
   {
      m_optionUnpavedRoads = optionUnpavedRoads;
   }

   UseIPDValue getUseIPD() const
   {
      return m_optionIPD;
   }

   void setUseIPD(UseIPDValue optionIPD)
   {
      m_optionIPD = optionIPD;
   }

   UseRealTimeTrafficInfoValue getUseRealTimeTrafficInfo() const
   {
      return m_optionRealTimeTraffic;
   }

   void setUseRealTimeTrafficInfo(UseRealTimeTrafficInfoValue optionRealTimeTraffic)
   {
      m_optionRealTimeTraffic = optionRealTimeTraffic;
   }

   UseBorderCrossingsValue getUseBorderCrossings() const
   {
      return m_optionBorderCrossings;
   }

   void setUseBorderCrossings(UseBorderCrossingsValue optionBorderCrossings)
   {
      m_optionBorderCrossings = optionBorderCrossings;
   }

   UseAvoidAreaValue getUseAvoidArea() const
   {
      return m_optionAvoidArea;
   }

   void setUseAvoidArea(UseAvoidAreaValue optionAvoidArea)
   {
      m_optionAvoidArea = optionAvoidArea;
   }

   UseCarShuttleTrainValue getUseCarShuttleTrain() const
   {
      return m_optionCarShuttleTrain;
   }

   void setUseCarShuttleTrain(UseCarShuttleTrainValue optionCarShuttleTrain)
   {
      m_optionCarShuttleTrain = optionCarShuttleTrain;
   }

   unsigned int getNumPassengersHOVLanes() const
   {
      return m_numPassengersHOVLanes;
   }

   void setNumPassengersHOVLanes(unsigned int numPassengersHOVLanes)
   {
      m_numPassengersHOVLanes = numPassengersHOVLanes;
   }

   UseLearnedRoutesValue getUseLearnedRoutes() const
   {
      return m_optionLearnedRoutes;
   }

   void setUseLearnedRoutes(UseLearnedRoutesValue optionLearnedRoutes)
   {
      m_optionLearnedRoutes = optionLearnedRoutes;
   }

   UseLearningNavigationValue getUseLearningNavigation() const
   {
      return m_optionLearningNavigation;
   }

   void setUseLearningNavigation(UseLearningNavigationValue optionLearningNavigation)
   {
      m_optionLearningNavigation = optionLearningNavigation;
   }

   MoreRoutesTypeValue getMoreRoutesType() const
   {
      return m_moreRoutesType;
   }

   void setMoreRoutesType(MoreRoutesTypeValue moreRoutesType)
   {
      m_moreRoutesType = moreRoutesType;
   }

   UseFreewaysValue getUseFreeways() const
   {
      return m_optionFreeways;
   }

   void setUseFreeways(UseFreewaysValue optionFreeways)
   {
      m_optionFreeways = optionFreeways;
   }

   UseHOVLanesValue isHOVLaneGuidanceActive() const
   {
      return m_isHOVLaneGuidanceActive;
   }

   void setHOVLaneGuidanceActive(UseHOVLanesValue optionHOVLaneValue)
   {
      m_isHOVLaneGuidanceActive = optionHOVLaneValue;
   }

   void setUseRouteAlternative(UseRouteAlternativeValue optionUseRouteAlternative)
   {
      m_optionUseRouteAlternative = optionUseRouteAlternative;
   }

   UseRouteAlternativeValue getUseRouteAlternative() const
   {
      return m_optionUseRouteAlternative;
   }

   UseEndToEndNavigationValue getUseEndToEndNavigation() const
   {
      return m_optionUseEndToEndNavigation;
   }

   void setUseEndToEndNavigation(UseEndToEndNavigationValue optionUseEndToEndNavigation)
   {
      m_optionUseEndToEndNavigation = optionUseEndToEndNavigation;
   }

   UseAutomaticGuidanceProposalValue getUseAutomaticGuidanceProposal() const
   {
      return m_optionUseAutomaticGuidanceProposal;
   }

   void setUseAutomaticGuidanceProposal(UseAutomaticGuidanceProposalValue optionUseAutomaticGuidanceProposal)
   {
      m_optionUseAutomaticGuidanceProposal = optionUseAutomaticGuidanceProposal;
   }

   UseChargingTimeEPois getUseOptionChargingTimeEPois() const
   {
      return m_optionChargingTimeEPois;
   }

   void setUseOptionChargingTimeEPois(UseChargingTimeEPois optionChargingTimeEPois)
   {
      m_optionChargingTimeEPois = optionChargingTimeEPois;
   }

   UseChargingStationAssistantValue getUseChargingStationAssistant() const
   {
      return m_optionUseChargingStationAssistant;
   }

   void setUseChargingStationAssistant(UseChargingStationAssistantValue optionUseChargingStationAssistant)
   {
      m_optionUseChargingStationAssistant = optionUseChargingStationAssistant;
   }

   void setEstimatedTime(EstimatedTime estimatedTime)
   {
      m_estimatedTime = estimatedTime;
   }

   EstimatedTime getEstimatedTime() const
   {
      return m_estimatedTime;
   }

   void setDestinationTimeDisplay(DestinationTimeDisplay destinationTimeDisplay)
   {
      m_destinationTimeDisplay = destinationTimeDisplay;
   }

   DestinationTimeDisplay getDestinationTimeDisplay() const
   {
      return m_destinationTimeDisplay;
   }

   //<--JP
   void setClosedForWinter(UseSeasonalRestrictedRoadsValue optionClosedForWinter)
   {
      m_optionClosedForWinter = optionClosedForWinter;
   }

   UseSeasonalRestrictedRoadsValue getClosedForWinter() const
   {
      return m_optionClosedForWinter;
   }

   UseAutoRerouteValue getAutoReroute() const
   {
      return m_optionAutoReroute;
   }

   void setAutoReroute(UseAutoRerouteValue optionRerouteValue)
   {
      m_optionAutoReroute = optionRerouteValue;
   }

   UseSmartIC getUseSmartIC() const
   {
      return m_optionSmartIC;
   }

   void setUseSmartIC(UseSmartIC optionSmartICValue)
   {
      m_optionSmartIC = optionSmartICValue;
   }

   UseAlongside getUseAlongside() const
   {
      return m_optionAlongside;
   }

   void setUseAlongside(UseAlongside optionAlongside)
   {
      m_optionAlongside = optionAlongside;
   }
   //->

   //<--INF4CV
   UseTunnelsValue getUseTunnels() const
   {
      return m_optionTunnels;
   }

   void setUseTunnels(UseTunnelsValue optionTunnels)
   {
      m_optionTunnels = optionTunnels;
   }

   UseUTurns getUseUTurns() const
   {
      return m_optionUTurns;
   }

   void setUseUTurns(UseUTurns optionUTurns)
   {
      m_optionUTurns = optionUTurns;
   }

   UseLowClassifiedRoads getUseLowClassifiedRoads() const
   {
      return m_optionLowClassifiedRoads;
   }

   void setUseLowClassifiedRoads(UseLowClassifiedRoads lowClassifiedRoads)
   {
      m_optionLowClassifiedRoads = lowClassifiedRoads;
   }
   //->

   bool operator == (const RouteOptionsSettings& rhs) const
   {
      return (m_optionMotorways == rhs.m_optionMotorways) &&
             (m_optionSubscriptionRoads == rhs.m_optionSubscriptionRoads) &&
             (m_optionTimeRestrictedRoads == rhs.m_optionTimeRestrictedRoads) &&
             (m_optionSeasonalRestrictedRoads == rhs.m_optionSeasonalRestrictedRoads) &&
             (m_optionTollRoads == rhs.m_optionTollRoads) &&
             (m_optionFerries == rhs.m_optionFerries) &&
             (m_optionUnpavedRoads == rhs.m_optionUnpavedRoads) &&
             (m_optionIPD == rhs.m_optionIPD)  &&
             (m_optionRealTimeTraffic == rhs.m_optionRealTimeTraffic) &&
             (m_optionBorderCrossings == rhs.m_optionBorderCrossings) &&
             (m_optionAvoidArea == rhs.m_optionAvoidArea) &&
             (m_optionCarShuttleTrain == rhs.m_optionCarShuttleTrain) &&
             (m_numPassengersHOVLanes == rhs.m_numPassengersHOVLanes) &&
             (m_optionLearnedRoutes == rhs.m_optionLearnedRoutes) &&
             (m_optionLearningNavigation == rhs.m_optionLearningNavigation) &&
             (m_moreRoutesType == rhs.m_moreRoutesType) &&
             (m_optionFreeways == rhs.m_optionFreeways) &&
             (m_isHOVLaneGuidanceActive == rhs.m_isHOVLaneGuidanceActive) &&
             (m_optionUseRouteAlternative == rhs.m_optionUseRouteAlternative) &&
             (m_optionUseEndToEndNavigation == rhs.m_optionUseEndToEndNavigation) &&
             (m_optionUseAutomaticGuidanceProposal == rhs.m_optionUseAutomaticGuidanceProposal)&&
             (m_optionChargingTimeEPois == rhs.m_optionChargingTimeEPois) &&
             (m_optionUseChargingStationAssistant == rhs.m_optionUseChargingStationAssistant) &&
             (m_estimatedTime == rhs.m_estimatedTime) &&
             (m_destinationTimeDisplay == rhs.m_destinationTimeDisplay) &&
             (m_optionAutoReroute == rhs.m_optionAutoReroute) &&
             (m_optionSmartIC == rhs.m_optionSmartIC) &&
             (m_optionAlongside == rhs.m_optionAlongside) &&
             //<--INF4CV
             (m_optionTunnels == rhs.m_optionTunnels) &&
             (m_optionUTurns == rhs.m_optionUTurns) &&
             (m_optionLowClassifiedRoads == rhs.m_optionLowClassifiedRoads) &&
             //->
             (m_optionClosedForWinter == rhs.m_optionClosedForWinter);
   }

   bool operator != (const RouteOptionsSettings& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString()const
   {
      ::std::stringstream stream("RouteOptionsSettings payload:\n");
      stream << "m_optionMotorways = " << ::navmiddleware::settings::toString(m_optionMotorways)<< ::std::endl
             << "m_optionSubscriptionRoads = "<< ::navmiddleware::settings::toString(m_optionSubscriptionRoads)<< ::std::endl
             << "m_optionTimeRestrictedRoads = "<< ::navmiddleware::settings::toString(m_optionTimeRestrictedRoads)<< ::std::endl
             << "m_optionSeasonalRestrictedRoads = " << ::navmiddleware::settings::toString(m_optionSeasonalRestrictedRoads)<< ::std::endl
             << "m_optionTollRoads = " << ::navmiddleware::settings::toString(m_optionTollRoads)<< ::std::endl
             << "m_optionFerries = "<< ::navmiddleware::settings::toString(m_optionFerries)<< ::std::endl
             << "m_optionUnpavedRoads = "<< ::navmiddleware::settings::toString(m_optionUnpavedRoads)<< ::std::endl
             << "m_optionIPD = " << ::navmiddleware::settings::toString(m_optionIPD)<< ::std::endl
             << "m_optionRealTimeTraffic = " << ::navmiddleware::settings::toString(m_optionRealTimeTraffic)<< ::std::endl
             << "m_optionBorderCrossings = "<< ::navmiddleware::settings::toString(m_optionBorderCrossings)<< ::std::endl
             << "m_optionAvoidArea = "<< ::navmiddleware::settings::toString(m_optionAvoidArea)<< ::std::endl
             << "m_optionCarShuttleTrain = "<< ::navmiddleware::settings::toString(m_optionCarShuttleTrain)<< ::std::endl
             << "m_numPassengersHOVLanes = "<< m_numPassengersHOVLanes<< ::std::endl
             << "m_optionLearnedRoutes = "<< ::navmiddleware::settings::toString(m_optionLearnedRoutes)<< ::std::endl
             << "m_optionLearningNavigation = "<< ::navmiddleware::settings::toString(m_optionLearningNavigation)<< ::std::endl
             << "m_moreRoutesType = "<< ::navmiddleware::settings::toString(m_moreRoutesType)<< ::std::endl
             << "m_optionFreeways = "<< ::navmiddleware::settings::toString(m_optionFreeways)<< ::std::endl
             << "m_isHOVLaneGuidanceActive = " << ::navmiddleware::settings::toString(m_isHOVLaneGuidanceActive)<< ::std::endl
             << "m_optionUseRouteAlternative = "<< ::navmiddleware::settings::toString(m_optionUseRouteAlternative)<< ::std::endl
             << "m_optionUseEndToEndNavigation = "<< ::navmiddleware::settings::toString(m_optionUseEndToEndNavigation)<< ::std::endl
             << "m_optionUseAutomaticGuidanceProposal = "<< ::navmiddleware::settings::toString(m_optionUseAutomaticGuidanceProposal)<< ::std::endl
             << "m_optionChargingTimeEPois = "<< ::navmiddleware::settings::toString(m_optionChargingTimeEPois)<< ::std::endl
             << "m_optionUseChargingStationAssistant = "<< ::navmiddleware::settings::toString(m_optionUseChargingStationAssistant)<< ::std::endl
             << "m_estimatedTime = "<< ::navmiddleware::settings::toString(m_estimatedTime)<< ::std::endl
             << "m_destinationTimeDisplay = "<< ::navmiddleware::settings::toString(m_destinationTimeDisplay)<< ::std::endl
             << "m_optionAutoReroute = "<< ::navmiddleware::settings::toString(m_optionAutoReroute)<< ::std::endl
             << "m_optionSmartIC = "<< ::navmiddleware::settings::toString(m_optionSmartIC)<< ::std::endl
             << "m_optionAlongside = "<< ::navmiddleware::settings::toString(m_optionAlongside)<< ::std::endl
             //<--INF4CV
             << "m_optionTunnels = " << ::navmiddleware::settings::toString(m_optionTunnels) << ::std::endl
             << "m_optionUTurns = " << ::navmiddleware::settings::toString(m_optionUTurns) << ::std::endl
             << "m_optionLowClassifiedRoads = " << ::navmiddleware::settings::toString(m_optionLowClassifiedRoads) << ::std::endl
             //->
             << "m_optionClosedForWinter = "<< ::navmiddleware::settings::toString(m_optionClosedForWinter)<< ::std::endl;
      return stream.str();
   }

private:
   UseMotorwaysValue m_optionMotorways;
   UseSubscriptionRoadsValue m_optionSubscriptionRoads;
   UseTimeRestrictedRoadsValue m_optionTimeRestrictedRoads;
   UseSeasonalRestrictedRoadsValue m_optionSeasonalRestrictedRoads;
   UseTollRoadsValue m_optionTollRoads;
   UseFerriesValue m_optionFerries;
   UseUnpavedRoadsValue m_optionUnpavedRoads;
   UseIPDValue m_optionIPD;
   UseRealTimeTrafficInfoValue m_optionRealTimeTraffic;
   UseBorderCrossingsValue m_optionBorderCrossings;
   UseAvoidAreaValue m_optionAvoidArea;
   UseCarShuttleTrainValue m_optionCarShuttleTrain;
   unsigned int m_numPassengersHOVLanes;
   UseLearnedRoutesValue m_optionLearnedRoutes;
   UseLearningNavigationValue m_optionLearningNavigation;
   MoreRoutesTypeValue m_moreRoutesType;
   UseFreewaysValue m_optionFreeways;
   UseHOVLanesValue m_isHOVLaneGuidanceActive;
   UseRouteAlternativeValue m_optionUseRouteAlternative;
   UseEndToEndNavigationValue m_optionUseEndToEndNavigation;
   UseAutomaticGuidanceProposalValue m_optionUseAutomaticGuidanceProposal;
   UseChargingTimeEPois m_optionChargingTimeEPois;
   UseChargingStationAssistantValue m_optionUseChargingStationAssistant;
   EstimatedTime m_estimatedTime;
   DestinationTimeDisplay m_destinationTimeDisplay;
   //<--JP
   UseAutoRerouteValue m_optionAutoReroute;
   UseSmartIC m_optionSmartIC;
   UseAlongside m_optionAlongside;
   UseSeasonalRestrictedRoadsValue m_optionClosedForWinter;
   //->
   //<--INF4CV
   UseTunnelsValue m_optionTunnels;
   UseUTurns m_optionUTurns;
   UseLowClassifiedRoads m_optionLowClassifiedRoads;
   //->
};


}
}
#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_ROUTESETTINGS_H_
