/**
* @copyright (c) 2015-2022 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_POSITIONINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_POSITIONINFOS_H_

#include <string>
#include <sstream>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{

enum PositionQuality
{
   POSITION_QUALITY__UNKNOWN,
   POSITION_QUALITY__SUFFICIENT,
   POSITION_QUALITY__INSUFFICIENT
};
inline ::std::string toString(PositionQuality quality)
{
   switch (quality)
   {
   case POSITION_QUALITY__UNKNOWN:
      return "POSITION_QUALITY__UNKNOWN";
   case POSITION_QUALITY__SUFFICIENT:
      return "POSITION_QUALITY__SUFFICIENT";
   case POSITION_QUALITY__INSUFFICIENT:
      return "POSITION_QUALITY__INSUFFICIENT";
   default:
      ::std::stringstream stream;
      stream << "POSITION_QUALITY__<" << static_cast<unsigned int>(quality) << ">";
      return stream.str();
   }
}

/**
 * Status about availability of RNW (RoadNetWork) data (at some location),
 * i.e. is there some street (map) data present which could be used for navigation
 */
enum RnwDataAvailability
{
   RNW_DATA_AVAILABILITY__UNKNOWN,
   RNW_DATA_AVAILABILITY__SUFFICIENT,
   RNW_DATA_AVAILABILITY__INSUFFICIENT
};
inline ::std::string toString(RnwDataAvailability availability)
{
   switch (availability)
   {
   case RNW_DATA_AVAILABILITY__UNKNOWN:
      return "RNW_DATA_AVAILABILITY__UNKNOWN";
   case RNW_DATA_AVAILABILITY__SUFFICIENT:
      return "RNW_DATA_AVAILABILITY__SUFFICIENT";
   case RNW_DATA_AVAILABILITY__INSUFFICIENT:
      return "RNW_DATA_AVAILABILITY__INSUFFICIENT";
   default:
      ::std::stringstream stream;
      stream << "RNW_DATA_AVAILABILITY__<" << static_cast<unsigned int>(availability) << ">";
      return stream.str();
   }
}

/**
 * Status about the mobility of the vehicle,
 * i.e. is the real vehicle is moving or not (independent of possibly active demo mode)
 */
enum VehicleMovingStatus
{
   VEHICLE_MOVING_STATUS__UNKNOWN,
   VEHICLE_MOVING_STATUS__NOT_MOVING,
   VEHICLE_MOVING_STATUS__MOVING
};
inline ::std::string toString(VehicleMovingStatus status)
{
   switch (status)
   {
   case VEHICLE_MOVING_STATUS__UNKNOWN:
      return "VEHICLE_MOVING_STATUS__UNKNOWN";
   case VEHICLE_MOVING_STATUS__NOT_MOVING:
      return "VEHICLE_MOVING_STATUS__NOT_MOVING";
   case VEHICLE_MOVING_STATUS__MOVING:
      return "VEHICLE_MOVING_STATUS__MOVING";
   default:
      ::std::stringstream stream;
      stream << "VEHICLE_MOVING_STATUS__<" << static_cast<unsigned int>(status) << ">";
      return stream.str();
   }
}

/**
 * Status about the vehicle position in relation with HOME location,
 * i.e. is the vehicle at HOME location or not?
 */
enum HomePositionStatus
{
   /* No Navigation is present in the system */
   HOME_POSITION_STATUS__NAVI_NOT_ADOPTED,
   /* Navigation is present, but is currently unable to provide the Home Status */
   HOME_POSITION_STATUS__UNDER_OPERATION,
   /* Vehicle position is less than 100m away from the home location stored in the address book */
   HOME_POSITION_STATUS__IN_HOME,
   /* Vehicle position is more than 100m away from the home location stored in the address book */
   HOME_POSITION_STATUS__OUT_OF_HOME,
   /* No home location is stored in the address book */
   HOME_POSITION_STATUS__HOME_NOT_SET
};
inline ::std::string toString(HomePositionStatus status)
{
   switch (status)
   {
   case HOME_POSITION_STATUS__NAVI_NOT_ADOPTED:
      return "HOME_POSITION_STATUS__NAVI_NOT_ADOPTED";
   case HOME_POSITION_STATUS__UNDER_OPERATION:
      return "HOME_POSITION_STATUS__UNDER_OPERATION";
   case HOME_POSITION_STATUS__IN_HOME:
      return "HOME_POSITION_STATUS__IN_HOME";
   case HOME_POSITION_STATUS__OUT_OF_HOME:
      return "HOME_POSITION_STATUS__OUT_OF_HOME";
   case HOME_POSITION_STATUS__HOME_NOT_SET:
      return "HOME_POSITION_STATUS__HOME_NOT_SET";
   default:
      ::std::stringstream stream;
      stream << "HOME_POSITION_STATUS__<" << static_cast<unsigned int>(status) << ">";
      return stream.str();
   }
}

class PositionInfo
{
public:
   class TimeZoneInfo
   {
   public:
      TimeZoneInfo(int32_t timeZoneOffset, int32_t daylightSavingTime)
         : m_timeZoneOffset(timeZoneOffset),
           m_daylightSavingTime(daylightSavingTime)
      {}

      void setTimeZoneOffset(int32_t timeZoneOffset)
      {
         m_timeZoneOffset = timeZoneOffset;
      }

      int32_t getTimeZoneOffset() const
      {
         return (m_timeZoneOffset);
      }

      void setDaylightSavingTime(int32_t daylightSavingTime)
      {
         m_daylightSavingTime = daylightSavingTime;
      }

      int32_t getDaylightSavingTime() const
      {
         return (m_daylightSavingTime);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("TimeZoneInfo payload:\n");
          stream << ::std::endl
                 << "m_timeZoneOffset = " << m_timeZoneOffset << ::std::endl
                 << "m_daylightSavingTime = " << m_daylightSavingTime << ::std::endl;

          return stream.str();
      }

      bool operator==(const TimeZoneInfo& other) const
      {
         return (this->m_timeZoneOffset == other.m_timeZoneOffset)
                &&  (this->m_daylightSavingTime == other.m_daylightSavingTime);
      }

      bool operator!=(const TimeZoneInfo& other) const
      {
         return !(*this == other);
      }

   private:
      int32_t m_timeZoneOffset;
      int32_t m_daylightSavingTime;
   };
   
   PositionInfo()
      : m_timeZoneInfo(0, 0)
      {}

   void setCurrentCountry(const ::std::string& currentCountry)
   {
      m_currentCountry = currentCountry;
   }
   const ::std::string& getCurrentCountry() const
   {
      return m_currentCountry;
   }

   void setCurrentState(const ::std::string& currentState)
   {
      m_currentState = currentState;
   }
   const ::std::string& getCurrentState() const
   {
      return m_currentState;
   }

   void setCurrentCity(const ::std::string& currentCity)
   {
      m_currentCity = currentCity;
   }
   const ::std::string& getCurrentCity() const
   {
      return m_currentCity;
   }

   void setCurrentCityDistrict(const ::std::string& currentCityDistrict)
   {
      m_currentCityDistrict = currentCityDistrict;
   }
   const ::std::string& getCurrentCityDistrict() const
   {
      return m_currentCityDistrict;
   }

   void setCurrentStreet(const ::std::string& currentStreet)
   {
      m_currentStreet = currentStreet;
   }
   const ::std::string& getCurrentStreet() const
   {
      return m_currentStreet;
   }

   void setCurrentStreetNumber(const ::std::string& currentStreetNumber)
   {
      m_currentStreetNumber = currentStreetNumber;
   }
   const ::std::string& getCurrentStreetNumber() const
   {
      return m_currentStreetNumber;
   }

   void setPositionInfo(const ::std::string& positionInfo)
   {
      m_positionInfo = positionInfo;
   }
   const ::std::string& getPositionInfo() const
   {
      return m_positionInfo;
   }

   void setCurrentIntersection(const ::std::string& currentIntersection)
   {
      m_currentIntersection = currentIntersection;
   }
   const ::std::string& getCurrentIntersection() const
   {
      return m_currentIntersection;
   }

   void setCurrentZipCode(const ::std::string& currentZipCode)
   {
      m_currentZipCode = currentZipCode;
   }
   const ::std::string& getCurrentZipCode() const
   {
      return m_currentZipCode;
   }
   //[JP]->
   void setTrafficLightAheadState(const ValidValue<bool>& value)
   {
      m_isTrafficLightAhead = value;
   }
   const ValidValue<bool>& isTrafficLightAhead() const
   {
      return m_isTrafficLightAhead;
   }
   //[JP] <-

   void setCountryCode(const ::std::string& countryCode)
   {
      m_countryCode = countryCode;
   }
   const ::std::string& getCountryCode() const
   {
      return m_countryCode;
   }

   void setStateCode(const ::std::string& stateCode)
   {
      m_stateCode = stateCode;
   }
   const ::std::string& getStateCode() const
   {
      return m_stateCode;
   }

   void setUniqueID(const ValidValue<uint32_t>& uniqueID)
   {
      m_uniqueID = uniqueID;
   }

   const ValidValue<uint32_t>& getUniqueID() const
   {
      return m_uniqueID;
   }

   void setTimeZoneInfo(const int32_t timeZoneOffset, const int32_t daylightSavingTime)
   {
      m_timeZoneInfo.setTimeZoneOffset(timeZoneOffset);
      m_timeZoneInfo.setDaylightSavingTime(daylightSavingTime);
   }

   const TimeZoneInfo& getTimeZoneInfo() const
   {
      return (m_timeZoneInfo);
   }


   ::std::string toString() const
   {
      ::std::stringstream stream("PositionInfo payload:\n");
      stream << ::std::endl
             << "Country = " << m_currentCountry << ", "
             << "State = " << m_currentState << ", "
             << "City = " << m_currentCity << ", "
             << "ZIP code = " << m_currentZipCode << ::std::endl
             << "District = " << m_currentCityDistrict << ", "
             << "Street = " << m_currentStreet << ", "
             << "Street Number = " << m_currentStreetNumber << ", "
             << "Intersection = " << m_currentIntersection << ::std::endl
             << "Position Info = " << m_positionInfo << ::std::endl
             << "Traffic Light Ahead = " << m_isTrafficLightAhead.toString() << ::std::endl
             << "CountryCode = " << m_countryCode << ::std::endl
             << "StateCode = " << m_stateCode << ::std::endl
             << "UniqueID = " << m_uniqueID.toString() << ::std::endl
             << "m_timeZoneInfo = " << m_timeZoneInfo.toString() << ::std::endl;

      return stream.str();
   }

   bool operator==(const PositionInfo& other) const
   {
      return (this->m_currentCountry == other.m_currentCountry)
             &&  (this->m_currentState == other.m_currentState)
             &&  (this->m_currentCity == other.m_currentCity)
             &&  (this->m_currentCityDistrict == other.m_currentCityDistrict)
             &&  (this->m_currentStreet == other.m_currentStreet)
             &&  (this->m_currentStreetNumber == other.m_currentStreetNumber)
             &&  (this->m_positionInfo == other.m_positionInfo)
             &&  (this->m_currentIntersection == other.m_currentIntersection)
             &&  (this->m_currentZipCode == other.m_currentZipCode)
             &&  (this->m_isTrafficLightAhead == other.m_isTrafficLightAhead)
             &&  (this->m_countryCode == other.m_countryCode)
             &&  (this->m_stateCode == other.m_stateCode)
             &&  (this->m_uniqueID == other.m_uniqueID)
             &&  (this->m_timeZoneInfo == other.m_timeZoneInfo);
   }
   bool operator!=(const PositionInfo& other) const
   {
      return !(*this == other);
   }

private:
   ::std::string m_currentCountry;
   ::std::string m_currentState;
   ::std::string m_currentCity;
   ::std::string m_currentCityDistrict;
   ::std::string m_currentStreet;
   ::std::string m_currentStreetNumber;
   ::std::string m_positionInfo;
   ::std::string m_currentIntersection; /*JP only*/
   ::std::string m_currentZipCode;
   ValidValue<bool> m_isTrafficLightAhead; /*JP only*/
   ::std::string m_countryCode;
   ::std::string m_stateCode;
   /** m_uniqueID of PositionInfo and PositionStatusInfo have to be same
    *  if content inside the info objects have to be merged **/
   ValidValue<uint32_t> m_uniqueID;
   TimeZoneInfo m_timeZoneInfo;
};

class SdsPositionInfo
{
public:
   class NamedObjectData
   {
   public:
      NamedObjectData()
         : m_namedObjectId(0)
         , m_updateRegionSpecifier(0)
      {}

      NamedObjectData(const ::std::string& objectName, NdsNamedObjectId namedObjectId, NdsUpdateRegionSpecifier updateRegionSpecifier)
         : m_objectName(objectName)
         , m_namedObjectId(namedObjectId)
         , m_updateRegionSpecifier(updateRegionSpecifier)
      {}

      bool operator==(const NamedObjectData& other) const
      {
         return (this->m_objectName == other.m_objectName)
                &&  (this->m_namedObjectId == other.m_namedObjectId)
                &&  (this->m_updateRegionSpecifier == other.m_updateRegionSpecifier);
      }

      bool operator!=(const NamedObjectData& other) const
      {
         return !(*this == other);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "\"" << m_objectName << "\" ("
                << "NamedObjectId=" << m_namedObjectId << ", "
                << "UpdRegionSpec=" << m_updateRegionSpecifier << ")";
         return stream.str();
      }

      ::std::string m_objectName;
      NdsNamedObjectId m_namedObjectId;
      NdsUpdateRegionSpecifier m_updateRegionSpecifier;
   };

   void setCurrentCountry(const NamedObjectData& currentCountry)
   {
      m_currentCountry = currentCountry;
   }
   const NamedObjectData& getCurrentCountry() const
   {
      return m_currentCountry;
   }

   void setCurrentState(const NamedObjectData& currentState)
   {
      m_currentState = currentState;
   }
   const NamedObjectData& getCurrentState() const
   {
      return m_currentState;
   }

   void setCurrentCity(const NamedObjectData& currentCity)
   {
      m_currentCity = currentCity;
   }
   const NamedObjectData& getCurrentCity() const
   {
      return m_currentCity;
   }

   void setCurrentCityDistrict(const NamedObjectData& currentCityDistrict)
   {
      m_currentCityDistrict = currentCityDistrict;
   }
   const NamedObjectData& getCurrentCityDistrict() const
   {
      return m_currentCityDistrict;
   }

   void setCurrentStreet(const NamedObjectData& currentStreet)
   {
      m_currentStreet = currentStreet;
   }
   const NamedObjectData& getCurrentStreet() const
   {
      return m_currentStreet;
   }

   void setCurrentIntersection(const NamedObjectData& currentIntersection)
   {
      m_currentIntersection = currentIntersection;
   }
   const NamedObjectData& getCurrentIntersection() const
   {
      return m_currentIntersection;
   }

   void setCurrentZipCode(const NamedObjectData& currentZipCode)
   {
      m_currentZipCode = currentZipCode;
   }
   const NamedObjectData& getCurrentZipCode() const
   {
      return m_currentZipCode;
   }

   bool operator==(const SdsPositionInfo& other) const
   {
      return (this->m_currentCountry == other.m_currentCountry)
             &&  (this->m_currentState == other.m_currentState)
             &&  (this->m_currentCity == other.m_currentCity)
             &&  (this->m_currentCityDistrict == other.m_currentCityDistrict)
             &&  (this->m_currentStreet == other.m_currentStreet)
             &&  (this->m_currentIntersection == other.m_currentIntersection)
             &&  (this->m_currentZipCode == other.m_currentZipCode);
   }
   bool operator!=(const SdsPositionInfo& other) const
   {
      return !(*this == other);
   }
   ::std::string toString() const
   {
      ::std::stringstream stream("SdsPositionInfo payload:\n");
      stream << ::std::endl
             << "   Country = " << m_currentCountry.toString() <<  ::std::endl
             << "   State = " << m_currentState.toString() <<  ::std::endl
             << "   City = " << m_currentCity.toString() << ::std::endl
             << "   District = " << m_currentCityDistrict.toString() <<  ::std::endl
             << "   Street = " << m_currentStreet.toString() <<  ::std::endl
             << "   Intersection = " << m_currentIntersection.toString() << ::std::endl
             << "   ZIP Code = " << m_currentZipCode.toString() << ::std::endl;
      return stream.str();
   }

private:
   NamedObjectData m_currentCountry;
   NamedObjectData m_currentState;
   NamedObjectData m_currentCity;
   NamedObjectData m_currentCityDistrict;
   NamedObjectData m_currentStreet;
   NamedObjectData m_currentIntersection; /*JP only*/
   NamedObjectData m_currentZipCode;
};

enum PositionStatus
{
   POSITION_STATUS__NO_INFO,
   POSITION_STATUS__NO_DATABASE_AVAILABLE,
   POSITION_STATUS__OFF_MAP,
   POSITION_STATUS__OFF_ROAD,
   POSITION_STATUS__ON_ROAD
};
inline ::std::string toString(PositionStatus status)
{
   switch (status)
   {
   case POSITION_STATUS__NO_INFO:
      return "POSITION_STATUS__NO_INFO";
   case POSITION_STATUS__NO_DATABASE_AVAILABLE:
      return "POSITION_STATUS__NO_DATABASE_AVAILABLE";
   case POSITION_STATUS__OFF_MAP:
      return "POSITION_STATUS__OFF_MAP";
   case POSITION_STATUS__OFF_ROAD:
      return "POSITION_STATUS__OFF_ROAD";
   case POSITION_STATUS__ON_ROAD:
      return "POSITION_STATUS__ON_ROAD";
   default:
      ::std::stringstream stream;
      stream << "POSITION_STATUS__<" << static_cast<unsigned int>(status) << ">";
      return stream.str();
   }
}

enum GnssMode
{
   GNSS_MODE__UNKNOWN,
   GNSS_MODE__NO_FIX,
   GNSS_MODE__2D_FIX,
   GNSS_MODE__3D_FIX
};
inline ::std::string toString(GnssMode mode)
{
   switch (mode)
   {
   case GNSS_MODE__UNKNOWN:
      return "GNSS_MODE__UNKNOWN";
   case GNSS_MODE__NO_FIX:
      return "GNSS_MODE__NO_FIX";
   case GNSS_MODE__2D_FIX:
      return "GNSS_MODE__2D_FIX";
   case GNSS_MODE__3D_FIX:
      return "GNSS_MODE__3D_FIX";
   default:
      ::std::stringstream stream;
      stream << "GNSS_MODE__<" << static_cast<unsigned int>(mode) << ">";
      return stream.str();
   }
}

enum RoadType
{
   ROAD_TYPE__UNKNOWN,
   ROAD_TYPE__NON_DIVIDED_ROAD,
   ROAD_TYPE__DIVIDED_ROAD,
   ROAD_TYPE__CONNECTED_ROAD_JUNCTION,
   ROAD_TYPE__CONNECTED_ROAD_RAMP,
   ROAD_TYPE__SLIP_ROAD,
   ROAD_TYPE__CAR_PARK,
   ROAD_TYPE__ROUNDABOUT,
   ROAD_TYPE__OTHER
};
inline ::std::string toString(RoadType roadType)
{
   switch ( roadType )
   {
      case ROAD_TYPE__UNKNOWN:
         return "ROAD_TYPE__UNKNOWN";
      case ROAD_TYPE__NON_DIVIDED_ROAD:
         return "ROAD_TYPE__NON_DIVIDED_ROAD";
      case ROAD_TYPE__DIVIDED_ROAD:
         return "ROAD_TYPE__DIVIDED_ROAD";
      case ROAD_TYPE__CONNECTED_ROAD_JUNCTION:
         return "ROAD_TYPE__CONNECTED_ROAD_JUNCTION";
      case ROAD_TYPE__CONNECTED_ROAD_RAMP:
         return "ROAD_TYPE__CONNECTED_ROAD_RAMP";
      case ROAD_TYPE__SLIP_ROAD:
         return "ROAD_TYPE__SLIP_ROAD";
      case ROAD_TYPE__CAR_PARK:
         return "ROAD_TYPE__CAR_PARK";
      case ROAD_TYPE__ROUNDABOUT:
         return "ROAD_TYPE__ROUNDABOUT";
      case ROAD_TYPE__OTHER:
         return "ROAD_TYPE__OTHER";
   }
   ::std::stringstream stream;
   stream << "ROAD_TYPE__" << static_cast<unsigned int>(roadType) << ">";
   return stream.str();
}

class PositionStatusInfo
{
public:
   class RoadProperties
   {
   public:
      RoadProperties():
         m_isRestricted(false),
         m_isBuildUp(false),
         m_isSpeedLimit(false),
         m_isJunctionFree(false),
         m_isMultiDigitized(false),
         m_isBasicAttributed(false),
         m_isComplexIntersection(false),
         m_isFerry(false),
         m_isTollRoad(false),
         m_isSeasonalClosed(false),
         m_isTunnel(false),
         m_isNoPass(false),
         m_isHardOnewayInDigitized(false),
         m_isHardOnewayRevDigitized(false),
         m_isBridge(false),
         m_isStubble(false),
         m_isGateway(false),
         m_isAutoTrain(false),
         m_isUnpavedRoad(false),
         m_isRightHandDriving(false),
         m_isHOVRoad(false) {}

      void setIsRestricted (bool isRestricted)
      {
         m_isRestricted = isRestricted;
      }

      bool isRestricted () const
      {
         return m_isRestricted;
      }
      void setIsBuildUp (bool isBuildUp)
      {
         m_isBuildUp = isBuildUp;
      }

      bool isBuildUp () const
      {
         return m_isBuildUp;
      }

      void setIsSpeedLimit (bool isSpeedLimit)
      {
         m_isSpeedLimit = isSpeedLimit;
      }

      bool isSpeedLimit () const
      {
         return m_isSpeedLimit;
      }

      void setIsJunctionFree (bool isJunctionFree)
      {
         m_isJunctionFree = isJunctionFree;
      }

      bool isJunctionFree () const
      {
         return m_isJunctionFree;
      }

      void setIsMultiDigitized (bool isMultiDigitized)
      {
         m_isMultiDigitized = isMultiDigitized;
      }

      bool isMultiDigitized () const
      {
         return m_isMultiDigitized;
      }

      void setIsBasicAttributed (bool isBasicAttributed)
      {
         m_isBasicAttributed = isBasicAttributed;
      }

      bool isBasicAttributed () const
      {
         return m_isBasicAttributed;
      }

      void setIsComplexIntersection (bool isComplexIntersection)
      {
         m_isComplexIntersection = isComplexIntersection;
      }

      bool isComplexIntersection () const
      {
         return m_isComplexIntersection;
      }

      void setIsFerry (bool isFerry)
      {
         m_isFerry = isFerry;
      }

      bool isFerry () const
      {
         return m_isFerry;
      }

      void setIsTollRoad (bool isTollRoad)
      {
         m_isTollRoad = isTollRoad;
      }

      bool isTollRoad () const
      {
         return m_isTollRoad;
      }

      void setIsSeasonalClosed (bool isSeasonalClosed)
      {
         m_isSeasonalClosed = isSeasonalClosed;
      }

      bool isSeasonalClosed () const
      {
         return m_isSeasonalClosed;
      }

      void setIsTunnel (bool isTunnel)
      {
         m_isTunnel = isTunnel;
      }

      bool isTunnel () const
      {
         return m_isTunnel;
      }

      void setIsNoPass (bool isNoPass)
      {
         m_isNoPass = isNoPass;
      }

      bool isNoPass () const
      {
         return m_isNoPass;
      }

      void setIsHardOnewayInDigitized (bool isHardOnewayInDigitized)
      {
         m_isHardOnewayInDigitized = isHardOnewayInDigitized;
      }

      bool isHardOnewayInDigitized () const
      {
         return m_isHardOnewayInDigitized;
      }

      void setIsHardOnewayRevDigitized (bool isHardOnewayRevDigitized)
      {
         m_isHardOnewayRevDigitized = isHardOnewayRevDigitized;
      }

      bool isHardOnewayRevDigitized () const
      {
         return m_isHardOnewayRevDigitized;
      }

      void setIsBridge (bool isBridge)
      {
         m_isBridge = isBridge;
      }

      bool isBridge () const
      {
         return m_isBridge;
      }

      void setIsStubble (bool isStubble)
      {
         m_isStubble = isStubble;
      }

      bool isStubble () const
      {
         return m_isStubble;
      }

      void setIsGateway (bool isGateway)
      {
         m_isGateway = isGateway;
      }

      bool isGateway () const
      {
         return m_isGateway;
      }

      void setIsAutoTrain (bool isAutoTrain)
      {
         m_isAutoTrain = isAutoTrain;
      }

      bool isAutoTrain () const
      {
         return m_isAutoTrain;
      }

      void setIsUnpavedRoad (bool isUnpavedRoad)
      {
         m_isUnpavedRoad = isUnpavedRoad;
      }

      bool isUnpavedRoad () const
      {
         return m_isUnpavedRoad;
      }

      void setIsRightHandDriving (bool isRightHandDriving)
      {
         m_isRightHandDriving = isRightHandDriving;
      }

      bool isRightHandDriving () const
      {
         return m_isRightHandDriving;
      }

      void setIsHOVRoad (bool isHOVRoad)
      {
         m_isHOVRoad = isHOVRoad;
      }

      bool isHOVRoad () const
      {
         return m_isHOVRoad;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("RoadProperties (Values == true):\n");
         stream << ::std::endl;

         if(m_isRestricted)
         {
            stream << "restricted Road" << ::std::endl;
         }
         if(m_isBuildUp)
         {
            stream << "BuildUp" << ::std::endl;
         }
         if(m_isSpeedLimit)
         {
            stream << "SpeedLimit" << ::std::endl;
         }
         if(m_isJunctionFree)
         {
            stream << "JunctionFree" << ::std::endl;
         }
         if(m_isMultiDigitized)
         {
            stream << "MultiDigitized" << ::std::endl;
         }
         if(m_isBasicAttributed)
         {
            stream << "BasicAttributed" << ::std::endl;
         }
         if(m_isComplexIntersection)
         {
            stream << "ComplexIntersection" << ::std::endl;
         }
         if(m_isFerry)
         {
            stream << "Ferry" << ::std::endl;
         }
         if(m_isTollRoad)
         {
            stream << "TollRoad" << ::std::endl;
         }
         if(m_isSeasonalClosed)
         {
            stream << "SeasonalClosed" << ::std::endl;
         }
         if(m_isTunnel)
         {
            stream << "isTunnel" << ::std::endl;
         }
         if(m_isNoPass)
         {
            stream << "NoPass" << ::std::endl;
         }
         if(m_isHardOnewayInDigitized)
         {
            stream << "HardOnewayInDigitized" << ::std::endl;
         }
         if(m_isHardOnewayRevDigitized)
         {
            stream << "HardOnewayRevDigitized" << ::std::endl;
         }
         if(m_isBridge)
         {
            stream << "Bridge" << ::std::endl;
         }
         if(m_isStubble)
         {
            stream << "Stubble" << ::std::endl;
         }
         if(m_isGateway)
         {
            stream << "Gateway" << ::std::endl;
         }
         if(m_isAutoTrain)
         {
            stream << "AutoTrain" << ::std::endl;
         }
         if(m_isUnpavedRoad)
         {
            stream << "UnpavedRoad" << ::std::endl;
         }
         if(m_isRightHandDriving)
         {
            stream << "RightHandDriving" << ::std::endl;
         }
         if(m_isHOVRoad)
         {
            stream << "HOVRoad" << ::std::endl;
         }

         return stream.str();
      }

   private:
      bool m_isRestricted:1;
      bool m_isBuildUp:1;
      bool m_isSpeedLimit:1;
      bool m_isJunctionFree:1;
      bool m_isMultiDigitized:1;
      bool m_isBasicAttributed:1;
      bool m_isComplexIntersection:1;
      bool m_isFerry:1;
      bool m_isTollRoad:1;
      bool m_isSeasonalClosed:1;
      bool m_isTunnel:1;
      bool m_isNoPass:1;
      bool m_isHardOnewayInDigitized:1;
      bool m_isHardOnewayRevDigitized:1;
      bool m_isBridge:1;
      bool m_isStubble:1;
      bool m_isGateway:1;
      bool m_isAutoTrain:1;
      bool m_isUnpavedRoad:1;
      bool m_isRightHandDriving:1;
      bool m_isHOVRoad:1;
   };



public:
   PositionStatusInfo()
      :m_positionStatusInfo(POSITION_STATUS__NO_INFO),
       m_hemisphereLatitude(HEMISPHERE_LATITUDE__NONE),
       m_hemisphereLongitude(HEMISPHERE_LONGITUDE__NONE),
       m_latitude(0.0),
       m_longitude(0.0),
       m_altitude(0),
       m_heading(0.0f),
       m_speed(0.0f),
       m_roadClass(ROAD_CLASS__NONE),
       m_roadType(ROAD_TYPE__UNKNOWN),
       m_roadProperties(),
       m_directionOfPrayer(0.0f),
       m_linkId(0) {}

   void setPositionStatus(const PositionStatus positionStatusIndex)
   {
      m_positionStatusInfo = positionStatusIndex;
   }
   PositionStatus getPositionStatus()  const
   {
      return m_positionStatusInfo;
   }

   void setSpeedInfo(const ::std::string& speedValue)
   {
      m_speedAsString = speedValue;
   }
   const ::std::string& getSpeedInfoAsString() const
   {
      return m_speedAsString;
   }

   void setSpeedInfo(float speedValue)
   {
      m_speed = speedValue;
   }
   float getSpeedInfo() const
   {
      return m_speed;
   }

   void setHeadingInfo(const ::std::string& headingValue)
   {
      m_headingAsString = headingValue;
   }
   const ::std::string& getHeadingInfoAsString() const
   {
      return m_headingAsString;
   }

   void setHeadingInfo(const float& headingValue)
   {
      m_heading = headingValue;
   }
   float getHeadingInfo() const
   {
      return m_heading;
   }

   void setLatitude(const ::std::string& latitudeValue)
   {
      m_latitudeAsString = latitudeValue;
   }
   void setHemisphereLatitude(HemisphereLatitude latitude)
   {
      m_hemisphereLatitude = latitude;
   }
   void setLongitude(const ::std::string& longitudeValue)
   {
      m_longitudeAsString = longitudeValue;
   }
   void setHemisphereLongitude(HemisphereLongitude longitude)
   {
      m_hemisphereLongitude = longitude;
   }
   void setAltitude(const ::std::string& altitudeString)
   {
      m_altitudeAsString = altitudeString;
   }

   void setLatitude(double latitudeValue)
   {
      m_latitude = latitudeValue;
   }
   void setLongitude(double longitudeValue)
   {
      m_longitude = longitudeValue;
   }
   void setAltitude(int altitudeValue)
   {
      m_altitude = altitudeValue;
   }

   const ::std::string& getLatitudeAsString() const
   {
      return m_latitudeAsString;
   }
   HemisphereLatitude getHemisphereLatitude() const
   {
      return m_hemisphereLatitude;
   }
   const ::std::string& getLongitudeAsString() const
   {
      return m_longitudeAsString;
   }
   HemisphereLongitude getHemisphereLongitude() const
   {
      return m_hemisphereLongitude;
   }
   const ::std::string& getAltitudeAsString() const
   {
      return m_altitudeAsString;
   }

   double getLatitude() const
   {
      return m_latitude;
   }
   double getLongitude() const
   {
      return m_longitude;
   }
   int getAltitude() const
   {
      return m_altitude;
   }

   void setCountryCode(const ::std::string& countryCode)
   {
      m_countryCode = countryCode;
   }
   const ::std::string& getCountryCode() const
   {
      return m_countryCode;
   }

   void setStateCode(const ::std::string& stateCode)
   {
      m_stateCode = stateCode;
   }
   const ::std::string& getStateCode() const
   {
      return m_stateCode;
   }

   void setRoadClass(RoadClass roadClass)
   {
      m_roadClass = roadClass;
   }
   RoadClass getRoadClass() const
   {
      return m_roadClass;
   }

   void setRoadType(RoadType roadType)
   {
      m_roadType = roadType;
   }
   RoadType getRoadType() const
   {
      return m_roadType;
   }

   void setRoadProperties(RoadProperties roadProperties)
   {
      m_roadProperties = roadProperties;
   }
   const RoadProperties& getRoadProperties() const
   {
      return m_roadProperties;
   }

   RoadProperties& getRoadProperties()
   {
      return m_roadProperties;
   }

   void setDirectionOfPrayer(float directionOfPrayer)
   {
      m_directionOfPrayer = directionOfPrayer;
   }

   float getDirectionOfPrayer() const
   {
      return m_directionOfPrayer;
   }

   void setMapMatchedPositionAccuracy(const ValidValue<double>& mapMatchedPositionAccuracy)
   {
      m_mapMatchedPositionAccuracy = mapMatchedPositionAccuracy;
   }

   const ValidValue<double>& getMapMatchedPositionAccuracy() const
   {
      return m_mapMatchedPositionAccuracy;
   }

   void setLinkId( LinkId linkId)
   {
      m_linkId = linkId;
   }

   LinkId getLinkId() const
   {
      return m_linkId;
   }

   void setUniqueID(const ValidValue<uint32_t>& uniqueID)
   {
      m_uniqueID = uniqueID;
   }

   const ValidValue<uint32_t>& getUniqueID() const
   {
      return m_uniqueID;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PositionStatusInfo payload:\n");
      stream << ::std::endl
         << "Position Status = " << ::navmiddleware::toString(m_positionStatusInfo) << ::std::endl

         << "Speed as string = " << m_speedAsString << ", "
         << "Heading as string  = " << m_headingAsString << ::std::endl

         << "Latitude as string = " << m_latitudeAsString << ::navmiddleware::toString(m_hemisphereLatitude) << ", "
         << "Longitude as string = " << m_longitudeAsString << ::navmiddleware::toString(m_hemisphereLongitude) << ::std::endl
         << "Altitude as string = " << m_altitudeAsString << ::std::endl

         << "Speed = " << m_speed << ", "
         << "Heading = " << m_heading << ::std::endl
         << "Latitude = " << m_latitude << ", "
         << "Longitude = " << m_longitude << ::std::endl
         << "Altitude = " << m_altitude << ::std::endl

         << "Country code = " << m_countryCode << ::std::endl
         << "State code = " << m_stateCode << ::std::endl
         << "Road class = " << ::navmiddleware::toString(m_roadClass) << ::std::endl
         << "Road type = " << ::navmiddleware::toString(m_roadType) << ::std::endl
         << "RoadProperties (Values == true) = " << m_roadProperties.toString() << ::std::endl
         << "Direction Of Prayer = " << m_directionOfPrayer << ::std::endl
         << "Map Matched Position Accuracy = " << m_mapMatchedPositionAccuracy.toString() << ::std::endl
         << "Link id = " << m_linkId << ::std::endl
         << "Unique ID =" << m_uniqueID.toString() << ::std::endl;

      return stream.str();
   }

private:

   PositionStatus m_positionStatusInfo;
   ::std::string m_speedAsString;
   ::std::string m_headingAsString;
   ::std::string m_latitudeAsString;
   HemisphereLatitude m_hemisphereLatitude; //Hemisphere information of m_latitudeAsString
   ::std::string m_longitudeAsString;
   HemisphereLongitude m_hemisphereLongitude; //Hemisphere information of m_longitudeAsString
   ::std::string m_altitudeAsString; // including unit (e.g. 'm' or 'ft')
   double m_latitude;
   double m_longitude;
   int m_altitude; // in m
   float m_heading;
   float m_speed; // in km/h

   ::std::string m_countryCode;
   ::std::string m_stateCode;
   RoadClass m_roadClass;
   RoadType m_roadType;
   RoadProperties m_roadProperties;
   float m_directionOfPrayer; // in degree relative to heading, clockwise
   ValidValue<double> m_mapMatchedPositionAccuracy; // distance between GNSS-position and MM-position expressed in meter
   LinkId m_linkId; // link id of the current car position

   /** m_uniqueID of PositionInfo and PositionStatusInfo have to be same
    *  if content inside the info objects have to be merged **/
   ValidValue<uint32_t> m_uniqueID;
};

class WhereAmIInfo
{
public:
   struct SingleStreetInfo
   {
      ::std::string m_cityName;
      ::std::string m_streetName;
      ::std::string m_streetNumber;
      ::std::vector< Image> m_roadNrImageData;
   };

   struct SideStreetInfo
   {
      SingleStreetInfo m_info;
      ::std::string m_distance;
   };

   const SingleStreetInfo& getCurrentStreetInfo() const
   {
      return m_currentStreet;
   }
   void setCurrentStreetInfo(const SingleStreetInfo& info)
   {
      m_currentStreet = info;
   }

   const ValidValue<SideStreetInfo>& getNextSideStreetInfo() const
   {
      return m_nextSideStreet;
   }
   void setNextSideStreetInfo(const ValidValue<SideStreetInfo>& info)
   {
      m_nextSideStreet = info;
   }

   const ValidValue<SideStreetInfo>& getPreviousSideStreetInfo() const
   {
      return m_previousSideStreet;
   }
   void setPreviousSideStreetInfo(const ValidValue<SideStreetInfo>& info)
   {
      m_previousSideStreet = info;
   }

   const ::std::string& getHouseNumberOnLeftHandSide() const
   {
      return m_houseNumberOnLeftHandSide;
   }
   void setHouseNumberOnLeftHandSide(const ::std::string& houseNumber)
   {
      m_houseNumberOnLeftHandSide = houseNumber;
   }

   const ::std::string& getHouseNumberOnRightHandSide() const
   {
      return m_houseNumberOnRightHandSide;
   }
   void setHouseNumberOnRightHandSide(const ::std::string& houseNumber)
   {
      m_houseNumberOnRightHandSide = houseNumber;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("WhereAmIInfo payload:\n");
      stream << ::std::endl
             << "CurrentStreet = ("
             << "street: \"" << m_currentStreet.m_streetName << "\", "
             << "city: \"" << m_currentStreet.m_cityName << "\", "
             << "number: \"" <<  m_currentStreet.m_streetNumber << "\")" << ::std::endl;

      if (m_nextSideStreet.isValid())
      {
         stream  << "NextSideStreet = ("
                 << "street: \"" <<  m_nextSideStreet.getValue().m_info.m_streetName << "\", "
                 << "city: \"" << m_nextSideStreet.getValue().m_info.m_cityName << "\", "
                 << "number: \"" << m_nextSideStreet.getValue().m_info.m_streetNumber << "\", "
                 << "distance: \"" << m_nextSideStreet.getValue().m_distance << "\")" << ::std::endl;
      }
      else
      {
         stream << "NextSideStreet empty" << ::std::endl;
      }

      if (m_previousSideStreet.isValid())
      {
         stream << "PreviousSideStreet = ("
                << "street: \"" <<  m_previousSideStreet.getValue().m_info.m_streetName << "\", "
                << "city: \"" << m_previousSideStreet.getValue().m_info.m_cityName << "\", "
                << "number: \"" << m_previousSideStreet.getValue().m_info.m_streetNumber << "\", "
                << "distance: \"" << m_previousSideStreet.getValue().m_distance << "\")"<< ::std::endl;
      }
      else
      {
         stream << "PreviousSideStreet empty" << ::std::endl;
      }

      stream << "HousenumberOnLeftHandSide = \"" << m_houseNumberOnLeftHandSide << "\"" << ::std::endl
             << "HousenumberOnrightHandSide = \"" << m_houseNumberOnRightHandSide << "\"" << ::std::endl;

      return stream.str();
   }

private:
   SingleStreetInfo m_currentStreet;
   ValidValue<SideStreetInfo> m_nextSideStreet;
   ValidValue<SideStreetInfo> m_previousSideStreet;
   ::std::string m_houseNumberOnLeftHandSide;
   ::std::string m_houseNumberOnRightHandSide;
};

class GnssDataInfo
{
public:
   enum GnssSystem
   {
      GNSS_SYSTEM__GPS,
      GNSS_SYSTEM__GLONASS,
      GNSS_SYSTEM__GALILEO,
      GNSS_SYSTEM__COMPASS,
      GNSS_SYSTEM__SBAS,
      GNSS_SYSTEM__QZSS,
      GNSS_SYSTEM__UNKNOWN
   };
   static ::std::string toString(GnssSystem gnssSystem)
   {
      switch (gnssSystem)
      {
      case GNSS_SYSTEM__GPS:
         return "GPS";
      case GNSS_SYSTEM__GLONASS:
         return "GLONASS";
      case GNSS_SYSTEM__GALILEO:
         return "Galileo";
      case GNSS_SYSTEM__COMPASS:
         return "Beidou";
      case GNSS_SYSTEM__SBAS:
         return "SBAS";
      case GNSS_SYSTEM__QZSS:
         return "QZSS";
      default:
         ::std::stringstream stream;
         stream << "Unknown GNSS system <" << static_cast<unsigned int>(gnssSystem) << ">";
         return stream.str();
      }
   }

   enum SatelliteUsage
   {
      SATELLITE_USAGE__NONE,
      SATELLITE_USAGE__TRACKED,
      SATELLITE_USAGE__USED_FOR_POS_CALC,
   };
   static ::std::string toString(SatelliteUsage usage)
   {
      switch (usage)
      {
      case SATELLITE_USAGE__NONE:
         return "none";
      case SATELLITE_USAGE__TRACKED:
         return "tracked";
      case SATELLITE_USAGE__USED_FOR_POS_CALC:
         return "used";
      default:
         ::std::stringstream stream;
         stream << "Unknown SAT usage <" << static_cast<unsigned int>(usage) << ">";
         return stream.str();
      }
   }

   struct GnssChannelInfo
   {
      GnssChannelInfo():
         m_gnssSystem(GNSS_SYSTEM__GPS),
         m_gnssSystemLetter('\0'),
         m_satelliteNum(0),
         m_posX(0),
         m_posY(0),
         m_status(SATELLITE_USAGE__NONE),
         m_cnr(0)
      { }

      GnssChannelInfo(
         GnssSystem     gnssSystem,
         char           gnssSystemLetter,
         uint8_t        satelliteNum,
         float          posX,
         float          posY,
         SatelliteUsage status,
         uint8_t        cnr
      ):
         m_gnssSystem(gnssSystem),
         m_gnssSystemLetter(gnssSystemLetter),
         m_satelliteNum(satelliteNum),
         m_posX(posX),
         m_posY(posY),
         m_status(status),
         m_cnr(cnr)
      { }

      ::std::string toString() const
      {
         ::std::stringstream stream("GnssChannelInfo payload:\n");
         stream << "GNSS system = " << GnssDataInfo::toString(m_gnssSystem)
                << ", sat id = " << m_gnssSystemLetter << static_cast<unsigned>(m_satelliteNum) << ::std::endl
                << "pos_x = " << m_posX << ", pos_y = " << m_posY << ::std::endl
                << "status = " << GnssDataInfo::toString(m_status) << ", CNR = " << static_cast<unsigned>(m_cnr) << ::std::endl;
         return stream.str();
      }

      GnssSystem     m_gnssSystem;
      char           m_gnssSystemLetter;  // 'G'=GPS, 'R'=GLONASS, 'E'=Galileo, 'C'=Beidou/COMPASS, 'J'=QZSS
      uint8_t        m_satelliteNum;  // satellite number within specified GNSS system
      float          m_posX;  // position on x axis of unit circle
      float          m_posY;  // position on y axis of unit circle
      SatelliteUsage m_status;
      uint8_t        m_cnr;  // carrier to noise ratio
   };

   GnssDataInfo()
      : m_gnssPosition()
      , m_hemisphereLatitude(HEMISPHERE_LATITUDE__NONE)
      , m_hemisphereLongitude(HEMISPHERE_LONGITUDE__NONE)
      , m_altitude(0)
      , m_hDOP(0.0)
      , m_pDOP(0.0)
      , m_vDOP(0.0)
      , m_speed(0)
      , m_speedReliability(0)
      , m_gnssMode(GNSS_MODE__UNKNOWN)
      , m_numberOfVisibleSatellites(0)
      , m_numberOfReceivedSatellites(0)
      , m_numberOfUsedSatellites(0)
      , m_almanacValid(false)
   {}

   GnssDataInfo(const ::std::string& gpsTime,
                const ::std::string& dataInfo,
                const ::std::string& latitudeAsString,
                HemisphereLatitude hemLatitude,
                const ::std::string& longitudeAsString,
                HemisphereLongitude hemLongitude,
                const ::std::string& altitudeAsString,
                double hDOP,
                const ::std::string& hDOPAsString,
                double pDOP,
                const ::std::string& pDOPAsString,
                double vDOP,
                const ::std::string& vDOPAsString,
                const ::std::string& heading,
                const ::std::string& speedAsString,
                const ::std::string& satelliteTypeUsed,
                GnssMode gnssMode,
                const GeoCoordinateDegree& gnssPosition,
                int16_t speed,
                uint8_t speedReliability,
                uint16_t numVisibleSatellites,
                uint16_t numReceivedSatellites,
                uint16_t numUsedSatellites,
                int16_t altitude,
                bool almanacValid,
                const ::std::vector<GnssChannelInfo>& gnssChannelInfo = ::std::vector<GnssChannelInfo>())
      : m_time(gpsTime)
      , m_gnssDataInfo(dataInfo)
      , m_gnssPosition(gnssPosition)
      , m_latitude(latitudeAsString)
      , m_hemisphereLatitude(hemLatitude)
      , m_longitude(longitudeAsString)
      , m_hemisphereLongitude(hemLongitude)
      , m_altitude(altitude)
      , m_altitudeAsString(altitudeAsString)
      , m_hDOP(hDOP)
      , m_hDOPAsString(hDOPAsString)
      , m_pDOP(pDOP)
      , m_pDOPAsString(pDOPAsString)
      , m_vDOP(vDOP)
      , m_vDOPAsString(vDOPAsString)
      , m_heading(heading)
      , m_speed(speed)
      , m_speedReliability(speedReliability)
      , m_speedAsString(speedAsString)
      , m_satelliteTypeUsed(satelliteTypeUsed)
      , m_gnssMode(gnssMode)
      , m_numberOfVisibleSatellites(numVisibleSatellites)
      , m_numberOfReceivedSatellites(numReceivedSatellites)
      , m_numberOfUsedSatellites(numUsedSatellites)
      , m_almanacValid(almanacValid)
      , m_gnssChannelInfos(gnssChannelInfo)
   {}

   void setTime(const ::std::string& time)
   {
      m_time = time;
   }
   const ::std::string& getTime() const
   {
      return m_time;
   }

   void setGnssDataInfo(const ::std::string& gnssDataInfo)
   {
      m_gnssDataInfo = gnssDataInfo;
   }
   const ::std::string& getGnssDataInfo() const
   {
      return m_gnssDataInfo;
   }

   void setGnssPosition(const GeoCoordinateDegree& gnssPosition)
   {
      m_gnssPosition = gnssPosition;
   }
   const GeoCoordinateDegree& getGnssPosition() const
   {
      return m_gnssPosition;
   }

   void setLatitude(const ::std::string& latitude)
   {
      m_latitude = latitude;
   }
   const ::std::string& getLatitude() const
   {
      return m_latitude;
   }

   void setHemisphereLatitude(HemisphereLatitude latitude)
   {
      m_hemisphereLatitude = latitude;
   }
   HemisphereLatitude getHemisphereLatitude() const
   {
      return m_hemisphereLatitude;
   }

   void setLongitude(const ::std::string& longitude)
   {
      m_longitude = longitude;
   }
   const ::std::string& getLongitude() const
   {
      return m_longitude;
   }

   void setHemisphereLongitude(HemisphereLongitude longitude)
   {
      m_hemisphereLongitude = longitude;
   }
   HemisphereLongitude getHemisphereLongitude() const
   {
      return m_hemisphereLongitude;
   }

   void setAltitude(int altitude)
   {
      m_altitude = altitude;
   }
   int getAltitude() const
   {
      return m_altitude;
   }

   void setAltitude(const ::std::string& altitude)
   {
      m_altitudeAsString = altitude;
   }
   const ::std::string& getAltitudeAsString() const
   {
      return m_altitudeAsString;
   }

   void setHDOP(double hDOP, const ::std::string& hDOPAsString)
   {
      m_hDOP = hDOP;
      m_hDOPAsString = hDOPAsString;
   }
   double getHDOP() const
   {
      return m_hDOP;
   }
   const ::std::string& getHDOPAsString() const
   {
      return m_hDOPAsString;
   }

   void setPDOP(double pDOP, const ::std::string& pDOPAsString)
   {
      m_pDOP = pDOP;
      m_pDOPAsString = pDOPAsString;
   }
   double getPDOP() const
   {
      return m_pDOP;
   }
   const ::std::string& getPDOPAsString() const
   {
      return m_pDOPAsString;
   }

   void setVDOP(double vDOP, const ::std::string& vDOPAsString)
   {
      m_vDOP = vDOP;
      m_vDOPAsString = vDOPAsString;
   }
   double getVDOP() const
   {
      return m_vDOP;
   }
   const ::std::string& getVDOPAsString() const
   {
      return m_vDOPAsString;
   }

   void setHeading(const ::std::string& heading)
   {
      m_heading = heading;
   }
   const ::std::string& getHeading() const
   {
      return m_heading;
   }


   // GNSS speed in m/s. Range: >= 0 m/s
   void setSpeed(int speed)
   {
      m_speed = speed;
   }
   int getSpeed() const
   {
      return m_speed;
   }

   // Reliability of the GNSS speed information in percent. Range: 0-100%
   void setSpeedReliability(uint8_t speedReliability)
   {
      m_speedReliability = speedReliability;
   }
   uint8_t getSpeedReliability() const
   {
      return m_speedReliability;
   }

   void setSpeedAsString(const ::std::string& speedAsString)
   {
      m_speedAsString = speedAsString;
   }
   const ::std::string& getSpeedAsString() const
   {
      return m_speedAsString;
   }

   void setSatelliteTypeUsed(const ::std::string& satelliteTypeUsed)
   {
      m_satelliteTypeUsed = satelliteTypeUsed;
   }
   const ::std::string& getSatelliteTypeUsed() const
   {
      return m_satelliteTypeUsed;
   }

   void setGnssMode(const GnssMode gnssMode)
   {
      m_gnssMode = gnssMode;
   }
   GnssMode getGnssMode() const
   {
      return m_gnssMode;
   }

   void setNumberOfVisibleSatellites(int numberOfVisibleSatellites)
   {
      m_numberOfVisibleSatellites = numberOfVisibleSatellites;
   }
   int getNumberOfVisibleSatellites() const
   {
      return m_numberOfVisibleSatellites;
   }

   void setNumberOfReceivedSatellites(int numberOfReceivedSatellites)
   {
      m_numberOfReceivedSatellites = numberOfReceivedSatellites;
   }
   int getNumberOfReceivedSatellites() const
   {
      return m_numberOfReceivedSatellites;
   }

   void setNumberOfUsedSatellites(int numberOfUsedSatellites)
   {
      m_numberOfUsedSatellites = numberOfUsedSatellites;
   }
   int getNumberOfUsedSatellites() const
   {
      return m_numberOfUsedSatellites;
   }

   void setAlmanacValid(bool almanacValid)
   {
      m_almanacValid = almanacValid;
   }
   bool isAlmanacValid() const
   {
      return m_almanacValid;
   }

   const ::std::vector<GnssChannelInfo>& getGnssChannelInfos() const
   {
      return m_gnssChannelInfos;
   }
   ::std::vector<GnssChannelInfo>& getGnssChannelInfos()
   {
      return m_gnssChannelInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("GnssDataInfo payload:\n");
      stream << ::std::endl
         << "time = " << m_time << ::std::endl

         << "Gnss Position = " << m_gnssPosition.toString() << ::std::endl

         << "Latitude = " << m_hemisphereLatitude << " " << m_latitude << ", "
         << "Longitude = " << m_hemisphereLongitude << " " << m_longitude << ", "
         << "Altitude = " << m_altitude << " / " << m_altitudeAsString << ::std::endl

         << "HDOP = " << m_hDOP << " / " << m_hDOPAsString << ", "
         << "PDOP = " << m_pDOP << " / " << m_pDOPAsString << ", "
         << "VDOP = " << m_vDOP << " / " << m_vDOPAsString << std::endl

         << "Heading = " << m_heading << ", "
         << "Speed = " << m_speedAsString << " / " << m_speed << "m/s, speed reliability = " << static_cast<uint16_t>(m_speedReliability) << "%" << std::endl

         << "Satellite type used = " << m_satelliteTypeUsed << ", "
         << "Gnss mode = " << ::navmiddleware::toString(m_gnssMode) << std::endl

         << "Number of satellites = " << m_numberOfVisibleSatellites << " visible, "
         << m_numberOfReceivedSatellites << " received, "
         << m_numberOfUsedSatellites << " used" << std::endl

         << "Almanach valid = " << (m_almanacValid ? "yes" : "no") << std::endl

         << "Gnss data Info = " << m_gnssDataInfo << ::std::endl;

      stream << "GNSS channel infos (num channels = " << m_gnssChannelInfos.size() << "):" << ::std::endl;
      for (::std::vector<GnssChannelInfo>::const_iterator iter = m_gnssChannelInfos.begin();
            iter != m_gnssChannelInfos.end(); ++iter)
      {
         stream << iter->toString();
      }

      return stream.str();
   }

private:
   ::std::string       m_time;
   ::std::string       m_gnssDataInfo;
   GeoCoordinateDegree m_gnssPosition;
   ::std::string       m_latitude;
   HemisphereLatitude  m_hemisphereLatitude;
   ::std::string       m_longitude;
   HemisphereLongitude m_hemisphereLongitude;
   int                 m_altitude;
   ::std::string       m_altitudeAsString;
   double              m_hDOP; // Horizontal Dilution of Precision
   ::std::string       m_hDOPAsString;
   double              m_pDOP; // (3D) Position Dilution of Precision
   ::std::string       m_pDOPAsString;
   double              m_vDOP; // Vertical Delution of Precision
   ::std::string       m_vDOPAsString;
   ::std::string       m_heading;
   int                 m_speed; // Speed in m/s
   uint8_t             m_speedReliability; // reliability of the speed value in %
   ::std::string       m_speedAsString;
   ::std::string       m_satelliteTypeUsed;
   GnssMode            m_gnssMode;
   int                 m_numberOfVisibleSatellites;
   int                 m_numberOfReceivedSatellites;
   int                 m_numberOfUsedSatellites;
   bool                m_almanacValid;

   ::std::vector<GnssChannelInfo> m_gnssChannelInfos;
};

class TripReplayStatusInfo
{
public:
   enum TripReplayStatus
   {
      TRIP_REPLAY_STATUS__UNKNOWN,
      TRIP_REPLAY_STATUS__RESERVED,
      TRIP_REPLAY_STATUS__NOT_ACTIVE,
      TRIP_REPLAY_STATUS__ACTIVE,
      TRIP_REPLAY_STATUS__FILE_NOT_FOUND_ERROR,
      TRIP_REPLAY_STATUS__INTERNAL_ERROR
   };
   static ::std::string toString(TripReplayStatus trReplayStatus)
   {
      ::std::string tripReplayStatus;
      switch (trReplayStatus)
      {
      case TRIP_REPLAY_STATUS__UNKNOWN:
         tripReplayStatus = "TRIP_REPLAY_STATUS__UNKNOWN";
         break;
      case TRIP_REPLAY_STATUS__RESERVED:
         tripReplayStatus = "TRIP_REPLAY_STATUS__RESERVED";
         break;
      case TRIP_REPLAY_STATUS__NOT_ACTIVE:
         tripReplayStatus = "TRIP_REPLAY_STATUS__NOT_ACTIVE";
         break;
      case TRIP_REPLAY_STATUS__ACTIVE:
         tripReplayStatus = "TRIP_REPLAY_STATUS__ACTIVE";
         break;
      case TRIP_REPLAY_STATUS__FILE_NOT_FOUND_ERROR:
         tripReplayStatus = "TRIP_REPLAY_STATUS__FILE_NOT_FOUND_ERROR";
         break;
      case TRIP_REPLAY_STATUS__INTERNAL_ERROR:
         tripReplayStatus = "TRIP_REPLAY_STATUS__INTERNAL_ERROR";
         break;
      default:
         ::std::stringstream stream;
         stream << "TRIP_REPLAY_STATUS__<" << static_cast<unsigned int>(trReplayStatus) << ">";
         tripReplayStatus = stream.str();
         break;
      }
      return tripReplayStatus;
   }

   TripReplayStatusInfo()
      : m_tripReplayStatus(TRIP_REPLAY_STATUS__UNKNOWN)
      , m_loopModeActive(false)
      , m_tripFileName("")
      , m_progress(0)
      , m_syncSegment(0)
      , m_countSyncSegments(0)
      , m_speed(0)
   {}

   void setTripReplayStatus(TripReplayStatus tripReplayStatus)
   {
      m_tripReplayStatus = tripReplayStatus;
   }

   TripReplayStatus getTripReplayStatus() const
   {
      return m_tripReplayStatus;
   }

   void setLoopModeActive(bool loopModeActive)
   {
      m_loopModeActive = loopModeActive;
   }

   bool getLoopModeActive() const
   {
      return m_loopModeActive;
   }

   void setTripFileName(const ::std::string& tripFileName)
   {
      m_tripFileName = tripFileName;
   }
   const ::std::string& getTripFileName() const
   {
      return m_tripFileName;
   }

   void setProgress(uint8_t progress)
   {
      m_progress = progress;
   }

   uint8_t getProgress() const
   {
      return m_progress;
   }

   void setSyncSegment(uint16_t syncSegment)
   {
      m_syncSegment = syncSegment;
   }

   uint16_t getSyncSegment() const
   {
      return m_syncSegment;
   }

   void setCountSyncSegments(uint16_t countSyncSegments)
   {
      m_countSyncSegments = countSyncSegments;
   }

   uint16_t getCountSyncSegments() const
   {
      return m_countSyncSegments;
   }

   void setSpeed(uint16_t speed)
   {
      m_speed = speed;
   }

   uint16_t getSpeed() const
   {
      return m_speed;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("TripReplayStatusInfo payload:\n");
      stream << ::std::endl
             << "Trip Replay Status = "   << toString(m_tripReplayStatus)  << ::std::endl
             << "Loop Mode Active = "     << m_loopModeActive    << ::std::endl
             << "Trip-File Name = "       << m_tripFileName      << ::std::endl
             << "Progress = "             << m_progress          << ::std::endl
             << "Sync Segment = "         << m_syncSegment       << ::std::endl
             << "Count Sync Segments = "  << m_countSyncSegments << ::std::endl
             << "Speed = "                << m_speed             << ::std::endl;

      return stream.str();
   }

private:
   TripReplayStatus m_tripReplayStatus;
   bool             m_loopModeActive;
   ::std::string    m_tripFileName;
   uint8_t          m_progress;          //in percent (Begin=0%, End=100%)
   uint16_t         m_syncSegment;
   uint16_t         m_countSyncSegments;
   uint16_t         m_speed;             //in percent
};

class TripRecordingStatusInfo
{
public:
   enum TripRecordingStatus
   {
      TRIP_RECORDING_STATUS__UNKNOWN,
      TRIP_RECORDING_STATUS__NOT_ACTIVE,
      TRIP_RECORDING_STATUS__ACTIVE
   };
   static ::std::string toString(TripRecordingStatus trRecordingStatus)
   {
      ::std::string tripRecordingStatus;
      switch (trRecordingStatus)
      {
      case TRIP_RECORDING_STATUS__UNKNOWN:
         tripRecordingStatus = "TRIP_RECORDING_STATUS__UNKNOWN";
         break;
      case TRIP_RECORDING_STATUS__NOT_ACTIVE:
         tripRecordingStatus = "TRIP_RECORDING_STATUS__NOT_ACTIVE";
         break;
      case TRIP_RECORDING_STATUS__ACTIVE:
         tripRecordingStatus = "TRIP_RECORDING_STATUS__ACTIVE";
         break;
      default:
         ::std::stringstream stream;
         stream << "TRIP_RECORDING_STATUS__<" << static_cast<unsigned int>(trRecordingStatus) << ">";
         tripRecordingStatus = stream.str();
         break;
      }
      return tripRecordingStatus;
   }

   TripRecordingStatusInfo()
      : m_tripRecordingStatus(TRIP_RECORDING_STATUS__UNKNOWN)
   {}

   void setTripRecordingStatus(TripRecordingStatus tripRecordingStatus)
   {
      m_tripRecordingStatus = tripRecordingStatus;
   }

   TripRecordingStatus getTripRecordingStatus() const
   {
      return m_tripRecordingStatus;
   }

   void setTripFileName(const ::std::string& tripFileName)
   {
      m_tripFileName = tripFileName;
   }
   const ::std::string& getTripFileName() const
   {
      return m_tripFileName;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("TripRecordingStatusInfo payload:\n");
      stream << ::std::endl
             << "Trip Recording Status = "   << toString(m_tripRecordingStatus)  << ::std::endl
             << "Trip-File Name = "          << m_tripFileName                   << ::std::endl;

      return stream.str();
   }

private:
   TripRecordingStatus m_tripRecordingStatus;
   ::std::string    m_tripFileName; // provides the trip file name (and path)
};

class PositionDebugInfo
{
public:
   PositionDebugInfo() {}
   virtual ~PositionDebugInfo() {}

   bool operator == (const PositionDebugInfo& rhs) const
   {
      return (m_gnssInfo == rhs.m_gnssInfo &&
              m_odometerInfo == rhs.m_odometerInfo &&
              m_gyroInfo == rhs.m_gyroInfo &&
              m_accInfo == rhs.m_accInfo) &&
             m_mapMatchingInfo == rhs.m_mapMatchingInfo &&
             m_absInfo == rhs.m_absInfo &&
             m_deadReckoningInfo == rhs.m_deadReckoningInfo &&
             m_calibrationInfo == rhs.m_calibrationInfo;
   }

   bool operator != (const PositionDebugInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setGnssInfo(const ::std::string& gnssInfo)
   {
      m_gnssInfo = gnssInfo;
   }
   const ::std::string& getGnssInfo() const
   {
      return m_gnssInfo;
   }
   void setOdometerInfo(const ::std::string& odometerInfo)
   {
      m_odometerInfo = odometerInfo;
   }
   const ::std::string& getOdometerInfo() const
   {
      return m_odometerInfo;
   }
   void setGyroInfo(const ::std::string& gyroInfo)
   {
      m_gyroInfo = gyroInfo;
   }
   const ::std::string& getGyroInfo() const
   {
      return m_gyroInfo;
   }
   void setAccInfo(const ::std::string& accInfo)
   {
      m_accInfo = accInfo;
   }
   const ::std::string& getAccInfo() const
   {
      return m_accInfo;
   }
   void setMapMatchingInfo(const ::std::string& mapMatchingInfo)
   {
      m_mapMatchingInfo = mapMatchingInfo;
   }
   const ::std::string& getMapMatchingInfo() const
   {
      return m_mapMatchingInfo;
   }
   void setAbsInfo(const ::std::string& absInfo)
   {
      m_absInfo = absInfo;
   }
   const ::std::string& getAbsInfo() const
   {
      return m_absInfo;
   }
   void setDeadReckoningInfo(const ::std::string& deadReckoningInfo)
   {
      m_deadReckoningInfo = deadReckoningInfo;
   }
   const ::std::string& getDeadReckoningInfo() const
   {
      return m_deadReckoningInfo;
   }
   void setCalibrationInfo(const ::std::string& calibrationInfo)
   {
      m_calibrationInfo = calibrationInfo;
   }
   const ::std::string& getCalibrationInfo() const
   {
      return m_calibrationInfo;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PositionDebugInfo payload:\n");
      stream << ::std::endl
             << "GNSS Info = "          << m_gnssInfo          << ::std::endl
             << "Odometer Info = "      << m_odometerInfo      << ::std::endl
             << "Gyro Info = "          << m_gyroInfo          << ::std::endl
             << "Acc Info= "            << m_accInfo           << ::std::endl
             << "Map matching Info = "  << m_mapMatchingInfo   << ::std::endl
             << "ABS Info = "           << m_absInfo           << ::std::endl
             << "DeadReckoning Info = " << m_deadReckoningInfo << ::std::endl
             << "Calibration Info = "   << m_calibrationInfo   << ::std::endl;
      return stream.str();
   }

private:
   ::std::string  m_gnssInfo;
   ::std::string  m_odometerInfo;
   ::std::string  m_gyroInfo;
   ::std::string  m_accInfo;
   ::std::string  m_mapMatchingInfo;
   ::std::string  m_absInfo;
   ::std::string  m_deadReckoningInfo;
   ::std::string  m_calibrationInfo;
};
}

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_POSITIONINFOS_H_
