/**
* @file MapSettings.h
* @copyright (c) 2013-2020 Robert Bosch Car Multimedia GmbH
* @brief useful description
* @addtogroup
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_MAPSETTINGS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_MAPSETTINGS_H_

#include <set>
#include <sstream>
#include <string>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{
namespace settings
{

enum MapOrientation
{
   MAP_ORIENTATION_INVALID,
   MAP_ORIENTATION_NORTH_UP,
   MAP_ORIENTATION_HEAD_UP
};

inline ::std::string toString(MapOrientation mapOrientation)
{
   switch (mapOrientation)
   {
   case MAP_ORIENTATION_INVALID:
      return "MAP_ORIENTATION_INVALID";
   case MAP_ORIENTATION_NORTH_UP:
      return "MAP_ORIENTATION_NORTH_UP";
   case MAP_ORIENTATION_HEAD_UP:
      return "MAP_ORIENTATION_HEAD_UP";
   default:
      ::std::stringstream stream;
      stream << "MAP_ORIENTATION_<" << static_cast<unsigned int>(mapOrientation) << ">";
      return stream.str();
   }
}

enum MapColorStyle
{
   MAPCOLORSTYLE_TYPE_1=0,
   MAPCOLORSTYLE_TYPE_2,
   MAPCOLORSTYLE_TYPE_3,
   MAPCOLORSTYLE_TYPE_4,
   MAPCOLORSTYLE_TYPE_5
};

inline ::std::string toString(MapColorStyle mapColorStyle)
{
   switch (mapColorStyle)
   {
   case MAPCOLORSTYLE_TYPE_1:
      return "MAPCOLORSTYLE_TYPE_1";
   case MAPCOLORSTYLE_TYPE_2:
      return "MAPCOLORSTYLE_TYPE_2";
   case MAPCOLORSTYLE_TYPE_3:
      return "MAPCOLORSTYLE_TYPE_3";
   case MAPCOLORSTYLE_TYPE_4:
      return "MAPCOLORSTYLE_TYPE_4";
   case MAPCOLORSTYLE_TYPE_5:
      return "MAPCOLORSTYLE_TYPE_5";
   default:
      ::std::stringstream stream;
      stream << "MAPCOLORSTYLE_TYPE_<" << static_cast<unsigned int>(mapColorStyle) << ">";
      return stream.str();
   }
}

enum MapLightMode
{
   MAPLIGHTMODE_AUTO,
   MAPLIGHTMODE_DAY,
   MAPLIGHTMODE_NIGHT,
};

inline ::std::string toString(MapLightMode mapLightMode)
{
   switch (mapLightMode)
   {
   case MAPLIGHTMODE_AUTO:
      return "MAPLIGHTMODE_AUTO";
   case MAPLIGHTMODE_DAY:
      return "MAPLIGHTMODE_DAY";
   case MAPLIGHTMODE_NIGHT:
      return "MAPLIGHTMODE_NIGHT";
   default:
      ::std::stringstream stream;
      stream << "MAPLIGHTMODE_<" << static_cast<unsigned int>(mapLightMode) << ">";
      return stream.str();
   }
}

enum MapTextSize
{
   MAPTEXTSIZE_MEDIUM=0,
   MAPTEXTSIZE_LARGE,
   MAPTEXTSIZE_SMALL
};

inline ::std::string toString(MapTextSize mapTextSize)
{
   switch (mapTextSize)
   {
   case MAPTEXTSIZE_MEDIUM:
      return "MAPTEXTSIZE_MEDIUM";
   case MAPTEXTSIZE_LARGE:
      return "MAPTEXTSIZE_LARGE";
   case MAPTEXTSIZE_SMALL:
      return "MAPTEXTSIZE_SMALL";
   default:
      ::std::stringstream stream;
      stream << "MAPTEXTSIZE_<" << static_cast<unsigned int>(mapTextSize) << ">";
      return stream.str();
   }
}

enum MapView
{
   MAPVIEW_2D,
   MAPVIEW_2D_SPLIT,
   MAPVIEW_3D,
   MAPVIEW_3D_SPLIT
};

inline ::std::string toString(MapView mapView)
{
   switch (mapView)
   {
   case MAPVIEW_2D:
      return "MAPVIEW_2D";
   case MAPVIEW_2D_SPLIT:
      return "MAPVIEW_2D_SPLIT";
   case MAPVIEW_3D:
      return "MAPVIEW_3D";
   case MAPVIEW_3D_SPLIT:
      return "MAPVIEW_3D_SPLIT";
   default:
      ::std::stringstream stream;
      stream << "MAPVIEW_<" << static_cast<unsigned int>(mapView) << ">";
      return stream.str();
   }
}

enum MapWeatherTypes
{
   MAPWEATHER_RADAR,
   MAPWEATHER_CHART,
   MAPWEATHER_STORM,
   MAPWEATHER_HURRICANE,
   MAPWEATHER_WINDS
};

inline ::std::string toString(MapWeatherTypes mapWeatherTypes)
{
   switch (mapWeatherTypes)
   {
   case MAPWEATHER_RADAR:
      return "MAPWEATHER_RADAR";
   case MAPWEATHER_CHART:
      return "MAPWEATHER_CHART";
   case MAPWEATHER_STORM:
      return "MAPWEATHER_STORM";
   case MAPWEATHER_HURRICANE:
      return "MAPWEATHER_HURRICANE";
   case MAPWEATHER_WINDS:
      return "MAPWEATHER_WINDS";
   default:
      ::std::stringstream stream;
      stream << "MAPWEATHER_<" << static_cast<unsigned int>(mapWeatherTypes) << ">";
      return stream.str();
   }
}

enum MapHMISplitMode
{
   MAP_HMI_SPLITMODE_INTERSECTION,
   MAP_HMI_SPLITMODE_DRIVE_REPORT,
   MAP_HMI_SPLITMODE_TURN_LIST,
   MAP_HMI_SPLITMODE_FULL_MAP
};

inline ::std::string toString(MapHMISplitMode mapHMISplitMode)
{
   switch (mapHMISplitMode)
   {
   case MAP_HMI_SPLITMODE_INTERSECTION:
      return "MAP_HMI_SPLITMODE_INTERSECTION";
   case MAP_HMI_SPLITMODE_DRIVE_REPORT:
      return "MAP_HMI_SPLITMODE_DRIVE_REPORT";
   case MAP_HMI_SPLITMODE_TURN_LIST:
      return "MAP_HMI_SPLITMODE_TURN_LIST";
   case MAP_HMI_SPLITMODE_FULL_MAP:
      return "MAP_HMI_SPLITMODE_FULL_MAP";
   default:
      ::std::stringstream stream;
      stream << "MAP_HMI_SPLITMODE_<" << static_cast<unsigned int>(mapHMISplitMode) << ">";
      return stream.str();
   }
}

enum TimeDisplay
{
   TIME_DISPLAY__ARRIVAL_TIME_TO_DESTINATION = 0,
   TIME_DISPLAY__TIME_LEFT_TO_DESTINATION
};

inline ::std::string toString(TimeDisplay timeDisplay)
{
   switch (timeDisplay)
   {
   case TIME_DISPLAY__ARRIVAL_TIME_TO_DESTINATION:
      return "TIME_DISPLAY__ARRIVAL_TIME_TO_DESTINATION";
   case TIME_DISPLAY__TIME_LEFT_TO_DESTINATION:
      return "TIME_DISPLAY__TIME_LEFT_TO_DESTINATION";
   default:
      ::std::stringstream stream;
      stream << "TIME_DISPLAY_" << static_cast<unsigned int>(timeDisplay) << ">";
      return stream.str();
   }
}

class MapSettings
{
public:
   MapSettings():
      m_mapOrientation(MAP_ORIENTATION_HEAD_UP),
      m_mapLongRange(true),
      m_mapColorStyle(MAPCOLORSTYLE_TYPE_1),
      m_mapLightMode(MAPLIGHTMODE_AUTO),
      m_mapTextSize(MAPTEXTSIZE_MEDIUM),
      m_northUpZoomOut(true),
      m_showStreetViewOnArrival(true),
      m_isAutoZoomEnabled(false),
      m_isAutoShowTurnListOnFreewayEnabled(true),
      m_isAudioDisplayOnMapEnabled(true),
      m_isShowAllFreewayExitsOnMapEnabled(true),
      m_mapSecondaryViewOrientation(MAP_ORIENTATION_HEAD_UP),
      m_mapSecondaryViewLongRange(true),
      m_mapSecondaryViewMapScale(0),
      m_isSmallStepZoomingOnMapEnabled(true),
      m_isWeatherDisplayEnabled(true),
      m_isTrafficDisplayEnabled(true),
      m_isJunctionViewEnabled(true),
      m_timeDisplay(TIME_DISPLAY__ARRIVAL_TIME_TO_DESTINATION),
      m_isShowCurrentStreetOnMapEnabled(true),
      m_isShowFuelPricesEnabled(true)
   {
   }

   MapSettings(
      const MapOrientation& mapOrientation,
      bool mapLongRange,
      const MapColorStyle& mapColorStyle,
      const MapLightMode& mapLightMode,
      const MapTextSize& mapTextSize,
      bool northUpZoomOut,
      bool showStreetViewOnArrival,
      bool isAutoZoomEnabled,
      bool isAutoShowTurnListOnFreewayEnabled,
      bool isAudioDisplayOnMapEnabled,
      bool isShowAllFreewayExitsOnMapEnabled,
      const MapOrientation& mapSecondaryViewOrientation,
      bool mapSecondaryViewLongRange,
      int mapSecondaryViewMapScale,
      bool isSmallStepZoomingOnMapEnabled,
      bool isWeatherDisplayEnabled,
      bool isTrafficDisplayEnabled,
      bool isJunctionViewEnabled,
      const TimeDisplay& timeDisplay,
      bool isShowCurrentStreetOnMapEnabled,
      bool isShowFuelPricesEnabled)

      : m_mapOrientation(mapOrientation)
      , m_mapLongRange(mapLongRange)
      , m_mapColorStyle(mapColorStyle)
      , m_mapLightMode(mapLightMode)
      , m_mapTextSize(mapTextSize)
      , m_northUpZoomOut(northUpZoomOut)
      , m_showStreetViewOnArrival(showStreetViewOnArrival)
      , m_isAutoZoomEnabled(isAutoZoomEnabled)
      , m_isAutoShowTurnListOnFreewayEnabled(isAutoShowTurnListOnFreewayEnabled)
      , m_isAudioDisplayOnMapEnabled(isAudioDisplayOnMapEnabled)
      , m_isShowAllFreewayExitsOnMapEnabled(isShowAllFreewayExitsOnMapEnabled)
      , m_mapSecondaryViewOrientation(mapSecondaryViewOrientation)
      , m_mapSecondaryViewLongRange(mapSecondaryViewLongRange)
      , m_mapSecondaryViewMapScale(mapSecondaryViewMapScale)
      , m_isSmallStepZoomingOnMapEnabled(isSmallStepZoomingOnMapEnabled)
      , m_isWeatherDisplayEnabled(isWeatherDisplayEnabled)
      , m_isTrafficDisplayEnabled(isTrafficDisplayEnabled)
      , m_isJunctionViewEnabled(isJunctionViewEnabled)
      , m_timeDisplay(timeDisplay)
      , m_isShowCurrentStreetOnMapEnabled(isShowCurrentStreetOnMapEnabled)
      , m_isShowFuelPricesEnabled(isShowFuelPricesEnabled)
   {
   }

   MapOrientation getMapOrientation() const
   {
      return m_mapOrientation;
   }

   void setMapOrientation(MapOrientation mapOrientation)
   {
      m_mapOrientation = mapOrientation;
   }

   void setMapLongRange(bool mapLongRange)
   {
      m_mapLongRange = mapLongRange;
   }

   bool isMapLongRangeEnabled() const
   {
      return m_mapLongRange;
   }

   MapColorStyle getMapColorStyle() const
   {
      return m_mapColorStyle;
   }
   void setMapColorStyle(MapColorStyle mapColorStyle)
   {
      m_mapColorStyle = mapColorStyle;
   }

   MapLightMode getMapLightMode() const
   {
      return m_mapLightMode;
   }

   void setMapLightMode(MapLightMode mapLightMode)
   {
      m_mapLightMode = mapLightMode;
   }

   MapTextSize getMapTextSize() const
   {
      return m_mapTextSize;
   }
   void setMapTextSize(MapTextSize mapTextSize)
   {
      m_mapTextSize = mapTextSize;
   }

   bool isNorthUpZoomOutEnabled() const
   {
      return m_northUpZoomOut;
   }
   void setNorthUpZoomOut(bool northUpZoomOut)
   {
      m_northUpZoomOut = northUpZoomOut;
   }

   bool isShowStreetViewOnArrivalEnabled() const
   {
      return m_showStreetViewOnArrival;
   }
   void setShowStreetViewOnArrival(bool showStreetViewOnArrival)
   {
      m_showStreetViewOnArrival = showStreetViewOnArrival;
   }

   bool isAutoZoomEnabled() const
   {
      return m_isAutoZoomEnabled;
   }
   void setAutoZoomEnabled(bool isAutoZoomEnabled)
   {
      m_isAutoZoomEnabled = isAutoZoomEnabled;
   }
   bool isAutoShowTurnListOnFreewayEnabled() const
   {
      return m_isAutoShowTurnListOnFreewayEnabled;
   }
   void setAutoShowTurnListOnFreeway(bool isAutoShowTurnListOnFreewayEnabled)
   {
      m_isAutoShowTurnListOnFreewayEnabled = isAutoShowTurnListOnFreewayEnabled;
   }

   void setAudioDisplayOnMap(bool mapAudioDisplay)
   {
      m_isAudioDisplayOnMapEnabled = mapAudioDisplay;
   }

   bool isAudioDisplayOnMapEnabled() const
   {
      return m_isAudioDisplayOnMapEnabled;
   }

   void setShowAllFreewayExitsOnMap(bool isShowAllFreewayExitsOnMapEnabled)
   {
      m_isShowAllFreewayExitsOnMapEnabled = isShowAllFreewayExitsOnMapEnabled;
   }

   bool isShowAllFreewayExitsOnMapEnabled() const
   {
      return m_isShowAllFreewayExitsOnMapEnabled;
   }

   MapOrientation getMapSecondaryViewOrientation() const
   {
      return m_mapSecondaryViewOrientation;
   }

   void setMapSecondaryViewOrientation(MapOrientation mapOrientation)
   {
      m_mapSecondaryViewOrientation = mapOrientation;
   }

   void setMapSecondaryViewLongRange(bool mapLongRange)
   {
      m_mapSecondaryViewLongRange = mapLongRange;
   }

   bool isMapSecondaryViewLongRangeEnabled() const
   {
      return m_mapSecondaryViewLongRange;
   }

   void setMapSecondaryViewMapScale(int mapScale)
   {
      m_mapSecondaryViewMapScale = mapScale;
   }

   int getMapSecondaryViewMapScale() const
   {
      return m_mapSecondaryViewMapScale;
   }

   void setSmallStepZoomingOnMap(bool smallStepZoomingOnMap)
   {
      m_isSmallStepZoomingOnMapEnabled = smallStepZoomingOnMap;
   }

   bool isWeatherMapDisplayEnabled() const
   {
      return m_isWeatherDisplayEnabled;
   }

   void setWeatherMapDisplay(bool isEnabled)
   {
      m_isWeatherDisplayEnabled = isEnabled;
   }

   bool isTrafficDisplayEnabled() const
   {
      return m_isTrafficDisplayEnabled;
   }

   void setTrafficDisplay(bool isEnabled)
   {
      m_isTrafficDisplayEnabled = isEnabled;
   }

   bool isSmallStepZoomingOnMapEnabled() const
   {
      return m_isSmallStepZoomingOnMapEnabled;
   }

   bool isJunctionViewEnabled() const
   {
      return m_isJunctionViewEnabled;
   }

   void setJunctionView(bool isEnabled)
   {
      m_isJunctionViewEnabled = isEnabled;
   }

   TimeDisplay getTimeDisplay() const
   {
      return m_timeDisplay;
   }

   void setTimeDisplay(TimeDisplay timeDisplay)
   {
      m_timeDisplay = timeDisplay;
   }

   void setShowCurrentStreetOnMap(bool showCurrentStreetOnMap)
   {
      m_isShowCurrentStreetOnMapEnabled = showCurrentStreetOnMap;
   }

   bool isShowCurrentStreetOnMapEnabled() const
   {
      return m_isShowCurrentStreetOnMapEnabled;
   }

   void setIsFuelPricesEnabled(bool isShowFuelPricesEnabled)
   {
      m_isShowFuelPricesEnabled = isShowFuelPricesEnabled;
   }

   bool isFuelPricesEnabled() const
   {
      return m_isShowFuelPricesEnabled;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapSettings payload:\n");
      stream << ::std::endl
             << "MapOrientation = " << ::navmiddleware::settings::toString(m_mapOrientation) << ::std::endl
             << "MapLongRange = " << m_mapLongRange << ::std::endl
             << "MapColorStyle = " << ::navmiddleware::settings::toString(m_mapColorStyle) << ::std::endl
             << "MapLightMode = " << ::navmiddleware::settings::toString(m_mapLightMode) << ::std::endl
             << "MapTextSize = " << ::navmiddleware::settings::toString(m_mapTextSize) << ::std::endl
             << "NorthUpZoomOut = " << m_northUpZoomOut << ::std::endl
             << "ShowStreetViewOnArrival = " << m_showStreetViewOnArrival << ::std::endl
             << "IsAutoZoomEnabled = " << m_isAutoZoomEnabled << ::std::endl
             << "IsAutoTurnShowOnFreewayEnabled = " << m_isAutoShowTurnListOnFreewayEnabled << ::std::endl
             << "IsAudioDisplayOnMapEnabled = " << m_isAudioDisplayOnMapEnabled << ::std::endl
             << "IsShowAllFreewayExitsOnMapEnabled = " << m_isShowAllFreewayExitsOnMapEnabled << ::std::endl
             << "MapSecondaryViewOrientation = " << ::navmiddleware::settings::toString(m_mapSecondaryViewOrientation) << ::std::endl
             << "MapSecondaryViewLongRange = " << m_mapSecondaryViewLongRange << ::std::endl
             << "MapSecondaryViewMapScale = " << m_mapSecondaryViewMapScale << ::std::endl
             << "IsSmallStepZoomingOnMapEnabled = " << m_isSmallStepZoomingOnMapEnabled << ::std::endl
             << "IsWeatherDisplayEnabled = " << m_isWeatherDisplayEnabled << ::std::endl
             << "IsTrafficDisplayEnabled = " << m_isTrafficDisplayEnabled << ::std::endl
             << "IsJunctionViewEnabled = " << m_isJunctionViewEnabled << ::std::endl
             << "Time Display = " << ::navmiddleware::settings::toString(m_timeDisplay) << ::std::endl
             << "IsShowCurrentStreetOnMapEnabled = " << m_isShowCurrentStreetOnMapEnabled << ::std::endl
             << "Show Fuel Prices = " << m_isShowFuelPricesEnabled << ::std::endl;


      return stream.str();
   }

   bool operator==(const MapSettings& rhs) const
   {
      return ((m_mapOrientation == rhs.m_mapOrientation) &&
              (m_mapLongRange == rhs.m_mapLongRange) &&
              (m_mapColorStyle == rhs.m_mapColorStyle) &&
              (m_mapLightMode == rhs.m_mapLightMode) &&
              (m_mapTextSize == rhs.m_mapTextSize) &&
              (m_northUpZoomOut == rhs.m_northUpZoomOut) &&
              (m_showStreetViewOnArrival == rhs.m_showStreetViewOnArrival) &&
              (m_isAutoZoomEnabled == rhs.m_isAutoZoomEnabled) &&
              (m_isAutoShowTurnListOnFreewayEnabled == rhs.m_isAutoShowTurnListOnFreewayEnabled) &&
              (m_isAudioDisplayOnMapEnabled == rhs.m_isAudioDisplayOnMapEnabled) &&
              (m_isShowAllFreewayExitsOnMapEnabled == rhs.m_isShowAllFreewayExitsOnMapEnabled)&&
              (m_mapSecondaryViewOrientation == rhs.m_mapSecondaryViewOrientation) &&
              (m_mapSecondaryViewLongRange == rhs.m_mapSecondaryViewLongRange) &&
              (m_mapSecondaryViewMapScale == rhs.m_mapSecondaryViewMapScale) &&
              (m_isSmallStepZoomingOnMapEnabled == rhs.m_isSmallStepZoomingOnMapEnabled) &&
              (m_isWeatherDisplayEnabled == rhs.m_isWeatherDisplayEnabled) &&
              (m_isTrafficDisplayEnabled == rhs.m_isTrafficDisplayEnabled) &&
              (m_isJunctionViewEnabled == rhs.m_isJunctionViewEnabled) &&
              (m_timeDisplay == rhs.m_timeDisplay) &&
              (m_isShowCurrentStreetOnMapEnabled == rhs.m_isShowCurrentStreetOnMapEnabled) &&
              (m_isShowFuelPricesEnabled == rhs.m_isShowFuelPricesEnabled));
   }

   bool operator!=(const MapSettings& rhs) const
   {
      return (!(*this == rhs));
   }

private:
   MapOrientation m_mapOrientation;
   bool m_mapLongRange;
   MapColorStyle m_mapColorStyle;
   MapLightMode m_mapLightMode;
   MapTextSize m_mapTextSize;
   bool m_northUpZoomOut;
   bool m_showStreetViewOnArrival;
   bool m_isAutoZoomEnabled;
   bool m_isAutoShowTurnListOnFreewayEnabled;
   bool m_isAudioDisplayOnMapEnabled;
   bool m_isShowAllFreewayExitsOnMapEnabled;
   MapOrientation m_mapSecondaryViewOrientation;
   bool m_mapSecondaryViewLongRange;
   int m_mapSecondaryViewMapScale;
   bool m_isSmallStepZoomingOnMapEnabled;
   bool m_isWeatherDisplayEnabled;
   bool m_isTrafficDisplayEnabled;
   bool m_isJunctionViewEnabled;
   TimeDisplay m_timeDisplay;
   bool m_isShowCurrentStreetOnMapEnabled;
   bool m_isShowFuelPricesEnabled;
};

class MapTextFilterSettings
{
public:
   MapTextFilterSettings():
      m_roadNameFilter(true),
      m_facilityNameFilter(true),
      m_placeNameFilter(true),
      m_intersectionNameFilter(false)
   {
   }

   MapTextFilterSettings(
      bool roadNameFilter,
      bool facilityNameFilter,
      bool placeNameFilter,
      bool intersectionNameFilter)

      : m_roadNameFilter(roadNameFilter)
      , m_facilityNameFilter(facilityNameFilter)
      , m_placeNameFilter(placeNameFilter)
      , m_intersectionNameFilter(intersectionNameFilter)
   {
   }

   bool isPlaceNameFilterEnabled() const
   {
      return m_placeNameFilter;
   }
   void setPlaceNameFilter(bool placeName)
   {
      m_placeNameFilter = placeName;
   }
   bool isFacilityNameFilterEnabled() const
   {
      return m_facilityNameFilter;
   }
   void setFacilityNameFilter(bool facilityName)
   {
      m_facilityNameFilter = facilityName;
   }
   bool isRoadNameFilterEnabled() const
   {
      return m_roadNameFilter;
   }
   void setRoadNameFilter(bool roadName)
   {
      m_roadNameFilter = roadName;
   }

   void setIntersectionNameFilter(bool intersectionNameFilter)
   {
      m_intersectionNameFilter = intersectionNameFilter;
   }

   bool getIntersectionNameFilter() const
   {
      return m_intersectionNameFilter;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapTextFilterSettings payload:\n");
      stream << ::std::endl
             << "RoadNameFilter = " << m_roadNameFilter << ::std::endl
             << "FacilityNameFilter = " << m_facilityNameFilter << ::std::endl
             << "PlaceNameFilter = " << m_placeNameFilter << ::std::endl
             << "IntersectionNameFilter = " << m_intersectionNameFilter << ::std::endl;
      return stream.str();
   }

   bool operator==(const MapTextFilterSettings& rhs) const
   {
      return ((m_roadNameFilter == rhs.m_roadNameFilter) &&
              (m_facilityNameFilter == rhs.m_facilityNameFilter) &&
              (m_placeNameFilter == rhs.m_placeNameFilter) &&
              (m_intersectionNameFilter == rhs.m_intersectionNameFilter));
   }

   bool operator!=(const MapTextFilterSettings& rhs) const
   {
      return (!(*this == rhs));
   }

private:
   bool m_roadNameFilter;
   bool m_facilityNameFilter;
   bool m_placeNameFilter;
   // [JP]: If true, intersection names will be shown on map
   bool m_intersectionNameFilter;
};

class MapDetailTriggerSettings
{
public:
   MapDetailTriggerSettings():
      m_addressBookIcons(true),
      m_tmcTrafficIcons(true),
      m_destinationWaypointIcons(true),
      m_showPopUpForPOIs(true),
      m_latLongInfo(false),
      m_streets(true)
   {
   }

   MapDetailTriggerSettings(
      bool addressBookIcons,
      bool tmcTrafficIcons,
      bool destinationWaypointIcons,
      bool showPopUpForPOIs,
      bool latLongInfo,
      bool streets)

      : m_addressBookIcons(addressBookIcons)
      , m_tmcTrafficIcons(tmcTrafficIcons)
      , m_destinationWaypointIcons(destinationWaypointIcons)
      , m_showPopUpForPOIs(showPopUpForPOIs)
      , m_latLongInfo(latLongInfo)
      , m_streets(streets)
   {
   }

   bool isAddressBookIconsEnabled() const
   {
      return m_addressBookIcons;
   }

   void setAddressBookIcons(bool addressBookIcons)
   {
      m_addressBookIcons = addressBookIcons;
   }

   bool isTMCTrafficIconsEnabled() const
   {
      return m_tmcTrafficIcons;
   }

   void setTMCTrafficIcons(bool tmcTrafficIcons)
   {
      m_tmcTrafficIcons = tmcTrafficIcons;
   }

   bool isDestinationWaypointIconsEnabled() const
   {
      return m_destinationWaypointIcons;
   }

   void setDestinationWaypointIcons(bool destinationWaypointIcons)
   {
      m_destinationWaypointIcons = destinationWaypointIcons;
   }

   bool isShowPopUpForPOIsEnabled() const
   {
      return m_showPopUpForPOIs;
   }

   void setShowPopUpForPOIs(bool showPopUpForPOIs)
   {
      m_showPopUpForPOIs = showPopUpForPOIs;
   }

   bool isLatLongInfoEnabled() const
   {
      return m_latLongInfo;
   }

   void setLatLongInfo(bool latLongInfo)
   {
      m_latLongInfo = latLongInfo;
   }

   bool isStreetsEnabled() const
   {
      return m_streets;
   }

   void setStreets(bool streets)
   {
      m_streets = streets;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapDetailTriggerSettings payload:\n");
      stream << ::std::endl
             << "AddressBookIcons = " << m_addressBookIcons << ::std::endl
             << "TmcTrafficIcons = " << m_tmcTrafficIcons << ::std::endl
             << "DestinationWaypointIcons = " << m_destinationWaypointIcons << ::std::endl
             << "ShowPopUpForPOIs = " << m_showPopUpForPOIs << ::std::endl
             << "LatLongInfo = " << m_latLongInfo << ::std::endl
             << "Streets = " << m_streets << ::std::endl;
      return stream.str();
   }

   bool operator==(const MapDetailTriggerSettings& rhs) const
   {
      return ((m_addressBookIcons == rhs.m_addressBookIcons) &&
              (m_tmcTrafficIcons == rhs.m_tmcTrafficIcons) &&
              (m_destinationWaypointIcons == rhs.m_destinationWaypointIcons) &&
              (m_showPopUpForPOIs == rhs.m_showPopUpForPOIs) &&
              (m_latLongInfo == rhs.m_latLongInfo) &&
              (m_streets == rhs.m_streets));
   }

   bool operator!=(const MapDetailTriggerSettings& rhs) const
   {
      return (!(*this == rhs));
   }

private:
   bool m_addressBookIcons;
   bool m_tmcTrafficIcons;
   bool m_destinationWaypointIcons;
   bool m_showPopUpForPOIs;
   bool m_latLongInfo;
   bool m_streets;
};

class ChangeMapViewSettings
{
public:
   ChangeMapViewSettings():
      m_mapView(MAPVIEW_2D)
   {
   }

   ChangeMapViewSettings(
      MapView mapView)

      : m_mapView(mapView)
   {
   }

   void setMapView(MapView mapView)
   {
      m_mapView = mapView;
   }
   MapView getMapView() const
   {
      return m_mapView;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ChangeMapViewSettings payload:\n");
      stream << ::std::endl
             << "MapView = " << ::navmiddleware::settings::toString(m_mapView) << ::std::endl;
      return stream.str();
   }

   bool operator==(const ChangeMapViewSettings& rhs) const
   {
      return (m_mapView == rhs.m_mapView);
   }

   bool operator!=(const ChangeMapViewSettings& rhs) const
   {
      return (!(*this == rhs));
   }

private:
   MapView m_mapView;
};

class MapViewSettings
{
public:
   MapViewSettings():
      m_mapHMISplitMode(MAP_HMI_SPLITMODE_FULL_MAP)
   {
   }

   MapViewSettings(
      MapHMISplitMode mapHMISplitMode)

      : m_mapHMISplitMode(mapHMISplitMode)
   {
   }

   void setMapHMISplitMode(MapHMISplitMode mapHMISplitMode)
   {
      m_mapHMISplitMode = mapHMISplitMode;
   }
   MapHMISplitMode getMapHMISplitMode() const
   {
      return m_mapHMISplitMode;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapViewSettings payload:\n");
      stream << ::std::endl
             << "m_mapHMISplitMode = " << ::navmiddleware::settings::toString(m_mapHMISplitMode) << ::std::endl;
      return stream.str();
   }

   bool operator==(const MapViewSettings& rhs) const
   {
      return (m_mapHMISplitMode == rhs.m_mapHMISplitMode);
   }

   bool operator!=(const MapViewSettings& rhs) const
   {
      return (!(*this == rhs));
   }

private:
   MapHMISplitMode m_mapHMISplitMode;
};

class MapWeatherSettings
{
public:
   MapWeatherSettings():
      m_mapWeatherTypes(MAPWEATHER_RADAR)
   {
   }

   MapWeatherSettings(
      MapWeatherTypes mapWeatherTypes)

      : m_mapWeatherTypes(mapWeatherTypes)
   {
   }

   void setMapWeather(MapWeatherTypes mapWeatherTypes)
   {
      m_mapWeatherTypes = mapWeatherTypes;
   }

   MapWeatherTypes getMapWeather() const
   {
      return m_mapWeatherTypes;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapWeatherSettings payload:\n");
      stream << ::std::endl
             << "MapWeatherTypes = " << ::navmiddleware::settings::toString(m_mapWeatherTypes) << ::std::endl;
      return stream.str();
   }

   bool operator==(const MapWeatherSettings& rhs) const
   {
      return (m_mapWeatherTypes == rhs.m_mapWeatherTypes);
   }

   bool operator!=(const MapWeatherSettings& rhs) const
   {
      return (!(*this == rhs));
   }

private:
   MapWeatherTypes m_mapWeatherTypes;
};

class MapTrackingDotsSettings
{
public:
   // enum for the 4 distance values available for the tracking dot distance
   enum MapTrackingDotsDistance
   {
      MAP_TRACKING_DOT_DISTANCE_FIFTY_METER = 50,
      MAP_TRACKING_DOT_DISTANCE_HUNDRED_METER = 100,
      MAP_TRACKING_DOT_DISTANCE_TWO_HUNDRED_METER = 200,
      MAP_TRACKING_DOT_DISTANCE_FIVE_HUNDRED_METER = 500
   };

   static ::std::string toString(MapTrackingDotsDistance mapTrackingDotsDistance)
   {
      switch (mapTrackingDotsDistance)
      {
      case MAP_TRACKING_DOT_DISTANCE_FIFTY_METER:
         return "MAP_TRACKING_DOT_DISTANCE_FIFTY_METER";
      case MAP_TRACKING_DOT_DISTANCE_HUNDRED_METER:
         return "MAP_TRACKING_DOT_DISTANCE_HUNDRED_METER";
      case MAP_TRACKING_DOT_DISTANCE_TWO_HUNDRED_METER:
         return "MAP_TRACKING_DOT_DISTANCE_TWO_HUNDRED_METER";
      case MAP_TRACKING_DOT_DISTANCE_FIVE_HUNDRED_METER:
         return "MAP_TRACKING_DOT_DISTANCE_FIVE_HUNDRED_METER";
      default:
         ::std::stringstream stream;
         stream << "MAP_TRACKING_DOT_DISTANCE_<" << static_cast<unsigned int>(mapTrackingDotsDistance) << ">";
         return stream.str();
      }
   }

   MapTrackingDotsSettings() :
      m_isTrackingDotDisplayOnMapEnabled(true),
      m_mapTrackingDotsDistance(MAP_TRACKING_DOT_DISTANCE_TWO_HUNDRED_METER)
   {
   }

   MapTrackingDotsSettings(
      bool isTrackingDotDisplayOnMapEnabled,
      MapTrackingDotsDistance mapTrackingDotsDistance)

      : m_isTrackingDotDisplayOnMapEnabled(isTrackingDotDisplayOnMapEnabled)
      , m_mapTrackingDotsDistance(mapTrackingDotsDistance)
   {
   }

   bool operator==(const MapTrackingDotsSettings& rhs) const
   {
      return ((m_isTrackingDotDisplayOnMapEnabled == rhs.m_isTrackingDotDisplayOnMapEnabled)
              && (m_mapTrackingDotsDistance  == rhs.m_mapTrackingDotsDistance));
   }

   bool operator!=(const MapTrackingDotsSettings& rhs) const
   {
      return (!(*this == rhs));
   }

   void setTrackingDotDisplayState(bool state)
   {
      m_isTrackingDotDisplayOnMapEnabled = state;
   }

   bool getTrackingDotDisplayState() const
   {
      return m_isTrackingDotDisplayOnMapEnabled;
   }

   void setMapTrackingDotDistance(MapTrackingDotsDistance mapTrackingDotDistance)
   {
      m_mapTrackingDotsDistance = mapTrackingDotDistance;
   }

   MapTrackingDotsDistance getMapTrackingDotDistance() const
   {
      return m_mapTrackingDotsDistance;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapTrackingDotsSettings payload:\n");
      stream << ::std::endl
             << "IsTrackingDotDisplayOnMapEnabled = " << m_isTrackingDotDisplayOnMapEnabled << ::std::endl
             << "MapTrackingDotDistance = " << toString(m_mapTrackingDotsDistance) << ::std::endl;
      return stream.str();
   }

private:
   bool                       m_isTrackingDotDisplayOnMapEnabled;
   MapTrackingDotsDistance    m_mapTrackingDotsDistance;
};

// [JP]: Brand information associated with a POI category
struct BrandInfo
{
   BrandInfo()
      : m_enabled(false)
   {
   }

   BrandInfo(const ::std::string& brandName, const Image& brandImage, bool enabled)
      : m_brandName(brandName)
      , m_brandImage(brandImage)
      , m_enabled(enabled)
   {
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("BrandInfo:\n");
      stream << "m_brandName = " << m_brandName << ", "
             << "m_enabled = " << m_enabled;

      return stream.str();
   }

   bool operator!=(const BrandInfo& rhs) const
   {
      return (!(*this == rhs));
   }

   bool operator==(const BrandInfo& rhs) const
   {
      // Brand image comparison intentionally omitted
      return ((m_brandName == rhs.m_brandName) && (m_enabled == rhs.m_enabled));
   }

   ::std::string m_brandName;
   Image m_brandImage;
   bool m_enabled;
};


class MapDynamicIconSettings
{
public:
   struct PoiCategoryInfo
   {
      PoiCategoryInfo()
         : m_poiCategoryId(0)
         , m_poiCategoryVisibilityOnMap(false)
      {
      }

      PoiCategoryInfo(const ::std::string& poiCategoryName, unsigned int poiCategoryId, bool poiCategoryVisibilityOnMap,
                      const ::std::vector<BrandInfo>& availableBrands, const ValidValue<Image>& poiCategoryIcon,
                      const ValidValue<SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfo)
         : m_poiCategoryName(poiCategoryName)
         , m_poiCategoryId(poiCategoryId)
         , m_poiCategoryVisibilityOnMap(poiCategoryVisibilityOnMap)
         , m_availableBrands(availableBrands)
         , m_poiCategoryIcon(poiCategoryIcon)
         , m_sdsGlobalNamedObjectIdInfo(sdsGlobalNamedObjectIdInfo)
      {
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("MapDynamicIconSettings::PoiCategoryInfo:\n");
         stream << "CategoryName = " << m_poiCategoryName << ", "
                << "PoiCategoryId = " << m_poiCategoryId << ", "
                << "VisibilityOnMap = " << m_poiCategoryVisibilityOnMap
                << "Available brands count: " << m_availableBrands.size()
                << "Image :" << (m_poiCategoryIcon.isValid() ? m_poiCategoryIcon.getValue().toString() : "")
                << "m_sdsGlobalNamedObjectIdInfo = " << (m_sdsGlobalNamedObjectIdInfo.isValid() ?
                      m_sdsGlobalNamedObjectIdInfo.getValue().toString() : "No valid NDSId") << ::std::endl;

         for (::std::vector<BrandInfo>::const_iterator it = m_availableBrands.begin(); it != m_availableBrands.end(); ++it)
         {
            stream << ::std::endl << it->toString();
         }
         stream << ::std::endl;
         return stream.str();
      }

      bool operator!=(const PoiCategoryInfo& rhs) const
      {
         return (!(*this == rhs));
      }

      bool operator==(const PoiCategoryInfo& rhs) const
      {
         return ((m_poiCategoryId == rhs.m_poiCategoryId) &&
                 (m_poiCategoryName == rhs.m_poiCategoryName) &&
                 (m_poiCategoryVisibilityOnMap == rhs.m_poiCategoryVisibilityOnMap) &&
                 (m_availableBrands == rhs.m_availableBrands) &&
                 (m_poiCategoryIcon == rhs.m_poiCategoryIcon) &&
                 (m_sdsGlobalNamedObjectIdInfo == rhs.m_sdsGlobalNamedObjectIdInfo));
      }

      ::std::string m_poiCategoryName;
      unsigned int m_poiCategoryId;
      bool m_poiCategoryVisibilityOnMap;
      // [JP]: POI category may have 0...n available brands
      ::std::vector<BrandInfo> m_availableBrands;
      ValidValue<Image> m_poiCategoryIcon;
      ValidValue<SdsGlobalNamedObjectIdInfo> m_sdsGlobalNamedObjectIdInfo;
   };

   MapDynamicIconSettings()
   {
   }

   MapDynamicIconSettings(
      const ::std::vector<PoiCategoryInfo>& poiCategoryInfoList,
      const LanguageEntryInfo& languageEntryInfo)

      : m_poiCategoryInfoList(poiCategoryInfoList)
      , m_languageEntryInfo(languageEntryInfo)
   {
   }

   /*
    * get a read-only list of the dynamic POI category info.
    */
   const ::std::vector<PoiCategoryInfo>& getPoiCategoryInfoList() const
   {
      return m_poiCategoryInfoList;
   }

   /*
    * get a modifiable list of the dynamic POI category info.
    */
   ::std::vector<PoiCategoryInfo>& getPoiCategoryInfoList()
   {
      return m_poiCategoryInfoList;
   }

   void setLanguageEntryInfo(const LanguageEntryInfo& languageEntryInfo)
   {
      m_languageEntryInfo = languageEntryInfo;
   }

   const LanguageEntryInfo& getLanguageEntryInfo() const
   {
      return m_languageEntryInfo;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapDynamicIconSettings payload:\n");
      for (::std::vector<PoiCategoryInfo>::const_iterator poiCategoryInfoItr = m_poiCategoryInfoList.begin();
            poiCategoryInfoItr != m_poiCategoryInfoList.end(); ++poiCategoryInfoItr)
      {
         stream << ::std::endl << poiCategoryInfoItr->toString();
      }
      return stream.str();
   }

   bool operator!=(const MapDynamicIconSettings& rhs) const
   {
      return (!(*this == rhs));
   }

   bool operator==(const MapDynamicIconSettings& rhs) const
   {
      return (m_poiCategoryInfoList == rhs.m_poiCategoryInfoList)
             && (m_languageEntryInfo == rhs.m_languageEntryInfo);
   }

private:
   ::std::vector<PoiCategoryInfo> m_poiCategoryInfoList;
   /** Language code used for Translation of POI Categories (Used in SDS context only) */
   LanguageEntryInfo m_languageEntryInfo;
};

class MapEvSettings
{
public:

   MapEvSettings() :
      m_showChargingStations(true),
      m_showChargingStationsOpen24Hours(true),
      m_showClosedChargingStations(false),
      m_showChargingStationsNewOrUpdated(false),
      m_showChargingStationsNCS(false)
   {
   }

   MapEvSettings(
      bool showChargingStations,
      bool showChargingStationsOpen24Hours,
      bool showClosedChargingStations,
      bool showChargingStationsNewOrUpdated,
      bool showChargingStationsNCS)

      : m_showChargingStations(showChargingStations)
      , m_showChargingStationsOpen24Hours(showChargingStationsOpen24Hours)
      , m_showClosedChargingStations(showClosedChargingStations)
      , m_showChargingStationsNewOrUpdated(showChargingStationsNewOrUpdated)
      , m_showChargingStationsNCS(showChargingStationsNCS)
   {
   }

   bool operator==(const MapEvSettings& rhs) const
   {
      return ((m_showChargingStations == rhs.m_showChargingStations)
              && (m_showChargingStationsOpen24Hours  == rhs.m_showChargingStationsOpen24Hours)
              && (m_showClosedChargingStations == rhs.m_showClosedChargingStations)
              && (m_showChargingStationsNewOrUpdated == rhs.m_showChargingStationsNewOrUpdated)
              && (m_showChargingStationsNCS == rhs.m_showChargingStationsNCS));
   }

   bool operator!=(const MapEvSettings& rhs) const
   {
      return (!(*this == rhs));
   }

   void setShowChargingStations(bool chargingStations)
   {
      m_showChargingStations = chargingStations;
   }

   bool isChargingStationsEnabled() const
   {
      return m_showChargingStations;
   }

   void setShowChargingStationsOpen24Hours(bool chargingStationsOpen24Hours )
   {
      m_showChargingStationsOpen24Hours = chargingStationsOpen24Hours;
   }

   bool isShowChargingStationsOpen24HoursEnabled() const
   {
      return m_showChargingStationsOpen24Hours;
   }

   void setShowClosedChargingStations(bool closedChargingStations)
   {
      m_showClosedChargingStations = closedChargingStations;
   }

   bool isShowClosedChargingStationsEnabled() const
   {
      return m_showClosedChargingStations;
   }

   void setShowChargingStationsNewOrUpdated(bool chargingStationsNewOrUpdated)
   {
      m_showChargingStationsNewOrUpdated = chargingStationsNewOrUpdated;
   }

   bool isShowChargingStationsNewOrUpdatedEnabled() const
   {
      return m_showChargingStationsNewOrUpdated;
   }

   void setShowChargingStationsNCS(bool chargingStationsNCS)
   {
      m_showChargingStationsNCS = chargingStationsNCS;
   }

   bool isShowChargingStationsNCSEnabled() const
   {
      return m_showChargingStationsNCS;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("EVSettings payload:\n");
      stream << ::std::endl
             << "ShowChargingStations = " << m_showChargingStations << ::std::endl
             << "ShowChargingStationsOpen24Hours = " << m_showChargingStationsOpen24Hours << ::std::endl
             << "ShowClosedChargingStations = " << m_showClosedChargingStations << ::std::endl
             << "ShowChargingStationsNewOrUpdated = " << m_showChargingStationsNewOrUpdated << ::std::endl
             << "ShowChargingStationsNCS = " << m_showChargingStationsNCS << ::std::endl;
      return stream.str();
   }

private:
   bool                       m_showChargingStations;
   bool                       m_showChargingStationsOpen24Hours;
   bool                       m_showClosedChargingStations;
   bool                       m_showChargingStationsNewOrUpdated;
   bool                       m_showChargingStationsNCS;
};

class MapCarsorSettings
{
public:
   enum MapCarsorType
   {
      MAP_CARSOR_TYPE__DEFAULT = 0,
      MAP_CARSOR_TYPE__CONCEPT_CAR_1,
      MAP_CARSOR_TYPE__CONCEPT_CAR_2,
      MAP_CARSOR_TYPE__CONCEPT_CAR_3,
      MAP_CARSOR_TYPE__CONCEPT_CAR_4,
      MAP_CARSOR_TYPE__CONCEPT_CAR_5,
      MAP_CARSOR_TYPE__CONCEPT_CAR_6,
      MAP_CARSOR_TYPE__SPECIAL        // if this type is set, the HMI will also specify a path to the carsor model
   };

   MapCarsorSettings() :
      m_mapCarsorType(MAP_CARSOR_TYPE__DEFAULT)
   {
   }

   MapCarsorSettings(
      MapCarsorType mapCarsorType)

      : m_mapCarsorType(mapCarsorType)
   {
   }

   bool operator==(const MapCarsorSettings& rhs) const
   {
      return ((m_mapCarsorType == rhs.m_mapCarsorType)
              && (m_specialCarsorPath  == rhs.m_specialCarsorPath));
   }

   bool operator!=(const MapCarsorSettings& rhs) const
   {
      return (!(*this == rhs));
   }

   void setMapCarsorType(MapCarsorType mapCarsorType)
   {
      m_mapCarsorType = mapCarsorType;
   }

   MapCarsorType getMapCarsorType() const
   {
      return m_mapCarsorType;
   }

   // this path is only taken into account for MAP_CARSOR_TYPE__SPECIAL
   void setSpecialCarsorPath(const ::std::string& specialCarsorPath)
   {
      m_specialCarsorPath = specialCarsorPath;
   }

   const ::std::string& getSpecialCarsorPath() const
   {
      return m_specialCarsorPath;
   }

   static ::std::string toString(MapCarsorType mapCarsorType)
   {
      switch (mapCarsorType)
      {
      case MAP_CARSOR_TYPE__DEFAULT:
         return "MAP_CARSOR_TYPE__DEFAULT";
      case MAP_CARSOR_TYPE__CONCEPT_CAR_1:
         return "MAP_CARSOR_TYPE__CONCEPT_CAR_1";
      case MAP_CARSOR_TYPE__CONCEPT_CAR_2:
         return "MAP_CARSOR_TYPE__CONCEPT_CAR_2";
      case MAP_CARSOR_TYPE__CONCEPT_CAR_3:
         return "MAP_CARSOR_TYPE__CONCEPT_CAR_3";
      case MAP_CARSOR_TYPE__CONCEPT_CAR_4:
         return "MAP_CARSOR_TYPE__CONCEPT_CAR_4";
      case MAP_CARSOR_TYPE__CONCEPT_CAR_5:
         return "MAP_CARSOR_TYPE__CONCEPT_CAR_5";
      case MAP_CARSOR_TYPE__CONCEPT_CAR_6:
         return "MAP_CARSOR_TYPE__CONCEPT_CAR_6";
      case MAP_CARSOR_TYPE__SPECIAL:
         return "MAP_CARSOR_TYPE__SPECIAL";
      default:
         ::std::stringstream stream;
         stream << "MAP_CARSOR_TYPE<" << static_cast<unsigned int>(mapCarsorType) << ">";
         return stream.str();
      }
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapCarsorSettings payload:\n");
      stream << ::std::endl
             << "MapCarsorType = " << toString(m_mapCarsorType) << ::std::endl
             << "SpecialCarsorPath = " << m_specialCarsorPath << ::std::endl;
      return stream.str();
   }

private:
   MapCarsorType m_mapCarsorType;
   ::std::string m_specialCarsorPath;
};

}
}

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_MAPSETTINGS_H_
