/**
* @copyright (c) 2015-2021 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_GUIDANCEINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_GUIDANCEINFOS_H_

#include <string>
#include <sstream>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{

enum ManeuverSymbol
{
   MANEUVER_SYMBOL_NO_SYMBOL = 0,
   MANEUVER_SYMBOL_NO_INFO,
   MANEUVER_SYMBOL_FOLLOW,
   MANEUVER_SYMBOL_RECALC,
   MANEUVER_SYMBOL_DEST_REACHED,
   MANEUVER_SYMBOL_DEST_REACHED_LEFT,
   MANEUVER_SYMBOL_DEST_REACHED_RIGHT,
   MANEUVER_SYMBOL_INTERMEDIATEDEST_REACHED,
   MANEUVER_SYMBOL_INTERMEDIATEDEST_REACHED_LEFT,
   MANEUVER_SYMBOL_INTERMEDIATEDEST_REACHED_RIGHT,
   MANEUVER_SYMBOL_EXIT_LEFT,
   MANEUVER_SYMBOL_EXIT_RIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_STRAIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_HALFRIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_RIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_SHARPRIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_REVERSE,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_SHARPLEFT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_LEFT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_LEFT_HALFLEFT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_STRAIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_HALFRIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_RIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_SHARPRIGHT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_REVERSE,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_SHARPLEFT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_LEFT,
   MANEUVER_SYMBOL_ROUNDABOUT_TRAFFICSENSE_RIGHT_HALFLEFT,
   MANEUVER_SYMBOL_STRAIGHT,
   MANEUVER_SYMBOL_TURN_HALFRIGHT,
   MANEUVER_SYMBOL_TURN_RIGHT,
   MANEUVER_SYMBOL_TURN_SHARPRIGHT,
   MANEUVER_SYMBOL_TURN_SHARPLEFT,
   MANEUVER_SYMBOL_TURN_LEFT,
   MANEUVER_SYMBOL_TURN_HALFLEFT,
   MANEUVER_SYMBOL_UTURN_LEFT,
   MANEUVER_SYMBOL_UTURN_RIGHT,
   MANEUVER_SYMBOL_KEEP_LEFT,
   MANEUVER_SYMBOL_KEEP_RIGHT,
   MANEUVER_SYMBOL_SIZE
};

enum RecommendationType
{
   RECOMMENDATION_TYPE_PREVIEW = 0,
   RECOMMENDATION_TYPE_PRELIMINARY,
   RECOMMENDATION_TYPE_REGULAR,
   RECOMMENDATION_TYPE_REPEATED
};

enum ManeuverSymbolType
{
   MANEUVER_SYMBOL = 0,
   DESTINATION_DIRECTION_SYMBOL,
   ROAD_ENTRY_POINT_DIRECTION_SYMBOL
};

enum SamplePhrase
{
   SAMPLE_PHRASE__GUIDANCE_WILL_BE_AT_THIS_VOLUME,
   SAMPLE_PHRASE__LOUDER,
   SAMPLE_PHRASE__QUIETER,
   SAMPLE_PHRASE__VOICE_OUTPUT
};

enum DeformedImageRoadType
{
   DEFORMEDIMAGEROADTYPE__NONE,
   DEFORMEDIMAGEROADTYPE__FORK,
   DEFORMEDIMAGEROADTYPE__EXIT_LEFT,
   DEFORMEDIMAGEROADTYPE__EXIT_RIGHT
};

enum DeformedImageArrowType
{
   DEFORMEDIMAGEARROWTYPE__NONE,
   DEFORMEDIMAGEARROWTYPE__STRAIGHT,
   DEFORMEDIMAGEARROWTYPE__LEFT,
   DEFORMEDIMAGEARROWTYPE__RIGHT
};

enum JunctionViewImageRoadType
{
   JUNCTIONVIEWIMAGEROADTYPE__NONE,
   JUNCTIONVIEWIMAGEROADTYPE__STRAIGHT_RIGHT,
   JUNCTIONVIEWIMAGEROADTYPE__HALF_RIGHT,
   JUNCTIONVIEWIMAGEROADTYPE__RIGHT,
   JUNCTIONVIEWIMAGEROADTYPE__LEFT,
   JUNCTIONVIEWIMAGEROADTYPE__HALF_LEFT,
   JUNCTIONVIEWIMAGEROADTYPE__STRAIGHT_LEFT
};

enum JunctionViewImageArrowType
{
   JUNCTIONVIEWIMAGEARROWTYPE__NONE,
   JUNCTIONVIEWIMAGEARROWTYPE__NORMAL,
   JUNCTIONVIEWIMAGEARROWTYPE__EXIT
};

enum JunctionViewImageType
{
   JUNCTIONVIEWIMAGETYPE__BLOB,
   JUNCTIONVIEWIMAGETYPE__DESCRIPTION
};

enum ManeuverViewType
{
   MANEUVERVIEWTYPE_NONE,
   MANEUVERVIEWTYPE_JUNCTION_VIEW,
   MANEUVERVIEWTYPE_INTERSECTION_MAP
};

inline ::std::string toString(JunctionViewImageRoadType junctionViewImageRoadType)
{
   switch (junctionViewImageRoadType)
   {
   case JUNCTIONVIEWIMAGEROADTYPE__NONE:
      return "JUNCTIONVIEWIMAGEROADTYPE__NONE";
   case JUNCTIONVIEWIMAGEROADTYPE__STRAIGHT_RIGHT:
      return "JUNCTIONVIEWIMAGEROADTYPE__STRAIGHT_RIGHT";
   case JUNCTIONVIEWIMAGEROADTYPE__STRAIGHT_LEFT:
      return "JUNCTIONVIEWIMAGEROADTYPE__STRAIGHT_LEFT";
   case JUNCTIONVIEWIMAGEROADTYPE__HALF_RIGHT:
      return "JUNCTIONVIEWIMAGEROADTYPE__HALF_RIGHT";
   case JUNCTIONVIEWIMAGEROADTYPE__HALF_LEFT:
      return "JUNCTIONVIEWIMAGEROADTYPE__HALF_LEFT";
   case JUNCTIONVIEWIMAGEROADTYPE__RIGHT:
      return "JUNCTIONVIEWIMAGEROADTYPE__RIGHT";
   case JUNCTIONVIEWIMAGEROADTYPE__LEFT:
      return "JUNCTIONVIEWIMAGEROADTYPE__LEFT";
   default:
      ::std::stringstream stream;
      stream << "JUNCTIONVIEWIMAGEROADTYPE__<" << static_cast<unsigned int>(junctionViewImageRoadType) << ">";
      return stream.str();
   }
}

inline ::std::string toString(JunctionViewImageArrowType junctionViewImageArrowType)
{
   switch (junctionViewImageArrowType)
   {
   case JUNCTIONVIEWIMAGEARROWTYPE__NONE:
      return "JUNCTIONVIEWIMAGEARROWTYPE__NONE";
   case JUNCTIONVIEWIMAGEARROWTYPE__NORMAL:
      return "JUNCTIONVIEWIMAGEARROWTYPE__NORMAL";
   case JUNCTIONVIEWIMAGEARROWTYPE__EXIT:
      return "JUNCTIONVIEWIMAGEARROWTYPE__EXIT";
   default:
      ::std::stringstream stream;
      stream << "JUNCTIONVIEWIMAGEARROWTYPE__<" << static_cast<unsigned int>(junctionViewImageArrowType) << ">";
      return stream.str();
   }
}

inline ::std::string toString(DeformedImageRoadType deformedImageRoadType)
{
   switch (deformedImageRoadType)
   {
   case DEFORMEDIMAGEROADTYPE__NONE:
      return "DEFORMEDIMAGEROADTYPE__NONE";
   case DEFORMEDIMAGEROADTYPE__FORK:
      return "DEFORMEDIMAGEROADTYPE__FORK";
   case DEFORMEDIMAGEROADTYPE__EXIT_LEFT:
      return "DEFORMEDIMAGEROADTYPE__EXIT_LEFT";
   case DEFORMEDIMAGEROADTYPE__EXIT_RIGHT:
      return "DEFORMEDIMAGEROADTYPE__EXIT_RIGHT";
   default:
      ::std::stringstream stream;
      stream << "DEFORMEDIMAGEROADTYPE__<" << static_cast<unsigned int>(deformedImageRoadType) << ">";
      return stream.str();
   }
}

inline ::std::string toString(DeformedImageArrowType deformedImageArrowType)
{
   switch (deformedImageArrowType)
   {
   case DEFORMEDIMAGEARROWTYPE__NONE:
      return "DEFORMEDIMAGEARROWTYPE__NONE";
   case DEFORMEDIMAGEARROWTYPE__STRAIGHT:
      return "DEFORMEDIMAGEARROWTYPE__STRAIGHT";
   case DEFORMEDIMAGEARROWTYPE__LEFT:
      return "DEFORMEDIMAGEARROWTYPE__LEFT";
   case DEFORMEDIMAGEARROWTYPE__RIGHT:
      return "DEFORMEDIMAGEARROWTYPE__RIGHT";
   default:
      ::std::stringstream stream;
      stream << "DEFORMEDIMAGEARROWTYPE__<" << static_cast<unsigned int>(deformedImageArrowType) << ">";
      return stream.str();
   }
}

class NextManeuverInfos
{
public:
   NextManeuverInfos()
      :m_maneuverId(0)
      ,m_barGraphValue(0)
      ,m_isBarGraphValid(false)
      ,m_isNewManeuver(true)
      ,m_isCombinedWithNextManeuver(false)
      ,m_distanceUnit(DISTANCEUNIT_KILOMETERS)
      ,m_roadNumberImageData()
      ,m_destinationDirectionSymbol()
      ,m_nextManeuverSymbolInfo()
      ,m_followingManeuverSymbolInfo()
      ,m_turnFromStreetRoadClass(ROAD_CLASS__NONE)
   {}
   bool operator == (const NextManeuverInfos& rhs) const
   {
      return (m_barGraphValue == rhs.m_barGraphValue) &&
             (m_isBarGraphValid == rhs.m_isBarGraphValid) &&
             (m_bargraphScaleValues == rhs.m_bargraphScaleValues);
   }

   bool operator != (const NextManeuverInfos& rhs) const
   {
      return !(*this == rhs);
   }

   class ManeuverSymbolInfo
   {
   public:
      ManeuverSymbolInfo() :
         m_maneuverSymbolType(MANEUVER_SYMBOL),
         m_symbolIndex(MANEUVER_SYMBOL_NO_SYMBOL)
      {}

      ManeuverSymbolInfo(ManeuverSymbol symbolIndex,
                         ::navmiddleware::ValidValue< unsigned int> roundaboutExitNumber)
         : m_maneuverSymbolType(MANEUVER_SYMBOL),
           m_symbolIndex(symbolIndex),
           m_roundaboutExitNumber(roundaboutExitNumber)
      {}

      ManeuverSymbolType m_maneuverSymbolType;
      ManeuverSymbol m_symbolIndex;
      ValidValue<unsigned int> m_roundaboutExitNumber;
   };

   void setManeuverId(uint32_t maneuverId)
   {
      m_maneuverId = maneuverId;
   }

   uint32_t getManeuverId() const
   {
      return m_maneuverId;
   }

   void setBarGraphValue(unsigned int barGraphValue)
   {
      m_barGraphValue = barGraphValue;
   }
   unsigned int getBarGraphValue() const
   {
      return m_barGraphValue;
   }
   void setBarGraphValid(bool barGraphValid)
   {
      m_isBarGraphValid = barGraphValid;
   }
   bool isBarGraphValid() const
   {
      return m_isBarGraphValid;
   }

   void setBargraphMinScale(const ::std::string& bargraphMinScale)
   {
      m_bargraphMinScale = bargraphMinScale;
   }

   const ::std::string& getBargraphMinScale() const
   {
      return m_bargraphMinScale;
   }

   void setBargraphMaxScale(const ::std::string& bargraphMaxScale)
   {
      m_bargraphMaxScale = bargraphMaxScale;
   }

   const ::std::string& getBargraphMaxScale() const
   {
      return m_bargraphMaxScale;
   }

   void setBargraphScaleValues(const ::std::vector < ::std::string>&  bargraphScaleValues)
   {
      m_bargraphScaleValues = bargraphScaleValues;
   }

   const ::std::vector < ::std::string>& getBargraphScaleValues() const
   {
      return m_bargraphScaleValues;
   }

   void setNewManeuver(bool isNewManeuver)
   {
      m_isNewManeuver = isNewManeuver;
   }
   bool isNewManeuver() const
   {
      return m_isNewManeuver;
   }

   void setTurnToStreet(const ::std::string& turnToStreet)
   {
      m_turnToStreets = turnToStreet;
   }
   const ::std::string& getTurnToStreet() const
   {
      return m_turnToStreets;
   }

   void setCombinedWithNextManeuver(bool isCombinedWithNextManeuver)
   {
      m_isCombinedWithNextManeuver = isCombinedWithNextManeuver;
   }
   bool isCombinedWithNextManeuver() const
   {
      return m_isCombinedWithNextManeuver;
   }

   void setDistanceToManeuver(const ::std::string& distanceToManeuver)
   {
      m_distanceToManeuver = distanceToManeuver;
   }
   const ::std::string& getDistanceToManeuver() const
   {
      return m_distanceToManeuver;
   }

   void setDistanceToManeuverNoUnit(const ::std::string& distanceToManeuverNoUnit)
   {
      m_distanceToManeuverNoUnit = distanceToManeuverNoUnit;
   }
   const ::std::string& getDistanceToManeuverNoUnit() const
   {
      return m_distanceToManeuverNoUnit;
   }

   void setDistanceToManeuverUnit(DistanceUnit distanceUnit)
   {
      m_distanceUnit = distanceUnit;
   }
   const DistanceUnit& getDistanceToManeuverUnit() const
   {
      return m_distanceUnit;
   }

   void setRecommendationType(ValidValue<RecommendationType> recommendationType)
   {
      m_recommendationType = recommendationType;
   }
   const ValidValue<RecommendationType>& getRecommendationType() const
   {
      return m_recommendationType;
   }

   void setTurnToRoadNumberImageData(::std::vector< Image> roadNumberImage)
   {
      m_roadNumberImageData = roadNumberImage;
   }

   ::std::vector<Image>& getTurnToRoadNumberImageData()
   {
      return m_roadNumberImageData;
   }
   const ::std::vector<Image>& getTurnToRoadNumberImageData() const
   {
      return m_roadNumberImageData;
   }

   void setNextManeuverSymbolInfo(ManeuverSymbolInfo nextManeueverInfo)
   {
      m_nextManeuverSymbolInfo = nextManeueverInfo;
   }

   ManeuverSymbolInfo& getNextManeuverSymbolInfo()
   {
      return m_nextManeuverSymbolInfo;
   }
   const ManeuverSymbolInfo& getNextManeuverSymbolInfo() const
   {
      return m_nextManeuverSymbolInfo;
   }

   void setDestinationDirection(DirectionDescription destinationDirection)
   {
      m_destinationDirectionSymbol = destinationDirection;
   }
   const DirectionDescription& getDestinationDirection() const
   {
      return m_destinationDirectionSymbol;
   }

   void setRoadEntryPointDirection(DirectionDescription roadEntryPointDirection)
   {
      m_roadEntryPointDirection.setValue(roadEntryPointDirection);
   }
   const ValidValue <DirectionDescription>& getRoadEntryPointDirection() const
   {
      return m_roadEntryPointDirection;
   }
   ValidValue <DirectionDescription>& getRoadEntryPointDirection()
   {
      return m_roadEntryPointDirection;
   }

   void setFollowingManeuverSymbolInfo(ValidValue<ManeuverSymbolInfo> followingManeueverInfo)
   {
      m_followingManeuverSymbolInfo = followingManeueverInfo;
   }

   ValidValue<ManeuverSymbolInfo>& getFollowingManeuverSymbolInfo()
   {
      return m_followingManeuverSymbolInfo;
   }
   const ValidValue<ManeuverSymbolInfo>& getFollowingManeuverSymbolInfo() const
   {
      return m_followingManeuverSymbolInfo;
   }

   void setTurnFromStreetRoadClass(RoadClass roadClass)
   {
      m_turnFromStreetRoadClass = roadClass;
   }
   RoadClass getTurnFromStreetRoadClass() const
   {
      return m_turnFromStreetRoadClass;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("NextManeuverInfos payload:\n");
      stream << ::std::endl
             << "Maneuver ID: " << m_maneuverId << ::std::endl
             << "Bar graph value: " << m_barGraphValue << ::std::endl
             << "is bargraph valid: " << m_isBarGraphValid << ::std::endl
             << "Bargraph min scale: " << m_bargraphMinScale << ::std::endl
             << "Bargraph Max scale: " << m_bargraphMaxScale << ::std::endl
             << "Bargraph scale values: ";
      for(::std::vector< ::std::string>::size_type bargraphScaleIndex = 0;
            bargraphScaleIndex < m_bargraphScaleValues.size();
            bargraphScaleIndex++)
      {
         stream << m_bargraphScaleValues.at(bargraphScaleIndex) << "  ";
      }
      stream << ::std::endl
             << "Turn to street name: " << m_turnToStreets << ::std::endl
             << "Distance to maneuver: " << m_distanceToManeuver << ::std::endl
             << "Recommendation Type: " << m_recommendationType.getValue() << ::std::endl
             << "is combined with next maneuver route: " << m_isCombinedWithNextManeuver << ::std::endl
             << "is new maneuver: " << m_isNewManeuver << ::std::endl
             << "Destination direction symbol: " << m_destinationDirectionSymbol << ::std::endl
             << "Next Maneuver symbol Type: " << m_nextManeuverSymbolInfo.m_maneuverSymbolType << ::std::endl
             << "Next Maneuver symbol Index: " << m_nextManeuverSymbolInfo.m_symbolIndex << ::std::endl
             << "Next Maneuver symbol ExitNumber: " << m_nextManeuverSymbolInfo.m_roundaboutExitNumber.toString() << ::std::endl
             << "Road entry point Direction: " << m_roadEntryPointDirection.toString() << ::std::endl;

      if(m_followingManeuverSymbolInfo.isValid())
      {
         stream << "Following maneuver symbol :" << m_followingManeuverSymbolInfo.getValue().m_symbolIndex << ::std::endl;
      }
      for(unsigned int roadNrImageIndex = 0; roadNrImageIndex < m_roadNumberImageData.size(); roadNrImageIndex++)
      {
         stream << "RoadNumber Image size : " << m_roadNumberImageData.at(roadNrImageIndex).getBlobSize() << ::std::endl;
      }
      stream << "Road class of turn from street : " << navmiddleware::toString(m_turnFromStreetRoadClass) << ::std::endl;
      return stream.str();
   }

private:
   uint32_t m_maneuverId;
   unsigned int m_barGraphValue;
   bool m_isBarGraphValid;
   ::std::string m_bargraphMinScale;
   ::std::string m_bargraphMaxScale;
   /** List containing bargraph min scale, max scale and intermediate scale values. All entries are formatted strings.
    * First entry in the list contains the min scale value.
    * Last entry in the list contains the max scale value.
    * If only the min and max scale should be shown without intermediate scale values,
    * m_bargraphMinScale and m_bargraphMaxScale should be used
    */
   ::std::vector< ::std::string> m_bargraphScaleValues;

   ::std::string m_turnToStreets;
   bool m_isNewManeuver;
   bool m_isCombinedWithNextManeuver;
   ::std::string m_distanceToManeuver;
   ::std::string m_distanceToManeuverNoUnit;
   DistanceUnit m_distanceUnit;
   ValidValue<RecommendationType> m_recommendationType;
   ::std::vector<Image> m_roadNumberImageData;
   ValidValue <DirectionDescription> m_roadEntryPointDirection;
   DirectionDescription m_destinationDirectionSymbol;

   ManeuverSymbolInfo m_nextManeuverSymbolInfo;
   ValidValue <ManeuverSymbolInfo> m_followingManeuverSymbolInfo;
   RoadClass m_turnFromStreetRoadClass;
};

enum LaneStatus
{
   LaneStatus__UNKNOWN,
   LaneStatus__FORBIDDEN,
   LaneStatus__NOT_RECOMMENDED,
   LaneStatus__RECOMMENDED
};

enum LaneDirection
{
   LaneDirection__NONE = 0x0,
   LaneDirection__STRAIGHT =0x1,
   LaneDirection__LEFT =0x2,
   LaneDirection__RIGHT =0x4,
   LaneDirection__HALF_LEFT =0x8,
   LaneDirection__HALF_RIGHT =0x10,
   LaneDirection__SHARP_LEFT =0x20,
   LaneDirection__SHARP_RIGHT =0x40,
   LaneDirection__UTURN_LEFT =0x80,
   LaneDirection__UTURN_RIGHT =0x100
};

class LaneInfos
{
public:
   class LaneInfo
   {
   public:
      LaneInfo():
         laneStatus(LaneStatus__UNKNOWN),
         combinedFollowingLaneDirection(0),
         guidedFollowingLaneDirection(0),
         isHOVLane(0),
         laneOccupancy(1) {}

      bool operator==(const LaneInfo& rhs) const
      {
         if ( &rhs == this )
         {
            return true;
         }
         return (laneStatus == rhs.laneStatus)
            && (combinedFollowingLaneDirection == rhs.combinedFollowingLaneDirection)
            && (guidedFollowingLaneDirection == rhs.guidedFollowingLaneDirection)
            && (isHOVLane == rhs.isHOVLane)
            && (laneOccupancy == rhs.laneOccupancy);
      }

      bool operator!=(const LaneInfo& rhs) const
      {
         return !(rhs == *this);
      }

      LaneStatus laneStatus;
      int combinedFollowingLaneDirection;
      int guidedFollowingLaneDirection;
      bool isHOVLane;
      int laneOccupancy;
   };

   const ::std::vector<LaneInfo>& getLaneInfos() const
   {
      return m_LaneInfos;
   }
   ::std::vector<LaneInfo>& getLaneInfos()
   {
      return m_LaneInfos;
   }

   bool operator==(const LaneInfos& rhs) const
   {
      if ( &rhs == this )
      {
         return true;
      }
      return (m_LaneInfos == rhs.m_LaneInfos);
   }

   bool operator!=(const LaneInfos& rhs) const
   {
      return !(rhs == *this);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("LaneInfos payload:\n");
      stream << ::std::endl
             << "Total number of Lanes " << getLaneInfos().size() << ::std::endl;

      for(unsigned int laneIndex = 0; laneIndex < getLaneInfos().size(); laneIndex++)
      {
         stream << ::std::endl
                << "Current Lane Index: " << laneIndex << ::std::endl
                << "lane status: " << getLaneInfos().at(laneIndex).laneStatus << ::std::endl
                << "lane combined FollowingLane Direction: " << getLaneInfos().at(laneIndex).combinedFollowingLaneDirection << ::std::endl
                << "lane guided FollowingLane Direction: " << getLaneInfos().at(laneIndex).guidedFollowingLaneDirection << ::std::endl
                << "is HOV lane: " << getLaneInfos().at(laneIndex).isHOVLane << ::std::endl
                << "lane Occupancy: " << getLaneInfos().at(laneIndex).laneOccupancy << ::std::endl;
      }

      return stream.str();
   }

private:
   ::std::vector<LaneInfo> m_LaneInfos;
};

enum SignPostPosition
{
   SignPostPosition_RIGHT,
   SignPostPosition_MIDDLE,
   SignPostPosition_LEFT
};

class SignPostInfo
{
public:

   SignPostInfo()
      :m_isSignPostAvailable(false)
      ,m_signPostPosition(SignPostPosition_MIDDLE) {}

   bool operator == (const SignPostInfo& rhs) const
   {
      return (m_isSignPostAvailable == rhs.m_isSignPostAvailable &&
              m_signPostPosition == rhs.m_signPostPosition &&
              m_signpostRoadNumberInfos == rhs.m_signpostRoadNumberInfos &&
              m_textLines == rhs.m_textLines &&
              m_exitNumber == rhs.m_exitNumber &&
              m_backgroundColor == rhs.m_backgroundColor &&
              m_textColor == rhs.m_textColor &&
              m_countryCode == rhs.m_countryCode);
   }

   bool operator != (const SignPostInfo& rhs) const
   {
      return !(*this == rhs);
   }

   const Color& getTextColor() const
   {
      return m_textColor;
   }
   Color& getTextColor()
   {
      return m_textColor;
   }

   const Color& getBackgroundColor() const
   {
      return m_backgroundColor;
   }
   Color& getBackgroundColor()
   {
      return m_backgroundColor;
   }

   const ::std::vector<RoadNumberInfo>& getSignpostRoadNumberInfos() const
   {
      return m_signpostRoadNumberInfos;
   }

   ::std::vector<RoadNumberInfo>& getSignpostRoadNumberInfos()
   {
      return m_signpostRoadNumberInfos;
   }

   const ::std::vector< ::std::string>& getTextLines() const
   {
      return m_textLines;
   }

   ::std::vector< ::std::string>& getTextLines()
   {
      return m_textLines;
   }

   const ::std::string& getExitNumber() const
   {
      return m_exitNumber;
   }

   ::std::string& getExitNumber()
   {
      return m_exitNumber;
   }

   bool isSignPostAvailable() const
   {
      return m_isSignPostAvailable;
   }

   void setSignPost(bool available)
   {
      m_isSignPostAvailable = available;
   }

   void setCountryCode(const ::std::string& countryCode)
   {
      m_countryCode = countryCode;
   }

   const ::std::string& getCountryCode() const
   {
      return m_countryCode;
   }

   void setSignPostPosition(SignPostPosition signPostPosition)
   {
      m_signPostPosition = signPostPosition;
   }

   SignPostPosition getSignPostPosition() const
   {
      return m_signPostPosition;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("SignPostInfo payload:\n");
      stream << ::std::endl
             << "is signpost available: " << m_isSignPostAvailable << ::std::endl
             << "exit number: " << m_exitNumber << ::std::endl
             << "country code: " << m_countryCode << ::std::endl
             << "signpost background color(R,G,B,Alpha): " << static_cast<int>(m_backgroundColor.m_red)<< ","<< static_cast<int>(m_backgroundColor.m_green)
             << ","<< static_cast<int>(m_backgroundColor.m_blue)<<","<< static_cast<int>(m_backgroundColor.m_alpha)<< ::std::endl
             << "signpost text color(R,G,B,Alpha): " << static_cast<int>(m_textColor.m_red)<< ","<< static_cast<int>(m_textColor.m_green)
             << ","<< static_cast<int>(m_textColor.m_blue)<<","<< static_cast<int>(m_textColor.m_alpha)<< ::std::endl
             << "signpost position: " << m_signPostPosition << ::std::endl
             << "total number of Road Number icons " << getSignpostRoadNumberInfos().size() << ::std::endl;

      for(unsigned int textLineIndexIndex = 0; textLineIndexIndex < getTextLines().size(); textLineIndexIndex++)
      {
         stream <<::std::endl<< "text line index" << textLineIndexIndex << ": "
                << getTextLines().at(textLineIndexIndex)<< ::std::endl;
      }

      for(unsigned int roadNumberIndex = 0; roadNumberIndex < getSignpostRoadNumberInfos().size(); roadNumberIndex++)
      {
         stream <<::std::endl<< "road Number info at list index: " << roadNumberIndex << ::std::endl
                << getSignpostRoadNumberInfos().at(roadNumberIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   bool m_isSignPostAvailable;
   ::std::vector<RoadNumberInfo> m_signpostRoadNumberInfos;
   ::std::vector< ::std::string> m_textLines;
   ::std::string m_exitNumber;
   Color m_backgroundColor;
   Color m_textColor;
   ::std::string m_countryCode;
   SignPostPosition m_signPostPosition;
};

class ManeuverViewInfo
{
public:

   class JunctionViewImageDescription
   {
   public:
      JunctionViewImageDescription()
         :m_junctionViewImageRoadType(JUNCTIONVIEWIMAGEROADTYPE__NONE)
         ,m_junctionViewImageArrowType(JUNCTIONVIEWIMAGEARROWTYPE__NONE) {}

      bool operator == (const JunctionViewImageDescription& rhs) const
      {
         return (m_junctionViewImageRoadType == rhs.m_junctionViewImageRoadType)&&
                (m_junctionViewImageArrowType == rhs.m_junctionViewImageArrowType);
      }

      bool operator != (const JunctionViewImageDescription& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("JunctionViewImageDescription info:\n");
         stream   << ::std::endl
            << "Image Road Type = "  << ::navmiddleware::toString(m_junctionViewImageRoadType) << ::std::endl
            << "Image Arrow Type = " << ::navmiddleware::toString(m_junctionViewImageArrowType) << ::std::endl;
         return stream.str();
      }

      JunctionViewImageRoadType  m_junctionViewImageRoadType;
      JunctionViewImageArrowType m_junctionViewImageArrowType;
   };

   class DeformedImageDescription
   {
   public:
      DeformedImageDescription()
         :m_deformedImageRoadType(DEFORMEDIMAGEROADTYPE__NONE)
         ,m_deformedImageArrowType(DEFORMEDIMAGEARROWTYPE__NONE) {}

      bool operator == (const DeformedImageDescription& rhs) const
      {
         return (m_deformedImageRoadType == rhs.m_deformedImageRoadType)&&
             (m_deformedImageArrowType == rhs.m_deformedImageArrowType);
      }

      bool operator != (const DeformedImageDescription& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("DeformedImageDescription info:\n");
         stream   << ::std::endl
            << "Image Road Type = "  << ::navmiddleware::toString(m_deformedImageRoadType) << ::std::endl
            << "Image Arrow Type = " << ::navmiddleware::toString(m_deformedImageArrowType) << ::std::endl;
         return stream.str();
      }

      DeformedImageRoadType m_deformedImageRoadType;
      DeformedImageArrowType m_deformedImageArrowType;
   };

   ManeuverViewInfo()
      :m_maneuverViewType(MANEUVERVIEWTYPE_NONE)
      ,m_junctionViewImageType(JUNCTIONVIEWIMAGETYPE__BLOB)
      ,m_maneuverId(0) {}

   void setBlobImageData(const unsigned char* blobData)
   {
      m_junctionViewImage.setBlob(blobData);
   }
   const unsigned char* getBlobImageData() const
   {
      return m_junctionViewImage.getBlob();
   }
   void setBlobImageDataSize(int blobDataSize)
   {
      m_junctionViewImage.setBlobSize(blobDataSize);
   }
   int getBlobImageDataSize() const
   {
      return m_junctionViewImage.getBlobSize();
   }

   void setDeformedImageDescription(const DeformedImageDescription& deformedImageDescription)
   {
      m_deformedImageDescription = deformedImageDescription;
   }
   const DeformedImageDescription& getDeformedImageDescription() const
   {
      return m_deformedImageDescription;
   }

   void setJunctionViewImageDescription(const ValidValue<JunctionViewImageDescription>& junctionViewImageDescription)
   {
      m_junctionViewImageDescription = junctionViewImageDescription;
   }
   const ValidValue<JunctionViewImageDescription>& getJunctionViewImageDescription() const
   {
      return m_junctionViewImageDescription;
   }

   void setManeuverViewType(ManeuverViewType maneuverViewType)
   {
      m_maneuverViewType = maneuverViewType;
   }
   ManeuverViewType getManeuverViewType() const
   {
      return m_maneuverViewType;
   }

   void setJunctionViewImageType(JunctionViewImageType junctionViewImageType)
   {
      m_junctionViewImageType = junctionViewImageType;
   }
   JunctionViewImageType getJunctionViewImageType() const
   {
      return m_junctionViewImageType;
   }

   void setManeuverId(uint32_t maneuverId)
   {
      m_maneuverId = maneuverId;
   }
   uint32_t getManeuverId() const
   {
      return m_maneuverId;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ManeuverViewInfo payload:\n");
      stream   << ::std::endl
               << "Maneuver view type : "  << m_maneuverViewType << ::std::endl
               << "JunctionView image type: "  << m_junctionViewImageType << ::std::endl
               << "JunctionView image info: " << m_junctionViewImage.toString() << ::std::endl
               << ::navmiddleware::toString(m_junctionViewImageDescription) << ::std::endl
               << m_deformedImageDescription.toString() << ::std::endl
               << "Intersection Map Id : " << m_maneuverId << ::std::endl;
      return stream.str();
   }

private:

   ManeuverViewType m_maneuverViewType;
   /**
    * JunctionViewImageType must be checked to indicate that JunctionViewImageDescription or Image  will be used for displaying JunctionView.
    * If JunctionViewImageType is JUNCTIONVIEWIMAGETYPE__DESCRIPTION, JunctionViewImageDescription will be used for displaying JunctionView
    * If JunctionViewImageType is JUNCTIONVIEWIMAGETYPE__BLOB, Image will be used for displaying JunctionView
   */
   JunctionViewImageType m_junctionViewImageType;
   /** JunctionViewImageDescription is valid when the JunctionViewImageType is set to JUNCTIONVIEWIMAGETYPE__DESCRIPTION */
   ValidValue <JunctionViewImageDescription> m_junctionViewImageDescription;
   DeformedImageDescription m_deformedImageDescription;
   Image m_junctionViewImage;
   uint32_t m_maneuverId;
};

enum SpeedUnit
{
   KM_PER_HOUR,
   MI_PER_HOUR
};

enum SpeedLimitType
{
   SPEED_LIMIT_TYPE__UNKNOWN,
   SPEED_LIMIT_TYPE__IMPLICIT,
   SPEED_LIMIT_TYPE__EXPLICIT,
   SPEED_LIMIT_TYPE__EXPLICIT_BY_NIGHT,
   SPEED_LIMIT_TYPE__EXPLICIT_BY_DAY,
   SPEED_LIMIT_TYPE__EXPLICIT_BY_TIME_OF_DAY,
   SPEED_LIMIT_TYPE__EXPLICIT_ON_RAIN,
   SPEED_LIMIT_TYPE__EXPLICIT_ON_SNOW
};

class SpeedInfo
{
public:
   SpeedInfo():
      m_isSpeedLimitPresent(false),
      m_speedLimit(0),
      m_speedLimitUnit(KM_PER_HOUR),
      m_speedLimitType(SPEED_LIMIT_TYPE__UNKNOWN),
      m_countryCode(),
      m_overSpeedWarning(false),
      m_isUnlimitedSpeed(false) {}

   bool operator == (const SpeedInfo& rhs) const
   {
      return (m_isSpeedLimitPresent == rhs.m_isSpeedLimitPresent &&
              m_speedLimit == rhs.m_speedLimit &&
              m_speedLimitUnit == rhs.m_speedLimitUnit &&
              m_speedLimitInHmiDistanceUnit == rhs.m_speedLimitInHmiDistanceUnit &&
              m_speedLimitType == rhs.m_speedLimitType &&
              m_countryCode == rhs.m_countryCode &&
              m_overSpeedWarning == rhs.m_overSpeedWarning &&
              m_isUnlimitedSpeed == rhs.m_isUnlimitedSpeed);
   }

   bool operator != (const SpeedInfo& rhs) const
   {
      return !(*this == rhs);
   }

   bool isSpeedLimitPresent() const
   {
      return m_isSpeedLimitPresent;
   }
   void setSpeedLimitPresent(bool present)
   {
      m_isSpeedLimitPresent = present;
   }

   void setCountryCode(const ::std::string& countryCode)
   {
      m_countryCode = countryCode;
   }
   const ::std::string& getCountryCode() const
   {
      return m_countryCode;
   }

   void setSpeedLimitUnit(SpeedUnit speedLimit)
   {
      m_speedLimitUnit = speedLimit;
   }
   SpeedUnit getSpeedLimitUnit() const
   {
      return m_speedLimitUnit;
   }

   void setSpeedLimit(unsigned int speed)
   {
      m_speedLimit = speed;
   }
   unsigned int getSpeedLimit() const
   {
      return m_speedLimit;
   }

   void setSpeedLimitInHmiDistanceUnit(const ::std::string& speedLimitInHmiDistanceUnit)
   {
      m_speedLimitInHmiDistanceUnit.setValue(speedLimitInHmiDistanceUnit);
   }
   ValidValue< ::std::string> getSpeedLimitInHmiDistanceUnit() const
   {
      return m_speedLimitInHmiDistanceUnit;
   }
   void invalidateSpeedLimitInHmiDistanceUnit()
   {
      m_speedLimitInHmiDistanceUnit.invalidate();
   }

   void setSpeedLimitType(SpeedLimitType speedLimitType)
   {
      m_speedLimitType = speedLimitType;
   }
   SpeedLimitType getSpeedLimitType() const
   {
      return m_speedLimitType;
   }

   void setOverSpeedWarning(bool overSpeedWarning)
   {
      m_overSpeedWarning = overSpeedWarning;
   }
   bool getOverSpeedWarning() const
   {
      return m_overSpeedWarning;
   }

   void setUnlimitedSpeed(bool isUnlimitedSpeed)
   {
      m_isUnlimitedSpeed = isUnlimitedSpeed;
   }
   bool isUnlimitedSpeed() const
   {
      return m_isUnlimitedSpeed;
   }

   ::std::string toString(SpeedUnit speedUnit) const
   {
      switch ( speedUnit )
      {
         case KM_PER_HOUR:
            return "km/h";
         case MI_PER_HOUR:
            return "mph";
      }
   }

   ::std::string toString(SpeedLimitType speedLimitType) const
   {
      switch (speedLimitType)
      {
         case SPEED_LIMIT_TYPE__UNKNOWN:
            return "SPEED_LIMIT_TYPE__UNKNOWN";
         case SPEED_LIMIT_TYPE__IMPLICIT:
            return "SPEED_LIMIT_TYPE__IMPLICIT";
         case SPEED_LIMIT_TYPE__EXPLICIT:
            return "SPEED_LIMIT_TYPE__EXPLICIT";
         case SPEED_LIMIT_TYPE__EXPLICIT_BY_NIGHT:
            return "SPEED_LIMIT_TYPE__EXPLICIT_BY_NIGHT";
         case SPEED_LIMIT_TYPE__EXPLICIT_BY_DAY:
            return "SPEED_LIMIT_TYPE__EXPLICIT_BY_DAY";
         case SPEED_LIMIT_TYPE__EXPLICIT_BY_TIME_OF_DAY:
            return "SPEED_LIMIT_TYPE__EXPLICIT_BY_TIME_OF_DAY";
         case SPEED_LIMIT_TYPE__EXPLICIT_ON_RAIN:
            return "SPEED_LIMIT_TYPE__EXPLICIT_ON_RAIN";
         case SPEED_LIMIT_TYPE__EXPLICIT_ON_SNOW:
            return "SPEED_LIMIT_TYPE__EXPLICIT_ON_SNOW";
      }
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("SpeedInfo payload:\n");
      stream << ::std::endl
             << "is SpeedLimit Present: " << m_isSpeedLimitPresent << ::std::endl
             << "speedLimit value: " << m_speedLimit << ::std::endl
             << "speedLimit Unit: " << toString(m_speedLimitUnit) << ::std::endl
             << "speedLimit Type:" << toString(m_speedLimitType) << ::std::endl
             << "country Code: " << m_countryCode << ::std::endl
             << "overspeed warning: " << m_overSpeedWarning << ::std::endl
             << "is unmlimited speed: " << m_isUnlimitedSpeed << ::std::endl;

      return stream.str();
   }

private:
   bool m_isSpeedLimitPresent;
   unsigned int m_speedLimit;
   SpeedUnit m_speedLimitUnit;
   ValidValue < ::std::string> m_speedLimitInHmiDistanceUnit;
   SpeedLimitType m_speedLimitType;
   ::std::string m_countryCode;
   bool m_overSpeedWarning;
   bool m_isUnlimitedSpeed;
};

enum TrafficSignId
{
   TRAFFIC_SIGN_ID_NO_OVERTAKING =1,
   TRAFFIC_SIGN_ID_PROTECTED_PASSING
};

class TrafficSignInfo
{
public:
   TrafficSignInfo():
      m_beep(false) {}

   ValidValue< TrafficSignId> getSignId() const
   {
      return m_signId;
   }

   void setSignId( TrafficSignId id)
   {
      m_signId.setValue(id);
   }

   void setSignId( ValidValue< TrafficSignId> id)
   {
      m_signId = id;
   }

   void invalidateSignId( )
   {
      m_signId.invalidate();
   }

   const ::std::string& getDistanceWithUnit() const
   {
      return m_distanceWithUnit;
   }

   void setDistanceWithUnit( const ::std::string& distanceWithUnit)
   {
      m_distanceWithUnit = distanceWithUnit;
   }

   bool getBeepTrigger() const
   {
      return m_beep;
   }

   void setBeepTrigger(bool beep)
   {
      m_beep = beep;
   }

   void setBlobImageData(const unsigned char* blobData)
   {
      m_trafficSignImage.setBlob(blobData);
   }

   const unsigned char* getBlobImageData() const
   {
      return m_trafficSignImage.getBlob() ;
   }

   void setBlobImageDataSize(int blobDataSize)
   {
      m_trafficSignImage.setBlobSize(blobDataSize);
   }

   int getBlobImageDataSize() const
   {
      return m_trafficSignImage.getBlobSize();
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("TrafficSignInfo payload:\n");
      stream << ::std::endl
             << "signId: " << m_signId.toString() << ::std::endl
             << "distanceWithUnit: " << m_distanceWithUnit << ::std::endl
             << "beep: " << m_beep << ::std::endl
             << "image Blob size: " << m_trafficSignImage.getBlobSize() << ::std::endl;

      return stream.str();
   }

private:
   ValidValue< TrafficSignId> m_signId;
   ::std::string m_distanceWithUnit;
   bool m_beep;
   Image m_trafficSignImage;
};

/** the values in this enumeration will be used to indicate presence of more than one event
 * The available events are indicated by the field m_horizonEventTypeMask which is a bitwise 'or' of the corresponding enum values.*/
enum HorizonEventType
{
   HORIZON_EVENT_TYPE__NO_EVENT = 0x00,
   HORIZON_EVENT_TYPE__AVOID_AREA_APPROACH = 0x01,
   HORIZON_EVENT_TYPE__AVOID_AREA_ON_ROUTE = 0x02,
   HORIZON_EVENT_TYPE__ADDRESSBOOK_POI_APPROACH = 0x04,
   HORIZON_EVENT_TYPE__BORDER_CROSSING = 0x08,
   HORIZON_EVENT_TYPE__TOLL_BOOTH_APPROACH = 0x10,
   //<-- INF4CV
   HORIZON_EVENT_TYPE__VEHICLE_RESTRICTIONS_APPROACH = 0x20,
   HORIZON_EVENT_TYPE__USB_POI_APPROACH = 0x40
   //->
};

class HorizonEventInfo
{
public:
   HorizonEventInfo():
      m_horizonEventTypeMask(0)
   {}

   bool operator != (const HorizonEventInfo& rhs) const
   {
      return ((m_horizonEventTypeMask != rhs.m_horizonEventTypeMask) ||
              (m_listOfSoundFilePaths != rhs.m_listOfSoundFilePaths) ||
              (m_approachingAvoidAreaNameList != rhs.m_approachingAvoidAreaNameList) ||
              (m_enteringCountryCode != rhs.m_enteringCountryCode) ||
              (m_enteringSubCountryCode != rhs.m_enteringSubCountryCode) ||
              (m_enteringCountryName != rhs.m_enteringCountryName) ||
              (m_enteringSubCountryName != rhs.m_enteringSubCountryName) ||
              (m_vehicleRestrictionsInfo != rhs.m_vehicleRestrictionsInfo));   //<-- INF4CV
   }

   void setHorizonEventTypeMask(uint16_t horizonEventTypeMask)
   {
      m_horizonEventTypeMask = horizonEventTypeMask;
   }

   /** The present horizon events is a bitwise AND of one or more elements of enum HorizonEventType. */
   uint16_t getHorizonEventTypeMask() const
   {
      return m_horizonEventTypeMask;
   }

   void setListOfSoundFilePaths(::std::vector< ::std::string> listOfSoundFilePaths)
   {
      m_listOfSoundFilePaths = listOfSoundFilePaths;
   }

   ::std::vector< ::std::string> getListOfSoundFilePaths() const
   {
      return m_listOfSoundFilePaths;
   }

   void setApproachingAvoidAreaNameList(const ::std::vector< ::std::string>& approachingAvoidAreaNameList)
   {
      m_approachingAvoidAreaNameList = approachingAvoidAreaNameList;
   }

   const ::std::vector< ::std::string>& getApproachingAvoidAreaNameList() const
   {
      return m_approachingAvoidAreaNameList;
   }

   void setEnteringCountryCode(const ValidValue< ::std::string>& enteringCountryCode)
   {
      m_enteringCountryCode = enteringCountryCode;
   }

   const ValidValue< ::std::string>& getEnteringCountryCode() const
   {
      return m_enteringCountryCode;
   }

   void setEnteringSubCountryCode(const ValidValue< ::std::string>& enteringSubCountryCode)
   {
      m_enteringSubCountryCode = enteringSubCountryCode;
   }

   const ValidValue< ::std::string>& getEnteringSubCountryCode() const
   {
      return m_enteringSubCountryCode;
   }

   void setEnteringCountryName(const ValidValue< ::std::string>& enteringCountryName)
   {
      m_enteringCountryName = enteringCountryName;
   }

   const ValidValue< ::std::string>& getEnteringCountryName() const
   {
      return m_enteringCountryName;
   }

   void setEnteringSubCountryName(const ValidValue< ::std::string>& enteringSubCountryName)
   {
      m_enteringSubCountryName = enteringSubCountryName;
   }

   const ValidValue< ::std::string>& getEnteringSubCountryName() const
   {
      return m_enteringSubCountryName;
   }

   //<-- INF4CV
   void setVehicleRestrictionsInfo(const VehicleRestrictionsInfo& vehicleRestrictionsInfo)
   {
      m_vehicleRestrictionsInfo = vehicleRestrictionsInfo;
   }

   const VehicleRestrictionsInfo& getVehicleRestrictionsInfo() const
   {
      return m_vehicleRestrictionsInfo;
   }
   //->

   ::std::string toString() const
   {
      ::std::stringstream stream("TriggerEventInfo payload:\n");
      stream << ::std::endl
             << "HorizonEvent: ";

      if(HORIZON_EVENT_TYPE__NO_EVENT == m_horizonEventTypeMask)
      {
         stream << "NO_EVENT" << ::std::endl;
      }
      if(HORIZON_EVENT_TYPE__AVOID_AREA_APPROACH & m_horizonEventTypeMask)
      {
         stream << "AVOID_AREA_APROACH" << ::std::endl;
         stream << "List of avoid area names: "<< ::std::endl;
         for(unsigned int avoidAreaNameIndex = 0; avoidAreaNameIndex < getApproachingAvoidAreaNameList().size(); avoidAreaNameIndex++)
         {
            stream <<::std::endl<< getApproachingAvoidAreaNameList().at(avoidAreaNameIndex) << ::std::endl;
         }
      }
      if(HORIZON_EVENT_TYPE__AVOID_AREA_ON_ROUTE & m_horizonEventTypeMask)
      {
         stream << "AVOID_AREA_ON_ROUTE" << ::std::endl;
      }
      if(HORIZON_EVENT_TYPE__ADDRESSBOOK_POI_APPROACH & m_horizonEventTypeMask)
      {
         stream << "ADDRESSBOOK_POI_APPROACH" << ::std::endl;
         stream << "List of SoundFile Paths: "<< ::std::endl;
         for(unsigned int filePathIndex = 0; filePathIndex < getListOfSoundFilePaths().size(); filePathIndex++)
         {
            stream <<::std::endl<< getListOfSoundFilePaths().at(filePathIndex) << ::std::endl;
         }
      }
      if(HORIZON_EVENT_TYPE__BORDER_CROSSING & m_horizonEventTypeMask)
      {
         stream << "BORDER_CROSSING" << ::std::endl;
         stream << "Entering Country code: "<< m_enteringCountryCode.toString() << ::std::endl;
         stream << "Entering Sub country code: "<< m_enteringSubCountryCode.toString() << ::std::endl;
         stream << "Entering Country name: "<< m_enteringCountryName.toString() << ::std::endl;
         stream << "Entering Sub country name: "<< m_enteringSubCountryName.toString() << ::std::endl;
      }
      if(HORIZON_EVENT_TYPE__TOLL_BOOTH_APPROACH & m_horizonEventTypeMask)
      {
         stream << "HORIZON_EVENT_TYPE__TOLL_BOOTH_APPROACH" << ::std::endl;
      }
      //<-- INF4CV
      if (HORIZON_EVENT_TYPE__VEHICLE_RESTRICTIONS_APPROACH & m_horizonEventTypeMask)
      {
         stream << "HORIZON_EVENT_TYPE__VEHICLE_RESTRICTIONS_APPROACH" << ::std::endl;
         stream << m_vehicleRestrictionsInfo.toString() << ::std::endl;
      }
      if(HORIZON_EVENT_TYPE__USB_POI_APPROACH & m_horizonEventTypeMask)
      {
         stream << "HORIZON_EVENT_TYPE__USB_POI_APPROACH" << ::std::endl;
      }
      //->
      return stream.str();
   }

private:
   uint16_t m_horizonEventTypeMask;
   ::std::vector< ::std::string> m_listOfSoundFilePaths;
   ::std::vector< ::std::string> m_approachingAvoidAreaNameList;
   ValidValue < ::std::string> m_enteringCountryCode;
   ValidValue < ::std::string> m_enteringSubCountryCode;
   ValidValue < ::std::string> m_enteringCountryName;
   ValidValue < ::std::string> m_enteringSubCountryName;
   VehicleRestrictionsInfo m_vehicleRestrictionsInfo;   //<-- INF4CV
};

class SpeedCamInfo
{
public:
   SpeedCamInfo()
      :m_speedCamWarning(false)
      ,m_speedCamCount(0)
      ,m_speedCamSpeedlimit(0)
      ,m_speedLimitUnit(KM_PER_HOUR)
      ,m_beep(false)
      ,m_blobData(NULL)
      ,m_blobDataSize(0) {}

   bool operator == (const SpeedCamInfo& rhs) const
   {
      return (m_speedCamWarning == rhs.m_speedCamWarning &&
              m_speedCamCount == rhs.m_speedCamCount &&
              m_distanceWithUnit == rhs.m_distanceWithUnit &&
              m_speedCamSpeedlimit == rhs.m_speedCamSpeedlimit &&
              m_speedLimitUnit == rhs.m_speedLimitUnit &&
              m_beep == rhs.m_beep &&
              m_blobData == rhs.m_blobData &&
              m_blobDataSize == rhs.m_blobDataSize);
   }

   bool operator != (const SpeedCamInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setSpeedCamCount(unsigned int speedCamCount)
   {
      m_speedCamCount = speedCamCount;
   }

   unsigned int getSpeedCamCount() const
   {
      return m_speedCamCount;
   }

   void setSpeedCamDistance(const ::std::string& speedCamDistance)
   {
      m_distanceWithUnit = speedCamDistance;
   }

   const ::std::string& getSpeedCamDistance() const
   {
      return m_distanceWithUnit;
   }

   void setSpeedCamSpeedlimit(unsigned int speedCamSpeedlimit)
   {
      m_speedCamSpeedlimit = speedCamSpeedlimit;
   }

   unsigned int getSpeedCamSpeedlimit() const
   {
      return m_speedCamSpeedlimit;
   }

   void setSpeedLimitUnit(SpeedUnit speedLimit)
   {
      m_speedLimitUnit = speedLimit;
   }

   SpeedUnit getSpeedLimitUnit() const
   {
      return m_speedLimitUnit;
   }

   void setBeep(bool beep)
   {
      m_beep = beep;
   }

   bool getBeep() const
   {
      return m_beep;
   }

   void setBlobData(const unsigned char* blobData)
   {
      m_blobData = blobData;
   }
   const unsigned char* getBlobData() const
   {
      return m_blobData;
   }

   void setBlobDataSize(int blobDataSize)
   {
      m_blobDataSize = blobDataSize;
   }
   int getBlobDataSize() const
   {
      return m_blobDataSize;
   }

   void setSpeedCamWarning(bool SpeedCamPresent)
   {
      m_speedCamWarning = SpeedCamPresent;
   }
   bool getSpeedCamWarning() const
   {
      return m_speedCamWarning;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("SpeedCamInfo payload:\n");
      stream << ::std::endl
             << "speed cam warning: " << m_speedCamWarning << ::std::endl
             << "speed cam count: " << m_speedCamCount << ::std::endl
             << "speed cam distance: " << m_distanceWithUnit << ::std::endl
             << "speed cam speed limit " << m_speedCamSpeedlimit << ::std::endl
             << "speed limit unit: " << m_speedLimitUnit << ::std::endl
             << "beep: " << m_beep << ::std::endl
             << "speed cam icon data: " << static_cast<const void*>(m_blobData) << ::std::endl;

      return stream.str();
   }

private:
   bool m_speedCamWarning;
   unsigned int m_speedCamCount;
   ::std::string m_distanceWithUnit;
   unsigned int m_speedCamSpeedlimit;
   SpeedUnit m_speedLimitUnit;
   bool m_beep;
   const unsigned char* m_blobData;
   int m_blobDataSize;

};

enum TrafficEnforcementAlertState
{
   TRAFFIC_ENFORCEMENT_ALERT_STATE__NO_ALERT,
   TRAFFIC_ENFORCEMENT_ALERT_STATE__PRE_ALERT,
   TRAFFIC_ENFORCEMENT_ALERT_STATE__ALERT
};

enum TrafficEnforcementCameraType
{
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__FIXED_SPEED,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__MOBILE_SPEED,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__REDLIGHT,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__TOLL_ROAD,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__PECCANCY,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__SPECIAL_LANE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__FIXED_SPEED_AND_REDLIGHT,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__BUSLANE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__FAKE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__PARKING_VIOLATION,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__OTHER,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__AVERAGE_SPEED_ZONE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__SPEED_ENFORCEMENT_ZONE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__DANGER_ZONE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__MOBILE_SPEED_HOTSPOT_ZONE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__ACCIDENT_BLACKSPOT_ZONE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__RISK_ZONE,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__RESTRICTED_ACCESS,
   TRAFFIC_ENFORCEMENT_CAMERA_TYPE__UNKNOWN
};

enum TrafficEnforcementDistanceType
{
   TRAFFIC_ENFORCEMENT_DISTANCE_TYPE__TO_ZONE_BEGIN,
   TRAFFIC_ENFORCEMENT_DISTANCE_TYPE__TO_ZONE_END
};

class TrafficEnforcementInfo
{
public:
   TrafficEnforcementInfo()
      :m_trafficEnforcementAlertState(TRAFFIC_ENFORCEMENT_ALERT_STATE__NO_ALERT)
      ,m_trafficEnforcementCameraType(TRAFFIC_ENFORCEMENT_CAMERA_TYPE__UNKNOWN)
      ,m_isHighlighted(false)
      ,m_isStacked(false)
      ,m_beep(false)
      ,m_distanceUnit(DISTANCEUNIT_KILOMETERS)
      ,m_trafficEnforcementDistanceType(TRAFFIC_ENFORCEMENT_DISTANCE_TYPE__TO_ZONE_BEGIN)
      ,m_speedLimit(0)
      ,m_speedLimitUnit(KM_PER_HOUR) {}

   bool operator == (const TrafficEnforcementInfo& rhs) const
   {
      return ((m_trafficEnforcementAlertState == rhs.m_trafficEnforcementAlertState) &&
              (m_trafficEnforcementCameraType == rhs.m_trafficEnforcementCameraType) &&
              (m_isHighlighted == rhs.m_isHighlighted) &&
              (m_isStacked == rhs.m_isStacked) &&
              (m_beep == rhs.m_beep) &&
              (m_distance == rhs.m_distance) &&
              (m_distanceUnit == rhs.m_distanceUnit) &&
              (m_trafficEnforcementDistanceType == rhs.m_trafficEnforcementDistanceType) &&
              (m_speedLimit == rhs.m_speedLimit) &&
              (m_speedLimitUnit == rhs.m_speedLimitUnit));
   }

   bool operator != (const TrafficEnforcementInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setTrafficEnforcementAlertState(TrafficEnforcementAlertState trafficEnforcementAlertState)
   {
      m_trafficEnforcementAlertState = trafficEnforcementAlertState;
   }

   TrafficEnforcementAlertState getTrafficEnforcementAlertState() const
   {
      return m_trafficEnforcementAlertState;
   }

   void setTrafficEnforcementCameraType(TrafficEnforcementCameraType trafficEnforcementCameraType)
   {
      m_trafficEnforcementCameraType = trafficEnforcementCameraType;
   }

   TrafficEnforcementCameraType getTrafficEnforcementCameraType() const
   {
      return m_trafficEnforcementCameraType;
   }

   void setHighlighted(bool isHighlighted)
   {
      m_isHighlighted = isHighlighted;
   }

   bool isHighlighted() const
   {
      return m_isHighlighted;
   }

   void setStacked(bool isStacked)
   {
      m_isStacked = isStacked;
   }

   bool isStacked() const
   {
      return m_isStacked;
   }

   void setBeep(bool beep)
   {
      m_beep = beep;
   }

   bool getBeep() const
   {
      return m_beep;
   }

   void setDistance(const ::std::string& distance)
   {
      m_distance = distance;
   }

   ::std::string getDistance() const
   {
      return m_distance;
   }

   void setDistanceUnit(DistanceUnit distanceUnit)
   {
      m_distanceUnit = distanceUnit;
   }

   DistanceUnit getDistanceUnit() const
   {
      return m_distanceUnit;
   }

   void setTrafficEnforcementDistanceType(TrafficEnforcementDistanceType trafficEnforcementDistanceType)
   {
      m_trafficEnforcementDistanceType = trafficEnforcementDistanceType;
   }

   TrafficEnforcementDistanceType getTrafficEnforcementDistanceType() const
   {
      return m_trafficEnforcementDistanceType;
   }

   void setSpeedLimit(unsigned int speedLimit)
   {
      m_speedLimit = speedLimit;
   }

   unsigned int getSpeedLimit() const
   {
      return m_speedLimit;
   }

   void setSpeedLimitUnit(SpeedUnit speedLimitUnit)
   {
      m_speedLimitUnit = speedLimitUnit;
   }

   SpeedUnit getSpeedLimitUnit() const
   {
      return m_speedLimitUnit;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("TrafficEnforcementInfo payload:\n");
      stream << ::std::endl
             << "traffic enforcement alert state: " << m_trafficEnforcementAlertState << ::std::endl
             << "traffic enforcement camera type " << m_trafficEnforcementCameraType << ::std::endl
             << "is highlighted: " << m_isHighlighted << ::std::endl
             << "is stacked: " << m_isStacked << ::std::endl
             << "beep trigger: " << m_beep << ::std::endl
             << "distance: " << m_distance << ::std::endl
             << "distance unit " << m_distanceUnit << ::std::endl
             << "distance type: " << m_trafficEnforcementDistanceType << ::std::endl
             << "speed limit: " << m_speedLimit << ::std::endl
             << "speed limit unit: " << m_speedLimitUnit << ::std::endl;

      return stream.str();
   }

private:
   TrafficEnforcementAlertState m_trafficEnforcementAlertState;
   TrafficEnforcementCameraType m_trafficEnforcementCameraType;
   bool m_isHighlighted;
   bool m_isStacked;
   bool m_beep;
   ::std::string m_distance;
   DistanceUnit m_distanceUnit; // Unit which is part of Distance string can't be used in Cluster so sent Separately
   TrafficEnforcementDistanceType m_trafficEnforcementDistanceType;
   unsigned int m_speedLimit;
   SpeedUnit m_speedLimitUnit;
};


enum SpeedLimitUsage
{
   SPEEDLIMITUSAGE_SPEEDLIMIT_FROM_EXTERNAL_SOURCE_UNAVAILABLE,
   SPEEDLIMITUSAGE_SPEEDLIMIT_FROM_EXTERNAL_SOURCE,
   SPEEDLIMITUSAGE_SPEEDLIMIT_AND_OVERSPEEDWARNING_FROM_EXTERNAL_SOURCE
};

struct SpeedLimitData
{
   SpeedLimitData(
      SpeedLimitUsage speedLimitUsage,
      unsigned int signPostSpeedLimit,
      ::navmiddleware::SpeedUnit signpostSpeedUnit,
      bool overspeedWarningFromExternalSource,
      bool isUnlimitedSpeed)
      :m_speedLimitUsage(speedLimitUsage),
       m_speedLimitFromExternalSource(signPostSpeedLimit),
       m_speedUnit(signpostSpeedUnit),
       m_overspeedWarningFromExternalSource(overspeedWarningFromExternalSource),
       m_isUnlimitedSpeed(isUnlimitedSpeed) {}
   SpeedLimitData()
      :m_speedLimitUsage(SPEEDLIMITUSAGE_SPEEDLIMIT_FROM_EXTERNAL_SOURCE_UNAVAILABLE),
       m_speedLimitFromExternalSource(0),
       m_speedUnit(::navmiddleware::KM_PER_HOUR),
       m_overspeedWarningFromExternalSource(false),
       m_isUnlimitedSpeed(false) {}

   SpeedLimitUsage m_speedLimitUsage;
   unsigned int m_speedLimitFromExternalSource;
   ::navmiddleware::SpeedUnit m_speedUnit;
   bool m_overspeedWarningFromExternalSource;
   bool m_isUnlimitedSpeed;

   ::std::string toString() const
   {
      ::std::stringstream stream("SpeedLimitData payload:\n");
      stream << ::std::endl
             << "use data from external camera: " << m_speedLimitUsage << ::std::endl
             << "speed limit: " << m_speedLimitFromExternalSource << ::std::endl
             << "speed limit unit: " << m_speedUnit << ::std::endl
             << "overspeed warning: " << m_overspeedWarningFromExternalSource << ::std::endl
             << "is unlimited speed" << m_isUnlimitedSpeed << ::std::endl;
      return stream.str();
   }
};

enum ReachedDestinationType
{
   REACHED_DESTINATION_TYPE__WAYPOINT,
   REACHED_DESTINATION_TYPE__FINAL_DESTINATION
};

class DestinationReachedInfo
{
public:
   DestinationReachedInfo()
      :m_reachedDestinationType(REACHED_DESTINATION_TYPE__FINAL_DESTINATION) {}

   ReachedDestinationType getReachedDestinationType() const
   {
      return m_reachedDestinationType;
   }
   void setReachedDestinationType(const ReachedDestinationType reachedDestinationtype)
   {
      m_reachedDestinationType = reachedDestinationtype;
   }

   static ::std::string toString(ReachedDestinationType reachedDestinationtype)
   {
      switch (reachedDestinationtype)
      {
      case REACHED_DESTINATION_TYPE__WAYPOINT:
         return "REACHED_DESTINATION_TYPE__WAYPOINT";
      case REACHED_DESTINATION_TYPE__FINAL_DESTINATION:
         return "REACHED_DESTINATION_TYPE__FINAL_DESTINATION";
      default:
         return "unsupported reachedDestinationType";
      }
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationReachedInfo payload:\n");
      stream << "reachedDestinationType: " << toString(m_reachedDestinationType) << ::std::endl;
      return stream.str();
   }

private:
   ReachedDestinationType m_reachedDestinationType;
};

enum AlertCategory
{
   ALERT_CATEGORY__UNKNOWN,
   ALERT_CATEGORY__EV_CHARGING_STATION
};
inline ::std::string toString(AlertCategory alertCategory)
{
   switch (alertCategory)
   {
   case ALERT_CATEGORY__UNKNOWN:
      return "ALERT_CATEGORY__UNKNOWN";
   case ALERT_CATEGORY__EV_CHARGING_STATION:
      return "ALERT_CATEGORY__EV_CHARGING_STATION";
   default:
      ::std::stringstream stream;
      stream << "ALERT_CATEGORY__<" << static_cast<unsigned int>(alertCategory) << ">";
      return stream.str();
   }
}

enum PoiAlertType
{
   POI_ALERT_TYPE__UNKNOWN,
   POI_ALERT_TYPE__UNREACHABLE_POI,
   POI_ALERT_TYPE__UNREACHABLE_DESTINATION,
   POI_ALERT_TYPE__LOW_BATTERY
};
inline ::std::string toString(PoiAlertType poiAlertType)
{
   switch (poiAlertType)
   {
   case POI_ALERT_TYPE__UNKNOWN:
      return "POI_ALERT_TYPE__UNKNOWN";
   case POI_ALERT_TYPE__UNREACHABLE_POI:
      return "POI_ALERT_TYPE__UNREACHABLE_POI";
   case POI_ALERT_TYPE__UNREACHABLE_DESTINATION:
      return "POI_ALERT_TYPE__UNREACHABLE_DESTINATION";
   case POI_ALERT_TYPE__LOW_BATTERY:
      return "POI_ALERT_TYPE__LOW_BATTERY";
   default:
      ::std::stringstream stream;
      stream << "POI_ALERT_TYPE__<" << static_cast<unsigned int>(poiAlertType) << ">";
      return stream.str();
   }
}

enum EvAlertState
{
   EV_ALERT_STATE__INACTIVE,
   EV_ALERT_STATE__ACTIVE
};
inline ::std::string toString(EvAlertState evAlertState)
{
   switch (evAlertState)
   {
   case EV_ALERT_STATE__INACTIVE:
      return "EV_ALERT_STATE__INACTIVE";
   case EV_ALERT_STATE__ACTIVE:
      return "EV_ALERT_STATE__ACTIVE";
   default:
      ::std::stringstream stream;
      stream << "EV_ALERT_STATE__<" << static_cast<unsigned int>(evAlertState) << ">";
      return stream.str();
   }
}

enum NumReachablePoi
{
   NUM_REACHABLE_POI__UNKNOWN,
   NUM_REACHABLE_POI__NONE,
   NUM_REACHABLE_POI__FEW,
   NUM_REACHABLE_POI__ENOUGH
};

inline ::std::string toString(NumReachablePoi numReachablePoi)
{
   switch (numReachablePoi)
   {
   case NUM_REACHABLE_POI__UNKNOWN:
      return "NUM_REACHABLE_POI__UNKNOWN";
   case NUM_REACHABLE_POI__NONE:
      return "NUM_REACHABLE_POI__NONE";
   case NUM_REACHABLE_POI__FEW:
      return "NUM_REACHABLE_POI__FEW";
   case NUM_REACHABLE_POI__ENOUGH:
      return "NUM_REACHABLE_POI__ENOUGH";
   default:
      ::std::stringstream stream;
      stream << "NUM_REACHABLE_POI__<" << static_cast<unsigned int>(numReachablePoi) << ">";
      return stream.str();
   }
}

enum PoiAlertUnreachableDestinationType
{
   POI_ALERT_UNREACHABLE_DESTINATION_TYPE__UNKNOWN,
   /**The destination is a POI (typical ePOI for EV-Alerts) and it is the final destination.*/
   POI_ALERT_UNREACHABLE_DESTINATION_TYPE__POI_DESTINATION,
   /**The destination is a POI (typical ePOI for EV-Alerts) and it is is an intermediate destination.*/
   POI_ALERT_UNREACHABLE_DESTINATION_TYPE__POI_WAYPOINT,
   /**The destination is no POI (typical non ePOI for EV-Alerts) and it is the final destination.*/
   POI_ALERT_UNREACHABLE_DESTINATION_TYPE__DESTINATION,
   /**The destination is no POI (typical non ePOI for EV-Alerts) and it is an intermediate destination.*/
   POI_ALERT_UNREACHABLE_DESTINATION_TYPE__WAYPOINT
};

inline ::std::string toString(PoiAlertUnreachableDestinationType  poiAlertUnreachableDestinationType)
{
   switch (poiAlertUnreachableDestinationType)
   {
   case POI_ALERT_UNREACHABLE_DESTINATION_TYPE__UNKNOWN:
      return "POI_ALERT_UNREACHABLE_DESTINATION_TYPE__UNKNOWN";
   case POI_ALERT_UNREACHABLE_DESTINATION_TYPE__POI_DESTINATION:
      return "POI_ALERT_UNREACHABLE_DESTINATION_TYPE__POI_DESTINATION";
   case POI_ALERT_UNREACHABLE_DESTINATION_TYPE__POI_WAYPOINT:
      return "POI_ALERT_UNREACHABLE_DESTINATION_TYPE__POI_WAYPOINT";
   case POI_ALERT_UNREACHABLE_DESTINATION_TYPE__DESTINATION:
      return "UNREACHABLE_DESTINATION_TYPE__DESTINATION";
   case POI_ALERT_UNREACHABLE_DESTINATION_TYPE__WAYPOINT:
      return "POI_ALERT_UNREACHABLE_DESTINATION_TYPE__WAYPOINT";
   default:
      ::std::stringstream stream;
      stream << "POI_ALERT_UNREACHABLE_DESTINATION_TYPE__<" << static_cast<unsigned int>(poiAlertUnreachableDestinationType) << ">";
      return stream.str();
   }
}

class EVAlertInfo
{
public:
   EVAlertInfo()
      :m_alertIdentifier(0)
      ,m_alertCategory(ALERT_CATEGORY__UNKNOWN)
      ,m_alertType(POI_ALERT_TYPE__UNKNOWN)
      ,m_alertState(EV_ALERT_STATE__INACTIVE)
      ,m_numberOfStandardReachablePois(0)
      ,m_numberOfEcoReachablePois(0)
      ,m_alertReminder(0)
      ,m_poiDetour(0)
      ,m_searchAreaType(SEARCH_AREA_TYPE__REACHABLE_AREA)
      ,m_isAlertDue(false)
      ,m_numStandardReachablePoi(NUM_REACHABLE_POI__UNKNOWN)
      ,m_numEcoReachablePoi(NUM_REACHABLE_POI__UNKNOWN)
      ,m_driveMode(DRIVEMODE__NORMAL) {}

   enum SearchAreaType
   {
      SEARCH_AREA_TYPE__REACHABLE_AREA,
      SEARCH_AREA_TYPE__ALONG_ROUTE,
      SEARCH_AREA_TYPE__ALONG_CURRENT_HIGHWAY
   };

   static ::std::string toString(SearchAreaType searchAreaType)
   {
      switch (searchAreaType)
      {
      case SEARCH_AREA_TYPE__REACHABLE_AREA:
         return "SEARCH_AREA_TYPE__REACHABLE_AREA";
      case SEARCH_AREA_TYPE__ALONG_ROUTE:
         return "SEARCH_AREA_TYPE__ALONG_ROUTE";
      case SEARCH_AREA_TYPE__ALONG_CURRENT_HIGHWAY:
         return "SEARCH_AREA_TYPE__ALONG_CURRENT_HIGHWAY";
      default:
         ::std::stringstream stream;
         stream << "SEARCH_AREA_TYPE__<" << static_cast<unsigned int>(searchAreaType) << ">";
         return stream.str();
      }
   }

   bool operator == (const EVAlertInfo& rhs) const
   {
      return (m_alertIdentifier == rhs.m_alertIdentifier &&
              m_alertCategory == rhs.m_alertCategory &&
              m_alertType == rhs.m_alertType &&
              m_alertState == rhs.m_alertState &&
              m_numberOfStandardReachablePois == rhs.m_numberOfStandardReachablePois &&
              m_numberOfEcoReachablePois == rhs.m_numberOfEcoReachablePois &&
              m_alertReminder == rhs.m_alertReminder &&
              m_poiDetour == rhs.m_poiDetour &&
              m_searchAreaType == rhs.m_searchAreaType &&
              m_isAlertDue == rhs.m_isAlertDue &&
              m_numStandardReachablePoi == rhs.m_numStandardReachablePoi &&
              m_numEcoReachablePoi == rhs.m_numEcoReachablePoi &&
              m_driveMode == rhs.m_driveMode &&
              m_poiAlertUnreachableDestinationType == rhs.m_poiAlertUnreachableDestinationType);
   }

   bool operator != (const EVAlertInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setAlertIdentifier(unsigned int alertIdentifier)
   {
      m_alertIdentifier = alertIdentifier;
   }

   unsigned int getAlertIdentifier() const
   {
      return m_alertIdentifier;
   }

   void setAlertCategory(AlertCategory alertCategory)
   {
      m_alertCategory = alertCategory;
   }

   AlertCategory getAlertCategory() const
   {
      return m_alertCategory;
   }

   void setAlertType(PoiAlertType alertType)
   {
      m_alertType = alertType;
   }

   PoiAlertType getAlertType() const
   {
      return m_alertType;
   }

   void setEvAlertState(const EvAlertState alertState)
   {
      m_alertState = alertState;
   }

   EvAlertState getEvAlertState() const
   {
      return m_alertState ;
   }

   void setNumberOfStandardReachablePois(unsigned int numberOfStandardReachablePois)
   {
      m_numberOfStandardReachablePois = numberOfStandardReachablePois;
   }

   unsigned int getNumberOfStandardReachablePois() const
   {
      return m_numberOfStandardReachablePois;
   }

   void setNumberOfEcoReachablePois(unsigned int numberOfEcoReachablePois)
   {
      m_numberOfEcoReachablePois = numberOfEcoReachablePois;
   }

   unsigned int getNumberOfEcoReachablePois() const
   {
      return m_numberOfEcoReachablePois;
   }

   void setAlertReminder(unsigned int alertReminder)
   {
      m_alertReminder = alertReminder ;
   }

   unsigned int getAlertReminder() const
   {
      return m_alertReminder;
   }

   void setPoiDetour(unsigned int poiDetour)
   {
      m_poiDetour = poiDetour;
   }

   unsigned int getPoiDetour() const
   {
      return m_poiDetour;
   }

   void setSearchAreaType(SearchAreaType searchAreaType)
   {
      m_searchAreaType = searchAreaType;
   }

   SearchAreaType getSearchAreaType() const
   {
      return m_searchAreaType;
   }

   void setAlertDue(bool isAlertDue)
   {
      m_isAlertDue = isAlertDue;
   }

   bool isAlertDue() const
   {
      return m_isAlertDue;
   }

   void setNumStandardReachablePOI(NumReachablePoi numStandardReachablePoi)
   {
      m_numStandardReachablePoi = numStandardReachablePoi;
   }

   NumReachablePoi getNumStandardReachablePOI() const
   {
      return m_numStandardReachablePoi;
   }

   void setNumEcoReachablePOI(NumReachablePoi numEcoReachablePoi)
   {
      m_numEcoReachablePoi = numEcoReachablePoi;
   }

   NumReachablePoi getNumEcoReachablePOI() const
   {
      return m_numEcoReachablePoi;
   }

   void setDriveMode(DriveMode driveMode)
   {
      m_driveMode = driveMode;
   }

   DriveMode getDriveMode() const
   {
      return m_driveMode;
   }

   void setPoiAlertUnreachableDestinationType(const ::navmiddleware::ValidValue<PoiAlertUnreachableDestinationType>& poiAlertUnreachableDestinationType)
   {
      m_poiAlertUnreachableDestinationType = poiAlertUnreachableDestinationType;
   }

   const ::navmiddleware::ValidValue<PoiAlertUnreachableDestinationType>& getPoiAlertUnreachableDestinationType() const
   {
      return m_poiAlertUnreachableDestinationType;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("EVAlertInfo payload:\n");
      stream << ::std::endl
             << "alert identifier" << m_alertIdentifier << ::std::endl
             << "alert category: " << ::navmiddleware::toString(m_alertCategory) << ::std::endl
             << "alert type: " << ::navmiddleware::toString(m_alertType) << ::std::endl
             << "alert state: " << ::navmiddleware::toString(m_alertState) << ::std::endl
             << "number of standard reachable POIs: " << m_numberOfStandardReachablePois << ::std::endl
             << "number of eco reachable POIs: " << m_numberOfEcoReachablePois << ::std::endl
             << "alert reminder: " << m_alertReminder << ::std::endl
             << "poi detour: " << m_poiDetour << ::std::endl
             << "search area type: " << toString(m_searchAreaType) << ::std::endl
             << "alert due: " << (m_isAlertDue ? "true" : "false") << ::std::endl
             << "num standard reachable POI: " << ::navmiddleware::toString(m_numStandardReachablePoi) << ::std::endl
             << "num eco reachable POI: " << ::navmiddleware::toString(m_numEcoReachablePoi) << ::std::endl
             << "drive mode: " << ::navmiddleware::toString(m_driveMode) << ::std::endl
             << "poi alert unreachable destination type: " << (m_poiAlertUnreachableDestinationType.isValid()?
                   ::navmiddleware::toString(m_poiAlertUnreachableDestinationType.getValue()): "None") << ::std::endl;
      return stream.str();
   }

private:
   /** Unique identifier for POI Alert*/
   unsigned int m_alertIdentifier;
   AlertCategory m_alertCategory;
   PoiAlertType m_alertType;
   EvAlertState m_alertState;
   /** (@DEPRECATED, use m_numStandardReachablePoi instead) indicates the number of high(standard) reachable pois*/
   unsigned int m_numberOfStandardReachablePois;
   /** (@DEPRECATED, use m_numEcoReachablePoi instead) indicates the number of low(Eco) reachable pois*/
   unsigned int m_numberOfEcoReachablePois;
   /** An event counter that is increased on important events that should remind the driver of the alert
       even though he might have refused adding POIs to his route earlier*/
   unsigned int m_alertReminder;
   /** An event counter that is increased if the driver has added a new POI waypoint for detour,
       which is considered for alert situation from now on*/
   unsigned int m_poiDetour;
   /** indicates type of area in which reachable POIs were counted*/
   SearchAreaType m_searchAreaType;
   /** indicates if alert pop-up is due*/
   bool m_isAlertDue;
   /** Number of standard-reachable POI from current position as an abstract enum value*/
   NumReachablePoi m_numStandardReachablePoi;
   /** Number of eco-reachable POI from current position as an abstract enum value*/
   NumReachablePoi m_numEcoReachablePoi;
   /** Indicates the status of the drive mode switch*/
   DriveMode m_driveMode;
   /** Indicates unreachable destination type.
       Only necessary for UnreachableDestinationAlert*/
   ::navmiddleware::ValidValue<PoiAlertUnreachableDestinationType> m_poiAlertUnreachableDestinationType;
};

class PredictedDestinationInfo
{
public:
   PredictedDestinationInfo()
   {

   }

   bool operator == (const PredictedDestinationInfo& rhs) const
   {
      return (m_destinationMemoryEntryId == rhs.m_destinationMemoryEntryId &&
              m_locationAttributes == rhs.m_locationAttributes);
   }

   bool operator != (const PredictedDestinationInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setDestinationMemoryEntryId(const ::navmiddleware::ValidValue<uint64_t>& destinationMemoryEntryId)
   {
      m_destinationMemoryEntryId = destinationMemoryEntryId;
   }

   const ::navmiddleware::ValidValue<uint64_t>& getDestinationMemoryEntryId() const
   {
      return m_destinationMemoryEntryId;
   }

   void setLocationAttributes(const LocationAttributeInfos& locationAttributes)
   {
      m_locationAttributes = locationAttributes;
   }

   const LocationAttributeInfos& getLocationAttributes() const
   {
      return m_locationAttributes;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PredictedDestinationInfo payload:\n");
      stream << ::std::endl;
      if(m_destinationMemoryEntryId.isValid())
      {
         stream << "m_destinationMemoryEntryId : " << m_destinationMemoryEntryId.getValue() << ::std::endl;
      }
      else
      {
         stream << "m_destinationMemoryEntryId is not set"<< ::std::endl;
      }

      stream << "m_locationAttributes:  " <<  m_locationAttributes.toString() << ::std::endl;
      return stream.str();
   }

private:
   ::navmiddleware::ValidValue<uint64_t> m_destinationMemoryEntryId;
   LocationAttributeInfos m_locationAttributes;
};

class NearestChargingStationInfo
{
public:
   NearestChargingStationInfo()
      :m_chargingSpeed(EV_CHARGING_SPEED__UNKNOWN)
      ,m_connectorType(EV_CONNECTOR_TYPE__UNSPECIFIED) {}


   bool operator == (const NearestChargingStationInfo& rhs) const
   {
      return (m_name == rhs.m_name) &&
             (m_airDistance == rhs.m_airDistance) &&
             (m_chargingSpeed == rhs.m_chargingSpeed) &&
             (m_connectorType == rhs.m_connectorType);
   }

   bool operator != (const NearestChargingStationInfo& rhs) const
   {
      return !(*this == rhs);
   }

   const ::std::string& getName() const
   {
      return m_name;
   }

   void setName(const ::std::string& name)
   {
      m_name = name;
   }

   const ::std::string& getAirDistance() const
   {
      return m_airDistance;
   }

   void setAirDistance(const ::std::string& airDistance)
   {
      m_airDistance = airDistance;
   }

   EvChargingSpeed getChargingSpeed() const
   {
      return m_chargingSpeed;
   }

   void setChargingSpeed(EvChargingSpeed chargingSpeed)
   {
      m_chargingSpeed = chargingSpeed;
   }

   EvConnectorType getConnectorType() const
   {
      return m_connectorType;
   }

   void setConnectorType(EvConnectorType connectorType)
   {
      m_connectorType = connectorType;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("NearestChargingStationInfo payload:\n");
      stream << "charging station name: " << m_name << ::std::endl
             << "air distance: " << m_airDistance << ::std::endl
             << "charging speed: " << ::navmiddleware::toString(m_chargingSpeed) << ::std::endl
             << "connector type: " << ::navmiddleware::toString(m_connectorType) << ::std::endl;
      return stream.str();
   }

private:
   ::std::string m_name;
   ::std::string m_airDistance;
   EvChargingSpeed m_chargingSpeed;
   EvConnectorType m_connectorType;
};

enum EcoNavSwitchState
{
   ECONAV_SWITCH_STATE__ON,
   ECONAV_SWITCH_STATE__OFF
};

inline ::std::string toString(EcoNavSwitchState ecoNavSwitchState)
{
   switch (ecoNavSwitchState)
   {
   case ECONAV_SWITCH_STATE__ON:
      return "ECONAV_SWITCH_STATE__ON";
   case ECONAV_SWITCH_STATE__OFF:
      return "ECONAV_SWITCH_STATE__OFF";
   default:
      ::std::stringstream stream;
      stream << "ECONAV_SWITCH_STATE__<" << static_cast<unsigned int>(ecoNavSwitchState) << ">";
      return stream.str();
   }
}

enum EcoNavState
{
   ECONAV_STATE__NOT_ACTIVATED,
   ECONAV_STATE__AWAITING,
   ECONAV_STATE__ON_COMPUTING,
   ECONAV_STATE__ON_REGULATION,
   ECONAV_STATE__ON_UPDATING,
   ECONAV_STATE__COMPUTATION_FAILED,
   ECONAV_STATE__RECHARGE_ON_DRIVING,
   ECONAV_STATE__RECHARGE_ON_STATION,
   ECONAV_STATE__NO_ZEV_ZONE_FUNCTION
};


inline ::std::string toString(EcoNavState ecoNavState)
{
   switch (ecoNavState)
   {
   case ECONAV_STATE__NOT_ACTIVATED:
      return "ECONAV_STATE__NOT_ACTIVATED";
   case ECONAV_STATE__AWAITING:
      return "ECONAV_STATE__AWAITING";
   case ECONAV_STATE__ON_COMPUTING:
      return "ECONAV_STATE__ON_COMPUTING";
   case ECONAV_STATE__ON_REGULATION:
      return "ECONAV_STATE__ON_REGULATION";
   case ECONAV_STATE__ON_UPDATING:
      return "ECONAV_STATE__ON_UPDATING";
   case ECONAV_STATE__COMPUTATION_FAILED:
      return "ECONAV_STATE__COMPUTATION_FAILED";
   case ECONAV_STATE__RECHARGE_ON_DRIVING:
      return "ECONAV_STATE__RECHARGE_ON_DRIVING";
   case ECONAV_STATE__RECHARGE_ON_STATION:
      return "ECONAV_STATE__RECHARGE_ON_STATION";
   case ECONAV_STATE__NO_ZEV_ZONE_FUNCTION:
      return "ECONAV_STATE__NO_ZEV_ZONE_FUNCTION";
   default:
      ::std::stringstream stream;
      stream << "ECONAV_STATE__<" << static_cast<unsigned int>(ecoNavState) << ">";
      return stream.str();
   }
}

enum UrbanZevDrivingOption
{
   URBAN_ZEV_DRIVING_OPTION__OFF,
   URBAN_ZEV_DRIVING_OPTION__ON
};

inline ::std::string toString(UrbanZevDrivingOption urbanZevDrivingOption)
{
   switch (urbanZevDrivingOption)
   {
   case URBAN_ZEV_DRIVING_OPTION__OFF:
      return "URBAN_ZEV_DRIVING_OPTION__OFF";
   case URBAN_ZEV_DRIVING_OPTION__ON:
      return "URBAN_ZEV_DRIVING_OPTION__ON";
   default:
      ::std::stringstream stream;
      stream << "URBAN_ZEV_DRIVING_OPTION__<" << static_cast<unsigned int>(urbanZevDrivingOption) << ">";
      return stream.str();
   }
}

enum VehicleRestrictionsStatus
{
   VEHICLE_RESTRICTIONS_STATUS__UNKNOWN,
   VEHICLE_RESTRICTIONS_STATUS__VALID,
   VEHICLE_RESTRICTIONS_STATUS__INVALID
};

inline ::std::string toString(VehicleRestrictionsStatus vehicleRestrictionsStatus)
{
   switch (vehicleRestrictionsStatus)
   {
   case VEHICLE_RESTRICTIONS_STATUS__UNKNOWN:
      return "VEHICLE_RESTRICTIONS_STATUS__UNKNOWN";
   case VEHICLE_RESTRICTIONS_STATUS__VALID:
      return "VEHICLE_RESTRICTIONS_STATUS__VALID";
   case VEHICLE_RESTRICTIONS_STATUS__INVALID:
      return "VEHICLE_RESTRICTIONS_STATUS__INVALID";
   default:
      ::std::stringstream stream;
      stream << "VEHICLE_RESTRICTIONS_STATUS__<" << static_cast<unsigned int>(vehicleRestrictionsStatus) << ">";
      return stream.str();
   }
}

}

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_GUIDANCEINFOS_H_
