/**
  * @copyright (c) 2015-2021 Robert Bosch Car Multimedia GmbH
 * @brief Adapts the AIVI Presentation Controller to the old Phone HMI interface
 * @addtogroup aivi_presentation_controller
 * @{
 */

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_AIVI_PRES_CTRL_ADAPTER_AIVIPRESCTRLINTERFACE_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_AIVI_PRES_CTRL_ADAPTER_AIVIPRESCTRLINTERFACE_H_

#include <boost/noncopyable.hpp>
#include <boost/shared_ptr.hpp>
#include <set>
#include <string>
#include <utility>
#include <vector>
#include <list>

#if defined(_MSC_VER)
#define AIVI_HMI_BASE_EXPORT __declspec(dllexport)
#else
#define AIVI_HMI_BASE_EXPORT
#endif

#ifdef _MSC_VER
#include <windows.h>
#else
#include <unistd.h>
#define Sleep(x) usleep((x)*1000)
#endif

#ifdef ASF_LOCK_EXTERNAL_THREAD
#include <asf/core/ProcessMessageHookIF.h>
#include <boost/thread/recursive_mutex.hpp>
#endif

#include "LocationInfos.h"
#include "MapInfos.h"
#include "MapPickingResults.h"
#include "NavDataUpdateInfos.h"
#include "RouteInfos.h"
#include "TrafficMessageInfos.h"
#include "DestinationMemoryInfos.h" // ::navmiddleware::DestinationMemoryEntry::Id
#include "GuidanceInfos.h"
#include "InfoTypes.h"
#include "ConfigurationInfos.h" // ::navmiddleware::ConfigurationInfos::UserProfile
#include "SettingTypes.h"
#include "PositionInfos.h"
#include "SdsLocationInfos.h"
#include "WeatherInfos.h"
#include "ElectricVehicleInfos.h"
#include "DiagnosticInfos.h"
#include "NavDataUpdateSettings.h"
#include "ConnectedServicesInfos.h"

namespace aivi_pres_ctrl
{
namespace controls
{
class AIVIPresCtrl;
}
}

namespace navmiddleware
{
class PropertyUpdateCallback;
class EngineeringInfos;
class EngineeringDebugInfos;
class PickingResultFilter;

namespace settings
{
class ChangeMapViewSettings;
class EvAutoSaveChargingStationSettings;
class EvBatteryStatusWarningSettings;
class EvMiscellaneousSettings;
class EvChargingSpeedFilterSettings;
class EvChargingStationFilterSettings;
class EvConnectorTypeFilterSettings;
class GuidanceSettings;
class LocationBasedServicesSettings;
class MapDetailTriggerSettings;
class MapSettings;
class MapTextFilterSettings;
class MapTrackingDotsSettings;
class MapViewSettings;
class MapWeatherSettings;
class NavDataUpdateFavoriteRegionSettings;
class ParkingSettings;
class PaymentMethodSettings;
class SafetyCameraSettings;
class SafetyGuidanceSettings;
class TrafficSettings;
}
}

namespace aivi_pres_ctrl
{
namespace adapter
{
class AIVIPresCtrlInterface : private boost::noncopyable
{
public:
   AIVIPresCtrlInterface();
   ~AIVIPresCtrlInterface();

   // ================= application state handling ======
   bool initialize(const ::navmiddleware::ConfigurationInfos& configurationInfos);
   void initializeASFApplication();
   void stop();
   void stopASFApplication();
   void restoreFactorySettings(::navmiddleware::settings::RestoreType restoreType,
                               ::navmiddleware::settings::UserProfileGroup userProfileGroup);
   void restoreDefaultSettings(::navmiddleware::settings::SettingsGroup settingsGroup);
   void applySettings();

   void activateLimitedMode(const ::std::set< ::navmiddleware::RequiredFeatures>& requiredFeatures);
   void deactivateLimitedMode();

   // ================= property update callback handling =======================================
   void registerLifeCyclePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterLifeCyclePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);

   void registerPositionPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterPositionPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerGuidancePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterGuidancePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerLocationPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterLocationPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerMapPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterMapPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerRoutePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterRoutePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerElectricVehiclePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterElectricVehiclePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerTrafficMessagePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterTrafficMessagePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerEngineeringMenuPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterEngineeringMenuPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerDestinationMemoryPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterDestinationMemoryPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerNavDataUpdatePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterNavDataUpdatePropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerConfigurationPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterConfigurationPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterIconCacheControlPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerIconCacheControlPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void registerLocationBasedServicesPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);
   void unregisterLocationBasedServicesPropertyUpdateCallback(::navmiddleware::PropertyUpdateCallback& callback);

   // ================= configuration =============
   const ::navmiddleware::ConfigurationInfos& getConfigurationInfos();
   void setLanguage(const ::std::string& languageCodeHMIAsString, ::navmiddleware::LanguageCode languageCodeHMI,
                    const ::std::string& languageCodeSDSAsString, ::navmiddleware::LanguageCode languageCodeSDS,
                    const ::std::string& languageCodeTrafficTextAsString, ::navmiddleware::LanguageCode trafficTextLanguageCode,
                    const ::std::string& languageCodeClusterAsString, ::navmiddleware::LanguageCode languageCodeCluster);
   void setDisplayTimeOffset(int32_t displayTimeOffset,
                             ::navmiddleware::ConfigurationInfos::TimeZoneAndDSTConsideration timeZoneAndDSTConsideration);
   void setDistanceUnitSystem(::navmiddleware::DistanceUnitSystem unitSystem);
   void setDateFormat(::navmiddleware::DateFormat dateFormat);
   void setTimeFormat(::navmiddleware::TimeFormat timeFormat);
   void setDefaultPosition(const ::navmiddleware::GeoCoordinateDegree& coordinate, float defaultPositionHeading);
   void setUserProfile(::navmiddleware::ConfigurationInfos::UserProfile userProfile);
   void copyUserProfileToUser(::navmiddleware::ConfigurationInfos::UserProfile targetUserProfile);
   void deleteUserProfile(::navmiddleware::ConfigurationInfos::UserProfile userProfile);
   const ::navmiddleware::UserProfileActionInfos& getUserProfileActionInfos() const;
   void setSystemPowerMode(::navmiddleware::ConfigurationInfos::SystemPowerMode systemPowerMode);
   void setSystemState(::navmiddleware::ConfigurationInfos::SystemState systemState);
   void setTCUStatus(::navmiddleware::ConfigurationInfos::TCUStatus tcuStatus);
   void setPrivacyMode(::navmiddleware::ConfigurationInfos::PrivacyMode privacyMode);

   // ================= map related methods =============

   /**
    * Zooms in stepwise based on the zoomType and step count.
    * Based on the MapSettings, HMI should call this interface with appropriate zoomType
    * If zoomType is ZOOMTYPE_ON_NOMINALSCALE_VALUES, then map is zoomed on fixed steps(from nominal scale property).
    * If zoomType is ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES, then map is zoomed on smaller points of
    * fixed steps (from nominal scale property).
    * Note: After the zooming is performed on fixed steps(from nominal scale property), a timer is activated internally and
    * after the timer is expired, Map will be auto zoomed to a fixed scale(from nominal scale property) if the auto zoom
    * setting is switched to ON.
    * @param[in] mapViewId         - view id{primary or secondary} of the map on which zooming in will be performed.
    * @param[in] stepCount         - stepCount determines the number of zoom in steps to be performed in nominal scales.
    *                                Applicable only on ZOOMTYPE_ON_NOMINALSCALE_VALUES.
    * @param[in] zoomType          - ZOOMTYPE_ON_NOMINALSCALE_VALUES means map is zoomed on fixed steps(from nominal scale property).
    *                                ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES means map is zoomed on smaller points of
    *                                fixed steps(from nominal scale property). i.e. 20% zoom in from the current map scale.
    * @param[in] cameraAnimation   - type of animation used for the zooming in operation.
    * \async ::MAP_CAMERAINFOS_CHANGED.
    */
   void zoomInStep(::navmiddleware::MapViewId mapViewId,
                   uint32_t stepCount,
                   ::navmiddleware::ZoomType zoomType,
                   ::navmiddleware::CameraAnimation cameraAnimation) const;

   /**
    * Zooms out stepwise based on the zoomType and step count.
    * Based on the MapSettings, HMI should call this interface with appropriate zoomType
    * If zoomType is ZOOMTYPE_ON_NOMINALSCALE_VALUES, then map is zoomed on fixed steps(from nominal scale property).
    * If zoomType is ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES, then map is zoomed on smaller points of
    * fixed steps (from nominal scale property).
    * Note: After the zooming is performed on fixed steps(from nominal scale property), a timer is activated internally and
    * after the timer is expired, Map will be auto zoomed to a fixed scale(from nominal scale property) if the auto zoom
    * setting is switched to ON.
    * @param[in] mapViewId         - view id{primary or secondary} of the map on which zooming in will be performed.
    * @param[in] stepCount         - stepCount determines the number of zoom in steps to be performed in nominal scales.
    *                                Applicable only on ZOOMTYPE_ON_NOMINALSCALE_VALUES.
    * @param[in] zoomType          - ZOOMTYPE_ON_NOMINALSCALE_VALUES means map is zoomed on fixed steps(from nominal scale property).
    *                                ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES means map is zoomed on smaller points of
    *                                fixed steps(from nominal scale property). i.e. 20% zoom out from the current map scale.
    * @param[in] cameraAnimation   - type of animation used for the zooming in operation.
    * \async ::MAP_CAMERAINFOS_CHANGED.
    */
   void zoomOutStep(::navmiddleware::MapViewId mapViewId,
                    uint32_t stepCount,
                    ::navmiddleware::ZoomType zoomType,
                    ::navmiddleware::CameraAnimation cameraAnimation) const;

   /**
    * @brief Toggle the Map Representation to 2D North from any other Map Representation.
    * @param[in] mapViewId         - view id {primary or secondary}
    * @param[in] cameraAnimation   - Animation type that is used to set the representation
    *
    */
   void setMapRepresentationMode2DNorth(::navmiddleware::MapViewId mapViewId,
                                        ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief Toggle the Map Representation to 2D HeadsUp from any other Map Representation.
    * @param[in] mapViewId         - view id {primary or secondary}
    * @param[in] cameraAnimation   - Animation type that is used to set the representation
    */
   void setMapRepresentationMode2DCarHeading(::navmiddleware::MapViewId mapViewId,
         ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief Toggle the Map Representation to 3D HeadsUp from any other Map Representation.
    */
   void setMapRepresentationMode3DCarHeading(::navmiddleware::MapViewId mapViewId,
         ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief Toggle the map representation between 2DNorth-2DCarHeading-3DCarHeading
    * @param[in] mapViewId          - view id {primary or secondary}
    * @param[in] mapRepresentation  - The representation type that is requested to be set
    * @param[in] cameraAnimation    - Animation type that is used to set the representation
    */
   void setMapRepresentation(::navmiddleware::MapViewId mapViewId,
                             const ::navmiddleware::MapRepresentation& mapRepresentation, ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief Locks the carsor as target
    * @param[in] mapViewId          - view id {primary or secondary}
    */
   void switchToLinkedMode(::navmiddleware::MapViewId mapViewId,
                           ::navmiddleware::CameraAnimation cameraAnimation);

   /**
   * @brief Bring the map into the Camera mode "FREE"
   * Changes the camera mode and optionally restores the camera setup (position, map scale, pitch, heading)
   * that was active when last exiting the free mode.
   *
   * @param[in] mapViewId           - view id primary/secondary in the map
   * @param[in] cameraConfiguration - indicates whether to use the current camera configuration or to
   *                                  restore the last free mode camera configuration
   */
   void switchToFreeMode(::navmiddleware::MapViewId mapViewId, ::navmiddleware::CameraConfiguration cameraConfiguration);

   /**
    * @brief Freeze the Map View
    * The client can also specify the visibility of the view.
    * @param[in] mapViewId          - view id {primary or secondary}
    * @param[in] mapViewVisibility  - the visibility of the map view {can be visible or hidden}
    */
   void freezeMap(::navmiddleware::MapViewId mapViewId, ::navmiddleware::MapViewVisibility mapViewVisibility);

   /* Sets map to freeze mode
    * The client can also specify the visibility of the view.
    * @param[in] mapViewId           - view id of the map {primary or secondary}
    * @param[in] renderingStatus     - Should be Freeze/Freeze_Keep_Data to inform whether to clear the internal data or not,
    *                                  in case map freezes for a short duration.
    * @param[in] duration            - Specifies the freeze duration in milliseconds.
    *                                  Rendering will be active after the time out.
    *                                  The Value 0 means no timeout (Map is in freeze state till unfreeze() is called)
    * @param[in] mapViewVisibility   - the visibility of the map view {can be visible or hidden}
    * \async ::MAP_VIEWSTATUS_CHANGED
    */
   void freezeMap(::navmiddleware::MapViewId mapViewId,
                  ::navmiddleware::MapViewStatusInfos::DataRenderingStatus renderingStatus,
                  uint16_t durationInMilliSeconds,
                  ::navmiddleware::MapViewVisibility mapViewVisibility);

   /**
    * @brief unFreeze the Map View
    * @param[in] mapViewId          - view id {primary or secondary}
    */
   void unFreezeMap(::navmiddleware::MapViewId mapViewId);

   /**
    * @brief Getter to check if Map is frozen
    * @param[in] mapViewId          - view id {primary or secondary}
    */
   bool isMapFrozen(::navmiddleware::MapViewId mapViewId) const ;

   /**
    * @brief pick the element from Map with the provided screen coordinates
    * @param[in] mapViewId             - view id {primary or secondary}
    * @param[in] x1                    - x Position on screen
    * @param[in] y1                    - y Position on screen
    * @param[in] pickingResultFilter   - If valid, the picked elements will be further filtered as per the specified filter.
    * @return ValidValue<RequestId>    - Unique Id to identify and map the request to response
    */
   ::navmiddleware::ValidValue< ::navmiddleware::RequestId> pickInMap(::navmiddleware::MapViewId mapViewId, unsigned int posX,unsigned int posY,
         const ::navmiddleware::ValidValue< ::navmiddleware::PickingResultFilter>& pickingResultFilter);

   /**
    * @brief Getter for the Picking Results
    */
   const ::navmiddleware::PickingResults& getPickingResults() const;

   /**
    * @brief starts a map picking from the visible picking results or POI search results at the given pixel position.
    * This method can be used for the following use-cases:
    * - To pick indexed POIs from POI search results or
    * - To pick results from picking a multi POI, in this case HMI needs to call pickInMap() before!
    * @param[in] mapViewId    - view id {example: primary or secondary}
    * @param[in] posX         - x pixel position on the screen
    * @param[in] posY         - y pixel position on the screen
    * \async   ::MAP_PICKED_INDEX_FROM_POI_ICONS_CHANGED
    * @pre  If this method is used for picking a multi POI, it expects a previous pickInMap() call and can be used in use case
    *       where user can pick a POI from the multi POI picking results.
    * Note: Besides picking a mutli POI, this method should be used for picking indexed POI from the POI search result
    *       and not for picking normal POIs as there won't be any indexed returned for normal POIs from map-engine.
    */
   void requestPickedIndexFromPoiIcons(::navmiddleware::MapViewId mapViewId, unsigned int posX, unsigned int posY);

   /**
    * @brief   returns index of the POI icon that was picked from the map view using requestPickedIndexFromPoiIcons().
    * @return  - a valid pickedIndexFromPoiIcons     if the same picking element type exists in the result vector
    *                                                of previous pickInMap() call or in POI search results.
    *          - invalid pickedIndexFromPoiIcons     if the picking element type does not exist in the result vector
    *                                                of previous pickInMap() call or in POI search results.
    */
   const ::navmiddleware::PickedIndexFromPoiIcons& getPickedIndexFromPoiIcons() const;

   /**
    * @brief fetches details of elements, passed as list of arguments. Calls are made to different components to render information.
    * @param[in] mapViewId              - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    * @param[in] elementsListForDetails - list of objects holding info of elements whose details need to be fetched.
    */
   void requestPickingDetails(::navmiddleware::MapViewId mapViewId, const ::std::vector< ::navmiddleware::PickingResults::PickingResult >& elementsListForDetails);

   /**
    * @brief stops fetching of position details for the picked elements, and stops processing update of the distance
    * and direction info of picked elements.
    * @param[in] mapViewId - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    */
   void releasePickingDetails(::navmiddleware::MapViewId mapViewId) const;

   /**
   * @brief showPickedPOIsInMap method will show all the POIs from the picked result list.
   * For a better overview all other POI's will be switched off in this camera mode(MAP_MODE_LOCATION_OVERVIEW).
   * MapScale must be set if range parameter is set to 1.
   * Only in this combination the map scale has any effect to show a single POI in 2D-north-up
   * if range = 0 all the POI Icons on the map will be deleted
   * If range > the map engine will calculate the map scale automatically and the given map scale will be ignored.
   * To display list-item indexes with the POI-Icon, the Enum ListItemIndexDisplay (LIST_ITEM_INDEX_DISPLAY__ON) can be used.
   * @param[in] mapViewId         - view id of the map{primary or secondary}
   * @param[in] visibleListInfo   - specifies the startOffset of the list as well as the index of the
   *                                icon to be highlighted and the range of elements which will be displayed
   *                                with the listItemIndexDisplay as LIST_ITEM_INDEX_DISPLAY__ON.
   * @param[in] mapScale          - if only one element is to be shown, mapScale will be used, otherwise map scale is
   *                                determined automatically by map-engine and the value passed will have no effect.
   * @param[in] cameraAnimation   - Type of camera animation to be used.
   * Note : startOffset and range set in the visibleListInfo determines those POI icons that needs to be displayed in PRIORITY style.
   *        The POI icons which doesn't fall under the visibleListInfo range will displayed in NORMAL style.
   */
   void showPickedPOIsInMap(::navmiddleware::MapViewId mapViewId,
                            const ::navmiddleware::VisibleListInfo& pickedPOIsVisibilityListInfo,
                            const ::navmiddleware::ValidValue<int32_t>& mapScale,
                            ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief request all user poi icons asynchronously. See getUserPoiIcons() method below.
    * \async 'MAP_USER_POI_ICONS_AVAILABLE' will be sent if icons are available.
    */
   void requestUserPoiIcons();

   /**
    * @brief getter for the requested user poi icons after method 'requestUserPoiIcons' was called.
    *        Should be used after property update MAP_USER_POI_ICONS_AVAILABLE was sent.
    *        Returns a list of structures holding icon data, icon type and user icon id.
    */
   const ::std::vector< ::navmiddleware::Image >& getUserPoiIcons();

   /**
    * @brief highlights the list of picked map element provided as argument.
    * If empty list is provided, then elements previously highlighted are un-highlighted
    * @param[in] mapViewId         - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    * @param[in] pickingResultList - list of objects holding details of elements to be highlighted
    */
   void highlightPickingResult(::navmiddleware::MapViewId mapViewId,
                               const ::std::vector< ::navmiddleware::PickingResults::PickingResult >& pickingResultList);

   /**
    * @brief Start scrolling the MAP
    * @param[in] mapViewId           - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    * @param[in] scrollDirection     - One of the 8 scroll directions
    * @param[in] scrollSpeed         - Scroll speed for slow and fast scrolling
    * @param[in] maxScrollInPixels   - number of pixels to be scrolled in scrollDirection. If the MaxPixelDelta value is zero,
    *                                  the scroll would not be limited.
    */
   void startScrollingMap(::navmiddleware::MapViewId mapViewId,
                          ::navmiddleware::ScrollDirection scrollDirection,
                          ::navmiddleware::ScrollSpeed scrollSpeed,
                          uint16_t maxScrollInPixels);

   /**
    * @brief Stop scrolling the MAP
    * @param[in] mapViewId - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    */
   void stopScrollingMap(::navmiddleware::MapViewId mapViewId);

   /*
    * @brief Rotates the map around the up-axis.
    * If maxRotationAngle is given as 0 with clockwise/counter-clockwise, the rotation is continuous.
    * If the Rotation type is BOUNDED, rotates by the degree provided by maxRotationAngle
    * and stops automatically after rotation of maxRotationAngle is achieved.
    * If the rotation type is BOUNDED, then it means map rotates to a maximum of maxRotationAngle and
    * If the maxRotationAngle is negative, map rotates counter-clockwise to a maximum of maxRotationAngle.
    * If the maxRotationAngle is positive, map rotates clockwise to a maximum of maxRotationAngle.
    * If the rotation type is Clockwise or CounterClockwise then maxRotationAngle should be 0.
    * @param[in] mapViewId          - If mapViewId is MAP_VIEW_ID__PRIMARY, then rotation is performed on Primary view.
    *                                 If mapViewId is MAP_VIEW_ID__SECONDARY, then rotation is performed on Secondary view.
    * @param[in] rotationType       - Rotation is clockwise,counter-clockwise or bounded
    * @param[in] rotationSpeed      - Speed at which the camera will rotate Range-> 1 to 9 where
    *                                 ROTATION_SPEED_1 is slowest speed and
    *                                 ROTATION_SPEED_9 is fastest speed
    * @param[in] maxRotationAngle   - Degree of rotation(applied when RotationType is BOUNDED) Range-> -360 to +360
    */
   void startRotatingMap(::navmiddleware::MapViewId mapViewId, ::navmiddleware::RotationType rotationType,
                         ::navmiddleware::RotationSpeed rotationSpeed, int32_t maxRotationAngle);
   /**
    * Stop Rotating the MAP
    * @param[in] mapViewId - If mapViewId is MAP_VIEW_ID__PRIMARY, then rotation on Primary view is stopped.
    *                        If mapViewId is MAP_VIEW_ID__SECONDARY, then rotation on Secondary view is stopped.
    */
   void stopRotatingMap(::navmiddleware::MapViewId mapViewId);

   /**
    * @brief Rotates the camera of the given map view around the axis specified. Rotation
    * can be clockwise and counter-clockwise. The passed in angle will always undergo a modulus by 360 operation.
    *
    * @param[in] mapViewId        - view id of Map.
    * @param[in] coordinateAxis   - axis of rotation
    * @param[in] rotationType     - rotation type. ROTATION_TYPE_BOUNDED is not supported for this method
    * @param[in] angle            - angle of rotation with respect to the current camera heading
    *
    */
   void rotateCamera(::navmiddleware::MapViewId mapViewId, ::navmiddleware::CoordinateAxis coordinateAxis,
                     ::navmiddleware::RotationType rotationType, uint32_t angle);

   /**
    * @brief Getter to check if Camera is linked to Carsor
    * @param[in] mapViewId - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    */
   bool isInLinkedMode(::navmiddleware::MapViewId mapViewId) const;

   /**
    * @brief Getter for Map Camera Infos
    */
   const ::navmiddleware::MapCameraInfos& getMapCameraInfos() const;

   /**
    * @brief Getter for Map Representation Infos
    */
   const ::navmiddleware::MapRepresentationInfo& getMapRepresentationInfo() const;

   /**
    * Returns the map scale range for the current map screen.
    * If the current Map scale is outside the range of Map scale supplied by
    * this interface, then HMI can explicitly set the map scale to the nearest map
    * scale range value.
    * Update after callback ::MAP_RECOMMENDED_SCALE_RANGE_CHANGED
    */
   const ::navmiddleware::MapScaleRange& getMapScaleRange() const;

   /**
    * @brief Getter to obtain the result of a capture operation.
    * Update after property update "MAP_CAPTURE_RESULT_AVAILABLE"
    */
   const ::navmiddleware::MapCaptureResultInfo& getMapCaptureResultInfo() const;

   /*
    * @brief Getter for data loading status and data rendering status
    * */
   const ::navmiddleware::MapViewStatusInfos& getViewStatusInfos() const;

   void updateMapSettings(const ::navmiddleware::settings::MapSettings&);

   const ::navmiddleware::settings::MapSettings& getMapSettings() const;

   void updateMapTextFilterSettings(const ::navmiddleware::settings::MapTextFilterSettings&);

   const ::navmiddleware::settings::MapTextFilterSettings& getMapTextFilterSettings() const;

   void updateMapDynamicIconSettings(const ::navmiddleware::settings::MapDynamicIconSettings& mapDynamicIconSettings);

   const ::navmiddleware::settings::MapDynamicIconSettings& getMapDynamicIconSettings() const;

   void resetMapDynamicIconSettings();

   void updateMapCarsorSettings(const ::navmiddleware::settings::MapCarsorSettings& mapCarsorSettings);

   const ::navmiddleware::settings::MapCarsorSettings& getMapCarsorSettings() const;

   void updateMapDetailTriggerSettings(const ::navmiddleware::settings::MapDetailTriggerSettings& );

   const ::navmiddleware::settings::MapDetailTriggerSettings& getMapDetailTriggerSettings() const;

   void updateChangeMapViewSettings(const ::navmiddleware::settings::ChangeMapViewSettings&);

   const ::navmiddleware::settings::ChangeMapViewSettings& getChangeMapViewSettings() const;

   void updateMapViewSettings(const ::navmiddleware::settings::MapViewSettings&);

   const ::navmiddleware::settings::MapViewSettings& getMapViewSettings() const;

   /* @brief - This settings enables the tracking dots feature and
    * @stores the positions with a fix distance on the driven roads
    */
   void updateMapWeatherSettings(const ::navmiddleware::settings::MapWeatherSettings&);
   const navmiddleware::settings::MapWeatherSettings& getMapWeatherSettings() const;

   void updateMapEvSettings(const ::navmiddleware::settings::MapEvSettings&);
   const ::navmiddleware::settings::MapEvSettings& getMapEvSettings() const;

   void updateMapTrackingDotsSettings(const ::navmiddleware::settings::MapTrackingDotsSettings&);
   const navmiddleware::settings::MapTrackingDotsSettings& getMapTrackingDotsSettings() const;
   /**
   * @brief Deletes all tracking dots till current position
   */
   void deleteAllTrackingDots();

   /**
    * @brief Getter for Map camera mode Infos
    */
   const ::navmiddleware::MapCameraModeInfo& getMapCameraModeInfo() const;

   /** Set camera mode and view size for one or more views.
    *  If an addressed view does not exist it will be created.
    *  An existing view that is not addressed will be destroyed.
    * \async ::MAP_*_CAMERAMODE_CHANGED after the corresponding feature call has been issued.
    *
    * @param[in] mapViewGroup          - The view group that the given viewModeConfiguration is assigned to.
    * @param[in] viewModeConfiguration - The configuration information for the addressed views.
    */
   void setMapCameraAndViewModes(::navmiddleware::MapViewGroup mapViewGroup, const ::std::vector< ::navmiddleware::ViewModeConfiguration >& viewModeConfiguration);

   /**
    * @brief Show/Hide the alternatives for the specified route segment
    * @param[in] mapViewId             - view id of the map (PRIMARY or SECONDARY) for which alternatives are required
    * @param[in] showAlternatives      - indicates number of alternatives to show
    * @param[in] moreRouteTypesValue   - indicates which type of more routes (multiple or spatial alternatives) shall be used
    * @param[in] destinationPreference - indicates which route segment to display
    * @param[in] showPlannedAndGuidedRoute - indicates the display of planned and guided routes on map.
    *                                        By default the current active route is shown.
    *                                        If set to true both planned and guided route will be shown on map.
    *                                        In case the value is true and if planned route is not available,
    *                                        then guided route will be shown in the overview.
    */
   void showRouteOverview(::navmiddleware::MapViewId mapViewId,
                          ::navmiddleware::ShowAlternatives showAlternatives,
                          ::navmiddleware::settings::MoreRoutesTypeValue moreRouteTypesValue,
                          ::navmiddleware::DestinationPreference destinationPreference,
                          bool showPlannedAndGuidedRoute = false);

   /**
      * @brief Show an overview for the editing/creating of a route.
      * If a route has already been created, a route overview (2D, north-up) will be shown that
      * may automatically change during the editing of the route.
      * If no route exists when this method is called, the map view will show a carsor locked
      * map in the given mapscale in 2D, north-up.
      *
      * @param[in] mapViewId              - view id {primary or secondary} of the map
      * @param[in] mapScale               - The mapscale will be ignored if a route exists
      *                                     and a route overview will be shown
      * @param[in] cameraAnimation        - Type of camera animation to be used.
      * @param[in] routeVisibilityOptions - Visibility of map elements route/flags/carsor in RouteOverview
      *
      * Update after callback ::MAP_CAMERAMODE_CHANGED
      */
   void showEditRouteOverviewInMap(::navmiddleware::MapViewId mapViewId,
                                   const ::navmiddleware::ValidValue<int32_t>& mapScale,
                                   ::navmiddleware::CameraAnimation cameraAnimation,
                                   const ::navmiddleware::ValidValue< ::navmiddleware::RouteVisibilityOptions>& routeVisibilityOptions);

   /**
   * @brief Highlight route corresponding to the provided routeAlternativeIndex.
    * @param[in] mapViewId               -The map view id (PRIMARY or SECONDARY) to which this request is sent.
   * @param[in] routeAlternativeIndex    - Index corresponding to the alternative route to be highlighted.
   */
   void highlightRoute(::navmiddleware::MapViewId mapViewId, unsigned int routeAlternativeIndex);

   /**
    * @brief Getter for Touch event Infos , returns touch coordinate and tapped info
    */
   const ::navmiddleware::TouchEventInfo& getTouchEventInfo() const;

   /**
   * @brief Makes the highlighted route the selected one
   * @param[in] mapViewId     - view id of the map (PRIMARY or SECONDARY)
   */
   void selectHighlightedRoute(::navmiddleware::MapViewId mapViewId);

   /** Select the route that was active when the route overview was entered.
    * @param[in] mapViewId               - view id {primary or secondary} of Map.
    */
   void selectActiveRoute(::navmiddleware::MapViewId mapViewId);

   /**
   * @brief show weather overview in map for renault.
   * @param[in] mapViewId           - View id of the map (PRIMARY or SECONDARY)
   * @param[in] cameraAnimation     - Type of camera animation to be used.
   */
   void showWeatherOverviewMap(::navmiddleware::MapViewId mapViewId, ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief Getter for NominalScale Info , returns list of map scale and auto pitch
    */
   const ::navmiddleware::NominalScaleInfos& getNominalScaleInfos() const;

   /**
   * @brief Center the map around the specified geo-position
   * @param[in] mapViewId       - view id of the map (PRIMARY or SECONDARY)
   * @param[in] geoCoordinate   - geographical coordinate
   * @param[in] mapScale        - required mapScale value. If this parameter is not set or provided,
   *                              the previous scale value will be retained.
   *                              The valid range is as per the NominalScaleInfo and any value outside
   *                              the range would lead to an error.
   * @param[in] cameraAnimation - required animation for camera
   */
   void centerMapToGeoPosition(::navmiddleware::MapViewId mapViewId,
                               const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                               ::navmiddleware::ValidValue<int32_t> mapScale,
                               ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    *  Center the map around the location
    *  Pre-requisite to showSetLocationInMap() call is that one *MUST* use the setLocationWithXY() interface to
    *  select the location which is going to be shown on map. Usage is similar to startGuidance().
    *
    *  @param[in] MapViewId         - mapViewId(PRIMARY or SECONDARY) for which Camera has to be centered for already set location
    *  @param[in] MapScale          - required mapScale value
    *  @param[in] cameraAnimation   - required animation for camera
    */
   void showSetLocationInMap(::navmiddleware::MapViewId mapViewId,int32_t mapScale,::navmiddleware::CameraAnimation cameraAnimation);

   /** Center the map around the specified view-coordinate
   *
   * @param MapViewId       - mapViewId(PRIMARY or SECONDARY) for which mode is to be set
   * @param posX            - relative x coordinate of the screen based on either PRIMARY or SECONDARY view
   * @param posY            - relative y coordinate of the screen based on either PRIMARY or SECONDARY view
   * @param cameraAnimation - required animation for camera
   */
   void centerMapToViewCoordinate(::navmiddleware::MapViewId mapViewId,
                                  uint16_t posX, uint16_t posY,
                                  ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief The map center point of the current map view needs to move to [posX, posY] values with the given camera animation.
    * These screen positions are relative to the map center position which is [0, 0]
    * @param[in] mapViewId        - view id of the map{primary or secondary}
    * @param[in] posX             - x Position on screen
    * @param[in] posY             - y Position on screen
    * @param[in] cameraAnimation  - Type of camera animation to be used to show avoid area overview.
    */
   void moveMapCenterPoint(::navmiddleware::MapViewId mapViewId,
                           int16_t posX, int16_t posY,
                           ::navmiddleware::CameraAnimation cameraAnimation);

   /**
     * @brief show geographical weather overview in map corresponding to the provided mapViewId.
     * @param[in] mapViewId     - view id of the map(PRIMARY or SECONDARY)
     */
   void showWeatherInMap(::navmiddleware::MapViewId mapViewId);

   /**
   * @brief Configure Light Mode give you the option to switch map between DAY and NIGHT mode.
   * @param[in] LightMode     - light mode type (Automatic or Permanent) with (Day or Night) option.
   */
   void configureLightMode(::navmiddleware::ConfigurationInfos::LightMode lightMode);

   /**
   * @brief Configure the type of touch event that should be handled by the current map view.
   * It can also be used to switch off handling of all touch event by map.
   * @param[in] mapViewId        - view id of the map (PRIMARY or SECONDARY)
   * @param[in] touchActions     - indicates the kind of actions to be triggered by the touch event
   * @param[in] pitchRange       - minimum and maximum value of pitch to restrict map's camera movement
   * @param[in] mapScaleRange    - minimum and maximum value of map scale to restrict map's scale
   */
   void configureTouch(::navmiddleware::MapViewId mapViewId, ::navmiddleware::TouchActions touchActions, ::navmiddleware::PitchRange pitchRange, ::navmiddleware::MapScaleRange mapScaleRange);

   /**
   * @brief Show locations in map.
   * @param[in] mapViewId                      - view id of the map (PRIMARY or SECONDARY)
   * @param[in] geoCoordinates                 - list of geographical coordinates
                                                 These will be taken into account for the overview map scale and -
                                                 depending on the mapIconInfo parameter, for the display of numbered icons.
   * @param[in] optionalOverviewGeoCoordinates - list of geographical coordinates
                                                 These will be taken into account for the overview, but not for the
                                                 display of numbered icons
   * @param[in] mapIconInfo                    - specifies the startOffset of the list as well as the index of the
   *                                             icon to be highlighted. The user can also specify if they want to
   *                                             overview only the coordinates or they also want to show the numbered POIs.
   * @param[in] mapScale                       - this value is considered only when a single GeoCoordinateDegree is given.
   *                                             When a list of GeocoordinateDegrees are passed, the map scale will be automatically
   *                                             decided by map-engine and the value passed will have no effect.
   * @param[in] cameraAnimation                - Type of camera animation to be used.
   */
   void showLocationsInMap(::navmiddleware::MapViewId mapViewId, const std::vector< ::navmiddleware::GeoCoordinateDegree >& geoCoordinates,
                           const ::std::vector< ::navmiddleware::GeoCoordinateDegree >& optionalOverviewGeoCoordinates,
                           const ::navmiddleware::MapIconInfo& mapIconInfo, const navmiddleware::ValidValue<int32_t>& mapScale,
                           ::navmiddleware::CameraAnimation cameraAnimation);

   /**
   * @brief Show locations overview in map with the requested Icons displayed.
   * @param[in] mapViewId         - view id of the map (PRIMARY or SECONDARY)
   * @param[in] VisibleListInfo   - specifies the startOffset of the list as well as the index of the
   *                                icon to be highlighted and the range of elements which will be displayed.
   * @param[in] locationInputType - current Location input type which is active.
   * @param[in] mapScale          - if only one element is to be shown, mapScale will be used, otherwise map scale is
   *                                determined automatically by map-engine and the value passed will have no effect.
   * @param[in] cameraAnimation   - Type of camera animation to be used.
   */
   void showLocationsInMap(::navmiddleware::MapViewId mapViewId, const ::navmiddleware::VisibleListInfo& visibleListInfo,
                           ::navmiddleware::LocationInputType locationInputType,
                           const navmiddleware::ValidValue<int32_t>& mapScale,
                           ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief Store a capture/screenshot of the given map view to the specified file.
    */
   void storeMapCapture(::navmiddleware::MapViewId mapViewId, const ::std::string& path);

   /**
    * Returns the MapMultiViewInfo
    * Update after callback ::MAP_MULTIVIEWMODE_CHANGED
    */
   const ::navmiddleware::MapMultiViewInfo& getMapMultiViewInfo() const;

   /**
   * @brief Detailed information for Intersection Map.
   * @This mode is used for showing an intersection at a guidance manoeuvre.
   * @param[in] mapViewId      - view id of the map (PRIMARY or SECONDARY)
   */
   void showIntersectionMap(::navmiddleware::MapViewId mapViewId,
                            ::navmiddleware::CameraAnimation cameraAnimation,
                            ::navmiddleware::IntersectionMapMode mode);

   /**
    * @brief Show Geo Fence in Map
    * @param[in] mapViewId          - view id of the map (PRIMARY or SECONDARY)
    * @param[in] geoFence           - Has two GeoCoordinates, radius and geoFenceType where
    *                                 GeoFence will be drawn as a circle(using the radius that is provided) if geoCoordinate2 is empty.
    *                                 If geoCoordinate2 is not empty, then a rectangular geo fence will be drawn.
    *                                 GeoFenceType_Inclusion indicates if the car is leaving the fence and
    *                                 GeoFenceType_Exclusion indicates if the car is entering the fence.
    */
   void showGeoFenceInMap(::navmiddleware::MapViewId mapViewId, const ::navmiddleware::GeoFence& geoFence);

   /**
    * @brief Shows a selected route segment in map, when user selects a turnlist element
    * @param[in] mapViewId          - view id of the map (primary or secondary)
    * @param[in] indexInRouteList   - element index of route list for which the overview shall be shown
    * @param[in] cameraAnimation    - type of camera animation used.
    */
   void showRouteSegmentSelectionInMap(::navmiddleware::MapViewId mapViewId, const ::navmiddleware::ValidValue<uint32_t>& indexInRouteList,
                                       ::navmiddleware::CameraAnimation cameraAnimation);

   /**
   * @brief show geographical weather alerts in map corresponding to the provided mapViewId.
   * zoom-level : 500km
   * orientation : 2D North-up
   * map location : current carsor position in map
   *
   * @param[in] mapViewId     - view id of the map (PRIMARY or SECONDARY)
   */
   void showWeatherAlertsInMap(::navmiddleware::MapViewId mapViewId);

   /**
   * @brief Display the map view as a widget (like a button, for example)
   *
   * @param[in] mapViewId     - view id of the map (PRIMARY or SECONDARY)
   */
   void showMapWidget(::navmiddleware::MapViewId mapViewId);

   /**
   * @brief If there is a traffic incident on the route or if the user wants to block a section on the route.
   * Show route overview from current route(with incident) and one alternative(detour) route.
   * The TrafficRouteOverview mode with COMPLETE will show the whole route as a "detour all" feature or
   * with DETOUR, only a section where the route spreads due to a traffic incident on the route will be shown
   * and user can come back to the original route
   * @param[in] mapViewId                 - view id of the map{primary or secondary}
   * @param[in] trafficRouteOverviewMode  - TRAFFIC_ROUTE_OVERVIEW_MODE__COMPLETE to show the whole route or
   *                                        TRAFFIC_ROUTE_OVERVIEW_MODE__DETOUR  to show only a section of the route
   * @param[in] cameraAnimation           - required animation for camera for Traffic Route overview.
   */
   void showTrafficDetourInMap(::navmiddleware::MapViewId mapViewId,
                               ::navmiddleware::TrafficRouteOverviewMode trafficRouteOverviewMode,
                               ::navmiddleware::CameraAnimation cameraAnimation);

   /**
   * @brief Show address book icons in map.
   * @param[in] mapViewId        - view id of the map{primary or secondary}
   * @param[in] entryIds         - List of address book entryIds
   * @param[in] selectedIndex    - index in the list to be highlighted
   * @param[in] mapScale         - this value is considered only when a single entryId is given.
   *                                When a list of entryIds are passed, the map scale will be automatically
   *                                decided by map-engine and the value passed will have no effect.When mapScale
   *                                is invalid and we have a single entryId,we use the old mapScale.
   * @param[in] cameraAnimation  - Type of camera animation to be used.
   *
   * Note: When the entryIds vector is empty,all the dynamic address book icons will be deleted.
   * This method is also applicable for work and home entries besides address book entries.
   */
   void showAddressbookIconsInMap(::navmiddleware::MapViewId mapViewId,
                                  const ::std::vector< ::navmiddleware::DestinationMemoryEntry::Id >& entryIds,
                                  int32_t selectedIndex,
                                  const ::navmiddleware::ValidValue<int32_t>& mapScale,
                                  ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief show reachable area overview in map.
    * @param[in] mapViewId                 - view id of the map{primary or secondary}
    * @param[in] reachableAreaVisibility   - Reachable area visibility {carsor or destination}
    * @param[in] cameraAnimation           - required camera animation for reachable area overview.
    */
   void showReachableAreaOverviewInMap(::navmiddleware::MapViewId mapViewId,
                                       ::navmiddleware::ReachableAreaVisibility reachableAreaVisibility,
                                       ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief configures the visibility of routes on Map
    * @param[in] mapViewId              - view id of the map{primary or secondary}
    * @param[in] routeIndices           - list of route index to be hidden.
    * If the list is empty all the routes will be made visible.
    * Note: No property update will be sent for the request
    */
   void configureRouteVisibilityOnMap(::navmiddleware::MapViewId mapViewId, const ::std::vector<unsigned int>& routeIndices);

   /**
   * @brief Get the vector data for the upcoming intersections (for external display)
   *
    * Update after callback ::MAP_INTERSECTION_VECTOR_DATA_CHANGED
   */
   const ::navmiddleware::IntersectionMapVectorDataInfos& getIntersectionMapVectorDataInfos() const;

   /**
   * @brief Delete the avoid area previewed
   * @param[in] mapViewId             - view id of the map (PRIMARY or SECONDARY)
   */
   void deleteAvoidAreaPreview(::navmiddleware::MapViewId mapViewId);

   /**
   * @brief Adds and displays an avoid area preview on the map
   *
   * @param[in] mapViewId             - view id of the map (PRIMARY or SECONDARY)
   * @param[in] centerPoint           - point around which the avoid area is to be displayed
   * @param[in] edgeLength            - length of the edges of the avoid area
   * @param[in] avoidAreaPreviewStyle - provides information about if the avoid area includes freeways or not
   */
   void addAvoidAreaPreview(::navmiddleware::MapViewId mapViewId,
                            const ::navmiddleware::GeoCoordinateDegree& centerPoint, uint32_t edgeLength,
                            ::navmiddleware::AvoidAreaPreviewStyle avoidAreaPreviewStyle);

   /**
   * @brief Interface to inform NavMiddleware about the type of cluster that is connected.
   * @param[in] connectedClusterType     - Type of cluster that is connected
   */
   void setConnectedClusterType(::navmiddleware::ConnectedClusterType connectedClusterType);

   /**
    * @brief Switch ON/OFF crosshair visibility on MAP
    *
    * @param[in] mapViewId             - view id of the map (PRIMARY or SECONDARY)
    * @param[in] crossHairVisibility   - ON or OFF
    */
   void configureCrossHair(::navmiddleware::MapViewId mapViewId, ::navmiddleware::CrossHairVisibility crossHairVisibility);

   /**
    * @brief Set a required map scale
    * @param[in] mapViewId       - view id of the map (PRIMARY or SECONDARY)
    * @param[in] mapScale        - required mapScale value
    * @param[in] CameraAnimation - required animation for camera
    */
   void setMapScale(::navmiddleware::MapViewId mapViewId, int32_t mapScale, ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief sets the list of Marker(MarkerDisplayMode & GeoCoordinateDegree) for each MarkerType with MapLocationMarkerConfigurationOperation passed from the HMI.
    * HMI calls this method with list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MapLocationMarkerConfigurationOperation is set to
    * MARKER_CONFIGURATION_OPERATION__ADD, the new markers are added and appended to existing makers if any of same type.
    * HMI calls this method with list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MapLocationMarkerConfigurationOperation is set to
    * MARKER_CONFIGURATION_OPERATION__REPLACE, the new markers will be added and replace/remove any existing makers of same type.
    * HMI calls this method with empty list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MapLocationMarkerConfigurationOperation is set to
    * MARKER_CONFIGURATION_OPERATION__REPLACE, all existing location markers of type MarkerType will be deleted.
    * @param[in] mapViewId             - view id of the map{primary or secondary}
    * @param[in] mapLocationMarkers    - has details like list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MarkerType.
    * @param[in] operation             - to add/replace/remove location markers.
    */
   void configureMapLocationMarkers(
      ::navmiddleware::MapViewId mapViewId,
      const ::navmiddleware::MapLocationMarkers& mapLocationMarkers,
      ::navmiddleware::MapLocationMarkerConfigurationOperation operation);

   /**
   * @brief Get the carsor position for the current intersection(for external display)
   *
   * Update after callback ::MAP_INTERSECTION_MAP_CARSOR_POSITION_CHANGED
   */
   const ::navmiddleware::IntersectionMapCarsorPosition& getIntersectionMapCarsorPosition() const;

   /**
    * @brief Returns the current WeatherFeaturesStatusInfo
    *
    * Update after callback ::MAP_WEATHER_FEATURES_STATUS_CHANGED
    */
   const ::navmiddleware::WeatherFeaturesStatusInfo& getWeatherFeaturesStatusInfo() const;

   /**
    * @brief Returns the current MapOnlineFeaturesStatusInfo
    *
    * Update after callback ::MAP_ONLINE_FEATURE_AVAILABILITY_CHANGED
    */
   const ::navmiddleware::MapOnlineFeatureAvailabilityInfos& getMapOnlineFeatureAvailabilityInfos() const;

   /**
   * @brief Show satellite image on map.
   * @param[in] mapViewId        - view id of the map{primary or secondary}
   * @param[in] geoCoordinate    - geographical coordinate
   * @param[in] mapScale         - user should provide a valid map scale value, which is one of the requested map scales
   *                               set before in prepareSatelliteView().
   */
   void showSatelliteImagesInMap(::navmiddleware::MapViewId mapViewId, const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                                 int32_t mapScale);

   /**
    * @brief Switch on street view mode. Street view of the given position is shown
    * @param[in] mapViewId        - view id of the map{primary or secondary}
    * @param[in] streetViewMode   - display mode of the street view
    * @param[in] geoCoordinate    - geo position of where to show street view
    * @param[in] address          - address of the location for which the street view
    *                               image is required
    */
   void showStreetViewImagesInMap(::navmiddleware::MapViewId mapViewId,
                                  ::navmiddleware::StreetViewMode streetViewMode,
                                  const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                                  const ::navmiddleware::ValidValue< ::std::string >& address);

   /**
    * @brief Show Avoid area overview in map.
    * @param[in] mapViewId                       - view id of the map{primary or secondary}
    * @param[in] avoidAreaId                     - when set,avoidAreaId will contain the Id associated to a particular avoid area.
    * @param[in] avoidAreaVisibility             - if avoidAreaVisibility set to "OFF" no avoid area will be visible,
    *                                              if avoidAreaVisibility set to "MAP" show avoid area via stored context
    *                                              coordinates on map,if avoidAreaVisibility set to "HMI_OVERLAY" HMI shows
    *                                              the avoid area overlay in map.
    * @param[in] cameraAnimation                 - Type of camera animation to be used to show avoid area overview.
    */
   void showAvoidAreaOverviewInMap(::navmiddleware::MapViewId mapViewId, const ::navmiddleware::ValidValue< ::navmiddleware::AvoidAreaId>& avoidAreaId,
                                   ::navmiddleware::AvoidAreaVisibility avoidAreaVisibility, ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief getter for the AvoidAreaHMIOverlay which will be valid after notification
    * "MAP_AVOIDAREA_HMI_OVERLAY_INFO_CHANGED"
    *  Returns AvoidAreaHMIOverlay
    */
   const ::navmiddleware::AvoidAreaHMIOverlay& getAvoidAreaHMIOverlay() const;
   /**
    * @brief Transforms the given view positions on the map to its relative geographical coordinates
    * in reference to the map that is being displayed.
    * @param[in] mapViewId            - view id of the map{primary or secondary}
    * @param[in] screenCoordinates    - list of view positions to be transformed
    * @return unique RequestId
    * Note: The results won't be accurate if there any ongoing activities on map like moving,
    * zooming or animation.
    *
    * \async ::MAP_TRANSFORMED_VIEW_POSITIONS_AVAILABLE
    */
   ::navmiddleware::RequestId transformViewPositions(::navmiddleware::MapViewId mapViewId,
         const ::std::vector< ::navmiddleware::ScreenCoordinate >& screenCoordinates);

   /**
    * @brief getter for the requested geo coordinates after method 'transformViewPositions' was called.
    *        Returns transformed view positions holding list of geo coordinates and the request id.
    */
   const ::navmiddleware::TransformedViewPositionsInfo& getTransformedViewPositionsInfo() const;

   /**
    * @brief Request the street view preview image of the specified width and height for the specified
    * geo-coordinate.
    * The address is an optional parameter. If it is not specified, then the only geo-coordinate
    * will be considered while fetching the street view preview image.
    * @param[in] desiredWidth               - width of the street view preview image
    * @param[in] desiredHeight              - height of the street view preview image
    * @param[in] geoCoordinate              - geo-coordinate of the location for which the street view
    *                                         preview image is required
    * @param[in] address                    - address of the location for which the street view preview
    *                                         image is required
    *
    * \async ::MAP_STREET_VIEW_PREVIEW_IMAGE_AVAILABLE
    */
   void requestStreetViewPreviewImage(uint32_t desiredWidth,
                                      uint32_t desiredHeight,
                                      const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                                      const ::navmiddleware::ValidValue< ::std::string >& address);

   /**
    * @brief getter for the street view preview image after property update "MAP_STREET_VIEW_PREVIEW_IMAGE_AVAILABLE"
    * is received.
    * Note: For most cases, the dimensions of the image returned in the StreetViewPreviewImageInfo instance match the requested
    * desiredWidth and desiredHeight parameters. However, if the maximum supported image width/height is exceeded, the
    * StreetViewPreviewImageInfo::m_width and StreetViewPreviewImageInfo::m_height will still match the desired image dimensions,
    * but the actual dimensions of the image held by StreetViewPreviewImageInfo::m_image will be different.
    */
   const ::navmiddleware::StreetViewPreviewImageInfo& getStreetViewPreviewImage() const;

   /**
   * @brief show traffic overview in map.
   * If the list of trafficMessageID is empty and the guidance is ON/OFF, map shows a route overview with fix map scale of 30km radius.
   * If only one trafficMessageId is provided in the list, a valid map scale is also expected.
   * If more than one trafficMessageIDs are received, the map engine will calculate the map scale and the provided map scale is ignored.
   * @param[in] mapViewId           - View id of the map (PRIMARY or SECONDARY)
   * @param[in] trafficMessageID    - List of indices corresponding to the traffic message to be shown
   * @param[in] mapScale            - Required mapScale value.
   * @param[in] cameraAnimation     - Type of camera animation to be used.
   */
   void showTrafficOverviewMap(::navmiddleware::MapViewId mapViewId,
                               const ::std::vector< unsigned int >& trafficMessageID,
                               const ::navmiddleware::ValidValue<int32_t>& mapScale,
                               ::navmiddleware::CameraAnimation cameraAnimation);

   /**
    * @brief Method for pre-caching street view images BEFORE switching to street view with setMapCameraAndViewModes/showStreetViewImagesInMap.
    * Once the pre-caching is done, a property update will be triggered (MAP_PREPARE_STREETVIEW_DONE). The result must be checked
    * before switching to street view using getPrepareStreetViewStatusInfo(). Only if the result is PREPARE_STREETVIEW_STATUS__COMPLETE
    * should the street view be activated.
    * The address is an optional parameter. If it is not specified, then the only geo-coordinate
    * will be considered while fetching the street view preview image.
    *
    * @param[in] desiredWidth         - width of the street view preview image
    * @param[in] desiredHeight        - height of the street view preview image
    * @param[in] geoCoordinate        - geo-coordinate of the location for which the street view
    *                                   preview image is required
    * @param[in] address              - address of the location for which the street view preview
    *                                   image is required
    *
    * \async ::MAP_PREPARE_STREETVIEW_DONE
    */
   void prepareStreetView(uint32_t desiredWidth,
                          uint32_t desiredHeight,
                          const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                          const ::navmiddleware::ValidValue< ::std::string >& address);

   /**
    * @brief getter for the status of the pre-caching of the street view image after property update "MAP_PREPARE_STREETVIEW_DONE"
    * is received.
    */
   const ::navmiddleware::PrepareStreetViewStatusInfo& getPrepareStreetViewStatusInfo() const;

   /**
    * @brief Method to cancel all the on-going requests for prepareStreetView.
    */
   void cancelPrepareStreetView();

   /**
    * @brief Method to cancel all the on-going requests for requestStreetViewPreviewImage.
    */
   void cancelRequestStreetViewPreviewImage();

   /**
    * @brief Method to pre-cache the satellite view images for the list of map scales provided by HMI.
    * BEFORE switching to satellite view MAP_MODE_SATELLITE_VIEW, this pre-caching is necessary.
    * Once the pre-caching is done, a property update will be triggered (MAP_PREPARE_SATELLITE_VIEW_DONE). The result must be checked
    * before getting the satellite view info using getPrepareSatelliteViewInfo() and switching to satellite view state.
    *
    * @param[in] desiredWidthInPixels     - width of the satellite view preview image in pixels
    * @param[in] desiredHeightInPixels    - height of the satellite view preview image in pixels
    * @param[in] geoCoordinate            - geo-coordinate of the location for which the satellite view
    *                                       preview image is required
    * @param[in] mapScales                - list of map scales for which the satellite images should be precached.
    *
    *
    * \async ::MAP_PREPARE_SATELLITE_VIEW_DONE
    */
   void prepareSatelliteView(uint32_t desiredWidthInPixels,
                             uint32_t desiredHeightInPixels,
                             const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                             const ::std::vector<int32_t>& mapScales);

   /**
    * @brief getter for the status of the pre-caching of the satellite view image after property update "MAP_PREPARE_SATELLITE_VIEW_STATUS_CHANGED"
    * is received.
    */
   const ::navmiddleware::PrepareSatelliteViewInfo& getPrepareSatelliteViewInfo() const;

   /**
    * @brief Method to cancel all the on-going requests for prepareSatelliteView.
    */
   void cancelPrepareSatelliteView();

   /**
    * Returns the MapCameraAndViewModesStatusInfo
    * Update after callback ::MAP_SET_MAP_CAMERA_AND_VIEW_MODES_DONE
    */
   const ::navmiddleware::MapCameraAndViewModesStatusInfo& getMapCameraAndViewModesStatusInfo() const;

   /**
    * Returns the MapVideoStreamStatusInfo
    * Update after callback ::MAP_VIDEO_STREAM_STATUS_CHANGED
    */
   const ::navmiddleware::MapVideoStreamStatusInfo& getMapVideoStreamStatusInfo() const;

   /**
    * @brief Set the highlight state of a road in the given map view. The lineId is
    * used to identify the road whose highlight state is supposed to be modified.
    * @param[in] mapViewId        - view id {primary or secondary} of Map.
    * @param[in] lineId           - identifier of the road to be (de-)highlighted
    * @param[in] isHighlighted    - flag indicating whether highlighting is to turned on or off
    */
   void highlightRoad(::navmiddleware::MapViewId mapViewId, const ::navmiddleware::LineId& lineId, bool isHighlighted);

   /**
    * @brief Add the dynamic icons to map
    * @param[in] mapDynamicIconInfoList - List of the icons to be displayed in Map
    */
   void addDynamicIconsOnMap(const ::std::vector< ::navmiddleware::MapDynamicIconInfo>& mapDynamicIconInfoList);

   /**
    * @brief Delete the dynamic icons from map
    * @param[in] mapDynamicIconIdList - List of the MapDynamicIconIds which contains the
    *                               64bit unique id and intent for the icon
    */
   void deleteDynamicIconsOnMap(const ::std::vector< ::navmiddleware::MapDynamicIconId>& mapDynamicIconIdList);

   /**
    * @brief set FrameRate for Map
    * The client can also specify the frame rate reduction timeout
    * @param[in] mapViewId                - view id {primary or secondary} of Map.
    * @param[in] frameRate                - frame rate
    * @param[in] durationInMilliSeconds   - frame reduction timeout
   */
   void setMapFrameRate(::navmiddleware::MapViewId mapViewId, uint8_t frameRate, uint16_t durationInMilliSeconds);

   /**
    * @brief Getter to obtain the result of addDynamicIconsOnMap()/deleteDynamicIconsOnMap()
    * Update after callback :: "MAP_DYNAMICICON_REQUEST_RESULT_AVAILABLE"
    * Returns DynamicIconRequestResult which contains the mapping of
    * MapDynamicIconId and the corresponding icon addition status
    */
   const ::navmiddleware::DynamicIconRequestResult& getDynamicIconRequestResult() const;

   // ================ Positioning ==============
   const ::navmiddleware::PositionInfo& getPositionInfo() const;
   const ::navmiddleware::SdsPositionInfo& getSdsPositionInfo() const;
   const ::navmiddleware::GnssDataInfo& getGnssDataInfo() const;
   const ::navmiddleware::PositionStatusInfo& getPositionStatusInfo() const;
   const ::navmiddleware::RnwDataAvailability& getRnwDataAvailabilityAtCurrentPosition() const;
   const ::navmiddleware::PositionQuality& getPositionQuality() const;
   ::navmiddleware::VehicleMovingStatus getVehicleMovingState() const;
   const ::navmiddleware::PositionDebugInfo& getPositionDebugInfo() const;
   void requestPositionDebugInfo();
   void releasePositionDebugInfo();
   void setPositionByCoordinates(const ::navmiddleware::GeoCoordinateDegree& geoCoordinate, float heading);
   void startTripReplay(const ::std::string& tripFileName);
   void stopTripReplay();
   void seekTripReplay(int16_t seekValue);
   void setTripReplayMode(bool loopModeActive, uint16_t speed);
   const ::navmiddleware::TripReplayStatusInfo& getTripReplayStatusInfo() const;
   const ::navmiddleware::TripRecordingStatusInfo& getTripRecordingStatusInfo() const;
   ::navmiddleware::HomePositionStatus getHomeStatus() const;
   void writeTripMarker();

   // ================= WherAmI related methods ======
   void enableWhereAmIInfo() const;
   void disableWhereAmIInfo() const;
   const ::navmiddleware::WhereAmIInfo& getWhereAmIInfo() const;


   // ================= location input related methods ======================
   void setLocationWithCoordinates(double latitude, double longitude);
   void setLocationWithExternalLocation(const ::navmiddleware::ExternalLocation& externalLocation);
   void setLocationWithDestinationMemoryId(const ::navmiddleware::DestinationMemoryId& destinationMemoryId);
   void setLocationWithWaypointIndex(unsigned int waypointIndex);
   void setLocationWithPickingResult(const ::navmiddleware::PickingResults::PickingResult& pickedElement);
   void setLocationWithPoiData(const ::navmiddleware::PoiData& poiData);

   const ::std::string& getCurrentLocationString() const;

   ::navmiddleware::RequestId requestLocationAttributes();
   const ::navmiddleware::LocationAttributeInfos& getLocationAttributes() const;
   const ::navmiddleware::LocationAttributeErrorInfo& getRequestLocationAttributesError() const;
   const ::navmiddleware::LocationAttributeErrorInfo& getAddressAttributesOfCoordinateError() const;

   ::navmiddleware::RequestId requestResolvedCoordinate(const ::navmiddleware::GeoCoordinateDegree& geoCoordinate, bool isOnBoardMatchingRequired);
   const ::navmiddleware::LocationAttributeInfos& getResolvedCoordinateAttributeInfo() const;

   void setAllowPositionBasedDistanceAndDirectionUpdates(bool allow);
   void requestAddressFormatForAvailableCountries();
   const ::navmiddleware::AddressFormatInfos& getAddressFormatInfos() const;

   void requestAddressesNearPosition(const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                                     unsigned int radiusInMeters,
                                     unsigned int numberOfLocations);
   const ::navmiddleware::LocationInfos& getAddressesNearPositionInfos() const;

   void requestLocationImage(const ::navmiddleware::LocationAttributeInfos::PictureReferenceInfo& pictureRefInfo,
                             unsigned int requiredHeightInPixel,
                             unsigned int requiredWidthInPixel);
   const ::navmiddleware::Image& getLocationImageInfo() const;

   // Hierarchical / form-based address input
   void startFormBasedAddressInput(const ::navmiddleware::LocationContext& context,
                                   bool resetContents = false);
   void startAddressInputClone(const ::navmiddleware::LocationContext& context,
                               unsigned int instanceNumberToBeCloned);
   void startFormBasedAddressInputFromAddressDetails(const ::navmiddleware::LocationContext& context,
                                     const ::navmiddleware::LocationAttributeInfos::AddressDetailInfo& addressDetaiInfo);

   const ::navmiddleware::AddressInputInfo& getFormBasedAddressInputInfo() const;
   void selectAddressCategoryForInput(::navmiddleware::AddressCategory category, unsigned int numFirstMatches = 0);
   const ::navmiddleware::ElementSpellerInfo& getAddressSpellerInfo() const;
   void setAddressSpellerString(const ::std::string& spellerString, unsigned int numFirstMatches = 0);
   void requestAddressElements(unsigned int startIndex, unsigned int numElements);
   const ::navmiddleware::LocationInfos& getAddressElements() const;
   void selectAddressElement(const ::navmiddleware::LocationInfos::LocationInfo& locationInfo);
   void requestLastSelectedAddressElements();
   const ::navmiddleware::LocationInfos& getLastSelectedAddressElements();
   void selectFromLastSelectedAddressElements(const ::navmiddleware::LocationInfos::LocationInfo& locationInfo);
   void undoLastAddressInputStep();
   void clearAddressCategory(::navmiddleware::AddressCategory category);
   bool setLocationWithCurrentAddressInput();
   void leaveFormBasedAddressInput();
   void setPostValidationForHouseNumberInput(bool status);

   // Hierarchical / form-based POI input
   const ::navmiddleware::OnlinePoiAvailabilityInfo& getOnlinePoiAvailability() const;
   void requestPoiOnlineServerAvailability(::navmiddleware::LocationSearchSource server);
   void startHierarchicalPoiInput(
      const ::navmiddleware::LocationSearchScope& searchScope = ::navmiddleware::LocationSearchScope(),
      ::navmiddleware::LocationSearchSource searchSource = ::navmiddleware::LOCATIONSEARCHSOURCE__OFFLINE,
      const ::navmiddleware::LocationContext& context =
         ::navmiddleware::LocationContext(::navmiddleware::LocationContext::HMICONTEXT__GUI, ::navmiddleware::APPLICATIONID__NAVIGATION));
   void startPoiInputWithCategoryShortcut(
      unsigned int shortcutNum,
      const ::navmiddleware::LocationSearchScope& searchScope = ::navmiddleware::LocationSearchScope(),
      ::navmiddleware::LocationSearchSource searchSource = ::navmiddleware::LOCATIONSEARCHSOURCE__OFFLINE,
      ::navmiddleware::PoiCategoryFilter poiCategoryFilter = ::navmiddleware::POI_CATEGORY_FILTER__NO_FILTER,
      const ::navmiddleware::LocationContext& context =
         ::navmiddleware::LocationContext(::navmiddleware::LocationContext::HMICONTEXT__GUI, ::navmiddleware::APPLICATIONID__NAVIGATION));
   void startPoiInputWithinAllShortcutCategories(
      const ::navmiddleware::LocationSearchScope& searchScope = ::navmiddleware::LocationSearchScope(),
      ::navmiddleware::LocationSearchSource searchSource = ::navmiddleware::LOCATIONSEARCHSOURCE__OFFLINE,
      const ::navmiddleware::LocationContext& context =
         ::navmiddleware::LocationContext(::navmiddleware::LocationContext::HMICONTEXT__GUI, ::navmiddleware::APPLICATIONID__NAVIGATION));
   void startPoiInputWithPredefinedCategory(
      ::navmiddleware::PredefinedPoiCategory predefinedPoiCategory,
      const ::navmiddleware::LocationSearchScope& searchScope = ::navmiddleware::LocationSearchScope(),
      ::navmiddleware::LocationSearchSource searchSource = ::navmiddleware::LOCATIONSEARCHSOURCE__OFFLINE,
      ::navmiddleware::PoiCategoryFilter poiCategoryFilter = ::navmiddleware::POI_CATEGORY_FILTER__NO_FILTER,
      const ::navmiddleware::LocationContext& context =
         ::navmiddleware::LocationContext(::navmiddleware::LocationContext::HMICONTEXT__GUI, ::navmiddleware::APPLICATIONID__NAVIGATION));
   void startPoiInputWithInFtsChosenCategory(
      const ::navmiddleware::LocationSearchScope& searchScope = ::navmiddleware::LocationSearchScope(),
      ::navmiddleware::LocationSearchSource searchSource = ::navmiddleware::LOCATIONSEARCHSOURCE__OFFLINE,
      const ::navmiddleware::LocationContext& context =
         ::navmiddleware::LocationContext(::navmiddleware::LocationContext::HMICONTEXT__GUI, ::navmiddleware::APPLICATIONID__NAVIGATION));
   void resumePoiInputInstance(const ::navmiddleware::LocationContext& context);
   const ::navmiddleware::PoiInputInfo& getPoiInputInfo() const;
   void setPoiSortOrder(::navmiddleware::LocationSortOrder sortOrder);
   void setPoiSearchScope(const ::navmiddleware::LocationSearchScope& searchScope);
   void setPoiSearchSource(::navmiddleware::LocationSearchSource searchSource);
   void requestPoiEntries(unsigned int startIndex, unsigned int numEntries);
   const ::navmiddleware::LocationInfos& getPoiEntries() const;
   void requestPoiEntryAttributes(unsigned int listIndex);
   const ::navmiddleware::LocationAttributeInfos& getPoiEntryAttributes() const;
   const ::navmiddleware::LocationAttributeErrorInfo& getPoiEntryAttributesError() const;
   void selectPoiEntry(const ::navmiddleware::LocationInfos::LocationInfo& locationInfo);
   void selectPoiEntry(unsigned int listIndex);
   void selectAllPoiCategory();
   void undoLastPoiEntrySelection();
   bool setLocationWithCurrentPoiInput();
   void leaveHierarchicalPoiInput();

   // POI shortcut configuration
   const ::navmiddleware::LocationCategoryInfos& getPoiShortcutCategoryInfos(
      const ::navmiddleware::LocationContext& context) const;
   void resetPoiShortcutsToFactoryDefaults();
   void storeCurrentPoiCategoryAsShortcut(unsigned int shortCutNum, const ::navmiddleware::LocationContext& context);
   void movePoiShortcut(unsigned int shortCutNumDrag, unsigned int shortCutNumDrop, const ::navmiddleware::LocationContext& context);
   const ::std::map< ::navmiddleware::ExitPOICategory,
         ::navmiddleware::LocationCategoryInfos::LocationCategoryInfo>& getExitPoiCategoryInfos() const;
   void deletePoiShortcut(unsigned int shortCutNum, const ::navmiddleware::LocationContext& context);

   //Multiple icons for POI shortcuts
   void requestPoiIconInfos(const ::navmiddleware::LocationContext& context , const std::vector< ::navmiddleware::PoiIconInfos::ImageVariant>& variants,
                            const std::vector< ::navmiddleware::PoiCategoryId>& poiCategories);
   const ::navmiddleware::PoiIconInfos& getPoiIconInfos(const ::navmiddleware::LocationContext& context) const;

   // Charging Station filter settings
   const ::navmiddleware::settings::EvChargingStationFilterSettings& getEvChargingStationFilterSettings() const;
   void updateEvChargingStationFilterSettings(const ::navmiddleware::settings::EvChargingStationFilterSettings&);

   // Payment method settings
   const ::navmiddleware::settings::PaymentMethodSettings& getEvChargingStationPaymentMethodSettings() const;
   void updateEvChargingStationPaymentMethodSettings(const ::navmiddleware::settings::PaymentMethodSettings&);

   // Connector Type filter settings
   const ::navmiddleware::settings::EvConnectorTypeFilterSettings& getEvConnectorTypeFilterSettings() const;
   void updateEvConnectorTypeFilterSettings(const ::navmiddleware::settings::EvConnectorTypeFilterSettings&);

   // Parking settings
   const ::navmiddleware::settings::ParkingSettings& getParkingSettings() const;
   void updateParkingSettings(const ::navmiddleware::settings::ParkingSettings&);

   // ChargingSpeed filter settings
   const ::navmiddleware::settings::EvChargingSpeedFilterSettings& getEvChargingSpeedFilterSettings() const;
   void updateEvChargingSpeedFilterSettings(const ::navmiddleware::settings::EvChargingSpeedFilterSettings&);

   // Free Text Search (FTS) - Combined (addresses and POI)
   const ::navmiddleware::OnlineFreeTextSearchAvailabilityInfo& getOnlineFreeTextSearchAvailability() const;
   void requestFreeTextSearchOnlineServerAvailability(::navmiddleware::LocationSearchSource server);
   void startFreeTextSearchInput(
      ::navmiddleware::FreeTextSearchType searchType,
      const ::navmiddleware::LocationSearchScope& searchScope,
      ::navmiddleware::LocationSearchSource searchSource,
      const ::navmiddleware::LocationContext& context);
   void setFreeTextSearchString(const ::std::string& searchString);
   const ::std::string& getFreeTextSearchString() const;
   void setFreeTextSearchScope(const ::navmiddleware::LocationSearchScope& searchScope);
   void setFreeTextSearchSource(::navmiddleware::LocationSearchSource searchSource);
   void setFreeTextSearchSortOrder(::navmiddleware::LocationSortOrder sortOrder);
   void setFreeTextSearchAutoCompletionMode(::navmiddleware::AutoCompletionMode autoCompletionMode);
   void setFreeTextSearchAutoCompleteCategories(const ::navmiddleware::AutoCompletionConfiguration& autoCompletionConfiguration);
   void requestFreeTextSearchResults(unsigned int startIndex, unsigned int numElements,
                                     ::navmiddleware::FreeTextSearchType filterType);
   const ::navmiddleware::LocationInfos& getFreeTextSearchResultInfos() const;
   const ::navmiddleware::ElementSpellerInfo& getFreeTextSearchSpellerInfo() const;
   void requestFreeTextSearchResultAttributes(unsigned int listIndex);
   const ::navmiddleware::LocationAttributeInfos& getFreeTextSearchResultAttributes() const;
   const ::navmiddleware::LocationAttributeErrorInfo& getFreeTextSearchResultAttributesError() const;
   void setLocationWithFreeTextSearchResult(const ::navmiddleware::LocationInfos::LocationInfo& locationInfo);
   void setLocationWithFreeTextSearchResult(unsigned int listIndex);
   void undoFreeTextSearchElementSelection();
   void leaveFreeTextSearchInput();
   void clearFreeTextSearchHistory();
   const ::navmiddleware::RefreshInfo& getRefreshStatus() const;
   void requestRefreshForExpiredFTSResult(unsigned int listIndex);
   void requestOnboardMatchingForExpiredFTSResult(unsigned int listIndex);
   ::navmiddleware::EntrySelectionStatus getEntrySelectionStatus() const;
   void resumeFreeTextSearchInstance(const ::navmiddleware::LocationContext& context);
   const ::navmiddleware::ResumeFreeTextSearchStatusInfo& getResumeFreeTextSearchStatusInfo() const;

   // SDS (Speech Dialog System)
   void sdsCheckAddress(const ::navmiddleware::SDSAddress& sdsAddress);
   const ::navmiddleware::SDSAddressWithOptions& getCurrentSDSAddressWithOptions() const;
   void requestSdsRefinementList();
   void requestSdsRefinementList(::navmiddleware::SDSAddressElementType sdsAddressElementType);
   const ::navmiddleware::RefinementWithIdInfos& getSdsRefinementListInfo() const;
   void selectSdsRefinement(unsigned int id);
   void setLocationWithSdsInput();
   void requestSdsHouseNumberPatterns();
   const ::navmiddleware::SdsHouseNumberPatternsInfos& getSdsHouseNumberPatternsInfo() const;
   void requestGlobalSdsData();
   void requestIdListOfCountryAndSubCountry();
   ::navmiddleware::RequestId requestNamesForNdsIds(
      const ::std::vector< ::navmiddleware::SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfos,
      const ::std::vector< ::navmiddleware::LanguageEntryInfo>& languageEntryInfos,
      ::navmiddleware::NdsFeatureType ndsFeatureType,
      bool stopOnFirstMatch);
   ::navmiddleware::RequestId requestPhonemesForNdsIds(
      const ::std::vector< ::navmiddleware::SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfos,
      const ::std::vector< ::navmiddleware::LanguageEntryInfo>& languageEntryInfos,
      ::navmiddleware::NdsFeatureType ndsFeatureType,
      bool stopOnFirstMatch);
   void requestCityHasStreets(const ::std::vector< ::navmiddleware::SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfos);
   const ::navmiddleware::SdsGlobalDataInfo& getSdsGlobalDataInfo() const;
   const ::navmiddleware::SdsCountryAndSubCountryInfos& getSdsCountryAndSubCountryInfos() const;
   const ::navmiddleware::SdsNdsIdStringInfos& getSdsNdsIdNamedStringInfos() const;
   const ::navmiddleware::SdsNdsIdStringInfos& getSdsNdsIdPhonemeStringInfos() const;
   const ::std::vector< ::navmiddleware::SdsCityNamedObjectHasStreet>& getSdsCityNamedObjectHasStreetInfos() const;

   void denyMotorwayUsageInCurrentLocation();

   // ================= guidance related methods ======
   void setDemoModePosition(double latitude, double longitude, float heading);
   void setPositionSimulationMode(::navmiddleware::PositionSimulationMode simulationMode);
   void activateDemoMode();
   void deactivateDemoMode();
   bool isDemoModeActive() const;
   bool isSimulationActive() const;

   void startRouteGuidance();
   void startRouteCalculationAndGuidance();
   void restartGuidanceToLastDestination();
   const ::navmiddleware::DestinationInfos& getDestinationInfosForLastMileNavigation() const;
   const ::navmiddleware::DestinationMemoryItem& getLastGuidedItem() const;
   void stopRouteGuidance();
   bool isRouteGuidanceActive() const;
   bool isFinalDestinationReached() const;
   void setDemoModeSpeed(unsigned int speed);

   void setSpeedLimitData( const ::navmiddleware::SpeedLimitData& speedLimitData);

   void confirmPoiAlert(const unsigned int alertIdentifier) const;

   // =================acoustic related methods ======
   void retriggerAcousticOutput() const;
   void speakSamplePhrase(const ::navmiddleware::SamplePhrase samplePhraseType) const;
   void speakPoiAlert(const ::navmiddleware::PoiAlertType poiAlertType) const;
   void requestCountryAndStateDetails(
      const ::std::string& isoCountryCode, const ::std::string& isoSubCountryCode) const;
   // =================request road link restrictions ======
   void requestRoadRestrictions(const ::navmiddleware::ValidValue< ::navmiddleware::LinkId >& linkId) const;

   // ================ horizon event ===============
   const ::navmiddleware::HorizonEventInfo& getHorizonEventInfo() const;

   // ================ Eco nav related methods ===============
   void setEcoNavSwitchState(::navmiddleware::EcoNavSwitchState ecoNavSwitchState) const;
   ::navmiddleware::EcoNavSwitchState getEcoNavSwitchState() const;
   ::navmiddleware::EcoNavState getEcoNavStatus() const;

   // ================ Urban ZEV driving option ==============
   void setUrbanZevDrivingOption(::navmiddleware::UrbanZevDrivingOption urbanZevDrivingOption) const;
   ::navmiddleware::UrbanZevDrivingOption getUrbanZevDrivingOptionStatus() const;

   //================== guidance update related methods ================
   const ::navmiddleware::NextManeuverInfos& getNextManeuverInfos() const;
   const ::navmiddleware::LaneInfos& getLaneInfos() const;
   const ::navmiddleware::ManeuverViewInfo& getManeuverViewInfo() const;
   const ::navmiddleware::SignPostInfo& getSignPostInfo() const;
   const ::navmiddleware::SpeedInfo& getSpeedInfo() const;
   const ::navmiddleware::SpeedCamInfo& getSpeedCamInfo() const;
   const ::navmiddleware::TrafficSignInfo& getTrafficSignInfo() const;
   const ::navmiddleware::TrafficEnforcementInfo& getTrafficEnforcementInfo() const;
   const ::navmiddleware::DestinationReachedInfo& getDestinationReachedInfo() const;
   const ::navmiddleware::EVAlertInfo& getGuardianAngelAlertInfo() const;
   const ::navmiddleware::EVAlertInfo& getUnreachableDestinationAlertInfo() const;
   const ::navmiddleware::EVAlertInfo& getLowBatteryAlertInfo() const;
   const ::navmiddleware::PredictedDestinationInfo& getPredictedDestinationInfo() const;
   const ::navmiddleware::VehicleRestrictionsInfo& getVehicleRestrictionsInfo() const;
   ::navmiddleware::VehicleRestrictionsStatus getVehicleRestrictionsStatus() const;
   const ::navmiddleware::CountryAndStateDetailsInfo& getCountryAndStateDetailsInfo() const;

   //================== guidance settings related methods ================
   void updateGuidanceSettings(const ::navmiddleware::settings::GuidanceSettings&);
   const ::navmiddleware::settings::GuidanceSettings& getGuidanceSettings() const;

   void updateSafetyGuidanceSettings(const ::navmiddleware::settings::SafetyGuidanceSettings&);
   const ::navmiddleware::settings::SafetyGuidanceSettings& getSafetyGuidanceSettings() const;

   void updateSafetyCameraSettings(const ::navmiddleware::settings::SafetyCameraSettings&);
   const ::navmiddleware::settings::SafetyCameraSettings& getSafetyCameraSettings() const;

   void updateEvBatteryStatusWarningSettings(const ::navmiddleware::settings::EvBatteryStatusWarningSettings&);
   const ::navmiddleware::settings::EvBatteryStatusWarningSettings& getEvBatteryStatusWarningSettings() const;

   void updateEvMiscellaneousSettings(const ::navmiddleware::settings::EvMiscellaneousSettings&);
   const ::navmiddleware::settings::EvMiscellaneousSettings& getEvMiscellaneousSettings() const;

   // ================= electric vehicle related methods ======
   void setBatteryInfos(const ::navmiddleware::BatteryInfos& batteryInfos);

   void setChargingInfos(const ::navmiddleware::ChargingInfos& chargingInfos);

   void setDriveMode(::navmiddleware::DriveMode driveMode);

   const ::navmiddleware::ElectricVehicleInfos& getElectricVehicleInfos() const;

   void storeDataBeforeShutDown();

   // ================= route related methods ======
   const ::navmiddleware::ArrivalInfos& getArrivalInfos() const;
   const ::navmiddleware::DestinationInfos& getDestinationInfos() const;
   const ::navmiddleware::ArrivalInfos& getArrivalInfosGuidedRoute() const;
   const ::navmiddleware::DestinationInfos& getDestinationInfosGuidedRoute() const;
   const ::navmiddleware::ArrivalInfosAlternatives& getArrivalInfosAlternatives() const;
   const ::navmiddleware::WaypointInfos& getWaypointInfos() const;
   const ::navmiddleware::TrafficImpactRouteAlternativeInfos& getTrafficImpactRouteAlternativeInfos() const;
   const ::navmiddleware::RouteCalculationDurationInfos& getRouteCalcDurationInfos() const;
   const ::navmiddleware::RangePredictionInfos& getRangePredictionInfos() const;
   const ::navmiddleware::RouteCountryAndStateInfos& getRouteCountryAndStateInfos() const;
   const ::navmiddleware::ShapePointsAvailabilityInfo& getShapePointsAvailabilityInfo() const;
   const ::navmiddleware::ShapePointsInfo& getShapePointsInfo() const;

   const ::navmiddleware::settings::RouteCriteriaSettings& getRouteCriteriaSettings() const;
   void updateRouteCriteriaSettings(const ::navmiddleware::settings::RouteCriteriaSettings&);

   const ::navmiddleware::settings::RouteOptionsSettings& getRouteOptionSetting() const;
   void updateRouteOptionSettings(const ::navmiddleware::settings::RouteOptionsSettings&);

   ::navmiddleware::settings::RestoreStatus getLastRestoreStatusOfRouteCriteriaSettings() const;
   ::navmiddleware::settings::RestoreStatus getLastRestoreStatusOfRouteOptionsSettings() const;

   const ::navmiddleware::settings::VehicleProfileSettings& getVehicleProfileSettings() const;
   void updateVehicleProfileSettings(const ::navmiddleware::settings::VehicleProfileSettings&);
   void deleteLearnedRoutes();
   void deletePredictedDestinations();

   const ::navmiddleware::settings::RouteOptionsSettings& getRouteOptionSettingsFromRoute() const;
   void applySettingsToRoute(const ::navmiddleware::settings::RouteOptionsSettings& routeOptionsSettings);

   void startRouteCalculation(::navmiddleware::RouteCalculationMode routeCalculationMode);
   void stopRouteCalculation();
   void activatePrecalculatedRoute();
   bool hasActiveRoute() const;
   void calculateMoreRoutes();
   const ::navmiddleware::RouteCalculationInfo& getRouteCalculationInfo() const;
   const ::navmiddleware::PlannedRouteHistoryInfos& getPlannedRouteHistoryInfos() const;
   void activateHighlightedRoute();
   void removeUnusedRoutes();
   void requestWaypointList(::navmiddleware::WaypointListMode waypointListMode);
   void requestWaypointListPlannedRoute();
   void applyWaypointListChange(::navmiddleware::WaypointEditMode waypointEditMode);
   void setStartPosition();
   void deleteStartPosition();
   void insertWaypoint(unsigned int newIndex, ::navmiddleware::WaypointEditMode waypointEditMode);
   void deleteWaypoint(::std::vector<unsigned int>& waypointIndexList, ::navmiddleware::WaypointEditMode waypointEditMode);
   void moveWaypoint(unsigned int fromOldIndex, unsigned int toNewIndex, ::navmiddleware::WaypointEditMode waypointEditMode);
   void skipWaypoint(bool skipWaypoint, unsigned int waypointIndex);
   void setRouteCriterionForWaypoint(unsigned int waypointIndex, ::navmiddleware::settings::RouteCriterion routeCriterion);
   void setRouteCriterionForOptimization(::navmiddleware::settings::RouteCriterion routeCriterion);
   void setTimeWindowsForAllWaypoints(const ::std::vector< ::navmiddleware::WaypointOptimizationDetails >& waypointOptimizationDetails);
   void setFinalDestinationForOptimization(unsigned int waypointIndex, bool isFinalDest);
   void setOptimizationRange(unsigned int optimizeFromSegment, unsigned int optimizeToSegment);
   void applyRouteOptimization();
   void cancelRouteOptimization();
   void replaceWaypointLocation(unsigned int waypointIndex, ::navmiddleware::WaypointEditMode waypointEditMode);
   void cancelEditWaypointsMode();
   void acceptTrafficImpactRouteAlternative() const;
   void rejectTrafficImpactRouteAlternative() const;
   void startRouteListInfo(unsigned int indexAlternative, ::navmiddleware::RouteListConfiguration routeListConfiguration);
   void stopRouteListInfo(unsigned int indexAlternative);
   ::navmiddleware::ValidValue< ::navmiddleware::RequestId> requestRouteList(unsigned int indexAlternative, unsigned int startIndex, unsigned int numOfElements) const;
   const ::navmiddleware::RouteListInfos& getRouteListInfos() const;
   void avoidRouteListElements(unsigned int indexAlternative, unsigned int startIndex, unsigned int endIndex);
   void cancelAvoidRouteListElements();
   void requestJamAheadList();
   void releaseJamAheadList();
   void selectJamAheadListEntry(unsigned int listIndex);
   const ::navmiddleware::JamAheadListInfo& getJamAheadListInfo() const;
   void cancelJamAhead();
   void startStoredRouteCalculation(const ::std::vector< ::navmiddleware::DestinationMemoryWaypoint >& waypointList);
   void startStoredRouteCalculationWithDestinationMemoryId(::navmiddleware::DestinationMemoryRoute::Id routeId, uint32_t numberOfWaypointsToSkip);
   void startStoredRouteCalculationWithFtsSelection();
   void startStoredRouteCalculationForPlanning(::navmiddleware::DestinationMemoryRoute::Id routeId);
   void startRouteCalculationForPlanning();
   void startTourCalculation(const ::navmiddleware::Tour& tour);
   void startRangePredictionAtWaypointOrDestination(unsigned int indexAlternative, unsigned int waypointIndex);
   void stopRangePredictionAtWaypointOrDestination();
   void requestRouteCountryAndStateInfo(unsigned int indexAlternative);
   void requestShapePoints();

   void startAutomaticChargingStationDetection();
   void cancelAutomaticChargingStationDetection();
   const ::navmiddleware::AutomaticChargingStationDetectionInfo& getAutomaticChargingStationDetectionInfo() const;
   void insertChargingStationsToRoute(::navmiddleware::RouteCalculationMode routeCalculationMode);
   void removeChargingStationsFromRoute();

   // ================= traffic message related methods =====
   void requestTrafficMessageList(unsigned int startIndex, unsigned int numElements,
                                  ::navmiddleware::TrafficMessageListType listType,
                                  ::navmiddleware::TrafficMessageTextType textType);
   void requestTrafficMessageListForRange(unsigned int maxDistanceInMeters, ::navmiddleware::TrafficMessageListType listType,
                                          ::navmiddleware::TrafficMessageTextType textType,
                                          bool isFreeFlowMessagesRequired, bool isIncidentTrafficMessagesRequired);
   void requestTrafficMessageDetails(unsigned int trafficMessageID);
   const ::navmiddleware::TrafficMessageListInfos& getTrafficMessageListInfos() const;
   const ::navmiddleware::ValidValue< ::navmiddleware::TrafficMessageDetailsInfo>& getTrafficMessageDetailsInfo() const;
   void releaseTrafficMessageDetails(unsigned int trafficMessageID);
   const ::navmiddleware::TrafficMessageListInfos& getXUrgentTrafficMessageListInfos() const;
   void releaseXUrgentTrafficMessage(unsigned int trafficMessageID);
   const ::navmiddleware::ValidValue< ::navmiddleware::TrafficServiceReceptionStatusInfo>& getTrafficServiceReceptionStatusInfo() const;
   int getCountOfAllTrafficMessages() const;
   void requestTrafficIconSelectionList();
   const ::navmiddleware::TrafficIconSelectionListInfo& getTrafficIconSelectionListInfo() const;
   void setTrafficMessageListCurrentVisualRange(unsigned int startIndex, unsigned int numElements);
   void setTrafficMessageListCurrentFocusedElementIndex(const ::navmiddleware::ValidValue<unsigned int>& selectedIndex);
   void setTrafficMessageUserRule(unsigned int trafficMessageID, ::navmiddleware::TrafficMessageUserRule userRule);
   const ::navmiddleware::settings::TrafficSettings& getTrafficSettings() const;
   void updateTrafficSettings(const ::navmiddleware::settings::TrafficSettings&);
   void releaseAllTrafficMessageLists();
   void detourAllTrafficMessagesAlongRoute();
   void cancelDetourAllTrafficMessagesAlongRoute();
   ::navmiddleware::TrafficMessageDetourResult getTrafficMessageDetourResult() const;

   // ================ Location Based Services ==============
   ::navmiddleware::RequestId requestWeatherData(const ::std::vector< navmiddleware::ReportConfiguration>& reportConfigurations);
   const ::navmiddleware::WeatherDataInfos& getWeatherDataInfos(::navmiddleware::RequestId requestId) const;
   void releaseWeatherData(::navmiddleware::RequestId requestId);

   const ::navmiddleware::settings::LocationBasedServicesSettings& getLocationBasedServicesSettings() const;
   void updateLocationBasedServicesSettings(const ::navmiddleware::settings::LocationBasedServicesSettings&) const;
   ::navmiddleware::WeatherReceptionState getWeatherDataReceptionState() const;

   const ::navmiddleware::OnlineServicesDiagnosticInfos& getOnlineServicesDiagnosticInfos() const;
   void setAdvancedTrafficLoggingState(::navmiddleware::AdvancedLoggingState state);

   const ::navmiddleware::ConnectedServicesAvailabilityInfos& getConnectedServicesAvailabilityInfos() const;

   // ================ AvoidArea ==============
   void addAvoidArea(const ::std::string& avoidAreaName,
                     const ::navmiddleware::GeoCoordinateDegree& centerGeoCoordinate,
                     ::navmiddleware::AvoidAreaInfo::AvoidAreaSize avoidAreaSize,
                     uint32_t avoidAreaMapScale,
                     uint32_t edgeLength,
                     ::navmiddleware::AvoidAreaInfo::AvoidAreaOptions avoidAreaOptions );
   void modifyAvoidArea(const ::navmiddleware::AvoidAreaInfo& avoidAreaInfo);
   void deleteAvoidArea(const ::navmiddleware::AvoidAreaId avoidAreaId);
   void setAvoidAreaListFilter(const ::std::string& nameFilter);
   const ::navmiddleware::AvoidAreaInfos& getAvoidAreaInfos() const;
   void deleteAllAvoidAreas();
   void addAvoidArea(const ::std::string& avoidAreaName,
                     const ::std::vector< ::navmiddleware::ScreenCoordinate >& screenCoordinates,
                     ::navmiddleware::AvoidAreaInfo::AvoidAreaOptions avoidAreaOptions);
   void modifyAvoidArea(const ::navmiddleware::AvoidAreaInfo& avoidAreaInfo,
                        const ::std::vector< navmiddleware::ScreenCoordinate >& screenCoordinates);

   // ================ DynamicAreaUserStyle ==============
   void addDynamicAreaUserStyle(const ::std::vector< ::navmiddleware::DynamicAreaUserStyle >& dynamicAreaUserStyleList);
   ::navmiddleware::AddDynamicAreaUserStyleResult getAddDynamicAreaUserStyleResult() const;
   void deleteAllDynamicAreaUserStyle();

   // ================ Configuration Manager ==============

   const ::navmiddleware::RegisteredConfigItemsList& getConfigItemsToBeRegistered() const;
   const ::navmiddleware::ConfigItemsGetResult& getConfigItems(const ::std::string& configItemGroupName) const;
   void setConfigItems(const ::std::string& configItemGroupName, const ::std::vector< ::navmiddleware::ConfigItem>& configItems);
   ::navmiddleware::ConfigResultCode getSetConfigResult() const;
   void requestNavigationVersion();
   const ::std::string& getNavigationVersion() const;


   // ================ DestinationMemory ==============
   void storeEntry(::navmiddleware::DestinationMemoryEntryCategory category);
   void storeEntry(::navmiddleware::DestinationMemoryEntry::Id entryId, ::navmiddleware::DestinationMemoryEntryCategory category);
   void storeEntry(::navmiddleware::DestinationMemoryEntryCategory category, const ::navmiddleware::DestinationMemoryEntryAttributes& destinationMemAttributes);
   ::navmiddleware::DestinationMemoryStoreEntryStatus getStoreEntryStatus() const;
   void requestEntryDetails(::navmiddleware::DestinationMemoryEntry::Id id);
   const ::navmiddleware::DestinationMemoryEntry& getEntryDetails() const;
   void storeEntries(::navmiddleware::DestinationMemoryEntryCategory category,
                     const std::vector< ::navmiddleware::DestinationMemoryEntry >& entries);
   const ::navmiddleware::DestinationMemoryStoreEntriesStatus& getStoreEntriesStatus() const;
   void requestEntryList(::navmiddleware::DestinationMemoryEntryCategory category,
                         const ::navmiddleware::ValidValue< ::navmiddleware::DestinationMemoryAttributesGroup>& subGroup,
                         uint32_t startIndex, int32_t numElements,
                         ::navmiddleware::DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                         const ::std::string& freeTextSearchString);
   const ::navmiddleware::DestinationMemoryEntryList& getEntryList() const;
   void replaceEntryLocation(::navmiddleware::DestinationMemoryEntry::Id entryId);
   const ::navmiddleware::DestinationMemoryStatus& getDestinationMemoryStatus() const;
   void modifyEntryAttributes(::navmiddleware::DestinationMemoryEntry::Id entryId, const ::navmiddleware::DestinationMemoryEntryAttributes& destinationAttributesInfo);
   void setEntryPosition(::navmiddleware::DestinationMemoryEntry::Id entryId, uint64_t position);
   ::navmiddleware::ValidValue< ::std::string> getAddressbookMediaPath(::navmiddleware::DestinationMemoryEntryAttributes::Sound sound) const;
   void removeEntry(const ::std::vector< ::navmiddleware::DestinationMemoryEntry::Id >& entryId);
   void removeEntryByPoiId(const ::navmiddleware::PoiId& poiId);
   void removeAllEntries(::navmiddleware::DestinationMemoryEntryCategory category);
   void requestLastDestinationList(unsigned int startIndex,
                                   int numElements,
                                   ::navmiddleware::DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                                   ::navmiddleware::DestinationMemoryItemList::Filter requestFilter,
                                   const ::std::string& freeTextSearchString);
   const ::navmiddleware::DestinationMemoryItemList& getLastDestinationList() const;
   void updateLastDestinationHistoryList(const ::std::vector< ::navmiddleware::DestinationMemoryLastDestinationHistory >& lastDestinationHistoryList);
   const ::navmiddleware::DestinationMemoryUpdatedLastDestinationList& getUpdatedLastDestinationList() const;
   void insertLastDestinationList(const ::std::vector< ::navmiddleware::DestinationMemoryLastDestination >& lastDestionationList);
   const ::navmiddleware::DestinationMemoryUpdatedLastDestinationList& getInsertedLastDestinationList() const;
   void removeLastDestinationListItems(const ::std::vector< ::navmiddleware::DestinationMemoryItem::IdElement >& itemIdList);
   void removeAllLastDestinationListItems(::navmiddleware::DestinationMemoryItemList::Filter requestFilter);
   void requestLastDeparture();
   const ::navmiddleware::DepartureInfo& getLastDeparture() const;
   void removeLastDeparture();
   void storeRoute(::std::string name);
   void storeRoute(::navmiddleware::DestinationMemoryRoute::Id routeId, ::navmiddleware::DestinationMemoryRoute::Category category);
   void storeRoute(const ::std::string& name, const ::navmiddleware::Tour& tour,::navmiddleware::DestinationMemoryRoute::Category category);
   const ::navmiddleware::DestinationMemoryStoreRouteStatus& getStoreRouteStatus() const;
   void requestStoredRouteDetails(::navmiddleware::DestinationMemoryRoute::Id routeId);
   const ::navmiddleware::DestinationMemoryRoute& getStoredRouteInfo() const;
   void requestStoredRoutes(uint32_t startIndex,
                            int32_t numElements,
                            ::navmiddleware::DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                            const ::std::string& freeTextSearchString);
   const ::navmiddleware::DestinationMemoryRouteList& getStoredRoutes() const;
   void removeRoute(const ::std::vector< ::navmiddleware::DestinationMemoryRoute::Id >& routeIdList);
   void removeAllRoutes();
   void replaceStoredRoute(::navmiddleware::DestinationMemoryRoute::Id routeId);
   void renameStoredRoute(::navmiddleware::DestinationMemoryRoute::Id routeId, const ::std::string& name);
   uint32_t getLastGuidedRouteNumberOfWaypointsReached() const;
   void updateEvAutoSaveChargingStationSettings(const ::navmiddleware::settings::EvAutoSaveChargingStationSettings& );
   const ::navmiddleware::settings::EvAutoSaveChargingStationSettings& getEvAutoSaveChargingStationSettings() const;
   void requestLastDestinationsHistory(uint32_t startIndex,
                                       int32_t numElements,
                                       ::navmiddleware::DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                                       ::navmiddleware::DestinationMemoryItemList::Filter requestFilter);
   const ::navmiddleware::DestinationMemoryLastDestinationHistoryList& getLastDestinationsHistory() const;
   void setIncludeGooglePlacesInListRequests(bool includeGooglePlaces);
   void requestRefreshForExpiredDestinationEntry(::navmiddleware::DestinationMemoryEntry::Id entryId);
   void requestOnboardMatchingForExpiredDestinationEntry(::navmiddleware::DestinationMemoryEntry::Id entryId);
   ::navmiddleware::EntrySelectionStatus getDestinationSelectionStatus() const;
   const ::navmiddleware::DestinationMemoryAttributesGroupList& getAddressBookUsedGroups() const;
   void clearVoiceTags();

   // ================ NavDataUpdate ==============
   void requestNavigationDataVersionInfo();
   void checkNavDataCompatibility(const ::std::string& navigationVersionString, const ::std::vector<uint8_t>& compatibilityInfoData);
   void requestAvailableUpdates(uint32_t deviceId, const ::std::string& path,
                                ::navmiddleware::AvailableUpdatesRequestType requestType);
   void requestStopAvailableUpdates();
   void requestStopUpdate();
   void requestAvailableUpdatesList(uint32_t startIndex, int32_t nbrOfElements);
   void requestUpdateBySelection();
   void requestUpdateBySelection(const ::std::string& path);
   void requestPrepareUSBDevice(uint32_t deviceId);

   const ::navmiddleware::NavDataVersionInfo& getNavigationDataVersionInfo() const;
   const ::navmiddleware::NavDataUpdateSelection& getUpdateSelectionStatus() const;
   const ::navmiddleware::NavDataCompatibilityInfo& getNavDataCompatibilityInfo() const;
   const ::navmiddleware::NavDataUpdateInfos& getAvailableUpdatesList() const;
   const ::navmiddleware::NavDataUpdateStatus& getUpdateStatus() const;
   ::navmiddleware::NavDataUpdateUSBDeviceStatus getPreparedUSBUpdateDeviceResult();
   const ::navmiddleware::NavDataDeviceInfos& getNavDataDeviceInfos() const;
   void setUpdateSelection(uint32_t updateInfoId, bool select);
   void setSelectionOfAllUpdates(bool selected);
   const ::navmiddleware::NavDataUpdateOverviewInfo& getAvailableUpdateInfo() const;
   const ::navmiddleware::NavDataStatus& getNavDataStatus() const;

   // --- SOTA/SCOMO specific interfaces ---
   void requestScomoComponentStatus(::navmiddleware::ScomoComponent scomoComponent);
   const ::navmiddleware::ScomoComponentStatus& getScomoComponentStatus() const;
   void registerOnNavDataContextChanges();
   const ::navmiddleware::NavDataContext& getScomoNavDataContext() const;

   void setScomoComponentRegistration(::navmiddleware::ScomoComponent scomoComponent,
                                      ::navmiddleware::ScomoComponentRegistrationStatus registrationStatus);
   void requestScomoInstallComponent(::navmiddleware::NavDataUpdateInstallStage installStage,
                                     ::navmiddleware::ScomoComponent scomoComponent, const ::std::string& updateURL,
                                     const ::std::string& componentName);
   const ::navmiddleware::ScomoInstallComponentResult& getScomoInstallComponentResult() const;
   const ::navmiddleware::ScomoAvailableUpdates& getAvailableScomoUpdates() const;

   void requestAbortScomoInstallComponent(::navmiddleware::ScomoComponent scomoComponent);
   void setFotaInventoryUpdateStatus(bool isFotaInventoryUpdated);
   void updateNavDataUpdateFavoriteRegion(const ::navmiddleware::settings::NavDataUpdateFavoriteRegionSettings& favoriteRegionSetting);
   const ::navmiddleware::settings::NavDataUpdateFavoriteRegionSettings& getNavDataUpdateFavoriteRegion() const;
   ::navmiddleware::UpdateFavoriteUpdateRegionsResult getUpdateNavDataUpdateFavoriteRegionSettingsResult() const;

   ::navmiddleware::CatalogueStatus getCatalogueStatus() const;

   void requestNavDataPartNumber();
   const ::navmiddleware::NavDataPartNumber& getNavDataPartNumber() const;

   void requestNavDataUpdateHistory();
   const ::navmiddleware::NavDataUpdateHistory& getNavDataUpdateHistory() const;

   const ::navmiddleware::NavDatasetInfo& getNavDatasetInfo() const;
   const ::navmiddleware::RegisterDatasetUserResult& getDatasetUserRegisterStatus() const;
   void registerDatasetUser(::navmiddleware::NdsDatasetUser ndsDatasetUser, uint32_t datasetId);
   void deregisterDatasetUser(::navmiddleware::NdsDatasetUser ndsDatasetUser, uint32_t datasetId);
   void datasetUserAccess(::navmiddleware::NdsDatasetUser ndsDatasetUser, uint32_t datasetId, ::navmiddleware::NdsDataAccessType ndsDataAccessType);

   void setAutomaticMapUpdateStatus(::navmiddleware::settings::AutomaticMapUpdateStatus automaticMapUpdateStatus);
   ::navmiddleware::settings::AutomaticMapUpdateStatus getAutomaticMapUpdateStatus() const;
   ::navmiddleware::AutomaticMapUpdateStatusResult getAutomaticMapUpdateStatusResult() const;

   ::navmiddleware::AutomaticFullUpdateStatus getAutomaticFullUpdateStatus() const;
   void setAutomaticFullUpdateFailed(::navmiddleware::AutomaticFullUpdateFailureReason failureReason);

   // ================ EngineeringMenu ==============
   const ::navmiddleware::EngineeringInfos& getEngineeringInfos();
   void selectEngineeringMenuEntry(unsigned int selectedIndex);
   void deselectEngineeringMenuEntry();
   void resetEngineeringMenu();
   void enterEngineeringMenu();
   void exitEngineeringMenu();
   void updateUserInputByValue(const ::std::string& userInputValue);

   // =============EngineeringDebugInformation=================
   const ::navmiddleware::EngineeringDebugInfos& getTrafficDebugInformation();
   void requestTrafficDebugInformation();
   void releaseTrafficDebugInformation();

   // ================ DEBUG ===============
   void lock();
   void unlock();

   //<-- INF4CV
   // =============User POIs=================
   void importUserPois(const ::std::string& usbPath);
   const ::navmiddleware::ImportUserPoisStatus& getImportUserPoisStatus() const;
   void deleteUserPois();

   void muteGuidance(bool guidanceMode);
   //->
private:
   bool isFullNav() const;
   ::aivi_pres_ctrl::controls::AIVIPresCtrl* m_aiviPresCtrl;

#ifdef ASF_LOCK_EXTERNAL_THREAD

   class ProcessMessageHookImpl : public ::asf::core::ProcessMessageHookIF
   {
   public:
      ProcessMessageHookImpl(::boost::recursive_mutex& mutex)
         : m_mutex(mutex)
         , m_hasBeenLocked(false)
      {}

      virtual void onPreProcess()
      {
         m_mutex.lock();
         m_hasBeenLocked = true;
      }

      virtual void onPostProcess()
      {
         if (m_hasBeenLocked)
         {
            m_hasBeenLocked = false;
            m_mutex.unlock();
         }
      }

   private:

      ::boost::recursive_mutex& m_mutex;
      bool m_hasBeenLocked;
   };

   mutable ::boost::recursive_mutex m_mutex;

#endif
};

} //namespace adapter
} //namespace aivi_pres_ctrl

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_AIVI_PRES_CTRL_ADAPTER_AIVIPRESCTRLINTERFACE_H_

