/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     Thread.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_THREADING_THREAD_H
#define ASF_THREADING_THREAD_H

#include "asf/core/Types.h"
#include "asf/threading/ThreadTypes.h"
#include "boost/shared_ptr.hpp"

namespace asf {
namespace threading {

class ThreadImpl;
class ThreadPosix;

/**
 * A Thread represents a thread in the system.
 *
 * Every thread has a name and an id. The name is just informational and used
 * for logging purposes only. ASF assigns each thread a unique id of type
 * uint32. This ASF id should not be confused with the native id of the thread
 * in the system. We introduce our own id to guarantee a width of 32 bit for our id.
 * The size of the native thread id is unspecified. Each thread is logging its
 * native thread id and the systems native id during creation.
 */
class Thread {
public:
    typedef void* (*ThreadFunction)(void*);

    // lightweight, trivially copyable, serves as a unique identifier of ::asf::Thread
    // objects.
    typedef unsigned int id;

    /**
     * Creates and starts a new thread.
     */
    Thread(const std::string& name, ThreadFunction threadFunction, void* arg);

    virtual ~Thread();

    /**
     * Returns thread id
     */
    Thread::id getId() const;

    /**
     * Returns the native handle of the thread (e.g. pthread handle)
     */
    thread_t getNativeHandle() const;

    /**
     * Returns the name of the thread.
     */
    const std::string& getName() const;

    /**
     * Waits for the thread to terminate.
     */
    virtual void* join();

    /*
     * Causes the currently executing thread to sleep (temporarily cease execution)
     * for the specified number of milliseconds, subject to the precision and accuracy
     * of system timers and schedulers.
     *
     * @param millis time in milliseconds how long the thread should sleep.
     *
     * @return Returns true unless the ongoing sleep has not been interrupted.
     */
    static bool sleep(ms_t millis);

    /**
     * Interrupts this thread.
     *
     * If this thread is blocked in an invocation of the sleep(uint32) method, then the interrupt
     * status
     * will be cleared and the sleep() returns to the caller with a return value "false".
     *
     * If non of the the previous conditions hold then this thread's interrupt status will be set.
     */
    virtual void interrupt();

    /**
     * Returns whether the current thread has been interrupted.
     */
    static bool isCurrentThreadInterrupted();

    /**
     * @deprecated use getCurrentThread() instead
     */
    static Thread* getCurrent();

    /**
     * Returns the current thread of execution. May return null when the current thread
     * was not created by ASF.
     */
    static Thread* getCurrentThread();

    /**
     * Returns the id of the current thread. Only allowed to call on a thread that was
     * created by the ::asf::Thread class.
     */
    static id getCurrentThreadId();

    /**
     * Returns the native handle of the current thread (e.g. pthread handle).
     */
    static thread_t getCurrentThreadNativeHandle();

    /**
     * Attach a thread object to a running thread which was created outside the
     * scope of ASF. This is for example used to create a Thread instance for
     * the main thread in order to get the thread name into the logging output.
     */
    static ::boost::shared_ptr< Thread > attach();

private:
    Thread(const std::string& name);

    Thread(const Thread&);

    Thread& operator=(const Thread&);

    void doAttach();

    bool doSleep(ms_t millis);

    /**
     * Returns whether this thread has been interrupted.
     */
    bool isInterrupted() const;

    ThreadImpl* _poImpl;

    friend class ThreadPosix;
};

class ThreadImpl {
public:
    virtual ~ThreadImpl() {}

    virtual Thread::id getId() const = 0;

    virtual thread_t getNativeHandle() const = 0;

    virtual const std::string& getName() const = 0;

    virtual void* join() = 0;

    virtual void attach() = 0;

    virtual bool sleep(ms_t millis) = 0;

    virtual void interrupt() = 0;

    virtual bool isInterrupted() const = 0;
};

}  // namespace threading
}  // namespace asf

#endif  // THREAD_H
