/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     AtomicCounter.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ATOMIC_COUNTER_H
#define ATOMIC_COUNTER_H

#include "asf/threading/Guard.h"

namespace asf {
namespace threading {

/**
 * This class represents an atomic counter.
 *
 * TODO:
 * We make use of a own atomic counter implementation because the ::boost::detail::atomic_count
 * type has a fix width of 32 bits (long). It is not possible to pass a uint64_t as template
 * argument if internally a 32 bit value is used.
 * The disadvantage is that the current implementation is not lock-free. In the future we may make
 * use of C++ 11. In this case we should switch to the look-free variant std::atomic< T >.
 */
template < typename T >
class AtomicCounter {
public:
    AtomicCounter(T n = 0) : _value(n) {}

    AtomicCounter& operator=(const AtomicCounter& other) {
        if (this != &other) {
            ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lock);
            ::asf::threading::Guard< ::asf::threading::Mutex > guard2(other._lock);

            _value = other._value;
            // _lock is not assigned as each instance shall have a separate mutex
        }
        return *this;
    }

    /**
     * Increments the counter. This function can be called from any number
     * of threads in parallel.
     */
    T nextValue() {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lock);
        return ++_value;
    }

private:
    ::asf::threading::Mutex _lock;

    T _value;
};

}  // namespace threading
}  // namespace asf

#endif
