/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     ProtocolFramingAbstract.h
 *\brief    Handles incoming byte streams from different connections.
 *          If a frame is detected in the byte stream the containing message
 *          is given to the next stack element.
 *          The class provides for each connection a buffer to receive the
 *          data.
 *
 *\author   CM-AI/PJ-CF11.1
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_PROTOCOL_PROTOCOLFRAMINGABSTRACT_H
#define ASF_PROTOCOL_PROTOCOLFRAMINGABSTRACT_H

#include <cassert>
#include <cstdlib>
#include <cstring>
#include <map>
#include "asf/core/IncomingOutgoingIF.h"
#include "asf/core/Logger.h"
#include "asf/core/Types.h"
#include "asf/threading/Guard.h"
#include "asf/threading/Mutex.h"

namespace asf {
namespace protocol {

#define MAX_MESSAGE_SIZE (128 * 1024 * 1024)
#define BUFFER_CHUNK_SIZE (10 * 1024)

class ProtocolFramingAbstract : public ::asf::core::IncomingIF, public ::asf::core::OutgoingIF {
public:
    class ReceiveBuffer {
    public:
        ReceiveBuffer(size_t chunkSize, size_t maxMessageSize)
            : _chunkSize(chunkSize),
              _maxMessageSize(maxMessageSize),
              _buffer(0),
              _bufferPeak(0),
              _prevConsumedBytes(0) {
            _buffer = (uint8*)realloc(_buffer, _chunkSize);
            LOG_ASSERT_FATAL(_buffer);
            _bufferSize = _chunkSize;
        }

        virtual ~ReceiveBuffer() { free(_buffer); }

        bool attach(size_t size, const uint8* data) {
            uint8* bufferReallocated = 0;
            if ((_bufferPeak + size) > _bufferSize) {
                _bufferSize = ((_bufferPeak + size + _chunkSize - 1) / _chunkSize) * _chunkSize;
                if (_bufferSize <= _maxMessageSize) {
                    bufferReallocated = (uint8*)realloc(_buffer, _bufferSize);
                } else {
                    LOG_ERROR("The received message reached the limit of %" ASF_PRINT_SIZE_T
                              " bytes.",
                              _maxMessageSize);
                }
                if (!bufferReallocated) {
                    LOG_ERROR(
                        "Can't reallocate the receive buffer. The system will delete the complete "
                        "buffer (loose messages).");
                    free(_buffer);
                    _buffer = 0;
                    _bufferPeak = 0;
                    _prevConsumedBytes = 0;
                    _bufferSize = 0;
                    return false;
                }
                _buffer = bufferReallocated;
            }
            memcpy(_buffer + _bufferPeak, data, size);
            _bufferPeak += size;
            return true;
        }

        size_t getSize() const { return _bufferPeak; }

        size_t getBufferSize() const { return _bufferSize; }

        size_t getPrevConsumedBytes() const { return _prevConsumedBytes; }

        const char* getData() const { return (const char*)_buffer; }

        void discardConsumedBytes(size_t consumedBytes) {
            if (consumedBytes > 0) {
                LOG_SYSTEM_ASSERT_FATAL(_bufferPeak >= consumedBytes);
                memmove(_buffer, _buffer + consumedBytes, _bufferPeak - consumedBytes);
                _bufferPeak -= consumedBytes;
            }
            _prevConsumedBytes = consumedBytes;
        }

    private:
        ReceiveBuffer(const ReceiveBuffer&);
        ReceiveBuffer& operator=(const ReceiveBuffer&);

        size_t _chunkSize;
        size_t _maxMessageSize;
        uint8* _buffer;
        size_t _bufferPeak;
        size_t _bufferSize;
        size_t _prevConsumedBytes;
    };

    ProtocolFramingAbstract(::asf::core::IncomingIF* pIncoming,
                            size_t chunkSize = BUFFER_CHUNK_SIZE,
                            size_t maxMessageSize = MAX_MESSAGE_SIZE)
        : _pIncoming(pIncoming),
          _pOutgoing(0),
          _chunkSize(chunkSize),
          _maxMessageSize(maxMessageSize) {
        LOG_ASSERT(_chunkSize <= maxMessageSize);
        _pIncoming->setOutgoingIF(this);
    }

    virtual ~ProtocolFramingAbstract() {
        ReceiveBufferMap::iterator it = _connectionBuffers.begin();
        while (it != _connectionBuffers.end()) {
            ReceiveBuffer* receiveBuffer = it->second;
            delete receiveBuffer;
            ++it;
        }
        _pIncoming = 0;
        _pOutgoing = 0;
    }

    // implement OutgoingIF
    ::asf::core::ConnectionIFSharedPtr connectServer(const std::string& address) {
        LOG_ASSERT_FATAL(_pOutgoing);
        return _pOutgoing->connectServer(address);
    }

    void disconnectServer(::asf::core::ConnectionIFSharedPtr c) {
        LOG_ASSERT(c);
        LOG_ASSERT_FATAL(_pOutgoing);
        _pOutgoing->disconnectServer(c);
    }

    ssize_t sendData(::asf::core::ConnectionIFSharedPtr c, size_t size, const uint8* pu8Data) {
        LOG_ASSERT(c);
        return sendFramedData(c, size, pu8Data);
    }

    // implementation of  IncomingIF
    void setOutgoingIF(::asf::core::OutgoingIF* pOutgoing) { _pOutgoing = pOutgoing; }

    void onConnected(::asf::core::ConnectionIFSharedPtr c) {
        LOG_ASSERT(c);
        LOG_ASSERT(_pIncoming);
        _pIncoming->onConnected(c);
    }

    void onDisconnected(::asf::core::ConnectionIFSharedPtr c) {
        LOG_ASSERT(c);
        LOG_ASSERT(_pIncoming);
        _pIncoming->onDisconnected(c);

        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lockReceiverBufferMap);

        ReceiveBufferMap::iterator it = _connectionBuffers.find(c);
        if (it != _connectionBuffers.end()) {
            ReceiveBuffer* receiveBuffer = it->second;
            delete receiveBuffer;
            _connectionBuffers.erase(it);
        }
    }

    void onReceivedData(::asf::core::ConnectionIFSharedPtr c, size_t size, const uint8* pu8Data) {
        LOG_ASSERT(c);
        _lockReceiverBufferMap.lock();
        ReceiveBuffer* receiveBuffer = 0;

        ReceiveBufferMap::iterator it = _connectionBuffers.find(c);
        if (it != _connectionBuffers.end()) {
            receiveBuffer = it->second;
        } else {
            std::pair< ReceiveBufferMap::iterator, bool > retMap;
            receiveBuffer = new ReceiveBuffer(_chunkSize, _maxMessageSize);
            CHECK_ALLOCATION(receiveBuffer);
            retMap = _connectionBuffers.insert(ReceiveBufferPair(c, receiveBuffer));
            if (retMap.second == false) {
                // todo: add error trace or assert
                delete receiveBuffer;
                return;
            }
        }
        _lockReceiverBufferMap.unlock();
        LOG_ASSERT(receiveBuffer);
        if (receiveBuffer->attach(size, pu8Data)) {
            size_t consumedBytes =
                onReceivedFramedData(c, receiveBuffer->getSize(), receiveBuffer->getData());
            receiveBuffer->discardConsumedBytes(consumedBytes);
        } else {
            LOG_ERROR(
                "Can't attach data to the receive buffer (connection=%p). Maybe the system has run "
                "out of memory.",
                c.get());
        }
    }

    void onError(::asf::core::ConnectionIFSharedPtr c, enError err) {
        LOG_ASSERT(c);
        LOG_ASSERT(_pIncoming);
        _pIncoming->onError(c, err);
    }

    virtual ssize_t sendFramedData(::asf::core::ConnectionIFSharedPtr c,
                                   size_t size,
                                   const uint8* pu8Data) = 0;

    virtual size_t onReceivedFramedData(::asf::core::ConnectionIFSharedPtr c,
                                        size_t size,
                                        const char* buffer) = 0;

    // the following functions are for unit tests
    size_t getAllocationChunkSize() const { return _chunkSize; }

    size_t getConnectionCount() const { return _connectionBuffers.size(); }

    size_t getBufferSize(::asf::core::ConnectionIFSharedPtr c) {
        ReceiveBufferMap::iterator it = _connectionBuffers.find(c);
        if (it != _connectionBuffers.end()) {
            ReceiveBuffer* receiveBuffer = it->second;
            LOG_ASSERT(receiveBuffer);
            return receiveBuffer->getBufferSize();
        }
        return 0;
    }

    size_t getBufferPeak(::asf::core::ConnectionIFSharedPtr c) {
        ReceiveBufferMap::iterator it = _connectionBuffers.find(c);
        if (it != _connectionBuffers.end()) {
            ReceiveBuffer* receiveBuffer = it->second;
            LOG_ASSERT(receiveBuffer);
            return receiveBuffer->getSize();
        }
        return 0;
    }

    size_t getPrevConsumedBytes(::asf::core::ConnectionIFSharedPtr c) {
        ReceiveBufferMap::iterator it = _connectionBuffers.find(c);
        if (it != _connectionBuffers.end()) {
            ReceiveBuffer* receiveBuffer = it->second;
            LOG_ASSERT(receiveBuffer);
            return receiveBuffer->getPrevConsumedBytes();
        }
        return 0;
    }

protected:
    ::asf::core::IncomingIF* _pIncoming;

    ::asf::core::OutgoingIF* _pOutgoing;

private:
    ProtocolFramingAbstract(const ProtocolFramingAbstract&);

    ProtocolFramingAbstract& operator=(const ProtocolFramingAbstract&);

    ::asf::threading::Mutex _lockReceiverBufferMap;

    typedef std::map< ::asf::core::ConnectionIFSharedPtr, ReceiveBuffer* > ReceiveBufferMap;

    typedef std::pair< ::asf::core::ConnectionIFSharedPtr, ReceiveBuffer* > ReceiveBufferPair;

    ReceiveBufferMap _connectionBuffers;

    size_t _chunkSize;

    size_t _maxMessageSize;

    DECLARE_CLASS_LOGGER();
};

}  // namespace protocol
}  // namespace asf

#endif  // ASF_PROTOCOL_PROTOCOLFRAMINGABSTRACT_H
