/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DBusVariant.cpp
 *\brief
 *
 *\author   CM-AI/PJ-EAR-S
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_DBUS_DBUSVARIANT_H_
#define ASF_DBUS_DBUSVARIANT_H_

#include "asf/core/Logger.h"

#include <dbus/dbus.h>

namespace asf {
namespace dbus {

/**
 * The ASF DBusVariant data type corresponds to the native D-Bus variant type.
 * The content of the variant is known at runtime only. Therefore the user of the
 * variant is responsible for the de/-serialization of the variant payload.
 * Please refer to the ASF User Guide for more information about the usage.
 *
 * The content of the variant will be hold in a temporary D-Bus message. When the content is
 * de/-serialized using the getIterator() methods by the user, the variant type will be created
 * and copied to the provided proxy or stub message.
 */
class DBusVariant {
public:
    DBusVariant();

    ~DBusVariant();

    DBusVariant(const DBusVariant& variant);

    DBusVariant& operator=(const DBusVariant& v);

    /**
     * The equality operator checks the equality of the variant content.
     */
    bool operator==(const DBusVariant& v) const;

    bool operator!=(const DBusVariant& v) const;

    /**
     * Not supported, do not use!
     */
    bool operator<(const DBusVariant& /*v*/) const;

    /**
     * Not supported, do not use!
     */
    bool operator>(const DBusVariant& /*v*/) const;

    /**
     * This function provides the signature of the variant's content.
     */
    void getSignature(std::string& sig) const;

    /**
     * Return the iterator of the variant, which can be used to read the content.
     */
    ::DBusMessageIter* getReadIterator();

    /**
     * Return the iterator of the variant, which can be used to write content.
     */
    ::DBusMessageIter* getWriteIterator();

    /**
     * Returns the pointer to the internal dbus message.
     */
    ::DBusMessage* getMessage();

    /**
     * Copies variant content.
     */
    void copyPayload(DBusMessageIter* in, DBusMessageIter* out);

    /**
     * The init() method creates the temporary dbus message.
     */
    void init();

    bool deserialize(DBusMessageIter* in);

    void serialize(DBusMessageIter* out);

private:
    void createEmptyVariantPayload(DBusMessageIter* iter) const;

    bool hasValidContent(DBusMessageIter* iter) const;

    ::DBusMessage* _dbusMessage;

    ::DBusMessageIter _readIter;

    ::DBusMessageIter _writeIter;

    DECLARE_CLASS_LOGGER();
};
}  // namespace dbus
}  // namespace asf

#endif  // ASF_DBUS_DBUSVARIANT_H_
