/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     ProxyDelegateBase.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_PROXYDELEGATEBASE_H
#define ASF_CORE_PROXYDELEGATEBASE_H

#include <boost/weak_ptr.hpp>
#include <cassert>
#include "asf/core/CommunicationStack.h"
#include "asf/core/Logger.h"
#include "asf/core/Proxy.h"
#include "asf/core/Types.h"
#include "asf/core/Uri.h"
#include "asf/threading/Guard.h"
#include "asf/threading/Mutex.h"

namespace asf {
namespace core {

class ComponentDescription;
class ServiceMessage;

/**
 * Do make use of this logger macros in any class which is derived from ProxyDelegateBase!
 * It enables to find out the originator of the delegate proxy.
 *
 * Keep in mind that __VA_ARGS__ isn't portable. E.g. MSVC compiler does not allow the
 * variable argument part to be empty. You will get a compiler error if you call this macro
 * in the following way: LOG_ERROR_PROXY("My Error message").
 * In GCC this isn't a problem.
 */

#ifdef _MSC_VER
/**
 * Don't trace proxy pointer in case of windows, because there are some problems with
 * the macro expansion.
 */
#define LOG_DEBUG_PROXY(FORMAT, ...) LOG_DEBUG(FORMAT, __VA_ARGS__)
#define LOG_INFO_PROXY(FORMAT, ...) LOG_INFO(FORMAT, __VA_ARGS__)
#define LOG_WARN_PROXY(FORMAT, ...) LOG_WARN(FORMAT, __VA_ARGS__)
#define LOG_ERROR_PROXY(FORMAT, ...) LOG_ERROR(FORMAT, __VA_ARGS__)
#define LOG_FATAL_PROXY(FORMAT, ...) LOG_FATAL(FORMAT, __VA_ARGS__)
#else
#define LOG_DEBUG_PROXY(FORMAT, ...) LOG_DEBUG(FORMAT ", proxy=%p", ##__VA_ARGS__, getProxy())
#define LOG_INFO_PROXY(FORMAT, ...) LOG_INFO(FORMAT ", proxy=%p", ##__VA_ARGS__, getProxy())
#define LOG_WARN_PROXY(FORMAT, ...) LOG_WARN(FORMAT ", proxy=%p", ##__VA_ARGS__, getProxy())
#define LOG_ERROR_PROXY(FORMAT, ...) LOG_ERROR(FORMAT ", proxy=%p", ##__VA_ARGS__, getProxy())
#define LOG_FATAL_PROXY(FORMAT, ...) LOG_FATAL(FORMAT ", proxy=%p", ##__VA_ARGS__, getProxy())
#endif

class ProxyDelegateBase {
public:
    ProxyDelegateBase(Proxy* proxy);

    virtual ~ProxyDelegateBase();

    /**
     * The port name is defined in the CMC (Car Multimedia Component) model.
     * The port is bound with help of the application model (CMA) to service
     * provider.
     */
    const std::string& getPortName() const {
        assert(_proxy);
        return _proxy->getPortName();
    }

    /*
     * Returns the pointer of the proxy which is created by the user.
     * The user uses the generated factory function
     * "MyProxy* MyProxy::createProxy(...)" to instantiate concrete proxies.
     * The pointer is the same which is passed by the ServiceAvailableIF
     * interface.
     */
    Proxy* getProxy() const { return _proxy; }

    const ::boost::shared_ptr< Proxy > getProxyShared() const { return _proxyShared.lock(); }

    bool isProxySharedExpired() const { return _proxyShared.expired(); }

    void setProxyShared(::boost::shared_ptr< Proxy > proxyShared) { _proxyShared = proxyShared; }

    /*
     * The function returns the service state.
     * It is only possible to use the proxy if the service reached the state
     * "ServiceState__Available".
     * In case of state "ServiceState__Disconnected" and "ServiceState__Suspended"
     * the service is not usable.
     */
    virtual ::asf::core::ServiceState getServiceState() const {
        assert(_proxy);
        return _proxy->getServiceState();
    }

    void setServiceState(::asf::core::ServiceState state) {
        assert(_proxy);
        _proxy->_serviceState = state;
    }

    ::asf::core::ServiceAvailableIF* getServiceAvailableCallback() const {
        assert(_proxy);
        return &(_proxy->_serviceAvailable);
    }

    /*
     * The function checks if the proxy uses a internal or external service.
     * If the function returns false the service is provided by a component
     * which is in the same process (ASF application).
     * If the function returns true the service is provided by another process.
     * In this case the ASF core uses IPC (inter process communication).
     */
    bool isRemote() const {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lockRemote);
        return ((_remote && _stack != 0));
    }

    /*
     * Set the URI address of the service.
     * The proxy needs the information to bind a remote service. If the
     * proxies has no address then the system tries to find the service
     * internally.
     */
    void setRemoteAddress(const std::string& address) {
        _remoteAddress = address;
        Uri add(_remoteAddress);
        _scheme = add.getScheme();
    }

    const std::string& getRemoteAddress() const { return _remoteAddress; }

    /*
     * Returns the protocol. The protocol is part of the URI address.
     */
    const std::string& getScheme() const { return _scheme; }

    /*
     * The function returns a pointer of the communication stack,
     * if the proxy is bound with a remote service.
     * If the proxy uses a internal service no communication stack is
     * necessary for the communication.
     */
    ::asf::core::CommunicationStack* getCommunicationStack() const { return _stack; }

    void setCommunicationStack(CommunicationStack* stack) { _stack = stack; }

    /*
     * If the proxy is bound with a remote service then the function
     * returns the concrete connection instance.
     * One communication stack can handle "n" concrete connections.
     */
    ::asf::core::ConnectionIFSharedPtr getRemoteConnection() const {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lockRemote);
        return _remote;
    }

    void setRemoteConnection(ConnectionIFSharedPtr con) {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lockRemote);
        _remote = con;
    }

    /*
     * This is a convenience function to check if the service is usable.
     */
    bool isConnected() const {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lockRemote);
        if (isRemote())
            return (_remote && _serviceConnected);
        else
            return _serviceConnected;
    }

    void setServiceConnected(bool con) { _serviceConnected = con; }

    /*
     * The client component ID is the context in which the proxy acts.
     */
    void setClientComponentId(::asf::identifier_t clientCompId) { _clientCompId = clientCompId; }

    ::asf::identifier_t getClientComponentId() const { return _clientCompId; }

    ComponentDescription* getComponentDescription() { return _componentDescription; }

    /**
     * send message to service provider
     */
    virtual void send(::asf::core::ServiceMessage& msg);

    /**
     * process incoming answers
     */
    virtual void process(::asf::core::ServiceMessage& msg) = 0;

private:
    DECLARE_CLASS_LOGGER();

    Proxy* _proxy;

    ::boost::weak_ptr< Proxy > _proxyShared;

    CommunicationStack* _stack;

    ::asf::threading::Mutex _lockRemote;

    ConnectionIFSharedPtr _remote;

    ::asf::identifier_t _clientCompId;

    ComponentDescription* _componentDescription;

    std::string _remoteAddress;

    std::string _scheme;

    bool _serviceConnected;
};

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_PROXYDELEGATEBASE_H
