/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     PayloadTuple.h
 *\brief
 *
 *\author   CM/ESA2
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2016-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_PAYLOADTUPLE_H
#define ASF_CORE_PAYLOADTUPLE_H

#include "asf/core/Payload.h"

#include <sstream>
#include <string>
#include <vector>

namespace std {
template < typename T >
ostream& operator<<(ostream& o, const vector< T >& value) {
    o << "[";
    for (size_t x = 0; x < value.size(); ++x) {
        o << value[x];
        if (x < value.size() - 1) o << ",";
    }
    o << "]";
    return o;
}
}  // namespace std

namespace asf {
namespace core {

std::string stringsToJsonDictionary(const std::vector< std::string >& v);

// The PayloadTuple#n classes represent tuples which are used as payload
// for local messages. The number as in PayloadTuple2 represent the number of
// elements in the tuple. Currently the largest tuple is PayloadTuple3.
// Larger tuple classes will be created when there is demand for it.

// These tuples serialize their content with the << operator to a json dictionary.
// The values of the elements are encoded as strings with help of their respective
// << operator. For example:
//
// {
//    "0" : "42",
//    "1" : "foo"
// }
//
// This is used for logging the payload of local messages.

class PayloadTuple0 : public ::asf::core::Payload {
public:
    PayloadTuple0(){};

    friend std::ostream& operator<<(std::ostream& o, const PayloadTuple0& /*tuple*/) {
        std::vector< std::string > v;
        o << stringsToJsonDictionary(v);
        return o;
    }
};

template < typename T0 >
class PayloadTuple1 : public ::asf::core::Payload {
public:
    PayloadTuple1(const T0& t0) : _0(t0) {}

    const T0& get0() const { return _0; }

    friend std::ostream& operator<<(std::ostream& o, const PayloadTuple1< T0 >& tuple) {
        std::stringstream t0;
        t0 << tuple.get0();
        std::vector< std::string > v;
        v.push_back(t0.str());
        o << stringsToJsonDictionary(v);
        return o;
    }

private:
    T0 _0;
};

template < typename T0, typename T1 >
class PayloadTuple2 : public ::asf::core::Payload {
public:
    PayloadTuple2(const T0& t0, const T1& t1) : _0(t0), _1(t1) {}

    const T0& get0() const { return _0; }

    const T1& get1() const { return _1; }

    friend std::ostream& operator<<(std::ostream& o, const PayloadTuple2< T0, T1 >& tuple) {
        std::stringstream t0;
        std::stringstream t1;
        t0 << tuple.get0();
        t1 << tuple.get1();
        std::vector< std::string > v;
        v.push_back(t0.str());
        v.push_back(t1.str());
        o << stringsToJsonDictionary(v);
        return o;
    }

private:
    T0 _0;

    T1 _1;
};

template < typename T0, typename T1, typename T2 >
class PayloadTuple3 : public ::asf::core::Payload {
public:
    PayloadTuple3(const T0& t0, const T1& t1, const T2& t2) : _0(t0), _1(t1), _2(t2) {}

    const T0& get0() const { return _0; }

    const T1& get1() const { return _1; }

    const T2& get2() const { return _2; }

    friend std::ostream& operator<<(std::ostream& o, const PayloadTuple3< T0, T1, T2 >& tuple) {
        std::stringstream t0;
        std::stringstream t1;
        std::stringstream t2;
        t0 << tuple.get0();
        t1 << tuple.get1();
        t2 << tuple.get2();
        std::vector< std::string > v;
        v.push_back(t0.str());
        v.push_back(t1.str());
        v.push_back(t2.str());
        o << stringsToJsonDictionary(v);
        return o;
    }

private:
    T0 _0;

    T1 _1;

    T2 _2;
};

}  // namespace core
}  // namespace asf

#endif
