/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2015
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file    LoggingBackendIF.h
 *\brief
 *
 *\author  CM-AI/ESA2
 *         christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2015 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_LOGGINGBACKENDIF_H
#define ASF_CORE_LOGGINGBACKENDIF_H

#include "asf/core/LoggerAccessorIF.h"
#include "asf/core/LoggingConst.h"
#include "asf/core/Types.h"
#include "asf/threading/ThreadTypes.h"

#include <cstdio>
#include <map>
#include <string>

#ifdef __QNX__
#include <stdarg.h>
#endif

namespace asf {
namespace core {

/**
 * It is possible to attach an external logging backend to the ASF
 * core on runtime by name. To provide a logging backend, it is
 * necessary to implement the abstract interface LoggingBackendIF.
 * The external logging backend must be delivered in form of a
 * shared library. To cause the ASF core to load the backend, it is
 * necessary to add the name of the backend to the logger
 * configuration file. If a backend name is given in the configuration
 * file, the core uses the external backend instead of the default
 * backend.
 *
 * Additionally, it is possible to add backend specific configuration
 * data. For that the json key "backendConfiguration" is reserved.
 * The ASF core reads the configuration and delegates the data
 * in form of a std::map<std::string, std::string> to the backend.
 *
 * Example:
 * {
 *      "loggerLevel" : {
 *          "*" : "Info"
 *      },
 *
 *      "backend" : "myloggingbackend",
 *      "backendLibraryPath" : "/my/path", (optional)
 *      "backendConfiguration" : {
 *          "myParamter1" : "true"
 *          "myParamter2" : "42"
 *      }
 * }
 */
class LoggingBackendIF {
public:
    /**
     * The major version of this interface. Whenever this interface will
     * be changed, the version number will be incremented.
     *
     * The shared library which implements this interface has to report its implemented version
     * of the interface via the getLoggingBackendVersion() function.
     */
    static const uint32 API_MAJOR_VERSION = 1;

    virtual ~LoggingBackendIF() {}

    struct InitParameters {
        /**
         * The package name of the application
         */
        const std::string& appPackageName;
        /**
         * The name of the application
         */
        const std::string& appName;
        /**
         * The backendConfiguration of the configuration file (e.g. logConfig.json)
         */
        const std::map< std::string, std::string >& backendConfiguration;
        /**
         * For logger access (e.g. set logging level)
         */
        LoggerAccessorIF& accessor;
    };

    /**
     * The init() function is called by ASF to initialize the backend. This will be
     * the first function being called from this interface.
     *
     * @param InitParameters Includes all initialization parameters
     *
     * @returns Returns true if it was possible to initialize the logging backend,
     * otherwise false.
     */
    virtual bool init(struct InitParameters& parameters) = 0;

    /**
     * Returns a description of the backend. The description is logged by ASF
     * in order to let users identify the used backend. This function will be
     * called after invoking the init() function.
     */
    virtual ::std::string getBackendDescription() = 0;

    /**
     * Returns the name of the backend.
     */
    virtual ::std::string getBackendName() = 0;

    struct LoggingHeader {
        /**
         * Time elapsed since the start of the application. Format: hh:mm:ss.ms
         */
        ms_t timeStamp;
        /**
         *  Level of the message: Debug, Info, Warning, Error or Fatal.
         */
        Logging::Level level;
        /**
         * Name of the thread currently being executed.
         */
        const std::string& threadName;
        /**
         * Id of the thread currently being executed.
         */
        ::asf::threading::thread_t threadHandle;
        /**
         * The activityIndex represents the number of messages that have been processed by
         * the currently executing thread. The index is increment for every message.
         */
        size_t activityIndex;
        /**
         * The component instance name, as defined in the application, which is currently active.
         * Empty if no component is currently active.
         */
        const std::string& componentName;
        /**
         * The id of the currently executing component. A value of zero means that no component
         * is currently executing.
         */
        ::asf::identifier_t componentId;
        /**
         * A unique identifier of the received message which is currently processed.
         */
        ::asf::msgId_t messageId;
        /**
         * The logger name (or path) of the log message.
         */
        const std::string& logger;
        /**
         * The user data associated with the logger (see setLoggerUserData()).
         */
        void* userData;
        /**
         * The this pointer of the object which created the log message.
         */
        const void* _this;
        /**
         * Line number of the log statement
         */
        size_t line;
        /**
         * The source code file name of the log statement
         */
        const char* file;
    };

    /**
     * The logFormat() function is called by log macros like LOG_DEBUG to emit a log message.
     * This function runs inside the ASF message processing loop and has therefore to return as
     * soon as possible. Otherwise the execution of ASF components will be slowed down.
     *
     * This function will only be called when the log level of the message is of the same or a
     * a higher priority as the the log level of the logger. This means log levels need not
     * be compared again by the implementation.
     */
    virtual void logFormat(const struct LoggingHeader& header,
                           const char* format,
                           va_list argptr) = 0;

    /**
     * The logBuffer() function is called by log macros like LOG_INFO_MEMORY_DUMP to log the content
     * of a memory region. This function runs inside the ASF message processing loop and has
     * therefore
     * to return as soon as possible. Otherwise the execution of ASF components will be slowed down.
     *
     * This function will only be called when the log level of the message is of the same or a
     * a higher priority as the the log level of the logger. This means log levels need not
     * be compared again by the implementation.
     */
    virtual void logBuffer(const struct LoggingHeader& header,
                           const uint8* buffer,
                           size_t length) = 0;
};

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_LOGGINGBACKENDIF_H
