/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2015
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     LoggerAccessorIF.h
 *\brief
 *
 *\author   CM-AI/ESA2
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2015 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_LOGGERSACCESSORIF_H
#define ASF_CORE_LOGGERSACCESSORIF_H

#include "asf/core/LoggingConst.h"
#include "asf/core/Types.h"

#include <map>
#include <string>

namespace asf {
namespace core {

/**
 * This interface gives a logging backend access to the loggers.
 */
class LoggerAccessorIF {
public:
    virtual ~LoggerAccessorIF() {}

    /**
     * Set the log level of one or more loggers. It is possible to use a wildcard
     * to address more than one logger.
     *
     * @param logger The name of the logger (e.g. "/my/logger/test", "/my/logger*").
     * @param level The log level which should be set.
     *
     * @returns Returns the number of loggers on which the level was set
     */
    virtual size_t setLoggingLevel(const ::std::string& logger, Logging::Level level) = 0;

    /**
     * The function returns a snapshot of the current log level configuration.
     *
     * @returns loggers List of all loggers and their current log level.
     */
    virtual const ::std::map< ::std::string, Logging::Level >& getCurrentLoggerLevels() const = 0;

    /**
     * The setLoggerUserData() function assigns a user defined pointer to a logger.
     * This pointer is passed later to the logFormat() and logBuffer() functions. A logging
     * backend can use this mechanism to save lookups via the logger name.
     *
     * @param logger The name of the logger. It must be a full match, wildcards
     * are not allowed.
     * @param userData A void pointer, returned later to the logFormat() and logBuffer()
     * functions
     *
     * @returns Returns true if it was possible to set the user data.
     */
    virtual bool setLoggerUserData(const ::std::string& logger, void* userData) = 0;

    /**
     * Get the user data of a logger.
     *
     * @param logger The name of the logger. It must be a full match, wildcards
     * are not allowed.
     *
     * @returns Returns the user data which was set previously. The function returns a
     * a null pointer if no user data was set before.
     */
    virtual void* getLoggerUserData(const ::std::string& logger) = 0;
};

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_LOGGERSACCESSORIF_H
