/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2014
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*
 *\file     ApplicationLauncher.h
 *\brief    helper class to start and stop ASF application
 *
 *\author   CM-AI/ECA2
 *          christoph.perick@de.bosch.com
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2014-2014 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef APPLICATIONLAUNCHER_H
#define APPLICATIONLAUNCHER_H

#include "asf/core/Logger.h"
#include "asf/core/MessageQueue.h"

#include "asf/threading/Mutex.h"
#include "asf/threading/Signal.h"
#include "asf/threading/Thread.h"

namespace asf {
namespace core {

/**
 * ApplicationLauncher
 *
 * Starts an ASF Application in a separate thread in the background.
 *
 * This class is only required when starting a ASF application which is
 * configured as a library and thus part of a foreign process.
 *
 * Only a single instance of this class should exists in a process.
 */
class ApplicationLauncher {
public:
    ApplicationLauncher()
        : _thread(0), _queue("ApplicationLauncherQueue"), _main(0), _argc(0), _argv(0) {
        LOG_ASSERT_FATAL_MSG(0 == _instance, "Can't create another ApplicationLauncher instance");
        _instance = this;
    }

    ~ApplicationLauncher();

    typedef int (*mainfptr)(int argc, char* argv[]);

    /**
     * Convenience method without argc and argv
     */
    void launch(mainfptr main) { launch(main, 0, 0); }

    /**
     * Launches the application in a new thread. The call will return
     * when the application is launched in the background and ready to
     * process messages.
     *
     * The application will be running until shutdown() is called on the
     * application.
     *
     * @param mainfptr Entry point of the ASF application. This is usually
     * the generate executeMyApplication function of the cpp file generated
     * out of the cma file.
     *
     * @param argc argument count
     *
     * @param argv list of arguments
     */
    void launch(mainfptr main, const char* name, int argc, char** argv) {
        LOG_DEBUG("Entering launch");
        LOG_ASSERT_MSG(_thread == 0, "Can't launch, application is already launched");

        _main = main;
        _argc = argc;
        _argv = argv;

        // this lock makes sure that _thread variable is initialized before the thread starts
        // to run and may call the signalApplicationIsLaunched or shutdown method, see (*)
        _threadCreationLock.lock();
        _thread = new ::asf::threading::Thread(name, threadFunction, this);
        CHECK_ALLOCATION(_thread);
        _threadCreationLock.unlock();

        LOG_DEBUG("Wait for application is launched");
        _queue.dequeue();
        LOG_DEBUG("Application is launched");
    }

    void launch(mainfptr main, int argc, char** argv) {
        launch(main, "ApplicationLauncher", argc, argv);
    }

    /**
     * Shuts down the application and wait for the background thread to join
     */
    void shutdown(int exitCode = 0);

    /**
     * Wait until application is closed (background thread finished).
     * The application shutdown must be triggered from another place.
     * This can be done by the command "Application::getApplication()->shutdown()".
     *
     * @return The exit code of the application.
     */
    int waitForCompletion();

    /**
     * This function is called by the ASF application to signal that the application
     * is started and is ready to process messages. The launch method waits for the
     * background thread to call this method.
     */
    void signalApplicationIsLaunched() {
        LOG_DEBUG("Entering signalApplicationIsLaunched");
        LOG_ASSERT(_thread);  // make sure launch was called before
        _queue.enqueue(::boost::shared_ptr< bool >(new bool));
    }

    /**
     * Returns the current ApplicationLauncher instance.
     */
    static ApplicationLauncher* getCurrentInstance() { return _instance; }

private:
    static void* threadFunction(void* p) {
        LOG_DEBUG_STATIC("Starting threadFunction");
        ApplicationLauncher* al = (ApplicationLauncher*)p;

        // (*) this lock makes sure that the thread starts after the _thread variable has been
        // initialized in the launch method
        al->_threadCreationLock.lock();
        al->_threadCreationLock.unlock();
        return reinterpret_cast< void* >(al->_main(al->_argc, al->_argv));
    }

    ::asf::threading::Thread* _thread;
    ::asf::core::MessageQueue< bool > _queue;
    ::asf::threading::Mutex _threadCreationLock;

    mainfptr _main;
    int _argc;
    char** _argv;

    static ApplicationLauncher* _instance;

    DECLARE_CLASS_LOGGER();
};
}  // namespace core
}  // namespace asf

#endif  // APPLICATIONLAUNCHER_H
