/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CmsStubProperties.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CMS_CMSSTUBPROPERTIES_H
#define ASF_CMS_CMSSTUBPROPERTIES_H

#include <map>
#include "asf/cms/CmsMessage.h"
#include "asf/cms/CmsSystemErrorIF.h"
#include "asf/threading/Mutex.h"

namespace asf {
namespace cms {

/**
 * Stores all property registrations
 *
 * It is used by the CmsStubDelegate class to:
 * - store registrations on incoming REGISTER messages
 * - remove registrations on incoming DEREGISTER messages
 * - Query all registrations to inform all related clients if a
 *   property value has changed
 */
class CmsStubProperties {
public:
    enum RegistrationState {
        E_REGISTRATION_NONE = 0,
        E_REGISTRATION_STARTED,
        E_REGISTRATION_FINISHED
    };

    /**
     * Registration entry. Each registration includes a message and
     * a registration state.
     */
    class Registration {
    public:
        Registration(CmsMessage& msg, RegistrationState state) : _msg(msg), _state(state) {}

        void setState(RegistrationState& state) { _state = state; }

        const RegistrationState& getState() const { return _state; }

        CmsMessage& getMessage() { return _msg; }

        CmsMessage _msg;

    private:
        RegistrationState _state;
    };

    typedef std::map< act_t, Registration > RegistrationMap;

    typedef std::pair< act_t, Registration > RegistrationPair;

    CmsStubProperties(CmsSystemErrorIF& systemError);

    virtual ~CmsStubProperties();

    /*
     * Adds a REGISTER message to the registration map. The function returns a unique
     * act for each registration. The function returns 0, if the registration fails.
     */
    act_t add(CmsMessage& msg);

    /**
     * Returns a list of registrations which relates to the passed functionId.
     */
    void getRegistrations(std::string functionId,
                          std::vector< CmsStubProperties::Registration >& result);

    /**
     * Deregister all messages which have a relation to the passed connection.
     * All related registration entries will be removed and DEREGISTER messages
     * will be sent to the related clients.
     */
    void deregisterConnection(::asf::core::ConnectionIFSharedPtr remote);

    /**
     * Deregister all messages which have a relation to the serviceId, clientId
     * and connection of the passed message.
     * All related registration entries will be removed and DEREGISTER messages
     * will be sent to the related clients.
     */
    void deregisterService(CmsMessage& msg);

    /**
     * Deregister message which relates to the cct, serviceId, functionId
     * clientId and connection.
     * All related registration entries will be removed and DEREGISTER messages
     * will be sent to the related clients.
     */
    void deregisterFunction(CmsMessage& msg);

    /**
     * Tries to find a registered message by the given act. The complete message of the search
     * hit will be copied to the passed message reference. Additionally the registration state
     * will be set. The function returns TRUE if a message was found and the state was set,
     * otherwise FALSE.
     */
    bool getMessageAndSetState(act_t act,
                               const std::string& serviceId,
                               const std::string& functionId,
                               RegistrationState state,
                               CmsMessage& retMsg);

    /**
     * Tries to find the corresponding message to the passed act and functionId. Afterwards
     * the function removes the entry and returns a TRUE.
     * If act=0 is assigned the function tries to read it from the thread local storage.
     * If it was not possible to reconstruct the act the function tries to find a unique entry
     * which has the same functionId like the passed one. If the search result is ambiguous the
     * system can't evaluate the registration entry without the act.
     * The function returns FALSE if no corresponding message was found.
     */
    bool getAndRemove(act_t act,
                      const std::string& serviceId,
                      const std::string& functionId,
                      CmsMessage& retMsg);

private:
    void deregister(RegistrationMap::iterator& it);

    bool isRegistered(CmsMessage& msg);

    uint32 getRegistrationCount(const std::string& serviceId, const std::string& functionId);

    void cleanUpAllActiveRegistrations();

    ::asf::threading::Mutex _msgMutex;

    RegistrationMap _registrations;

    CmsSystemErrorIF& _systemError;

    DECLARE_CLASS_LOGGER();
};
}  // namespace cms
}  // namespace asf

#endif  // ASF_CMS_CMSSTUBPROPERTIES_H
