/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CmsMessage.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CMS_CMSMESSAGE_H
#define ASF_CMS_CMSMESSAGE_H

#include <boost/weak_ptr.hpp>
#include "asf/cms/CmsTypes.h"
#include "asf/core/Blob.h"
#include "asf/core/Logger.h"
#include "asf/core/Message.h"
#include "asf/core/Payload.h"
#include "asf/core/Proxy.h"
#include "asf/core/ServiceMessage.h"
#include "asf/core/TokenPool.h"
#include "asf/core/Types.h"
#include "asf/threading/Thread.h"

namespace asf {
namespace core {

class ServiceStreamer;
class ProvidedPort;
}  // namespace core
}  // namespace asf

namespace asf {
namespace cms {

class CmsServiceStreamer;
class CmsServiceConnector;
class CmsMessage;
class CmsMessageAdapter;
class CmsProxyDelegate;
class CmsProxy;

typedef void (*ProxyCallback)(void* obj, CmsMessage* msg);

class CmsMessage : public ::asf::core::ServiceMessage {
public:
    CmsMessage()
        : _msgHeader(new CmsTypes::CmsMessageHeader()),
          _act(0),
          _payloadOffset(0),
          _streamer(0),
          _connector(0),
          _adapter(0),
          _localProvidedPort(0),
          _systemError(CmsTypes::SystemError__Undefined),
          _initiatorMessageType(CmsTypes::CmsMessageHeader::CmsMessageType__Request),
          _registrationCount(0),
          _trigger(false),
          _serviceRegistered(true) {
        CHECK_ALLOCATION(_msgHeader);
    }

    CmsMessage(const CmsTypes::CmsMessageHeader::CmsMessageType messageType,
               const ::std::string& serviceId,
               const ::std::string& functionId,
               act_t cct,
               uint32 clientId)
        :  // cast is allowed because the act/cct of a CMS message is never greater than UINT32_MAX
           // (created by token pool)
          _msgHeader(new CmsTypes::CmsMessageHeader(
              messageType, serviceId, functionId, static_cast< uint32 >(cct), clientId)),
          _act(0),
          _payloadOffset(0),
          _streamer(0),
          _connector(0),
          _adapter(0),
          _localProvidedPort(0),
          _systemError(CmsTypes::SystemError__Undefined),
          _initiatorMessageType(CmsTypes::CmsMessageHeader::CmsMessageType__Request),
          _registrationCount(0),
          _trigger(false),
          _serviceRegistered(true) {
        CHECK_ALLOCATION(_msgHeader);
    }

    CmsMessage(const CmsMessage& c)
        : ServiceMessage(c),
          _act(0),
          _payloadOffset(0),
          _streamer(0),
          _connector(0),
          _adapter(0),
          _localProvidedPort(0),
          _systemError(CmsTypes::SystemError__Undefined),
          _initiatorMessageType(CmsTypes::CmsMessageHeader::CmsMessageType__Request),
          _registrationCount(0),
          _trigger(false),
          _serviceRegistered(true) {
        *this = c;
    }

    virtual ~CmsMessage() {
        _streamer = 0;
        _connector = 0;
        _adapter = 0;
        _localProvidedPort = 0;
    }

    virtual ::asf::core::MessageSharedPtr clone() const;

    void setHeader(const CmsTypes::CmsMessageHeader& msgHeader) {
        _msgHeader.reset(new CmsTypes::CmsMessageHeader(msgHeader));
    }

    const CmsTypes::CmsMessageHeader& getHeader() const { return *_msgHeader; }

    CmsTypes::CmsMessageHeader& getHeaderMutable() const { return *_msgHeader; }

    void setServiceName(const std::string& serviceName) { _serviceName = serviceName; }

    const std::string& getServiceName() const { return _serviceName; }

    void setStreamer(CmsServiceStreamer* streamer) { _streamer = streamer; }

    CmsServiceStreamer* getStreamer() const { return _streamer; }

    void setStubConnector(CmsServiceConnector* connector) { _connector = connector; }

    CmsServiceConnector* getStubConnector() const { return _connector; }

    void setLocalProvidedPort(::asf::core::ProvidedPort* localProvidedPort) {
        _localProvidedPort = localProvidedPort;
    }

    ::asf::core::ProvidedPort* getLocalProviderPort() const { return _localProvidedPort; }

    void setAdapter(CmsMessageAdapter* cmsMessageAdapter) { _adapter = cmsMessageAdapter; }

    void setRawMessage(const ::boost::shared_ptr< ::asf::core::Blob >& blobMessage) {
        _blobMessage = blobMessage;
    }

    void appendRawMessage(const uint8* data, size_t size) { _blobMessage->push_back(data, size); }

    void appendRawMessage(const ::boost::shared_ptr< ::asf::core::Blob >& blobMessage) {
        _blobMessage->push_back(blobMessage->getBytes(), blobMessage->getSize());
    }

    const ::boost::shared_ptr< ::asf::core::Blob >& getRawMessage() const { return _blobMessage; }

    ::boost::shared_ptr< ::asf::core::Blob >& getRawMessageMutable() { return _blobMessage; }

    const uint8* getRawMessagePayload() const {
        uint8* rawMessage = _blobMessage->getBytes();
        return &rawMessage[_payloadOffset];
    }

    bool hasRawMessagePayload() const {
        return (_blobMessage.get() && (_blobMessage->getSize() > _payloadOffset));
    }

    size_t getRawMessagePayloadSize() const {
        if (_blobMessage.get()) return _blobMessage->getSize() - _payloadOffset;

        return 0;
    }

    void setPayloadOffset(size_t payloadOffset) { _payloadOffset = payloadOffset; }

    size_t getPayloadOffset() const { return _payloadOffset; }

    void setPayload(const ::asf::core::PayloadSharedPtr& payload) { _payload = payload; }

    const ::asf::core::PayloadSharedPtr getPayload() const { return _payload; }

    bool hasPayload() const { return !(_payload.get() == 0); }

    void resetPayload() { _payload.reset(); }

    virtual const ::std::string& getServiceId() const { return _msgHeader->getServiceId(); }

    const ::std::string& getFunctionId() const { return _msgHeader->getFunctionId(); }

    void setServiceId(const std::string& serviceId) const { _msgHeader->setServiceId(serviceId); }

    void setFunctionId(const std::string& functionId) const {
        _msgHeader->setFunctionId(functionId);
    }

    void setClientId(uint32 clientId) const { _msgHeader->setClientId(clientId); }

    uint32 getClientId() const { return _msgHeader->getClientId(); }

    void setCct(act_t cct) const {
        // cast is allowed because the act of a CMS message is never greater than UINT32_MAX
        // (created by token pool)
        _msgHeader->setCct(static_cast< uint32 >(cct));
    }

    act_t getCct() const { return _msgHeader->getCct(); }

    void setAct(act_t act) { _act = act; }

    act_t getAct() const { return _act; }

    CmsTypes::CmsMessageHeader::CmsMessageType getCmsMessageType() const {
        return _msgHeader->getMessageType();
    }

    const char* getCmsMessageTypeName() const {
        return _msgHeader->CmsMessageType_Name(_msgHeader->getMessageType());
    }

    void setCmsMessageType(CmsTypes::CmsMessageHeader::CmsMessageType type) const {
        return _msgHeader->setMessageType(type);
    }

    void setProxy(::boost::shared_ptr< ::asf::cms::CmsProxy > proxy) { _weakProxy = proxy; }

    ::boost::shared_ptr< ::asf::cms::CmsProxy > getProxy() const { return _weakProxy.lock(); }

    bool isProxyAlive() const;

    bool isProxyConnected() const;

    void setSystemError(CmsTypes::SystemError error) { _systemError = error; }

    CmsTypes::SystemError getSystemError() const { return _systemError; }

    void setInitiatorMessageType(CmsTypes::CmsMessageHeader::CmsMessageType initiatorMessageType) {
        _initiatorMessageType = initiatorMessageType;
    }

    CmsTypes::CmsMessageHeader::CmsMessageType getInitiatorMessageType() const {
        return _initiatorMessageType;
    }

    void setRegistrationCount(uint32 count) { _registrationCount = count; }

    uint32 getRegistrationCount() const { return _registrationCount; }

    bool isTrigger() const { return _trigger; }

    void setTrigger(bool trigger) { _trigger = trigger; }

    void setServiceRegistered(bool serviceRegistered) { _serviceRegistered = serviceRegistered; }

    bool isServiceRegistered() const { return _serviceRegistered; }

    virtual void processMessage();

    virtual bool sendRemote();

    bool isProxyAnswer() const;

    static act_t getUniqueCct();

    static void returnCct(act_t cct);

    static void deleteTokenPool();

    static act_t getTokenPoolPeak();

    static ::asf::core::TokenPool< act_t >* getTokenPool();

    void serializeHeader();

    CmsMessage& operator=(const CmsMessage& rhs);

private:
    ::boost::shared_ptr< CmsTypes::CmsMessageHeader > _msgHeader;

    act_t _act;

    // The service name of the CMS description (e.g. "example.echocms.EchoService")
    // The ServiceId can differentiate if it is exported by another name.
    std::string _serviceName;

    ::boost::shared_ptr< ::asf::core::Blob > _blobMessage;

    size_t _payloadOffset;

    ::asf::core::PayloadSharedPtr _payload;

    CmsServiceStreamer* _streamer;

    CmsServiceConnector* _connector;

    CmsMessageAdapter* _adapter;

    ::asf::core::ProvidedPort* _localProvidedPort;

    ::boost::weak_ptr< ::asf::cms::CmsProxy > _weakProxy;

    CmsTypes::SystemError _systemError;

    CmsTypes::CmsMessageHeader::CmsMessageType _initiatorMessageType;

    uint32 _registrationCount;

    bool _trigger;

    bool _serviceRegistered;

    static ::asf::core::TokenPool< act_t >* _cctTokens;

    DECLARE_CLASS_LOGGER();
};

}  // namespace cms
}  // namespace asf

#endif  // ASF_CMS_CMSMESSAGE_H
