/************************************************************************
 
 *FILE:           alltypes.h
 
 *SW-COMPONENT:   FI, framework for Functional Interface
 
 *DESCRIPTION:    Interface for all types defined in the different FIs
                  
 *AUTHOR:         Jochen Bruns (CM-DI/ENS41)
 
 *COPYRIGHT:      (c) 2001 Blaupunkt GmbH
 
 *HISTORY:
           $ Log $
 * 
 *    Rev 1.0   Oct 7 2001 17:16:00   bnj2hi
 * Initial revision
 *  
 ************************************************************************/

#ifndef SPM_ALLFITYPES_HEADER
#define SPM_ALLFITYPES_HEADER

#include <vector>

class spm_fi_tclVisitorBase;

const tU16 SPM_FI_C_U16_ERROR_INTERNALFAILURE = 1;
const tU16 SPM_FI_C_U16_ERROR_INVALIDBINCONTAINERWRONGVERSION = 2;
const tU16 SPM_FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA = 3;

class spm_fi_tclTypeBase : public fi_tclTypeBase
{
public:
  spm_fi_tclTypeBase() {}
  virtual ~spm_fi_tclTypeBase() {}
   
  virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& /*oVisitor*/) {}
};

class spm_fi_tclVisitableTypeBase :public spm_fi_tclTypeBase
{
public:
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& /*oVisitor*/) {}
};

class spm_fi_tclU8 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclU8():u8Value(0) {}
   tU8 u8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclU8& roRef) const
      { return (u8Value == roRef.u8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_U8;}
};

class spm_fi_tclS8 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclS8():s8Value(0) {}
   tS8 s8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclS8& roRef) const
      { return (s8Value == roRef.s8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_S8;}
};

class spm_fi_tclU16 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclU16():u16Value(0) {}
   tU16 u16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclU16& roRef) const
      { return (u16Value == roRef.u16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_U16;}
};

class spm_fi_tclS16 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclS16():s16Value(0) {}
   tS16 s16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclS16& roRef) const
      { return (s16Value == roRef.s16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_S16;}
};

class spm_fi_tclU24 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclU24():u24Value(0) {}
   tU24 u24Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclU24& roRef) const
      { return (u24Value == roRef.u24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_U24;}
};

class spm_fi_tclS24 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclS24():s24Value(0) {}
   tS24 s24Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclS24& roRef) const
      { return (s24Value == roRef.s24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_S24;}
};

class spm_fi_tclU32 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclU32():u32Value(0) {}
   tU32 u32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclU32& roRef) const
      { return (u32Value == roRef.u32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_U32;}
};

class spm_fi_tclS32 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclS32():s32Value(0) {}
   tS32 s32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclS32& roRef) const
      { return (s32Value == roRef.s32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_S32;}
};

class spm_fi_tclU64 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclU64():u64Value(0) {}
   tU64 u64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclU64& roRef) const
      { return (u64Value == roRef.u64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_U64;}
};

class spm_fi_tclS64 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclS64():s64Value(0) {}
   tS64 s64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclS64& roRef) const
      { return (s64Value == roRef.s64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_S64;}
};

class spm_fi_tclF32 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclF32():f32Value(0) {}
   tFloat f32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclF32& roRef) const
      { return (f32Value == roRef.f32Value);  } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_F32;}
};

class spm_fi_tclF64 : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclF64():f64Value(0) {}
   tDouble f64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclF64& roRef) const
      { return (f64Value == roRef.f64Value); } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_F64;}
};

class spm_fi_tclBool : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclBool():bValue(0) {}
   tBool bValue;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const spm_fi_tclBool& roRef) const
      { return (bValue == roRef.bValue); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_BOOL;}
};

class spm_fi_tclString : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tclString(const spm_fi_tclString& coRef);
   spm_fi_tclString& operator=(const spm_fi_tclString& coRef);
   virtual ~spm_fi_tclString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const spm_fi_tclString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   spm_fi_tclString& operator=(tCString coszString); // ADDED
   spm_fi_tclString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8,
      FI_EN_ISO8859_1M
   };
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const spm_fi_tclString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};

class spm_fi_tclList : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclList():u32ListSize(0) {}
   tU32 u32ListSize;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const spm_fi_tclList& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oIn);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_LIST;}
};

class spm_fi_tclMultiLanguageString : public spm_fi_tclVisitableTypeBase
{
public:
   spm_fi_tclMultiLanguageString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tclMultiLanguageString(const spm_fi_tclMultiLanguageString& coRef);
   spm_fi_tclMultiLanguageString& operator=(const spm_fi_tclMultiLanguageString& coRef);
   virtual ~spm_fi_tclMultiLanguageString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const spm_fi_tclMultiLanguageString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(spm_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   spm_fi_tclMultiLanguageString& operator=(tCString coszString); // ADDED
   spm_fi_tclMultiLanguageString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8
   };
   virtual tS32 s32GetTypeId() const { return spm_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const spm_fi_tclMultiLanguageString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};
class spm_fi_tcl_BinContainer;
class spm_fi_tcl_Reserved;
class spm_fi_tcl_u8_Percent;
class spm_fi_tcl_s16_PerThousand;
class spm_fi_tcl_u16_IndexInReferenceList;
class spm_fi_tcl_u16_ErrorCode;
class spm_fi_tcl_u16_RPM;
class spm_fi_tcl_s32_Kilogram;
class spm_fi_tcl_s32_Pound;
class spm_fi_tcl_s16_Meter;
class spm_fi_tcl_s32_Meter;
class spm_fi_tcl_s32_Yards;
class spm_fi_tcl_s32_Foot;
class spm_fi_tcl_s32_Centimeter;
class spm_fi_tcl_s32_Inch;
class spm_fi_tcl_s32_Millimeter;
class spm_fi_tcl_s32_Microliter;
class spm_fi_tcl_s32_Second;
class spm_fi_tcl_u16_MilliSecond;
class spm_fi_tcl_s32_MilliSecond;
class spm_fi_tcl_POSIXTime;
class spm_fi_tcl_e8_DST_State;
class spm_fi_tcl_TimeZoneElement;
class spm_fi_tcl_s16_KmPerHour;
class spm_fi_tcl_s16_MilesPerHour;
class spm_fi_tcl_s16_Knot;
class spm_fi_tcl_s16_CentimeterSecond;
class spm_fi_tcl_s16_Kelvin;
class spm_fi_tcl_s16_Fahrenheit;
class spm_fi_tcl_s32_Pascal;
class spm_fi_tcl_s32_Watt;
class spm_fi_tcl_FIVersion;
class spm_fi_tcl_s32_Longitude;
class spm_fi_tcl_s32_Latitude;
class spm_fi_tcl_PositionWGS84;
class spm_fi_tcl_RectangleWGS84;
class spm_fi_tcl_e16_ISOCountryCode;
class spm_fi_tcl_e16_ISOLanguageCode;
class spm_fi_tcl_LanguageID;
class spm_fi_tcl_u8_DirectionAbsolute;
class spm_fi_tcl_u8_DirectionRelative;
class spm_fi_tcl_s16_CentiDegree;
class spm_fi_tcl_s16_Degree;
class spm_fi_tcl_u16_DirectionAbsolute_CentiDegree;
class spm_fi_tcl_s16_DirectionRelative_CentiDegree;
class spm_fi_tcl_e8_FuelType;
class spm_fi_tcl_DistTimeFuel;
class spm_fi_tclTimeDate;
class spm_fi_tclSwVersionInfo;
class spm_fi_tcl_b32_SubState0;
class spm_fi_tcl_b32_SubState1;
class spm_fi_tcl_b32_SubState2;
class spm_fi_tcl_b32_SubState3;
class spm_fi_tcl_SPM_e32_SubStateType;
class spm_fi_tcl_SPM_e32_OnOffReason;
class spm_fi_tcl_SPM_e32_SYSTEM_STATES;
class spm_fi_tcl_SPM_e32_SYSPWR_BEHAVIOUR;
class spm_fi_tcl_SPM_e32_HMI_STATE;
class spm_fi_tcl_SPM_e32_DIM_STATE;
class spm_fi_tcl_SPM_b32_DIM_STATE;
class spm_fi_tcl_SPM_e32_BT_STATE;
class spm_fi_tcl_SPM_e32_WIFI_STATE;
class spm_fi_tcl_SPM_e32_AUDIO_STATE;
class spm_fi_tcl_SPM_e32_SPI_STATE;
class spm_fi_tcl_SPM_e32_TUNER_STATE;
class spm_fi_tcl_SPM_e32_AUDIO_SOURCE;
class spm_fi_tcl_SPM_e32_CD_INSERT_STATE;
class spm_fi_tcl_SPM_e32_SYS_STATE;
class spm_fi_tcl_SPM_e32_MOST_EVENT;
class spm_fi_tcl_SPM_e32_MOST_CONFIG_STATE;
class spm_fi_tcl_SPM_e32_TLA_STATE;
class spm_fi_tcl_SPM_e32_PROCESSOR_STATE;
class spm_fi_tcl_SPM_e32_STARTUP_STATE;
class spm_fi_tcl_SPM_HistoryTrigger;
class spm_fi_tcl_SPM_e32_RESTART_TYPE;
class spm_fi_tcl_SPM_e32_CODE_REQUEST;
class spm_fi_tcl_SPM_e32_WAKEUP_REASON;
class spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE;
class spm_fi_tcl_SPM_e32_KEY_STATE;
class spm_fi_tcl_SPM_e32_SECURITY_STATE;
class spm_fi_tcl_SPM_e32_STARTUP_REASON;
class spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE;
class spm_fi_tcl_e8_ResetReason;
class spm_fi_tcl_e8_LcmResetReason;
class spm_fi_tcl_SPM_e32_CVM_EVENT;
class spm_fi_tcl_SPM_e32_SYSPWRMODE;
class spm_fi_tcl_SPM_e8_WAKEUP_LIST_REASON;
class spm_fi_tcl_SPM_TransPortMode;
class spm_fi_tcl_SPM_TransPortModeList;
class spm_fi_tcl_e8_StateModes;
class spm_fi_tcl_e8_StateEntryType;
class spm_fi_tcl_SubStateEntry;
class spm_fi_tcl_e8_DownloadLockStates;
class spm_fi_tcl_e8_ProcessState;
class spm_fi_tcl_ProcessElement;
class spm_fi_tcl_SPM_ProcessInfo;
class spm_fi_tcl_SPM_Vin;
class spm_fi_tcl_SPM_VinHistory;
class spm_fi_tcl_ResetElement;
class spm_fi_tcl_WakeupElement;
class spm_fi_tcl_StateChangeElement;
class spm_fi_tcl_SwitchOnElement;
class spm_fi_tcl_SPM_e32_START_PROC_TYPE;
class spm_fi_tcl_SPM_e32_MRESULT;
class spm_fi_tcl_SPM_e32_RUNLEVEL;
class spm_fi_tcl_SPM_b32_SYSPWRSAVEMODE;
class spm_fi_tcl_SPM_PartnumberItem;
class spm_fi_tcl_SPM_PartnumberList;
class spm_fi_tcl_SPM_AlphacodeItem;
class spm_fi_tcl_SPM_AlphacodeList;
class spm_fi_tcl_SPM_e32_REDUCTION_LEVEL;
class spm_fi_tcl_SPM_e32_CLEAR_LIST;
class spm_fi_tcl_SPM_AppIdList;
class spm_fi_tcl_SPM_RestrictedDlLock;
class spm_fi_tcl_e8_UpdateLockCommand;
class spm_fi_tcl_e8_UpdateLockState;
class spm_fi_tcl_SPM_UpdateLockState;
class spm_fi_tcl_SPM_VoltageEventCounter;
class spm_fi_tcl_SPM_VoltageRange;
class spm_fi_tcl_SPM_SystemStatisticsData;
class spm_fi_tcl_SPM_ResetInfoStatisticData;
class spm_fi_tcl_SPM_SWVersionStatisticData;
class spm_fi_tcl_e8_DefSetMode;
class spm_fi_tcl_e8_DefSetServiceStatus;
class spm_fi_tcl_e8_DiaglogDefsetType;
class spm_fi_tcl_e8_DefSetGroup;
class spm_fi_tcl_BinContainer : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_BinContainer();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_BinContainer& operator=(const spm_fi_tcl_BinContainer& coRef);
   spm_fi_tcl_BinContainer(const spm_fi_tcl_BinContainer& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<tU8, std::allocator<tU8> > ContainerDataList;
   virtual ~spm_fi_tcl_BinContainer();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_BinContainer& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_Reserved : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_Reserved();

   virtual ~spm_fi_tcl_Reserved();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_Reserved& /* roRef */) const
      {  return true; }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u8_Percent : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u8_Percent(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~spm_fi_tcl_u8_Percent();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u8_Percent& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_PerThousand : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_PerThousand(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_PerThousand();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_PerThousand& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u16_IndexInReferenceList : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u16_IndexInReferenceList(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~spm_fi_tcl_u16_IndexInReferenceList();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u16_IndexInReferenceList& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u16_ErrorCode : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u16_ErrorCode(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~spm_fi_tcl_u16_ErrorCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u16_ErrorCode& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u16_RPM : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u16_RPM(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~spm_fi_tcl_u16_RPM();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u16_RPM& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Kilogram : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Kilogram(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Kilogram();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Kilogram& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Pound : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Pound(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Pound();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Pound& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_Meter : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_Meter(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_Meter& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Meter : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Meter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Meter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Yards : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Yards(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Yards();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Yards& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Foot : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Foot(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Foot();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Foot& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Centimeter : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Centimeter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Centimeter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Centimeter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Inch : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Inch(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Inch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Inch& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Millimeter : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Millimeter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Millimeter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Millimeter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Microliter : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Microliter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Microliter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Microliter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Second : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Second(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Second();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Second& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u16_MilliSecond : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u16_MilliSecond(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~spm_fi_tcl_u16_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u16_MilliSecond& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_MilliSecond : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_MilliSecond(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_MilliSecond& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_POSIXTime : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_POSIXTime();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_POSIXTime& operator=(const spm_fi_tcl_POSIXTime& coRef);
   spm_fi_tcl_POSIXTime(const spm_fi_tcl_POSIXTime& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS32 Time;
   virtual ~spm_fi_tcl_POSIXTime();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_POSIXTime& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_DST_State : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_DST_State();
   enum tenType {
      FI_EN_T_STANDARDTIME = 0UL,
      FI_EN_T_DAYLIGHTSAVINGTIME = 1UL,
      FI_EN_T_EVALUATIONNECESSARY = 2UL
   };
   tenType enType;

   spm_fi_tcl_e8_DST_State(spm_fi_tcl_e8_DST_State::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_DST_State();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_DST_State& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_TimeZoneElement : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_TimeZoneElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_TimeZoneElement& operator=(const spm_fi_tcl_TimeZoneElement& coRef);
   spm_fi_tcl_TimeZoneElement(const spm_fi_tcl_TimeZoneElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclMultiLanguageString Abbreviation_StandardTime;
   spm_fi_tclMultiLanguageString Abbreviation_DST_in_effect;
   spm_fi_tcl_s32_Second TimeZoneDifference;
   spm_fi_tcl_e8_DST_State DST_State;
   spm_fi_tcl_s32_Second DSTDifference;
   std::vector<tU8, std::allocator<tU8> > DST_TimeDomain;
   virtual ~spm_fi_tcl_TimeZoneElement();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_TimeZoneElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_KmPerHour : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_KmPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_KmPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_KmPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_MilesPerHour : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_MilesPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_MilesPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_MilesPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_Knot : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_Knot(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_Knot();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_Knot& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_CentimeterSecond : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_CentimeterSecond(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_CentimeterSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_CentimeterSecond& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_Kelvin : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_Kelvin(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_Kelvin();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_Kelvin& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_Fahrenheit : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_Fahrenheit(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_Fahrenheit();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_Fahrenheit& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Pascal : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Pascal(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Pascal();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Pascal& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Watt : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Watt(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Watt();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Watt& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_FIVersion : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_FIVersion();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_FIVersion& operator=(const spm_fi_tcl_FIVersion& coRef);
   spm_fi_tcl_FIVersion(const spm_fi_tcl_FIVersion& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 MajorVersion;
   tU8 MinorVersion;
   tU8 PatchVersion;
   virtual ~spm_fi_tcl_FIVersion();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_FIVersion& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Longitude : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Longitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Longitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Longitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s32_Latitude : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s32_Latitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~spm_fi_tcl_s32_Latitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s32_Latitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_PositionWGS84 : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_PositionWGS84();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_PositionWGS84& operator=(const spm_fi_tcl_PositionWGS84& coRef);
   spm_fi_tcl_PositionWGS84(const spm_fi_tcl_PositionWGS84& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tcl_s32_Longitude Longitude;
   spm_fi_tcl_s32_Latitude Latitude;
   virtual ~spm_fi_tcl_PositionWGS84();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_PositionWGS84& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_RectangleWGS84 : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_RectangleWGS84();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_RectangleWGS84& operator=(const spm_fi_tcl_RectangleWGS84& coRef);
   spm_fi_tcl_RectangleWGS84(const spm_fi_tcl_RectangleWGS84& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tcl_s32_Longitude LongitudeMin;
   spm_fi_tcl_s32_Latitude LatitudeMin;
   spm_fi_tcl_s32_Longitude LongitudeMax;
   spm_fi_tcl_s32_Latitude LatitudeMax;
   virtual ~spm_fi_tcl_RectangleWGS84();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_RectangleWGS84& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e16_ISOCountryCode : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e16_ISOCountryCode();
   enum tenType {
      FI_EN_ISO_ALPHA_3_UNDEFINED = 0UL,
      FI_EN_ISO_ALPHA_3_AFG = 1223UL,
      FI_EN_ISO_ALPHA_3_ALB = 1410UL,
      FI_EN_ISO_ALPHA_3_DZA = 4929UL,
      FI_EN_ISO_ALPHA_3_ASM = 1645UL,
      FI_EN_ISO_ALPHA_3_AND = 1476UL,
      FI_EN_ISO_ALPHA_3_AGO = 1263UL,
      FI_EN_ISO_ALPHA_3_AIA = 1313UL,
      FI_EN_ISO_ALPHA_3_ATG = 1671UL,
      FI_EN_ISO_ALPHA_3_ARG = 1607UL,
      FI_EN_ISO_ALPHA_3_ARM = 1613UL,
      FI_EN_ISO_ALPHA_3_ABW = 1111UL,
      FI_EN_ISO_ALPHA_3_AUS = 1715UL,
      FI_EN_ISO_ALPHA_3_AUT = 1716UL,
      FI_EN_ISO_ALPHA_3_AZE = 1861UL,
      FI_EN_ISO_ALPHA_3_BHS = 2323UL,
      FI_EN_ISO_ALPHA_3_BHR = 2322UL,
      FI_EN_ISO_ALPHA_3_BGD = 2276UL,
      FI_EN_ISO_ALPHA_3_BRB = 2626UL,
      FI_EN_ISO_ALPHA_3_BLR = 2450UL,
      FI_EN_ISO_ALPHA_3_BEL = 2220UL,
      FI_EN_ISO_ALPHA_3_BLZ = 2458UL,
      FI_EN_ISO_ALPHA_3_BEN = 2222UL,
      FI_EN_ISO_ALPHA_3_BMU = 2485UL,
      FI_EN_ISO_ALPHA_3_BTN = 2702UL,
      FI_EN_ISO_ALPHA_3_BOL = 2540UL,
      FI_EN_ISO_ALPHA_3_BIH = 2344UL,
      FI_EN_ISO_ALPHA_3_BWA = 2785UL,
      FI_EN_ISO_ALPHA_3_BRA = 2625UL,
      FI_EN_ISO_ALPHA_3_BRN = 2638UL,
      FI_EN_ISO_ALPHA_3_BGR = 2290UL,
      FI_EN_ISO_ALPHA_3_BFA = 2241UL,
      FI_EN_ISO_ALPHA_3_BDI = 2185UL,
      FI_EN_ISO_ALPHA_3_KHM = 11533UL,
      FI_EN_ISO_ALPHA_3_CMR = 3506UL,
      FI_EN_ISO_ALPHA_3_CAN = 3118UL,
      FI_EN_ISO_ALPHA_3_CPV = 3606UL,
      FI_EN_ISO_ALPHA_3_CYM = 3885UL,
      FI_EN_ISO_ALPHA_3_CAF = 3110UL,
      FI_EN_ISO_ALPHA_3_TCD = 20580UL,
      FI_EN_ISO_ALPHA_3_CHL = 3340UL,
      FI_EN_ISO_ALPHA_3_CHN = 3342UL,
      FI_EN_ISO_ALPHA_3_COL = 3564UL,
      FI_EN_ISO_ALPHA_3_COM = 3565UL,
      FI_EN_ISO_ALPHA_3_COG = 3559UL,
      FI_EN_ISO_ALPHA_3_COD = 3556UL,
      FI_EN_ISO_ALPHA_3_COK = 3563UL,
      FI_EN_ISO_ALPHA_3_CRI = 3657UL,
      FI_EN_ISO_ALPHA_3_CIV = 3382UL,
      FI_EN_ISO_ALPHA_3_HRV = 8790UL,
      FI_EN_ISO_ALPHA_3_CUB = 3746UL,
      FI_EN_ISO_ALPHA_3_CYP = 3888UL,
      FI_EN_ISO_ALPHA_3_CZE = 3909UL,
      FI_EN_ISO_ALPHA_3_DNK = 4555UL,
      FI_EN_ISO_ALPHA_3_DJI = 4425UL,
      FI_EN_ISO_ALPHA_3_DMA = 4513UL,
      FI_EN_ISO_ALPHA_3_DOM = 4589UL,
      FI_EN_ISO_ALPHA_3_ECU = 5237UL,
      FI_EN_ISO_ALPHA_3_EGY = 5369UL,
      FI_EN_ISO_ALPHA_3_SLV = 19862UL,
      FI_EN_ISO_ALPHA_3_GNQ = 7633UL,
      FI_EN_ISO_ALPHA_3_ERI = 5705UL,
      FI_EN_ISO_ALPHA_3_EST = 5748UL,
      FI_EN_ISO_ALPHA_3_ETH = 5768UL,
      FI_EN_ISO_ALPHA_3_FLK = 6539UL,
      FI_EN_ISO_ALPHA_3_FRO = 6735UL,
      FI_EN_ISO_ALPHA_3_FJI = 6473UL,
      FI_EN_ISO_ALPHA_3_FIN = 6446UL,
      FI_EN_ISO_ALPHA_3_FRA = 6721UL,
      FI_EN_ISO_ALPHA_3_GUF = 7846UL,
      FI_EN_ISO_ALPHA_3_PYF = 17190UL,
      FI_EN_ISO_ALPHA_3_GAB = 7202UL,
      FI_EN_ISO_ALPHA_3_GMB = 7586UL,
      FI_EN_ISO_ALPHA_3_GEO = 7343UL,
      FI_EN_ISO_ALPHA_3_DEU = 4277UL,
      FI_EN_ISO_ALPHA_3_GHA = 7425UL,
      FI_EN_ISO_ALPHA_3_GIB = 7458UL,
      FI_EN_ISO_ALPHA_3_GRC = 7747UL,
      FI_EN_ISO_ALPHA_3_GRL = 7756UL,
      FI_EN_ISO_ALPHA_3_GRD = 7748UL,
      FI_EN_ISO_ALPHA_3_GLP = 7568UL,
      FI_EN_ISO_ALPHA_3_GUM = 7853UL,
      FI_EN_ISO_ALPHA_3_GTM = 7821UL,
      FI_EN_ISO_ALPHA_3_GIN = 7470UL,
      FI_EN_ISO_ALPHA_3_GNB = 7618UL,
      FI_EN_ISO_ALPHA_3_GUY = 7865UL,
      FI_EN_ISO_ALPHA_3_HTI = 8841UL,
      FI_EN_ISO_ALPHA_3_VAT = 22580UL,
      FI_EN_ISO_ALPHA_3_HND = 8644UL,
      FI_EN_ISO_ALPHA_3_HKG = 8551UL,
      FI_EN_ISO_ALPHA_3_HUN = 8878UL,
      FI_EN_ISO_ALPHA_3_ISL = 9836UL,
      FI_EN_ISO_ALPHA_3_IND = 9668UL,
      FI_EN_ISO_ALPHA_3_IDN = 9358UL,
      FI_EN_ISO_ALPHA_3_INT = 9684UL,
      FI_EN_ISO_ALPHA_3_IRN = 9806UL,
      FI_EN_ISO_ALPHA_3_IRQ = 9809UL,
      FI_EN_ISO_ALPHA_3_IRL = 9804UL,
      FI_EN_ISO_ALPHA_3_ISR = 9842UL,
      FI_EN_ISO_ALPHA_3_ITA = 9857UL,
      FI_EN_ISO_ALPHA_3_JAM = 10285UL,
      FI_EN_ISO_ALPHA_3_JPN = 10766UL,
      FI_EN_ISO_ALPHA_3_JOR = 10738UL,
      FI_EN_ISO_ALPHA_3_KAZ = 11322UL,
      FI_EN_ISO_ALPHA_3_KEN = 11438UL,
      FI_EN_ISO_ALPHA_3_KIR = 11570UL,
      FI_EN_ISO_ALPHA_3_PRK = 16971UL,
      FI_EN_ISO_ALPHA_3_KOR = 11762UL,
      FI_EN_ISO_ALPHA_3_KWT = 12020UL,
      FI_EN_ISO_ALPHA_3_KGZ = 11514UL,
      FI_EN_ISO_ALPHA_3_LAO = 12335UL,
      FI_EN_ISO_ALPHA_3_LVA = 12993UL,
      FI_EN_ISO_ALPHA_3_LBN = 12366UL,
      FI_EN_ISO_ALPHA_3_LSO = 12911UL,
      FI_EN_ISO_ALPHA_3_LBR = 12370UL,
      FI_EN_ISO_ALPHA_3_LBY = 12377UL,
      FI_EN_ISO_ALPHA_3_LIE = 12581UL,
      FI_EN_ISO_ALPHA_3_LTU = 12949UL,
      FI_EN_ISO_ALPHA_3_LUX = 12984UL,
      FI_EN_ISO_ALPHA_3_MAC = 13347UL,
      FI_EN_ISO_ALPHA_3_MKD = 13668UL,
      FI_EN_ISO_ALPHA_3_MDG = 13447UL,
      FI_EN_ISO_ALPHA_3_MWI = 14057UL,
      FI_EN_ISO_ALPHA_3_MYS = 14131UL,
      FI_EN_ISO_ALPHA_3_MDV = 13462UL,
      FI_EN_ISO_ALPHA_3_MLI = 13705UL,
      FI_EN_ISO_ALPHA_3_MLT = 13716UL,
      FI_EN_ISO_ALPHA_3_MHL = 13580UL,
      FI_EN_ISO_ALPHA_3_MTQ = 13969UL,
      FI_EN_ISO_ALPHA_3_MRT = 13908UL,
      FI_EN_ISO_ALPHA_3_MUS = 14003UL,
      FI_EN_ISO_ALPHA_3_MYT = 14132UL,
      FI_EN_ISO_ALPHA_3_MEX = 13496UL,
      FI_EN_ISO_ALPHA_3_FSM = 6765UL,
      FI_EN_ISO_ALPHA_3_MDA = 13441UL,
      FI_EN_ISO_ALPHA_3_MCO = 13423UL,
      FI_EN_ISO_ALPHA_3_MNG = 13767UL,
      FI_EN_ISO_ALPHA_3_MNE = 13765UL,
      FI_EN_ISO_ALPHA_3_MSR = 13938UL,
      FI_EN_ISO_ALPHA_3_MAR = 13362UL,
      FI_EN_ISO_ALPHA_3_MOZ = 13818UL,
      FI_EN_ISO_ALPHA_3_MMR = 13746UL,
      FI_EN_ISO_ALPHA_3_NAM = 14381UL,
      FI_EN_ISO_ALPHA_3_NRU = 14933UL,
      FI_EN_ISO_ALPHA_3_NPL = 14860UL,
      FI_EN_ISO_ALPHA_3_NLD = 14724UL,
      FI_EN_ISO_ALPHA_3_ANT = 1492UL,
      FI_EN_ISO_ALPHA_3_NCL = 14444UL,
      FI_EN_ISO_ALPHA_3_NZL = 15180UL,
      FI_EN_ISO_ALPHA_3_NIC = 14627UL,
      FI_EN_ISO_ALPHA_3_NER = 14514UL,
      FI_EN_ISO_ALPHA_3_NGA = 14561UL,
      FI_EN_ISO_ALPHA_3_NIU = 14645UL,
      FI_EN_ISO_ALPHA_3_NFK = 14539UL,
      FI_EN_ISO_ALPHA_3_MNP = 13776UL,
      FI_EN_ISO_ALPHA_3_NOR = 14834UL,
      FI_EN_ISO_ALPHA_3_PSE = 16997UL,
      FI_EN_ISO_ALPHA_3_OMN = 15790UL,
      FI_EN_ISO_ALPHA_3_PAK = 16427UL,
      FI_EN_ISO_ALPHA_3_PLW = 16791UL,
      FI_EN_ISO_ALPHA_3_PAN = 16430UL,
      FI_EN_ISO_ALPHA_3_PNG = 16839UL,
      FI_EN_ISO_ALPHA_3_PRY = 16985UL,
      FI_EN_ISO_ALPHA_3_PER = 16562UL,
      FI_EN_ISO_ALPHA_3_PHL = 16652UL,
      FI_EN_ISO_ALPHA_3_PCN = 16494UL,
      FI_EN_ISO_ALPHA_3_POL = 16876UL,
      FI_EN_ISO_ALPHA_3_PRT = 16980UL,
      FI_EN_ISO_ALPHA_3_PRI = 16969UL,
      FI_EN_ISO_ALPHA_3_QAT = 17460UL,
      FI_EN_ISO_ALPHA_3_REU = 18613UL,
      FI_EN_ISO_ALPHA_3_ROU = 18933UL,
      FI_EN_ISO_ALPHA_3_RUS = 19123UL,
      FI_EN_ISO_ALPHA_3_RWA = 19169UL,
      FI_EN_ISO_ALPHA_3_KNA = 11713UL,
      FI_EN_ISO_ALPHA_3_LCA = 12385UL,
      FI_EN_ISO_ALPHA_3_VCT = 22644UL,
      FI_EN_ISO_ALPHA_3_WSM = 24173UL,
      FI_EN_ISO_ALPHA_3_SMR = 19890UL,
      FI_EN_ISO_ALPHA_3_STP = 20112UL,
      FI_EN_ISO_ALPHA_3_SAU = 19509UL,
      FI_EN_ISO_ALPHA_3_SEN = 19630UL,
      FI_EN_ISO_ALPHA_3_SRB = 20034UL,
      FI_EN_ISO_ALPHA_3_SYC = 20259UL,
      FI_EN_ISO_ALPHA_3_SLE = 19845UL,
      FI_EN_ISO_ALPHA_3_SGP = 19696UL,
      FI_EN_ISO_ALPHA_3_SVK = 20171UL,
      FI_EN_ISO_ALPHA_3_SVN = 20174UL,
      FI_EN_ISO_ALPHA_3_SLB = 19842UL,
      FI_EN_ISO_ALPHA_3_SOM = 19949UL,
      FI_EN_ISO_ALPHA_3_ZAF = 26662UL,
      FI_EN_ISO_ALPHA_3_ESP = 5744UL,
      FI_EN_ISO_ALPHA_3_LKA = 12641UL,
      FI_EN_ISO_ALPHA_3_SHN = 19726UL,
      FI_EN_ISO_ALPHA_3_SPM = 19981UL,
      FI_EN_ISO_ALPHA_3_SDN = 19598UL,
      FI_EN_ISO_ALPHA_3_SUR = 20146UL,
      FI_EN_ISO_ALPHA_3_SJM = 19789UL,
      FI_EN_ISO_ALPHA_3_SWZ = 20218UL,
      FI_EN_ISO_ALPHA_3_SWE = 20197UL,
      FI_EN_ISO_ALPHA_3_CHE = 3333UL,
      FI_EN_ISO_ALPHA_3_SYR = 20274UL,
      FI_EN_ISO_ALPHA_3_TWN = 21230UL,
      FI_EN_ISO_ALPHA_3_TJK = 20811UL,
      FI_EN_ISO_ALPHA_3_TZA = 21313UL,
      FI_EN_ISO_ALPHA_3_THA = 20737UL,
      FI_EN_ISO_ALPHA_3_TLS = 20883UL,
      FI_EN_ISO_ALPHA_3_TGO = 20719UL,
      FI_EN_ISO_ALPHA_3_TKL = 20844UL,
      FI_EN_ISO_ALPHA_3_TON = 20974UL,
      FI_EN_ISO_ALPHA_3_TTO = 21135UL,
      FI_EN_ISO_ALPHA_3_TUN = 21166UL,
      FI_EN_ISO_ALPHA_3_TUR = 21170UL,
      FI_EN_ISO_ALPHA_3_TKM = 20845UL,
      FI_EN_ISO_ALPHA_3_TCA = 20577UL,
      FI_EN_ISO_ALPHA_3_TUV = 21174UL,
      FI_EN_ISO_ALPHA_3_UGA = 21729UL,
      FI_EN_ISO_ALPHA_3_UKR = 21874UL,
      FI_EN_ISO_ALPHA_3_ARE = 1605UL,
      FI_EN_ISO_ALPHA_3_GBR = 7250UL,
      FI_EN_ISO_ALPHA_3_USA = 22113UL,
      FI_EN_ISO_ALPHA_3_URY = 22105UL,
      FI_EN_ISO_ALPHA_3_UZB = 22338UL,
      FI_EN_ISO_ALPHA_3_VUT = 23220UL,
      FI_EN_ISO_ALPHA_3_VEN = 22702UL,
      FI_EN_ISO_ALPHA_3_VNM = 22989UL,
      FI_EN_ISO_ALPHA_3_VGB = 22754UL,
      FI_EN_ISO_ALPHA_3_VIR = 22834UL,
      FI_EN_ISO_ALPHA_3_WLF = 23942UL,
      FI_EN_ISO_ALPHA_3_ESH = 5736UL,
      FI_EN_ISO_ALPHA_3_YEM = 25773UL,
      FI_EN_ISO_ALPHA_3_YUG = 26279UL,
      FI_EN_ISO_ALPHA_3_ZMB = 27042UL,
      FI_EN_ISO_ALPHA_3_ZWE = 27365UL
   };
   tenType enType;

   spm_fi_tcl_e16_ISOCountryCode(spm_fi_tcl_e16_ISOCountryCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e16_ISOCountryCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e16_ISOCountryCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e16_ISOLanguageCode : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e16_ISOLanguageCode();
   enum tenType {
      FI_EN_ISO_639_1_UNDEFINED = 0UL,
      FI_EN_ISO_639_1_AA = 24929UL,
      FI_EN_ISO_639_1_AB = 24930UL,
      FI_EN_ISO_639_1_AE = 24933UL,
      FI_EN_ISO_639_1_AF = 24934UL,
      FI_EN_ISO_639_1_AK = 24939UL,
      FI_EN_ISO_639_1_AM = 24941UL,
      FI_EN_ISO_639_1_AN = 24942UL,
      FI_EN_ISO_639_1_AR = 24946UL,
      FI_EN_ISO_639_1_AS = 24947UL,
      FI_EN_ISO_639_1_AV = 24950UL,
      FI_EN_ISO_639_1_AY = 24953UL,
      FI_EN_ISO_639_1_AZ = 24954UL,
      FI_EN_ISO_639_1_BA = 25185UL,
      FI_EN_ISO_639_1_BE = 25189UL,
      FI_EN_ISO_639_1_BG = 25191UL,
      FI_EN_ISO_639_1_BH = 25192UL,
      FI_EN_ISO_639_1_BI = 25193UL,
      FI_EN_ISO_639_1_BL = 25196UL,
      FI_EN_ISO_639_1_BM = 25197UL,
      FI_EN_ISO_639_1_BN = 25198UL,
      FI_EN_ISO_639_1_BO = 25199UL,
      FI_EN_ISO_639_1_BR = 25202UL,
      FI_EN_ISO_639_1_BS = 25203UL,
      FI_EN_ISO_639_1_BT = 25204UL,
      FI_EN_ISO_639_1_CA = 25441UL,
      FI_EN_ISO_639_1_CE = 25445UL,
      FI_EN_ISO_639_1_CF = 25446UL,
      FI_EN_ISO_639_1_CH = 25448UL,
      FI_EN_ISO_639_1_CL = 25452UL,
      FI_EN_ISO_639_1_CN = 25454UL,
      FI_EN_ISO_639_1_CO = 25455UL,
      FI_EN_ISO_639_1_CR = 25458UL,
      FI_EN_ISO_639_1_CS = 25459UL,
      FI_EN_ISO_639_1_CT = 25460UL,
      FI_EN_ISO_639_1_CU = 25461UL,
      FI_EN_ISO_639_1_CV = 25462UL,
      FI_EN_ISO_639_1_CY = 25465UL,
      FI_EN_ISO_639_1_DA = 25697UL,
      FI_EN_ISO_639_1_DE = 25701UL,
      FI_EN_ISO_639_1_DV = 25718UL,
      FI_EN_ISO_639_1_DZ = 25722UL,
      FI_EN_ISO_639_1_EE = 25957UL,
      FI_EN_ISO_639_1_EG = 25959UL,
      FI_EN_ISO_639_1_EL = 25964UL,
      FI_EN_ISO_639_1_EN = 25966UL,
      FI_EN_ISO_639_1_EO = 25967UL,
      FI_EN_ISO_639_1_ES = 25971UL,
      FI_EN_ISO_639_1_ET = 25972UL,
      FI_EN_ISO_639_1_EU = 25973UL,
      FI_EN_ISO_639_1_FA = 26209UL,
      FI_EN_ISO_639_1_FF = 26214UL,
      FI_EN_ISO_639_1_FI = 26217UL,
      FI_EN_ISO_639_1_FJ = 26218UL,
      FI_EN_ISO_639_1_FO = 26223UL,
      FI_EN_ISO_639_1_FR = 26226UL,
      FI_EN_ISO_639_1_FY = 26233UL,
      FI_EN_ISO_639_1_GA = 26465UL,
      FI_EN_ISO_639_1_GD = 26468UL,
      FI_EN_ISO_639_1_GL = 26476UL,
      FI_EN_ISO_639_1_GN = 26478UL,
      FI_EN_ISO_639_1_GU = 26485UL,
      FI_EN_ISO_639_1_GV = 26486UL,
      FI_EN_ISO_639_1_HA = 26721UL,
      FI_EN_ISO_639_1_HE = 26725UL,
      FI_EN_ISO_639_1_HI = 26729UL,
      FI_EN_ISO_639_1_HO = 26735UL,
      FI_EN_ISO_639_1_HR = 26738UL,
      FI_EN_ISO_639_1_HU = 26741UL,
      FI_EN_ISO_639_1_HY = 26745UL,
      FI_EN_ISO_639_1_HZ = 26746UL,
      FI_EN_ISO_639_1_IA = 26977UL,
      FI_EN_ISO_639_1_ID = 26980UL,
      FI_EN_ISO_639_1_IE = 26981UL,
      FI_EN_ISO_639_1_IG = 26983UL,
      FI_EN_ISO_639_1_II = 26985UL,
      FI_EN_ISO_639_1_IK = 26987UL,
      FI_EN_ISO_639_1_IO = 26991UL,
      FI_EN_ISO_639_1_IS = 26995UL,
      FI_EN_ISO_639_1_IT = 26996UL,
      FI_EN_ISO_639_1_IU = 26997UL,
      FI_EN_ISO_639_1_JA = 27233UL,
      FI_EN_ISO_639_1_JV = 27254UL,
      FI_EN_ISO_639_1_KA = 27489UL,
      FI_EN_ISO_639_1_KG = 27495UL,
      FI_EN_ISO_639_1_KI = 27497UL,
      FI_EN_ISO_639_1_KJ = 27498UL,
      FI_EN_ISO_639_1_KK = 27499UL,
      FI_EN_ISO_639_1_KL = 27500UL,
      FI_EN_ISO_639_1_KM = 27501UL,
      FI_EN_ISO_639_1_KN = 27502UL,
      FI_EN_ISO_639_1_KO = 27503UL,
      FI_EN_ISO_639_1_KR = 27506UL,
      FI_EN_ISO_639_1_KS = 27507UL,
      FI_EN_ISO_639_1_KU = 27509UL,
      FI_EN_ISO_639_1_KV = 27510UL,
      FI_EN_ISO_639_1_KW = 27511UL,
      FI_EN_ISO_639_1_KY = 27513UL,
      FI_EN_ISO_639_1_LA = 27745UL,
      FI_EN_ISO_639_1_LB = 27746UL,
      FI_EN_ISO_639_1_LG = 27751UL,
      FI_EN_ISO_639_1_LI = 27753UL,
      FI_EN_ISO_639_1_LN = 27758UL,
      FI_EN_ISO_639_1_LO = 27759UL,
      FI_EN_ISO_639_1_LT = 27764UL,
      FI_EN_ISO_639_1_LU = 27765UL,
      FI_EN_ISO_639_1_LV = 27766UL,
      FI_EN_ISO_639_1_MC = 28003UL,
      FI_EN_ISO_639_1_MG = 28007UL,
      FI_EN_ISO_639_1_MH = 28008UL,
      FI_EN_ISO_639_1_MI = 28009UL,
      FI_EN_ISO_639_1_MK = 28011UL,
      FI_EN_ISO_639_1_ML = 28012UL,
      FI_EN_ISO_639_1_MN = 28014UL,
      FI_EN_ISO_639_1_MO = 28015UL,
      FI_EN_ISO_639_1_MR = 28018UL,
      FI_EN_ISO_639_1_MS = 28019UL,
      FI_EN_ISO_639_1_MT = 28020UL,
      FI_EN_ISO_639_1_MY = 28025UL,
      FI_EN_ISO_639_1_NA = 28257UL,
      FI_EN_ISO_639_1_NB = 28258UL,
      FI_EN_ISO_639_1_ND = 28260UL,
      FI_EN_ISO_639_1_NE = 28261UL,
      FI_EN_ISO_639_1_NG = 28263UL,
      FI_EN_ISO_639_1_NL = 28268UL,
      FI_EN_ISO_639_1_NN = 28270UL,
      FI_EN_ISO_639_1_NO = 28271UL,
      FI_EN_ISO_639_1_NR = 28274UL,
      FI_EN_ISO_639_1_NV = 28278UL,
      FI_EN_ISO_639_1_NY = 28281UL,
      FI_EN_ISO_639_1_OC = 28515UL,
      FI_EN_ISO_639_1_OJ = 28522UL,
      FI_EN_ISO_639_1_OM = 28525UL,
      FI_EN_ISO_639_1_OR = 28530UL,
      FI_EN_ISO_639_1_OS = 28531UL,
      FI_EN_ISO_639_1_PA = 28769UL,
      FI_EN_ISO_639_1_PI = 28777UL,
      FI_EN_ISO_639_1_PL = 28780UL,
      FI_EN_ISO_639_1_PS = 28787UL,
      FI_EN_ISO_639_1_PT = 28788UL,
      FI_EN_ISO_639_1_QU = 29045UL,
      FI_EN_ISO_639_1_RM = 29293UL,
      FI_EN_ISO_639_1_RN = 29294UL,
      FI_EN_ISO_639_1_RO = 29295UL,
      FI_EN_ISO_639_1_RT = 29300UL,
      FI_EN_ISO_639_1_RU = 29301UL,
      FI_EN_ISO_639_1_RW = 29303UL,
      FI_EN_ISO_639_1_SA = 29537UL,
      FI_EN_ISO_639_1_SB = 29538UL,
      FI_EN_ISO_639_1_SC = 29539UL,
      FI_EN_ISO_639_1_SD = 29540UL,
      FI_EN_ISO_639_1_SE = 29541UL,
      FI_EN_ISO_639_1_SG = 29543UL,
      FI_EN_ISO_639_1_SH = 29544UL,
      FI_EN_ISO_639_1_SI = 29545UL,
      FI_EN_ISO_639_1_SK = 29547UL,
      FI_EN_ISO_639_1_SL = 29548UL,
      FI_EN_ISO_639_1_SM = 29549UL,
      FI_EN_ISO_639_1_SN = 29550UL,
      FI_EN_ISO_639_1_SO = 29551UL,
      FI_EN_ISO_639_1_SQ = 29553UL,
      FI_EN_ISO_639_1_SR = 29554UL,
      FI_EN_ISO_639_1_SS = 29555UL,
      FI_EN_ISO_639_1_ST = 29556UL,
      FI_EN_ISO_639_1_SU = 29557UL,
      FI_EN_ISO_639_1_SV = 29558UL,
      FI_EN_ISO_639_1_SW = 29559UL,
      FI_EN_ISO_639_1_SX = 29560UL,
      FI_EN_ISO_639_1_TA = 29793UL,
      FI_EN_ISO_639_1_TE = 29797UL,
      FI_EN_ISO_639_1_TG = 29799UL,
      FI_EN_ISO_639_1_TH = 29800UL,
      FI_EN_ISO_639_1_TI = 29801UL,
      FI_EN_ISO_639_1_TK = 29803UL,
      FI_EN_ISO_639_1_TL = 29804UL,
      FI_EN_ISO_639_1_TN = 29806UL,
      FI_EN_ISO_639_1_TO = 29807UL,
      FI_EN_ISO_639_1_TR = 29810UL,
      FI_EN_ISO_639_1_TS = 29811UL,
      FI_EN_ISO_639_1_TT = 29812UL,
      FI_EN_ISO_639_1_TW = 29815UL,
      FI_EN_ISO_639_1_TY = 29817UL,
      FI_EN_ISO_639_1_UE = 30053UL,
      FI_EN_ISO_639_1_UG = 30055UL,
      FI_EN_ISO_639_1_UK = 30059UL,
      FI_EN_ISO_639_1_UL = 30060UL,
      FI_EN_ISO_639_1_UR = 30066UL,
      FI_EN_ISO_639_1_UZ = 30074UL,
      FI_EN_ISO_639_1_VA = 30305UL,
      FI_EN_ISO_639_1_VE = 30309UL,
      FI_EN_ISO_639_1_VI = 30313UL,
      FI_EN_ISO_639_1_VO = 30319UL,
      FI_EN_ISO_639_1_WA = 30561UL,
      FI_EN_ISO_639_1_WO = 30575UL,
      FI_EN_ISO_639_1_XH = 30824UL,
      FI_EN_ISO_639_1_YI = 31081UL,
      FI_EN_ISO_639_1_YO = 31087UL,
      FI_EN_ISO_639_1_ZA = 31329UL,
      FI_EN_ISO_639_1_ZH = 31336UL,
      FI_EN_ISO_639_1_ZL = 31340UL,
      FI_EN_ISO_639_1_ZU = 31349UL
   };
   tenType enType;

   spm_fi_tcl_e16_ISOLanguageCode(spm_fi_tcl_e16_ISOLanguageCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e16_ISOLanguageCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e16_ISOLanguageCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_LanguageID : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_LanguageID();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_LanguageID& operator=(const spm_fi_tcl_LanguageID& coRef);
   spm_fi_tcl_LanguageID(const spm_fi_tcl_LanguageID& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tcl_e16_ISOLanguageCode ISO639_LanguageCode;
   spm_fi_tcl_e16_ISOCountryCode ISO3166_CountryCode;
   virtual ~spm_fi_tcl_LanguageID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_LanguageID& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u8_DirectionAbsolute : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u8_DirectionAbsolute(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~spm_fi_tcl_u8_DirectionAbsolute();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u8_DirectionAbsolute& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u8_DirectionRelative : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u8_DirectionRelative(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~spm_fi_tcl_u8_DirectionRelative();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u8_DirectionRelative& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_CentiDegree : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_Degree : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_Degree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_Degree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_Degree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_u16_DirectionAbsolute_CentiDegree : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_u16_DirectionAbsolute_CentiDegree(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~spm_fi_tcl_u16_DirectionAbsolute_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_u16_DirectionAbsolute_CentiDegree& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_s16_DirectionRelative_CentiDegree : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_s16_DirectionRelative_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~spm_fi_tcl_s16_DirectionRelative_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_s16_DirectionRelative_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_FuelType : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_FuelType();
   enum tenType {
      FI_EN_RESERVED = 0UL,
      FI_EN_PETROL = 1UL,
      FI_EN_DIESEL = 2UL,
      FI_EN_HYDROGEN = 3UL,
      FI_EN_COMPRESSEDNATURALGAS = 4UL,
      FI_EN_LIQUIFIEDPETROLEUMGAS = 5UL
   };
   tenType enType;

   spm_fi_tcl_e8_FuelType(spm_fi_tcl_e8_FuelType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_FuelType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_FuelType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_DistTimeFuel : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_DistTimeFuel();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_DistTimeFuel& operator=(const spm_fi_tcl_DistTimeFuel& coRef);
   spm_fi_tcl_DistTimeFuel(const spm_fi_tcl_DistTimeFuel& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tcl_s32_Meter Distance;
   spm_fi_tcl_s32_Second Time;
   spm_fi_tcl_s32_Microliter Fuel;
   virtual ~spm_fi_tcl_DistTimeFuel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_DistTimeFuel& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tclTimeDate : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tclTimeDate();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tclTimeDate& operator=(const spm_fi_tclTimeDate& coRef);
   spm_fi_tclTimeDate(const spm_fi_tclTimeDate& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS16 s16Year;
   tU8 u8Month;
   tU8 u8Day;
   tU8 u8Hour;
   tU8 u8Minute;
   tU8 u8Second;
   virtual ~spm_fi_tclTimeDate();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tclTimeDate& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tclSwVersionInfo : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tclSwVersionInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tclSwVersionInfo& operator=(const spm_fi_tclSwVersionInfo& coRef);
   spm_fi_tclSwVersionInfo(const spm_fi_tclSwVersionInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclString tSwBuildVersion;
   spm_fi_tclString tSwCustomerVersion;
   virtual ~spm_fi_tclSwVersionInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tclSwVersionInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_b32_SubState0 : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_b32_SubState0();

   enum {
      FI_C_U32_BIT_SPM_E32_SET_SD_CARD_ACCESS = 0x1UL,
      FI_C_U32_BIT_SPM_E32_EMERGENCY_OFF = 0x2UL,
      FI_C_U32_BIT_SPM_E32_FAST_SHUTDOWN = 0x4UL,
      FI_C_U32_BIT_SPM_E32_DIMMING = 0x8UL,
      FI_C_U32_BIT_SPM_E32_DOWNLOAD = 0x10UL,
      FI_C_U32_BIT_SPM_E32_DIAG_ACTIVITY = 0x20UL,
      FI_C_U32_BIT_SPM_E32_DIAGNOSIS = 0x40UL,
      FI_C_U32_BIT_SPM_E32_PHONE = 0x80UL,
      FI_C_U32_BIT_SPM_E32_ACCESSORY = 0x100UL,
      FI_C_U32_BIT_SPM_E32_IGNITION = 0x200UL,
      FI_C_U32_BIT_SPM_E32_TRANSPORTMODE = 0x400UL,
      FI_C_U32_BIT_SPM_E32_DOOROPEN = 0x800UL,
      FI_C_U32_BIT_SPM_E32_DOORLOCK = 0x1000UL,
      FI_C_U32_BIT_SPM_E32_TRIGGER_FORCE_ON = 0x2000UL,
      FI_C_U32_BIT_SPM_E32_CLIMATE_INFO = 0x4000UL,
      FI_C_U32_BIT_SPM_E32_PARK_DISTANCE_CTRL = 0x8000UL,
      FI_C_U32_BIT_SPM_E32_FACTORY_MODE = 0x10000UL,
      FI_C_U32_BIT_SPM_E32_CD_INSERT = 0x20000UL,
      FI_C_U32_BIT_SPM_E32_SD_INSERT = 0x40000UL,
      FI_C_U32_BIT_SPM_E32_EJECT = 0x80000UL,
      FI_C_U32_BIT_SPM_E32_ON_TIPPER = 0x100000UL,
      FI_C_U32_BIT_SPM_E32_SOFTTOP_OPEN = 0x200000UL,
      FI_C_U32_BIT_SPM_E32_AIRBAG_DEPLOYED = 0x400000UL,
      FI_C_U32_BIT_SPM_E32_KEY_PRESENT = 0x800000UL,
      FI_C_U32_BIT_SPM_E32_NETWORK_ACTIVITY = 0x1000000UL,
      FI_C_U32_BIT_SPM_E32_OFF_TIPPER = 0x2000000UL,
      FI_C_U32_BIT_SPM_E32_OFF_LASTSTATE_CYCLE = 0x4000000UL,
      FI_C_U32_BIT_SPM_E32_DIAG_REQUEST = 0x8000000UL,
      FI_C_U32_BIT_SPM_E32_BACKLIGHT = 0x10000000UL,
      FI_C_U32_BIT_SPM_E32_HIGH_VOLTAGE = 0x20000000UL,
      FI_C_U32_BIT_SPM_E32_RCV_CONNECTED = 0x40000000UL,
      FI_C_U32_BIT_SPM_E32_OFF_LEARN = 0x80000000UL
   };
   tBool bSPM_e32_SET_SD_CARD_ACCESS() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SET_SD_CARD_ACCESS) != 0; }
   tVoid vSetSPM_e32_SET_SD_CARD_ACCESS(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SET_SD_CARD_ACCESS):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SET_SD_CARD_ACCESS)); }
   tBool bSPM_e32_EMERGENCY_OFF() const {return (u32Value & FI_C_U32_BIT_SPM_E32_EMERGENCY_OFF) != 0; }
   tVoid vSetSPM_e32_EMERGENCY_OFF(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_EMERGENCY_OFF):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_EMERGENCY_OFF)); }
   tBool bSPM_e32_FAST_SHUTDOWN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_FAST_SHUTDOWN) != 0; }
   tVoid vSetSPM_e32_FAST_SHUTDOWN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_FAST_SHUTDOWN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_FAST_SHUTDOWN)); }
   tBool bSPM_e32_DIMMING() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DIMMING) != 0; }
   tVoid vSetSPM_e32_DIMMING(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DIMMING):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DIMMING)); }
   tBool bSPM_e32_DOWNLOAD() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DOWNLOAD) != 0; }
   tVoid vSetSPM_e32_DOWNLOAD(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DOWNLOAD):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DOWNLOAD)); }
   tBool bSPM_e32_DIAG_ACTIVITY() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DIAG_ACTIVITY) != 0; }
   tVoid vSetSPM_e32_DIAG_ACTIVITY(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DIAG_ACTIVITY):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DIAG_ACTIVITY)); }
   tBool bSPM_e32_DIAGNOSIS() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DIAGNOSIS) != 0; }
   tVoid vSetSPM_e32_DIAGNOSIS(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DIAGNOSIS):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DIAGNOSIS)); }
   tBool bSPM_e32_PHONE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_PHONE) != 0; }
   tVoid vSetSPM_e32_PHONE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_PHONE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_PHONE)); }
   tBool bSPM_e32_ACCESSORY() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ACCESSORY) != 0; }
   tVoid vSetSPM_e32_ACCESSORY(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ACCESSORY):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ACCESSORY)); }
   tBool bSPM_e32_IGNITION() const {return (u32Value & FI_C_U32_BIT_SPM_E32_IGNITION) != 0; }
   tVoid vSetSPM_e32_IGNITION(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_IGNITION):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_IGNITION)); }
   tBool bSPM_e32_TRANSPORTMODE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_TRANSPORTMODE) != 0; }
   tVoid vSetSPM_e32_TRANSPORTMODE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_TRANSPORTMODE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_TRANSPORTMODE)); }
   tBool bSPM_e32_DOOROPEN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DOOROPEN) != 0; }
   tVoid vSetSPM_e32_DOOROPEN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DOOROPEN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DOOROPEN)); }
   tBool bSPM_e32_DOORLOCK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DOORLOCK) != 0; }
   tVoid vSetSPM_e32_DOORLOCK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DOORLOCK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DOORLOCK)); }
   tBool bSPM_e32_TRIGGER_FORCE_ON() const {return (u32Value & FI_C_U32_BIT_SPM_E32_TRIGGER_FORCE_ON) != 0; }
   tVoid vSetSPM_e32_TRIGGER_FORCE_ON(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_TRIGGER_FORCE_ON):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_TRIGGER_FORCE_ON)); }
   tBool bSPM_e32_CLIMATE_INFO() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CLIMATE_INFO) != 0; }
   tVoid vSetSPM_e32_CLIMATE_INFO(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CLIMATE_INFO):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CLIMATE_INFO)); }
   tBool bSPM_e32_PARK_DISTANCE_CTRL() const {return (u32Value & FI_C_U32_BIT_SPM_E32_PARK_DISTANCE_CTRL) != 0; }
   tVoid vSetSPM_e32_PARK_DISTANCE_CTRL(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_PARK_DISTANCE_CTRL):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_PARK_DISTANCE_CTRL)); }
   tBool bSPM_e32_FACTORY_MODE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_FACTORY_MODE) != 0; }
   tVoid vSetSPM_e32_FACTORY_MODE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_FACTORY_MODE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_FACTORY_MODE)); }
   tBool bSPM_e32_CD_INSERT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CD_INSERT) != 0; }
   tVoid vSetSPM_e32_CD_INSERT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CD_INSERT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CD_INSERT)); }
   tBool bSPM_e32_SD_INSERT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SD_INSERT) != 0; }
   tVoid vSetSPM_e32_SD_INSERT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SD_INSERT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SD_INSERT)); }
   tBool bSPM_e32_EJECT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_EJECT) != 0; }
   tVoid vSetSPM_e32_EJECT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_EJECT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_EJECT)); }
   tBool bSPM_e32_ON_TIPPER() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ON_TIPPER) != 0; }
   tVoid vSetSPM_e32_ON_TIPPER(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ON_TIPPER):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ON_TIPPER)); }
   tBool bSPM_e32_SOFTTOP_OPEN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SOFTTOP_OPEN) != 0; }
   tVoid vSetSPM_e32_SOFTTOP_OPEN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SOFTTOP_OPEN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SOFTTOP_OPEN)); }
   tBool bSPM_e32_AIRBAG_DEPLOYED() const {return (u32Value & FI_C_U32_BIT_SPM_E32_AIRBAG_DEPLOYED) != 0; }
   tVoid vSetSPM_e32_AIRBAG_DEPLOYED(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_AIRBAG_DEPLOYED):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_AIRBAG_DEPLOYED)); }
   tBool bSPM_e32_KEY_PRESENT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_KEY_PRESENT) != 0; }
   tVoid vSetSPM_e32_KEY_PRESENT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_KEY_PRESENT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_KEY_PRESENT)); }
   tBool bSPM_e32_NETWORK_ACTIVITY() const {return (u32Value & FI_C_U32_BIT_SPM_E32_NETWORK_ACTIVITY) != 0; }
   tVoid vSetSPM_e32_NETWORK_ACTIVITY(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_NETWORK_ACTIVITY):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_NETWORK_ACTIVITY)); }
   tBool bSPM_e32_OFF_TIPPER() const {return (u32Value & FI_C_U32_BIT_SPM_E32_OFF_TIPPER) != 0; }
   tVoid vSetSPM_e32_OFF_TIPPER(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_OFF_TIPPER):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_OFF_TIPPER)); }
   tBool bSPM_e32_OFF_LASTSTATE_CYCLE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_OFF_LASTSTATE_CYCLE) != 0; }
   tVoid vSetSPM_e32_OFF_LASTSTATE_CYCLE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_OFF_LASTSTATE_CYCLE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_OFF_LASTSTATE_CYCLE)); }
   tBool bSPM_e32_DIAG_REQUEST() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DIAG_REQUEST) != 0; }
   tVoid vSetSPM_e32_DIAG_REQUEST(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DIAG_REQUEST):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DIAG_REQUEST)); }
   tBool bSPM_e32_BACKLIGHT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_BACKLIGHT) != 0; }
   tVoid vSetSPM_e32_BACKLIGHT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_BACKLIGHT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_BACKLIGHT)); }
   tBool bSPM_e32_HIGH_VOLTAGE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_HIGH_VOLTAGE) != 0; }
   tVoid vSetSPM_e32_HIGH_VOLTAGE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_HIGH_VOLTAGE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_HIGH_VOLTAGE)); }
   tBool bSPM_e32_RCV_CONNECTED() const {return (u32Value & FI_C_U32_BIT_SPM_E32_RCV_CONNECTED) != 0; }
   tVoid vSetSPM_e32_RCV_CONNECTED(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_RCV_CONNECTED):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_RCV_CONNECTED)); }
   tBool bSPM_e32_OFF_LEARN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_OFF_LEARN) != 0; }
   tVoid vSetSPM_e32_OFF_LEARN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_OFF_LEARN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_OFF_LEARN)); }
   struct rBitValues {
      tBool SPM_e32_SET_SD_CARD_ACCESS:1;
      tBool SPM_e32_EMERGENCY_OFF:1;
      tBool SPM_e32_FAST_SHUTDOWN:1;
      tBool SPM_e32_DIMMING:1;
      tBool SPM_e32_DOWNLOAD:1;
      tBool SPM_e32_DIAG_ACTIVITY:1;
      tBool SPM_e32_DIAGNOSIS:1;
      tBool SPM_e32_PHONE:1;
      tBool SPM_e32_ACCESSORY:1;
      tBool SPM_e32_IGNITION:1;
      tBool SPM_e32_TRANSPORTMODE:1;
      tBool SPM_e32_DOOROPEN:1;
      tBool SPM_e32_DOORLOCK:1;
      tBool SPM_e32_TRIGGER_FORCE_ON:1;
      tBool SPM_e32_CLIMATE_INFO:1;
      tBool SPM_e32_PARK_DISTANCE_CTRL:1;
      tBool SPM_e32_FACTORY_MODE:1;
      tBool SPM_e32_CD_INSERT:1;
      tBool SPM_e32_SD_INSERT:1;
      tBool SPM_e32_EJECT:1;
      tBool SPM_e32_ON_TIPPER:1;
      tBool SPM_e32_SOFTTOP_OPEN:1;
      tBool SPM_e32_AIRBAG_DEPLOYED:1;
      tBool SPM_e32_KEY_PRESENT:1;
      tBool SPM_e32_NETWORK_ACTIVITY:1;
      tBool SPM_e32_OFF_TIPPER:1;
      tBool SPM_e32_OFF_LASTSTATE_CYCLE:1;
      tBool SPM_e32_DIAG_REQUEST:1;
      tBool SPM_e32_BACKLIGHT:1;
      tBool SPM_e32_HIGH_VOLTAGE:1;
      tBool SPM_e32_RCV_CONNECTED:1;
      tBool SPM_e32_OFF_LEARN:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~spm_fi_tcl_b32_SubState0();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_b32_SubState0& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_b32_SubState1 : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_b32_SubState1();

   enum {
      FI_C_U32_BIT_SPM_E32_HEARTBEAT = 0x1UL,
      FI_C_U32_BIT_SPM_E32_USER_ON_REQ = 0x2UL,
      FI_C_U32_BIT_SPM_E32_BACKGROUND_ON_REQ = 0x4UL,
      FI_C_U32_BIT_SPM_E32_MULTIMEDIA_ON = 0x8UL,
      FI_C_U32_BIT_SPM_E32_ANIMATION = 0x10UL,
      FI_C_U32_BIT_SPM_E32_MULTIMEDIA_OPERATION = 0x20UL,
      FI_C_U32_BIT_SPM_E32_TELEMATIC_ACTIVE = 0x40UL,
      FI_C_U32_BIT_SPM_E32_STANDBY_REQ = 0x80UL,
      FI_C_U32_BIT_SPM_E32_ACCESSORY_ACTIVE = 0x100UL,
      FI_C_U32_BIT_SPM_E32_SHUTDOWN_BATTERY_SAVING = 0x200UL,
      FI_C_U32_BIT_SPM_E32_TML_REQ_BACKGROUND = 0x400UL,
      FI_C_U32_BIT_SPM_E32_IGNITION_TIMEOUT = 0x800UL,
      FI_C_U32_BIT_SPM_E32_IGNITION_INVALID = 0x1000UL,
      FI_C_U32_BIT_SPM_E32_FIRST_PIC_DRAWN = 0x2000UL,
      FI_C_U32_BIT_SPM_E32_OVERTEMP = 0x4000UL,
      FI_C_U32_BIT_SPM_E32_LOCK_ACTIVITY = 0x8000UL,
      FI_C_U32_BIT_SPM_E32_SYSSTATE_CHECK = 0x10000UL,
      FI_C_U32_BIT_SPM_E32_CR_TEMPERATURE = 0x20000UL,
      FI_C_U32_BIT_SPM_E32_PUSH_TO_TALK = 0x40000UL,
      FI_C_U32_BIT_SPM_E32_ROUTEGUIDANCE_ACTIVE = 0x80000UL,
      FI_C_U32_BIT_SPM_E32_HW_IGNITION = 0x100000UL,
      FI_C_U32_BIT_SPM_E32_CAN_WAKEUP = 0x200000UL,
      FI_C_U32_BIT_SPM_E32_DOWNLOAD_PERMISSION = 0x400000UL,
      FI_C_U32_BIT_SPM_E32_IGNITION_LOGIC = 0x800000UL,
      FI_C_U32_BIT_SPM_E32_LOW_VOLTAGE = 0x1000000UL,
      FI_C_U32_BIT_SPM_E32_STATE_DEBOUNCE = 0x2000000UL,
      FI_C_U32_BIT_SPM_E32_NETWORK_FAILURE = 0x4000000UL,
      FI_C_U32_BIT_SPM_E32_STATE_TO = 0x8000000UL,
      FI_C_U32_BIT_SPM_E32_DEBUG_WAKEUP = 0x10000000UL,
      FI_C_U32_BIT_SPM_E32_AUTOMATIC = 0x20000000UL,
      FI_C_U32_BIT_SPM_E32_CODE_REQUEST = 0x40000000UL,
      FI_C_U32_BIT_SPM_E32_SYNC_SHUTDOWN_ACK = 0x80000000UL
   };
   tBool bSPM_e32_HEARTBEAT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_HEARTBEAT) != 0; }
   tVoid vSetSPM_e32_HEARTBEAT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_HEARTBEAT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_HEARTBEAT)); }
   tBool bSPM_e32_USER_ON_REQ() const {return (u32Value & FI_C_U32_BIT_SPM_E32_USER_ON_REQ) != 0; }
   tVoid vSetSPM_e32_USER_ON_REQ(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_USER_ON_REQ):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_USER_ON_REQ)); }
   tBool bSPM_e32_BACKGROUND_ON_REQ() const {return (u32Value & FI_C_U32_BIT_SPM_E32_BACKGROUND_ON_REQ) != 0; }
   tVoid vSetSPM_e32_BACKGROUND_ON_REQ(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_BACKGROUND_ON_REQ):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_BACKGROUND_ON_REQ)); }
   tBool bSPM_e32_MULTIMEDIA_ON() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MULTIMEDIA_ON) != 0; }
   tVoid vSetSPM_e32_MULTIMEDIA_ON(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MULTIMEDIA_ON):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MULTIMEDIA_ON)); }
   tBool bSPM_e32_ANIMATION() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ANIMATION) != 0; }
   tVoid vSetSPM_e32_ANIMATION(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ANIMATION):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ANIMATION)); }
   tBool bSPM_e32_MULTIMEDIA_OPERATION() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MULTIMEDIA_OPERATION) != 0; }
   tVoid vSetSPM_e32_MULTIMEDIA_OPERATION(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MULTIMEDIA_OPERATION):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MULTIMEDIA_OPERATION)); }
   tBool bSPM_e32_TELEMATIC_ACTIVE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_TELEMATIC_ACTIVE) != 0; }
   tVoid vSetSPM_e32_TELEMATIC_ACTIVE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_TELEMATIC_ACTIVE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_TELEMATIC_ACTIVE)); }
   tBool bSPM_e32_STANDBY_REQ() const {return (u32Value & FI_C_U32_BIT_SPM_E32_STANDBY_REQ) != 0; }
   tVoid vSetSPM_e32_STANDBY_REQ(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_STANDBY_REQ):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_STANDBY_REQ)); }
   tBool bSPM_e32_ACCESSORY_ACTIVE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ACCESSORY_ACTIVE) != 0; }
   tVoid vSetSPM_e32_ACCESSORY_ACTIVE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ACCESSORY_ACTIVE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ACCESSORY_ACTIVE)); }
   tBool bSPM_e32_SHUTDOWN_BATTERY_SAVING() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SHUTDOWN_BATTERY_SAVING) != 0; }
   tVoid vSetSPM_e32_SHUTDOWN_BATTERY_SAVING(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SHUTDOWN_BATTERY_SAVING):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SHUTDOWN_BATTERY_SAVING)); }
   tBool bSPM_e32_TML_REQ_BACKGROUND() const {return (u32Value & FI_C_U32_BIT_SPM_E32_TML_REQ_BACKGROUND) != 0; }
   tVoid vSetSPM_e32_TML_REQ_BACKGROUND(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_TML_REQ_BACKGROUND):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_TML_REQ_BACKGROUND)); }
   tBool bSPM_e32_IGNITION_TIMEOUT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_IGNITION_TIMEOUT) != 0; }
   tVoid vSetSPM_e32_IGNITION_TIMEOUT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_IGNITION_TIMEOUT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_IGNITION_TIMEOUT)); }
   tBool bSPM_e32_IGNITION_INVALID() const {return (u32Value & FI_C_U32_BIT_SPM_E32_IGNITION_INVALID) != 0; }
   tVoid vSetSPM_e32_IGNITION_INVALID(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_IGNITION_INVALID):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_IGNITION_INVALID)); }
   tBool bSPM_e32_FIRST_PIC_DRAWN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_FIRST_PIC_DRAWN) != 0; }
   tVoid vSetSPM_e32_FIRST_PIC_DRAWN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_FIRST_PIC_DRAWN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_FIRST_PIC_DRAWN)); }
   tBool bSPM_e32_OVERTEMP() const {return (u32Value & FI_C_U32_BIT_SPM_E32_OVERTEMP) != 0; }
   tVoid vSetSPM_e32_OVERTEMP(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_OVERTEMP):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_OVERTEMP)); }
   tBool bSPM_e32_LOCK_ACTIVITY() const {return (u32Value & FI_C_U32_BIT_SPM_E32_LOCK_ACTIVITY) != 0; }
   tVoid vSetSPM_e32_LOCK_ACTIVITY(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_LOCK_ACTIVITY):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_LOCK_ACTIVITY)); }
   tBool bSPM_e32_SYSSTATE_CHECK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SYSSTATE_CHECK) != 0; }
   tVoid vSetSPM_e32_SYSSTATE_CHECK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SYSSTATE_CHECK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SYSSTATE_CHECK)); }
   tBool bSPM_e32_CR_TEMPERATURE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CR_TEMPERATURE) != 0; }
   tVoid vSetSPM_e32_CR_TEMPERATURE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CR_TEMPERATURE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CR_TEMPERATURE)); }
   tBool bSPM_e32_PUSH_TO_TALK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_PUSH_TO_TALK) != 0; }
   tVoid vSetSPM_e32_PUSH_TO_TALK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_PUSH_TO_TALK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_PUSH_TO_TALK)); }
   tBool bSPM_e32_ROUTEGUIDANCE_ACTIVE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ROUTEGUIDANCE_ACTIVE) != 0; }
   tVoid vSetSPM_e32_ROUTEGUIDANCE_ACTIVE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ROUTEGUIDANCE_ACTIVE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ROUTEGUIDANCE_ACTIVE)); }
   tBool bSPM_e32_HW_IGNITION() const {return (u32Value & FI_C_U32_BIT_SPM_E32_HW_IGNITION) != 0; }
   tVoid vSetSPM_e32_HW_IGNITION(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_HW_IGNITION):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_HW_IGNITION)); }
   tBool bSPM_e32_CAN_WAKEUP() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CAN_WAKEUP) != 0; }
   tVoid vSetSPM_e32_CAN_WAKEUP(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CAN_WAKEUP):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CAN_WAKEUP)); }
   tBool bSPM_e32_DOWNLOAD_PERMISSION() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DOWNLOAD_PERMISSION) != 0; }
   tVoid vSetSPM_e32_DOWNLOAD_PERMISSION(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DOWNLOAD_PERMISSION):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DOWNLOAD_PERMISSION)); }
   tBool bSPM_e32_IGNITION_LOGIC() const {return (u32Value & FI_C_U32_BIT_SPM_E32_IGNITION_LOGIC) != 0; }
   tVoid vSetSPM_e32_IGNITION_LOGIC(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_IGNITION_LOGIC):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_IGNITION_LOGIC)); }
   tBool bSPM_e32_LOW_VOLTAGE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_LOW_VOLTAGE) != 0; }
   tVoid vSetSPM_e32_LOW_VOLTAGE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_LOW_VOLTAGE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_LOW_VOLTAGE)); }
   tBool bSPM_e32_STATE_DEBOUNCE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_STATE_DEBOUNCE) != 0; }
   tVoid vSetSPM_e32_STATE_DEBOUNCE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_STATE_DEBOUNCE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_STATE_DEBOUNCE)); }
   tBool bSPM_e32_NETWORK_FAILURE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_NETWORK_FAILURE) != 0; }
   tVoid vSetSPM_e32_NETWORK_FAILURE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_NETWORK_FAILURE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_NETWORK_FAILURE)); }
   tBool bSPM_e32_STATE_TO() const {return (u32Value & FI_C_U32_BIT_SPM_E32_STATE_TO) != 0; }
   tVoid vSetSPM_e32_STATE_TO(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_STATE_TO):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_STATE_TO)); }
   tBool bSPM_e32_DEBUG_WAKEUP() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DEBUG_WAKEUP) != 0; }
   tVoid vSetSPM_e32_DEBUG_WAKEUP(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DEBUG_WAKEUP):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DEBUG_WAKEUP)); }
   tBool bSPM_e32_AUTOMATIC() const {return (u32Value & FI_C_U32_BIT_SPM_E32_AUTOMATIC) != 0; }
   tVoid vSetSPM_e32_AUTOMATIC(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_AUTOMATIC):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_AUTOMATIC)); }
   tBool bSPM_e32_CODE_REQUEST() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CODE_REQUEST) != 0; }
   tVoid vSetSPM_e32_CODE_REQUEST(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CODE_REQUEST):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CODE_REQUEST)); }
   tBool bSPM_e32_SYNC_SHUTDOWN_ACK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SYNC_SHUTDOWN_ACK) != 0; }
   tVoid vSetSPM_e32_SYNC_SHUTDOWN_ACK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SYNC_SHUTDOWN_ACK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SYNC_SHUTDOWN_ACK)); }
   struct rBitValues {
      tBool SPM_e32_HEARTBEAT:1;
      tBool SPM_e32_USER_ON_REQ:1;
      tBool SPM_e32_BACKGROUND_ON_REQ:1;
      tBool SPM_e32_MULTIMEDIA_ON:1;
      tBool SPM_e32_ANIMATION:1;
      tBool SPM_e32_MULTIMEDIA_OPERATION:1;
      tBool SPM_e32_TELEMATIC_ACTIVE:1;
      tBool SPM_e32_STANDBY_REQ:1;
      tBool SPM_e32_ACCESSORY_ACTIVE:1;
      tBool SPM_e32_SHUTDOWN_BATTERY_SAVING:1;
      tBool SPM_e32_TML_REQ_BACKGROUND:1;
      tBool SPM_e32_IGNITION_TIMEOUT:1;
      tBool SPM_e32_IGNITION_INVALID:1;
      tBool SPM_e32_FIRST_PIC_DRAWN:1;
      tBool SPM_e32_OVERTEMP:1;
      tBool SPM_e32_LOCK_ACTIVITY:1;
      tBool SPM_e32_SYSSTATE_CHECK:1;
      tBool SPM_e32_CR_TEMPERATURE:1;
      tBool SPM_e32_PUSH_TO_TALK:1;
      tBool SPM_e32_ROUTEGUIDANCE_ACTIVE:1;
      tBool SPM_e32_HW_IGNITION:1;
      tBool SPM_e32_CAN_WAKEUP:1;
      tBool SPM_e32_DOWNLOAD_PERMISSION:1;
      tBool SPM_e32_IGNITION_LOGIC:1;
      tBool SPM_e32_LOW_VOLTAGE:1;
      tBool SPM_e32_STATE_DEBOUNCE:1;
      tBool SPM_e32_NETWORK_FAILURE:1;
      tBool SPM_e32_STATE_TO:1;
      tBool SPM_e32_DEBUG_WAKEUP:1;
      tBool SPM_e32_AUTOMATIC:1;
      tBool SPM_e32_CODE_REQUEST:1;
      tBool SPM_e32_SYNC_SHUTDOWN_ACK:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~spm_fi_tcl_b32_SubState1();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_b32_SubState1& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_b32_SubState2 : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_b32_SubState2();

   enum {
      FI_C_U32_BIT_SPM_E32_FRIENDLY_USER = 0x1UL,
      FI_C_U32_BIT_SPM_E32_NISSAN_CMF1 = 0x2UL,
      FI_C_U32_BIT_SPM_E32_PREPARE_SHUTDOWN = 0x4UL,
      FI_C_U32_BIT_SPM_E32_SHUTDOWN = 0x8UL,
      FI_C_U32_BIT_SPM_E32_HKEY_PRESSED = 0x10UL,
      FI_C_U32_BIT_SPM_E32_CD_INSERT_CLAMPS = 0x20UL,
      FI_C_U32_BIT_SPM_E32_CRITICAL_LOW_VOLTAGE = 0x40UL,
      FI_C_U32_BIT_SPM_E32_ENDURING_LOW_VOLTAGE = 0x80UL,
      FI_C_U32_BIT_SPM_E32_DIAG_FAST_SHUTDOWN = 0x100UL,
      FI_C_U32_BIT_SPM_E32_USING_ACCESSORY = 0x200UL,
      FI_C_U32_BIT_SPM_E32_BACKGROUND_IMM_OFF = 0x400UL,
      FI_C_U32_BIT_SPM_E32_ROLLER_RIG_MODE = 0x800UL,
      FI_C_U32_BIT_SPM_E32_REMOTE_ON_ACTIVE = 0x1000UL,
      FI_C_U32_BIT_SPM_E32_ENDURING_CR_VOLTAGE = 0x2000UL,
      FI_C_U32_BIT_SPM_E32_NORMAL_VOLT = 0x4000UL,
      FI_C_U32_BIT_SPM_E32_Q_CRITICAL = 0x8000UL,
      FI_C_U32_BIT_SPM_E32_TRIGGER_RESET = 0x10000UL,
      FI_C_U32_BIT_SPM_E32_CAR_WAKEUP = 0x20000UL,
      FI_C_U32_BIT_SPM_E32_PHONE_CONNECTED = 0x40000UL,
      FI_C_U32_BIT_SPM_E32_CUSTOMER_DOWNLOAD = 0x80000UL,
      FI_C_U32_BIT_SPM_E32_STARTUP = 0x100000UL,
      FI_C_U32_BIT_SPM_E32_MOST_OFF = 0x200000UL,
      FI_C_U32_BIT_SPM_E32_MOST_OFF_ERR = 0x400000UL,
      FI_C_U32_BIT_SPM_E32_MOST_WAIT_FOR_SHUTDOWN = 0x800000UL,
      FI_C_U32_BIT_SPM_E32_MOST_RBD = 0x1000000UL,
      FI_C_U32_BIT_SPM_E32_MOST_SHUTDOWN_EXECUTE = 0x2000000UL,
      FI_C_U32_BIT_SPM_E32_MOST_PWR_SWITCH_OFF_DELAY = 0x4000000UL,
      FI_C_U32_BIT_SPM_E32_SYSSTATE_NETWORK_FAILED = 0x8000000UL,
      FI_C_U32_BIT_SPM_E32_SAFE = 0x10000000UL,
      FI_C_U32_BIT_SPM_E32_FSM_SHUTDOWN_FINISHED = 0x20000000UL,
      FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_SHUTDOWN_ACK = 0x40000000UL,
      FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_RESTART_ACK = 0x80000000UL
   };
   tBool bSPM_e32_FRIENDLY_USER() const {return (u32Value & FI_C_U32_BIT_SPM_E32_FRIENDLY_USER) != 0; }
   tVoid vSetSPM_e32_FRIENDLY_USER(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_FRIENDLY_USER):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_FRIENDLY_USER)); }
   tBool bSPM_e32_NISSAN_CMF1() const {return (u32Value & FI_C_U32_BIT_SPM_E32_NISSAN_CMF1) != 0; }
   tVoid vSetSPM_e32_NISSAN_CMF1(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_NISSAN_CMF1):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_NISSAN_CMF1)); }
   tBool bSPM_e32_PREPARE_SHUTDOWN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_PREPARE_SHUTDOWN) != 0; }
   tVoid vSetSPM_e32_PREPARE_SHUTDOWN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_PREPARE_SHUTDOWN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_PREPARE_SHUTDOWN)); }
   tBool bSPM_e32_SHUTDOWN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SHUTDOWN) != 0; }
   tVoid vSetSPM_e32_SHUTDOWN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SHUTDOWN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SHUTDOWN)); }
   tBool bSPM_e32_HKEY_PRESSED() const {return (u32Value & FI_C_U32_BIT_SPM_E32_HKEY_PRESSED) != 0; }
   tVoid vSetSPM_e32_HKEY_PRESSED(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_HKEY_PRESSED):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_HKEY_PRESSED)); }
   tBool bSPM_e32_CD_INSERT_CLAMPS() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CD_INSERT_CLAMPS) != 0; }
   tVoid vSetSPM_e32_CD_INSERT_CLAMPS(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CD_INSERT_CLAMPS):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CD_INSERT_CLAMPS)); }
   tBool bSPM_e32_CRITICAL_LOW_VOLTAGE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CRITICAL_LOW_VOLTAGE) != 0; }
   tVoid vSetSPM_e32_CRITICAL_LOW_VOLTAGE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CRITICAL_LOW_VOLTAGE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CRITICAL_LOW_VOLTAGE)); }
   tBool bSPM_e32_ENDURING_LOW_VOLTAGE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ENDURING_LOW_VOLTAGE) != 0; }
   tVoid vSetSPM_e32_ENDURING_LOW_VOLTAGE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ENDURING_LOW_VOLTAGE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ENDURING_LOW_VOLTAGE)); }
   tBool bSPM_e32_DIAG_FAST_SHUTDOWN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_DIAG_FAST_SHUTDOWN) != 0; }
   tVoid vSetSPM_e32_DIAG_FAST_SHUTDOWN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_DIAG_FAST_SHUTDOWN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_DIAG_FAST_SHUTDOWN)); }
   tBool bSPM_e32_USING_ACCESSORY() const {return (u32Value & FI_C_U32_BIT_SPM_E32_USING_ACCESSORY) != 0; }
   tVoid vSetSPM_e32_USING_ACCESSORY(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_USING_ACCESSORY):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_USING_ACCESSORY)); }
   tBool bSPM_e32_BACKGROUND_IMM_OFF() const {return (u32Value & FI_C_U32_BIT_SPM_E32_BACKGROUND_IMM_OFF) != 0; }
   tVoid vSetSPM_e32_BACKGROUND_IMM_OFF(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_BACKGROUND_IMM_OFF):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_BACKGROUND_IMM_OFF)); }
   tBool bSPM_e32_ROLLER_RIG_MODE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ROLLER_RIG_MODE) != 0; }
   tVoid vSetSPM_e32_ROLLER_RIG_MODE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ROLLER_RIG_MODE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ROLLER_RIG_MODE)); }
   tBool bSPM_e32_REMOTE_ON_ACTIVE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_REMOTE_ON_ACTIVE) != 0; }
   tVoid vSetSPM_e32_REMOTE_ON_ACTIVE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_REMOTE_ON_ACTIVE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_REMOTE_ON_ACTIVE)); }
   tBool bSPM_e32_ENDURING_CR_VOLTAGE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ENDURING_CR_VOLTAGE) != 0; }
   tVoid vSetSPM_e32_ENDURING_CR_VOLTAGE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ENDURING_CR_VOLTAGE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ENDURING_CR_VOLTAGE)); }
   tBool bSPM_e32_NORMAL_VOLT() const {return (u32Value & FI_C_U32_BIT_SPM_E32_NORMAL_VOLT) != 0; }
   tVoid vSetSPM_e32_NORMAL_VOLT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_NORMAL_VOLT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_NORMAL_VOLT)); }
   tBool bSPM_e32_Q_CRITICAL() const {return (u32Value & FI_C_U32_BIT_SPM_E32_Q_CRITICAL) != 0; }
   tVoid vSetSPM_e32_Q_CRITICAL(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_Q_CRITICAL):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_Q_CRITICAL)); }
   tBool bSPM_e32_TRIGGER_RESET() const {return (u32Value & FI_C_U32_BIT_SPM_E32_TRIGGER_RESET) != 0; }
   tVoid vSetSPM_e32_TRIGGER_RESET(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_TRIGGER_RESET):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_TRIGGER_RESET)); }
   tBool bSPM_e32_CAR_WAKEUP() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CAR_WAKEUP) != 0; }
   tVoid vSetSPM_e32_CAR_WAKEUP(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CAR_WAKEUP):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CAR_WAKEUP)); }
   tBool bSPM_e32_PHONE_CONNECTED() const {return (u32Value & FI_C_U32_BIT_SPM_E32_PHONE_CONNECTED) != 0; }
   tVoid vSetSPM_e32_PHONE_CONNECTED(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_PHONE_CONNECTED):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_PHONE_CONNECTED)); }
   tBool bSPM_e32_CUSTOMER_DOWNLOAD() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CUSTOMER_DOWNLOAD) != 0; }
   tVoid vSetSPM_e32_CUSTOMER_DOWNLOAD(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CUSTOMER_DOWNLOAD):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CUSTOMER_DOWNLOAD)); }
   tBool bSPM_e32_STARTUP() const {return (u32Value & FI_C_U32_BIT_SPM_E32_STARTUP) != 0; }
   tVoid vSetSPM_e32_STARTUP(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_STARTUP):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_STARTUP)); }
   tBool bSPM_e32_MOST_OFF() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MOST_OFF) != 0; }
   tVoid vSetSPM_e32_MOST_OFF(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MOST_OFF):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MOST_OFF)); }
   tBool bSPM_e32_MOST_OFF_ERR() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MOST_OFF_ERR) != 0; }
   tVoid vSetSPM_e32_MOST_OFF_ERR(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MOST_OFF_ERR):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MOST_OFF_ERR)); }
   tBool bSPM_e32_MOST_WAIT_FOR_SHUTDOWN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MOST_WAIT_FOR_SHUTDOWN) != 0; }
   tVoid vSetSPM_e32_MOST_WAIT_FOR_SHUTDOWN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MOST_WAIT_FOR_SHUTDOWN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MOST_WAIT_FOR_SHUTDOWN)); }
   tBool bSPM_e32_MOST_RBD() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MOST_RBD) != 0; }
   tVoid vSetSPM_e32_MOST_RBD(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MOST_RBD):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MOST_RBD)); }
   tBool bSPM_e32_MOST_SHUTDOWN_EXECUTE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MOST_SHUTDOWN_EXECUTE) != 0; }
   tVoid vSetSPM_e32_MOST_SHUTDOWN_EXECUTE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MOST_SHUTDOWN_EXECUTE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MOST_SHUTDOWN_EXECUTE)); }
   tBool bSPM_e32_MOST_PWR_SWITCH_OFF_DELAY() const {return (u32Value & FI_C_U32_BIT_SPM_E32_MOST_PWR_SWITCH_OFF_DELAY) != 0; }
   tVoid vSetSPM_e32_MOST_PWR_SWITCH_OFF_DELAY(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_MOST_PWR_SWITCH_OFF_DELAY):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_MOST_PWR_SWITCH_OFF_DELAY)); }
   tBool bSPM_e32_SYSSTATE_NETWORK_FAILED() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SYSSTATE_NETWORK_FAILED) != 0; }
   tVoid vSetSPM_e32_SYSSTATE_NETWORK_FAILED(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SYSSTATE_NETWORK_FAILED):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SYSSTATE_NETWORK_FAILED)); }
   tBool bSPM_e32_SAFE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_SAFE) != 0; }
   tVoid vSetSPM_e32_SAFE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_SAFE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_SAFE)); }
   tBool bSPM_e32_FSM_SHUTDOWN_FINISHED() const {return (u32Value & FI_C_U32_BIT_SPM_E32_FSM_SHUTDOWN_FINISHED) != 0; }
   tVoid vSetSPM_e32_FSM_SHUTDOWN_FINISHED(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_FSM_SHUTDOWN_FINISHED):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_FSM_SHUTDOWN_FINISHED)); }
   tBool bSPM_e32_WAIT_FOR_FSM_SHUTDOWN_ACK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_SHUTDOWN_ACK) != 0; }
   tVoid vSetSPM_e32_WAIT_FOR_FSM_SHUTDOWN_ACK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_SHUTDOWN_ACK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_SHUTDOWN_ACK)); }
   tBool bSPM_e32_WAIT_FOR_FSM_RESTART_ACK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_RESTART_ACK) != 0; }
   tVoid vSetSPM_e32_WAIT_FOR_FSM_RESTART_ACK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_RESTART_ACK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_WAIT_FOR_FSM_RESTART_ACK)); }
   struct rBitValues {
      tBool SPM_e32_FRIENDLY_USER:1;
      tBool SPM_e32_NISSAN_CMF1:1;
      tBool SPM_e32_PREPARE_SHUTDOWN:1;
      tBool SPM_e32_SHUTDOWN:1;
      tBool SPM_e32_HKEY_PRESSED:1;
      tBool SPM_e32_CD_INSERT_CLAMPS:1;
      tBool SPM_e32_CRITICAL_LOW_VOLTAGE:1;
      tBool SPM_e32_ENDURING_LOW_VOLTAGE:1;
      tBool SPM_e32_DIAG_FAST_SHUTDOWN:1;
      tBool SPM_e32_USING_ACCESSORY:1;
      tBool SPM_e32_BACKGROUND_IMM_OFF:1;
      tBool SPM_e32_ROLLER_RIG_MODE:1;
      tBool SPM_e32_REMOTE_ON_ACTIVE:1;
      tBool SPM_e32_ENDURING_CR_VOLTAGE:1;
      tBool SPM_e32_NORMAL_VOLT:1;
      tBool SPM_e32_Q_CRITICAL:1;
      tBool SPM_e32_TRIGGER_RESET:1;
      tBool SPM_e32_CAR_WAKEUP:1;
      tBool SPM_e32_PHONE_CONNECTED:1;
      tBool SPM_e32_CUSTOMER_DOWNLOAD:1;
      tBool SPM_e32_STARTUP:1;
      tBool SPM_e32_MOST_OFF:1;
      tBool SPM_e32_MOST_OFF_ERR:1;
      tBool SPM_e32_MOST_WAIT_FOR_SHUTDOWN:1;
      tBool SPM_e32_MOST_RBD:1;
      tBool SPM_e32_MOST_SHUTDOWN_EXECUTE:1;
      tBool SPM_e32_MOST_PWR_SWITCH_OFF_DELAY:1;
      tBool SPM_e32_SYSSTATE_NETWORK_FAILED:1;
      tBool SPM_e32_SAFE:1;
      tBool SPM_e32_FSM_SHUTDOWN_FINISHED:1;
      tBool SPM_e32_WAIT_FOR_FSM_SHUTDOWN_ACK:1;
      tBool SPM_e32_WAIT_FOR_FSM_RESTART_ACK:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~spm_fi_tcl_b32_SubState2();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_b32_SubState2& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_b32_SubState3 : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_b32_SubState3();

   enum {
      FI_C_U32_BIT_SPM_E32_VALET_MODE = 0x1UL,
      FI_C_U32_BIT_SPM_E32_CRANK = 0x2UL,
      FI_C_U32_BIT_SPM_E32_EMERGENCY_CALL = 0x4UL,
      FI_C_U32_BIT_SPM_E32_LIN = 0x8UL,
      FI_C_U32_BIT_SPM_E32_CELLNETWORK = 0x10UL,
      FI_C_U32_BIT_SPM_E32_EXTERNAL_GPIO = 0x20UL,
      FI_C_U32_BIT_SPM_E32_BLUETOOTH = 0x40UL,
      FI_C_U32_BIT_SPM_E32_UART = 0x80UL,
      FI_C_U32_BIT_SPM_E32_USB = 0x100UL,
      FI_C_U32_BIT_SPM_E32_CRITICAL_HIGH_VOLTAGE = 0x200UL,
      FI_C_U32_BIT_SPM_E32_ENDURING_HIGH_VOLTAGE = 0x400UL,
      FI_C_U32_BIT_SPM_E32_OVER_THE_AIR = 0x800UL
   };
   tBool bSPM_e32_VALET_MODE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_VALET_MODE) != 0; }
   tVoid vSetSPM_e32_VALET_MODE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_VALET_MODE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_VALET_MODE)); }
   tBool bSPM_e32_CRANK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CRANK) != 0; }
   tVoid vSetSPM_e32_CRANK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CRANK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CRANK)); }
   tBool bSPM_e32_EMERGENCY_CALL() const {return (u32Value & FI_C_U32_BIT_SPM_E32_EMERGENCY_CALL) != 0; }
   tVoid vSetSPM_e32_EMERGENCY_CALL(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_EMERGENCY_CALL):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_EMERGENCY_CALL)); }
   tBool bSPM_e32_LIN() const {return (u32Value & FI_C_U32_BIT_SPM_E32_LIN) != 0; }
   tVoid vSetSPM_e32_LIN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_LIN):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_LIN)); }
   tBool bSPM_e32_CELLNETWORK() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CELLNETWORK) != 0; }
   tVoid vSetSPM_e32_CELLNETWORK(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CELLNETWORK):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CELLNETWORK)); }
   tBool bSPM_e32_EXTERNAL_GPIO() const {return (u32Value & FI_C_U32_BIT_SPM_E32_EXTERNAL_GPIO) != 0; }
   tVoid vSetSPM_e32_EXTERNAL_GPIO(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_EXTERNAL_GPIO):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_EXTERNAL_GPIO)); }
   tBool bSPM_e32_BLUETOOTH() const {return (u32Value & FI_C_U32_BIT_SPM_E32_BLUETOOTH) != 0; }
   tVoid vSetSPM_e32_BLUETOOTH(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_BLUETOOTH):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_BLUETOOTH)); }
   tBool bSPM_e32_UART() const {return (u32Value & FI_C_U32_BIT_SPM_E32_UART) != 0; }
   tVoid vSetSPM_e32_UART(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_UART):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_UART)); }
   tBool bSPM_e32_USB() const {return (u32Value & FI_C_U32_BIT_SPM_E32_USB) != 0; }
   tVoid vSetSPM_e32_USB(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_USB):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_USB)); }
   tBool bSPM_e32_CRITICAL_HIGH_VOLTAGE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_CRITICAL_HIGH_VOLTAGE) != 0; }
   tVoid vSetSPM_e32_CRITICAL_HIGH_VOLTAGE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_CRITICAL_HIGH_VOLTAGE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_CRITICAL_HIGH_VOLTAGE)); }
   tBool bSPM_e32_ENDURING_HIGH_VOLTAGE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ENDURING_HIGH_VOLTAGE) != 0; }
   tVoid vSetSPM_e32_ENDURING_HIGH_VOLTAGE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ENDURING_HIGH_VOLTAGE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ENDURING_HIGH_VOLTAGE)); }
   tBool bSPM_e32_OVER_THE_AIR() const {return (u32Value & FI_C_U32_BIT_SPM_E32_OVER_THE_AIR) != 0; }
   tVoid vSetSPM_e32_OVER_THE_AIR(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_OVER_THE_AIR):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_OVER_THE_AIR)); }
   struct rBitValues {
      tBool SPM_e32_VALET_MODE:1;
      tBool SPM_e32_CRANK:1;
      tBool SPM_e32_EMERGENCY_CALL:1;
      tBool SPM_e32_LIN:1;
      tBool SPM_e32_CELLNETWORK:1;
      tBool SPM_e32_EXTERNAL_GPIO:1;
      tBool SPM_e32_BLUETOOTH:1;
      tBool SPM_e32_UART:1;
      tBool SPM_e32_USB:1;
      tBool SPM_e32_CRITICAL_HIGH_VOLTAGE:1;
      tBool SPM_e32_ENDURING_HIGH_VOLTAGE:1;
      tBool SPM_e32_OVER_THE_AIR:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
      tBool dummy16:1;
      tBool dummy17:1;
      tBool dummy18:1;
      tBool dummy19:1;
      tBool dummy20:1;
      tBool dummy21:1;
      tBool dummy22:1;
      tBool dummy23:1;
      tBool dummy24:1;
      tBool dummy25:1;
      tBool dummy26:1;
      tBool dummy27:1;
      tBool dummy28:1;
      tBool dummy29:1;
      tBool dummy30:1;
      tBool dummy31:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~spm_fi_tcl_b32_SubState3();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_b32_SubState3& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_SubStateType : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_SubStateType();
   enum tenType {
      FI_EN_SPM_U32_SUBSTATE_SET_SD_CARD_ACCESS = 0UL,
      FI_EN_SPM_U32_SUBSTATE_EMERGENCY_OFF = 1UL,
      FI_EN_SPM_U32_SUBSTATE_FAST_SHUTDOWN = 2UL,
      FI_EN_SPM_U32_SUBSTATE_DIMMING = 3UL,
      FI_EN_SPM_U32_SUBSTATE_DOWNLOAD = 4UL,
      FI_EN_SPM_U32_SUBSTATE_DIAG_ACTIVITY = 5UL,
      FI_EN_SPM_U32_SUBSTATE_DIAGNOSIS = 6UL,
      FI_EN_SPM_U32_SUBSTATE_PHONE = 7UL,
      FI_EN_SPM_U32_SUBSTATE_S_CONTACT = 8UL,
      FI_EN_SPM_U32_SUBSTATE_IGNITION = 9UL,
      FI_EN_SPM_U32_SUBSTATE_TRANSPORTMODE = 10UL,
      FI_EN_SPM_U32_SUBSTATE_DOOROPEN = 11UL,
      FI_EN_SPM_U32_SUBSTATE_DOORLOCK = 12UL,
      FI_EN_SPM_U32_SUBSTATE_TRIGGER_FORCE_ON = 13UL,
      FI_EN_SPM_U32_SUBSTATE_CLIMATE_INFO = 14UL,
      FI_EN_SPM_U32_SUBSTATE_PARK_DISTANCE_CTRL = 15UL,
      FI_EN_SPM_U32_SUBSTATE_FACTORY_MODE = 16UL,
      FI_EN_SPM_U32_SUBSTATE_CD_INSERT = 17UL,
      FI_EN_SPM_U32_SUBSTATE_SD_INSERT = 18UL,
      FI_EN_SPM_U32_SUBSTATE_EJECT = 19UL,
      FI_EN_SPM_U32_SUBSTATE_ON_TIPPER = 20UL,
      FI_EN_SPM_U32_SUBSTATE_SOFTTOP_OPEN = 21UL,
      FI_EN_SPM_U32_SUBSTATE_AIRBAG_DEPLOYED = 22UL,
      FI_EN_SPM_U32_SUBSTATE_KEY_PRESENT = 23UL,
      FI_EN_SPM_U32_SUBSTATE_NETWORK_ACTIVITY = 24UL,
      FI_EN_SPM_U32_SUBSTATE_OFF_TIPPER = 25UL,
      FI_EN_SPM_U32_SUBSTATE_OFF_LASTSTATE_CYCLE = 26UL,
      FI_EN_SPM_U32_SUBSTATE_DIAG_REQUEST = 27UL,
      FI_EN_SPM_U32_SUBSTATE_BACKLIGHT = 28UL,
      FI_EN_SPM_U32_SUBSTATE_HIGH_VOLTAGE = 29UL,
      FI_EN_SPM_U32_SUBSTATE_RCV_CONNECTED = 30UL,
      FI_EN_SPM_U32_SUBSTATE_OFF_LEARN = 31UL,
      FI_EN_SPM_U32_SUBSTATE_HEARTBEAT = 32UL,
      FI_EN_SPM_U32_SUBSTATE_USER_ON_REQ = 33UL,
      FI_EN_SPM_U32_SUBSTATE_BACKGROUND_ON_REQ = 34UL,
      FI_EN_SPM_U32_SUBSTATE_MULTIMEDIA_ON = 35UL,
      FI_EN_SPM_U32_SUBSTATE_ANIMATION = 36UL,
      FI_EN_SPM_U32_SUBSTATE_MULTIMEDIA_OPERATION = 37UL,
      FI_EN_SPM_U32_SUBSTATE_TELEMATIC_ACTIVE = 38UL,
      FI_EN_SPM_U32_SUBSTATE_STANDBY_REQ = 39UL,
      FI_EN_SPM_U32_SUBSTATE_ACCESSORY_ACTIVE = 40UL,
      FI_EN_SPM_U32_SUBSTATE_SHUTDOWN_BATTERY_SAVING = 41UL,
      FI_EN_SPM_U32_SUBSTATE_TML_REQ_BACKGROUND = 42UL,
      FI_EN_SPM_U32_SUBSTATE_IGNITION_TIMEOUT = 43UL,
      FI_EN_SPM_U32_SUBSTATE_IGNITION_INVALID = 44UL,
      FI_EN_SPM_U32_SUBSTATE_FIRST_PIC_DRAWN = 45UL,
      FI_EN_SPM_U32_SUBSTATE_OVERTEMP = 46UL,
      FI_EN_SPM_U32_SUBSTATE_LOCK_ACTIVITY = 47UL,
      FI_EN_SPM_U32_SUBSTATE_SYSSTATE_CHECK = 48UL,
      FI_EN_SPM_U32_SUBSTATE_CR_TEMPERATURE = 49UL,
      FI_EN_SPM_U32_SUBSTATE_PUSH_TO_TALK = 50UL,
      FI_EN_SPM_U32_SUBSTATE_ROUTEGUIDANCE_ACTIVE = 51UL,
      FI_EN_SPM_U32_SUBSTATE_HW_IGNITION = 52UL,
      FI_EN_SPM_U32_SUBSTATE_CAN_WAKEUP = 53UL,
      FI_EN_SPM_U32_SUBSTATE_DOWNLOAD_PERMISSION = 54UL,
      FI_EN_SPM_U32_SUBSTATE_IGNITION_LOGIC = 55UL,
      FI_EN_SPM_U32_SUBSTATE_LOW_VOLTAGE = 56UL,
      FI_EN_SPM_U32_SUBSTATE_STATE_DEBOUNCE = 57UL,
      FI_EN_SPM_U32_SUBSTATE_NETWORK_FAILURE = 58UL,
      FI_EN_SPM_U32_SUBSTATE_STATE_TO = 59UL,
      FI_EN_SPM_U32_SUBSTATE_DEBUG_WAKEUP = 60UL,
      FI_EN_SPM_U32_SUBSTATE_AUTOMATIC = 61UL,
      FI_EN_SPM_U32_SUBSTATE_CODE_REQUEST = 62UL,
      FI_EN_SPM_U32_SUBSTATE_SYNC_SHUTDOWN_ACK = 63UL,
      FI_EN_SPM_U32_SUBSTATE_FRIENDLY_USER = 64UL,
      FI_EN_SPM_U32_SUBSTATE_NISSAN_CMF1 = 65UL,
      FI_EN_SPM_U32_SUBSTATE_PREPARE_SHUTDOWN = 66UL,
      FI_EN_SPM_U32_SUBSTATE_SHUTDOWN = 67UL,
      FI_EN_SPM_U32_SUBSTATE_HKEY_PRESSED = 68UL,
      FI_EN_SPM_U32_SUBSTATE_CD_INSERT_CLAMPS = 69UL,
      FI_EN_SPM_U32_SUBSTATE_CRITICAL_LOW_VOLTAGE = 70UL,
      FI_EN_SPM_U32_SUBSTATE_ENDURING_LOW_VOLTAGE = 71UL,
      FI_EN_SPM_U32_SUBSTATE_DIAG_FAST_SHUTDOWN = 72UL,
      FI_EN_SPM_U32_SUBSTATE_USING_ACCESSORY = 73UL,
      FI_EN_SPM_U32_SUBSTATE_BACKGROUND_IMM_OFF = 74UL,
      FI_EN_SPM_U32_SUBSTATE_ROLLER_RIG_MODE = 75UL,
      FI_EN_SPM_U32_SUBSTATE_REMOTE_ON_ACTIVE = 76UL,
      FI_EN_SPM_U32_SUBSTATE_ENDURING_CR_VOLTAGE = 77UL,
      FI_EN_SPM_U32_SUBSTATE_NORMAL_VOLT = 78UL,
      FI_EN_SPM_U32_SUBSTATE_Q_CRITICAL = 79UL,
      FI_EN_SPM_U32_SUBSTATE_TRIGGER_RESET = 80UL,
      FI_EN_SPM_U32_SUBSTATE_CAR_WAKEUP = 81UL,
      FI_EN_SPM_U32_SUBSTATE_PHONE_CONNECTED = 82UL,
      FI_EN_SPM_U32_SUBSTATE_CUSTOMER_DOWNLOAD = 83UL,
      FI_EN_SPM_U32_SUBSTATE_STARTUP = 84UL,
      FI_EN_SPM_U32_SUBSTATE_MOST_OFF = 85UL,
      FI_EN_SPM_U32_SUBSTATE_MOST_OFF_ERR = 86UL,
      FI_EN_SPM_U32_SUBSTATE_MOST_WAIT_FOR_SHUTDOWN = 87UL,
      FI_EN_SPM_U32_SUBSTATE_MOST_RBD = 88UL,
      FI_EN_SPM_U32_SUBSTATE_MOST_SHUTDOWN_EXECUTE = 89UL,
      FI_EN_SPM_U32_SUBSTATE_MOST_PWR_SWITCH_OFF_DELAY = 90UL,
      FI_EN_SPM_U32_SUBSTATE_SYSSTATE_NETWORK_FAILED = 91UL,
      FI_EN_SPM_U32_SUBSTATE_SAFE = 92UL,
      FI_EN_SPM_U32_SUBSTATE_FSM_SHUTDOWN_FINISHED = 93UL,
      FI_EN_SPM_U32_SUBSTATE_WAIT_FOR_FSM_SHUTDOWN_ACK = 94UL,
      FI_EN_SPM_U32_SUBSTATE_WAIT_FOR_FSM_RESTART_ACK = 95UL,
      FI_EN_SPM_U32_SUBSTATE_VALET_MODE = 96UL,
      FI_EN_SPM_U32_SUBSTATE_CRANK = 97UL,
      FI_EN_SPM_U32_SUBSTATE_EMERGENCY_CALL = 98UL,
      FI_EN_SPM_U32_SUBSTATE_LIN = 99UL,
      FI_EN_SPM_U32_SUBSTATE_CELLNETWORK = 100UL,
      FI_EN_SPM_U32_SUBSTATE_EXTERNAL_GPIO = 101UL,
      FI_EN_SPM_U32_SUBSTATE_BLUETOOTH = 102UL,
      FI_EN_SPM_U32_SUBSTATE_UART = 103UL,
      FI_EN_SPM_U32_SUBSTATE_USB = 104UL,
      FI_EN_SPM_U32_SUBSTATE_CRITICAL_HIGH_VOLTAGE = 105UL,
      FI_EN_SPM_U32_SUBSTATE_ENDURING_HIGH_VOLTAGE = 106UL,
      FI_EN_SPM_U32_SUBSTATE_OVER_THE_AIR_ACTIVE = 107UL,
      FI_EN_SPM_U32_SUBSTATE_ENGINE_RUNNING = 108UL,
      FI_EN_SPM_U32_SUBSTATE_UNBALLAST_SWITCH_OFF = 109UL,
      FI_EN_SPM_U32_SUBSTATE_CUSTOMER_FEEDBACK_NEEDED = 110UL,
      FI_EN_SPM_U32_SUBSTATE_SWITCH_OFF_REQUEST = 111UL,
      FI_EN_SPM_U32_SUBSTATE_DARK_SCREEN_REQ = 112UL,
      FI_EN_SPM_U32_SUBSTATE_DISPLAY_REQUEST = 113UL,
      FI_EN_SPM_U32_SUBSTATE_TOUCH_ACTIVITY = 114UL,
      FI_EN_SPM_U32_SUBSTATE_STORAGE_MODE = 115UL,
      FI_EN_SPM_U32_SUBSTATE_ECO_MODE = 116UL,
      FI_EN_SPM_U32_SUBSTATE_ECO_MODE_TIMEOUT = 117UL,
      FI_EN_SPM_U32_SUBSTATE_PHONE_ECO_MODE_TIMEOUT = 118UL,
      FI_EN_SPM_U32_SUBSTATE_PHONE_ECO_MODE_POPUP_TIMEOUT = 119UL,
      FI_EN_SPM_U32_SUBSTATE_EXTERNAL_SHUTDOWN_REQ = 120UL,
      FI_EN_SPM_U32_SUBSTATE_EXTERNAL_SHUTDOWN_CONFIG = 121UL,
      FI_EN_SPM_U32_SUBSTATE_DEMAND_AUDIO_MUTE = 122UL,
      FI_EN_SPM_U32_SUBSTATE_PERTE_COM = 123UL,
      FI_EN_SPM_U32_SUBSTATE_SHUTDOWN_SCREEN_TIMEOUT = 124UL,
      FI_EN_SPM_U32_SUBSTATE_OFF_REQUEST = 125UL,
      FI_EN_SPM_U32_SUBSTATE_ALERT_MODE = 126UL,
      FI_EN_SPM_U32_SUBSTATE_ECO_MODE_ACTIVATED = 127UL,
      FI_EN_SPM_U32_SUBSTATE_SMARTAPP_COMM = 128UL,
      FI_EN_SPM_U32_SUBSTATE_PHONE_MUTE = 129UL,
      FI_EN_SPM_U32_SUBSTATE_WAITING = 130UL,
      FI_EN_SPM_U32_SUBSTATE_MAPLEARN_ONGOING = 131UL,
      FI_EN_SPM_U32_SUBSTATE_SPI_PHONE = 132UL,
      FI_EN_SPM_U32_SUBSTATE_DMVERITY_CHECK = 133UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_SubStateType(spm_fi_tcl_SPM_e32_SubStateType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_SubStateType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_SubStateType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_OnOffReason : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_OnOffReason();
   enum tenType {
      FI_EN_SPM_SUBSTATE_TIPPER_ON = 1UL,
      FI_EN_SPM_SUBSTATE_NO_DIAG = 2UL,
      FI_EN_SPM_SUBSTATE_DIAG = 2147483650UL,
      FI_EN_SPM_SUBSTATE_NO_DOWNLOAD = 4UL,
      FI_EN_SPM_SUBSTATE_DOWNLOAD = 2147483652UL,
      FI_EN_SPM_SUBSTATE_NO_TRANSPORT_MODE = 8UL,
      FI_EN_SPM_SUBSTATE_TRANSPORT_MODE = 2147483656UL,
      FI_EN_SPM_SUBSTATE_NO_IGNITION = 16UL,
      FI_EN_SPM_SUBSTATE_IGNITION = 2147483664UL,
      FI_EN_SPM_SUBSTATE_NO_S_CONTACT = 32UL,
      FI_EN_SPM_SUBSTATE_S_CONTACT = 2147483680UL,
      FI_EN_SPM_SUBSTATE_NO_DIAGNOSTICS_ACTIVITY = 64UL,
      FI_EN_SPM_SUBSTATE_DIAGNOSTICS_ACTIVITY = 2147483712UL,
      FI_EN_SPM_SUBSTATE_NO_PHONE = 128UL,
      FI_EN_SPM_SUBSTATE_PHONE = 2147483776UL,
      FI_EN_SPM_SUBSTATE_NO_NETWORK_ACTIVITY = 256UL,
      FI_EN_SPM_SUBSTATE_NETWORK_ACTIVITY = 2147483904UL,
      FI_EN_SPM_SUBSTATE_CD_INSERT = 512UL,
      FI_EN_SPM_SUBSTATE_NO_DIM = 1024UL,
      FI_EN_SPM_SUBSTATE_DIM = 2147484672UL,
      FI_EN_SPM_SUBSTATE_CAN_BUS_WAKEUP = 2048UL,
      FI_EN_SPM_SUBSTATE_DOWNLOAD_PERMISSION = 4096UL,
      FI_EN_SPM_SUBSTATE_EJECT_KEY_DETECTED = 8192UL,
      FI_EN_SPM_SUBSTATE_DRIVER_DOOR = 16384UL,
      FI_EN_SPM_SUBSTATE_NO_UNDER_VOLTAGE = 32768UL,
      FI_EN_SPM_SUBSTATE_UNDER_VOLTAGE = 2147516416UL,
      FI_EN_SPM_SUBSTATE_EMERGENCY_OFF = 65536UL,
      FI_EN_SPM_SUBSTATE_1H_LOGIC_TIMEOUT = 131072UL,
      FI_EN_SPM_SUBSTATE_FAST_SHUTDOWN = 1048576UL,
      FI_EN_SPM_SUBSTATE_HW_IGNITION = 2097152UL,
      FI_EN_SPM_SUBSTATE_PHONE_MUTE = 4194304UL,
      FI_EN_SPM_SUBSTATE_MAUS_ACTIVITY = 8388608UL,
      FI_EN_SPM_SUBSTATE_PUSH_TO_TALK = 33554432UL,
      FI_EN_SPM_SUBSTATE_CR_TEMPERATURE = 67108864UL,
      FI_EN_SPM_SUBSTATE_NETWORK_FAILURE = 134217728UL,
      FI_EN_SPM_SUBSTATE_SAFE = 2415919104UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_OnOffReason(spm_fi_tcl_SPM_e32_OnOffReason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_OnOffReason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_OnOffReason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_SYSTEM_STATES : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_SYSTEM_STATES();
   enum tenType {
      FI_EN_SPM_SYSTEM_BACKGROUND = 5UL,
      FI_EN_SPM_SYSTEM_PREPARE_SHUTDOWN = 16UL,
      FI_EN_SPM_SYSTEM_SHUTDOWN = 17UL,
      FI_EN_SPM_SYSTEM_SUSPEND = 0UL,
      FI_EN_SPM_SYSTEM_STANDBY = 1UL,
      FI_EN_SPM_SYSTEM_OFF = 2UL,
      FI_EN_SPM_SYSTEM_DOWNLOAD = 3UL,
      FI_EN_SPM_SYSTEM_ON = 4UL,
      FI_EN_SPM_SYSTEM_DOOR_OPEN = 6UL,
      FI_EN_SPM_SYSTEM_IGNITION = 7UL,
      FI_EN_SPM_SYSTEM_DIAGNOSIS = 8UL,
      FI_EN_SPM_SYSTEM_STATE_PROFILE = 9UL,
      FI_EN_SPM_SYSTEM_MMI_STANDBY_PWR_SAVE_1 = 32UL,
      FI_EN_SPM_SYSTEM_MMI_STANDBY_PWR_SAVE_2 = 33UL,
      FI_EN_SPM_SYSTEM_MMI_STANDBY = 34UL,
      FI_EN_SPM_SYSTEM_MMI_ON = 35UL,
      FI_EN_SPM_SYSTEM_MMI_STANDBY_RESTRICTED = 36UL,
      FI_EN_SPM_SYSTEM_MMI_ON_DIAG = 37UL,
      FI_EN_SPM_SYSTEM_MMI_ON_TEL = 38UL,
      FI_EN_SPM_SYSTEM_MMI_ON_SWDL = 39UL,
      FI_EN_SPM_SYSTEM_MMI_STANDBY_CUSTOMER_SWDL = 40UL,
      FI_EN_SPM_SYSTEM_MMI_STANDBY_PWR_SAVE = 41UL,
      FI_EN_SPM_SYSTEM_OVERTEMP = 42UL,
      FI_EN_SPM_SYSTEM_SAFE = 43UL,
      FI_EN_SPM_SYSTEM_STATE_MAX = 44UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_SYSTEM_STATES(spm_fi_tcl_SPM_e32_SYSTEM_STATES::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_SYSTEM_STATES();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_SYSTEM_STATES& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_SYSPWR_BEHAVIOUR : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_SYSPWR_BEHAVIOUR();
   enum tenType {
      FI_EN_SPM_U32_SYSPWR_NORMAL = 0UL,
      FI_EN_SPM_U32_SYSPWR_BATTERY_LOW_SHUTDOWN = 1UL,
      FI_EN_SPM_U32_SYSPWR_BATTERY_LOW_WARNING = 2UL,
      FI_EN_SPM_U32_SYSPWR_PHONE_HANDOVER_WARN = 3UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_SYSPWR_BEHAVIOUR(spm_fi_tcl_SPM_e32_SYSPWR_BEHAVIOUR::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_SYSPWR_BEHAVIOUR();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_SYSPWR_BEHAVIOUR& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_HMI_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_HMI_STATE();
   enum tenType {
      FI_EN_SPM_U32_HMISTATE_SHOW_OFF = 1UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_HMI = 2UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_CLOCK = 3UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_GOODBYE = 4UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_PHONE = 5UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_CLIMATE_CLOCK = 6UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE = 7UL,
      FI_EN_SPM_U32_HMISTATE_ANIMATION = 8UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_ON_INFO_WARN = 9UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_OFF_PHONE = 10UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_OFF_OPS = 11UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_OFF_CLIMATE = 12UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_ON_INFO_OFF = 13UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_OFF_DIAG = 14UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF = 15UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_CLOCK_OVERTEMP = 16UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_CLIMATE_CLOCK_OVERTEMP = 17UL,
      FI_EN_SPM_U32_HMISTATE_SECURE_GREEN = 18UL,
      FI_EN_SPM_U32_HMISTATE_SECURE_YELLOW = 19UL,
      FI_EN_SPM_U32_HMISTATE_SECURE_RED = 20UL,
      FI_EN_SPM_U32_HMISTATE_SECURE_WAIT = 21UL,
      FI_EN_SPM_U32_HMISTATE_SLEEP_SCREEN = 22UL,
      FI_EN_SPM_U32_HMISTATE_SLEEP = 23UL,
      FI_EN_SPM_U32_HMISTATE_COMM_ENABLED = 24UL,
      FI_EN_SPM_U32_HMISTATE_INACTIVE = 25UL,
      FI_EN_SPM_U32_HMISTATE_FULL_OPERATION = 26UL,
      FI_EN_SPM_U32_HMISTATE_THEFTLOCKED = 27UL,
      FI_EN_SPM_U32_HMISTATE_NO_CALIBRATION = 28UL,
      FI_EN_SPM_U32_HMISTATE_NOVIN_MODE = 29UL,
      FI_EN_SPM_U32_HMISTATE_VALET_MODE = 30UL,
      FI_EN_SPM_U32_HMISTATE_LOCAL_PHONE = 31UL,
      FI_EN_SPM_U32_HMISTATE_LOCAL_INFORMATION = 32UL,
      FI_EN_SPM_U32_HMISTATE_LOCAL_EJECT = 33UL,
      FI_EN_SPM_U32_HMISTATE_LOCAL_INFOTAINMENT = 34UL,
      FI_EN_SPM_U32_HMISTATE_CRANK = 35UL,
      FI_EN_SPM_U32_HMISTATE_REMOTE_PROGRAMMING = 36UL,
      FI_EN_SPM_U32_HMISTATE_SEAT_BELT_RESTRICTIONS = 37UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_BATTERY_SAVER = 38UL,
      FI_EN_SPM_U32_HMISTATE_LOCAL_VALET = 39UL,
      FI_EN_SPM_U32_HMISTATE_SYSTEM_INIT = 40UL,
      FI_EN_SPM_U32_HMISTATE_STARTUP_ANIMATION = 41UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_OVERTEMPERATURE = 42UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_STANDBY_RESTRICTED = 43UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_STANDBY = 44UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_ON_PARTIAL_STANDBY = 45UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER = 46UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_ON_PARTIAL_DARK = 47UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_ON_FULL_STANDBY = 48UL,
      FI_EN_SPM_U32_HMISTATE_SHOW_ON_FULL_DARK = 49UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_HMI_STATE(spm_fi_tcl_SPM_e32_HMI_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_HMI_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_HMI_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_DIM_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_DIM_STATE();
   enum tenType {
      FI_EN_SPM_U32_DIMSTATE = 1UL,
      FI_EN_SPM_U32_DIMSTATE_CLIMATE = 2UL,
      FI_EN_SPM_U32_DIMSTATE_BACKLIGHT = 4UL,
      FI_EN_SPM_U32_DIMSTATE_HOUR_MODE = 8UL,
      FI_EN_SPM_U32_DIMSTATE_ALL_OFF = 16UL,
      FI_EN_SPM_U32_DIMSTATE_TLIN_ON_BACKLIGHT_OFF = 32UL,
      FI_EN_SPM_U32_DIMSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED = 64UL,
      FI_EN_SPM_U32_DIMSTATE_ALL_ON = 128UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_DIM_STATE(spm_fi_tcl_SPM_e32_DIM_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_DIM_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_DIM_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_b32_DIM_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_b32_DIM_STATE();

   enum {
      FI_C_U32_BIT_SPM_U32_DIMMINGSTATE = 0x1UL,
      FI_C_U32_BIT_SPM_U32_DIMSMINGTATE_CLIMATE = 0x2UL,
      FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_BACKLIGHT = 0x4UL,
      FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_HOUR_MODE = 0x8UL,
      FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_ALL_OFF = 0x10UL,
      FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_ON_BACKLIGHT_OFF = 0x20UL,
      FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED = 0x40UL,
      FI_C_U32_BIT_SPM_U32DIMMINGSTATE_ALL_ON = 0x80UL
   };
   tBool bSPM_U32_DIMMINGSTATE() const {return (u32Value & FI_C_U32_BIT_SPM_U32_DIMMINGSTATE) != 0; }
   tVoid vSetSPM_U32_DIMMINGSTATE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32_DIMMINGSTATE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32_DIMMINGSTATE)); }
   tBool bSPM_U32_DIMSMINGTATE_CLIMATE() const {return (u32Value & FI_C_U32_BIT_SPM_U32_DIMSMINGTATE_CLIMATE) != 0; }
   tVoid vSetSPM_U32_DIMSMINGTATE_CLIMATE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32_DIMSMINGTATE_CLIMATE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32_DIMSMINGTATE_CLIMATE)); }
   tBool bSPM_U32_DIMMINGSTATE_BACKLIGHT() const {return (u32Value & FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_BACKLIGHT) != 0; }
   tVoid vSetSPM_U32_DIMMINGSTATE_BACKLIGHT(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_BACKLIGHT):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_BACKLIGHT)); }
   tBool bSPM_U32_DIMMINGSTATE_HOUR_MODE() const {return (u32Value & FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_HOUR_MODE) != 0; }
   tVoid vSetSPM_U32_DIMMINGSTATE_HOUR_MODE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_HOUR_MODE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_HOUR_MODE)); }
   tBool bSPM_U32_DIMMINGSTATE_ALL_OFF() const {return (u32Value & FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_ALL_OFF) != 0; }
   tVoid vSetSPM_U32_DIMMINGSTATE_ALL_OFF(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_ALL_OFF):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_ALL_OFF)); }
   tBool bSPM_U32_DIMMINGSTATE_TLIN_ON_BACKLIGHT_OFF() const {return (u32Value & FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_ON_BACKLIGHT_OFF) != 0; }
   tVoid vSetSPM_U32_DIMMINGSTATE_TLIN_ON_BACKLIGHT_OFF(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_ON_BACKLIGHT_OFF):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_ON_BACKLIGHT_OFF)); }
   tBool bSPM_U32_DIMMINGSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED() const {return (u32Value & FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED) != 0; }
   tVoid vSetSPM_U32_DIMMINGSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32_DIMMINGSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED)); }
   tBool bSPM_U32DIMMINGSTATE_ALL_ON() const {return (u32Value & FI_C_U32_BIT_SPM_U32DIMMINGSTATE_ALL_ON) != 0; }
   tVoid vSetSPM_U32DIMMINGSTATE_ALL_ON(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_U32DIMMINGSTATE_ALL_ON):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_U32DIMMINGSTATE_ALL_ON)); }
   struct rBitValues {
      tBool SPM_U32_DIMMINGSTATE:1;
      tBool SPM_U32_DIMSMINGTATE_CLIMATE:1;
      tBool SPM_U32_DIMMINGSTATE_BACKLIGHT:1;
      tBool SPM_U32_DIMMINGSTATE_HOUR_MODE:1;
      tBool SPM_U32_DIMMINGSTATE_ALL_OFF:1;
      tBool SPM_U32_DIMMINGSTATE_TLIN_ON_BACKLIGHT_OFF:1;
      tBool SPM_U32_DIMMINGSTATE_TLIN_OFF_BACKLIGHT_UNCHANGED:1;
      tBool SPM_U32DIMMINGSTATE_ALL_ON:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
      tBool dummy16:1;
      tBool dummy17:1;
      tBool dummy18:1;
      tBool dummy19:1;
      tBool dummy20:1;
      tBool dummy21:1;
      tBool dummy22:1;
      tBool dummy23:1;
      tBool dummy24:1;
      tBool dummy25:1;
      tBool dummy26:1;
      tBool dummy27:1;
      tBool dummy28:1;
      tBool dummy29:1;
      tBool dummy30:1;
      tBool dummy31:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~spm_fi_tcl_SPM_b32_DIM_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_b32_DIM_STATE& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_BT_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_BT_STATE();
   enum tenType {
      FI_EN_BLUETOOTHSTATE_BT_OFF = 0UL,
      FI_EN_BLUETOOTHSTATE_BLOCK = 1UL,
      FI_EN_BLUETOOTHSTATE_VALET_MODE = 2UL,
      FI_EN_BLUETOOTHSTATE_DISCONNECT = 3UL,
      FI_EN_BLUETOOTHSTATE_NO_ACTION = 4UL,
      FI_EN_BLUETOOTHSTATE_NORMAL = 5UL,
      FI_EN_BLUETOOTHSTATE_NORMAL_SWITCH_OFF_BT = 6UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_BT_STATE(spm_fi_tcl_SPM_e32_BT_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_BT_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_BT_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_WIFI_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_WIFI_STATE();
   enum tenType {
      FI_EN_WIFISTATE_OFF = 0UL,
      FI_EN_WIFISTATE_BLOCK = 1UL,
      FI_EN_WIFISTATE_NORMAL = 2UL,
      FI_EN_WIFISTATE_NORMAL_SWITCH_OFF_WIFI = 3UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_WIFI_STATE(spm_fi_tcl_SPM_e32_WIFI_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_WIFI_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_WIFI_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_AUDIO_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_AUDIO_STATE();
   enum tenType {
      FI_EN_SPM_U32_AUDIOSTATE_OFF = 0UL,
      FI_EN_SPM_U32_AUDIOSTATE_EMERGENCY = 1UL,
      FI_EN_SPM_U32_AUDIOSTATE_NORMAL = 2UL,
      FI_EN_SPM_U32_AUDIOSTATE_LIMITED = 3UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_AUDIO_STATE(spm_fi_tcl_SPM_e32_AUDIO_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_AUDIO_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_AUDIO_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_SPI_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_SPI_STATE();
   enum tenType {
      FI_EN_SPM_U32_SPISTATE_OFF = 0UL,
      FI_EN_SPM_U32_SPISTATE_ON = 1UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_SPI_STATE(spm_fi_tcl_SPM_e32_SPI_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_SPI_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_SPI_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_TUNER_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_TUNER_STATE();
   enum tenType {
      FI_EN_SPM_U32_TUNER_OFF = 0UL,
      FI_EN_SPM_U32_TUNER_ON = 1UL,
      FI_EN_SPM_U32_TUNER_LANDSCAPE_SCAN = 2UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_TUNER_STATE(spm_fi_tcl_SPM_e32_TUNER_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_TUNER_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_TUNER_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_AUDIO_SOURCE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_AUDIO_SOURCE();
   enum tenType {
      FI_EN_SPM_U32_AUDIOSRC_NONE = 0UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_FM1 = 1UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_FM2 = 2UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_AM = 3UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_DAB = 4UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDCOMP = 5UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_SD = 6UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_BTAUDIO = 7UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_MDI = 10UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_AUXPLATE = 12UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_TA = 13UL,
      FI_EN_SPM_U32_AUDIOSRC_NAVISPEECH = 14UL,
      FI_EN_SPM_U32_AUDIOSRC_TELEPHONE = 15UL,
      FI_EN_SPM_U32_AUDIOSRC_TELEPHONE_MUTE = 16UL,
      FI_EN_SPM_U32_AUDIOSRC_OPS_PDC = 19UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDDA = 20UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_CDC = 21UL,
      FI_EN_SPM_U32_AUDIOSRC_ENTERTAINMENT_MUTE = 22UL,
      FI_EN_SPM_U32_AUDIOSRC_DIAGNOSIS_MUTE = 23UL,
      FI_EN_SPM_U32_AUDIOSRC_SYSTEM_MUTE = 24UL,
      FI_EN_SPM_U32_AUDIOSRC_CLUSTER_MUTE = 25UL,
      FI_EN_SPM_U32_AUDIOSRC_PIN_MUTE = 26UL,
      FI_EN_SPM_U32_AUDIOSRC_LIMITED_SYSTEM_MUTE = 27UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_IPOD = 28UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_USB = 29UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_XM = 30UL,
      FI_EN_SPM_U32_AUDIOSRC_NAVIINFO = 31UL,
      FI_EN_SPM_U32_AUDIOSRC_AUDIO_OFF_MUTE = 32UL,
      FI_EN_SPM_U32_AUDIOSRC_SDS = 33UL,
      FI_EN_SPM_U32_AUDIOSRC_DOV = 34UL,
      FI_EN_SPM_U32_AUDIOSRC_DOV_TEST = 35UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_PANODRA = 36UL,
      FI_EN_SPM_U32_AUDIOSRC_SDS_SMS = 37UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_DAB_TA = 38UL,
      FI_EN_SPM_U32_AUDIOSRC_RADIO_DABSERVICEFOLLOWING = 39UL,
      FI_EN_SPM_U32_AUDIOSRC_SPI_BT_AUDIO = 40UL,
      FI_EN_SPM_U32_AUDIOSRC_SPI_USB = 41UL,
      FI_EN_SPM_U32_AUDIOSRC_SPI_VR_USB = 42UL,
      FI_EN_SPM_U32_AUDIOSRC_SPI_VR_BT_AUDIO = 43UL,
      FI_EN_SPM_U32_AUDIOSRC_SPI_TTS = 44UL,
      FI_EN_SPM_U32_AUDIOSRC_MEDIA_DB_PLAYER = 45UL,
      FI_EN_SPM_U32_AUDIOSRC_CUE = 46UL,
      FI_EN_SPM_U32_AUDIOSRC_VOLTAGE_MUTE = 47UL,
      FI_EN_SPM_U32_AUDIOSRC_AUD_DEV__NOT_AVAIL = 48UL,
      FI_EN_SPM_U32_AUDIOSRC_MIC_LOOPBACK = 49UL,
      FI_EN_SPM_U32_AUDIOSRC_IPA_MUTE = 50UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_AUDIO_SOURCE(spm_fi_tcl_SPM_e32_AUDIO_SOURCE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_AUDIO_SOURCE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_AUDIO_SOURCE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_CD_INSERT_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_CD_INSERT_STATE();
   enum tenType {
      FI_EN_SPM_U32_CD_INSERT_INACTIVE = 0UL,
      FI_EN_SPM_U32_CD_INSERT_ACTIVE = 1UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_CD_INSERT_STATE(spm_fi_tcl_SPM_e32_CD_INSERT_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_CD_INSERT_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_CD_INSERT_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_SYS_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_SYS_STATE();
   enum tenType {
      FI_EN_SPM_U32_SYSSTATE_SLEEP = 0UL,
      FI_EN_SPM_U32_SYSSTATE_COMM_ENABLED = 1UL,
      FI_EN_SPM_U32_SYSSTATE_INACTIVE = 2UL,
      FI_EN_SPM_U32_SYSSTATE_FULL_OPERATION = 3UL,
      FI_EN_SPM_U32_SYSSTATE_THEFTLOCKED = 4UL,
      FI_EN_SPM_U32_SYSSTATE_NO_CALIBRATION = 5UL,
      FI_EN_SPM_U32_SYSSTATE_NOVIN_MODE = 6UL,
      FI_EN_SPM_U32_SYSSTATE_VALET_MODE = 7UL,
      FI_EN_SPM_U32_SYSSTATE_LOCAL_PHONE = 8UL,
      FI_EN_SPM_U32_SYSSTATE_LOCAL_INFORMATION = 9UL,
      FI_EN_SPM_U32_SYSSTATE_LOCAL_EJECT = 10UL,
      FI_EN_SPM_U32_SYSSTATE_LOCAL_INFOTAINMENT = 11UL,
      FI_EN_SPM_U32_SYSSTATE_CRANK = 12UL,
      FI_EN_SPM_U32_SYSSTATE_REMOTE_PROGRAMMING = 13UL,
      FI_EN_SPM_U32_SYSSTATE_SEAT_BELT_RESTRICTIONS = 14UL,
      FI_EN_SPM_U32_SYSSTATE_SHOW_BATTERY_SAVER = 15UL,
      FI_EN_SPM_U32_SYSSTATE_LOCAL_VALET = 16UL,
      FI_EN_SPM_U32_SYSSTATE_SYSTEM_INIT = 17UL,
      FI_EN_SPM_U32_SYSSTATE_STARTUP_ANIMATION = 18UL,
      FI_EN_SPM_U32_SYSSTATE_SHOW_OFF = 128UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_SYS_STATE(spm_fi_tcl_SPM_e32_SYS_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_SYS_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_SYS_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_MOST_EVENT : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_MOST_EVENT();
   enum tenType {
      FI_EN_SPM_U32_MOST_EVENT_NO_EVENT = 0UL,
      FI_EN_SPM_U32_MOST_EVENT_NET_ON = 1UL,
      FI_EN_SPM_U32_MOST_EVENT_NET_OFF = 2UL,
      FI_EN_SPM_U32_MOST_EVENT_UNLOCK = 3UL,
      FI_EN_SPM_U32_MOST_EVENT_LOCK_STABLE = 4UL,
      FI_EN_SPM_U32_MOST_EVENT_SHUTDOWN_EXECUTE = 5UL,
      FI_EN_SPM_U32_MOST_EVENT_CONFIG_STATE = 6UL,
      FI_EN_SPM_U32_MOST_EVENT_RING_OFF = 7UL,
      FI_EN_SPM_U32_MOST_EVENT_RING_ON = 8UL,
      FI_EN_SPM_U32_MOST_EVENT_TLA_STATE = 9UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_MOST_EVENT(spm_fi_tcl_SPM_e32_MOST_EVENT::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_MOST_EVENT();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_MOST_EVENT& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_MOST_CONFIG_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_MOST_CONFIG_STATE();
   enum tenType {
      FI_EN_SPM_U32_MOST_CONFIG_STATE_UNAVAILABLE = 0UL,
      FI_EN_SPM_U32_MOST_CONFIG_STATE_NOT_OK = 1UL,
      FI_EN_SPM_U32_MOST_CONFIG_STATE_OK = 2UL,
      FI_EN_SPM_U32_MOST_CONFIG_STATE_INVALID = 3UL,
      FI_EN_SPM_U32_MOST_CONFIG_STATE_NEW = 4UL,
      FI_EN_SPM_U32_MOST_CONFIG_STATE_NEW_EXT = 5UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_MOST_CONFIG_STATE(spm_fi_tcl_SPM_e32_MOST_CONFIG_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_MOST_CONFIG_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_MOST_CONFIG_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_TLA_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_TLA_STATE();
   enum tenType {
      FI_EN_SPM_U32_TLA_STATE_UNAVAILABLE = 0UL,
      FI_EN_SPM_U32_TLA_STATE_DISCONNECTED = 1UL,
      FI_EN_SPM_U32_TLA_STATE_CONNECTED = 2UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_TLA_STATE(spm_fi_tcl_SPM_e32_TLA_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_TLA_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_TLA_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_PROCESSOR_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_PROCESSOR_STATE();
   enum tenType {
      FI_EN_SPM_U32_PROC_STATE_NORMAL = 0UL,
      FI_EN_SPM_U32_PROC_STATE_DNL = 1UL,
      FI_EN_SPM_U32_PROC_STATE_TM = 2UL,
      FI_EN_SPM_U32_PROC_STATE_STANDBY = 3UL,
      FI_EN_SPM_U32_PROC_STATE_DOWN = 255UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_PROCESSOR_STATE(spm_fi_tcl_SPM_e32_PROCESSOR_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_PROCESSOR_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_PROCESSOR_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_STARTUP_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_STARTUP_STATE();
   enum tenType {
      FI_EN_SPM_U32_STARTUP_STATE_INIT = 0UL,
      FI_EN_SPM_U32_STARTUP_STATE_BASE = 1UL,
      FI_EN_SPM_U32_STARTUP_STATE_VD = 2UL,
      FI_EN_SPM_U32_STARTUP_STATE_FC = 4UL,
      FI_EN_SPM_U32_STARTUP_STATE_NAVI = 8UL,
      FI_EN_SPM_U32_STARTUP_STATE_SDS = 16UL,
      FI_EN_SPM_U32_STARTUP_STATE_HMI = 32UL,
      FI_EN_SPM_U32_STARTUP_STATE_MM = 64UL,
      FI_EN_SPM_U32_STARTUP_STATE_MAP = 128UL,
      FI_EN_SPM_U32_STARTUP_STATE_TUNER = 256UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_STARTUP_STATE(spm_fi_tcl_SPM_e32_STARTUP_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_STARTUP_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_STARTUP_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_HistoryTrigger : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_HistoryTrigger();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_HistoryTrigger& operator=(const spm_fi_tcl_SPM_HistoryTrigger& coRef);
   spm_fi_tcl_SPM_HistoryTrigger(const spm_fi_tcl_SPM_HistoryTrigger& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32Timestamp;
   spm_fi_tcl_SPM_e32_SubStateType eTriggerType;
   virtual ~spm_fi_tcl_SPM_HistoryTrigger();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_HistoryTrigger& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_RESTART_TYPE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_RESTART_TYPE();
   enum tenType {
      FI_EN_SPM_U32_RESTART_ECU = 0UL,
      FI_EN_SPM_U32_RESTART_WDT = 1UL,
      FI_EN_SPM_U32_RESTART_ECU_FAST = 2UL,
      FI_EN_SPM_U32_RESTART_DOWNLOAD_ERROR = 3UL,
      FI_EN_SPM_U32_RESTART_DOWNLOAD_FINISHED = 4UL,
      FI_EN_SPM_U32_RESTART_ECU_WITH_SYNC = 5UL,
      FI_EN_SPM_U32_RESTART_DOWNLOAD_ABORTED = 6UL,
      FI_EN_SPM_U32_RESTART_ECU_ALTSW_TESTMANAGER = 7UL,
      FI_EN_SPM_U32_SYNC_WITHOUT_RESTART = 8UL,
      FI_EN_SPM_U32_RESTART_APP = 9UL,
      FI_EN_SPM_U32_RESTART_APP_FAST = 10UL,
      FI_EN_SPM_U32_RESTART_APP_WITH_SYNC = 11UL,
      FI_EN_SPM_U32_RESTART_SCC = 12UL,
      FI_EN_SPM_U32_RESTART_USER_RESET = 13UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_RESTART_TYPE(spm_fi_tcl_SPM_e32_RESTART_TYPE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_RESTART_TYPE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_RESTART_TYPE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_CODE_REQUEST : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_CODE_REQUEST();
   enum tenType {
      FI_EN_SPM_U32_CODE_IDLE = 0UL,
      FI_EN_SPM_U32_CODE_REQUESTED = 1UL,
      FI_EN_SPM_U32_CODE_PROCESSED = 2UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_CODE_REQUEST(spm_fi_tcl_SPM_e32_CODE_REQUEST::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_CODE_REQUEST();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_CODE_REQUEST& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_WAKEUP_REASON : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_WAKEUP_REASON();
   enum tenType {
      FI_EN_SPM_U32_WAKEUP_INVALID = 0UL,
      FI_EN_SPM_U32_WAKEUP_CAN = 1UL,
      FI_EN_SPM_U32_WAKEUP_EJECT = 2UL,
      FI_EN_SPM_U32_WAKEUP_INSERT = 3UL,
      FI_EN_SPM_U32_WAKEUP_IGNITION = 4UL,
      FI_EN_SPM_U32_WAKEUP_RTC = 5UL,
      FI_EN_SPM_U32_WAKEUP_POWER_ON = 6UL,
      FI_EN_SPM_U32_WAKEUP_ON_TIPPER = 7UL,
      FI_EN_SPM_U32_WAKEUP_MOST = 8UL,
      FI_EN_SPM_U32_WAKEUP_DEBUG_WAKEUP = 9UL,
      FI_EN_SPM_U32_WAKEUP_DOOR_OPEN = 10UL,
      FI_EN_SPM_U32_SDCARD_INSERT = 11UL,
      FI_EN_SPM_U32_WAKEUP_PHONE_MUTE = 12UL,
      FI_EN_SPM_U32_WAKEUP_ACCESSORY = 13UL,
      FI_EN_SPM_U32_WAKEUP_CELLNETWORK = 14UL,
      FI_EN_SPM_U32_WAKEUP_CAN2 = 15UL,
      FI_EN_SPM_U32_WAKEUP_CAN3 = 16UL,
      FI_EN_SPM_U32_WAKEUP_CAN4 = 17UL,
      FI_EN_SPM_U32_WAKEUP_ODOMETER = 18UL,
      FI_EN_SPM_U32_WAKEUP_LIN = 19UL,
      FI_EN_SPM_U32_WAKEUP_ACCELLERATOR_SENSOR = 20UL,
      FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO = 21UL,
      FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO2 = 22UL,
      FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO3 = 23UL,
      FI_EN_SPM_U32_WAKEUP_EXTERNAL_GPIO4 = 24UL,
      FI_EN_SPM_U32_WAKEUP_BLUETOOTH = 25UL,
      FI_EN_SPM_U32_WAKEUP_USB = 26UL,
      FI_EN_SPM_U32_WAKEUP_UART = 27UL,
      FI_EN_SPM_U32_WAKEUP_UART2 = 28UL,
      FI_EN_SPM_U32_WAKEUP_ILLUMINATION = 29UL,
      FI_EN_SPM_U32_WAKEUP_RESET = 255UL,
      FI_EN_SPM_U32_WAKEUP_RESTART = 4294967295UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_WAKEUP_REASON(spm_fi_tcl_SPM_e32_WAKEUP_REASON::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_WAKEUP_REASON();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_WAKEUP_REASON& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE();
   enum tenType {
      FI_EN_SPM_U32_EJECT_RELEASED = 0UL,
      FI_EN_SPM_U32_EJECT_PRESSED = 2UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE(spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_KEY_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_KEY_STATE();
   enum tenType {
      FI_EN_SPM_U32_RELEASED = 0UL,
      FI_EN_SPM_U32_PRESSED = 1UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_KEY_STATE(spm_fi_tcl_SPM_e32_KEY_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_KEY_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_KEY_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_SECURITY_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_SECURITY_STATE();
   enum tenType {
      FI_EN_SPM_U32_SECURITY_STATE_NO_VIN = 0UL,
      FI_EN_SPM_U32_SECURITY_STATE_HAS_VIN = 1UL,
      FI_EN_SPM_U32_SECURITY_STATE_THEFT_PROTECTED = 2UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_SECURITY_STATE(spm_fi_tcl_SPM_e32_SECURITY_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_SECURITY_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_SECURITY_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_STARTUP_REASON : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_STARTUP_REASON();
   enum tenType {
      FI_EN_SPM_U32_STARTUP_ON_EJECT = 1UL,
      FI_EN_SPM_U32_STARTUP_ON_INSERT = 2UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_STARTUP_REASON(spm_fi_tcl_SPM_e32_STARTUP_REASON::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_STARTUP_REASON();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_STARTUP_REASON& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE();
   enum tenType {
      FI_EN_SPM_U32_PHONEMUTE_INACTIVE = 0UL,
      FI_EN_SPM_U32_PHONEMUTE_ACTIVE = 1UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE(spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_ResetReason : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_ResetReason();
   enum tenType {
      FI_EN_SPM_EHMI_COPRO_NO_REASON = 0UL,
      FI_EN_SPM_EHMI_COPRO_SPI_ERROR = 1UL,
      FI_EN_SPM_EHMI_COPRO_WATCHDOG_TIMEOUT = 2UL,
      FI_EN_SPM_EHMI_COPRO_SMACK_TIMEOUT = 3UL,
      FI_EN_SPM_EHMI_COPRO_RSTOUT_INACTIVE = 4UL,
      FI_EN_SPM_EHMI_COPRO_RSTOUT_ACTIVE_TIMEOUT = 5UL,
      FI_EN_SPM_EHMI_COPRO_RSTOUT_INACTIVE_TIMEOUT = 6UL,
      FI_EN_SPM_EHMI_COPRO_WAKEUP_MSG_TIMEOUT = 7UL,
      FI_EN_SPM_EHMI_COPRO_NO_MODE_TIMEOUT = 8UL,
      FI_EN_SPM_EHMI_APP_NOT_INITIALIZED = 9UL,
      FI_EN_SPM_EHMI_APP_NOT_ACK = 10UL,
      FI_EN_SPM_EHMI_APP_END_WANTED = 11UL,
      FI_EN_SPM_EHMI_APP_WDT_ERROR = 12UL,
      FI_EN_SPM_EHMI_APP_NOTIFY_ERROR = 13UL,
      FI_EN_SPM_EHMI_APP_DOWNLOAD_ERROR = 14UL,
      FI_EN_SPM_EHMI_APP_DOWNLOAD = 15UL,
      FI_EN_SPM_EHMI_WDT = 16UL,
      FI_EN_SPM_EHMI_MPU_WDT = 17UL,
      FI_EN_SPM_EHMI_DSP_WDT = 18UL,
      FI_EN_SPM_EHMI_RST_SOFTWARE = 19UL,
      FI_EN_SPM_EHMI_MPU = 20UL,
      FI_EN_SPM_EHMI_RST_UNKNOWN = 21UL,
      FI_EN_SPM_EHMI_EXTERN = 22UL,
      FI_EN_SPM_EHMI_RST_USER = 23UL,
      FI_EN_SPM_ERU_UNKNOWN = 24UL,
      FI_EN_SPM_ERU_UNDERVOLTAGE = 25UL,
      FI_EN_SPM_ERU_HMI = 26UL,
      FI_EN_SPM_EFGS_HMI = 27UL,
      FI_EN_SPM_EFGS_FGS = 28UL,
      FI_EN_SPM_EHMI_RST_OVERTEMPERATURE = 29UL,
      FI_EN_SPM_EHMI_SLV_SUPERVISION = 30UL,
      FI_EN_SPM_EHMI_RST_ECU = 31UL,
      FI_EN_SPM_EHMI_ECU_FAST = 32UL,
      FI_EN_SPM_EHMI_APP_WDT_HEARTBEAT_ERROR = 33UL,
      FI_EN_SPM_EHMI_SLV_PROCESS_SUPERVISION = 34UL,
      FI_EN_SPM_EHMI_SLV_REQUESTED_RESET = 35UL,
      FI_EN_SPM_EHMI_RST_POWER_ON_RESET = 36UL
   };
   tenType enType;

   spm_fi_tcl_e8_ResetReason(spm_fi_tcl_e8_ResetReason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_ResetReason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_ResetReason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_LcmResetReason : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_LcmResetReason();
   enum tenType {
      FI_EN_SPM_EHMI_APP_ERROR = 0UL,
      FI_EN_SPM_EHMI_ECU = 1UL,
      FI_EN_SPM_EHMI_USER = 2UL,
      FI_EN_SPM_EHMI_DOWNLOAD = 3UL,
      FI_EN_SPM_EHMI_DOWNLOAD_ERROR = 4UL,
      FI_EN_SPM_EHMI_SOFTWARE = 5UL,
      FI_EN_SPM_EHMI_UNKNOWN = 6UL,
      FI_EN_SPM_EHMI_OVERTEMPERATURE = 7UL,
      FI_EN_SPM_EHMI_LCM_INTERN = 8UL,
      FI_EN_SPM_EHMI_POWER_ON_RESET = 60UL,
      FI_EN_SPM_EHMI_POWER_LOSS = 61UL,
      FI_EN_SPM_EHMI_POWER_OFF = 62UL,
      FI_EN_SPM_EHMI_SCC = 63UL,
      FI_EN_SPM_EHMI_WARMSTART = 99UL,
      FI_EN_SPM_ESCC_HW_WATCHDOG = 100UL,
      FI_EN_SPM_ESCC_POWER_ON = 101UL,
      FI_EN_SPM_ESCC_COLDSTART = 102UL,
      FI_EN_SPM_ESCC_APPMODE_CHANGE = 103UL,
      FI_EN_SPM_ESCC_DURING_LPW = 104UL,
      FI_EN_SPM_ESCC_DURING_PLL_OSZ = 105UL,
      FI_EN_SPM_ESCC_DURING_SW = 106UL,
      FI_EN_SPM_ESCC_WARMSTART = 107UL,
      FI_EN_SPM_ESCC_USER = 108UL
   };
   tenType enType;

   spm_fi_tcl_e8_LcmResetReason(spm_fi_tcl_e8_LcmResetReason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_LcmResetReason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_LcmResetReason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_CVM_EVENT : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_CVM_EVENT();
   enum tenType {
      FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_START = 0UL,
      FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_END = 1UL,
      FI_EN_SPM_U32_CVM_LOW_VOLTAGE_START = 2UL,
      FI_EN_SPM_U32_CVM_LOW_VOLTAGE_END = 3UL,
      FI_EN_SPM_U32_CVM_HIGH_VOLTAGE_START = 4UL,
      FI_EN_SPM_U32_CVM_HIGH_VOLTAGE_END = 5UL,
      FI_EN_SPM_U32_CVM_CRITICAL_HIGH_VOLTAGE_START = 6UL,
      FI_EN_SPM_U32_CVM_CRITICAL_HIGH_VOLTAGE_END = 7UL,
      FI_EN_SPM_U32_CVM_NO_EVENT = 15UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_CVM_EVENT(spm_fi_tcl_SPM_e32_CVM_EVENT::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_CVM_EVENT();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_CVM_EVENT& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_SYSPWRMODE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_SYSPWRMODE();
   enum tenType {
      FI_EN_SPM_U32_SYSPWRMODE_UNDEFINED = 0UL,
      FI_EN_SPM_U32_SYSPWRMODE_OFF = 1UL,
      FI_EN_SPM_U32_SYSPWRMODE_ACCESSORY = 2UL,
      FI_EN_SPM_U32_SYSPWRMODE_RUN = 3UL,
      FI_EN_SPM_U32_SYSPWRMODE_CRANK = 4UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_SYSPWRMODE(spm_fi_tcl_SPM_e32_SYSPWRMODE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_SYSPWRMODE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_SYSPWRMODE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e8_WAKEUP_LIST_REASON : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e8_WAKEUP_LIST_REASON();
   enum tenType {
      FI_EN_SPM_U8_WAKEUP_INVALID = 0UL,
      FI_EN_SPM_U8_WAKEUP_CAN = 1UL,
      FI_EN_SPM_U8_WAKEUP_EJECT = 2UL,
      FI_EN_SPM_U8_WAKEUP_INSERT = 3UL,
      FI_EN_SPM_U8_WAKEUP_IGNITION = 4UL,
      FI_EN_SPM_U8_WAKEUP_RTC = 5UL,
      FI_EN_SPM_U8_WAKEUP_POWER_ON = 6UL,
      FI_EN_SPM_U8_WAKEUP_ON_TIPPER = 7UL,
      FI_EN_SPM_U8_WAKEUP_COPRO_REQ = 8UL,
      FI_EN_SPM_U8_WAKEUP_MAUS_BUS = 9UL,
      FI_EN_SPM_U8_WAKEUP_DOOR_OPEN = 10UL,
      FI_EN_SPM_U8_SDCARD_INSERT = 11UL,
      FI_EN_SPM_U8_WAKEUP_PHONE_MUTE = 12UL,
      FI_EN_SPM_U8_WAKEUP_DIMMING = 13UL,
      FI_EN_SPM_U8_WAKEUP_MOST = 14UL,
      FI_EN_SPM_U8_WAKEUP_RESTART = 255UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e8_WAKEUP_LIST_REASON(spm_fi_tcl_SPM_e8_WAKEUP_LIST_REASON::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e8_WAKEUP_LIST_REASON();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e8_WAKEUP_LIST_REASON& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_TransPortMode : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_TransPortMode();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_TransPortMode& operator=(const spm_fi_tcl_SPM_TransPortMode& coRef);
   spm_fi_tcl_SPM_TransPortMode(const spm_fi_tcl_SPM_TransPortMode& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 YEAR;
   tU32 MONTH;
   tU32 DAY;
   tU32 HOUR;
   tU32 MINUTE;
   virtual ~spm_fi_tcl_SPM_TransPortMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_TransPortMode& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_TransPortModeList : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_TransPortModeList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_TransPortModeList& operator=(const spm_fi_tcl_SPM_TransPortModeList& coRef);
   spm_fi_tcl_SPM_TransPortModeList(const spm_fi_tcl_SPM_TransPortModeList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<spm_fi_tcl_SPM_TransPortMode, std::allocator<spm_fi_tcl_SPM_TransPortMode> > TransPortModeList;
   virtual ~spm_fi_tcl_SPM_TransPortModeList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_TransPortModeList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_StateModes : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_StateModes();
   enum tenType {
      FI_EN_SPM_E8_SYSMODEDOWNLOAD = 0UL,
      FI_EN_SPM_E8_SYSMODEDIAG = 1UL,
      FI_EN_SPM_E8_SYSMODEDOWNLOADEXIT = 2UL,
      FI_EN_SPM_E8_SYSMODECUSTOMERDOWNLOADSTART = 3UL,
      FI_EN_SPM_E8_SYSMODECUSTOMERDOWNLOADEXIT = 4UL,
      FI_EN_SPM_E8_SYSMODEDIAGACTIVITYSTART = 5UL,
      FI_EN_SPM_E8_SYSMODEDIAGACTIVITYEXIT = 6UL,
      FI_EN_SPM_E8_SYSMODEDIAGEXIT = 7UL,
      FI_EN_SPM_E8_SYSMODEMAPDOWNLOADLOCK = 8UL,
      FI_EN_SPM_E8_SYSMODEMAPDOWNLOADUNLOCK = 9UL,
      FI_EN_SPM_E8_SYSMODEMAPDOWNLOADMASTERUNLOCK = 10UL
   };
   tenType enType;

   spm_fi_tcl_e8_StateModes(spm_fi_tcl_e8_StateModes::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_StateModes();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_StateModes& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_StateEntryType : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_StateEntryType();
   enum tenType {
      FI_EN_SPM_E8_HISTORYENTRYWAKEUP = 0UL,
      FI_EN_SPM_E8_HISTORYENTRYSUBSTATE = 1UL,
      FI_EN_SPM_E8_HISTORYENTRYRESET = 2UL
   };
   tenType enType;

   spm_fi_tcl_e8_StateEntryType(spm_fi_tcl_e8_StateEntryType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_StateEntryType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_StateEntryType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SubStateEntry : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SubStateEntry();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SubStateEntry& operator=(const spm_fi_tcl_SubStateEntry& coRef);
   spm_fi_tcl_SubStateEntry(const spm_fi_tcl_SubStateEntry& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tcl_b32_SubState0 tSubStateMask0;
   spm_fi_tcl_b32_SubState0 tSubStateData0;
   spm_fi_tcl_b32_SubState1 tSubStateMask1;
   spm_fi_tcl_b32_SubState1 tSubStateData1;
   spm_fi_tcl_b32_SubState2 tSubStateMask2;
   spm_fi_tcl_b32_SubState2 tSubStateData2;
   spm_fi_tcl_b32_SubState3 tSubStateMask3;
   spm_fi_tcl_b32_SubState3 tSubStateData3;
   virtual ~spm_fi_tcl_SubStateEntry();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SubStateEntry& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_DownloadLockStates : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_DownloadLockStates();
   enum tenType {
      FI_EN_SPM_E8_DOWNLOADLOCK = 0UL,
      FI_EN_SPM_E8_DOWNLOADUNLOCK = 1UL,
      FI_EN_SPM_E8_DOWNLOADMASTERUNLOCK = 2UL
   };
   tenType enType;

   spm_fi_tcl_e8_DownloadLockStates(spm_fi_tcl_e8_DownloadLockStates::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_DownloadLockStates();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_DownloadLockStates& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_ProcessState : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_ProcessState();
   enum tenType {
      FI_EN_SPM_E8_PROCESSTOSTART = 0UL,
      FI_EN_SPM_E8_PROCESSRUNNING = 1UL,
      FI_EN_SPM_E8_PROCESSDOWN = 2UL,
      FI_EN_SPM_E8_PROCESSCONFIGUREDOUT = 3UL
   };
   tenType enType;

   spm_fi_tcl_e8_ProcessState(spm_fi_tcl_e8_ProcessState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_ProcessState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_ProcessState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_ProcessElement : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_ProcessElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_ProcessElement& operator=(const spm_fi_tcl_ProcessElement& coRef);
   spm_fi_tcl_ProcessElement(const spm_fi_tcl_ProcessElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclString tProcessName;
   spm_fi_tclString tConfiguredProcessName;
   spm_fi_tcl_e8_ProcessState eProcState;
   tU32 u32StartTimeSinceStart;
   spm_fi_tclTimeDate tSwitchOnDate;
   virtual ~spm_fi_tcl_ProcessElement();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_ProcessElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_ProcessInfo : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_ProcessInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_ProcessInfo& operator=(const spm_fi_tcl_SPM_ProcessInfo& coRef);
   spm_fi_tcl_SPM_ProcessInfo(const spm_fi_tcl_SPM_ProcessInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<spm_fi_tcl_ProcessElement, std::allocator<spm_fi_tcl_ProcessElement> > tProcesses;
   virtual ~spm_fi_tcl_SPM_ProcessInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_ProcessInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_Vin : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_Vin();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_Vin& operator=(const spm_fi_tcl_SPM_Vin& coRef);
   spm_fi_tcl_SPM_Vin(const spm_fi_tcl_SPM_Vin& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<tU8, std::allocator<tU8> > Vin;
   virtual ~spm_fi_tcl_SPM_Vin();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_Vin& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_VinHistory : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_VinHistory();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_VinHistory& operator=(const spm_fi_tcl_SPM_VinHistory& coRef);
   spm_fi_tcl_SPM_VinHistory(const spm_fi_tcl_SPM_VinHistory& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<spm_fi_tcl_SPM_Vin, std::allocator<spm_fi_tcl_SPM_Vin> > VinHistory;
   virtual ~spm_fi_tcl_SPM_VinHistory();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_VinHistory& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_ResetElement : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_ResetElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_ResetElement& operator=(const spm_fi_tcl_ResetElement& coRef);
   spm_fi_tcl_ResetElement(const spm_fi_tcl_ResetElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclTimeDate tResetDate;
   spm_fi_tclSwVersionInfo tSwVersionInfo;
   spm_fi_tcl_e8_LcmResetReason eReset;
   virtual ~spm_fi_tcl_ResetElement();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_ResetElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_WakeupElement : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_WakeupElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_WakeupElement& operator=(const spm_fi_tcl_WakeupElement& coRef);
   spm_fi_tcl_WakeupElement(const spm_fi_tcl_WakeupElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclTimeDate tWakeupDate;
   spm_fi_tcl_SPM_e32_WAKEUP_REASON eWakeupReason;
   virtual ~spm_fi_tcl_WakeupElement();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_WakeupElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_StateChangeElement : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_StateChangeElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_StateChangeElement& operator=(const spm_fi_tcl_StateChangeElement& coRef);
   spm_fi_tcl_StateChangeElement(const spm_fi_tcl_StateChangeElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclTimeDate tWakeupDate;
   spm_fi_tcl_SPM_e32_SYSTEM_STATES tSysState;
   spm_fi_tcl_e8_StateEntryType tElemType;
   spm_fi_tcl_SPM_e32_WAKEUP_REASON tWakeup;
   spm_fi_tcl_e8_LcmResetReason tReset;
   spm_fi_tcl_SubStateEntry tSubStates;
   virtual ~spm_fi_tcl_StateChangeElement();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_StateChangeElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SwitchOnElement : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SwitchOnElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SwitchOnElement& operator=(const spm_fi_tcl_SwitchOnElement& coRef);
   spm_fi_tcl_SwitchOnElement(const spm_fi_tcl_SwitchOnElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclTimeDate tSwitchOnDate;
   spm_fi_tcl_SPM_e32_SubStateType eSwitchOnReason;
   virtual ~spm_fi_tcl_SwitchOnElement();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SwitchOnElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_START_PROC_TYPE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_START_PROC_TYPE();
   enum tenType {
      FI_EN_SPM_U32_START_PROC_TYPE_OSAL_SPAWN = 0UL,
      FI_EN_SPM_U32_START_PROC_TYPE_NATIVE_SPAWN = 1UL,
      FI_EN_SPM_U32_START_PROC_TYPE_SCRIPT = 2UL,
      FI_EN_SPM_U32_START_PROC_TYPE_ADD_SUPERVISION = 3UL,
      FI_EN_SPM_U32_START_PROC_TYPE_SYSTEMD = 4UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_START_PROC_TYPE(spm_fi_tcl_SPM_e32_START_PROC_TYPE::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_START_PROC_TYPE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_START_PROC_TYPE& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_MRESULT : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_MRESULT();
   enum tenType {
      FI_EN_SPM_U32_MRESULT_NOK = 4294967295UL,
      FI_EN_SPM_U32_MRESULT_OK = 0UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_MRESULT(spm_fi_tcl_SPM_e32_MRESULT::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_MRESULT();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_MRESULT& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_RUNLEVEL : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_RUNLEVEL();
   enum tenType {
      FI_EN_SPM_U32_RUNLEVEL_LCM_STARTED = 0UL,
      FI_EN_SPM_U32_RUNLEVEL_SYSTEM_STARTED = 1UL,
      FI_EN_SPM_U32_RUNLEVEL_SYSTEM_UP = 2UL,
      FI_EN_SPM_U32_RUNLEVEL_SYSTEM_RUNNING = 3UL,
      FI_EN_SPM_U32_RUNLEVEL_RESTART_REQUESTED = 4UL,
      FI_EN_SPM_U32_RUNLEVEL_SYNC_STARTED = 5UL,
      FI_EN_SPM_U32_RUNLEVEL_RESTART_NOW = 6UL,
      FI_EN_SPM_U32_RUNLEVEL_SHUTDOWN_STARTED = 7UL,
      FI_EN_SPM_U32_RUNLEVEL_SHUTDOWN_UNMOUNT = 8UL,
      FI_EN_SPM_U32_RUNLEVEL_SHUTDOWN_NOW = 9UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_RUNLEVEL(spm_fi_tcl_SPM_e32_RUNLEVEL::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_RUNLEVEL();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_RUNLEVEL& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_b32_SYSPWRSAVEMODE : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_b32_SYSPWRSAVEMODE();

   enum {
      FI_C_U32_BIT_SPM_E32_TRANSPORTMODE = 0x1UL,
      FI_C_U32_BIT_SPM_E32_ROLLER = 0x2UL,
      FI_C_U32_BIT_SPM_E32_FACTORY = 0x4UL,
      FI_C_U32_BIT_SPM_E32_QCRITICAL = 0x8UL
   };
   tBool bSPM_e32_TRANSPORTMODE() const {return (u32Value & FI_C_U32_BIT_SPM_E32_TRANSPORTMODE) != 0; }
   tVoid vSetSPM_e32_TRANSPORTMODE(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_TRANSPORTMODE):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_TRANSPORTMODE)); }
   tBool bSPM_e32_ROLLER() const {return (u32Value & FI_C_U32_BIT_SPM_E32_ROLLER) != 0; }
   tVoid vSetSPM_e32_ROLLER(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_ROLLER):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_ROLLER)); }
   tBool bSPM_e32_FACTORY() const {return (u32Value & FI_C_U32_BIT_SPM_E32_FACTORY) != 0; }
   tVoid vSetSPM_e32_FACTORY(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_FACTORY):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_FACTORY)); }
   tBool bSPM_e32_QCRITICAL() const {return (u32Value & FI_C_U32_BIT_SPM_E32_QCRITICAL) != 0; }
   tVoid vSetSPM_e32_QCRITICAL(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_SPM_E32_QCRITICAL):(u32Value & ~(tU32)FI_C_U32_BIT_SPM_E32_QCRITICAL)); }
   struct rBitValues {
      tBool SPM_e32_TRANSPORTMODE:1;
      tBool SPM_e32_ROLLER:1;
      tBool SPM_e32_FACTORY:1;
      tBool SPM_e32_QCRITICAL:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
      tBool dummy16:1;
      tBool dummy17:1;
      tBool dummy18:1;
      tBool dummy19:1;
      tBool dummy20:1;
      tBool dummy21:1;
      tBool dummy22:1;
      tBool dummy23:1;
      tBool dummy24:1;
      tBool dummy25:1;
      tBool dummy26:1;
      tBool dummy27:1;
      tBool dummy28:1;
      tBool dummy29:1;
      tBool dummy30:1;
      tBool dummy31:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~spm_fi_tcl_SPM_b32_SYSPWRSAVEMODE();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_b32_SYSPWRSAVEMODE& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_PartnumberItem : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_PartnumberItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_PartnumberItem& operator=(const spm_fi_tcl_SPM_PartnumberItem& coRef);
   spm_fi_tcl_SPM_PartnumberItem(const spm_fi_tcl_SPM_PartnumberItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclString Part;
   tU32 Number;
   virtual ~spm_fi_tcl_SPM_PartnumberItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_PartnumberItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_PartnumberList : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_PartnumberList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_PartnumberList& operator=(const spm_fi_tcl_SPM_PartnumberList& coRef);
   spm_fi_tcl_SPM_PartnumberList(const spm_fi_tcl_SPM_PartnumberList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<spm_fi_tcl_SPM_PartnumberItem, std::allocator<spm_fi_tcl_SPM_PartnumberItem> > Items;
   virtual ~spm_fi_tcl_SPM_PartnumberList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_PartnumberList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_AlphacodeItem : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_AlphacodeItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_AlphacodeItem& operator=(const spm_fi_tcl_SPM_AlphacodeItem& coRef);
   spm_fi_tcl_SPM_AlphacodeItem(const spm_fi_tcl_SPM_AlphacodeItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclString Part;
   tU8 Alphacode1;
   tU8 Alphacode2;
   virtual ~spm_fi_tcl_SPM_AlphacodeItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_AlphacodeItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_AlphacodeList : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_AlphacodeList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_AlphacodeList& operator=(const spm_fi_tcl_SPM_AlphacodeList& coRef);
   spm_fi_tcl_SPM_AlphacodeList(const spm_fi_tcl_SPM_AlphacodeList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<spm_fi_tcl_SPM_AlphacodeItem, std::allocator<spm_fi_tcl_SPM_AlphacodeItem> > Items;
   virtual ~spm_fi_tcl_SPM_AlphacodeList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_AlphacodeList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_REDUCTION_LEVEL : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_REDUCTION_LEVEL();
   enum tenType {
      FI_EN_SPM_U32_REDUCTION_LEVEL_INACTIVE = 0UL,
      FI_EN_SPM_U32_REDUCTION_LEVEL_ACTIVE_1 = 1UL,
      FI_EN_SPM_U32_REDUCTION_LEVEL_ACTIVE_2 = 2UL,
      FI_EN_SPM_U32_REDUCTION_LEVEL_ACTIVE_3 = 3UL,
      FI_EN_SPM_U32_REDUCTION_LEVEL_ACTIVE_4 = 4UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_REDUCTION_LEVEL(spm_fi_tcl_SPM_e32_REDUCTION_LEVEL::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_REDUCTION_LEVEL();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_REDUCTION_LEVEL& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_e32_CLEAR_LIST : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_e32_CLEAR_LIST();
   enum tenType {
      FI_EN_SPM_U32_CLEAR_LIST_ALL = 0UL,
      FI_EN_SPM_U32_CLEAR_LIST_ERROR_HISTORY = 1UL,
      FI_EN_SPM_U32_CLEAR_LIST_WAKEUP_HISTORY = 2UL,
      FI_EN_SPM_U32_CLEAR_LIST_RESET_HISTORY = 3UL,
      FI_EN_SPM_U32_CLEAR_LIST_STATE_HISTORY = 4UL,
      FI_EN_SPM_U32_CLEAR_LIST_TRANSPORT_HISTORY = 5UL,
      FI_EN_SPM_U32_CLEAR_LIST_SWITCH_ON_HISTORY = 6UL
   };
   tenType enType;

   spm_fi_tcl_SPM_e32_CLEAR_LIST(spm_fi_tcl_SPM_e32_CLEAR_LIST::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_SPM_e32_CLEAR_LIST();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_SPM_e32_CLEAR_LIST& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_AppIdList : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_AppIdList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_AppIdList& operator=(const spm_fi_tcl_SPM_AppIdList& coRef);
   spm_fi_tcl_SPM_AppIdList(const spm_fi_tcl_SPM_AppIdList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<tU16, std::allocator<tU16> > tAppIdList;
   virtual ~spm_fi_tcl_SPM_AppIdList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_AppIdList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_RestrictedDlLock : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_RestrictedDlLock();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_RestrictedDlLock& operator=(const spm_fi_tcl_SPM_RestrictedDlLock& coRef);
   spm_fi_tcl_SPM_RestrictedDlLock(const spm_fi_tcl_SPM_RestrictedDlLock& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclString strRestrictedLockArea;
   tU16 u16AppIdLockOwner;
   tU16 u16AppIdLockRequester;
   virtual ~spm_fi_tcl_SPM_RestrictedDlLock();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_RestrictedDlLock& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_UpdateLockCommand : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_UpdateLockCommand();
   enum tenType {
      FI_EN_SPM_E8_USE = 0UL,
      FI_EN_SPM_E8_RESERVE = 1UL,
      FI_EN_SPM_E8_LOCK = 2UL,
      FI_EN_SPM_E8_MASTERLOCK = 3UL,
      FI_EN_SPM_E8_RELEASE = 4UL
   };
   tenType enType;

   spm_fi_tcl_e8_UpdateLockCommand(spm_fi_tcl_e8_UpdateLockCommand::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_UpdateLockCommand();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_UpdateLockCommand& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_UpdateLockState : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_UpdateLockState();
   enum tenType {
      FI_EN_SPM_E8_USED = 1UL,
      FI_EN_SPM_E8_RESERVED = 2UL,
      FI_EN_SPM_E8_LOCKING = 3UL,
      FI_EN_SPM_E8_LOCKED = 4UL
   };
   tenType enType;

   spm_fi_tcl_e8_UpdateLockState(spm_fi_tcl_e8_UpdateLockState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_UpdateLockState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_UpdateLockState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_UpdateLockState : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_UpdateLockState();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_UpdateLockState& operator=(const spm_fi_tcl_SPM_UpdateLockState& coRef);
   spm_fi_tcl_SPM_UpdateLockState(const spm_fi_tcl_SPM_UpdateLockState& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclString strLockName;
   spm_fi_tcl_e8_UpdateLockState eLockState;
   tU16 u16LockOwner;
   std::vector<tU16, std::allocator<tU16> > lu16LockUsers;
   virtual ~spm_fi_tcl_SPM_UpdateLockState();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_UpdateLockState& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_VoltageEventCounter : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_VoltageEventCounter();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_VoltageEventCounter& operator=(const spm_fi_tcl_SPM_VoltageEventCounter& coRef);
   spm_fi_tcl_SPM_VoltageEventCounter(const spm_fi_tcl_SPM_VoltageEventCounter& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 LowVoltageCount;
   tU32 CrLowVoltageCount;
   tU32 HighVoltageCount;
   tU32 CrHighVoltageCount;
   virtual ~spm_fi_tcl_SPM_VoltageEventCounter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_VoltageEventCounter& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_VoltageRange : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_VoltageRange();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_VoltageRange& operator=(const spm_fi_tcl_SPM_VoltageRange& coRef);
   spm_fi_tcl_SPM_VoltageRange(const spm_fi_tcl_SPM_VoltageRange& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 VoltageValue;
   tU32 OperationTime;
   virtual ~spm_fi_tcl_SPM_VoltageRange();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_VoltageRange& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_SystemStatisticsData : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_SystemStatisticsData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_SystemStatisticsData& operator=(const spm_fi_tcl_SPM_SystemStatisticsData& coRef);
   spm_fi_tcl_SPM_SystemStatisticsData(const spm_fi_tcl_SPM_SystemStatisticsData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 MaxValue;
   tU32 AvgValue;
   tU32 MinValue;
   tBool StatisticDataDisabled;
   virtual ~spm_fi_tcl_SPM_SystemStatisticsData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_SystemStatisticsData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_ResetInfoStatisticData : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_ResetInfoStatisticData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_ResetInfoStatisticData& operator=(const spm_fi_tcl_SPM_ResetInfoStatisticData& coRef);
   spm_fi_tcl_SPM_ResetInfoStatisticData(const spm_fi_tcl_SPM_ResetInfoStatisticData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclTimeDate tLastResetDate;
   spm_fi_tcl_e8_LcmResetReason tLastResetReason;
   tU32 u32ApApplicationError;
   tU32 u32ApSoftware;
   tU32 u32ApEcuReset;
   tU32 u32ApDownload;
   tU32 u32ApDownloadError;
   tU32 u32ApOvertemperature;
   tU32 u32ApUser;
   tU32 u32ApLcmIntern;
   tU32 u32ApWarmstartCount;
   tU32 u32ApPowerOnCount;
   tU32 u32ApPowerLossCount;
   tU32 u32ApPowerOffCount;
   tU32 u32ApSccCount;
   tU32 u32ApUnknownCount;
   tU32 u32SccHwWdtCount;
   tU32 u32SccPowerOnCount;
   tU32 u32SccColdstartCount;
   tU32 u32SccAppmodeChangeCount;
   tU32 u32SccDuringLpwCount;
   tU32 u32SccDuringPllOszCount;
   tU32 u32SccDuringSWCount;
   tU32 u32SccWarmstartCount;
   virtual ~spm_fi_tcl_SPM_ResetInfoStatisticData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_ResetInfoStatisticData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_SPM_SWVersionStatisticData : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_SPM_SWVersionStatisticData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   spm_fi_tcl_SPM_SWVersionStatisticData& operator=(const spm_fi_tcl_SPM_SWVersionStatisticData& coRef);
   spm_fi_tcl_SPM_SWVersionStatisticData(const spm_fi_tcl_SPM_SWVersionStatisticData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   spm_fi_tclString strBuildVersion;
   spm_fi_tclString strBuildVersionCustomer;
   spm_fi_tclString strCustomerVersion;
   spm_fi_tcl_SPM_ResetInfoStatisticData tResetInfo;
   std::vector<spm_fi_tcl_ResetElement, std::allocator<spm_fi_tcl_ResetElement> > tResetHistoryList;
   std::vector<spm_fi_tcl_WakeupElement, std::allocator<spm_fi_tcl_WakeupElement> > tWakeupHistoryList;
   std::vector<spm_fi_tcl_SwitchOnElement, std::allocator<spm_fi_tcl_SwitchOnElement> > tSwitchOnList;
   std::vector<spm_fi_tcl_StateChangeElement, std::allocator<spm_fi_tcl_StateChangeElement> > tStateHistoryList;
   std::vector<spm_fi_tcl_StateChangeElement, std::allocator<spm_fi_tcl_StateChangeElement> > tErrStateHistory01List;
   std::vector<spm_fi_tcl_StateChangeElement, std::allocator<spm_fi_tcl_StateChangeElement> > tErrStateHistory02List;
   std::vector<spm_fi_tcl_StateChangeElement, std::allocator<spm_fi_tcl_StateChangeElement> > tErrStateHistory03List;
   std::vector<spm_fi_tcl_StateChangeElement, std::allocator<spm_fi_tcl_StateChangeElement> > tErrStateHistory04List;
   std::vector<spm_fi_tcl_StateChangeElement, std::allocator<spm_fi_tcl_StateChangeElement> > tErrStateHistory05List;
   tU32 u32SystemOperationTime;
   spm_fi_tcl_SPM_SystemStatisticsData tCpuUsage;
   spm_fi_tcl_SPM_SystemStatisticsData tRamUsage;
   spm_fi_tcl_SPM_SystemStatisticsData tStartupTime;
   virtual ~spm_fi_tcl_SPM_SWVersionStatisticData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const spm_fi_tcl_SPM_SWVersionStatisticData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_DefSetMode : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_DefSetMode();
   enum tenType {
      FI_EN_DEFSET_SRC_NOT_DEF = 255UL,
      FI_EN_DEFSET_SRC_SET_USER = 0UL,
      FI_EN_DEFSET_SRC_SET_TEF = 1UL,
      FI_EN_DEFSET_SRC_UPREG = 2UL,
      FI_EN_DEFSET_SRC_RELUPREG = 3UL,
      FI_EN_DEFSET_SRC_GET = 4UL,
      FI_EN_DEFSET_SRC_SET_OEM = 5UL
   };
   tenType enType;

   spm_fi_tcl_e8_DefSetMode(spm_fi_tcl_e8_DefSetMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_DefSetMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_DefSetMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_DefSetServiceStatus : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_DefSetServiceStatus();
   enum tenType {
      FI_EN_DEFSET_STATE_UNDEFINED = 255UL,
      FI_EN_DEFSET_STATE_CONFIRMED = 0UL,
      FI_EN_DEFSET_STATE_FAILED = 1UL
   };
   tenType enType;

   spm_fi_tcl_e8_DefSetServiceStatus(spm_fi_tcl_e8_DefSetServiceStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_DefSetServiceStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_DefSetServiceStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_DiaglogDefsetType : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_DiaglogDefsetType();
   enum tenType {
      FI_EN_CLEAR_ERRMEM = 1UL,
      FI_EN_ACTIVATE_DEACTIVATE_DTC = 2UL
   };
   tenType enType;

   spm_fi_tcl_e8_DiaglogDefsetType(spm_fi_tcl_e8_DiaglogDefsetType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_DiaglogDefsetType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_DiaglogDefsetType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

class spm_fi_tcl_e8_DefSetGroup : public spm_fi_tclVisitableTypeBase
{
public:

   spm_fi_tcl_e8_DefSetGroup();
   enum tenType {
      FI_EN_DEFSET_GROUP_SPM = 21UL,
      FI_EN_DEFSET_GROUP_INFOTAINMENT_AUDIO = 9UL,
      FI_EN_DEFSET_GROUP_MAINTENANCE = 13UL,
      FI_EN_DEFSET_GROUP_COMPLETE = 255UL
   };
   tenType enType;

   spm_fi_tcl_e8_DefSetGroup(spm_fi_tcl_e8_DefSetGroup::tenType newValue)
      :enType(newValue)
   {}

   virtual ~spm_fi_tcl_e8_DefSetGroup();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const spm_fi_tcl_e8_DefSetGroup& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(spm_fi_tclVisitorBase& rfoVisitor);
};

#endif
