/************************************************************************
 
 *FILE:           alltypes.h
 
 *SW-COMPONENT:   FI, framework for Functional Interface
 
 *DESCRIPTION:    Interface for all types defined in the different FIs
                  
 *AUTHOR:         Jochen Bruns (CM-DI/ENS41)
 
 *COPYRIGHT:      (c) 2001 Blaupunkt GmbH
 
 *HISTORY:
           $ Log $
 * 
 *    Rev 1.0   Oct 7 2001 17:16:00   bnj2hi
 * Initial revision
 *  
 ************************************************************************/

#ifndef SENSOR_ALLFITYPES_HEADER
#define SENSOR_ALLFITYPES_HEADER

#include <vector>

class sensor_fi_tclVisitorBase;

const tU16 SENSOR_FI_C_U16_ERROR_INTERNALFAILURE = 1;
const tU16 SENSOR_FI_C_U16_ERROR_INVALIDBINCONTAINERWRONGVERSION = 2;
const tU16 SENSOR_FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA = 3;

class sensor_fi_tclTypeBase : public fi_tclTypeBase
{
public:
  sensor_fi_tclTypeBase() {}
  virtual ~sensor_fi_tclTypeBase() {}
   
  virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& /*oVisitor*/) {}
};

class sensor_fi_tclVisitableTypeBase :public sensor_fi_tclTypeBase
{
public:
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& /*oVisitor*/) {}
};

class sensor_fi_tclU8 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclU8():u8Value(0) {}
   tU8 u8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclU8& roRef) const
      { return (u8Value == roRef.u8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_U8;}
};

class sensor_fi_tclS8 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclS8():s8Value(0) {}
   tS8 s8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclS8& roRef) const
      { return (s8Value == roRef.s8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_S8;}
};

class sensor_fi_tclU16 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclU16():u16Value(0) {}
   tU16 u16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclU16& roRef) const
      { return (u16Value == roRef.u16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_U16;}
};

class sensor_fi_tclS16 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclS16():s16Value(0) {}
   tS16 s16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclS16& roRef) const
      { return (s16Value == roRef.s16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_S16;}
};

class sensor_fi_tclU24 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclU24():u24Value(0) {}
   tU24 u24Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclU24& roRef) const
      { return (u24Value == roRef.u24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_U24;}
};

class sensor_fi_tclS24 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclS24():s24Value(0) {}
   tS24 s24Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclS24& roRef) const
      { return (s24Value == roRef.s24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_S24;}
};

class sensor_fi_tclU32 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclU32():u32Value(0) {}
   tU32 u32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclU32& roRef) const
      { return (u32Value == roRef.u32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_U32;}
};

class sensor_fi_tclS32 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclS32():s32Value(0) {}
   tS32 s32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclS32& roRef) const
      { return (s32Value == roRef.s32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_S32;}
};

class sensor_fi_tclU64 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclU64():u64Value(0) {}
   tU64 u64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclU64& roRef) const
      { return (u64Value == roRef.u64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_U64;}
};

class sensor_fi_tclS64 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclS64():s64Value(0) {}
   tS64 s64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclS64& roRef) const
      { return (s64Value == roRef.s64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_S64;}
};

class sensor_fi_tclF32 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclF32():f32Value(0) {}
   tFloat f32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclF32& roRef) const
      { return (f32Value == roRef.f32Value);  } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_F32;}
};

class sensor_fi_tclF64 : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclF64():f64Value(0) {}
   tDouble f64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclF64& roRef) const
      { return (f64Value == roRef.f64Value); } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_F64;}
};

class sensor_fi_tclBool : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclBool():bValue(0) {}
   tBool bValue;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const sensor_fi_tclBool& roRef) const
      { return (bValue == roRef.bValue); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_BOOL;}
};

class sensor_fi_tclString : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tclString(const sensor_fi_tclString& coRef);
   sensor_fi_tclString& operator=(const sensor_fi_tclString& coRef);
   virtual ~sensor_fi_tclString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const sensor_fi_tclString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   sensor_fi_tclString& operator=(tCString coszString); // ADDED
   sensor_fi_tclString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8,
      FI_EN_ISO8859_1M
   };
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const sensor_fi_tclString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};

class sensor_fi_tclList : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclList():u32ListSize(0) {}
   tU32 u32ListSize;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const sensor_fi_tclList& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oIn);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_LIST;}
};

class sensor_fi_tclMultiLanguageString : public sensor_fi_tclVisitableTypeBase
{
public:
   sensor_fi_tclMultiLanguageString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tclMultiLanguageString(const sensor_fi_tclMultiLanguageString& coRef);
   sensor_fi_tclMultiLanguageString& operator=(const sensor_fi_tclMultiLanguageString& coRef);
   virtual ~sensor_fi_tclMultiLanguageString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const sensor_fi_tclMultiLanguageString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(sensor_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   sensor_fi_tclMultiLanguageString& operator=(tCString coszString); // ADDED
   sensor_fi_tclMultiLanguageString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8
   };
   virtual tS32 s32GetTypeId() const { return sensor_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const sensor_fi_tclMultiLanguageString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};
class sensor_fi_tcl_BinContainer;
class sensor_fi_tcl_Reserved;
class sensor_fi_tcl_u8_Percent;
class sensor_fi_tcl_s16_PerThousand;
class sensor_fi_tcl_u16_IndexInReferenceList;
class sensor_fi_tcl_u16_ErrorCode;
class sensor_fi_tcl_u16_RPM;
class sensor_fi_tcl_s32_Kilogram;
class sensor_fi_tcl_s32_Pound;
class sensor_fi_tcl_s16_Meter;
class sensor_fi_tcl_s32_Meter;
class sensor_fi_tcl_s32_Yards;
class sensor_fi_tcl_s32_Foot;
class sensor_fi_tcl_s32_Centimeter;
class sensor_fi_tcl_s32_Inch;
class sensor_fi_tcl_s32_Millimeter;
class sensor_fi_tcl_s32_Microliter;
class sensor_fi_tcl_s32_Second;
class sensor_fi_tcl_u16_MilliSecond;
class sensor_fi_tcl_s32_MilliSecond;
class sensor_fi_tcl_POSIXTime;
class sensor_fi_tcl_e8_DST_State;
class sensor_fi_tcl_TimeZoneElement;
class sensor_fi_tcl_s16_KmPerHour;
class sensor_fi_tcl_s16_MilesPerHour;
class sensor_fi_tcl_s16_Knot;
class sensor_fi_tcl_s16_CentimeterSecond;
class sensor_fi_tcl_s16_Kelvin;
class sensor_fi_tcl_s16_Fahrenheit;
class sensor_fi_tcl_s32_Pascal;
class sensor_fi_tcl_s32_Watt;
class sensor_fi_tcl_FIVersion;
class sensor_fi_tcl_s32_Longitude;
class sensor_fi_tcl_s32_Latitude;
class sensor_fi_tcl_PositionWGS84;
class sensor_fi_tcl_RectangleWGS84;
class sensor_fi_tcl_e16_ISOCountryCode;
class sensor_fi_tcl_e16_ISOLanguageCode;
class sensor_fi_tcl_LanguageID;
class sensor_fi_tcl_u8_DirectionAbsolute;
class sensor_fi_tcl_u8_DirectionRelative;
class sensor_fi_tcl_s16_CentiDegree;
class sensor_fi_tcl_s16_Degree;
class sensor_fi_tcl_u16_DirectionAbsolute_CentiDegree;
class sensor_fi_tcl_s16_DirectionRelative_CentiDegree;
class sensor_fi_tcl_e8_FuelType;
class sensor_fi_tcl_DistTimeFuel;
class sensor_fi_tcl_VdsVersionInfo;
class sensor_fi_tcl_e8_OdometerStatus;
class sensor_fi_tcl_e8_OdometerDirection;
class sensor_fi_tcl_OdometerData;
class sensor_fi_tcl_e8_AbsStatus;
class sensor_fi_tcl_e8_AbsDirection;
class sensor_fi_tcl_AbsData;
class sensor_fi_tcl_SensorHwInfo;
class sensor_fi_tcl_3dMountAngles;
class sensor_fi_tcl_e16_GyroStatus;
class sensor_fi_tcl_e16_AccStatus;
class sensor_fi_tcl_3dGyroHwInfo;
class sensor_fi_tcl_3dAccHwInfo;
class sensor_fi_tcl_3dGyroData;
class sensor_fi_tcl_3dAccData;
class sensor_fi_tcl_e8_GnssHw;
class sensor_fi_tcl_b8_GnssSatSys;
class sensor_fi_tcl_b8_GnssSatStatus;
class sensor_fi_tcl_GnssConfigData;
class sensor_fi_tcl_e8_GnssQuality;
class sensor_fi_tcl_e8_GnssMode;
class sensor_fi_tcl_GnssStatus;
class sensor_fi_tcl_CovarianceMatrixUpperTriangle;
class sensor_fi_tcl_TimeDate;
class sensor_fi_tcl_GnssChanData;
class sensor_fi_tcl_GnssPvtData;
class sensor_fi_tcl_GnssData;
class sensor_fi_tcl_e8_timeMode;
class sensor_fi_tcl_e8_DefSetGroups;
class sensor_fi_tcl_e8_DefSetMode;
class sensor_fi_tcl_AllSensorData;
class sensor_fi_tcl_e8_velocity_source;
class sensor_fi_tcl_e8_position_source;
class sensor_fi_tcl_e8_countrycode_source;
class sensor_fi_tcl_e8_timezone_source;
class sensor_fi_tcl_e8_deadreckoning_info_source;
class sensor_fi_tcl_b16_deadreckoning_info_validity;
class sensor_fi_tcl_s32_pv_Second;
class sensor_fi_tcl_e8_pv_DST_State;
class sensor_fi_tcl_pv_TimeZone;
class sensor_fi_tcl_e16_pv_ISOCountryCode;
class sensor_fi_tcl_mount_angles;
class sensor_fi_tcl_e8_mounting_source;
class sensor_fi_tcl_BinContainer : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_BinContainer();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_BinContainer& operator=(const sensor_fi_tcl_BinContainer& coRef);
   sensor_fi_tcl_BinContainer(const sensor_fi_tcl_BinContainer& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<tU8, std::allocator<tU8> > ContainerDataList;
   virtual ~sensor_fi_tcl_BinContainer();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_BinContainer& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_Reserved : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_Reserved();

   virtual ~sensor_fi_tcl_Reserved();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_Reserved& /* roRef */) const
      {  return true; }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u8_Percent : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u8_Percent(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~sensor_fi_tcl_u8_Percent();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u8_Percent& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_PerThousand : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_PerThousand(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_PerThousand();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_PerThousand& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u16_IndexInReferenceList : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u16_IndexInReferenceList(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~sensor_fi_tcl_u16_IndexInReferenceList();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u16_IndexInReferenceList& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u16_ErrorCode : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u16_ErrorCode(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~sensor_fi_tcl_u16_ErrorCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u16_ErrorCode& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u16_RPM : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u16_RPM(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~sensor_fi_tcl_u16_RPM();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u16_RPM& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Kilogram : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Kilogram(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Kilogram();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Kilogram& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Pound : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Pound(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Pound();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Pound& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_Meter : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_Meter(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_Meter& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Meter : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Meter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Meter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Yards : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Yards(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Yards();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Yards& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Foot : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Foot(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Foot();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Foot& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Centimeter : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Centimeter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Centimeter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Centimeter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Inch : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Inch(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Inch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Inch& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Millimeter : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Millimeter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Millimeter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Millimeter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Microliter : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Microliter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Microliter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Microliter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Second : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Second(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Second();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Second& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u16_MilliSecond : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u16_MilliSecond(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~sensor_fi_tcl_u16_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u16_MilliSecond& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_MilliSecond : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_MilliSecond(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_MilliSecond& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_POSIXTime : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_POSIXTime();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_POSIXTime& operator=(const sensor_fi_tcl_POSIXTime& coRef);
   sensor_fi_tcl_POSIXTime(const sensor_fi_tcl_POSIXTime& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS32 Time;
   virtual ~sensor_fi_tcl_POSIXTime();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_POSIXTime& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_DST_State : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_DST_State();

   enum tenType {
      FI_EN_T_STANDARDTIME = 0UL,
      FI_EN_T_DAYLIGHTSAVINGTIME = 1UL,
      FI_EN_T_EVALUATIONNECESSARY = 2UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_DST_State();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_DST_State& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_TimeZoneElement : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_TimeZoneElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_TimeZoneElement& operator=(const sensor_fi_tcl_TimeZoneElement& coRef);
   sensor_fi_tcl_TimeZoneElement(const sensor_fi_tcl_TimeZoneElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tclMultiLanguageString Abbreviation_StandardTime;
   sensor_fi_tclMultiLanguageString Abbreviation_DST_in_effect;
   sensor_fi_tcl_s32_Second TimeZoneDifference;
   sensor_fi_tcl_e8_DST_State DST_State;
   sensor_fi_tcl_s32_Second DSTDifference;
   std::vector<tU8, std::allocator<tU8> > DST_TimeDomain;
   virtual ~sensor_fi_tcl_TimeZoneElement();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_TimeZoneElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_KmPerHour : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_KmPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_KmPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_KmPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_MilesPerHour : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_MilesPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_MilesPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_MilesPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_Knot : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_Knot(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_Knot();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_Knot& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_CentimeterSecond : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_CentimeterSecond(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_CentimeterSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_CentimeterSecond& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_Kelvin : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_Kelvin(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_Kelvin();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_Kelvin& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_Fahrenheit : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_Fahrenheit(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_Fahrenheit();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_Fahrenheit& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Pascal : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Pascal(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Pascal();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Pascal& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Watt : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Watt(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Watt();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Watt& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_FIVersion : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_FIVersion();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_FIVersion& operator=(const sensor_fi_tcl_FIVersion& coRef);
   sensor_fi_tcl_FIVersion(const sensor_fi_tcl_FIVersion& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 MajorVersion;
   tU8 MinorVersion;
   tU8 PatchVersion;
   virtual ~sensor_fi_tcl_FIVersion();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_FIVersion& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Longitude : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Longitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Longitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Longitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_Latitude : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_Latitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_Latitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_Latitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_PositionWGS84 : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_PositionWGS84();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_PositionWGS84& operator=(const sensor_fi_tcl_PositionWGS84& coRef);
   sensor_fi_tcl_PositionWGS84(const sensor_fi_tcl_PositionWGS84& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_s32_Longitude Longitude;
   sensor_fi_tcl_s32_Latitude Latitude;
   virtual ~sensor_fi_tcl_PositionWGS84();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_PositionWGS84& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_RectangleWGS84 : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_RectangleWGS84();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_RectangleWGS84& operator=(const sensor_fi_tcl_RectangleWGS84& coRef);
   sensor_fi_tcl_RectangleWGS84(const sensor_fi_tcl_RectangleWGS84& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_s32_Longitude LongitudeMin;
   sensor_fi_tcl_s32_Latitude LatitudeMin;
   sensor_fi_tcl_s32_Longitude LongitudeMax;
   sensor_fi_tcl_s32_Latitude LatitudeMax;
   virtual ~sensor_fi_tcl_RectangleWGS84();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_RectangleWGS84& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e16_ISOCountryCode : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e16_ISOCountryCode();

   enum tenType {
      FI_EN_ISO_ALPHA_3_UNDEFINED = 0UL,
      FI_EN_ISO_ALPHA_3_AFG = 1223UL,
      FI_EN_ISO_ALPHA_3_ALB = 1410UL,
      FI_EN_ISO_ALPHA_3_DZA = 4929UL,
      FI_EN_ISO_ALPHA_3_ASM = 1645UL,
      FI_EN_ISO_ALPHA_3_AND = 1476UL,
      FI_EN_ISO_ALPHA_3_AGO = 1263UL,
      FI_EN_ISO_ALPHA_3_AIA = 1313UL,
      FI_EN_ISO_ALPHA_3_ATG = 1671UL,
      FI_EN_ISO_ALPHA_3_ARG = 1607UL,
      FI_EN_ISO_ALPHA_3_ARM = 1613UL,
      FI_EN_ISO_ALPHA_3_ABW = 1111UL,
      FI_EN_ISO_ALPHA_3_AUS = 1715UL,
      FI_EN_ISO_ALPHA_3_AUT = 1716UL,
      FI_EN_ISO_ALPHA_3_AZE = 1861UL,
      FI_EN_ISO_ALPHA_3_BHS = 2323UL,
      FI_EN_ISO_ALPHA_3_BHR = 2322UL,
      FI_EN_ISO_ALPHA_3_BGD = 2276UL,
      FI_EN_ISO_ALPHA_3_BRB = 2626UL,
      FI_EN_ISO_ALPHA_3_BLR = 2450UL,
      FI_EN_ISO_ALPHA_3_BEL = 2220UL,
      FI_EN_ISO_ALPHA_3_BLZ = 2458UL,
      FI_EN_ISO_ALPHA_3_BEN = 2222UL,
      FI_EN_ISO_ALPHA_3_BMU = 2485UL,
      FI_EN_ISO_ALPHA_3_BTN = 2702UL,
      FI_EN_ISO_ALPHA_3_BOL = 2540UL,
      FI_EN_ISO_ALPHA_3_BIH = 2344UL,
      FI_EN_ISO_ALPHA_3_BWA = 2785UL,
      FI_EN_ISO_ALPHA_3_BRA = 2625UL,
      FI_EN_ISO_ALPHA_3_BRN = 2638UL,
      FI_EN_ISO_ALPHA_3_BGR = 2290UL,
      FI_EN_ISO_ALPHA_3_BFA = 2241UL,
      FI_EN_ISO_ALPHA_3_BDI = 2185UL,
      FI_EN_ISO_ALPHA_3_KHM = 11533UL,
      FI_EN_ISO_ALPHA_3_CMR = 3506UL,
      FI_EN_ISO_ALPHA_3_CAN = 3118UL,
      FI_EN_ISO_ALPHA_3_CPV = 3606UL,
      FI_EN_ISO_ALPHA_3_CYM = 3885UL,
      FI_EN_ISO_ALPHA_3_CAF = 3110UL,
      FI_EN_ISO_ALPHA_3_TCD = 20580UL,
      FI_EN_ISO_ALPHA_3_CHL = 3340UL,
      FI_EN_ISO_ALPHA_3_CHN = 3342UL,
      FI_EN_ISO_ALPHA_3_COL = 3564UL,
      FI_EN_ISO_ALPHA_3_COM = 3565UL,
      FI_EN_ISO_ALPHA_3_COG = 3559UL,
      FI_EN_ISO_ALPHA_3_COD = 3556UL,
      FI_EN_ISO_ALPHA_3_COK = 3563UL,
      FI_EN_ISO_ALPHA_3_CRI = 3657UL,
      FI_EN_ISO_ALPHA_3_CIV = 3382UL,
      FI_EN_ISO_ALPHA_3_HRV = 8790UL,
      FI_EN_ISO_ALPHA_3_CUB = 3746UL,
      FI_EN_ISO_ALPHA_3_CYP = 3888UL,
      FI_EN_ISO_ALPHA_3_CZE = 3909UL,
      FI_EN_ISO_ALPHA_3_DNK = 4555UL,
      FI_EN_ISO_ALPHA_3_DJI = 4425UL,
      FI_EN_ISO_ALPHA_3_DMA = 4513UL,
      FI_EN_ISO_ALPHA_3_DOM = 4589UL,
      FI_EN_ISO_ALPHA_3_ECU = 5237UL,
      FI_EN_ISO_ALPHA_3_EGY = 5369UL,
      FI_EN_ISO_ALPHA_3_SLV = 19862UL,
      FI_EN_ISO_ALPHA_3_GNQ = 7633UL,
      FI_EN_ISO_ALPHA_3_ERI = 5705UL,
      FI_EN_ISO_ALPHA_3_EST = 5748UL,
      FI_EN_ISO_ALPHA_3_ETH = 5768UL,
      FI_EN_ISO_ALPHA_3_FLK = 6539UL,
      FI_EN_ISO_ALPHA_3_FRO = 6735UL,
      FI_EN_ISO_ALPHA_3_FJI = 6473UL,
      FI_EN_ISO_ALPHA_3_FIN = 6446UL,
      FI_EN_ISO_ALPHA_3_FRA = 6721UL,
      FI_EN_ISO_ALPHA_3_GUF = 7846UL,
      FI_EN_ISO_ALPHA_3_PYF = 17190UL,
      FI_EN_ISO_ALPHA_3_GAB = 7202UL,
      FI_EN_ISO_ALPHA_3_GMB = 7586UL,
      FI_EN_ISO_ALPHA_3_GEO = 7343UL,
      FI_EN_ISO_ALPHA_3_DEU = 4277UL,
      FI_EN_ISO_ALPHA_3_GHA = 7425UL,
      FI_EN_ISO_ALPHA_3_GIB = 7458UL,
      FI_EN_ISO_ALPHA_3_GRC = 7747UL,
      FI_EN_ISO_ALPHA_3_GRL = 7756UL,
      FI_EN_ISO_ALPHA_3_GRD = 7748UL,
      FI_EN_ISO_ALPHA_3_GLP = 7568UL,
      FI_EN_ISO_ALPHA_3_GUM = 7853UL,
      FI_EN_ISO_ALPHA_3_GTM = 7821UL,
      FI_EN_ISO_ALPHA_3_GIN = 7470UL,
      FI_EN_ISO_ALPHA_3_GNB = 7618UL,
      FI_EN_ISO_ALPHA_3_GUY = 7865UL,
      FI_EN_ISO_ALPHA_3_HTI = 8841UL,
      FI_EN_ISO_ALPHA_3_VAT = 22580UL,
      FI_EN_ISO_ALPHA_3_HND = 8644UL,
      FI_EN_ISO_ALPHA_3_HKG = 8551UL,
      FI_EN_ISO_ALPHA_3_HUN = 8878UL,
      FI_EN_ISO_ALPHA_3_ISL = 9836UL,
      FI_EN_ISO_ALPHA_3_IND = 9668UL,
      FI_EN_ISO_ALPHA_3_IDN = 9358UL,
      FI_EN_ISO_ALPHA_3_INT = 9684UL,
      FI_EN_ISO_ALPHA_3_IRN = 9806UL,
      FI_EN_ISO_ALPHA_3_IRQ = 9809UL,
      FI_EN_ISO_ALPHA_3_IRL = 9804UL,
      FI_EN_ISO_ALPHA_3_ISR = 9842UL,
      FI_EN_ISO_ALPHA_3_ITA = 9857UL,
      FI_EN_ISO_ALPHA_3_JAM = 10285UL,
      FI_EN_ISO_ALPHA_3_JPN = 10766UL,
      FI_EN_ISO_ALPHA_3_JOR = 10738UL,
      FI_EN_ISO_ALPHA_3_KAZ = 11322UL,
      FI_EN_ISO_ALPHA_3_KEN = 11438UL,
      FI_EN_ISO_ALPHA_3_KIR = 11570UL,
      FI_EN_ISO_ALPHA_3_PRK = 16971UL,
      FI_EN_ISO_ALPHA_3_KOR = 11762UL,
      FI_EN_ISO_ALPHA_3_KWT = 12020UL,
      FI_EN_ISO_ALPHA_3_KGZ = 11514UL,
      FI_EN_ISO_ALPHA_3_LAO = 12335UL,
      FI_EN_ISO_ALPHA_3_LVA = 12993UL,
      FI_EN_ISO_ALPHA_3_LBN = 12366UL,
      FI_EN_ISO_ALPHA_3_LSO = 12911UL,
      FI_EN_ISO_ALPHA_3_LBR = 12370UL,
      FI_EN_ISO_ALPHA_3_LBY = 12377UL,
      FI_EN_ISO_ALPHA_3_LIE = 12581UL,
      FI_EN_ISO_ALPHA_3_LTU = 12949UL,
      FI_EN_ISO_ALPHA_3_LUX = 12984UL,
      FI_EN_ISO_ALPHA_3_MAC = 13347UL,
      FI_EN_ISO_ALPHA_3_MKD = 13668UL,
      FI_EN_ISO_ALPHA_3_MDG = 13447UL,
      FI_EN_ISO_ALPHA_3_MWI = 14057UL,
      FI_EN_ISO_ALPHA_3_MYS = 14131UL,
      FI_EN_ISO_ALPHA_3_MDV = 13462UL,
      FI_EN_ISO_ALPHA_3_MLI = 13705UL,
      FI_EN_ISO_ALPHA_3_MLT = 13716UL,
      FI_EN_ISO_ALPHA_3_MHL = 13580UL,
      FI_EN_ISO_ALPHA_3_MTQ = 13969UL,
      FI_EN_ISO_ALPHA_3_MRT = 13908UL,
      FI_EN_ISO_ALPHA_3_MUS = 14003UL,
      FI_EN_ISO_ALPHA_3_MYT = 14132UL,
      FI_EN_ISO_ALPHA_3_MEX = 13496UL,
      FI_EN_ISO_ALPHA_3_FSM = 6765UL,
      FI_EN_ISO_ALPHA_3_MDA = 13441UL,
      FI_EN_ISO_ALPHA_3_MCO = 13423UL,
      FI_EN_ISO_ALPHA_3_MNG = 13767UL,
      FI_EN_ISO_ALPHA_3_MNE = 13765UL,
      FI_EN_ISO_ALPHA_3_MSR = 13938UL,
      FI_EN_ISO_ALPHA_3_MAR = 13362UL,
      FI_EN_ISO_ALPHA_3_MOZ = 13818UL,
      FI_EN_ISO_ALPHA_3_MMR = 13746UL,
      FI_EN_ISO_ALPHA_3_NAM = 14381UL,
      FI_EN_ISO_ALPHA_3_NRU = 14933UL,
      FI_EN_ISO_ALPHA_3_NPL = 14860UL,
      FI_EN_ISO_ALPHA_3_NLD = 14724UL,
      FI_EN_ISO_ALPHA_3_ANT = 1492UL,
      FI_EN_ISO_ALPHA_3_NCL = 14444UL,
      FI_EN_ISO_ALPHA_3_NZL = 15180UL,
      FI_EN_ISO_ALPHA_3_NIC = 14627UL,
      FI_EN_ISO_ALPHA_3_NER = 14514UL,
      FI_EN_ISO_ALPHA_3_NGA = 14561UL,
      FI_EN_ISO_ALPHA_3_NIU = 14645UL,
      FI_EN_ISO_ALPHA_3_NFK = 14539UL,
      FI_EN_ISO_ALPHA_3_MNP = 13776UL,
      FI_EN_ISO_ALPHA_3_NOR = 14834UL,
      FI_EN_ISO_ALPHA_3_PSE = 16997UL,
      FI_EN_ISO_ALPHA_3_OMN = 15790UL,
      FI_EN_ISO_ALPHA_3_PAK = 16427UL,
      FI_EN_ISO_ALPHA_3_PLW = 16791UL,
      FI_EN_ISO_ALPHA_3_PAN = 16430UL,
      FI_EN_ISO_ALPHA_3_PNG = 16839UL,
      FI_EN_ISO_ALPHA_3_PRY = 16985UL,
      FI_EN_ISO_ALPHA_3_PER = 16562UL,
      FI_EN_ISO_ALPHA_3_PHL = 16652UL,
      FI_EN_ISO_ALPHA_3_PCN = 16494UL,
      FI_EN_ISO_ALPHA_3_POL = 16876UL,
      FI_EN_ISO_ALPHA_3_PRT = 16980UL,
      FI_EN_ISO_ALPHA_3_PRI = 16969UL,
      FI_EN_ISO_ALPHA_3_QAT = 17460UL,
      FI_EN_ISO_ALPHA_3_REU = 18613UL,
      FI_EN_ISO_ALPHA_3_ROU = 18933UL,
      FI_EN_ISO_ALPHA_3_RUS = 19123UL,
      FI_EN_ISO_ALPHA_3_RWA = 19169UL,
      FI_EN_ISO_ALPHA_3_KNA = 11713UL,
      FI_EN_ISO_ALPHA_3_LCA = 12385UL,
      FI_EN_ISO_ALPHA_3_VCT = 22644UL,
      FI_EN_ISO_ALPHA_3_WSM = 24173UL,
      FI_EN_ISO_ALPHA_3_SMR = 19890UL,
      FI_EN_ISO_ALPHA_3_STP = 20112UL,
      FI_EN_ISO_ALPHA_3_SAU = 19509UL,
      FI_EN_ISO_ALPHA_3_SEN = 19630UL,
      FI_EN_ISO_ALPHA_3_SRB = 20034UL,
      FI_EN_ISO_ALPHA_3_SYC = 20259UL,
      FI_EN_ISO_ALPHA_3_SLE = 19845UL,
      FI_EN_ISO_ALPHA_3_SGP = 19696UL,
      FI_EN_ISO_ALPHA_3_SVK = 20171UL,
      FI_EN_ISO_ALPHA_3_SVN = 20174UL,
      FI_EN_ISO_ALPHA_3_SLB = 19842UL,
      FI_EN_ISO_ALPHA_3_SOM = 19949UL,
      FI_EN_ISO_ALPHA_3_ZAF = 26662UL,
      FI_EN_ISO_ALPHA_3_ESP = 5744UL,
      FI_EN_ISO_ALPHA_3_LKA = 12641UL,
      FI_EN_ISO_ALPHA_3_SHN = 19726UL,
      FI_EN_ISO_ALPHA_3_SPM = 19981UL,
      FI_EN_ISO_ALPHA_3_SDN = 19598UL,
      FI_EN_ISO_ALPHA_3_SUR = 20146UL,
      FI_EN_ISO_ALPHA_3_SJM = 19789UL,
      FI_EN_ISO_ALPHA_3_SWZ = 20218UL,
      FI_EN_ISO_ALPHA_3_SWE = 20197UL,
      FI_EN_ISO_ALPHA_3_CHE = 3333UL,
      FI_EN_ISO_ALPHA_3_SYR = 20274UL,
      FI_EN_ISO_ALPHA_3_TWN = 21230UL,
      FI_EN_ISO_ALPHA_3_TJK = 20811UL,
      FI_EN_ISO_ALPHA_3_TZA = 21313UL,
      FI_EN_ISO_ALPHA_3_THA = 20737UL,
      FI_EN_ISO_ALPHA_3_TLS = 20883UL,
      FI_EN_ISO_ALPHA_3_TGO = 20719UL,
      FI_EN_ISO_ALPHA_3_TKL = 20844UL,
      FI_EN_ISO_ALPHA_3_TON = 20974UL,
      FI_EN_ISO_ALPHA_3_TTO = 21135UL,
      FI_EN_ISO_ALPHA_3_TUN = 21166UL,
      FI_EN_ISO_ALPHA_3_TUR = 21170UL,
      FI_EN_ISO_ALPHA_3_TKM = 20845UL,
      FI_EN_ISO_ALPHA_3_TCA = 20577UL,
      FI_EN_ISO_ALPHA_3_TUV = 21174UL,
      FI_EN_ISO_ALPHA_3_UGA = 21729UL,
      FI_EN_ISO_ALPHA_3_UKR = 21874UL,
      FI_EN_ISO_ALPHA_3_ARE = 1605UL,
      FI_EN_ISO_ALPHA_3_GBR = 7250UL,
      FI_EN_ISO_ALPHA_3_USA = 22113UL,
      FI_EN_ISO_ALPHA_3_URY = 22105UL,
      FI_EN_ISO_ALPHA_3_UZB = 22338UL,
      FI_EN_ISO_ALPHA_3_VUT = 23220UL,
      FI_EN_ISO_ALPHA_3_VEN = 22702UL,
      FI_EN_ISO_ALPHA_3_VNM = 22989UL,
      FI_EN_ISO_ALPHA_3_VGB = 22754UL,
      FI_EN_ISO_ALPHA_3_VIR = 22834UL,
      FI_EN_ISO_ALPHA_3_WLF = 23942UL,
      FI_EN_ISO_ALPHA_3_ESH = 5736UL,
      FI_EN_ISO_ALPHA_3_YEM = 25773UL,
      FI_EN_ISO_ALPHA_3_YUG = 26279UL,
      FI_EN_ISO_ALPHA_3_ZMB = 27042UL,
      FI_EN_ISO_ALPHA_3_ZWE = 27365UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e16_ISOCountryCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e16_ISOCountryCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e16_ISOLanguageCode : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e16_ISOLanguageCode();

   enum tenType {
      FI_EN_ISO_639_1_UNDEFINED = 0UL,
      FI_EN_ISO_639_1_AA = 24929UL,
      FI_EN_ISO_639_1_AB = 24930UL,
      FI_EN_ISO_639_1_AE = 24933UL,
      FI_EN_ISO_639_1_AF = 24934UL,
      FI_EN_ISO_639_1_AK = 24939UL,
      FI_EN_ISO_639_1_AM = 24941UL,
      FI_EN_ISO_639_1_AN = 24942UL,
      FI_EN_ISO_639_1_AR = 24946UL,
      FI_EN_ISO_639_1_AS = 24947UL,
      FI_EN_ISO_639_1_AV = 24950UL,
      FI_EN_ISO_639_1_AY = 24953UL,
      FI_EN_ISO_639_1_AZ = 24954UL,
      FI_EN_ISO_639_1_BA = 25185UL,
      FI_EN_ISO_639_1_BE = 25189UL,
      FI_EN_ISO_639_1_BG = 25191UL,
      FI_EN_ISO_639_1_BH = 25192UL,
      FI_EN_ISO_639_1_BI = 25193UL,
      FI_EN_ISO_639_1_BL = 25196UL,
      FI_EN_ISO_639_1_BM = 25197UL,
      FI_EN_ISO_639_1_BN = 25198UL,
      FI_EN_ISO_639_1_BO = 25199UL,
      FI_EN_ISO_639_1_BR = 25202UL,
      FI_EN_ISO_639_1_BS = 25203UL,
      FI_EN_ISO_639_1_BT = 25204UL,
      FI_EN_ISO_639_1_CA = 25441UL,
      FI_EN_ISO_639_1_CE = 25445UL,
      FI_EN_ISO_639_1_CF = 25446UL,
      FI_EN_ISO_639_1_CH = 25448UL,
      FI_EN_ISO_639_1_CL = 25452UL,
      FI_EN_ISO_639_1_CN = 25454UL,
      FI_EN_ISO_639_1_CO = 25455UL,
      FI_EN_ISO_639_1_CR = 25458UL,
      FI_EN_ISO_639_1_CS = 25459UL,
      FI_EN_ISO_639_1_CT = 25460UL,
      FI_EN_ISO_639_1_CU = 25461UL,
      FI_EN_ISO_639_1_CV = 25462UL,
      FI_EN_ISO_639_1_CY = 25465UL,
      FI_EN_ISO_639_1_DA = 25697UL,
      FI_EN_ISO_639_1_DE = 25701UL,
      FI_EN_ISO_639_1_DV = 25718UL,
      FI_EN_ISO_639_1_DZ = 25722UL,
      FI_EN_ISO_639_1_EE = 25957UL,
      FI_EN_ISO_639_1_EG = 25959UL,
      FI_EN_ISO_639_1_EL = 25964UL,
      FI_EN_ISO_639_1_EN = 25966UL,
      FI_EN_ISO_639_1_EO = 25967UL,
      FI_EN_ISO_639_1_ES = 25971UL,
      FI_EN_ISO_639_1_ET = 25972UL,
      FI_EN_ISO_639_1_EU = 25973UL,
      FI_EN_ISO_639_1_FA = 26209UL,
      FI_EN_ISO_639_1_FF = 26214UL,
      FI_EN_ISO_639_1_FI = 26217UL,
      FI_EN_ISO_639_1_FJ = 26218UL,
      FI_EN_ISO_639_1_FO = 26223UL,
      FI_EN_ISO_639_1_FR = 26226UL,
      FI_EN_ISO_639_1_FY = 26233UL,
      FI_EN_ISO_639_1_GA = 26465UL,
      FI_EN_ISO_639_1_GD = 26468UL,
      FI_EN_ISO_639_1_GL = 26476UL,
      FI_EN_ISO_639_1_GN = 26478UL,
      FI_EN_ISO_639_1_GU = 26485UL,
      FI_EN_ISO_639_1_GV = 26486UL,
      FI_EN_ISO_639_1_HA = 26721UL,
      FI_EN_ISO_639_1_HE = 26725UL,
      FI_EN_ISO_639_1_HI = 26729UL,
      FI_EN_ISO_639_1_HO = 26735UL,
      FI_EN_ISO_639_1_HR = 26738UL,
      FI_EN_ISO_639_1_HU = 26741UL,
      FI_EN_ISO_639_1_HY = 26745UL,
      FI_EN_ISO_639_1_HZ = 26746UL,
      FI_EN_ISO_639_1_IA = 26977UL,
      FI_EN_ISO_639_1_ID = 26980UL,
      FI_EN_ISO_639_1_IE = 26981UL,
      FI_EN_ISO_639_1_IG = 26983UL,
      FI_EN_ISO_639_1_II = 26985UL,
      FI_EN_ISO_639_1_IK = 26987UL,
      FI_EN_ISO_639_1_IO = 26991UL,
      FI_EN_ISO_639_1_IS = 26995UL,
      FI_EN_ISO_639_1_IT = 26996UL,
      FI_EN_ISO_639_1_IU = 26997UL,
      FI_EN_ISO_639_1_JA = 27233UL,
      FI_EN_ISO_639_1_JV = 27254UL,
      FI_EN_ISO_639_1_KA = 27489UL,
      FI_EN_ISO_639_1_KG = 27495UL,
      FI_EN_ISO_639_1_KI = 27497UL,
      FI_EN_ISO_639_1_KJ = 27498UL,
      FI_EN_ISO_639_1_KK = 27499UL,
      FI_EN_ISO_639_1_KL = 27500UL,
      FI_EN_ISO_639_1_KM = 27501UL,
      FI_EN_ISO_639_1_KN = 27502UL,
      FI_EN_ISO_639_1_KO = 27503UL,
      FI_EN_ISO_639_1_KR = 27506UL,
      FI_EN_ISO_639_1_KS = 27507UL,
      FI_EN_ISO_639_1_KU = 27509UL,
      FI_EN_ISO_639_1_KV = 27510UL,
      FI_EN_ISO_639_1_KW = 27511UL,
      FI_EN_ISO_639_1_KY = 27513UL,
      FI_EN_ISO_639_1_LA = 27745UL,
      FI_EN_ISO_639_1_LB = 27746UL,
      FI_EN_ISO_639_1_LG = 27751UL,
      FI_EN_ISO_639_1_LI = 27753UL,
      FI_EN_ISO_639_1_LN = 27758UL,
      FI_EN_ISO_639_1_LO = 27759UL,
      FI_EN_ISO_639_1_LT = 27764UL,
      FI_EN_ISO_639_1_LU = 27765UL,
      FI_EN_ISO_639_1_LV = 27766UL,
      FI_EN_ISO_639_1_MC = 28003UL,
      FI_EN_ISO_639_1_MG = 28007UL,
      FI_EN_ISO_639_1_MH = 28008UL,
      FI_EN_ISO_639_1_MI = 28009UL,
      FI_EN_ISO_639_1_MK = 28011UL,
      FI_EN_ISO_639_1_ML = 28012UL,
      FI_EN_ISO_639_1_MN = 28014UL,
      FI_EN_ISO_639_1_MO = 28015UL,
      FI_EN_ISO_639_1_MR = 28018UL,
      FI_EN_ISO_639_1_MS = 28019UL,
      FI_EN_ISO_639_1_MT = 28020UL,
      FI_EN_ISO_639_1_MY = 28025UL,
      FI_EN_ISO_639_1_NA = 28257UL,
      FI_EN_ISO_639_1_NB = 28258UL,
      FI_EN_ISO_639_1_ND = 28260UL,
      FI_EN_ISO_639_1_NE = 28261UL,
      FI_EN_ISO_639_1_NG = 28263UL,
      FI_EN_ISO_639_1_NL = 28268UL,
      FI_EN_ISO_639_1_NN = 28270UL,
      FI_EN_ISO_639_1_NO = 28271UL,
      FI_EN_ISO_639_1_NR = 28274UL,
      FI_EN_ISO_639_1_NV = 28278UL,
      FI_EN_ISO_639_1_NY = 28281UL,
      FI_EN_ISO_639_1_OC = 28515UL,
      FI_EN_ISO_639_1_OJ = 28522UL,
      FI_EN_ISO_639_1_OM = 28525UL,
      FI_EN_ISO_639_1_OR = 28530UL,
      FI_EN_ISO_639_1_OS = 28531UL,
      FI_EN_ISO_639_1_PA = 28769UL,
      FI_EN_ISO_639_1_PI = 28777UL,
      FI_EN_ISO_639_1_PL = 28780UL,
      FI_EN_ISO_639_1_PS = 28787UL,
      FI_EN_ISO_639_1_PT = 28788UL,
      FI_EN_ISO_639_1_QU = 29045UL,
      FI_EN_ISO_639_1_RM = 29293UL,
      FI_EN_ISO_639_1_RN = 29294UL,
      FI_EN_ISO_639_1_RO = 29295UL,
      FI_EN_ISO_639_1_RT = 29300UL,
      FI_EN_ISO_639_1_RU = 29301UL,
      FI_EN_ISO_639_1_RW = 29303UL,
      FI_EN_ISO_639_1_SA = 29537UL,
      FI_EN_ISO_639_1_SB = 29538UL,
      FI_EN_ISO_639_1_SC = 29539UL,
      FI_EN_ISO_639_1_SD = 29540UL,
      FI_EN_ISO_639_1_SE = 29541UL,
      FI_EN_ISO_639_1_SG = 29543UL,
      FI_EN_ISO_639_1_SH = 29544UL,
      FI_EN_ISO_639_1_SI = 29545UL,
      FI_EN_ISO_639_1_SK = 29547UL,
      FI_EN_ISO_639_1_SL = 29548UL,
      FI_EN_ISO_639_1_SM = 29549UL,
      FI_EN_ISO_639_1_SN = 29550UL,
      FI_EN_ISO_639_1_SO = 29551UL,
      FI_EN_ISO_639_1_SQ = 29553UL,
      FI_EN_ISO_639_1_SR = 29554UL,
      FI_EN_ISO_639_1_SS = 29555UL,
      FI_EN_ISO_639_1_ST = 29556UL,
      FI_EN_ISO_639_1_SU = 29557UL,
      FI_EN_ISO_639_1_SV = 29558UL,
      FI_EN_ISO_639_1_SW = 29559UL,
      FI_EN_ISO_639_1_SX = 29560UL,
      FI_EN_ISO_639_1_TA = 29793UL,
      FI_EN_ISO_639_1_TE = 29797UL,
      FI_EN_ISO_639_1_TG = 29799UL,
      FI_EN_ISO_639_1_TH = 29800UL,
      FI_EN_ISO_639_1_TI = 29801UL,
      FI_EN_ISO_639_1_TK = 29803UL,
      FI_EN_ISO_639_1_TL = 29804UL,
      FI_EN_ISO_639_1_TN = 29806UL,
      FI_EN_ISO_639_1_TO = 29807UL,
      FI_EN_ISO_639_1_TR = 29810UL,
      FI_EN_ISO_639_1_TS = 29811UL,
      FI_EN_ISO_639_1_TT = 29812UL,
      FI_EN_ISO_639_1_TW = 29815UL,
      FI_EN_ISO_639_1_TY = 29817UL,
      FI_EN_ISO_639_1_UE = 30053UL,
      FI_EN_ISO_639_1_UG = 30055UL,
      FI_EN_ISO_639_1_UK = 30059UL,
      FI_EN_ISO_639_1_UL = 30060UL,
      FI_EN_ISO_639_1_UR = 30066UL,
      FI_EN_ISO_639_1_UZ = 30074UL,
      FI_EN_ISO_639_1_VA = 30305UL,
      FI_EN_ISO_639_1_VE = 30309UL,
      FI_EN_ISO_639_1_VI = 30313UL,
      FI_EN_ISO_639_1_VO = 30319UL,
      FI_EN_ISO_639_1_WA = 30561UL,
      FI_EN_ISO_639_1_WO = 30575UL,
      FI_EN_ISO_639_1_XH = 30824UL,
      FI_EN_ISO_639_1_YI = 31081UL,
      FI_EN_ISO_639_1_YO = 31087UL,
      FI_EN_ISO_639_1_ZA = 31329UL,
      FI_EN_ISO_639_1_ZH = 31336UL,
      FI_EN_ISO_639_1_ZL = 31340UL,
      FI_EN_ISO_639_1_ZU = 31349UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e16_ISOLanguageCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e16_ISOLanguageCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_LanguageID : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_LanguageID();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_LanguageID& operator=(const sensor_fi_tcl_LanguageID& coRef);
   sensor_fi_tcl_LanguageID(const sensor_fi_tcl_LanguageID& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_e16_ISOLanguageCode ISO639_LanguageCode;
   sensor_fi_tcl_e16_ISOCountryCode ISO3166_CountryCode;
   virtual ~sensor_fi_tcl_LanguageID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_LanguageID& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u8_DirectionAbsolute : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u8_DirectionAbsolute(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~sensor_fi_tcl_u8_DirectionAbsolute();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u8_DirectionAbsolute& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u8_DirectionRelative : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u8_DirectionRelative(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~sensor_fi_tcl_u8_DirectionRelative();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u8_DirectionRelative& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_CentiDegree : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_Degree : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_Degree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_Degree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_Degree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_u16_DirectionAbsolute_CentiDegree : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_u16_DirectionAbsolute_CentiDegree(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~sensor_fi_tcl_u16_DirectionAbsolute_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_u16_DirectionAbsolute_CentiDegree& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s16_DirectionRelative_CentiDegree : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s16_DirectionRelative_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~sensor_fi_tcl_s16_DirectionRelative_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s16_DirectionRelative_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_FuelType : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_FuelType();

   enum tenType {
      FI_EN_RESERVED = 0UL,
      FI_EN_PETROL = 1UL,
      FI_EN_DIESEL = 2UL,
      FI_EN_HYDROGEN = 3UL,
      FI_EN_COMPRESSEDNATURALGAS = 4UL,
      FI_EN_LIQUIFIEDPETROLEUMGAS = 5UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_FuelType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_FuelType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_DistTimeFuel : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_DistTimeFuel();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_DistTimeFuel& operator=(const sensor_fi_tcl_DistTimeFuel& coRef);
   sensor_fi_tcl_DistTimeFuel(const sensor_fi_tcl_DistTimeFuel& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_s32_Meter Distance;
   sensor_fi_tcl_s32_Second Time;
   sensor_fi_tcl_s32_Microliter Fuel;
   virtual ~sensor_fi_tcl_DistTimeFuel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_DistTimeFuel& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_VdsVersionInfo : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_VdsVersionInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_VdsVersionInfo& operator=(const sensor_fi_tcl_VdsVersionInfo& coRef);
   sensor_fi_tcl_VdsVersionInfo(const sensor_fi_tcl_VdsVersionInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tclString vdsVersion;
   sensor_fi_tclString boardVersion;
   virtual ~sensor_fi_tcl_VdsVersionInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_VdsVersionInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_OdometerStatus : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_OdometerStatus();

   enum tenType {
      FI_EN_ODOMSTATE_CONNECTED_NORMAL = 0UL,
      FI_EN_ODOMSTATE_CONNECTED_NO_CALIBRATION = 248UL,
      FI_EN_ODOMSTATE_CONNECTED_DATA_INVALID = 249UL,
      FI_EN_ODOMSTATE_CONNECTED_INTERNALERROR = 250UL,
      FI_EN_ODOMSTATE_CONNECTED_NOINFO = 251UL,
      FI_EN_ODOMSTATE_CONNECTED_NOMOVEMENT = 252UL,
      FI_EN_ODOMSTATE_CONNECTED_ERROR = 253UL,
      FI_EN_ODOMSTATE_DISCONNECTED = 254UL,
      FI_EN_ODOMSTATE_UNKNOWN = 255UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_OdometerStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_OdometerStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_OdometerDirection : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_OdometerDirection();

   enum tenType {
      FI_EN_ODOMDIR_UNKNOWN = 0UL,
      FI_EN_ODOMDIR_FORWARD = 1UL,
      FI_EN_ODOMDIR_REVERSE = 2UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_OdometerDirection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_OdometerDirection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_OdometerData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_OdometerData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_OdometerData& operator=(const sensor_fi_tcl_OdometerData& coRef);
   sensor_fi_tcl_OdometerData(const sensor_fi_tcl_OdometerData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   tU16 OdometerCount;
   sensor_fi_tcl_e8_OdometerStatus OdometerStatus;
   sensor_fi_tcl_e8_OdometerDirection OdometerCount_Direction;
   virtual ~sensor_fi_tcl_OdometerData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_OdometerData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_AbsStatus : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_AbsStatus();

   enum tenType {
      FI_EN_ABS_STATUS_NORMAL = 0UL,
      FI_EN_ABS_STATUS_DATA_INVALID = 249UL,
      FI_EN_ABS_STATUS_INTERNAL_ERROR = 250UL,
      FI_EN_ABS_STATUS_NO_INFO = 251UL,
      FI_EN_ABS_STATUS_NO_MOVEMENT = 252UL,
      FI_EN_ABS_STATUS_ERROR = 253UL,
      FI_EN_ABS_STATUS_DISCONNECTED = 254UL,
      FI_EN_ABS_STATUS_UNKNOWN = 255UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_AbsStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_AbsStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_AbsDirection : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_AbsDirection();

   enum tenType {
      FI_EN_ABS_DIRECTION_UNKNOWN = 0UL,
      FI_EN_ABS_DIRECTION_FORWARD = 1UL,
      FI_EN_ABS_DIRECTION_REVERSE = 2UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_AbsDirection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_AbsDirection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_AbsData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_AbsData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_AbsData& operator=(const sensor_fi_tcl_AbsData& coRef);
   sensor_fi_tcl_AbsData(const sensor_fi_tcl_AbsData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 AbsTimeStamp;
   tU16 WheelCounterFrontLeft;
   tU16 WheelCounterFrontRight;
   tU16 WheelCounterRearLeft;
   tU16 WheelCounterRearRight;
   sensor_fi_tcl_e8_AbsStatus StatusFrontLeft;
   sensor_fi_tcl_e8_AbsStatus StatusFrontRight;
   sensor_fi_tcl_e8_AbsStatus StatusRearLeft;
   sensor_fi_tcl_e8_AbsStatus StatusRearRight;
   sensor_fi_tcl_e8_AbsDirection DirectionFrontLeft;
   sensor_fi_tcl_e8_AbsDirection DirectionFrontRight;
   sensor_fi_tcl_e8_AbsDirection DirectionRearLeft;
   sensor_fi_tcl_e8_AbsDirection DirectionRearRight;
   virtual ~sensor_fi_tcl_AbsData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_AbsData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_SensorHwInfo : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_SensorHwInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_SensorHwInfo& operator=(const sensor_fi_tcl_SensorHwInfo& coRef);
   sensor_fi_tcl_SensorHwInfo(const sensor_fi_tcl_SensorHwInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 AdcRangeMin;
   tU32 AdcRangeMax;
   tU32 SampleMin;
   tU32 SampleMax;
   tF32 MinNoiseValue;
   tF32 EstimOffset;
   tF32 MinOffset;
   tF32 MaxOffset;
   tF32 DriftOffset;
   tF32 MaxUnsteadOffset;
   tF32 BestCalibOffset;
   tF32 EstimScaleFactor;
   tF32 MinScaleFactor;
   tF32 MaxScaleFactor;
   tF32 DriftScaleFactor;
   tF32 MaxUnsteadScaleFactor;
   tF32 BestCalibScaleFactor;
   tF32 DriftOffsetTime;
   tF32 DriftScaleFactorTime;
   virtual ~sensor_fi_tcl_SensorHwInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_SensorHwInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_3dMountAngles : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_3dMountAngles();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_3dMountAngles& operator=(const sensor_fi_tcl_3dMountAngles& coRef);
   sensor_fi_tcl_3dMountAngles(const sensor_fi_tcl_3dMountAngles& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 AngRX;
   tU8 AngRY;
   tU8 AngRZ;
   tU8 AngSX;
   tU8 AngSY;
   tU8 AngSZ;
   tU8 AngTX;
   tU8 AngTY;
   tU8 AngTZ;
   virtual ~sensor_fi_tcl_3dMountAngles();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_3dMountAngles& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e16_GyroStatus : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e16_GyroStatus();

   enum tenType {
      FI_EN_GYROSTATE_CONNECTED_NORMAL = 0UL,
      FI_EN_GYROSTATE_OUT_OF_ORDER = 65531UL,
      FI_EN_GYROSTATE_OVERHEAT = 65532UL,
      FI_EN_GYROSTATE_SENSOR_NOT_PRESENT = 65533UL,
      FI_EN_GYROSTATE_CONNECTED_DATA_INVALID = 65534UL,
      FI_EN_GYROSTATE_CONNECTED_INTERNALERROR = 65535UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e16_GyroStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e16_GyroStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e16_AccStatus : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e16_AccStatus();

   enum tenType {
      FI_EN_ACCSTATE_CONNECTED_NORMAL = 0UL,
      FI_EN_ACCSTATE_OUT_OF_ORDER = 65531UL,
      FI_EN_ACCSTATE_OVERHEAT = 65532UL,
      FI_EN_ACCSTATE_SENSOR_NOT_PRESENT = 65533UL,
      FI_EN_ACCSTATE_CONNECTED_DATA_INVALID = 65534UL,
      FI_EN_ACCSTATE_CONNECTED_INTERNALERROR = 65535UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e16_AccStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e16_AccStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_3dGyroHwInfo : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_3dGyroHwInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_3dGyroHwInfo& operator=(const sensor_fi_tcl_3dGyroHwInfo& coRef);
   sensor_fi_tcl_3dGyroHwInfo(const sensor_fi_tcl_3dGyroHwInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 SampleRate;
   sensor_fi_tcl_3dMountAngles MountAngles;
   sensor_fi_tcl_SensorHwInfo RAxes;
   sensor_fi_tcl_SensorHwInfo SAxes;
   sensor_fi_tcl_SensorHwInfo TAxes;
   virtual ~sensor_fi_tcl_3dGyroHwInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_3dGyroHwInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_3dAccHwInfo : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_3dAccHwInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_3dAccHwInfo& operator=(const sensor_fi_tcl_3dAccHwInfo& coRef);
   sensor_fi_tcl_3dAccHwInfo(const sensor_fi_tcl_3dAccHwInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 SampleRate;
   sensor_fi_tcl_3dMountAngles MountAngles;
   sensor_fi_tcl_SensorHwInfo RAxes;
   sensor_fi_tcl_SensorHwInfo SAxes;
   sensor_fi_tcl_SensorHwInfo TAxes;
   virtual ~sensor_fi_tcl_3dAccHwInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_3dAccHwInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_3dGyroData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_3dGyroData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_3dGyroData& operator=(const sensor_fi_tcl_3dGyroData& coRef);
   sensor_fi_tcl_3dGyroData(const sensor_fi_tcl_3dGyroData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 TimeStamp3dGyro;
   tU32 RVal;
   tU32 SVal;
   tU32 TVal;
   sensor_fi_tcl_e16_GyroStatus RStatus;
   sensor_fi_tcl_e16_GyroStatus SStatus;
   sensor_fi_tcl_e16_GyroStatus TStatus;
   virtual ~sensor_fi_tcl_3dGyroData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_3dGyroData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_3dAccData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_3dAccData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_3dAccData& operator=(const sensor_fi_tcl_3dAccData& coRef);
   sensor_fi_tcl_3dAccData(const sensor_fi_tcl_3dAccData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 TimeStamp3dAcc;
   tU32 RVal;
   tU32 SVal;
   tU32 TVal;
   sensor_fi_tcl_e16_AccStatus RStatus;
   sensor_fi_tcl_e16_AccStatus SStatus;
   sensor_fi_tcl_e16_AccStatus TStatus;
   virtual ~sensor_fi_tcl_3dAccData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_3dAccData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_GnssHw : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_GnssHw();

   enum tenType {
      FI_EN_GNSS_HW_GPS1_4 = 0UL,
      FI_EN_GNSS_HW_GPS5 = 1UL,
      FI_EN_GNSS_HW_GPS6_7 = 2UL,
      FI_EN_GNSS_HW_GPS_ELENA = 10UL,
      FI_EN_GNSS_HW_BPGPS_ARION = 11UL,
      FI_EN_GNSS_HW_BPGPS_DRAGON_ES1 = 12UL,
      FI_EN_GNSS_HW_BPGPS_DRAGON = 13UL,
      FI_EN_GNSS_HW_NMEA_GPS = 14UL,
      FI_EN_GNSS_HW_IPHONE_GPS = 15UL,
      FI_EN_GNSS_HW_STA8088 = 16UL,
      FI_EN_GNSS_HW_STA8089 = 17UL,
      FI_EN_GNSS_HW_UNKNOWN = 255UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_GnssHw();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_GnssHw& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_b8_GnssSatSys : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_b8_GnssSatSys();

   enum {
      FI_C_U8_BIT_GNSS_SATSYS_GPS = 0x1UL,
      FI_C_U8_BIT_GNSS_SATSYS_GLONASS = 0x2UL,
      FI_C_U8_BIT_GNSS_SATSYS_GALILEO = 0x4UL,
      FI_C_U8_BIT_GNSS_SATSYS_COMPASS = 0x8UL,
      FI_C_U8_BIT_GNSS_SATSYS_SBAS = 0x10UL,
      FI_C_U8_BIT_GNSS_SATSYS_QZSS = 0x20UL
   };
   tBool bGNSS_SATSYS_GPS() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_GPS) != 0; }
   tVoid vSetGNSS_SATSYS_GPS(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_GPS):(u8Value & ~FI_C_U8_BIT_GNSS_SATSYS_GPS); }
   tBool bGNSS_SATSYS_GLONASS() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_GLONASS) != 0; }
   tVoid vSetGNSS_SATSYS_GLONASS(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_GLONASS):(u8Value & ~FI_C_U8_BIT_GNSS_SATSYS_GLONASS); }
   tBool bGNSS_SATSYS_GALILEO() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_GALILEO) != 0; }
   tVoid vSetGNSS_SATSYS_GALILEO(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_GALILEO):(u8Value & ~FI_C_U8_BIT_GNSS_SATSYS_GALILEO); }
   tBool bGNSS_SATSYS_COMPASS() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_COMPASS) != 0; }
   tVoid vSetGNSS_SATSYS_COMPASS(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_COMPASS):(u8Value & ~FI_C_U8_BIT_GNSS_SATSYS_COMPASS); }
   tBool bGNSS_SATSYS_SBAS() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_SBAS) != 0; }
   tVoid vSetGNSS_SATSYS_SBAS(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_SBAS):(u8Value & ~FI_C_U8_BIT_GNSS_SATSYS_SBAS); }
   tBool bGNSS_SATSYS_QZSS() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_QZSS) != 0; }
   tVoid vSetGNSS_SATSYS_QZSS(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_QZSS):(u8Value & ~FI_C_U8_BIT_GNSS_SATSYS_QZSS); }
   struct rBitValues {
      tBool GNSS_SATSYS_GPS:1;
      tBool GNSS_SATSYS_GLONASS:1;
      tBool GNSS_SATSYS_GALILEO:1;
      tBool GNSS_SATSYS_COMPASS:1;
      tBool GNSS_SATSYS_SBAS:1;
      tBool GNSS_SATSYS_QZSS:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~sensor_fi_tcl_b8_GnssSatSys();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_b8_GnssSatSys& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_b8_GnssSatStatus : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_b8_GnssSatStatus();

   enum {
      FI_C_U8_BIT_SAT_HEALTHY = 0x1UL,
      FI_C_U8_BIT_SAT_ALMANAC_AVAILABLE = 0x2UL,
      FI_C_U8_BIT_SAT_EPHEMERIS_AVAILABLE = 0x4UL,
      FI_C_U8_BIT_SAT_DGPS_AVAILABLE = 0x8UL,
      FI_C_U8_BIT_SAT_TRACKED = 0x10UL,
      FI_C_U8_BIT_SAT_USED_FOR_POSCALC = 0x20UL
   };
   tBool bSAT_HEALTHY() const {return (u8Value & FI_C_U8_BIT_SAT_HEALTHY) != 0; }
   tVoid vSetSAT_HEALTHY(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_SAT_HEALTHY):(u8Value & ~FI_C_U8_BIT_SAT_HEALTHY); }
   tBool bSAT_ALMANAC_AVAILABLE() const {return (u8Value & FI_C_U8_BIT_SAT_ALMANAC_AVAILABLE) != 0; }
   tVoid vSetSAT_ALMANAC_AVAILABLE(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_SAT_ALMANAC_AVAILABLE):(u8Value & ~FI_C_U8_BIT_SAT_ALMANAC_AVAILABLE); }
   tBool bSAT_EPHEMERIS_AVAILABLE() const {return (u8Value & FI_C_U8_BIT_SAT_EPHEMERIS_AVAILABLE) != 0; }
   tVoid vSetSAT_EPHEMERIS_AVAILABLE(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_SAT_EPHEMERIS_AVAILABLE):(u8Value & ~FI_C_U8_BIT_SAT_EPHEMERIS_AVAILABLE); }
   tBool bSAT_DGPS_AVAILABLE() const {return (u8Value & FI_C_U8_BIT_SAT_DGPS_AVAILABLE) != 0; }
   tVoid vSetSAT_DGPS_AVAILABLE(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_SAT_DGPS_AVAILABLE):(u8Value & ~FI_C_U8_BIT_SAT_DGPS_AVAILABLE); }
   tBool bSAT_TRACKED() const {return (u8Value & FI_C_U8_BIT_SAT_TRACKED) != 0; }
   tVoid vSetSAT_TRACKED(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_SAT_TRACKED):(u8Value & ~FI_C_U8_BIT_SAT_TRACKED); }
   tBool bSAT_USED_FOR_POSCALC() const {return (u8Value & FI_C_U8_BIT_SAT_USED_FOR_POSCALC) != 0; }
   tVoid vSetSAT_USED_FOR_POSCALC(tBool bSetBit) { u8Value = (bSetBit)?(u8Value | FI_C_U8_BIT_SAT_USED_FOR_POSCALC):(u8Value & ~FI_C_U8_BIT_SAT_USED_FOR_POSCALC); }
   struct rBitValues {
      tBool SAT_HEALTHY:1;
      tBool SAT_ALMANAC_AVAILABLE:1;
      tBool SAT_EPHEMERIS_AVAILABLE:1;
      tBool SAT_DGPS_AVAILABLE:1;
      tBool SAT_TRACKED:1;
      tBool SAT_USED_FOR_POSCALC:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~sensor_fi_tcl_b8_GnssSatStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_b8_GnssSatStatus& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_GnssConfigData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_GnssConfigData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_GnssConfigData& operator=(const sensor_fi_tcl_GnssConfigData& coRef);
   sensor_fi_tcl_GnssConfigData(const sensor_fi_tcl_GnssConfigData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_e8_GnssHw GnssHardwareType;
   tU32 GnssReceiverFirmwareVersion;
   tU32 GnssReceiverFirmwareCrc;
   tU16 NumberOfChannels;
   sensor_fi_tcl_b8_GnssSatStatus UsedSatStatusBits;
   tU8 UpdateFrequency;
   virtual ~sensor_fi_tcl_GnssConfigData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_GnssConfigData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_GnssQuality : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_GnssQuality();

   enum tenType {
      FI_EN_GNSSQUALITY_NOFIX = 0UL,
      FI_EN_GNSSQUALITY_AUTONOMOUS = 1UL,
      FI_EN_GNSSQUALITY_DIFFERENTIAL = 2UL,
      FI_EN_GNSSQUALITY_UNKNOWN = 255UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_GnssQuality();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_GnssQuality& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_GnssMode : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_GnssMode();

   enum tenType {
      FI_EN_GNSSMODE_NOFIX = 1UL,
      FI_EN_GNSSMODE_2DFIX = 2UL,
      FI_EN_GNSSMODE_3DFIX = 3UL,
      FI_EN_GNSSMODE_UNKNOWN = 255UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_GnssMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_GnssMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_GnssStatus : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_GnssStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_GnssStatus& operator=(const sensor_fi_tcl_GnssStatus& coRef);
   sensor_fi_tcl_GnssStatus(const sensor_fi_tcl_GnssStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_e8_GnssQuality GnssQuality;
   sensor_fi_tcl_e8_GnssMode GnssMode;
   virtual ~sensor_fi_tcl_GnssStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_GnssStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_CovarianceMatrixUpperTriangle : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_CovarianceMatrixUpperTriangle();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_CovarianceMatrixUpperTriangle& operator=(const sensor_fi_tcl_CovarianceMatrixUpperTriangle& coRef);
   sensor_fi_tcl_CovarianceMatrixUpperTriangle(const sensor_fi_tcl_CovarianceMatrixUpperTriangle& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF32 Elem0;
   tF32 Elem4;
   tF32 Elem5;
   tF32 Elem8;
   tF32 Elem9;
   tF32 Elem10;
   virtual ~sensor_fi_tcl_CovarianceMatrixUpperTriangle();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_CovarianceMatrixUpperTriangle& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_TimeDate : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_TimeDate();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_TimeDate& operator=(const sensor_fi_tcl_TimeDate& coRef);
   sensor_fi_tcl_TimeDate(const sensor_fi_tcl_TimeDate& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 td_year;
   tU8 td_month;
   tU8 td_day;
   tU8 td_hour;
   tU8 td_minute;
   tU8 td_second;
   tU16 td_milliSeconds;
   virtual ~sensor_fi_tcl_TimeDate();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_TimeDate& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_GnssChanData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_GnssChanData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_GnssChanData& operator=(const sensor_fi_tcl_GnssChanData& coRef);
   sensor_fi_tcl_GnssChanData(const sensor_fi_tcl_GnssChanData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 SvId;
   sensor_fi_tcl_b8_GnssSatStatus SatStatus;
   tF32 Azimuth;
   tF32 Elevation;
   tU8 CarrierToNoiseRatio;
   virtual ~sensor_fi_tcl_GnssChanData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_GnssChanData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_GnssPvtData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_GnssPvtData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_GnssPvtData& operator=(const sensor_fi_tcl_GnssPvtData& coRef);
   sensor_fi_tcl_GnssPvtData(const sensor_fi_tcl_GnssPvtData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_TimeDate UtcTimeDate;
   tF64 Latitude;
   tF64 Longitude;
   tF32 AltitudeWGS84;
   tF32 GeoidalSeparation;
   tF32 VelocityNorth;
   tF32 VelocityEast;
   tF32 VelocityUp;
   sensor_fi_tcl_CovarianceMatrixUpperTriangle PositionCovarianceMatrix;
   sensor_fi_tcl_CovarianceMatrixUpperTriangle VelocityCovarianceMatrix;
   sensor_fi_tcl_GnssStatus GnssStatus;
   tF32 GDOP;
   tF32 PDOP;
   tF32 HDOP;
   tF32 TDOP;
   tF32 VDOP;
   sensor_fi_tcl_b8_GnssSatSys SatSysUsed;
   tU16 SatellitesVisible;
   tU16 SatellitesReceived;
   tU16 SatellitesUsed;
   virtual ~sensor_fi_tcl_GnssPvtData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_GnssPvtData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_GnssData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_GnssData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_GnssData& operator=(const sensor_fi_tcl_GnssData& coRef);
   sensor_fi_tcl_GnssData(const sensor_fi_tcl_GnssData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   sensor_fi_tcl_GnssPvtData GnssPvtData;
   std::vector<sensor_fi_tcl_GnssChanData, std::allocator<sensor_fi_tcl_GnssChanData> > GnssChanDataList;
   virtual ~sensor_fi_tcl_GnssData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_GnssData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_timeMode : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_timeMode();

   enum tenType {
      FI_EN_TIMEMODE_DEFAULT = 0UL,
      FI_EN_TIMEMODE_AUTO = 1UL,
      FI_EN_TIMEMODE_MANUAL = 2UL,
      FI_EN_TIMEMODE_INVALID = 254UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_timeMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_timeMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_DefSetGroups : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_DefSetGroups();

   enum tenType {
      FI_EN_DEFSET_GROUP_MAINTENANCE = 13UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_DefSetGroups();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_DefSetGroups& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_DefSetMode : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_DefSetMode();

   enum tenType {
      FI_EN_DEFSET_SRC_CODING = 5UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_DefSetMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_DefSetMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_AllSensorData : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_AllSensorData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_AllSensorData& operator=(const sensor_fi_tcl_AllSensorData& coRef);
   sensor_fi_tcl_AllSensorData(const sensor_fi_tcl_AllSensorData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<sensor_fi_tcl_GnssData, std::allocator<sensor_fi_tcl_GnssData> > GnssUpdate;
   std::vector<sensor_fi_tcl_OdometerData, std::allocator<sensor_fi_tcl_OdometerData> > OdometerUpdate;
   std::vector<sensor_fi_tcl_AbsData, std::allocator<sensor_fi_tcl_AbsData> > AbsUpdate;
   std::vector<sensor_fi_tcl_3dGyroData, std::allocator<sensor_fi_tcl_3dGyroData> > GyroUpdate;
   tF32 GyroTemperature;
   std::vector<sensor_fi_tcl_3dAccData, std::allocator<sensor_fi_tcl_3dAccData> > AccUpdate;
   tF32 AccTemperature;
   virtual ~sensor_fi_tcl_AllSensorData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_AllSensorData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_velocity_source : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_velocity_source();

   enum tenType {
      FI_EN_SRC_V_INVALID = 0UL,
      FI_EN_SRC_V_ODO = 1UL,
      FI_EN_SRC_V_CANSPEED = 2UL,
      FI_EN_SRC_V_GNSS = 3UL,
      FI_EN_SRC_V_DR = 4UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_velocity_source();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_velocity_source& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_position_source : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_position_source();

   enum tenType {
      FI_EN_SRC_POS_INVALID = 0UL,
      FI_EN_SRC_POS_CAN = 1UL,
      FI_EN_SRC_POS_GNSS = 2UL,
      FI_EN_SRC_POS_DR = 3UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_position_source();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_position_source& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_countrycode_source : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_countrycode_source();

   enum tenType {
      FI_EN_SRC_CC_INVALID = 0UL,
      FI_EN_SRC_CC_MAPDATA = 1UL,
      FI_EN_SRC_CC_EXTERNAL = 2UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_countrycode_source();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_countrycode_source& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_timezone_source : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_timezone_source();

   enum tenType {
      FI_EN_SRC_TZ_INVALID = 0UL,
      FI_EN_SRC_TZ_MAPDATA = 1UL,
      FI_EN_SRC_TZ_EXTERNAL = 2UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_timezone_source();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_timezone_source& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_deadreckoning_info_source : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_deadreckoning_info_source();

   enum tenType {
      FI_EN_SRC_DRPOS_INVALID = 0UL,
      FI_EN_SRC_DRPOS_GNSS = 1UL,
      FI_EN_SRC_DRPOS_DR = 2UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_deadreckoning_info_source();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_deadreckoning_info_source& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_b16_deadreckoning_info_validity : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_b16_deadreckoning_info_validity();

   enum {
      FI_C_U16_BIT_TIMESTAMPISVALID = 0x1UL,
      FI_C_U16_BIT_POSITIONISVALID = 0x2UL,
      FI_C_U16_BIT_HEADINGISVALID = 0x4UL,
      FI_C_U16_BIT_SPEEDISVALID = 0x8UL,
      FI_C_U16_BIT_ALTITUDEISVALID = 0x10UL,
      FI_C_U16_BIT_TURNRATEXISVALID = 0x20UL,
      FI_C_U16_BIT_TURNRATEYISVALID = 0x40UL,
      FI_C_U16_BIT_TURNRATEZISVALID = 0x80UL,
      FI_C_U16_BIT_ACCELERATIONXISVALID = 0x100UL,
      FI_C_U16_BIT_ACCELERATIONYISVALID = 0x200UL,
      FI_C_U16_BIT_ACCELERATIONZISVALID = 0x400UL
   };
   tBool bTimestampIsValid() const {return (u16Value & FI_C_U16_BIT_TIMESTAMPISVALID) != 0; }
   tVoid vSetTimestampIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_TIMESTAMPISVALID):(u16Value & ~FI_C_U16_BIT_TIMESTAMPISVALID); }
   tBool bPositionIsValid() const {return (u16Value & FI_C_U16_BIT_POSITIONISVALID) != 0; }
   tVoid vSetPositionIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_POSITIONISVALID):(u16Value & ~FI_C_U16_BIT_POSITIONISVALID); }
   tBool bHeadingIsValid() const {return (u16Value & FI_C_U16_BIT_HEADINGISVALID) != 0; }
   tVoid vSetHeadingIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_HEADINGISVALID):(u16Value & ~FI_C_U16_BIT_HEADINGISVALID); }
   tBool bSpeedIsValid() const {return (u16Value & FI_C_U16_BIT_SPEEDISVALID) != 0; }
   tVoid vSetSpeedIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_SPEEDISVALID):(u16Value & ~FI_C_U16_BIT_SPEEDISVALID); }
   tBool bAltitudeIsValid() const {return (u16Value & FI_C_U16_BIT_ALTITUDEISVALID) != 0; }
   tVoid vSetAltitudeIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_ALTITUDEISVALID):(u16Value & ~FI_C_U16_BIT_ALTITUDEISVALID); }
   tBool bTurnrateXIsValid() const {return (u16Value & FI_C_U16_BIT_TURNRATEXISVALID) != 0; }
   tVoid vSetTurnrateXIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_TURNRATEXISVALID):(u16Value & ~FI_C_U16_BIT_TURNRATEXISVALID); }
   tBool bTurnrateYIsValid() const {return (u16Value & FI_C_U16_BIT_TURNRATEYISVALID) != 0; }
   tVoid vSetTurnrateYIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_TURNRATEYISVALID):(u16Value & ~FI_C_U16_BIT_TURNRATEYISVALID); }
   tBool bTurnrateZIsValid() const {return (u16Value & FI_C_U16_BIT_TURNRATEZISVALID) != 0; }
   tVoid vSetTurnrateZIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_TURNRATEZISVALID):(u16Value & ~FI_C_U16_BIT_TURNRATEZISVALID); }
   tBool bAccelerationXIsValid() const {return (u16Value & FI_C_U16_BIT_ACCELERATIONXISVALID) != 0; }
   tVoid vSetAccelerationXIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_ACCELERATIONXISVALID):(u16Value & ~FI_C_U16_BIT_ACCELERATIONXISVALID); }
   tBool bAccelerationYIsValid() const {return (u16Value & FI_C_U16_BIT_ACCELERATIONYISVALID) != 0; }
   tVoid vSetAccelerationYIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_ACCELERATIONYISVALID):(u16Value & ~FI_C_U16_BIT_ACCELERATIONYISVALID); }
   tBool bAccelerationZIsValid() const {return (u16Value & FI_C_U16_BIT_ACCELERATIONZISVALID) != 0; }
   tVoid vSetAccelerationZIsValid(tBool bSetBit) { u16Value = (bSetBit)?(u16Value | FI_C_U16_BIT_ACCELERATIONZISVALID):(u16Value & ~FI_C_U16_BIT_ACCELERATIONZISVALID); }
   struct rBitValues {
      tBool TimestampIsValid:1;
      tBool PositionIsValid:1;
      tBool HeadingIsValid:1;
      tBool SpeedIsValid:1;
      tBool AltitudeIsValid:1;
      tBool TurnrateXIsValid:1;
      tBool TurnrateYIsValid:1;
      tBool TurnrateZIsValid:1;
      tBool AccelerationXIsValid:1;
      tBool AccelerationYIsValid:1;
      tBool AccelerationZIsValid:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~sensor_fi_tcl_b16_deadreckoning_info_validity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_b16_deadreckoning_info_validity& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_s32_pv_Second : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_s32_pv_Second(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~sensor_fi_tcl_s32_pv_Second();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_s32_pv_Second& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_pv_DST_State : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_pv_DST_State();

   enum tenType {
      FI_EN_PV_STANDARD_TIME = 0UL,
      FI_EN_PV_DAYLIGHT_SAVING_TIME = 1UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_pv_DST_State();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_pv_DST_State& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_pv_TimeZone : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_pv_TimeZone();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_pv_TimeZone& operator=(const sensor_fi_tcl_pv_TimeZone& coRef);
   sensor_fi_tcl_pv_TimeZone(const sensor_fi_tcl_pv_TimeZone& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   sensor_fi_tcl_s32_pv_Second TimeZoneDifference;
   sensor_fi_tcl_e8_pv_DST_State DST_State;
   sensor_fi_tcl_s32_pv_Second DSTDifference;
   virtual ~sensor_fi_tcl_pv_TimeZone();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_pv_TimeZone& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e16_pv_ISOCountryCode : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e16_pv_ISOCountryCode();

   enum tenType {
      FI_EN_ISO_ALPHA_3_UNDEFINED = 0UL,
      FI_EN_ISO_ALPHA_3_AFG = 1223UL,
      FI_EN_ISO_ALPHA_3_ALB = 1410UL,
      FI_EN_ISO_ALPHA_3_DZA = 4929UL,
      FI_EN_ISO_ALPHA_3_ASM = 1645UL,
      FI_EN_ISO_ALPHA_3_AND = 1476UL,
      FI_EN_ISO_ALPHA_3_AGO = 1263UL,
      FI_EN_ISO_ALPHA_3_AIA = 1313UL,
      FI_EN_ISO_ALPHA_3_ATG = 1671UL,
      FI_EN_ISO_ALPHA_3_ARG = 1607UL,
      FI_EN_ISO_ALPHA_3_ARM = 1613UL,
      FI_EN_ISO_ALPHA_3_ABW = 1111UL,
      FI_EN_ISO_ALPHA_3_AUS = 1715UL,
      FI_EN_ISO_ALPHA_3_AUT = 1716UL,
      FI_EN_ISO_ALPHA_3_AZE = 1861UL,
      FI_EN_ISO_ALPHA_3_BHS = 2323UL,
      FI_EN_ISO_ALPHA_3_BHR = 2322UL,
      FI_EN_ISO_ALPHA_3_BGD = 2276UL,
      FI_EN_ISO_ALPHA_3_BRB = 2626UL,
      FI_EN_ISO_ALPHA_3_BLR = 2450UL,
      FI_EN_ISO_ALPHA_3_BEL = 2220UL,
      FI_EN_ISO_ALPHA_3_BLZ = 2458UL,
      FI_EN_ISO_ALPHA_3_BEN = 2222UL,
      FI_EN_ISO_ALPHA_3_BMU = 2485UL,
      FI_EN_ISO_ALPHA_3_BTN = 2702UL,
      FI_EN_ISO_ALPHA_3_BOL = 2540UL,
      FI_EN_ISO_ALPHA_3_BIH = 2344UL,
      FI_EN_ISO_ALPHA_3_BWA = 2785UL,
      FI_EN_ISO_ALPHA_3_BRA = 2625UL,
      FI_EN_ISO_ALPHA_3_BRN = 2638UL,
      FI_EN_ISO_ALPHA_3_BGR = 2290UL,
      FI_EN_ISO_ALPHA_3_BFA = 2241UL,
      FI_EN_ISO_ALPHA_3_BDI = 2185UL,
      FI_EN_ISO_ALPHA_3_KHM = 11533UL,
      FI_EN_ISO_ALPHA_3_CMR = 3506UL,
      FI_EN_ISO_ALPHA_3_CAN = 3118UL,
      FI_EN_ISO_ALPHA_3_CPV = 3606UL,
      FI_EN_ISO_ALPHA_3_CYM = 3885UL,
      FI_EN_ISO_ALPHA_3_CAF = 3110UL,
      FI_EN_ISO_ALPHA_3_TCD = 20580UL,
      FI_EN_ISO_ALPHA_3_CHL = 3340UL,
      FI_EN_ISO_ALPHA_3_CHN = 3342UL,
      FI_EN_ISO_ALPHA_3_COL = 3564UL,
      FI_EN_ISO_ALPHA_3_COM = 3565UL,
      FI_EN_ISO_ALPHA_3_COG = 3559UL,
      FI_EN_ISO_ALPHA_3_COD = 3556UL,
      FI_EN_ISO_ALPHA_3_COK = 3563UL,
      FI_EN_ISO_ALPHA_3_CRI = 3657UL,
      FI_EN_ISO_ALPHA_3_CIV = 3382UL,
      FI_EN_ISO_ALPHA_3_HRV = 8790UL,
      FI_EN_ISO_ALPHA_3_CUB = 3746UL,
      FI_EN_ISO_ALPHA_3_CYP = 3888UL,
      FI_EN_ISO_ALPHA_3_CZE = 3909UL,
      FI_EN_ISO_ALPHA_3_DNK = 4555UL,
      FI_EN_ISO_ALPHA_3_DJI = 4425UL,
      FI_EN_ISO_ALPHA_3_DMA = 4513UL,
      FI_EN_ISO_ALPHA_3_DOM = 4589UL,
      FI_EN_ISO_ALPHA_3_ECU = 5237UL,
      FI_EN_ISO_ALPHA_3_EGY = 5369UL,
      FI_EN_ISO_ALPHA_3_SLV = 19862UL,
      FI_EN_ISO_ALPHA_3_GNQ = 7633UL,
      FI_EN_ISO_ALPHA_3_ERI = 5705UL,
      FI_EN_ISO_ALPHA_3_EST = 5748UL,
      FI_EN_ISO_ALPHA_3_ETH = 5768UL,
      FI_EN_ISO_ALPHA_3_FLK = 6539UL,
      FI_EN_ISO_ALPHA_3_FRO = 6735UL,
      FI_EN_ISO_ALPHA_3_FJI = 6473UL,
      FI_EN_ISO_ALPHA_3_FIN = 6446UL,
      FI_EN_ISO_ALPHA_3_FRA = 6721UL,
      FI_EN_ISO_ALPHA_3_GUF = 7846UL,
      FI_EN_ISO_ALPHA_3_PYF = 17190UL,
      FI_EN_ISO_ALPHA_3_GAB = 7202UL,
      FI_EN_ISO_ALPHA_3_GMB = 7586UL,
      FI_EN_ISO_ALPHA_3_GEO = 7343UL,
      FI_EN_ISO_ALPHA_3_DEU = 4277UL,
      FI_EN_ISO_ALPHA_3_GHA = 7425UL,
      FI_EN_ISO_ALPHA_3_GIB = 7458UL,
      FI_EN_ISO_ALPHA_3_GRC = 7747UL,
      FI_EN_ISO_ALPHA_3_GRL = 7756UL,
      FI_EN_ISO_ALPHA_3_GRD = 7748UL,
      FI_EN_ISO_ALPHA_3_GLP = 7568UL,
      FI_EN_ISO_ALPHA_3_GUM = 7853UL,
      FI_EN_ISO_ALPHA_3_GTM = 7821UL,
      FI_EN_ISO_ALPHA_3_GIN = 7470UL,
      FI_EN_ISO_ALPHA_3_GNB = 7618UL,
      FI_EN_ISO_ALPHA_3_GUY = 7865UL,
      FI_EN_ISO_ALPHA_3_HTI = 8841UL,
      FI_EN_ISO_ALPHA_3_VAT = 22580UL,
      FI_EN_ISO_ALPHA_3_HND = 8644UL,
      FI_EN_ISO_ALPHA_3_HKG = 8551UL,
      FI_EN_ISO_ALPHA_3_HUN = 8878UL,
      FI_EN_ISO_ALPHA_3_ISL = 9836UL,
      FI_EN_ISO_ALPHA_3_IND = 9668UL,
      FI_EN_ISO_ALPHA_3_IDN = 9358UL,
      FI_EN_ISO_ALPHA_3_INT = 9684UL,
      FI_EN_ISO_ALPHA_3_IRN = 9806UL,
      FI_EN_ISO_ALPHA_3_IRQ = 9809UL,
      FI_EN_ISO_ALPHA_3_IRL = 9804UL,
      FI_EN_ISO_ALPHA_3_ISR = 9842UL,
      FI_EN_ISO_ALPHA_3_ITA = 9857UL,
      FI_EN_ISO_ALPHA_3_JAM = 10285UL,
      FI_EN_ISO_ALPHA_3_JPN = 10766UL,
      FI_EN_ISO_ALPHA_3_JOR = 10738UL,
      FI_EN_ISO_ALPHA_3_KAZ = 11322UL,
      FI_EN_ISO_ALPHA_3_KEN = 11438UL,
      FI_EN_ISO_ALPHA_3_KIR = 11570UL,
      FI_EN_ISO_ALPHA_3_PRK = 16971UL,
      FI_EN_ISO_ALPHA_3_KOR = 11762UL,
      FI_EN_ISO_ALPHA_3_KWT = 12020UL,
      FI_EN_ISO_ALPHA_3_KGZ = 11514UL,
      FI_EN_ISO_ALPHA_3_LAO = 12335UL,
      FI_EN_ISO_ALPHA_3_LVA = 12993UL,
      FI_EN_ISO_ALPHA_3_LBN = 12366UL,
      FI_EN_ISO_ALPHA_3_LSO = 12911UL,
      FI_EN_ISO_ALPHA_3_LBR = 12370UL,
      FI_EN_ISO_ALPHA_3_LBY = 12377UL,
      FI_EN_ISO_ALPHA_3_LIE = 12581UL,
      FI_EN_ISO_ALPHA_3_LTU = 12949UL,
      FI_EN_ISO_ALPHA_3_LUX = 12984UL,
      FI_EN_ISO_ALPHA_3_MAC = 13347UL,
      FI_EN_ISO_ALPHA_3_MKD = 13668UL,
      FI_EN_ISO_ALPHA_3_MDG = 13447UL,
      FI_EN_ISO_ALPHA_3_MWI = 14057UL,
      FI_EN_ISO_ALPHA_3_MYS = 14131UL,
      FI_EN_ISO_ALPHA_3_MDV = 13462UL,
      FI_EN_ISO_ALPHA_3_MLI = 13705UL,
      FI_EN_ISO_ALPHA_3_MLT = 13716UL,
      FI_EN_ISO_ALPHA_3_MHL = 13580UL,
      FI_EN_ISO_ALPHA_3_MTQ = 13969UL,
      FI_EN_ISO_ALPHA_3_MRT = 13908UL,
      FI_EN_ISO_ALPHA_3_MUS = 14003UL,
      FI_EN_ISO_ALPHA_3_MYT = 14132UL,
      FI_EN_ISO_ALPHA_3_MEX = 13496UL,
      FI_EN_ISO_ALPHA_3_FSM = 6765UL,
      FI_EN_ISO_ALPHA_3_MDA = 13441UL,
      FI_EN_ISO_ALPHA_3_MCO = 13423UL,
      FI_EN_ISO_ALPHA_3_MNG = 13767UL,
      FI_EN_ISO_ALPHA_3_MNE = 13765UL,
      FI_EN_ISO_ALPHA_3_MSR = 13938UL,
      FI_EN_ISO_ALPHA_3_MAR = 13362UL,
      FI_EN_ISO_ALPHA_3_MOZ = 13818UL,
      FI_EN_ISO_ALPHA_3_MMR = 13746UL,
      FI_EN_ISO_ALPHA_3_NAM = 14381UL,
      FI_EN_ISO_ALPHA_3_NRU = 14933UL,
      FI_EN_ISO_ALPHA_3_NPL = 14860UL,
      FI_EN_ISO_ALPHA_3_NLD = 14724UL,
      FI_EN_ISO_ALPHA_3_ANT = 1492UL,
      FI_EN_ISO_ALPHA_3_NCL = 14444UL,
      FI_EN_ISO_ALPHA_3_NZL = 15180UL,
      FI_EN_ISO_ALPHA_3_NIC = 14627UL,
      FI_EN_ISO_ALPHA_3_NER = 14514UL,
      FI_EN_ISO_ALPHA_3_NGA = 14561UL,
      FI_EN_ISO_ALPHA_3_NIU = 14645UL,
      FI_EN_ISO_ALPHA_3_NFK = 14539UL,
      FI_EN_ISO_ALPHA_3_MNP = 13776UL,
      FI_EN_ISO_ALPHA_3_NOR = 14834UL,
      FI_EN_ISO_ALPHA_3_PSE = 16997UL,
      FI_EN_ISO_ALPHA_3_OMN = 15790UL,
      FI_EN_ISO_ALPHA_3_PAK = 16427UL,
      FI_EN_ISO_ALPHA_3_PLW = 16791UL,
      FI_EN_ISO_ALPHA_3_PAN = 16430UL,
      FI_EN_ISO_ALPHA_3_PNG = 16839UL,
      FI_EN_ISO_ALPHA_3_PRY = 16985UL,
      FI_EN_ISO_ALPHA_3_PER = 16562UL,
      FI_EN_ISO_ALPHA_3_PHL = 16652UL,
      FI_EN_ISO_ALPHA_3_PCN = 16494UL,
      FI_EN_ISO_ALPHA_3_POL = 16876UL,
      FI_EN_ISO_ALPHA_3_PRT = 16980UL,
      FI_EN_ISO_ALPHA_3_PRI = 16969UL,
      FI_EN_ISO_ALPHA_3_QAT = 17460UL,
      FI_EN_ISO_ALPHA_3_REU = 18613UL,
      FI_EN_ISO_ALPHA_3_ROU = 18933UL,
      FI_EN_ISO_ALPHA_3_RUS = 19123UL,
      FI_EN_ISO_ALPHA_3_RWA = 19169UL,
      FI_EN_ISO_ALPHA_3_KNA = 11713UL,
      FI_EN_ISO_ALPHA_3_LCA = 12385UL,
      FI_EN_ISO_ALPHA_3_VCT = 22644UL,
      FI_EN_ISO_ALPHA_3_WSM = 24173UL,
      FI_EN_ISO_ALPHA_3_SMR = 19890UL,
      FI_EN_ISO_ALPHA_3_STP = 20112UL,
      FI_EN_ISO_ALPHA_3_SAU = 19509UL,
      FI_EN_ISO_ALPHA_3_SEN = 19630UL,
      FI_EN_ISO_ALPHA_3_SRB = 20034UL,
      FI_EN_ISO_ALPHA_3_SYC = 20259UL,
      FI_EN_ISO_ALPHA_3_SLE = 19845UL,
      FI_EN_ISO_ALPHA_3_SGP = 19696UL,
      FI_EN_ISO_ALPHA_3_SVK = 20171UL,
      FI_EN_ISO_ALPHA_3_SVN = 20174UL,
      FI_EN_ISO_ALPHA_3_SLB = 19842UL,
      FI_EN_ISO_ALPHA_3_SOM = 19949UL,
      FI_EN_ISO_ALPHA_3_ZAF = 26662UL,
      FI_EN_ISO_ALPHA_3_ESP = 5744UL,
      FI_EN_ISO_ALPHA_3_LKA = 12641UL,
      FI_EN_ISO_ALPHA_3_SHN = 19726UL,
      FI_EN_ISO_ALPHA_3_SPM = 19981UL,
      FI_EN_ISO_ALPHA_3_SDN = 19598UL,
      FI_EN_ISO_ALPHA_3_SUR = 20146UL,
      FI_EN_ISO_ALPHA_3_SJM = 19789UL,
      FI_EN_ISO_ALPHA_3_SWZ = 20218UL,
      FI_EN_ISO_ALPHA_3_SWE = 20197UL,
      FI_EN_ISO_ALPHA_3_CHE = 3333UL,
      FI_EN_ISO_ALPHA_3_SYR = 20274UL,
      FI_EN_ISO_ALPHA_3_TWN = 21230UL,
      FI_EN_ISO_ALPHA_3_TJK = 20811UL,
      FI_EN_ISO_ALPHA_3_TZA = 21313UL,
      FI_EN_ISO_ALPHA_3_THA = 20737UL,
      FI_EN_ISO_ALPHA_3_TLS = 20883UL,
      FI_EN_ISO_ALPHA_3_TGO = 20719UL,
      FI_EN_ISO_ALPHA_3_TKL = 20844UL,
      FI_EN_ISO_ALPHA_3_TON = 20974UL,
      FI_EN_ISO_ALPHA_3_TTO = 21135UL,
      FI_EN_ISO_ALPHA_3_TUN = 21166UL,
      FI_EN_ISO_ALPHA_3_TUR = 21170UL,
      FI_EN_ISO_ALPHA_3_TKM = 20845UL,
      FI_EN_ISO_ALPHA_3_TCA = 20577UL,
      FI_EN_ISO_ALPHA_3_TUV = 21174UL,
      FI_EN_ISO_ALPHA_3_UGA = 21729UL,
      FI_EN_ISO_ALPHA_3_UKR = 21874UL,
      FI_EN_ISO_ALPHA_3_ARE = 1605UL,
      FI_EN_ISO_ALPHA_3_GBR = 7250UL,
      FI_EN_ISO_ALPHA_3_USA = 22113UL,
      FI_EN_ISO_ALPHA_3_URY = 22105UL,
      FI_EN_ISO_ALPHA_3_UZB = 22338UL,
      FI_EN_ISO_ALPHA_3_VUT = 23220UL,
      FI_EN_ISO_ALPHA_3_VEN = 22702UL,
      FI_EN_ISO_ALPHA_3_VNM = 22989UL,
      FI_EN_ISO_ALPHA_3_VGB = 22754UL,
      FI_EN_ISO_ALPHA_3_VIR = 22834UL,
      FI_EN_ISO_ALPHA_3_WLF = 23942UL,
      FI_EN_ISO_ALPHA_3_ESH = 5736UL,
      FI_EN_ISO_ALPHA_3_YEM = 25773UL,
      FI_EN_ISO_ALPHA_3_YUG = 26279UL,
      FI_EN_ISO_ALPHA_3_ZMB = 27042UL,
      FI_EN_ISO_ALPHA_3_ZWE = 27365UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e16_pv_ISOCountryCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e16_pv_ISOCountryCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_mount_angles : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_mount_angles();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   sensor_fi_tcl_mount_angles& operator=(const sensor_fi_tcl_mount_angles& coRef);
   sensor_fi_tcl_mount_angles(const sensor_fi_tcl_mount_angles& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 AngRX;
   tU8 AngRY;
   tU8 AngRZ;
   tU8 AngSX;
   tU8 AngSY;
   tU8 AngSZ;
   tU8 AngTX;
   tU8 AngTY;
   tU8 AngTZ;
   virtual ~sensor_fi_tcl_mount_angles();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const sensor_fi_tcl_mount_angles& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

class sensor_fi_tcl_e8_mounting_source : public sensor_fi_tclVisitableTypeBase
{
public:

   sensor_fi_tcl_e8_mounting_source();

   enum tenType {
      FI_EN_SRC_MOUNTING_INVALID = 0UL,
      FI_EN_SRC_MOUNTING_KDS = 1UL,
      FI_EN_SRC_MOUNTING_CALIB = 2UL
   };
   tenType enType;
   virtual ~sensor_fi_tcl_e8_mounting_source();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const sensor_fi_tcl_e8_mounting_source& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(sensor_fi_tclVisitorBase& rfoVisitor);
};

#endif
