/* ******************************FileHeaderBegin*******************************
 *
 * @file        emmcif.c
 *
 * This file contains all functions for accessing emmc device 
 * 
 * 
 * global function:
 * -- 
 *
 * local function:
 * -- 
 *
 * @date        2018-03-07
 *
 * @note
 *
 *  &copy; Copyright RBEI. All Rights reserved!
 *
 *******************************FileHeaderEnd*********************************/

 
/*****************************************************************************/
/* include the system interface                                              */
/*****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <semaphore.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <grp.h>
#include "system_types.h"
#include "system_definition.h"
#include <errno.h>
#include "pdd_trace.h"
#include <dirent.h>

#ifdef __cplusplus
extern "C" {
#endif
/******************************************************************************
|defines and macros 
|----------------------------------------------------------------------------*/
#define MAX_LEN_PARTITION_NAME              24
#define MAX_LEN_DEVICE_NAME                 17

#define ROOT_DEVICE_NODE                    "mount  | grep -w / | awk {'print $1'}"
#define NODE_LENGTH                         32
#define eMMC_DEV_NODE_OFFSET                6
#define MAX_MMC_SLOT_FOR_GEN4               4
#define PDD_RAW_EMMC_USER_CLUSTER_SIZE      0x8000


#define EMMC_USER_SUBPART_CONFIG_FILENAME   "/etc/default/RaweMMCUserPartition.conf"
#define MAX_MMC_SLOT_FOR_GEN4    4
#define SYS_DIR_MMCHOST_PATH "/sys/class/mmc_host"
#define MAXLENGTH     255



/* sample for config file:
# Config file for RaweMMCUserPartition
#  start and end values in sector (1 sector = 512bytes)
#
# Partiotion name       DeviceNode           start        end
PDD_eMMCUserEarly      /dev/mmcblk1p128      524352       534592
*/

#define RAW_EMMC_ACCESS_RIGTHS              S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP
#define CLUSTER_SIZE                        32768//32KB
typedef int eMMCDevHndl;

/******************************************************************************** 
|typedefs and struct defs
|------------------------------------------------------------------------------*/
typedef struct 
{
    eMMCDevHndl DevHndl;
    tChar Partitionname[MAX_LEN_PARTITION_NAME];
    tChar DeviceNode[MAX_LEN_DEVICE_NAME];
    tU32 u32start, u32end;
}eMMCsubpart_info;

typedef struct 
{
    eMMCsubpart_info *subParts;
    tU32 NoOfeMMCUserParts;
}eMMCPartition;
eMMCPartition *eMMCPartitionInfo;

typedef enum
{
    eMMC_NOERROR            =  0,
    eMMC_INVALID_ARGUMENT   = -1,
    eMMC_ACCESS_ERROR       = -2,
    eMMC_OPEN_ERROR         = -3,
    eMMC_NO_MEMORY          = -4,
    eMMC_READ_ERROR         = -5,
    eMMC_SEEK_ERROR         = -6,
    eMMC_CONVERSION_ERROR   = -7,
    eMMC_INVALID_FILESIZE   = -8,
    eMMC_EMPTY_METAFILE     = -9,
    eMMC_CORRUPTED_FILE     = -10,
    eMMC_OUT_OF_RANGE       = -11,
    eMMC_RENAME_ERROR       = -12,
    eMMC_TOTALSIZE_ERROR    = -13

}ErrorValues;

/******************************************************************************/
/* static  variable                                                           */
/******************************************************************************/
static tS32 s32CheckMMCDevNode(void);
static tBool bPDDPartitionExist =TRUE;


/******************************************************************************
| variable definition (scope: module-global)
|----------------------------------------------------------------------------*/
eMMCDevHndl eMMC_open (char *devicename);
void eMMC_close (eMMCDevHndl DevHndl);
int eMMC_read (eMMCDevHndl handle, void *Readbuffer,unsigned int size, unsigned int offset);
int eMMC_write (eMMCDevHndl handle, void *Writebuffer,unsigned int size, unsigned int offset);

/*****************************************************************************/
/* declaration local function                                                */
/*****************************************************************************/

/******************************************************************************
* FUNCTION: eMMC_open()
*
* DESCRIPTION: open the device for RAW eMMC area access
*
* PARAMETERS:
*     Devicename - Device name is nothing but partition name (PDD_eMMCUserEarly)
*
* RETURNS: 
*      Success: eMMCDevHndl 
*      Failure: error code 
*
* HISTORY:Created  2018 07 06
*****************************************************************************/

eMMCDevHndl eMMC_open(char *Devicename)
{
    tS32   fd = 0;
    tU32   len = 0;
    tS32   s32NodeNum;
    char   *line = NULL;
    FILE   *fpRaweMMCUserDetails = NULL;
    static tBool bConfigFileFound = FALSE;

    if (FALSE == bConfigFileFound)
    {
        eMMCPartitionInfo = (eMMCPartition*)malloc(sizeof(eMMCPartition));
        if (eMMCPartitionInfo)
        {
            memset(eMMCPartitionInfo, 0, sizeof(eMMCPartition));
            // get/parse the eMMC config file.
            fpRaweMMCUserDetails = fopen(EMMC_USER_SUBPART_CONFIG_FILENAME, "r");

            if (NULL == fpRaweMMCUserDetails)
            {
                fprintf(stderr, "pdd_emmcif: open failed for RaweMMCUserPartition.conf\n");
                return 0;
            }
            else
            {
                //fprintf(stderr, "\n EMMC_USER_SUBPART_CONFIG_FILENAME open passed" );
                while (getline(&line, (size_t *)&len, fpRaweMMCUserDetails) != -1)
                {
                    char* VcpStr = NULL;

                    VcpStr = strchr(line, '\n');
                    if (VcpStr != NULL)
                    {
                        *VcpStr = '\0';
                    }
                    if ((line[0] != '#') && (len > 1))
                    {
                        eMMCPartitionInfo->subParts = (eMMCsubpart_info*)malloc(sizeof(eMMCsubpart_info));
                        if (NULL != eMMCPartitionInfo->subParts)
                        {
                            //fprintf(stderr, "\n EMMC_USER_SUBPART_CONFIG_FILENAME parsing started" );
                            memset(eMMCPartitionInfo->subParts, 0, sizeof(eMMCsubpart_info));
                            sscanf(line, "%s %s %d %d", eMMCPartitionInfo->subParts->Partitionname, eMMCPartitionInfo->subParts->DeviceNode, &eMMCPartitionInfo->subParts->u32start, &eMMCPartitionInfo->subParts->u32end);

                            fprintf(stderr, "\n Partition Name %s DeviceNode %s start %d end %d", eMMCPartitionInfo->subParts->Partitionname, eMMCPartitionInfo->subParts->DeviceNode, eMMCPartitionInfo->subParts->u32start, eMMCPartitionInfo->subParts->u32end );
                            eMMCPartitionInfo->NoOfeMMCUserParts++;
                            bConfigFileFound = TRUE;
                            break;
                        }
                    }
                }
                if (line)
                    free(line);
                line = NULL;
                fclose(fpRaweMMCUserDetails);                
            }
        }
        else
        {
            fprintf(stderr, "\nemmcif:Memory allocation failed");
            return 0;
        }
    }
    if (TRUE == bConfigFileFound)
    {
        if (!strcmp(eMMCPartitionInfo->subParts->Partitionname, Devicename))
        {
            //fprintf(stderr, "\n Device Node is %s",eMMCPartitionInfo->subParts->DeviceNode );
            fd = (int)open(eMMCPartitionInfo->subParts->DeviceNode, O_RDWR, RAW_EMMC_ACCESS_RIGTHS);
            
            if(fd>=0)
            {
                eMMCPartitionInfo->subParts->DevHndl = fd;
                //fprintf(stderr,"\n Device open passed");
                
            }
            else
            {
                fprintf(stderr,"\n Device open failed for %s",eMMCPartitionInfo->subParts->DeviceNode);				
               
                memset (eMMCPartitionInfo->subParts->DeviceNode,0,MAX_LEN_DEVICE_NAME );
				s32NodeNum = s32CheckMMCDevNode();
				snprintf(eMMCPartitionInfo->subParts->DeviceNode, (size_t)(MAX_LEN_DEVICE_NAME-1), "/dev/mmcblk%d", s32NodeNum);
                //Open the eMMC device /dev/mmcblk1 or /dev/mmcblk2  by offset
                fd = (int)open(eMMCPartitionInfo->subParts->DeviceNode, O_RDWR, RAW_EMMC_ACCESS_RIGTHS);

            	if (fd > 0)
            	{
                	fprintf(stderr,"\n Open via OFFSET\n");
                	eMMCPartitionInfo->subParts->DevHndl = fd;
                	fprintf(stderr, "\n eMMCPartitionInfo->subParts->DevHndl is %d", eMMCPartitionInfo->subParts->DevHndl );
                	bPDDPartitionExist=FALSE;
            	}
            	else
            	{
                	eMMCPartitionInfo->subParts->DevHndl = -1;
                	fprintf(stderr, "\nEMMC: Device open %s is failed with errno string %s", eMMCPartitionInfo->subParts->DeviceNode, strerror(errno));
            	}             
        	}
        	
    	}
    }	
    return fd;
}

/******************************************************************************
* FUNCTION: eMMC_close()
*
* DESCRIPTION: Close the device 
*
* PARAMETERS:
*     DeviceHandl - DeviceHandl is used to close the device
*
* RETURNS: 
*      None
*
* HISTORY:Created  2018 07 06
*****************************************************************************/

void eMMC_close(eMMCDevHndl DeviceHandl)
{
    if ( DeviceHandl > 0)
    {
        if (eMMCPartitionInfo->subParts->DevHndl == DeviceHandl)
        {
            close(DeviceHandl);
            eMMCPartitionInfo->subParts->DevHndl = -1;
            if (eMMCPartitionInfo->subParts != NULL)
            {
                free(eMMCPartitionInfo->subParts);
                eMMCPartitionInfo->subParts = NULL;
            }
            if (eMMCPartitionInfo != NULL)
            {
                free(eMMCPartitionInfo);
                eMMCPartitionInfo = NULL;
            }
        }
    }
}

/******************************************************************************
* FUNCTION: eMMC_read()
*
* DESCRIPTION: Read the data from the RAW eMMC area 
*
* PARAMETERS:
*     handle         - DeviceHandl is used to read the device
*     Readbuffer   - used to store the read data
*     size             - size of the data to be read
*     offset           - location of device to read
*
* RETURNS: 
*      success       - TRUE
*      Failure        -  FALSE (or) Read Error code
*
* HISTORY:Created  2018 07 06
*****************************************************************************/

int eMMC_read(eMMCDevHndl handle, void *Readbuffer, unsigned int size, unsigned int offset)
{
    int Result = 0;
    tBool bValidHndl = FALSE;
    long long ExactOffset;

    if ((NULL != Readbuffer) && (handle > 0))
    {
        if (eMMCPartitionInfo->subParts->DevHndl == handle)
        {
            bValidHndl = TRUE;
        }

        if (TRUE == bValidHndl)
        {
            
            if (((offset + size) > (((eMMCPartitionInfo->subParts->u32end) - (eMMCPartitionInfo->subParts->u32start)) * 512)) && ( size > CLUSTER_SIZE))
            {
                fprintf(stderr, "\nPartition Boundry crossed");
            }
            else
            {
                if ( bPDDPartitionExist )        	
                   ExactOffset=(long long)offset;
                else
                   ExactOffset = ((long long)offset + ((long long)eMMCPartitionInfo->subParts->u32start * 512));
                   
                if (ExactOffset == lseek(handle, ExactOffset, SEEK_SET))
                {
                    Result = (int)read(handle, Readbuffer, size);
                    if (Result != (int)size)
                    {
                        fprintf(stderr, "\nread failed at the offset %lld with the return code %d & errno in string %s", ExactOffset, Result, strerror(errno));
                        Result = 0;
                    }
                    else
                        Result = 1;
                }
                else
                    fprintf(stderr, "\nseek error with errno string %s", strerror(errno));
            }
        }
    }
    else
        fprintf(stderr, "\neither read buffer or handle is NULL");
    return Result;
}

/******************************************************************************
* FUNCTION: eMMC_write()
*
* DESCRIPTION: Write the data to the RAW eMMC area 
*
* PARAMETERS:
*     handle         - DeviceHandl is used to write the device
*     Writebuffer   - data to be written
*     size             - size of the data to be write
*     offset           - location of device to write
*
* RETURNS: 
*      success       - TRUE
*      Failure        -  FALSE (or) write Error code
*
* HISTORY:Created  2018 07 06
*****************************************************************************/


int eMMC_write(eMMCDevHndl handle, void *Writebuffer, unsigned int size, unsigned int offset)
{
    int Result = 0;
    tBool bValidHndl = FALSE;
    long long ExactOffset;

    if ((NULL != Writebuffer) && ( handle > 0))
    {
        // find handle
        if (eMMCPartitionInfo->subParts->DevHndl == handle)
        {
            bValidHndl = TRUE;
        }
        if (TRUE == bValidHndl)
        {
            if (((offset + size) > (((eMMCPartitionInfo->subParts->u32end) - (eMMCPartitionInfo->subParts->u32start)) * 512)) && ( size > CLUSTER_SIZE))
            {
                fprintf(stderr, "\nPartition Boundry crossed ");
            }
            else
            {
               if ( bPDDPartitionExist )        	
                  ExactOffset=(long long)offset;
               else
                  ExactOffset = ((long long)offset + ((long long)eMMCPartitionInfo->subParts->u32start * 512));
                
                if ((lseek(handle,ExactOffset,SEEK_SET)) == ExactOffset)
                {
                    Result = (int)write(handle, Writebuffer, size);
                    if (Result != (int)size)
                    {
                        fprintf(stderr, "\nwrite failed at the offset %lld with the return code %d errno in string %s", ExactOffset, Result,strerror(errno));
                         Result = 0;
                    }
                    else
                        Result = 1;
                }
                else
                    printf("\nseek error with errno %s", strerror(errno));
            }
        }
    }
    else
        fprintf(stderr, "\neither write buffer or handle is NULL");
    return Result;
}

/******************************************************************************
* FUNCTION: s32CheckMMCDevNode()
*
* DESCRIPTION: Find the eMMC device node
*
* PARAMETERS:
*     None
*
* RETURNS: 
*      success       - Device Node no
*      Failure        -  FALSE (or)  Error code
*
* HISTORY:Created  2018 07 06
*****************************************************************************/

static tS32 s32CheckMMCDevNode(void)
{
    int    s32Result = -1;
/*
Bug 386215: The value of rootstat.st_dev and p1stat.st_rdev does not match when the target boots in bootchain2 mode
so we used different mechanism to identify the mmc device node with the help of /sys/class..
*/


    DIR             *sysDirMmcHostPath,
		                    *sysDirMmcCardPath;	
    struct dirent   *dirMmcHostEntry,
                    *dirMmcCardEntry;
    FILE            *fpMmcDevType;
    /* Since, we know the file format for mmc_card amd host info , 
       so we have limited the length by 32 bytes*/
    unsigned char             devType [8]                 = {0};
    char            	mmcHostCardPath [MAXLENGTH] = {0};
    unsigned char           bFoundPath = FALSE;
	/*MMC_HOST: Open the sysfs directory /sys/class/mmc_host/ */
	sysDirMmcHostPath = opendir(SYS_DIR_MMCHOST_PATH);
	if (NULL == sysDirMmcHostPath)
    {
		fprintf(stderr, "%s: Open Dir %s Fails!!! E:%d\n",__func__,SYS_DIR_MMCHOST_PATH,errno);
    }
	else
    {
        /* MMC_HOST: Read the entries in the host direcotry and find the device type as MMC */
        while (((dirMmcHostEntry= readdir(sysDirMmcHostPath)) != NULL) && (bFoundPath == FALSE))
        {
            if (!strncmp(dirMmcHostEntry->d_name,"mmc",3)) 
            {
                snprintf(mmcHostCardPath,sizeof(mmcHostCardPath)-1,"%s/%s/"
                                                                        ,SYS_DIR_MMCHOST_PATH
                                                                        ,dirMmcHostEntry->d_name);
                sysDirMmcCardPath = opendir (mmcHostCardPath);
                if (NULL == sysDirMmcCardPath) 
                {
					fprintf(stderr, "%s: Open Dir %s Fails!!! E:%d\n",__func__,mmcHostCardPath,errno);
                    continue;
                }
                else
                {
                    while (((dirMmcCardEntry= readdir(sysDirMmcCardPath)) != NULL)  && (bFoundPath == FALSE))
                    {
                        if (!strncmp (dirMmcCardEntry->d_name, dirMmcHostEntry->d_name, strlen (dirMmcHostEntry->d_name))) 
                        {
                            snprintf(mmcHostCardPath,sizeof(mmcHostCardPath)-1,"%s/%s/%s/type"
                                                                            ,SYS_DIR_MMCHOST_PATH
                                                                            ,dirMmcHostEntry->d_name
                                                                            ,dirMmcCardEntry->d_name);
                            fpMmcDevType = fopen(mmcHostCardPath,"r");
                            if(fpMmcDevType == NULL )
                            {
								fprintf(stderr, "%s: File open fails %s E:%d\n",__func__,mmcHostCardPath,errno);
                                break;
                            }
                            fgets( (char *)devType, sizeof(devType)-1, fpMmcDevType);
                            if (!strncmp((const char*)devType,(const char*)"MMC",3))
                            {
								if (!strncmp((const char*)dirMmcHostEntry->d_name,(const char*)"mmc0",4))
								{
									s32Result=0;
								}
								else if (!strncmp((const char*)dirMmcHostEntry->d_name,(const char*)"mmc1",4))
								{
									s32Result=1;
								}
								else if (!strncmp((const char*)dirMmcHostEntry->d_name,(const char*)"mmc2",4))
								{
									s32Result=2;
								}
								else if (!strncmp((const char*)dirMmcHostEntry->d_name,(const char*)"mmc3",4))
								{
									s32Result=3;
								}
                                bFoundPath = TRUE;
                            }
                            fclose(fpMmcDevType);
                        }
                    }
                    closedir(sysDirMmcCardPath);
                }
            }
        }
        closedir(sysDirMmcHostPath);
    }
    return s32Result;
}

#ifdef __cplusplus
}
#endif
